declare class Accelerometer {
/**
 * Associates a comment with a accelerometer.
 * @param Comment Comment that will be attached to the accelerometer
 */
    AssociateComment(Comment: Comment): void;

/**
 * Blanks the accelerometer
 */
    Blank(): void;

/**
 * Blanks all of the accelerometers in the model.
 * @param Model Model that all accelerometers will be blanked in
 * @param redraw If model should be redrawn or not. If omitted redraw is false. If you want to do several (un)blanks and only redraw after the last one then use false for all redraws apart from the last one. Alternatively you can redraw using View.Redraw().
 */
    static BlankAll(Model: Model, redraw?: boolean): void;

/**
 * Blanks all of the flagged accelerometers in the model.
 * @param Model Model that all the flagged accelerometers will be blanked in
 * @param flag Flag set on the accelerometers that you want to blank
 * @param redraw If model should be redrawn or not. If omitted redraw is false. If you want to do several (un)blanks and only redraw after the last one then use false for all redraws apart from the last one. Alternatively you can redraw using View.Redraw().
 */
    static BlankFlagged(Model: Model, flag: number, redraw?: boolean): void;

/**
 * Checks if the accelerometer is blanked or not.
 * @returns true if blanked, false if not.
 */
    Blanked(): boolean;

/**
 * Starts an edit panel in Browse mode.
 * @param modal If this window is modal (blocks the user from doing anything else in PRIMER until this window is dismissed). If omitted the window will be modal.
 */
    Browse(modal?: boolean): void;

/**
 * Clears a flag on the accelerometer.
 * @param flag Flag to clear on the accelerometer
 */
    ClearFlag(flag: number): void;

/**
 * Copies the accelerometer. The target include of the copied accelerometer can be set using Options.copy_target_include.
 * @param range If you want to keep the copied item in the range specified for the current include. Default value is false. To set current include, use Include.MakeCurrentLayer().
 * @returns Accelerometer object
 */
    Copy(range?: boolean): Accelerometer;

/**
 * Starts an interactive editing panel to create a accelerometer
 * @param Model Model that the accelerometer will be created in.
 * @param modal If this window is modal (blocks the user from doing anything else in PRIMER until this window is dismissed). If omitted the window will be modal.
 * @returns Accelerometer object (or null if not made).
 */
    static Create(Model: Model, modal?: boolean): Accelerometer;

/**
 * Detaches a comment from a accelerometer.
 * @param Comment Comment that will be detached from the accelerometer
 */
    DetachComment(Comment: Comment): void;

/**
 * Starts an interactive editing panel.
 * @param modal If this window is modal (blocks the user from doing anything else in PRIMER until this window is dismissed). If omitted the window will be modal.
 */
    Edit(modal?: boolean): void;

/**
 * Adds an error for accelerometer. For more details on checking see the Check class.
 * @param message The error message to give
 * @param details An optional detailed error message
 */
    Error(message: string, details?: string): void;

/**
 * Extracts the actual colour used for accelerometer. By default in PRIMER many entities such as elements get their colour automatically from the part that they are in. PRIMER cycles through 13 default colours based on the label of the entity. In this case the accelerometer colour property will return the value Colour.PART instead of the actual colour. This method will return the actual colour which is used for drawing the accelerometer.
 * @returns colour value (integer)
 */
    ExtractColour(): number;

/**
 * Returns the first accelerometer in the model.
 * @param Model Model to get first accelerometer in
 * @returns Accelerometer object (or null if there are no accelerometers in the model).
 */
    static First(Model: Model): Accelerometer;

/**
 * Returns the first free accelerometer label in the model. Also see Accelerometer.LastFreeLabel(), Accelerometer.NextFreeLabel() and Model.FirstFreeItemLabel().
 * @param Model Model to get first free accelerometer label in
 * @param layer Include file (0 for the main file) to search for labels in (Equivalent to First free in layer in editing panels). If omitted the whole model will be used (Equivalent to First free in editing panels).
 * @returns Accelerometer label.
 */
    static FirstFreeLabel(Model: Model, layer?: number): number;

/**
 * Flags all of the accelerometers in the model with a defined flag.
 * @param Model Model that all accelerometers will be flagged in
 * @param flag Flag to set on the accelerometers
 */
    static FlagAll(Model: Model, flag: number): void;

/**
 * Checks if the accelerometer is flagged or not.
 * @param flag Flag to test on the accelerometer
 * @returns true if flagged, false if not.
 */
    Flagged(flag: number): boolean;

/**
 * Calls a function for each accelerometer in the model. Note that ForEach has been designed to make looping over accelerometers as fast as possible and so has some limitations. Firstly, a single temporary Accelerometer object is created and on each function call it is updated with the current accelerometer data. This means that you should not try to store the Accelerometer object for later use (e.g. in an array) as it is temporary. Secondly, you cannot create new accelerometers inside a ForEach loop. 
 * @param Model Model that all accelerometers are in
 * @param func Function to call for each accelerometer
 * @param extra An optional extra object/array/string etc that will appended to arguments when calling the function
 */
    static ForEach(Model: Model, func: () => void, extra?: any): void;

/**
 * Returns an array of Accelerometer objects or properties for all of the accelerometers in a model in PRIMER. If the optional property argument is not given then an array of Accelerometer objects is returned. If the property argument is given, that property value for each accelerometer is returned in the array instead of a Accelerometer object
 * @param Model Model to get accelerometers from
 * @param property Name for property to get for all accelerometers in the model
 * @returns Array of Accelerometer objects or properties
 */
    static GetAll(Model: Model, property?: string): Accelerometer[];

/**
 * Extracts the comments associated to a accelerometer.
 * @returns Array of Comment objects (or null if there are no comments associated to the node).
 */
    GetComments(): Comment[];

/**
 * Returns an array of Accelerometer objects for all of the flagged accelerometers in a model in PRIMER If the optional property argument is not given then an array of Accelerometer objects is returned. If the property argument is given, then that property value for each accelerometer is returned in the array instead of a Accelerometer object
 * @param Model Model to get accelerometers from
 * @param flag Flag set on the accelerometers that you want to retrieve
 * @param property Name for property to get for all flagged accelerometers in the model
 * @returns Array of Accelerometer objects or properties
 */
    static GetFlagged(Model: Model, flag: number, property?: string): Accelerometer[];

/**
 * Returns the Accelerometer object for a accelerometer ID.
 * @param Model Model to find the accelerometer in
 * @param number number of the accelerometer you want the Accelerometer object for
 * @returns Accelerometer object (or null if accelerometer does not exist).
 */
    static GetFromID(Model: Model, number: number): Accelerometer;

/**
 * Checks if a Accelerometer property is a parameter or not. Note that object properties that are parameters are normally returned as the integer or float parameter values as that is virtually always what the user would want. For this function to work the JavaScript interpreter must use the parameter name instead of the value. This can be done by setting the Options.property_parameter_names option to true before calling the function and then resetting it to false afterwards.. This behaviour can also temporarily be switched by using the Accelerometer.ViewParameters() method and 'method chaining' (see the examples below).
 * @param prop accelerometer property to get parameter for
 * @returns Parameter object if property is a parameter, null if not.
 */
    GetParameter(prop: string): Parameter;

/**
 * Returns the keyword for this accelerometer (*ELEMENT_SEATBELT_ACCELEROMETER) Note that a carriage return is not added. See also Accelerometer.KeywordCards()
 * @returns string containing the keyword.
 */
    Keyword(): string;

/**
 * Returns the keyword cards for the accelerometer. Note that a carriage return is not added. See also Accelerometer.Keyword()
 * @returns string containing the cards.
 */
    KeywordCards(): string;

/**
 * Returns the last accelerometer in the model.
 * @param Model Model to get last accelerometer in
 * @returns Accelerometer object (or null if there are no accelerometers in the model).
 */
    static Last(Model: Model): Accelerometer;

/**
 * Returns the last free accelerometer label in the model. Also see Accelerometer.FirstFreeLabel(), Accelerometer.NextFreeLabel() and see Model.LastFreeItemLabel()
 * @param Model Model to get last free accelerometer label in
 * @param layer Include file (0 for the main file) to search for labels in (Equivalent to Highest free in layer in editing panels). If omitted the whole model will be used.
 * @returns Accelerometer label.
 */
    static LastFreeLabel(Model: Model, layer?: number): number;

/**
 * Returns the next accelerometer in the model.
 * @returns Accelerometer object (or null if there are no more accelerometers in the model).
 */
    Next(): Accelerometer;

/**
 * Returns the next free (highest+1) accelerometer label in the model. Also see Accelerometer.FirstFreeLabel(), Accelerometer.LastFreeLabel() and Model.NextFreeItemLabel()
 * @param Model Model to get next free accelerometer label in
 * @param layer Include file (0 for the main file) to search for labels in (Equivalent to Highest+1 in layer in editing panels). If omitted the whole model will be used (Equivalent to Highest+1 in editing panels).
 * @returns Accelerometer label.
 */
    static NextFreeLabel(Model: Model, layer?: number): number;

/**
 * Allows the user to pick a accelerometer.
 * @param prompt Text to display as a prompt to the user
 * @param limit If the argument is a Model then only accelerometers from that model can be picked. If the argument is a Flag then only accelerometers that are flagged with limit can be selected. If omitted, or null, any accelerometers from any model can be selected. from any model.
 * @param modal If picking is modal (blocks the user from doing anything else in PRIMER until this window is dismissed). If omitted the pick will be modal.
 * @param button_text By default the window with the prompt will have a button labelled 'Cancel' which if pressed will cancel the pick and return null. If you want to change the text on the button use this argument. If omitted 'Cancel' will be used.
 * @returns Accelerometer object (or null if not picked)
 */
    static Pick(prompt: string, limit?: Model | number, modal?: boolean, button_text?: string): Accelerometer;

/**
 * Returns the previous accelerometer in the model.
 * @returns Accelerometer object (or null if there are no more accelerometers in the model).
 */
    Previous(): Accelerometer;

/**
 * Renumbers all of the accelerometers in the model.
 * @param Model Model that all accelerometers will be renumbered in
 * @param start Start point for renumbering
 */
    static RenumberAll(Model: Model, start: number): void;

/**
 * Renumbers all of the flagged accelerometers in the model.
 * @param Model Model that all the flagged accelerometers will be renumbered in
 * @param flag Flag set on the accelerometers that you want to renumber
 * @param start Start point for renumbering
 */
    static RenumberFlagged(Model: Model, flag: number, start: number): void;

/**
 * Allows the user to select accelerometers using standard PRIMER object menus.
 * @param flag Flag to use when selecting accelerometers
 * @param prompt Text to display as a prompt to the user
 * @param limit If the argument is a Model then only accelerometers from that model can be selected. If the argument is a Flag then only accelerometers that are flagged with limit can be selected (limit should be different to flag). If omitted, or null, any accelerometers can be selected. from any model.
 * @param modal If selection is modal (blocks the user from doing anything else in PRIMER until this window is dismissed). If omitted the selection will be modal.
 * @returns Number of accelerometers selected or null if menu cancelled
 */
    static Select(flag: number, prompt: string, limit?: Model | number, modal?: boolean): number;

/**
 * Sets a flag on the accelerometer.
 * @param flag Flag to set on the accelerometer
 */
    SetFlag(flag: number): void;

/**
 * Sketches the accelerometer. The accelerometer will be sketched until you either call Accelerometer.Unsketch(), Accelerometer.UnsketchAll(), Model.UnsketchAll(), or delete the model
 * @param redraw If model should be redrawn or not after the accelerometer is sketched. If omitted redraw is true. If you want to sketch several accelerometers and only redraw after the last one then use false for redraw and call View.Redraw().
 */
    Sketch(redraw?: boolean): void;

/**
 * Sketches all of the flagged accelerometers in the model. The accelerometers will be sketched until you either call Accelerometer.Unsketch(), Accelerometer.UnsketchFlagged(), Model.UnsketchAll(), or delete the model
 * @param Model Model that all the flagged accelerometers will be sketched in
 * @param flag Flag set on the accelerometers that you want to sketch
 * @param redraw If model should be redrawn or not after the accelerometers are sketched. If omitted redraw is true. If you want to sketch flagged accelerometers several times and only redraw after the last one then use false for redraw and call View.Redraw().
 */
    static SketchFlagged(Model: Model, flag: number, redraw?: boolean): void;

/**
 * Returns the total number of accelerometers in the model.
 * @param Model Model to get total for
 * @param exists true if only existing accelerometers should be counted. If false or omitted referenced but undefined accelerometers will also be included in the total.
 * @returns number of accelerometers
 */
    static Total(Model: Model, exists?: boolean): number;

/**
 * Unblanks the accelerometer
 */
    Unblank(): void;

/**
 * Unblanks all of the accelerometers in the model.
 * @param Model Model that all accelerometers will be unblanked in
 * @param redraw If model should be redrawn or not. If omitted redraw is false. If you want to do several (un)blanks and only redraw after the last one then use false for all redraws apart from the last one. Alternatively you can redraw using View.Redraw().
 */
    static UnblankAll(Model: Model, redraw?: boolean): void;

/**
 * Unblanks all of the flagged accelerometers in the model.
 * @param Model Model that the flagged accelerometers will be unblanked in
 * @param flag Flag set on the accelerometers that you want to unblank
 * @param redraw If model should be redrawn or not. If omitted redraw is false. If you want to do several (un)blanks and only redraw after the last one then use false for all redraws apart from the last one. Alternatively you can redraw using View.Redraw().
 */
    static UnblankFlagged(Model: Model, flag: number, redraw?: boolean): void;

/**
 * Unsets a defined flag on all of the accelerometers in the model.
 * @param Model Model that the defined flag for all accelerometers will be unset in
 * @param flag Flag to unset on the accelerometers
 */
    static UnflagAll(Model: Model, flag: number): void;

/**
 * Unsketches the accelerometer.
 * @param redraw If model should be redrawn or not after the accelerometer is unsketched. If omitted redraw is true. If you want to unsketch several accelerometers and only redraw after the last one then use false for redraw and call View.Redraw().
 */
    Unsketch(redraw?: boolean): void;

/**
 * Unsketches all accelerometers.
 * @param Model Model that all accelerometers will be unblanked in
 * @param redraw If model should be redrawn or not after the accelerometers are unsketched. If omitted redraw is true. If you want to unsketch several things and only redraw after the last one then use false for redraw and call View.Redraw().
 */
    static UnsketchAll(Model: Model, redraw?: boolean): void;

/**
 * Unsketches all flagged accelerometers in the model.
 * @param Model Model that all accelerometers will be unsketched in
 * @param flag Flag set on the accelerometers that you want to unsketch
 * @param redraw If model should be redrawn or not after the accelerometers are unsketched. If omitted redraw is true. If you want to unsketch several things and only redraw after the last one then use false for redraw and call View.Redraw().
 */
    static UnsketchFlagged(Model: Model, flag: number, redraw?: boolean): void;

/**
 * Object properties that are parameters are normally returned as the integer or float parameter values as that is virtually always what the user would want. This function temporarily changes the behaviour so that if a property is a parameter the parameter name is returned instead. This can be used with 'method chaining' (see the example below) to make sure a property argument is correct.
 * @returns Accelerometer object.
 */
    ViewParameters(): Accelerometer;

/**
 * Adds a warning for accelerometer. For more details on checking see the Check class.
 * @param message The warning message to give
 * @param details An optional detailed warning message
 */
    Warning(message: string, details?: string): void;

/**
 * Returns the cross references for this accelerometer.
 * @returns Xrefs object.
 */
    Xrefs(): Xrefs;

/**
 * Create a new Seatbelt Accelerometer object.
 * @param Model Model that accelerometer will be created in
 * @param sbacid Accelerometer number. Also see the label property which is an alternative name for this.
 * @param nid1 Node number 1
 * @param nid2 Node number 2
 * @param nid3 Node number 3
 * @param igrav Gravitational acceleration due to body force loads is included in acceleration output if igrav is 0, removed if igrav is 1.
 * @param intopt Integration option; velocities are integrated from global accelerations and transformed into local system if intopt is 0, they are integrated directly from local accelerations if intopt is 1.
 * @param mass Optional added mass for accelerometer
 * @returns Accelerometer object
 */
    constructor(Model: Model, sbacid: number, nid1: number, nid2: number, nid3: number, igrav?: number, intopt?: number, mass?: number);

/**
 * Creates a string containing the accelerometer data in keyword format. Note that this contains the keyword header and the keyword cards. See also Accelerometer.Keyword() and Accelerometer.KeywordCards().
 * @returns string
 */
    toString(): string;

/** The colour of the accelerometer */
    colour: Colour;
/** true if accelerometer exists, false if referred to but not defined. */
    readonly exists: boolean;
/** Gravitational acceleration due to body force loads is included in acceleration output if igrav is 0, removed if igrav is 1. */
    igrav: number;
/** The Include file number that the accelerometer is in. */
    include: number;
/** Integration option; velocities are integrated from global accelerations and transformed into local system if intopt is 0, they are integrated directly from local accelerations if intopt is 1. */
    intopt: number;
/** Accelerometer number. Also see the sbacid property which is an alternative name for this. */
    label: number;
/** Optional added mass for accelerometer */
    mass: number;
/** The Model number that the accelerometer is in. */
    readonly model: number;
/** Node number 1 */
    nid1: number;
/** Node number 2 */
    nid2: number;
/** Node number 3 */
    nid3: number;
/** Accelerometer number. Also see the label property which is an alternative name for this. */
    sbacid: number;
/** The transparency of the accelerometer (0-100) 0% is opaque, 100% is transparent. */
    transparency: number;
}

declare class Airbag {
/**
 * Associates a comment with a airbag.
 * @param Comment Comment that will be attached to the airbag
 */
    AssociateComment(Comment: Comment): void;

/**
 * Starts an edit panel in Browse mode.
 * @param modal If this window is modal (blocks the user from doing anything else in PRIMER until this window is dismissed). If omitted the window will be modal.
 */
    Browse(modal?: boolean): void;

/**
 * Clears a flag on the airbag.
 * @param flag Flag to clear on the airbag
 */
    ClearFlag(flag: number): void;

/**
 * Copies the airbag. The target include of the copied airbag can be set using Options.copy_target_include.
 * @param range If you want to keep the copied item in the range specified for the current include. Default value is false. To set current include, use Include.MakeCurrentLayer().
 * @returns Airbag object
 */
    Copy(range?: boolean): Airbag;

/**
 * Starts an interactive editing panel to create a airbag
 * @param Model Model that the airbag will be created in.
 * @param modal If this window is modal (blocks the user from doing anything else in PRIMER until this window is dismissed). If omitted the window will be modal.
 * @returns Airbag object (or null if not made).
 */
    static Create(Model: Model, modal?: boolean): Airbag;

/**
 * Detaches a comment from a airbag.
 * @param Comment Comment that will be detached from the airbag
 */
    DetachComment(Comment: Comment): void;

/**
 * Starts an interactive editing panel.
 * @param modal If this window is modal (blocks the user from doing anything else in PRIMER until this window is dismissed). If omitted the window will be modal.
 */
    Edit(modal?: boolean): void;

/**
 * Adds an error for airbag. For more details on checking see the Check class.
 * @param message The error message to give
 * @param details An optional detailed error message
 */
    Error(message: string, details?: string): void;

/**
 * Extracts the actual colour used for airbag. By default in PRIMER many entities such as elements get their colour automatically from the part that they are in. PRIMER cycles through 13 default colours based on the label of the entity. In this case the airbag colour property will return the value Colour.PART instead of the actual colour. This method will return the actual colour which is used for drawing the airbag.
 * @returns colour value (integer)
 */
    ExtractColour(): number;

/**
 * Returns the first airbag in the model.
 * @param Model Model to get first airbag in
 * @returns Airbag object (or null if there are no airbags in the model).
 */
    static First(Model: Model): Airbag;

/**
 * Returns the first free airbag label in the model. Also see Airbag.LastFreeLabel(), Airbag.NextFreeLabel() and Model.FirstFreeItemLabel().
 * @param Model Model to get first free airbag label in
 * @param layer Include file (0 for the main file) to search for labels in (Equivalent to First free in layer in editing panels). If omitted the whole model will be used (Equivalent to First free in editing panels).
 * @returns Airbag label.
 */
    static FirstFreeLabel(Model: Model, layer?: number): number;

/**
 * Flags all of the airbags in the model with a defined flag.
 * @param Model Model that all airbags will be flagged in
 * @param flag Flag to set on the airbags
 */
    static FlagAll(Model: Model, flag: number): void;

/**
 * Checks if the airbag is flagged or not.
 * @param flag Flag to test on the airbag
 * @returns true if flagged, false if not.
 */
    Flagged(flag: number): boolean;

/**
 * Calls a function for each airbag in the model. Note that ForEach has been designed to make looping over airbags as fast as possible and so has some limitations. Firstly, a single temporary Airbag object is created and on each function call it is updated with the current airbag data. This means that you should not try to store the Airbag object for later use (e.g. in an array) as it is temporary. Secondly, you cannot create new airbags inside a ForEach loop. 
 * @param Model Model that all airbags are in
 * @param func Function to call for each airbag
 * @param extra An optional extra object/array/string etc that will appended to arguments when calling the function
 */
    static ForEach(Model: Model, func: () => void, extra?: any): void;

/**
 * Returns an array of Airbag objects or properties for all of the airbags in a model in PRIMER. If the optional property argument is not given then an array of Airbag objects is returned. If the property argument is given, that property value for each airbag is returned in the array instead of a Airbag object
 * @param Model Model to get airbags from
 * @param property Name for property to get for all airbags in the model
 * @returns Array of Airbag objects or properties
 */
    static GetAll(Model: Model, property?: string): Airbag[];

/**
 * Extracts the comments associated to a airbag.
 * @returns Array of Comment objects (or null if there are no comments associated to the node).
 */
    GetComments(): Comment[];

/**
 * Returns an array of Airbag objects for all of the flagged airbags in a model in PRIMER If the optional property argument is not given then an array of Airbag objects is returned. If the property argument is given, then that property value for each airbag is returned in the array instead of a Airbag object
 * @param Model Model to get airbags from
 * @param flag Flag set on the airbags that you want to retrieve
 * @param property Name for property to get for all flagged airbags in the model
 * @returns Array of Airbag objects or properties
 */
    static GetFlagged(Model: Model, flag: number, property?: string): Airbag[];

/**
 * Returns the Airbag object for a airbag ID.
 * @param Model Model to find the airbag in
 * @param number number of the airbag you want the Airbag object for
 * @returns Airbag object (or null if airbag does not exist).
 */
    static GetFromID(Model: Model, number: number): Airbag;

/**
 * Checks if a Airbag property is a parameter or not. Note that object properties that are parameters are normally returned as the integer or float parameter values as that is virtually always what the user would want. For this function to work the JavaScript interpreter must use the parameter name instead of the value. This can be done by setting the Options.property_parameter_names option to true before calling the function and then resetting it to false afterwards.. This behaviour can also temporarily be switched by using the Airbag.ViewParameters() method and 'method chaining' (see the examples below).
 * @param prop airbag property to get parameter for
 * @returns Parameter object if property is a parameter, null if not.
 */
    GetParameter(prop: string): Parameter;

/**
 * Returns the value of property at index index for this Airbag object or null if no property exists.
 * @param index The index of the property value to retrieve. (the number of properties can be found from properties) Note that indices start at 0. There is no link between indices and rows/columns so adjacent fields on a line for an airbag may not have adjacent indices.
 * @returns Property value (float/integer)
 */
    GetPropertyByIndex(index: number): number;

/**
 * Returns the value of property string acronym for this Airbag object or null if no property exists.
 * @param acronym The acronym of the property value to retrieve
 * @returns Property value (float/integer)
 */
    GetPropertyByName(acronym: string): number;

/**
 * Returns the value of the property for row and col for this Airbag object or null if no property exists. Note that columns start at 0. Rows start at 1 if the _ID option is set, at 0 otherwise.
 * @param row The row of the property value to retrieve
 * @param col The column of the property value to retrieve
 * @returns Property value (float/integer)
 */
    GetPropertyByRowCol(row: number, col: number): number;

/**
 * Returns the name of the property at index index for this Airbag object or null if there is no property.
 * @param index The index of the property name to retrieve. (the number of properties can be found from properties) Note that indices start at 0. There is no link between indices and rows/columns so adjacent fields on a line for an airbag may not have adjacent indices.
 * @returns Property name (string)
 */
    GetPropertyNameForIndex(index: number): string;

/**
 * Returns the name of the property at row and col for this Airbag object or null if there is no property. Note that columns start at 0. Rows start at 1 if the _ID option is set, at 0 otherwise.
 * @param row The row of the property name to retrieve
 * @param col The column of the property name to retrieve
 * @returns Property name (string)
 */
    GetPropertyNameForRowCol(row: number, col: number): string;

/**
 * Returns the keyword for this airbag (e.g. *AIRBAG_SIMPLE_PRESSURE_VOLUME, *AIRBAG_SIMPLE_AIRBAG_MODEL etc). Note that a carriage return is not added. See also Airbag.KeywordCards()
 * @returns string containing the keyword.
 */
    Keyword(): string;

/**
 * Returns the keyword cards for the airbag. Note that a carriage return is not added. See also Airbag.Keyword()
 * @returns string containing the cards.
 */
    KeywordCards(): string;

/**
 * Returns the last airbag in the model.
 * @param Model Model to get last airbag in
 * @returns Airbag object (or null if there are no airbags in the model).
 */
    static Last(Model: Model): Airbag;

/**
 * Returns the last free airbag label in the model. Also see Airbag.FirstFreeLabel(), Airbag.NextFreeLabel() and see Model.LastFreeItemLabel()
 * @param Model Model to get last free airbag label in
 * @param layer Include file (0 for the main file) to search for labels in (Equivalent to Highest free in layer in editing panels). If omitted the whole model will be used.
 * @returns Airbag label.
 */
    static LastFreeLabel(Model: Model, layer?: number): number;

/**
 * Returns the next airbag in the model.
 * @returns Airbag object (or null if there are no more airbags in the model).
 */
    Next(): Airbag;

/**
 * Returns the next free (highest+1) airbag label in the model. Also see Airbag.FirstFreeLabel(), Airbag.LastFreeLabel() and Model.NextFreeItemLabel()
 * @param Model Model to get next free airbag label in
 * @param layer Include file (0 for the main file) to search for labels in (Equivalent to Highest+1 in layer in editing panels). If omitted the whole model will be used (Equivalent to Highest+1 in editing panels).
 * @returns Airbag label.
 */
    static NextFreeLabel(Model: Model, layer?: number): number;

/**
 * Returns the previous airbag in the model.
 * @returns Airbag object (or null if there are no more airbags in the model).
 */
    Previous(): Airbag;

/**
 * Renumbers all of the airbags in the model.
 * @param Model Model that all airbags will be renumbered in
 * @param start Start point for renumbering
 */
    static RenumberAll(Model: Model, start: number): void;

/**
 * Renumbers all of the flagged airbags in the model.
 * @param Model Model that all the flagged airbags will be renumbered in
 * @param flag Flag set on the airbags that you want to renumber
 * @param start Start point for renumbering
 */
    static RenumberFlagged(Model: Model, flag: number, start: number): void;

/**
 * Allows the user to select airbags using standard PRIMER object menus.
 * @param flag Flag to use when selecting airbags
 * @param prompt Text to display as a prompt to the user
 * @param limit If the argument is a Model then only airbags from that model can be selected. If the argument is a Flag then only airbags that are flagged with limit can be selected (limit should be different to flag). If omitted, or null, any airbags can be selected. from any model.
 * @param modal If selection is modal (blocks the user from doing anything else in PRIMER until this window is dismissed). If omitted the selection will be modal.
 * @returns Number of airbags selected or null if menu cancelled
 */
    static Select(flag: number, prompt: string, limit?: Model | number, modal?: boolean): number;

/**
 * Sets a flag on the airbag.
 * @param flag Flag to set on the airbag
 */
    SetFlag(flag: number): void;

/**
 * Sets the value of property at index index for this Airbag object
 * @param index The index of the property value to set. (the number of properties can be found from properties) Note that indices start at 0. There is no link between indices and rows/columns so adjacent fields on a line for an airbag may not have adjacent indices.
 * @param value The value of the property to set.
 */
    SetPropertyByIndex(index: number, value: number): void;

/**
 * Sets the value of property string acronym for this Airbag object
 * @param acronym The acronym of the property value to set
 * @param value The value of the property to set.
 */
    SetPropertyByName(acronym: string, value: number): void;

/**
 * Sets the value of the property for row and col for this Airbag object. Note that columns start at 0. Rows start at 1 if the _ID option is set, at 0 otherwise.
 * @param row The row of the property value to set
 * @param col The column of the property value to set
 * @param value The value of the property to set.
 */
    SetPropertyByRowCol(row: number, col: number, value: number): void;

/**
 * Sketches the airbag. The airbag will be sketched until you either call Airbag.Unsketch(), Airbag.UnsketchAll(), Model.UnsketchAll(), or delete the model
 * @param redraw If model should be redrawn or not after the airbag is sketched. If omitted redraw is true. If you want to sketch several airbags and only redraw after the last one then use false for redraw and call View.Redraw().
 */
    Sketch(redraw?: boolean): void;

/**
 * Sketches all of the flagged airbags in the model. The airbags will be sketched until you either call Airbag.Unsketch(), Airbag.UnsketchFlagged(), Model.UnsketchAll(), or delete the model
 * @param Model Model that all the flagged airbags will be sketched in
 * @param flag Flag set on the airbags that you want to sketch
 * @param redraw If model should be redrawn or not after the airbags are sketched. If omitted redraw is true. If you want to sketch flagged airbags several times and only redraw after the last one then use false for redraw and call View.Redraw().
 */
    static SketchFlagged(Model: Model, flag: number, redraw?: boolean): void;

/**
 * Returns the total number of airbags in the model.
 * @param Model Model to get total for
 * @param exists true if only existing airbags should be counted. If false or omitted referenced but undefined airbags will also be included in the total.
 * @returns number of airbags
 */
    static Total(Model: Model, exists?: boolean): number;

/**
 * Unsets a defined flag on all of the airbags in the model.
 * @param Model Model that the defined flag for all airbags will be unset in
 * @param flag Flag to unset on the airbags
 */
    static UnflagAll(Model: Model, flag: number): void;

/**
 * Unsketches the airbag.
 * @param redraw If model should be redrawn or not after the airbag is unsketched. If omitted redraw is true. If you want to unsketch several airbags and only redraw after the last one then use false for redraw and call View.Redraw().
 */
    Unsketch(redraw?: boolean): void;

/**
 * Unsketches all airbags.
 * @param Model Model that all airbags will be unblanked in
 * @param redraw If model should be redrawn or not after the airbags are unsketched. If omitted redraw is true. If you want to unsketch several things and only redraw after the last one then use false for redraw and call View.Redraw().
 */
    static UnsketchAll(Model: Model, redraw?: boolean): void;

/**
 * Unsketches all flagged airbags in the model.
 * @param Model Model that all airbags will be unsketched in
 * @param flag Flag set on the airbags that you want to unsketch
 * @param redraw If model should be redrawn or not after the airbags are unsketched. If omitted redraw is true. If you want to unsketch several things and only redraw after the last one then use false for redraw and call View.Redraw().
 */
    static UnsketchFlagged(Model: Model, flag: number, redraw?: boolean): void;

/**
 * Object properties that are parameters are normally returned as the integer or float parameter values as that is virtually always what the user would want. This function temporarily changes the behaviour so that if a property is a parameter the parameter name is returned instead. This can be used with 'method chaining' (see the example below) to make sure a property argument is correct.
 * @returns Airbag object.
 */
    ViewParameters(): Airbag;

/**
 * Adds a warning for airbag. For more details on checking see the Check class.
 * @param message The warning message to give
 * @param details An optional detailed warning message
 */
    Warning(message: string, details?: string): void;

/**
 * Returns the cross references for this airbag.
 * @returns Xrefs object.
 */
    Xrefs(): Xrefs;

/**
 * Create a new Airbag object.
 * @param Model Model that airbag will be created in
 * @param type Airbag type. Can be Airbag.SIMPLE_PRESSURE_VOLUME, Airbag.SIMPLE_AIRBAG_MODEL, Airbag.ADIABATIC_GAS_MODEL, Airbag.WANG_NEFSKE, Airbag.WANG_NEFSKE_JETTING, Airbag.WANG_NEFSKE_MULTIPLE_JETTING, Airbag.LOAD_CURVE, Airbag.LINEAR_FLUID, Airbag.HYBRID, Airbag.HYBRID_JETTING, Airbag.HYBRID_CHEMKIN, Airbag.ALE, Airbag.ADVANCED_ALE or Airbag.PARTICLE
 * @param sid Set ID
 * @param sidtyp Set type: segment/part set ID
 * @param abid Airbag number
 * @param heading Airbag title
 * @returns Airbag object
 */
    constructor(Model: Model, type: string, sid: number, sidtyp?: number, abid?: number, heading?: string);

/**
 * Creates a string containing the airbag data in keyword format. Note that this contains the keyword header and the keyword cards. See also Airbag.Keyword() and Airbag.KeywordCards().
 * @returns string
 */
    toString(): string;

/** Airbag number. Also see the label property which is an alternative name for this. */
    abid: number;
/** The colour of the airbag */
    colour: Colour;
/** The number of columns of data the airbag has */
    readonly cols: number;
/** true if airbag exists, false if referred to but not defined. */
    readonly exists: boolean;
/** Turns _ID on or OFF */
    id: boolean;
/** The Include file number that the airbag is in. */
    include: number;
/** Airbag number. Also see the abid property which is an alternative name for this. */
    label: number;
/** The Model number that the airbag is in. */
    readonly model: number;
/** The total number of properties that the airbag has */
    properties: number;
/** The number of rows of data the airbag has. This includes the _ID card if it is set. */
    readonly rows: number;
/** Airbag title */
    title: string;
/** Airbag type. Can be Airbag.SIMPLE_PRESSURE_VOLUME, Airbag.SIMPLE_AIRBAG_MODEL, Airbag.ADIABATIC_GAS_MODEL, Airbag.WANG_NEFSKE, Airbag.WANG_NEFSKE_JETTING, Airbag.WANG_NEFSKE_MULTIPLE_JETTING, Airbag.LOAD_CURVE, Airbag.LINEAR_FLUID, Airbag.HYBRID, Airbag.HYBRID_JETTING, Airbag.HYBRID_CHEMKIN, Airbag.ALE, Airbag.ADVANCED_ALE or Airbag.PARTICLE */
    type: number;
/** Airbag adiabatic gas model type */
    static ADIABATIC_GAS_MODEL: number;
/** Airbag advanced ALE type */
    static ADVANCED_ALE: number;
/** Airbag ALE type */
    static ALE: number;
/** Airbag hybrid type */
    static HYBRID: number;
/** Airbag hybrid chemkin type */
    static HYBRID_CHEMKIN: number;
/** Airbag hybrid jetting type */
    static HYBRID_JETTING: number;
/** Airbag linear fluid type */
    static LINEAR_FLUID: number;
/** Airbag load curve type */
    static LOAD_CURVE: number;
/** Airbag particle type */
    static PARTICLE: number;
/** Airbag simple airbag model type */
    static SIMPLE_AIRBAG_MODEL: number;
/** Airbag simple pressure volume type */
    static SIMPLE_PRESSURE_VOLUME: number;
/** Airbag Wang Nefske type */
    static WANG_NEFSKE: number;
/** Airbag Wang Nefske jetting type */
    static WANG_NEFSKE_JETTING: number;
/** Airbag Wang Nefske multiple jetting type */
    static WANG_NEFSKE_MULTIPLE_JETTING: number;
}

declare class Attached {
/**
 * Sets the find attached option for beam 3rd nodes on or off
 * @param Setting If true beam 3rd nodes are considered for find attached, if false, they are not.
 */
    static Beam3rdNodes(Setting: boolean): void;

/**
 * Sets the find attached option for beam pid on or off
 * @param Setting If true beam pid's are considered for find attached, if false, they are not.
 */
    static BeamPid(Setting: boolean): void;

/**
 * Sets the deformable option for find attached
 * @param Setting Option. Can be Attached.WHOLE, Attached.SINGLE
 */
    static Deformable(Setting: number): void;

/**
 * Sets an option to flag parts after a find attached if any elements within that part are flagged
 * @param Setting If true, parts are flagged after a find attached if any elements within that part are flagged, if false, they are not.
 */
    static FlagPart(Setting: boolean): void;

/**
 * Sets the find attached option for recursive on or off
 * @param Setting If true recursive is on, if false, it is off.
 * @param Number Option to set the number of find attached iterations used when the recursive option is set.
 */
    static Recursive(Setting: boolean, Number?: number): void;

/**
 * Sets the rigid option for find attached
 * @param Setting Option. Can be Attached.WHOLE, Attached.SINGLE
 */
    static Rigid(Setting: number): void;

/**
 * Sets entity to be on or off to find attached through.
 * @param Type The type of the item to switch on or off (for a list of types see Appendix A of the PRIMER manual). Use "ALL" to switch all entities or "CONSTRAINEDALL" to switch all constrained entities.
 * @param Setting If true you turn the entity switch on, if false you turn it off.
 */
    static SetEntity(Type: string, Setting: boolean): void;

/**
 * Sets the find attached option for tied contacts on or off
 * @param Setting If true tied contacts are considered for find attached, if false, they are not.
 */
    static TiedContacts(Setting: boolean): void;

/** Find attached option - find attached through single elements only */
    static SINGLE: number;
/** Find attached option - find through whole attached part */
    static WHOLE: number;
}

declare class AxialForceBeam {
/**
 * Associates a comment with a axial force beam.
 * @param Comment Comment that will be attached to the axial force beam
 */
    AssociateComment(Comment: Comment): void;

/**
 * Clears a flag on the axial force beam.
 * @param flag Flag to clear on the axial force beam
 */
    ClearFlag(flag: number): void;

/**
 * Copies the axial force beam. The target include of the copied axial force beam can be set using Options.copy_target_include.
 * @param range If you want to keep the copied item in the range specified for the current include. Default value is false. To set current include, use Include.MakeCurrentLayer().
 * @returns AxialForceBeam object
 */
    Copy(range?: boolean): AxialForceBeam;

/**
 * Detaches a comment from a axial force beam.
 * @param Comment Comment that will be detached from the axial force beam
 */
    DetachComment(Comment: Comment): void;

/**
 * Adds an error for axial force beam. For more details on checking see the Check class.
 * @param message The error message to give
 * @param details An optional detailed error message
 */
    Error(message: string, details?: string): void;

/**
 * Returns the first axial force beam in the model.
 * @param Model Model to get first axial force beam in
 * @returns AxialForceBeam object (or null if there are no axial force beams in the model).
 */
    static First(Model: Model): AxialForceBeam;

/**
 * Flags all of the axial force beams in the model with a defined flag.
 * @param Model Model that all axial force beams will be flagged in
 * @param flag Flag to set on the axial force beams
 */
    static FlagAll(Model: Model, flag: number): void;

/**
 * Checks if the axial force beam is flagged or not.
 * @param flag Flag to test on the axial force beam
 * @returns true if flagged, false if not.
 */
    Flagged(flag: number): boolean;

/**
 * Calls a function for each axial force beam in the model. Note that ForEach has been designed to make looping over axial force beams as fast as possible and so has some limitations. Firstly, a single temporary AxialForceBeam object is created and on each function call it is updated with the current axial force beam data. This means that you should not try to store the AxialForceBeam object for later use (e.g. in an array) as it is temporary. Secondly, you cannot create new axial force beams inside a ForEach loop. 
 * @param Model Model that all axial force beams are in
 * @param func Function to call for each axial force beam
 * @param extra An optional extra object/array/string etc that will appended to arguments when calling the function
 */
    static ForEach(Model: Model, func: () => void, extra?: any): void;

/**
 * Returns an array of AxialForceBeam objects or properties for all of the axial force beams in a model in PRIMER. If the optional property argument is not given then an array of AxialForceBeam objects is returned. If the property argument is given, that property value for each axial force beam is returned in the array instead of a AxialForceBeam object
 * @param Model Model to get axial force beams from
 * @param property Name for property to get for all axial force beams in the model
 * @returns Array of AxialForceBeam objects or properties
 */
    static GetAll(Model: Model, property?: string): AxialForceBeam[];

/**
 * Extracts the comments associated to a axial force beam.
 * @returns Array of Comment objects (or null if there are no comments associated to the node).
 */
    GetComments(): Comment[];

/**
 * Returns an array of AxialForceBeam objects for all of the flagged axial force beams in a model in PRIMER If the optional property argument is not given then an array of AxialForceBeam objects is returned. If the property argument is given, then that property value for each axial force beam is returned in the array instead of a AxialForceBeam object
 * @param Model Model to get axial force beams from
 * @param flag Flag set on the axial force beams that you want to retrieve
 * @param property Name for property to get for all flagged axial force beams in the model
 * @returns Array of AxialForceBeam objects or properties
 */
    static GetFlagged(Model: Model, flag: number, property?: string): AxialForceBeam[];

/**
 * Returns the AxialForceBeam object for a axial force beam ID.
 * @param Model Model to find the axial force beam in
 * @param number number of the axial force beam you want the AxialForceBeam object for
 * @returns AxialForceBeam object (or null if axial force beam does not exist).
 */
    static GetFromID(Model: Model, number: number): AxialForceBeam;

/**
 * Checks if a AxialForceBeam property is a parameter or not. Note that object properties that are parameters are normally returned as the integer or float parameter values as that is virtually always what the user would want. For this function to work the JavaScript interpreter must use the parameter name instead of the value. This can be done by setting the Options.property_parameter_names option to true before calling the function and then resetting it to false afterwards.. This behaviour can also temporarily be switched by using the AxialForceBeam.ViewParameters() method and 'method chaining' (see the examples below).
 * @param prop axial force beam property to get parameter for
 * @returns Parameter object if property is a parameter, null if not.
 */
    GetParameter(prop: string): Parameter;

/**
 * Returns the keyword for this axial force beam (*INITIAL_AXIAL_FORCE_BEAM). Note that a carriage return is not added. See also AxialForceBeam.KeywordCards()
 * @returns string containing the keyword.
 */
    Keyword(): string;

/**
 * Returns the keyword cards for the axial force beam. Note that a carriage return is not added. See also AxialForceBeam.Keyword()
 * @returns string containing the cards.
 */
    KeywordCards(): string;

/**
 * Returns the last axial force beam in the model.
 * @param Model Model to get last axial force beam in
 * @returns AxialForceBeam object (or null if there are no axial force beams in the model).
 */
    static Last(Model: Model): AxialForceBeam;

/**
 * Returns the next axial force beam in the model.
 * @returns AxialForceBeam object (or null if there are no more axial force beams in the model).
 */
    Next(): AxialForceBeam;

/**
 * Returns the previous axial force beam in the model.
 * @returns AxialForceBeam object (or null if there are no more axial force beams in the model).
 */
    Previous(): AxialForceBeam;

/**
 * Allows the user to select axial force beams using standard PRIMER object menus.
 * @param flag Flag to use when selecting axial force beams
 * @param prompt Text to display as a prompt to the user
 * @param limit If the argument is a Model then only axial force beams from that model can be selected. If the argument is a Flag then only axial force beams that are flagged with limit can be selected (limit should be different to flag). If omitted, or null, any axial force beams can be selected. from any model.
 * @param modal If selection is modal (blocks the user from doing anything else in PRIMER until this window is dismissed). If omitted the selection will be modal.
 * @returns Number of axial force beams selected or null if menu cancelled
 */
    static Select(flag: number, prompt: string, limit?: Model | number, modal?: boolean): number;

/**
 * Sets a flag on the axial force beam.
 * @param flag Flag to set on the axial force beam
 */
    SetFlag(flag: number): void;

/**
 * Sketches the axial force beam. The axial force beam will be sketched until you either call AxialForceBeam.Unsketch(), AxialForceBeam.UnsketchAll(), Model.UnsketchAll(), or delete the model
 * @param redraw If model should be redrawn or not after the axial force beam is sketched. If omitted redraw is true. If you want to sketch several axial force beams and only redraw after the last one then use false for redraw and call View.Redraw().
 */
    Sketch(redraw?: boolean): void;

/**
 * Sketches all of the flagged axial force beams in the model. The axial force beams will be sketched until you either call AxialForceBeam.Unsketch(), AxialForceBeam.UnsketchFlagged(), Model.UnsketchAll(), or delete the model
 * @param Model Model that all the flagged axial force beams will be sketched in
 * @param flag Flag set on the axial force beams that you want to sketch
 * @param redraw If model should be redrawn or not after the axial force beams are sketched. If omitted redraw is true. If you want to sketch flagged axial force beams several times and only redraw after the last one then use false for redraw and call View.Redraw().
 */
    static SketchFlagged(Model: Model, flag: number, redraw?: boolean): void;

/**
 * Returns the total number of axial force beams in the model.
 * @param Model Model to get total for
 * @param exists true if only existing axial force beams should be counted. If false or omitted referenced but undefined axial force beams will also be included in the total.
 * @returns number of axial force beams
 */
    static Total(Model: Model, exists?: boolean): number;

/**
 * Unsets a defined flag on all of the axial force beams in the model.
 * @param Model Model that the defined flag for all axial force beams will be unset in
 * @param flag Flag to unset on the axial force beams
 */
    static UnflagAll(Model: Model, flag: number): void;

/**
 * Unsketches the axial force beam.
 * @param redraw If model should be redrawn or not after the axial force beam is unsketched. If omitted redraw is true. If you want to unsketch several axial force beams and only redraw after the last one then use false for redraw and call View.Redraw().
 */
    Unsketch(redraw?: boolean): void;

/**
 * Unsketches all axial force beams.
 * @param Model Model that all axial force beams will be unblanked in
 * @param redraw If model should be redrawn or not after the axial force beams are unsketched. If omitted redraw is true. If you want to unsketch several things and only redraw after the last one then use false for redraw and call View.Redraw().
 */
    static UnsketchAll(Model: Model, redraw?: boolean): void;

/**
 * Unsketches all flagged axial force beams in the model.
 * @param Model Model that all axial force beams will be unsketched in
 * @param flag Flag set on the axial force beams that you want to unsketch
 * @param redraw If model should be redrawn or not after the axial force beams are unsketched. If omitted redraw is true. If you want to unsketch several things and only redraw after the last one then use false for redraw and call View.Redraw().
 */
    static UnsketchFlagged(Model: Model, flag: number, redraw?: boolean): void;

/**
 * Object properties that are parameters are normally returned as the integer or float parameter values as that is virtually always what the user would want. This function temporarily changes the behaviour so that if a property is a parameter the parameter name is returned instead. This can be used with 'method chaining' (see the example below) to make sure a property argument is correct.
 * @returns AxialForceBeam object.
 */
    ViewParameters(): AxialForceBeam;

/**
 * Adds a warning for axial force beam. For more details on checking see the Check class.
 * @param message The warning message to give
 * @param details An optional detailed warning message
 */
    Warning(message: string, details?: string): void;

/**
 * Returns the cross references for this axial force beam.
 * @returns Xrefs object.
 */
    Xrefs(): Xrefs;

/**
 * Create a new AxialForceBeam object.
 * @param Model Model that axial force beam will be created in
 * @param bsid BeamSet ID.
 * @param lcid Loadcurve ID defining preload versus time.
 * @param scale Scale factor on curve
 * @returns AxialForceBeam object
 */
    constructor(Model: Model, bsid: number, lcid: number, scale?: number);

/**
 * Creates a string containing the axial force data in keyword format. Note that this contains the keyword header and the keyword cards. See also AxialForceBeam.Keyword() and AxialForceBeam.KeywordCards().
 * @returns string
 */
    toString(): string;

/** Beam set ID. */
    bsid: number;
/** true if axial force beam exists, false if referred to but not defined. */
    readonly exists: boolean;
/** ID of the axial force beam. Only used in PRIMER. */
    readonly id: number;
/** The Include file number that the axial force beam is in. */
    include: number;
/** Bending stiffness flag. */
    kbend: number;
/** Loadcurve ID. */
    lcid: number;
/** The Model number that the axial force beam is in. */
    readonly model: number;
/** Scale factor on loadcurve. */
    scale: number;
}

declare class Beam {
/**
 * Associates a comment with a beam.
 * @param Comment Comment that will be attached to the beam
 */
    AssociateComment(Comment: Comment): void;

/**
 * Blanks the beam
 */
    Blank(): void;

/**
 * Blanks all of the beams in the model.
 * @param Model Model that all beams will be blanked in
 * @param redraw If model should be redrawn or not. If omitted redraw is false. If you want to do several (un)blanks and only redraw after the last one then use false for all redraws apart from the last one. Alternatively you can redraw using View.Redraw().
 */
    static BlankAll(Model: Model, redraw?: boolean): void;

/**
 * Blanks all of the flagged beams in the model.
 * @param Model Model that all the flagged beams will be blanked in
 * @param flag Flag set on the beams that you want to blank
 * @param redraw If model should be redrawn or not. If omitted redraw is false. If you want to do several (un)blanks and only redraw after the last one then use false for all redraws apart from the last one. Alternatively you can redraw using View.Redraw().
 */
    static BlankFlagged(Model: Model, flag: number, redraw?: boolean): void;

/**
 * Checks if the beam is blanked or not.
 * @returns true if blanked, false if not.
 */
    Blanked(): boolean;

/**
 * Starts an edit panel in Browse mode.
 * @param modal If this window is modal (blocks the user from doing anything else in PRIMER until this window is dismissed). If omitted the window will be modal.
 */
    Browse(modal?: boolean): void;

/**
 * Clears a flag on the beam.
 * @param flag Flag to clear on the beam
 */
    ClearFlag(flag: number): void;

/**
 * Copies the beam. The target include of the copied beam can be set using Options.copy_target_include.
 * @param range If you want to keep the copied item in the range specified for the current include. Default value is false. To set current include, use Include.MakeCurrentLayer().
 * @returns Beam object
 */
    Copy(range?: boolean): Beam;

/**
 * Starts an interactive editing panel to create a beam
 * @param Model Model that the beam will be created in.
 * @param modal If this window is modal (blocks the user from doing anything else in PRIMER until this window is dismissed). If omitted the window will be modal.
 * @returns Beam object (or null if not made).
 */
    static Create(Model: Model, modal?: boolean): Beam;

/**
 * Detaches a comment from a beam.
 * @param Comment Comment that will be detached from the beam
 */
    DetachComment(Comment: Comment): void;

/**
 * Starts an interactive editing panel.
 * @param modal If this window is modal (blocks the user from doing anything else in PRIMER until this window is dismissed). If omitted the window will be modal.
 */
    Edit(modal?: boolean): void;

/**
 * Returns coordinates of the intersections between a beam and a database cross section.Note this function does not check that the beam is in the cross section definition (part set)
 * @param Database_cross_section_label The label of the database cross section.
 * @returns An array containing the x,y,z coordinates of the cut point, or NULL if it does not cut
 */
    ElemCut(Database_cross_section_label: number): number[];

/**
 * Adds an error for beam. For more details on checking see the Check class.
 * @param message The error message to give
 * @param details An optional detailed error message
 */
    Error(message: string, details?: string): void;

/**
 * Extracts the actual colour used for beam. By default in PRIMER many entities such as elements get their colour automatically from the part that they are in. PRIMER cycles through 13 default colours based on the label of the entity. In this case the beam colour property will return the value Colour.PART instead of the actual colour. This method will return the actual colour which is used for drawing the beam.
 * @returns colour value (integer)
 */
    ExtractColour(): number;

/**
 * Returns an array of Beam objects for the beams within a box. Please note this function provides a list of all beams that could potentially be in the box (using computationally cheap bounding box comparison) it is not a rigorous test of whether the beam is actually in the box. Note an extension of "spot_thickness" is applied to each beam. This may include beams that are ostensibly outside box. The user should apply their own test. (this function is intended to provide an upper bound of elems to test) Setting the "excl" flag will require that the beam is fully contained, but this may not capture all the beams you want to process.
 * @param Model Model designated model
 * @param xmin Minimum bound in global x
 * @param xmax Maximum bound in global x
 * @param ymin Minimum bound in global y
 * @param ymax Maximum bound in global y
 * @param zmin Minimum bound in global z
 * @param zmax Maximum bound in global z
 * @param flag Optional flag to restrict beams considered, if 0 all beams considered
 * @param excl Optional flag ( 0) Apply inclusive selection ( 1) Apply exclusive selection inclusive selection means elements intersect box exclusive selection means elements contained in box
 * @param vis_only Optional flag to consider visible elements only (1), if (0) all elements considered
 * @returns Array of Beam objects
 */
    static FindBeamInBox(Model: Model, xmin: number, xmax: number, ymin: number, ymax: number, zmin: number, zmax: number, flag?: number, excl?: number, vis_only?: number): Beam[];

/**
 * Returns the first beam in the model.
 * @param Model Model to get first beam in
 * @returns Beam object (or null if there are no beams in the model).
 */
    static First(Model: Model): Beam;

/**
 * Returns the first free beam label in the model. Also see Beam.LastFreeLabel(), Beam.NextFreeLabel() and Model.FirstFreeItemLabel().
 * @param Model Model to get first free beam label in
 * @param layer Include file (0 for the main file) to search for labels in (Equivalent to First free in layer in editing panels). If omitted the whole model will be used (Equivalent to First free in editing panels).
 * @returns Beam label.
 */
    static FirstFreeLabel(Model: Model, layer?: number): number;

/**
 * Flags all of the beams in the model with a defined flag.
 * @param Model Model that all beams will be flagged in
 * @param flag Flag to set on the beams
 */
    static FlagAll(Model: Model, flag: number): void;

/**
 * Checks if the beam is flagged or not.
 * @param flag Flag to test on the beam
 * @returns true if flagged, false if not.
 */
    Flagged(flag: number): boolean;

/**
 * Calls a function for each beam in the model. Note that ForEach has been designed to make looping over beams as fast as possible and so has some limitations. Firstly, a single temporary Beam object is created and on each function call it is updated with the current beam data. This means that you should not try to store the Beam object for later use (e.g. in an array) as it is temporary. Secondly, you cannot create new beams inside a ForEach loop. 
 * @param Model Model that all beams are in
 * @param func Function to call for each beam
 * @param extra An optional extra object/array/string etc that will appended to arguments when calling the function
 */
    static ForEach(Model: Model, func: () => void, extra?: any): void;

/**
 * Returns an array of Beam objects or properties for all of the beams in a model in PRIMER. If the optional property argument is not given then an array of Beam objects is returned. If the property argument is given, that property value for each beam is returned in the array instead of a Beam object
 * @param Model Model to get beams from
 * @param property Name for property to get for all beams in the model
 * @returns Array of Beam objects or properties
 */
    static GetAll(Model: Model, property?: string): Beam[];

/**
 * Extracts the comments associated to a beam.
 * @returns Array of Comment objects (or null if there are no comments associated to the node).
 */
    GetComments(): Comment[];

/**
 * Returns an array of Beam objects for all of the flagged beams in a model in PRIMER If the optional property argument is not given then an array of Beam objects is returned. If the property argument is given, then that property value for each beam is returned in the array instead of a Beam object
 * @param Model Model to get beams from
 * @param flag Flag set on the beams that you want to retrieve
 * @param property Name for property to get for all flagged beams in the model
 * @returns Array of Beam objects or properties
 */
    static GetFlagged(Model: Model, flag: number, property?: string): Beam[];

/**
 * Returns the Beam object for a beam ID.
 * @param Model Model to find the beam in
 * @param number number of the beam you want the Beam object for
 * @returns Beam object (or null if beam does not exist).
 */
    static GetFromID(Model: Model, number: number): Beam;

/**
 * Checks if a Beam property is a parameter or not. Note that object properties that are parameters are normally returned as the integer or float parameter values as that is virtually always what the user would want. For this function to work the JavaScript interpreter must use the parameter name instead of the value. This can be done by setting the Options.property_parameter_names option to true before calling the function and then resetting it to false afterwards.. This behaviour can also temporarily be switched by using the Beam.ViewParameters() method and 'method chaining' (see the examples below).
 * @param prop beam property to get parameter for
 * @returns Parameter object if property is a parameter, null if not.
 */
    GetParameter(prop: string): Parameter;

/**
 * Returns the keyword for this beam (*BEAM, *BEAM_SCALAR or *BEAM_SCALAR_VALUE). Note that a carriage return is not added. See also Beam.KeywordCards()
 * @returns string containing the keyword.
 */
    Keyword(): string;

/**
 * Returns the keyword cards for the beam. Note that a carriage return is not added. See also Beam.Keyword()
 * @returns string containing the cards.
 */
    KeywordCards(): string;

/**
 * Returns the last beam in the model.
 * @param Model Model to get last beam in
 * @returns Beam object (or null if there are no beams in the model).
 */
    static Last(Model: Model): Beam;

/**
 * Returns the last free beam label in the model. Also see Beam.FirstFreeLabel(), Beam.NextFreeLabel() and see Model.LastFreeItemLabel()
 * @param Model Model to get last free beam label in
 * @param layer Include file (0 for the main file) to search for labels in (Equivalent to Highest free in layer in editing panels). If omitted the whole model will be used.
 * @returns Beam label.
 */
    static LastFreeLabel(Model: Model, layer?: number): number;

/**
 * Returns the next beam in the model.
 * @returns Beam object (or null if there are no more beams in the model).
 */
    Next(): Beam;

/**
 * Returns the next free (highest+1) beam label in the model. Also see Beam.FirstFreeLabel(), Beam.LastFreeLabel() and Model.NextFreeItemLabel()
 * @param Model Model to get next free beam label in
 * @param layer Include file (0 for the main file) to search for labels in (Equivalent to Highest+1 in layer in editing panels). If omitted the whole model will be used (Equivalent to Highest+1 in editing panels).
 * @returns Beam label.
 */
    static NextFreeLabel(Model: Model, layer?: number): number;

/**
 * Allows the user to pick a beam.
 * @param prompt Text to display as a prompt to the user
 * @param limit If the argument is a Model then only beams from that model can be picked. If the argument is a Flag then only beams that are flagged with limit can be selected. If omitted, or null, any beams from any model can be selected. from any model.
 * @param modal If picking is modal (blocks the user from doing anything else in PRIMER until this window is dismissed). If omitted the pick will be modal.
 * @param button_text By default the window with the prompt will have a button labelled 'Cancel' which if pressed will cancel the pick and return null. If you want to change the text on the button use this argument. If omitted 'Cancel' will be used.
 * @returns Beam object (or null if not picked)
 */
    static Pick(prompt: string, limit?: Model | number, modal?: boolean, button_text?: string): Beam;

/**
 * Returns the previous beam in the model.
 * @returns Beam object (or null if there are no more beams in the model).
 */
    Previous(): Beam;

/**
 * Renumbers all of the beams in the model.
 * @param Model Model that all beams will be renumbered in
 * @param start Start point for renumbering
 */
    static RenumberAll(Model: Model, start: number): void;

/**
 * Renumbers all of the flagged beams in the model.
 * @param Model Model that all the flagged beams will be renumbered in
 * @param flag Flag set on the beams that you want to renumber
 * @param start Start point for renumbering
 */
    static RenumberFlagged(Model: Model, flag: number, start: number): void;

/**
 * Returns the indices of the points for a faces to plot the true section of the beam. Note face numbers start at 0. Beam.SectionPoints must be called before this method.
 * @param face Face to get indices for
 * @returns Array of integers
 */
    SectionFacePoints(face: number): number[];

/**
 * Returns the number of faces to plot the true section of the beam. Beam.SectionPoints must be called before this method.
 * @returns integer
 */
    SectionFaces(): number;

/**
 * Returns the point coordinates to plot the true section of the beam. They are returned in a single array of numbers.
 * @returns Array of floats
 */
    SectionPoints(): number[];

/**
 * Allows the user to select beams using standard PRIMER object menus.
 * @param flag Flag to use when selecting beams
 * @param prompt Text to display as a prompt to the user
 * @param limit If the argument is a Model then only beams from that model can be selected. If the argument is a Flag then only beams that are flagged with limit can be selected (limit should be different to flag). If omitted, or null, any beams can be selected. from any model.
 * @param modal If selection is modal (blocks the user from doing anything else in PRIMER until this window is dismissed). If omitted the selection will be modal.
 * @returns Number of beams selected or null if menu cancelled
 */
    static Select(flag: number, prompt: string, limit?: Model | number, modal?: boolean): number;

/**
 * Sets a flag on the beam.
 * @param flag Flag to set on the beam
 */
    SetFlag(flag: number): void;

/**
 * Sketches the beam. The beam will be sketched until you either call Beam.Unsketch(), Beam.UnsketchAll(), Model.UnsketchAll(), or delete the model
 * @param redraw If model should be redrawn or not after the beam is sketched. If omitted redraw is true. If you want to sketch several beams and only redraw after the last one then use false for redraw and call View.Redraw().
 */
    Sketch(redraw?: boolean): void;

/**
 * Sketches all of the flagged beams in the model. The beams will be sketched until you either call Beam.Unsketch(), Beam.UnsketchFlagged(), Model.UnsketchAll(), or delete the model
 * @param Model Model that all the flagged beams will be sketched in
 * @param flag Flag set on the beams that you want to sketch
 * @param redraw If model should be redrawn or not after the beams are sketched. If omitted redraw is true. If you want to sketch flagged beams several times and only redraw after the last one then use false for redraw and call View.Redraw().
 */
    static SketchFlagged(Model: Model, flag: number, redraw?: boolean): void;

/**
 * Checks if nodes of beam are tied by contact or directly attached (non-zero option1)
 * @param Contact_label The label of the tied contact. If zero the tied contact is found for the beam by reverse lookup.
 * @param Flag flag bit
 * @param Option1 Directly tied node (logical OR) 0:NONE 1:NRB/C_EXNO 2:BEAM 4:SHELL 8:SOLID 16:TSHELL
 * @param Option2 0:No action 1:report error if directly attached node (acc. option1) also captured by contact
 * @returns string
 */
    TiedNodeCheck(Contact_label: number, Flag: number, Option1: number, Option2: number): string;

/**
 * Calculates the timestep for the beam
 * @returns float
 */
    Timestep(): number;

/**
 * Returns the total number of beams in the model.
 * @param Model Model to get total for
 * @param exists true if only existing beams should be counted. If false or omitted referenced but undefined beams will also be included in the total.
 * @returns number of beams
 */
    static Total(Model: Model, exists?: boolean): number;

/**
 * Unblanks the beam
 */
    Unblank(): void;

/**
 * Unblanks all of the beams in the model.
 * @param Model Model that all beams will be unblanked in
 * @param redraw If model should be redrawn or not. If omitted redraw is false. If you want to do several (un)blanks and only redraw after the last one then use false for all redraws apart from the last one. Alternatively you can redraw using View.Redraw().
 */
    static UnblankAll(Model: Model, redraw?: boolean): void;

/**
 * Unblanks all of the flagged beams in the model.
 * @param Model Model that the flagged beams will be unblanked in
 * @param flag Flag set on the beams that you want to unblank
 * @param redraw If model should be redrawn or not. If omitted redraw is false. If you want to do several (un)blanks and only redraw after the last one then use false for all redraws apart from the last one. Alternatively you can redraw using View.Redraw().
 */
    static UnblankFlagged(Model: Model, flag: number, redraw?: boolean): void;

/**
 * Unsets a defined flag on all of the beams in the model.
 * @param Model Model that the defined flag for all beams will be unset in
 * @param flag Flag to unset on the beams
 */
    static UnflagAll(Model: Model, flag: number): void;

/**
 * Unsketches the beam.
 * @param redraw If model should be redrawn or not after the beam is unsketched. If omitted redraw is true. If you want to unsketch several beams and only redraw after the last one then use false for redraw and call View.Redraw().
 */
    Unsketch(redraw?: boolean): void;

/**
 * Unsketches all beams.
 * @param Model Model that all beams will be unblanked in
 * @param redraw If model should be redrawn or not after the beams are unsketched. If omitted redraw is true. If you want to unsketch several things and only redraw after the last one then use false for redraw and call View.Redraw().
 */
    static UnsketchAll(Model: Model, redraw?: boolean): void;

/**
 * Unsketches all flagged beams in the model.
 * @param Model Model that all beams will be unsketched in
 * @param flag Flag set on the beams that you want to unsketch
 * @param redraw If model should be redrawn or not after the beams are unsketched. If omitted redraw is true. If you want to unsketch several things and only redraw after the last one then use false for redraw and call View.Redraw().
 */
    static UnsketchFlagged(Model: Model, flag: number, redraw?: boolean): void;

/**
 * Object properties that are parameters are normally returned as the integer or float parameter values as that is virtually always what the user would want. This function temporarily changes the behaviour so that if a property is a parameter the parameter name is returned instead. This can be used with 'method chaining' (see the example below) to make sure a property argument is correct.
 * @returns Beam object.
 */
    ViewParameters(): Beam;

/**
 * Adds a warning for beam. For more details on checking see the Check class.
 * @param message The warning message to give
 * @param details An optional detailed warning message
 */
    Warning(message: string, details?: string): void;

/**
 * Returns the cross references for this beam.
 * @returns Xrefs object.
 */
    Xrefs(): Xrefs;

/**
 * Create a new Beam object. Use either 1, 2 or 3 nodes when creating a new beam.
 * @param Model Model that beam will be created in
 * @param eid Beam number
 * @param pid Part number
 * @param n1 Node number 1
 * @param n2 Node number 2
 * @param n3 Node number 3
 * @returns Beam object
 */
    constructor(Model: Model, eid: number, pid: number, n1: number, n2?: number, n3?: number);

/**
 * Creates a string containing the beam data in keyword format. Note that this contains the keyword header and the keyword cards. See also Beam.Keyword() and Beam.KeywordCards().
 * @returns string
 */
    toString(): string;

/** Coordinate system ID (_SCALAR) */
    cid: number;
/** Coordinate system ID at node 1 (_SCALR) */
    cid_1: number;
/** The colour of the beam */
    colour: Colour;
/** Section parameter 1 */
    d1: number;
/** Section parameter 2 */
    d2: number;
/** Section parameter 3 */
    d3: number;
/** Section parameter 4 */
    d4: number;
/** Section parameter 5 */
    d5: number;
/** Section parameter 6 */
    d6: number;
/** Active degree of freedom at node 1 (_SCALAR) */
    dofn1: number;
/** Active degree of freedom at node 2 (_SCALAR) */
    dofn2: number;
/** Active degrees of freedom at nodes 1 and 2 (_SCALR) */
    dofns: number;
/** Beam number. Also see the label property which is an alternative name for this. */
    eid: number;
/** If ELBOW option is set. Can be true or false */
    elbow: boolean;
/** true if beam exists, false if referred to but not defined. */
    readonly exists: boolean;
/** The Include file number that the beam is in. */
    include: number;
/** Mass moment of inertia for beam */
    iner: number;
/** Beam number. Also see the eid property which is an alternative name for this. */
    label: number;
/** Coordinate system option */
    local: number;
/** Middle Node for elbow beam */
    mn: number;
/** The Model number that the beam is in. */
    readonly model: number;
/** Node number 1 */
    n1: number;
/** Node number 2 */
    n2: number;
/** Node number 3 */
    n3: number;
/** Number of nodes beam has */
    readonly nodes: number;
/** If _OFFSET option is set. Can be true or false */
    offset: number;
/** If _ORIENTATION option is set. Can be true or false */
    orientation: number;
/** Thickness parameter 1 */
    parm1: number;
/** Thickness parameter 2 */
    parm2: number;
/** Thickness parameter 3 */
    parm3: number;
/** Thickness parameter 4 */
    parm4: number;
/** Thickness parameter 5 */
    parm5: number;
/** Part number */
    pid: number;
/** Part number 1 for spotweld beam */
    pid1: number;
/** Part number 2 for spotweld beam */
    pid2: number;
/** If _PID option is set. Can be true or false */
    pid_opt: boolean;
/** Rotational release code at node 1 */
    rr1: number;
/** Rotational release code at node 2 */
    rr2: number;
/** Translational release code at node 1 */
    rt1: number;
/** Translational release code at node 2 */
    rt2: number;
/** If _SCALAR option is set. Can be true or false */
    scalar: boolean;
/** If _SCALR option is set. Can be true or false */
    scalr: boolean;
/** If _SECTION option is set. Can be true or false */
    section: boolean;
/** Scalar Node number 1 */
    sn1: number;
/** Scalar Node number 2 */
    sn2: number;
/** Section type */
    stype: string;
/** If _THICKNESS option is set. Can be true or false */
    thickness: boolean;
/** The transparency of the beam (0-100) 0% is opaque, 100% is transparent. */
    transparency: number;
/** Volume of beam */
    vol: number;
/** Orientation vector X at node 1 */
    vx: number;
/** Orientation vector Y at node 1 */
    vy: number;
/** Orientation vector Z at node 1 */
    vz: number;
/** If WARPAGE option is set. Can be true or false */
    warpage: boolean;
/** Offset vector X at node 1 */
    wx1: number;
/** Offset vector X at node 2 */
    wx2: number;
/** Offset vector Y at node 1 */
    wy1: number;
/** Offset vector Y at node 2 */
    wy2: number;
/** Offset vector Z at node 1 */
    wz1: number;
/** Offset vector Z at node 2 */
    wz2: number;
}


/** Object function argument in InsertPoint */
interface InsertPointArgument_data {
    /** Point fixity type. Bitwise 'or' of the Path point fixity constants: Belt.B_POST_SLIPRING, Belt.FREE_SLIPRING, Belt.KNOWN, Belt.RETRACTOR, Belt.TWIST, Belt.XSEC */
    fixity: number;
    /** Node label (not required if using x, y and z) */
    node?: number;
    /** Twist node 1 label */
    tnode1?: number;
    /** Twist node 2 label */
    tnode2?: number;
    /** X component of normal vector 1 */
    tnx1?: number;
    /** X component of normal vector 2 */
    tnx2?: number;
    /** Y component of normal vector 1 */
    tny1?: number;
    /** Y component of normal vector 2 */
    tny2?: number;
    /** Z component of normal vector 1 */
    tnz1?: number;
    /** Z component of normal vector 2 */
    tnz2?: number;
    /** X component of twist radial vector 1 */
    trx1?: number;
    /** X component of twist radial vector 2 */
    trx2?: number;
    /** Y component of twist radial vector 1 */
    try1?: number;
    /** Y component of twist radial vector 2 */
    try2?: number;
    /** Z component of twist radial vector 1 */
    trz1?: number;
    /** Z component of twist radial vector 2 */
    trz2?: number;
    /** X coordinate (not required if using node) */
    x?: number;
    /** Y coordinate (not required if using node) */
    y?: number;
    /** Z coordinate (not required if using node) */
    z?: number;
}


/** Object function argument in SetMesh */
interface SetMeshArgument_data {
    /** 1st base point number */
    base_pt1: number;
    /** 2nd base point number */
    base_pt2: number;
    /** Number of belt elements at the 1st end for mixed modes */
    lb1: number;
    /** Number of belt elements at the 2nd end for mixed modes */
    lb2: number;
    /** Meshing modes can be of old style or new style. The following old style constants are available: Belt.MSEG_B1_ONLY, Belt.MSEG_B2_ONLY, Belt.MSEG_SH_ONLY, Belt.MSEG_MIX_SB1, Belt.MSEG_MIX_SB2 The following constant must be invoked in order to use the new style: Belt.MSEG_BD_NEW The following new style constants are available: Belt.MSEG_E1_1D, Belt.MSEG_E1_2D, Belt.MSEG_E1_SH, Belt.MSEG_E2_1D, Belt.MSEG_E2_2D, Belt.MSEG_E2_SH, Belt.MSEG_CE_1D, Belt.MSEG_CE_2D, Belt.MSEG_CE_SH */
    mode: number;
    /** 1st path point number */
    path_pt1: number;
    /** 2nd path point number */
    path_pt2: number;
    /** Retractor id to be used */
    retractor?: number;
}


/** Object function argument in SetPoint */
interface SetPointArgument_data {
    /** Point fixity type. Bitwise 'or' of the Path point fixity constants: Belt.B_POST_SLIPRING, Belt.FREE_SLIPRING, Belt.KNOWN, Belt.RETRACTOR, Belt.TWIST, Belt.XSEC */
    fixity: number;
    /** Node label (not required if using x, y and z) */
    node?: number;
    /** Twist node 1 label */
    tnode1?: number;
    /** Twist node 2 label */
    tnode2?: number;
    /** X component of normal vector 1 */
    tnx1?: number;
    /** X component of normal vector 2 */
    tnx2?: number;
    /** Y component of normal vector 1 */
    tny1?: number;
    /** Y component of normal vector 2 */
    tny2?: number;
    /** Z component of normal vector 1 */
    tnz1?: number;
    /** Z component of normal vector 2 */
    tnz2?: number;
    /** X component of twist radial vector 1 */
    trx1?: number;
    /** X component of twist radial vector 2 */
    trx2?: number;
    /** Y component of twist radial vector 1 */
    try1?: number;
    /** Y component of twist radial vector 2 */
    try2?: number;
    /** Z component of twist radial vector 1 */
    trz1?: number;
    /** Z component of twist radial vector 2 */
    trz2?: number;
    /** X coordinate (not required if using node) */
    x?: number;
    /** Y coordinate (not required if using node) */
    y?: number;
    /** Z coordinate (not required if using node) */
    z?: number;
}

declare class Belt {
/**
 * Associates a comment with a belt.
 * @param Comment Comment that will be attached to the belt
 */
    AssociateComment(Comment: Comment): void;

/**
 * Blanks the belt
 */
    Blank(): void;

/**
 * Blanks all of the belts in the model.
 * @param Model Model that all belts will be blanked in
 * @param redraw If model should be redrawn or not. If omitted redraw is false. If you want to do several (un)blanks and only redraw after the last one then use false for all redraws apart from the last one. Alternatively you can redraw using View.Redraw().
 */
    static BlankAll(Model: Model, redraw?: boolean): void;

/**
 * Blanks all of the flagged belts in the model.
 * @param Model Model that all the flagged belts will be blanked in
 * @param flag Flag set on the belts that you want to blank
 * @param redraw If model should be redrawn or not. If omitted redraw is false. If you want to do several (un)blanks and only redraw after the last one then use false for all redraws apart from the last one. Alternatively you can redraw using View.Redraw().
 */
    static BlankFlagged(Model: Model, flag: number, redraw?: boolean): void;

/**
 * Checks if the belt is blanked or not.
 * @returns true if blanked, false if not.
 */
    Blanked(): boolean;

/**
 * Clears a flag on the belt.
 * @param flag Flag to clear on the belt
 */
    ClearFlag(flag: number): void;

/**
 * Copies the belt. The target include of the copied belt can be set using Options.copy_target_include.
 * @param range If you want to keep the copied item in the range specified for the current include. Default value is false. To set current include, use Include.MakeCurrentLayer().
 * @returns Belt object
 */
    Copy(range?: boolean): Belt;

/**
 * Detaches a comment from a belt.
 * @param Comment Comment that will be detached from the belt
 */
    DetachComment(Comment: Comment): void;

/**
 * Adds an error for belt. For more details on checking see the Check class.
 * @param message The error message to give
 * @param details An optional detailed error message
 */
    Error(message: string, details?: string): void;

/**
 * Returns the first belt in the model.
 * @param Model Model to get first belt in
 * @returns Belt object (or null if there are no belts in the model).
 */
    static First(Model: Model): Belt;

/**
 * Returns the first free belt label in the model. Also see Belt.LastFreeLabel(), Belt.NextFreeLabel() and Model.FirstFreeItemLabel().
 * @param Model Model to get first free belt label in
 * @param layer Include file (0 for the main file) to search for labels in (Equivalent to First free in layer in editing panels). If omitted the whole model will be used (Equivalent to First free in editing panels).
 * @returns Belt label.
 */
    static FirstFreeLabel(Model: Model, layer?: number): number;

/**
 * (Re)fits belt
 */
    Fit(): void;

/**
 * Flags all of the belts in the model with a defined flag.
 * @param Model Model that all belts will be flagged in
 * @param flag Flag to set on the belts
 */
    static FlagAll(Model: Model, flag: number): void;

/**
 * Checks if the belt is flagged or not.
 * @param flag Flag to test on the belt
 * @returns true if flagged, false if not.
 */
    Flagged(flag: number): boolean;

/**
 * Calls a function for each belt in the model. Note that ForEach has been designed to make looping over belts as fast as possible and so has some limitations. Firstly, a single temporary Belt object is created and on each function call it is updated with the current belt data. This means that you should not try to store the Belt object for later use (e.g. in an array) as it is temporary. Secondly, you cannot create new belts inside a ForEach loop. 
 * @param Model Model that all belts are in
 * @param func Function to call for each belt
 * @param extra An optional extra object/array/string etc that will appended to arguments when calling the function
 */
    static ForEach(Model: Model, func: () => void, extra?: any): void;

/**
 * Generates belt mesh. Extracts and uses existing mesh properties when a mesh is present; inserts a default mesh otherwise.
 */
    Generate(): void;

/**
 * Returns an array of Belt objects or properties for all of the belts in a model in PRIMER. If the optional property argument is not given then an array of Belt objects is returned. If the property argument is given, that property value for each belt is returned in the array instead of a Belt object
 * @param Model Model to get belts from
 * @param property Name for property to get for all belts in the model
 * @returns Array of Belt objects or properties
 */
    static GetAll(Model: Model, property?: string): Belt[];

/**
 * Extracts the comments associated to a belt.
 * @returns Array of Comment objects (or null if there are no comments associated to the node).
 */
    GetComments(): Comment[];

/**
 * Returns an array of Belt objects for all of the flagged belts in a model in PRIMER If the optional property argument is not given then an array of Belt objects is returned. If the property argument is given, then that property value for each belt is returned in the array instead of a Belt object
 * @param Model Model to get belts from
 * @param flag Flag set on the belts that you want to retrieve
 * @param property Name for property to get for all flagged belts in the model
 * @returns Array of Belt objects or properties
 */
    static GetFlagged(Model: Model, flag: number, property?: string): Belt[];

/**
 * Returns the Belt object for a belt ID.
 * @param Model Model to find the belt in
 * @param number number of the belt you want the Belt object for
 * @returns Belt object (or null if belt does not exist).
 */
    static GetFromID(Model: Model, number: number): Belt;

/**
 * Returns the information for a belt mesh section (properties base_pt1, base_pt2, path_pt1, path_pt2, mode, lb1, lb2). See Belt.SetMesh() for more information on supported properties. Must be preceded by a call to Belt.Generate().
 * @param index The index of the mesh section you want the information for. Note that mesh segments start at 0, not 1. 0 &lt;= index &lt; meshSegs
 * @returns Object containing the mesh section information
 */
    GetMesh(index: number): object;

/**
 * Checks if a Belt property is a parameter or not. Note that object properties that are parameters are normally returned as the integer or float parameter values as that is virtually always what the user would want. For this function to work the JavaScript interpreter must use the parameter name instead of the value. This can be done by setting the Options.property_parameter_names option to true before calling the function and then resetting it to false afterwards.. This behaviour can also temporarily be switched by using the Belt.ViewParameters() method and 'method chaining' (see the examples below).
 * @param prop belt property to get parameter for
 * @returns Parameter object if property is a parameter, null if not.
 */
    GetParameter(prop: string): Parameter;

/**
 * Returns the information for a path point (properties fixity, x, y, z, node, trx1, try1, trz1, tnx1, tny1, tnz1, tnode1, trx2, try2, trz2, tnx2, tny2, tnz2, tnode2). Properties fixity, x, y, z and node will always be returned. Twist properties trx1, try1, trz1, tnx1, tny1, tnz1, tnode1, trx2, try2, trz2, tnx2, tny2, tnz2 and tnode2 will only be returned if defined for the point.
 * @param index The index of the path point you want the information for. Note that path points start at 0, not 1. 0 &lt;= index &lt; points
 * @returns Object containing the path point information
 */
    GetPoint(index: number): object;

/**
 * Inserts a path point before/after an existing one. Subsequent path points will be moved 'up' as required.
 * @param index The index of an existing path point. Note that path points start at 0, not 1. 0 &lt;= index &lt; points
 * @param position Do we want to insert before or after the path point denoted by index? The position can be Belt.INSERT_AFTER or Belt.INSERT_BEFORE
 * @param data Object containing the path point data.
 */
    InsertPoint(index: number, position: number, data: InsertPointArgument_data): void;

/**
 * Returns the last belt in the model.
 * @param Model Model to get last belt in
 * @returns Belt object (or null if there are no belts in the model).
 */
    static Last(Model: Model): Belt;

/**
 * Returns the last free belt label in the model. Also see Belt.FirstFreeLabel(), Belt.NextFreeLabel() and see Model.LastFreeItemLabel()
 * @param Model Model to get last free belt label in
 * @param layer Include file (0 for the main file) to search for labels in (Equivalent to Highest free in layer in editing panels). If omitted the whole model will be used.
 * @returns Belt label.
 */
    static LastFreeLabel(Model: Model, layer?: number): number;

/**
 * Returns the next belt in the model.
 * @returns Belt object (or null if there are no more belts in the model).
 */
    Next(): Belt;

/**
 * Returns the next free (highest+1) belt label in the model. Also see Belt.FirstFreeLabel(), Belt.LastFreeLabel() and Model.NextFreeItemLabel()
 * @param Model Model to get next free belt label in
 * @param layer Include file (0 for the main file) to search for labels in (Equivalent to Highest+1 in layer in editing panels). If omitted the whole model will be used (Equivalent to Highest+1 in editing panels).
 * @returns Belt label.
 */
    static NextFreeLabel(Model: Model, layer?: number): number;

/**
 * Allows the user to pick a belt.
 * @param prompt Text to display as a prompt to the user
 * @param limit If the argument is a Model then only belts from that model can be picked. If the argument is a Flag then only belts that are flagged with limit can be selected. If omitted, or null, any belts from any model can be selected. from any model.
 * @param modal If picking is modal (blocks the user from doing anything else in PRIMER until this window is dismissed). If omitted the pick will be modal.
 * @param button_text By default the window with the prompt will have a button labelled 'Cancel' which if pressed will cancel the pick and return null. If you want to change the text on the button use this argument. If omitted 'Cancel' will be used.
 * @returns Belt object (or null if not picked)
 */
    static Pick(prompt: string, limit?: Model | number, modal?: boolean, button_text?: string): Belt;

/**
 * Returns the previous belt in the model.
 * @returns Belt object (or null if there are no more belts in the model).
 */
    Previous(): Belt;

/**
 * Removes a path point from a belt
 * @param index The index of the path point you want to remove. Note that path points start at 0, not 1. 0 &lt;= index &lt; points
 */
    RemovePoint(index: number): void;

/**
 * Renumbers all of the belts in the model.
 * @param Model Model that all belts will be renumbered in
 * @param start Start point for renumbering
 */
    static RenumberAll(Model: Model, start: number): void;

/**
 * Renumbers all of the flagged belts in the model.
 * @param Model Model that all the flagged belts will be renumbered in
 * @param flag Flag set on the belts that you want to renumber
 * @param start Start point for renumbering
 */
    static RenumberFlagged(Model: Model, flag: number, start: number): void;

/**
 * Allows the user to select belts using standard PRIMER object menus.
 * @param flag Flag to use when selecting belts
 * @param prompt Text to display as a prompt to the user
 * @param limit If the argument is a Model then only belts from that model can be selected. If the argument is a Flag then only belts that are flagged with limit can be selected (limit should be different to flag). If omitted, or null, any belts can be selected. from any model.
 * @param modal If selection is modal (blocks the user from doing anything else in PRIMER until this window is dismissed). If omitted the selection will be modal.
 * @returns Number of belts selected or null if menu cancelled
 */
    static Select(flag: number, prompt: string, limit?: Model | number, modal?: boolean): number;

/**
 * Sets a flag on the belt.
 * @param flag Flag to set on the belt
 */
    SetFlag(flag: number): void;

/**
 * Sets the data for various properties for a mesh section in a belt. Values for properties not invoked will be retained as is. Must be preceded by a call to Belt.Generate()
 * @param index The index of the mesh section you want to set. Note that mesh segments start at 0, not 1
 * @param data Object containing the mesh section data.
 */
    SetMesh(index: number, data: SetMeshArgument_data): void;

/**
 * Set the start labels for the entities created for a Seat Belt.
 * @param entity_type The Meshing label can be Belt.MESH_NODE,  Belt.MESH_SHELL, Belt.MESH_SET_NODE, Belt.MESH_SET_SHELL, Belt.MESH_SEATBELT, Belt.MESH_NRBC, BELT.MESH_RETRACTOR, Belt.MESH_XSEC, Belt.MESH_SLIPRING, Belt.MESH_SET_PART, Belt.MESH_2D_SLIPRING_SET_NODE, Belt.MESH_ALL.
 * @param label_value The initial label value to be assigned for the entity type.
 */
    static SetMeshingLabels(entity_type: number, label_value: number): void;

/**
 * Sets the data for a path point in a belt
 * @param index The index of the path point you want to set. Note that path points start at 0, not 1. To add a new point use index points
 * @param data Object containing the path point data.
 */
    SetPoint(index: number, data: SetPointArgument_data): void;

/**
 * Sketches the belt. The belt will be sketched until you either call Belt.Unsketch(), Belt.UnsketchAll(), Model.UnsketchAll(), or delete the model
 * @param redraw If model should be redrawn or not after the belt is sketched. If omitted redraw is true. If you want to sketch several belts and only redraw after the last one then use false for redraw and call View.Redraw().
 */
    Sketch(redraw?: boolean): void;

/**
 * Sketches all of the flagged belts in the model. The belts will be sketched until you either call Belt.Unsketch(), Belt.UnsketchFlagged(), Model.UnsketchAll(), or delete the model
 * @param Model Model that all the flagged belts will be sketched in
 * @param flag Flag set on the belts that you want to sketch
 * @param redraw If model should be redrawn or not after the belts are sketched. If omitted redraw is true. If you want to sketch flagged belts several times and only redraw after the last one then use false for redraw and call View.Redraw().
 */
    static SketchFlagged(Model: Model, flag: number, redraw?: boolean): void;

/**
 * Returns the total number of belts in the model.
 * @param Model Model to get total for
 * @param exists true if only existing belts should be counted. If false or omitted referenced but undefined belts will also be included in the total.
 * @returns number of belts
 */
    static Total(Model: Model, exists?: boolean): number;

/**
 * Unblanks the belt
 */
    Unblank(): void;

/**
 * Unblanks all of the belts in the model.
 * @param Model Model that all belts will be unblanked in
 * @param redraw If model should be redrawn or not. If omitted redraw is false. If you want to do several (un)blanks and only redraw after the last one then use false for all redraws apart from the last one. Alternatively you can redraw using View.Redraw().
 */
    static UnblankAll(Model: Model, redraw?: boolean): void;

/**
 * Unblanks all of the flagged belts in the model.
 * @param Model Model that the flagged belts will be unblanked in
 * @param flag Flag set on the belts that you want to unblank
 * @param redraw If model should be redrawn or not. If omitted redraw is false. If you want to do several (un)blanks and only redraw after the last one then use false for all redraws apart from the last one. Alternatively you can redraw using View.Redraw().
 */
    static UnblankFlagged(Model: Model, flag: number, redraw?: boolean): void;

/**
 * Unsets a defined flag on all of the belts in the model.
 * @param Model Model that the defined flag for all belts will be unset in
 * @param flag Flag to unset on the belts
 */
    static UnflagAll(Model: Model, flag: number): void;

/**
 * Unsketches the belt.
 * @param redraw If model should be redrawn or not after the belt is unsketched. If omitted redraw is true. If you want to unsketch several belts and only redraw after the last one then use false for redraw and call View.Redraw().
 */
    Unsketch(redraw?: boolean): void;

/**
 * Unsketches all belts.
 * @param Model Model that all belts will be unblanked in
 * @param redraw If model should be redrawn or not after the belts are unsketched. If omitted redraw is true. If you want to unsketch several things and only redraw after the last one then use false for redraw and call View.Redraw().
 */
    static UnsketchAll(Model: Model, redraw?: boolean): void;

/**
 * Unsketches all flagged belts in the model.
 * @param Model Model that all belts will be unsketched in
 * @param flag Flag set on the belts that you want to unsketch
 * @param redraw If model should be redrawn or not after the belts are unsketched. If omitted redraw is true. If you want to unsketch several things and only redraw after the last one then use false for redraw and call View.Redraw().
 */
    static UnsketchFlagged(Model: Model, flag: number, redraw?: boolean): void;

/**
 * Object properties that are parameters are normally returned as the integer or float parameter values as that is virtually always what the user would want. This function temporarily changes the behaviour so that if a property is a parameter the parameter name is returned instead. This can be used with 'method chaining' (see the example below) to make sure a property argument is correct.
 * @returns Belt object.
 */
    ViewParameters(): Belt;

/**
 * Adds a warning for belt. For more details on checking see the Check class.
 * @param message The warning message to give
 * @param details An optional detailed warning message
 */
    Warning(message: string, details?: string): void;

/**
 * Returns the cross references for this belt.
 * @returns Xrefs object.
 */
    Xrefs(): Xrefs;

/**
 * Create a new Belt object.
 * @param model Model that the belt definition will be created in
 * @param id Belt number
 * @param title Title for the belt
 * @param structural_type Seatbelt will be fitted around this entity type. This will trigger creation of sets as required. Type can be one of MODEL, DUMMY, PART, any ELEMENT subtype such as SHELL, or any SET subtype such as SET_PART. See Appendix I of the PRIMER manual for more information on PRIMER types
 * @param flag Flag used to identify entities that the belt should fit around. This argument is ignored if structural_type is MODEL. Instead, the current model is used
 * @returns Belt object
 */
    constructor(model: Model, id: number, title?: string, structural_type?: string, flag?: number);

/** Limiting angle to be considered "acute" (0 means 90) */
    acuteAngle: number;
/** Maximum permitted transverse belt curvature in degrees */
    curvature: number;
/** Set of shell or 2D seatbelt elements. Only created if the option to generate a contact for the belt is used */
    readonly elemSet: number;
/** true if belt exists, false if referred to but not defined. */
    readonly exists: boolean;
/** Transverse friction coefficient */
    friction: number;
/** Belt number. Also see the label property which is an alternative name for this. */
    id: number;
/** The Include file number that the belt is in. */
    include: number;
/** The number of fitting iterations between contact bucket resorts */
    iterations: number;
/** Belt number. Also see the id property which is an alternative name for this. */
    label: number;
/** The characteristic length of each belt element */
    length: number;
/** Number of mesh segments defined */
    readonly meshSegs: number;
/** The Model number that the belt is in. */
    readonly model: number;
/** Nodes to Surface contact used between nodes on 1D belt elements and dummy structure. Only set if the AUTOMATIC_NODES_TO_SURFACE contact is defined in Seatbelts contact panel. Optional contact, see "Contact: Creating a Contact between Belt and Dummy" section in the PRIMER manual for more information. */
    readonly n2sContact: number;
/** Set of all nodes in seatbelt. Only created if the option to generate a contact for the belt is used */
    readonly nodeSet: number;
/** First nodal rigid body ID */
    readonly nrbFirst: number;
/** Last nodal rigid body ID */
    readonly nrbLast: number;
/** Set of nodes on 1D seatbelt elements only. Only created if the option to generate a contact for the belt is used */
    readonly nsboSet: number;
/** Fraction by which facets are extended during contact checking to stop nodes "falling into gaps" */
    overlap: number;
/** Part set ID defining structure. Note that if you are creating the seatbelt definition from scratch in JavaScript you must define a shell, solid or thick shell set. */
    parts: number;
/** Maximum penetration distance considered for contact into solid and thick shell elements */
    penetration: number;
/** The part ID for any 2D seatbelt elements */
    pidShell: number;
/** The part ID for any 1D seatbelt elements */
    pid_1d: number;
/** The part ID for any 2D seatbelt elements */
    pid_2d: number;
/** Number of path points defined */
    readonly points: number;
/** Initial projection distance by which belt path is "thrown outwards" at start of fitting */
    projection: number;
/** Optional orthotropic angle for any shell elements */
    psiShell: number;
/** Optional orthotropic angle for any 2D seatbelt elements */
    psi_2d: number;
/** First retractor ID */
    readonly retractorFirst: number;
/** Last retractor ID */
    readonly retractorLast: number;
/** The number of rows of 2D elements across the belt */
    rows: number;
/** Surface to Surface contact used between shell/2D belt elements and dummy structure. Only set if the AUTOMATIC_SURFACE_TO_SURFACE contact is defined in Seatbelts contact panel. Optional contact, see "Contact: Creating a Contact between Belt and Dummy" section in the PRIMER manual for more information. */
    readonly s2sContact: number;
/** First 1D seatbelt ID */
    readonly seatbeltFirst: number;
/** Last 1D seatbelt ID */
    readonly seatbeltLast: number;
/** Segment set created for contact */
    readonly segments: number;
/** Shell set ID defining structure. Note that if you are creating the seatbelt definition from scratch in JavaScript you must define a shell, solid or thick shell set. */
    shells: number;
/** The initial slack length for any 1D seatbelt elements */
    slen_1d: number;
/** First slipring ID */
    readonly slipringFirst: number;
/** Last slipring ID */
    readonly slipringLast: number;
/** Solid set ID defining structure. Note that if you are creating the seatbelt definition from scratch in JavaScript you must define a shell, solid or thick shell set. */
    solids: number;
/** Optional thickness at n1 for any shell elements */
    t1Shell: number;
/** Optional thickness at n1 for any 2D seatbelt elements */
    t1_2d: number;
/** Optional thickness at n2 for any shell elements */
    t2Shell: number;
/** Optional thickness at n2 for any 2D seatbelt elements */
    t2_2d: number;
/** Optional thickness at n3 for any shell elements */
    t3Shell: number;
/** Optional thickness at n3 for any 2D seatbelt elements */
    t3_2d: number;
/** Optional thickness at n4 for any shell elements */
    t4Shell: number;
/** Optional thickness at n4 for any 2D seatbelt elements */
    t4_2d: number;
/** Factor used when thickFlag is 1 */
    thickFactor: number;
/** Thickness used during fitting: 0 (default)=use true thickness; 1=use true thickness x factor; 2=use neutral axis (no thickness) */
    thickFlag: number;
/** The thickness of 2D belt elements */
    thickness: number;
/** Belt title. */
    title: string;
/** The convergence tolerance at which fitting halts */
    tolerance: number;
/** Thick shell set ID defining structure. Note that if you are creating the seatbelt definition from scratch in JavaScript you must define a shell, solid or thick shell set. */
    tshells: number;
/** The overall belt width */
    width: number;
/** If X-Section pretext option is set to 2 then string for additional pretext */
    xsect_pretext: string;
/** X-Section pretext option, 0: None, 1: Automatic, 2: Manual */
    xsect_pretext_option: number;
/** First cross section ID */
    readonly xsectionFirst: number;
/** Last cross section ID */
    readonly xsectionLast: number;
/** There is a B-Post slipring at this point. */
    static B_POST_SLIPRING: number;
/** Point is fixed */
    static FIXED: number;
/** There is a free (eg pelvis) slipring at this point. */
    static FREE_SLIPRING: number;
/** Insert after given path point. */
    static INSERT_AFTER: number;
/** Insert before given path point. */
    static INSERT_BEFORE: number;
/** The belt path is known to pass through this point */
    static KNOWN: number;
/** Set meshing start Labels for 2D slipring node sets */
    static MESH_2D_SLIPRING_SET_NODE: number;
/** Set meshing start Labels for everything used in the seatbelt definition */
    static MESH_ALL: number;
/** Set meshing start Labels for nodes */
    static MESH_NODE: number;
/** Set meshing start Labels for nodal rigid bodies */
    static MESH_NRBC: number;
/** Set meshing start Labels for retractors */
    static MESH_RETRACTOR: number;
/** Set meshing start Labels for seatbelt elements */
    static MESH_SEATBELT: number;
/** Set meshing start Labels for node sets */
    static MESH_SET_NODE: number;
/** Set meshing start Labels for part sets */
    static MESH_SET_PART: number;
/** Set meshing start Labels for shell sets */
    static MESH_SET_SHELL: number;
/** Set meshing start Labels for shells */
    static MESH_SHELL: number;
/** Set meshing start Labels for slipring elements */
    static MESH_SLIPRING: number;
/** Set meshing start Labels for Database cross sections */
    static MESH_XSEC: number;
/** Old style all 1D belt */
    static MSEG_B1_ONLY: number;
/** Old style all 2D belt */
    static MSEG_B2_ONLY: number;
/** Indicates new mode. This must be set before any of the new style constants can be used */
    static MSEG_BD_NEW: number;
/** New style 1D at centre */
    static MSEG_CE_1D: number;
/** New style 2D at centre */
    static MSEG_CE_2D: number;
/** New style SH at centre */
    static MSEG_CE_SH: number;
/** New style 1D at end 1 */
    static MSEG_E1_1D: number;
/** New style 2D at end 1 */
    static MSEG_E1_2D: number;
/** New style shells at end 1 */
    static MSEG_E1_SH: number;
/** New style 1D at end 2 */
    static MSEG_E2_1D: number;
/** New style 2D at end 2 */
    static MSEG_E2_2D: number;
/** New style shells at end 2 */
    static MSEG_E2_SH: number;
/** Old style 1D at ends, shells in middle */
    static MSEG_MIX_SB1: number;
/** Old style 2D at ends, shells in middle */
    static MSEG_MIX_SB2: number;
/** Old style all shell belt */
    static MSEG_SH_ONLY: number;
/** There is a retractor at this point */
    static RETRACTOR: number;
/** There is a slipring at this point. (Deprecated from V12 onwards, use FREE_SLIPRING or B_POST_SLIPRING instead) */
    static SLIPRING: number;
/** Point has twist vectors or twist nodes defined */
    static TWIST: number;
/** There is a database cross section at this point */
    static XSEC: number;
}

declare class Box {
/**
 * Associates a comment with a box.
 * @param Comment Comment that will be attached to the box
 */
    AssociateComment(Comment: Comment): void;

/**
 * Blanks the box
 */
    Blank(): void;

/**
 * Blanks all of the boxs in the model.
 * @param Model Model that all boxs will be blanked in
 * @param redraw If model should be redrawn or not. If omitted redraw is false. If you want to do several (un)blanks and only redraw after the last one then use false for all redraws apart from the last one. Alternatively you can redraw using View.Redraw().
 */
    static BlankAll(Model: Model, redraw?: boolean): void;

/**
 * Blanks all of the flagged boxs in the model.
 * @param Model Model that all the flagged boxs will be blanked in
 * @param flag Flag set on the boxs that you want to blank
 * @param redraw If model should be redrawn or not. If omitted redraw is false. If you want to do several (un)blanks and only redraw after the last one then use false for all redraws apart from the last one. Alternatively you can redraw using View.Redraw().
 */
    static BlankFlagged(Model: Model, flag: number, redraw?: boolean): void;

/**
 * Checks if the box is blanked or not.
 * @returns true if blanked, false if not.
 */
    Blanked(): boolean;

/**
 * Starts an edit panel in Browse mode.
 * @param modal If this window is modal (blocks the user from doing anything else in PRIMER until this window is dismissed). If omitted the window will be modal.
 */
    Browse(modal?: boolean): void;

/**
 * Clears a flag on the box.
 * @param flag Flag to clear on the box
 */
    ClearFlag(flag: number): void;

/**
 * Copies the box. The target include of the copied box can be set using Options.copy_target_include.
 * @param range If you want to keep the copied item in the range specified for the current include. Default value is false. To set current include, use Include.MakeCurrentLayer().
 * @returns Box object
 */
    Copy(range?: boolean): Box;

/**
 * Starts an interactive editing panel to create a box
 * @param Model Model that the box will be created in.
 * @param modal If this window is modal (blocks the user from doing anything else in PRIMER until this window is dismissed). If omitted the window will be modal.
 * @returns Box object (or null if not made).
 */
    static Create(Model: Model, modal?: boolean): Box;

/**
 * Detaches a comment from a box.
 * @param Comment Comment that will be detached from the box
 */
    DetachComment(Comment: Comment): void;

/**
 * Starts an interactive editing panel.
 * @param modal If this window is modal (blocks the user from doing anything else in PRIMER until this window is dismissed). If omitted the window will be modal.
 */
    Edit(modal?: boolean): void;

/**
 * Adds an error for box. For more details on checking see the Check class.
 * @param message The error message to give
 * @param details An optional detailed error message
 */
    Error(message: string, details?: string): void;

/**
 * Returns the first box in the model.
 * @param Model Model to get first box in
 * @returns Box object (or null if there are no boxs in the model).
 */
    static First(Model: Model): Box;

/**
 * Returns the first free box label in the model. Also see Box.LastFreeLabel(), Box.NextFreeLabel() and Model.FirstFreeItemLabel().
 * @param Model Model to get first free box label in
 * @param layer Include file (0 for the main file) to search for labels in (Equivalent to First free in layer in editing panels). If omitted the whole model will be used (Equivalent to First free in editing panels).
 * @returns Box label.
 */
    static FirstFreeLabel(Model: Model, layer?: number): number;

/**
 * Flags all of the boxs in the model with a defined flag.
 * @param Model Model that all boxs will be flagged in
 * @param flag Flag to set on the boxs
 */
    static FlagAll(Model: Model, flag: number): void;

/**
 * Checks if the box is flagged or not.
 * @param flag Flag to test on the box
 * @returns true if flagged, false if not.
 */
    Flagged(flag: number): boolean;

/**
 * Calls a function for each box in the model. Note that ForEach has been designed to make looping over boxs as fast as possible and so has some limitations. Firstly, a single temporary Box object is created and on each function call it is updated with the current box data. This means that you should not try to store the Box object for later use (e.g. in an array) as it is temporary. Secondly, you cannot create new boxs inside a ForEach loop. 
 * @param Model Model that all boxs are in
 * @param func Function to call for each box
 * @param extra An optional extra object/array/string etc that will appended to arguments when calling the function
 */
    static ForEach(Model: Model, func: () => void, extra?: any): void;

/**
 * Returns an array of Box objects or properties for all of the boxs in a model in PRIMER. If the optional property argument is not given then an array of Box objects is returned. If the property argument is given, that property value for each box is returned in the array instead of a Box object
 * @param Model Model to get boxs from
 * @param property Name for property to get for all boxs in the model
 * @returns Array of Box objects or properties
 */
    static GetAll(Model: Model, property?: string): Box[];

/**
 * Extracts the comments associated to a box.
 * @returns Array of Comment objects (or null if there are no comments associated to the node).
 */
    GetComments(): Comment[];

/**
 * Returns an array of Box objects for all of the flagged boxs in a model in PRIMER If the optional property argument is not given then an array of Box objects is returned. If the property argument is given, then that property value for each box is returned in the array instead of a Box object
 * @param Model Model to get boxs from
 * @param flag Flag set on the boxs that you want to retrieve
 * @param property Name for property to get for all flagged boxs in the model
 * @returns Array of Box objects or properties
 */
    static GetFlagged(Model: Model, flag: number, property?: string): Box[];

/**
 * Returns the Box object for a box ID.
 * @param Model Model to find the box in
 * @param number number of the box you want the Box object for
 * @returns Box object (or null if box does not exist).
 */
    static GetFromID(Model: Model, number: number): Box;

/**
 * Checks if a Box property is a parameter or not. Note that object properties that are parameters are normally returned as the integer or float parameter values as that is virtually always what the user would want. For this function to work the JavaScript interpreter must use the parameter name instead of the value. This can be done by setting the Options.property_parameter_names option to true before calling the function and then resetting it to false afterwards.. This behaviour can also temporarily be switched by using the Box.ViewParameters() method and 'method chaining' (see the examples below).
 * @param prop box property to get parameter for
 * @returns Parameter object if property is a parameter, null if not.
 */
    GetParameter(prop: string): Parameter;

/**
 * Returns the keyword for this box (*DEFINE_BOX). Note that a carriage return is not added. See also Box.KeywordCards()
 * @returns string containing the keyword.
 */
    Keyword(): string;

/**
 * Returns the keyword cards for the box. Note that a carriage return is not added. See also Box.Keyword()
 * @returns string containing the cards.
 */
    KeywordCards(): string;

/**
 * Returns the last box in the model.
 * @param Model Model to get last box in
 * @returns Box object (or null if there are no boxs in the model).
 */
    static Last(Model: Model): Box;

/**
 * Returns the last free box label in the model. Also see Box.FirstFreeLabel(), Box.NextFreeLabel() and see Model.LastFreeItemLabel()
 * @param Model Model to get last free box label in
 * @param layer Include file (0 for the main file) to search for labels in (Equivalent to Highest free in layer in editing panels). If omitted the whole model will be used.
 * @returns Box label.
 */
    static LastFreeLabel(Model: Model, layer?: number): number;

/**
 * Returns the next box in the model.
 * @returns Box object (or null if there are no more boxs in the model).
 */
    Next(): Box;

/**
 * Returns the next free (highest+1) box label in the model. Also see Box.FirstFreeLabel(), Box.LastFreeLabel() and Model.NextFreeItemLabel()
 * @param Model Model to get next free box label in
 * @param layer Include file (0 for the main file) to search for labels in (Equivalent to Highest+1 in layer in editing panels). If omitted the whole model will be used (Equivalent to Highest+1 in editing panels).
 * @returns Box label.
 */
    static NextFreeLabel(Model: Model, layer?: number): number;

/**
 * Allows the user to pick a box.
 * @param prompt Text to display as a prompt to the user
 * @param limit If the argument is a Model then only boxs from that model can be picked. If the argument is a Flag then only boxs that are flagged with limit can be selected. If omitted, or null, any boxs from any model can be selected. from any model.
 * @param modal If picking is modal (blocks the user from doing anything else in PRIMER until this window is dismissed). If omitted the pick will be modal.
 * @param button_text By default the window with the prompt will have a button labelled 'Cancel' which if pressed will cancel the pick and return null. If you want to change the text on the button use this argument. If omitted 'Cancel' will be used.
 * @returns Box object (or null if not picked)
 */
    static Pick(prompt: string, limit?: Model | number, modal?: boolean, button_text?: string): Box;

/**
 * Returns the previous box in the model.
 * @returns Box object (or null if there are no more boxs in the model).
 */
    Previous(): Box;

/**
 * Renumbers all of the boxs in the model.
 * @param Model Model that all boxs will be renumbered in
 * @param start Start point for renumbering
 */
    static RenumberAll(Model: Model, start: number): void;

/**
 * Renumbers all of the flagged boxs in the model.
 * @param Model Model that all the flagged boxs will be renumbered in
 * @param flag Flag set on the boxs that you want to renumber
 * @param start Start point for renumbering
 */
    static RenumberFlagged(Model: Model, flag: number, start: number): void;

/**
 * Allows the user to select boxs using standard PRIMER object menus.
 * @param flag Flag to use when selecting boxs
 * @param prompt Text to display as a prompt to the user
 * @param limit If the argument is a Model then only boxs from that model can be selected. If the argument is a Flag then only boxs that are flagged with limit can be selected (limit should be different to flag). If omitted, or null, any boxs can be selected. from any model.
 * @param modal If selection is modal (blocks the user from doing anything else in PRIMER until this window is dismissed). If omitted the selection will be modal.
 * @returns Number of boxs selected or null if menu cancelled
 */
    static Select(flag: number, prompt: string, limit?: Model | number, modal?: boolean): number;

/**
 * Sets a flag on the box.
 * @param flag Flag to set on the box
 */
    SetFlag(flag: number): void;

/**
 * Sketches the box. The box will be sketched until you either call Box.Unsketch(), Box.UnsketchAll(), Model.UnsketchAll(), or delete the model
 * @param redraw If model should be redrawn or not after the box is sketched. If omitted redraw is true. If you want to sketch several boxs and only redraw after the last one then use false for redraw and call View.Redraw().
 */
    Sketch(redraw?: boolean): void;

/**
 * Sketches all of the flagged boxs in the model. The boxs will be sketched until you either call Box.Unsketch(), Box.UnsketchFlagged(), Model.UnsketchAll(), or delete the model
 * @param Model Model that all the flagged boxs will be sketched in
 * @param flag Flag set on the boxs that you want to sketch
 * @param redraw If model should be redrawn or not after the boxs are sketched. If omitted redraw is true. If you want to sketch flagged boxs several times and only redraw after the last one then use false for redraw and call View.Redraw().
 */
    static SketchFlagged(Model: Model, flag: number, redraw?: boolean): void;

/**
 * Returns the total number of boxs in the model.
 * @param Model Model to get total for
 * @param exists true if only existing boxs should be counted. If false or omitted referenced but undefined boxs will also be included in the total.
 * @returns number of boxs
 */
    static Total(Model: Model, exists?: boolean): number;

/**
 * Unblanks the box
 */
    Unblank(): void;

/**
 * Unblanks all of the boxs in the model.
 * @param Model Model that all boxs will be unblanked in
 * @param redraw If model should be redrawn or not. If omitted redraw is false. If you want to do several (un)blanks and only redraw after the last one then use false for all redraws apart from the last one. Alternatively you can redraw using View.Redraw().
 */
    static UnblankAll(Model: Model, redraw?: boolean): void;

/**
 * Unblanks all of the flagged boxs in the model.
 * @param Model Model that the flagged boxs will be unblanked in
 * @param flag Flag set on the boxs that you want to unblank
 * @param redraw If model should be redrawn or not. If omitted redraw is false. If you want to do several (un)blanks and only redraw after the last one then use false for all redraws apart from the last one. Alternatively you can redraw using View.Redraw().
 */
    static UnblankFlagged(Model: Model, flag: number, redraw?: boolean): void;

/**
 * Unsets a defined flag on all of the boxs in the model.
 * @param Model Model that the defined flag for all boxs will be unset in
 * @param flag Flag to unset on the boxs
 */
    static UnflagAll(Model: Model, flag: number): void;

/**
 * Unsketches the box.
 * @param redraw If model should be redrawn or not after the box is unsketched. If omitted redraw is true. If you want to unsketch several boxs and only redraw after the last one then use false for redraw and call View.Redraw().
 */
    Unsketch(redraw?: boolean): void;

/**
 * Unsketches all boxs.
 * @param Model Model that all boxs will be unblanked in
 * @param redraw If model should be redrawn or not after the boxs are unsketched. If omitted redraw is true. If you want to unsketch several things and only redraw after the last one then use false for redraw and call View.Redraw().
 */
    static UnsketchAll(Model: Model, redraw?: boolean): void;

/**
 * Unsketches all flagged boxs in the model.
 * @param Model Model that all boxs will be unsketched in
 * @param flag Flag set on the boxs that you want to unsketch
 * @param redraw If model should be redrawn or not after the boxs are unsketched. If omitted redraw is true. If you want to unsketch several things and only redraw after the last one then use false for redraw and call View.Redraw().
 */
    static UnsketchFlagged(Model: Model, flag: number, redraw?: boolean): void;

/**
 * Object properties that are parameters are normally returned as the integer or float parameter values as that is virtually always what the user would want. This function temporarily changes the behaviour so that if a property is a parameter the parameter name is returned instead. This can be used with 'method chaining' (see the example below) to make sure a property argument is correct.
 * @returns Box object.
 */
    ViewParameters(): Box;

/**
 * Adds a warning for box. For more details on checking see the Check class.
 * @param message The warning message to give
 * @param details An optional detailed warning message
 */
    Warning(message: string, details?: string): void;

/**
 * Returns the cross references for this box.
 * @returns Xrefs object.
 */
    Xrefs(): Xrefs;

/**
 * Create a new Box object.
 * @param Model Model that box will be created in
 * @param bid Box number
 * @param xmn Minimum X coordinate
 * @param xmx Maximum X coordinate
 * @param ymn Minimum Y coordinate
 * @param ymx Maximum Y coordinate
 * @param zmn Minimum Z coordinate
 * @param zmx Maximum Z coordinate
 * @param heading Title for the box
 * @returns Box object
 */
    constructor(Model: Model, bid: number, xmn: number, xmx: number, ymn: number, ymx: number, zmn: number, zmx: number, heading?: string);

/**
 * Creates a string containing the box data in keyword format. Note that this contains the keyword header and the keyword cards. See also Box.Keyword() and Box.KeywordCards().
 * @returns string
 */
    toString(): string;

/** Box number. Also see the label property which is an alternative name for this. */
    bid: number;
/** Maximum mesh size in 3D tetrahedron adaptivity */
    brmax: number;
/** Minimum mesh size in 3D tetrahedron adaptivity */
    brmin: number;
/** Optional coordinate system ID for tubular drawbead */
    cid: number;
/** X coordinate of offset vector to local origin */
    cx: number;
/** Y coordinate of offset vector to local origin */
    cy: number;
/** Z coordinate of offset vector to local origin */
    cz: number;
/** true if box exists, false if referred to but not defined. */
    readonly exists: boolean;
/** Box heading */
    heading: string;
/** Direction of tooling movement. 1: x-direction, 2: y-direction, 3: z-direction */
    idir: number;
/** Element protection flag. 0: elements inside, 1: elements outside box cannot be coarsened. */
    iflag: number;
/** The Include file number that the box is in. */
    include: number;
/** Box number. Also see the bid property which is an alternative name for this. */
    label: number;
/** Load curve ID to describe motion value versus time */
    lcid: number;
/** Maximum number of refinement levels for elements contained in box */
    level: number;
/** Box movement in global X axis or by node. The ndid property is an alternative name for this. */
    lidx: number;
/** Box movement in global Y axis */
    lidy: number;
/** Box movement in global Z axis */
    lidz: number;
/** Turns _LOCAL on or off */
    local: boolean;
/** The Model number that the box is in. */
    readonly model: number;
/** Box movement in global X axis or by node. The lidx property is an alternative name for this. */
    ndid: number;
/** Referential nodal ID for vd = 2 */
    nid: number;
/** The box option. Can be Box.BOX, Box.BOX_ADAPTIVE, Box.BOX_COARSEN, Box.BOX_DRAWBEAD or Box.BOX_SPH. */
    option: number;
/** Part ID for Box.BOX_ADAPTIVE option */
    pid_adaptive: number;
/** Part ID of blank for Box.BOX_DRAWBEAD option */
    pid_drawbead: number;
/** Radius of tube centered around draw bead */
    radius: number;
/** Part set, part or node set defining the nodal points along draw bead */
    sid: number;
/** Set type for stype. 2: part set ID, 3: part ID, 4: node set ID */
    stype: number;
/** Velocity/Displacement flag. 0: velocity, 1: displacement, 2: referential node */
    vd: number;
/** Vector ID of DOF */
    vid: number;
/** Minimum X coordinate */
    xmn: number;
/** Maximum X coordinate */
    xmx: number;
/** Local V vector X coordinate */
    xv: number;
/** Local X vector X coordinate */
    xx: number;
/** Minimum Y coordinate */
    ymn: number;
/** Maximum Y coordinate */
    ymx: number;
/** Local V vector Y coordinate */
    yv: number;
/** Local X vector Y coordinate */
    yx: number;
/** Minimum Z coordinate */
    zmn: number;
/** Maximum Z coordinate */
    zmx: number;
/** Local V vector Z coordinate */
    zv: number;
/** Local X vector Z coordinate */
    zx: number;
/** Box is *DEFINE_BOX. */
    static BOX: number;
/** Box is *DEFINE_BOX_ADAPTIVE. */
    static BOX_ADAPTIVE: number;
/** Box is *DEFINE_BOX_COARSEN. */
    static BOX_COARSEN: number;
/** Box is *DEFINE_BOX_DRAWBEAD. */
    static BOX_DRAWBEAD: number;
/** Box is *DEFINE_BOX_SPH. */
    static BOX_SPH: number;
}

declare class Check {
/**
 * Adds a comment for a user dashboard check. Multiple comments can be added. Call this function as many times as required.  This function should only be called from a user JavaScript dashboard script.
 * @param comment The comment to add.
 */
    static AddDashboardComment(comment: string): void;

/**
 * Allows the user to add the value of model health based on the other dashboard results  This function should only be called from model_health.config.js which should be placed with the other user defined dashboard scripts.
 * @param model_health Text which will be displayed on the dashboard panel and the summary files.
 * @param Health_text_colour Colour of the model health text. The default colour is Black.
 * @param Health_button_colour Colour of the model health button. The default colour is dark grey.
 */
    static AddDashboardHealth(model_health: string, Health_text_colour?: number, Health_button_colour?: number): void;

/**
 * Adds a custom error. This function should only be called from a custom JavaScript check script. See the details in the Check class for how to do this.
 * @param message The error message to give
 * @param details An optional detailed error message
 */
    static Error(message: string, details?: string): void;

/**
 * Returns data from all the dashboards that are defined.  This function should only be called from model_health.config.js which should be placed with the other user defined dashboard scripts.  The dashboard properties are:  result (Overall result of the dashboard) title (dashboard title) message1 (First message of the dashboard) message2 (Second message of the dashboard) comments (Array of comments on the dashboard) 
 * @returns Array of dashboard objects
 */
    static GetAllDashboards(): dashboard[];

/**
 * Used to proceed with or abort the keyout operation (Ansys LS-DYNA output) from the keyout_hook.js script. The current hooks are launched just before the keyout operation from the model write tab, writing from the dialogue box and during keyout from the include tree. Please look at the example_keyout_script.js for an example of its usage.
 * @param interrupt_flag If this flag is set to true then keyout is aborted else keyout proceeds as usual.
 */
    static KeyoutHook(interrupt_flag: boolean): void;

/**
 * Adds a message for a user dashboard check. Each dashboard can currently show two messages. This function should only be called from a user JavaScript dashboard script.
 * @param first The first message to add.
 * @param second The second message to add.
 */
    static SetDashboardMessage(first: string, second?: string): void;

/**
 * Sets the status of a user dashboard check. This function should only be called from a user JavaScript dashboard script.
 * @param status The status. Can be Check.OK, Check.WARNING, Check.ERROR or Check.UNKNOWN.
 */
    static SetDashboardStatus(status: number): void;

/**
 * Adds a custom warning. This function should only be called from a custom JavaScript check script. See the details in the Check class for how to do this.
 * @param message The warning message to give
 * @param details An optional detailed warning message
 */
    static Warning(message: string, details?: string): void;

/** Colour black */
    static BLACK: number;
/** Colour blue */
    static BLUE: number;
/** Colour cyan */
    static CYAN: number;
/** Colour dark blue */
    static DARKBLUE: number;
/** Colour dark green */
    static DARKGREEN: number;
/** Colour dark grey */
    static DARKGREY: number;
/** Colour dark red */
    static DARKRED: number;
/** Dashboard check gave error(s) */
    static ERROR: number;
/** Colour green */
    static GREEN: number;
/** Colour grey */
    static GREY: number;
/** Colour light grey */
    static LIGHTGREY: number;
/** Colour magenta */
    static MAGENTA: number;
/** Dashboard check status OK */
    static OK: number;
/** Colour orange */
    static ORANGE: number;
/** Colour red */
    static RED: number;
/** Dashboard check status unknown (not run) */
    static UNKNOWN: number;
/** Dashboard check gave warning(s) */
    static WARNING: number;
/** Colour white */
    static WHITE: number;
/** Colour yellow */
    static YELLOW: number;
}

declare class Colour {
/**
 * Returns the colour for a given core or user colour name
 * @param name The name of the colour, for example red or user_green or green/cyan.
 * @returns colour value (integer)
 */
    static GetFromName(name: string): number;

/**
 * Creates a colour from red, green and blue components
 * @param red red component of colour (0-255).
 * @param green green component of colour (0-255).
 * @param blue blue component of colour (0-255).
 * @returns colour value (integer)
 */
    static RGB(red: number, green: number, blue: number): number;

/** Base colour on assembly */
    static ASSEMBLY: number;
/** Background colour */
    static BACKGROUND: number;
/** Colour black */
    static BLACK: number;
/** Colour blue */
    static BLUE: number;
/** Colour cyan */
    static CYAN: number;
/** Colour dark orange */
    static DARK_ORANGE: number;
/** Default colour for objects */
    static DEFAULT: number;
/** Colour green */
    static GREEN: number;
/** Colour green/cyan */
    static GREEN_CYAN: number;
/** Colour grey */
    static GREY: number;
/** Base colour on include file */
    static INCLUDE: number;
/** Colour light blue */
    static LIGHT_BLUE: number;
/** Colour magenta */
    static MAGENTA: number;
/** For elements with part IDs base colour on material ID */
    static MATERIAL: number;
/** Colour medium blue */
    static MEDIUM_BLUE: number;
/** Base colour on model */
    static MODEL: number;
/** Not the background colour */
    static NOT_BACKGROUND: number;
/** Colour orange */
    static ORANGE: number;
/** For elements with part IDs base colour on part ID */
    static PART: number;
/** Colour red */
    static RED: number;
/** Colour red/magenta */
    static RED_MAGENTA: number;
/** For elements with part IDs base colour on section ID */
    static SECTION: number;
/** Sketch colour */
    static SKETCH: number;
/** Text colour */
    static TEXT: number;
/** Colour white */
    static WHITE: number;
/** Colour yellow */
    static YELLOW: number;
/** Colour yellow/green */
    static YELLOW_GREEN: number;
}

declare class Comment {
/**
 * Adds a line, or an array of lines, to a comment object.
 * @param Line_content String that will be added to a line
 * @param Line_number 0: First line, 1: Second line, etc. If array of lines has been passed in the first argument, the first line of the array will be inserted in the line number specified in second argument, the second line of the array will be inserted in the following line number, etc. If that line already exists, that line and rest of them below will be shifted down. If greater than number of existing lines, blank lines will be added. If lower than 0, not valid argument. If no argument, the line(s) will be appended at the end.
 */
    AddLine(Line_content: string | string[], Line_number?: number): void;

/**
 * Starts an edit panel in Browse mode.
 * @param modal If this window is modal (blocks the user from doing anything else in PRIMER until this window is dismissed). If omitted the window will be modal.
 */
    Browse(modal?: boolean): void;

/**
 * Clears a flag on the comment.
 * @param flag Flag to clear on the comment
 */
    ClearFlag(flag: number): void;

/**
 * Copies the comment. The target include of the copied comment can be set using Options.copy_target_include.
 * @param range If you want to keep the copied item in the range specified for the current include. Default value is false. To set current include, use Include.MakeCurrentLayer().
 * @returns Comment object
 */
    Copy(range?: boolean): Comment;

/**
 * Starts an interactive editing panel to create a comment
 * @param Model Model that the comment will be created in.
 * @param modal If this window is modal (blocks the user from doing anything else in PRIMER until this window is dismissed). If omitted the window will be modal.
 * @returns Comment object (or null if not made).
 */
    static Create(Model: Model, modal?: boolean): Comment;

/**
 * Deletes a line of a comment.
 * @param Line_number Line number to delete (starting at 0). The following lines will be shifted up.
 */
    DeleteLine(Line_number: number): void;

/**
 * Starts an interactive editing panel.
 * @param modal If this window is modal (blocks the user from doing anything else in PRIMER until this window is dismissed). If omitted the window will be modal.
 */
    Edit(modal?: boolean): void;

/**
 * Adds an error for comment. For more details on checking see the Check class.
 * @param message The error message to give
 * @param details An optional detailed error message
 */
    Error(message: string, details?: string): void;

/**
 * Returns the first comment in the model.
 * @param Model Model to get first comment in
 * @returns Comment object (or null if there are no comments in the model).
 */
    static First(Model: Model): Comment;

/**
 * Flags all of the comments in the model with a defined flag.
 * @param Model Model that all comments will be flagged in
 * @param flag Flag to set on the comments
 */
    static FlagAll(Model: Model, flag: number): void;

/**
 * Checks if the comment is flagged or not.
 * @param flag Flag to test on the comment
 * @returns true if flagged, false if not.
 */
    Flagged(flag: number): boolean;

/**
 * Calls a function for each comment in the model. Note that ForEach has been designed to make looping over comments as fast as possible and so has some limitations. Firstly, a single temporary Comment object is created and on each function call it is updated with the current comment data. This means that you should not try to store the Comment object for later use (e.g. in an array) as it is temporary. Secondly, you cannot create new comments inside a ForEach loop. 
 * @param Model Model that all comments are in
 * @param func Function to call for each comment
 * @param extra An optional extra object/array/string etc that will appended to arguments when calling the function
 */
    static ForEach(Model: Model, func: () => void, extra?: any): void;

/**
 * Returns an array of Comment objects or properties for all of the comments in a model in PRIMER. If the optional property argument is not given then an array of Comment objects is returned. If the property argument is given, that property value for each comment is returned in the array instead of a Comment object
 * @param Model Model to get comments from
 * @param property Name for property to get for all comments in the model
 * @returns Array of Comment objects or properties
 */
    static GetAll(Model: Model, property?: string): Comment[];

/**
 * Returns an array of Comment objects for all of the flagged comments in a model in PRIMER If the optional property argument is not given then an array of Comment objects is returned. If the property argument is given, then that property value for each comment is returned in the array instead of a Comment object
 * @param Model Model to get comments from
 * @param flag Flag set on the comments that you want to retrieve
 * @param property Name for property to get for all flagged comments in the model
 * @returns Array of Comment objects or properties
 */
    static GetFlagged(Model: Model, flag: number, property?: string): Comment[];

/**
 * Returns the Comment object for a comment ID.
 * @param Model Model to find the comment in
 * @param number number of the comment you want the Comment object for
 * @returns Comment object (or null if comment does not exist).
 */
    static GetFromID(Model: Model, number: number): Comment;

/**
 * Extracts the lines (the strings) from a comment object.
 * @param Line Line number to be extracted. Default value: 0 (first line)
 * @returns String (or null if no lines in the comment and not argument passed)
 */
    GetLine(Line?: number): string;

/**
 * Checks if a Comment property is a parameter or not. Note that object properties that are parameters are normally returned as the integer or float parameter values as that is virtually always what the user would want. For this function to work the JavaScript interpreter must use the parameter name instead of the value. This can be done by setting the Options.property_parameter_names option to true before calling the function and then resetting it to false afterwards.. This behaviour can also temporarily be switched by using the Comment.ViewParameters() method and 'method chaining' (see the examples below).
 * @param prop comment property to get parameter for
 * @returns Parameter object if property is a parameter, null if not.
 */
    GetParameter(prop: string): Parameter;

/**
 * Returns the keyword for this comment (*COMMENT) and the header of the comment if there is one. Note that a carriage return is not added. See also Comment.KeywordCards()
 * @returns string containing the keyword.
 */
    Keyword(): string;

/**
 * Returns the keyword cards for the comment. Note that a carriage return is not added. See also Comment.Keyword()
 * @returns string containing the cards.
 */
    KeywordCards(): string;

/**
 * Returns the last comment in the model.
 * @param Model Model to get last comment in
 * @returns Comment object (or null if there are no comments in the model).
 */
    static Last(Model: Model): Comment;

/**
 * Modifies the content of a line in a comment.
 * @param Line_number Line number to modify (starting at 0)
 * @param New_line_content String that replaces the existing one in a line
 */
    ModifyLine(Line_number: number, New_line_content: string): void;

/**
 * Returns the next comment in the model.
 * @returns Comment object (or null if there are no more comments in the model).
 */
    Next(): Comment;

/**
 * Returns the previous comment in the model.
 * @returns Comment object (or null if there are no more comments in the model).
 */
    Previous(): Comment;

/**
 * Allows the user to select comments using standard PRIMER object menus.
 * @param flag Flag to use when selecting comments
 * @param prompt Text to display as a prompt to the user
 * @param limit If the argument is a Model then only comments from that model can be selected. If the argument is a Flag then only comments that are flagged with limit can be selected (limit should be different to flag). If omitted, or null, any comments can be selected. from any model.
 * @param modal If selection is modal (blocks the user from doing anything else in PRIMER until this window is dismissed). If omitted the selection will be modal.
 * @returns Number of comments selected or null if menu cancelled
 */
    static Select(flag: number, prompt: string, limit?: Model | number, modal?: boolean): number;

/**
 * Sets a flag on the comment.
 * @param flag Flag to set on the comment
 */
    SetFlag(flag: number): void;

/**
 * Returns the total number of comments in the model.
 * @param Model Model to get total for
 * @param exists true if only existing comments should be counted. If false or omitted referenced but undefined comments will also be included in the total.
 * @returns number of comments
 */
    static Total(Model: Model, exists?: boolean): number;

/**
 * Unsets a defined flag on all of the comments in the model.
 * @param Model Model that the defined flag for all comments will be unset in
 * @param flag Flag to unset on the comments
 */
    static UnflagAll(Model: Model, flag: number): void;

/**
 * Object properties that are parameters are normally returned as the integer or float parameter values as that is virtually always what the user would want. This function temporarily changes the behaviour so that if a property is a parameter the parameter name is returned instead. This can be used with 'method chaining' (see the example below) to make sure a property argument is correct.
 * @returns Comment object.
 */
    ViewParameters(): Comment;

/**
 * Adds a warning for comment. For more details on checking see the Check class.
 * @param message The warning message to give
 * @param details An optional detailed warning message
 */
    Warning(message: string, details?: string): void;

/**
 * Returns the cross references for this comment.
 * @returns Xrefs object.
 */
    Xrefs(): Xrefs;

/**
 * Create a new Comment object.
 * @param Model Model that comment will be created in
 * @param Header Comment number
 * @param Mode Anchor: single or multiple
 * @returns Comment object
 */
    constructor(Model: Model, Header?: string, Mode?: number);

/**
 * Creates a string containing the comment data in keyword format. Note that this contains the keyword header and the keyword cards. See also Comment.Keyword() and Comment.KeywordCards().
 * @returns string
 */
    toString(): string;

/** Anchor mode. Can be Comment.SINGLE, Comment.MULTIPLE. */
    anchor_mode: number;
/** true if comment exists, false if referred to but not defined. */
    readonly exists: boolean;
/** The header of the comment, or empty if the comment has no header. */
    header: string;
/** The Include file number that the comment is in. */
    include: number;
/** The Model number that the comment is in. */
    readonly model: number;
/** Number of lines in the comment. */
    nlines: number;
/** true if _NOECHO option is set, false if not. */
    noecho: boolean;
/** The *COMMENT is associated with all cards in the next block of keywords. */
    static MULTIPLE: number;
/** The *COMMENT is associated with just the one immediately following keyword. */
    static SINGLE: number;
}

declare class ConnectionProperties {
/**
 * Allows user to add material data line in *DEFINE_CONNECTION_PROPERTIES.
 */
    AddMaterialDataLine(): void;

/**
 * Associates a comment with a *DEFINE_CONNECTION_PROPERTIES.
 * @param Comment Comment that will be attached to the *DEFINE_CONNECTION_PROPERTIES
 */
    AssociateComment(Comment: Comment): void;

/**
 * Starts an edit panel in Browse mode.
 * @param modal If this window is modal (blocks the user from doing anything else in PRIMER until this window is dismissed). If omitted the window will be modal.
 */
    Browse(modal?: boolean): void;

/**
 * Clears a flag on the *DEFINE_CONNECTION_PROPERTIES.
 * @param flag Flag to clear on the *DEFINE_CONNECTION_PROPERTIES
 */
    ClearFlag(flag: number): void;

/**
 * Copies the *DEFINE_CONNECTION_PROPERTIES. The target include of the copied *DEFINE_CONNECTION_PROPERTIES can be set using Options.copy_target_include.
 * @param range If you want to keep the copied item in the range specified for the current include. Default value is false. To set current include, use Include.MakeCurrentLayer().
 * @returns ConnectionProperties object
 */
    Copy(range?: boolean): ConnectionProperties;

/**
 * Starts an interactive editing panel to create a *DEFINE_CONNECTION_PROPERTIES
 * @param Model Model that the *DEFINE_CONNECTION_PROPERTIES will be created in.
 * @param modal If this window is modal (blocks the user from doing anything else in PRIMER until this window is dismissed). If omitted the window will be modal.
 * @returns ConnectionProperties object (or null if not made).
 */
    static Create(Model: Model, modal?: boolean): ConnectionProperties;

/**
 * Detaches a comment from a *DEFINE_CONNECTION_PROPERTIES.
 * @param Comment Comment that will be detached from the *DEFINE_CONNECTION_PROPERTIES
 */
    DetachComment(Comment: Comment): void;

/**
 * Starts an interactive editing panel.
 * @param modal If this window is modal (blocks the user from doing anything else in PRIMER until this window is dismissed). If omitted the window will be modal.
 */
    Edit(modal?: boolean): void;

/**
 * Adds an error for *DEFINE_CONNECTION_PROPERTIES. For more details on checking see the Check class.
 * @param message The error message to give
 * @param details An optional detailed error message
 */
    Error(message: string, details?: string): void;

/**
 * Returns the first *DEFINE_CONNECTION_PROPERTIES in the model.
 * @param Model Model to get first *DEFINE_CONNECTION_PROPERTIES in
 * @returns ConnectionProperties object (or null if there are no *DEFINE_CONNECTION_PROPERTIESs in the model).
 */
    static First(Model: Model): ConnectionProperties;

/**
 * Returns the first free *DEFINE_CONNECTION_PROPERTIES label in the model. Also see ConnectionProperties.LastFreeLabel(), ConnectionProperties.NextFreeLabel() and Model.FirstFreeItemLabel().
 * @param Model Model to get first free *DEFINE_CONNECTION_PROPERTIES label in
 * @param layer Include file (0 for the main file) to search for labels in (Equivalent to First free in layer in editing panels). If omitted the whole model will be used (Equivalent to First free in editing panels).
 * @returns ConnectionProperties label.
 */
    static FirstFreeLabel(Model: Model, layer?: number): number;

/**
 * Flags all of the *DEFINE_CONNECTION_PROPERTIESs in the model with a defined flag.
 * @param Model Model that all *DEFINE_CONNECTION_PROPERTIESs will be flagged in
 * @param flag Flag to set on the *DEFINE_CONNECTION_PROPERTIESs
 */
    static FlagAll(Model: Model, flag: number): void;

/**
 * Checks if the *DEFINE_CONNECTION_PROPERTIES is flagged or not.
 * @param flag Flag to test on the *DEFINE_CONNECTION_PROPERTIES
 * @returns true if flagged, false if not.
 */
    Flagged(flag: number): boolean;

/**
 * Calls a function for each *DEFINE_CONNECTION_PROPERTIES in the model. Note that ForEach has been designed to make looping over *DEFINE_CONNECTION_PROPERTIESs as fast as possible and so has some limitations. Firstly, a single temporary ConnectionProperties object is created and on each function call it is updated with the current *DEFINE_CONNECTION_PROPERTIES data. This means that you should not try to store the ConnectionProperties object for later use (e.g. in an array) as it is temporary. Secondly, you cannot create new *DEFINE_CONNECTION_PROPERTIESs inside a ForEach loop. 
 * @param Model Model that all *DEFINE_CONNECTION_PROPERTIESs are in
 * @param func Function to call for each *DEFINE_CONNECTION_PROPERTIES
 * @param extra An optional extra object/array/string etc that will appended to arguments when calling the function
 */
    static ForEach(Model: Model, func: () => void, extra?: any): void;

/**
 * Returns an array of ConnectionProperties objects or properties for all of the *DEFINE_CONNECTION_PROPERTIESs in a model in PRIMER. If the optional property argument is not given then an array of ConnectionProperties objects is returned. If the property argument is given, that property value for each *DEFINE_CONNECTION_PROPERTIES is returned in the array instead of a ConnectionProperties object
 * @param Model Model to get *DEFINE_CONNECTION_PROPERTIESs from
 * @param property Name for property to get for all *DEFINE_CONNECTION_PROPERTIESs in the model
 * @returns Array of ConnectionProperties objects or properties
 */
    static GetAll(Model: Model, property?: string): ConnectionProperties[];

/**
 * Extracts the comments associated to a *DEFINE_CONNECTION_PROPERTIES.
 * @returns Array of Comment objects (or null if there are no comments associated to the node).
 */
    GetComments(): Comment[];

/**
 * Returns an array of ConnectionProperties objects for all of the flagged *DEFINE_CONNECTION_PROPERTIESs in a model in PRIMER If the optional property argument is not given then an array of ConnectionProperties objects is returned. If the property argument is given, then that property value for each *DEFINE_CONNECTION_PROPERTIES is returned in the array instead of a ConnectionProperties object
 * @param Model Model to get *DEFINE_CONNECTION_PROPERTIESs from
 * @param flag Flag set on the *DEFINE_CONNECTION_PROPERTIESs that you want to retrieve
 * @param property Name for property to get for all flagged *DEFINE_CONNECTION_PROPERTIESs in the model
 * @returns Array of ConnectionProperties objects or properties
 */
    static GetFlagged(Model: Model, flag: number, property?: string): ConnectionProperties[];

/**
 * Returns the ConnectionProperties object for a *DEFINE_CONNECTION_PROPERTIES ID.
 * @param Model Model to find the *DEFINE_CONNECTION_PROPERTIES in
 * @param number number of the *DEFINE_CONNECTION_PROPERTIES you want the ConnectionProperties object for
 * @returns ConnectionProperties object (or null if *DEFINE_CONNECTION_PROPERTIES does not exist).
 */
    static GetFromID(Model: Model, number: number): ConnectionProperties;

/**
 * Returns the material data at given row in *DEFINE_CONNECTION_PROPERTIES.
 * @param row Material data row number, eg. for first material data, row = 0
 * @returns Array of numbers containing the material id, sigy, e_tan etc. .
 */
    GetMaterialDataLine(row: number): number[];

/**
 * Checks if a ConnectionProperties property is a parameter or not. Note that object properties that are parameters are normally returned as the integer or float parameter values as that is virtually always what the user would want. For this function to work the JavaScript interpreter must use the parameter name instead of the value. This can be done by setting the Options.property_parameter_names option to true before calling the function and then resetting it to false afterwards.. This behaviour can also temporarily be switched by using the ConnectionProperties.ViewParameters() method and 'method chaining' (see the examples below).
 * @param prop *DEFINE_CONNECTION_PROPERTIES property to get parameter for
 * @returns Parameter object if property is a parameter, null if not.
 */
    GetParameter(prop: string): Parameter;

/**
 * Returns the keyword for this *DEFINE_CONNECTION_PROPERTIES Note that a carriage return is not added. See also ConnectionProperties.KeywordCards()
 * @returns string containing the keyword.
 */
    Keyword(): string;

/**
 * Returns the keyword cards for the *DEFINE_CONNECTION_PROPERTIES. Note that a carriage return is not added. See also ConnectionProperties.Keyword()
 * @returns string containing the cards.
 */
    KeywordCards(): string;

/**
 * Returns the last *DEFINE_CONNECTION_PROPERTIES in the model.
 * @param Model Model to get last *DEFINE_CONNECTION_PROPERTIES in
 * @returns ConnectionProperties object (or null if there are no *DEFINE_CONNECTION_PROPERTIESs in the model).
 */
    static Last(Model: Model): ConnectionProperties;

/**
 * Returns the last free *DEFINE_CONNECTION_PROPERTIES label in the model. Also see ConnectionProperties.FirstFreeLabel(), ConnectionProperties.NextFreeLabel() and see Model.LastFreeItemLabel()
 * @param Model Model to get last free *DEFINE_CONNECTION_PROPERTIES label in
 * @param layer Include file (0 for the main file) to search for labels in (Equivalent to Highest free in layer in editing panels). If omitted the whole model will be used.
 * @returns ConnectionProperties label.
 */
    static LastFreeLabel(Model: Model, layer?: number): number;

/**
 * Returns the next *DEFINE_CONNECTION_PROPERTIES in the model.
 * @returns ConnectionProperties object (or null if there are no more *DEFINE_CONNECTION_PROPERTIESs in the model).
 */
    Next(): ConnectionProperties;

/**
 * Returns the next free (highest+1) *DEFINE_CONNECTION_PROPERTIES label in the model. Also see ConnectionProperties.FirstFreeLabel(), ConnectionProperties.LastFreeLabel() and Model.NextFreeItemLabel()
 * @param Model Model to get next free *DEFINE_CONNECTION_PROPERTIES label in
 * @param layer Include file (0 for the main file) to search for labels in (Equivalent to Highest+1 in layer in editing panels). If omitted the whole model will be used (Equivalent to Highest+1 in editing panels).
 * @returns ConnectionProperties label.
 */
    static NextFreeLabel(Model: Model, layer?: number): number;

/**
 * Returns the previous *DEFINE_CONNECTION_PROPERTIES in the model.
 * @returns ConnectionProperties object (or null if there are no more *DEFINE_CONNECTION_PROPERTIESs in the model).
 */
    Previous(): ConnectionProperties;

/**
 * Allows user to remove material data line in *DEFINE_CONNECTION_PROPERTIES.
 * @param row Material data row number, eg. for first material data, row = 0
 */
    RemoveMaterialDataLine(row: number): void;

/**
 * Renumbers all of the *DEFINE_CONNECTION_PROPERTIESs in the model.
 * @param Model Model that all *DEFINE_CONNECTION_PROPERTIESs will be renumbered in
 * @param start Start point for renumbering
 */
    static RenumberAll(Model: Model, start: number): void;

/**
 * Renumbers all of the flagged *DEFINE_CONNECTION_PROPERTIESs in the model.
 * @param Model Model that all the flagged *DEFINE_CONNECTION_PROPERTIESs will be renumbered in
 * @param flag Flag set on the *DEFINE_CONNECTION_PROPERTIESs that you want to renumber
 * @param start Start point for renumbering
 */
    static RenumberFlagged(Model: Model, flag: number, start: number): void;

/**
 * Allows the user to select *DEFINE_CONNECTION_PROPERTIESs using standard PRIMER object menus.
 * @param flag Flag to use when selecting *DEFINE_CONNECTION_PROPERTIESs
 * @param prompt Text to display as a prompt to the user
 * @param limit If the argument is a Model then only *DEFINE_CONNECTION_PROPERTIESs from that model can be selected. If the argument is a Flag then only *DEFINE_CONNECTION_PROPERTIESs that are flagged with limit can be selected (limit should be different to flag). If omitted, or null, any *DEFINE_CONNECTION_PROPERTIESs can be selected. from any model.
 * @param modal If selection is modal (blocks the user from doing anything else in PRIMER until this window is dismissed). If omitted the selection will be modal.
 * @returns Number of *DEFINE_CONNECTION_PROPERTIESs selected or null if menu cancelled
 */
    static Select(flag: number, prompt: string, limit?: Model | number, modal?: boolean): number;

/**
 * Sets a flag on the *DEFINE_CONNECTION_PROPERTIES.
 * @param flag Flag to set on the *DEFINE_CONNECTION_PROPERTIES
 */
    SetFlag(flag: number): void;

/**
 * Allows user to set fields for material data line at given row in *DEFINE_CONNECTION_PROPERTIES.
 * @param row Material data row number, eg. for first material data, row = 0
 * @param mid Material ID
 * @param sigy Default yield stress
 * @param etan Default tangent modulus
 * @param dg_pr Default damage parameter
 * @param rank Default rank value
 * @param sn Default normal strength
 * @param sb Default bending strength
 * @param ss Default shear strength
 * @param exsn Default normal stress exponent
 * @param exsb Default bending stress exponent
 * @param exss Default shear stress exponent
 * @param lcsn Default LC of normal stress scale factor wrt strain rate
 * @param lcsb Default LC of bending stress scale factor wrt strain rate
 * @param lcss Default LC of shear stress scale factor wrt strain rate
 * @param gfad Default fading energy
 * @param sclmrr Default scaling factor for torsional moment in failure function
 */
    SetMaterialDataLine(row: number, mid: number, sigy?: number, etan?: number, dg_pr?: number, rank?: number, sn?: number, sb?: number, ss?: number, exsn?: number, exsb?: number, exss?: number, lcsn?: number, lcsb?: number, lcss?: number, gfad?: number, sclmrr?: number): void;

/**
 * Returns the total number of *DEFINE_CONNECTION_PROPERTIESs in the model.
 * @param Model Model to get total for
 * @param exists true if only existing *DEFINE_CONNECTION_PROPERTIESs should be counted. If false or omitted referenced but undefined *DEFINE_CONNECTION_PROPERTIESs will also be included in the total.
 * @returns number of *DEFINE_CONNECTION_PROPERTIESs
 */
    static Total(Model: Model, exists?: boolean): number;

/**
 * Unsets a defined flag on all of the *DEFINE_CONNECTION_PROPERTIESs in the model.
 * @param Model Model that the defined flag for all *DEFINE_CONNECTION_PROPERTIESs will be unset in
 * @param flag Flag to unset on the *DEFINE_CONNECTION_PROPERTIESs
 */
    static UnflagAll(Model: Model, flag: number): void;

/**
 * Object properties that are parameters are normally returned as the integer or float parameter values as that is virtually always what the user would want. This function temporarily changes the behaviour so that if a property is a parameter the parameter name is returned instead. This can be used with 'method chaining' (see the example below) to make sure a property argument is correct.
 * @returns ConnectionProperties object.
 */
    ViewParameters(): ConnectionProperties;

/**
 * Adds a warning for *DEFINE_CONNECTION_PROPERTIES. For more details on checking see the Check class.
 * @param message The warning message to give
 * @param details An optional detailed warning message
 */
    Warning(message: string, details?: string): void;

/**
 * Returns the cross references for this *DEFINE_CONNECTION_PROPERTIES.
 * @returns Xrefs object.
 */
    Xrefs(): Xrefs;

/**
 * Create a new  *DEFINE_CONNECTION_PROPERTIES object.
 * @param Model Model that *DEFINE_CONNECTION_PROPERTIES will be created in
 * @param con_id *DEFINE_CONNECTION_PROPERTIES id.
 * @param heading Title for the *DEFINE_CONNECTION_PROPERTIES
 * @returns ConnectionProperties object
 */
    constructor(Model: Model, con_id: number, heading?: string);

/**
 * Creates a string containing the connection_properties data in keyword format. Note that this contains the keyword header and the keyword cards. See also ConnectionProperties.Keyword() and ConnectionProperties.KeywordCards().
 * @returns string
 */
    toString(): string;

/** To _ADD case's parent definition */
    add: number;
/** Area equation number. */
    areaeq: number;
/** *DEFINE_CONNECTION_PROPERTIES id. */
    con_id: number;
/** Default damage parameter. */
    d_dg_pr: number;
/** Default damage parameter(function if proprul == 2). */
    d_dg_prf: number;
/** Default tangent modulus. */
    d_etan: number;
/** Default tangent modulus(function if proprul == 2). */
    d_etanf: number;
/** Default bending stress exponent. */
    d_exsb: number;
/** Default bending stress exponent(function if proprul == 2). */
    d_exsbf: number;
/** Default normal stress exponent. */
    d_exsn: number;
/** Default normal stress exponent(function if proprul == 2). */
    d_exsnf: number;
/** Default shear stress exponent. */
    d_exss: number;
/** Default shear stress exponent(function if proprul == 2). */
    d_exssf: number;
/** Default fading energy. */
    d_gfad: number;
/** Default fading energy(function if proprul == 2). */
    d_gfadf: number;
/** Default LC of bending stress scale factor wrt strain rate. */
    d_lcsb: number;
/** Default LC of normal stress scale factor wrt strain rate. */
    d_lcsn: number;
/** Default LC of shear stress scale factor wrt strain rate. */
    d_lcss: number;
/** Default rank value. */
    d_rank: number;
/** Default bending strength. */
    d_sb: number;
/** Default bending strength(function if proprul == 2). */
    d_sbf: number;
/** Default scaling factor for torsional moment in failure function. */
    d_sclmrr: number;
/** Default yield stress. */
    d_sigy: number;
/** Default yield stress(function if proprul == 2). */
    d_sigyf: number;
/** Default normal strength. */
    d_sn: number;
/** Default normal strength(function if proprul == 2). */
    d_snf: number;
/** Default shear strength. */
    d_ss: number;
/** Default shear strength(function if proprul == 2). */
    d_ssf: number;
/** Damage type. */
    dg_typ: number;
/** true if *DEFINE_CONNECTION_PROPERTIES exists, false if referred to but not defined. */
    readonly exists: boolean;
/** The title of the *DEFINE_CONNECTION_PROPERTIES or the empty string if _TITLE is not set */
    heading: string;
/** The Include file number that the *DEFINE_CONNECTION_PROPERTIES is in. */
    include: number;
/** Modelled area flag. */
    moarfl: number;
/** The Model number that the *DEFINE_CONNECTION_PROPERTIES is in. */
    readonly model: number;
/** Property rule number. */
    proprul: number;
}

declare class ConstructionStages {
/**
 * Associates a comment with a *DEFINE_CONSTRUCTION_STAGES.
 * @param Comment Comment that will be attached to the *DEFINE_CONSTRUCTION_STAGES
 */
    AssociateComment(Comment: Comment): void;

/**
 * Starts an edit panel in Browse mode.
 * @param modal If this window is modal (blocks the user from doing anything else in PRIMER until this window is dismissed). If omitted the window will be modal.
 */
    Browse(modal?: boolean): void;

/**
 * Clears a flag on the *DEFINE_CONSTRUCTION_STAGES.
 * @param flag Flag to clear on the *DEFINE_CONSTRUCTION_STAGES
 */
    ClearFlag(flag: number): void;

/**
 * Copies the *DEFINE_CONSTRUCTION_STAGES. The target include of the copied *DEFINE_CONSTRUCTION_STAGES can be set using Options.copy_target_include.
 * @param range If you want to keep the copied item in the range specified for the current include. Default value is false. To set current include, use Include.MakeCurrentLayer().
 * @returns ConstructionStages object
 */
    Copy(range?: boolean): ConstructionStages;

/**
 * Starts an interactive editing panel to create a *DEFINE_CONSTRUCTION_STAGES
 * @param Model Model that the *DEFINE_CONSTRUCTION_STAGES will be created in.
 * @param modal If this window is modal (blocks the user from doing anything else in PRIMER until this window is dismissed). If omitted the window will be modal.
 * @returns ConstructionStages object (or null if not made).
 */
    static Create(Model: Model, modal?: boolean): ConstructionStages;

/**
 * Detaches a comment from a *DEFINE_CONSTRUCTION_STAGES.
 * @param Comment Comment that will be detached from the *DEFINE_CONSTRUCTION_STAGES
 */
    DetachComment(Comment: Comment): void;

/**
 * Starts an interactive editing panel.
 * @param modal If this window is modal (blocks the user from doing anything else in PRIMER until this window is dismissed). If omitted the window will be modal.
 */
    Edit(modal?: boolean): void;

/**
 * Adds an error for *DEFINE_CONSTRUCTION_STAGES. For more details on checking see the Check class.
 * @param message The error message to give
 * @param details An optional detailed error message
 */
    Error(message: string, details?: string): void;

/**
 * Returns the first *DEFINE_CONSTRUCTION_STAGES in the model.
 * @param Model Model to get first *DEFINE_CONSTRUCTION_STAGES in
 * @returns ConstructionStages object (or null if there are no *DEFINE_CONSTRUCTION_STAGESs in the model).
 */
    static First(Model: Model): ConstructionStages;

/**
 * Returns the first free *DEFINE_CONSTRUCTION_STAGES label in the model. Also see ConstructionStages.LastFreeLabel(), ConstructionStages.NextFreeLabel() and Model.FirstFreeItemLabel().
 * @param Model Model to get first free *DEFINE_CONSTRUCTION_STAGES label in
 * @param layer Include file (0 for the main file) to search for labels in (Equivalent to First free in layer in editing panels). If omitted the whole model will be used (Equivalent to First free in editing panels).
 * @returns ConstructionStages label.
 */
    static FirstFreeLabel(Model: Model, layer?: number): number;

/**
 * Flags all of the *DEFINE_CONSTRUCTION_STAGESs in the model with a defined flag.
 * @param Model Model that all *DEFINE_CONSTRUCTION_STAGESs will be flagged in
 * @param flag Flag to set on the *DEFINE_CONSTRUCTION_STAGESs
 */
    static FlagAll(Model: Model, flag: number): void;

/**
 * Checks if the *DEFINE_CONSTRUCTION_STAGES is flagged or not.
 * @param flag Flag to test on the *DEFINE_CONSTRUCTION_STAGES
 * @returns true if flagged, false if not.
 */
    Flagged(flag: number): boolean;

/**
 * Calls a function for each *DEFINE_CONSTRUCTION_STAGES in the model. Note that ForEach has been designed to make looping over *DEFINE_CONSTRUCTION_STAGESs as fast as possible and so has some limitations. Firstly, a single temporary ConstructionStages object is created and on each function call it is updated with the current *DEFINE_CONSTRUCTION_STAGES data. This means that you should not try to store the ConstructionStages object for later use (e.g. in an array) as it is temporary. Secondly, you cannot create new *DEFINE_CONSTRUCTION_STAGESs inside a ForEach loop. 
 * @param Model Model that all *DEFINE_CONSTRUCTION_STAGESs are in
 * @param func Function to call for each *DEFINE_CONSTRUCTION_STAGES
 * @param extra An optional extra object/array/string etc that will appended to arguments when calling the function
 */
    static ForEach(Model: Model, func: () => void, extra?: any): void;

/**
 * Returns an array of ConstructionStages objects or properties for all of the *DEFINE_CONSTRUCTION_STAGESs in a model in PRIMER. If the optional property argument is not given then an array of ConstructionStages objects is returned. If the property argument is given, that property value for each *DEFINE_CONSTRUCTION_STAGES is returned in the array instead of a ConstructionStages object
 * @param Model Model to get *DEFINE_CONSTRUCTION_STAGESs from
 * @param property Name for property to get for all *DEFINE_CONSTRUCTION_STAGESs in the model
 * @returns Array of ConstructionStages objects or properties
 */
    static GetAll(Model: Model, property?: string): ConstructionStages[];

/**
 * Extracts the comments associated to a *DEFINE_CONSTRUCTION_STAGES.
 * @returns Array of Comment objects (or null if there are no comments associated to the node).
 */
    GetComments(): Comment[];

/**
 * Returns an array of ConstructionStages objects for all of the flagged *DEFINE_CONSTRUCTION_STAGESs in a model in PRIMER If the optional property argument is not given then an array of ConstructionStages objects is returned. If the property argument is given, then that property value for each *DEFINE_CONSTRUCTION_STAGES is returned in the array instead of a ConstructionStages object
 * @param Model Model to get *DEFINE_CONSTRUCTION_STAGESs from
 * @param flag Flag set on the *DEFINE_CONSTRUCTION_STAGESs that you want to retrieve
 * @param property Name for property to get for all flagged *DEFINE_CONSTRUCTION_STAGESs in the model
 * @returns Array of ConstructionStages objects or properties
 */
    static GetFlagged(Model: Model, flag: number, property?: string): ConstructionStages[];

/**
 * Returns the ConstructionStages object for a *DEFINE_CONSTRUCTION_STAGES ID.
 * @param Model Model to find the *DEFINE_CONSTRUCTION_STAGES in
 * @param number number of the *DEFINE_CONSTRUCTION_STAGES you want the ConstructionStages object for
 * @returns ConstructionStages object (or null if *DEFINE_CONSTRUCTION_STAGES does not exist).
 */
    static GetFromID(Model: Model, number: number): ConstructionStages;

/**
 * Checks if a ConstructionStages property is a parameter or not. Note that object properties that are parameters are normally returned as the integer or float parameter values as that is virtually always what the user would want. For this function to work the JavaScript interpreter must use the parameter name instead of the value. This can be done by setting the Options.property_parameter_names option to true before calling the function and then resetting it to false afterwards.. This behaviour can also temporarily be switched by using the ConstructionStages.ViewParameters() method and 'method chaining' (see the examples below).
 * @param prop *DEFINE_CONSTRUCTION_STAGES property to get parameter for
 * @returns Parameter object if property is a parameter, null if not.
 */
    GetParameter(prop: string): Parameter;

/**
 * Returns the keyword for this *DEFINE_CONSTRUCTION_STAGES. Note that a carriage return is not added. See also ConstructionStages.KeywordCards()
 * @returns string containing the keyword.
 */
    Keyword(): string;

/**
 * Returns the keyword cards for the *DEFINE_CONSTRUCTION_STAGES. Note that a carriage return is not added. See also ConstructionStages.Keyword()
 * @returns string containing the cards.
 */
    KeywordCards(): string;

/**
 * Returns the last *DEFINE_CONSTRUCTION_STAGES in the model.
 * @param Model Model to get last *DEFINE_CONSTRUCTION_STAGES in
 * @returns ConstructionStages object (or null if there are no *DEFINE_CONSTRUCTION_STAGESs in the model).
 */
    static Last(Model: Model): ConstructionStages;

/**
 * Returns the last free *DEFINE_CONSTRUCTION_STAGES label in the model. Also see ConstructionStages.FirstFreeLabel(), ConstructionStages.NextFreeLabel() and see Model.LastFreeItemLabel()
 * @param Model Model to get last free *DEFINE_CONSTRUCTION_STAGES label in
 * @param layer Include file (0 for the main file) to search for labels in (Equivalent to Highest free in layer in editing panels). If omitted the whole model will be used.
 * @returns ConstructionStages label.
 */
    static LastFreeLabel(Model: Model, layer?: number): number;

/**
 * Returns the next *DEFINE_CONSTRUCTION_STAGES in the model.
 * @returns ConstructionStages object (or null if there are no more *DEFINE_CONSTRUCTION_STAGESs in the model).
 */
    Next(): ConstructionStages;

/**
 * Returns the next free (highest+1) *DEFINE_CONSTRUCTION_STAGES label in the model. Also see ConstructionStages.FirstFreeLabel(), ConstructionStages.LastFreeLabel() and Model.NextFreeItemLabel()
 * @param Model Model to get next free *DEFINE_CONSTRUCTION_STAGES label in
 * @param layer Include file (0 for the main file) to search for labels in (Equivalent to Highest+1 in layer in editing panels). If omitted the whole model will be used (Equivalent to Highest+1 in editing panels).
 * @returns ConstructionStages label.
 */
    static NextFreeLabel(Model: Model, layer?: number): number;

/**
 * Returns the previous *DEFINE_CONSTRUCTION_STAGES in the model.
 * @returns ConstructionStages object (or null if there are no more *DEFINE_CONSTRUCTION_STAGESs in the model).
 */
    Previous(): ConstructionStages;

/**
 * Renumbers all of the *DEFINE_CONSTRUCTION_STAGESs in the model.
 * @param Model Model that all *DEFINE_CONSTRUCTION_STAGESs will be renumbered in
 * @param start Start point for renumbering
 */
    static RenumberAll(Model: Model, start: number): void;

/**
 * Renumbers all of the flagged *DEFINE_CONSTRUCTION_STAGESs in the model.
 * @param Model Model that all the flagged *DEFINE_CONSTRUCTION_STAGESs will be renumbered in
 * @param flag Flag set on the *DEFINE_CONSTRUCTION_STAGESs that you want to renumber
 * @param start Start point for renumbering
 */
    static RenumberFlagged(Model: Model, flag: number, start: number): void;

/**
 * Allows the user to select *DEFINE_CONSTRUCTION_STAGESs using standard PRIMER object menus.
 * @param flag Flag to use when selecting *DEFINE_CONSTRUCTION_STAGESs
 * @param prompt Text to display as a prompt to the user
 * @param limit If the argument is a Model then only *DEFINE_CONSTRUCTION_STAGESs from that model can be selected. If the argument is a Flag then only *DEFINE_CONSTRUCTION_STAGESs that are flagged with limit can be selected (limit should be different to flag). If omitted, or null, any *DEFINE_CONSTRUCTION_STAGESs can be selected. from any model.
 * @param modal If selection is modal (blocks the user from doing anything else in PRIMER until this window is dismissed). If omitted the selection will be modal.
 * @returns Number of *DEFINE_CONSTRUCTION_STAGESs selected or null if menu cancelled
 */
    static Select(flag: number, prompt: string, limit?: Model | number, modal?: boolean): number;

/**
 * Sets a flag on the *DEFINE_CONSTRUCTION_STAGES.
 * @param flag Flag to set on the *DEFINE_CONSTRUCTION_STAGES
 */
    SetFlag(flag: number): void;

/**
 * Returns the total number of *DEFINE_CONSTRUCTION_STAGESs in the model.
 * @param Model Model to get total for
 * @param exists true if only existing *DEFINE_CONSTRUCTION_STAGESs should be counted. If false or omitted referenced but undefined *DEFINE_CONSTRUCTION_STAGESs will also be included in the total.
 * @returns number of *DEFINE_CONSTRUCTION_STAGESs
 */
    static Total(Model: Model, exists?: boolean): number;

/**
 * Unsets a defined flag on all of the *DEFINE_CONSTRUCTION_STAGESs in the model.
 * @param Model Model that the defined flag for all *DEFINE_CONSTRUCTION_STAGESs will be unset in
 * @param flag Flag to unset on the *DEFINE_CONSTRUCTION_STAGESs
 */
    static UnflagAll(Model: Model, flag: number): void;

/**
 * Object properties that are parameters are normally returned as the integer or float parameter values as that is virtually always what the user would want. This function temporarily changes the behaviour so that if a property is a parameter the parameter name is returned instead. This can be used with 'method chaining' (see the example below) to make sure a property argument is correct.
 * @returns ConstructionStages object.
 */
    ViewParameters(): ConstructionStages;

/**
 * Adds a warning for *DEFINE_CONSTRUCTION_STAGES. For more details on checking see the Check class.
 * @param message The warning message to give
 * @param details An optional detailed warning message
 */
    Warning(message: string, details?: string): void;

/**
 * Returns the cross references for this *DEFINE_CONSTRUCTION_STAGES.
 * @returns Xrefs object.
 */
    Xrefs(): Xrefs;

/**
 * Create a new ConstructionStages object.
 * @param Model Model that *DEFINE_CONSTRUCTION_STAGES will be created in
 * @param Stage_ID ConstructionStages id.
 * @param heading Title for the *DEFINE_CONSTRUCTION_STAGES
 * @returns ConstructionStages object
 */
    constructor(Model: Model, Stage_ID: number, heading?: string);

/**
 * Creates a string containing the construction stages data in keyword format. Note that this contains the keyword header and the keyword cards. See also ConstructionStages.Keyword() and ConstructionStages.KeywordCards().
 * @returns string
 */
    toString(): string;

/** Analysis time at end of stage. */
    ate: number;
/** Analysis time duration of stage. */
    atr: number;
/** Analysis time at start of stage. */
    ats: number;
/** true if *DEFINE_CONSTRUCTION_STAGES exists, false if referred to but not defined. */
    readonly exists: boolean;
/** The title of the *DEFINE_CONSTRUCTION_STAGES or the empty string if _TITLE is not set */
    heading: string;
/** The Include file number that the *DEFINE_CONSTRUCTION_STAGES is in. */
    include: number;
/** ConstructionStages number. The label is an alternative name for this. */
    istage: number;
/** Flag to set velocities to zero at start of stage. */
    ivel0: number;
/** ConstructionStages number. The istage is an alternative name for this. */
    label: number;
/** The Model number that the *DEFINE_CONSTRUCTION_STAGES is in. */
    readonly model: number;
/** Real time at end of stage. */
    rte: number;
/** Real time at start of stage. */
    rts: number;
}


/** Object returned by Interactions */
interface InteractionsReturn {
    /** End coordinate of intersection line (for Contact.CROSSED_EDGES) */
    end: number[];
    /** escape vector X component (for Contact.PENETRATIONS) */
    ex: number;
    /** escape vector Y component (for Contact.PENETRATIONS) */
    ey: number;
    /** escape vector Z component (for Contact.PENETRATIONS) */
    ez: number;
    /** Node 1 of reference segment (for Contact.PENETRATIONS) */
    n1: Node;
    /** Node 2 of reference segment (for Contact.PENETRATIONS) */
    n2: Node;
    /** Node 3 of reference segment (for Contact.PENETRATIONS) */
    n3: Node;
    /** Node 4 of reference segment (for Contact.PENETRATIONS) */
    n4: Node;
    /** Penetrating node (for Contact.PENETRATIONS) */
    node: Node;
    /** Depth of penetration (for Contact.PENETRATIONS) */
    pen: number;
    /** Remaining thickness ratio (for Contact.PENETRATIONS) */
    qthick: number;
    /** Remaining unpenetrated thickness (for Contact.PENETRATIONS) */
    rthick: number;
    /** s parametric coordinate of the tracked node projected onto the shell (for Contact.PENETRATIONS) */
    s: number;
    /** Penetrated shell (for Contact.PENETRATIONS) */
    shell: Shell;
    /** First segment if shell (for Contact.CROSSED_EDGES) */
    shell1: Shell;
    /** Second segment if shell (for Contact.CROSSED_EDGES) */
    shell2: Shell;
    /** Penetrated solid (for Contact.PENETRATIONS) */
    solid: Solid;
    /** First segment if solid (for Contact.CROSSED_EDGES) */
    solid1: Solid;
    /** Second segment if solid (for Contact.CROSSED_EDGES) */
    solid2: Solid;
    /** Start coordinate of intersection line (for Contact.CROSSED_EDGES) */
    start: number[];
    /** t parametric coordinate of the tracked node projected onto the shell (for Contact.PENETRATIONS) */
    t: number;
    /** Thickness of contact segment, i.e. 0.5*(t1+t2) (for Contact.PENETRATIONS) */
    thick: number;
    /** Penetrated thick shell (for Contact.PENETRATIONS) */
    thickshell: Tshell;
    /** First segment if thick shell (for Contact.CROSSED_EDGES) */
    thickshell1: Tshell;
    /** Second segment if thick shell (for Contact.CROSSED_EDGES) */
    thickshell2: Tshell;
    /** The interaction type. Either Contact.PENETRATIONS or Contact.CROSSED_EDGES. */
    type: number;
}

declare class Contact {
/**
 * Associates a comment with a contact.
 * @param Comment Comment that will be attached to the contact
 */
    AssociateComment(Comment: Comment): void;

/**
 * Blanks the contact
 */
    Blank(): void;

/**
 * Blanks all of the contacts in the model.
 * @param Model Model that all contacts will be blanked in
 * @param redraw If model should be redrawn or not. If omitted redraw is false. If you want to do several (un)blanks and only redraw after the last one then use false for all redraws apart from the last one. Alternatively you can redraw using View.Redraw().
 */
    static BlankAll(Model: Model, redraw?: boolean): void;

/**
 * Blanks all of the flagged contacts in the model.
 * @param Model Model that all the flagged contacts will be blanked in
 * @param flag Flag set on the contacts that you want to blank
 * @param redraw If model should be redrawn or not. If omitted redraw is false. If you want to do several (un)blanks and only redraw after the last one then use false for all redraws apart from the last one. Alternatively you can redraw using View.Redraw().
 */
    static BlankFlagged(Model: Model, flag: number, redraw?: boolean): void;

/**
 * Checks if the contact is blanked or not.
 * @returns true if blanked, false if not.
 */
    Blanked(): boolean;

/**
 * Starts an edit panel in Browse mode.
 * @param modal If this window is modal (blocks the user from doing anything else in PRIMER until this window is dismissed). If omitted the window will be modal.
 */
    Browse(modal?: boolean): void;

/**
 * Clears a flag on the contact.
 * @param flag Flag to clear on the contact
 */
    ClearFlag(flag: number): void;

/**
 * see if tied/spotweld contact uses constrained formulation
 * @param connection if true will only consider contacts used for PRIMER connections. The default is false.
 * @returns logical
 */
    Constrained(connection?: boolean): boolean;

/**
 * Copies the contact. The target include of the copied contact can be set using Options.copy_target_include.
 * @param range If you want to keep the copied item in the range specified for the current include. Default value is false. To set current include, use Include.MakeCurrentLayer().
 * @returns Contact object
 */
    Copy(range?: boolean): Contact;

/**
 * Starts an interactive editing panel to create a contact
 * @param Model Model that the contact will be created in.
 * @param modal If this window is modal (blocks the user from doing anything else in PRIMER until this window is dismissed). If omitted the window will be modal.
 * @returns Contact object (or null if not made).
 */
    static Create(Model: Model, modal?: boolean): Contact;

/**
 * Detaches a comment from a contact.
 * @param Comment Comment that will be detached from the contact
 */
    DetachComment(Comment: Comment): void;

/**
 * Starts an interactive editing panel.
 * @param modal If this window is modal (blocks the user from doing anything else in PRIMER until this window is dismissed). If omitted the window will be modal.
 */
    Edit(modal?: boolean): void;

/**
 * Adds an error for contact. For more details on checking see the Check class.
 * @param message The error message to give
 * @param details An optional detailed error message
 */
    Error(message: string, details?: string): void;

/**
 * Extracts the actual colour used for contact. By default in PRIMER many entities such as elements get their colour automatically from the part that they are in. PRIMER cycles through 13 default colours based on the label of the entity. In this case the contact colour property will return the value Colour.PART instead of the actual colour. This method will return the actual colour which is used for drawing the contact.
 * @returns colour value (integer)
 */
    ExtractColour(): number;

/**
 * Returns the first contact in the model.
 * @param Model Model to get first contact in
 * @returns Contact object (or null if there are no contacts in the model).
 */
    static First(Model: Model): Contact;

/**
 * Returns the first free contact label in the model. Also see Contact.LastFreeLabel(), Contact.NextFreeLabel() and Model.FirstFreeItemLabel().
 * @param Model Model to get first free contact label in
 * @param layer Include file (0 for the main file) to search for labels in (Equivalent to First free in layer in editing panels). If omitted the whole model will be used (Equivalent to First free in editing panels).
 * @returns Contact label.
 */
    static FirstFreeLabel(Model: Model, layer?: number): number;

/**
 * Flags all of the contacts in the model with a defined flag.
 * @param Model Model that all contacts will be flagged in
 * @param flag Flag to set on the contacts
 */
    static FlagAll(Model: Model, flag: number): void;

/**
 * Checks if the contact is flagged or not.
 * @param flag Flag to test on the contact
 * @returns true if flagged, false if not.
 */
    Flagged(flag: number): boolean;

/**
 * Calls a function for each contact in the model. Note that ForEach has been designed to make looping over contacts as fast as possible and so has some limitations. Firstly, a single temporary Contact object is created and on each function call it is updated with the current contact data. This means that you should not try to store the Contact object for later use (e.g. in an array) as it is temporary. Secondly, you cannot create new contacts inside a ForEach loop. 
 * @param Model Model that all contacts are in
 * @param func Function to call for each contact
 * @param extra An optional extra object/array/string etc that will appended to arguments when calling the function
 */
    static ForEach(Model: Model, func: () => void, extra?: any): void;

/**
 * Returns an array of Contact objects or properties for all of the contacts in a model in PRIMER. If the optional property argument is not given then an array of Contact objects is returned. If the property argument is given, that property value for each contact is returned in the array instead of a Contact object
 * @param Model Model to get contacts from
 * @param property Name for property to get for all contacts in the model
 * @returns Array of Contact objects or properties
 */
    static GetAll(Model: Model, property?: string): Contact[];

/**
 * Extracts the comments associated to a contact.
 * @returns Array of Comment objects (or null if there are no comments associated to the node).
 */
    GetComments(): Comment[];

/**
 * Returns an array of Contact objects for all of the flagged contacts in a model in PRIMER If the optional property argument is not given then an array of Contact objects is returned. If the property argument is given, then that property value for each contact is returned in the array instead of a Contact object
 * @param Model Model to get contacts from
 * @param flag Flag set on the contacts that you want to retrieve
 * @param property Name for property to get for all flagged contacts in the model
 * @returns Array of Contact objects or properties
 */
    static GetFlagged(Model: Model, flag: number, property?: string): Contact[];

/**
 * Returns the Contact object for a contact ID.
 * @param Model Model to find the contact in
 * @param number number of the contact you want the Contact object for
 * @returns Contact object (or null if contact does not exist).
 */
    static GetFromID(Model: Model, number: number): Contact;

/**
 * Checks if a Contact property is a parameter or not. Note that object properties that are parameters are normally returned as the integer or float parameter values as that is virtually always what the user would want. For this function to work the JavaScript interpreter must use the parameter name instead of the value. This can be done by setting the Options.property_parameter_names option to true before calling the function and then resetting it to false afterwards.. This behaviour can also temporarily be switched by using the Contact.ViewParameters() method and 'method chaining' (see the examples below).
 * @param prop contact property to get parameter for
 * @returns Parameter object if property is a parameter, null if not.
 */
    GetParameter(prop: string): Parameter;

/**
 * Returns an array of objects describing the interactions which can either be penetrations (tracked nodes that are tied to or penetrate elements in the contact) or crossed edges (contact segments that cross).
 * @param type What type of interactions to return. Can be bitwise code of Contact.PENETRATIONS to return penetrations and Contact.CROSSED_EDGES to return crossed edges. If omitted penetrations will be returned.
 */
    Interactions(type?: number): InteractionsReturn[];

/**
 * Returns the keyword for this Contact (*BOUNDARY_PRESCRIBED_MOTION_xxxx). Note that a carriage return is not added. See also Contact.KeywordCards()
 * @returns string containing the keyword.
 */
    Keyword(): string;

/**
 * Returns the keyword cards for the Contact. Note that a carriage return is not added. See also Contact.Keyword()
 * @returns string containing the cards.
 */
    KeywordCards(): string;

/**
 * Returns the last contact in the model.
 * @param Model Model to get last contact in
 * @returns Contact object (or null if there are no contacts in the model).
 */
    static Last(Model: Model): Contact;

/**
 * Returns the last free contact label in the model. Also see Contact.FirstFreeLabel(), Contact.NextFreeLabel() and see Model.LastFreeItemLabel()
 * @param Model Model to get last free contact label in
 * @param layer Include file (0 for the main file) to search for labels in (Equivalent to Highest free in layer in editing panels). If omitted the whole model will be used.
 * @returns Contact label.
 */
    static LastFreeLabel(Model: Model, layer?: number): number;

/**
 * Returns the next contact in the model.
 * @returns Contact object (or null if there are no more contacts in the model).
 */
    Next(): Contact;

/**
 * Returns the next free (highest+1) contact label in the model. Also see Contact.FirstFreeLabel(), Contact.LastFreeLabel() and Model.NextFreeItemLabel()
 * @param Model Model to get next free contact label in
 * @param layer Include file (0 for the main file) to search for labels in (Equivalent to Highest+1 in layer in editing panels). If omitted the whole model will be used (Equivalent to Highest+1 in editing panels).
 * @returns Contact label.
 */
    static NextFreeLabel(Model: Model, layer?: number): number;

/**
 * Flags nodes that penetrate (or tie) in contact
 * @param flag Flag to be set on penetrating (or tied) node.
 * @param eflag Optional flag for elements. If supplied, node will be flagged only if it penetrates (or ties to) an element that is flagged. Node and element flag may be the same.
 * @returns zero if contact successfully checked
 */
    PenCheck(flag: number, eflag: number): number;

/**
 * launches the interactive edit panel for penetration check on the con
 * @param modal If this window is modal (blocks the user from doing anything else in PRIMER until this window is dismissed). If omitted the window will be modal.
 * @param check_mode Check mode. Can be Model.MPP_MODE or Model.SMP_MODE. Default is set to the oa pref contact_check_mode
 * @param mpp_threshold Can set the MPP threshold, by default this is set to the oa pref contact_mpp_penetration_threshold
 * @param report_crossed_3d_elems Can set the value of reporting crossed elements to TRUE or FALSE, by default this is set to the oa pref report_crossed_3d_elems
 * @param contact_penchk_dup_shells Duplicate shell treatment Can be Model.SHELL_AUTO, Model.SHELL_THICK or Model.SHELL_THIN. Default is set to the oa pref contact_penchk_dup_shells
 */
    PenCheckEdit(modal?: boolean, check_mode?: number, mpp_threshold?: number, report_crossed_3d_elems?: boolean, contact_penchk_dup_shells?: number): void;

/**
 * Allows the user to pick a contact.
 * @param prompt Text to display as a prompt to the user
 * @param limit If the argument is a Model then only contacts from that model can be picked. If the argument is a Flag then only contacts that are flagged with limit can be selected. If omitted, or null, any contacts from any model can be selected. from any model.
 * @param modal If picking is modal (blocks the user from doing anything else in PRIMER until this window is dismissed). If omitted the pick will be modal.
 * @param button_text By default the window with the prompt will have a button labelled 'Cancel' which if pressed will cancel the pick and return null. If you want to change the text on the button use this argument. If omitted 'Cancel' will be used.
 * @returns Contact object (or null if not picked)
 */
    static Pick(prompt: string, limit?: Model | number, modal?: boolean, button_text?: string): Contact;

/**
 * Returns the previous contact in the model.
 * @returns Contact object (or null if there are no more contacts in the model).
 */
    Previous(): Contact;

/**
 * Renumbers all of the contacts in the model.
 * @param Model Model that all contacts will be renumbered in
 * @param start Start point for renumbering
 */
    static RenumberAll(Model: Model, start: number): void;

/**
 * Renumbers all of the flagged contacts in the model.
 * @param Model Model that all the flagged contacts will be renumbered in
 * @param flag Flag set on the contacts that you want to renumber
 * @param start Start point for renumbering
 */
    static RenumberFlagged(Model: Model, flag: number, start: number): void;

/**
 * Allows the user to select contacts using standard PRIMER object menus.
 * @param flag Flag to use when selecting contacts
 * @param prompt Text to display as a prompt to the user
 * @param limit If the argument is a Model then only contacts from that model can be selected. If the argument is a Flag then only contacts that are flagged with limit can be selected (limit should be different to flag). If omitted, or null, any contacts can be selected. from any model.
 * @param modal If selection is modal (blocks the user from doing anything else in PRIMER until this window is dismissed). If omitted the selection will be modal.
 * @returns Number of contacts selected or null if menu cancelled
 */
    static Select(flag: number, prompt: string, limit?: Model | number, modal?: boolean): number;

/**
 * Sets a flag on the contact.
 * @param flag Flag to set on the contact
 */
    SetFlag(flag: number): void;

/**
 * Sketches the contact. The contact will be sketched until you either call Contact.Unsketch(), Contact.UnsketchAll(), Model.UnsketchAll(), or delete the model
 * @param redraw If model should be redrawn or not after the contact is sketched. If omitted redraw is true. If you want to sketch several contacts and only redraw after the last one then use false for redraw and call View.Redraw().
 */
    Sketch(redraw?: boolean): void;

/**
 * Sketches all of the flagged contacts in the model. The contacts will be sketched until you either call Contact.Unsketch(), Contact.UnsketchFlagged(), Model.UnsketchAll(), or delete the model
 * @param Model Model that all the flagged contacts will be sketched in
 * @param flag Flag set on the contacts that you want to sketch
 * @param redraw If model should be redrawn or not after the contacts are sketched. If omitted redraw is true. If you want to sketch flagged contacts several times and only redraw after the last one then use false for redraw and call View.Redraw().
 */
    static SketchFlagged(Model: Model, flag: number, redraw?: boolean): void;

/**
 * Checks sliding contact for crossed edges and penetrations
 * @returns An array containing count of crossed edges, count of penetrations (note if a node penetrates more than one segment, it is only reported once here)
 */
    StatusCheck(): number[];

/**
 * Returns the total number of contacts in the model.
 * @param Model Model to get total for
 * @param exists true if only existing contacts should be counted. If false or omitted referenced but undefined contacts will also be included in the total.
 * @returns number of contacts
 */
    static Total(Model: Model, exists?: boolean): number;

/**
 * Unblanks the contact
 */
    Unblank(): void;

/**
 * Unblanks all of the contacts in the model.
 * @param Model Model that all contacts will be unblanked in
 * @param redraw If model should be redrawn or not. If omitted redraw is false. If you want to do several (un)blanks and only redraw after the last one then use false for all redraws apart from the last one. Alternatively you can redraw using View.Redraw().
 */
    static UnblankAll(Model: Model, redraw?: boolean): void;

/**
 * Unblanks all of the flagged contacts in the model.
 * @param Model Model that the flagged contacts will be unblanked in
 * @param flag Flag set on the contacts that you want to unblank
 * @param redraw If model should be redrawn or not. If omitted redraw is false. If you want to do several (un)blanks and only redraw after the last one then use false for all redraws apart from the last one. Alternatively you can redraw using View.Redraw().
 */
    static UnblankFlagged(Model: Model, flag: number, redraw?: boolean): void;

/**
 * Unsets a defined flag on all of the contacts in the model.
 * @param Model Model that the defined flag for all contacts will be unset in
 * @param flag Flag to unset on the contacts
 */
    static UnflagAll(Model: Model, flag: number): void;

/**
 * Unsketches the contact.
 * @param redraw If model should be redrawn or not after the contact is unsketched. If omitted redraw is true. If you want to unsketch several contacts and only redraw after the last one then use false for redraw and call View.Redraw().
 */
    Unsketch(redraw?: boolean): void;

/**
 * Unsketches all contacts.
 * @param Model Model that all contacts will be unblanked in
 * @param redraw If model should be redrawn or not after the contacts are unsketched. If omitted redraw is true. If you want to unsketch several things and only redraw after the last one then use false for redraw and call View.Redraw().
 */
    static UnsketchAll(Model: Model, redraw?: boolean): void;

/**
 * Unsketches all flagged contacts in the model.
 * @param Model Model that all contacts will be unsketched in
 * @param flag Flag set on the contacts that you want to unsketch
 * @param redraw If model should be redrawn or not after the contacts are unsketched. If omitted redraw is true. If you want to unsketch several things and only redraw after the last one then use false for redraw and call View.Redraw().
 */
    static UnsketchFlagged(Model: Model, flag: number, redraw?: boolean): void;

/**
 * Object properties that are parameters are normally returned as the integer or float parameter values as that is virtually always what the user would want. This function temporarily changes the behaviour so that if a property is a parameter the parameter name is returned instead. This can be used with 'method chaining' (see the example below) to make sure a property argument is correct.
 * @returns Contact object.
 */
    ViewParameters(): Contact;

/**
 * Adds a warning for contact. For more details on checking see the Check class.
 * @param message The warning message to give
 * @param details An optional detailed warning message
 */
    Warning(message: string, details?: string): void;

/**
 * Returns the cross references for this contact.
 * @returns Xrefs object.
 */
    Xrefs(): Xrefs;

/**
 * Create a new Contact object.
 * @param Model Model that Contact will be created in
 * @param type Type of contact
 * @param id Contact number
 * @param heading Title for the Contact
 * @returns Contact object
 */
    constructor(Model: Model, type: string, id?: number, heading?: string);

/**
 * Creates a string containing the Contact data in keyword format. Note that this contains the keyword header and the keyword cards. See also Contact.Keyword() and Contact.KeywordCards().
 * @returns string
 */
    toString(): string;

/** Loadcurve ID for a */
    a: number;
/** contact algorithm */
    algo: number;
/** Key amplitude parameter A */
    alpha: number;
/** Loadcurve ID for b */
    b: number;
/** boundary condition flag */
    bc_flg: number;
/** Key amplitude parameter B */
    beta: number;
/** Loadcurve for #cycles between bucket sorts */
    bsort: number;
/** Contact birth time */
    bt: number;
/** Bucket sorting frequency */
    bucket: number;
/** Loadcurve ID for c */
    c: number;
/** Checking mode on the pen check edit panel. (Can be Contact.MPP_METHOD, Contact.MPP_METHOD or Contact.SMP_METHOD) */
    check_mode: number;
/** Special check for inverted elements */
    chksegs: number;
/** Contact number (identical to label. */
    cid: number;
/** Coordinate system ID to output rcforc force resultants and ncforc data in a local system */
    cid_rcf: number;
/** Curve ID for the viscosity */
    cideta: number;
/** Curve ID for the coefficient of friction */
    cidmu: number;
/** Surface closeness parameter */
    close: number;
/** Normal stiffness */
    cn: number;
/** Normal stiffness */
    cn_1: number;
/** The colour of the contact */
    colour: Colour;
/** Shell treatment on the pen check edit panel. (Can be Contact.SHELL_AUTO, Contact.SHELL_AUTO or Contact.SHELL_THIN or Contact.SHELL_THICK) */
    contact_penchk_dup_shells: number;
/** Exclude beam to beam contact flag */
    cparm8: number;
/** Spotweld beam flag for SMP */
    cparm8smp: number;
/** Ratio of tangential stiffness to normal stiffness */
    ct2cn: number;
/** Ratio of tangential to normal stiff */
    ct2cn_1: number;
/** Loadcurve ID for d */
    d: number;
/** Composite film thickness */
    d_comp: number;
/** Draw bead depth */
    dbdth: number;
/** 2dbinr - Flag to include 2d belt elements in contact (note properties cannot start with a number, so 2 has been removed). */
    dbinr: number;
/** optional Part ID */
    dbpid: number;
/** Exponential decay coeff */
    dc: number;
/** Loadcurve for search depth in automatic contact */
    depth: number;
/** Scale factor on lcidrf */
    dfscl: number;
/** Distance for nonlinear force scaling */
    dnlscl: number;
/** Depth of penetration reduction factor */
    dprfac: number;
/** Contact death time */
    dt: number;
/** Time interval between penetration reports */
    dtpchk: number;
/** Timestep used in stiffness calc */
    dtstif: number;
/** Scale factor for penalty stiffness of edge-to-edge contact */
    edgek: number;
/** optional element id offset */
    eloff: number;
/** Parameter to define the length of the bead */
    ending: number;
/** Maximum strain the blank will experience when it passes the bead */
    epm: number;
/** Scale factor to weaken the stress-strain curve */
    epscale: number;
/** Normal energy release rate used in damage calculation */
    eraten: number;
/** Shear energy release rate used in damage calculation */
    erates: number;
/** Erosion/interior node option */
    erosop: number;
/** true if contact exists, false if referred to but not defined. */
    readonly exists: boolean;
/** Force calculation method */
    fcm: number;
/** Dynamic coeff of friction */
    fd: number;
/** Angle tolerance in radians for feature lines option in smooth contact */
    flangl: number;
/** Scale factor for nonlinear force scaling */
    fnlscl: number;
/** formula id */
    formula: number;
/** Radiation conductance across gap */
    frad: number;
/** #cycles between penalty force updates */
    frcfrq: number;
/** Scale factor for frictional stiffness */
    fricsf: number;
/** Static coeff of friction */
    fs: number;
/** Coulomb friction scale factor */
    fsf: number;
/** Tolerance for determining flat segments. */
    fstol: number;
/** Beam torsional force computation flag */
    ftorq: number;
/** Fraction of sliding friction energy partitioned to surface A */
    ftosa: number;
/** Experimental contact algorithm */
    grpable: number;
/** Heat transfer coefficient */
    h0: number;
/** Thermal contact conductivity */
    hclose: number;
/** Contact heading */
    heading: string;
/** Segment searching option */
    i2d3d: number;
/** Adjacent matl treatment for solids */
    iadj: number;
/** coefficient of restitution expressed as a percentage */
    icor: number;
/** true if _ID option is set, false if not */
    id: boolean;
/** Implicit convergence flag */
    igap: number;
/** Ignore initial pens in automatic types */
    ignore: number;
/** Flag to ignore the thickness offset for shells in the calculation of the shell contact penetration depth. */
    ignroff: number;
/** The Include file number that the contact is in. */
    include: number;
/** Number of iterations for initial penetration checking */
    inititer: number;
/** Create backup penalty tied contact */
    ipback: number;
/** Symmetry plane option */
    isym: number;
/** Symmetry plane option */
    isym_1: number;
/** Conductivity of gap fluid */
    k: number;
/** Kinematic partition factor */
    kpf: number;
/** Contact number. */
    label: number;
/** Bucket sorting frequency loadcurve ID */
    lcbucket: number;
/** Loadcurve ID for plastic strain vs. parametric coord */
    lceps: number;
/** Loadcurve ID for plastic strain vs. parametric coord (elements moved &gt; offset) */
    lceps2: number;
/** Loadcurve ID for dynamic friction vs. temp */
    lcfdt: number;
/** Loadcurve ID for static friction vs. temp */
    lcfst: number;
/** Loadcurve ID for lch */
    lch: number;
/** Loadcurve ID for Force vs penetration curve */
    lcid: number;
/** Loadcurve ID for Dyn rel stiffness */
    lcid1: number;
/** Loadcurve ID for Transient stiffness */
    lcid2: number;
/** Loadcurve ID for airbag thickness vs time */
    lcidab: number;
/** Loadcurve ID for Normal force per unit length */
    lcidnf: number;
/** Loadcurve ID for Force due to bending per unit length */
    lcidrf: number;
/** Max size for thermal contact */
    lmax: number;
/** Critical gap size */
    lmin: number;
/** Max parametric coord overlap */
    maxpar: number;
/** Shear force exponent */
    mes: number;
/** The Model number that the contact is in. */
    readonly model: number;
/** _MORTAR keyword option - true if set, false if not. */
    mortar: boolean;
/** true if _MPP option is set, false if not */
    mpp: boolean;
/** The method option for the gap function */
    mtcj: number;
/** Normal force exponent */
    nen: number;
/** Normal failure stress */
    nfls: number;
/** Number of history variables */
    nhv: number;
/** Number of material history variables */
    nmhis: number;
/** Number of surface B tied weld history variables */
    nmtwh: number;
/** Number of segments to track per surface A node */
    ns2track: number;
/** Number of surface A tied weld history variables */
    nstwh: number;
/** Number of user tied weld parameters */
    ntprm: number;
/** #int points along drawbead */
    numint: number;
/** Flag for offset treatment. This is only valid for *CONTACT_AUTOMATIC(_ONE_WAY)_SURFACE_TO_SURFACE_TIEBREAK_USER and should not be confused with the 'offset' property for other contact types. */
    offset: number;
/** distance offset */
    offset_1: number;
/** _OFFSET option. (Can be Contact.NO_OFFSET, Contact.SIMPLE_OFFSET, Contact.CONSTR_OFFSET or Contact.BEAM_OFFSET) */
    offset_flag: number;
/** Response option */
    option: number;
/** User tiebreak type */
    option_1: number;
/** Critical distance */
    param: number;
/** The parametric extension distance for contact segments */
    parmax: number;
/** Penetration search flag */
    penchk: number;
/** Max pen distance for "old" types 3, 5, 10 */
    penmax: number;
/** Ignore penetration scale factor */
    pensf: number;
/** Flag to choose the method for calculating the penalty stiffness */
    pstiff: number;
/** Split quads into 2 trias */
    q2tri: number;
/** Region to limit contact volume */
    region: number;
/** Surface A box id */
    saboxid: number;
/** Surface A side printout flag */
    sapr: number;
/** Optional surface A side shell thickness */
    sast: number;
/** Surface B box id */
    sbboxid: number;
/** segment based contact option */
    sbopt: number;
/** Surface B side printout flag */
    sbpr: number;
/** Optional surface B side shell thickness */
    sbst: number;
/** Shear failure stress */
    sfls: number;
/** Scale factor for neighbour segment contact */
    sfnbr: number;
/** Scale factor on surface A penalty stiffness */
    sfsa: number;
/** Scale factor on true surface A shell thickness */
    sfsat: number;
/** Scale factor on surface B penalty stiffness */
    sfsb: number;
/** Scale factor on true surface B shell thickness */
    sfsbt: number;
/** Shared constraint flag */
    sharec: number;
/** Edge shape for shells when measuring penetration */
    shledg: number;
/** Flag affecting the location of the contact surfaces. */
    shloff: number;
/** Thickness consideration flag */
    shlthk: number;
/** Optional solid stiffness */
    sldstf: number;
/** Optional solid thickness */
    sldthk: number;
/** Shooting node logic flag */
    snlog: number;
/** Soft constraint scale factor */
    sofscl: number;
/** Soft constraint flag */
    soft: number;
/** Model for shear response */
    srmodel: number;
/** Flag for non-extended exterior shell edges */
    srnde: number;
/** Flag to determine how the SSF option on *PART_CONTACT behaves when SOFT = 2 on optional card A */
    ssftyp: number;
/** Surface A set id */
    surfa: number;
/** Surface A set type */
    surfatyp: number;
/** Surface B set id */
    surfb: number;
/** Surface B set type */
    surfbtyp: number;
/** Flag to use tapered shell contact segments adjacent to segments that are thinned by the SPOTHIN option on *CONTROL_CONTACT */
    swtpr: number;
/** Loadcurve ID for stress vs gap post failure */
    tblcid: number;
/** Segment treatment only flag */
    tcso: number;
/** Minimum temperature required. */
    temp: number;
/** Scale factor for the computed volume of tetrahedral solid elements for the mass calculation in SOFT = 2 contact.. */
    tetfac: number;
/** Tensile traction for failure */
    tfail: number;
/** If _THERMAL option is set. Can be true or false */
    thermal: boolean;
/** flag for thickness offset */
    thkoff: number;
/** Thickness option for "old" types 3, 5, 10 */
    thkopt: number;
/** Incremental displacement update for tied contacts */
    tiedid: number;
/** Minimum time required */
    time: number;
/**  */
    tscale: number;
/** X component of the free sliding direction T */
    tsvx: number;
/** Y component of the free sliding direction T */
    tsvy: number;
/** Z component of the free sliding direction T */
    tsvz: number;
/** Contact type ("AUTOMATIC_GENERAL", "SINGLE_SURFACE" etc). */
    type: string;
/** User parameter */
    up1: number;
/** User parameter */
    up10: number;
/** User parameter */
    up11: number;
/** User parameter */
    up12: number;
/** User parameter */
    up13: number;
/** User parameter */
    up14: number;
/** User parameter */
    up15: number;
/** User parameter */
    up16: number;
/** User parameter */
    up2: number;
/** User parameter */
    up3: number;
/** User parameter */
    up4: number;
/** User parameter */
    up5: number;
/** User parameter */
    up6: number;
/** User parameter */
    up7: number;
/** User parameter */
    up8: number;
/** User parameter */
    up9: number;
/** Optional unloading stiffness */
    us: number;
/** Coeff for viscous friction */
    vc: number;
/** Visous damping coefficient */
    vdc: number;
/** Viscous friction scale factor */
    vsf: number;
/** Adds _BEAM_OFFSET option */
    static BEAM_OFFSET: number;
/** Adds _CONSTRAINED_OFFSET option */
    static CONSTR_OFFSET: number;
/** Return crossed edges in Contact.Interactions() */
    static CROSSED_EDGES: number;
/** Launches the penetration edit panel with the MPP methodology turned on */
    static MPP_METHOD: number;
/** MPP penetration check mode */
    static MPP_MODE: number;
/** No offset option added. */
    static NO_OFFSET: number;
/** Return penetrations in Contact.Interactions() */
    static PENETRATIONS: number;
/** Launches the penetration edit panel with Automatic shell treatment of duplicate shells. */
    static SHELL_AUTO: number;
/** Launches the penetration edit panel with the thickest always option for duplicate shells. */
    static SHELL_THICK: number;
/** Launches the penetration edit panel with the thinnest always option for duplicate shells. */
    static SHELL_THIN: number;
/** Adds _OFFSET option */
    static SIMPLE_OFFSET: number;
/** Launches the penetration edit panel with the SMP methodology turned on */
    static SMP_METHOD: number;
/** SMP penetration check mode */
    static SMP_MODE: number;
}

declare class ContactGuidedCable {
/**
 * Associates a comment with a contact guided_cable.
 * @param Comment Comment that will be attached to the contact guided_cable
 */
    AssociateComment(Comment: Comment): void;

/**
 * Blanks the contact guided_cable
 */
    Blank(): void;

/**
 * Blanks all of the contact guided_cables in the model.
 * @param Model Model that all contact guided_cables will be blanked in
 * @param redraw If model should be redrawn or not. If omitted redraw is false. If you want to do several (un)blanks and only redraw after the last one then use false for all redraws apart from the last one. Alternatively you can redraw using View.Redraw().
 */
    static BlankAll(Model: Model, redraw?: boolean): void;

/**
 * Blanks all of the flagged contact guided_cables in the model.
 * @param Model Model that all the flagged contact guided_cables will be blanked in
 * @param flag Flag set on the contact guided_cables that you want to blank
 * @param redraw If model should be redrawn or not. If omitted redraw is false. If you want to do several (un)blanks and only redraw after the last one then use false for all redraws apart from the last one. Alternatively you can redraw using View.Redraw().
 */
    static BlankFlagged(Model: Model, flag: number, redraw?: boolean): void;

/**
 * Checks if the contact guided_cable is blanked or not.
 * @returns true if blanked, false if not.
 */
    Blanked(): boolean;

/**
 * Clears a flag on the contact guided_cable.
 * @param flag Flag to clear on the contact guided_cable
 */
    ClearFlag(flag: number): void;

/**
 * Copies the contact guided_cable. The target include of the copied contact guided_cable can be set using Options.copy_target_include.
 * @param range If you want to keep the copied item in the range specified for the current include. Default value is false. To set current include, use Include.MakeCurrentLayer().
 * @returns ContactGuidedCable object
 */
    Copy(range?: boolean): ContactGuidedCable;

/**
 * Detaches a comment from a contact guided_cable.
 * @param Comment Comment that will be detached from the contact guided_cable
 */
    DetachComment(Comment: Comment): void;

/**
 * Adds an error for contact guided_cable. For more details on checking see the Check class.
 * @param message The error message to give
 * @param details An optional detailed error message
 */
    Error(message: string, details?: string): void;

/**
 * Returns the first contact guided_cable in the model.
 * @param Model Model to get first contact guided_cable in
 * @returns ContactGuidedCable object (or null if there are no contact guided_cables in the model).
 */
    static First(Model: Model): ContactGuidedCable;

/**
 * Returns the first free contact guided_cable label in the model. Also see ContactGuidedCable.LastFreeLabel(), ContactGuidedCable.NextFreeLabel() and Model.FirstFreeItemLabel().
 * @param Model Model to get first free contact guided_cable label in
 * @param layer Include file (0 for the main file) to search for labels in (Equivalent to First free in layer in editing panels). If omitted the whole model will be used (Equivalent to First free in editing panels).
 * @returns ContactGuidedCable label.
 */
    static FirstFreeLabel(Model: Model, layer?: number): number;

/**
 * Flags all of the contact guided_cables in the model with a defined flag.
 * @param Model Model that all contact guided_cables will be flagged in
 * @param flag Flag to set on the contact guided_cables
 */
    static FlagAll(Model: Model, flag: number): void;

/**
 * Checks if the contact guided_cable is flagged or not.
 * @param flag Flag to test on the contact guided_cable
 * @returns true if flagged, false if not.
 */
    Flagged(flag: number): boolean;

/**
 * Calls a function for each contact guided_cable in the model. Note that ForEach has been designed to make looping over contact guided_cables as fast as possible and so has some limitations. Firstly, a single temporary ContactGuidedCable object is created and on each function call it is updated with the current contact guided_cable data. This means that you should not try to store the ContactGuidedCable object for later use (e.g. in an array) as it is temporary. Secondly, you cannot create new contact guided_cables inside a ForEach loop. 
 * @param Model Model that all contact guided_cables are in
 * @param func Function to call for each contact guided_cable
 * @param extra An optional extra object/array/string etc that will appended to arguments when calling the function
 */
    static ForEach(Model: Model, func: () => void, extra?: any): void;

/**
 * Returns an array of ContactGuidedCable objects or properties for all of the contact guided_cables in a model in PRIMER. If the optional property argument is not given then an array of ContactGuidedCable objects is returned. If the property argument is given, that property value for each contact guided_cable is returned in the array instead of a ContactGuidedCable object
 * @param Model Model to get contact guided_cables from
 * @param property Name for property to get for all contact guided_cables in the model
 * @returns Array of ContactGuidedCable objects or properties
 */
    static GetAll(Model: Model, property?: string): ContactGuidedCable[];

/**
 * Extracts the comments associated to a contact guided_cable.
 * @returns Array of Comment objects (or null if there are no comments associated to the node).
 */
    GetComments(): Comment[];

/**
 * Returns an array of ContactGuidedCable objects for all of the flagged contact guided_cables in a model in PRIMER If the optional property argument is not given then an array of ContactGuidedCable objects is returned. If the property argument is given, then that property value for each contact guided_cable is returned in the array instead of a ContactGuidedCable object
 * @param Model Model to get contact guided_cables from
 * @param flag Flag set on the contact guided_cables that you want to retrieve
 * @param property Name for property to get for all flagged contact guided_cables in the model
 * @returns Array of ContactGuidedCable objects or properties
 */
    static GetFlagged(Model: Model, flag: number, property?: string): ContactGuidedCable[];

/**
 * Returns the ContactGuidedCable object for a contact guided_cable ID.
 * @param Model Model to find the contact guided_cable in
 * @param number number of the contact guided_cable you want the ContactGuidedCable object for
 * @returns ContactGuidedCable object (or null if contact guided_cable does not exist).
 */
    static GetFromID(Model: Model, number: number): ContactGuidedCable;

/**
 * Checks if a ContactGuidedCable property is a parameter or not. Note that object properties that are parameters are normally returned as the integer or float parameter values as that is virtually always what the user would want. For this function to work the JavaScript interpreter must use the parameter name instead of the value. This can be done by setting the Options.property_parameter_names option to true before calling the function and then resetting it to false afterwards.. This behaviour can also temporarily be switched by using the ContactGuidedCable.ViewParameters() method and 'method chaining' (see the examples below).
 * @param prop contact guided_cable property to get parameter for
 * @returns Parameter object if property is a parameter, null if not.
 */
    GetParameter(prop: string): Parameter;

/**
 * Returns the keyword for this ContactGuidedCable (*contact_guided_cable). Note that a carriage return is not added. See also ContactGuidedCable.KeywordCards()
 * @returns string containing the keyword.
 */
    Keyword(): string;

/**
 * Returns the keyword cards for the ContactGuidedCable. Note that a carriage return is not added. See also ContactGuidedCable.Keyword()
 * @returns string containing the cards.
 */
    KeywordCards(): string;

/**
 * Returns the last contact guided_cable in the model.
 * @param Model Model to get last contact guided_cable in
 * @returns ContactGuidedCable object (or null if there are no contact guided_cables in the model).
 */
    static Last(Model: Model): ContactGuidedCable;

/**
 * Returns the last free contact guided_cable label in the model. Also see ContactGuidedCable.FirstFreeLabel(), ContactGuidedCable.NextFreeLabel() and see Model.LastFreeItemLabel()
 * @param Model Model to get last free contact guided_cable label in
 * @param layer Include file (0 for the main file) to search for labels in (Equivalent to Highest free in layer in editing panels). If omitted the whole model will be used.
 * @returns ContactGuidedCable label.
 */
    static LastFreeLabel(Model: Model, layer?: number): number;

/**
 * Returns the next contact guided_cable in the model.
 * @returns ContactGuidedCable object (or null if there are no more contact guided_cables in the model).
 */
    Next(): ContactGuidedCable;

/**
 * Returns the next free (highest+1) contact guided_cable label in the model. Also see ContactGuidedCable.FirstFreeLabel(), ContactGuidedCable.LastFreeLabel() and Model.NextFreeItemLabel()
 * @param Model Model to get next free contact guided_cable label in
 * @param layer Include file (0 for the main file) to search for labels in (Equivalent to Highest+1 in layer in editing panels). If omitted the whole model will be used (Equivalent to Highest+1 in editing panels).
 * @returns ContactGuidedCable label.
 */
    static NextFreeLabel(Model: Model, layer?: number): number;

/**
 * Allows the user to pick a contact guided_cable.
 * @param prompt Text to display as a prompt to the user
 * @param limit If the argument is a Model then only contact guided_cables from that model can be picked. If the argument is a Flag then only contact guided_cables that are flagged with limit can be selected. If omitted, or null, any contact guided_cables from any model can be selected. from any model.
 * @param modal If picking is modal (blocks the user from doing anything else in PRIMER until this window is dismissed). If omitted the pick will be modal.
 * @param button_text By default the window with the prompt will have a button labelled 'Cancel' which if pressed will cancel the pick and return null. If you want to change the text on the button use this argument. If omitted 'Cancel' will be used.
 * @returns ContactGuidedCable object (or null if not picked)
 */
    static Pick(prompt: string, limit?: Model | number, modal?: boolean, button_text?: string): ContactGuidedCable;

/**
 * Returns the previous contact guided_cable in the model.
 * @returns ContactGuidedCable object (or null if there are no more contact guided_cables in the model).
 */
    Previous(): ContactGuidedCable;

/**
 * Renumbers all of the contact guided_cables in the model.
 * @param Model Model that all contact guided_cables will be renumbered in
 * @param start Start point for renumbering
 */
    static RenumberAll(Model: Model, start: number): void;

/**
 * Renumbers all of the flagged contact guided_cables in the model.
 * @param Model Model that all the flagged contact guided_cables will be renumbered in
 * @param flag Flag set on the contact guided_cables that you want to renumber
 * @param start Start point for renumbering
 */
    static RenumberFlagged(Model: Model, flag: number, start: number): void;

/**
 * Allows the user to select contact guided_cables using standard PRIMER object menus.
 * @param flag Flag to use when selecting contact guided_cables
 * @param prompt Text to display as a prompt to the user
 * @param limit If the argument is a Model then only contact guided_cables from that model can be selected. If the argument is a Flag then only contact guided_cables that are flagged with limit can be selected (limit should be different to flag). If omitted, or null, any contact guided_cables can be selected. from any model.
 * @param modal If selection is modal (blocks the user from doing anything else in PRIMER until this window is dismissed). If omitted the selection will be modal.
 * @returns Number of contact guided_cables selected or null if menu cancelled
 */
    static Select(flag: number, prompt: string, limit?: Model | number, modal?: boolean): number;

/**
 * Sets a flag on the contact guided_cable.
 * @param flag Flag to set on the contact guided_cable
 */
    SetFlag(flag: number): void;

/**
 * Sketches the contact guided_cable. The contact guided_cable will be sketched until you either call ContactGuidedCable.Unsketch(), ContactGuidedCable.UnsketchAll(), Model.UnsketchAll(), or delete the model
 * @param redraw If model should be redrawn or not after the contact guided_cable is sketched. If omitted redraw is true. If you want to sketch several contact guided_cables and only redraw after the last one then use false for redraw and call View.Redraw().
 */
    Sketch(redraw?: boolean): void;

/**
 * Sketches all of the flagged contact guided_cables in the model. The contact guided_cables will be sketched until you either call ContactGuidedCable.Unsketch(), ContactGuidedCable.UnsketchFlagged(), Model.UnsketchAll(), or delete the model
 * @param Model Model that all the flagged contact guided_cables will be sketched in
 * @param flag Flag set on the contact guided_cables that you want to sketch
 * @param redraw If model should be redrawn or not after the contact guided_cables are sketched. If omitted redraw is true. If you want to sketch flagged contact guided_cables several times and only redraw after the last one then use false for redraw and call View.Redraw().
 */
    static SketchFlagged(Model: Model, flag: number, redraw?: boolean): void;

/**
 * Returns the total number of contact guided_cables in the model.
 * @param Model Model to get total for
 * @param exists true if only existing contact guided_cables should be counted. If false or omitted referenced but undefined contact guided_cables will also be included in the total.
 * @returns number of contact guided_cables
 */
    static Total(Model: Model, exists?: boolean): number;

/**
 * Unblanks the contact guided_cable
 */
    Unblank(): void;

/**
 * Unblanks all of the contact guided_cables in the model.
 * @param Model Model that all contact guided_cables will be unblanked in
 * @param redraw If model should be redrawn or not. If omitted redraw is false. If you want to do several (un)blanks and only redraw after the last one then use false for all redraws apart from the last one. Alternatively you can redraw using View.Redraw().
 */
    static UnblankAll(Model: Model, redraw?: boolean): void;

/**
 * Unblanks all of the flagged contact guided_cables in the model.
 * @param Model Model that the flagged contact guided_cables will be unblanked in
 * @param flag Flag set on the contact guided_cables that you want to unblank
 * @param redraw If model should be redrawn or not. If omitted redraw is false. If you want to do several (un)blanks and only redraw after the last one then use false for all redraws apart from the last one. Alternatively you can redraw using View.Redraw().
 */
    static UnblankFlagged(Model: Model, flag: number, redraw?: boolean): void;

/**
 * Unsets a defined flag on all of the contact guided_cables in the model.
 * @param Model Model that the defined flag for all contact guided_cables will be unset in
 * @param flag Flag to unset on the contact guided_cables
 */
    static UnflagAll(Model: Model, flag: number): void;

/**
 * Unsketches the contact guided_cable.
 * @param redraw If model should be redrawn or not after the contact guided_cable is unsketched. If omitted redraw is true. If you want to unsketch several contact guided_cables and only redraw after the last one then use false for redraw and call View.Redraw().
 */
    Unsketch(redraw?: boolean): void;

/**
 * Unsketches all contact guided_cables.
 * @param Model Model that all contact guided_cables will be unblanked in
 * @param redraw If model should be redrawn or not after the contact guided_cables are unsketched. If omitted redraw is true. If you want to unsketch several things and only redraw after the last one then use false for redraw and call View.Redraw().
 */
    static UnsketchAll(Model: Model, redraw?: boolean): void;

/**
 * Unsketches all flagged contact guided_cables in the model.
 * @param Model Model that all contact guided_cables will be unsketched in
 * @param flag Flag set on the contact guided_cables that you want to unsketch
 * @param redraw If model should be redrawn or not after the contact guided_cables are unsketched. If omitted redraw is true. If you want to unsketch several things and only redraw after the last one then use false for redraw and call View.Redraw().
 */
    static UnsketchFlagged(Model: Model, flag: number, redraw?: boolean): void;

/**
 * Object properties that are parameters are normally returned as the integer or float parameter values as that is virtually always what the user would want. This function temporarily changes the behaviour so that if a property is a parameter the parameter name is returned instead. This can be used with 'method chaining' (see the example below) to make sure a property argument is correct.
 * @returns ContactGuidedCable object.
 */
    ViewParameters(): ContactGuidedCable;

/**
 * Adds a warning for contact guided_cable. For more details on checking see the Check class.
 * @param message The warning message to give
 * @param details An optional detailed warning message
 */
    Warning(message: string, details?: string): void;

/**
 * Returns the cross references for this contact guided_cable.
 * @returns Xrefs object.
 */
    Xrefs(): Xrefs;

/**
 * Create a new ContactGuidedCable object.
 * @param Model Model that ContactGuidedCable will be created in
 * @param ptype Specify the type of ContactGuidedCable (Can be ContactGuidedCable.PART or ContactGuidedCable.SET_PART
 * @param nsid Node Set ID that guides the 1D elements.
 * @param pid Part ID or Part Set ID
 * @param soft Flag for soft constraint option. Set to 1 for soft constraint.
 * @param ssfac Stiffness scale factor for penalty stiffness value. The default value is unity. This applies to SOFT set to 0 and 1.
 * @param fric Contact friction.
 * @param cid ContactGuidedCable number (Same as label).
 * @param heading ContactGuidedCable heading (Same as title).
 * @param endtol Tolerance, in length units.
 * @returns ContactGuidedCable object
 */
    constructor(Model: Model, ptype: number, nsid: number, pid: number, soft?: number, ssfac?: number, fric?: number, cid?: number, heading?: string, endtol?: number);

/**
 * Creates a string containing the ContactGuidedCable data in keyword format. Note that this contains the keyword header and the keyword cards. See also ContactGuidedCable.Keyword() and ContactGuidedCable.KeywordCards().
 * @returns string
 */
    toString(): string;

/** ContactGuidedCable number. */
    cid: number;
/** Tolerance, in length units. */
    endtol: number;
/** true if contact guided_cable exists, false if referred to but not defined. */
    readonly exists: boolean;
/** Contact friction. */
    fric: number;
/** ContactGuidedCable heading */
    heading: string;
/** TRUE if _ID option is set, FALSE if not */
    id: boolean;
/** The Include file number that the contact guided_cable is in. */
    include: number;
/** The Model number that the contact guided_cable is in. */
    readonly model: number;
/** Node Set ID that guides the 1D elements. */
    nsid: number;
/** Part ID or Part Set ID */
    pid: number;
/** The Contact Part type. Can be ContactGuidedCable.PART or ContactGuidedCable.SET_PART. */
    ptype: number;
/** Flag for soft constraint option. Set to 1 for soft constraint. */
    soft: number;
/** Stiffness scale factor for penalty stiffness value. The default value is unity. This applies to SOFT set to 0 and 1. */
    ssfac: number;
/** CONTACT is *CONTACT_GUIDED_CABLE. */
    static PART: number;
/** CONTACT is *CONTACT_GUIDED_CABLE_SET. */
    static SET_PART: number;
}

declare class Control {
/**
 * Returns the damping coefficient data for an index in *CONTROL_IMPLICIT_MODAL_DYNAMIC_DAMPING.
 * @param index The index you want the data for. Note that indices start at 0, not 1.
 * @returns An array containing the mode id/frequency and damping coefficient values.
 */
    GetCoefficient(index: number): number[];

/**
 * Returns the transformation for a row in *CONTROL_MPP_DECOMPOSITION_TRANSFORMATION.
 * @param row The row you want the data for. Note that rows start at 0, not 1.
 * @returns An array containing the transformation type and the parameters V1 to V9.
 */
    GetTransformation(row: number): any[];

/**
 * Removes the damping coefficient data for an index in *CONTROL_IMPLICIT_MODAL_DYNAMIC_DAMPING.
 * @param index The index you want to delete damping data for. Note that indices start at 0, not 1.
 */
    RemoveCoefficient(index: number): void;

/**
 * Removes a row in *CONTROL_MPP_DECOMPOSITION_TRANSFORMATION.
 * @param row The row number to be deleted. Note that rows start at 0, not 1. If there are rows under this one, they will be shifted up.
 */
    RemoveTransformation(row: number): void;

/**
 * Sets the damping coefficient data for an index in *CONTROL_IMPLICIT_MODAL_DYNAMIC_DAMPING.
 * @param index The index you want to set the data for. Note that indices start at 0, not 1.
 * @param mode_frequency The mode ID (_SPECIFIC) or frequency (_FREQUENCY_RANGE).
 * @param zeta Damping coefficient
 */
    SetCoefficient(index: number, mode_frequency: number, zeta: number): void;

/**
 * Sets the type and the parameters V1 to V9 for a row in *CONTROL_MPP_DECOMPOSITION_TRANSFORMATION. Note: If the row already exists, the type and the parameters will be overwritten.
 * @param row The row you want to set the data for. Note that rows start at 0, not 1.
 * @param type_or_array The string repesenting the type of the transformation. Alternatively an array with this and the following parameters on it.
 * @param V1 The float repesenting the parameter V1
 * @param V2_to_V9 The floats repesenting the parameters V2 to V9. These are only used for VEC3, C2R, S2R and MAT.
 */
    SetTransformation(row: number, type_or_array: string | array[], V1: number, V2_to_V9?: number): void;

/** Load curve ID for the a coefficient used in the formula */
    a: number;
/** ALE advection factor */
    aafac: number;
/** Absolute upper bound for the dynamic memory allocated for factorization */
    absmem: string;
/** Terminate based on absolute total displacement in the Euclidean norm. */
    absol: number;
/** absolute convergence tol */
    abstol: number;
/** Absolute convergence tolerance */
    abstol_1: number;
/** Absolute convergence tolerance */
    abstol_2: number;
/** gravity */
    accel: number;
/** *CONTROL_ACCURACY card */
    accuracy: object;
/** Multiplier on proximity test. */
    acecf1: number;
/** Angle between normal vectors in an orientation test. */
    acecf2: number;
/** Multiplier on ceiling test. */
    acecf3: number;
/** Area equilibration threshold. */
    acecf4: number;
/** Artificial compressibility term */
    acmp: number;
/** *CONTROL_ACOUSTIC card */
    acoustic: object;
/** *CONTROL_ACOUSTIC_COUPLING card */
    acoustic_coupling: object;
/** *CONTROL_ACOUSTIC_SPECTRAL card */
    acoustic_spectral: object;
/** *CONTROL_ADAPSTEP card */
    adapstep: object;
/** *CONTROL_ADAPTIVE card */
    adaptive: object;
/** *CONTROL_ADAPTIVE_CURVE card */
    adaptive_curve: object;
/** Total number of adaptive steps during the forming simulation */
    adatims: number;
/** 1 or 2 pass adaptivity flag */
    adpass: number;
/** Adaptivity error tolerance in degrees for activating fusion */
    adpctl: number;
/** Flag to control the output of adaptive mesh &amp; d3plot files */
    adpd3p: number;
/** Type of damage accumulation in the workpiece */
    adpdam: number;
/** Nodal penetration at which to refine elem */
    adpene: number;
/** Options for recovery techniques and error estimators */
    adperr: number;
/** Time interval between refinements */
    adpfreq: number;
/** Adaptive options */
    adpopt: number;
/** Strain rate scale factor */
    adpscl: number;
/** Min element edge size for adaptivity */
    adpsize: number;
/** Absolute shell thickness below which remeshing should begin */
    adpth: number;
/** Adaptive error tolerance (degrees) */
    adptol: number;
/** Adaptive options */
    adptyp: number;
/** Smoothing weight factor: simple average */
    afac: number;
/** Pressure of atmospheric air */
    air_p: number;
/** Density of atmospheric air */
    air_ro: number;
/** *CONTROL_AIRBAG card */
    airbag: object;
/** A=Aluminium blank, F=steel */
    al_fe: string;
/** A=Aluminium blank, F=steel */
    al_fe_1: string;
/** *CONTROL_ALE card */
    ale: object;
/** Contact algorithm type */
    algo: number;
/** Composite time integration constant */
    alpha: number;
/** Maximum allowable acceleration */
    amax: number;
/** Maximum allowable acceleration */
    amax_1: number;
/** Flag to turn off printing of pore air analysis status message */
    anamsg: number;
/** contact angle */
    ang: number;
/** Permitted angle between neighbours */
    angle: number;
/** relative influence predictor step */
    arcalf: number;
/** Arc length controlling node ID */
    arcctl: number;
/** Arc length controlling node direction */
    arcdir: number;
/** Arc length damping option */
    arcdmp: number;
/** Arc length size */
    arclen: number;
/** Arc length method */
    arcmth: number;
/** relative influence load/time parameter */
    arcpsi: number;
/** initiation time */
    arctim: number;
/** Analysis type */
    atype: number;
/** Thermal analysis type */
    atype_1: number;
/** AUTOSPC switch */
    autospc: number;
/** AUTOSPC tolerance */
    autotol: number;
/** weight factor */
    awgt: number;
/** Load curve ID for the b coefficient used in the formula */
    b: number;
/** Thermal boundary condition flag */
    bc_flg: number;
/** Method to extract buckling modes */
    bckmth: number;
/** Flag for aligning beam dynamics */
    beamin: number;
/** start time */
    begtim: number;
/** Newmark time integration constant */
    beta: number;
/** Smoothing weight factor: volume weighting */
    bfac: number;
/** flag to set _BINARY option */
    binary: number;
/** birth time */
    birth: number;
/** Part (stype=0) or part set (stype=1) ID for blank */
    blank: number;
/** Part (stype=0) or part set (stype=1) ID that defines the blank */
    blkid: number;
/** Flag to allocate additional memory for contact nodal forces */
    blkv: number;
/** beam set for convert to hex assembly */
    bmsid: number;
/** Multi-Material ALE group set if positive or -1 */
    bndflx: number;
/** Part that defines the lower binder */
    bndl: number;
/** Part that defines the upper binder */
    bndu: number;
/** Box limiting application */
    boxid: number;
/** Birth time */
    bt: number;
/** beam bulk viscosity type */
    btype: number;
/** *CONTROL_BULK_VISCOSITY card */
    bulk_viscosity: object;
/** Warping stiffness flag for Belytschko-Tsay shells */
    bwc: number;
/** Load curve ID for the c coefficient used in the formula */
    c: number;
/** dry/wet particle flag */
    cap: number;
/** Birth time for adaptive fusion */
    cbirth: number;
/** Death time for adaptive fusion */
    cdeath: number;
/** Tolerance for output of *DEFINE_CURVE discretization warnings */
    cdetol: number;
/** Centre frequency */
    center: number;
/** Smoothing weight factor: isoparametric */
    cfac: number;
/** Convergence tolerance for iterative solver */
    cgtol: number;
/** Max flange height */
    charlen: number;
/** *CONTROL_CHECK card */
    check: object;
/** Parameter for ALE pressure locking */
    checkr: number;
/** ID of coordinate system (only for the LOCAL option) */
    cid: number;
/** coordinate system id */
    cid_1: number;
/** Flag to check and report open edge of CV airbag */
    ckerr: number;
/** Limit angle for corner nodes */
    cnla: number;
/** Consistent tangent stiffness flag */
    cnstn: number;
/** include shell ref surface offset */
    cntco: number;
/** *CONTROL_COARSEN card */
    coarsen: object;
/** Number of coefficients for SPECIFIC and FREQUENCY_RANGE options */
    readonly coefficients: number;
/** Cohesive element quality check */
    coheqc: number;
/** Flag to treat how the mass from SURFB of a tied contact affects the time step estimation of cohesive elements */
    cohtiem: number;
/** damping factor */
    conmax: number;
/** *CONTROL_CONSTRAINED card */
    constrained: object;
/** Consistency (Accuracy) flag */
    consty: number;
/** Particle approx method */
    cont: number;
/** *CONTROL_CONTACT card */
    contact: object;
/** conduction factor */
    conv: number;
/** *CONTROL_COUPLING card */
    coupling: object;
/** DES thermal property */
    cp: number;
/** Heat Capacity. Constant float value if CPTYP = 0,  Curve ID if CPTYP = 1. */
    cp_1: number;
/** Contact penetration check flag */
    cpchk: number;
/** *CONTROL_CPG card */
    cpg: object;
/** *CONTROL_CPM card */
    cpm: object;
/** Disable/enable error checking */
    cpmerr: number;
/** Flag to consider airbag system velocity */
    cpmmf: number;
/** Type of CP. Valid values: 0-Constant, 1-Temperature vs Heat Capacity Curve. */
    cptyp: number;
/** *CONTROL_CPU card */
    cpu: object;
/** Max permitted cpu time */
    cputim: number;
/** Bypass time-based evaluation of non-time-dependent curves */
    crvp: number;
/** Coord sys for type 6 element */
    cstyp6: number;
/** X component of centre of most-bent location */
    cx: number;
/** Y component of centre of most-bent location */
    cy: number;
/** Z component of centre of most-bent location */
    cz: number;
/** Explicit accuracy parameter */
    d: number;
/** D3ITER database control */
    d3itctl: number;
/** Number of output states in the D3PLOT database */
    d3plot: number;
/** Number of output states in the D3PLOT database */
    d3plt: number;
/** Flag for writing out D3PLOT state */
    d3trace: number;
/** Modal damping coefficient */
    dampf: number;
/** Z elevation of datum */
    datum: number;
/** Exponential decay coefficient */
    dc: number;
/** Divergence control parameter */
    dcp: number;
/** Default continuum treatment */
    dct: number;
/** Displacement convergence tolerance */
    dctol: number;
/** death time */
    death: number;
/** *CONTROL_DEBUG card */
    debug: object;
/** Geometry for decomposition */
    defgeo: number;
/** delete shells where neighbours fail */
    delfr: number;
/** Terminate based on rel total displacement in max norm */
    delta1: number;
/** Terminate based on rel total displacement in Euclidean norm */
    deltau: number;
/** Output DEM density data to D3PLOT database */
    demden: number;
/** Density */
    density: number;
/** Density */
    density_1: number;
/** Time integration type */
    deriv: number;
/** Smoothing weight factor: equipotential */
    dfac: number;
/** Incremental increase in factin */
    dfactr: number;
/** Default dynamic coefficient of friction */
    dfric: number;
/** Part that defines the die */
    dieid: number;
/** Directional tie for MPP non-groupable tied contacts */
    dir_tie: number;
/** Number of automatically determined directions */
    direct: number;
/** *CONTROL_DISCRETE_ELEMENT card */
    discrete_element: object;
/** Dissipation energy calculation calc flag */
    disen: number;
/** Distance tolerance for auto-SPC along flange roots */
    dist: number;
/** Divergence flag */
    diverg: number;
/** Mass proportional damping constant in Rayleigh damping */
    dmpmas: number;
/** Stiffness proportional damping constant in Rayleigh damping */
    dmpstf: number;
/** Maximum displacement convergence tolerance */
    dmtol: number;
/** Displacement norm for convergence test */
    dnorm: number;
/** Applicable degrees-of-freedom for excitation input */
    dof1: number;
/** Applicable degrees-of-freedom for response output */
    dof2: number;
/** Dormant part treatment in D3PLOT file */
    dordel: number;
/** Max rate of change of pwp water head (m/s) */
    dpwmax: number;
/** Drilling rotation constraint method */
    drcm: number;
/** drilling rotation constraint method. */
    drcmth: number;
/** Drilling rotation constraint parameter */
    drcprm: number;
/** part set for drilling rotation constraint method. */
    drcpsid: number;
/** drilling rotation constraint parameter. */
    drcpsrm: number;
/** Dyn relaxation factor */
    drfctr: number;
/** Drilling energy calculation flag */
    drlen: number;
/** Part set used to check for convergence */
    drpset: number;
/** Optional DR termination time */
    drterm: number;
/** Convergence tolerance */
    drtol: number;
/** Death time */
    dt: number;
/** initial time step size */
    dt0: number;
/** Initial timestep for implicit analysis */
    dt0_1: number;
/** Timestep for mass scaling */
    dt2ms: number;
/** Scale factor for initial timestep size to determine min permitted time step size */
    dt2msf: number;
/** Loadcurve: DT2MS vs time */
    dt2mslc: number;
/** Death time */
    dt_1: number;
/** Flag for output option (time interval or cycle number) */
    dt_cycle: number;
/** Flag for output option (time interval or cycle number) */
    dt_cycle_1: number;
/** Flag to consider the effects of dynamic viscosity on the critical time step */
    dtdynv: number;
/** Max delta temp permitted before timestep decrease, of LC of dt vs time if -ve */
    dtemp: number;
/** time in explicit before switch */
    dtexp: number;
/** Time step factor. */
    dtfac: number;
/** Initial timestep size */
    dtinit: number;
/** Maximum allowable timestep. Loadcurve if negative */
    dtmax: number;
/** Minimum allowable timestep */
    dtmin: number;
/** timestep size for remesh */
    dtmin_1: number;
/** Scale factor on initial dt size for termination */
    dtmin_2: number;
/** Scale time step called DTMU */
    dtmufac: number;
/** Time interval between outputs. Constant float value if DTOUTYP = 0,  Curve ID if DTOUTYP = 1. */
    dtout: number;
/** Modal dynamics output interval */
    dtout_1: number;
/** Type of DTOUT. Valid values: 0-Constant, 1-Time vs DTOUT Curve. */
    dtoutyp: number;
/** User-defined time step for explicit analysis */
    dtusr: number;
/** Time interval between view factor updates */
    dtvf: number;
/** Not used */
    dvid: number;
/** *CONTROL_DYNAMIC_RELAXATION card */
    dynamic_relaxation: object;
/** Youngs modulus */
    e: number;
/** Youngs modulus */
    e_1: number;
/** Automatic Euler boundary condition */
    ebc: number;
/** Timestep override for eroding contacts */
    ecdt: number;
/** Energy convergence tolerance */
    ectol: number;
/** Default exponential decay coefficient */
    edc: number;
/** Convergence tolerance on auto control */
    edttl: number;
/** Smoothing weight factor: equipotential */
    efac: number;
/** *CONTROL_EFG card */
    efg: object;
/** efg keyword option */
    efg_1: number;
/** Control whether to use the original (unscaled) or the scaled mass */
    eigmscl: number;
/** Eigenvalue extraction method */
    eigmth: number;
/** Fraction of added mass from mass scaling that contributes to gravity loads */
    emscl: number;
/** Maximum energy convergence tolerance */
    emtol: number;
/** Flag for dumping elemental stiffness and mass matrices */
    emxdmp: number;
/** End time for smoothing */
    end: number;
/** Termination cycle # */
    endcyc: number;
/** %age change in energy for termination */
    endeng: number;
/** %age change in mass for termination. Loadcurve if negative */
    endmas: number;
/** Termination time */
    endtim: number;
/** *CONTROL_ENERGY card */
    energy: object;
/** Flag to output contact sliding energy densities for mortar contact */
    engout: number;
/** Treatment of mass of eroded nodes */
    enmass: number;
/** Elout coordinate system option */
    eocs: number;
/** Max effective plastic strain, beyond which elements are deleted */
    epsmx: number;
/** Flag to determine if electric flux and electric field at the element center of piezoelectric material is outputed to D3PLOT */
    epzmsg: number;
/** Mechanical equivelent of heat (J/Nm etc). Loadcurve if negative */
    eqheat: number;
/** Erosion flag for solids &amp; shells @ DTMIN */
    erode: number;
/** Degenerate shell sorting flag (was ITRIST) */
    esort: number;
/** Automatic sort of tetra &amp; penta flag */
    esort_1: number;
/** Event termination time */
    eterm: number;
/** event time termination */
    eterm_1: number;
/** Flag for interpretation of time */
    etflag: number;
/** Error tolerance in the IMLM */
    etol: number;
/** Flag for writing eigenvalues and eigenvectors */
    evdump: number;
/** Explicit accuracy parameter */
    exacc: number;
/** .eq.1 if excl above */
    excl: number;
/** true if control card exists */
    exists: boolean;
/** true if control card exists */
    exists_1: boolean;
/** true if control card exists */
    exists_10: boolean;
/** true if control card exists */
    exists_100: boolean;
/** true if control card exists */
    exists_101: boolean;
/** true if control card exists */
    exists_102: boolean;
/** true if control card exists */
    exists_103: boolean;
/** true if control card exists */
    exists_104: boolean;
/** true if control card exists */
    exists_105: boolean;
/** true if control card exists */
    exists_106: boolean;
/** true if control card exists */
    exists_107: boolean;
/** true if control card exists */
    exists_108: boolean;
/** true if control card exists */
    exists_109: boolean;
/** true if control card exists */
    exists_11: boolean;
/** true if control card exists */
    exists_110: boolean;
/** true if control card exists */
    exists_111: boolean;
/** true if control card exists */
    exists_112: boolean;
/** true if control card exists */
    exists_113: boolean;
/** true if control card exists */
    exists_114: boolean;
/** true if control card exists */
    exists_115: boolean;
/** true if control card exists */
    exists_116: boolean;
/** true if control card exists */
    exists_117: boolean;
/** true if control card exists */
    exists_118: boolean;
/** true if control card exists */
    exists_119: boolean;
/** true if control card exists */
    exists_12: boolean;
/** true if control card exists */
    exists_120: boolean;
/** true if control card exists */
    exists_121: boolean;
/** true if control card exists */
    exists_122: boolean;
/** true if control card exists */
    exists_123: boolean;
/** true if control card exists */
    exists_124: boolean;
/** true if control card exists */
    exists_125: boolean;
/** true if control card exists */
    exists_126: boolean;
/** true if control card exists */
    exists_127: boolean;
/** true if control card exists */
    exists_128: boolean;
/** true if control card exists */
    exists_129: boolean;
/** true if control card exists */
    exists_13: boolean;
/** true if control card exists */
    exists_130: boolean;
/** true if control card exists */
    exists_14: boolean;
/** true if control card exists */
    exists_15: boolean;
/** true if control card exists */
    exists_16: boolean;
/** true if control card exists */
    exists_17: boolean;
/** true if control card exists */
    exists_18: boolean;
/** true if control card exists */
    exists_19: boolean;
/** true if control card exists */
    exists_2: boolean;
/** true if control card exists */
    exists_20: boolean;
/** true if control card exists */
    exists_21: boolean;
/** true if control card exists */
    exists_22: boolean;
/** true if control card exists */
    exists_23: boolean;
/** true if control card exists */
    exists_24: boolean;
/** true if control card exists */
    exists_25: boolean;
/** true if control card exists */
    exists_26: boolean;
/** true if control card exists */
    exists_27: boolean;
/** true if control card exists */
    exists_28: boolean;
/** true if control card exists */
    exists_29: boolean;
/** true if control card exists */
    exists_3: boolean;
/** true if control card exists */
    exists_30: boolean;
/** true if control card exists */
    exists_31: boolean;
/** true if control card exists */
    exists_32: boolean;
/** true if control card exists */
    exists_33: boolean;
/** true if control card exists */
    exists_34: boolean;
/** true if control card exists */
    exists_35: boolean;
/** true if control card exists */
    exists_36: boolean;
/** true if control card exists */
    exists_37: boolean;
/** true if control card exists */
    exists_38: boolean;
/** true if control card exists */
    exists_39: boolean;
/** true if control card exists */
    exists_4: boolean;
/** true if control card exists */
    exists_40: boolean;
/** true if control card exists */
    exists_41: boolean;
/** true if control card exists */
    exists_42: boolean;
/** true if control card exists */
    exists_43: boolean;
/** true if control card exists */
    exists_44: boolean;
/** true if control card exists */
    exists_45: boolean;
/** true if control card exists */
    exists_46: boolean;
/** true if control card exists */
    exists_47: boolean;
/** true if control card exists */
    exists_48: boolean;
/** true if control card exists */
    exists_49: boolean;
/** true if control card exists */
    exists_5: boolean;
/** true if control card exists */
    exists_50: boolean;
/** true if control card exists */
    exists_51: boolean;
/** true if control card exists */
    exists_52: boolean;
/** true if control card exists */
    exists_53: boolean;
/** true if control card exists */
    exists_54: boolean;
/** true if control card exists */
    exists_55: boolean;
/** true if control card exists */
    exists_56: boolean;
/** true if control card exists */
    exists_57: boolean;
/** true if control card exists */
    exists_58: boolean;
/** true if control card exists */
    exists_59: boolean;
/** true if control card exists */
    exists_6: boolean;
/** true if control card exists */
    exists_60: boolean;
/** true if control card exists */
    exists_61: boolean;
/** true if control card exists */
    exists_62: boolean;
/** true if control card exists */
    exists_63: boolean;
/** true if control card exists */
    exists_64: boolean;
/** true if control card exists */
    exists_65: boolean;
/** true if control card exists */
    exists_66: boolean;
/** true if control card exists */
    exists_67: boolean;
/** true if control card exists */
    exists_68: boolean;
/** true if control card exists */
    exists_69: boolean;
/** true if control card exists */
    exists_7: boolean;
/** true if control card exists */
    exists_70: boolean;
/** true if control card exists */
    exists_71: boolean;
/** true if control card exists */
    exists_72: boolean;
/** true if control card exists */
    exists_73: boolean;
/** true if control card exists */
    exists_74: boolean;
/** true if control card exists */
    exists_75: boolean;
/** true if control card exists */
    exists_76: boolean;
/** true if control card exists */
    exists_77: boolean;
/** true if control card exists */
    exists_78: boolean;
/** true if control card exists */
    exists_79: boolean;
/** true if control card exists */
    exists_8: boolean;
/** true if control card exists */
    exists_80: boolean;
/** true if control card exists */
    exists_81: boolean;
/** true if control card exists */
    exists_82: boolean;
/** true if control card exists */
    exists_83: boolean;
/** true if control card exists */
    exists_84: boolean;
/** true if control card exists */
    exists_85: boolean;
/** true if control card exists */
    exists_86: boolean;
/** true if control card exists */
    exists_87: boolean;
/** true if control card exists */
    exists_88: boolean;
/** true if control card exists */
    exists_89: boolean;
/** true if control card exists */
    exists_9: boolean;
/** true if control card exists */
    exists_90: boolean;
/** true if control card exists */
    exists_91: boolean;
/** true if control card exists */
    exists_92: boolean;
/** true if control card exists */
    exists_93: boolean;
/** true if control card exists */
    exists_94: boolean;
/** true if control card exists */
    exists_95: boolean;
/** true if control card exists */
    exists_96: boolean;
/** true if control card exists */
    exists_97: boolean;
/** true if control card exists */
    exists_98: boolean;
/** true if control card exists */
    exists_99: boolean;
/** *CONTROL_EXPLICIT_THERMAL_PROPERTIES card */
    explicit_thermal: object;
/** *CONTROL_EXPLICIT_THERMAL_ALE_COUPLING card */
    explicit_thermal_ale_coupling: object;
/** *CONTROL_EXPLICIT_THERMAL_BOUNDARY card */
    explicit_thermal_boundary: object;
/** *CONTROL_EXPLICIT_THERMAL_CONTACT card */
    explicit_thermal_contact: object;
/** *CONTROL_EXPLICIT_THERMAL_INITIAL card */
    explicit_thermal_initial: object;
/** *CONTROL_EXPLICIT_THERMAL_OUTPUT card */
    explicit_thermal_output: object;
/** *CONTROL_EXPLICIT_THERMAL_SOLVER card */
    explicit_thermal_solver: object;
/** *CONTROL_EXPLOSIVE_SHADOW card */
    explosive_shadow: object;
/** Internal flag to set 936 compatibility */
    f_936: number;
/** default stiffness/gravity factor */
    fact: number;
/** Initial relaxation factor for contact force */
    factin: number;
/** Target mesh file in keyword format */
    filename: string;
/** Constraint modes file name */
    filename2: string;
/** Residual vectors file name */
    filename3: string;
/** Eigen modes file name */
    filename_1: string;
/** Flag to flip X coords */
    flipx: number;
/** Flag to flip Y coords */
    flipy: number;
/** Flag to flip Z coords */
    flipz: number;
/** calculation method for deformation gradient */
    fmatrix: number;
/** Maximum frequency for FRF output */
    fmax: number;
/** Maximum frequency in the solution */
    fmax_1: number;
/** max seepage factor */
    fmax_2: number;
/** Maximum factor on seepage calc */
    fmax_3: number;
/** Minimum frequency for FRF output */
    fmin: number;
/** Minimum frequency in the solution */
    fmin_1: number;
/** min seepage factor */
    fmin_2: number;
/** Minimum factor on seepage calc */
    fmin_3: number;
/** Optional maximum natural frequency */
    fnmax: number;
/** Part for tool to follow */
    follow: number;
/** Element formulation to use. */
    form: number;
/** particle theory */
    form_1: number;
/** *CONTROL_FORMING_BESTFIT card */
    forming_bestfit: object;
/** *CONTROL_FORMING_INITIAL_THICKNESS card */
    forming_initial_thickness: object;
/** *CONTROL_FORMING_MAXID card */
    forming_maxid: object;
/** *CONTROL_FORMING_POSITION card */
    forming_position: object;
/** *CONTROL_FORMING_PRE_BENDING card */
    forming_pre_bending: object;
/** *CONTROL_FORMING_PROJECTION card */
    forming_projection: object;
/** *CONTROL_FORMING_REMOVE_ADAPTIVE_CONSTRAINTS card */
    forming_remove_adaptive_constraints: object;
/** *CONTROL_FORMING_SHELL_TO_TSHELL card */
    forming_shell_to_tshell: object;
/** *CONTROL_FORMING_STONING card */
    forming_stoning: object;
/** *CONTROL_FORMING_STRAIN_RATIO_SMOOTH card */
    forming_strain_ratio_smooth: object;
/** *CONTROL_FORMING_TEMPLATE card */
    forming_template: object;
/** *CONTROL_FORMING_TOLERANC card */
    forming_toleranc: object;
/** *CONTROL_FORMING_TRAVEL card */
    forming_travel: object;
/** *CONTROL_FORMING_TRIM_MERGE card */
    forming_trim_merge: object;
/** *CONTROL_FORMING_TRIM_SOLID_REFINEMENT card */
    forming_trim_solid_refinement: object;
/** *CONTROL_FORMING_UNFLANGING card */
    forming_unflanging: object;
/** *CONTROL_FORMING_USER card */
    forming_user: object;
/** Formula that defines the contact heat conductance as a function of temperature and pressure */
    formula: number;
/** Octave fraction */
    fractn: number;
/** Radiation factor between the contact surfaces */
    frad: number;
/** Flag to calculate internal friction energy */
    frceng: number;
/** Time interval between redecomposition */
    freq: number;
/** *CONTROL_FREQUENCY_DOMAIN card */
    frequency_domain: object;
/** If FREQUENCY_RANGE option is used */
    frequency_range: boolean;
/** *CONTROL_FREQUENCY_RESPONSE_FUNCTION card */
    frequency_response_function: object;
/** Output frequency for failed element report */
    frfreq: number;
/** Dynamic coefficient of friction */
    fricd: number;
/** rolling friction coefficient */
    fricr: number;
/** friction coefficient */
    frics: number;
/** Friction coefficient */
    fs: number;
/** Friction coefficient */
    fs_1: number;
/** Solution frequency assignment strategy */
    fspace: number;
/** output contact forces to rcforc */
    ftall: number;
/** Analysis type */
    ftied: number;
/** Fraction of sliding friction energy partitioned to the SURFA surface */
    ftosa: number;
/** Fraction of mechanical heat converted into heat */
    fwork: number;
/** Fraction of mechanical work converted into heat */
    fwork_1: number;
/** liquid surface tension */
    gamma: number;
/** Newmark time integration constant */
    gamma_1: number;
/** parameter affecting spatial limit of liquid bridge */
    gap: number;
/** Minimum gap */
    gap_1: number;
/** Home gap between rigid tools */
    gap_2: number;
/** Min distance between tool and target in the home position */
    gap_3: number;
/** Minimum gap between tools */
    gap_4: number;
/** Gap distance between two open ends of a trim loop curve in the model */
    gapm: number;
/** Separation distance calculation flag */
    gaponly: number;
/** Joint rotational stiffness */
    gjadstf: number;
/** Joint rotational damping */
    gjadvsc: number;
/** output interval for *INTERFACE_SSI_AUX */
    gmdt: number;
/** #gauss points in solids */
    gpt: number;
/** Gravitational acceleration for Ro.g.h */
    grav: number;
/** GROUPABLE algorithm options */
    grp: number;
/** Heat transfer conductance for closed gaps */
    h0: number;
/** Mid-point relative Euclidian residual norm max tolerance */
    hcmax: number;
/** Mid-point relative Euclidian residual norm min tolerance */
    hcmin: number;
/** Hourglass energy calc flag */
    hgen: number;
/** Flag to invoke output of extra history variable names */
    hisnout: number;
/** Mid-point relative maximum residual norm max tolerance */
    hmmax: number;
/** Mid-point relative maximum residual norm min tolerance */
    hmmin: number;
/** Mid-point absolute Nodal Rotational norm tolerance */
    hnrmax: number;
/** Mid-point absolute Nodal Translational norm tolerance */
    hntmax: number;
/** *CONTROL_HOURGLASS card */
    hourglass: object;
/** Mid-point absolute Rigid body Rotational norm tolerance */
    hrrmax: number;
/** Mid-point absolute Rigid body Translational norm tolerance */
    hrtmax: number;
/** Not used */
    hsort: number;
/** Keyword to be output to a dynain file */
    i2dynain: number;
/** interactive adaptivity adjustable tolerance */
    iaat: number;
/** Implicit accuracy flag */
    iacc: number;
/** Flag for accels in d3thdt to be averaged */
    iaccop: number;
/** Fission level that fusion will start at */
    iadpcl: number;
/** Maximum no. of elements covering 90degree of radii */
    iadpe90: number;
/** Fiffion flag for neighbour splitting */
    iadpgh: number;
/** Include ALE computations in the dynamic relaxation analysis */
    ialedr: number;
/** Artificial stabilization flag */
    ias: number;
/** interactive adaptivity */
    iat: number;
/** tolerance of shear distortion indicator for interactive adaptivity */
    iat1: number;
/** tolerance of unbalanced nodal distribution indicator for interactive adaptivity */
    iat2: number;
/** tolerance of volumetric change indicator for interactive adaptivity */
    iat3: number;
/** Automatic timestep control flag. Loadcurve if negative */
    iauto: number;
/** artificial viscosity formulation */
    iavis: number;
/** Offset for numbering */
    ibase: number;
/** Mesh integrity check */
    ibchk: number;
/** Beam element formulation for implicit */
    ibeam: number;
/** Flag providing options for the block low-rank (BLR) computations */
    iblropt: number;
/** Pressure treatment of boundary particles */
    ibndp: number;
/** Default bulk viscosity type (m#PR035) */
    ibq: number;
/** Flag to invoke output of *SET_BEAM data */
    ibsf: number;
/** On/Off flag */
    icoarse: number;
/** global flag for cohesive element deletion */
    icoh: number;
/** contact option */
    icont: number;
/** Coordinates used in rebalance */
    icoor: number;
/** Correction of density and total energy */
    icorr: number;
/** Element costs used in rebalance */
    icost: number;
/** Invokes the covariant formulation of Konyukhov and Schweizerhof */
    icov: number;
/** Output node and element sets used in computing secforc data */
    icrfile: number;
/** number of elements along radius */
    icurv: number;
/** Treatment of cylindrical joints */
    icylin: number;
/** If less than 0 then Element ID if greater than 0 then  Set ID. Can be SOLID, SHELL, BEAM or THICK SHELL based on value of idtyp. */
    id: number;
/** Contact ID 1 to distrubute */
    id1: number;
/** Contact ID 10 to distrubute */
    id10: number;
/** Contact ID 11 to distrubute */
    id11: number;
/** Contact ID 12 to distrubute */
    id12: number;
/** Contact ID 13 to distrubute */
    id13: number;
/** Contact ID 14 to distrubute */
    id14: number;
/** Contact ID 15 to distrubute */
    id15: number;
/** Contact ID 1 to distrubute */
    id1_1: number;
/** Contact ID 2 to distrubute */
    id2: number;
/** Contact ID 2 to distrubute */
    id2_1: number;
/** Contact ID 3 to distrubute */
    id3: number;
/** Contact ID 3 to distrubute */
    id3_1: number;
/** Write d3mode file flag */
    id3mode: number;
/** Contact ID 4 to distrubute */
    id4: number;
/** Contact ID 4 to distrubute */
    id4_1: number;
/** Contact ID 5 to distrubute */
    id5: number;
/** Contact ID 5 to distrubute */
    id5_1: number;
/** Contact ID 6 to distrubute */
    id6: number;
/** Contact ID 7 to distrubute */
    id7: number;
/** Contact ID 8 to distrubute */
    id8: number;
/** Contact ID 9 to distrubute */
    id9: number;
/** part id offset */
    id_off: number;
/** Output internal debug message */
    ideb: number;
/** Flag for soft constraint formulation */
    idesoft: number;
/** dilation param */
    idila: number;
/** Space system flag */
    idim: number;
/** Stress initialisation flag */
    idrflg: number;
/** Set id of the segments to be generated at the bottom layer */
    idsegb: number;
/** Set id of the segments to be generated at the top layer */
    idsegt: number;
/** Shell/Part set id */
    idset: number;
/** Type of forming process */
    idtemp: number;
/** Type of ID. Valid values: 1-Solid, 2-Shell, 3-Beam, 4-Thick shell . */
    idtyp: number;
/** Flag to invoke diagnostic output to error file during the initialisation phase */
    ielogini: number;
/** Flag to invoke diagnostic output to error file during the input phase */
    ielogkey: number;
/** Flag to invoke diagnostic output to error file during the solution phase */
    ielogsol: number;
/** remeshing with element erosion */
    ier: number;
/** erosion option */
    ierod: number;
/** output eroded energy */
    ierode: number;
/** Terminate based on internal energy */
    ietol: number;
/** Computing performance optimisation flag */
    ifast: number;
/** Best fit flag */
    ifit: number;
/** Consistent mass matrix flag */
    iflag: number;
/** Iteration limit for first phase of unfolding */
    iflimit: number;
/** i/o buffer flushing interval (t-steps) */
    iflush: number;
/** Format for processing eigenmodes */
    iformat: number;
/** Flag for forming of sandwiched parts with adaptive blank mesh */
    ifsand: number;
/** Optional flag to define a node set to be included or excluded */
    ifset: number;
/** option to use isogeometric shells for contact detection */
    igactc: number;
/** Method for calculating time steps for IGA elements */
    igado: number;
/** glstat data flag */
    iglst: number;
/** Ignore initial penetrations flag */
    ignore: number;
/** Ignore penetration flag */
    ignore_1: number;
/** Geometric (initial stress) stiffness flag */
    igs: number;
/** Method for calculating solid element time steps */
    ihdo: number;
/** Hourglass viscosity type */
    ihq: number;
/** Flag for Heat Transfer Coefficient calculation */
    ihtc: number;
/** Status report interval to d3hsp */
    ikedit: number;
/** Adaptive refinement level */
    ilevel: number;
/** Iteration limit between automatic stiffness reformations */
    ilimit: number;
/** Unfolding algorithm selection flag */
    ilinear: number;
/** Flag for mass scaling for ALE parts */
    imascl: number;
/** Implicit analysis type */
    imass: number;
/** Inertia switching flag */
    imass_1: number;
/** Flag for *MAT_SPH_INCOMPRESSIBLE_FLUID formulations */
    imat: number;
/** Activation flag */
    imerge: number;
/** Implicit/explicit switching flag; loadcurve if negative */
    imflag: number;
/** Element formulation switching flag */
    imform: number;
/** Choice for matrix operation, linear solving and memory usage */
    imlm: number;
/** *CONTROL_IMPLICIT_AUTO card */
    implicit_auto: object;
/** *CONTROL_IMPLICIT_BUCKLE card */
    implicit_buckle: object;
/** *CONTROL_IMPLICIT_CONSISTENT_MASS card */
    implicit_consistent_mass: object;
/** *CONTROL_IMPLICIT_DYNAMICS card */
    implicit_dynamics: object;
/** *CONTROL_IMPLICIT_EIGENVALUE card */
    implicit_eigenvalue: object;
/** *CONTROL_IMPLICIT_EXPLICIT_HYBRID card */
    implicit_explicit_hybrid: object;
/** *CONTROL_IMPLICIT_FORMING card */
    implicit_forming: object;
/** *CONTROL_IMPLICIT_GENERAL card */
    implicit_general: object;
/** *CONTROL_IMPLICIT_INERTIA_RELIEF card */
    implicit_inertia_relief: object;
/** *CONTROL_IMPLICIT_JOINTS card */
    implicit_joints: object;
/** *CONTROL_IMPLICIT_MODAL_DYNAMIC card */
    implicit_modal_dynamic: object;
/** *CONTROL_IMPLICIT_MODAL_DYNAMIC_DAMPING card */
    implicit_modal_dynamic_damping: object;
/** *CONTROL_IMPLICIT_MODES card */
    implicit_modes: object;
/** *CONTROL_IMPLICIT_ORDERING card */
    implicit_ordering: object;
/** *CONTROL_IMPLICIT_RESIDUAL_VECTOR card */
    implicit_residual_vector: object;
/** *CONTROL_IMPLICIT_SOLUTION card */
    implicit_solution: object;
/** *CONTROL_IMPLICIT_SOLVER card */
    implicit_solver: object;
/** *CONTROL_IMPLICIT_SSD_DIRECT card */
    implicit_ssd_direct: object;
/** *CONTROL_IMPLICIT_STABILIZATION card */
    implicit_stabilization: object;
/** *CONTROL_IMPLICIT_STATIC_CONDENSATION card */
    implicit_static_condensation: object;
/** *CONTROL_IMPLICIT_TERMINATION card */
    implicit_termination: object;
/** Selective mass scaling. Part set if negative */
    imscl: number;
/** The Include file number that the control card is in. */
    include: number;
/** The Include file number that the control card is in. */
    include_1: number;
/** The Include file number that the control card is in. */
    include_10: number;
/** The Include file number that the control card is in. */
    include_100: number;
/** The Include file number that the control card is in. */
    include_101: number;
/** The Include file number that the control card is in. */
    include_102: number;
/** The Include file number that the control card is in. */
    include_103: number;
/** The Include file number that the control card is in. */
    include_104: number;
/** The Include file number that the control card is in. */
    include_105: number;
/** The Include file number that the control card is in. */
    include_106: number;
/** The Include file number that the control card is in. */
    include_107: number;
/** The Include file number that the control card is in. */
    include_108: number;
/** the Include file number that the control card is in. */
    include_109: number;
/** The Include file number that the control card is in. */
    include_11: number;
/** The Include file number that the control card is in. */
    include_110: number;
/** The Include file number that the control card is in. */
    include_111: number;
/** The Include file number that the control card is in. */
    include_112: number;
/** The Include file number that the control card is in. */
    include_113: number;
/** The Include file number that the control card is in. */
    include_114: number;
/** The Include file number that the control card is in. */
    include_115: number;
/** The Include file number that the control card is in. */
    include_116: number;
/** The Include file number that the control card is in. */
    include_117: number;
/** The Include file number that the control card is in. */
    include_118: number;
/** The Include file number that the control card is in. */
    include_119: number;
/** The Include file number that the control card is in. */
    include_12: number;
/** The Include file number that the control card is in. */
    include_120: number;
/** The Include file number that the control card is in. */
    include_121: number;
/** The Include file number that the control card is in. */
    include_122: number;
/** The Include file number that the control card is in. */
    include_123: number;
/** The Include file number that the control card is in. */
    include_124: number;
/** The Include file number that the control card is in. */
    include_125: number;
/** The Include file number that the control card is in. */
    include_126: number;
/** The Include file number that the control card is in. */
    include_127: number;
/** The Include file number that the control card is in. */
    include_128: number;
/** The Include file number that the control card is in. */
    include_129: number;
/** The Include file number that the control card is in. */
    include_13: number;
/** The Include file number that the control card is in. */
    include_130: number;
/** The Include file number that the control card is in. */
    include_14: number;
/** The Include file number that the control card is in. */
    include_15: number;
/** The Include file number that the control card is in. */
    include_16: number;
/** The Include file number that the control card is in. */
    include_17: number;
/** The Include file number that the control card is in. */
    include_18: number;
/** The Include file number that the control card is in. */
    include_19: number;
/** The Include file number that the control card is in. */
    include_2: number;
/** The Include file number that the control card is in. */
    include_20: number;
/** The Include file number that the control card is in. */
    include_21: number;
/** The Include file number that the control card is in. */
    include_22: number;
/** The Include file number that the control card is in. */
    include_23: number;
/** The Include file number that the control card is in. */
    include_24: number;
/** The Include file number that the control card is in. */
    include_25: number;
/** The Include file number that the control card is in. */
    include_26: number;
/** The Include file number that the control card is in. */
    include_27: number;
/** The Include file number that the control card is in. */
    include_28: number;
/** The Include file number that the control card is in. */
    include_29: number;
/** The Include file number that the control card is in. */
    include_3: number;
/** The Include file number that the control card is in. */
    include_30: number;
/** The Include file number that the control card is in. */
    include_31: number;
/** The Include file number that the control card is in. */
    include_32: number;
/** The Include file number that the control card is in. */
    include_33: number;
/** The Include file number that the control card is in. */
    include_34: number;
/** The Include file number that the control card is in. */
    include_35: number;
/** The Include file number that the control card is in. */
    include_36: number;
/** The Include file number that the control card is in. */
    include_37: number;
/** The Include file number that the control card is in. */
    include_38: number;
/** The Include file number that the control card is in. */
    include_39: number;
/** The Include file number that the control card is in. */
    include_4: number;
/** The Include file number that the control card is in. */
    include_40: number;
/** The Include file number that the control card is in. */
    include_41: number;
/** The Include file number that the control card is in. */
    include_42: number;
/** The Include file number that the control card is in. */
    include_43: number;
/** The Include file number that the control card is in. */
    include_44: number;
/** The Include file number that the control card is in. */
    include_45: number;
/** The Include file number that the control card is in. */
    include_46: number;
/** The Include file number that the control card is in. */
    include_47: number;
/** The Include file number that the control card is in. */
    include_48: number;
/** The Include file number that the control card is in. */
    include_49: number;
/** The Include file number that the control card is in. */
    include_5: number;
/** The Include file number that the control card is in. */
    include_50: number;
/** The Include file number that the control card is in. */
    include_51: number;
/** The Include file number that the control card is in. */
    include_52: number;
/** The Include file number that the control card is in. */
    include_53: number;
/** The Include file number that the control card is in. */
    include_54: number;
/** The Include file number that the control card is in. */
    include_55: number;
/** The Include file number that the control card is in. */
    include_56: number;
/** The Include file number that the control card is in. */
    include_57: number;
/** The Include file number that the control card is in. */
    include_58: number;
/** The Include file number that the control card is in. */
    include_59: number;
/** The Include file number that the control card is in. */
    include_6: number;
/** The Include file number that the control card is in. */
    include_60: number;
/** The Include file number that the control card is in. */
    include_61: number;
/** The Include file number that the control card is in. */
    include_62: number;
/** The Include file number that the control card is in. */
    include_63: number;
/** The Include file number that the control card is in. */
    include_64: number;
/** The Include file number that the control card is in. */
    include_65: number;
/** The Include file number that the control card is in. */
    include_66: number;
/** The Include file number that the control card is in. */
    include_67: number;
/** The Include file number that the control card is in. */
    include_68: number;
/** The Include file number that the control card is in. */
    include_69: number;
/** The Include file number that the control card is in. */
    include_7: number;
/** The Include file number that the control card is in. */
    include_70: number;
/** The Include file number that the control card is in. */
    include_71: number;
/** The Include file number that the control card is in. */
    include_72: number;
/** The Include file number that the control card is in. */
    include_73: number;
/** The Include file number that the control card is in. */
    include_74: number;
/** The Include file number that the control card is in. */
    include_75: number;
/** The Include file number that the control card is in. */
    include_76: number;
/** The Include file number that the control card is in. */
    include_77: number;
/** The Include file number that the control card is in. */
    include_78: number;
/** The Include file number that the control card is in. */
    include_79: number;
/** The Include file number that the control card is in. */
    include_8: number;
/** The Include file number that the control card is in. */
    include_80: number;
/** The Include file number that the control card is in. */
    include_81: number;
/** The Include file number that the control card is in. */
    include_82: number;
/** The Include file number that the control card is in. */
    include_83: number;
/** The Include file number that the control card is in. */
    include_84: number;
/** The Include file number that the control card is in. */
    include_85: number;
/** The Include file number that the control card is in. */
    include_86: number;
/** The Include file number that the control card is in. */
    include_87: number;
/** The Include file number that the control card is in. */
    include_88: number;
/** The Include file number that the control card is in. */
    include_89: number;
/** The Include file number that the control card is in. */
    include_9: number;
/** The Include file number that the control card is in. */
    include_90: number;
/** The Include file number that the control card is in. */
    include_91: number;
/** The Include file number that the control card is in. */
    include_92: number;
/** The Include file number that the control card is in. */
    include_93: number;
/** The Include file number that the control card is in. */
    include_94: number;
/** The Include file number that the control card is in. */
    include_95: number;
/** The Include file number that the control card is in. */
    include_96: number;
/** The Include file number that the control card is in. */
    include_97: number;
/** The Include file number that the control card is in. */
    include_98: number;
/** The Include file number that the control card is in. */
    include_99: number;
/** bucket or global smoothing */
    ini: number;
/** Factor needed for the estimation of maximum workspace used during initialization */
    inint: number;
/** Invariant node numbering for shell element */
    inn: number;
/** Flag to invoke output of *SET_NODE data */
    insf: number;
/** Integration method */
    integ: number;
/** Intermittent searching flag for old contacts */
    interm: number;
/** Gauss/Lobatto intg rule switch */
    intgrd: number;
/** Flag for behavior in case of unwanted interp. or extrap. of initial stresses */
    intperr: number;
/** Flag to generate adaptive mesh */
    ioflag: number;
/** 1:gravity 2:binder */
    ioption: number;
/** output of 1D seatbelt created for 2D seatbelt to sbtout */
    ip1dblt: number;
/** Number of strips in U direction */
    ipanelu: number;
/** Number of strips in V direction */
    ipanelv: number;
/** Minimum block size for the Cholesky factorization (for eigmth=101) or Maximum number of iterations (for eigmth=102) */
    iparm1: number;
/** Maximum number of iterations */
    iparm1_1: number;
/** Maximum block size for the Cholesky factorization (for eigmth=101) or Block size (for eigmth=102) */
    iparm2: number;
/** Node set ID */
    iparm3: number;
/** MCMS minimum group/substructure size */
    iparm4: number;
/** Harmonic index */
    iparm5: number;
/** Vector ID for the axis of rotation */
    iparm6: number;
/** output curve data flag */
    ipcurv: number;
/** Flag to print initial timesteps at cycle #1 */
    ipnint: number;
/** Print flag for RBDOUT and MATSUM files */
    iprtf: number;
/** curve factor */
    irad: number;
/** rate effect switch */
    irate: number;
/** Lowest IRCNT modes */
    ircnt: number;
/** Flag to activate trimming of a multi-layer sandwiched part */
    irefine: number;
/** Uniform refinement level. Loadcurve if negative */
    ireflg: number;
/** Automatic control flag */
    irelal: number;
/** Converting the attachment modes to residual vectors flag */
    iresvec: number;
/** Residual vector control flag */
    iresvec_1: number;
/** Treatment of revolute joints */
    irevol: number;
/** Flag to revert the spot weld thinning behavior */
    irevspt: number;
/** Inertia relief flag */
    irflag: number;
/** Initial reference geometry calc flag */
    irgen: number;
/** Flag to remove initially interpenetrated particles */
    irmv: number;
/** Hughes-Liu shell normal update option */
    irnxx: number;
/** intg rule */
    irquad: number;
/** dt calc method for 4 noded shells */
    isdo: number;
/** Continuity level in applying interface linking data */
    isfcnt: number;
/** Shell element formulation for implicit */
    ishell: number;
/** apply shifting algorithm */
    ishift: number;
/** display option */
    ishow: number;
/** Controls tasks performed in single precision */
    isingle: number;
/** Flag for skipping the calculation of contact force between DES: */
    iskip: number;
/** Initial penetration check flag */
    islchk: number;
/** Flag to check for a NaN in force and moment arrays */
    isnan: number;
/** Solid element formulation for implicit */
    isolid: number;
/** Flag to invoke output of *SET_SOLID data */
    isolsf: number;
/** Symmetric positive definite flag */
    ispd: number;
/** Treatment of spherical joints */
    ispher: number;
/** kernel function */
    ispline: number;
/** Flag to invoke output of *SET_SHELL data */
    issf: number;
/** Steady state vibration flag */
    issflg: number;
/** stabilisation type */
    istab: number;
/** Initial stiffness formulation flag */
    istif: number;
/** Stoning calculation option */
    istone: number;
/** Flag to compute stresses */
    istress: number;
/** Shell thickness change option */
    istupd: number;
/** flag for switching between formulations 16 and 30. */
    iswshl: number;
/** symmetry option. Node set if negative */
    isym: number;
/** percentage of sph */
    isymp: number;
/** Optimum equilibrium iteration count per timestep */
    iteopt: number;
/** iter keyword option */
    iter: number;
/** base name of two files for the ITER keyword option */
    iterfile: string;
/** Allowable iteration window (no. of iterations) */
    itewin: number;
/** thermal contact heat transfer mode */
    ithcnt: number;
/** contact thickness option */
    ithk: number;
/** Flag for offsetting thermal contact surfaces for thick thermal shells */
    ithoff: number;
/** Flag for offsetting thermal contact surfaces for thick thermal shells */
    ithoff_1: number;
/** Treatment of "Real time" on *DEFIN_CONSTRUCTION_STAGES */
    itime: number;
/** Refinement option for enclosed area of trim curve */
    itriopt: number;
/** Initial thermal timestep */
    its: number;
/** Initial thermal time step size */
    its_1: number;
/** initial transverse shear stress */
    itsflg: number;
/** Thick shell element formulation for implicit */
    itshell: number;
/** Turbulence model */
    iturb: number;
/** Set type */
    itype: number;
/** Set type designation */
    itype_1: number;
/** Hardware specific cost profile */
    itype_2: number;
/** Database format */
    itype_3: number;
/** Velocity switching flag */
    ivel: number;
/** internal variable transfer in adaptive EFG */
    ivt: number;
/** Generalized joint stiffness formulation */
    jntf: number;
/** Strength coefficient for exponential hardening */
    k: number;
/** Strength coefficient for exponential hardening */
    k_1: number;
/** Thermal conductivity of fluid between the contact surfaces */
    k_2: number;
/** number of explicit cycles before switch */
    kcycle: number;
/** keep contact segs of failed shells */
    keepcs: number;
/** Terminate based on kinetic energy */
    ketol: number;
/** number of failed implicit attempts before switch */
    kfail: number;
/** Flag to output kinetic energy density as a nodal field */
    kineng: number;
/** size of krylov space */
    kssize: number;
/** Heat conductivity matrix. Constant float value if respective KxxTYP = 0,  Curve ID if respective KxxTYP = 1. */
    kxx: number;
/** Types of Kxx. Valid values: 0-Constant, 1-Temperature vs Heat Conductivity Curve. */
    kxxtyp: number;
/** Heat conductivity matrix. Constant float value if respective KxyTYP = 0,  Curve ID if respective KxyTYP = 1. */
    kxy: number;
/** Types of Kxy. Valid values: 0-Constant, 1-Temperature vs Heat Conductivity Curve. */
    kxytyp: number;
/** Heat conductivity matrix. Constant float value if respective KxzTYP = 0,  Curve ID if respective KxzTYP = 1. */
    kxz: number;
/** Types of Kxz. Valid values: 0-Constant, 1-Temperature vs Heat Conductivity Curve. */
    kxztyp: number;
/** Heat conductivity matrix. Constant float value if respective KyxTYP = 0,  Curve ID if respective KyxTYP = 1. */
    kyx: number;
/** Types of Kyx. Valid values: 0-Constant, 1-Temperature vs Heat Conductivity Curve. */
    kyxtyp: number;
/** Heat conductivity matrix. Constant float value if respective KyyTYP = 0,  Curve ID if respective KyyTYP = 1. */
    kyy: number;
/** Types of Kyy. Valid values: 0-Constant, 1-Temperature vs Heat Conductivity Curve. */
    kyytyp: number;
/** Heat conductivity matrix. Constant float value if respective KyzTYP = 0,  Curve ID if respective KyzTYP = 1. */
    kyz: number;
/** Types of Kyz. Valid values: 0-Constant, 1-Temperature vs Heat Conductivity Curve. */
    kyztyp: number;
/** Heat conductivity matrix. Constant float value if respective KzxTYP = 0,  Curve ID if respective KzxTYP = 1. */
    kzx: number;
/** Types of Kzx. Valid values: 0-Constant, 1-Temperature vs Heat Conductivity Curve. */
    kzxtyp: number;
/** Heat conductivity matrix. Constant float value if respective KzyTYP = 0,  Curve ID if respective KzyTYP = 1. */
    kzy: number;
/** Types of Kzy. Valid values: 0-Constant, 1-Temperature vs Heat Conductivity Curve. */
    kzytyp: number;
/** Heat conductivity matrix. Constant float value if respective KzzTYP = 0,  Curve ID if respective KzzTYP = 1. */
    kzz: number;
/** Types of Kzz. Valid values: 0-Constant, 1-Temperature vs Heat Conductivity Curve. */
    kzztyp: number;
/** Laminated shell theory update flag */
    lamsht: number;
/** Flag to truncate curves: 0 = no truncation; otherwise = truncate */
    lcacc: number;
/** Loadcurve: Adaptive interval vs time */
    lcadp: number;
/** Loadcurve ID defining modal damping coefficient */
    lcdam: number;
/** Loadcurve for scaling friction forces */
    lcdmu: number;
/** Loadcurve for scaling friction forces during dynamic relaxation */
    lcdmur: number;
/** Load curve number for dynamic coefficient of friction as a function of temperature */
    lcfdt: number;
/** Load curve number for static coefficient of friction as a function of temperature */
    lcfst: number;
/** Load curve ID for h (can be curve ID or function ID) */
    lch: number;
/** Load curve ID defining thickness vs distance */
    lcid: number;
/** The Curve ID specifying Temperature vs Time. */
    lcid_1: number;
/** Number of points in load curve discretization */
    lcint: number;
/** Loadcurve of maximum refinement level vs. time */
    lclvl: number;
/** Matrix assembly package */
    lcpack: number;
/** Loadcurve: shear response vs. shell size */
    lcs: number;
/** Loadcurve for stress-strain relationship */
    lcss: number;
/** Loadcurve for stress-strain relationship */
    lcss_1: number;
/** Loadcurve: tension response vs. shell size */
    lct: number;
/** Loadcurve: Max timestep vs time */
    lctm: number;
/** Loadcurve: timestep vs time */
    lcts: number;
/** Type of load curve */
    lctyp: number;
/** Length of the stone */
    length: number;
/** m = meter, mm = millimeter, cm = centimeter, in = inch, ft = foot */
    length_1: string;
/** Number of meters in the length unit for the input deck */
    length_scale: number;
/** Left end point finite flag */
    lflag: number;
/** Left end point of interval */
    lftend: number;
/** Part set ID related to *INITIAL_STRESS_SHELL. */
    lispsid: number;
/** No thermal contact if gap is greater than this value */
    lmax: number;
/** Switch explicit/implicit joint formulation */
    lmf: number;
/** Minimum gap */
    lmin: number;
/** LCID that defines the function for normal stiffness vs norm pen ratio */
    lnorm: number;
/** Flag to activate an element csys. Valid values: 0-Vecids are considered in Global csys, 1-Vecids are considered in Local Csys. */
    local: number;
/** Structural loss factor */
    loss: number;
/** Linear solver print flag */
    lprint: number;
/** search direction */
    lsdir: number;
/** LCID that defines the function for shear stiffness vs norm pen ratio */
    lshear: number;
/** search method */
    lsmtd: number;
/** Linear equation solver method */
    lsolvr: number;
/** Line search convergence tolerance */
    lstol: number;
/** lump boundary condition */
    lumpbc: number;
/** Maximum adaptive level */
    lvlada: number;
/** Maximum adaptive level */
    lvlada_1: number;
/** Coupling method. */
    maccpl: number;
/** Acoustic nodal motions will be calculated or not. */
    macdvp: boolean;
/** *CONTROL_MAPPING_SHIFT_BY_DT card */
    mapping_shift_by_dt: object;
/** Optional h-refinement. */
    masehrf: number;
/** Approach to element time step calculation. */
    maseigx: number;
/** Dump flag for h-refined and spectral element meshes. */
    masekfl: number;
/** Spectral element integration order. */
    maseord: number;
/** Flag to output a high-resolution plot state form. */
    maseplt: number;
/** kg = kilogram, g = gram, mg = milligram, lb = pound, slug = pound x sec2/foot, slinch = pound x sec2/inch, mtrc_ton = metric_ton */
    mass: string;
/** Number of kilograms in the mass unit for the input deck */
    mass_scale: number;
/** Detailed material energies flag */
    maten: number;
/** Max number of elements for adaptivity */
    maxel: number;
/** Element ID number */
    maxide: number;
/** Node ID number */
    maxidn: number;
/** max number of iterations */
    maxiter: number;
/** Maximum number of iterations 2 */
    maxitr: number;
/** Maximum number of iterations */
    maxitr_1: number;
/** Max #refinement levels */
    maxlvl: number;
/** Number of neighbors to be tracked for DES contact and capillary force calculation */
    maxnei: number;
/** Stiffness reformation limit per time step */
    maxref: number;
/** max velocity */
    maxv: number;
/** Flag for writing out MCF (Modal Coefficient File) from SSD analysis */
    mcf: number;
/** Modal dynamic stress flag */
    md_strs: number;
/** Modal dynamic flag */
    mdflag: number;
/** Last mode employed in FRF computation */
    mdmax: number;
/** First mode employed in FRF computation */
    mdmin: number;
/** %age increase in memory for *MAT_NONLOCAL usage */
    mem: number;
/** Control the load-balancing behavior */
    memlvl: number;
/** Memory limit beyond which adaptivity will cease */
    memory: number;
/** memory alloc */
    memory_1: number;
/** metalforming option */
    metalf: number;
/** Advection method */
    meth: number;
/** Stoning method */
    method: number;
/** iterative method */
    method_1: number;
/** mass ratio gain required for remesh */
    mfrac: number;
/** Mid-plane position flag */
    midsf: number;
/** Output penetration information */
    minfo: number;
/** Factor of the minimum mass allowed in an element */
    minmas: number;
/** Plane stress plasticity option */
    miter: number;
/** Flag to invoke output of accumulated airbag mass leakage in ABSTAT */
    mlkbag: number;
/** monotonic mesh resizing */
    mm: number;
/** Selects the method that is used to include a reference pressure in a calculation involving ALE multi-material groups */
    mmgpref: number;
/** Option for merging common boundaries of all adapted materials */
    mmm2d: number;
/** The Multi-material Set ID. */
    mmset: number;
/** Large mass added per node. */
    mpn: number;
/** *CONTROL_MPP_CONTACT_GROUPABLE card */
    mpp_contact_groupable: object;
/** *CONTROL_MPP_DECOMPOSITION_AUTOMATIC card */
    mpp_decomposition_automatic: object;
/** *CONTROL_MPP_DECOMPOSITION_BAGREF card */
    mpp_decomposition_bagref: object;
/** *CONTROL_MPP_DECOMPOSITION_CHECK_SPEED card */
    mpp_decomposition_check_speed: object;
/** *CONTROL_MPP_DECOMPOSITION_CONTACT_ISOLATE card */
    mpp_decomposition_contact_isolate: object;
/** *CONTROL_MPP_DECOMPOSITION_DISABLE_UNREF_CURVES card */
    mpp_decomposition_disable_unref_curves: object;
/** *CONTROL_MPP_DECOMPOSITION_DISTRIBUTE_ALE_ELEMENTS card */
    mpp_decomposition_distribute_ale_elements: object;
/** *CONTROL_MPP_DECOMPOSITION_DISTRIBUTE_SPH_ELEMENTS card */
    mpp_decomposition_distribute_sph_elements: object;
/** *CONTROL_MPP_DECOMPOSITION_ELCOST card */
    mpp_decomposition_elcost: object;
/** *CONTROL_MPP_DECOMPOSITION_FILE card */
    mpp_decomposition_file: object;
/** *CONTROL_MPP_DECOMPOSITION_FLAG_STRESS_STRAIN_CURVE card */
    mpp_decomposition_flag_stress_strain_curve: object;
/** *CONTROL_MPP_DECOMPOSITION_METHOD card */
    mpp_decomposition_method: object;
/** *CONTROL_MPP_DECOMPOSITION_NUMPROC card */
    mpp_decomposition_numproc: object;
/** *CONTROL_MPP_DECOMPOSITION_OUTDECOMP card */
    mpp_decomposition_outdecomp: object;
/** *CONTROL_MPP_DECOMPOSITION_RCBLOG card */
    mpp_decomposition_rcblog: object;
/** *CONTROL_MPP_DECOMPOSITION_SCALE_CONTACT_COST card */
    mpp_decomposition_scale_contact_cost: object;
/** *CONTROL_MPP_DECOMPOSITION_SCALE_FACTOR_SPH card */
    mpp_decomposition_scale_factor_sph: object;
/** *CONTROL_MPP_DECOMPOSITION_SHOW card */
    mpp_decomposition_show: object;
/** *CONTROL_MPP_DECOMPOSITION_TRANSFORMATION card */
    mpp_decomposition_transformation: object;
/** *CONTROL_MPP_IO_BINOUTONLY card */
    mpp_io_binoutonly: object;
/** *CONTROL_MPP_IO_LSTC_REDUCE card */
    mpp_io_lstc_reduce: object;
/** *CONTROL_MPP_IO_NOD3DUMP card */
    mpp_io_nod3dump: object;
/** *CONTROL_MPP_IO_NODUMP card */
    mpp_io_nodump: object;
/** *CONTROL_MPP_IO_NOFAIL card */
    mpp_io_nofail: object;
/** *CONTROL_MPP_IO_NOFULL card */
    mpp_io_nofull: object;
/** *CONTROL_MPP_IO_SWAPBYTES card */
    mpp_io_swapbytes: object;
/** *CONTROL_MPP_MATERIAL_MODEL_DRIVER card */
    mpp_mat_model_driver: object;
/** *CONTROL_MPP_REBALANCE card */
    mpp_rebalance: object;
/** Limit mass scaling to 1st timestep flag */
    ms1st: number;
/** Option for printing detail message to d3msg */
    msgflg: number;
/** Output iteration message level */
    msgitr: number;
/** Output message level */
    msglvl: number;
/** max num messags */
    msgmax: number;
/** stress compute flag */
    mstres: number;
/** Scaling for computing velocity */
    mstrscl: number;
/** flag to dump matrix */
    mtxdmp: number;
/** Material type */
    mtyp: number;
/** Material type */
    mtype: number;
/** Matrix dumping. */
    mxdmp: number;
/** Refinement option */
    n: number;
/** Optional seed node ID 1 */
    n1: number;
/** Node (n1typ=0) / node set (n1typ=1) /segment set (n1typ=2) ID for excitation input */
    n1_1: number;
/** Type of N1 */
    n1typ: number;
/** Optional seed node ID 2 */
    n2: number;
/** Node (n2typ=0) /node set (n2typ=1) /segment set (n2typ=2) ID for response output */
    n2_1: number;
/** Type of N2 */
    n2typ: number;
/** Optional seed node ID 3 */
    n3: number;
/** Optional seed node ID 4 */
    n4: number;
/** Optional seed node ID 5 */
    n5: number;
/** Optional seed node ID 6 */
    n6: number;
/** Optional seed node ID 7 */
    n7: number;
/** Optional seed node ID 8 */
    n8: number;
/** Exponent for exponential hardening */
    n_1: number;
/** Exponent for exponential hardening */
    n_2: number;
/** number of processors */
    n_3: number;
/** Number of cycles between advections */
    nadv: number;
/** decomposition file */
    name: string;
/** decomposition method */
    name_1: string;
/** decomposition file */
    name_2: string;
/** Start node ID on a flange root boundary */
    nb1: number;
/** ID of a node in the middl of the flange root boundary */
    nb2: number;
/** End node ID on a flange root boundary */
    nb3: number;
/** Number of Lagrangian cycles between bucket sort searches */
    nbkt: number;
/** Asynchronous scheme and memory buffer option */
    nbuf: number;
/** Number of cycles between particle sorting */
    ncbs: number;
/** Cycle number at which to evaluate DEFINE_CURVE_FUNCTION */
    ncdcf: number;
/** Frequency of fission to fusion steps */
    ncfreq: number;
/** Number of cycles between point cloud checks. A negative value will point to a load curve ID function of time */
    ncpc: number;
/** Number of Lagrangian cycles between coupling calculations */
    ncpl: number;
/** Control CPM output database to D3PLOT */
    ncpmout: number;
/** Timestep size estimation */
    ncpmts: number;
/** #cpus to use */
    ncpu: number;
/** Rebalancing frequency */
    ncrb: number;
/** Thermal matrix reassembly frequency */
    ncycl: number;
/** Number of cycle between checks of new contact. */
    ncycle: number;
/** Number of cycles between rebalance steps */
    ncycle_1: number;
/** normal damping coefficient */
    ndamp: number;
/** A node ID on the outer flange boundary */
    ndouter: number;
/** Reform the dielectric stiffness matrix every NDTRFK time steps */
    ndtrfk: number;
/** Print suppression during input: echo file */
    neecho: number;
/** Negative eigenvalue flag */
    negev: number;
/** #eigenvalues to extract; loadcurve if negative */
    neig: number;
/** Number of eigenmodes */
    neig_1: number;
/** Number of eigenmodes to compute for the purpose of orthogonalizing the computed load */
    neig_2: number;
/** Number of eigen values to compute. */
    neig_3: number;
/** New legends */
    newleg: number;
/** Flag for distorted 1 intg point shell check */
    nfail1: number;
/** Flag for distorted 4 intg point shell check */
    nfail4: number;
/** Number of frequencies for FRF output */
    nfreq: number;
/** Number of frequencies in the solution */
    nfreq_1: number;
/** Number of inner iterations for GMRES solve */
    ninner: number;
/** #intg points for quadratic tets */
    niptets: number;
/** non-linear convergence type */
    nlnorm: number;
/** flag for time step handling for shell elements with offset. */
    nlocdt: number;
/** non-linear solver print flag */
    nlprint: number;
/** Vector length */
    nlq: number;
/** Thermal nonlinear printout level */
    nlthpr: number;
/** Number of modes in modal stress/strain output */
    nmdstr: number;
/** Number of times to use Metis */
    nmetis: number;
/** memory alloc */
    nmneigh: number;
/** number of buckling modes to calculate */
    nmode: number;
/** Avoid copying material history variables to temporary buffers for constitutive evaluations */
    nocopy: number;
/** Tail node defining stone moving direction */
    node1: number;
/** Head node defining stone moving direction */
    node1_1: number;
/** *CONTROL_NONLOCAL card */
    nonlocal: object;
/** Treatment of pressure loads on deleted elements */
    nopdel: number;
/** Flag to turn on unfolding simulation */
    noption: number;
/** Circumvent rigid body inertia check */
    norbic: number;
/** scale factor for normal spring constant */
    normk: number;
/** flag for non-solution run */
    nosol: number;
/** Number of outer iterations for GMRES solve */
    nouter: number;
/** Number of cycles for repartition particles */
    np2p: number;
/** Print suppression during input: printer file */
    npopt: number;
/** Normal direction of blank */
    nrbst: number;
/** #iterations between convergence checks */
    nrcyck: number;
/** Flag to update individual beam 3rd nodes */
    nrefup: number;
/** Nodal rotational convergence tolerance */
    nrtol: number;
/** Normal direction of tool */
    nrtst: number;
/** #cycles between 3D bucket sorts */
    nsbcs: number;
/** Number of steps in non-linear springback */
    nsbs: number;
/** #extra "seed" nodes below */
    nseed: number;
/** erosion option */
    nserod: number;
/** An optional node set ID of three nodes from the source mesh */
    nsets: number;
/** An optional node set ID of three nodes from the target mesh */
    nsett: number;
/** Node set ID of the nodes in the modal model that are subjected to loads */
    nsid: number;
/** node set for attachment modes */
    nsida: number;
/** node set constraint modes */
    nsidc: number;
/** Optional excluded node set */
    nsidebc: number;
/** Optional skipping scheme */
    nskip: number;
/** Slip condition applied at walls */
    nslip: number;
/** max number of implicit steps */
    nsmax: number;
/** min number of implicit steps */
    nsmin: number;
/** Non-linear equation solver method */
    nsolvr: number;
/** Consecutive implicit time steps */
    nstep: number;
/** Nodal translational convergence tolerance */
    nttol: number;
/** #rh sides written */
    numrhs: number;
/** Estimated number of unbending */
    nunbend: number;
/** Relaxation parameter */
    omega: number;
/** Output interval for interface file */
    opifs: number;
/** Can be &lt;BLANK&gt; or _BINARY */
    opt: number;
/** Optimize the MPP communications (Range 0/1) */
    optimpp: number;
/** Keyword option */
    option: number;
/** Ordering option */
    order: number;
/** Ordering option */
    order_1: number;
/** Automatic contact segment orientation flag */
    orien: number;
/** Flag to set the global orientation of a forming contact */
    orient: number;
/** Orthogonalise modes wrt each other */
    orthmd: number;
/** Objective stress update for large timestep */
    osu: number;
/** *CONTROL_OUTPUT card */
    output: object;
/** Output Flag */
    output_1: number;
/** TRUE if _&lt;OPTION&gt; is OUTPUT. */
    output_2: boolean;
/** Output flag for stresses */
    output_3: number;
/** Spotweld output flag */
    outseg: number;
/** Decompose the structure and ALE domains together? */
    overlap: boolean;
/** Control the energy transfer during particle-to-particle collision */
    p2pmix: number;
/** Flag for parallel force assembly */
    para: number;
/** *CONTROL_PARALLEL card */
    parallel: object;
/** Option to force calculation of bonded DES */
    parallel_1: number;
/** Use global mass matrix for mass distribution */
    partm: number;
/** The Part Set ID. */
    partset: number;
/** The Part Set ID. */
    partset_1: number;
/** The  Part Set ID. */
    partset_2: number;
/** The Part Set ID. */
    partset_3: number;
/** Velocity profile of moving tool */
    patern: number;
/** Velocity profile of moving tool */
    patern_1: number;
/** Max pressure difference for stress zeroing */
    pdifmx: number;
/** Default local penalty scale factor */
    pen_sf: number;
/** penetration allowed as ratio of part thickness */
    penchk: number;
/** Penalty stiffness option flag */
    penopt: number;
/** Flag to output contact penetration for mortar contact */
    penout: number;
/** Default bulk modulus of pore fluid */
    pf_bulk: number;
/** Default pore water density */
    pf_rho: number;
/** Phase number */
    phase: number;
/** Phase change penalty parameter */
    phchpn: number;
/** Message to messag file for phase change on materials 216, 217 and 218 */
    phschng: number;
/** Part ID */
    pid: number;
/** Part ID of the sheet blank */
    pid_1: number;
/** Part ID of the sheet blank */
    pid_2: number;
/** Part id to remove adaptive constraints from */
    pid_3: number;
/** Part id of the thin shell elements */
    pid_4: number;
/** Part ID of tool */
    pid_5: number;
/** Part id for blank */
    pidb: number;
/** Part set id for objective stress updates */
    pidosu: number;
/** Part id for tool */
    pidt: number;
/** Automatic generation of *ELEMENT_PLOTEL */
    plotel: number;
/** 10 noded tetrahedral solid node ID 1 */
    pm1: number;
/** 10 noded tetrahedral solid node ID 10 */
    pm10: number;
/** 10 noded tetrahedral solid node ID 2 */
    pm2: number;
/** 10 noded tetrahedral solid node ID 3 */
    pm3: number;
/** 10 noded tetrahedral solid node ID 4 */
    pm4: number;
/** 10 noded tetrahedral solid node ID 5 */
    pm5: number;
/** 10 noded tetrahedral solid node ID 6 */
    pm6: number;
/** 10 noded tetrahedral solid node ID 7 */
    pm7: number;
/** 10 noded tetrahedral solid node ID 8 */
    pm8: number;
/** 10 noded tetrahedral solid node ID 9 */
    pm9: number;
/** Flag for choosing logic to use when a particle leaks out due to undetected contact */
    pmis: number;
/** Part that defines the punch */
    pnch: number;
/** *CONTROL_PORE_AIR card */
    pore_air: object;
/** *CONTROL_PORE_FLUID card */
    pore_fluid: object;
/** Poissons ratio */
    pr: number;
/** Poissons ratio */
    pr_1: number;
/** Distance between lower binder and punch */
    prebd: number;
/** ref pressure on boundary */
    pref: number;
/** Distance to pre-move tool in reverse direction */
    premove: number;
/** Pressure equalibrium flag */
    prit: number;
/** Projection method for warping stiffness */
    proj: number;
/** Flag to print data for spotwelds */
    prtflg: number;
/** Part set ID */
    pset: number;
/** Optional part set id */
    psfail: number;
/** excluded part set */
    psid: number;
/** Part set ID */
    psid_1: number;
/** part set id for check */
    psnfail: number;
/** part set for thichness update, -ve to exclude */
    psstupd: number;
/** method for penalty stiff calc */
    pstiff: number;
/** scale factor on the contact stress exerted onto shells */
    ptscl: number;
/** Thermal problem type */
    ptype: number;
/** Thermal problem type */
    ptype_1: number;
/** *CONTROL_PWP_AUTO_TMF card */
    pwp_auto_tmf: object;
/** *CONTROL_PZELECTRIC card */
    pzelectric: object;
/** Default linear viscosity coefficient */
    q1: number;
/** Default quadratic viscosity coefficient */
    q2: number;
/** Default hourglass coefficient */
    qh: number;
/** quasi-linear coefficient */
    ql: number;
/** Material anisotropic parameter R00 */
    r00: number;
/** Material anisotropic parameter R00 */
    r00_1: number;
/** Material anisotropic parameter R45 */
    r45: number;
/** Material anisotropic parameter R45 */
    r45_1: number;
/** Material anisotropic parameter R90 */
    r90: number;
/** Material anisotropic parameter R90 */
    r90_1: number;
/** Radius of pre-bending */
    radius: number;
/** Flag to apply consistent treatment of rigid bodies in selective mass scaling */
    rbsms: number;
/** Residual (force) convergence tolerance */
    rctol: number;
/** Recover the lead rigid body of constrained rigid bodies */
    rcvlr2d: number;
/** Factor for capping the amount of dynamic memory requested */
    rdcmem: number;
/** *CONTROL_REFERENCE_CONFIGURATION card */
    ref_config: object;
/** Flag for reference geometry in acoustic eigenvalue analysis */
    refgeo: number;
/** Max #matrix reformations per timestep */
    refmax: number;
/** Relative convergence tolerance */
    reltol: number;
/** Relative convergence tolerance */
    reltol_1: number;
/** *CONTROL_REMESHING card */
    remesh: object;
/** Flag to remove deactivated SPH particles */
    remsph: number;
/** Restart option */
    restrt: number;
/** Restart option */
    restrt_1: number;
/** Surface normal reversing option */
    reverse: number;
/** Right end point finite flag */
    rflag: number;
/** Right end point of interval */
    rhtend: number;
/** *CONTROL_RIGID card */
    rigid: object;
/** Option to compute rotational inertia for the nodes of solid elements */
    rinrt: number;
/** Maximum edge length */
    rmax: number;
/** Minimum edge length. Loadcurve if negative. */
    rmin: number;
/** flag to activate scaling of rotational inertia */
    rmscl: number;
/** Maximum residual convergence tolerance */
    rmtol: number;
/** Deactivation criteria */
    rol: number;
/** Scale factor for rotary shell mass */
    rotascl: number;
/** Scale factor for the inertia of rotational degrees of freedom */
    rotscl: number;
/** Number of rows. */
    rows: number;
/** Eigenvalue expansion factor (for eigmth=101) or Convergence tolerance (for eigmth=102) */
    rparm1: number;
/** Absolute tolerance for convergence */
    rparm1_1: number;
/** BLR preconditioner tolerance */
    rparm2: number;
/** Relative tolerance for convergence */
    rparm2_1: number;
/** Control output of eigenvectors to the d3eigv database */
    rparm4: number;
/** Nominal block size */
    rparm4_1: number;
/** Compression tolerance for the low-rank approximation */
    rparm5: number;
/** Replace each beam with a cluster of RPBHX solids */
    rpbhx: number;
/** Rigid body rotational convergence tolerance */
    rrtol: number;
/** Rigid body translational convergence tolerance */
    rttol: number;
/** Rigid wall energy calc flag */
    rwen: number;
/** flag for gap stiffness */
    rwgaps: number;
/** death time for gap stiffness */
    rwgdth: number;
/** penalty scale factor */
    rwksf: number;
/** Scale factor for rigid wall penalties */
    rwpnal: number;
/** Rayleigh energy calc flag */
    rylen: number;
/** Time interval for collecting element cost profile to use in the next REDECOMP step. */
    sampt: number;
/** Stefan Boltzman constant (w/m**2/K) */
    sbc: number;
/** Static condensation control flag */
    sc_flag: number;
/** Node set ID for nodes to be preserved in the procedure */
    sc_nsid: number;
/** Part set ID for parts to be included in the procedure */
    sc_psid: number;
/** scale factor for artificial stabilization. Loadcurve if negative */
    scale: number;
/** Scale factor for friction stiffness */
    scl_k: number;
/** Name of superelement damping matrix */
    se_damp: string;
/** File name */
    se_filename: string;
/** File name */
    se_filename_1: string;
/** Name of superelement inertia matrix */
    se_inert: string;
/** Name of superelement inertia matrix */
    se_inert_1: string;
/** Name of superelement mass matrix */
    se_mass: string;
/** Name of superelement mass matrix */
    se_mass_1: string;
/** Name of superelement stiffness matrix */
    se_stiff: string;
/** Name of superelement stiffness matrix */
    se_stiff_1: string;
/** angular mesh size in 3-D axisymmetric remeshing */
    segang: number;
/** The  Set ID. Can be SOLID, SHELL or BEAM Set based on value of setyp. */
    set: number;
/** true if _SET option is present. */
    set_option: boolean;
/** Set ID of *SET_SHELL or *SET_SOLID. */
    setid: number;
/** Type of Set. Valid values: 1-Solid Set, 2-Shell Set, 3-Beam Set. */
    setyp: number;
/** Scale factor */
    sf: number;
/** Scale factor */
    sf_1: number;
/** Scale factor of force decay constant */
    sffdc: number;
/** Default static coefficient of friction */
    sfric: number;
/** The Segment Set ID. */
    sgset: number;
/** ratio between sheark/normk */
    sheark: number;
/** *CONTROL_SHELL card */
    shell: object;
/** Shift scale */
    shfscl: number;
/** Flag for assuming edge shape for shells */
    shledg: number;
/** Flag to extrapolate stresses for shells with 8 integration points to nodes */
    shlsig: number;
/** Shell thickness consideration flag */
    shlthk: number;
/** Shell thickness scale factor */
    shltrw: number;
/** Node/Shell set id */
    sid: number;
/** part set for type 4 thickness update where elastic strains are ignored. */
    sidt4tu: number;
/** Minimum element size permitted in the adaptive mesh */
    sizeada: number;
/** Minimum element size permitted in the adaptive mesh */
    sizeada_1: number;
/** Display rigidwall flag */
    skiprwg: number;
/** Contact energy calc flag */
    slnten: number;
/** Scale factor for sliding penalties */
    slsfac: number;
/** Maximum element size */
    smax: number;
/** Element dimension limit for refining */
    smin: number;
/** Scale factor applied to the contact stiffness */
    sofscl: number;
/** *CONTROL_SOLID card */
    solid: object;
/** TRUE if a plain (no _SHELL suffix) card exists */
    solitary: number;
/** Solution type flag */
    soln: number;
/** Flag to extrapolate stresses/history variables */
    solsig: number;
/** *CONTROL_SOLUTION card */
    solution: object;
/** Piezoelectric solver type */
    solver: number;
/** Thermal analysis solver type */
    solver_1: number;
/** Thermal analysis solver type */
    solver_2: number;
/** Use sparse xply routines for modal &amp; stiffness damping matrices */
    sparse: number;
/** Flag to convert constraints on rigid bodies to equivalent *BOUNDARY_PRESCRIBED_MOTION_RIGID motion */
    spc2bnd: number;
/** If SPECIFIC option is used */
    specific: boolean;
/** *CONTROL_SPH card */
    sph: object;
/** *CONTROL_SPH_INCOMPRESSIBLE card */
    sph_incompressible: object;
/** sort and move SPH */
    sphsort: number;
/** Spotweld deletion flag */
    spotdel: number;
/** Optional thickness scale factor */
    spothin: number;
/** Error termination flag on unfound spotweld */
    spotstp: number;
/** *CONTROL_SPOTWELD_BEAM card */
    spotweld_beam: object;
/** Flag to check and report open edge of CV constrained */
    sprchk: number;
/** factor for reducing feedback */
    sprfac: number;
/** Shear moment distribution behavior for SPR3 */
    sprsmd: number;
/** Search method for SPR2 and SPR3 */
    sprsrch: number;
/** radius of influence */
    srad: number;
/** step reduction factor */
    sred: number;
/** Flag for automatic sort of background triangular shells */
    ssort: number;
/** Shell thickness use flag for type 4 contacts */
    ssthk: number;
/** *CONTROL_STAGED_CONSTRUCTION card */
    staged_construction: object;
/** *CONTROL_START card */
    start: object;
/** Start time for smoothing */
    start_1: number;
/** start time */
    start_2: number;
/** *CONTROL_STEADY_STATE_ROLLING card */
    steady_state_rolling: object;
/** Stepping size of moving stone */
    step: number;
/** step size used in iterations */
    step_1: number;
/** Unflanging stiffness */
    stfbend: number;
/** Normal stiffness */
    stfcnt: number;
/** end stage */
    stge: number;
/** start stage */
    stgs: number;
/** Start time for redecomposition */
    stime: number;
/** ref stage */
    stref: number;
/** Stretch ratio of element diagonals for element deletion */
    stretch: number;
/** *CONTROL_STRUCTURED card */
    structured: object;
/** 0-&gt;blkid is PART, 1-&gt;PARTSET NOTE don't use &lt;type&gt; as in stat_header */
    stype: number;
/** Flag for part/part set */
    stype_1: number;
/** Subcycling flag */
    subcyl: number;
/** output flag for stresses in solid spotwelds */
    swlocl: number;
/** Spot weld radius scale factor */
    swradf: number;
/** tolerance for jacobian in 4-point 10-noded quadratic tetrahedra */
    t10jtol: number;
/** Table ID for scaling shear response */
    t_ors: number;
/** Table for scaling response */
    t_ort: number;
/** target for change of excess pressure */
    targ: number;
/** Target max pwp change/thermal timestep */
    targ_1: number;
/**  */
    target: number;
/** Move tool PID to meet part TARGET */
    target_1: number;
/** file containing all nodes of the target geometry */
    targetfile: string;
/** Tolerance criteria for average relative density */
    tavg: number;
/** Birth time for adaptivity */
    tbirth: number;
/** DES thermal property */
    tc: number;
/** tangential damping coefficient */
    tdamp: number;
/** tied constraint offset contact update option */
    tdcnof: number;
/** Death time for adaptivity */
    tdeath: number;
/** birth time for dynamic terms. Loadcurve if negative */
    tdybir: number;
/** burial */
    tdybur: number;
/** death */
    tdydth: number;
/** K = Kelvin, C = Celsius, F = Fahrenheit, R = Rankine */
    temp: string;
/** Initial Temperature. */
    tempini: number;
/** End time */
    tend: number;
/** _TERM flag */
    term: number;
/** *CONTROL_TERMINATION card */
    termination: object;
/** tet connectivity output */
    tet10s8: number;
/** global flag for cohesive element deletion */
    tet13k: number;
/** Choice of type 13 solid implementation */
    tet13v: number;
/** Terminate based on total energy */
    tetol: number;
/** DES thermal property */
    tfac: number;
/** Default contact thickness */
    th: number;
/** Default thickness scale factor */
    th_sf: number;
/** Shell theory to use */
    theory: number;
/** thermal vol expansion coeff */
    therm: number;
/** *CONTROL_THERMAL_EIGENVALUE card */
    thermal_eigenvalue: object;
/** *CONTROL_THERMAL_FORMING card */
    thermal_forming: object;
/** *CONTROL_THERMAL_NONLINEAR card */
    thermal_nonlinear: object;
/** *CONTROL_THERMAL_SOLVER card */
    thermal_solver: object;
/** *CONTROL_THERMAL_TIMESTEP card */
    thermal_timestep: object;
/** Thickness of the thick shell elements */
    thick: number;
/** Blank thickness */
    thick_1: number;
/** Blank thickness */
    thick_2: number;
/** Consider shell thickness change flag */
    thkchg: number;
/** Line search convergence tolerance */
    thlstl: number;
/** Min thickness below which elements are deleted */
    thmn: number;
/** Max thickness beyond which elements are deleted */
    thmx: number;
/** Percent threshold for rebalancing */
    thres: number;
/** Threshold for rigid body node */
    thresh: number;
/** Thermal shell option */
    thshel: number;
/** Projection bypass flag for TIED types */
    tiedprj: number;
/** sec = second, ms = msec/millisec, micro_s = microsec */
    time: string;
/** Number of seconds in the time unit for the input deck */
    time_scale: number;
/** *CONTROL_TIMESTEP card */
    timestep: object;
/** Idle time value */
    timidl: number;
/** Total number of adaptive steps during the forming simulation */
    timsada: number;
/** Thermal time integration parameter */
    tip: number;
/** Joint translational stiffness */
    tjadstf: number;
/** Joint translational damping */
    tjadvsc: number;
/** Tolerance criteria for maximum relative density */
    tmax: number;
/** Maximum thermal timestep, or LC of tmax vs time if -ve */
    tmax_1: number;
/** Time magnification factor on seepage. Loadcurve if negative */
    tmf: number;
/** Minimum thermal timestep, or LC of tmin vs time if -ve */
    tmin: number;
/** tolerance used to determine convergence */
    tol: number;
/** Convergence tolerance for temperature */
    tol_1: number;
/** Timing output levels */
    tolev: number;
/** Distance to move tool along VID */
    travel: number;
/** Thermal timestep control flag */
    ts: number;
/** Timestep control parameter */
    tscp: number;
/** Thermal speedup factor. Loadcurve if negative */
    tsf: number;
/** Thermal Speedup Factor */
    tsf_1: number;
/** Thermal shell option */
    tshell: number;
/** Min timestep for shell modulus change */
    tslimt: number;
/** Scale factor for computed timestep */
    tssfac: number;
/** Optional timestep factor during DR */
    tssfdr: number;
/** Start time */
    tstart: number;
/** start time */
    tstart_1: number;
/** Thick shell bulk viscosity type */
    tstype: number;
/** Force conversion factor */
    unforc: number;
/** Units for simulation */
    unit: number;
/** Units for simulation */
    unit_1: number;
/** *CONTROL_UNITS card */
    units: object;
/** Length conversion factor */
    unleng: number;
/** Time conversion factor */
    untime: number;
/** Storage for user-controlled friction subroutine */
    usrfrc: number;
/** Storage for user-controlled control subroutine */
    usrstr: number;
/** Vector component defining stoning direction */
    v1: number;
/** Vector component defining stoning direction */
    v2: number;
/** Vector component defining stoning direction */
    v3: number;
/** Excitation input type */
    vad1: number;
/** Response output type */
    vad2: number;
/** Loading type */
    vaflag: number;
/** Flag for PSD broadband plots */
    vaplot: number;
/** Flag for including preload */
    vaprld: number;
/** Flag for PSD output */
    vapsd: number;
/** Variable thermal density flag */
    varden: number;
/** Flag for RMS output */
    varms: number;
/** Flag for including stress analysis */
    vastrs: number;
/** The Vector ID to define x-direction. */
    vecid1: number;
/** The Vector ID to define y-direction. */
    vecid2: number;
/** true if _VECTOR option is set */
    vector: boolean;
/** CPG verbosity control */
    verb: number;
/** Void factor */
    vfact: number;
/** Default viscous friction coefficient */
    vfc: number;
/** necessary VF loss for remesh */
    vfloss: number;
/** *CONTROL_VIBRO_ACOUSTIC card */
    vibro_acoustic: object;
/** Vector ID defining direction of movement */
    vid: number;
/** Vector ID defining direction of travel */
    vid_1: number;
/** Vector ID for DOF1=4 */
    vid_2: number;
/** Maximum allowable tool velocity */
    vmax: number;
/** Maximum allowable tool velocity */
    vmax_1: number;
/** volume fraction */
    vol: number;
/** max number of subcycling cycles */
    vtk: number;
/** X component of vector defining the direction of distance in load curve */
    vx: number;
/** X component of axis about which blank will be bent */
    vx_1: number;
/** X vector component of movement of punch */
    vx_2: number;
/** Y component of vector defining the direction of distance in load curve */
    vy: number;
/** Y component of axis about which blank will be bent */
    vy_1: number;
/** Y vector component of movement of punch */
    vy_2: number;
/** Z component of vector defining the direction of distance in load curve */
    vz: number;
/** Z component of axis about which blank will be bent */
    vz_1: number;
/** Z vector component of movement of punch */
    vz_2: number;
/** Coefficient in equation */
    weight: number;
/** Coefficient in equation */
    weight_1: number;
/** Element cost scale factor for element in contact */
    weight_2: number;
/** Width of the stone */
    width: number;
/** W-mode amplitude for element deletion (deg) */
    wmode: number;
/** Shell warpage angle (deg) */
    wrpang: number;
/** Default elevation of water table */
    wtable: number;
/** Starting position x coordinate */
    x0: number;
/** Surface max penetration check multiplier */
    xpene: number;
/** Starting position y coordinate */
    y0: number;
/** Starting position z coordinate */
    z0: number;
/** flag to zero vels before switch to implicit */
    zero_v: number;
/** Modal dynamic damping constant */
    zeta: number;
/** Modal dynamic damping constant */
    zeta_1: number;
}


/** Object returned by GetSettings */
interface GetSettingsReturn {
    /** Angle tolerance */
    angle_tolerance: number;
    /** Adjust bolt mass when creating bolt entitites */
    bolt_adjust_mass: boolean;
    /** Add database history beam when bolt beam is created */
    bolt_dth_beam: boolean;
    /** Consider feature line for bolt holes */
    bolt_feature_line: boolean;
    /** Bolt rigid NRB minimum mass */
    bolt_nrb_min_mass: number;
    /** Bolt rigid part minimum mass */
    bolt_part_min_mass: number;
    /** Allow connections to join a clinch type connection */
    clinch: boolean;
    /** Turn on consistent weld area so multihex welds are pi*d*d/4 */
    consistent_weld_area: boolean;
    /** Edge distance */
    edge_distance: number;
    /** Glue break angle */
    glue_break_angle: number;
    /** Glue hard aspect ratio */
    glue_hard_aspect: number;
    /** Glue soft aspect ratio */
    glue_soft_aspect: number;
    /** Check length */
    length_check: boolean;
    /** Maximum length */
    max_length: number;
    /** Maximum number of panels */
    max_panels: number;
    /** Maximum warpage */
    max_warpage: number;
    /** Minimum length */
    min_length: number;
    /** Create arcweld using NRBs (rather than contact) */
    nrb_arcweld (optional): boolean;
    /** Check for maximum number of panels */
    panel_check: boolean;
    /** Patch angle setting */
    patch_angle: number;
    /** Turn on or off patch angle check */
    patch_angle_check: boolean;
    /** Allow connections to join a part to itself */
    same_part: boolean;
    /** Consider free edges when orienting single solid spotwelds */
    solid_free_edges: boolean;
    /** Spotweld line search tolerance */
    spot_line_tol: number;
    /** Search thickness */
    spot_thickness: number;
    /** Total length */
    total_length: number;
    /** Use _PID for beam connections */
    use_pid: boolean;
    /** Check warpage value */
    warpage_check: boolean;
}


/** Object returned by GetShellThickness */
interface GetShellThicknessReturn {
    /** Shell object in the specified layer in contact with the nth node the solid adhesive element, where n can be a value from 1 to 8. This corresponds to the maximum number of nodes in the solid element. */
    shell<i>n</i>: Shell;
    /** Solid object associated with the shells in the return object. */
    solid: Solid;
    /** Thickness of shell in the specified layer in contact with the nth node of the solid adhesive element, where n can be a value from 1 to 8. */
    sthk<i>n</i>: Double;
}


/** Object function argument in SetSettings */
interface SetSettingsArgument_data {
    /** Angle tolerance */
    angle_tolerance?: number;
    /** Adjust bolt mass when creating bolt entities */
    bolt_adjust_mass?: boolean;
    /** Add database history beam when bolt is created */
    bolt_dth_beam?: boolean;
    /** Consider feature line for bolt holes */
    bolt_feature_line?: boolean;
    /** Bolt NRB minimum mass */
    bolt_nrb_min_mass?: number;
    /** Bolt rigid part minimum mass */
    bolt_part_min_mass?: number;
    /** Allow connections to join a clinch type connection */
    clinch?: boolean;
    /** Use consistent area for multihex welds */
    consistent_weld_area?: boolean;
    /** Edge distance */
    edge_distance?: number;
    /** Glue break angle */
    glue_break_angle?: number;
    /** Glue hard aspect ratio */
    glue_hard_aspect?: number;
    /** Glue soft aspect ratio */
    glue_soft_aspect?: number;
    /** Check the connection length */
    length_check?: boolean;
    /** Maximum length of connection */
    max_length?: number;
    /** Maximum number of panels */
    max_panels?: number;
    /** Maximum warpage */
    max_warpage?: number;
    /** Minimum length of connection */
    min_length?: number;
    /** Create arcweld using NRBs (rather than contact) */
    nrb_arcweld?: boolean;
    /** Check for maximum number of panels */
    panel_check?: boolean;
    /** Patch angle */
    patch_angle?: number;
    /** Check the patch angle */
    patch_angle_check?: boolean;
    /** Allow connections to join a part to itself */
    same_part?: boolean;
    /** Consider free edges when orienting single solid spotwelds */
    solid_free_edges?: boolean;
    /** Spotweld line search tolerance */
    spot_line_tol?: number;
    /** Search thickness */
    spot_thickness?: number;
    /** Total length of connection */
    total_length?: number;
    /** Use _PID for beam connections */
    use_pid?: boolean;
    /** Check warpage value */
    warpage_check?: boolean;
}

declare class Conx {
/**
 * Associates a comment with a connection.
 * @param Comment Comment that will be attached to the connection
 */
    AssociateComment(Comment: Comment): void;

/**
 * Blanks the connection
 */
    Blank(): void;

/**
 * Blanks all of the connections in the model.
 * @param Model Model that all connections will be blanked in
 * @param redraw If model should be redrawn or not. If omitted redraw is false. If you want to do several (un)blanks and only redraw after the last one then use false for all redraws apart from the last one. Alternatively you can redraw using View.Redraw().
 */
    static BlankAll(Model: Model, redraw?: boolean): void;

/**
 * Blanks all of the flagged connections in the model.
 * @param Model Model that all the flagged connections will be blanked in
 * @param flag Flag set on the connections that you want to blank
 * @param redraw If model should be redrawn or not. If omitted redraw is false. If you want to do several (un)blanks and only redraw after the last one then use false for all redraws apart from the last one. Alternatively you can redraw using View.Redraw().
 */
    static BlankFlagged(Model: Model, flag: number, redraw?: boolean): void;

/**
 * Checks if the connection is blanked or not.
 * @returns true if blanked, false if not.
 */
    Blanked(): boolean;

/**
 * Clears a flag on the connection.
 * @param flag Flag to clear on the connection
 */
    ClearFlag(flag: number): void;

/**
 * Copies the connection. The target include of the copied connection can be set using Options.copy_target_include.
 * @param range If you want to keep the copied item in the range specified for the current include. Default value is false. To set current include, use Include.MakeCurrentLayer().
 * @returns Conx object
 */
    Copy(range?: boolean): Conx;

/**
 * Detaches a comment from a connection.
 * @param Comment Comment that will be detached from the connection
 */
    DetachComment(Comment: Comment): void;

/**
 * Empties the patch topology/coordinates data.
 */
    EmptyPatch(): void;

/**
 * Adds an error for connection. For more details on checking see the Check class.
 * @param message The error message to give
 * @param details An optional detailed error message
 */
    Error(message: string, details?: string): void;

/**
 * Extracts the actual colour used for connection. By default in PRIMER many entities such as elements get their colour automatically from the part that they are in. PRIMER cycles through 13 default colours based on the label of the entity. In this case the connection colour property will return the value Colour.PART instead of the actual colour. This method will return the actual colour which is used for drawing the connection.
 * @returns colour value (integer)
 */
    ExtractColour(): number;

/**
 * Returns the first connection in the model.
 * @param Model Model to get first connection in
 * @returns Conx object (or null if there are no connections in the model).
 */
    static First(Model: Model): Conx;

/**
 * Returns the first free connection label in the model. Also see Conx.LastFreeLabel(), Conx.NextFreeLabel() and Model.FirstFreeItemLabel().
 * @param Model Model to get first free connection label in
 * @param layer Include file (0 for the main file) to search for labels in (Equivalent to First free in layer in editing panels). If omitted the whole model will be used (Equivalent to First free in editing panels).
 * @returns Conx label.
 */
    static FirstFreeLabel(Model: Model, layer?: number): number;

/**
 * Flags all of the connections in the model with a defined flag.
 * @param Model Model that all connections will be flagged in
 * @param flag Flag to set on the connections
 */
    static FlagAll(Model: Model, flag: number): void;

/**
 * Checks if the connection is flagged or not.
 * @param flag Flag to test on the connection
 * @returns true if flagged, false if not.
 */
    Flagged(flag: number): boolean;

/**
 * Calls a function for each connection in the model. Note that ForEach has been designed to make looping over connections as fast as possible and so has some limitations. Firstly, a single temporary Conx object is created and on each function call it is updated with the current connection data. This means that you should not try to store the Conx object for later use (e.g. in an array) as it is temporary. Secondly, you cannot create new connections inside a ForEach loop. 
 * @param Model Model that all connections are in
 * @param func Function to call for each connection
 * @param extra An optional extra object/array/string etc that will appended to arguments when calling the function
 */
    static ForEach(Model: Model, func: () => void, extra?: any): void;

/**
 * Returns an array of Conx objects or properties for all of the connections in a model in PRIMER. If the optional property argument is not given then an array of Conx objects is returned. If the property argument is given, that property value for each connection is returned in the array instead of a Conx object
 * @param Model Model to get connections from
 * @param property Name for property to get for all connections in the model
 * @returns Array of Conx objects or properties
 */
    static GetAll(Model: Model, property?: string): Conx[];

/**
 * Extracts the comments associated to a connection.
 * @returns Array of Comment objects (or null if there are no comments associated to the node).
 */
    GetComments(): Comment[];

/**
 * Returns the beams/solids that are used in the connection weld.
 * @returns An array containing the element IDs (or null if no elements).
 */
    GetElements(): number[];

/**
 * Returns list of the entities of type that are used in the connection.
 * @param type The type of the item in the reference list (for a list of types see Appendix I of the PRIMER manual).
 * @returns An array containing the item IDs (or null if none).
 */
    GetEntities(type: string): number[];

/**
 * Returns an array of Conx objects for all of the flagged connections in a model in PRIMER If the optional property argument is not given then an array of Conx objects is returned. If the property argument is given, then that property value for each connection is returned in the array instead of a Conx object
 * @param Model Model to get connections from
 * @param flag Flag set on the connections that you want to retrieve
 * @param property Name for property to get for all flagged connections in the model
 * @returns Array of Conx objects or properties
 */
    static GetFlagged(Model: Model, flag: number, property?: string): Conx[];

/**
 * Returns the Conx object for a connection ID.
 * @param Model Model to find the connection in
 * @param number number of the connection you want the Conx object for
 * @returns Conx object (or null if connection does not exist).
 */
    static GetFromID(Model: Model, number: number): Conx;

/**
 * Returns the data for a layer of the connection.
 * @param layer The layer you want the data for. Note that layers start at 0, not 1.
 * @returns An array containing the layer data.
 */
    GetLayerData(layer: number): number[];

/**
 * Returns the attached shells for a layer of the connection.
 * @param layer The layer you want the data for. Note that layers start at 0, not 1.
 * @returns Array of Shell objects or null if not valid
 */
    GetLayerShells(layer: number): Shell[];

/**
 * Checks if a Conx property is a parameter or not. Note that object properties that are parameters are normally returned as the integer or float parameter values as that is virtually always what the user would want. For this function to work the JavaScript interpreter must use the parameter name instead of the value. This can be done by setting the Options.property_parameter_names option to true before calling the function and then resetting it to false afterwards.. This behaviour can also temporarily be switched by using the Conx.ViewParameters() method and 'method chaining' (see the examples below).
 * @param prop connection property to get parameter for
 * @returns Parameter object if property is a parameter, null if not.
 */
    GetParameter(prop: string): Parameter;

/**
 * Returns the data for a patch coordinate of an adhesive patch connection.
 * @param point The point you want the data for. Note that points start at 0, not 1.
 * @returns An array containing the patch coordinate.
 */
    GetPatchCoords(point: number): number[];

/**
 * Returns the topology for a patch quad/tria of an adhesive patch connection.
 * @param point The patch quad/tria you want the data for. Note that points start at 0, not 1.
 * @returns Array of numbers containing the patch topology information.
 */
    GetPatchTopol(point: number): number[];

/**
 * Returns the data for a path point of an adhesive/spotweld line connection.
 * @param point The point you want the data for. Note that points start at 0, not 1.
 * @returns An array containing the path data.
 */
    GetPathData(point: number): number[];

/**
 * Returns an array of Part objects for the connection FE entities. A connection can contain elements with different part ID's between different layers. If one part ID is returned, that part is used for all elements in the connection. Not applicable for bolts.
 * @returns Array of Part objects
 */
    GetPidData(): Part[];

/**
 * Returns an object of settings stored with the connection.
 */
    GetSettings(): GetSettingsReturn;

/**
 * Returns an array containing a number of objects equal to the number of solid elements in the connection. Each object contains the corresponding solid element object, and shell element objects and their thicknesses. The argument allows the user to output only shells from all layers, or a particular layer. Note that a carriage return is not added.
 * @param Layer ID of the connection layer containing the shells from which the thicknesses will be extracted. If a value of zero or lower is input, all layers will be considered in the output data.
 */
    GetShellThickness(Layer: number): GetShellThicknessReturn[];

/**
 * Returns the keyword for this connection (*CONNECTION_START_SPOTWELD etc). Note that a carriage return is not added. See also Conx.KeywordCards()
 * @returns string containing the keyword.
 */
    Keyword(): string;

/**
 * Returns the keyword cards for the connection. Note that a carriage return is not added. See also Conx.Keyword()
 * @returns string containing the cards.
 */
    KeywordCards(): string;

/**
 * Returns the last connection in the model.
 * @param Model Model to get last connection in
 * @returns Conx object (or null if there are no connections in the model).
 */
    static Last(Model: Model): Conx;

/**
 * Returns the last free connection label in the model. Also see Conx.FirstFreeLabel(), Conx.NextFreeLabel() and see Model.LastFreeItemLabel()
 * @param Model Model to get last free connection label in
 * @param layer Include file (0 for the main file) to search for labels in (Equivalent to Highest free in layer in editing panels). If omitted the whole model will be used.
 * @returns Conx label.
 */
    static LastFreeLabel(Model: Model, layer?: number): number;

/**
 * Returns the next connection in the model.
 * @returns Conx object (or null if there are no more connections in the model).
 */
    Next(): Conx;

/**
 * Returns the next free (highest+1) connection label in the model. Also see Conx.FirstFreeLabel(), Conx.LastFreeLabel() and Model.NextFreeItemLabel()
 * @param Model Model to get next free connection label in
 * @param layer Include file (0 for the main file) to search for labels in (Equivalent to Highest+1 in layer in editing panels). If omitted the whole model will be used (Equivalent to Highest+1 in editing panels).
 * @returns Conx label.
 */
    static NextFreeLabel(Model: Model, layer?: number): number;

/**
 * Allows the user to pick a connection.
 * @param prompt Text to display as a prompt to the user
 * @param limit If the argument is a Model then only connections from that model can be picked. If the argument is a Flag then only connections that are flagged with limit can be selected. If omitted, or null, any connections from any model can be selected. from any model.
 * @param modal If picking is modal (blocks the user from doing anything else in PRIMER until this window is dismissed). If omitted the pick will be modal.
 * @param button_text By default the window with the prompt will have a button labelled 'Cancel' which if pressed will cancel the pick and return null. If you want to change the text on the button use this argument. If omitted 'Cancel' will be used.
 * @returns Conx object (or null if not picked)
 */
    static Pick(prompt: string, limit?: Model | number, modal?: boolean, button_text?: string): Conx;

/**
 * Returns the previous connection in the model.
 * @returns Conx object (or null if there are no more connections in the model).
 */
    Previous(): Conx;

/**
 * Realizes all of the connections in the model.
 * @param Model Model that all connections will be realized in
 */
    static RealizeAll(Model: Model): void;

/**
 * Realizes all of the flagged connections in the model.
 * @param Model Model that the flagged connections will be realized in
 * @param flag Flag set on the connections that you want to realize
 */
    static RealizeFlagged(Model: Model, flag: number): void;

/**
 * Reload all modules from primer_library/connectors
 */
    static ReloadConnectors(): void;

/**
 * Deletes the topology at a particular location for patch type adhesive.
 * @param layer The topology location you want to remove. Note that layers start at 0, not 1.
 */
    RemovePatchTopol(layer: number): void;

/**
 * Deletes a pathc point for a line adhesive connection.
 * @param layer The point you want to remove. Note that layers start at 0, not 1.
 */
    RemovePathData(layer: number): void;

/**
 * Renumbers all of the connections in the model.
 * @param Model Model that all connections will be renumbered in
 * @param start Start point for renumbering
 */
    static RenumberAll(Model: Model, start: number): void;

/**
 * Renumbers all of the flagged connections in the model.
 * @param Model Model that all the flagged connections will be renumbered in
 * @param flag Flag set on the connections that you want to renumber
 * @param start Start point for renumbering
 */
    static RenumberFlagged(Model: Model, flag: number, start: number): void;

/**
 * Allows the user to select connections using standard PRIMER object menus.
 * @param flag Flag to use when selecting connections
 * @param prompt Text to display as a prompt to the user
 * @param limit If the argument is a Model then only connections from that model can be selected. If the argument is a Flag then only connections that are flagged with limit can be selected (limit should be different to flag). If omitted, or null, any connections can be selected. from any model.
 * @param modal If selection is modal (blocks the user from doing anything else in PRIMER until this window is dismissed). If omitted the selection will be modal.
 * @returns Number of connections selected or null if menu cancelled
 */
    static Select(flag: number, prompt: string, limit?: Model | number, modal?: boolean): number;

/**
 * Sets a flag on the connection.
 * @param flag Flag to set on the connection
 */
    SetFlag(flag: number): void;

/**
 * Sets the data for a layer of the connection.
 * @param layer The layer you want to set the data for. Note that layers start at 0, not 1.
 * @param item1 The first item for the layer definition. As layer definitions can be part IDs, part names, CAD names, part set IDs, part set names or assemby names the following logic is used. If the item is an integer it is assumed to be a part ID. If the item is a string then it must be in the format 'P&lt;part ID&gt;', 'P:&lt;part name&gt;', 'C:&lt;CAD name&gt;', 'S&lt;set ID&gt;', 'S:&lt;set name&gt;'or 'A:&lt;assembly name&gt;'.
 * @param item2 The second item for the layer definition. This must be type same type as item1. e.g. if item1 is a part ID, item2 must be a part ID (it cannot be a part name etc).
 * @param ...vars The nth item for the layer definition. This must be type same type as item1. e.g. if item1 is a part ID, this item must be a part ID (it cannot be a part name etc).
 */
    SetLayerData(layer: number, item1: number, item2?: number, ...vars?: number): void;

/**
 * Sets a coordinate used by the adhesive patch connection type.
 * @param point The point you want to set the data for. Note that points start at 0, not 1.
 * @param x X coordinate of point
 * @param y Y coordinate of point
 * @param z Z coordinate of point
 */
    SetPatchCoords(point: number, x: number, y: number, z: number): void;

/**
 * Sets the topology used by the adhesive patch connection type.
 * @param point The point you want to set the data for. Note that points start at 0, not 1.
 * @param c1 1st coordinate location point
 * @param c2 2nd coordinate location point
 * @param c3 3rd coordinate location point
 * @param c4 4th coordinate location point
 */
    SetPatchTopol(point: number, c1: number, c2: number, c3: number, c4?: number): void;

/**
 * Sets the data for a path point of the connection.
 * @param point The point you want to set the data for. Note that points start at 0, not 1.
 * @param x X coordinate of point
 * @param y Y coordinate of point
 * @param z Z coordinate of point
 */
    SetPathData(point: number, x: number, y: number, z: number): void;

/**
 * Sets the element part IDs for the connection. A different part can be defined for elements in the connection between different layers. Not applicable for bolts.
 * @param item1 Part label of the first item in the PID layer list.
 * @param item2 The second item for the layer definition.
 * @param ...vars The nth item for the layer definition.
 */
    SetPidData(item1: number, item2?: number, ...vars?: number): void;

/**
 * Set the diameter for a spotweld ring when running a rule. Note that this method can only be called when running a connection rule script. It will not have any effect if used in a 'normal' script.
 * @param diameter The diameter to set for the ring
 */
    static SetRuleDiameter(diameter: number): void;

/**
 * Set the PID for spotweld beam/solid elements or adhesive solids when running a rule. Note that this method can only be called when running a connection rule script. It will not have any effect if used in a 'normal' script.
 * @param pid The PID to set for the spotweld or adhesive elements
 */
    static SetRuleFEPID(pid: number): void;

/**
 * Set the PID for a spotweld ring when running a rule. Note that this method can only be called when running a connection rule script. It will not have any effect if used in a 'normal' script.
 * @param pid The PID to set for the ring
 */
    static SetRulePID(pid: number): void;

/**
 * Sets the settings stored on a connection entity. Not applicable for bolts.
 * @param data Object containing the connection settings data. The properties can be:
 */
    SetSettings(data: SetSettingsArgument_data): void;

/**
 * Sketches the connection. The connection will be sketched until you either call Conx.Unsketch(), Conx.UnsketchAll(), Model.UnsketchAll(), or delete the model
 * @param redraw If model should be redrawn or not after the connection is sketched. If omitted redraw is true. If you want to sketch several connections and only redraw after the last one then use false for redraw and call View.Redraw().
 */
    Sketch(redraw?: boolean): void;

/**
 * Sketches all of the flagged connections in the model. The connections will be sketched until you either call Conx.Unsketch(), Conx.UnsketchFlagged(), Model.UnsketchAll(), or delete the model
 * @param Model Model that all the flagged connections will be sketched in
 * @param flag Flag set on the connections that you want to sketch
 * @param redraw If model should be redrawn or not after the connections are sketched. If omitted redraw is true. If you want to sketch flagged connections several times and only redraw after the last one then use false for redraw and call View.Redraw().
 */
    static SketchFlagged(Model: Model, flag: number, redraw?: boolean): void;

/**
 * Returns the total number of connections in the model.
 * @param Model Model to get total for
 * @param exists true if only existing connections should be counted. If false or omitted referenced but undefined connections will also be included in the total.
 * @returns number of connections
 */
    static Total(Model: Model, exists?: boolean): number;

/**
 * Unblanks the connection
 */
    Unblank(): void;

/**
 * Unblanks all of the connections in the model.
 * @param Model Model that all connections will be unblanked in
 * @param redraw If model should be redrawn or not. If omitted redraw is false. If you want to do several (un)blanks and only redraw after the last one then use false for all redraws apart from the last one. Alternatively you can redraw using View.Redraw().
 */
    static UnblankAll(Model: Model, redraw?: boolean): void;

/**
 * Unblanks all of the flagged connections in the model.
 * @param Model Model that the flagged connections will be unblanked in
 * @param flag Flag set on the connections that you want to unblank
 * @param redraw If model should be redrawn or not. If omitted redraw is false. If you want to do several (un)blanks and only redraw after the last one then use false for all redraws apart from the last one. Alternatively you can redraw using View.Redraw().
 */
    static UnblankFlagged(Model: Model, flag: number, redraw?: boolean): void;

/**
 * Unsets a defined flag on all of the connections in the model.
 * @param Model Model that the defined flag for all connections will be unset in
 * @param flag Flag to unset on the connections
 */
    static UnflagAll(Model: Model, flag: number): void;

/**
 * Unsketches the connection.
 * @param redraw If model should be redrawn or not after the connection is unsketched. If omitted redraw is true. If you want to unsketch several connections and only redraw after the last one then use false for redraw and call View.Redraw().
 */
    Unsketch(redraw?: boolean): void;

/**
 * Unsketches all connections.
 * @param Model Model that all connections will be unblanked in
 * @param redraw If model should be redrawn or not after the connections are unsketched. If omitted redraw is true. If you want to unsketch several things and only redraw after the last one then use false for redraw and call View.Redraw().
 */
    static UnsketchAll(Model: Model, redraw?: boolean): void;

/**
 * Unsketches all flagged connections in the model.
 * @param Model Model that all connections will be unsketched in
 * @param flag Flag set on the connections that you want to unsketch
 * @param redraw If model should be redrawn or not after the connections are unsketched. If omitted redraw is true. If you want to unsketch several things and only redraw after the last one then use false for redraw and call View.Redraw().
 */
    static UnsketchFlagged(Model: Model, flag: number, redraw?: boolean): void;

/**
 * True (default) means put bolt FE into parent layer where possible.
 * @param option True (default) means put bolt FE into parent layer where possible.
 */
    static UseParentLayer(option: boolean): void;

/**
 * True (default) means use the pref settings for C_SPR2 created when rivet realized.
 * @param option True (default) means use the pref settings for C_SPR2 created when rivet realized.
 */
    static UseSPR2Pref(option: boolean): void;

/**
 * Object properties that are parameters are normally returned as the integer or float parameter values as that is virtually always what the user would want. This function temporarily changes the behaviour so that if a property is a parameter the parameter name is returned instead. This can be used with 'method chaining' (see the example below) to make sure a property argument is correct.
 * @returns Conx object.
 */
    ViewParameters(): Conx;

/**
 * Adds a warning for connection. For more details on checking see the Check class.
 * @param message The warning message to give
 * @param details An optional detailed warning message
 */
    Warning(message: string, details?: string): void;

/**
 * Returns the cross references for this connection.
 * @returns Xrefs object.
 */
    Xrefs(): Xrefs;

/**
 * Create a new Conx object.
 * @param Model Model that connection will be created in
 * @param x X coordinate
 * @param y Y coordinate
 * @param z Z coordinate
 * @param type Type of connection. Can be Conx.SPOTWELD, Conx.BOLT, Conx.ADHESIVE Conx.SPOTWELD_LINE or Conx.RIVET. If omitted type will be set to Conx.SPOTWELD.
 * @param subtype Subtype of connection. See property subtype for valid values. If omitted subtype will be set to the default subtype for this type of connection.
 * @param title Title for the connection
 * @returns Conx object
 */
    constructor(Model: Model, x: number, y: number, z: number, type?: number, subtype?: number, title?: string);

/**
 * Creates a string containing the connection data in keyword format. Note that this contains the keyword header and the keyword cards. See also Conx.Keyword() and Conx.KeywordCards().
 * @returns string
 */
    toString(): string;

/** Element size along the length of the adhesive run */
    adhesive_esize: number;
/** The number of elements across the width of the adhesive */
    adhesive_nelem: number;
/** The width of the adhesive run */
    adhesive_width: number;
/** angle tolerance for bolt */
    angtol: number;
/** angle tolerance at end 2 for 2 point bolt */
    angtol2: number;
/** Assembly used to specify panels connection together, rather than individual layers. Integer for a part set ID, string for a PRIMER assembly (name). */
    assembly: number;
/** The assembly type. Can be Conx.PART_SET or Conx.ASSEMBLY. */
    assembly_type: number;
/** The colour of the connection */
    colour: Colour;
/** Diameter of spotweld/rigid */
    diameter: number;
/** Diameter of rigid at end 2 */
    diameter2: number;
/** Spotweld line edge distance */
    edge_distance: number;
/** true if a spotweld line is locked to an edge, false if not */
    edge_lock: boolean;
/** Description of the error if the connection cannot be made */
    readonly error: string;
/** Details of the error if the connection cannot be made */
    readonly error_details: string;
/** true if connection exists, false if referred to but not defined. */
    readonly exists: boolean;
/** contact fitting method for library bolts */
    fit: number;
/** Conx number. Also see the label property which is an alternative name for this. */
    id: number;
/** The Include file number that the connection is in. */
    include: number;
/** Conx number. Also see the id property which is an alternative name for this. */
    label: number;
/** The number of layers the connection has. */
    layers: number;
/** Length of 1 point bolt, max thickness for 2 point bolt */
    length: number;
/** max thickness at end 2 for 2 point bolt */
    length2: number;
/** The ID of the Material used for 'merge' bolt connections. i.e. Conx.BOLT_MRG_CYL, Conx.BOLT_MRG_CYL_BEAM, */
    material: number;
/** The Model number that the connection is in. */
    readonly model: number;
/** name of library module for bolt */
    module: string;
/** The ID of the Part used for adhesive or spotweld connections. Note that in v11.0 and above you are able to specify a different part IDs for elements in the connection between different layers. If you only have one part for the elements in the connection, then this is the value of this property. If there is more than one used, then the value of this property is the first part. If you set this property to a new value, then the all the elements in the connection will have this new part ID when it is realized. To set and retrieve information on parts used between different layers, the functions GetPidData() and SetPidData() should be used. */
    part: number;
/** The number of patch coordinate points the connection has (Adhesive patch only). */
    patch_coords: number;
/** The number of patch topology entries the connection has (Adhesive patch only). */
    patch_topol: number;
/** The number of path points the connection has (Adhesive only). Note that these points do NOT include the start and end points for the adhesive run. These are defined using the properties x, y, z and x2, y2, z2 */
    path: number;
/** Spotweld line pitch */
    pitch: number;
/** snap to points fitting method for library bolts */
    resize: number;
/** Whether settings are saved for a connection or not */
    saved_settings: boolean;
/** shape for bolt attachment */
    shape: number;
/** shape for bolt attachment at end 2 for 2 point bolt */
    shape2: number;
/** Internal label of C_SPR2 which applied to this rivet connection */
    readonly spr2_id: number;
/** True to use matching C_SPR2 for this rivet. False to create new C_SPR2 for each rivet. IF unset, a new C_SPR2 will be created. */
    spr2_match: boolean;
/** True if C_SPR2 is unique for this rivet */
    readonly spr2_unshared: boolean;
/** The status of the connection. Can be Conx.DORMANT, Conx.MADE, Conx.INVALID, Conx.REALIZED or Conx.BAD. */
    status: number;
/** The connection subtype. For SPOTWELD and SPOTWELD_LINE connections the subtype can be:  Conx.SPOTWELD_BEAM Conx.SPOTWELD_MIG Conx.SPOTWELD_HEXA1 Conx.SPOTWELD_HEXA2 Conx.SPOTWELD_HEXA3 Conx.SPOTWELD_HEXA4 Conx.SPOTWELD_HEXA8 Conx.SPOTWELD_HEXA12 Conx.SPOTWELD_HEXA16  For BOLT connections the subtype can be:  Conx.BOLT_MRG_CYL Conx.BOLT_MRG_CYL_BEAM Conx.BOLT_MRG_CYL_BALL Conx.BOLT_MRG_2PTS Conx.BOLT_MRG_2PTS_RB Conx.BOLT_MRG_2PTS_RJ Conx.BOLT_MRG_CYL Conx.BOLT_NRB_CYL_BEAM Conx.BOLT_NRB_CYL_BALL Conx.BOLT_NRB_SPH Conx.BOLT_NRB_SPH_BALL Conx.BOLT_NRB_SPH_DISC Conx.BOLT_NRB_2PTS Conx.BOLT_MODULE  For ADHESIVE connections the subtype can be: Conx.ADHESIVE_SOLID. Conx.ADHESIVE_PATCH. */
    subtype: number;
/** Title for connection */
    title: string;
/** The transparency of the connection (0-100) 0% is opaque, 100% is transparent. */
    transparency: number;
/** The connection type. Can be Conx.SPOTWELD, Conx.BOLT or Conx.RIVET or Conx.ADHESIVE. */
    type: number;
/** User data for connection */
    user_data: string;
/** X coordinate */
    x: number;
/** X coordinate for second point (adhesive only) */
    x2: number;
/** Y coordinate */
    y: number;
/** Y coordinate for second point (adhesive only) */
    y2: number;
/** Z coordinate */
    z: number;
/** Z coordinate for second point (adhesive only) */
    z2: number;
/** Connection is adhesive. */
    static ADHESIVE: number;
/** Connection adhesive type is a patch. */
    static ADHESIVE_PATCH: number;
/** Connection adhesive type is a solid line. */
    static ADHESIVE_SOLID: number;
/** If the connection refers to an assembly rather than individual layers, the assembly is defined by part tree assembly. */
    static ASSEMBLY: number;
/** Connection is bad (e.g. necessary data is missing). */
    static BAD: number;
/** Connection is a bolt. */
    static BOLT: number;
/** Library bolt. */
    static BOLT_MODULE: number;
/** 2pt Patch Beam. */
    static BOLT_MRG_2PTS: number;
/** 2pt Patch (Rigid Beam). */
    static BOLT_MRG_2PTS_RB: number;
/** 2pt Patch Revolute joint. */
    static BOLT_MRG_2PTS_RJ: number;
/** Cylindrical Merge. */
    static BOLT_MRG_CYL: number;
/** Cylindrical Patch Ball joint. */
    static BOLT_MRG_CYL_BALL: number;
/** Cylindrical Patch Beam. */
    static BOLT_MRG_CYL_BEAM: number;
/** 2pt NRB Beam. */
    static BOLT_NRB_2PTS: number;
/** Cylindrical NRB. */
    static BOLT_NRB_CYL: number;
/** Cylindrical NRB Ball joint. */
    static BOLT_NRB_CYL_BALL: number;
/** Cylindrical NRB Beam. */
    static BOLT_NRB_CYL_BEAM: number;
/** Spherical NRB. */
    static BOLT_NRB_SPH: number;
/** Spherical NRB Ball joint. */
    static BOLT_NRB_SPH_BALL: number;
/** Spherical NRB Discrete Beam. */
    static BOLT_NRB_SPH_DISC: number;
/** Connection is dormant (not yet made). */
    static DORMANT: number;
/** Connection has been made but something is wrong (e.g. part moved). */
    static INVALID: number;
/** Connection has been made but status is unknown. */
    static MADE: number;
/** If the connection refers to an assembly rather than individual layers, the assembly is defined by part set. */
    static PART_SET: number;
/** Connection has been made and is OK (checks OK). */
    static REALIZED: number;
/** Connection is rivet. */
    static RIVET: number;
/** Connection is a spotweld. */
    static SPOTWELD: number;
/** Connection spotweld type is beam. */
    static SPOTWELD_BEAM: number;
/** Connection spotweld type is one hexa solid/spotweld layer. */
    static SPOTWELD_HEXA1: number;
/** Connection spotweld type is twelve hexa solids/spotweld layer. */
    static SPOTWELD_HEXA12: number;
/** Connection spotweld type is sixteen hexa solids/spotweld layer. */
    static SPOTWELD_HEXA16: number;
/** Connection spotweld type is two hexa solids/spotweld layer. */
    static SPOTWELD_HEXA2: number;
/** Connection spotweld type is three hexa solids/spotweld layer. */
    static SPOTWELD_HEXA3: number;
/** Connection spotweld type is four hexa solids/spotweld layer. */
    static SPOTWELD_HEXA4: number;
/** Connection spotweld type is eight hexa solids/spotweld layer. */
    static SPOTWELD_HEXA8: number;
/** Connection is a spotweld line. */
    static SPOTWELD_LINE: number;
/** Connection spotweld type is (beam) MIG weld. */
    static SPOTWELD_MIG: number;
}


/** Object function argument in CoordinateSystem constructor */
interface CoordinateSystem_constructor_Argument_details {
    /** Label of CoordinateSystem */
    cid: number;
    /** Array of coordinates of point on local X-axis [lx, ly, lz] (for option CoordinateSystem.SYSTEM) */
    cl?: array[];
    /** Array of coordinates of origin [ox, oy, oz] (for option CoordinateSystem.SYSTEM) */
    co?: array[];
    /** Array of coordinates of point in local X-Y plane [px, py, pz] (for option CoordinateSystem.SYSTEM) */
    cp?: array[];
    /** Array of coordinates of local X-Y vector [vx, vy, vz] (for option CoordinateSystem.VECTOR) */
    cv?: array[];
    /** Array of coordinates on local X-axis [xx, xy, xz] (for option CoordinateSystem.VECTOR) */
    cx?: array[];
    /** Axis defined by N1N2 (for option CoordinateSystem.NODES) */
    dir?: number;
    /** Flag for local system update for each time step (for option CoordinateSystem.NODES) */
    flag?: boolean;
    /** Title for the coordinate system */
    heading?: string;
    /** Optional Node ID for rotation (for option CoordinateSystem.VECTOR) */
    nid?: number;
    /** Array of Node IDs [n1, n2, n3] for the coordinate system (for option CoordinateSystem.NODES) */
    nodes?: array[];
    /** CoordinateSystem type (can be CoordinateSystem.NODES, CoordinateSystem.SYSTEM or CoordinateSystem.VECTOR) */
    option: number;
}

declare class CoordinateSystem {
/**
 * Associates a comment with a coordinate system.
 * @param Comment Comment that will be attached to the coordinate system
 */
    AssociateComment(Comment: Comment): void;

/**
 * Blanks the coordinate system
 */
    Blank(): void;

/**
 * Blanks all of the coordinate systems in the model.
 * @param Model Model that all coordinate systems will be blanked in
 * @param redraw If model should be redrawn or not. If omitted redraw is false. If you want to do several (un)blanks and only redraw after the last one then use false for all redraws apart from the last one. Alternatively you can redraw using View.Redraw().
 */
    static BlankAll(Model: Model, redraw?: boolean): void;

/**
 * Blanks all of the flagged coordinate systems in the model.
 * @param Model Model that all the flagged coordinate systems will be blanked in
 * @param flag Flag set on the coordinate systems that you want to blank
 * @param redraw If model should be redrawn or not. If omitted redraw is false. If you want to do several (un)blanks and only redraw after the last one then use false for all redraws apart from the last one. Alternatively you can redraw using View.Redraw().
 */
    static BlankFlagged(Model: Model, flag: number, redraw?: boolean): void;

/**
 * Checks if the coordinate system is blanked or not.
 * @returns true if blanked, false if not.
 */
    Blanked(): boolean;

/**
 * Starts an edit panel in Browse mode.
 * @param modal If this window is modal (blocks the user from doing anything else in PRIMER until this window is dismissed). If omitted the window will be modal.
 */
    Browse(modal?: boolean): void;

/**
 * Clears a flag on the coordinate system.
 * @param flag Flag to clear on the coordinate system
 */
    ClearFlag(flag: number): void;

/**
 * Copies the coordinate system. The target include of the copied coordinate system can be set using Options.copy_target_include.
 * @param range If you want to keep the copied item in the range specified for the current include. Default value is false. To set current include, use Include.MakeCurrentLayer().
 * @returns CoordinateSystem object
 */
    Copy(range?: boolean): CoordinateSystem;

/**
 * Starts an interactive editing panel to create a coordinate system
 * @param Model Model that the coordinate system will be created in.
 * @param modal If this window is modal (blocks the user from doing anything else in PRIMER until this window is dismissed). If omitted the window will be modal.
 * @returns CoordinateSystem object (or null if not made).
 */
    static Create(Model: Model, modal?: boolean): CoordinateSystem;

/**
 * Detaches a comment from a coordinate system.
 * @param Comment Comment that will be detached from the coordinate system
 */
    DetachComment(Comment: Comment): void;

/**
 * Starts an interactive editing panel.
 * @param modal If this window is modal (blocks the user from doing anything else in PRIMER until this window is dismissed). If omitted the window will be modal.
 */
    Edit(modal?: boolean): void;

/**
 * Adds an error for coordinate system. For more details on checking see the Check class.
 * @param message The error message to give
 * @param details An optional detailed error message
 */
    Error(message: string, details?: string): void;

/**
 * Returns the first coordinate system in the model.
 * @param Model Model to get first coordinate system in
 * @returns CoordinateSystem object (or null if there are no coordinate systems in the model).
 */
    static First(Model: Model): CoordinateSystem;

/**
 * Returns the first free coordinate system label in the model. Also see CoordinateSystem.LastFreeLabel(), CoordinateSystem.NextFreeLabel() and Model.FirstFreeItemLabel().
 * @param Model Model to get first free coordinate system label in
 * @param layer Include file (0 for the main file) to search for labels in (Equivalent to First free in layer in editing panels). If omitted the whole model will be used (Equivalent to First free in editing panels).
 * @returns CoordinateSystem label.
 */
    static FirstFreeLabel(Model: Model, layer?: number): number;

/**
 * Flags all of the coordinate systems in the model with a defined flag.
 * @param Model Model that all coordinate systems will be flagged in
 * @param flag Flag to set on the coordinate systems
 */
    static FlagAll(Model: Model, flag: number): void;

/**
 * Checks if the coordinate system is flagged or not.
 * @param flag Flag to test on the coordinate system
 * @returns true if flagged, false if not.
 */
    Flagged(flag: number): boolean;

/**
 * Calls a function for each coordinate system in the model. Note that ForEach has been designed to make looping over coordinate systems as fast as possible and so has some limitations. Firstly, a single temporary CoordinateSystem object is created and on each function call it is updated with the current coordinate system data. This means that you should not try to store the CoordinateSystem object for later use (e.g. in an array) as it is temporary. Secondly, you cannot create new coordinate systems inside a ForEach loop. 
 * @param Model Model that all coordinate systems are in
 * @param func Function to call for each coordinate system
 * @param extra An optional extra object/array/string etc that will appended to arguments when calling the function
 */
    static ForEach(Model: Model, func: () => void, extra?: any): void;

/**
 * Returns an array of CoordinateSystem objects or properties for all of the coordinate systems in a model in PRIMER. If the optional property argument is not given then an array of CoordinateSystem objects is returned. If the property argument is given, that property value for each coordinate system is returned in the array instead of a CoordinateSystem object
 * @param Model Model to get coordinate systems from
 * @param property Name for property to get for all coordinate systems in the model
 * @returns Array of CoordinateSystem objects or properties
 */
    static GetAll(Model: Model, property?: string): CoordinateSystem[];

/**
 * Extracts the comments associated to a coordinate system.
 * @returns Array of Comment objects (or null if there are no comments associated to the node).
 */
    GetComments(): Comment[];

/**
 * Returns an array of CoordinateSystem objects for all of the flagged coordinate systems in a model in PRIMER If the optional property argument is not given then an array of CoordinateSystem objects is returned. If the property argument is given, then that property value for each coordinate system is returned in the array instead of a CoordinateSystem object
 * @param Model Model to get coordinate systems from
 * @param flag Flag set on the coordinate systems that you want to retrieve
 * @param property Name for property to get for all flagged coordinate systems in the model
 * @returns Array of CoordinateSystem objects or properties
 */
    static GetFlagged(Model: Model, flag: number, property?: string): CoordinateSystem[];

/**
 * Returns the CoordinateSystem object for a coordinate system ID.
 * @param Model Model to find the coordinate system in
 * @param number number of the coordinate system you want the CoordinateSystem object for
 * @returns CoordinateSystem object (or null if coordinate system does not exist).
 */
    static GetFromID(Model: Model, number: number): CoordinateSystem;

/**
 * Checks if a CoordinateSystem property is a parameter or not. Note that object properties that are parameters are normally returned as the integer or float parameter values as that is virtually always what the user would want. For this function to work the JavaScript interpreter must use the parameter name instead of the value. This can be done by setting the Options.property_parameter_names option to true before calling the function and then resetting it to false afterwards.. This behaviour can also temporarily be switched by using the CoordinateSystem.ViewParameters() method and 'method chaining' (see the examples below).
 * @param prop coordinate system property to get parameter for
 * @returns Parameter object if property is a parameter, null if not.
 */
    GetParameter(prop: string): Parameter;

/**
 * Returns the keyword for this csys (*DEFINE_COORDINATE). Note that a carriage return is not added. See also CoordinateSystem.KeywordCards()
 * @returns string containing the keyword.
 */
    Keyword(): string;

/**
 * Returns the keyword cards for the csys. Note that a carriage return is not added. See also CoordinateSystem.Keyword()
 * @returns string containing the cards.
 */
    KeywordCards(): string;

/**
 * Returns the last coordinate system in the model.
 * @param Model Model to get last coordinate system in
 * @returns CoordinateSystem object (or null if there are no coordinate systems in the model).
 */
    static Last(Model: Model): CoordinateSystem;

/**
 * Returns the last free coordinate system label in the model. Also see CoordinateSystem.FirstFreeLabel(), CoordinateSystem.NextFreeLabel() and see Model.LastFreeItemLabel()
 * @param Model Model to get last free coordinate system label in
 * @param layer Include file (0 for the main file) to search for labels in (Equivalent to Highest free in layer in editing panels). If omitted the whole model will be used.
 * @returns CoordinateSystem label.
 */
    static LastFreeLabel(Model: Model, layer?: number): number;

/**
 * Returns the next coordinate system in the model.
 * @returns CoordinateSystem object (or null if there are no more coordinate systems in the model).
 */
    Next(): CoordinateSystem;

/**
 * Returns the next free (highest+1) coordinate system label in the model. Also see CoordinateSystem.FirstFreeLabel(), CoordinateSystem.LastFreeLabel() and Model.NextFreeItemLabel()
 * @param Model Model to get next free coordinate system label in
 * @param layer Include file (0 for the main file) to search for labels in (Equivalent to Highest+1 in layer in editing panels). If omitted the whole model will be used (Equivalent to Highest+1 in editing panels).
 * @returns CoordinateSystem label.
 */
    static NextFreeLabel(Model: Model, layer?: number): number;

/**
 * Allows the user to pick a coordinate system.
 * @param prompt Text to display as a prompt to the user
 * @param limit If the argument is a Model then only coordinate systems from that model can be picked. If the argument is a Flag then only coordinate systems that are flagged with limit can be selected. If omitted, or null, any coordinate systems from any model can be selected. from any model.
 * @param modal If picking is modal (blocks the user from doing anything else in PRIMER until this window is dismissed). If omitted the pick will be modal.
 * @param button_text By default the window with the prompt will have a button labelled 'Cancel' which if pressed will cancel the pick and return null. If you want to change the text on the button use this argument. If omitted 'Cancel' will be used.
 * @returns CoordinateSystem object (or null if not picked)
 */
    static Pick(prompt: string, limit?: Model | number, modal?: boolean, button_text?: string): CoordinateSystem;

/**
 * Returns the previous coordinate system in the model.
 * @returns CoordinateSystem object (or null if there are no more coordinate systems in the model).
 */
    Previous(): CoordinateSystem;

/**
 * Renumbers all of the coordinate systems in the model.
 * @param Model Model that all coordinate systems will be renumbered in
 * @param start Start point for renumbering
 */
    static RenumberAll(Model: Model, start: number): void;

/**
 * Renumbers all of the flagged coordinate systems in the model.
 * @param Model Model that all the flagged coordinate systems will be renumbered in
 * @param flag Flag set on the coordinate systems that you want to renumber
 * @param start Start point for renumbering
 */
    static RenumberFlagged(Model: Model, flag: number, start: number): void;

/**
 * Allows the user to select coordinate systems using standard PRIMER object menus.
 * @param flag Flag to use when selecting coordinate systems
 * @param prompt Text to display as a prompt to the user
 * @param limit If the argument is a Model then only coordinate systems from that model can be selected. If the argument is a Flag then only coordinate systems that are flagged with limit can be selected (limit should be different to flag). If omitted, or null, any coordinate systems can be selected. from any model.
 * @param modal If selection is modal (blocks the user from doing anything else in PRIMER until this window is dismissed). If omitted the selection will be modal.
 * @returns Number of coordinate systems selected or null if menu cancelled
 */
    static Select(flag: number, prompt: string, limit?: Model | number, modal?: boolean): number;

/**
 * Sets a flag on the coordinate system.
 * @param flag Flag to set on the coordinate system
 */
    SetFlag(flag: number): void;

/**
 * Sketches the coordinate system. The coordinate system will be sketched until you either call CoordinateSystem.Unsketch(), CoordinateSystem.UnsketchAll(), Model.UnsketchAll(), or delete the model
 * @param redraw If model should be redrawn or not after the coordinate system is sketched. If omitted redraw is true. If you want to sketch several coordinate systems and only redraw after the last one then use false for redraw and call View.Redraw().
 */
    Sketch(redraw?: boolean): void;

/**
 * Sketches all of the flagged coordinate systems in the model. The coordinate systems will be sketched until you either call CoordinateSystem.Unsketch(), CoordinateSystem.UnsketchFlagged(), Model.UnsketchAll(), or delete the model
 * @param Model Model that all the flagged coordinate systems will be sketched in
 * @param flag Flag set on the coordinate systems that you want to sketch
 * @param redraw If model should be redrawn or not after the coordinate systems are sketched. If omitted redraw is true. If you want to sketch flagged coordinate systems several times and only redraw after the last one then use false for redraw and call View.Redraw().
 */
    static SketchFlagged(Model: Model, flag: number, redraw?: boolean): void;

/**
 * Returns the total number of coordinate systems in the model.
 * @param Model Model to get total for
 * @param exists true if only existing coordinate systems should be counted. If false or omitted referenced but undefined coordinate systems will also be included in the total.
 * @returns number of coordinate systems
 */
    static Total(Model: Model, exists?: boolean): number;

/**
 * Unblanks the coordinate system
 */
    Unblank(): void;

/**
 * Unblanks all of the coordinate systems in the model.
 * @param Model Model that all coordinate systems will be unblanked in
 * @param redraw If model should be redrawn or not. If omitted redraw is false. If you want to do several (un)blanks and only redraw after the last one then use false for all redraws apart from the last one. Alternatively you can redraw using View.Redraw().
 */
    static UnblankAll(Model: Model, redraw?: boolean): void;

/**
 * Unblanks all of the flagged coordinate systems in the model.
 * @param Model Model that the flagged coordinate systems will be unblanked in
 * @param flag Flag set on the coordinate systems that you want to unblank
 * @param redraw If model should be redrawn or not. If omitted redraw is false. If you want to do several (un)blanks and only redraw after the last one then use false for all redraws apart from the last one. Alternatively you can redraw using View.Redraw().
 */
    static UnblankFlagged(Model: Model, flag: number, redraw?: boolean): void;

/**
 * Unsets a defined flag on all of the coordinate systems in the model.
 * @param Model Model that the defined flag for all coordinate systems will be unset in
 * @param flag Flag to unset on the coordinate systems
 */
    static UnflagAll(Model: Model, flag: number): void;

/**
 * Unsketches the coordinate system.
 * @param redraw If model should be redrawn or not after the coordinate system is unsketched. If omitted redraw is true. If you want to unsketch several coordinate systems and only redraw after the last one then use false for redraw and call View.Redraw().
 */
    Unsketch(redraw?: boolean): void;

/**
 * Unsketches all coordinate systems.
 * @param Model Model that all coordinate systems will be unblanked in
 * @param redraw If model should be redrawn or not after the coordinate systems are unsketched. If omitted redraw is true. If you want to unsketch several things and only redraw after the last one then use false for redraw and call View.Redraw().
 */
    static UnsketchAll(Model: Model, redraw?: boolean): void;

/**
 * Unsketches all flagged coordinate systems in the model.
 * @param Model Model that all coordinate systems will be unsketched in
 * @param flag Flag set on the coordinate systems that you want to unsketch
 * @param redraw If model should be redrawn or not after the coordinate systems are unsketched. If omitted redraw is true. If you want to unsketch several things and only redraw after the last one then use false for redraw and call View.Redraw().
 */
    static UnsketchFlagged(Model: Model, flag: number, redraw?: boolean): void;

/**
 * Object properties that are parameters are normally returned as the integer or float parameter values as that is virtually always what the user would want. This function temporarily changes the behaviour so that if a property is a parameter the parameter name is returned instead. This can be used with 'method chaining' (see the example below) to make sure a property argument is correct.
 * @returns CoordinateSystem object.
 */
    ViewParameters(): CoordinateSystem;

/**
 * Adds a warning for coordinate system. For more details on checking see the Check class.
 * @param message The warning message to give
 * @param details An optional detailed warning message
 */
    Warning(message: string, details?: string): void;

/**
 * Returns the cross references for this coordinate system.
 * @returns Xrefs object.
 */
    Xrefs(): Xrefs;

/**
 * Create a new CoordinateSystem object for *DEFINE_COORDINATE_NODES.
 * @param Model Model that csys will be created in
 * @param details Details for creating the CoordinateSystem
 * @returns CoordinateSystem object
 */
    constructor(Model: Model, details: CoordinateSystem_constructor_Argument_details);

/**
 * Creates a string containing the csys data in keyword format. Note that this contains the keyword header and the keyword cards. See also CoordinateSystem.Keyword() and CoordinateSystem.KeywordCards().
 * @returns string
 */
    toString(): string;

/** CoordinateSystem number. Also see the label number. */
    cid: number;
/** Optional local coordinate system to define the points in */
    cidl: number;
/** Axis defined by N1N2 */
    dir: number;
/** true if coordinate system exists, false if referred to but not defined. */
    readonly exists: boolean;
/** Flag for updating local system each timestep */
    flag: boolean;
/** CoordinateSystem heading */
    heading: string;
/** The Include file number that the coordinate system is in. */
    include: number;
/** CoordinateSystem number. Also see the cid property which is an alternative name for this. */
    label: number;
/** X-coordinate of point on local X-axis */
    lx: number;
/** Y-coordinate of point on local X-axis */
    ly: number;
/** Z-coordinate of point on local X-axis */
    lz: number;
/** The Model number that the coordinate system is in. */
    readonly model: number;
/** Node located at local origin */
    n1: number;
/** Node located along local (dir) axis */
    n2: number;
/** Node located in local plane determined by (dir) */
    n3: number;
/** Optional node id for rotation */
    nid: number;
/** CoordinateSystem type (Can be CoordinateSystem.NODES, CoordinateSystem.SYSTEM or CoordinateSystem.VECTOR). */
    option: number;
/** X-coordinate of origin */
    ox: number;
/** Y-coordinate of origin */
    oy: number;
/** Z-coordinate of origin */
    oz: number;
/** X-coordinate of point in local X-Y plane */
    px: number;
/** Y-coordinate of point in local X-Y plane */
    py: number;
/** Z-coordinate of point in local X-Y plane */
    pz: number;
/** X-coordinate of local X-Y vector */
    vx: number;
/** Y-coordinate of local X-Y vector */
    vy: number;
/** Z-coordinate of local X-Y vector */
    vz: number;
/** X-coordinate on local X-axis */
    xx: number;
/** Y-coordinate on local X-axis */
    xy: number;
/** Z-coordinate on local X-axis */
    xz: number;
/** Csys is *DEFINE_COORDINATE_NODES. */
    static NODES: number;
/** Csys is *DEFINE_COORDINATE_SYSTEM. */
    static SYSTEM: number;
/** Csys is *DEFINE_COORDINATE_VECTOR. */
    static VECTOR: number;
}


/** Object function argument in CoordinateSystem constructor */
interface CoordinateSystem_constructor_Argument_details {
    /** Label of CoordinateSystem */
    cid: number;
    /** Array of coordinates of point on local X-axis [lx, ly, lz] (for option CoordinateSystem.SYSTEM) */
    cl?: array[];
    /** Array of coordinates of origin [ox, oy, oz] (for option CoordinateSystem.SYSTEM) */
    co?: array[];
    /** Array of coordinates of point in local X-Y plane [px, py, pz] (for option CoordinateSystem.SYSTEM) */
    cp?: array[];
    /** Array of coordinates of local X-Y vector [vx, vy, vz] (for option CoordinateSystem.VECTOR) */
    cv?: array[];
    /** Array of coordinates on local X-axis [xx, xy, xz] (for option CoordinateSystem.VECTOR) */
    cx?: array[];
    /** Axis defined by N1N2 (for option CoordinateSystem.NODES) */
    dir?: number;
    /** Flag for local system update for each time step (for option CoordinateSystem.NODES) */
    flag?: boolean;
    /** Title for the coordinate system */
    heading?: string;
    /** Optional Node ID for rotation (for option CoordinateSystem.VECTOR) */
    nid?: number;
    /** Array of Node IDs [n1, n2, n3] for the coordinate system (for option CoordinateSystem.NODES) */
    nodes?: array[];
    /** CoordinateSystem type (can be CoordinateSystem.NODES, CoordinateSystem.SYSTEM or CoordinateSystem.VECTOR) */
    option: number;
}

declare class Csys {
/**
 * Associates a comment with a coordinate system.
 * @param Comment Comment that will be attached to the coordinate system
 */
    AssociateComment(Comment: Comment): void;

/**
 * Blanks the coordinate system
 */
    Blank(): void;

/**
 * Blanks all of the coordinate systems in the model.
 * @param Model Model that all coordinate systems will be blanked in
 * @param redraw If model should be redrawn or not. If omitted redraw is false. If you want to do several (un)blanks and only redraw after the last one then use false for all redraws apart from the last one. Alternatively you can redraw using View.Redraw().
 */
    static BlankAll(Model: Model, redraw?: boolean): void;

/**
 * Blanks all of the flagged coordinate systems in the model.
 * @param Model Model that all the flagged coordinate systems will be blanked in
 * @param flag Flag set on the coordinate systems that you want to blank
 * @param redraw If model should be redrawn or not. If omitted redraw is false. If you want to do several (un)blanks and only redraw after the last one then use false for all redraws apart from the last one. Alternatively you can redraw using View.Redraw().
 */
    static BlankFlagged(Model: Model, flag: number, redraw?: boolean): void;

/**
 * Checks if the coordinate system is blanked or not.
 * @returns true if blanked, false if not.
 */
    Blanked(): boolean;

/**
 * Starts an edit panel in Browse mode.
 * @param modal If this window is modal (blocks the user from doing anything else in PRIMER until this window is dismissed). If omitted the window will be modal.
 */
    Browse(modal?: boolean): void;

/**
 * Clears a flag on the coordinate system.
 * @param flag Flag to clear on the coordinate system
 */
    ClearFlag(flag: number): void;

/**
 * Copies the coordinate system. The target include of the copied coordinate system can be set using Options.copy_target_include.
 * @param range If you want to keep the copied item in the range specified for the current include. Default value is false. To set current include, use Include.MakeCurrentLayer().
 * @returns CoordinateSystem object
 */
    Copy(range?: boolean): CoordinateSystem;

/**
 * Starts an interactive editing panel to create a coordinate system
 * @param Model Model that the coordinate system will be created in.
 * @param modal If this window is modal (blocks the user from doing anything else in PRIMER until this window is dismissed). If omitted the window will be modal.
 * @returns CoordinateSystem object (or null if not made).
 */
    static Create(Model: Model, modal?: boolean): CoordinateSystem;

/**
 * Detaches a comment from a coordinate system.
 * @param Comment Comment that will be detached from the coordinate system
 */
    DetachComment(Comment: Comment): void;

/**
 * Starts an interactive editing panel.
 * @param modal If this window is modal (blocks the user from doing anything else in PRIMER until this window is dismissed). If omitted the window will be modal.
 */
    Edit(modal?: boolean): void;

/**
 * Adds an error for coordinate system. For more details on checking see the Check class.
 * @param message The error message to give
 * @param details An optional detailed error message
 */
    Error(message: string, details?: string): void;

/**
 * Returns the first coordinate system in the model.
 * @param Model Model to get first coordinate system in
 * @returns CoordinateSystem object (or null if there are no coordinate systems in the model).
 */
    static First(Model: Model): CoordinateSystem;

/**
 * Returns the first free coordinate system label in the model. Also see CoordinateSystem.LastFreeLabel(), CoordinateSystem.NextFreeLabel() and Model.FirstFreeItemLabel().
 * @param Model Model to get first free coordinate system label in
 * @param layer Include file (0 for the main file) to search for labels in (Equivalent to First free in layer in editing panels). If omitted the whole model will be used (Equivalent to First free in editing panels).
 * @returns CoordinateSystem label.
 */
    static FirstFreeLabel(Model: Model, layer?: number): number;

/**
 * Flags all of the coordinate systems in the model with a defined flag.
 * @param Model Model that all coordinate systems will be flagged in
 * @param flag Flag to set on the coordinate systems
 */
    static FlagAll(Model: Model, flag: number): void;

/**
 * Checks if the coordinate system is flagged or not.
 * @param flag Flag to test on the coordinate system
 * @returns true if flagged, false if not.
 */
    Flagged(flag: number): boolean;

/**
 * Calls a function for each coordinate system in the model. Note that ForEach has been designed to make looping over coordinate systems as fast as possible and so has some limitations. Firstly, a single temporary CoordinateSystem object is created and on each function call it is updated with the current coordinate system data. This means that you should not try to store the CoordinateSystem object for later use (e.g. in an array) as it is temporary. Secondly, you cannot create new coordinate systems inside a ForEach loop. 
 * @param Model Model that all coordinate systems are in
 * @param func Function to call for each coordinate system
 * @param extra An optional extra object/array/string etc that will appended to arguments when calling the function
 */
    static ForEach(Model: Model, func: () => void, extra?: any): void;

/**
 * Returns an array of CoordinateSystem objects or properties for all of the coordinate systems in a model in PRIMER. If the optional property argument is not given then an array of CoordinateSystem objects is returned. If the property argument is given, that property value for each coordinate system is returned in the array instead of a CoordinateSystem object
 * @param Model Model to get coordinate systems from
 * @param property Name for property to get for all coordinate systems in the model
 * @returns Array of CoordinateSystem objects or properties
 */
    static GetAll(Model: Model, property?: string): CoordinateSystem[];

/**
 * Extracts the comments associated to a coordinate system.
 * @returns Array of Comment objects (or null if there are no comments associated to the node).
 */
    GetComments(): Comment[];

/**
 * Returns an array of CoordinateSystem objects for all of the flagged coordinate systems in a model in PRIMER If the optional property argument is not given then an array of CoordinateSystem objects is returned. If the property argument is given, then that property value for each coordinate system is returned in the array instead of a CoordinateSystem object
 * @param Model Model to get coordinate systems from
 * @param flag Flag set on the coordinate systems that you want to retrieve
 * @param property Name for property to get for all flagged coordinate systems in the model
 * @returns Array of CoordinateSystem objects or properties
 */
    static GetFlagged(Model: Model, flag: number, property?: string): CoordinateSystem[];

/**
 * Returns the CoordinateSystem object for a coordinate system ID.
 * @param Model Model to find the coordinate system in
 * @param number number of the coordinate system you want the CoordinateSystem object for
 * @returns CoordinateSystem object (or null if coordinate system does not exist).
 */
    static GetFromID(Model: Model, number: number): CoordinateSystem;

/**
 * Checks if a CoordinateSystem property is a parameter or not. Note that object properties that are parameters are normally returned as the integer or float parameter values as that is virtually always what the user would want. For this function to work the JavaScript interpreter must use the parameter name instead of the value. This can be done by setting the Options.property_parameter_names option to true before calling the function and then resetting it to false afterwards.. This behaviour can also temporarily be switched by using the CoordinateSystem.ViewParameters() method and 'method chaining' (see the examples below).
 * @param prop coordinate system property to get parameter for
 * @returns Parameter object if property is a parameter, null if not.
 */
    GetParameter(prop: string): Parameter;

/**
 * Returns the keyword for this csys (*DEFINE_COORDINATE). Note that a carriage return is not added. See also CoordinateSystem.KeywordCards()
 * @returns string containing the keyword.
 */
    Keyword(): string;

/**
 * Returns the keyword cards for the csys. Note that a carriage return is not added. See also CoordinateSystem.Keyword()
 * @returns string containing the cards.
 */
    KeywordCards(): string;

/**
 * Returns the last coordinate system in the model.
 * @param Model Model to get last coordinate system in
 * @returns CoordinateSystem object (or null if there are no coordinate systems in the model).
 */
    static Last(Model: Model): CoordinateSystem;

/**
 * Returns the last free coordinate system label in the model. Also see CoordinateSystem.FirstFreeLabel(), CoordinateSystem.NextFreeLabel() and see Model.LastFreeItemLabel()
 * @param Model Model to get last free coordinate system label in
 * @param layer Include file (0 for the main file) to search for labels in (Equivalent to Highest free in layer in editing panels). If omitted the whole model will be used.
 * @returns CoordinateSystem label.
 */
    static LastFreeLabel(Model: Model, layer?: number): number;

/**
 * Returns the next coordinate system in the model.
 * @returns CoordinateSystem object (or null if there are no more coordinate systems in the model).
 */
    Next(): CoordinateSystem;

/**
 * Returns the next free (highest+1) coordinate system label in the model. Also see CoordinateSystem.FirstFreeLabel(), CoordinateSystem.LastFreeLabel() and Model.NextFreeItemLabel()
 * @param Model Model to get next free coordinate system label in
 * @param layer Include file (0 for the main file) to search for labels in (Equivalent to Highest+1 in layer in editing panels). If omitted the whole model will be used (Equivalent to Highest+1 in editing panels).
 * @returns CoordinateSystem label.
 */
    static NextFreeLabel(Model: Model, layer?: number): number;

/**
 * Allows the user to pick a coordinate system.
 * @param prompt Text to display as a prompt to the user
 * @param limit If the argument is a Model then only coordinate systems from that model can be picked. If the argument is a Flag then only coordinate systems that are flagged with limit can be selected. If omitted, or null, any coordinate systems from any model can be selected. from any model.
 * @param modal If picking is modal (blocks the user from doing anything else in PRIMER until this window is dismissed). If omitted the pick will be modal.
 * @param button_text By default the window with the prompt will have a button labelled 'Cancel' which if pressed will cancel the pick and return null. If you want to change the text on the button use this argument. If omitted 'Cancel' will be used.
 * @returns CoordinateSystem object (or null if not picked)
 */
    static Pick(prompt: string, limit?: Model | number, modal?: boolean, button_text?: string): CoordinateSystem;

/**
 * Returns the previous coordinate system in the model.
 * @returns CoordinateSystem object (or null if there are no more coordinate systems in the model).
 */
    Previous(): CoordinateSystem;

/**
 * Renumbers all of the coordinate systems in the model.
 * @param Model Model that all coordinate systems will be renumbered in
 * @param start Start point for renumbering
 */
    static RenumberAll(Model: Model, start: number): void;

/**
 * Renumbers all of the flagged coordinate systems in the model.
 * @param Model Model that all the flagged coordinate systems will be renumbered in
 * @param flag Flag set on the coordinate systems that you want to renumber
 * @param start Start point for renumbering
 */
    static RenumberFlagged(Model: Model, flag: number, start: number): void;

/**
 * Allows the user to select coordinate systems using standard PRIMER object menus.
 * @param flag Flag to use when selecting coordinate systems
 * @param prompt Text to display as a prompt to the user
 * @param limit If the argument is a Model then only coordinate systems from that model can be selected. If the argument is a Flag then only coordinate systems that are flagged with limit can be selected (limit should be different to flag). If omitted, or null, any coordinate systems can be selected. from any model.
 * @param modal If selection is modal (blocks the user from doing anything else in PRIMER until this window is dismissed). If omitted the selection will be modal.
 * @returns Number of coordinate systems selected or null if menu cancelled
 */
    static Select(flag: number, prompt: string, limit?: Model | number, modal?: boolean): number;

/**
 * Sets a flag on the coordinate system.
 * @param flag Flag to set on the coordinate system
 */
    SetFlag(flag: number): void;

/**
 * Sketches the coordinate system. The coordinate system will be sketched until you either call CoordinateSystem.Unsketch(), CoordinateSystem.UnsketchAll(), Model.UnsketchAll(), or delete the model
 * @param redraw If model should be redrawn or not after the coordinate system is sketched. If omitted redraw is true. If you want to sketch several coordinate systems and only redraw after the last one then use false for redraw and call View.Redraw().
 */
    Sketch(redraw?: boolean): void;

/**
 * Sketches all of the flagged coordinate systems in the model. The coordinate systems will be sketched until you either call CoordinateSystem.Unsketch(), CoordinateSystem.UnsketchFlagged(), Model.UnsketchAll(), or delete the model
 * @param Model Model that all the flagged coordinate systems will be sketched in
 * @param flag Flag set on the coordinate systems that you want to sketch
 * @param redraw If model should be redrawn or not after the coordinate systems are sketched. If omitted redraw is true. If you want to sketch flagged coordinate systems several times and only redraw after the last one then use false for redraw and call View.Redraw().
 */
    static SketchFlagged(Model: Model, flag: number, redraw?: boolean): void;

/**
 * Returns the total number of coordinate systems in the model.
 * @param Model Model to get total for
 * @param exists true if only existing coordinate systems should be counted. If false or omitted referenced but undefined coordinate systems will also be included in the total.
 * @returns number of coordinate systems
 */
    static Total(Model: Model, exists?: boolean): number;

/**
 * Unblanks the coordinate system
 */
    Unblank(): void;

/**
 * Unblanks all of the coordinate systems in the model.
 * @param Model Model that all coordinate systems will be unblanked in
 * @param redraw If model should be redrawn or not. If omitted redraw is false. If you want to do several (un)blanks and only redraw after the last one then use false for all redraws apart from the last one. Alternatively you can redraw using View.Redraw().
 */
    static UnblankAll(Model: Model, redraw?: boolean): void;

/**
 * Unblanks all of the flagged coordinate systems in the model.
 * @param Model Model that the flagged coordinate systems will be unblanked in
 * @param flag Flag set on the coordinate systems that you want to unblank
 * @param redraw If model should be redrawn or not. If omitted redraw is false. If you want to do several (un)blanks and only redraw after the last one then use false for all redraws apart from the last one. Alternatively you can redraw using View.Redraw().
 */
    static UnblankFlagged(Model: Model, flag: number, redraw?: boolean): void;

/**
 * Unsets a defined flag on all of the coordinate systems in the model.
 * @param Model Model that the defined flag for all coordinate systems will be unset in
 * @param flag Flag to unset on the coordinate systems
 */
    static UnflagAll(Model: Model, flag: number): void;

/**
 * Unsketches the coordinate system.
 * @param redraw If model should be redrawn or not after the coordinate system is unsketched. If omitted redraw is true. If you want to unsketch several coordinate systems and only redraw after the last one then use false for redraw and call View.Redraw().
 */
    Unsketch(redraw?: boolean): void;

/**
 * Unsketches all coordinate systems.
 * @param Model Model that all coordinate systems will be unblanked in
 * @param redraw If model should be redrawn or not after the coordinate systems are unsketched. If omitted redraw is true. If you want to unsketch several things and only redraw after the last one then use false for redraw and call View.Redraw().
 */
    static UnsketchAll(Model: Model, redraw?: boolean): void;

/**
 * Unsketches all flagged coordinate systems in the model.
 * @param Model Model that all coordinate systems will be unsketched in
 * @param flag Flag set on the coordinate systems that you want to unsketch
 * @param redraw If model should be redrawn or not after the coordinate systems are unsketched. If omitted redraw is true. If you want to unsketch several things and only redraw after the last one then use false for redraw and call View.Redraw().
 */
    static UnsketchFlagged(Model: Model, flag: number, redraw?: boolean): void;

/**
 * Object properties that are parameters are normally returned as the integer or float parameter values as that is virtually always what the user would want. This function temporarily changes the behaviour so that if a property is a parameter the parameter name is returned instead. This can be used with 'method chaining' (see the example below) to make sure a property argument is correct.
 * @returns CoordinateSystem object.
 */
    ViewParameters(): CoordinateSystem;

/**
 * Adds a warning for coordinate system. For more details on checking see the Check class.
 * @param message The warning message to give
 * @param details An optional detailed warning message
 */
    Warning(message: string, details?: string): void;

/**
 * Returns the cross references for this coordinate system.
 * @returns Xrefs object.
 */
    Xrefs(): Xrefs;

/**
 * Create a new CoordinateSystem object for *DEFINE_COORDINATE_NODES.
 * @param Model Model that csys will be created in
 * @param details Details for creating the CoordinateSystem
 * @returns CoordinateSystem object
 */
    constructor(Model: Model, details: CoordinateSystem_constructor_Argument_details);

/**
 * Creates a string containing the csys data in keyword format. Note that this contains the keyword header and the keyword cards. See also CoordinateSystem.Keyword() and CoordinateSystem.KeywordCards().
 * @returns string
 */
    toString(): string;

/** CoordinateSystem number. Also see the label number. */
    cid: number;
/** Optional local coordinate system to define the points in */
    cidl: number;
/** Axis defined by N1N2 */
    dir: number;
/** true if coordinate system exists, false if referred to but not defined. */
    readonly exists: boolean;
/** Flag for updating local system each timestep */
    flag: boolean;
/** CoordinateSystem heading */
    heading: string;
/** The Include file number that the coordinate system is in. */
    include: number;
/** CoordinateSystem number. Also see the cid property which is an alternative name for this. */
    label: number;
/** X-coordinate of point on local X-axis */
    lx: number;
/** Y-coordinate of point on local X-axis */
    ly: number;
/** Z-coordinate of point on local X-axis */
    lz: number;
/** The Model number that the coordinate system is in. */
    readonly model: number;
/** Node located at local origin */
    n1: number;
/** Node located along local (dir) axis */
    n2: number;
/** Node located in local plane determined by (dir) */
    n3: number;
/** Optional node id for rotation */
    nid: number;
/** CoordinateSystem type (Can be CoordinateSystem.NODES, CoordinateSystem.SYSTEM or CoordinateSystem.VECTOR). */
    option: number;
/** X-coordinate of origin */
    ox: number;
/** Y-coordinate of origin */
    oy: number;
/** Z-coordinate of origin */
    oz: number;
/** X-coordinate of point in local X-Y plane */
    px: number;
/** Y-coordinate of point in local X-Y plane */
    py: number;
/** Z-coordinate of point in local X-Y plane */
    pz: number;
/** X-coordinate of local X-Y vector */
    vx: number;
/** Y-coordinate of local X-Y vector */
    vy: number;
/** Z-coordinate of local X-Y vector */
    vz: number;
/** X-coordinate on local X-axis */
    xx: number;
/** Y-coordinate on local X-axis */
    xy: number;
/** Z-coordinate on local X-axis */
    xz: number;
/** Csys is *DEFINE_COORDINATE_NODES. */
    static NODES: number;
/** Csys is *DEFINE_COORDINATE_SYSTEM. */
    static SYSTEM: number;
/** Csys is *DEFINE_COORDINATE_VECTOR. */
    static VECTOR: number;
}


/** Object function argument in Autosize */
interface AutosizeArgument_options {
    /** Additional percentage that the autosized cross section will be scaled by. (default: same as value specified by the preference primer*cross_section_auto_size_percent) */
    percentage?: number;
    /** If true, autosizing will ignore non-visible cut elements, false otherwise. (default: true) */
    vis_only?: boolean;
}


/** Object function argument in CreateAlongFeatureLine */
interface CreateAlongFeatureLineArgument_options {
    /** true if cross sections are to be autosized. lenl and lenm will be ignored if set. */
    autosize?: boolean;
    /** If autosize is true, cross sections will be scaled by this additional percentage after being autosized. (default: same as value specified by the preference primer*cross_section_auto_size_percent) */
    autosize_pct?: number;
    /** Direction along feature line to create cross sections. +1 for positive direction (default), -1 for negative direction. Ignored for two node method (nid2 set). */
    direction?: number;
    /** Side length of plane along L vector - non-circular cross sections only. */
    lenl: number;
    /** Side length of plane along M vector - non-circular cross sections only. */
    lenm: number;
    /** ID of feature line ending node, can be the same as nid1. If provided, cross sections will be created between the nodes nid1 and nid2 along the shortest feature line path. */
    nid2?: number;
    /** Number of cross sections to be created. Required for single node method. For two node method either num or pitch must be specified. For two node method, the last cross section will be created at the location of node nid2 (provided nid1 != nid2). */
    num: number;
    /** Separation between adjacent cross sections. Required for single node method. For two node method either num or pitch must be specified. */
    pitch: number;
    /** ID of part set. */
    psid?: number;
    /** Radius of circular cross section. */
    radius?: number;
    /** If true, and autosize is true, autosizing will ignore non-visible cut elements, false otherwise (default: true). */
    vis_only?: boolean;
}


/** Object returned by Properties */
interface PropertiesReturn {
    /** Area of material sliced by the cut section */
    area: number;
    /** First yield axial force */
    first_yield_axial: number;
    /** First yield bending moment (Mxx) */
    first_yield_mxx: number;
    /** First yield bending moment (Myy) */
    first_yield_myy: number;
    /** Fully plastic bending moment (Mxx) */
    fully_plastic_mxx: number;
    /** Fully plastic bending moment (Myy) */
    fully_plastic_myy: number;
    /** X component of equal force axis */
    fully_plastic_xf: number;
    /** X component of equal force axis calculated in global coordinates */
    fully_plastic_xf_g: number;
    /** Fully plastic axial force */
    fully_plastic_yf: number;
    /** Y component of equal force axis calculated in global coordinates */
    fully_plastic_yf_g: number;
    /** Z component of equal force axis calculated in global coordinates */
    fully_plastic_zf_g: number;
    /** Iuu principal second moments (UU - major) */
    iuu: number;
    /** Ivv principal second moments (VV - minor) */
    ivv: number;
    /** Ixx component of second moment of area */
    ixx: number;
    /** Ixy component of second moment of area */
    ixy: number;
    /** Iyy component of second moment of area */
    iyy: number;
    /** X component of section origin */
    origin_x: number;
    /** Y component of section origin */
    origin_y: number;
    /** Z component of section origin */
    origin_z: number;
    /** Angle between Ixx and Iuu */
    theta: number;
    /** X component of X-axis vector */
    x_comp_axis_x: number;
    /** X component of Y-axis vector */
    x_comp_axis_y: number;
    /** X component of Z-axis vector */
    x_comp_axis_z: number;
    /** X component of centroid calculated from the first moment of area */
    xc: number;
    /** X component of centre of gravity calculated in global coordinates */
    xc_global: number;
    /** X component of equal area axis */
    xe: number;
    /** X component of equal area axis calculated in global coordinates */
    xe_global: number;
    /** Y component of x-axis vector */
    y_comp_axis_x: number;
    /** Y component of Y-axis vector */
    y_comp_axis_y: number;
    /** Y component of Z-axis vector */
    y_comp_axis_z: number;
    /** Y component of centroid calculated from the first moment of area */
    yc: number;
    /** Y component of centre of gravity calculated in global coordinates */
    yc_global: number;
    /** Y component of equal area axis */
    ye: number;
    /** Y component of equal area axis calculated in global coordinates */
    ye_global: number;
    /** Z component of X-axis vector */
    z_comp_axis_x: number;
    /** Z component of Y-axis vector */
    z_comp_axis_y: number;
    /** Z component of Z-axis vector */
    z_comp_axis_z: number;
    /** Z component of centre of gravity calculated in global coordinates */
    zc_global: number;
    /** Z component of equal area axis calculated in global coordinates */
    ze_global: number;
    /** Plastic moduli Zxx */
    zxx: number;
    /** Plastic moduli Zyy */
    zyy: number;
}


/** Object function argument in CrossSection constructor */
interface CrossSection_constructor_Argument_settings {
    /** true if the Database cross_section is to be autosized. */
    autosize__PLANE_only?: boolean;
    /** Additional post-autosize scaling percentage. (default: same as value specified by the preference primer*cross_section_auto_size_percent) */
    autosize_pct__PLANE_only?: number;
    /** Beam set number. */
    bsid_SET_only: number;
    /** Database cross_section number. */
    csid?: number;
    /** Discrete set number. */
    dsid_SET_only: number;
    /** Database cross_section title. */
    heading?: string;
    /** Solid set number. */
    hsid_SET_only: number;
    /** Rigid part or accelerometer or coordinate system number. */
    id?: number;
    /** Flag for local system type. */
    itype?: number;
    /** Length in l direction. */
    lenl__PLANE_only?: number;
    /** Length in m direction. */
    lenm__PLANE_only?: number;
    /** Node set number. */
    nsid_SET_only: number;
    /** Part set number. */
    psid_PLANE_only: number;
    /** Shell set number. */
    ssid_SET_only: number;
    /** Thick shell set number. */
    tsid_SET_only: number;
    /** true if autosizing should ignore non-visible elements, false otherwise. (default: true) */
    vis_only__PLANE_only?: number;
    /** X coordinate of head of normal vector. */
    xch_PLANE_only: number;
    /** X coordinate of tail of normal vector. */
    xct_PLANE_only: number;
    /** X coordinate of head of edge vector. */
    xhev_PLANE_only: number;
    /** Y coordinate of head of normal vector. */
    ych_PLANE_only: number;
    /** Y coordinate of tail of normal vector. */
    yct_PLANE_only: number;
    /** Y coordinate of head of edge vector. */
    yhev_PLANE_only: number;
    /** Z coordinate of head of normal vector. */
    zch_PLANE_only: number;
    /** Z coordinate of tail of normal vector. */
    zct_PLANE_only: number;
    /** Z coordinate of head of edge vector. */
    zhev_PLANE_only: number;
}

declare class CrossSection {
/**
 * Associates a comment with a cross section.
 * @param Comment Comment that will be attached to the cross section
 */
    AssociateComment(Comment: Comment): void;

/**
 * Autosizes a _PLANE cross section such that it cuts through all elements in model/psid along that plane.
 * @param options Object containing additional options
 */
    Autosize(options?: AutosizeArgument_options): void;

/**
 * Blanks the cross section
 */
    Blank(): void;

/**
 * Blanks all of the cross sections in the model.
 * @param Model Model that all cross sections will be blanked in
 * @param redraw If model should be redrawn or not. If omitted redraw is false. If you want to do several (un)blanks and only redraw after the last one then use false for all redraws apart from the last one. Alternatively you can redraw using View.Redraw().
 */
    static BlankAll(Model: Model, redraw?: boolean): void;

/**
 * Blanks all of the flagged cross sections in the model.
 * @param Model Model that all the flagged cross sections will be blanked in
 * @param flag Flag set on the cross sections that you want to blank
 * @param redraw If model should be redrawn or not. If omitted redraw is false. If you want to do several (un)blanks and only redraw after the last one then use false for all redraws apart from the last one. Alternatively you can redraw using View.Redraw().
 */
    static BlankFlagged(Model: Model, flag: number, redraw?: boolean): void;

/**
 * Checks if the cross section is blanked or not.
 * @returns true if blanked, false if not.
 */
    Blanked(): boolean;

/**
 * Starts an edit panel in Browse mode.
 * @param modal If this window is modal (blocks the user from doing anything else in PRIMER until this window is dismissed). If omitted the window will be modal.
 */
    Browse(modal?: boolean): void;

/**
 * Clears a flag on the cross section.
 * @param flag Flag to clear on the cross section
 */
    ClearFlag(flag: number): void;

/**
 * Copies the cross section. The target include of the copied cross section can be set using Options.copy_target_include.
 * @param range If you want to keep the copied item in the range specified for the current include. Default value is false. To set current include, use Include.MakeCurrentLayer().
 * @returns CrossSection object
 */
    Copy(range?: boolean): CrossSection;

/**
 * Starts an interactive editing panel to create a cross section
 * @param Model Model that the cross section will be created in.
 * @param modal If this window is modal (blocks the user from doing anything else in PRIMER until this window is dismissed). If omitted the window will be modal.
 * @returns CrossSection object (or null if not made).
 */
    static Create(Model: Model, modal?: boolean): CrossSection;

/**
 * Creates a set of cross sections along a feature line and returns them as an array of CrossSection objects. Use Options.edge_angle to control the break angle for the feature line search within this function.
 * @param Model Model that the cross_section will be created in
 * @param nid1 ID of feature line starting node. The first cross section will be created at this Node's location.
 * @param options Additional arguments for controlling how the cross sections are created.
 * @returns Array of CrossSection objects (or null if not made). Depending on the geometry of the model and the node provided for nid1, the array may contain less CrossSection objects than requested for the single node method.
 */
    static CreateAlongFeatureLine(Model: Model, nid1: number, options: CreateAlongFeatureLineArgument_options): CrossSection[];

/**
 * Detaches a comment from a cross section.
 * @param Comment Comment that will be detached from the cross section
 */
    DetachComment(Comment: Comment): void;

/**
 * Starts an interactive editing panel.
 * @param modal If this window is modal (blocks the user from doing anything else in PRIMER until this window is dismissed). If omitted the window will be modal.
 */
    Edit(modal?: boolean): void;

/**
 * Returns coordinates of the intersections between a shell and a database cross section. Note, ElemCut on the Shell class may be quicker
 * @param Shell_label The label of the shell.
 * @returns An array containing the x1,y1,z1,x2,y2,z2 coordinates of the cut line, or NULL if it does not cut. Note this function does not check that the shell is in the cross section definition (part set)
 */
    ElemCut(Shell_label: number): number[];

/**
 * Adds an error for cross section. For more details on checking see the Check class.
 * @param message The error message to give
 * @param details An optional detailed error message
 */
    Error(message: string, details?: string): void;

/**
 * Extracts the actual colour used for cross section. By default in PRIMER many entities such as elements get their colour automatically from the part that they are in. PRIMER cycles through 13 default colours based on the label of the entity. In this case the cross section colour property will return the value Colour.PART instead of the actual colour. This method will return the actual colour which is used for drawing the cross section.
 * @returns colour value (integer)
 */
    ExtractColour(): number;

/**
 * Returns the first cross section in the model.
 * @param Model Model to get first cross section in
 * @returns CrossSection object (or null if there are no cross sections in the model).
 */
    static First(Model: Model): CrossSection;

/**
 * Returns the first free cross section label in the model. Also see CrossSection.LastFreeLabel(), CrossSection.NextFreeLabel() and Model.FirstFreeItemLabel().
 * @param Model Model to get first free cross section label in
 * @param layer Include file (0 for the main file) to search for labels in (Equivalent to First free in layer in editing panels). If omitted the whole model will be used (Equivalent to First free in editing panels).
 * @returns CrossSection label.
 */
    static FirstFreeLabel(Model: Model, layer?: number): number;

/**
 * Flags all of the cross sections in the model with a defined flag.
 * @param Model Model that all cross sections will be flagged in
 * @param flag Flag to set on the cross sections
 */
    static FlagAll(Model: Model, flag: number): void;

/**
 * Flags every element (solid,shell,tshell,beam) cut by the cross section. Note this function does not check that the element is in the cross section definition (part set)
 * @param Flag Flag bit.
 * @returns Boolean.
 */
    FlagCut(Flag: number): boolean;

/**
 * Checks if the cross section is flagged or not.
 * @param flag Flag to test on the cross section
 * @returns true if flagged, false if not.
 */
    Flagged(flag: number): boolean;

/**
 * Calls a function for each cross section in the model. Note that ForEach has been designed to make looping over cross sections as fast as possible and so has some limitations. Firstly, a single temporary CrossSection object is created and on each function call it is updated with the current cross section data. This means that you should not try to store the CrossSection object for later use (e.g. in an array) as it is temporary. Secondly, you cannot create new cross sections inside a ForEach loop. 
 * @param Model Model that all cross sections are in
 * @param func Function to call for each cross section
 * @param extra An optional extra object/array/string etc that will appended to arguments when calling the function
 */
    static ForEach(Model: Model, func: () => void, extra?: any): void;

/**
 * Returns an array of CrossSection objects or properties for all of the cross sections in a model in PRIMER. If the optional property argument is not given then an array of CrossSection objects is returned. If the property argument is given, that property value for each cross section is returned in the array instead of a CrossSection object
 * @param Model Model to get cross sections from
 * @param property Name for property to get for all cross sections in the model
 * @returns Array of CrossSection objects or properties
 */
    static GetAll(Model: Model, property?: string): CrossSection[];

/**
 * Extracts the comments associated to a cross section.
 * @returns Array of Comment objects (or null if there are no comments associated to the node).
 */
    GetComments(): Comment[];

/**
 * Returns an array of CrossSection objects for all of the flagged cross sections in a model in PRIMER If the optional property argument is not given then an array of CrossSection objects is returned. If the property argument is given, then that property value for each cross section is returned in the array instead of a CrossSection object
 * @param Model Model to get cross sections from
 * @param flag Flag set on the cross sections that you want to retrieve
 * @param property Name for property to get for all flagged cross sections in the model
 * @returns Array of CrossSection objects or properties
 */
    static GetFlagged(Model: Model, flag: number, property?: string): CrossSection[];

/**
 * Returns the CrossSection object for a cross section ID.
 * @param Model Model to find the cross section in
 * @param number number of the cross section you want the CrossSection object for
 * @returns CrossSection object (or null if cross section does not exist).
 */
    static GetFromID(Model: Model, number: number): CrossSection;

/**
 * Checks if a CrossSection property is a parameter or not. Note that object properties that are parameters are normally returned as the integer or float parameter values as that is virtually always what the user would want. For this function to work the JavaScript interpreter must use the parameter name instead of the value. This can be done by setting the Options.property_parameter_names option to true before calling the function and then resetting it to false afterwards.. This behaviour can also temporarily be switched by using the CrossSection.ViewParameters() method and 'method chaining' (see the examples below).
 * @param prop cross section property to get parameter for
 * @returns Parameter object if property is a parameter, null if not.
 */
    GetParameter(prop: string): Parameter;

/**
 * Returns the keyword for this cross_section (*DATABASE_CROSS_SECTION). Note that a carriage return is not added. See also CrossSection.KeywordCards()
 * @returns string containing the keyword.
 */
    Keyword(): string;

/**
 * Returns the keyword cards for the cross_section. Note that a carriage return is not added. See also CrossSection.Keyword()
 * @returns string containing the cards.
 */
    KeywordCards(): string;

/**
 * Returns the last cross section in the model.
 * @param Model Model to get last cross section in
 * @returns CrossSection object (or null if there are no cross sections in the model).
 */
    static Last(Model: Model): CrossSection;

/**
 * Returns the last free cross section label in the model. Also see CrossSection.FirstFreeLabel(), CrossSection.NextFreeLabel() and see Model.LastFreeItemLabel()
 * @param Model Model to get last free cross section label in
 * @param layer Include file (0 for the main file) to search for labels in (Equivalent to Highest free in layer in editing panels). If omitted the whole model will be used.
 * @returns CrossSection label.
 */
    static LastFreeLabel(Model: Model, layer?: number): number;

/**
 * Returns the next cross section in the model.
 * @returns CrossSection object (or null if there are no more cross sections in the model).
 */
    Next(): CrossSection;

/**
 * Returns the next free (highest+1) cross section label in the model. Also see CrossSection.FirstFreeLabel(), CrossSection.LastFreeLabel() and Model.NextFreeItemLabel()
 * @param Model Model to get next free cross section label in
 * @param layer Include file (0 for the main file) to search for labels in (Equivalent to Highest+1 in layer in editing panels). If omitted the whole model will be used (Equivalent to Highest+1 in editing panels).
 * @returns CrossSection label.
 */
    static NextFreeLabel(Model: Model, layer?: number): number;

/**
 * Returns true if cross section is cutting the part, false otherwise. If option flag is active, will flag every element of the part cut by the cross section. Note this function does not check that the part is in the cross section definition (part set)
 * @param Part_label The label of the part.
 * @param Flag Optional Flag to flag the element which are cut by the cross section.
 * @returns Boolean.
 */
    PartCut(Part_label: number, Flag?: number): boolean;

/**
 * Allows the user to pick a cross section.
 * @param prompt Text to display as a prompt to the user
 * @param limit If the argument is a Model then only cross sections from that model can be picked. If the argument is a Flag then only cross sections that are flagged with limit can be selected. If omitted, or null, any cross sections from any model can be selected. from any model.
 * @param modal If picking is modal (blocks the user from doing anything else in PRIMER until this window is dismissed). If omitted the pick will be modal.
 * @param button_text By default the window with the prompt will have a button labelled 'Cancel' which if pressed will cancel the pick and return null. If you want to change the text on the button use this argument. If omitted 'Cancel' will be used.
 * @returns CrossSection object (or null if not picked)
 */
    static Pick(prompt: string, limit?: Model | number, modal?: boolean, button_text?: string): CrossSection;

/**
 * Returns the previous cross section in the model.
 * @returns CrossSection object (or null if there are no more cross sections in the model).
 */
    Previous(): CrossSection;

/**
 * Returns an object which describe various cross section properties
 */
    Properties(): PropertiesReturn;

/**
 * Renumbers all of the cross sections in the model.
 * @param Model Model that all cross sections will be renumbered in
 * @param start Start point for renumbering
 */
    static RenumberAll(Model: Model, start: number): void;

/**
 * Renumbers all of the flagged cross sections in the model.
 * @param Model Model that all the flagged cross sections will be renumbered in
 * @param flag Flag set on the cross sections that you want to renumber
 * @param start Start point for renumbering
 */
    static RenumberFlagged(Model: Model, flag: number, start: number): void;

/**
 * Allows the user to select cross sections using standard PRIMER object menus.
 * @param flag Flag to use when selecting cross sections
 * @param prompt Text to display as a prompt to the user
 * @param limit If the argument is a Model then only cross sections from that model can be selected. If the argument is a Flag then only cross sections that are flagged with limit can be selected (limit should be different to flag). If omitted, or null, any cross sections can be selected. from any model.
 * @param modal If selection is modal (blocks the user from doing anything else in PRIMER until this window is dismissed). If omitted the selection will be modal.
 * @returns Number of cross sections selected or null if menu cancelled
 */
    static Select(flag: number, prompt: string, limit?: Model | number, modal?: boolean): number;

/**
 * Sets a flag on the cross section.
 * @param flag Flag to set on the cross section
 */
    SetFlag(flag: number): void;

/**
 * Sketches the cross section. The cross section will be sketched until you either call CrossSection.Unsketch(), CrossSection.UnsketchAll(), Model.UnsketchAll(), or delete the model
 * @param redraw If model should be redrawn or not after the cross section is sketched. If omitted redraw is true. If you want to sketch several cross sections and only redraw after the last one then use false for redraw and call View.Redraw().
 */
    Sketch(redraw?: boolean): void;

/**
 * Sketches all of the flagged cross sections in the model. The cross sections will be sketched until you either call CrossSection.Unsketch(), CrossSection.UnsketchFlagged(), Model.UnsketchAll(), or delete the model
 * @param Model Model that all the flagged cross sections will be sketched in
 * @param flag Flag set on the cross sections that you want to sketch
 * @param redraw If model should be redrawn or not after the cross sections are sketched. If omitted redraw is true. If you want to sketch flagged cross sections several times and only redraw after the last one then use false for redraw and call View.Redraw().
 */
    static SketchFlagged(Model: Model, flag: number, redraw?: boolean): void;

/**
 * Returns the total number of cross sections in the model.
 * @param Model Model to get total for
 * @param exists true if only existing cross sections should be counted. If false or omitted referenced but undefined cross sections will also be included in the total.
 * @returns number of cross sections
 */
    static Total(Model: Model, exists?: boolean): number;

/**
 * Unblanks the cross section
 */
    Unblank(): void;

/**
 * Unblanks all of the cross sections in the model.
 * @param Model Model that all cross sections will be unblanked in
 * @param redraw If model should be redrawn or not. If omitted redraw is false. If you want to do several (un)blanks and only redraw after the last one then use false for all redraws apart from the last one. Alternatively you can redraw using View.Redraw().
 */
    static UnblankAll(Model: Model, redraw?: boolean): void;

/**
 * Unblanks all of the flagged cross sections in the model.
 * @param Model Model that the flagged cross sections will be unblanked in
 * @param flag Flag set on the cross sections that you want to unblank
 * @param redraw If model should be redrawn or not. If omitted redraw is false. If you want to do several (un)blanks and only redraw after the last one then use false for all redraws apart from the last one. Alternatively you can redraw using View.Redraw().
 */
    static UnblankFlagged(Model: Model, flag: number, redraw?: boolean): void;

/**
 * Unsets a defined flag on all of the cross sections in the model.
 * @param Model Model that the defined flag for all cross sections will be unset in
 * @param flag Flag to unset on the cross sections
 */
    static UnflagAll(Model: Model, flag: number): void;

/**
 * Unsketches the cross section.
 * @param redraw If model should be redrawn or not after the cross section is unsketched. If omitted redraw is true. If you want to unsketch several cross sections and only redraw after the last one then use false for redraw and call View.Redraw().
 */
    Unsketch(redraw?: boolean): void;

/**
 * Unsketches all cross sections.
 * @param Model Model that all cross sections will be unblanked in
 * @param redraw If model should be redrawn or not after the cross sections are unsketched. If omitted redraw is true. If you want to unsketch several things and only redraw after the last one then use false for redraw and call View.Redraw().
 */
    static UnsketchAll(Model: Model, redraw?: boolean): void;

/**
 * Unsketches all flagged cross sections in the model.
 * @param Model Model that all cross sections will be unsketched in
 * @param flag Flag set on the cross sections that you want to unsketch
 * @param redraw If model should be redrawn or not after the cross sections are unsketched. If omitted redraw is true. If you want to unsketch several things and only redraw after the last one then use false for redraw and call View.Redraw().
 */
    static UnsketchFlagged(Model: Model, flag: number, redraw?: boolean): void;

/**
 * Object properties that are parameters are normally returned as the integer or float parameter values as that is virtually always what the user would want. This function temporarily changes the behaviour so that if a property is a parameter the parameter name is returned instead. This can be used with 'method chaining' (see the example below) to make sure a property argument is correct.
 * @returns CrossSection object.
 */
    ViewParameters(): CrossSection;

/**
 * Adds a warning for cross section. For more details on checking see the Check class.
 * @param message The warning message to give
 * @param details An optional detailed warning message
 */
    Warning(message: string, details?: string): void;

/**
 * Returns the cross references for this cross section.
 * @returns Xrefs object.
 */
    Xrefs(): Xrefs;

/**
 * Create a new CrossSection object.
 * @param Model Model that database cross section will be created in
 * @param option Database cross section type. Must be CrossSection.SET or CrossSection.PLANE
 * @param settings Options specifying various properties used to create the keyword. If optional values are not specified then their default values will be used.
 * @returns CrossSection object
 */
    constructor(Model: Model, option: number, settings: CrossSection_constructor_Argument_settings);

/**
 * Creates a string containing the cross_section data in keyword format. Note that this contains the keyword header and the keyword cards. See also CrossSection.Keyword() and CrossSection.KeywordCards().
 * @returns string
 */
    toString(): string;

/** Beam set number. */
    bsid: number;
/** The colour of the cross section */
    colour: Colour;
/** Database cross section number (identical to label). */
    csid: number;
/** Discrete set number. */
    dsid: number;
/** true if cross section exists, false if referred to but not defined. */
    readonly exists: boolean;
/** Database cross section heading. */
    heading: string;
/** Solid set number. */
    hsid: number;
/** Rigid part or accelerometer or coordinate system number. */
    id: number;
/** true if _ID option is set, false if not */
    idset: boolean;
/** The Include file number that the cross section is in. */
    include: number;
/** Flag for local system type. */
    itype: number;
/** Database cross section number. */
    label: number;
/** Length of L edge. */
    lenl: number;
/** Length of M edge. */
    lenm: number;
/** The Model number that the cross section is in. */
    readonly model: number;
/** Node set number. */
    nsid: number;
/** The Database CrossSection option. Can be:  CrossSection.PLANE or CrossSection.SET  */
    option: number;
/** Part set number. */
    psid: number;
/** Radius. */
    radius: number;
/** Shell set number. */
    ssid: number;
/** Thick shell set number. */
    tsid: number;
/** Head X coord of N normal vector. If &lt;radius&gt; is negative &lt;xch&gt; is a node ID which, combined with &lt;xct&gt;, defines the normal vector of the cut plane. */
    xch: number;
/** Tail X coord of N normal vector. If &lt;radius&gt; is negative &lt;xct&gt; is a node ID whose coordinates define the centre of the circular cut plane. */
    xct: number;
/** Head X coord of L edge vector. */
    xhev: number;
/** Head Y coord of N normal vector. */
    ych: number;
/** Tail Y coord of N normal vector. */
    yct: number;
/** Head Y coord of L edge vector. */
    yhev: number;
/** Head Z coord of N normal vector. */
    zch: number;
/** Tail Z coord of N normal vector. */
    zct: number;
/** Head Z coord of L edge vector. */
    zhev: number;
/** PLANE is *DATABASE_CROSS_SECTION_PLANE. */
    static PLANE: number;
/** SET is *DATABASE_CROSS_SECTION_SET. */
    static SET: number;
}


/** Object function argument in Curve constructor */
interface Curve_constructor_Argument_options {
    /** Data type */
    dattyp?: number;
    /** Total distance tool will travel */
    dist_Type_of_load_curve_must_be_Curve.CURVE_SMOOTH?: number;
    /** Function expression */
    func_load_curve_type_has_to_be_Curve.FUNCTION_or_Curve.CURVE_FUNCTION?: string;
    /** Title for the curve */
    heading?: string;
    /** Curve number */
    lcid: number;
    /** Data type */
    lcint?: number;
    /** Offset on abscissa value */
    offa?: number;
    /** Offset on ordinate value */
    offo?: number;
    /** Scale factor on abscissa value */
    sfa?: number;
    /** Scale factor on ordinate value */
    sfo?: number;
    /** Stress initialisation by dynamic relaxation */
    sidr?: number;
    /** Time curve returns to zero */
    tend_Type_of_load_curve_must_be_Curve.CURVE_SMOOTH?: number;
    /** Rise time */
    trise_Type_of_load_curve_must_be_Curve.CURVE_SMOOTH?: number;
    /** Time curve starts to rise */
    tstart_Type_of_load_curve_must_be_Curve.CURVE_SMOOTH?: number;
    /** Type of load curve. Can be Curve.CURVE, Curve.TABLE, Note this does not have to be defined. In previous versions of PRIMER you could only construct a basic load curve type, therefore the type argument was not used. PRIMER is still backwards compatible with this method of load curve creation. */
    type: number;
    /** Maximum velocity */
    vmax_Type_of_load_curve_must_be_Curve.CURVE_SMOOTH?: number;
}

declare class Curve {
/**
 * Adds a point to a load curve.
 * @param xvalue The x value of the point.
 * @param yvalue The y value of the point.
 */
    AddPoint(xvalue: number, yvalue: number): void;

/**
 * Adds an entry line to a table.
 * @param value The value for for this entry in the table.
 * @param load_curve The load curve corresponding to the defined value.
 */
    AddTableEntry(value: number, load_curve: number): void;

/**
 * Associates a comment with a curve.
 * @param Comment Comment that will be attached to the curve
 */
    AssociateComment(Comment: Comment): void;

/**
 * Starts an edit panel in Browse mode.
 * @param modal If this window is modal (blocks the user from doing anything else in PRIMER until this window is dismissed). If omitted the window will be modal.
 */
    Browse(modal?: boolean): void;

/**
 * Clears a flag on the curve.
 * @param flag Flag to clear on the curve
 */
    ClearFlag(flag: number): void;

/**
 * Copies the curve. The target include of the copied curve can be set using Options.copy_target_include.
 * @param range If you want to keep the copied item in the range specified for the current include. Default value is false. To set current include, use Include.MakeCurrentLayer().
 * @returns Curve object
 */
    Copy(range?: boolean): Curve;

/**
 * Starts an interactive editing panel to create a curve.
 * @param Model Model that the curve will be created in
 * @param modal If this window is modal (blocks the user from doing anything else in PRIMER until this window is dismissed). If omitted the window will be modal.
 * @returns Curve object (or null if not made)
 */
    static Create(Model: Model, modal?: boolean): Curve;

/**
 * Starts an interactive editing panel to create a table.
 * @param Model Model that the curve will be created in
 * @param modal If this window is modal (blocks the user from doing anything else in PRIMER until this window is dismissed). If omitted the window will be modal.
 * @returns Curve object (or null if not made)
 */
    static CreateTable(Model: Model, modal?: boolean): Curve;

/**
 * Detaches a comment from a curve.
 * @param Comment Comment that will be detached from the curve
 */
    DetachComment(Comment: Comment): void;

/**
 * Starts an interactive editing panel.
 * @param modal If this window is modal (blocks the user from doing anything else in PRIMER until this window is dismissed). If omitted the window will be modal.
 */
    Edit(modal?: boolean): void;

/**
 * Adds an error for curve. For more details on checking see the Check class.
 * @param message The error message to give
 * @param details An optional detailed error message
 */
    Error(message: string, details?: string): void;

/**
 * Returns the first curve in the model.
 * @param Model Model to get first curve in
 * @returns Curve object (or null if there are no curves in the model).
 */
    static First(Model: Model): Curve;

/**
 * Returns the first free curve label in the model. Also see Curve.LastFreeLabel(), Curve.NextFreeLabel() and Model.FirstFreeItemLabel().
 * @param Model Model to get first free curve label in
 * @param layer Include file (0 for the main file) to search for labels in (Equivalent to First free in layer in editing panels). If omitted the whole model will be used (Equivalent to First free in editing panels).
 * @returns Curve label.
 */
    static FirstFreeLabel(Model: Model, layer?: number): number;

/**
 * Flags all of the curves in the model with a defined flag.
 * @param Model Model that all curves will be flagged in
 * @param flag Flag to set on the curves
 */
    static FlagAll(Model: Model, flag: number): void;

/**
 * Checks if the curve is flagged or not.
 * @param flag Flag to test on the curve
 * @returns true if flagged, false if not.
 */
    Flagged(flag: number): boolean;

/**
 * Calls a function for each curve in the model. Note that ForEach has been designed to make looping over curves as fast as possible and so has some limitations. Firstly, a single temporary Curve object is created and on each function call it is updated with the current curve data. This means that you should not try to store the Curve object for later use (e.g. in an array) as it is temporary. Secondly, you cannot create new curves inside a ForEach loop. 
 * @param Model Model that all curves are in
 * @param func Function to call for each curve
 * @param extra An optional extra object/array/string etc that will appended to arguments when calling the function
 */
    static ForEach(Model: Model, func: () => void, extra?: any): void;

/**
 * Returns an array of Curve objects or properties for all of the curves in a model in PRIMER. If the optional property argument is not given then an array of Curve objects is returned. If the property argument is given, that property value for each curve is returned in the array instead of a Curve object
 * @param Model Model to get curves from
 * @param property Name for property to get for all curves in the model
 * @returns Array of Curve objects or properties
 */
    static GetAll(Model: Model, property?: string): Curve[];

/**
 * Extracts the comments associated to a curve.
 * @returns Array of Comment objects (or null if there are no comments associated to the node).
 */
    GetComments(): Comment[];

/**
 * Returns an array of Curve objects for all of the flagged curves in a model in PRIMER If the optional property argument is not given then an array of Curve objects is returned. If the property argument is given, then that property value for each curve is returned in the array instead of a Curve object
 * @param Model Model to get curves from
 * @param flag Flag set on the curves that you want to retrieve
 * @param property Name for property to get for all flagged curves in the model
 * @returns Array of Curve objects or properties
 */
    static GetFlagged(Model: Model, flag: number, property?: string): Curve[];

/**
 * Returns the Curve object for a curve ID.
 * @param Model Model to find the curve in
 * @param number number of the curve you want the Curve object for
 * @returns Curve object (or null if curve does not exist).
 */
    static GetFromID(Model: Model, number: number): Curve;

/**
 * Checks if a Curve property is a parameter or not. Note that object properties that are parameters are normally returned as the integer or float parameter values as that is virtually always what the user would want. For this function to work the JavaScript interpreter must use the parameter name instead of the value. This can be done by setting the Options.property_parameter_names option to true before calling the function and then resetting it to false afterwards.. This behaviour can also temporarily be switched by using the Curve.ViewParameters() method and 'method chaining' (see the examples below).
 * @param prop curve property to get parameter for
 * @returns Parameter object if property is a parameter, null if not.
 */
    GetParameter(prop: string): Parameter;

/**
 * Returns x and y data for a point in a curve
 * @param row The row point you want the data for. Note that curve points start at 0, not 1.
 * @returns An array containing the x coordinate and the y coordinate.
 */
    GetPoint(row: number): number[];

/**
 * Returns the value and curve label for a row in a table
 * @param row The row point you want the data for. Note that curve points start at 0, not 1.
 * @returns An array containing the value and the load curve label.
 */
    GetTableEntry(row: number): number[];

/**
 * Inserts point values before or after a specified row of data on a load curve.
 * @param ipt The row you want to insert the data before or after. Note that the row data starts at 0, not 1.
 * @param xvalue The x value of the point.
 * @param yvalue The y value of the point.
 * @param position Specify either before or after the selected row. Use 'Curve.BEFORE' for before, and 'Curve.AFTER' for after.
 */
    InsertPoint(ipt: number, xvalue: number, yvalue: number, position: number): void;

/**
 * Inserts a table row before or after a specified row of data on a table.
 * @param ipt The row you want to insert the data before or after. Note that the row data starts at 0, not 1.
 * @param value The value of the row.
 * @param lcid The load curve corresponding to the defined value.
 * @param position Specify either before or after the selected row. Use 'Curve.BEFORE' for before, and 'Curve.AFTER' for after.
 */
    InsertTableEntry(ipt: number, value: number, lcid: number, position: number): void;

/**
 * Returns the keyword for this curve (*DEFINE_CURVE_xxxx). Note that a carriage return is not added. See also Curve.KeywordCards()
 * @returns string containing the keyword.
 */
    Keyword(): string;

/**
 * Returns the keyword cards for the curve. Note that a carriage return is not added. See also Curve.Keyword()
 * @returns string containing the cards.
 */
    KeywordCards(): string;

/**
 * Returns the last curve in the model.
 * @param Model Model to get last curve in
 * @returns Curve object (or null if there are no curves in the model).
 */
    static Last(Model: Model): Curve;

/**
 * Returns the last free curve label in the model. Also see Curve.FirstFreeLabel(), Curve.NextFreeLabel() and see Model.LastFreeItemLabel()
 * @param Model Model to get last free curve label in
 * @param layer Include file (0 for the main file) to search for labels in (Equivalent to Highest free in layer in editing panels). If omitted the whole model will be used.
 * @returns Curve label.
 */
    static LastFreeLabel(Model: Model, layer?: number): number;

/**
 * Returns the next curve in the model.
 * @returns Curve object (or null if there are no more curves in the model).
 */
    Next(): Curve;

/**
 * Returns the next free (highest+1) curve label in the model. Also see Curve.FirstFreeLabel(), Curve.LastFreeLabel() and Model.NextFreeItemLabel()
 * @param Model Model to get next free curve label in
 * @param layer Include file (0 for the main file) to search for labels in (Equivalent to Highest+1 in layer in editing panels). If omitted the whole model will be used (Equivalent to Highest+1 in editing panels).
 * @returns Curve label.
 */
    static NextFreeLabel(Model: Model, layer?: number): number;

/**
 * Returns the previous curve in the model.
 * @returns Curve object (or null if there are no more curves in the model).
 */
    Previous(): Curve;

/**
 * Removes a row of data from a curve
 * @param row The row point you want to remove. Note that curve points start at 0, not 1.
 */
    RemovePoint(row: number): void;

/**
 * Removes the value and loadcurve values for a specified row of data on a load curve.
 * @param ipt The row you want to remove the data for. Note that the row data starts at 0, not 1.
 */
    RemoveTableEntry(ipt: number): void;

/**
 * Renumbers all of the curves in the model.
 * @param Model Model that all curves will be renumbered in
 * @param start Start point for renumbering
 */
    static RenumberAll(Model: Model, start: number): void;

/**
 * Renumbers all of the flagged curves in the model.
 * @param Model Model that all the flagged curves will be renumbered in
 * @param flag Flag set on the curves that you want to renumber
 * @param start Start point for renumbering
 */
    static RenumberFlagged(Model: Model, flag: number, start: number): void;

/**
 * Allows the user to select curves using standard PRIMER object menus.
 * @param flag Flag to use when selecting curves
 * @param prompt Text to display as a prompt to the user
 * @param limit If the argument is a Model then only curves from that model can be selected. If the argument is a Flag then only curves that are flagged with limit can be selected (limit should be different to flag). If omitted, or null, any curves can be selected. from any model.
 * @param modal If selection is modal (blocks the user from doing anything else in PRIMER until this window is dismissed). If omitted the selection will be modal.
 * @returns Number of curves selected or null if menu cancelled
 */
    static Select(flag: number, prompt: string, limit?: Model | number, modal?: boolean): number;

/**
 * Sets a flag on the curve.
 * @param flag Flag to set on the curve
 */
    SetFlag(flag: number): void;

/**
 * Sets the x and y values for a specified row of data on a load curve.
 * @param ipt The row you want to set the data for. Note that the row data starts at 0, not 1.
 * @param xvalue The x value of the point.
 * @param yvalue The y value of the point.
 */
    SetPoint(ipt: number, xvalue: number, yvalue: number): void;

/**
 * Sets the value and loadcurve values for a specified row of data on a load curve.
 * @param ipt The row you want to set the data for. Note that the row data starts at 0, not 1.
 * @param value The value for for this entry in the table.
 * @param load_curve The load curve corresponding to the defined value.
 */
    SetTableEntry(ipt: number, value: number, load_curve: number): void;

/**
 * Returns the total number of curves in the model.
 * @param Model Model to get total for
 * @param exists true if only existing curves should be counted. If false or omitted referenced but undefined curves will also be included in the total.
 * @returns number of curves
 */
    static Total(Model: Model, exists?: boolean): number;

/**
 * Unsets a defined flag on all of the curves in the model.
 * @param Model Model that the defined flag for all curves will be unset in
 * @param flag Flag to unset on the curves
 */
    static UnflagAll(Model: Model, flag: number): void;

/**
 * Object properties that are parameters are normally returned as the integer or float parameter values as that is virtually always what the user would want. This function temporarily changes the behaviour so that if a property is a parameter the parameter name is returned instead. This can be used with 'method chaining' (see the example below) to make sure a property argument is correct.
 * @returns Curve object.
 */
    ViewParameters(): Curve;

/**
 * Adds a warning for curve. For more details on checking see the Check class.
 * @param message The warning message to give
 * @param details An optional detailed warning message
 */
    Warning(message: string, details?: string): void;

/**
 * Returns the cross references for this curve.
 * @returns Xrefs object.
 */
    Xrefs(): Xrefs;

/**
 * Create a new Curve object.
 * @param Model Model that curve will be created in
 * @param options Options for creating the curve
 * @returns Curve object
 */
    constructor(Model: Model, options: Curve_constructor_Argument_options);

/**
 * Creates a string containing the curve data in keyword format. Note that this contains the keyword header and the keyword cards. See also Curve.Keyword() and Curve.KeywordCards().
 * @returns string
 */
    toString(): string;

/** Data type */
    dattyp: number;
/** Total distance tool will travel */
    dist: number;
/** true if curve exists, false if referred to but not defined. */
    readonly exists: boolean;
/** Function expression for Curve.CURVE_FUNCTION */
    function: string;
/** Curve heading */
    heading: string;
/** The Include file number that the curve is in. */
    include: number;
/** Curve number. Also see the lcid property which is an alternative name for this. */
    label: number;
/** Curve number. Also see the label property which is an alternative name for this. */
    lcid: number;
/** Number of discretization points for the curve */
    lcint: number;
/** The Model number that the curve is in. */
    readonly model: number;
/** Number of points in curve or number of curves in table. The npoints property is an alternative name for this. (read only for tables) */
    ncurves: number;
/** Number of points in curve or number of curves in table. The ncurves property is an alternative name for this. (read only for tables) */
    npoints: number;
/** Offset for abscissa values */
    offa: number;
/** Offset for ordinate values */
    offo: number;
/** Scale factor on abscissa value */
    sfa: number;
/** Scale factor on ordinate value */
    sfo: number;
/** Stress initialisation by dynamic relaxation */
    sidr: number;
/** Time curve returns to zero */
    tend: number;
/** Rise time */
    trise: number;
/** Time curve starts to rise */
    tstart: number;
/** Load curve type (Can be Curve.CURVE, Curve.CURVE_FUNCTION, Curve.SMOOTH, Curve.FUNCTION or Curve.TABLE). */
    type: number;
/** Version for discretization. Can be blank, "3858" or "5434a" */
    version: string;
/** Maximum velocity */
    vmax: number;
/** Insertion of curve data option. */
    static AFTER: number;
/** Insertion of curve data option. */
    static BEFORE: number;
/** Load curve type *DEFINE_CURVE */
    static CURVE: number;
/** Load curve type *DEFINE_CURVE_FUNCTION */
    static CURVE_FUNCTION: number;
/** Load curve type *DEFINE_CURVE_SMOOTH */
    static CURVE_SMOOTH: number;
/** Load curve type *DEFINE_FUNCTION */
    static FUNCTION: number;
/** Load curve type *DEFINE_TABLE */
    static TABLE: number;
}

declare class Damping {
/** Calculation interval */
    calcdt: number;
/** true if damping card exists */
    exists: boolean;
/** true if damping card exists */
    exists_1: boolean;
/** true if damping card exists */
    exists_2: boolean;
/** Filename */
    f_name: number;
/** Constant structural damping coefficient */
    g: number;
/** *DAMPING_GLOBAL card */
    global: object;
/** The Include file number that the damping card is in. */
    include: number;
/** The Include file number that the damping card is in. */
    include_1: number;
/** The Include file number that the damping card is in. */
    include_2: number;
/** Curve ID specifying system damping constant */
    lcid: number;
/** Curve ID Curve ID to define frequency dependent structural damping coefficients */
    lcid_1: number;
/** Type of load curve defining structural damping coefficients */
    lctyp: number;
/** *DAMPING_MODAL card */
    modal: object;
/** Printout flag */
    prflg: number;
/** Scale factor on global x rotational damping moments */
    srx: number;
/** Scale factor on global y rotational damping moments */
    sry: number;
/** Scale factor on global z rotational damping moments */
    srz: number;
/** *DAMPING_STRUCTURAL card */
    structural: object;
/** Scale factor on global x translational damping forces */
    stx: number;
/** Scale factor on global y translational damping forces */
    sty: number;
/** Scale factor on global z translational damping forces */
    stz: number;
/** System damping constant */
    valdmp: number;
}

declare class DampingFrequencyRange {
/**
 * Associates a comment with a damping frequency range.
 * @param Comment Comment that will be attached to the damping frequency range
 */
    AssociateComment(Comment: Comment): void;

/**
 * Blanks the damping frequency range
 */
    Blank(): void;

/**
 * Blanks all of the damping frequency ranges in the model.
 * @param Model Model that all damping frequency ranges will be blanked in
 * @param redraw If model should be redrawn or not. If omitted redraw is false. If you want to do several (un)blanks and only redraw after the last one then use false for all redraws apart from the last one. Alternatively you can redraw using View.Redraw().
 */
    static BlankAll(Model: Model, redraw?: boolean): void;

/**
 * Blanks all of the flagged damping frequency ranges in the model.
 * @param Model Model that all the flagged damping frequency ranges will be blanked in
 * @param flag Flag set on the damping frequency ranges that you want to blank
 * @param redraw If model should be redrawn or not. If omitted redraw is false. If you want to do several (un)blanks and only redraw after the last one then use false for all redraws apart from the last one. Alternatively you can redraw using View.Redraw().
 */
    static BlankFlagged(Model: Model, flag: number, redraw?: boolean): void;

/**
 * Checks if the damping frequency range is blanked or not.
 * @returns true if blanked, false if not.
 */
    Blanked(): boolean;

/**
 * Clears a flag on the damping frequency range.
 * @param flag Flag to clear on the damping frequency range
 */
    ClearFlag(flag: number): void;

/**
 * Copies the damping frequency range. The target include of the copied damping frequency range can be set using Options.copy_target_include.
 * @param range If you want to keep the copied item in the range specified for the current include. Default value is false. To set current include, use Include.MakeCurrentLayer().
 * @returns DampingFrequencyRange object
 */
    Copy(range?: boolean): DampingFrequencyRange;

/**
 * Detaches a comment from a damping frequency range.
 * @param Comment Comment that will be detached from the damping frequency range
 */
    DetachComment(Comment: Comment): void;

/**
 * Adds an error for damping frequency range. For more details on checking see the Check class.
 * @param message The error message to give
 * @param details An optional detailed error message
 */
    Error(message: string, details?: string): void;

/**
 * Returns the first damping frequency range in the model.
 * @param Model Model to get first damping frequency range in
 * @returns DampingFrequencyRange object (or null if there are no damping frequency ranges in the model).
 */
    static First(Model: Model): DampingFrequencyRange;

/**
 * Flags all of the damping frequency ranges in the model with a defined flag.
 * @param Model Model that all damping frequency ranges will be flagged in
 * @param flag Flag to set on the damping frequency ranges
 */
    static FlagAll(Model: Model, flag: number): void;

/**
 * Checks if the damping frequency range is flagged or not.
 * @param flag Flag to test on the damping frequency range
 * @returns true if flagged, false if not.
 */
    Flagged(flag: number): boolean;

/**
 * Calls a function for each damping frequency range in the model. Note that ForEach has been designed to make looping over damping frequency ranges as fast as possible and so has some limitations. Firstly, a single temporary DampingFrequencyRange object is created and on each function call it is updated with the current damping frequency range data. This means that you should not try to store the DampingFrequencyRange object for later use (e.g. in an array) as it is temporary. Secondly, you cannot create new damping frequency ranges inside a ForEach loop. 
 * @param Model Model that all damping frequency ranges are in
 * @param func Function to call for each damping frequency range
 * @param extra An optional extra object/array/string etc that will appended to arguments when calling the function
 */
    static ForEach(Model: Model, func: () => void, extra?: any): void;

/**
 * Returns an array of DampingFrequencyRange objects or properties for all of the damping frequency ranges in a model in PRIMER. If the optional property argument is not given then an array of DampingFrequencyRange objects is returned. If the property argument is given, that property value for each damping frequency range is returned in the array instead of a DampingFrequencyRange object
 * @param Model Model to get damping frequency ranges from
 * @param property Name for property to get for all damping frequency ranges in the model
 * @returns Array of DampingFrequencyRange objects or properties
 */
    static GetAll(Model: Model, property?: string): DampingFrequencyRange[];

/**
 * Extracts the comments associated to a damping frequency range.
 * @returns Array of Comment objects (or null if there are no comments associated to the node).
 */
    GetComments(): Comment[];

/**
 * Returns an array of DampingFrequencyRange objects for all of the flagged damping frequency ranges in a model in PRIMER If the optional property argument is not given then an array of DampingFrequencyRange objects is returned. If the property argument is given, then that property value for each damping frequency range is returned in the array instead of a DampingFrequencyRange object
 * @param Model Model to get damping frequency ranges from
 * @param flag Flag set on the damping frequency ranges that you want to retrieve
 * @param property Name for property to get for all flagged damping frequency ranges in the model
 * @returns Array of DampingFrequencyRange objects or properties
 */
    static GetFlagged(Model: Model, flag: number, property?: string): DampingFrequencyRange[];

/**
 * Returns the DampingFrequencyRange object for a damping frequency range ID.
 * @param Model Model to find the damping frequency range in
 * @param number number of the damping frequency range you want the DampingFrequencyRange object for
 * @returns DampingFrequencyRange object (or null if damping frequency range does not exist).
 */
    static GetFromID(Model: Model, number: number): DampingFrequencyRange;

/**
 * Checks if a DampingFrequencyRange property is a parameter or not. Note that object properties that are parameters are normally returned as the integer or float parameter values as that is virtually always what the user would want. For this function to work the JavaScript interpreter must use the parameter name instead of the value. This can be done by setting the Options.property_parameter_names option to true before calling the function and then resetting it to false afterwards.. This behaviour can also temporarily be switched by using the DampingFrequencyRange.ViewParameters() method and 'method chaining' (see the examples below).
 * @param prop damping frequency range property to get parameter for
 * @returns Parameter object if property is a parameter, null if not.
 */
    GetParameter(prop: string): Parameter;

/**
 * Returns the keyword for this damping frequency range (*DAMPING_FREQUENCY_RANGE). Note that a carriage return is not added. See also DampingFrequencyRange.KeywordCards()
 * @returns string containing the keyword.
 */
    Keyword(): string;

/**
 * Returns the keyword cards for the damping frequency range. Note that a carriage return is not added. See also DampingFrequencyRange.Keyword()
 * @returns string containing the cards.
 */
    KeywordCards(): string;

/**
 * Returns the last damping frequency range in the model.
 * @param Model Model to get last damping frequency range in
 * @returns DampingFrequencyRange object (or null if there are no damping frequency ranges in the model).
 */
    static Last(Model: Model): DampingFrequencyRange;

/**
 * Returns the next damping frequency range in the model.
 * @returns DampingFrequencyRange object (or null if there are no more damping frequency ranges in the model).
 */
    Next(): DampingFrequencyRange;

/**
 * Allows the user to pick a damping frequency range.
 * @param prompt Text to display as a prompt to the user
 * @param limit If the argument is a Model then only damping frequency ranges from that model can be picked. If the argument is a Flag then only damping frequency ranges that are flagged with limit can be selected. If omitted, or null, any damping frequency ranges from any model can be selected. from any model.
 * @param modal If picking is modal (blocks the user from doing anything else in PRIMER until this window is dismissed). If omitted the pick will be modal.
 * @param button_text By default the window with the prompt will have a button labelled 'Cancel' which if pressed will cancel the pick and return null. If you want to change the text on the button use this argument. If omitted 'Cancel' will be used.
 * @returns DampingFrequencyRange object (or null if not picked)
 */
    static Pick(prompt: string, limit?: Model | number, modal?: boolean, button_text?: string): DampingFrequencyRange;

/**
 * Returns the previous damping frequency range in the model.
 * @returns DampingFrequencyRange object (or null if there are no more damping frequency ranges in the model).
 */
    Previous(): DampingFrequencyRange;

/**
 * Allows the user to select damping frequency ranges using standard PRIMER object menus.
 * @param flag Flag to use when selecting damping frequency ranges
 * @param prompt Text to display as a prompt to the user
 * @param limit If the argument is a Model then only damping frequency ranges from that model can be selected. If the argument is a Flag then only damping frequency ranges that are flagged with limit can be selected (limit should be different to flag). If omitted, or null, any damping frequency ranges can be selected. from any model.
 * @param modal If selection is modal (blocks the user from doing anything else in PRIMER until this window is dismissed). If omitted the selection will be modal.
 * @returns Number of damping frequency ranges selected or null if menu cancelled
 */
    static Select(flag: number, prompt: string, limit?: Model | number, modal?: boolean): number;

/**
 * Sets a flag on the damping frequency range.
 * @param flag Flag to set on the damping frequency range
 */
    SetFlag(flag: number): void;

/**
 * Sketches the damping frequency range. The damping frequency range will be sketched until you either call DampingFrequencyRange.Unsketch(), DampingFrequencyRange.UnsketchAll(), Model.UnsketchAll(), or delete the model
 * @param redraw If model should be redrawn or not after the damping frequency range is sketched. If omitted redraw is true. If you want to sketch several damping frequency ranges and only redraw after the last one then use false for redraw and call View.Redraw().
 */
    Sketch(redraw?: boolean): void;

/**
 * Sketches all of the flagged damping frequency ranges in the model. The damping frequency ranges will be sketched until you either call DampingFrequencyRange.Unsketch(), DampingFrequencyRange.UnsketchFlagged(), Model.UnsketchAll(), or delete the model
 * @param Model Model that all the flagged damping frequency ranges will be sketched in
 * @param flag Flag set on the damping frequency ranges that you want to sketch
 * @param redraw If model should be redrawn or not after the damping frequency ranges are sketched. If omitted redraw is true. If you want to sketch flagged damping frequency ranges several times and only redraw after the last one then use false for redraw and call View.Redraw().
 */
    static SketchFlagged(Model: Model, flag: number, redraw?: boolean): void;

/**
 * Returns the total number of damping frequency ranges in the model.
 * @param Model Model to get total for
 * @param exists true if only existing damping frequency ranges should be counted. If false or omitted referenced but undefined damping frequency ranges will also be included in the total.
 * @returns number of damping frequency ranges
 */
    static Total(Model: Model, exists?: boolean): number;

/**
 * Unblanks the damping frequency range
 */
    Unblank(): void;

/**
 * Unblanks all of the damping frequency ranges in the model.
 * @param Model Model that all damping frequency ranges will be unblanked in
 * @param redraw If model should be redrawn or not. If omitted redraw is false. If you want to do several (un)blanks and only redraw after the last one then use false for all redraws apart from the last one. Alternatively you can redraw using View.Redraw().
 */
    static UnblankAll(Model: Model, redraw?: boolean): void;

/**
 * Unblanks all of the flagged damping frequency ranges in the model.
 * @param Model Model that the flagged damping frequency ranges will be unblanked in
 * @param flag Flag set on the damping frequency ranges that you want to unblank
 * @param redraw If model should be redrawn or not. If omitted redraw is false. If you want to do several (un)blanks and only redraw after the last one then use false for all redraws apart from the last one. Alternatively you can redraw using View.Redraw().
 */
    static UnblankFlagged(Model: Model, flag: number, redraw?: boolean): void;

/**
 * Unsets a defined flag on all of the damping frequency ranges in the model.
 * @param Model Model that the defined flag for all damping frequency ranges will be unset in
 * @param flag Flag to unset on the damping frequency ranges
 */
    static UnflagAll(Model: Model, flag: number): void;

/**
 * Unsketches the damping frequency range.
 * @param redraw If model should be redrawn or not after the damping frequency range is unsketched. If omitted redraw is true. If you want to unsketch several damping frequency ranges and only redraw after the last one then use false for redraw and call View.Redraw().
 */
    Unsketch(redraw?: boolean): void;

/**
 * Unsketches all damping frequency ranges.
 * @param Model Model that all damping frequency ranges will be unblanked in
 * @param redraw If model should be redrawn or not after the damping frequency ranges are unsketched. If omitted redraw is true. If you want to unsketch several things and only redraw after the last one then use false for redraw and call View.Redraw().
 */
    static UnsketchAll(Model: Model, redraw?: boolean): void;

/**
 * Unsketches all flagged damping frequency ranges in the model.
 * @param Model Model that all damping frequency ranges will be unsketched in
 * @param flag Flag set on the damping frequency ranges that you want to unsketch
 * @param redraw If model should be redrawn or not after the damping frequency ranges are unsketched. If omitted redraw is true. If you want to unsketch several things and only redraw after the last one then use false for redraw and call View.Redraw().
 */
    static UnsketchFlagged(Model: Model, flag: number, redraw?: boolean): void;

/**
 * Object properties that are parameters are normally returned as the integer or float parameter values as that is virtually always what the user would want. This function temporarily changes the behaviour so that if a property is a parameter the parameter name is returned instead. This can be used with 'method chaining' (see the example below) to make sure a property argument is correct.
 * @returns DampingFrequencyRange object.
 */
    ViewParameters(): DampingFrequencyRange;

/**
 * Adds a warning for damping frequency range. For more details on checking see the Check class.
 * @param message The warning message to give
 * @param details An optional detailed warning message
 */
    Warning(message: string, details?: string): void;

/**
 * Returns the cross references for this damping frequency range.
 * @returns Xrefs object.
 */
    Xrefs(): Xrefs;

/**
 * Create a new DampingFrequencyRange object for *DAMPING_FREQUENCY_RANGE.
 * @param Model Model that damping frequency range will be created in
 * @param option Damping frequency range type. Can be DampingFrequencyRange.NO_OPT or DampingFrequencyRange.DEFORM.
 * @param cdamp Fraction of critical damping
 * @param flow Lowest frequency in range of interest
 * @param fhigh Highest frequency in range of interest.
 * @param psid Set Part set ID.
 * @param pidref Part ID.
 * @param iflg Method used for internal calculation of damping constants
 * @param icard2 Flag for reading the additional card
 * @param cdampv Fraction of critical damping applied to pressure/volume response of solid elements
 * @param ipwp Flag to determine whether damping is applied to excess pore pressure
 * @returns DampingFrequencyRange object
 */
    constructor(Model: Model, option?: number, cdamp?: number, flow?: number, fhigh?: number, psid?: number, pidref?: number, iflg?: number, icard2?: number, cdampv?: number, ipwp?: number);

/**
 * Creates a string containing the damping frequency range data in keyword format. Note that this contains the keyword header and the keyword cards. See also DampingFrequencyRange.Keyword() and DampingFrequencyRange.KeywordCards().
 * @returns string
 */
    toString(): string;

/** Fraction of critical damping */
    cdamp: number;
/** Fraction of critical damping applied to pressure/volume response of solid elements */
    cdampv: number;
/** true if damping frequency range exists, false if referred to but not defined. */
    readonly exists: boolean;
/** Highest frequency in range of interest. */
    fhigh: number;
/** Lowest frequency in range of interest */
    flow: number;
/** Flag for reading the additional card */
    icard2: number;
/** Method used for internal calculation of damping constants */
    iflg: number;
/** The Include file number that the damping frequency range is in. */
    include: number;
/** Flag to determine whether damping is applied to excess pore pressure */
    ipwp: number;
/** The Model number that the damping frequency range is in. */
    readonly model: number;
/** The DampingFrequencyRange option. Can be:  DampingFrequencyRange.NO_OPT or 0   DampingFrequencyRange.DEFORM  */
    option: number;
/** Part ID. */
    pidref: number;
/** Set Part set ID. */
    psid: number;
/** DEFORM is *DAMPING_FREQUENCY_RANGE_DEFORM. */
    static DEFORM: number;
/** NO_OPT is *DAMPING_FREQUENCY_RANGE. */
    static NO_OPT: number;
}

declare class DampingPartMass {
/**
 * Associates a comment with a damping part mass.
 * @param Comment Comment that will be attached to the damping part mass
 */
    AssociateComment(Comment: Comment): void;

/**
 * Blanks the damping part mass
 */
    Blank(): void;

/**
 * Blanks all of the damping part masss in the model.
 * @param Model Model that all damping part masss will be blanked in
 * @param redraw If model should be redrawn or not. If omitted redraw is false. If you want to do several (un)blanks and only redraw after the last one then use false for all redraws apart from the last one. Alternatively you can redraw using View.Redraw().
 */
    static BlankAll(Model: Model, redraw?: boolean): void;

/**
 * Blanks all of the flagged damping part masss in the model.
 * @param Model Model that all the flagged damping part masss will be blanked in
 * @param flag Flag set on the damping part masss that you want to blank
 * @param redraw If model should be redrawn or not. If omitted redraw is false. If you want to do several (un)blanks and only redraw after the last one then use false for all redraws apart from the last one. Alternatively you can redraw using View.Redraw().
 */
    static BlankFlagged(Model: Model, flag: number, redraw?: boolean): void;

/**
 * Checks if the damping part mass is blanked or not.
 * @returns true if blanked, false if not.
 */
    Blanked(): boolean;

/**
 * Clears a flag on the damping part mass.
 * @param flag Flag to clear on the damping part mass
 */
    ClearFlag(flag: number): void;

/**
 * Copies the damping part mass. The target include of the copied damping part mass can be set using Options.copy_target_include.
 * @param range If you want to keep the copied item in the range specified for the current include. Default value is false. To set current include, use Include.MakeCurrentLayer().
 * @returns DampingPartMass object
 */
    Copy(range?: boolean): DampingPartMass;

/**
 * Detaches a comment from a damping part mass.
 * @param Comment Comment that will be detached from the damping part mass
 */
    DetachComment(Comment: Comment): void;

/**
 * Adds an error for damping part mass. For more details on checking see the Check class.
 * @param message The error message to give
 * @param details An optional detailed error message
 */
    Error(message: string, details?: string): void;

/**
 * Returns the first damping part mass in the model.
 * @param Model Model to get first damping part mass in
 * @returns DampingPartMass object (or null if there are no damping part masss in the model).
 */
    static First(Model: Model): DampingPartMass;

/**
 * Flags all of the damping part masss in the model with a defined flag.
 * @param Model Model that all damping part masss will be flagged in
 * @param flag Flag to set on the damping part masss
 */
    static FlagAll(Model: Model, flag: number): void;

/**
 * Checks if the damping part mass is flagged or not.
 * @param flag Flag to test on the damping part mass
 * @returns true if flagged, false if not.
 */
    Flagged(flag: number): boolean;

/**
 * Calls a function for each damping part mass in the model. Note that ForEach has been designed to make looping over damping part masss as fast as possible and so has some limitations. Firstly, a single temporary DampingPartMass object is created and on each function call it is updated with the current damping part mass data. This means that you should not try to store the DampingPartMass object for later use (e.g. in an array) as it is temporary. Secondly, you cannot create new damping part masss inside a ForEach loop. 
 * @param Model Model that all damping part masss are in
 * @param func Function to call for each damping part mass
 * @param extra An optional extra object/array/string etc that will appended to arguments when calling the function
 */
    static ForEach(Model: Model, func: () => void, extra?: any): void;

/**
 * Returns an array of DampingPartMass objects or properties for all of the damping part masss in a model in PRIMER. If the optional property argument is not given then an array of DampingPartMass objects is returned. If the property argument is given, that property value for each damping part mass is returned in the array instead of a DampingPartMass object
 * @param Model Model to get damping part masss from
 * @param property Name for property to get for all damping part masss in the model
 * @returns Array of DampingPartMass objects or properties
 */
    static GetAll(Model: Model, property?: string): DampingPartMass[];

/**
 * Extracts the comments associated to a damping part mass.
 * @returns Array of Comment objects (or null if there are no comments associated to the node).
 */
    GetComments(): Comment[];

/**
 * Returns an array of DampingPartMass objects for all of the flagged damping part masss in a model in PRIMER If the optional property argument is not given then an array of DampingPartMass objects is returned. If the property argument is given, then that property value for each damping part mass is returned in the array instead of a DampingPartMass object
 * @param Model Model to get damping part masss from
 * @param flag Flag set on the damping part masss that you want to retrieve
 * @param property Name for property to get for all flagged damping part masss in the model
 * @returns Array of DampingPartMass objects or properties
 */
    static GetFlagged(Model: Model, flag: number, property?: string): DampingPartMass[];

/**
 * Returns the DampingPartMass object for a damping part mass ID.
 * @param Model Model to find the damping part mass in
 * @param number number of the damping part mass you want the DampingPartMass object for
 * @returns DampingPartMass object (or null if damping part mass does not exist).
 */
    static GetFromID(Model: Model, number: number): DampingPartMass;

/**
 * Checks if a DampingPartMass property is a parameter or not. Note that object properties that are parameters are normally returned as the integer or float parameter values as that is virtually always what the user would want. For this function to work the JavaScript interpreter must use the parameter name instead of the value. This can be done by setting the Options.property_parameter_names option to true before calling the function and then resetting it to false afterwards.. This behaviour can also temporarily be switched by using the DampingPartMass.ViewParameters() method and 'method chaining' (see the examples below).
 * @param prop damping part mass property to get parameter for
 * @returns Parameter object if property is a parameter, null if not.
 */
    GetParameter(prop: string): Parameter;

/**
 * Returns the keyword for this damping part mass (*DAMPING_PART_MASS). Note that a carriage return is not added. See also DampingPartMass.KeywordCards()
 * @returns string containing the keyword.
 */
    Keyword(): string;

/**
 * Returns the keyword cards for the damping part mass. Note that a carriage return is not added. See also DampingPartMass.Keyword()
 * @returns string containing the cards.
 */
    KeywordCards(): string;

/**
 * Returns the last damping part mass in the model.
 * @param Model Model to get last damping part mass in
 * @returns DampingPartMass object (or null if there are no damping part masss in the model).
 */
    static Last(Model: Model): DampingPartMass;

/**
 * Returns the next damping part mass in the model.
 * @returns DampingPartMass object (or null if there are no more damping part masss in the model).
 */
    Next(): DampingPartMass;

/**
 * Allows the user to pick a damping part mass.
 * @param prompt Text to display as a prompt to the user
 * @param limit If the argument is a Model then only damping part masss from that model can be picked. If the argument is a Flag then only damping part masss that are flagged with limit can be selected. If omitted, or null, any damping part masss from any model can be selected. from any model.
 * @param modal If picking is modal (blocks the user from doing anything else in PRIMER until this window is dismissed). If omitted the pick will be modal.
 * @param button_text By default the window with the prompt will have a button labelled 'Cancel' which if pressed will cancel the pick and return null. If you want to change the text on the button use this argument. If omitted 'Cancel' will be used.
 * @returns DampingPartMass object (or null if not picked)
 */
    static Pick(prompt: string, limit?: Model | number, modal?: boolean, button_text?: string): DampingPartMass;

/**
 * Returns the previous damping part mass in the model.
 * @returns DampingPartMass object (or null if there are no more damping part masss in the model).
 */
    Previous(): DampingPartMass;

/**
 * Allows the user to select damping part masss using standard PRIMER object menus.
 * @param flag Flag to use when selecting damping part masss
 * @param prompt Text to display as a prompt to the user
 * @param limit If the argument is a Model then only damping part masss from that model can be selected. If the argument is a Flag then only damping part masss that are flagged with limit can be selected (limit should be different to flag). If omitted, or null, any damping part masss can be selected. from any model.
 * @param modal If selection is modal (blocks the user from doing anything else in PRIMER until this window is dismissed). If omitted the selection will be modal.
 * @returns Number of damping part masss selected or null if menu cancelled
 */
    static Select(flag: number, prompt: string, limit?: Model | number, modal?: boolean): number;

/**
 * Sets a flag on the damping part mass.
 * @param flag Flag to set on the damping part mass
 */
    SetFlag(flag: number): void;

/**
 * Sketches the damping part mass. The damping part mass will be sketched until you either call DampingPartMass.Unsketch(), DampingPartMass.UnsketchAll(), Model.UnsketchAll(), or delete the model
 * @param redraw If model should be redrawn or not after the damping part mass is sketched. If omitted redraw is true. If you want to sketch several damping part masss and only redraw after the last one then use false for redraw and call View.Redraw().
 */
    Sketch(redraw?: boolean): void;

/**
 * Sketches all of the flagged damping part masss in the model. The damping part masss will be sketched until you either call DampingPartMass.Unsketch(), DampingPartMass.UnsketchFlagged(), Model.UnsketchAll(), or delete the model
 * @param Model Model that all the flagged damping part masss will be sketched in
 * @param flag Flag set on the damping part masss that you want to sketch
 * @param redraw If model should be redrawn or not after the damping part masss are sketched. If omitted redraw is true. If you want to sketch flagged damping part masss several times and only redraw after the last one then use false for redraw and call View.Redraw().
 */
    static SketchFlagged(Model: Model, flag: number, redraw?: boolean): void;

/**
 * Returns the total number of damping part masss in the model.
 * @param Model Model to get total for
 * @param exists true if only existing damping part masss should be counted. If false or omitted referenced but undefined damping part masss will also be included in the total.
 * @returns number of damping part masss
 */
    static Total(Model: Model, exists?: boolean): number;

/**
 * Unblanks the damping part mass
 */
    Unblank(): void;

/**
 * Unblanks all of the damping part masss in the model.
 * @param Model Model that all damping part masss will be unblanked in
 * @param redraw If model should be redrawn or not. If omitted redraw is false. If you want to do several (un)blanks and only redraw after the last one then use false for all redraws apart from the last one. Alternatively you can redraw using View.Redraw().
 */
    static UnblankAll(Model: Model, redraw?: boolean): void;

/**
 * Unblanks all of the flagged damping part masss in the model.
 * @param Model Model that the flagged damping part masss will be unblanked in
 * @param flag Flag set on the damping part masss that you want to unblank
 * @param redraw If model should be redrawn or not. If omitted redraw is false. If you want to do several (un)blanks and only redraw after the last one then use false for all redraws apart from the last one. Alternatively you can redraw using View.Redraw().
 */
    static UnblankFlagged(Model: Model, flag: number, redraw?: boolean): void;

/**
 * Unsets a defined flag on all of the damping part masss in the model.
 * @param Model Model that the defined flag for all damping part masss will be unset in
 * @param flag Flag to unset on the damping part masss
 */
    static UnflagAll(Model: Model, flag: number): void;

/**
 * Unsketches the damping part mass.
 * @param redraw If model should be redrawn or not after the damping part mass is unsketched. If omitted redraw is true. If you want to unsketch several damping part masss and only redraw after the last one then use false for redraw and call View.Redraw().
 */
    Unsketch(redraw?: boolean): void;

/**
 * Unsketches all damping part masss.
 * @param Model Model that all damping part masss will be unblanked in
 * @param redraw If model should be redrawn or not after the damping part masss are unsketched. If omitted redraw is true. If you want to unsketch several things and only redraw after the last one then use false for redraw and call View.Redraw().
 */
    static UnsketchAll(Model: Model, redraw?: boolean): void;

/**
 * Unsketches all flagged damping part masss in the model.
 * @param Model Model that all damping part masss will be unsketched in
 * @param flag Flag set on the damping part masss that you want to unsketch
 * @param redraw If model should be redrawn or not after the damping part masss are unsketched. If omitted redraw is true. If you want to unsketch several things and only redraw after the last one then use false for redraw and call View.Redraw().
 */
    static UnsketchFlagged(Model: Model, flag: number, redraw?: boolean): void;

/**
 * Object properties that are parameters are normally returned as the integer or float parameter values as that is virtually always what the user would want. This function temporarily changes the behaviour so that if a property is a parameter the parameter name is returned instead. This can be used with 'method chaining' (see the example below) to make sure a property argument is correct.
 * @returns DampingPartMass object.
 */
    ViewParameters(): DampingPartMass;

/**
 * Adds a warning for damping part mass. For more details on checking see the Check class.
 * @param message The warning message to give
 * @param details An optional detailed warning message
 */
    Warning(message: string, details?: string): void;

/**
 * Returns the cross references for this damping part mass.
 * @returns Xrefs object.
 */
    Xrefs(): Xrefs;

/**
 * Create a new DampingPartMass object for *DAMPING_PART_MASS.
 * @param Model Model that damping part mass will be created in
 * @param type Damping part mass type. Can be DampingPartMass.PART or DampingPartMass.SET.
 * @param id Part/part set id
 * @param lcid LC: Damping vs time
 * @param sf Scale factor on loadcurve
 * @param flag Flag for scale factors
 * @param stx Translational x scale factor
 * @param sty Translational y scale factor
 * @param stz Translational z scale factor
 * @param srx Rotational x scale factor
 * @param sry Rotational y scale factor
 * @param srz Rotational z scale factor
 * @returns DampingPartMass object
 */
    constructor(Model: Model, type: number, id: number, lcid: number, sf?: number, flag?: number, stx?: number, sty?: number, stz?: number, srx?: number, sry?: number, srz?: number);

/**
 * Creates a string containing the damping part mass data in keyword format. Note that this contains the keyword header and the keyword cards. See also DampingPartMass.Keyword() and DampingPartMass.KeywordCards().
 * @returns string
 */
    toString(): string;

/** true if damping part mass exists, false if referred to but not defined. */
    readonly exists: boolean;
/** Flag for scale factors */
    flag: number;
/** Part/part set id */
    id: number;
/** The Include file number that the damping part mass is in. */
    include: number;
/** LC: Damping vs time */
    lcid: number;
/** The Model number that the damping part mass is in. */
    readonly model: number;
/** Scale factor on loadcurve */
    sf: number;
/** Rotational x scale factor */
    srx: number;
/** Rotational y scale factor */
    sry: number;
/** Rotational z scale factor */
    srz: number;
/** Translational x scale factor */
    stx: number;
/** Translational y scale factor */
    sty: number;
/** Translational z scale factor */
    stz: number;
/** The DampingPartMass type. Can be:  DampingPartMass.PART or DampingPartMass.SET  */
    type: number;
/** PART is *DAMPING_PART_MASS. */
    static PART: number;
/** SET is *DAMPING_PART_MASS_SET. */
    static SET: number;
}

declare class DampingPartStiffness {
/**
 * Associates a comment with a damping part stiffness.
 * @param Comment Comment that will be attached to the damping part stiffness
 */
    AssociateComment(Comment: Comment): void;

/**
 * Blanks the damping part stiffness
 */
    Blank(): void;

/**
 * Blanks all of the damping part stiffnesss in the model.
 * @param Model Model that all damping part stiffnesss will be blanked in
 * @param redraw If model should be redrawn or not. If omitted redraw is false. If you want to do several (un)blanks and only redraw after the last one then use false for all redraws apart from the last one. Alternatively you can redraw using View.Redraw().
 */
    static BlankAll(Model: Model, redraw?: boolean): void;

/**
 * Blanks all of the flagged damping part stiffnesss in the model.
 * @param Model Model that all the flagged damping part stiffnesss will be blanked in
 * @param flag Flag set on the damping part stiffnesss that you want to blank
 * @param redraw If model should be redrawn or not. If omitted redraw is false. If you want to do several (un)blanks and only redraw after the last one then use false for all redraws apart from the last one. Alternatively you can redraw using View.Redraw().
 */
    static BlankFlagged(Model: Model, flag: number, redraw?: boolean): void;

/**
 * Checks if the damping part stiffness is blanked or not.
 * @returns true if blanked, false if not.
 */
    Blanked(): boolean;

/**
 * Clears a flag on the damping part stiffness.
 * @param flag Flag to clear on the damping part stiffness
 */
    ClearFlag(flag: number): void;

/**
 * Copies the damping part stiffness. The target include of the copied damping part stiffness can be set using Options.copy_target_include.
 * @param range If you want to keep the copied item in the range specified for the current include. Default value is false. To set current include, use Include.MakeCurrentLayer().
 * @returns DampingPartStiffness object
 */
    Copy(range?: boolean): DampingPartStiffness;

/**
 * Detaches a comment from a damping part stiffness.
 * @param Comment Comment that will be detached from the damping part stiffness
 */
    DetachComment(Comment: Comment): void;

/**
 * Adds an error for damping part stiffness. For more details on checking see the Check class.
 * @param message The error message to give
 * @param details An optional detailed error message
 */
    Error(message: string, details?: string): void;

/**
 * Returns the first damping part stiffness in the model.
 * @param Model Model to get first damping part stiffness in
 * @returns DampingPartStiffness object (or null if there are no damping part stiffnesss in the model).
 */
    static First(Model: Model): DampingPartStiffness;

/**
 * Flags all of the damping part stiffnesss in the model with a defined flag.
 * @param Model Model that all damping part stiffnesss will be flagged in
 * @param flag Flag to set on the damping part stiffnesss
 */
    static FlagAll(Model: Model, flag: number): void;

/**
 * Checks if the damping part stiffness is flagged or not.
 * @param flag Flag to test on the damping part stiffness
 * @returns true if flagged, false if not.
 */
    Flagged(flag: number): boolean;

/**
 * Calls a function for each damping part stiffness in the model. Note that ForEach has been designed to make looping over damping part stiffnesss as fast as possible and so has some limitations. Firstly, a single temporary DampingPartStiffness object is created and on each function call it is updated with the current damping part stiffness data. This means that you should not try to store the DampingPartStiffness object for later use (e.g. in an array) as it is temporary. Secondly, you cannot create new damping part stiffnesss inside a ForEach loop. 
 * @param Model Model that all damping part stiffnesss are in
 * @param func Function to call for each damping part stiffness
 * @param extra An optional extra object/array/string etc that will appended to arguments when calling the function
 */
    static ForEach(Model: Model, func: () => void, extra?: any): void;

/**
 * Returns an array of DampingPartStiffness objects or properties for all of the damping part stiffnesss in a model in PRIMER. If the optional property argument is not given then an array of DampingPartStiffness objects is returned. If the property argument is given, that property value for each damping part stiffness is returned in the array instead of a DampingPartStiffness object
 * @param Model Model to get damping part stiffnesss from
 * @param property Name for property to get for all damping part stiffnesss in the model
 * @returns Array of DampingPartStiffness objects or properties
 */
    static GetAll(Model: Model, property?: string): DampingPartStiffness[];

/**
 * Extracts the comments associated to a damping part stiffness.
 * @returns Array of Comment objects (or null if there are no comments associated to the node).
 */
    GetComments(): Comment[];

/**
 * Returns an array of DampingPartStiffness objects for all of the flagged damping part stiffnesss in a model in PRIMER If the optional property argument is not given then an array of DampingPartStiffness objects is returned. If the property argument is given, then that property value for each damping part stiffness is returned in the array instead of a DampingPartStiffness object
 * @param Model Model to get damping part stiffnesss from
 * @param flag Flag set on the damping part stiffnesss that you want to retrieve
 * @param property Name for property to get for all flagged damping part stiffnesss in the model
 * @returns Array of DampingPartStiffness objects or properties
 */
    static GetFlagged(Model: Model, flag: number, property?: string): DampingPartStiffness[];

/**
 * Returns the DampingPartStiffness object for a damping part stiffness ID.
 * @param Model Model to find the damping part stiffness in
 * @param number number of the damping part stiffness you want the DampingPartStiffness object for
 * @returns DampingPartStiffness object (or null if damping part stiffness does not exist).
 */
    static GetFromID(Model: Model, number: number): DampingPartStiffness;

/**
 * Checks if a DampingPartStiffness property is a parameter or not. Note that object properties that are parameters are normally returned as the integer or float parameter values as that is virtually always what the user would want. For this function to work the JavaScript interpreter must use the parameter name instead of the value. This can be done by setting the Options.property_parameter_names option to true before calling the function and then resetting it to false afterwards.. This behaviour can also temporarily be switched by using the DampingPartStiffness.ViewParameters() method and 'method chaining' (see the examples below).
 * @param prop damping part stiffness property to get parameter for
 * @returns Parameter object if property is a parameter, null if not.
 */
    GetParameter(prop: string): Parameter;

/**
 * Returns the keyword for this damping part stiffness (*DAMPING_PART_STIFFNESS). Note that a carriage return is not added. See also DampingPartStiffness.KeywordCards()
 * @returns string containing the keyword.
 */
    Keyword(): string;

/**
 * Returns the keyword cards for the damping part stiffness. Note that a carriage return is not added. See also DampingPartStiffness.Keyword()
 * @returns string containing the cards.
 */
    KeywordCards(): string;

/**
 * Returns the last damping part stiffness in the model.
 * @param Model Model to get last damping part stiffness in
 * @returns DampingPartStiffness object (or null if there are no damping part stiffnesss in the model).
 */
    static Last(Model: Model): DampingPartStiffness;

/**
 * Returns the next damping part stiffness in the model.
 * @returns DampingPartStiffness object (or null if there are no more damping part stiffnesss in the model).
 */
    Next(): DampingPartStiffness;

/**
 * Allows the user to pick a damping part stiffness.
 * @param prompt Text to display as a prompt to the user
 * @param limit If the argument is a Model then only damping part stiffnesss from that model can be picked. If the argument is a Flag then only damping part stiffnesss that are flagged with limit can be selected. If omitted, or null, any damping part stiffnesss from any model can be selected. from any model.
 * @param modal If picking is modal (blocks the user from doing anything else in PRIMER until this window is dismissed). If omitted the pick will be modal.
 * @param button_text By default the window with the prompt will have a button labelled 'Cancel' which if pressed will cancel the pick and return null. If you want to change the text on the button use this argument. If omitted 'Cancel' will be used.
 * @returns DampingPartStiffness object (or null if not picked)
 */
    static Pick(prompt: string, limit?: Model | number, modal?: boolean, button_text?: string): DampingPartStiffness;

/**
 * Returns the previous damping part stiffness in the model.
 * @returns DampingPartStiffness object (or null if there are no more damping part stiffnesss in the model).
 */
    Previous(): DampingPartStiffness;

/**
 * Allows the user to select damping part stiffnesss using standard PRIMER object menus.
 * @param flag Flag to use when selecting damping part stiffnesss
 * @param prompt Text to display as a prompt to the user
 * @param limit If the argument is a Model then only damping part stiffnesss from that model can be selected. If the argument is a Flag then only damping part stiffnesss that are flagged with limit can be selected (limit should be different to flag). If omitted, or null, any damping part stiffnesss can be selected. from any model.
 * @param modal If selection is modal (blocks the user from doing anything else in PRIMER until this window is dismissed). If omitted the selection will be modal.
 * @returns Number of damping part stiffnesss selected or null if menu cancelled
 */
    static Select(flag: number, prompt: string, limit?: Model | number, modal?: boolean): number;

/**
 * Sets a flag on the damping part stiffness.
 * @param flag Flag to set on the damping part stiffness
 */
    SetFlag(flag: number): void;

/**
 * Sketches the damping part stiffness. The damping part stiffness will be sketched until you either call DampingPartStiffness.Unsketch(), DampingPartStiffness.UnsketchAll(), Model.UnsketchAll(), or delete the model
 * @param redraw If model should be redrawn or not after the damping part stiffness is sketched. If omitted redraw is true. If you want to sketch several damping part stiffnesss and only redraw after the last one then use false for redraw and call View.Redraw().
 */
    Sketch(redraw?: boolean): void;

/**
 * Sketches all of the flagged damping part stiffnesss in the model. The damping part stiffnesss will be sketched until you either call DampingPartStiffness.Unsketch(), DampingPartStiffness.UnsketchFlagged(), Model.UnsketchAll(), or delete the model
 * @param Model Model that all the flagged damping part stiffnesss will be sketched in
 * @param flag Flag set on the damping part stiffnesss that you want to sketch
 * @param redraw If model should be redrawn or not after the damping part stiffnesss are sketched. If omitted redraw is true. If you want to sketch flagged damping part stiffnesss several times and only redraw after the last one then use false for redraw and call View.Redraw().
 */
    static SketchFlagged(Model: Model, flag: number, redraw?: boolean): void;

/**
 * Returns the total number of damping part stiffnesss in the model.
 * @param Model Model to get total for
 * @param exists true if only existing damping part stiffnesss should be counted. If false or omitted referenced but undefined damping part stiffnesss will also be included in the total.
 * @returns number of damping part stiffnesss
 */
    static Total(Model: Model, exists?: boolean): number;

/**
 * Unblanks the damping part stiffness
 */
    Unblank(): void;

/**
 * Unblanks all of the damping part stiffnesss in the model.
 * @param Model Model that all damping part stiffnesss will be unblanked in
 * @param redraw If model should be redrawn or not. If omitted redraw is false. If you want to do several (un)blanks and only redraw after the last one then use false for all redraws apart from the last one. Alternatively you can redraw using View.Redraw().
 */
    static UnblankAll(Model: Model, redraw?: boolean): void;

/**
 * Unblanks all of the flagged damping part stiffnesss in the model.
 * @param Model Model that the flagged damping part stiffnesss will be unblanked in
 * @param flag Flag set on the damping part stiffnesss that you want to unblank
 * @param redraw If model should be redrawn or not. If omitted redraw is false. If you want to do several (un)blanks and only redraw after the last one then use false for all redraws apart from the last one. Alternatively you can redraw using View.Redraw().
 */
    static UnblankFlagged(Model: Model, flag: number, redraw?: boolean): void;

/**
 * Unsets a defined flag on all of the damping part stiffnesss in the model.
 * @param Model Model that the defined flag for all damping part stiffnesss will be unset in
 * @param flag Flag to unset on the damping part stiffnesss
 */
    static UnflagAll(Model: Model, flag: number): void;

/**
 * Unsketches the damping part stiffness.
 * @param redraw If model should be redrawn or not after the damping part stiffness is unsketched. If omitted redraw is true. If you want to unsketch several damping part stiffnesss and only redraw after the last one then use false for redraw and call View.Redraw().
 */
    Unsketch(redraw?: boolean): void;

/**
 * Unsketches all damping part stiffnesss.
 * @param Model Model that all damping part stiffnesss will be unblanked in
 * @param redraw If model should be redrawn or not after the damping part stiffnesss are unsketched. If omitted redraw is true. If you want to unsketch several things and only redraw after the last one then use false for redraw and call View.Redraw().
 */
    static UnsketchAll(Model: Model, redraw?: boolean): void;

/**
 * Unsketches all flagged damping part stiffnesss in the model.
 * @param Model Model that all damping part stiffnesss will be unsketched in
 * @param flag Flag set on the damping part stiffnesss that you want to unsketch
 * @param redraw If model should be redrawn or not after the damping part stiffnesss are unsketched. If omitted redraw is true. If you want to unsketch several things and only redraw after the last one then use false for redraw and call View.Redraw().
 */
    static UnsketchFlagged(Model: Model, flag: number, redraw?: boolean): void;

/**
 * Object properties that are parameters are normally returned as the integer or float parameter values as that is virtually always what the user would want. This function temporarily changes the behaviour so that if a property is a parameter the parameter name is returned instead. This can be used with 'method chaining' (see the example below) to make sure a property argument is correct.
 * @returns DampingPartStiffness object.
 */
    ViewParameters(): DampingPartStiffness;

/**
 * Adds a warning for damping part stiffness. For more details on checking see the Check class.
 * @param message The warning message to give
 * @param details An optional detailed warning message
 */
    Warning(message: string, details?: string): void;

/**
 * Returns the cross references for this damping part stiffness.
 * @returns Xrefs object.
 */
    Xrefs(): Xrefs;

/**
 * Create a new DampingPartStiffness object for *DAMPING_PART_STIFFNESS.
 * @param Model Model that damping part stiffness will be created in
 * @param type Damping part stiffness type. Can be DampingPartStiffness.PART or DampingPartStiffness.SET.
 * @param id Part/part set id
 * @param coef Rayleigh damping coefficient
 * @returns DampingPartStiffness object
 */
    constructor(Model: Model, type: number, id: number, coef?: number);

/**
 * Creates a string containing the damping part stiffness data in keyword format. Note that this contains the keyword header and the keyword cards. See also DampingPartStiffness.Keyword() and DampingPartStiffness.KeywordCards().
 * @returns string
 */
    toString(): string;

/** Rayleigh damping coefficient */
    coef: number;
/** true if damping part stiffness exists, false if referred to but not defined. */
    readonly exists: boolean;
/** Part/part set id */
    id: number;
/** The Include file number that the damping part stiffness is in. */
    include: number;
/** The Model number that the damping part stiffness is in. */
    readonly model: number;
/** The DampingPartStiffness type. Can be:  DampingPartStiffness.PART or DampingPartStiffness.SET  */
    type: number;
/** PART is *DAMPING_PART_STIFFNESS. */
    static PART: number;
/** SET is *DAMPING_PART_STIFFNESS_SET. */
    static SET: number;
}

declare class DampingPartStructural {
/**
 * Associates a comment with a damping part structural.
 * @param Comment Comment that will be attached to the damping part structural
 */
    AssociateComment(Comment: Comment): void;

/**
 * Blanks the damping part structural
 */
    Blank(): void;

/**
 * Blanks all of the damping part structurals in the model.
 * @param Model Model that all damping part structurals will be blanked in
 * @param redraw If model should be redrawn or not. If omitted redraw is false. If you want to do several (un)blanks and only redraw after the last one then use false for all redraws apart from the last one. Alternatively you can redraw using View.Redraw().
 */
    static BlankAll(Model: Model, redraw?: boolean): void;

/**
 * Blanks all of the flagged damping part structurals in the model.
 * @param Model Model that all the flagged damping part structurals will be blanked in
 * @param flag Flag set on the damping part structurals that you want to blank
 * @param redraw If model should be redrawn or not. If omitted redraw is false. If you want to do several (un)blanks and only redraw after the last one then use false for all redraws apart from the last one. Alternatively you can redraw using View.Redraw().
 */
    static BlankFlagged(Model: Model, flag: number, redraw?: boolean): void;

/**
 * Checks if the damping part structural is blanked or not.
 * @returns true if blanked, false if not.
 */
    Blanked(): boolean;

/**
 * Clears a flag on the damping part structural.
 * @param flag Flag to clear on the damping part structural
 */
    ClearFlag(flag: number): void;

/**
 * Copies the damping part structural. The target include of the copied damping part structural can be set using Options.copy_target_include.
 * @param range If you want to keep the copied item in the range specified for the current include. Default value is false. To set current include, use Include.MakeCurrentLayer().
 * @returns DampingPartStructural object
 */
    Copy(range?: boolean): DampingPartStructural;

/**
 * Detaches a comment from a damping part structural.
 * @param Comment Comment that will be detached from the damping part structural
 */
    DetachComment(Comment: Comment): void;

/**
 * Adds an error for damping part structural. For more details on checking see the Check class.
 * @param message The error message to give
 * @param details An optional detailed error message
 */
    Error(message: string, details?: string): void;

/**
 * Returns the first damping part structural in the model.
 * @param Model Model to get first damping part structural in
 * @returns DampingPartStructural object (or null if there are no damping part structurals in the model).
 */
    static First(Model: Model): DampingPartStructural;

/**
 * Flags all of the damping part structurals in the model with a defined flag.
 * @param Model Model that all damping part structurals will be flagged in
 * @param flag Flag to set on the damping part structurals
 */
    static FlagAll(Model: Model, flag: number): void;

/**
 * Checks if the damping part structural is flagged or not.
 * @param flag Flag to test on the damping part structural
 * @returns true if flagged, false if not.
 */
    Flagged(flag: number): boolean;

/**
 * Calls a function for each damping part structural in the model. Note that ForEach has been designed to make looping over damping part structurals as fast as possible and so has some limitations. Firstly, a single temporary DampingPartStructural object is created and on each function call it is updated with the current damping part structural data. This means that you should not try to store the DampingPartStructural object for later use (e.g. in an array) as it is temporary. Secondly, you cannot create new damping part structurals inside a ForEach loop. 
 * @param Model Model that all damping part structurals are in
 * @param func Function to call for each damping part structural
 * @param extra An optional extra object/array/string etc that will appended to arguments when calling the function
 */
    static ForEach(Model: Model, func: () => void, extra?: any): void;

/**
 * Returns an array of DampingPartStructural objects or properties for all of the damping part structurals in a model in PRIMER. If the optional property argument is not given then an array of DampingPartStructural objects is returned. If the property argument is given, that property value for each damping part structural is returned in the array instead of a DampingPartStructural object
 * @param Model Model to get damping part structurals from
 * @param property Name for property to get for all damping part structurals in the model
 * @returns Array of DampingPartStructural objects or properties
 */
    static GetAll(Model: Model, property?: string): DampingPartStructural[];

/**
 * Extracts the comments associated to a damping part structural.
 * @returns Array of Comment objects (or null if there are no comments associated to the node).
 */
    GetComments(): Comment[];

/**
 * Returns an array of DampingPartStructural objects for all of the flagged damping part structurals in a model in PRIMER If the optional property argument is not given then an array of DampingPartStructural objects is returned. If the property argument is given, then that property value for each damping part structural is returned in the array instead of a DampingPartStructural object
 * @param Model Model to get damping part structurals from
 * @param flag Flag set on the damping part structurals that you want to retrieve
 * @param property Name for property to get for all flagged damping part structurals in the model
 * @returns Array of DampingPartStructural objects or properties
 */
    static GetFlagged(Model: Model, flag: number, property?: string): DampingPartStructural[];

/**
 * Returns the DampingPartStructural object for a damping part structural ID.
 * @param Model Model to find the damping part structural in
 * @param number number of the damping part structural you want the DampingPartStructural object for
 * @returns DampingPartStructural object (or null if damping part structural does not exist).
 */
    static GetFromID(Model: Model, number: number): DampingPartStructural;

/**
 * Checks if a DampingPartStructural property is a parameter or not. Note that object properties that are parameters are normally returned as the integer or float parameter values as that is virtually always what the user would want. For this function to work the JavaScript interpreter must use the parameter name instead of the value. This can be done by setting the Options.property_parameter_names option to true before calling the function and then resetting it to false afterwards.. This behaviour can also temporarily be switched by using the DampingPartStructural.ViewParameters() method and 'method chaining' (see the examples below).
 * @param prop damping part structural property to get parameter for
 * @returns Parameter object if property is a parameter, null if not.
 */
    GetParameter(prop: string): Parameter;

/**
 * Returns the keyword for this damping part structural (*DAMPING_PART_STRUCTURAL). Note that a carriage return is not added. See also DampingPartStructural.KeywordCards()
 * @returns string containing the keyword.
 */
    Keyword(): string;

/**
 * Returns the keyword cards for the damping part structural. Note that a carriage return is not added. See also DampingPartStructural.Keyword()
 * @returns string containing the cards.
 */
    KeywordCards(): string;

/**
 * Returns the last damping part structural in the model.
 * @param Model Model to get last damping part structural in
 * @returns DampingPartStructural object (or null if there are no damping part structurals in the model).
 */
    static Last(Model: Model): DampingPartStructural;

/**
 * Returns the next damping part structural in the model.
 * @returns DampingPartStructural object (or null if there are no more damping part structurals in the model).
 */
    Next(): DampingPartStructural;

/**
 * Allows the user to pick a damping part structural.
 * @param prompt Text to display as a prompt to the user
 * @param limit If the argument is a Model then only damping part structurals from that model can be picked. If the argument is a Flag then only damping part structurals that are flagged with limit can be selected. If omitted, or null, any damping part structurals from any model can be selected. from any model.
 * @param modal If picking is modal (blocks the user from doing anything else in PRIMER until this window is dismissed). If omitted the pick will be modal.
 * @param button_text By default the window with the prompt will have a button labelled 'Cancel' which if pressed will cancel the pick and return null. If you want to change the text on the button use this argument. If omitted 'Cancel' will be used.
 * @returns DampingPartStructural object (or null if not picked)
 */
    static Pick(prompt: string, limit?: Model | number, modal?: boolean, button_text?: string): DampingPartStructural;

/**
 * Returns the previous damping part structural in the model.
 * @returns DampingPartStructural object (or null if there are no more damping part structurals in the model).
 */
    Previous(): DampingPartStructural;

/**
 * Allows the user to select damping part structurals using standard PRIMER object menus.
 * @param flag Flag to use when selecting damping part structurals
 * @param prompt Text to display as a prompt to the user
 * @param limit If the argument is a Model then only damping part structurals from that model can be selected. If the argument is a Flag then only damping part structurals that are flagged with limit can be selected (limit should be different to flag). If omitted, or null, any damping part structurals can be selected. from any model.
 * @param modal If selection is modal (blocks the user from doing anything else in PRIMER until this window is dismissed). If omitted the selection will be modal.
 * @returns Number of damping part structurals selected or null if menu cancelled
 */
    static Select(flag: number, prompt: string, limit?: Model | number, modal?: boolean): number;

/**
 * Sets a flag on the damping part structural.
 * @param flag Flag to set on the damping part structural
 */
    SetFlag(flag: number): void;

/**
 * Sketches the damping part structural. The damping part structural will be sketched until you either call DampingPartStructural.Unsketch(), DampingPartStructural.UnsketchAll(), Model.UnsketchAll(), or delete the model
 * @param redraw If model should be redrawn or not after the damping part structural is sketched. If omitted redraw is true. If you want to sketch several damping part structurals and only redraw after the last one then use false for redraw and call View.Redraw().
 */
    Sketch(redraw?: boolean): void;

/**
 * Sketches all of the flagged damping part structurals in the model. The damping part structurals will be sketched until you either call DampingPartStructural.Unsketch(), DampingPartStructural.UnsketchFlagged(), Model.UnsketchAll(), or delete the model
 * @param Model Model that all the flagged damping part structurals will be sketched in
 * @param flag Flag set on the damping part structurals that you want to sketch
 * @param redraw If model should be redrawn or not after the damping part structurals are sketched. If omitted redraw is true. If you want to sketch flagged damping part structurals several times and only redraw after the last one then use false for redraw and call View.Redraw().
 */
    static SketchFlagged(Model: Model, flag: number, redraw?: boolean): void;

/**
 * Returns the total number of damping part structurals in the model.
 * @param Model Model to get total for
 * @param exists true if only existing damping part structurals should be counted. If false or omitted referenced but undefined damping part structurals will also be included in the total.
 * @returns number of damping part structurals
 */
    static Total(Model: Model, exists?: boolean): number;

/**
 * Unblanks the damping part structural
 */
    Unblank(): void;

/**
 * Unblanks all of the damping part structurals in the model.
 * @param Model Model that all damping part structurals will be unblanked in
 * @param redraw If model should be redrawn or not. If omitted redraw is false. If you want to do several (un)blanks and only redraw after the last one then use false for all redraws apart from the last one. Alternatively you can redraw using View.Redraw().
 */
    static UnblankAll(Model: Model, redraw?: boolean): void;

/**
 * Unblanks all of the flagged damping part structurals in the model.
 * @param Model Model that the flagged damping part structurals will be unblanked in
 * @param flag Flag set on the damping part structurals that you want to unblank
 * @param redraw If model should be redrawn or not. If omitted redraw is false. If you want to do several (un)blanks and only redraw after the last one then use false for all redraws apart from the last one. Alternatively you can redraw using View.Redraw().
 */
    static UnblankFlagged(Model: Model, flag: number, redraw?: boolean): void;

/**
 * Unsets a defined flag on all of the damping part structurals in the model.
 * @param Model Model that the defined flag for all damping part structurals will be unset in
 * @param flag Flag to unset on the damping part structurals
 */
    static UnflagAll(Model: Model, flag: number): void;

/**
 * Unsketches the damping part structural.
 * @param redraw If model should be redrawn or not after the damping part structural is unsketched. If omitted redraw is true. If you want to unsketch several damping part structurals and only redraw after the last one then use false for redraw and call View.Redraw().
 */
    Unsketch(redraw?: boolean): void;

/**
 * Unsketches all damping part structurals.
 * @param Model Model that all damping part structurals will be unblanked in
 * @param redraw If model should be redrawn or not after the damping part structurals are unsketched. If omitted redraw is true. If you want to unsketch several things and only redraw after the last one then use false for redraw and call View.Redraw().
 */
    static UnsketchAll(Model: Model, redraw?: boolean): void;

/**
 * Unsketches all flagged damping part structurals in the model.
 * @param Model Model that all damping part structurals will be unsketched in
 * @param flag Flag set on the damping part structurals that you want to unsketch
 * @param redraw If model should be redrawn or not after the damping part structurals are unsketched. If omitted redraw is true. If you want to unsketch several things and only redraw after the last one then use false for redraw and call View.Redraw().
 */
    static UnsketchFlagged(Model: Model, flag: number, redraw?: boolean): void;

/**
 * Object properties that are parameters are normally returned as the integer or float parameter values as that is virtually always what the user would want. This function temporarily changes the behaviour so that if a property is a parameter the parameter name is returned instead. This can be used with 'method chaining' (see the example below) to make sure a property argument is correct.
 * @returns DampingPartStructural object.
 */
    ViewParameters(): DampingPartStructural;

/**
 * Adds a warning for damping part structural. For more details on checking see the Check class.
 * @param message The warning message to give
 * @param details An optional detailed warning message
 */
    Warning(message: string, details?: string): void;

/**
 * Returns the cross references for this damping part structural.
 * @returns Xrefs object.
 */
    Xrefs(): Xrefs;

/**
 * Create a new DampingPartStructural object for *DAMPING_PART_STRUCTURAL.
 * @param Model Model that damping part structural will be created in
 * @param type Damping part structural type. Can be DampingPartStructural.PART or DampingPartStructural.SET.
 * @param id Part/part set id
 * @param coef Structural damping coefficient
 * @returns DampingPartStructural object
 */
    constructor(Model: Model, type: number, id: number, coef?: number);

/**
 * Creates a string containing the damping part structural data in keyword format. Note that this contains the keyword header and the keyword cards. See also DampingPartStructural.Keyword() and DampingPartStructural.KeywordCards().
 * @returns string
 */
    toString(): string;

/** Structural damping coefficient */
    coef: number;
/** true if damping part structural exists, false if referred to but not defined. */
    readonly exists: boolean;
/** Part/part set id */
    id: number;
/** The Include file number that the damping part structural is in. */
    include: number;
/** The Model number that the damping part structural is in. */
    readonly model: number;
/** The DampingPartStructural type. Can be:  DampingPartStructural.PART or DampingPartStructural.SET  */
    type: number;
/** PART is *DAMPING_PART_STRUCTURAL. */
    static PART: number;
/** SET is *DAMPING_PART_STRUCTURAL_SET. */
    static SET: number;
}

declare class DampingRelative {
/**
 * Associates a comment with a damping relative.
 * @param Comment Comment that will be attached to the damping relative
 */
    AssociateComment(Comment: Comment): void;

/**
 * Blanks the damping relative
 */
    Blank(): void;

/**
 * Blanks all of the damping relatives in the model.
 * @param Model Model that all damping relatives will be blanked in
 * @param redraw If model should be redrawn or not. If omitted redraw is false. If you want to do several (un)blanks and only redraw after the last one then use false for all redraws apart from the last one. Alternatively you can redraw using View.Redraw().
 */
    static BlankAll(Model: Model, redraw?: boolean): void;

/**
 * Blanks all of the flagged damping relatives in the model.
 * @param Model Model that all the flagged damping relatives will be blanked in
 * @param flag Flag set on the damping relatives that you want to blank
 * @param redraw If model should be redrawn or not. If omitted redraw is false. If you want to do several (un)blanks and only redraw after the last one then use false for all redraws apart from the last one. Alternatively you can redraw using View.Redraw().
 */
    static BlankFlagged(Model: Model, flag: number, redraw?: boolean): void;

/**
 * Checks if the damping relative is blanked or not.
 * @returns true if blanked, false if not.
 */
    Blanked(): boolean;

/**
 * Clears a flag on the damping relative.
 * @param flag Flag to clear on the damping relative
 */
    ClearFlag(flag: number): void;

/**
 * Copies the damping relative. The target include of the copied damping relative can be set using Options.copy_target_include.
 * @param range If you want to keep the copied item in the range specified for the current include. Default value is false. To set current include, use Include.MakeCurrentLayer().
 * @returns DampingRelative object
 */
    Copy(range?: boolean): DampingRelative;

/**
 * Detaches a comment from a damping relative.
 * @param Comment Comment that will be detached from the damping relative
 */
    DetachComment(Comment: Comment): void;

/**
 * Adds an error for damping relative. For more details on checking see the Check class.
 * @param message The error message to give
 * @param details An optional detailed error message
 */
    Error(message: string, details?: string): void;

/**
 * Returns the first damping relative in the model.
 * @param Model Model to get first damping relative in
 * @returns DampingRelative object (or null if there are no damping relatives in the model).
 */
    static First(Model: Model): DampingRelative;

/**
 * Flags all of the damping relatives in the model with a defined flag.
 * @param Model Model that all damping relatives will be flagged in
 * @param flag Flag to set on the damping relatives
 */
    static FlagAll(Model: Model, flag: number): void;

/**
 * Checks if the damping relative is flagged or not.
 * @param flag Flag to test on the damping relative
 * @returns true if flagged, false if not.
 */
    Flagged(flag: number): boolean;

/**
 * Calls a function for each damping relative in the model. Note that ForEach has been designed to make looping over damping relatives as fast as possible and so has some limitations. Firstly, a single temporary DampingRelative object is created and on each function call it is updated with the current damping relative data. This means that you should not try to store the DampingRelative object for later use (e.g. in an array) as it is temporary. Secondly, you cannot create new damping relatives inside a ForEach loop. 
 * @param Model Model that all damping relatives are in
 * @param func Function to call for each damping relative
 * @param extra An optional extra object/array/string etc that will appended to arguments when calling the function
 */
    static ForEach(Model: Model, func: () => void, extra?: any): void;

/**
 * Returns an array of DampingRelative objects or properties for all of the damping relatives in a model in PRIMER. If the optional property argument is not given then an array of DampingRelative objects is returned. If the property argument is given, that property value for each damping relative is returned in the array instead of a DampingRelative object
 * @param Model Model to get damping relatives from
 * @param property Name for property to get for all damping relatives in the model
 * @returns Array of DampingRelative objects or properties
 */
    static GetAll(Model: Model, property?: string): DampingRelative[];

/**
 * Extracts the comments associated to a damping relative.
 * @returns Array of Comment objects (or null if there are no comments associated to the node).
 */
    GetComments(): Comment[];

/**
 * Returns an array of DampingRelative objects for all of the flagged damping relatives in a model in PRIMER If the optional property argument is not given then an array of DampingRelative objects is returned. If the property argument is given, then that property value for each damping relative is returned in the array instead of a DampingRelative object
 * @param Model Model to get damping relatives from
 * @param flag Flag set on the damping relatives that you want to retrieve
 * @param property Name for property to get for all flagged damping relatives in the model
 * @returns Array of DampingRelative objects or properties
 */
    static GetFlagged(Model: Model, flag: number, property?: string): DampingRelative[];

/**
 * Returns the DampingRelative object for a damping relative ID.
 * @param Model Model to find the damping relative in
 * @param number number of the damping relative you want the DampingRelative object for
 * @returns DampingRelative object (or null if damping relative does not exist).
 */
    static GetFromID(Model: Model, number: number): DampingRelative;

/**
 * Checks if a DampingRelative property is a parameter or not. Note that object properties that are parameters are normally returned as the integer or float parameter values as that is virtually always what the user would want. For this function to work the JavaScript interpreter must use the parameter name instead of the value. This can be done by setting the Options.property_parameter_names option to true before calling the function and then resetting it to false afterwards.. This behaviour can also temporarily be switched by using the DampingRelative.ViewParameters() method and 'method chaining' (see the examples below).
 * @param prop damping relative property to get parameter for
 * @returns Parameter object if property is a parameter, null if not.
 */
    GetParameter(prop: string): Parameter;

/**
 * Returns the keyword for this damping relative (*DAMPING_RELATIVE). Note that a carriage return is not added. See also DampingRelative.KeywordCards()
 * @returns string containing the keyword.
 */
    Keyword(): string;

/**
 * Returns the keyword cards for the damping relative. Note that a carriage return is not added. See also DampingRelative.Keyword()
 * @returns string containing the cards.
 */
    KeywordCards(): string;

/**
 * Returns the last damping relative in the model.
 * @param Model Model to get last damping relative in
 * @returns DampingRelative object (or null if there are no damping relatives in the model).
 */
    static Last(Model: Model): DampingRelative;

/**
 * Returns the next damping relative in the model.
 * @returns DampingRelative object (or null if there are no more damping relatives in the model).
 */
    Next(): DampingRelative;

/**
 * Allows the user to pick a damping relative.
 * @param prompt Text to display as a prompt to the user
 * @param limit If the argument is a Model then only damping relatives from that model can be picked. If the argument is a Flag then only damping relatives that are flagged with limit can be selected. If omitted, or null, any damping relatives from any model can be selected. from any model.
 * @param modal If picking is modal (blocks the user from doing anything else in PRIMER until this window is dismissed). If omitted the pick will be modal.
 * @param button_text By default the window with the prompt will have a button labelled 'Cancel' which if pressed will cancel the pick and return null. If you want to change the text on the button use this argument. If omitted 'Cancel' will be used.
 * @returns DampingRelative object (or null if not picked)
 */
    static Pick(prompt: string, limit?: Model | number, modal?: boolean, button_text?: string): DampingRelative;

/**
 * Returns the previous damping relative in the model.
 * @returns DampingRelative object (or null if there are no more damping relatives in the model).
 */
    Previous(): DampingRelative;

/**
 * Allows the user to select damping relatives using standard PRIMER object menus.
 * @param flag Flag to use when selecting damping relatives
 * @param prompt Text to display as a prompt to the user
 * @param limit If the argument is a Model then only damping relatives from that model can be selected. If the argument is a Flag then only damping relatives that are flagged with limit can be selected (limit should be different to flag). If omitted, or null, any damping relatives can be selected. from any model.
 * @param modal If selection is modal (blocks the user from doing anything else in PRIMER until this window is dismissed). If omitted the selection will be modal.
 * @returns Number of damping relatives selected or null if menu cancelled
 */
    static Select(flag: number, prompt: string, limit?: Model | number, modal?: boolean): number;

/**
 * Sets a flag on the damping relative.
 * @param flag Flag to set on the damping relative
 */
    SetFlag(flag: number): void;

/**
 * Sketches the damping relative. The damping relative will be sketched until you either call DampingRelative.Unsketch(), DampingRelative.UnsketchAll(), Model.UnsketchAll(), or delete the model
 * @param redraw If model should be redrawn or not after the damping relative is sketched. If omitted redraw is true. If you want to sketch several damping relatives and only redraw after the last one then use false for redraw and call View.Redraw().
 */
    Sketch(redraw?: boolean): void;

/**
 * Sketches all of the flagged damping relatives in the model. The damping relatives will be sketched until you either call DampingRelative.Unsketch(), DampingRelative.UnsketchFlagged(), Model.UnsketchAll(), or delete the model
 * @param Model Model that all the flagged damping relatives will be sketched in
 * @param flag Flag set on the damping relatives that you want to sketch
 * @param redraw If model should be redrawn or not after the damping relatives are sketched. If omitted redraw is true. If you want to sketch flagged damping relatives several times and only redraw after the last one then use false for redraw and call View.Redraw().
 */
    static SketchFlagged(Model: Model, flag: number, redraw?: boolean): void;

/**
 * Returns the total number of damping relatives in the model.
 * @param Model Model to get total for
 * @param exists true if only existing damping relatives should be counted. If false or omitted referenced but undefined damping relatives will also be included in the total.
 * @returns number of damping relatives
 */
    static Total(Model: Model, exists?: boolean): number;

/**
 * Unblanks the damping relative
 */
    Unblank(): void;

/**
 * Unblanks all of the damping relatives in the model.
 * @param Model Model that all damping relatives will be unblanked in
 * @param redraw If model should be redrawn or not. If omitted redraw is false. If you want to do several (un)blanks and only redraw after the last one then use false for all redraws apart from the last one. Alternatively you can redraw using View.Redraw().
 */
    static UnblankAll(Model: Model, redraw?: boolean): void;

/**
 * Unblanks all of the flagged damping relatives in the model.
 * @param Model Model that the flagged damping relatives will be unblanked in
 * @param flag Flag set on the damping relatives that you want to unblank
 * @param redraw If model should be redrawn or not. If omitted redraw is false. If you want to do several (un)blanks and only redraw after the last one then use false for all redraws apart from the last one. Alternatively you can redraw using View.Redraw().
 */
    static UnblankFlagged(Model: Model, flag: number, redraw?: boolean): void;

/**
 * Unsets a defined flag on all of the damping relatives in the model.
 * @param Model Model that the defined flag for all damping relatives will be unset in
 * @param flag Flag to unset on the damping relatives
 */
    static UnflagAll(Model: Model, flag: number): void;

/**
 * Unsketches the damping relative.
 * @param redraw If model should be redrawn or not after the damping relative is unsketched. If omitted redraw is true. If you want to unsketch several damping relatives and only redraw after the last one then use false for redraw and call View.Redraw().
 */
    Unsketch(redraw?: boolean): void;

/**
 * Unsketches all damping relatives.
 * @param Model Model that all damping relatives will be unblanked in
 * @param redraw If model should be redrawn or not after the damping relatives are unsketched. If omitted redraw is true. If you want to unsketch several things and only redraw after the last one then use false for redraw and call View.Redraw().
 */
    static UnsketchAll(Model: Model, redraw?: boolean): void;

/**
 * Unsketches all flagged damping relatives in the model.
 * @param Model Model that all damping relatives will be unsketched in
 * @param flag Flag set on the damping relatives that you want to unsketch
 * @param redraw If model should be redrawn or not after the damping relatives are unsketched. If omitted redraw is true. If you want to unsketch several things and only redraw after the last one then use false for redraw and call View.Redraw().
 */
    static UnsketchFlagged(Model: Model, flag: number, redraw?: boolean): void;

/**
 * Object properties that are parameters are normally returned as the integer or float parameter values as that is virtually always what the user would want. This function temporarily changes the behaviour so that if a property is a parameter the parameter name is returned instead. This can be used with 'method chaining' (see the example below) to make sure a property argument is correct.
 * @returns DampingRelative object.
 */
    ViewParameters(): DampingRelative;

/**
 * Adds a warning for damping relative. For more details on checking see the Check class.
 * @param message The warning message to give
 * @param details An optional detailed warning message
 */
    Warning(message: string, details?: string): void;

/**
 * Returns the cross references for this damping relative.
 * @returns Xrefs object.
 */
    Xrefs(): Xrefs;

/**
 * Create a new DampingRelative object for *DAMPING_RELATIVE.
 * @param Model Model that damping relative will be created in
 * @param pidrb Part Rigid body ID
 * @param psid Set Part set ID.
 * @param cdamp Fraction of critical damping
 * @param freq Target frequency
 * @param dv2 Constant for velocity squared term
 * @param lcid Fraction of cricitcal damping vs time
 * @returns DampingRelative object
 */
    constructor(Model: Model, pidrb: number, psid: number, cdamp?: number, freq?: number, dv2?: number, lcid?: number);

/**
 * Creates a string containing the damping relative data in keyword format. Note that this contains the keyword header and the keyword cards. See also DampingRelative.Keyword() and DampingRelative.KeywordCards().
 * @returns string
 */
    toString(): string;

/** Fraction of critical damping */
    cdamp: number;
/** Constant for velocity squared term */
    dv2: number;
/** true if damping relative exists, false if referred to but not defined. */
    readonly exists: boolean;
/** Target frequency */
    freq: number;
/** The Include file number that the damping relative is in. */
    include: number;
/** Fraction of cricitcal damping vs time */
    lcid: number;
/** The Model number that the damping relative is in. */
    readonly model: number;
/** Part Rigid body ID */
    pidrb: number;
/** Set Part set ID. */
    psid: number;
}

declare class Database {
/** *DATABASE_ABSTAT card */
    abstat: object;
/** *DATABASE_ATDOUT card */
    atdout: object;
/** Beam option */
    beam: number;
/** Beam option */
    beam_1: number;
/** #beam int points to output */
    beamip: number;
/** *DATABASE_BEARING card */
    bearing: object;
/** *DATABASE_BINARY card */
    binary: object;
/** Flag for binary file */
    binary_1: number;
/** Flag for binary file */
    binary_10: number;
/** Flag for binary file */
    binary_11: number;
/** Flag for binary file */
    binary_12: number;
/** Flag for binary file */
    binary_13: number;
/** Flag for binary file */
    binary_14: number;
/** Flag for binary file */
    binary_15: number;
/** Flag for binary file */
    binary_16: number;
/** Flag for binary file */
    binary_17: number;
/** Flag for binary file */
    binary_18: number;
/** Flag for binary file */
    binary_19: number;
/** Flag for binary file */
    binary_2: number;
/** Flag for binary file */
    binary_20: number;
/** Flag for binary file */
    binary_21: number;
/** Flag for binary file */
    binary_22: number;
/** Flag for binary file */
    binary_23: number;
/** Flag for binary file */
    binary_24: number;
/** Flag for binary file */
    binary_25: number;
/** Flag for binary file */
    binary_26: number;
/** Flag for binary file */
    binary_27: number;
/** Flag for binary file */
    binary_28: number;
/** Flag for binary file */
    binary_29: number;
/** Flag for binary file */
    binary_3: number;
/** Flag for binary file */
    binary_30: number;
/** Flag for binary file */
    binary_31: number;
/** Flag for binary file */
    binary_32: number;
/** Flag for binary file */
    binary_33: number;
/** Flag for binary file */
    binary_34: number;
/** Flag for binary file */
    binary_35: number;
/** Flag for binary file */
    binary_36: number;
/** Flag for binary file */
    binary_37: number;
/** Flag for binary file */
    binary_38: number;
/** Flag for binary file */
    binary_39: number;
/** Flag for binary file */
    binary_4: number;
/** Flag for binary file */
    binary_5: number;
/** Flag for binary file */
    binary_6: number;
/** Flag for binary file */
    binary_7: number;
/** Flag for binary file */
    binary_8: number;
/** Flag for binary file */
    binary_9: number;
/** *DATABASE_BINARY_BLSTFOR card */
    blstfor: object;
/** *DATABASE_BNDOUT card */
    bndout: object;
/** Output for beam elements. +n is output for elements in beam set n, 0 no beam, -1 all elements */
    bsetid: number;
/** Beam Set ID */
    bsetid_1: number;
/** Flag to output composite material stress in local csys */
    cmpflg: number;
/** *DATABASE_BINARY_CPMFOR card */
    cpmfor: object;
/** Output flag for quadratic solid types */
    cubsld: number;
/** Frequency cut-off C in Hz */
    cutoff: number;
/** Output interval in cycles */
    cycl: number;
/** Output interval in cycles */
    cycl_1: number;
/** Output interval in cycles */
    cycl_2: number;
/** *DATABASE_BINARY_D3CRACK card */
    d3crack: object;
/** *DATABASE_BINARY_D3DRLF card */
    d3drlf: object;
/** *DATABASE_BINARY_D3DUMP card */
    d3dump: object;
/** *DATABASE_BINARY_D3MEAN card */
    d3mean: object;
/** *DATABASE_BINARY_D3PART card */
    d3part: object;
/** *DATABASE_BINARY_D3PLOT card */
    d3plot: object;
/** *DATABASE_BINARY_D3PROP card */
    d3prop: object;
/** *DATABASE_BINARY_D3THDT card */
    d3thdt: object;
/** *DATABASE_DCFAIL card */
    dcfail: object;
/** Data compression flag */
    dcomp: number;
/** *DATABASE_DEFGEO card */
    defgeo: object;
/** *DATABASE_DEFORC card */
    deforc: object;
/** Output flag for results of deleted elements */
    deleres: number;
/** *DATABASE_BINARY_DEMFOR card */
    demfor: object;
/** *DATABASE_DESTAT card */
    destat: object;
/** Time interval between output */
    dt: number;
/** Time interval between output */
    dt_1: number;
/** Time interval between output */
    dt_10: number;
/** Time interval between output */
    dt_11: number;
/** Time interval between output */
    dt_12: number;
/** Time interval between output */
    dt_13: number;
/** Time interval between output */
    dt_14: number;
/** Time interval between output */
    dt_15: number;
/** Time interval between output */
    dt_16: number;
/** Time interval between output */
    dt_17: number;
/** Time interval between output */
    dt_18: number;
/** Time interval between output */
    dt_19: number;
/** Time interval between output */
    dt_2: number;
/** Time interval between output */
    dt_20: number;
/** Time interval between output */
    dt_21: number;
/** Time interval between output */
    dt_22: number;
/** Time interval between output */
    dt_23: number;
/** Time interval between output */
    dt_24: number;
/** Time interval between output */
    dt_25: number;
/** Time interval between output */
    dt_26: number;
/** Time interval between output */
    dt_27: number;
/** Time interval between output */
    dt_28: number;
/** Time interval between output */
    dt_29: number;
/** Time interval between output */
    dt_3: number;
/** Time interval between output */
    dt_30: number;
/** Time interval between output */
    dt_31: number;
/** Time interval between output */
    dt_32: number;
/** Time interval between output */
    dt_33: number;
/** Time interval between output */
    dt_34: number;
/** Time interval between output */
    dt_35: number;
/** Time interval between output */
    dt_36: number;
/** Time interval between output */
    dt_37: number;
/** Time interval between output */
    dt_38: number;
/** Time interval between outputs */
    dt_39: number;
/** Time interval between output */
    dt_4: number;
/** Time interval between outputs */
    dt_40: number;
/** Time interval between outputs */
    dt_41: number;
/** Time interval between outputs */
    dt_42: number;
/** Time interval between outputs */
    dt_43: number;
/** Time interval between outputs */
    dt_44: number;
/** Time interval between outputs */
    dt_45: number;
/** Time interval between outputs */
    dt_46: number;
/** Time interval between outputs */
    dt_47: number;
/** Time interval between outputs */
    dt_48: number;
/** Time interval between outputs */
    dt_49: number;
/** Time interval between output */
    dt_5: number;
/** Time interval between outputs */
    dt_50: number;
/** Time interval between outputs */
    dt_51: number;
/** Time interval between output */
    dt_6: number;
/** Time interval between output */
    dt_7: number;
/** Time interval between output */
    dt_8: number;
/** Time interval between output */
    dt_9: number;
/** output of nodal temp */
    dtdt: number;
/** *DATABASE_ELOUT card */
    elout: object;
/** Flag to in/exclude shell energy &amp; thickness */
    engflg: number;
/** Flag to in/exclude shell energy &amp; thickness */
    engflg_1: number;
/** *DATABASE_ENVELOPE card */
    envelope: object;
/** Flag to in/exclude shell strains */
    epsflg: number;
/** Flag to in/exclude shell strains */
    epsflg_1: number;
/** true if database card exists */
    exists: boolean;
/** true if database card exists */
    exists_1: boolean;
/** true if database card exists */
    exists_10: boolean;
/** true if database card exists */
    exists_11: boolean;
/** true if database card exists */
    exists_12: boolean;
/** true if database card exists */
    exists_13: boolean;
/** true if database card exists */
    exists_14: boolean;
/** true if database card exists */
    exists_15: boolean;
/** true if database card exists */
    exists_16: boolean;
/** true if database card exists */
    exists_17: boolean;
/** true if database card exists */
    exists_18: boolean;
/** true if database card exists */
    exists_19: boolean;
/** true if database card exists */
    exists_2: boolean;
/** true if database card exists */
    exists_20: boolean;
/** true if database card exists */
    exists_21: boolean;
/** true if database card exists */
    exists_22: boolean;
/** true if database card exists */
    exists_23: boolean;
/** true if database card exists */
    exists_24: boolean;
/** true if database card exists */
    exists_25: boolean;
/** true if database card exists */
    exists_26: boolean;
/** true if database card exists */
    exists_27: boolean;
/** true if database card exists */
    exists_28: boolean;
/** true if database card exists */
    exists_29: boolean;
/** true if database card exists */
    exists_3: boolean;
/** true if database card exists */
    exists_30: boolean;
/** true if database card exists */
    exists_31: boolean;
/** true if database card exists */
    exists_32: boolean;
/** true if database card exists */
    exists_33: boolean;
/** true if database card exists */
    exists_34: boolean;
/** true if database card exists */
    exists_35: boolean;
/** true if database card exists */
    exists_36: boolean;
/** true if database card exists */
    exists_37: boolean;
/** true if database card exists */
    exists_38: boolean;
/** true if database card exists */
    exists_39: boolean;
/** true if database card exists */
    exists_4: boolean;
/** true if database binary card exists */
    exists_40: boolean;
/** true if database binary card exists */
    exists_41: boolean;
/** true if database binary card exists */
    exists_42: boolean;
/** true if database binary card exists */
    exists_43: boolean;
/** true if database binary card exists */
    exists_44: boolean;
/** true if database binary card exists */
    exists_45: boolean;
/** true if database binary card exists */
    exists_46: boolean;
/** true if database binary card exists */
    exists_47: boolean;
/** true if database binary card exists */
    exists_48: boolean;
/** true if database binary card exists */
    exists_49: boolean;
/** true if database card exists */
    exists_5: boolean;
/** true if database binary card exists */
    exists_50: boolean;
/** true if database binary card exists */
    exists_51: boolean;
/** true if database binary card exists */
    exists_52: boolean;
/** true if database binary card exists */
    exists_53: boolean;
/** true if database binary card exists */
    exists_54: boolean;
/** true if database binary card exists */
    exists_55: boolean;
/** true if database binary card exists */
    exists_56: boolean;
/** true if database card exists */
    exists_57: boolean;
/** true if database card exists */
    exists_58: boolean;
/** true if database card exists */
    exists_59: boolean;
/** true if database card exists */
    exists_6: boolean;
/** true if database card exists */
    exists_60: boolean;
/** true if database card exists */
    exists_61: boolean;
/** true if database card exists */
    exists_7: boolean;
/** true if database card exists */
    exists_8: boolean;
/** true if database card exists */
    exists_9: boolean;
/** *DATABASE_EXTENT_BINARY card */
    extent_binary: object;
/** *DATABASE_EXTENT_BINARY_COMP card */
    extent_binary_comp: object;
/** *DATABASE_EXTENT_D3PART card */
    extent_d3part: object;
/** *DATABASE_EXTENT_INTFOR card */
    extent_intfor: object;
/** Filename of the database for the INTFOR data */
    fname: string;
/** *DATABASE_FORMAT card */
    format: object;
/** *DATABASE_BINARY_FSIFOR card */
    fsifor: object;
/** *DATABASE_BINARY_FSILNK card */
    fsilnk: object;
/** *DATABASE_GCEOUT card */
    gceout: object;
/** *DATABASE_GLSTAT card */
    glstat: object;
/** *DATABASE_H3OUT card */
    h3out: object;
/** Solid Set ID */
    hsetid: number;
/** adds extra history variables */
    hydro: number;
/** output of accleration data */
    iacc: string;
/** output ale materials */
    ialemat: number;
/** Averaging time interval */
    iavg: number;
/** Word size for binary output files */
    ibinary: number;
/** *DATABASE_DEBOND card */
    icvout: object;
/** *DATABASE_ICVOUT card */
    icvout_1: object;
/** Every INTFOR database to separate file flag */
    ieverf: number;
/** Every D3PLOT file to separate database flag */
    ieverp: number;
/** Every D3PLOT file to separate database flag */
    ieverp_1: number;
/** Output data flag */
    ifile: number;
/** Output format for D3PLOT and D3THDT files */
    iform: number;
/** output of global data */
    iglb: string;
/** Output *EOS, *HOURGLASS, *MAT, *part and *SECTION data */
    imatl: number;
/** The Include file number that the database card is in. */
    include: number;
/** The Include file number that the database card is in. */
    include_1: number;
/** The Include file number that the database card is in. */
    include_10: number;
/** The Include file number that the database card is in. */
    include_11: number;
/** The Include file number that the database card is in. */
    include_12: number;
/** The Include file number that the database card is in. */
    include_13: number;
/** The Include file number that the database card is in. */
    include_14: number;
/** The Include file number that the database card is in. */
    include_15: number;
/** The Include file number that the database card is in. */
    include_16: number;
/** The Include file number that the database card is in. */
    include_17: number;
/** The Include file number that the database card is in. */
    include_18: number;
/** The Include file number that the database card is in. */
    include_19: number;
/** The Include file number that the database card is in. */
    include_2: number;
/** The Include file number that the database card is in. */
    include_20: number;
/** The Include file number that the database card is in. */
    include_21: number;
/** The Include file number that the database card is in. */
    include_22: number;
/** The Include file number that the database card is in. */
    include_23: number;
/** The Include file number that the database card is in. */
    include_24: number;
/** The Include file number that the database card is in. */
    include_25: number;
/** The Include file number that the database card is in. */
    include_26: number;
/** The Include file number that the database card is in. */
    include_27: number;
/** The Include file number that the database card is in. */
    include_28: number;
/** The Include file number that the database card is in. */
    include_29: number;
/** The Include file number that the database card is in. */
    include_3: number;
/** The Include file number that the database card is in. */
    include_30: number;
/** The Include file number that the database card is in. */
    include_31: number;
/** The Include file number that the database card is in. */
    include_32: number;
/** The Include file number that the database card is in. */
    include_33: number;
/** The Include file number that the database card is in. */
    include_34: number;
/** The Include file number that the database card is in. */
    include_35: number;
/** The Include file number that the database card is in. */
    include_36: number;
/** The Include file number that the database card is in. */
    include_37: number;
/** The Include file number that the database card is in. */
    include_38: number;
/** The Include file number that the database card is in. */
    include_39: number;
/** The Include file number that the database card is in. */
    include_4: number;
/** The Include file number that the database binary card is in. */
    include_40: number;
/** The Include file number that the database binary card is in. */
    include_41: number;
/** The Include file number that the database binary card is in. */
    include_42: number;
/** The Include file number that the database binary card is in. */
    include_43: number;
/** The Include file number that the database binary card is in. */
    include_44: number;
/** The Include file number that the database binary card is in. */
    include_45: number;
/** The Include file number that the database binary card is in. */
    include_46: number;
/** The Include file number that the database binary card is in. */
    include_47: number;
/** The Include file number that the database binary card is in. */
    include_48: number;
/** The Include file number that the database binary card is in. */
    include_49: number;
/** The Include file number that the database card is in. */
    include_5: number;
/** The Include file number that the database binary card is in. */
    include_50: number;
/** The Include file number that the database binary card is in. */
    include_51: number;
/** The Include file number that the database binary card is in. */
    include_52: number;
/** The Include file number that the database binary card is in. */
    include_53: number;
/** The Include file number that the database binary card is in. */
    include_54: number;
/** The Include file number that the database binary card is in. */
    include_55: number;
/** The Include file number that the database binary card is in. */
    include_56: number;
/** The Include file number that the database card is in. */
    include_57: number;
/** The Include file number that the database card is in. */
    include_58: number;
/** The Include file number that the database card is in. */
    include_59: number;
/** The Include file number that the database card is in. */
    include_6: number;
/** The Include file number that the database card is in. */
    include_60: number;
/** The Include file number that the database card is in. */
    include_61: number;
/** The Include file number that the database card is in. */
    include_7: number;
/** The Include file number that the database card is in. */
    include_8: number;
/** The Include file number that the database card is in. */
    include_9: number;
/** *DATABASE_BINARY_INTFOR card */
    intfor: object;
/** output of intg pt data */
    intout: string;
/** Flag for behaviour of load curve */
    ioopt: number;
/** Flag for behaviour of load curve */
    ioopt_1: number;
/** Flag for behaviour of load curve */
    ioopt_10: number;
/** Flag for behaviour of load curve */
    ioopt_11: number;
/** Flag for behaviour of load curve */
    ioopt_12: number;
/** Flag for behaviour of load curve */
    ioopt_13: number;
/** Flag for behaviour of load curve */
    ioopt_14: number;
/** Flag for behaviour of load curve */
    ioopt_15: number;
/** Flag for behaviour of load curve */
    ioopt_16: number;
/** Flag for behaviour of load curve */
    ioopt_17: number;
/** Flag for behaviour of load curve */
    ioopt_18: number;
/** Flag for behaviour of load curve */
    ioopt_19: number;
/** Flag for behaviour of load curve */
    ioopt_2: number;
/** Flag for behaviour of load curve */
    ioopt_20: number;
/** Flag for behaviour of load curve */
    ioopt_21: number;
/** Flag for behaviour of load curve */
    ioopt_22: number;
/** Flag for behaviour of load curve */
    ioopt_23: number;
/** Flag for behaviour of load curve */
    ioopt_24: number;
/** Flag for behaviour of load curve */
    ioopt_25: number;
/** Flag for behaviour of load curve */
    ioopt_26: number;
/** Flag for behaviour of load curve */
    ioopt_27: number;
/** Flag for behaviour of load curve */
    ioopt_28: number;
/** Flag for behaviour of load curve */
    ioopt_29: number;
/** Flag for behaviour of load curve */
    ioopt_3: number;
/** Flag for behaviour of load curve */
    ioopt_30: number;
/** Flag for behaviour of load curve */
    ioopt_31: number;
/** Flag for behaviour of load curve */
    ioopt_32: number;
/** Flag for behaviour of load curve */
    ioopt_33: number;
/** Flag for behaviour of load curve */
    ioopt_34: number;
/** Flag for behaviour of load curve */
    ioopt_35: number;
/** Flag for behaviour of load curve */
    ioopt_36: number;
/** Flag for behaviour of load curve */
    ioopt_37: number;
/** Flag for behaviour of load curve */
    ioopt_38: number;
/** Flag for lcdt behaviour */
    ioopt_39: number;
/** Flag for behaviour of load curve */
    ioopt_4: number;
/** governs how the plot state frequency is determined from curve LCDT */
    ioopt_40: number;
/** Flag for behaviour of load curve */
    ioopt_5: number;
/** Flag for behaviour of load curve */
    ioopt_6: number;
/** Flag for behaviour of load curve */
    ioopt_7: number;
/** Flag for behaviour of load curve */
    ioopt_8: number;
/** Flag for behaviour of load curve */
    ioopt_9: number;
/** output of strain energy density data */
    ised: string;
/** *DATABASE_BINARY_ISPHFOR card */
    isphfor: object;
/** Level of statistics */
    istats: number;
/** output of strain data */
    istra: string;
/** output of stress data */
    istrs: string;
/** output of velocity data */
    ivel: string;
/** Output *RIGIDWALL data */
    iwall: number;
/** output of geometry data */
    ixyz: string;
/** *DATABASE_JNTFORC card */
    jntforc: object;
/** Curve ID giving time interval between dumps */
    lcdt: number;
/** Curve ID giving time interval between dumps */
    lcdt_1: number;
/** Curve ID giving time interval between dumps */
    lcdt_2: number;
/** Curve ID giving time interval between dumps */
    lcdt_3: number;
/** Curve ID specifying time interval */
    lcur: number;
/** Curve ID specifying time interval */
    lcur_1: number;
/** Curve ID specifying time interval */
    lcur_10: number;
/** Curve ID specifying time interval */
    lcur_11: number;
/** Curve ID specifying time interval */
    lcur_12: number;
/** Curve ID specifying time interval */
    lcur_13: number;
/** Curve ID specifying time interval */
    lcur_14: number;
/** Curve ID specifying time interval */
    lcur_15: number;
/** Curve ID specifying time interval */
    lcur_16: number;
/** Curve ID specifying time interval */
    lcur_17: number;
/** Curve ID specifying time interval */
    lcur_18: number;
/** Curve ID specifying time interval */
    lcur_19: number;
/** Curve ID specifying time interval */
    lcur_2: number;
/** Curve ID specifying time interval */
    lcur_20: number;
/** Curve ID specifying time interval */
    lcur_21: number;
/** Curve ID specifying time interval */
    lcur_22: number;
/** Curve ID specifying time interval */
    lcur_23: number;
/** Curve ID specifying time interval */
    lcur_24: number;
/** Curve ID specifying time interval */
    lcur_25: number;
/** Curve ID specifying time interval */
    lcur_26: number;
/** Curve ID specifying time interval */
    lcur_27: number;
/** Curve ID specifying time interval */
    lcur_28: number;
/** Curve ID specifying time interval */
    lcur_29: number;
/** Curve ID specifying time interval */
    lcur_3: number;
/** Curve ID specifying time interval */
    lcur_30: number;
/** Curve ID specifying time interval */
    lcur_31: number;
/** Curve ID specifying time interval */
    lcur_32: number;
/** Curve ID specifying time interval */
    lcur_33: number;
/** Curve ID specifying time interval */
    lcur_34: number;
/** Curve ID specifying time interval */
    lcur_35: number;
/** Curve ID specifying time interval */
    lcur_36: number;
/** Curve ID specifying time interval */
    lcur_37: number;
/** Curve ID specifying time interval */
    lcur_38: number;
/** Curve ID specifying time interval */
    lcur_4: number;
/** Curve ID specifying time interval */
    lcur_5: number;
/** Curve ID specifying time interval */
    lcur_6: number;
/** Curve ID specifying time interval */
    lcur_7: number;
/** Curve ID specifying time interval */
    lcur_8: number;
/** Curve ID specifying time interval */
    lcur_9: number;
/** Flag to include mass and inertia properties */
    mass_properties: number;
/** *DATABASE_MATSUM card */
    matsum: object;
/** #integration points for shell output */
    maxint: number;
/** #integration points for shell output */
    maxint_1: number;
/** output nodal mass scaling data */
    msscl: number;
/** Output for material energies to D3THDT file */
    n3thdt: number;
/** *DATABASE_NCFORC card */
    ncforc: object;
/** Output of loop-stresses to D3PLOT */
    neipb: number;
/** #extra values for solids */
    neiph: number;
/** #extra values for solids */
    neiph_1: number;
/** #extra values for shells */
    neips: number;
/** #extra values for shells */
    neips_1: number;
/** Output contact energy density for mortar contact and SOFT = 2 contact */
    neng: number;
/** Display deleted contact segments flag */
    nfail: number;
/** Output forces */
    nforc: number;
/** Output contact gaps */
    ngapc: number;
/** Output global variables */
    nglbv: number;
/** Number of user friction history variables to output from user defined friction routines */
    nhuf: number;
/** number of solid integration pts */
    nintsld: number;
/** number of solid integration pts */
    nintsld_1: number;
/** *DATABASE_NODFOR card */
    nodfor: object;
/** *DATABASE_NODOUT card */
    nodout: object;
/** output of connectivity nodes */
    nodout_1: string;
/** Output penetration information for mortar contact */
    npen: number;
/** Number of plot files */
    npltc: number;
/** Number of plot files */
    npltc_1: number;
/** Output pressures */
    npresu: number;
/** Number of running restart files */
    nr: number;
/** Output shear stresses */
    nshear: number;
/** Output tied segments for Mortar contact */
    ntied: number;
/** Output nodal velocity */
    nvelo: number;
/** Output contact wear data mode */
    nwear: number;
/** Output (total) sliding interface energy density for mortar contact */
    nwrk: number;
/** Number of user wear history variables */
    nwusr: number;
/** extra history variables for solids */
    option1: number;
/** High frequency output interval */
    option1_1: number;
/** extra history variables for shells */
    option2: number;
/** Flag for binary file for high frequency output */
    option2_1: number;
/** extra history variables for thick shells */
    option3: number;
/** extra history variables for beams */
    option4: number;
/** Output format. Can be 0 or 1 */
    output: number;
/** *DATABASE_PBSTAT card */
    pbstat: object;
/** Flag to output peak pressure and surface energy for each contact interface */
    pkp_sen: number;
/** *DATABASE_PLLYOUT card */
    pllyout: object;
/** *DATABASE_PRTUBE card */
    prtube: object;
/** Part set ID for filtering */
    pset: number;
/** Part Set ID */
    psetid: number;
/** Part Set ID */
    psetid_1: number;
/** *DATABASE_PYRO card */
    pyro: object;
/** Output flag for cubic solid types */
    quadsld: number;
/** Time interval T between filter sampling */
    rate: number;
/** *DATABASE_RBDOUT card */
    rbdout: object;
/** *DATABASE_RCFORC card */
    rcforc: object;
/** Output of residual forces */
    resplt: number;
/** Flag to in/exclude shell force/moment resultants */
    rltflg: number;
/** Flag to in/exclude shell force/moment resultants */
    rltflg_1: number;
/** *DATABASE_BINARY_RUNRSF card */
    runrsf: object;
/** *DATABASE_RWFORC card */
    rwforc: object;
/** *DATABASE_SBTOUT card */
    sbtout: object;
/** Scaling parameter used in the computation of the peak pressure */
    sclp: number;
/** *DATABASE_SECFORC card */
    secforc: object;
/** Shell hourglass energy output flag */
    shge: number;
/** Shell hourglass energy output flag */
    shge_1: number;
/** Flag to in/exclude shell stress tensors */
    sigflg: number;
/** Flag to in/exclude shell stress tensors */
    sigflg_1: number;
/** *DATABASE_SLEOUT card */
    sleout: object;
/** *DATABASE_SNSROUT card */
    snsrout: object;
/** *DATABASE_SPCFORC card */
    spcforc: object;
/** *DATABASE_SPGSPL card */
    spgcpl: object;
/** *DATABASE_SPHMASSFLOW card */
    sphmassflow: object;
/** *DATABASE_SPHOUT card */
    sphout: object;
/** Output for shell elements. +n is output for elements in shell set n, 0 no shell, -1 all elements */
    ssetid: number;
/** Shell Set ID */
    ssetid_1: number;
/** Strain tensor output flag */
    strflg: number;
/** Strain tensor output flag */
    strflg_1: number;
/** Output shell element dt flag */
    stssz: number;
/** Output shell element dt flag */
    stssz_1: number;
/** *DATABASE_SWFORC card */
    swforc: object;
/** Time interval for backup output files during the analysis */
    tback: number;
/** Time interval for checking whether the previous maxima/minima are exceeded */
    tcheck: number;
/** Output of thermal data to D3PLOT */
    therm: number;
/** *DATABASE_TPRINT card */
    tprint: object;
/** *DATABASE_TRHIST card */
    trhist: object;
/** TShell Set ID */
    tsetid: number;
/** Start time */
    tstart: number;
/** Flag for filtering options */
    type: number;
/** Width of the window in units of time for storing single, forward filtering */
    window: number;
/** *DATABASE_BINARY_XTFILE card */
    xtfile: object;
}

declare class DeformableToRigid {
/**
 * Associates a comment with a deformable to rigid.
 * @param Comment Comment that will be attached to the deformable to rigid
 */
    AssociateComment(Comment: Comment): void;

/**
 * Blanks the deformable to rigid
 */
    Blank(): void;

/**
 * Blanks all of the deformable to rigids in the model.
 * @param Model Model that all deformable to rigids will be blanked in
 * @param redraw If model should be redrawn or not. If omitted redraw is false. If you want to do several (un)blanks and only redraw after the last one then use false for all redraws apart from the last one. Alternatively you can redraw using View.Redraw().
 */
    static BlankAll(Model: Model, redraw?: boolean): void;

/**
 * Blanks all of the flagged deformable to rigids in the model.
 * @param Model Model that all the flagged deformable to rigids will be blanked in
 * @param flag Flag set on the deformable to rigids that you want to blank
 * @param redraw If model should be redrawn or not. If omitted redraw is false. If you want to do several (un)blanks and only redraw after the last one then use false for all redraws apart from the last one. Alternatively you can redraw using View.Redraw().
 */
    static BlankFlagged(Model: Model, flag: number, redraw?: boolean): void;

/**
 * Checks if the deformable to rigid is blanked or not.
 * @returns true if blanked, false if not.
 */
    Blanked(): boolean;

/**
 * Starts an edit panel in Browse mode.
 * @param modal If this window is modal (blocks the user from doing anything else in PRIMER until this window is dismissed). If omitted the window will be modal.
 */
    Browse(modal?: boolean): void;

/**
 * Clears a flag on the deformable to rigid.
 * @param flag Flag to clear on the deformable to rigid
 */
    ClearFlag(flag: number): void;

/**
 * Copies the deformable to rigid. The target include of the copied deformable to rigid can be set using Options.copy_target_include.
 * @param range If you want to keep the copied item in the range specified for the current include. Default value is false. To set current include, use Include.MakeCurrentLayer().
 * @returns DeformableToRigid object
 */
    Copy(range?: boolean): DeformableToRigid;

/**
 * Starts an interactive editing panel to create a deformable to rigid
 * @param Model Model that the deformable to rigid will be created in.
 * @param modal If this window is modal (blocks the user from doing anything else in PRIMER until this window is dismissed). If omitted the window will be modal.
 * @returns DeformableToRigid object (or null if not made).
 */
    static Create(Model: Model, modal?: boolean): DeformableToRigid;

/**
 * Detaches a comment from a deformable to rigid.
 * @param Comment Comment that will be detached from the deformable to rigid
 */
    DetachComment(Comment: Comment): void;

/**
 * Starts an interactive editing panel.
 * @param modal If this window is modal (blocks the user from doing anything else in PRIMER until this window is dismissed). If omitted the window will be modal.
 */
    Edit(modal?: boolean): void;

/**
 * Adds an error for deformable to rigid. For more details on checking see the Check class.
 * @param message The error message to give
 * @param details An optional detailed error message
 */
    Error(message: string, details?: string): void;

/**
 * Returns the first deformable to rigid in the model.
 * @param Model Model to get first deformable to rigid in
 * @returns DeformableToRigid object (or null if there are no deformable to rigids in the model).
 */
    static First(Model: Model): DeformableToRigid;

/**
 * Returns the first free deformable to rigid label in the model. Also see DeformableToRigid.LastFreeLabel(), DeformableToRigid.NextFreeLabel() and Model.FirstFreeItemLabel().
 * @param Model Model to get first free deformable to rigid label in
 * @param layer Include file (0 for the main file) to search for labels in (Equivalent to First free in layer in editing panels). If omitted the whole model will be used (Equivalent to First free in editing panels).
 * @returns DeformableToRigid label.
 */
    static FirstFreeLabel(Model: Model, layer?: number): number;

/**
 * Flags all of the deformable to rigids in the model with a defined flag.
 * @param Model Model that all deformable to rigids will be flagged in
 * @param flag Flag to set on the deformable to rigids
 */
    static FlagAll(Model: Model, flag: number): void;

/**
 * Checks if the deformable to rigid is flagged or not.
 * @param flag Flag to test on the deformable to rigid
 * @returns true if flagged, false if not.
 */
    Flagged(flag: number): boolean;

/**
 * Calls a function for each deformable to rigid in the model. Note that ForEach has been designed to make looping over deformable to rigids as fast as possible and so has some limitations. Firstly, a single temporary DeformableToRigid object is created and on each function call it is updated with the current deformable to rigid data. This means that you should not try to store the DeformableToRigid object for later use (e.g. in an array) as it is temporary. Secondly, you cannot create new deformable to rigids inside a ForEach loop. 
 * @param Model Model that all deformable to rigids are in
 * @param func Function to call for each deformable to rigid
 * @param extra An optional extra object/array/string etc that will appended to arguments when calling the function
 */
    static ForEach(Model: Model, func: () => void, extra?: any): void;

/**
 * Returns an array of DeformableToRigid objects or properties for all of the deformable to rigids in a model in PRIMER. If the optional property argument is not given then an array of DeformableToRigid objects is returned. If the property argument is given, that property value for each deformable to rigid is returned in the array instead of a DeformableToRigid object
 * @param Model Model to get deformable to rigids from
 * @param property Name for property to get for all deformable to rigids in the model
 * @returns Array of DeformableToRigid objects or properties
 */
    static GetAll(Model: Model, property?: string): DeformableToRigid[];

/**
 * Extracts the comments associated to a deformable to rigid.
 * @returns Array of Comment objects (or null if there are no comments associated to the node).
 */
    GetComments(): Comment[];

/**
 * Returns the D2R or R2D cards for *DEFORMABLE_TO_RIGID_AUTOMATC.
 * @param ctype The card type you want the data for. Can be D2R or R2D.
 * @param index The card index you want the data for. Note that card indices start at 0, not 1.
 * @returns An array of numbers containing the 2 or 3 member (depending on Card type): Part or Part Set ID, LRB Part ID (only for card type D2R), and part type (PTYPE - Can be DeformableToRigid.PART or DeformableToRigid.PSET).
 */
    GetDefToRegAutoCard(ctype: number, index: number): number[];

/**
 * Returns an array of DeformableToRigid objects for all of the flagged deformable to rigids in a model in PRIMER If the optional property argument is not given then an array of DeformableToRigid objects is returned. If the property argument is given, then that property value for each deformable to rigid is returned in the array instead of a DeformableToRigid object
 * @param Model Model to get deformable to rigids from
 * @param flag Flag set on the deformable to rigids that you want to retrieve
 * @param property Name for property to get for all flagged deformable to rigids in the model
 * @returns Array of DeformableToRigid objects or properties
 */
    static GetFlagged(Model: Model, flag: number, property?: string): DeformableToRigid[];

/**
 * Returns the DeformableToRigid object for a deformable to rigid ID.
 * @param Model Model to find the deformable to rigid in
 * @param number number of the deformable to rigid you want the DeformableToRigid object for
 * @returns DeformableToRigid object (or null if deformable to rigid does not exist).
 */
    static GetFromID(Model: Model, number: number): DeformableToRigid;

/**
 * Checks if a DeformableToRigid property is a parameter or not. Note that object properties that are parameters are normally returned as the integer or float parameter values as that is virtually always what the user would want. For this function to work the JavaScript interpreter must use the parameter name instead of the value. This can be done by setting the Options.property_parameter_names option to true before calling the function and then resetting it to false afterwards.. This behaviour can also temporarily be switched by using the DeformableToRigid.ViewParameters() method and 'method chaining' (see the examples below).
 * @param prop deformable to rigid property to get parameter for
 * @returns Parameter object if property is a parameter, null if not.
 */
    GetParameter(prop: string): Parameter;

/**
 * Returns the keyword for this DeformableToRigid (*DEFORMABLE_TO_RIGID_xxxx) Note that a carriage return is not added. See also DeformableToRigid.KeywordCards()
 * @returns string containing the keyword.
 */
    Keyword(): string;

/**
 * Returns the keyword cards for the DeformableToRigid. Note that a carriage return is not added. See also DeformableToRigid.Keyword()
 * @returns string containing the cards.
 */
    KeywordCards(): string;

/**
 * Returns the last deformable to rigid in the model.
 * @param Model Model to get last deformable to rigid in
 * @returns DeformableToRigid object (or null if there are no deformable to rigids in the model).
 */
    static Last(Model: Model): DeformableToRigid;

/**
 * Returns the last free deformable to rigid label in the model. Also see DeformableToRigid.FirstFreeLabel(), DeformableToRigid.NextFreeLabel() and see Model.LastFreeItemLabel()
 * @param Model Model to get last free deformable to rigid label in
 * @param layer Include file (0 for the main file) to search for labels in (Equivalent to Highest free in layer in editing panels). If omitted the whole model will be used.
 * @returns DeformableToRigid label.
 */
    static LastFreeLabel(Model: Model, layer?: number): number;

/**
 * Returns the next deformable to rigid in the model.
 * @returns DeformableToRigid object (or null if there are no more deformable to rigids in the model).
 */
    Next(): DeformableToRigid;

/**
 * Returns the next free (highest+1) deformable to rigid label in the model. Also see DeformableToRigid.FirstFreeLabel(), DeformableToRigid.LastFreeLabel() and Model.NextFreeItemLabel()
 * @param Model Model to get next free deformable to rigid label in
 * @param layer Include file (0 for the main file) to search for labels in (Equivalent to Highest+1 in layer in editing panels). If omitted the whole model will be used (Equivalent to Highest+1 in editing panels).
 * @returns DeformableToRigid label.
 */
    static NextFreeLabel(Model: Model, layer?: number): number;

/**
 * Allows the user to pick a deformable to rigid.
 * @param prompt Text to display as a prompt to the user
 * @param limit If the argument is a Model then only deformable to rigids from that model can be picked. If the argument is a Flag then only deformable to rigids that are flagged with limit can be selected. If omitted, or null, any deformable to rigids from any model can be selected. from any model.
 * @param modal If picking is modal (blocks the user from doing anything else in PRIMER until this window is dismissed). If omitted the pick will be modal.
 * @param button_text By default the window with the prompt will have a button labelled 'Cancel' which if pressed will cancel the pick and return null. If you want to change the text on the button use this argument. If omitted 'Cancel' will be used.
 * @returns DeformableToRigid object (or null if not picked)
 */
    static Pick(prompt: string, limit?: Model | number, modal?: boolean, button_text?: string): DeformableToRigid;

/**
 * Returns the previous deformable to rigid in the model.
 * @returns DeformableToRigid object (or null if there are no more deformable to rigids in the model).
 */
    Previous(): DeformableToRigid;

/**
 * Removes the D2R or R2D cards for *DEFORMABLE_TO_RIGID_AUTOMATC.
 * @param ctype The card type you want removed. Can be D2R or R2D.
 * @param index The card index you want removed. Note that card indices start at 0, not 1.
 */
    RemoveDefToRegAutoCard(ctype: number, index: number): void;

/**
 * Renumbers all of the deformable to rigids in the model.
 * @param Model Model that all deformable to rigids will be renumbered in
 * @param start Start point for renumbering
 */
    static RenumberAll(Model: Model, start: number): void;

/**
 * Renumbers all of the flagged deformable to rigids in the model.
 * @param Model Model that all the flagged deformable to rigids will be renumbered in
 * @param flag Flag set on the deformable to rigids that you want to renumber
 * @param start Start point for renumbering
 */
    static RenumberFlagged(Model: Model, flag: number, start: number): void;

/**
 * Allows the user to select deformable to rigids using standard PRIMER object menus.
 * @param flag Flag to use when selecting deformable to rigids
 * @param prompt Text to display as a prompt to the user
 * @param limit If the argument is a Model then only deformable to rigids from that model can be selected. If the argument is a Flag then only deformable to rigids that are flagged with limit can be selected (limit should be different to flag). If omitted, or null, any deformable to rigids can be selected. from any model.
 * @param modal If selection is modal (blocks the user from doing anything else in PRIMER until this window is dismissed). If omitted the selection will be modal.
 * @returns Number of deformable to rigids selected or null if menu cancelled
 */
    static Select(flag: number, prompt: string, limit?: Model | number, modal?: boolean): number;

/**
 * Sets the D2r or R2D card data f*DEFORMABLE_TO_RIGID_AUTOMATIC.
 * @param ctype The card type you want to set. Can be D2R or R2D.
 * @param index The D2R or R2D card index you want to set. Note that cards start at 0, not 1.
 * @param ptype Part type (PTYPE). Can be DeformableToRigid.PART or DeformableToRigid.PSET.
 * @param pid Part or Part Set ID.
 * @param lrb LRB Part ID (only for card type D2R)
 */
    SetDefToRegAutoCard(ctype: number, index: number, ptype: number, pid: number, lrb?: number): void;

/**
 * Sets a flag on the deformable to rigid.
 * @param flag Flag to set on the deformable to rigid
 */
    SetFlag(flag: number): void;

/**
 * Sketches the deformable to rigid. The deformable to rigid will be sketched until you either call DeformableToRigid.Unsketch(), DeformableToRigid.UnsketchAll(), Model.UnsketchAll(), or delete the model
 * @param redraw If model should be redrawn or not after the deformable to rigid is sketched. If omitted redraw is true. If you want to sketch several deformable to rigids and only redraw after the last one then use false for redraw and call View.Redraw().
 */
    Sketch(redraw?: boolean): void;

/**
 * Sketches all of the flagged deformable to rigids in the model. The deformable to rigids will be sketched until you either call DeformableToRigid.Unsketch(), DeformableToRigid.UnsketchFlagged(), Model.UnsketchAll(), or delete the model
 * @param Model Model that all the flagged deformable to rigids will be sketched in
 * @param flag Flag set on the deformable to rigids that you want to sketch
 * @param redraw If model should be redrawn or not after the deformable to rigids are sketched. If omitted redraw is true. If you want to sketch flagged deformable to rigids several times and only redraw after the last one then use false for redraw and call View.Redraw().
 */
    static SketchFlagged(Model: Model, flag: number, redraw?: boolean): void;

/**
 * Returns the total number of deformable to rigids in the model.
 * @param Model Model to get total for
 * @param exists true if only existing deformable to rigids should be counted. If false or omitted referenced but undefined deformable to rigids will also be included in the total.
 * @returns number of deformable to rigids
 */
    static Total(Model: Model, exists?: boolean): number;

/**
 * Unblanks the deformable to rigid
 */
    Unblank(): void;

/**
 * Unblanks all of the deformable to rigids in the model.
 * @param Model Model that all deformable to rigids will be unblanked in
 * @param redraw If model should be redrawn or not. If omitted redraw is false. If you want to do several (un)blanks and only redraw after the last one then use false for all redraws apart from the last one. Alternatively you can redraw using View.Redraw().
 */
    static UnblankAll(Model: Model, redraw?: boolean): void;

/**
 * Unblanks all of the flagged deformable to rigids in the model.
 * @param Model Model that the flagged deformable to rigids will be unblanked in
 * @param flag Flag set on the deformable to rigids that you want to unblank
 * @param redraw If model should be redrawn or not. If omitted redraw is false. If you want to do several (un)blanks and only redraw after the last one then use false for all redraws apart from the last one. Alternatively you can redraw using View.Redraw().
 */
    static UnblankFlagged(Model: Model, flag: number, redraw?: boolean): void;

/**
 * Unsets a defined flag on all of the deformable to rigids in the model.
 * @param Model Model that the defined flag for all deformable to rigids will be unset in
 * @param flag Flag to unset on the deformable to rigids
 */
    static UnflagAll(Model: Model, flag: number): void;

/**
 * Unsketches the deformable to rigid.
 * @param redraw If model should be redrawn or not after the deformable to rigid is unsketched. If omitted redraw is true. If you want to unsketch several deformable to rigids and only redraw after the last one then use false for redraw and call View.Redraw().
 */
    Unsketch(redraw?: boolean): void;

/**
 * Unsketches all deformable to rigids.
 * @param Model Model that all deformable to rigids will be unblanked in
 * @param redraw If model should be redrawn or not after the deformable to rigids are unsketched. If omitted redraw is true. If you want to unsketch several things and only redraw after the last one then use false for redraw and call View.Redraw().
 */
    static UnsketchAll(Model: Model, redraw?: boolean): void;

/**
 * Unsketches all flagged deformable to rigids in the model.
 * @param Model Model that all deformable to rigids will be unsketched in
 * @param flag Flag set on the deformable to rigids that you want to unsketch
 * @param redraw If model should be redrawn or not after the deformable to rigids are unsketched. If omitted redraw is true. If you want to unsketch several things and only redraw after the last one then use false for redraw and call View.Redraw().
 */
    static UnsketchFlagged(Model: Model, flag: number, redraw?: boolean): void;

/**
 * Object properties that are parameters are normally returned as the integer or float parameter values as that is virtually always what the user would want. This function temporarily changes the behaviour so that if a property is a parameter the parameter name is returned instead. This can be used with 'method chaining' (see the example below) to make sure a property argument is correct.
 * @returns DeformableToRigid object.
 */
    ViewParameters(): DeformableToRigid;

/**
 * Adds a warning for deformable to rigid. For more details on checking see the Check class.
 * @param message The warning message to give
 * @param details An optional detailed warning message
 */
    Warning(message: string, details?: string): void;

/**
 * Returns the cross references for this deformable to rigid.
 * @returns Xrefs object.
 */
    Xrefs(): Xrefs;

/**
 * Create a new DeformableToRigid object.
 * @param Model Model that deformable to rigid will be created in
 * @param Type Specify the type of DeformableToRigid (Can be DeformableToRigid.SIMPLE or DeformableToRigid.AUTOMATIC or DeformableToRigid.INERTIA )
 * @param pid Part  or Part set  ID which is switched to a rigid material. Depends on value of ptype. Used only for DeformableToRigid.SIMPLE or DeformableToRigid.INERTIA.
 * @param lrb Part ID of the lead rigid body to which the part is merged. Used only for DeformableToRigid.SIMPLE.
 * @param ptype Type of PID. Valid values are: DeformableToRigid.PART or DeformableToRigid.PSET. Used only for DeformableToRigid.SIMPLE.
 * @returns DeformableToRigid object
 */
    constructor(Model: Model, Type: number, pid?: number, lrb?: number, ptype?: number);

/**
 * Creates a string containing the DeformableToRigid data in keyword format. Note that this contains the keyword header and the keyword cards. See also DeformableToRigid.Keyword() and DeformableToRigid.KeywordCards().
 * @returns string
 */
    toString(): string;

/** Activation switch code. (Valid values: 0-5). Used only for DeformableToRigid.AUTOMATIC. */
    code: number;
/** Number of deformable parts to be switched to rigid plus number of rigid parts for which new lead/constrained rigid body combinations will be defined. Used only for DeformableToRigid.AUTOMATIC. */
    d2r: number;
/** Maximum permitted time step size after switch. Used only for DeformableToRigid.AUTOMATIC. */
    dtmax: number;
/** Rigid wall/contact surface number for switch codes 1, 2, 3, 4. Used only for DeformableToRigid.AUTOMATIC. */
    entno: number;
/** true if deformable to rigid exists, false if referred to but not defined. */
    readonly exists: boolean;
/** The Include file number that the deformable to rigid is in. */
    include: number;
/** The xx component of inertia tensor. Used only for DeformableToRigid.INERTIA. */
    ixx: number;
/** The xx component of inertia tensor. Used only for DeformableToRigid.INERTIA. */
    ixx_1: number;
/** The xy component of inertia tensor. Used only for DeformableToRigid.INERTIA. */
    ixy: number;
/** The xz component of inertia tensor. Used only for DeformableToRigid.INERTIA. */
    ixz: number;
/** The yz component of inertia tensor. Used only for DeformableToRigid.INERTIA. */
    iyz: number;
/** The zz component of inertia tensor. Used only for DeformableToRigid.INERTIA. */
    izz: number;
/** Part ID of the lead rigid body to which the part is merged. Used only for DeformableToRigid.SIMPLE. */
    lrb: number;
/** The Model number that the deformable to rigid is in. */
    readonly model: number;
/** Nodal constraint body flag. (Valid values : 0, 1, 2). Used only for DeformableToRigid.AUTOMATIC. */
    ncsf: number;
/** Nodal rigid body flag. (Valid values : 0, 1, 2). Used only for DeformableToRigid.AUTOMATIC. */
    nrbf: number;
/** Optional contact thickness for switch to deformable. Used only for DeformableToRigid.AUTOMATIC. */
    offset: number;
/** Define a pair of related switches. (Valid values : -1, 0, 1). Used only for DeformableToRigid.AUTOMATIC. */
    paired: number;
/** Part or Part set  ID which is switched to a rigid material. Depends on value of ptype. Used only for DeformableToRigid.SIMPLE or DeformableToRigid.INERTIA. */
    pid: number;
/** Type of PID. Valid values are: DeformableToRigid.PART or DeformableToRigid.PSET. Used only for DeformableToRigid.SIMPLE. */
    ptype: number;
/** Number of rigid parts to be switched to deformable. Used only for DeformableToRigid.AUTOMATIC. */
    r2d: number;
/** Related switch set. Used only for DeformableToRigid.AUTOMATIC. */
    relsw: number;
/** Flag to delete or activate rigid walls. (Valid values : 0, 1, 2). Used only for DeformableToRigid.AUTOMATIC. */
    rwf: number;
/** Set number for this automatic switch set. Used only for DeformableToRigid.AUTOMATIC. */
    readonly swset: number;
/** Switch will not take place before this time. Used only for DeformableToRigid.AUTOMATIC. */
    time1: number;
/** Switch will not take place after this time. Used only for DeformableToRigid.AUTOMATIC. */
    time2: number;
/** After this part switch has taken place, another automatic switch will not take place for the duration of the delay period. Used only for DeformableToRigid.AUTOMATIC. */
    time3: number;
/** Translational mass. Used only for DeformableToRigid.INERTIA. */
    tm: number;
/** Gives the type of DeformableToRigid Object. */
    readonly type: number;
/** x-coordinate of center of mass. Used only for DeformableToRigid.INERTIA. */
    xc: number;
/** y-coordinate of center of mass. Used only for DeformableToRigid.INERTIA. */
    yc: number;
/** z-coordinate of center of mass. Used only for DeformableToRigid.INERTIA. */
    zc: number;
/** *DEFORMABLE_TO_RIGID_AUTOMATIC. */
    static AUTOMATIC: number;
/** Identifies that card is being written/retreived/removed as D2R card. Used in methods GetDefToRegAutoCard, SetDefToRegAutoCard and RemoveDefToRegAutoCard. Used only for DeformableToRigid.AUTOMATIC. */
    static D2R: number;
/** *DEFORMABLE_TO_RIGID_INERTIA. */
    static INERTIA: number;
/** Identifies the PID type as Part. Used for field ptype. Used only for DeformableToRigid.SIMPLE or DeformableToRigid.INERTIA. */
    static PART: number;
/** Identifies the PID type as Part Set. Used for field ptype. Used only for DeformableToRigid.SIMPLE or DeformableToRigid.INERTIA. */
    static PSET: number;
/** Identifies that card is being written/retreived/removed as R2D card. Used in methods GetDefToRegAutoCard, SetDefToRegAutoCard and RemoveDefToRegAutoCard. Used only for DeformableToRigid.AUTOMATIC. */
    static R2D: number;
/** *DEFORMABLE_TO_RIGID. */
    static SIMPLE: number;
}

declare class Discrete {
/**
 * Associates a comment with a discrete.
 * @param Comment Comment that will be attached to the discrete
 */
    AssociateComment(Comment: Comment): void;

/**
 * Blanks the discrete
 */
    Blank(): void;

/**
 * Blanks all of the discretes in the model.
 * @param Model Model that all discretes will be blanked in
 * @param redraw If model should be redrawn or not. If omitted redraw is false. If you want to do several (un)blanks and only redraw after the last one then use false for all redraws apart from the last one. Alternatively you can redraw using View.Redraw().
 */
    static BlankAll(Model: Model, redraw?: boolean): void;

/**
 * Blanks all of the flagged discretes in the model.
 * @param Model Model that all the flagged discretes will be blanked in
 * @param flag Flag set on the discretes that you want to blank
 * @param redraw If model should be redrawn or not. If omitted redraw is false. If you want to do several (un)blanks and only redraw after the last one then use false for all redraws apart from the last one. Alternatively you can redraw using View.Redraw().
 */
    static BlankFlagged(Model: Model, flag: number, redraw?: boolean): void;

/**
 * Checks if the discrete is blanked or not.
 * @returns true if blanked, false if not.
 */
    Blanked(): boolean;

/**
 * Starts an edit panel in Browse mode.
 * @param modal If this window is modal (blocks the user from doing anything else in PRIMER until this window is dismissed). If omitted the window will be modal.
 */
    Browse(modal?: boolean): void;

/**
 * Clears a flag on the discrete.
 * @param flag Flag to clear on the discrete
 */
    ClearFlag(flag: number): void;

/**
 * Copies the discrete. The target include of the copied discrete can be set using Options.copy_target_include.
 * @param range If you want to keep the copied item in the range specified for the current include. Default value is false. To set current include, use Include.MakeCurrentLayer().
 * @returns Discrete object
 */
    Copy(range?: boolean): Discrete;

/**
 * Starts an interactive editing panel to create a discrete
 * @param Model Model that the discrete will be created in.
 * @param modal If this window is modal (blocks the user from doing anything else in PRIMER until this window is dismissed). If omitted the window will be modal.
 * @returns Discrete object (or null if not made).
 */
    static Create(Model: Model, modal?: boolean): Discrete;

/**
 * Detaches a comment from a discrete.
 * @param Comment Comment that will be detached from the discrete
 */
    DetachComment(Comment: Comment): void;

/**
 * Starts an interactive editing panel.
 * @param modal If this window is modal (blocks the user from doing anything else in PRIMER until this window is dismissed). If omitted the window will be modal.
 */
    Edit(modal?: boolean): void;

/**
 * Adds an error for discrete. For more details on checking see the Check class.
 * @param message The error message to give
 * @param details An optional detailed error message
 */
    Error(message: string, details?: string): void;

/**
 * Extracts the actual colour used for discrete. By default in PRIMER many entities such as elements get their colour automatically from the part that they are in. PRIMER cycles through 13 default colours based on the label of the entity. In this case the discrete colour property will return the value Colour.PART instead of the actual colour. This method will return the actual colour which is used for drawing the discrete.
 * @returns colour value (integer)
 */
    ExtractColour(): number;

/**
 * Returns the first discrete in the model.
 * @param Model Model to get first discrete in
 * @returns Discrete object (or null if there are no discretes in the model).
 */
    static First(Model: Model): Discrete;

/**
 * Returns the first free discrete label in the model. Also see Discrete.LastFreeLabel(), Discrete.NextFreeLabel() and Model.FirstFreeItemLabel().
 * @param Model Model to get first free discrete label in
 * @param layer Include file (0 for the main file) to search for labels in (Equivalent to First free in layer in editing panels). If omitted the whole model will be used (Equivalent to First free in editing panels).
 * @returns Discrete label.
 */
    static FirstFreeLabel(Model: Model, layer?: number): number;

/**
 * Flags all of the discretes in the model with a defined flag.
 * @param Model Model that all discretes will be flagged in
 * @param flag Flag to set on the discretes
 */
    static FlagAll(Model: Model, flag: number): void;

/**
 * Checks if the discrete is flagged or not.
 * @param flag Flag to test on the discrete
 * @returns true if flagged, false if not.
 */
    Flagged(flag: number): boolean;

/**
 * Calls a function for each discrete in the model. Note that ForEach has been designed to make looping over discretes as fast as possible and so has some limitations. Firstly, a single temporary Discrete object is created and on each function call it is updated with the current discrete data. This means that you should not try to store the Discrete object for later use (e.g. in an array) as it is temporary. Secondly, you cannot create new discretes inside a ForEach loop. 
 * @param Model Model that all discretes are in
 * @param func Function to call for each discrete
 * @param extra An optional extra object/array/string etc that will appended to arguments when calling the function
 */
    static ForEach(Model: Model, func: () => void, extra?: any): void;

/**
 * Returns an array of Discrete objects or properties for all of the discretes in a model in PRIMER. If the optional property argument is not given then an array of Discrete objects is returned. If the property argument is given, that property value for each discrete is returned in the array instead of a Discrete object
 * @param Model Model to get discretes from
 * @param property Name for property to get for all discretes in the model
 * @returns Array of Discrete objects or properties
 */
    static GetAll(Model: Model, property?: string): Discrete[];

/**
 * Extracts the comments associated to a discrete.
 * @returns Array of Comment objects (or null if there are no comments associated to the node).
 */
    GetComments(): Comment[];

/**
 * Returns an array of Discrete objects for all of the flagged discretes in a model in PRIMER If the optional property argument is not given then an array of Discrete objects is returned. If the property argument is given, then that property value for each discrete is returned in the array instead of a Discrete object
 * @param Model Model to get discretes from
 * @param flag Flag set on the discretes that you want to retrieve
 * @param property Name for property to get for all flagged discretes in the model
 * @returns Array of Discrete objects or properties
 */
    static GetFlagged(Model: Model, flag: number, property?: string): Discrete[];

/**
 * Returns the Discrete object for a discrete ID.
 * @param Model Model to find the discrete in
 * @param number number of the discrete you want the Discrete object for
 * @returns Discrete object (or null if discrete does not exist).
 */
    static GetFromID(Model: Model, number: number): Discrete;

/**
 * Checks if a Discrete property is a parameter or not. Note that object properties that are parameters are normally returned as the integer or float parameter values as that is virtually always what the user would want. For this function to work the JavaScript interpreter must use the parameter name instead of the value. This can be done by setting the Options.property_parameter_names option to true before calling the function and then resetting it to false afterwards.. This behaviour can also temporarily be switched by using the Discrete.ViewParameters() method and 'method chaining' (see the examples below).
 * @param prop discrete property to get parameter for
 * @returns Parameter object if property is a parameter, null if not.
 */
    GetParameter(prop: string): Parameter;

/**
 * Returns the keyword for this discrete (*ELEMENT_DISCRETE). Note that a carriage return is not added. See also Discrete.KeywordCards()
 * @returns string containing the keyword.
 */
    Keyword(): string;

/**
 * Returns the keyword cards for the discrete. Note that a carriage return is not added. See also Discrete.Keyword()
 * @returns string containing the cards.
 */
    KeywordCards(): string;

/**
 * Returns the last discrete in the model.
 * @param Model Model to get last discrete in
 * @returns Discrete object (or null if there are no discretes in the model).
 */
    static Last(Model: Model): Discrete;

/**
 * Returns the last free discrete label in the model. Also see Discrete.FirstFreeLabel(), Discrete.NextFreeLabel() and see Model.LastFreeItemLabel()
 * @param Model Model to get last free discrete label in
 * @param layer Include file (0 for the main file) to search for labels in (Equivalent to Highest free in layer in editing panels). If omitted the whole model will be used.
 * @returns Discrete label.
 */
    static LastFreeLabel(Model: Model, layer?: number): number;

/**
 * Returns the next discrete in the model.
 * @returns Discrete object (or null if there are no more discretes in the model).
 */
    Next(): Discrete;

/**
 * Returns the next free (highest+1) discrete label in the model. Also see Discrete.FirstFreeLabel(), Discrete.LastFreeLabel() and Model.NextFreeItemLabel()
 * @param Model Model to get next free discrete label in
 * @param layer Include file (0 for the main file) to search for labels in (Equivalent to Highest+1 in layer in editing panels). If omitted the whole model will be used (Equivalent to Highest+1 in editing panels).
 * @returns Discrete label.
 */
    static NextFreeLabel(Model: Model, layer?: number): number;

/**
 * Allows the user to pick a discrete.
 * @param prompt Text to display as a prompt to the user
 * @param limit If the argument is a Model then only discretes from that model can be picked. If the argument is a Flag then only discretes that are flagged with limit can be selected. If omitted, or null, any discretes from any model can be selected. from any model.
 * @param modal If picking is modal (blocks the user from doing anything else in PRIMER until this window is dismissed). If omitted the pick will be modal.
 * @param button_text By default the window with the prompt will have a button labelled 'Cancel' which if pressed will cancel the pick and return null. If you want to change the text on the button use this argument. If omitted 'Cancel' will be used.
 * @returns Discrete object (or null if not picked)
 */
    static Pick(prompt: string, limit?: Model | number, modal?: boolean, button_text?: string): Discrete;

/**
 * Returns the previous discrete in the model.
 * @returns Discrete object (or null if there are no more discretes in the model).
 */
    Previous(): Discrete;

/**
 * Renumbers all of the discretes in the model.
 * @param Model Model that all discretes will be renumbered in
 * @param start Start point for renumbering
 */
    static RenumberAll(Model: Model, start: number): void;

/**
 * Renumbers all of the flagged discretes in the model.
 * @param Model Model that all the flagged discretes will be renumbered in
 * @param flag Flag set on the discretes that you want to renumber
 * @param start Start point for renumbering
 */
    static RenumberFlagged(Model: Model, flag: number, start: number): void;

/**
 * Allows the user to select discretes using standard PRIMER object menus.
 * @param flag Flag to use when selecting discretes
 * @param prompt Text to display as a prompt to the user
 * @param limit If the argument is a Model then only discretes from that model can be selected. If the argument is a Flag then only discretes that are flagged with limit can be selected (limit should be different to flag). If omitted, or null, any discretes can be selected. from any model.
 * @param modal If selection is modal (blocks the user from doing anything else in PRIMER until this window is dismissed). If omitted the selection will be modal.
 * @returns Number of discretes selected or null if menu cancelled
 */
    static Select(flag: number, prompt: string, limit?: Model | number, modal?: boolean): number;

/**
 * Sets a flag on the discrete.
 * @param flag Flag to set on the discrete
 */
    SetFlag(flag: number): void;

/**
 * Sketches the discrete. The discrete will be sketched until you either call Discrete.Unsketch(), Discrete.UnsketchAll(), Model.UnsketchAll(), or delete the model
 * @param redraw If model should be redrawn or not after the discrete is sketched. If omitted redraw is true. If you want to sketch several discretes and only redraw after the last one then use false for redraw and call View.Redraw().
 */
    Sketch(redraw?: boolean): void;

/**
 * Sketches all of the flagged discretes in the model. The discretes will be sketched until you either call Discrete.Unsketch(), Discrete.UnsketchFlagged(), Model.UnsketchAll(), or delete the model
 * @param Model Model that all the flagged discretes will be sketched in
 * @param flag Flag set on the discretes that you want to sketch
 * @param redraw If model should be redrawn or not after the discretes are sketched. If omitted redraw is true. If you want to sketch flagged discretes several times and only redraw after the last one then use false for redraw and call View.Redraw().
 */
    static SketchFlagged(Model: Model, flag: number, redraw?: boolean): void;

/**
 * Calculates the timestep for the discrete
 * @returns float
 */
    Timestep(): number;

/**
 * Returns the total number of discretes in the model.
 * @param Model Model to get total for
 * @param exists true if only existing discretes should be counted. If false or omitted referenced but undefined discretes will also be included in the total.
 * @returns number of discretes
 */
    static Total(Model: Model, exists?: boolean): number;

/**
 * Unblanks the discrete
 */
    Unblank(): void;

/**
 * Unblanks all of the discretes in the model.
 * @param Model Model that all discretes will be unblanked in
 * @param redraw If model should be redrawn or not. If omitted redraw is false. If you want to do several (un)blanks and only redraw after the last one then use false for all redraws apart from the last one. Alternatively you can redraw using View.Redraw().
 */
    static UnblankAll(Model: Model, redraw?: boolean): void;

/**
 * Unblanks all of the flagged discretes in the model.
 * @param Model Model that the flagged discretes will be unblanked in
 * @param flag Flag set on the discretes that you want to unblank
 * @param redraw If model should be redrawn or not. If omitted redraw is false. If you want to do several (un)blanks and only redraw after the last one then use false for all redraws apart from the last one. Alternatively you can redraw using View.Redraw().
 */
    static UnblankFlagged(Model: Model, flag: number, redraw?: boolean): void;

/**
 * Unsets a defined flag on all of the discretes in the model.
 * @param Model Model that the defined flag for all discretes will be unset in
 * @param flag Flag to unset on the discretes
 */
    static UnflagAll(Model: Model, flag: number): void;

/**
 * Unsketches the discrete.
 * @param redraw If model should be redrawn or not after the discrete is unsketched. If omitted redraw is true. If you want to unsketch several discretes and only redraw after the last one then use false for redraw and call View.Redraw().
 */
    Unsketch(redraw?: boolean): void;

/**
 * Unsketches all discretes.
 * @param Model Model that all discretes will be unblanked in
 * @param redraw If model should be redrawn or not after the discretes are unsketched. If omitted redraw is true. If you want to unsketch several things and only redraw after the last one then use false for redraw and call View.Redraw().
 */
    static UnsketchAll(Model: Model, redraw?: boolean): void;

/**
 * Unsketches all flagged discretes in the model.
 * @param Model Model that all discretes will be unsketched in
 * @param flag Flag set on the discretes that you want to unsketch
 * @param redraw If model should be redrawn or not after the discretes are unsketched. If omitted redraw is true. If you want to unsketch several things and only redraw after the last one then use false for redraw and call View.Redraw().
 */
    static UnsketchFlagged(Model: Model, flag: number, redraw?: boolean): void;

/**
 * Object properties that are parameters are normally returned as the integer or float parameter values as that is virtually always what the user would want. This function temporarily changes the behaviour so that if a property is a parameter the parameter name is returned instead. This can be used with 'method chaining' (see the example below) to make sure a property argument is correct.
 * @returns Discrete object.
 */
    ViewParameters(): Discrete;

/**
 * Adds a warning for discrete. For more details on checking see the Check class.
 * @param message The warning message to give
 * @param details An optional detailed warning message
 */
    Warning(message: string, details?: string): void;

/**
 * Returns the cross references for this discrete.
 * @returns Xrefs object.
 */
    Xrefs(): Xrefs;

/**
 * Create a new Discrete object.
 * @param Model Model that discrete will be created in
 * @param eid Discrete number
 * @param pid Part number
 * @param n1 Node number 1
 * @param n2 Node number 2
 * @param vid Orientation vector
 * @param s Scale factor on forces
 * @param pf Print flag. Set to write forces to the DEFORC file
 * @param offset Initial offset
 * @returns Discrete object
 */
    constructor(Model: Model, eid: number, pid: number, n1: number, n2: number, vid?: number, s?: number, pf?: number, offset?: number);

/**
 * Creates a string containing the discrete data in keyword format. Note that this contains the keyword header and the keyword cards. See also Discrete.Keyword() and Discrete.KeywordCards().
 * @returns string
 */
    toString(): string;

/** The colour of the discrete */
    colour: Colour;
/** Discrete number. Also see the label property which is an alternative name for this. */
    eid: number;
/** true if discrete exists, false if referred to but not defined. */
    readonly exists: boolean;
/** The Include file number that the discrete is in. */
    include: number;
/** Discrete number. Also see the eid property which is an alternative name for this. */
    label: number;
/** Loadcurve for offset vs time */
    lcid: number;
/** Loadcurve for offset vs time during dynamic relaxation */
    lciddr: number;
/** If LCO option is set. Can be true or false */
    lco: boolean;
/** The Model number that the discrete is in. */
    readonly model: number;
/** Node number 1 */
    n1: number;
/** Node number 2 */
    n2: number;
/** Initial offset */
    offset: number;
/** Print flag. Set to write forces to the DEFORC file */
    pf: number;
/** Part number */
    pid: number;
/** Scale factor on forces */
    s: number;
/** The transparency of the discrete (0-100) 0% is opaque, 100% is transparent. */
    transparency: number;
/** Orientation vector */
    vid: number;
}

declare class DiscreteSphere {
/**
 * Associates a comment with a discrete sphere.
 * @param Comment Comment that will be attached to the discrete sphere
 */
    AssociateComment(Comment: Comment): void;

/**
 * Blanks the discrete sphere
 */
    Blank(): void;

/**
 * Blanks all of the discrete spheres in the model.
 * @param Model Model that all discrete spheres will be blanked in
 * @param redraw If model should be redrawn or not. If omitted redraw is false. If you want to do several (un)blanks and only redraw after the last one then use false for all redraws apart from the last one. Alternatively you can redraw using View.Redraw().
 */
    static BlankAll(Model: Model, redraw?: boolean): void;

/**
 * Blanks all of the flagged discrete spheres in the model.
 * @param Model Model that all the flagged discrete spheres will be blanked in
 * @param flag Flag set on the discrete spheres that you want to blank
 * @param redraw If model should be redrawn or not. If omitted redraw is false. If you want to do several (un)blanks and only redraw after the last one then use false for all redraws apart from the last one. Alternatively you can redraw using View.Redraw().
 */
    static BlankFlagged(Model: Model, flag: number, redraw?: boolean): void;

/**
 * Checks if the discrete sphere is blanked or not.
 * @returns true if blanked, false if not.
 */
    Blanked(): boolean;

/**
 * Starts an edit panel in Browse mode.
 * @param modal If this window is modal (blocks the user from doing anything else in PRIMER until this window is dismissed). If omitted the window will be modal.
 */
    Browse(modal?: boolean): void;

/**
 * Clears a flag on the discrete sphere.
 * @param flag Flag to clear on the discrete sphere
 */
    ClearFlag(flag: number): void;

/**
 * Copies the discrete sphere. The target include of the copied discrete sphere can be set using Options.copy_target_include.
 * @param range If you want to keep the copied item in the range specified for the current include. Default value is false. To set current include, use Include.MakeCurrentLayer().
 * @returns DiscreteSphere object
 */
    Copy(range?: boolean): DiscreteSphere;

/**
 * Starts an interactive editing panel to create a discrete sphere
 * @param Model Model that the discrete sphere will be created in.
 * @param modal If this window is modal (blocks the user from doing anything else in PRIMER until this window is dismissed). If omitted the window will be modal.
 * @returns DiscreteSphere object (or null if not made).
 */
    static Create(Model: Model, modal?: boolean): DiscreteSphere;

/**
 * Detaches a comment from a discrete sphere.
 * @param Comment Comment that will be detached from the discrete sphere
 */
    DetachComment(Comment: Comment): void;

/**
 * Starts an interactive editing panel.
 * @param modal If this window is modal (blocks the user from doing anything else in PRIMER until this window is dismissed). If omitted the window will be modal.
 */
    Edit(modal?: boolean): void;

/**
 * Adds an error for discrete sphere. For more details on checking see the Check class.
 * @param message The error message to give
 * @param details An optional detailed error message
 */
    Error(message: string, details?: string): void;

/**
 * Extracts the actual colour used for discrete sphere. By default in PRIMER many entities such as elements get their colour automatically from the part that they are in. PRIMER cycles through 13 default colours based on the label of the entity. In this case the discrete sphere colour property will return the value Colour.PART instead of the actual colour. This method will return the actual colour which is used for drawing the discrete sphere.
 * @returns colour value (integer)
 */
    ExtractColour(): number;

/**
 * Returns the first discrete sphere in the model.
 * @param Model Model to get first discrete sphere in
 * @returns DiscreteSphere object (or null if there are no discrete spheres in the model).
 */
    static First(Model: Model): DiscreteSphere;

/**
 * Returns the first free discrete sphere label in the model. Also see DiscreteSphere.LastFreeLabel(), DiscreteSphere.NextFreeLabel() and Model.FirstFreeItemLabel().
 * @param Model Model to get first free discrete sphere label in
 * @param layer Include file (0 for the main file) to search for labels in (Equivalent to First free in layer in editing panels). If omitted the whole model will be used (Equivalent to First free in editing panels).
 * @returns DiscreteSphere label.
 */
    static FirstFreeLabel(Model: Model, layer?: number): number;

/**
 * Flags all of the discrete spheres in the model with a defined flag.
 * @param Model Model that all discrete spheres will be flagged in
 * @param flag Flag to set on the discrete spheres
 */
    static FlagAll(Model: Model, flag: number): void;

/**
 * Checks if the discrete sphere is flagged or not.
 * @param flag Flag to test on the discrete sphere
 * @returns true if flagged, false if not.
 */
    Flagged(flag: number): boolean;

/**
 * Calls a function for each discrete sphere in the model. Note that ForEach has been designed to make looping over discrete spheres as fast as possible and so has some limitations. Firstly, a single temporary DiscreteSphere object is created and on each function call it is updated with the current discrete sphere data. This means that you should not try to store the DiscreteSphere object for later use (e.g. in an array) as it is temporary. Secondly, you cannot create new discrete spheres inside a ForEach loop. 
 * @param Model Model that all discrete spheres are in
 * @param func Function to call for each discrete sphere
 * @param extra An optional extra object/array/string etc that will appended to arguments when calling the function
 */
    static ForEach(Model: Model, func: () => void, extra?: any): void;

/**
 * Returns an array of DiscreteSphere objects or properties for all of the discrete spheres in a model in PRIMER. If the optional property argument is not given then an array of DiscreteSphere objects is returned. If the property argument is given, that property value for each discrete sphere is returned in the array instead of a DiscreteSphere object
 * @param Model Model to get discrete spheres from
 * @param property Name for property to get for all discrete spheres in the model
 * @returns Array of DiscreteSphere objects or properties
 */
    static GetAll(Model: Model, property?: string): DiscreteSphere[];

/**
 * Extracts the comments associated to a discrete sphere.
 * @returns Array of Comment objects (or null if there are no comments associated to the node).
 */
    GetComments(): Comment[];

/**
 * Returns an array of DiscreteSphere objects for all of the flagged discrete spheres in a model in PRIMER If the optional property argument is not given then an array of DiscreteSphere objects is returned. If the property argument is given, then that property value for each discrete sphere is returned in the array instead of a DiscreteSphere object
 * @param Model Model to get discrete spheres from
 * @param flag Flag set on the discrete spheres that you want to retrieve
 * @param property Name for property to get for all flagged discrete spheres in the model
 * @returns Array of DiscreteSphere objects or properties
 */
    static GetFlagged(Model: Model, flag: number, property?: string): DiscreteSphere[];

/**
 * Returns the DiscreteSphere object for a discrete sphere ID.
 * @param Model Model to find the discrete sphere in
 * @param number number of the discrete sphere you want the DiscreteSphere object for
 * @returns DiscreteSphere object (or null if discrete sphere does not exist).
 */
    static GetFromID(Model: Model, number: number): DiscreteSphere;

/**
 * Checks if a DiscreteSphere property is a parameter or not. Note that object properties that are parameters are normally returned as the integer or float parameter values as that is virtually always what the user would want. For this function to work the JavaScript interpreter must use the parameter name instead of the value. This can be done by setting the Options.property_parameter_names option to true before calling the function and then resetting it to false afterwards.. This behaviour can also temporarily be switched by using the DiscreteSphere.ViewParameters() method and 'method chaining' (see the examples below).
 * @param prop discrete sphere property to get parameter for
 * @returns Parameter object if property is a parameter, null if not.
 */
    GetParameter(prop: string): Parameter;

/**
 * Returns the keyword for this discrete sphere (*ELEMENT_DISCRETE_SPHERE or *ELEMENT_DISCRETE_SPHERE_VOLUME). Note that a carriage return is not added. See also DiscreteSphere.KeywordCards()
 * @returns string containing the keyword.
 */
    Keyword(): string;

/**
 * Returns the keyword cards for the discrete sphere. Note that a carriage return is not added. See also DiscreteSphere.Keyword()
 * @returns string containing the cards.
 */
    KeywordCards(): string;

/**
 * Returns the last discrete sphere in the model.
 * @param Model Model to get last discrete sphere in
 * @returns DiscreteSphere object (or null if there are no discrete spheres in the model).
 */
    static Last(Model: Model): DiscreteSphere;

/**
 * Returns the last free discrete sphere label in the model. Also see DiscreteSphere.FirstFreeLabel(), DiscreteSphere.NextFreeLabel() and see Model.LastFreeItemLabel()
 * @param Model Model to get last free discrete sphere label in
 * @param layer Include file (0 for the main file) to search for labels in (Equivalent to Highest free in layer in editing panels). If omitted the whole model will be used.
 * @returns DiscreteSphere label.
 */
    static LastFreeLabel(Model: Model, layer?: number): number;

/**
 * Returns the next discrete sphere in the model.
 * @returns DiscreteSphere object (or null if there are no more discrete spheres in the model).
 */
    Next(): DiscreteSphere;

/**
 * Returns the next free (highest+1) discrete sphere label in the model. Also see DiscreteSphere.FirstFreeLabel(), DiscreteSphere.LastFreeLabel() and Model.NextFreeItemLabel()
 * @param Model Model to get next free discrete sphere label in
 * @param layer Include file (0 for the main file) to search for labels in (Equivalent to Highest+1 in layer in editing panels). If omitted the whole model will be used (Equivalent to Highest+1 in editing panels).
 * @returns DiscreteSphere label.
 */
    static NextFreeLabel(Model: Model, layer?: number): number;

/**
 * Allows the user to pick a discrete sphere.
 * @param prompt Text to display as a prompt to the user
 * @param limit If the argument is a Model then only discrete spheres from that model can be picked. If the argument is a Flag then only discrete spheres that are flagged with limit can be selected. If omitted, or null, any discrete spheres from any model can be selected. from any model.
 * @param modal If picking is modal (blocks the user from doing anything else in PRIMER until this window is dismissed). If omitted the pick will be modal.
 * @param button_text By default the window with the prompt will have a button labelled 'Cancel' which if pressed will cancel the pick and return null. If you want to change the text on the button use this argument. If omitted 'Cancel' will be used.
 * @returns DiscreteSphere object (or null if not picked)
 */
    static Pick(prompt: string, limit?: Model | number, modal?: boolean, button_text?: string): DiscreteSphere;

/**
 * Returns the previous discrete sphere in the model.
 * @returns DiscreteSphere object (or null if there are no more discrete spheres in the model).
 */
    Previous(): DiscreteSphere;

/**
 * Allows the user to select discrete spheres using standard PRIMER object menus.
 * @param flag Flag to use when selecting discrete spheres
 * @param prompt Text to display as a prompt to the user
 * @param limit If the argument is a Model then only discrete spheres from that model can be selected. If the argument is a Flag then only discrete spheres that are flagged with limit can be selected (limit should be different to flag). If omitted, or null, any discrete spheres can be selected. from any model.
 * @param modal If selection is modal (blocks the user from doing anything else in PRIMER until this window is dismissed). If omitted the selection will be modal.
 * @returns Number of discrete spheres selected or null if menu cancelled
 */
    static Select(flag: number, prompt: string, limit?: Model | number, modal?: boolean): number;

/**
 * Sets a flag on the discrete sphere.
 * @param flag Flag to set on the discrete sphere
 */
    SetFlag(flag: number): void;

/**
 * Sketches the discrete sphere. The discrete sphere will be sketched until you either call DiscreteSphere.Unsketch(), DiscreteSphere.UnsketchAll(), Model.UnsketchAll(), or delete the model
 * @param redraw If model should be redrawn or not after the discrete sphere is sketched. If omitted redraw is true. If you want to sketch several discrete spheres and only redraw after the last one then use false for redraw and call View.Redraw().
 */
    Sketch(redraw?: boolean): void;

/**
 * Sketches all of the flagged discrete spheres in the model. The discrete spheres will be sketched until you either call DiscreteSphere.Unsketch(), DiscreteSphere.UnsketchFlagged(), Model.UnsketchAll(), or delete the model
 * @param Model Model that all the flagged discrete spheres will be sketched in
 * @param flag Flag set on the discrete spheres that you want to sketch
 * @param redraw If model should be redrawn or not after the discrete spheres are sketched. If omitted redraw is true. If you want to sketch flagged discrete spheres several times and only redraw after the last one then use false for redraw and call View.Redraw().
 */
    static SketchFlagged(Model: Model, flag: number, redraw?: boolean): void;

/**
 * Returns the total number of discrete spheres in the model.
 * @param Model Model to get total for
 * @param exists true if only existing discrete spheres should be counted. If false or omitted referenced but undefined discrete spheres will also be included in the total.
 * @returns number of discrete spheres
 */
    static Total(Model: Model, exists?: boolean): number;

/**
 * Unblanks the discrete sphere
 */
    Unblank(): void;

/**
 * Unblanks all of the discrete spheres in the model.
 * @param Model Model that all discrete spheres will be unblanked in
 * @param redraw If model should be redrawn or not. If omitted redraw is false. If you want to do several (un)blanks and only redraw after the last one then use false for all redraws apart from the last one. Alternatively you can redraw using View.Redraw().
 */
    static UnblankAll(Model: Model, redraw?: boolean): void;

/**
 * Unblanks all of the flagged discrete spheres in the model.
 * @param Model Model that the flagged discrete spheres will be unblanked in
 * @param flag Flag set on the discrete spheres that you want to unblank
 * @param redraw If model should be redrawn or not. If omitted redraw is false. If you want to do several (un)blanks and only redraw after the last one then use false for all redraws apart from the last one. Alternatively you can redraw using View.Redraw().
 */
    static UnblankFlagged(Model: Model, flag: number, redraw?: boolean): void;

/**
 * Unsets a defined flag on all of the discrete spheres in the model.
 * @param Model Model that the defined flag for all discrete spheres will be unset in
 * @param flag Flag to unset on the discrete spheres
 */
    static UnflagAll(Model: Model, flag: number): void;

/**
 * Unsketches the discrete sphere.
 * @param redraw If model should be redrawn or not after the discrete sphere is unsketched. If omitted redraw is true. If you want to unsketch several discrete spheres and only redraw after the last one then use false for redraw and call View.Redraw().
 */
    Unsketch(redraw?: boolean): void;

/**
 * Unsketches all discrete spheres.
 * @param Model Model that all discrete spheres will be unblanked in
 * @param redraw If model should be redrawn or not after the discrete spheres are unsketched. If omitted redraw is true. If you want to unsketch several things and only redraw after the last one then use false for redraw and call View.Redraw().
 */
    static UnsketchAll(Model: Model, redraw?: boolean): void;

/**
 * Unsketches all flagged discrete spheres in the model.
 * @param Model Model that all discrete spheres will be unsketched in
 * @param flag Flag set on the discrete spheres that you want to unsketch
 * @param redraw If model should be redrawn or not after the discrete spheres are unsketched. If omitted redraw is true. If you want to unsketch several things and only redraw after the last one then use false for redraw and call View.Redraw().
 */
    static UnsketchFlagged(Model: Model, flag: number, redraw?: boolean): void;

/**
 * Object properties that are parameters are normally returned as the integer or float parameter values as that is virtually always what the user would want. This function temporarily changes the behaviour so that if a property is a parameter the parameter name is returned instead. This can be used with 'method chaining' (see the example below) to make sure a property argument is correct.
 * @returns DiscreteSphere object.
 */
    ViewParameters(): DiscreteSphere;

/**
 * Adds a warning for discrete sphere. For more details on checking see the Check class.
 * @param message The warning message to give
 * @param details An optional detailed warning message
 */
    Warning(message: string, details?: string): void;

/**
 * Returns the cross references for this discrete sphere.
 * @returns Xrefs object.
 */
    Xrefs(): Xrefs;

/**
 * Create a new DiscreteSphere object.
 * @param Model Model that discrete sphere will be created in
 * @param nid Node ID and Element ID are the same for discrete spheres.
 * @param pid Part ID to which this element belongs.
 * @param mass Mass or volume value.
 * @param inertia Mass moment of inertia.
 * @param radius Particle radius.
 * @returns DiscreteSphere object
 */
    constructor(Model: Model, nid: number, pid: number, mass: number, inertia: number, radius: number);

/**
 * Creates a string containing the discrete sphere data in keyword format. Note that this contains the keyword header and the keyword cards. See also DiscreteSphere.Keyword() and DiscreteSphere.KeywordCards().
 * @returns string
 */
    toString(): string;

/** The colour of the discrete sphere */
    colour: Colour;
/** true if discrete sphere exists, false if referred to but not defined. */
    readonly exists: boolean;
/** Distribution of DES properties. */
    idist: number;
/** The Include file number that the discrete sphere is in. */
    include: number;
/** Mass moment of inertia. */
    inertia: number;
/** Standard deviation for mass moment of inertia or inertia per unit density(depending on whether the _VOLUME option is set), if idist is -1 or 1. Shape parameter for mass moment of inertia or inertia per unit density(depending on whether the _VOLUME option is set), if idist is -2 or 2. */
    isd_mi: number;
/** Mass or volume value (depending on whether the _VOLUME option is set). */
    mass: number;
/** The Model number that the discrete sphere is in. */
    readonly model: number;
/** Standard deviation for mass or volume(depending on whether the _VOLUME option is set), if idist is -1 or 1. Shape parameter for mass or volume(depending on whether the _VOLUME option is set), if idist is -2 or 2. */
    msd_mm: number;
/** Node ID. */
    nid: number;
/** Part ID to which this element belongs. */
    pid: number;
/** Particle radius. */
    radius: number;
/** Standard deviation for radius, if idist is -1 or 1. Shape parameter for radius, if idist is -2 or 2. */
    rsd_mr: number;
/** The transparency of the discrete sphere (0-100) 0% is opaque, 100% is transparent. */
    transparency: number;
/** Turns _VOLUME on or OFF. Note that this does NOT refer to the data field VOLUME. For the latter see the mass property. */
    volume: boolean;
}


/** Object returned by GetAssembly */
interface GetAssemblyReturn {
    /** Assembly label */
    label: number;
    /** Parent assembly label */
    parent: number;
    /** Assembly title */
    title: string;
}


/** Object returned by GetAssemblyChildInfo */
interface GetAssemblyChildInfoReturn {
    /** Degree of freedom codes */
    dof_code: number;
    /** Constrained joint stiffness label */
    joint_stiff: number;
    /** Label of the child assembly */
    label: number;
    /** Node A label */
    node_a: number;
    /** Node B label */
    node_b: number;
}


/** Object returned by GetAssemblyFromID */
interface GetAssemblyFromIDReturn {
    /** Total number of child assembly(s). */
    child: number;
    /** true if assembly exists, false if not defined. */
    exists: boolean;
    /** Assembly index */
    index: number;
    /** Assembly label */
    label: number;
    /** Parent assembly label */
    parent: number;
    /** x-angle of the assembly. */
    rx: number;
    /** y-angle of the assembly. */
    ry: number;
    /** z-angle of the assembly. */
    rz: number;
    /** Assembly title */
    title: string;
}


/** Object returned by GetPoint */
interface GetPointReturn {
    /** Assembly label */
    assembly: number;
    /** Coordinate system */
    csys: number;
    /** If point has been automatically created by PRIMER at the H-point */
    hpt: boolean;
    /** Point label */
    label: number;
    /** Node label (0 if coordinate) */
    node: number;
    /** Point restrained rotationally in X */
    rx: boolean;
    /** Point restrained rotationally in Y */
    ry: boolean;
    /** Point restrained rotationally in Z */
    rz: boolean;
    /** Point title */
    title: string;
    /** Point restrained translationally in X */
    tx: boolean;
    /** Point restrained translationally in Y */
    ty: boolean;
    /** Point restrained translationally in Z */
    tz: boolean;
    /** Node/point x coordinate */
    x: number;
    /** Node/point y coordinate */
    y: number;
    /** Node/point z coordinate */
    z: number;
}


/** Object returned by SelectAssembly */
interface SelectAssemblyReturn {
    /** Assembly label */
    label: number;
    /** Parent assembly label */
    parent: number;
    /** Assembly title */
    title: string;
}


/** Object function argument in SetPoint */
interface SetPointArgument_data {
    /** Assembly label */
    assembly: number;
    /** Coordinate system label */
    csys?: number;
    /** Node label (not rewuired if using x, y and z) */
    node?: number;
    /** Point restrained rotationally in X */
    rx?: boolean;
    /** Point restrained rotationally in Y */
    ry?: boolean;
    /** Point restrained rotationally in Z */
    rz?: boolean;
    /** Title */
    title?: string;
    /** Point restrained translationally in X */
    tx?: boolean;
    /** Point restrained translationally in Y */
    ty?: boolean;
    /** Point restrained translationally in Z */
    tz?: boolean;
    /** X coordinate (not required if using node) */
    x?: number;
    /** Y coordinate (not required if using node) */
    y?: number;
    /** Z coordinate (not required if using node) */
    z?: number;
}

declare class Dummy {
/**
 * Associates a comment with a dummy.
 * @param Comment Comment that will be attached to the dummy
 */
    AssociateComment(Comment: Comment): void;

/**
 * Blanks the dummy
 */
    Blank(): void;

/**
 * Blanks all of the dummys in the model.
 * @param Model Model that all dummys will be blanked in
 * @param redraw If model should be redrawn or not. If omitted redraw is false. If you want to do several (un)blanks and only redraw after the last one then use false for all redraws apart from the last one. Alternatively you can redraw using View.Redraw().
 */
    static BlankAll(Model: Model, redraw?: boolean): void;

/**
 * Blanks all of the flagged dummys in the model.
 * @param Model Model that all the flagged dummys will be blanked in
 * @param flag Flag set on the dummys that you want to blank
 * @param redraw If model should be redrawn or not. If omitted redraw is false. If you want to do several (un)blanks and only redraw after the last one then use false for all redraws apart from the last one. Alternatively you can redraw using View.Redraw().
 */
    static BlankFlagged(Model: Model, flag: number, redraw?: boolean): void;

/**
 * Checks if the dummy is blanked or not.
 * @returns true if blanked, false if not.
 */
    Blanked(): boolean;

/**
 * Clears a flag on the dummy.
 * @param flag Flag to clear on the dummy
 */
    ClearFlag(flag: number): void;

/**
 * Copies the dummy. The target include of the copied dummy can be set using Options.copy_target_include.
 * @param range If you want to keep the copied item in the range specified for the current include. Default value is false. To set current include, use Include.MakeCurrentLayer().
 * @returns Dummy object
 */
    Copy(range?: boolean): Dummy;

/**
 * Detaches a comment from a dummy.
 * @param Comment Comment that will be detached from the dummy
 */
    DetachComment(Comment: Comment): void;

/**
 * Adds an error for dummy. For more details on checking see the Check class.
 * @param message The error message to give
 * @param details An optional detailed error message
 */
    Error(message: string, details?: string): void;

/**
 * Returns the first dummy in the model.
 * @param Model Model to get first dummy in
 * @returns Dummy object (or null if there are no dummys in the model).
 */
    static First(Model: Model): Dummy;

/**
 * Returns the first free dummy label in the model. Also see Dummy.LastFreeLabel(), Dummy.NextFreeLabel() and Model.FirstFreeItemLabel().
 * @param Model Model to get first free dummy label in
 * @param layer Include file (0 for the main file) to search for labels in (Equivalent to First free in layer in editing panels). If omitted the whole model will be used (Equivalent to First free in editing panels).
 * @returns Dummy label.
 */
    static FirstFreeLabel(Model: Model, layer?: number): number;

/**
 * Flags all of the dummys in the model with a defined flag.
 * @param Model Model that all dummys will be flagged in
 * @param flag Flag to set on the dummys
 */
    static FlagAll(Model: Model, flag: number): void;

/**
 * Checks if the dummy is flagged or not.
 * @param flag Flag to test on the dummy
 * @returns true if flagged, false if not.
 */
    Flagged(flag: number): boolean;

/**
 * Calls a function for each dummy in the model. Note that ForEach has been designed to make looping over dummys as fast as possible and so has some limitations. Firstly, a single temporary Dummy object is created and on each function call it is updated with the current dummy data. This means that you should not try to store the Dummy object for later use (e.g. in an array) as it is temporary. Secondly, you cannot create new dummys inside a ForEach loop. 
 * @param Model Model that all dummys are in
 * @param func Function to call for each dummy
 * @param extra An optional extra object/array/string etc that will appended to arguments when calling the function
 */
    static ForEach(Model: Model, func: () => void, extra?: any): void;

/**
 * Returns an array of Dummy objects or properties for all of the dummys in a model in PRIMER. If the optional property argument is not given then an array of Dummy objects is returned. If the property argument is given, that property value for each dummy is returned in the array instead of a Dummy object
 * @param Model Model to get dummys from
 * @param property Name for property to get for all dummys in the model
 * @returns Array of Dummy objects or properties
 */
    static GetAll(Model: Model, property?: string): Dummy[];

/**
 * Returns the information for an assembly
 * @param index The index of the assembly you want the coordinates for. Note that reference points start at 0, not 1. 0 &lt;= index &lt; assemblies
 */
    GetAssembly(index: number): GetAssemblyReturn;

/**
 * Get information about a child assembly from its parent assembly.
 * @param label The label of the parent assembly.
 * @param index index of the child (start with 0 till n-1, where n is total number of child).
 */
    GetAssemblyChildInfo(label: number, index: number): GetAssemblyChildInfoReturn;

/**
 * Get assembly information for a given assembly ID and returns an object containing the details.
 * @param label The label of the assembly you want the Assembly object for.
 */
    GetAssemblyFromID(label: number): GetAssemblyFromIDReturn;

/**
 * Returns an array of Part objects representing all the parts within the assembly.
 * @param label The label of the assembly.
 * @returns Array of Part objects
 */
    GetAssemblyPart(label: number): Part[];

/**
 * Extracts the comments associated to a dummy.
 * @returns Array of Comment objects (or null if there are no comments associated to the node).
 */
    GetComments(): Comment[];

/**
 * Returns an array of Dummy objects for all of the flagged dummys in a model in PRIMER If the optional property argument is not given then an array of Dummy objects is returned. If the property argument is given, then that property value for each dummy is returned in the array instead of a Dummy object
 * @param Model Model to get dummys from
 * @param flag Flag set on the dummys that you want to retrieve
 * @param property Name for property to get for all flagged dummys in the model
 * @returns Array of Dummy objects or properties
 */
    static GetFlagged(Model: Model, flag: number, property?: string): Dummy[];

/**
 * Returns the Dummy object for a dummy ID.
 * @param Model Model to find the dummy in
 * @param number number of the dummy you want the Dummy object for
 * @returns Dummy object (or null if dummy does not exist).
 */
    static GetFromID(Model: Model, number: number): Dummy;

/**
 * Checks if a Dummy property is a parameter or not. Note that object properties that are parameters are normally returned as the integer or float parameter values as that is virtually always what the user would want. For this function to work the JavaScript interpreter must use the parameter name instead of the value. This can be done by setting the Options.property_parameter_names option to true before calling the function and then resetting it to false afterwards.. This behaviour can also temporarily be switched by using the Dummy.ViewParameters() method and 'method chaining' (see the examples below).
 * @param prop dummy property to get parameter for
 * @returns Parameter object if property is a parameter, null if not.
 */
    GetParameter(prop: string): Parameter;

/**
 * Returns the information for a reference point
 * @param index The index of the reference point you want the information for. Note that reference points start at 0, not 1. 0 &lt;= index &lt; points
 */
    GetPoint(index: number): GetPointReturn;

/**
 * Returns the coordinates of a reference point
 * @param rpt The reference point you want the coordinates for. Note that reference points start at 0, not 1.
 * @returns Array containing the reference point coordinates
 */
    GetPointData(rpt: number): number[];

/**
 * Returns the title of a reference point
 * @param rpt The reference point you want the title for. Note that reference points start at 0, not 1.
 * @returns The reference point title
 */
    GetPointTitle(rpt: number): string;

/**
 * Returns the last dummy in the model.
 * @param Model Model to get last dummy in
 * @returns Dummy object (or null if there are no dummys in the model).
 */
    static Last(Model: Model): Dummy;

/**
 * Returns the last free dummy label in the model. Also see Dummy.FirstFreeLabel(), Dummy.NextFreeLabel() and see Model.LastFreeItemLabel()
 * @param Model Model to get last free dummy label in
 * @param layer Include file (0 for the main file) to search for labels in (Equivalent to Highest free in layer in editing panels). If omitted the whole model will be used.
 * @returns Dummy label.
 */
    static LastFreeLabel(Model: Model, layer?: number): number;

/**
 * Returns the next dummy in the model.
 * @returns Dummy object (or null if there are no more dummys in the model).
 */
    Next(): Dummy;

/**
 * Returns the next free (highest+1) dummy label in the model. Also see Dummy.FirstFreeLabel(), Dummy.LastFreeLabel() and Model.NextFreeItemLabel()
 * @param Model Model to get next free dummy label in
 * @param layer Include file (0 for the main file) to search for labels in (Equivalent to Highest+1 in layer in editing panels). If omitted the whole model will be used (Equivalent to Highest+1 in editing panels).
 * @returns Dummy label.
 */
    static NextFreeLabel(Model: Model, layer?: number): number;

/**
 * Allows the user to pick a dummy.
 * @param prompt Text to display as a prompt to the user
 * @param limit If the argument is a Model then only dummys from that model can be picked. If the argument is a Flag then only dummys that are flagged with limit can be selected. If omitted, or null, any dummys from any model can be selected. from any model.
 * @param modal If picking is modal (blocks the user from doing anything else in PRIMER until this window is dismissed). If omitted the pick will be modal.
 * @param button_text By default the window with the prompt will have a button labelled 'Cancel' which if pressed will cancel the pick and return null. If you want to change the text on the button use this argument. If omitted 'Cancel' will be used.
 * @returns Dummy object (or null if not picked)
 */
    static Pick(prompt: string, limit?: Model | number, modal?: boolean, button_text?: string): Dummy;

/**
 * Returns the previous dummy in the model.
 * @returns Dummy object (or null if there are no more dummys in the model).
 */
    Previous(): Dummy;

/**
 * Removes a reference point from a dummy
 * @param index The index of the reference point you want to remove. Note that reference points start at 0, not 1. 0 &lt;= index &lt; points
 */
    RemovePoint(index: number): void;

/**
 * Renumbers all of the dummys in the model.
 * @param Model Model that all dummys will be renumbered in
 * @param start Start point for renumbering
 */
    static RenumberAll(Model: Model, start: number): void;

/**
 * Renumbers all of the flagged dummys in the model.
 * @param Model Model that all the flagged dummys will be renumbered in
 * @param flag Flag set on the dummys that you want to renumber
 * @param start Start point for renumbering
 */
    static RenumberFlagged(Model: Model, flag: number, start: number): void;

/**
 * Allows the user to select dummys using standard PRIMER object menus.
 * @param flag Flag to use when selecting dummys
 * @param prompt Text to display as a prompt to the user
 * @param limit If the argument is a Model then only dummys from that model can be selected. If the argument is a Flag then only dummys that are flagged with limit can be selected (limit should be different to flag). If omitted, or null, any dummys can be selected. from any model.
 * @param modal If selection is modal (blocks the user from doing anything else in PRIMER until this window is dismissed). If omitted the selection will be modal.
 * @returns Number of dummys selected or null if menu cancelled
 */
    static Select(flag: number, prompt: string, limit?: Model | number, modal?: boolean): number;

/**
 * Returns an array of objects containing the assembly informaitons or null if menu cancelled.
 */
    SelectAssembly(): SelectAssemblyReturn[];

/**
 * Sets a set node for a Dummy/HBM assembly.
 * @param label The label of the assembly.
 * @param nsid The label of the set node to be added into the assembly.
 */
    SetAssemblyNodeSet(label: number, nsid: number): void;

/**
 * Sets a part for a Dummy assembly.
 * @param label The label of the assembly.
 * @param pid The label of the set part to be added into the assembly.
 */
    SetAssemblyPart(label: number, pid: number): void;

/**
 * Sets a set part for a Dummy/HBM assembly.
 * @param label The label of the assembly.
 * @param psid The label of the set part to be added into the assembly.
 */
    SetAssemblyPartSet(label: number, psid: number): void;

/**
 * Sets -ve and +ve stop angles in the assembly.
 * @param label The label of the assembly.
 * @param axis Axis (0 = X, 1 = Y, or 2 = Z) on which to define stop angles.
 * @param stop_neg -ve stop angle.
 * @param stop_pos +ve stop angle.
 */
    SetAssemblyStopAngle(label: number, axis: number, stop_neg: number, stop_pos: number): void;

/**
 * Sets a flag on the dummy.
 * @param flag Flag to set on the dummy
 */
    SetFlag(flag: number): void;

/**
 * Sets the data for a reference point in a dummy
 * @param index The index of the reference point you want to set. Note that reference points start at 0, not 1. To add a new point use index points
 * @param data Object containing the reference point data. The properties can be:
 */
    SetPoint(index: number, data: SetPointArgument_data): void;

/**
 * Sketches the dummy. The dummy will be sketched until you either call Dummy.Unsketch(), Dummy.UnsketchAll(), Model.UnsketchAll(), or delete the model
 * @param redraw If model should be redrawn or not after the dummy is sketched. If omitted redraw is true. If you want to sketch several dummys and only redraw after the last one then use false for redraw and call View.Redraw().
 */
    Sketch(redraw?: boolean): void;

/**
 * Sketches the assembly
 * @param label The label of the assembly you want to sketch.
 * @param redraw If model should be redrawn or not. If omitted redraw is false. If you want to do several (un)blanks and only redraw after the last one then use false for all redraws apart from the last one. Alternatively you can redraw using View.Redraw().
 */
    SketchAssembly(label: number, redraw?: boolean): void;

/**
 * Sketches all of the flagged dummys in the model. The dummys will be sketched until you either call Dummy.Unsketch(), Dummy.UnsketchFlagged(), Model.UnsketchAll(), or delete the model
 * @param Model Model that all the flagged dummys will be sketched in
 * @param flag Flag set on the dummys that you want to sketch
 * @param redraw If model should be redrawn or not after the dummys are sketched. If omitted redraw is true. If you want to sketch flagged dummys several times and only redraw after the last one then use false for redraw and call View.Redraw().
 */
    static SketchFlagged(Model: Model, flag: number, redraw?: boolean): void;

/**
 * Returns the total number of dummys in the model.
 * @param Model Model to get total for
 * @param exists true if only existing dummys should be counted. If false or omitted referenced but undefined dummys will also be included in the total.
 * @returns number of dummys
 */
    static Total(Model: Model, exists?: boolean): number;

/**
 * Unblanks the dummy
 */
    Unblank(): void;

/**
 * Unblanks all of the dummys in the model.
 * @param Model Model that all dummys will be unblanked in
 * @param redraw If model should be redrawn or not. If omitted redraw is false. If you want to do several (un)blanks and only redraw after the last one then use false for all redraws apart from the last one. Alternatively you can redraw using View.Redraw().
 */
    static UnblankAll(Model: Model, redraw?: boolean): void;

/**
 * Unblanks all of the flagged dummys in the model.
 * @param Model Model that the flagged dummys will be unblanked in
 * @param flag Flag set on the dummys that you want to unblank
 * @param redraw If model should be redrawn or not. If omitted redraw is false. If you want to do several (un)blanks and only redraw after the last one then use false for all redraws apart from the last one. Alternatively you can redraw using View.Redraw().
 */
    static UnblankFlagged(Model: Model, flag: number, redraw?: boolean): void;

/**
 * Unsets a defined flag on all of the dummys in the model.
 * @param Model Model that the defined flag for all dummys will be unset in
 * @param flag Flag to unset on the dummys
 */
    static UnflagAll(Model: Model, flag: number): void;

/**
 * Unsketches the dummy.
 * @param redraw If model should be redrawn or not after the dummy is unsketched. If omitted redraw is true. If you want to unsketch several dummys and only redraw after the last one then use false for redraw and call View.Redraw().
 */
    Unsketch(redraw?: boolean): void;

/**
 * Unsketches all dummys.
 * @param Model Model that all dummys will be unblanked in
 * @param redraw If model should be redrawn or not after the dummys are unsketched. If omitted redraw is true. If you want to unsketch several things and only redraw after the last one then use false for redraw and call View.Redraw().
 */
    static UnsketchAll(Model: Model, redraw?: boolean): void;

/**
 * Unsketches the assembly
 * @param label The label of the assembly you want to unsketch.
 * @param redraw If model should be redrawn or not. If omitted redraw is false. If you want to do several (un)blanks and only redraw after the last one then use false for all redraws apart from the last one. Alternatively you can redraw using View.Redraw().
 */
    UnsketchAssembly(label: number, redraw?: boolean): void;

/**
 * Unsketches all flagged dummys in the model.
 * @param Model Model that all dummys will be unsketched in
 * @param flag Flag set on the dummys that you want to unsketch
 * @param redraw If model should be redrawn or not after the dummys are unsketched. If omitted redraw is true. If you want to unsketch several things and only redraw after the last one then use false for redraw and call View.Redraw().
 */
    static UnsketchFlagged(Model: Model, flag: number, redraw?: boolean): void;

/**
 * Object properties that are parameters are normally returned as the integer or float parameter values as that is virtually always what the user would want. This function temporarily changes the behaviour so that if a property is a parameter the parameter name is returned instead. This can be used with 'method chaining' (see the example below) to make sure a property argument is correct.
 * @returns Dummy object.
 */
    ViewParameters(): Dummy;

/**
 * Adds a warning for dummy. For more details on checking see the Check class.
 * @param message The warning message to give
 * @param details An optional detailed warning message
 */
    Warning(message: string, details?: string): void;

/**
 * Returns the cross references for this dummy.
 * @returns Xrefs object.
 */
    Xrefs(): Xrefs;

/** Number of assemblies defined. */
    readonly assemblies: number;
/** true if dummy exists, false if referred to but not defined. */
    readonly exists: boolean;
/** Dummy number. Also see the label property which is an alternative name for this. */
    readonly id: number;
/** The Include file number that the dummy is in. */
    include: number;
/** Dummy number. Also see the id property which is an alternative name for this. */
    readonly label: number;
/** The Model number that the dummy is in. */
    readonly model: number;
/** Number of reference points defined. */
    readonly points: number;
/** Dummy title. */
    title: string;
/** H-Point X coordinate. */
    readonly xhpoint: number;
/** H-Point Y coordinate. */
    readonly yhpoint: number;
/** H-Point Z coordinate. */
    readonly zhpoint: number;
}

declare class ElementDeath {
/**
 * Associates a comment with a element death.
 * @param Comment Comment that will be attached to the element death
 */
    AssociateComment(Comment: Comment): void;

/**
 * Starts an edit panel in Browse mode.
 * @param modal If this window is modal (blocks the user from doing anything else in PRIMER until this window is dismissed). If omitted the window will be modal.
 */
    Browse(modal?: boolean): void;

/**
 * Clears a flag on the element death.
 * @param flag Flag to clear on the element death
 */
    ClearFlag(flag: number): void;

/**
 * Copies the element death. The target include of the copied element death can be set using Options.copy_target_include.
 * @param range If you want to keep the copied item in the range specified for the current include. Default value is false. To set current include, use Include.MakeCurrentLayer().
 * @returns ElementDeath object
 */
    Copy(range?: boolean): ElementDeath;

/**
 * Starts an interactive editing panel to create a element death
 * @param Model Model that the element death will be created in.
 * @param modal If this window is modal (blocks the user from doing anything else in PRIMER until this window is dismissed). If omitted the window will be modal.
 * @returns ElementDeath object (or null if not made).
 */
    static Create(Model: Model, modal?: boolean): ElementDeath;

/**
 * Detaches a comment from a element death.
 * @param Comment Comment that will be detached from the element death
 */
    DetachComment(Comment: Comment): void;

/**
 * Starts an interactive editing panel.
 * @param modal If this window is modal (blocks the user from doing anything else in PRIMER until this window is dismissed). If omitted the window will be modal.
 */
    Edit(modal?: boolean): void;

/**
 * Adds an error for element death. For more details on checking see the Check class.
 * @param message The error message to give
 * @param details An optional detailed error message
 */
    Error(message: string, details?: string): void;

/**
 * Returns the first element death in the model.
 * @param Model Model to get first element death in
 * @returns ElementDeath object (or null if there are no element deaths in the model).
 */
    static First(Model: Model): ElementDeath;

/**
 * Flags all of the element deaths in the model with a defined flag.
 * @param Model Model that all element deaths will be flagged in
 * @param flag Flag to set on the element deaths
 */
    static FlagAll(Model: Model, flag: number): void;

/**
 * Checks if the element death is flagged or not.
 * @param flag Flag to test on the element death
 * @returns true if flagged, false if not.
 */
    Flagged(flag: number): boolean;

/**
 * Calls a function for each element death in the model. Note that ForEach has been designed to make looping over element deaths as fast as possible and so has some limitations. Firstly, a single temporary ElementDeath object is created and on each function call it is updated with the current element death data. This means that you should not try to store the ElementDeath object for later use (e.g. in an array) as it is temporary. Secondly, you cannot create new element deaths inside a ForEach loop. 
 * @param Model Model that all element deaths are in
 * @param func Function to call for each element death
 * @param extra An optional extra object/array/string etc that will appended to arguments when calling the function
 */
    static ForEach(Model: Model, func: () => void, extra?: any): void;

/**
 * Returns an array of ElementDeath objects or properties for all of the element deaths in a model in PRIMER. If the optional property argument is not given then an array of ElementDeath objects is returned. If the property argument is given, that property value for each element death is returned in the array instead of a ElementDeath object
 * @param Model Model to get element deaths from
 * @param property Name for property to get for all element deaths in the model
 * @returns Array of ElementDeath objects or properties
 */
    static GetAll(Model: Model, property?: string): ElementDeath[];

/**
 * Extracts the comments associated to a element death.
 * @returns Array of Comment objects (or null if there are no comments associated to the node).
 */
    GetComments(): Comment[];

/**
 * Returns an array of ElementDeath objects for all of the flagged element deaths in a model in PRIMER If the optional property argument is not given then an array of ElementDeath objects is returned. If the property argument is given, then that property value for each element death is returned in the array instead of a ElementDeath object
 * @param Model Model to get element deaths from
 * @param flag Flag set on the element deaths that you want to retrieve
 * @param property Name for property to get for all flagged element deaths in the model
 * @returns Array of ElementDeath objects or properties
 */
    static GetFlagged(Model: Model, flag: number, property?: string): ElementDeath[];

/**
 * Returns the ElementDeath object for a element death ID.
 * @param Model Model to find the element death in
 * @param number number of the element death you want the ElementDeath object for
 * @returns ElementDeath object (or null if element death does not exist).
 */
    static GetFromID(Model: Model, number: number): ElementDeath;

/**
 * Checks if a ElementDeath property is a parameter or not. Note that object properties that are parameters are normally returned as the integer or float parameter values as that is virtually always what the user would want. For this function to work the JavaScript interpreter must use the parameter name instead of the value. This can be done by setting the Options.property_parameter_names option to true before calling the function and then resetting it to false afterwards.. This behaviour can also temporarily be switched by using the ElementDeath.ViewParameters() method and 'method chaining' (see the examples below).
 * @param prop element death property to get parameter for
 * @returns Parameter object if property is a parameter, null if not.
 */
    GetParameter(prop: string): Parameter;

/**
 * Returns the keyword for this element death (*DEFINE_ELEMENT_DEATH). Note that a carriage return is not added. See also ElementDeath.KeywordCards()
 * @returns string containing the keyword.
 */
    Keyword(): string;

/**
 * Returns the keyword cards for the element death. Note that a carriage return is not added. See also ElementDeath.Keyword()
 * @returns string containing the cards.
 */
    KeywordCards(): string;

/**
 * Returns the last element death in the model.
 * @param Model Model to get last element death in
 * @returns ElementDeath object (or null if there are no element deaths in the model).
 */
    static Last(Model: Model): ElementDeath;

/**
 * Returns the next element death in the model.
 * @returns ElementDeath object (or null if there are no more element deaths in the model).
 */
    Next(): ElementDeath;

/**
 * Returns the previous element death in the model.
 * @returns ElementDeath object (or null if there are no more element deaths in the model).
 */
    Previous(): ElementDeath;

/**
 * Allows the user to select element deaths using standard PRIMER object menus.
 * @param flag Flag to use when selecting element deaths
 * @param prompt Text to display as a prompt to the user
 * @param limit If the argument is a Model then only element deaths from that model can be selected. If the argument is a Flag then only element deaths that are flagged with limit can be selected (limit should be different to flag). If omitted, or null, any element deaths can be selected. from any model.
 * @param modal If selection is modal (blocks the user from doing anything else in PRIMER until this window is dismissed). If omitted the selection will be modal.
 * @returns Number of element deaths selected or null if menu cancelled
 */
    static Select(flag: number, prompt: string, limit?: Model | number, modal?: boolean): number;

/**
 * Sets a flag on the element death.
 * @param flag Flag to set on the element death
 */
    SetFlag(flag: number): void;

/**
 * Returns the total number of element deaths in the model.
 * @param Model Model to get total for
 * @param exists true if only existing element deaths should be counted. If false or omitted referenced but undefined element deaths will also be included in the total.
 * @returns number of element deaths
 */
    static Total(Model: Model, exists?: boolean): number;

/**
 * Unsets a defined flag on all of the element deaths in the model.
 * @param Model Model that the defined flag for all element deaths will be unset in
 * @param flag Flag to unset on the element deaths
 */
    static UnflagAll(Model: Model, flag: number): void;

/**
 * Object properties that are parameters are normally returned as the integer or float parameter values as that is virtually always what the user would want. This function temporarily changes the behaviour so that if a property is a parameter the parameter name is returned instead. This can be used with 'method chaining' (see the example below) to make sure a property argument is correct.
 * @returns ElementDeath object.
 */
    ViewParameters(): ElementDeath;

/**
 * Adds a warning for element death. For more details on checking see the Check class.
 * @param message The warning message to give
 * @param details An optional detailed warning message
 */
    Warning(message: string, details?: string): void;

/**
 * Returns the cross references for this element death.
 * @returns Xrefs object.
 */
    Xrefs(): Xrefs;

/**
 * Create a new ElementDeath object.
 * @param Model Model that element death will be created in
 * @param type ElementDeath type. Can be ElementDeath.SOLID, ElementDeath.SOLID_SET, ElementDeath.BEAM, ElementDeath.BEAM_SET, ElementDeath.SHELL, ElementDeath.SHELL_SET, ElementDeath.THICK_SHELL or ElementDeath.THICK_SHELL_SET
 * @param eid_sid Element or element set ID
 * @returns ElementDeath object
 */
    constructor(Model: Model, type: string, eid_sid: number);

/**
 * Creates a string containing the element death data in keyword format. Note that this contains the keyword header and the keyword cards. See also ElementDeath.Keyword() and ElementDeath.KeywordCards().
 * @returns string
 */
    toString(): string;

/** Box restricting element deletion */
    boxid: number;
/** Coordinate ID for transforming boxid. */
    cid: number;
/** Element ID or element set ID. The sid property is an alternative name for this. */
    eid: number;
/** true if element death exists, false if referred to but not defined. */
    readonly exists: boolean;
/** Group ID for simultaneous deletion. */
    idgrp: number;
/** The Include file number that the element death is in. */
    include: number;
/** If true, Ansys LS-DYNA deletes elements outside box, otherwise inside box. */
    inout: boolean;
/** The Model number that the element death is in. */
    readonly model: number;
/** ElementDeath option. Can be ElementDeath.SOLID, ElementDeath.SOLID_SET, ElementDeath.BEAM, ElementDeath.BEAM_SET, ElementDeath.SHELL, ElementDeath.SHELL_SET, ElementDeath.THICK_SHELL or ElementDeath.THICK_SHELL_SET. The type property is an alternative name for this. */
    option: number;
/** Deletion percentage. */
    percent: number;
/** Element ID or element set ID. The eid property is an alternative name for this. */
    sid: number;
/** Deletion time for elimination */
    time: number;
/** ElementDeath title */
    title: string;
/** ElementDeath option. Can be ElementDeath.SOLID, ElementDeath.SOLID_SET, ElementDeath.BEAM, ElementDeath.BEAM_SET, ElementDeath.SHELL, ElementDeath.SHELL_SET, ElementDeath.THICK_SHELL or ElementDeath.THICK_SHELL_SET The option property is an alternative name for this. */
    type: number;
/** Beam option */
    static BEAM: number;
/** Beam set option */
    static BEAM_SET: number;
/** Shell option */
    static SHELL: number;
/** Shell set option */
    static SHELL_SET: number;
/** Solid option */
    static SOLID: number;
/** Solid set option */
    static SOLID_SET: number;
/** Thick shell option */
    static THICK_SHELL: number;
/** Thick shell set option */
    static THICK_SHELL_SET: number;
}

declare class ExtraNodes {
/**
 * Associates a comment with a constrained extra node.
 * @param Comment Comment that will be attached to the constrained extra node
 */
    AssociateComment(Comment: Comment): void;

/**
 * Blanks the constrained extra node
 */
    Blank(): void;

/**
 * Blanks all of the constrained extra nodes in the model.
 * @param Model Model that all constrained extra nodes will be blanked in
 * @param redraw If model should be redrawn or not. If omitted redraw is false. If you want to do several (un)blanks and only redraw after the last one then use false for all redraws apart from the last one. Alternatively you can redraw using View.Redraw().
 */
    static BlankAll(Model: Model, redraw?: boolean): void;

/**
 * Blanks all of the flagged constrained extra nodes in the model.
 * @param Model Model that all the flagged constrained extra nodes will be blanked in
 * @param flag Flag set on the constrained extra nodes that you want to blank
 * @param redraw If model should be redrawn or not. If omitted redraw is false. If you want to do several (un)blanks and only redraw after the last one then use false for all redraws apart from the last one. Alternatively you can redraw using View.Redraw().
 */
    static BlankFlagged(Model: Model, flag: number, redraw?: boolean): void;

/**
 * Checks if the constrained extra node is blanked or not.
 * @returns true if blanked, false if not.
 */
    Blanked(): boolean;

/**
 * Starts an edit panel in Browse mode.
 * @param modal If this window is modal (blocks the user from doing anything else in PRIMER until this window is dismissed). If omitted the window will be modal.
 */
    Browse(modal?: boolean): void;

/**
 * Clears a flag on the constrained extra node.
 * @param flag Flag to clear on the constrained extra node
 */
    ClearFlag(flag: number): void;

/**
 * Copies the constrained extra node. The target include of the copied constrained extra node can be set using Options.copy_target_include.
 * @param range If you want to keep the copied item in the range specified for the current include. Default value is false. To set current include, use Include.MakeCurrentLayer().
 * @returns ExtraNodes object
 */
    Copy(range?: boolean): ExtraNodes;

/**
 * Starts an interactive editing panel to create a constrained extra node
 * @param Model Model that the constrained extra node will be created in.
 * @param modal If this window is modal (blocks the user from doing anything else in PRIMER until this window is dismissed). If omitted the window will be modal.
 * @returns ExtraNodes object (or null if not made).
 */
    static Create(Model: Model, modal?: boolean): ExtraNodes;

/**
 * Detaches a comment from a constrained extra node.
 * @param Comment Comment that will be detached from the constrained extra node
 */
    DetachComment(Comment: Comment): void;

/**
 * Starts an interactive editing panel.
 * @param modal If this window is modal (blocks the user from doing anything else in PRIMER until this window is dismissed). If omitted the window will be modal.
 */
    Edit(modal?: boolean): void;

/**
 * Adds an error for constrained extra node. For more details on checking see the Check class.
 * @param message The error message to give
 * @param details An optional detailed error message
 */
    Error(message: string, details?: string): void;

/**
 * Extracts the actual colour used for constrained extra node. By default in PRIMER many entities such as elements get their colour automatically from the part that they are in. PRIMER cycles through 13 default colours based on the label of the entity. In this case the constrained extra node colour property will return the value Colour.PART instead of the actual colour. This method will return the actual colour which is used for drawing the constrained extra node.
 * @returns colour value (integer)
 */
    ExtractColour(): number;

/**
 * Returns the first constrained extra node in the model.
 * @param Model Model to get first constrained extra node in
 * @returns ExtraNodes object (or null if there are no constrained extra nodes in the model).
 */
    static First(Model: Model): ExtraNodes;

/**
 * Flags all of the constrained extra nodes in the model with a defined flag.
 * @param Model Model that all constrained extra nodes will be flagged in
 * @param flag Flag to set on the constrained extra nodes
 */
    static FlagAll(Model: Model, flag: number): void;

/**
 * Checks if the constrained extra node is flagged or not.
 * @param flag Flag to test on the constrained extra node
 * @returns true if flagged, false if not.
 */
    Flagged(flag: number): boolean;

/**
 * Calls a function for each constrained extra node in the model. Note that ForEach has been designed to make looping over constrained extra nodes as fast as possible and so has some limitations. Firstly, a single temporary ExtraNodes object is created and on each function call it is updated with the current constrained extra node data. This means that you should not try to store the ExtraNodes object for later use (e.g. in an array) as it is temporary. Secondly, you cannot create new constrained extra nodes inside a ForEach loop. 
 * @param Model Model that all constrained extra nodes are in
 * @param func Function to call for each constrained extra node
 * @param extra An optional extra object/array/string etc that will appended to arguments when calling the function
 */
    static ForEach(Model: Model, func: () => void, extra?: any): void;

/**
 * Returns an array of ExtraNodes objects or properties for all of the constrained extra nodes in a model in PRIMER. If the optional property argument is not given then an array of ExtraNodes objects is returned. If the property argument is given, that property value for each constrained extra node is returned in the array instead of a ExtraNodes object
 * @param Model Model to get constrained extra nodes from
 * @param property Name for property to get for all constrained extra nodes in the model
 * @returns Array of ExtraNodes objects or properties
 */
    static GetAll(Model: Model, property?: string): ExtraNodes[];

/**
 * Extracts the comments associated to a constrained extra node.
 * @returns Array of Comment objects (or null if there are no comments associated to the node).
 */
    GetComments(): Comment[];

/**
 * Returns an array of ExtraNodes objects for all of the flagged constrained extra nodes in a model in PRIMER If the optional property argument is not given then an array of ExtraNodes objects is returned. If the property argument is given, then that property value for each constrained extra node is returned in the array instead of a ExtraNodes object
 * @param Model Model to get constrained extra nodes from
 * @param flag Flag set on the constrained extra nodes that you want to retrieve
 * @param property Name for property to get for all flagged constrained extra nodes in the model
 * @returns Array of ExtraNodes objects or properties
 */
    static GetFlagged(Model: Model, flag: number, property?: string): ExtraNodes[];

/**
 * Returns the ExtraNodes object for a constrained extra node ID.
 * @param Model Model to find the constrained extra node in
 * @param number number of the constrained extra node you want the ExtraNodes object for
 * @returns ExtraNodes object (or null if constrained extra node does not exist).
 */
    static GetFromID(Model: Model, number: number): ExtraNodes;

/**
 * Checks if a ExtraNodes property is a parameter or not. Note that object properties that are parameters are normally returned as the integer or float parameter values as that is virtually always what the user would want. For this function to work the JavaScript interpreter must use the parameter name instead of the value. This can be done by setting the Options.property_parameter_names option to true before calling the function and then resetting it to false afterwards.. This behaviour can also temporarily be switched by using the ExtraNodes.ViewParameters() method and 'method chaining' (see the examples below).
 * @param prop constrained extra node property to get parameter for
 * @returns Parameter object if property is a parameter, null if not.
 */
    GetParameter(prop: string): Parameter;

/**
 * Returns the keyword for this constrained extra nodes (*CONSTRAINED_EXTRA_NODES). Note that a carriage return is not added. See also ExtraNodes.KeywordCards()
 * @returns string containing the keyword.
 */
    Keyword(): string;

/**
 * Returns the keyword cards for the constrained extra nodes. Note that a carriage return is not added. See also ExtraNodes.Keyword()
 * @returns string containing the cards.
 */
    KeywordCards(): string;

/**
 * Returns the last constrained extra node in the model.
 * @param Model Model to get last constrained extra node in
 * @returns ExtraNodes object (or null if there are no constrained extra nodes in the model).
 */
    static Last(Model: Model): ExtraNodes;

/**
 * Returns the next constrained extra node in the model.
 * @returns ExtraNodes object (or null if there are no more constrained extra nodes in the model).
 */
    Next(): ExtraNodes;

/**
 * Allows the user to pick a constrained extra node.
 * @param prompt Text to display as a prompt to the user
 * @param limit If the argument is a Model then only constrained extra nodes from that model can be picked. If the argument is a Flag then only constrained extra nodes that are flagged with limit can be selected. If omitted, or null, any constrained extra nodes from any model can be selected. from any model.
 * @param modal If picking is modal (blocks the user from doing anything else in PRIMER until this window is dismissed). If omitted the pick will be modal.
 * @param button_text By default the window with the prompt will have a button labelled 'Cancel' which if pressed will cancel the pick and return null. If you want to change the text on the button use this argument. If omitted 'Cancel' will be used.
 * @returns ExtraNodes object (or null if not picked)
 */
    static Pick(prompt: string, limit?: Model | number, modal?: boolean, button_text?: string): ExtraNodes;

/**
 * Returns the previous constrained extra node in the model.
 * @returns ExtraNodes object (or null if there are no more constrained extra nodes in the model).
 */
    Previous(): ExtraNodes;

/**
 * Allows the user to select constrained extra nodes using standard PRIMER object menus.
 * @param flag Flag to use when selecting constrained extra nodes
 * @param prompt Text to display as a prompt to the user
 * @param limit If the argument is a Model then only constrained extra nodes from that model can be selected. If the argument is a Flag then only constrained extra nodes that are flagged with limit can be selected (limit should be different to flag). If omitted, or null, any constrained extra nodes can be selected. from any model.
 * @param modal If selection is modal (blocks the user from doing anything else in PRIMER until this window is dismissed). If omitted the selection will be modal.
 * @returns Number of constrained extra nodes selected or null if menu cancelled
 */
    static Select(flag: number, prompt: string, limit?: Model | number, modal?: boolean): number;

/**
 * Sets a flag on the constrained extra node.
 * @param flag Flag to set on the constrained extra node
 */
    SetFlag(flag: number): void;

/**
 * Sketches the constrained extra node. The constrained extra node will be sketched until you either call ExtraNodes.Unsketch(), ExtraNodes.UnsketchAll(), Model.UnsketchAll(), or delete the model
 * @param redraw If model should be redrawn or not after the constrained extra node is sketched. If omitted redraw is true. If you want to sketch several constrained extra nodes and only redraw after the last one then use false for redraw and call View.Redraw().
 */
    Sketch(redraw?: boolean): void;

/**
 * Sketches all of the flagged constrained extra nodes in the model. The constrained extra nodes will be sketched until you either call ExtraNodes.Unsketch(), ExtraNodes.UnsketchFlagged(), Model.UnsketchAll(), or delete the model
 * @param Model Model that all the flagged constrained extra nodes will be sketched in
 * @param flag Flag set on the constrained extra nodes that you want to sketch
 * @param redraw If model should be redrawn or not after the constrained extra nodes are sketched. If omitted redraw is true. If you want to sketch flagged constrained extra nodes several times and only redraw after the last one then use false for redraw and call View.Redraw().
 */
    static SketchFlagged(Model: Model, flag: number, redraw?: boolean): void;

/**
 * Returns the total number of constrained extra nodes in the model.
 * @param Model Model to get total for
 * @param exists true if only existing constrained extra nodes should be counted. If false or omitted referenced but undefined constrained extra nodes will also be included in the total.
 * @returns number of constrained extra nodes
 */
    static Total(Model: Model, exists?: boolean): number;

/**
 * Unblanks the constrained extra node
 */
    Unblank(): void;

/**
 * Unblanks all of the constrained extra nodes in the model.
 * @param Model Model that all constrained extra nodes will be unblanked in
 * @param redraw If model should be redrawn or not. If omitted redraw is false. If you want to do several (un)blanks and only redraw after the last one then use false for all redraws apart from the last one. Alternatively you can redraw using View.Redraw().
 */
    static UnblankAll(Model: Model, redraw?: boolean): void;

/**
 * Unblanks all of the flagged constrained extra nodes in the model.
 * @param Model Model that the flagged constrained extra nodes will be unblanked in
 * @param flag Flag set on the constrained extra nodes that you want to unblank
 * @param redraw If model should be redrawn or not. If omitted redraw is false. If you want to do several (un)blanks and only redraw after the last one then use false for all redraws apart from the last one. Alternatively you can redraw using View.Redraw().
 */
    static UnblankFlagged(Model: Model, flag: number, redraw?: boolean): void;

/**
 * Unsets a defined flag on all of the constrained extra nodes in the model.
 * @param Model Model that the defined flag for all constrained extra nodes will be unset in
 * @param flag Flag to unset on the constrained extra nodes
 */
    static UnflagAll(Model: Model, flag: number): void;

/**
 * Unsketches the constrained extra node.
 * @param redraw If model should be redrawn or not after the constrained extra node is unsketched. If omitted redraw is true. If you want to unsketch several constrained extra nodes and only redraw after the last one then use false for redraw and call View.Redraw().
 */
    Unsketch(redraw?: boolean): void;

/**
 * Unsketches all constrained extra nodes.
 * @param Model Model that all constrained extra nodes will be unblanked in
 * @param redraw If model should be redrawn or not after the constrained extra nodes are unsketched. If omitted redraw is true. If you want to unsketch several things and only redraw after the last one then use false for redraw and call View.Redraw().
 */
    static UnsketchAll(Model: Model, redraw?: boolean): void;

/**
 * Unsketches all flagged constrained extra nodes in the model.
 * @param Model Model that all constrained extra nodes will be unsketched in
 * @param flag Flag set on the constrained extra nodes that you want to unsketch
 * @param redraw If model should be redrawn or not after the constrained extra nodes are unsketched. If omitted redraw is true. If you want to unsketch several things and only redraw after the last one then use false for redraw and call View.Redraw().
 */
    static UnsketchFlagged(Model: Model, flag: number, redraw?: boolean): void;

/**
 * Object properties that are parameters are normally returned as the integer or float parameter values as that is virtually always what the user would want. This function temporarily changes the behaviour so that if a property is a parameter the parameter name is returned instead. This can be used with 'method chaining' (see the example below) to make sure a property argument is correct.
 * @returns ExtraNodes object.
 */
    ViewParameters(): ExtraNodes;

/**
 * Adds a warning for constrained extra node. For more details on checking see the Check class.
 * @param message The warning message to give
 * @param details An optional detailed warning message
 */
    Warning(message: string, details?: string): void;

/**
 * Returns the cross references for this constrained extra node.
 * @returns Xrefs object.
 */
    Xrefs(): Xrefs;

/**
 * Create a new ExtraNodes object.
 * @param Model Model that constrained extra nodes will be created in
 * @param option Specify the type of constrained extra nodes. Can be ExtraNodes.NODE or ExtraNodes.SET)
 * @param pid Part ID of rigid body
 * @param id Node node ID or node set ID
 * @param iflag Flag for adding node mass inertia to PART_INERTIA
 * @returns ExtraNodes object
 */
    constructor(Model: Model, option: number, pid: number, id: number, iflag: boolean);

/**
 * Creates a string containing the constrained extra nodes data in keyword format. Note that this contains the keyword header and the keyword cards. See also ExtraNodes.Keyword() and ExtraNodes.KeywordCards().
 * @returns string
 */
    toString(): string;

/** The colour of the extra nodes */
    colour: Colour;
/** true if constrained extra node exists, false if referred to but not defined. */
    readonly exists: boolean;
/** Node ID or node set ID (not internal label) */
    id: number;
/** Flag for adding node mass inertia to PART_INERTIA */
    iflag: boolean;
/** The Include file number that the constrained extra node is in. */
    include: number;
/** The label the constrained extra nodes has in PRIMER */
    readonly label: number;
/** The Model number that the constrained extra node is in. */
    readonly model: number;
/** The Constrained Extra Nodes option. Can be ExtraNodes.NODE or ExtraNodes.SET. */
    option: number;
/** Part ID of rigid body. */
    pid: number;
/** CNST is *CONSTRAINED_EXTRA_NODES_NODE. */
    static NODE: number;
/** CNST is *CONSTRAINED_EXTRA_NODES_SET. */
    static SET: number;
}


/** Object function argument in Get */
interface GetArgument_options {
    /** Password */
    password?: string;
    /** If set to true, then the response code will be returned instead of true/false. This can be used to retieve error messages and codes when the file is not returned successfully. */
    response?: boolean;
    /** Username */
    username?: string;
}


/** Object function argument in Upload */
interface UploadArgument_options {
    /** Password */
    password?: string;
    /** Username */
    username?: string;
}

declare class File {
/**
 * Close a file opened by a File object.
 */
    Close(): void;

/**
 * Copies a file
 * @param source Source filename you want to copy.
 * @param dest Destination filename you want to copy source file to.
 * @returns true if copy successful, false otherwise.
 */
    static Copy(source: string, dest: string): boolean;

/**
 * Deletes a file
 * @param filename Filename you want to delete.
 * @returns true if successful, false if not.
 */
    static Delete(filename: string): boolean;

/**
 * Changes a filename or directory name to the correct format for a specific operating system using the directory mappings (if present)
 * @param filename Filename you want to drive map.
 * @param format The format for the file/directory name. Can be Include.NATIVE, Include.UNIX or Include.WINDOWS
 * @returns string containing drive mapped filename
 */
    static DriveMapFilename(filename: string, format: number): string;

/**
 * Check if a file exists. See also File.IsDirectory() and See also File.IsFile().
 * @param filename Filename you want to check for existance.
 * @returns true/false
 */
    static Exists(filename: string): boolean;

/**
 * Find any files and/or directories in a directory.
 * @param directory Directory to look for files/directories in.
 * @param type Type of things to find. Can be bitwise OR of File.FILE and File.DIRECTORY. If omitted only files will be returned.
 * @returns Array of filenames/directories
 */
    static FindFiles(directory: string, type?: number): string[];

/**
 * Reads a line from a file which contains contain, opened for reading by a File object. Although this is possible using core JavaScript functions this function should be significantly faster as most of the processing is done by PRIMER in C rather than in the JavaScript interpreter. To enable this function to be as fast as possible a maximum line length of 512 characters is used. If you expect a file to have lines longer than 512 characters then use ReadLongLine which allows lines of any length. If one argument is used then the line must contain that string. If more than one argument is used then lines which contain any of the arguments will be returned
 * @param contain String which matching lines must contain
 *        This argument can be repeated if required.
 * @returns string read from file or undefined if end of file
 */
    FindLineContaining(...contain: string[]): string;

/**
 * Reads a line from a file which contains contain, opened for reading by a File object. Although this is possible using core JavaScript functions this function should be significantly faster as most of the processing is done by PRIMER in C rather than in the JavaScript interpreter. To enable this function to be as fast as possible a maximum line length of 512 characters is used. If you expect a file to have lines longer than 512 characters then use ReadLongLine which allows lines of any length. If one argument is used then the line must contain that string. If more than one argument is used then lines which contain any of the arguments will be returned
 * @param contain Array where each element specifies a String which matching lines must contain
 * @returns string read from file or undefined if end of file
 */
    FindLineContaining(contain: string[]): string;

/**
 * Reads a line from a file which starts with start, opened for reading by a File object. Although this is possible using core JavaScript functions this function should be significantly faster as most of the processing is done by PRIMER in C rather than in the JavaScript interpreter. To enable this function to be as fast as possible a maximum line length of 512 characters is used. If you expect a file to have lines longer than 512 characters then use ReadLongLine which allows lines of any length. If one argument is used then the line must start with that string. If more than one argument is used then lines which start with any of the arguments will be returned
 * @param start String which matching lines must start with
 *        This argument can be repeated if required.
 * @returns string read from file or undefined if end of file
 */
    FindLineStarting(...start: string[]): string;

/**
 * Reads a line from a file which starts with start, opened for reading by a File object. Although this is possible using core JavaScript functions this function should be significantly faster as most of the processing is done by PRIMER in C rather than in the JavaScript interpreter. To enable this function to be as fast as possible a maximum line length of 512 characters is used. If you expect a file to have lines longer than 512 characters then use ReadLongLine which allows lines of any length. If one argument is used then the line must start with that string. If more than one argument is used then lines which start with any of the arguments will be returned
 * @param start Array where each element specifies a String which matching lines must start with
 * @returns string read from file or undefined if end of file
 */
    FindLineStarting(start: string[]): string;

/**
 * Flushes a file opened for writing by a File object.
 */
    Flush(): void;

/**
 * Get a file from a remote location. See also File.Proxy(), File.ProxyPassword() and File.ProxyUsername().
 * @param url URL (uniform resource locator) of remote file you want to get. Currently http and ftp are supported. For http give the full address including the leading 'http://'. e.g. 'http://www.example.com/file.html'. For ftp an optional username and password can be given. e.g. 'ftp://ftp.example.com' retrieves the directory listing for the root directory. 'ftp://ftp.example.com/readme.txt' downloads the file readme.txt from the root directory. 'ftp://user:password@ftp.example.com/readme.txt' retrieves the readme.txt file from the user's home directory.
 * @param filename Filename you want to save the file to.
 * @param options Options for get. If 'username' and 'password' are set then basic authorization using the username and password will be used.
 * @returns true if file was successfully got, false otherwise.
 */
    static Get(url: string, filename: string, options?: GetArgument_options): boolean;

/**
 * Check if a filename is absolute or relative.
 * @param filename Filename you want to check.
 * @returns true/false
 */
    static IsAbsolute(filename: string): boolean;

/**
 * Check if a filename is a directory. See also File.Exists(), File.IsFile(), File.IsReadable() and File.IsWritable().
 * @param filename Filename you want to check.
 * @returns true/false
 */
    static IsDirectory(filename: string): boolean;

/**
 * Check if a filename is a file. See also File.Exists(), File.IsDirectory(), File.IsReadable() and File.IsWritable().
 * @param filename Filename you want to check.
 * @returns true/false
 */
    static IsFile(filename: string): boolean;

/**
 * Check if a filename has read permissions. See also File.Exists(), File.IsDirectory() and File.IsWritable().
 * @param filename Filename you want to check.
 * @returns true/false
 */
    static IsReadable(filename: string): boolean;

/**
 * Check if a filename has write permissions. If filename exists and it is a file then it is checked to see if it can be opened with write (File.APPEND permissions). If filename exists and it is a directory then the directory is checked for write permission (can files be created in the directory). If filename does not exist then it is assumed to be a file and is checked to see if it can be opened for writing (File.WRITE permissions). See also File.Exists(), File.IsDirectory() and File.IsReadable().
 * @param filename Filename you want to check.
 * @returns true/false
 */
    static IsWritable(filename: string): boolean;

/**
 * Make a directory. If PRIMER preference 'directory_permission' is set e.g.755 then this will apply (same as if set by chmod 755) ignoring any setting of umask. If there is no preference then the users current setting of umask will control permissions (same as system mkdir)
 * @param directory The name of the directory you want to create.
 * @returns true if successfully created, false if not.
 */
    static Mkdir(directory: string): boolean;

/**
 * Make a temporary filename for writing a temporary file.
 * @returns String name of temporary filename that can be used.
 */
    static Mktemp(): string;

/**
 * Set a proxy for files opened by http, ftp etc. See also File.Get(), File.ProxyPassword() and File.ProxyUsername().
 * @param name The name of the proxy.
 */
    static Proxy(name: string): void;

/**
 * Set a proxy password for files opened by http, ftp etc. See also File.Get(), File.Proxy() and File.ProxyUsername().
 * @param name Password for the proxy server.
 */
    static ProxyPassword(name: string): void;

/**
 * Set a proxy username for files opened by http, ftp etc. See also File.Get(), File.Proxy() and File.ProxyPassword().
 * @param username The username for the proxy.
 */
    static ProxyUsername(username: string): void;

/**
 * Reads all the remaining characters from a file opened for reading by a File object. As this function can read the entire file as a string be careful when reading large files as it will consume large amounts of memory.
 * @returns String. Characters read from file or undefined if end of file
 */
    ReadAll(): string;

/**
 * Reads binary data from a file opened for reading by a File object. The data is returned as an ArrayBuffer object. For more details on how to use an ArrayBuffer see the following links: https://developer.mozilla.org/en-US/docs/Web/JavaScript/Typed_arrays https://developer.mozilla.org/en-US/docs/Web/JavaScript/Reference/Global_Objects/ArrayBuffer https://developer.mozilla.org/en-US/docs/Web/JavaScript/Reference/Global_Objects/TypedArray https://developer.mozilla.org/en-US/docs/Web/JavaScript/Reference/Global_Objects/DataView.
 * @param length Number of bytes to try to read from the file. If omitted all the remaining data from the file will be read.
 * @returns ArrayBuffer object or undefined if end of file
 */
    ReadArrayBuffer(length?: number): ArrayBuffer;

/**
 * Reads the input CSV file and returns an array of string arrays. If the CSV file has legitimate records the function returns an Array object containing sub-arrays of strings otherwise the function returns NULL. The lengths of all the sub-arrays are the same and equal to maximum number of fields in any of the records. For records in a CSV file having fewer fields, the respective sub-arrays are padded with NULL elements to the maximum array length.
 * @param filename Filename you want to read CSV options from.
 * @param delimiter Delimiter string to be used. Default is a comma (",").
 * @param comment Comment string to be used. Default is a dollar sign ("$").
 * @returns 2d array of strings.
 */
    static ReadCSV(filename: string, delimiter?: string, comment?: string): string[][];

/**
 * Reads a single character from a file opened for reading by a File object.
 * @returns character read from file or undefined if end of file
 */
    ReadChar(): string;

/**
 * Reads a line from a file opened for reading by a File object. To enable this function to be as fast as possible a maximum line length of 512 characters is used. If you expect a file to have lines longer than 512 characters then use ReadLongLine which allows lines of any length.
 * @returns string read from file or undefined if end of file
 */
    ReadLine(): string;

/**
 * Reads a line from a file opened for reading by a File object. The line can be any length. If your file has lines shorter than 512 characters then you may want to use ReadLine instead which is faster.
 * @returns string read from file or undefined if end of file
 */
    ReadLongLine(): string;

/**
 * Rename an existing file to have a different name.
 * @param oldname Existing filename you want to rename
 * @param newname New filename you want to rename to
 * @returns true if successful, false if not.
 */
    static Rename(oldname: string, newname: string): boolean;

/**
 * Set the current position for reading or writing in a File object.
 * @param offset Offset to seek to in the file
 * @param origin Origin for offset. Must be one of File.START, File.END or File.CURRENT. If omitted File.START will be used.
 */
    Seek(offset: number, origin?: number): void;

/**
 * Return the size of a file in bytes
 * @param filename Filename you want the size of.
 * @returns size in bytes
 */
    static Size(filename: string): number;

/**
 * Return the current file position for a File object. Note that on Windows when reading files if the file is not opened with File.BINARY this may not return the correct file position for files with unix line endings.
 * @returns integer
 */
    Tell(): number;

/**
 * Uploads a file to a remote location. See also File.Proxy(), File.ProxyPassword() and File.ProxyUsername().
 * @param filename Filename you want to upload.
 * @param url URL (uniform resource locator) of the remote location you want to upload the file to. Currently only http is supported. Give the full address including the leading 'http://'. e.g. 'http://www.example.com/file.html'.
 * @param options Options for upload. If both of these are set then basic authorization using the username and password will be used.
 * @returns true if file was successfully uploaded, false otherwise.
 */
    static Upload(filename: string, url: string, options?: UploadArgument_options): boolean;

/**
 * Write a string to a file opened for writing by a File object. Note that a carriage return is not added.
 * @param string The string/item that you want to write
 */
    Write(string: any): void;

/**
 * Writes binary data to a file opened for writing by a File object. The data to write is an ArrayBuffer object. For more details on how to use an ArrayBuffer see the following links: https://developer.mozilla.org/en-US/docs/Web/JavaScript/Typed_arrays https://developer.mozilla.org/en-US/docs/Web/JavaScript/Reference/Global_Objects/ArrayBuffer https://developer.mozilla.org/en-US/docs/Web/JavaScript/Reference/Global_Objects/TypedArray https://developer.mozilla.org/en-US/docs/Web/JavaScript/Reference/Global_Objects/DataView.
 * @param buffer ArrayBuffer to write to file
 * @param length Number of bytes to write to the file. If omitted all the data in the ArrayBuffer will be written (buffer.byteLength bytes)
 */
    WriteArrayBuffer(buffer: ArrayBuffer, length?: number): void;

/**
 * Write a string to a file opened for writing by a File object adding a carriage return.
 * @param string The string/item that you want to write
 */
    Writeln(string: any): void;

/**
 * Create a new File object for reading and writing text files.
 * @param filename Filename of the file you want to read/write. If reading, the file must exist. If writing, the file will be overwritten (if it exists) if mode is File.WRITE, or if mode is File.APPEND it will be appended to if it exists, or created if it does not. When reading a file the filename can also be a URL (uniform resource locator) in which case the file will be read from the remote site. See File.Get() for more details on the format of the URL.
 * @param mode The mode to open the file with. Can be File.READ, File.WRITE or File.APPEND. For File.WRITE or File.APPEND it can also be ORed with File.BINARY if required. By default text is read and written as ASCII. To read/write text in utf-8 mode can also be ORed with File.UTF8 if required.
 * @returns File object
 */
    constructor(filename: string, mode: number);

/** Name of the file */
    readonly filename: string;
/** Mode the file was opened with (File.READ, File.WRITE etc) */
    readonly mode: number;
/** Flag to open file for appending */
    static APPEND: number;
/** Flag to open file in binary mode. This will have no effect on unix/linux but for windows if a file is opened for writing with binary mode \n will not be translated to \r\n (CRLF), it will be written as \n (LF) */
    static BINARY: number;
/** Seek relative to current file position */
    static CURRENT: number;
/** Find directories */
    static DIRECTORY: number;
/** Seek relative to end of the file */
    static END: number;
/** Find files */
    static FILE: number;
/** Flag to open file for reading */
    static READ: number;
/** Seek relative to start of the file */
    static START: number;
/** Flag to open file for reading as UTF-8 encoding. */
    static UTF8: number;
/** Flag to open file for writing */
    static WRITE: number;
}

declare class FreqFRF {
/**
 * Allows user to add new subcase cards in *FREQUENCY_DOMAIN_SSFRF. This method is only applicable when option is FreqFRF.SUBCASE. The new cards have uninitialised fields and should be updated by FreqFRF.SetSubcaseData().
 * @returns Index of the new subcase
 */
    AddSubcaseData(): number;

/**
 * Associates a comment with a *FREQUENCY_DOMAIN_FRF.
 * @param Comment Comment that will be attached to the *FREQUENCY_DOMAIN_FRF
 */
    AssociateComment(Comment: Comment): void;

/**
 * Starts an edit panel in Browse mode.
 * @param modal If this window is modal (blocks the user from doing anything else in PRIMER until this window is dismissed). If omitted the window will be modal.
 */
    Browse(modal?: boolean): void;

/**
 * Clears a flag on the *FREQUENCY_DOMAIN_FRF.
 * @param flag Flag to clear on the *FREQUENCY_DOMAIN_FRF
 */
    ClearFlag(flag: number): void;

/**
 * Copies the *FREQUENCY_DOMAIN_FRF. The target include of the copied *FREQUENCY_DOMAIN_FRF can be set using Options.copy_target_include.
 * @param range If you want to keep the copied item in the range specified for the current include. Default value is false. To set current include, use Include.MakeCurrentLayer().
 * @returns FreqFRF object
 */
    Copy(range?: boolean): FreqFRF;

/**
 * Detaches a comment from a *FREQUENCY_DOMAIN_FRF.
 * @param Comment Comment that will be detached from the *FREQUENCY_DOMAIN_FRF
 */
    DetachComment(Comment: Comment): void;

/**
 * Starts an interactive editing panel.
 * @param modal If this window is modal (blocks the user from doing anything else in PRIMER until this window is dismissed). If omitted the window will be modal.
 */
    Edit(modal?: boolean): void;

/**
 * Adds an error for *FREQUENCY_DOMAIN_FRF. For more details on checking see the Check class.
 * @param message The error message to give
 * @param details An optional detailed error message
 */
    Error(message: string, details?: string): void;

/**
 * Returns the first *FREQUENCY_DOMAIN_FRF in the model.
 * @param Model Model to get first *FREQUENCY_DOMAIN_FRF in
 * @returns FreqFRF object (or null if there are no *FREQUENCY_DOMAIN_FRFs in the model).
 */
    static First(Model: Model): FreqFRF;

/**
 * Flags all of the *FREQUENCY_DOMAIN_FRFs in the model with a defined flag.
 * @param Model Model that all *FREQUENCY_DOMAIN_FRFs will be flagged in
 * @param flag Flag to set on the *FREQUENCY_DOMAIN_FRFs
 */
    static FlagAll(Model: Model, flag: number): void;

/**
 * Checks if the *FREQUENCY_DOMAIN_FRF is flagged or not.
 * @param flag Flag to test on the *FREQUENCY_DOMAIN_FRF
 * @returns true if flagged, false if not.
 */
    Flagged(flag: number): boolean;

/**
 * Calls a function for each *FREQUENCY_DOMAIN_FRF in the model. Note that ForEach has been designed to make looping over *FREQUENCY_DOMAIN_FRFs as fast as possible and so has some limitations. Firstly, a single temporary FreqFRF object is created and on each function call it is updated with the current *FREQUENCY_DOMAIN_FRF data. This means that you should not try to store the FreqFRF object for later use (e.g. in an array) as it is temporary. Secondly, you cannot create new *FREQUENCY_DOMAIN_FRFs inside a ForEach loop. 
 * @param Model Model that all *FREQUENCY_DOMAIN_FRFs are in
 * @param func Function to call for each *FREQUENCY_DOMAIN_FRF
 * @param extra An optional extra object/array/string etc that will appended to arguments when calling the function
 */
    static ForEach(Model: Model, func: () => void, extra?: any): void;

/**
 * Returns an array of FreqFRF objects or properties for all of the *FREQUENCY_DOMAIN_FRFs in a model in PRIMER. If the optional property argument is not given then an array of FreqFRF objects is returned. If the property argument is given, that property value for each *FREQUENCY_DOMAIN_FRF is returned in the array instead of a FreqFRF object
 * @param Model Model to get *FREQUENCY_DOMAIN_FRFs from
 * @param property Name for property to get for all *FREQUENCY_DOMAIN_FRFs in the model
 * @returns Array of FreqFRF objects or properties
 */
    static GetAll(Model: Model, property?: string): FreqFRF[];

/**
 * Extracts the comments associated to a *FREQUENCY_DOMAIN_FRF.
 * @returns Array of Comment objects (or null if there are no comments associated to the node).
 */
    GetComments(): Comment[];

/**
 * Returns an array of FreqFRF objects for all of the flagged *FREQUENCY_DOMAIN_FRFs in a model in PRIMER If the optional property argument is not given then an array of FreqFRF objects is returned. If the property argument is given, then that property value for each *FREQUENCY_DOMAIN_FRF is returned in the array instead of a FreqFRF object
 * @param Model Model to get *FREQUENCY_DOMAIN_FRFs from
 * @param flag Flag set on the *FREQUENCY_DOMAIN_FRFs that you want to retrieve
 * @param property Name for property to get for all flagged *FREQUENCY_DOMAIN_FRFs in the model
 * @returns Array of FreqFRF objects or properties
 */
    static GetFlagged(Model: Model, flag: number, property?: string): FreqFRF[];

/**
 * Returns the FreqFRF object for a *FREQUENCY_DOMAIN_FRF ID.
 * @param Model Model to find the *FREQUENCY_DOMAIN_FRF in
 * @param number number of the *FREQUENCY_DOMAIN_FRF you want the FreqFRF object for
 * @returns FreqFRF object (or null if *FREQUENCY_DOMAIN_FRF does not exist).
 */
    static GetFromID(Model: Model, number: number): FreqFRF;

/**
 * Checks if a FreqFRF property is a parameter or not. Note that object properties that are parameters are normally returned as the integer or float parameter values as that is virtually always what the user would want. For this function to work the JavaScript interpreter must use the parameter name instead of the value. This can be done by setting the Options.property_parameter_names option to true before calling the function and then resetting it to false afterwards.. This behaviour can also temporarily be switched by using the FreqFRF.ViewParameters() method and 'method chaining' (see the examples below).
 * @param prop *FREQUENCY_DOMAIN_FRF property to get parameter for
 * @returns Parameter object if property is a parameter, null if not.
 */
    GetParameter(prop: string): Parameter;

/**
 * Returns the data for a specific subcase as an array. For each subcase there will be 13 values when vad1=12 else 11 values . There can be as many subcases as needed. This method is only applicable when option is FreqFRF.SUBCASE.
 * @param index Index you want the subcase data for. Note that indices start at 0.
 * @returns An array containing the subcase data (values: title[string], n1[integer], n1typ[integer], n1typ[integer], dof1[integer], vad1[integer], vid1[integer], n2[integer], n2typ[integer],dof2[integer], vad2[integer], vid2[integer], n11[integer], n11typ[integer])  n11 and n11typ are present only when vad1 =12
 */
    GetSubcaseData(index: number): number[];

/**
 * Returns the keyword for this *FREQUENCY_DOMAIN_FRF Note that a carriage return is not added. See also FreqFRF.KeywordCards()
 * @returns string containing the keyword.
 */
    Keyword(): string;

/**
 * Returns the keyword cards for the *FREQUENCY_DOMAIN_FRF. Note that a carriage return is not added. See also FreqFRF.Keyword()
 * @returns string containing the cards.
 */
    KeywordCards(): string;

/**
 * Returns the last *FREQUENCY_DOMAIN_FRF in the model.
 * @param Model Model to get last *FREQUENCY_DOMAIN_FRF in
 * @returns FreqFRF object (or null if there are no *FREQUENCY_DOMAIN_FRFs in the model).
 */
    static Last(Model: Model): FreqFRF;

/**
 * Returns the next *FREQUENCY_DOMAIN_FRF in the model.
 * @returns FreqFRF object (or null if there are no more *FREQUENCY_DOMAIN_FRFs in the model).
 */
    Next(): FreqFRF;

/**
 * Returns the previous *FREQUENCY_DOMAIN_FRF in the model.
 * @returns FreqFRF object (or null if there are no more *FREQUENCY_DOMAIN_FRFs in the model).
 */
    Previous(): FreqFRF;

/**
 * Allows the user to select *FREQUENCY_DOMAIN_FRFs using standard PRIMER object menus.
 * @param flag Flag to use when selecting *FREQUENCY_DOMAIN_FRFs
 * @param prompt Text to display as a prompt to the user
 * @param limit If the argument is a Model then only *FREQUENCY_DOMAIN_FRFs from that model can be selected. If the argument is a Flag then only *FREQUENCY_DOMAIN_FRFs that are flagged with limit can be selected (limit should be different to flag). If omitted, or null, any *FREQUENCY_DOMAIN_FRFs can be selected. from any model.
 * @param modal If selection is modal (blocks the user from doing anything else in PRIMER until this window is dismissed). If omitted the selection will be modal.
 * @returns Number of *FREQUENCY_DOMAIN_FRFs selected or null if menu cancelled
 */
    static Select(flag: number, prompt: string, limit?: Model | number, modal?: boolean): number;

/**
 * Sets a flag on the *FREQUENCY_DOMAIN_FRF.
 * @param flag Flag to set on the *FREQUENCY_DOMAIN_FRF
 */
    SetFlag(flag: number): void;

/**
 * Set the data for a specific subcase. There can be as many subcases as needed. This method is only applicable when option is FreqFRF.SUBCASE.
 * @param index Index you want to set subcase data for. Note that indices start at 0.
 * @param title A description of the current subcase (can be blank).
 * @param vad1 Value of vad1.
 * @param data An array containing the subcase data (values: n1[integer], n1typ[integer], dof1[integer], vad1[integer], vid1[integer], n2[integer], n2typ[integer], dof2[integer], vad2[integer], vid2[integer])  For vad1=12 Extra 2 arguments to be given n1[integer], n11typ[integer] 
 */
    SetSubcaseData(index: number, title: string, vad1: number, data: number[]): void;

/**
 * Returns the total number of *FREQUENCY_DOMAIN_FRFs in the model.
 * @param Model Model to get total for
 * @param exists true if only existing *FREQUENCY_DOMAIN_FRFs should be counted. If false or omitted referenced but undefined *FREQUENCY_DOMAIN_FRFs will also be included in the total.
 * @returns number of *FREQUENCY_DOMAIN_FRFs
 */
    static Total(Model: Model, exists?: boolean): number;

/**
 * Unsets a defined flag on all of the *FREQUENCY_DOMAIN_FRFs in the model.
 * @param Model Model that the defined flag for all *FREQUENCY_DOMAIN_FRFs will be unset in
 * @param flag Flag to unset on the *FREQUENCY_DOMAIN_FRFs
 */
    static UnflagAll(Model: Model, flag: number): void;

/**
 * Object properties that are parameters are normally returned as the integer or float parameter values as that is virtually always what the user would want. This function temporarily changes the behaviour so that if a property is a parameter the parameter name is returned instead. This can be used with 'method chaining' (see the example below) to make sure a property argument is correct.
 * @returns FreqFRF object.
 */
    ViewParameters(): FreqFRF;

/**
 * Adds a warning for *FREQUENCY_DOMAIN_FRF. For more details on checking see the Check class.
 * @param message The warning message to give
 * @param details An optional detailed warning message
 */
    Warning(message: string, details?: string): void;

/**
 * Returns the cross references for this *FREQUENCY_DOMAIN_FRF.
 * @returns Xrefs object.
 */
    Xrefs(): Xrefs;

/**
 * Create a new FreqFRF object.
 * @param Model Model that *FREQUENCY_DOMAIN_FRF will be created in.
 * @param option Specify the type of *FREQUENCY_DOMAIN_FRF. Can be FreqFRF.BLANK, FreqFRF.SUBCASE.
 */
    constructor(Model: Model, option: number);

/**
 * Creates a string containing the *FREQUENCY_DOMAIN_FRF data in keyword format. Note that this contains the keyword header and the keyword cards. See also FreqFRF.Keyword() and FreqFRF.KeywordCards().
 * @returns string
 */
    toString(): string;

/** Modal damping coefficient. */
    dampf: number;
/** Mass proportional damping constant in Rayleigh damping. */
    dmpmas: number;
/** Stiffness proportional damping constant in Rayleigh damping. */
    dmpstf: number;
/** Applicable degrees-of-freedom for excitation input (ignored if vad1 = 4). */
    dof1: number;
/** Applicable degrees-of-freedom for response output. */
    dof2: number;
/** true if *FREQUENCY_DOMAIN_FRF exists, false if referred to but not defined. */
    readonly exists: boolean;
/** Maximum frequency for FRF output (cycles/time). */
    fmax: number;
/** Minimum frequency for FRF output (cycles/time). */
    fmin: number;
/** Optional maximum natural frequency employed in FRF computation. */
    fnmax: number;
/** Frequency spacing option for FRF output. */
    fspace: number;
/** The Include file number that the *FREQUENCY_DOMAIN_FRF is in. */
    include: number;
/** The label the *FREQUENCY_DOMAIN_FRF has in PRIMER. */
    readonly label: number;
/** Load Curve ID defining mode dependent modal damping coefficient. */
    lcdam: number;
/** Load Curve ID defining the frequencies for FRF output. */
    lcfreq: number;
/** Type of load curve defining modal damping coefficient. */
    lctyp: number;
/** The last mode employed in FRF computation (optional). */
    mdmax: number;
/** The first mode employed in FRF computation (optional). */
    mdmin: number;
/** The Model number that the *FREQUENCY_DOMAIN_FRF is in. */
    readonly model: number;
/** Node/ Node Set/ Segment Set ID for excitation input. */
    n1: number;
/** Node/ Node Set/ Segment Set ID for excitation input. */
    n11: number;
/** Type of n11. Values can be FreqFRF.NODE, FreqFRF.NODE_SET, FreqFRF.SEGMENT_SET, FreqFRF.IGA_POINT_UVW, FreqFRF.IGA_POINT_UVW_SET, FreqFRF.IGA_EDGE_UVW, FreqFRF.IGA_EDGE_UVW_SET, FreqFRF.IGA_FACE_XYZ or FreqFRF.IGA_FACE_XYZ_SET */
    n11typ: number;
/** Type of n1. Values can be FreqFRF.NODE, FreqFRF.NODE_SET, FreqFRF.SEGMENT_SET. FreqFRF.IGA_POINT_UVW, FreqFRF.IGA_POINT_UVW_SET, FreqFRF.IGA_EDGE_UVW, FreqFRF.IGA_EDGE_UVW_SET, FreqFRF.IGA_FACE_XYZ or FreqFRF.IGA_FACE_XYZ_SET */
    n1typ: number;
/** Node/ Node Set/ Segment Set ID for response output. */
    n2: number;
/** Type of n2. Values can be FreqFRF.NODE, FreqFRF.NODE_SET , FreqFRF.SEGMENT_SET, FreqFRF.IGA_POINT_UVW or FreqFRF.IGA_POINT_UVW_SET, */
    n2typ: number;
/** Number of subcases. */
    ncases: number;
/** Number of frequencies for FRF output. */
    nfreq: number;
/** The *FREQUENCY_DOMAIN_FRF option. Can be FreqFRF.BLANK or FreqFRF.SUBCASE */
    option: number;
/** Output option. */
    output: number;
/** Flag for displacement, velocity and acceleration results. */
    relatv: number;
/** Restart option. */
    restrt: number;
/** Excitation input type. */
    vad1: number;
/** Response output type. */
    vad2: number;
/** Vector ID (for dof1 = 0) for excitation input. */
    vid1: number;
/** Vector ID (for dof2 = 0) for response direction. */
    vid2: number;
/** FreqFRF option for *FREQUENCY_DOMAIN_FRF_BLANK. */
    static BLANK: number;
/** Property n1typ/ n11typ value EQ.5.0: n1/ n11 is IGA edge UVW ID. */
    static IGA_EDGE_UVW: number;
/** Property n1typ/ n11typ value EQ.6.0: n1/ n11 is IGA edge UVW Set ID. */
    static IGA_EDGE_UVW_SET: number;
/** Property n1typ/ n11typ value EQ.7.0: n1/ n11 is IGA face XYZ Set ID. */
    static IGA_FACE_XYZ: number;
/** Property n1typ/ n11typ value EQ.8.0: n1/ n11 is IGA face XYZ Set ID. */
    static IGA_FACE_XYZ_SET: number;
/** Property n1typ/ n11typ value EQ.3.0: n2 is IGA point UVW ID. */
    static IGA_POINT_UVW: number;
/** Property n1typ/ n11typ value EQ.4.0: n2 is IGA point UVW Set ID. */
    static IGA_POINT_UVW_SET: number;
/** Property n1typ/ n2typ value EQ.0.0: n2 is Node ID (default). */
    static NODE: number;
/** Property n1typ/ n11typ value EQ.1.0: n2 is Node Set ID. */
    static NODE_SET: number;
/** Property n1typ/ n11typ value EQ.2.0: n2 is Segment Set ID. */
    static SEGMENT_SET: number;
/** FreqFRF option for *FREQUENCY_DOMAIN_FRF_SUBCASE. */
    static SUBCASE: number;
}

declare class FreqSSD {
/**
 * Allows user to add a new load card in *FREQUENCY_DOMAIN_SSD. This method is only applicable when option is not FreqSSD.SUBCASE. The new card has uninitialised fields and should be updated by FreqSSD.SetLoadData().
 * @returns Index of the new load
 */
    AddLoadData(): number;

/**
 * Allows user to add new subcase cards in *FREQUENCY_DOMAIN_SSD. This method is only applicable when option is FreqSSD.SUBCASE. The new cards have uninitialised fields and should be updated by FreqSSD.SetSubcaseData().
 * @returns Index of the new subcase
 */
    AddSubcaseData(): number;

/**
 * Associates a comment with a *FREQUENCY_DOMAIN_SSD.
 * @param Comment Comment that will be attached to the *FREQUENCY_DOMAIN_SSD
 */
    AssociateComment(Comment: Comment): void;

/**
 * Starts an edit panel in Browse mode.
 * @param modal If this window is modal (blocks the user from doing anything else in PRIMER until this window is dismissed). If omitted the window will be modal.
 */
    Browse(modal?: boolean): void;

/**
 * Clears a flag on the *FREQUENCY_DOMAIN_SSD.
 * @param flag Flag to clear on the *FREQUENCY_DOMAIN_SSD
 */
    ClearFlag(flag: number): void;

/**
 * Copies the *FREQUENCY_DOMAIN_SSD. The target include of the copied *FREQUENCY_DOMAIN_SSD can be set using Options.copy_target_include.
 * @param range If you want to keep the copied item in the range specified for the current include. Default value is false. To set current include, use Include.MakeCurrentLayer().
 * @returns FreqSSD object
 */
    Copy(range?: boolean): FreqSSD;

/**
 * Starts an interactive editing panel to create a *FREQUENCY_DOMAIN_SSD
 * @param Model Model that the *FREQUENCY_DOMAIN_SSD will be created in.
 * @param modal If this window is modal (blocks the user from doing anything else in PRIMER until this window is dismissed). If omitted the window will be modal.
 * @returns FreqSSD object (or null if not made).
 */
    static Create(Model: Model, modal?: boolean): FreqSSD;

/**
 * Detaches a comment from a *FREQUENCY_DOMAIN_SSD.
 * @param Comment Comment that will be detached from the *FREQUENCY_DOMAIN_SSD
 */
    DetachComment(Comment: Comment): void;

/**
 * Starts an interactive editing panel.
 * @param modal If this window is modal (blocks the user from doing anything else in PRIMER until this window is dismissed). If omitted the window will be modal.
 */
    Edit(modal?: boolean): void;

/**
 * Adds an error for *FREQUENCY_DOMAIN_SSD. For more details on checking see the Check class.
 * @param message The error message to give
 * @param details An optional detailed error message
 */
    Error(message: string, details?: string): void;

/**
 * Returns the first *FREQUENCY_DOMAIN_SSD in the model.
 * @param Model Model to get first *FREQUENCY_DOMAIN_SSD in
 * @returns FreqSSD object (or null if there are no *FREQUENCY_DOMAIN_SSDs in the model).
 */
    static First(Model: Model): FreqSSD;

/**
 * Flags all of the *FREQUENCY_DOMAIN_SSDs in the model with a defined flag.
 * @param Model Model that all *FREQUENCY_DOMAIN_SSDs will be flagged in
 * @param flag Flag to set on the *FREQUENCY_DOMAIN_SSDs
 */
    static FlagAll(Model: Model, flag: number): void;

/**
 * Checks if the *FREQUENCY_DOMAIN_SSD is flagged or not.
 * @param flag Flag to test on the *FREQUENCY_DOMAIN_SSD
 * @returns true if flagged, false if not.
 */
    Flagged(flag: number): boolean;

/**
 * Calls a function for each *FREQUENCY_DOMAIN_SSD in the model. Note that ForEach has been designed to make looping over *FREQUENCY_DOMAIN_SSDs as fast as possible and so has some limitations. Firstly, a single temporary FreqSSD object is created and on each function call it is updated with the current *FREQUENCY_DOMAIN_SSD data. This means that you should not try to store the FreqSSD object for later use (e.g. in an array) as it is temporary. Secondly, you cannot create new *FREQUENCY_DOMAIN_SSDs inside a ForEach loop. 
 * @param Model Model that all *FREQUENCY_DOMAIN_SSDs are in
 * @param func Function to call for each *FREQUENCY_DOMAIN_SSD
 * @param extra An optional extra object/array/string etc that will appended to arguments when calling the function
 */
    static ForEach(Model: Model, func: () => void, extra?: any): void;

/**
 * Returns an array of FreqSSD objects or properties for all of the *FREQUENCY_DOMAIN_SSDs in a model in PRIMER. If the optional property argument is not given then an array of FreqSSD objects is returned. If the property argument is given, that property value for each *FREQUENCY_DOMAIN_SSD is returned in the array instead of a FreqSSD object
 * @param Model Model to get *FREQUENCY_DOMAIN_SSDs from
 * @param property Name for property to get for all *FREQUENCY_DOMAIN_SSDs in the model
 * @returns Array of FreqSSD objects or properties
 */
    static GetAll(Model: Model, property?: string): FreqSSD[];

/**
 * Extracts the comments associated to a *FREQUENCY_DOMAIN_SSD.
 * @returns Array of Comment objects (or null if there are no comments associated to the node).
 */
    GetComments(): Comment[];

/**
 * Returns the ERP data for a specific ERP part as an array. For each ERP part there will be 2 values. There are nerp ERP parts. This method is only applicable when option is FreqSSD.ERP.
 * @param index Index you want the ERP data for. Note that indices start at 0.
 * @returns An array containing the ERP data (values: pid[integer], ptyp[integer]). The array length will be 2.
 */
    GetErpData(index: number): number[];

/**
 * Returns an array of FreqSSD objects for all of the flagged *FREQUENCY_DOMAIN_SSDs in a model in PRIMER If the optional property argument is not given then an array of FreqSSD objects is returned. If the property argument is given, then that property value for each *FREQUENCY_DOMAIN_SSD is returned in the array instead of a FreqSSD object
 * @param Model Model to get *FREQUENCY_DOMAIN_SSDs from
 * @param flag Flag set on the *FREQUENCY_DOMAIN_SSDs that you want to retrieve
 * @param property Name for property to get for all flagged *FREQUENCY_DOMAIN_SSDs in the model
 * @returns Array of FreqSSD objects or properties
 */
    static GetFlagged(Model: Model, flag: number, property?: string): FreqSSD[];

/**
 * Returns the FreqSSD object for a *FREQUENCY_DOMAIN_SSD ID.
 * @param Model Model to find the *FREQUENCY_DOMAIN_SSD in
 * @param number number of the *FREQUENCY_DOMAIN_SSD you want the FreqSSD object for
 * @returns FreqSSD object (or null if *FREQUENCY_DOMAIN_SSD does not exist).
 */
    static GetFromID(Model: Model, number: number): FreqSSD;

/**
 * Returns the data for a specific excitation load as an array. For each load there will be 8 values. There can be as many loads as needed. This method is only applicable when option is not FreqSSD.SUBCASE.
 * @param index Index you want the load data for. Note that indices start at 0.
 * @returns An array containing the load data (values: nid[integer], ntyp[integer], dof[integer], vad[integer], lc1[integer], lc2[integer], sf[real], vid[integer]). The array length will be 8.
 */
    GetLoadData(index: number): number[];

/**
 * Checks if a FreqSSD property is a parameter or not. Note that object properties that are parameters are normally returned as the integer or float parameter values as that is virtually always what the user would want. For this function to work the JavaScript interpreter must use the parameter name instead of the value. This can be done by setting the Options.property_parameter_names option to true before calling the function and then resetting it to false afterwards.. This behaviour can also temporarily be switched by using the FreqSSD.ViewParameters() method and 'method chaining' (see the examples below).
 * @param prop *FREQUENCY_DOMAIN_SSD property to get parameter for
 * @returns Parameter object if property is a parameter, null if not.
 */
    GetParameter(prop: string): Parameter;

/**
 * Returns the data for a specific subcase as an array. For each subcase there will be 3 + 8 x nload values. There can be as many subcases as needed. This method is only applicable when option is FreqSSD.SUBCASE.
 * @param index Index you want the subcase data for. Note that indices start at 0.
 * @returns An array containing the subcase data (values: caseid[string], title[string], nload[integer], nid[integer], ntyp[integer], dof[integer], vad[integer], lc1[integer], lc2[integer], sf[real], vid[integer], ...)  Where values nid to vid are repeated nload times in the array. The array length will be 3 + 8 x nload.
 */
    GetSubcaseData(index: number): number[];

/**
 * Returns the keyword for this *FREQUENCY_DOMAIN_SSD. Note that a carriage return is not added. See also FreqSSD.KeywordCards()
 * @returns string containing the keyword.
 */
    Keyword(): string;

/**
 * Returns the keyword cards for the *FREQUENCY_DOMAIN_SSD. Note that a carriage return is not added. See also FreqSSD.Keyword()
 * @returns string containing the cards.
 */
    KeywordCards(): string;

/**
 * Returns the last *FREQUENCY_DOMAIN_SSD in the model.
 * @param Model Model to get last *FREQUENCY_DOMAIN_SSD in
 * @returns FreqSSD object (or null if there are no *FREQUENCY_DOMAIN_SSDs in the model).
 */
    static Last(Model: Model): FreqSSD;

/**
 * Returns the next *FREQUENCY_DOMAIN_SSD in the model.
 * @returns FreqSSD object (or null if there are no more *FREQUENCY_DOMAIN_SSDs in the model).
 */
    Next(): FreqSSD;

/**
 * Returns the previous *FREQUENCY_DOMAIN_SSD in the model.
 * @returns FreqSSD object (or null if there are no more *FREQUENCY_DOMAIN_SSDs in the model).
 */
    Previous(): FreqSSD;

/**
 * Allows user to remove a specified load card in *FREQUENCY_DOMAIN_SSD. This method is only applicable when option is not FreqSSD.SUBCASE.
 * @param index Index of the load card you want to remove. Note that indices start at 0.
 */
    RemoveLoadData(index: number): void;

/**
 * Allows user to remove cards for a specified subcase in *FREQUENCY_DOMAIN_SSD. This method is only applicable when option is FreqSSD.SUBCASE.
 * @param index Index of the subcase you want to remove cards for. Note that indices start at 0.
 */
    RemoveSubcaseData(index: number): void;

/**
 * Allows the user to select *FREQUENCY_DOMAIN_SSDs using standard PRIMER object menus.
 * @param flag Flag to use when selecting *FREQUENCY_DOMAIN_SSDs
 * @param prompt Text to display as a prompt to the user
 * @param limit If the argument is a Model then only *FREQUENCY_DOMAIN_SSDs from that model can be selected. If the argument is a Flag then only *FREQUENCY_DOMAIN_SSDs that are flagged with limit can be selected (limit should be different to flag). If omitted, or null, any *FREQUENCY_DOMAIN_SSDs can be selected. from any model.
 * @param modal If selection is modal (blocks the user from doing anything else in PRIMER until this window is dismissed). If omitted the selection will be modal.
 * @returns Number of *FREQUENCY_DOMAIN_SSDs selected or null if menu cancelled
 */
    static Select(flag: number, prompt: string, limit?: Model | number, modal?: boolean): number;

/**
 * Set the data for a specific ERP part. For each ERP part there will be 2 values. There are nerp ERP parts. This method is only applicable when option is FreqSSD.ERP.
 * @param index Index you want to set ERP data for. Note that indices start at 0.
 * @param data An array containing the ERP data (values: pid[integer], ptyp[integer]). The array length should be 2.
 */
    SetErpData(index: number, data: number[]): void;

/**
 * Sets a flag on the *FREQUENCY_DOMAIN_SSD.
 * @param flag Flag to set on the *FREQUENCY_DOMAIN_SSD
 */
    SetFlag(flag: number): void;

/**
 * Set the data for a specific excitation load. For each load there will be 8 values. There can be as many loads as needed. This method is only applicable when option is not FreqSSD.SUBCASE.
 * @param index Index you want to set load data for. Note that indices start at 0.
 * @param data An array containing the load data (values: nid[integer], ntyp[integer], dof[integer], vad[integer], lc1[integer], lc2[integer], sf[real], vid[integer]). The array length should be 8.
 */
    SetLoadData(index: number, data: number[]): void;

/**
 * Set the data for a specific subcase. For each subcase, data will have 8 x nload values. There can be as many subcases as needed. This method is only applicable when option is FreqSSD.SUBCASE.
 * @param index Index you want to set subcase data for. Note that indices start at 0.
 * @param caseid Identification string to be used as the case ID (must include at least one letter).
 * @param title A description of the current loading case (can be blank).
 * @param nload Number of loads for this loading case.
 * @param data An array containing the subcase load data (values: nid[integer], ntyp[integer], dof[integer], vad[integer], lc1[integer], lc2[integer], sf[real], vid[integer], ...)  Where values nid to vid are repeated nload times in the array. The array length should be 8 x nload.
 */
    SetSubcaseData(index: number, caseid: string, title: string, nload: number, data: number[]): void;

/**
 * Returns the total number of *FREQUENCY_DOMAIN_SSDs in the model.
 * @param Model Model to get total for
 * @param exists true if only existing *FREQUENCY_DOMAIN_SSDs should be counted. If false or omitted referenced but undefined *FREQUENCY_DOMAIN_SSDs will also be included in the total.
 * @returns number of *FREQUENCY_DOMAIN_SSDs
 */
    static Total(Model: Model, exists?: boolean): number;

/**
 * Unsets a defined flag on all of the *FREQUENCY_DOMAIN_SSDs in the model.
 * @param Model Model that the defined flag for all *FREQUENCY_DOMAIN_SSDs will be unset in
 * @param flag Flag to unset on the *FREQUENCY_DOMAIN_SSDs
 */
    static UnflagAll(Model: Model, flag: number): void;

/**
 * Object properties that are parameters are normally returned as the integer or float parameter values as that is virtually always what the user would want. This function temporarily changes the behaviour so that if a property is a parameter the parameter name is returned instead. This can be used with 'method chaining' (see the example below) to make sure a property argument is correct.
 * @returns FreqSSD object.
 */
    ViewParameters(): FreqSSD;

/**
 * Adds a warning for *FREQUENCY_DOMAIN_SSD. For more details on checking see the Check class.
 * @param message The warning message to give
 * @param details An optional detailed warning message
 */
    Warning(message: string, details?: string): void;

/**
 * Returns the cross references for this *FREQUENCY_DOMAIN_SSD.
 * @returns Xrefs object.
 */
    Xrefs(): Xrefs;

/**
 * Create a new FreqSSD object.
 * @param Model Model that *FREQUENCY_DOMAIN_SSD will be created in.
 * @param option Specify the type of *FREQUENCY_DOMAIN_SSD. Can be FreqSSD.DIRECT, FreqSSD.DIRECT_FD, FreqSSD.FATIGUE, FreqSSD.FRF, FreqSSD.ERP or FreqSSD.SUBCASE.
 * @returns FreqSSD object
 */
    constructor(Model: Model, option: number);

/**
 * Creates a string containing the *FREQUENCY_DOMAIN_SSD data in keyword format. Note that this contains the keyword header and the keyword cards. See also FreqSSD.Keyword() and FreqSSD.KeywordCards().
 * @returns string
 */
    toString(): string;

/** Sound speed of the fluid (for option FreqSSD.ERP only). */
    c: number;
/** Modal damping coefficient. */
    dampf: number;
/** Damping flag. */
    dmpflg: number;
/** Mass proportional damping constant in Rayleigh damping. */
    dmpmas: number;
/** Stiffness proportional damping constant in Rayleigh damping. */
    dmpstf: number;
/** ERP reference value (for option FreqSSD.ERP only). */
    erpref: number;
/** ERP radiation loss factor (for option FreqSSD.ERP only). */
    erprlf: number;
/** true if *FREQUENCY_DOMAIN_SSD exists, false if referred to but not defined. */
    readonly exists: boolean;
/** The maximum natural frequency in modal superposition method (optional). */
    fnmax: number;
/** The minimum natural frequency in modal superposition method (optional). */
    fnmin: number;
/** The Include file number that the *FREQUENCY_DOMAIN_SSD is in. */
    include: number;
/** Stress computation flag (for option FreqSSD.DIRECT only). */
    istress: number;
/** The label the *FREQUENCY_DOMAIN_SSD has in PRIMER. */
    readonly label: number;
/** Load Curve ID defining mode dependent modal damping coefficient. */
    lcdam: number;
/** Load curve definition flag. */
    lcflag: number;
/** Load Curve ID defining duration of excitation for each frequency (for option FreqSSD.FATIGUE only). */
    lcftg: number;
/** Type of load curve defining modal damping coefficient. */
    lctyp: number;
/** The last mode in modal superposition method (optional). */
    mdmax: number;
/** The first mode in modal superposition method (optional). */
    mdmin: number;
/** Memory flag. */
    memory: number;
/** The Model number that the *FREQUENCY_DOMAIN_SSD is in. */
    readonly model: number;
/** Number of ERP panels. */
    nerp: number;
/** Type of NOUT. */
    notyp: number;
/** Part, part set, segment set, or node set ID for response output. */
    nout: number;
/** Response output type. */
    nova: number;
/** The *FREQUENCY_DOMAIN_SSD option. Can be FreqSSD.DIRECT, FreqSSD.DIRECT_FD, FreqSSD.FATIGUE, FreqSSD.FRF, FreqSSD.ERP or FreqSSD.SUBCASE. */
    option: number;
/** Radiation efficiency computation flag (for option FreqSSD.ERP only). */
    radeff: number;
/** Flag for displacement, velocity and acceleration results. */
    relatv: number;
/** Restart option. */
    restdp: number;
/** Restart option. */
    restmd: number;
/** Fluid density (for option FreqSSD.ERP only). */
    ro: number;
/** Stress used in fatigue analysis. */
    strtyp: number;
/** FreqSSD option for *FREQUENCY_DOMAIN_SSD_DIRECT. */
    static DIRECT: number;
/** FreqSSD option for *FREQUENCY_DOMAIN_SSD_DIRECT_FREQUENCY_DEPENDENT. */
    static DIRECT_FD: number;
/** FreqSSD option for *FREQUENCY_DOMAIN_SSD_ERP. */
    static ERP: number;
/** FreqSSD option for *FREQUENCY_DOMAIN_SSD_FATIGUE. */
    static FATIGUE: number;
/** FreqSSD option for *FREQUENCY_DOMAIN_SSD_FRF. */
    static FRF: number;
/** FreqSSD option for *FREQUENCY_DOMAIN_SSD_SUBCASE. */
    static SUBCASE: number;
}

declare class FreqVibration {
/**
 * Allows user to add a new Auto PSD load card in *FREQUENCY_DOMAIN_RANDOM_VIBRATION. The new card has uninitialised fields and should be updated by FreqVibration.SetAutoPsdLoadData().
 * @returns Index of the new auto PSD load
 */
    AddAutoPsdLoadData(): number;

/**
 * Allows user to add a new Cross PSD load card in *FREQUENCY_DOMAIN_RANDOM_VIBRATION. The new card has uninitialised fields and should be updated by FreqVibration.SetCrossPsdLoadData().
 * @returns Index of the new cross PSD load
 */
    AddCrossPsdLoadData(): number;

/**
 * Allows user to add new Initial Damage cards in *FREQUENCY_DOMAIN_RANDOM_VIBRATION. This method is only applicable when option is FreqVibration.FATIGUE. The new cards have uninitialised fields and should be updated by FreqVibration.SetInftgData().
 * @returns Index of the new initial damage card
 */
    AddInftgData(): number;

/**
 * Allows user to add new S-N curve cards in *FREQUENCY_DOMAIN_RANDOM_VIBRATION. This method is only applicable when option is FreqVibration.FATIGUE. The new cards have uninitialised fields and should be updated by FreqVibration.SetSNCurveData().
 * @returns Index of the new S-N curve card
 */
    AddSNCurveData(): number;

/**
 * Associates a comment with a *FREQUENCY_DOMAIN_RANDOM_VIBRATION.
 * @param Comment Comment that will be attached to the *FREQUENCY_DOMAIN_RANDOM_VIBRATION
 */
    AssociateComment(Comment: Comment): void;

/**
 * Starts an edit panel in Browse mode.
 * @param modal If this window is modal (blocks the user from doing anything else in PRIMER until this window is dismissed). If omitted the window will be modal.
 */
    Browse(modal?: boolean): void;

/**
 * Clears a flag on the *FREQUENCY_DOMAIN_RANDOM_VIBRATION.
 * @param flag Flag to clear on the *FREQUENCY_DOMAIN_RANDOM_VIBRATION
 */
    ClearFlag(flag: number): void;

/**
 * Copies the *FREQUENCY_DOMAIN_RANDOM_VIBRATION. The target include of the copied *FREQUENCY_DOMAIN_RANDOM_VIBRATION can be set using Options.copy_target_include.
 * @param range If you want to keep the copied item in the range specified for the current include. Default value is false. To set current include, use Include.MakeCurrentLayer().
 * @returns FreqVibration object
 */
    Copy(range?: boolean): FreqVibration;

/**
 * Starts an interactive editing panel to create a *FREQUENCY_DOMAIN_RANDOM_VIBRATION
 * @param Model Model that the *FREQUENCY_DOMAIN_RANDOM_VIBRATION will be created in.
 * @param modal If this window is modal (blocks the user from doing anything else in PRIMER until this window is dismissed). If omitted the window will be modal.
 * @returns FreqVibration object (or null if not made).
 */
    static Create(Model: Model, modal?: boolean): FreqVibration;

/**
 * Detaches a comment from a *FREQUENCY_DOMAIN_RANDOM_VIBRATION.
 * @param Comment Comment that will be detached from the *FREQUENCY_DOMAIN_RANDOM_VIBRATION
 */
    DetachComment(Comment: Comment): void;

/**
 * Starts an interactive editing panel.
 * @param modal If this window is modal (blocks the user from doing anything else in PRIMER until this window is dismissed). If omitted the window will be modal.
 */
    Edit(modal?: boolean): void;

/**
 * Adds an error for *FREQUENCY_DOMAIN_RANDOM_VIBRATION. For more details on checking see the Check class.
 * @param message The error message to give
 * @param details An optional detailed error message
 */
    Error(message: string, details?: string): void;

/**
 * Returns the first *FREQUENCY_DOMAIN_RANDOM_VIBRATION in the model.
 * @param Model Model to get first *FREQUENCY_DOMAIN_RANDOM_VIBRATION in
 * @returns FreqVibration object (or null if there are no *FREQUENCY_DOMAIN_RANDOM_VIBRATIONs in the model).
 */
    static First(Model: Model): FreqVibration;

/**
 * Flags all of the *FREQUENCY_DOMAIN_RANDOM_VIBRATIONs in the model with a defined flag.
 * @param Model Model that all *FREQUENCY_DOMAIN_RANDOM_VIBRATIONs will be flagged in
 * @param flag Flag to set on the *FREQUENCY_DOMAIN_RANDOM_VIBRATIONs
 */
    static FlagAll(Model: Model, flag: number): void;

/**
 * Checks if the *FREQUENCY_DOMAIN_RANDOM_VIBRATION is flagged or not.
 * @param flag Flag to test on the *FREQUENCY_DOMAIN_RANDOM_VIBRATION
 * @returns true if flagged, false if not.
 */
    Flagged(flag: number): boolean;

/**
 * Calls a function for each *FREQUENCY_DOMAIN_RANDOM_VIBRATION in the model. Note that ForEach has been designed to make looping over *FREQUENCY_DOMAIN_RANDOM_VIBRATIONs as fast as possible and so has some limitations. Firstly, a single temporary FreqVibration object is created and on each function call it is updated with the current *FREQUENCY_DOMAIN_RANDOM_VIBRATION data. This means that you should not try to store the FreqVibration object for later use (e.g. in an array) as it is temporary. Secondly, you cannot create new *FREQUENCY_DOMAIN_RANDOM_VIBRATIONs inside a ForEach loop. 
 * @param Model Model that all *FREQUENCY_DOMAIN_RANDOM_VIBRATIONs are in
 * @param func Function to call for each *FREQUENCY_DOMAIN_RANDOM_VIBRATION
 * @param extra An optional extra object/array/string etc that will appended to arguments when calling the function
 */
    static ForEach(Model: Model, func: () => void, extra?: any): void;

/**
 * Returns an array of FreqVibration objects or properties for all of the *FREQUENCY_DOMAIN_RANDOM_VIBRATIONs in a model in PRIMER. If the optional property argument is not given then an array of FreqVibration objects is returned. If the property argument is given, that property value for each *FREQUENCY_DOMAIN_RANDOM_VIBRATION is returned in the array instead of a FreqVibration object
 * @param Model Model to get *FREQUENCY_DOMAIN_RANDOM_VIBRATIONs from
 * @param property Name for property to get for all *FREQUENCY_DOMAIN_RANDOM_VIBRATIONs in the model
 * @returns Array of FreqVibration objects or properties
 */
    static GetAll(Model: Model, property?: string): FreqVibration[];

/**
 * Returns the Auto PSD load data for a specific Auto PSD Load definition as an array. For each Auto PSD load definition there will be 8 values. There are napsd Auto PSD load definitions.
 * @param index Index you want the Auto PSD load data for. Note that indices start at 0.
 * @returns An array containing the Auto PSD load data (values: sid[integer], stype[integer], dof[integer], ldpsd[integer], ldvel[integer], ldflw[integer], ldspn[integer], cid[integer]). The array length will be 8.
 */
    static GetAutoPsdLoadData(index: number): number[];

/**
 * Extracts the comments associated to a *FREQUENCY_DOMAIN_RANDOM_VIBRATION.
 * @returns Array of Comment objects (or null if there are no comments associated to the node).
 */
    GetComments(): Comment[];

/**
 * Returns the Cross PSD load data for a specific Cross PSD Load definition as an array. For each Cross PSD load definition there will be 5 values. There are ncpsd Cross PSD load definitions.
 * @param index Index you want the Cross PSD load data for. Note that indices start at 0.
 * @returns An array containing the Cross PSD load data (values: load_i[integer], load_j[integer], lctyp2[integer], ldpsd1[integer], ldpsd2[integer]). The array length will be 5.
 */
    static GetCrossPsdLoadData(index: number): number[];

/**
 * Returns an array of FreqVibration objects for all of the flagged *FREQUENCY_DOMAIN_RANDOM_VIBRATIONs in a model in PRIMER If the optional property argument is not given then an array of FreqVibration objects is returned. If the property argument is given, then that property value for each *FREQUENCY_DOMAIN_RANDOM_VIBRATION is returned in the array instead of a FreqVibration object
 * @param Model Model to get *FREQUENCY_DOMAIN_RANDOM_VIBRATIONs from
 * @param flag Flag set on the *FREQUENCY_DOMAIN_RANDOM_VIBRATIONs that you want to retrieve
 * @param property Name for property to get for all flagged *FREQUENCY_DOMAIN_RANDOM_VIBRATIONs in the model
 * @returns Array of FreqVibration objects or properties
 */
    static GetFlagged(Model: Model, flag: number, property?: string): FreqVibration[];

/**
 * Returns the FreqVibration object for a *FREQUENCY_DOMAIN_RANDOM_VIBRATION ID.
 * @param Model Model to find the *FREQUENCY_DOMAIN_RANDOM_VIBRATION in
 * @param number number of the *FREQUENCY_DOMAIN_RANDOM_VIBRATION you want the FreqVibration object for
 * @returns FreqVibration object (or null if *FREQUENCY_DOMAIN_RANDOM_VIBRATION does not exist).
 */
    static GetFromID(Model: Model, number: number): FreqVibration;

/**
 * Returns the path and filename of a binary database for fatigue information from a specific initial damage card. There are inftg filenames. This method is only applicable when option is FreqVibration.FATIGUE.
 * @param index Index of an initial damage card that you want the filename from. Note that indices start at 0.
 * @returns Return value from an initial damage card (values: filename[string])
 */
    static GetInftgData(index: number): string;

/**
 * Checks if a FreqVibration property is a parameter or not. Note that object properties that are parameters are normally returned as the integer or float parameter values as that is virtually always what the user would want. For this function to work the JavaScript interpreter must use the parameter name instead of the value. This can be done by setting the Options.property_parameter_names option to true before calling the function and then resetting it to false afterwards.. This behaviour can also temporarily be switched by using the FreqVibration.ViewParameters() method and 'method chaining' (see the examples below).
 * @param prop *FREQUENCY_DOMAIN_RANDOM_VIBRATION property to get parameter for
 * @returns Parameter object if property is a parameter, null if not.
 */
    GetParameter(prop: string): Parameter;

/**
 * Returns the data of a specific zone for fatigue analysis as an array. For each zone there will be 8 values. There are nftg zone definitions for fatigue analysis. This method is only applicable when option is FreqVibration.FATIGUE.
 * @param index Index you want the zone data for. Note that indices start at 0.
 * @returns An array containing the zone data (values: pid[integer], lcid[integer], ptype[integer], ltype[integer], a[float], b[float], sthres[float], snlimt[integer]). The array length will be 8.
 */
    static GetSNCurveData(index: number): number[];

/**
 * Returns the keyword for this *FREQUENCY_DOMAIN_RANDOM_VIBRATION. Note that a carriage return is not added. See also FreqVibration.KeywordCards()
 * @returns string containing the keyword.
 */
    Keyword(): string;

/**
 * Returns the keyword cards for the *FREQUENCY_DOMAIN_RANDOM_VIBRATION. Note that a carriage return is not added. See also FreqVibration.Keyword()
 * @returns string containing the cards.
 */
    KeywordCards(): string;

/**
 * Returns the last *FREQUENCY_DOMAIN_RANDOM_VIBRATION in the model.
 * @param Model Model to get last *FREQUENCY_DOMAIN_RANDOM_VIBRATION in
 * @returns FreqVibration object (or null if there are no *FREQUENCY_DOMAIN_RANDOM_VIBRATIONs in the model).
 */
    static Last(Model: Model): FreqVibration;

/**
 * Returns the next *FREQUENCY_DOMAIN_RANDOM_VIBRATION in the model.
 * @returns FreqVibration object (or null if there are no more *FREQUENCY_DOMAIN_RANDOM_VIBRATIONs in the model).
 */
    Next(): FreqVibration;

/**
 * Returns the previous *FREQUENCY_DOMAIN_RANDOM_VIBRATION in the model.
 * @returns FreqVibration object (or null if there are no more *FREQUENCY_DOMAIN_RANDOM_VIBRATIONs in the model).
 */
    Previous(): FreqVibration;

/**
 * Allows user to remove a specified Auto PSD load card in *FREQUENCY_DOMAIN_RANDOM_VIBRATION.
 * @param index Index of the auto PSD load card you want to remove. Note that indices start at 0.
 */
    RemoveAutoPsdLoadData(index: number): void;

/**
 * Allows user to remove a specified Cross PSD load card in *FREQUENCY_DOMAIN_RANDOM_VIBRATION.
 * @param index Index of the cross PSD load card you want to remove. Note that indices start at 0.
 */
    RemoveCrossPsdLoadData(index: number): void;

/**
 * Allows user to remove a specified Initial Damage card in *FREQUENCY_DOMAIN_RANDOM_VIBRATION. This method is only applicable when option is FreqVibration.FATIGUE.
 * @param index Index of the Initrial Damage card you want to remove. Note that indices start at 0.
 */
    RemoveInftgData(index: number): void;

/**
 * Allows user to remove a specified S-N curve card in *FREQUENCY_DOMAIN_RANDOM_VIBRATION. This method is only applicable when option is FreqVibration.FATIGUE.
 * @param index Index of the S-N curve card you want to remove. Note that indices start at 0.
 */
    RemoveSNCurveData(index: number): void;

/**
 * Allows the user to select *FREQUENCY_DOMAIN_RANDOM_VIBRATIONs using standard PRIMER object menus.
 * @param flag Flag to use when selecting *FREQUENCY_DOMAIN_RANDOM_VIBRATIONs
 * @param prompt Text to display as a prompt to the user
 * @param limit If the argument is a Model then only *FREQUENCY_DOMAIN_RANDOM_VIBRATIONs from that model can be selected. If the argument is a Flag then only *FREQUENCY_DOMAIN_RANDOM_VIBRATIONs that are flagged with limit can be selected (limit should be different to flag). If omitted, or null, any *FREQUENCY_DOMAIN_RANDOM_VIBRATIONs can be selected. from any model.
 * @param modal If selection is modal (blocks the user from doing anything else in PRIMER until this window is dismissed). If omitted the selection will be modal.
 * @returns Number of *FREQUENCY_DOMAIN_RANDOM_VIBRATIONs selected or null if menu cancelled
 */
    static Select(flag: number, prompt: string, limit?: Model | number, modal?: boolean): number;

/**
 * Set the data for a specific Auto PSD load card. For each Auto PSD load card there will be 8 values. There are napsd Auto PSD load cards.
 * @param index Index you want to set Auto PSD load data for. Note that indices start at 0.
 * @param data An array containing the Auto PSD load data (values: sid[integer], stype[integer], dof[integer], ldpsd[integer], ldvel[integer], ldflw[integer], ldspn[integer], cid[integer]). The array length should be 8.
 */
    static SetAutoPsdLoadData(index: number, data: number[]): void;

/**
 * Set the data for a specific Cross PSD load card. For each Cross PSD load card there will be 5 values. There are ncpsd Cross PSD load cards.
 * @param index Index you want to set Cross PSD load data for. Note that indices start at 0.
 * @param data An array containing the Cross PSD load data (values: load_i[integer], load_j[integer], lctyp2[integer], ldpsd1[integer], ldpsd2[integer]). The array length should be 5.
 */
    static SetCrossPsdLoadData(index: number, data: number[]): void;

/**
 * Sets a flag on the *FREQUENCY_DOMAIN_RANDOM_VIBRATION.
 * @param flag Flag to set on the *FREQUENCY_DOMAIN_RANDOM_VIBRATION
 */
    SetFlag(flag: number): void;

/**
 * Set the filename data for an existing binary database for fatigue infromation for a specific initial damage card. There are inftg filenames. This method is only applicable when option is FreqVibration.FATIGUE.
 * @param index Index of an initial damage card that you want the filename for. Note that indices start at 0.
 * @param filename Path and name of existing binary database fro fatigue information.
 */
    static SetInftgData(index: number, filename: string): void;

/**
 * Set the data for a specific zone for fatigue analysis. For each zone there will be 8 values. There are nftg zone definitions for fatigue analysis. This method is only applicable when option is FreqVibration.FATIGUE.
 * @param index Index you want to set the fatigue analysis zone data for. Note that indices start at 0.
 * @param data An array containing the zone data (values: pid[integer], lcid[integer], ptype[integer], ltype[integer], a[float], b[float], sthres[float], snlimt[integer]). The array length will be 8.
 */
    static SetSNCurveData(index: number, data: number[]): void;

/**
 * Returns the total number of *FREQUENCY_DOMAIN_RANDOM_VIBRATIONs in the model.
 * @param Model Model to get total for
 * @param exists true if only existing *FREQUENCY_DOMAIN_RANDOM_VIBRATIONs should be counted. If false or omitted referenced but undefined *FREQUENCY_DOMAIN_RANDOM_VIBRATIONs will also be included in the total.
 * @returns number of *FREQUENCY_DOMAIN_RANDOM_VIBRATIONs
 */
    static Total(Model: Model, exists?: boolean): number;

/**
 * Unsets a defined flag on all of the *FREQUENCY_DOMAIN_RANDOM_VIBRATIONs in the model.
 * @param Model Model that the defined flag for all *FREQUENCY_DOMAIN_RANDOM_VIBRATIONs will be unset in
 * @param flag Flag to unset on the *FREQUENCY_DOMAIN_RANDOM_VIBRATIONs
 */
    static UnflagAll(Model: Model, flag: number): void;

/**
 * Object properties that are parameters are normally returned as the integer or float parameter values as that is virtually always what the user would want. This function temporarily changes the behaviour so that if a property is a parameter the parameter name is returned instead. This can be used with 'method chaining' (see the example below) to make sure a property argument is correct.
 * @returns FreqVibration object.
 */
    ViewParameters(): FreqVibration;

/**
 * Adds a warning for *FREQUENCY_DOMAIN_RANDOM_VIBRATION. For more details on checking see the Check class.
 * @param message The warning message to give
 * @param details An optional detailed warning message
 */
    Warning(message: string, details?: string): void;

/**
 * Returns the cross references for this *FREQUENCY_DOMAIN_RANDOM_VIBRATION.
 * @returns Xrefs object.
 */
    Xrefs(): Xrefs;

/**
 * Create a new FreqVibration object.
 * @param Model Model that *FREQUENCY_DOMAIN_RANDOM_VIBRATION will be created in.
 * @param option Specify the type of *FREQUENCY_DOMAIN_RANDOM_VIBRATION. Can be FreqVibration.VIBRATION, FreqVibration.FATIGUE.
 * @returns FreqVibration object
 */
    constructor(Model: Model, option: number);

/**
 * Creates a string containing the *FREQUENCY_DOMAIN_RANDOM_VIBRATION data in keyword format. Note that this contains the keyword header and the keyword cards. See also FreqVibration.Keyword() and FreqVibration.KeywordCards().
 * @returns string
 */
    toString(): string;

/** Modal damping coefficient. */
    dampf: number;
/** Mass proportional damping constant in Rayleigh damping. */
    dmpmas: number;
/** Stiffness proportional damping constant in Rayleigh damping. */
    dmpstf: number;
/** Tytpe of Damping. */
    dmptyp: number;
/** true if *FREQUENCY_DOMAIN_RANDOM_VIBRATION exists, false if referred to but not defined. */
    readonly exists: boolean;
/** The maximum natural frequency in modal superposition method (optional). */
    fnmax: number;
/** The minimum natural frequency in modal superposition method (optional). */
    fnmin: number;
/** Option for PSD curve coarsening. */
    icoarse: number;
/** The Include file number that the *FREQUENCY_DOMAIN_RANDOM_VIBRATION is in. */
    include: number;
/** Flag for including initial damage ratio. */
    inftg: number;
/** Number of strips in U-direction (used only for vaflag = 5,6,7). */
    ipanelu: number;
/** Number of strips in V-direction (used only for vaflag = 5,6,7). */
    ipanelv: number;
/** The label the *FREQUENCY_DOMAIN_RANDOM_VIBRATION has in PRIMER. */
    readonly label: number;
/** Load Curve ID defining mode dependent modal damping coefficient. */
    lcdam: number;
/** Type of load curve defining modal damping coefficient. */
    lctyp: number;
/** Type of loading curves. */
    ldflag: number;
/** Excitation load type. */
    ldtyp: number;
/** The last mode in modal superposition method (optional). */
    mdmax: number;
/** The first mode in modal superposition method (optional). */
    mdmin: number;
/** Method for modal response analysis. */
    method: number;
/** Method for random fatigue analysis. */
    mftg: number;
/** The Model number that the *FREQUENCY_DOMAIN_RANDOM_VIBRATION is in. */
    readonly model: number;
/** Number of auto PSD load definition. */
    napsd: number;
/** Number of cross PSD load definition. */
    ncpsd: number;
/** Field specifying the number of S-N curves to be defined. */
    nftg: number;
/** The *FREQUENCY_DOMAIN_RANDOM_VIBRATION option. Can be FreqVibration.VIBRATION, FreqVibration.FATIGUE. */
    option: number;
/** Reference pressure */
    pref: number;
/** Restart option. */
    restrm: number;
/** Restart option. */
    restrt: number;
/** Stress scale factor to accommodate different ordinates in S-N curve (not used if nftg = -999). */
    strsf: number;
/** Stress type of S-N curve in fatigue analysis. */
    strtyp: number;
/** Tolerance for slope change percentage for removing intermediate points from PSD curve for icoarse = 2. */
    tcoarse: number;
/** Temperature. */
    temper: number;
/** Exposure time. */
    texpos: number;
/** Multiplier for converting g to [length unit]/[time unit]^2 (used only for UNIT = -1). */
    umlt: number;
/** Flag for acceleration unit conversion. */
    unit: number;
/** Loading type. */
    vaflag: number;
/** Flag for PSD output. */
    vapsd: number;
/** Flag for RMS output. */
    varms: number;
/** FreqVibration option for *FREQUENCY_DOMAIN_RANDOM_VIBRATION_FATIGUE. */
    static FATIGUE: number;
/** FreqVibration option for *FREQUENCY_DOMAIN_RANDOM_VIBRATION. */
    static VIBRATION: number;
}

declare class GeneralizedWeld {
/**
 * Associates a comment with a generalized weld.
 * @param Comment Comment that will be attached to the generalized weld
 */
    AssociateComment(Comment: Comment): void;

/**
 * Blanks the generalized weld
 */
    Blank(): void;

/**
 * Blanks all of the generalized welds in the model.
 * @param Model Model that all generalized welds will be blanked in
 * @param redraw If model should be redrawn or not. If omitted redraw is false. If you want to do several (un)blanks and only redraw after the last one then use false for all redraws apart from the last one. Alternatively you can redraw using View.Redraw().
 */
    static BlankAll(Model: Model, redraw?: boolean): void;

/**
 * Blanks all of the flagged generalized welds in the model.
 * @param Model Model that all the flagged generalized welds will be blanked in
 * @param flag Flag set on the generalized welds that you want to blank
 * @param redraw If model should be redrawn or not. If omitted redraw is false. If you want to do several (un)blanks and only redraw after the last one then use false for all redraws apart from the last one. Alternatively you can redraw using View.Redraw().
 */
    static BlankFlagged(Model: Model, flag: number, redraw?: boolean): void;

/**
 * Checks if the generalized weld is blanked or not.
 * @returns true if blanked, false if not.
 */
    Blanked(): boolean;

/**
 * Starts an edit panel in Browse mode.
 * @param modal If this window is modal (blocks the user from doing anything else in PRIMER until this window is dismissed). If omitted the window will be modal.
 */
    Browse(modal?: boolean): void;

/**
 * Clears a flag on the generalized weld.
 * @param flag Flag to clear on the generalized weld
 */
    ClearFlag(flag: number): void;

/**
 * Copies the generalized weld. The target include of the copied generalized weld can be set using Options.copy_target_include.
 * @param range If you want to keep the copied item in the range specified for the current include. Default value is false. To set current include, use Include.MakeCurrentLayer().
 * @returns GeneralizedWeld object
 */
    Copy(range?: boolean): GeneralizedWeld;

/**
 * Starts an interactive editing panel to create a generalized weld
 * @param Model Model that the generalized weld will be created in.
 * @param modal If this window is modal (blocks the user from doing anything else in PRIMER until this window is dismissed). If omitted the window will be modal.
 * @returns GeneralizedWeld object (or null if not made).
 */
    static Create(Model: Model, modal?: boolean): GeneralizedWeld;

/**
 * Detaches a comment from a generalized weld.
 * @param Comment Comment that will be detached from the generalized weld
 */
    DetachComment(Comment: Comment): void;

/**
 * Starts an interactive editing panel.
 * @param modal If this window is modal (blocks the user from doing anything else in PRIMER until this window is dismissed). If omitted the window will be modal.
 */
    Edit(modal?: boolean): void;

/**
 * Adds an error for generalized weld. For more details on checking see the Check class.
 * @param message The error message to give
 * @param details An optional detailed error message
 */
    Error(message: string, details?: string): void;

/**
 * Returns the first generalized weld in the model.
 * @param Model Model to get first generalized weld in
 * @returns GeneralizedWeld object (or null if there are no generalized welds in the model).
 */
    static First(Model: Model): GeneralizedWeld;

/**
 * Returns the first free generalized weld label in the model. Also see GeneralizedWeld.LastFreeLabel(), GeneralizedWeld.NextFreeLabel() and Model.FirstFreeItemLabel().
 * @param Model Model to get first free generalized weld label in
 * @param layer Include file (0 for the main file) to search for labels in (Equivalent to First free in layer in editing panels). If omitted the whole model will be used (Equivalent to First free in editing panels).
 * @returns GeneralizedWeld label.
 */
    static FirstFreeLabel(Model: Model, layer?: number): number;

/**
 * Flags all of the generalized welds in the model with a defined flag.
 * @param Model Model that all generalized welds will be flagged in
 * @param flag Flag to set on the generalized welds
 */
    static FlagAll(Model: Model, flag: number): void;

/**
 * Checks if the generalized weld is flagged or not.
 * @param flag Flag to test on the generalized weld
 * @returns true if flagged, false if not.
 */
    Flagged(flag: number): boolean;

/**
 * Calls a function for each generalized weld in the model. Note that ForEach has been designed to make looping over generalized welds as fast as possible and so has some limitations. Firstly, a single temporary GeneralizedWeld object is created and on each function call it is updated with the current generalized weld data. This means that you should not try to store the GeneralizedWeld object for later use (e.g. in an array) as it is temporary. Secondly, you cannot create new generalized welds inside a ForEach loop. 
 * @param Model Model that all generalized welds are in
 * @param func Function to call for each generalized weld
 * @param extra An optional extra object/array/string etc that will appended to arguments when calling the function
 */
    static ForEach(Model: Model, func: () => void, extra?: any): void;

/**
 * Returns an array of GeneralizedWeld objects or properties for all of the generalized welds in a model in PRIMER. If the optional property argument is not given then an array of GeneralizedWeld objects is returned. If the property argument is given, that property value for each generalized weld is returned in the array instead of a GeneralizedWeld object
 * @param Model Model to get generalized welds from
 * @param property Name for property to get for all generalized welds in the model
 * @returns Array of GeneralizedWeld objects or properties
 */
    static GetAll(Model: Model, property?: string): GeneralizedWeld[];

/**
 * Returns the combined data for a specific nodal pair as an array.
 * @param index Index you want the data for. Note that indices start at 0.
 * @returns An array containing the data (tfail, epsf, sigy, beta, l, w, a, alpha, nodea, nodeb, ncid, wtyp).
 */
    GetCombinedData(index: number): number[];

/**
 * Extracts the comments associated to a generalized weld.
 * @returns Array of Comment objects (or null if there are no comments associated to the node).
 */
    GetComments(): Comment[];

/**
 * Returns the cross fillet data for a specific nodal pair as an array.
 * @param index Index you want the data for. Note that indices start at 0.
 * @returns An array containing the data (nodea, nodeb, ncid).
 */
    GetCrossFilletData(index: number): number[];

/**
 * Returns an array of GeneralizedWeld objects for all of the flagged generalized welds in a model in PRIMER If the optional property argument is not given then an array of GeneralizedWeld objects is returned. If the property argument is given, then that property value for each generalized weld is returned in the array instead of a GeneralizedWeld object
 * @param Model Model to get generalized welds from
 * @param flag Flag set on the generalized welds that you want to retrieve
 * @param property Name for property to get for all flagged generalized welds in the model
 * @returns Array of GeneralizedWeld objects or properties
 */
    static GetFlagged(Model: Model, flag: number, property?: string): GeneralizedWeld[];

/**
 * Returns the GeneralizedWeld object for a generalized weld ID.
 * @param Model Model to find the generalized weld in
 * @param number number of the generalized weld you want the GeneralizedWeld object for
 * @returns GeneralizedWeld object (or null if generalized weld does not exist).
 */
    static GetFromID(Model: Model, number: number): GeneralizedWeld;

/**
 * Checks if a GeneralizedWeld property is a parameter or not. Note that object properties that are parameters are normally returned as the integer or float parameter values as that is virtually always what the user would want. For this function to work the JavaScript interpreter must use the parameter name instead of the value. This can be done by setting the Options.property_parameter_names option to true before calling the function and then resetting it to false afterwards.. This behaviour can also temporarily be switched by using the GeneralizedWeld.ViewParameters() method and 'method chaining' (see the examples below).
 * @param prop generalized weld property to get parameter for
 * @returns Parameter object if property is a parameter, null if not.
 */
    GetParameter(prop: string): Parameter;

/**
 * Returns the keyword for this gwld (*CONSTRAINED_GENERALIZED_WELD_xxxx). Note that a carriage return is not added. See also GeneralizedWeld.KeywordCards()
 * @returns string containing the keyword.
 */
    Keyword(): string;

/**
 * Returns the keyword cards for the gwld. Note that a carriage return is not added. See also GeneralizedWeld.Keyword()
 * @returns string containing the cards.
 */
    KeywordCards(): string;

/**
 * Returns the last generalized weld in the model.
 * @param Model Model to get last generalized weld in
 * @returns GeneralizedWeld object (or null if there are no generalized welds in the model).
 */
    static Last(Model: Model): GeneralizedWeld;

/**
 * Returns the last free generalized weld label in the model. Also see GeneralizedWeld.FirstFreeLabel(), GeneralizedWeld.NextFreeLabel() and see Model.LastFreeItemLabel()
 * @param Model Model to get last free generalized weld label in
 * @param layer Include file (0 for the main file) to search for labels in (Equivalent to Highest free in layer in editing panels). If omitted the whole model will be used.
 * @returns GeneralizedWeld label.
 */
    static LastFreeLabel(Model: Model, layer?: number): number;

/**
 * Returns the next generalized weld in the model.
 * @returns GeneralizedWeld object (or null if there are no more generalized welds in the model).
 */
    Next(): GeneralizedWeld;

/**
 * Returns the next free (highest+1) generalized weld label in the model. Also see GeneralizedWeld.FirstFreeLabel(), GeneralizedWeld.LastFreeLabel() and Model.NextFreeItemLabel()
 * @param Model Model to get next free generalized weld label in
 * @param layer Include file (0 for the main file) to search for labels in (Equivalent to Highest+1 in layer in editing panels). If omitted the whole model will be used (Equivalent to Highest+1 in editing panels).
 * @returns GeneralizedWeld label.
 */
    static NextFreeLabel(Model: Model, layer?: number): number;

/**
 * Allows the user to pick a generalized weld.
 * @param prompt Text to display as a prompt to the user
 * @param limit If the argument is a Model then only generalized welds from that model can be picked. If the argument is a Flag then only generalized welds that are flagged with limit can be selected. If omitted, or null, any generalized welds from any model can be selected. from any model.
 * @param modal If picking is modal (blocks the user from doing anything else in PRIMER until this window is dismissed). If omitted the pick will be modal.
 * @param button_text By default the window with the prompt will have a button labelled 'Cancel' which if pressed will cancel the pick and return null. If you want to change the text on the button use this argument. If omitted 'Cancel' will be used.
 * @returns GeneralizedWeld object (or null if not picked)
 */
    static Pick(prompt: string, limit?: Model | number, modal?: boolean, button_text?: string): GeneralizedWeld;

/**
 * Returns the previous generalized weld in the model.
 * @returns GeneralizedWeld object (or null if there are no more generalized welds in the model).
 */
    Previous(): GeneralizedWeld;

/**
 * Renumbers all of the generalized welds in the model.
 * @param Model Model that all generalized welds will be renumbered in
 * @param start Start point for renumbering
 */
    static RenumberAll(Model: Model, start: number): void;

/**
 * Renumbers all of the flagged generalized welds in the model.
 * @param Model Model that all the flagged generalized welds will be renumbered in
 * @param flag Flag set on the generalized welds that you want to renumber
 * @param start Start point for renumbering
 */
    static RenumberFlagged(Model: Model, flag: number, start: number): void;

/**
 * Allows the user to select generalized welds using standard PRIMER object menus.
 * @param flag Flag to use when selecting generalized welds
 * @param prompt Text to display as a prompt to the user
 * @param limit If the argument is a Model then only generalized welds from that model can be selected. If the argument is a Flag then only generalized welds that are flagged with limit can be selected (limit should be different to flag). If omitted, or null, any generalized welds can be selected. from any model.
 * @param modal If selection is modal (blocks the user from doing anything else in PRIMER until this window is dismissed). If omitted the selection will be modal.
 * @returns Number of generalized welds selected or null if menu cancelled
 */
    static Select(flag: number, prompt: string, limit?: Model | number, modal?: boolean): number;

/**
 * Sets the combined data for a specific nodal pair.
 * @param index Index you want to set the data for. Note that indices start at 0.
 * @param data Array containing the data. The array length should be 12 (tfail, epsf, sigy, beta, l, w, a, alpha, nodea, nodeb, ncid, wtyp)
 */
    SetCombinedData(index: number, data: number[]): void;

/**
 * Sets the cross fillet data for a specific nodal pair.
 * @param index Index you want to set the data for. Note that indices start at 0.
 * @param data Array containing the data. The array length should be 3 (nodea, nodeb, ncid)
 */
    SetCrossFilletData(index: number, data: number[]): void;

/**
 * Sets a flag on the generalized weld.
 * @param flag Flag to set on the generalized weld
 */
    SetFlag(flag: number): void;

/**
 * Sketches the generalized weld. The generalized weld will be sketched until you either call GeneralizedWeld.Unsketch(), GeneralizedWeld.UnsketchAll(), Model.UnsketchAll(), or delete the model
 * @param redraw If model should be redrawn or not after the generalized weld is sketched. If omitted redraw is true. If you want to sketch several generalized welds and only redraw after the last one then use false for redraw and call View.Redraw().
 */
    Sketch(redraw?: boolean): void;

/**
 * Sketches all of the flagged generalized welds in the model. The generalized welds will be sketched until you either call GeneralizedWeld.Unsketch(), GeneralizedWeld.UnsketchFlagged(), Model.UnsketchAll(), or delete the model
 * @param Model Model that all the flagged generalized welds will be sketched in
 * @param flag Flag set on the generalized welds that you want to sketch
 * @param redraw If model should be redrawn or not after the generalized welds are sketched. If omitted redraw is true. If you want to sketch flagged generalized welds several times and only redraw after the last one then use false for redraw and call View.Redraw().
 */
    static SketchFlagged(Model: Model, flag: number, redraw?: boolean): void;

/**
 * Returns the total number of generalized welds in the model.
 * @param Model Model to get total for
 * @param exists true if only existing generalized welds should be counted. If false or omitted referenced but undefined generalized welds will also be included in the total.
 * @returns number of generalized welds
 */
    static Total(Model: Model, exists?: boolean): number;

/**
 * Unblanks the generalized weld
 */
    Unblank(): void;

/**
 * Unblanks all of the generalized welds in the model.
 * @param Model Model that all generalized welds will be unblanked in
 * @param redraw If model should be redrawn or not. If omitted redraw is false. If you want to do several (un)blanks and only redraw after the last one then use false for all redraws apart from the last one. Alternatively you can redraw using View.Redraw().
 */
    static UnblankAll(Model: Model, redraw?: boolean): void;

/**
 * Unblanks all of the flagged generalized welds in the model.
 * @param Model Model that the flagged generalized welds will be unblanked in
 * @param flag Flag set on the generalized welds that you want to unblank
 * @param redraw If model should be redrawn or not. If omitted redraw is false. If you want to do several (un)blanks and only redraw after the last one then use false for all redraws apart from the last one. Alternatively you can redraw using View.Redraw().
 */
    static UnblankFlagged(Model: Model, flag: number, redraw?: boolean): void;

/**
 * Unsets a defined flag on all of the generalized welds in the model.
 * @param Model Model that the defined flag for all generalized welds will be unset in
 * @param flag Flag to unset on the generalized welds
 */
    static UnflagAll(Model: Model, flag: number): void;

/**
 * Unsketches the generalized weld.
 * @param redraw If model should be redrawn or not after the generalized weld is unsketched. If omitted redraw is true. If you want to unsketch several generalized welds and only redraw after the last one then use false for redraw and call View.Redraw().
 */
    Unsketch(redraw?: boolean): void;

/**
 * Unsketches all generalized welds.
 * @param Model Model that all generalized welds will be unblanked in
 * @param redraw If model should be redrawn or not after the generalized welds are unsketched. If omitted redraw is true. If you want to unsketch several things and only redraw after the last one then use false for redraw and call View.Redraw().
 */
    static UnsketchAll(Model: Model, redraw?: boolean): void;

/**
 * Unsketches all flagged generalized welds in the model.
 * @param Model Model that all generalized welds will be unsketched in
 * @param flag Flag set on the generalized welds that you want to unsketch
 * @param redraw If model should be redrawn or not after the generalized welds are unsketched. If omitted redraw is true. If you want to unsketch several things and only redraw after the last one then use false for redraw and call View.Redraw().
 */
    static UnsketchFlagged(Model: Model, flag: number, redraw?: boolean): void;

/**
 * Object properties that are parameters are normally returned as the integer or float parameter values as that is virtually always what the user would want. This function temporarily changes the behaviour so that if a property is a parameter the parameter name is returned instead. This can be used with 'method chaining' (see the example below) to make sure a property argument is correct.
 * @returns GeneralizedWeld object.
 */
    ViewParameters(): GeneralizedWeld;

/**
 * Adds a warning for generalized weld. For more details on checking see the Check class.
 * @param message The warning message to give
 * @param details An optional detailed warning message
 */
    Warning(message: string, details?: string): void;

/**
 * Returns the cross references for this generalized weld.
 * @returns Xrefs object.
 */
    Xrefs(): Xrefs;

/**
 * Create a new GeneralizedWeld object.
 * @param Model Model that gwld will be created in
 * @param option Constrained generalized weld type (any).
 * @param nsid Set Node Set ID.
 * @param cid Coordinate System ID.
 * @param filter Number of force vectors saved for filtering.
 * @param window Filter time window.
 * @param npr Number of individual nodal pairs in cross fillet and combined weld.
 * @param nprt Printout option.
 * @param wid Constrained Generalized weld number.
 * @returns GeneralizedWeld object
 */
    constructor(Model: Model, option: number, nsid: number, cid?: number, filter?: number, window?: number, npr?: number, nprt?: number, wid?: number);

/**
 * Creates a string containing the gwld data in keyword format. Note that this contains the keyword header and the keyword cards. See also GeneralizedWeld.Keyword() and GeneralizedWeld.KeywordCards().
 * @returns string
 */
    toString(): string;

/** Width of fillet (GeneralizedWeld.FILLET, GeneralizedWeld.CROSS_FILLET) */
    a: number;
/** Weld angle (GeneralizedWeld.FILLET, GeneralizedWeld.CROSS_FILLET) */
    alpha: number;
/** Failure parameter (GeneralizedWeld.FILLET, GeneralizedWeld.BUTT, GeneralizedWeld.CROSS_FILLET) */
    beta: number;
/** Coordinate System ID. */
    cid: number;
/** Thickness of weld (GeneralizedWeld.BUTT) */
    d: number;
/** Effective plastic strain at failure (GeneralizedWeld.SPOT, GeneralizedWeld.FILLET, GeneralizedWeld.BUTT, GeneralizedWeld.CROSS_FILLET) */
    epsf: number;
/** true if generalized weld exists, false if referred to but not defined. */
    readonly exists: boolean;
/** Number of force vectors saved for filtering. */
    filter: number;
/** true if _ID option is set, false if not */
    id: boolean;
/** The Include file number that the generalized weld is in. */
    include: number;
/** Length of weld (GeneralizedWeld.FILLET, GeneralizedWeld.BUTT, GeneralizedWeld.CROSS_FILLET) */
    l: number;
/** Constrained Generalized weld number. */
    label: number;
/** Transverse length (GeneralizedWeld.BUTT) */
    lt: number;
/** Exponent for shear force (GeneralizedWeld.SPOT) */
    m: number;
/** The Model number that the generalized weld is in. */
    readonly model: number;
/** Exponent for normal force (GeneralizedWeld.SPOT) */
    n: number;
/** Number of individual nodal pairs in cross fillet and combined weld. */
    npr: number;
/** Printout option. */
    nprt: number;
/** Set Node Set ID. */
    nsid: number;
/** GeneralizedWeld type. Can be GeneralizedWeld.SPOT, GeneralizedWeld.FILLET, GeneralizedWeld.BUTT, GeneralizedWeld.CROSS_FILLET, GeneralizedWeld.COMBINED */
    option: number;
/** Stress at failure (GeneralizedWeld.FILLET) */
    sigf: number;
/** Stress at failure (GeneralizedWeld.BUTT, GeneralizedWeld.CROSS_FILLET) */
    sigy: number;
/** Normal force at failure (GeneralizedWeld.SPOT) */
    sn: number;
/** Shear force at faliure (GeneralizedWeld.SPOT) */
    ss: number;
/** Failure time for constraint set (GeneralizedWeld.SPOT, GeneralizedWeld.FILLET, GeneralizedWeld.BUTT, GeneralizedWeld.CROSS_FILLET) */
    tfail: number;
/** Width of flange (GeneralizedWeld.FILLET, GeneralizedWeld.CROSS_FILLET) */
    w: number;
/** Constrained Generalized weld number (identical to label). */
    wid: number;
/** Filter time window. */
    window: number;
/** GeneralizedWeld is *CONSTRAINED_GENERALIZED_WELD_BUTT. */
    static BUTT: number;
/** GeneralizedWeld is *CONSTRAINED_GENERALIZED_WELD_COMBINED. */
    static COMBINED: number;
/** GeneralizedWeld is *CONSTRAINED_GENERALIZED_WELD_CROSS_FILLET. */
    static CROSS_FILLET: number;
/** GeneralizedWeld is *CONSTRAINED_GENERALIZED_WELD_FILLET. */
    static FILLET: number;
/** GeneralizedWeld is *CONSTRAINED_GENERALIZED_WELD_SPOT. */
    static SPOT: number;
}

declare class Gwld {
/**
 * Associates a comment with a generalized weld.
 * @param Comment Comment that will be attached to the generalized weld
 */
    AssociateComment(Comment: Comment): void;

/**
 * Blanks the generalized weld
 */
    Blank(): void;

/**
 * Blanks all of the generalized welds in the model.
 * @param Model Model that all generalized welds will be blanked in
 * @param redraw If model should be redrawn or not. If omitted redraw is false. If you want to do several (un)blanks and only redraw after the last one then use false for all redraws apart from the last one. Alternatively you can redraw using View.Redraw().
 */
    static BlankAll(Model: Model, redraw?: boolean): void;

/**
 * Blanks all of the flagged generalized welds in the model.
 * @param Model Model that all the flagged generalized welds will be blanked in
 * @param flag Flag set on the generalized welds that you want to blank
 * @param redraw If model should be redrawn or not. If omitted redraw is false. If you want to do several (un)blanks and only redraw after the last one then use false for all redraws apart from the last one. Alternatively you can redraw using View.Redraw().
 */
    static BlankFlagged(Model: Model, flag: number, redraw?: boolean): void;

/**
 * Checks if the generalized weld is blanked or not.
 * @returns true if blanked, false if not.
 */
    Blanked(): boolean;

/**
 * Starts an edit panel in Browse mode.
 * @param modal If this window is modal (blocks the user from doing anything else in PRIMER until this window is dismissed). If omitted the window will be modal.
 */
    Browse(modal?: boolean): void;

/**
 * Clears a flag on the generalized weld.
 * @param flag Flag to clear on the generalized weld
 */
    ClearFlag(flag: number): void;

/**
 * Copies the generalized weld. The target include of the copied generalized weld can be set using Options.copy_target_include.
 * @param range If you want to keep the copied item in the range specified for the current include. Default value is false. To set current include, use Include.MakeCurrentLayer().
 * @returns GeneralizedWeld object
 */
    Copy(range?: boolean): GeneralizedWeld;

/**
 * Starts an interactive editing panel to create a generalized weld
 * @param Model Model that the generalized weld will be created in.
 * @param modal If this window is modal (blocks the user from doing anything else in PRIMER until this window is dismissed). If omitted the window will be modal.
 * @returns GeneralizedWeld object (or null if not made).
 */
    static Create(Model: Model, modal?: boolean): GeneralizedWeld;

/**
 * Detaches a comment from a generalized weld.
 * @param Comment Comment that will be detached from the generalized weld
 */
    DetachComment(Comment: Comment): void;

/**
 * Starts an interactive editing panel.
 * @param modal If this window is modal (blocks the user from doing anything else in PRIMER until this window is dismissed). If omitted the window will be modal.
 */
    Edit(modal?: boolean): void;

/**
 * Adds an error for generalized weld. For more details on checking see the Check class.
 * @param message The error message to give
 * @param details An optional detailed error message
 */
    Error(message: string, details?: string): void;

/**
 * Returns the first generalized weld in the model.
 * @param Model Model to get first generalized weld in
 * @returns GeneralizedWeld object (or null if there are no generalized welds in the model).
 */
    static First(Model: Model): GeneralizedWeld;

/**
 * Returns the first free generalized weld label in the model. Also see GeneralizedWeld.LastFreeLabel(), GeneralizedWeld.NextFreeLabel() and Model.FirstFreeItemLabel().
 * @param Model Model to get first free generalized weld label in
 * @param layer Include file (0 for the main file) to search for labels in (Equivalent to First free in layer in editing panels). If omitted the whole model will be used (Equivalent to First free in editing panels).
 * @returns GeneralizedWeld label.
 */
    static FirstFreeLabel(Model: Model, layer?: number): number;

/**
 * Flags all of the generalized welds in the model with a defined flag.
 * @param Model Model that all generalized welds will be flagged in
 * @param flag Flag to set on the generalized welds
 */
    static FlagAll(Model: Model, flag: number): void;

/**
 * Checks if the generalized weld is flagged or not.
 * @param flag Flag to test on the generalized weld
 * @returns true if flagged, false if not.
 */
    Flagged(flag: number): boolean;

/**
 * Calls a function for each generalized weld in the model. Note that ForEach has been designed to make looping over generalized welds as fast as possible and so has some limitations. Firstly, a single temporary GeneralizedWeld object is created and on each function call it is updated with the current generalized weld data. This means that you should not try to store the GeneralizedWeld object for later use (e.g. in an array) as it is temporary. Secondly, you cannot create new generalized welds inside a ForEach loop. 
 * @param Model Model that all generalized welds are in
 * @param func Function to call for each generalized weld
 * @param extra An optional extra object/array/string etc that will appended to arguments when calling the function
 */
    static ForEach(Model: Model, func: () => void, extra?: any): void;

/**
 * Returns an array of GeneralizedWeld objects or properties for all of the generalized welds in a model in PRIMER. If the optional property argument is not given then an array of GeneralizedWeld objects is returned. If the property argument is given, that property value for each generalized weld is returned in the array instead of a GeneralizedWeld object
 * @param Model Model to get generalized welds from
 * @param property Name for property to get for all generalized welds in the model
 * @returns Array of GeneralizedWeld objects or properties
 */
    static GetAll(Model: Model, property?: string): GeneralizedWeld[];

/**
 * Returns the combined data for a specific nodal pair as an array.
 * @param index Index you want the data for. Note that indices start at 0.
 * @returns An array containing the data (tfail, epsf, sigy, beta, l, w, a, alpha, nodea, nodeb, ncid, wtyp).
 */
    GetCombinedData(index: number): number[];

/**
 * Extracts the comments associated to a generalized weld.
 * @returns Array of Comment objects (or null if there are no comments associated to the node).
 */
    GetComments(): Comment[];

/**
 * Returns the cross fillet data for a specific nodal pair as an array.
 * @param index Index you want the data for. Note that indices start at 0.
 * @returns An array containing the data (nodea, nodeb, ncid).
 */
    GetCrossFilletData(index: number): number[];

/**
 * Returns an array of GeneralizedWeld objects for all of the flagged generalized welds in a model in PRIMER If the optional property argument is not given then an array of GeneralizedWeld objects is returned. If the property argument is given, then that property value for each generalized weld is returned in the array instead of a GeneralizedWeld object
 * @param Model Model to get generalized welds from
 * @param flag Flag set on the generalized welds that you want to retrieve
 * @param property Name for property to get for all flagged generalized welds in the model
 * @returns Array of GeneralizedWeld objects or properties
 */
    static GetFlagged(Model: Model, flag: number, property?: string): GeneralizedWeld[];

/**
 * Returns the GeneralizedWeld object for a generalized weld ID.
 * @param Model Model to find the generalized weld in
 * @param number number of the generalized weld you want the GeneralizedWeld object for
 * @returns GeneralizedWeld object (or null if generalized weld does not exist).
 */
    static GetFromID(Model: Model, number: number): GeneralizedWeld;

/**
 * Checks if a GeneralizedWeld property is a parameter or not. Note that object properties that are parameters are normally returned as the integer or float parameter values as that is virtually always what the user would want. For this function to work the JavaScript interpreter must use the parameter name instead of the value. This can be done by setting the Options.property_parameter_names option to true before calling the function and then resetting it to false afterwards.. This behaviour can also temporarily be switched by using the GeneralizedWeld.ViewParameters() method and 'method chaining' (see the examples below).
 * @param prop generalized weld property to get parameter for
 * @returns Parameter object if property is a parameter, null if not.
 */
    GetParameter(prop: string): Parameter;

/**
 * Returns the keyword for this gwld (*CONSTRAINED_GENERALIZED_WELD_xxxx). Note that a carriage return is not added. See also GeneralizedWeld.KeywordCards()
 * @returns string containing the keyword.
 */
    Keyword(): string;

/**
 * Returns the keyword cards for the gwld. Note that a carriage return is not added. See also GeneralizedWeld.Keyword()
 * @returns string containing the cards.
 */
    KeywordCards(): string;

/**
 * Returns the last generalized weld in the model.
 * @param Model Model to get last generalized weld in
 * @returns GeneralizedWeld object (or null if there are no generalized welds in the model).
 */
    static Last(Model: Model): GeneralizedWeld;

/**
 * Returns the last free generalized weld label in the model. Also see GeneralizedWeld.FirstFreeLabel(), GeneralizedWeld.NextFreeLabel() and see Model.LastFreeItemLabel()
 * @param Model Model to get last free generalized weld label in
 * @param layer Include file (0 for the main file) to search for labels in (Equivalent to Highest free in layer in editing panels). If omitted the whole model will be used.
 * @returns GeneralizedWeld label.
 */
    static LastFreeLabel(Model: Model, layer?: number): number;

/**
 * Returns the next generalized weld in the model.
 * @returns GeneralizedWeld object (or null if there are no more generalized welds in the model).
 */
    Next(): GeneralizedWeld;

/**
 * Returns the next free (highest+1) generalized weld label in the model. Also see GeneralizedWeld.FirstFreeLabel(), GeneralizedWeld.LastFreeLabel() and Model.NextFreeItemLabel()
 * @param Model Model to get next free generalized weld label in
 * @param layer Include file (0 for the main file) to search for labels in (Equivalent to Highest+1 in layer in editing panels). If omitted the whole model will be used (Equivalent to Highest+1 in editing panels).
 * @returns GeneralizedWeld label.
 */
    static NextFreeLabel(Model: Model, layer?: number): number;

/**
 * Allows the user to pick a generalized weld.
 * @param prompt Text to display as a prompt to the user
 * @param limit If the argument is a Model then only generalized welds from that model can be picked. If the argument is a Flag then only generalized welds that are flagged with limit can be selected. If omitted, or null, any generalized welds from any model can be selected. from any model.
 * @param modal If picking is modal (blocks the user from doing anything else in PRIMER until this window is dismissed). If omitted the pick will be modal.
 * @param button_text By default the window with the prompt will have a button labelled 'Cancel' which if pressed will cancel the pick and return null. If you want to change the text on the button use this argument. If omitted 'Cancel' will be used.
 * @returns GeneralizedWeld object (or null if not picked)
 */
    static Pick(prompt: string, limit?: Model | number, modal?: boolean, button_text?: string): GeneralizedWeld;

/**
 * Returns the previous generalized weld in the model.
 * @returns GeneralizedWeld object (or null if there are no more generalized welds in the model).
 */
    Previous(): GeneralizedWeld;

/**
 * Renumbers all of the generalized welds in the model.
 * @param Model Model that all generalized welds will be renumbered in
 * @param start Start point for renumbering
 */
    static RenumberAll(Model: Model, start: number): void;

/**
 * Renumbers all of the flagged generalized welds in the model.
 * @param Model Model that all the flagged generalized welds will be renumbered in
 * @param flag Flag set on the generalized welds that you want to renumber
 * @param start Start point for renumbering
 */
    static RenumberFlagged(Model: Model, flag: number, start: number): void;

/**
 * Allows the user to select generalized welds using standard PRIMER object menus.
 * @param flag Flag to use when selecting generalized welds
 * @param prompt Text to display as a prompt to the user
 * @param limit If the argument is a Model then only generalized welds from that model can be selected. If the argument is a Flag then only generalized welds that are flagged with limit can be selected (limit should be different to flag). If omitted, or null, any generalized welds can be selected. from any model.
 * @param modal If selection is modal (blocks the user from doing anything else in PRIMER until this window is dismissed). If omitted the selection will be modal.
 * @returns Number of generalized welds selected or null if menu cancelled
 */
    static Select(flag: number, prompt: string, limit?: Model | number, modal?: boolean): number;

/**
 * Sets the combined data for a specific nodal pair.
 * @param index Index you want to set the data for. Note that indices start at 0.
 * @param data Array containing the data. The array length should be 12 (tfail, epsf, sigy, beta, l, w, a, alpha, nodea, nodeb, ncid, wtyp)
 */
    SetCombinedData(index: number, data: number[]): void;

/**
 * Sets the cross fillet data for a specific nodal pair.
 * @param index Index you want to set the data for. Note that indices start at 0.
 * @param data Array containing the data. The array length should be 3 (nodea, nodeb, ncid)
 */
    SetCrossFilletData(index: number, data: number[]): void;

/**
 * Sets a flag on the generalized weld.
 * @param flag Flag to set on the generalized weld
 */
    SetFlag(flag: number): void;

/**
 * Sketches the generalized weld. The generalized weld will be sketched until you either call GeneralizedWeld.Unsketch(), GeneralizedWeld.UnsketchAll(), Model.UnsketchAll(), or delete the model
 * @param redraw If model should be redrawn or not after the generalized weld is sketched. If omitted redraw is true. If you want to sketch several generalized welds and only redraw after the last one then use false for redraw and call View.Redraw().
 */
    Sketch(redraw?: boolean): void;

/**
 * Sketches all of the flagged generalized welds in the model. The generalized welds will be sketched until you either call GeneralizedWeld.Unsketch(), GeneralizedWeld.UnsketchFlagged(), Model.UnsketchAll(), or delete the model
 * @param Model Model that all the flagged generalized welds will be sketched in
 * @param flag Flag set on the generalized welds that you want to sketch
 * @param redraw If model should be redrawn or not after the generalized welds are sketched. If omitted redraw is true. If you want to sketch flagged generalized welds several times and only redraw after the last one then use false for redraw and call View.Redraw().
 */
    static SketchFlagged(Model: Model, flag: number, redraw?: boolean): void;

/**
 * Returns the total number of generalized welds in the model.
 * @param Model Model to get total for
 * @param exists true if only existing generalized welds should be counted. If false or omitted referenced but undefined generalized welds will also be included in the total.
 * @returns number of generalized welds
 */
    static Total(Model: Model, exists?: boolean): number;

/**
 * Unblanks the generalized weld
 */
    Unblank(): void;

/**
 * Unblanks all of the generalized welds in the model.
 * @param Model Model that all generalized welds will be unblanked in
 * @param redraw If model should be redrawn or not. If omitted redraw is false. If you want to do several (un)blanks and only redraw after the last one then use false for all redraws apart from the last one. Alternatively you can redraw using View.Redraw().
 */
    static UnblankAll(Model: Model, redraw?: boolean): void;

/**
 * Unblanks all of the flagged generalized welds in the model.
 * @param Model Model that the flagged generalized welds will be unblanked in
 * @param flag Flag set on the generalized welds that you want to unblank
 * @param redraw If model should be redrawn or not. If omitted redraw is false. If you want to do several (un)blanks and only redraw after the last one then use false for all redraws apart from the last one. Alternatively you can redraw using View.Redraw().
 */
    static UnblankFlagged(Model: Model, flag: number, redraw?: boolean): void;

/**
 * Unsets a defined flag on all of the generalized welds in the model.
 * @param Model Model that the defined flag for all generalized welds will be unset in
 * @param flag Flag to unset on the generalized welds
 */
    static UnflagAll(Model: Model, flag: number): void;

/**
 * Unsketches the generalized weld.
 * @param redraw If model should be redrawn or not after the generalized weld is unsketched. If omitted redraw is true. If you want to unsketch several generalized welds and only redraw after the last one then use false for redraw and call View.Redraw().
 */
    Unsketch(redraw?: boolean): void;

/**
 * Unsketches all generalized welds.
 * @param Model Model that all generalized welds will be unblanked in
 * @param redraw If model should be redrawn or not after the generalized welds are unsketched. If omitted redraw is true. If you want to unsketch several things and only redraw after the last one then use false for redraw and call View.Redraw().
 */
    static UnsketchAll(Model: Model, redraw?: boolean): void;

/**
 * Unsketches all flagged generalized welds in the model.
 * @param Model Model that all generalized welds will be unsketched in
 * @param flag Flag set on the generalized welds that you want to unsketch
 * @param redraw If model should be redrawn or not after the generalized welds are unsketched. If omitted redraw is true. If you want to unsketch several things and only redraw after the last one then use false for redraw and call View.Redraw().
 */
    static UnsketchFlagged(Model: Model, flag: number, redraw?: boolean): void;

/**
 * Object properties that are parameters are normally returned as the integer or float parameter values as that is virtually always what the user would want. This function temporarily changes the behaviour so that if a property is a parameter the parameter name is returned instead. This can be used with 'method chaining' (see the example below) to make sure a property argument is correct.
 * @returns GeneralizedWeld object.
 */
    ViewParameters(): GeneralizedWeld;

/**
 * Adds a warning for generalized weld. For more details on checking see the Check class.
 * @param message The warning message to give
 * @param details An optional detailed warning message
 */
    Warning(message: string, details?: string): void;

/**
 * Returns the cross references for this generalized weld.
 * @returns Xrefs object.
 */
    Xrefs(): Xrefs;

/**
 * Create a new GeneralizedWeld object.
 * @param Model Model that gwld will be created in
 * @param option Constrained generalized weld type (any).
 * @param nsid Set Node Set ID.
 * @param cid Coordinate System ID.
 * @param filter Number of force vectors saved for filtering.
 * @param window Filter time window.
 * @param npr Number of individual nodal pairs in cross fillet and combined weld.
 * @param nprt Printout option.
 * @param wid Constrained Generalized weld number.
 * @returns GeneralizedWeld object
 */
    constructor(Model: Model, option: number, nsid: number, cid?: number, filter?: number, window?: number, npr?: number, nprt?: number, wid?: number);

/**
 * Creates a string containing the gwld data in keyword format. Note that this contains the keyword header and the keyword cards. See also GeneralizedWeld.Keyword() and GeneralizedWeld.KeywordCards().
 * @returns string
 */
    toString(): string;

/** Width of fillet (GeneralizedWeld.FILLET, GeneralizedWeld.CROSS_FILLET) */
    a: number;
/** Weld angle (GeneralizedWeld.FILLET, GeneralizedWeld.CROSS_FILLET) */
    alpha: number;
/** Failure parameter (GeneralizedWeld.FILLET, GeneralizedWeld.BUTT, GeneralizedWeld.CROSS_FILLET) */
    beta: number;
/** Coordinate System ID. */
    cid: number;
/** Thickness of weld (GeneralizedWeld.BUTT) */
    d: number;
/** Effective plastic strain at failure (GeneralizedWeld.SPOT, GeneralizedWeld.FILLET, GeneralizedWeld.BUTT, GeneralizedWeld.CROSS_FILLET) */
    epsf: number;
/** true if generalized weld exists, false if referred to but not defined. */
    readonly exists: boolean;
/** Number of force vectors saved for filtering. */
    filter: number;
/** true if _ID option is set, false if not */
    id: boolean;
/** The Include file number that the generalized weld is in. */
    include: number;
/** Length of weld (GeneralizedWeld.FILLET, GeneralizedWeld.BUTT, GeneralizedWeld.CROSS_FILLET) */
    l: number;
/** Constrained Generalized weld number. */
    label: number;
/** Transverse length (GeneralizedWeld.BUTT) */
    lt: number;
/** Exponent for shear force (GeneralizedWeld.SPOT) */
    m: number;
/** The Model number that the generalized weld is in. */
    readonly model: number;
/** Exponent for normal force (GeneralizedWeld.SPOT) */
    n: number;
/** Number of individual nodal pairs in cross fillet and combined weld. */
    npr: number;
/** Printout option. */
    nprt: number;
/** Set Node Set ID. */
    nsid: number;
/** GeneralizedWeld type. Can be GeneralizedWeld.SPOT, GeneralizedWeld.FILLET, GeneralizedWeld.BUTT, GeneralizedWeld.CROSS_FILLET, GeneralizedWeld.COMBINED */
    option: number;
/** Stress at failure (GeneralizedWeld.FILLET) */
    sigf: number;
/** Stress at failure (GeneralizedWeld.BUTT, GeneralizedWeld.CROSS_FILLET) */
    sigy: number;
/** Normal force at failure (GeneralizedWeld.SPOT) */
    sn: number;
/** Shear force at faliure (GeneralizedWeld.SPOT) */
    ss: number;
/** Failure time for constraint set (GeneralizedWeld.SPOT, GeneralizedWeld.FILLET, GeneralizedWeld.BUTT, GeneralizedWeld.CROSS_FILLET) */
    tfail: number;
/** Width of flange (GeneralizedWeld.FILLET, GeneralizedWeld.CROSS_FILLET) */
    w: number;
/** Constrained Generalized weld number (identical to label). */
    wid: number;
/** Filter time window. */
    window: number;
/** GeneralizedWeld is *CONSTRAINED_GENERALIZED_WELD_BUTT. */
    static BUTT: number;
/** GeneralizedWeld is *CONSTRAINED_GENERALIZED_WELD_COMBINED. */
    static COMBINED: number;
/** GeneralizedWeld is *CONSTRAINED_GENERALIZED_WELD_CROSS_FILLET. */
    static CROSS_FILLET: number;
/** GeneralizedWeld is *CONSTRAINED_GENERALIZED_WELD_FILLET. */
    static FILLET: number;
/** GeneralizedWeld is *CONSTRAINED_GENERALIZED_WELD_SPOT. */
    static SPOT: number;
}

declare class GeometrySurface {
/**
 * Associates a comment with a surface.
 * @param Comment Comment that will be attached to the surface
 */
    AssociateComment(Comment: Comment): void;

/**
 * Blanks the surface
 */
    Blank(): void;

/**
 * Blanks all of the surfaces in the model.
 * @param Model Model that all surfaces will be blanked in
 * @param redraw If model should be redrawn or not. If omitted redraw is false. If you want to do several (un)blanks and only redraw after the last one then use false for all redraws apart from the last one. Alternatively you can redraw using View.Redraw().
 */
    static BlankAll(Model: Model, redraw?: boolean): void;

/**
 * Blanks all of the flagged surfaces in the model.
 * @param Model Model that all the flagged surfaces will be blanked in
 * @param flag Flag set on the surfaces that you want to blank
 * @param redraw If model should be redrawn or not. If omitted redraw is false. If you want to do several (un)blanks and only redraw after the last one then use false for all redraws apart from the last one. Alternatively you can redraw using View.Redraw().
 */
    static BlankFlagged(Model: Model, flag: number, redraw?: boolean): void;

/**
 * Checks if the surface is blanked or not.
 * @returns true if blanked, false if not.
 */
    Blanked(): boolean;

/**
 * Calculate the normal vector for a parametric point on a surface.
 * @param u u parametric coordinate
 * @param y v parametric coordinate
 * @returns Array containing x, y and z values.
 */
    CalculateNormal(u: number, y: number): number[];

/**
 * Calculate the X, Y and Z coordinates for a parametric point on a surface.
 * @param u u parametric coordinate
 * @param v v parametric coordinate
 * @returns Array containing x, y and z values.
 */
    CalculatePoint(u: number, v: number): number[];

/**
 * Clears a flag on the surface.
 * @param flag Flag to clear on the surface
 */
    ClearFlag(flag: number): void;

/**
 * Copies the surface. The target include of the copied surface can be set using Options.copy_target_include.
 * @param range If you want to keep the copied item in the range specified for the current include. Default value is false. To set current include, use Include.MakeCurrentLayer().
 * @returns GeometrySurface object
 */
    Copy(range?: boolean): GeometrySurface;

/**
 * Detaches a comment from a surface.
 * @param Comment Comment that will be detached from the surface
 */
    DetachComment(Comment: Comment): void;

/**
 * Adds an error for surface. For more details on checking see the Check class.
 * @param message The error message to give
 * @param details An optional detailed error message
 */
    Error(message: string, details?: string): void;

/**
 * Returns the first surface in the model.
 * @param Model Model to get first surface in
 * @returns GeometrySurface object (or null if there are no surfaces in the model).
 */
    static First(Model: Model): GeometrySurface;

/**
 * Returns the first free surface label in the model. Also see GeometrySurface.LastFreeLabel(), GeometrySurface.NextFreeLabel() and Model.FirstFreeItemLabel().
 * @param Model Model to get first free surface label in
 * @param layer Include file (0 for the main file) to search for labels in (Equivalent to First free in layer in editing panels). If omitted the whole model will be used (Equivalent to First free in editing panels).
 * @returns GeometrySurface label.
 */
    static FirstFreeLabel(Model: Model, layer?: number): number;

/**
 * Flags all of the surfaces in the model with a defined flag.
 * @param Model Model that all surfaces will be flagged in
 * @param flag Flag to set on the surfaces
 */
    static FlagAll(Model: Model, flag: number): void;

/**
 * Checks if the surface is flagged or not.
 * @param flag Flag to test on the surface
 * @returns true if flagged, false if not.
 */
    Flagged(flag: number): boolean;

/**
 * Calls a function for each surface in the model. Note that ForEach has been designed to make looping over surfaces as fast as possible and so has some limitations. Firstly, a single temporary GeometrySurface object is created and on each function call it is updated with the current surface data. This means that you should not try to store the GeometrySurface object for later use (e.g. in an array) as it is temporary. Secondly, you cannot create new surfaces inside a ForEach loop. 
 * @param Model Model that all surfaces are in
 * @param func Function to call for each surface
 * @param extra An optional extra object/array/string etc that will appended to arguments when calling the function
 */
    static ForEach(Model: Model, func: () => void, extra?: any): void;

/**
 * Returns an array of GeometrySurface objects or properties for all of the surfaces in a model in PRIMER. If the optional property argument is not given then an array of GeometrySurface objects is returned. If the property argument is given, that property value for each surface is returned in the array instead of a GeometrySurface object
 * @param Model Model to get surfaces from
 * @param property Name for property to get for all surfaces in the model
 * @returns Array of GeometrySurface objects or properties
 */
    static GetAll(Model: Model, property?: string): GeometrySurface[];

/**
 * Extracts the comments associated to a surface.
 * @returns Array of Comment objects (or null if there are no comments associated to the node).
 */
    GetComments(): Comment[];

/**
 * Return an array of all the edge indices for a surface (in pairs).
 * @returns Array of indices
 */
    GetEdgeIndices(): number[];

/**
 * Returns an array of GeometrySurface objects for all of the flagged surfaces in a model in PRIMER If the optional property argument is not given then an array of GeometrySurface objects is returned. If the property argument is given, then that property value for each surface is returned in the array instead of a GeometrySurface object
 * @param Model Model to get surfaces from
 * @param flag Flag set on the surfaces that you want to retrieve
 * @param property Name for property to get for all flagged surfaces in the model
 * @returns Array of GeometrySurface objects or properties
 */
    static GetFlagged(Model: Model, flag: number, property?: string): GeometrySurface[];

/**
 * Returns the GeometrySurface object for a surface ID.
 * @param Model Model to find the surface in
 * @param number number of the surface you want the GeometrySurface object for
 * @returns GeometrySurface object (or null if surface does not exist).
 */
    static GetFromID(Model: Model, number: number): GeometrySurface;

/**
 * Checks if a GeometrySurface property is a parameter or not. Note that object properties that are parameters are normally returned as the integer or float parameter values as that is virtually always what the user would want. For this function to work the JavaScript interpreter must use the parameter name instead of the value. This can be done by setting the Options.property_parameter_names option to true before calling the function and then resetting it to false afterwards.. This behaviour can also temporarily be switched by using the GeometrySurface.ViewParameters() method and 'method chaining' (see the examples below).
 * @param prop surface property to get parameter for
 * @returns Parameter object if property is a parameter, null if not.
 */
    GetParameter(prop: string): Parameter;

/**
 * Return an array of all the tria indices for a surface (in triplets).
 * @returns Array of indices
 */
    GetTriaIndices(): number[];

/**
 * Return an array of all the vertex coordinates for a surface (in triplets).
 * @returns Array of indices
 */
    GetVertices(): number[];

/**
 * Returns the last surface in the model.
 * @param Model Model to get last surface in
 * @returns GeometrySurface object (or null if there are no surfaces in the model).
 */
    static Last(Model: Model): GeometrySurface;

/**
 * Returns the last free surface label in the model. Also see GeometrySurface.FirstFreeLabel(), GeometrySurface.NextFreeLabel() and see Model.LastFreeItemLabel()
 * @param Model Model to get last free surface label in
 * @param layer Include file (0 for the main file) to search for labels in (Equivalent to Highest free in layer in editing panels). If omitted the whole model will be used.
 * @returns GeometrySurface label.
 */
    static LastFreeLabel(Model: Model, layer?: number): number;

/**
 * Returns the next surface in the model.
 * @returns GeometrySurface object (or null if there are no more surfaces in the model).
 */
    Next(): GeometrySurface;

/**
 * Returns the next free (highest+1) surface label in the model. Also see GeometrySurface.FirstFreeLabel(), GeometrySurface.LastFreeLabel() and Model.NextFreeItemLabel()
 * @param Model Model to get next free surface label in
 * @param layer Include file (0 for the main file) to search for labels in (Equivalent to Highest+1 in layer in editing panels). If omitted the whole model will be used (Equivalent to Highest+1 in editing panels).
 * @returns GeometrySurface label.
 */
    static NextFreeLabel(Model: Model, layer?: number): number;

/**
 * Allows the user to pick a surface.
 * @param prompt Text to display as a prompt to the user
 * @param limit If the argument is a Model then only surfaces from that model can be picked. If the argument is a Flag then only surfaces that are flagged with limit can be selected. If omitted, or null, any surfaces from any model can be selected. from any model.
 * @param modal If picking is modal (blocks the user from doing anything else in PRIMER until this window is dismissed). If omitted the pick will be modal.
 * @param button_text By default the window with the prompt will have a button labelled 'Cancel' which if pressed will cancel the pick and return null. If you want to change the text on the button use this argument. If omitted 'Cancel' will be used.
 * @returns GeometrySurface object (or null if not picked)
 */
    static Pick(prompt: string, limit?: Model | number, modal?: boolean, button_text?: string): GeometrySurface;

/**
 * Returns the previous surface in the model.
 * @returns GeometrySurface object (or null if there are no more surfaces in the model).
 */
    Previous(): GeometrySurface;

/**
 * Project a point onto the surface.
 * @param x X coordinate of point to project
 * @param y Y coordinate of point to project
 * @param z Z coordinate of point to project
 * @returns Array containing u and v values.
 */
    ProjectPoint(x: number, y: number, z: number): number[];

/**
 * Renumbers all of the surfaces in the model.
 * @param Model Model that all surfaces will be renumbered in
 * @param start Start point for renumbering
 */
    static RenumberAll(Model: Model, start: number): void;

/**
 * Renumbers all of the flagged surfaces in the model.
 * @param Model Model that all the flagged surfaces will be renumbered in
 * @param flag Flag set on the surfaces that you want to renumber
 * @param start Start point for renumbering
 */
    static RenumberFlagged(Model: Model, flag: number, start: number): void;

/**
 * Allows the user to select surfaces using standard PRIMER object menus.
 * @param flag Flag to use when selecting surfaces
 * @param prompt Text to display as a prompt to the user
 * @param limit If the argument is a Model then only surfaces from that model can be selected. If the argument is a Flag then only surfaces that are flagged with limit can be selected (limit should be different to flag). If omitted, or null, any surfaces can be selected. from any model.
 * @param modal If selection is modal (blocks the user from doing anything else in PRIMER until this window is dismissed). If omitted the selection will be modal.
 * @returns Number of surfaces selected or null if menu cancelled
 */
    static Select(flag: number, prompt: string, limit?: Model | number, modal?: boolean): number;

/**
 * Sets a flag on the surface.
 * @param flag Flag to set on the surface
 */
    SetFlag(flag: number): void;

/**
 * Sketches the surface. The surface will be sketched until you either call GeometrySurface.Unsketch(), GeometrySurface.UnsketchAll(), Model.UnsketchAll(), or delete the model
 * @param redraw If model should be redrawn or not after the surface is sketched. If omitted redraw is true. If you want to sketch several surfaces and only redraw after the last one then use false for redraw and call View.Redraw().
 */
    Sketch(redraw?: boolean): void;

/**
 * Sketches all of the flagged surfaces in the model. The surfaces will be sketched until you either call GeometrySurface.Unsketch(), GeometrySurface.UnsketchFlagged(), Model.UnsketchAll(), or delete the model
 * @param Model Model that all the flagged surfaces will be sketched in
 * @param flag Flag set on the surfaces that you want to sketch
 * @param redraw If model should be redrawn or not after the surfaces are sketched. If omitted redraw is true. If you want to sketch flagged surfaces several times and only redraw after the last one then use false for redraw and call View.Redraw().
 */
    static SketchFlagged(Model: Model, flag: number, redraw?: boolean): void;

/**
 * Returns the total number of surfaces in the model.
 * @param Model Model to get total for
 * @param exists true if only existing surfaces should be counted. If false or omitted referenced but undefined surfaces will also be included in the total.
 * @returns number of surfaces
 */
    static Total(Model: Model, exists?: boolean): number;

/**
 * Unblanks the surface
 */
    Unblank(): void;

/**
 * Unblanks all of the surfaces in the model.
 * @param Model Model that all surfaces will be unblanked in
 * @param redraw If model should be redrawn or not. If omitted redraw is false. If you want to do several (un)blanks and only redraw after the last one then use false for all redraws apart from the last one. Alternatively you can redraw using View.Redraw().
 */
    static UnblankAll(Model: Model, redraw?: boolean): void;

/**
 * Unblanks all of the flagged surfaces in the model.
 * @param Model Model that the flagged surfaces will be unblanked in
 * @param flag Flag set on the surfaces that you want to unblank
 * @param redraw If model should be redrawn or not. If omitted redraw is false. If you want to do several (un)blanks and only redraw after the last one then use false for all redraws apart from the last one. Alternatively you can redraw using View.Redraw().
 */
    static UnblankFlagged(Model: Model, flag: number, redraw?: boolean): void;

/**
 * Unsets a defined flag on all of the surfaces in the model.
 * @param Model Model that the defined flag for all surfaces will be unset in
 * @param flag Flag to unset on the surfaces
 */
    static UnflagAll(Model: Model, flag: number): void;

/**
 * Unsketches the surface.
 * @param redraw If model should be redrawn or not after the surface is unsketched. If omitted redraw is true. If you want to unsketch several surfaces and only redraw after the last one then use false for redraw and call View.Redraw().
 */
    Unsketch(redraw?: boolean): void;

/**
 * Unsketches all surfaces.
 * @param Model Model that all surfaces will be unblanked in
 * @param redraw If model should be redrawn or not after the surfaces are unsketched. If omitted redraw is true. If you want to unsketch several things and only redraw after the last one then use false for redraw and call View.Redraw().
 */
    static UnsketchAll(Model: Model, redraw?: boolean): void;

/**
 * Unsketches all flagged surfaces in the model.
 * @param Model Model that all surfaces will be unsketched in
 * @param flag Flag set on the surfaces that you want to unsketch
 * @param redraw If model should be redrawn or not after the surfaces are unsketched. If omitted redraw is true. If you want to unsketch several things and only redraw after the last one then use false for redraw and call View.Redraw().
 */
    static UnsketchFlagged(Model: Model, flag: number, redraw?: boolean): void;

/**
 * Object properties that are parameters are normally returned as the integer or float parameter values as that is virtually always what the user would want. This function temporarily changes the behaviour so that if a property is a parameter the parameter name is returned instead. This can be used with 'method chaining' (see the example below) to make sure a property argument is correct.
 * @returns GeometrySurface object.
 */
    ViewParameters(): GeometrySurface;

/**
 * Adds a warning for surface. For more details on checking see the Check class.
 * @param message The warning message to give
 * @param details An optional detailed warning message
 */
    Warning(message: string, details?: string): void;

/**
 * Returns the cross references for this surface.
 * @returns Xrefs object.
 */
    Xrefs(): Xrefs;

/** true if surface exists, false if referred to but not defined. */
    readonly exists: boolean;
/** GeometrySurface number. Also see the label property which is an alternative name for this. */
    readonly id: number;
/** The Include file number that the surface is in. */
    include: number;
/** GeometrySurface number. Also see the id property which is an alternative name for this. */
    readonly label: number;
/** The Model number that the surface is in. */
    readonly model: number;
}

declare class Graphics {
/**
 * Allows depth testing (hidden surface removal) to be turned on or off. Temporarily turning depth testing off may be used to ensure that an item (e.g. some text) is always drawn in front and will not be obscured.
 * @param enable Whether depth testing (hidden surface removal) is performed (true) or not (false)
 */
    static DepthTest(enable: boolean): void;

/**
 * Set the function to draw graphics from javaScript. This function will be called each time the graphics are redrawn after PRIMER has finished drawing everything else. This allows you to add extra items to the graphics. To remove the graphics drawing function use Graphics.DrawingFunction(null). It is the responsibility of the script developer to ensure that any objects or variables that are used in the drawing function do not refer to items in PRIMER that no longer exist. Not doing so may cause PRIMER to crash. For example, if you use some Node objects in the drawing function that refer to nodes in model 1 and you delete the model, when the graphics are redrawn PRIMER may crash as the nodes referred to by the Node objects no longer exist. You should either remove the drawing function by calling Graphics.DrawingFunction(null) or set the Node variables to null (and test that they exist before using them) in your drawing function before deleting the model. If a drawing function is used in your script, you should reset it before the script terminates to avoid a "race condition" between the script terminating and the graphics function being called. Not doing so may cause PRIMER to crash.
 * @param name The name of the function (or null to remove a function)
 */
    static DrawingFunction(name: () => void): void;

/**
 * Sets the colour for drawing polygons. See the Colour class for more details on colours.
 * @param colour The colour you want to fill polygons with
 */
    static FillColour(colour: Colour): void;

/**
 * Finish any graphics. See also Graphics.Start(). This must be used to finish drawing.
 */
    static Finish(): void;

/**
 * Draws a line from (x1, y1, z1) to (x2, y2, z2). See also Graphics.LineTo() and Graphics.MoveTo()
 * @param x1 X coordinate of point 1
 * @param y1 Y coordinate of point 1
 * @param z1 Z coordinate of point 1
 * @param x2 X coordinate of point 2
 * @param y2 Y coordinate of point 2
 * @param z2 Z coordinate of point 2
 */
    static Line(x1: number, y1: number, z1: number, x2: number, y2: number, z2: number): void;

/**
 * Sets the colour for drawing lines. See the Colour class for more details on colours.
 * @param colour The colour you want to draw lines with
 */
    static LineColour(colour: Colour): void;

/**
 * Sets the style for drawing lines.
 * @param style The style to draw lines with. Can be: Graphics.SOLID_LINE, Graphics.DASH_LINE, Graphics.DASHDOT_LINE or Graphics.DOT_LINE
 */
    static LineStyle(style: number): void;

/**
 * Draws a line from the current point to (x, y, z). After drawing the line the current point will be (x, y, z). See also Graphics.Line() and Graphics.MoveTo()
 * @param x X coordinate
 * @param y Y coordinate
 * @param z Z coordinate
 */
    static LineTo(x: number, y: number, z: number): void;

/**
 * Sets the width for drawing lines.
 * @param width The width to draw lines with
 */
    static LineWidth(width: number): void;

/**
 * Sets the current point to (x, y, z). See also Graphics.Line() and Graphics.LineTo()
 * @param x X coordinate
 * @param y Y coordinate
 * @param z Z coordinate
 */
    static MoveTo(x: number, y: number, z: number): void;

/**
 * Ends drawing a polygon. See also Graphics.PolygonStart()
 */
    static PolygonFinish(): void;

/**
 * Starts drawing a convexpolygon. A maximum of 250 vertices are allowed. Drawing concave polygons is not supported by this function. To draw concave polygons split them into separate convex polygons. See also Graphics.PolygonFinish(). The only graphics command allowed between Graphics.PolygonStart() and Graphics.PolygonFinish() is Graphics.MoveTo(). Any other graphics drawing commands (e.g. Graphics.Text()) will be ignored.
 */
    static PolygonStart(): void;

/**
 * Draws a simple shape.
 * @param shape The style to draw lines with. Can be: Graphics.POINT, Graphics.SQUARE, Graphics.CIRCLE, Graphics.DIAMOND, Graphics.HOURGLASS, Graphics.FILLED_SQUARE, Graphics.FILLED_CIRCLE, Graphics.FILLED_DIAMOND or Graphics.FILLED_HOURGLASS
 * @param size Size the shape should be drawn at.
 */
    static Shape(shape: number, size: number): void;

/**
 * Start any graphics. See also Graphics.Finish(). This must be used before any drawing is done.
 */
    static Start(): void;

/**
 * Draws text at current position. See Graphics.MoveTo() to set the current position.
 * @param text The text to write
 */
    static Text(text: string): void;

/**
 * Sets the colour for drawing text. See the Colour class for more details on colours.
 * @param colour The colour you want to draw text with
 */
    static TextColour(colour: Colour): void;

/**
 * Sets the size for drawing text.
 * @param size The size to draw text with
 */
    static TextSize(size: number): void;

/** Circle shape. See Graphics.Shape() for use. */
    static CIRCLE: number;
/** Dashed and dotted lines. See Graphics.LineStyle() for use. */
    static DASHDOT_LINE: number;
/** Dashed lines. See Graphics.LineStyle() for use. */
    static DASH_LINE: number;
/** Diamond shape. See Graphics.Shape() for use. */
    static DIAMOND: number;
/** Dotted lines. See Graphics.LineStyle() for use. */
    static DOT_LINE: number;
/** Filled circle shape. See Graphics.Shape() for use. */
    static FILLED_CIRCLE: number;
/** Filled diamond shape. See Graphics.Shape() for use. */
    static FILLED_DIAMOND: number;
/** Filled hourglass shape. See Graphics.Shape() for use. */
    static FILLED_HOURGLASS: number;
/** Filled square shape. See Graphics.Shape() for use. */
    static FILLED_SQUARE: number;
/** Hourglass shape. See Graphics.Shape() for use. */
    static HOURGLASS: number;
/** Point shape. See Graphics.Shape() for use. */
    static POINT: number;
/** Solid lines. See Graphics.LineStyle() for use. */
    static SOLID_LINE: number;
/** Square shape. See Graphics.Shape() for use. */
    static SQUARE: number;
}

declare class Group {
/**
 * Associates a comment with a group.
 * @param Comment Comment that will be attached to the group
 */
    AssociateComment(Comment: Comment): void;

/**
 * Blanks the group
 */
    Blank(): void;

/**
 * Blanks all of the groups in the model.
 * @param Model Model that all groups will be blanked in
 * @param redraw If model should be redrawn or not. If omitted redraw is false. If you want to do several (un)blanks and only redraw after the last one then use false for all redraws apart from the last one. Alternatively you can redraw using View.Redraw().
 */
    static BlankAll(Model: Model, redraw?: boolean): void;

/**
 * Blanks all of the flagged groups in the model.
 * @param Model Model that all the flagged groups will be blanked in
 * @param flag Flag set on the groups that you want to blank
 * @param redraw If model should be redrawn or not. If omitted redraw is false. If you want to do several (un)blanks and only redraw after the last one then use false for all redraws apart from the last one. Alternatively you can redraw using View.Redraw().
 */
    static BlankFlagged(Model: Model, flag: number, redraw?: boolean): void;

/**
 * Checks if the group is blanked or not.
 * @returns true if blanked, false if not.
 */
    Blanked(): boolean;

/**
 * Starts an edit panel in Browse mode.
 * @param modal If this window is modal (blocks the user from doing anything else in PRIMER until this window is dismissed). If omitted the window will be modal.
 */
    Browse(modal?: boolean): void;

/**
 * Clears a flag on the group.
 * @param flag Flag to clear on the group
 */
    ClearFlag(flag: number): void;

/**
 * Copies the group. The target include of the copied group can be set using Options.copy_target_include.
 * @param range If you want to keep the copied item in the range specified for the current include. Default value is false. To set current include, use Include.MakeCurrentLayer().
 * @returns Group object
 */
    Copy(range?: boolean): Group;

/**
 * Starts an interactive editing panel to create a group
 * @param Model Model that the group will be created in.
 * @param modal If this window is modal (blocks the user from doing anything else in PRIMER until this window is dismissed). If omitted the window will be modal.
 * @returns Group object (or null if not made).
 */
    static Create(Model: Model, modal?: boolean): Group;

/**
 * Detaches a comment from a group.
 * @param Comment Comment that will be detached from the group
 */
    DetachComment(Comment: Comment): void;

/**
 * Starts an interactive editing panel.
 * @param modal If this window is modal (blocks the user from doing anything else in PRIMER until this window is dismissed). If omitted the window will be modal.
 */
    Edit(modal?: boolean): void;

/**
 * Adds an error for group. For more details on checking see the Check class.
 * @param message The error message to give
 * @param details An optional detailed error message
 */
    Error(message: string, details?: string): void;

/**
 * Returns the first group in the model.
 * @param Model Model to get first group in
 * @returns Group object (or null if there are no groups in the model).
 */
    static First(Model: Model): Group;

/**
 * Returns the first free group label in the model. Also see Group.LastFreeLabel(), Group.NextFreeLabel() and Model.FirstFreeItemLabel().
 * @param Model Model to get first free group label in
 * @param layer Include file (0 for the main file) to search for labels in (Equivalent to First free in layer in editing panels). If omitted the whole model will be used (Equivalent to First free in editing panels).
 * @returns Group label.
 */
    static FirstFreeLabel(Model: Model, layer?: number): number;

/**
 * Flags all of the groups in the model with a defined flag.
 * @param Model Model that all groups will be flagged in
 * @param flag Flag to set on the groups
 */
    static FlagAll(Model: Model, flag: number): void;

/**
 * Checks if the group is flagged or not.
 * @param flag Flag to test on the group
 * @returns true if flagged, false if not.
 */
    Flagged(flag: number): boolean;

/**
 * Calls a function for each group in the model. Note that ForEach has been designed to make looping over groups as fast as possible and so has some limitations. Firstly, a single temporary Group object is created and on each function call it is updated with the current group data. This means that you should not try to store the Group object for later use (e.g. in an array) as it is temporary. Secondly, you cannot create new groups inside a ForEach loop. 
 * @param Model Model that all groups are in
 * @param func Function to call for each group
 * @param extra An optional extra object/array/string etc that will appended to arguments when calling the function
 */
    static ForEach(Model: Model, func: () => void, extra?: any): void;

/**
 * Returns an array of Group objects or properties for all of the groups in a model in PRIMER. If the optional property argument is not given then an array of Group objects is returned. If the property argument is given, that property value for each group is returned in the array instead of a Group object
 * @param Model Model to get groups from
 * @param property Name for property to get for all groups in the model
 * @returns Array of Group objects or properties
 */
    static GetAll(Model: Model, property?: string): Group[];

/**
 * Extracts the comments associated to a group.
 * @returns Array of Comment objects (or null if there are no comments associated to the node).
 */
    GetComments(): Comment[];

/**
 * Returns 'all' data for a given row number and type in the group.
 * @param type The type of the item
 * @param index Index of 'all' row you want the data for. Note that indices start at 0, not 1. 0 &lt;= index &lt; Group.GetTotalAll()
 * @returns An array containing data [Group.ADD or Group.REMOVE, BOX (if defined)].
 */
    GetDataAll(type: string, index: number): number[];

/**
 * Returns 'list' data for a given row number and type in the group.
 * @param type The type of the item
 * @param index Index of 'list' row you want the data for. Note that indices start at 0, not 1. 0 &lt;= index &lt; Group.GetTotalList()
 * @returns An array containing data [Group.ADD or Group.REMOVE, ITEM1 (if defined), ITEM2 (if defined), ITEM3 (if defined), ITEM4 (if defined), ITEM5 (if defined), BOX (if defined)].
 */
    GetDataList(type: string, index: number): number[];

/**
 * Returns 'range' data for a given row number and type in the group.
 * @param type The type of the item
 * @param index Index of 'range' row you want the data for. Note that indices start at 0, not 1. 0 &lt;= index &lt; Group.GetTotalRange()
 * @returns An array containing data [Group.ADD or Group.REMOVE, START, END, BOX (if defined)].
 */
    GetDataRange(type: string, index: number): number[];

/**
 * Returns an array of Group objects for all of the flagged groups in a model in PRIMER If the optional property argument is not given then an array of Group objects is returned. If the property argument is given, then that property value for each group is returned in the array instead of a Group object
 * @param Model Model to get groups from
 * @param flag Flag set on the groups that you want to retrieve
 * @param property Name for property to get for all flagged groups in the model
 * @returns Array of Group objects or properties
 */
    static GetFlagged(Model: Model, flag: number, property?: string): Group[];

/**
 * Returns the Group object for a group ID.
 * @param Model Model to find the group in
 * @param number number of the group you want the Group object for
 * @returns Group object (or null if group does not exist).
 */
    static GetFromID(Model: Model, number: number): Group;

/**
 * Checks if a Group property is a parameter or not. Note that object properties that are parameters are normally returned as the integer or float parameter values as that is virtually always what the user would want. For this function to work the JavaScript interpreter must use the parameter name instead of the value. This can be done by setting the Options.property_parameter_names option to true before calling the function and then resetting it to false afterwards.. This behaviour can also temporarily be switched by using the Group.ViewParameters() method and 'method chaining' (see the examples below).
 * @param prop group property to get parameter for
 * @returns Parameter object if property is a parameter, null if not.
 */
    GetParameter(prop: string): Parameter;

/**
 * Returns the total number of 'all' rows for a type in a group
 * @param type The type of the item
 * @returns The number of 'all' rows defined
 */
    GetTotalAll(type: string): number;

/**
 * Returns the total number of 'list' rows for a type in a group
 * @param type The type of the item
 * @returns The number of 'list' rows defined
 */
    GetTotalList(type: string): number;

/**
 * Returns the total number of 'range' rows for a type in a group
 * @param type The type of the item
 * @returns The number of 'range' rows defined
 */
    GetTotalRange(type: string): number;

/**
 * Returns the total number of 'all', 'list' and 'range' rows for a type in a group
 * @param type The type of the item
 * @returns Array containing number of 'all', 'list' and 'range' rows defined or null if type not in group.
 */
    GetTotals(type: string): number[];

/**
 * Returns the type for an entry in a group
 * @param row The entry in the group types that you want the type for. Note that entries start at 0, not 1
 * @returns The type of the item (string)
 */
    GetType(row: number): string;

/**
 * Returns the keyword for this group. Note that a carriage return is not added. See also Group.KeywordCards()
 * @returns string containing the keyword.
 */
    Keyword(): string;

/**
 * Returns the keyword cards for the Group. Note that a carriage return is not added. See also Group.Keyword()
 * @returns string containing the cards.
 */
    KeywordCards(): string;

/**
 * Returns the last group in the model.
 * @param Model Model to get last group in
 * @returns Group object (or null if there are no groups in the model).
 */
    static Last(Model: Model): Group;

/**
 * Returns the last free group label in the model. Also see Group.FirstFreeLabel(), Group.NextFreeLabel() and see Model.LastFreeItemLabel()
 * @param Model Model to get last free group label in
 * @param layer Include file (0 for the main file) to search for labels in (Equivalent to Highest free in layer in editing panels). If omitted the whole model will be used.
 * @returns Group label.
 */
    static LastFreeLabel(Model: Model, layer?: number): number;

/**
 * Returns the next group in the model.
 * @returns Group object (or null if there are no more groups in the model).
 */
    Next(): Group;

/**
 * Returns the next free (highest+1) group label in the model. Also see Group.FirstFreeLabel(), Group.LastFreeLabel() and Model.NextFreeItemLabel()
 * @param Model Model to get next free group label in
 * @param layer Include file (0 for the main file) to search for labels in (Equivalent to Highest+1 in layer in editing panels). If omitted the whole model will be used (Equivalent to Highest+1 in editing panels).
 * @returns Group label.
 */
    static NextFreeLabel(Model: Model, layer?: number): number;

/**
 * Allows the user to pick a group.
 * @param prompt Text to display as a prompt to the user
 * @param limit If the argument is a Model then only groups from that model can be picked. If the argument is a Flag then only groups that are flagged with limit can be selected. If omitted, or null, any groups from any model can be selected. from any model.
 * @param modal If picking is modal (blocks the user from doing anything else in PRIMER until this window is dismissed). If omitted the pick will be modal.
 * @param button_text By default the window with the prompt will have a button labelled 'Cancel' which if pressed will cancel the pick and return null. If you want to change the text on the button use this argument. If omitted 'Cancel' will be used.
 * @returns Group object (or null if not picked)
 */
    static Pick(prompt: string, limit?: Model | number, modal?: boolean, button_text?: string): Group;

/**
 * Returns the previous group in the model.
 * @returns Group object (or null if there are no more groups in the model).
 */
    Previous(): Group;

/**
 * Removes 'all' data for a given row number and type in the group.
 * @param type The type of the item
 * @param index Index of 'all' row you want to Remove. Note that indices start at 0, not 1. 0 &lt;= index &lt; Group.GetTotalAll()
 */
    RemoveDataAll(type: string, index: number): void;

/**
 * Removes 'list' data for a given row number and type in the group.
 * @param type The type of the item
 * @param index Index of 'list' row you want to Remove. Note that indices start at 0, not 1. 0 &lt;= index &lt; Group.GetTotalList()
 */
    RemoveDataList(type: string, index: number): void;

/**
 * Removes 'range' data for a given row number and type in the group.
 * @param type The type of the item
 * @param index Index of 'range' row you want to Remove. Note that indices start at 0, not 1. 0 &lt;= index &lt; Group.GetTotalRange()
 */
    RemoveDataRange(type: string, index: number): void;

/**
 * Renumbers all of the groups in the model.
 * @param Model Model that all groups will be renumbered in
 * @param start Start point for renumbering
 */
    static RenumberAll(Model: Model, start: number): void;

/**
 * Renumbers all of the flagged groups in the model.
 * @param Model Model that all the flagged groups will be renumbered in
 * @param flag Flag set on the groups that you want to renumber
 * @param start Start point for renumbering
 */
    static RenumberFlagged(Model: Model, flag: number, start: number): void;

/**
 * Allows the user to select groups using standard PRIMER object menus.
 * @param flag Flag to use when selecting groups
 * @param prompt Text to display as a prompt to the user
 * @param limit If the argument is a Model then only groups from that model can be selected. If the argument is a Flag then only groups that are flagged with limit can be selected (limit should be different to flag). If omitted, or null, any groups can be selected. from any model.
 * @param modal If selection is modal (blocks the user from doing anything else in PRIMER until this window is dismissed). If omitted the selection will be modal.
 * @returns Number of groups selected or null if menu cancelled
 */
    static Select(flag: number, prompt: string, limit?: Model | number, modal?: boolean): number;

/**
 * Sets 'all' data for a given row number and type in the group.
 * @param type The type of the item
 * @param index Index of 'all' row you want the data for. Note that indices start at 0, not 1. 0 &lt;= index &lt;= Group.GetTotalAll()
 * @param data An array containing data [Group.ADD or Group.REMOVE, BOX (if defined)].
 */
    SetDataAll(type: string, index: number, data: number[]): void;

/**
 * Sets 'list' data for a given row number and type in the group.
 * @param type The type of the item
 * @param index Index of 'list' row you want the data for. Note that indices start at 0, not 1. 0 &lt;= index &lt;= Group.GetTotalList()
 * @param data An array containing data [Group.ADD or Group.REMOVE, ITEM1 (if defined), ITEM2 (if defined), ITEM3 (if defined), ITEM4 (if defined), ITEM5 (if defined), BOX (if defined)].
 */
    SetDataList(type: string, index: number, data: number[]): void;

/**
 * Sets 'range' data for a given row number and type in the group.
 * @param type The type of the item
 * @param index Index of 'all' row you want the data for. Note that indices start at 0, not 1. 0 &lt;= index &lt;= Group.GetTotalRange()
 * @param data An array containing data [Group.ADD or Group.REMOVE, START, END, BOX (if defined)].
 */
    SetDataRange(type: string, index: number, data: number[]): void;

/**
 * Sets a flag on the group.
 * @param flag Flag to set on the group
 */
    SetFlag(flag: number): void;

/**
 * Sketches the group. The group will be sketched until you either call Group.Unsketch(), Group.UnsketchAll(), Model.UnsketchAll(), or delete the model
 * @param redraw If model should be redrawn or not after the group is sketched. If omitted redraw is true. If you want to sketch several groups and only redraw after the last one then use false for redraw and call View.Redraw().
 */
    Sketch(redraw?: boolean): void;

/**
 * Sketches all of the flagged groups in the model. The groups will be sketched until you either call Group.Unsketch(), Group.UnsketchFlagged(), Model.UnsketchAll(), or delete the model
 * @param Model Model that all the flagged groups will be sketched in
 * @param flag Flag set on the groups that you want to sketch
 * @param redraw If model should be redrawn or not after the groups are sketched. If omitted redraw is true. If you want to sketch flagged groups several times and only redraw after the last one then use false for redraw and call View.Redraw().
 */
    static SketchFlagged(Model: Model, flag: number, redraw?: boolean): void;

/**
 * Returns the total number of groups in the model.
 * @param Model Model to get total for
 * @param exists true if only existing groups should be counted. If false or omitted referenced but undefined groups will also be included in the total.
 * @returns number of groups
 */
    static Total(Model: Model, exists?: boolean): number;

/**
 * Unblanks the group
 */
    Unblank(): void;

/**
 * Unblanks all of the groups in the model.
 * @param Model Model that all groups will be unblanked in
 * @param redraw If model should be redrawn or not. If omitted redraw is false. If you want to do several (un)blanks and only redraw after the last one then use false for all redraws apart from the last one. Alternatively you can redraw using View.Redraw().
 */
    static UnblankAll(Model: Model, redraw?: boolean): void;

/**
 * Unblanks all of the flagged groups in the model.
 * @param Model Model that the flagged groups will be unblanked in
 * @param flag Flag set on the groups that you want to unblank
 * @param redraw If model should be redrawn or not. If omitted redraw is false. If you want to do several (un)blanks and only redraw after the last one then use false for all redraws apart from the last one. Alternatively you can redraw using View.Redraw().
 */
    static UnblankFlagged(Model: Model, flag: number, redraw?: boolean): void;

/**
 * Unsets a defined flag on all of the groups in the model.
 * @param Model Model that the defined flag for all groups will be unset in
 * @param flag Flag to unset on the groups
 */
    static UnflagAll(Model: Model, flag: number): void;

/**
 * Unsketches the group.
 * @param redraw If model should be redrawn or not after the group is unsketched. If omitted redraw is true. If you want to unsketch several groups and only redraw after the last one then use false for redraw and call View.Redraw().
 */
    Unsketch(redraw?: boolean): void;

/**
 * Unsketches all groups.
 * @param Model Model that all groups will be unblanked in
 * @param redraw If model should be redrawn or not after the groups are unsketched. If omitted redraw is true. If you want to unsketch several things and only redraw after the last one then use false for redraw and call View.Redraw().
 */
    static UnsketchAll(Model: Model, redraw?: boolean): void;

/**
 * Unsketches all flagged groups in the model.
 * @param Model Model that all groups will be unsketched in
 * @param flag Flag set on the groups that you want to unsketch
 * @param redraw If model should be redrawn or not after the groups are unsketched. If omitted redraw is true. If you want to unsketch several things and only redraw after the last one then use false for redraw and call View.Redraw().
 */
    static UnsketchFlagged(Model: Model, flag: number, redraw?: boolean): void;

/**
 * Object properties that are parameters are normally returned as the integer or float parameter values as that is virtually always what the user would want. This function temporarily changes the behaviour so that if a property is a parameter the parameter name is returned instead. This can be used with 'method chaining' (see the example below) to make sure a property argument is correct.
 * @returns Group object.
 */
    ViewParameters(): Group;

/**
 * Adds a warning for group. For more details on checking see the Check class.
 * @param message The warning message to give
 * @param details An optional detailed warning message
 */
    Warning(message: string, details?: string): void;

/**
 * Returns the cross references for this group.
 * @returns Xrefs object.
 */
    Xrefs(): Xrefs;

/**
 * Create a new Group object.
 * @param Model Model that Group will be created in
 * @param label Group number.
 * @param title Title for the group
 * @returns Group object
 */
    constructor(Model: Model, label: number, title?: string);

/**
 * Creates a string containing the Group data in keyword format. Note that this contains the keyword header and the keyword cards. See also Group.Keyword() and Group.KeywordCards().
 * @returns string
 */
    toString(): string;

/** true if group exists, false if referred to but not defined. */
    readonly exists: boolean;
/** The Include file number that the group is in. */
    include: number;
/** Group number */
    label: number;
/** Whether Group contents are locked against deletion. */
    lock: boolean;
/** The Model number that the group is in. */
    readonly model: number;
/** Number of types in the group. */
    readonly numtypes: number;
/** Group title */
    title: string;
/** Add contents to group */
    static ADD: number;
/** Remove contents from group */
    static REMOVE: number;
}


/** Object function argument in HexSpotweldAssembly constructor */
interface HexSpotweldAssembly_constructor_Argument_options {
    /** HexSpotweldAssembly ID. */
    id: number;
    /** HexSpotweldAssembly option indicating the length of the solids array. opt can be 4, 8 or 16. */
    opt: number;
    /** Array of Solid IDs, at least 4 EIDs must be given. */
    solids: array[];
    /** Optional HexSpotweldAssembly title. */
    title?: string;
}

declare class HexSpotweldAssembly {
/**
 * Associates a comment with a DEFINE_HEX_SPOTWELD_ASSEMBLY.
 * @param Comment Comment that will be attached to the DEFINE_HEX_SPOTWELD_ASSEMBLY
 */
    AssociateComment(Comment: Comment): void;

/**
 * Starts an edit panel in Browse mode.
 * @param modal If this window is modal (blocks the user from doing anything else in PRIMER until this window is dismissed). If omitted the window will be modal.
 */
    Browse(modal?: boolean): void;

/**
 * Clears a flag on the DEFINE_HEX_SPOTWELD_ASSEMBLY.
 * @param flag Flag to clear on the DEFINE_HEX_SPOTWELD_ASSEMBLY
 */
    ClearFlag(flag: number): void;

/**
 * Copies the DEFINE_HEX_SPOTWELD_ASSEMBLY. The target include of the copied DEFINE_HEX_SPOTWELD_ASSEMBLY can be set using Options.copy_target_include.
 * @param range If you want to keep the copied item in the range specified for the current include. Default value is false. To set current include, use Include.MakeCurrentLayer().
 * @returns HexSpotweldAssembly object
 */
    Copy(range?: boolean): HexSpotweldAssembly;

/**
 * Starts an interactive editing panel to create a DEFINE_HEX_SPOTWELD_ASSEMBLY
 * @param Model Model that the DEFINE_HEX_SPOTWELD_ASSEMBLY will be created in.
 * @param modal If this window is modal (blocks the user from doing anything else in PRIMER until this window is dismissed). If omitted the window will be modal.
 * @returns HexSpotweldAssembly object (or null if not made).
 */
    static Create(Model: Model, modal?: boolean): HexSpotweldAssembly;

/**
 * Detaches a comment from a DEFINE_HEX_SPOTWELD_ASSEMBLY.
 * @param Comment Comment that will be detached from the DEFINE_HEX_SPOTWELD_ASSEMBLY
 */
    DetachComment(Comment: Comment): void;

/**
 * Starts an interactive editing panel.
 * @param modal If this window is modal (blocks the user from doing anything else in PRIMER until this window is dismissed). If omitted the window will be modal.
 */
    Edit(modal?: boolean): void;

/**
 * Adds an error for DEFINE_HEX_SPOTWELD_ASSEMBLY. For more details on checking see the Check class.
 * @param message The error message to give
 * @param details An optional detailed error message
 */
    Error(message: string, details?: string): void;

/**
 * Returns the first DEFINE_HEX_SPOTWELD_ASSEMBLY in the model.
 * @param Model Model to get first DEFINE_HEX_SPOTWELD_ASSEMBLY in
 * @returns HexSpotweldAssembly object (or null if there are no DEFINE_HEX_SPOTWELD_ASSEMBLYs in the model).
 */
    static First(Model: Model): HexSpotweldAssembly;

/**
 * Returns the first free DEFINE_HEX_SPOTWELD_ASSEMBLY label in the model. Also see HexSpotweldAssembly.LastFreeLabel(), HexSpotweldAssembly.NextFreeLabel() and Model.FirstFreeItemLabel().
 * @param Model Model to get first free DEFINE_HEX_SPOTWELD_ASSEMBLY label in
 * @param layer Include file (0 for the main file) to search for labels in (Equivalent to First free in layer in editing panels). If omitted the whole model will be used (Equivalent to First free in editing panels).
 * @returns HexSpotweldAssembly label.
 */
    static FirstFreeLabel(Model: Model, layer?: number): number;

/**
 * Flags all of the DEFINE_HEX_SPOTWELD_ASSEMBLYs in the model with a defined flag.
 * @param Model Model that all DEFINE_HEX_SPOTWELD_ASSEMBLYs will be flagged in
 * @param flag Flag to set on the DEFINE_HEX_SPOTWELD_ASSEMBLYs
 */
    static FlagAll(Model: Model, flag: number): void;

/**
 * Checks if the DEFINE_HEX_SPOTWELD_ASSEMBLY is flagged or not.
 * @param flag Flag to test on the DEFINE_HEX_SPOTWELD_ASSEMBLY
 * @returns true if flagged, false if not.
 */
    Flagged(flag: number): boolean;

/**
 * Calls a function for each DEFINE_HEX_SPOTWELD_ASSEMBLY in the model. Note that ForEach has been designed to make looping over DEFINE_HEX_SPOTWELD_ASSEMBLYs as fast as possible and so has some limitations. Firstly, a single temporary HexSpotweldAssembly object is created and on each function call it is updated with the current DEFINE_HEX_SPOTWELD_ASSEMBLY data. This means that you should not try to store the HexSpotweldAssembly object for later use (e.g. in an array) as it is temporary. Secondly, you cannot create new DEFINE_HEX_SPOTWELD_ASSEMBLYs inside a ForEach loop. 
 * @param Model Model that all DEFINE_HEX_SPOTWELD_ASSEMBLYs are in
 * @param func Function to call for each DEFINE_HEX_SPOTWELD_ASSEMBLY
 * @param extra An optional extra object/array/string etc that will appended to arguments when calling the function
 */
    static ForEach(Model: Model, func: () => void, extra?: any): void;

/**
 * Returns an array of HexSpotweldAssembly objects or properties for all of the DEFINE_HEX_SPOTWELD_ASSEMBLYs in a model in PRIMER. If the optional property argument is not given then an array of HexSpotweldAssembly objects is returned. If the property argument is given, that property value for each DEFINE_HEX_SPOTWELD_ASSEMBLY is returned in the array instead of a HexSpotweldAssembly object
 * @param Model Model to get DEFINE_HEX_SPOTWELD_ASSEMBLYs from
 * @param property Name for property to get for all DEFINE_HEX_SPOTWELD_ASSEMBLYs in the model
 * @returns Array of HexSpotweldAssembly objects or properties
 */
    static GetAll(Model: Model, property?: string): HexSpotweldAssembly[];

/**
 * Extracts the comments associated to a DEFINE_HEX_SPOTWELD_ASSEMBLY.
 * @returns Array of Comment objects (or null if there are no comments associated to the node).
 */
    GetComments(): Comment[];

/**
 * Returns an array of HexSpotweldAssembly objects for all of the flagged DEFINE_HEX_SPOTWELD_ASSEMBLYs in a model in PRIMER If the optional property argument is not given then an array of HexSpotweldAssembly objects is returned. If the property argument is given, then that property value for each DEFINE_HEX_SPOTWELD_ASSEMBLY is returned in the array instead of a HexSpotweldAssembly object
 * @param Model Model to get DEFINE_HEX_SPOTWELD_ASSEMBLYs from
 * @param flag Flag set on the DEFINE_HEX_SPOTWELD_ASSEMBLYs that you want to retrieve
 * @param property Name for property to get for all flagged DEFINE_HEX_SPOTWELD_ASSEMBLYs in the model
 * @returns Array of HexSpotweldAssembly objects or properties
 */
    static GetFlagged(Model: Model, flag: number, property?: string): HexSpotweldAssembly[];

/**
 * Returns the HexSpotweldAssembly object for a DEFINE_HEX_SPOTWELD_ASSEMBLY ID.
 * @param Model Model to find the DEFINE_HEX_SPOTWELD_ASSEMBLY in
 * @param number number of the DEFINE_HEX_SPOTWELD_ASSEMBLY you want the HexSpotweldAssembly object for
 * @returns HexSpotweldAssembly object (or null if DEFINE_HEX_SPOTWELD_ASSEMBLY does not exist).
 */
    static GetFromID(Model: Model, number: number): HexSpotweldAssembly;

/**
 * Checks if a HexSpotweldAssembly property is a parameter or not. Note that object properties that are parameters are normally returned as the integer or float parameter values as that is virtually always what the user would want. For this function to work the JavaScript interpreter must use the parameter name instead of the value. This can be done by setting the Options.property_parameter_names option to true before calling the function and then resetting it to false afterwards.. This behaviour can also temporarily be switched by using the HexSpotweldAssembly.ViewParameters() method and 'method chaining' (see the examples below).
 * @param prop DEFINE_HEX_SPOTWELD_ASSEMBLY property to get parameter for
 * @returns Parameter object if property is a parameter, null if not.
 */
    GetParameter(prop: string): Parameter;

/**
 * Returns the keyword for this HexSpotweldAssembly (*DEFINE_HEX_SPOTWELD_ASSEMBLY). Note that a carriage return is not added. See also HexSpotweldAssembly.KeywordCards()
 * @returns string containing the keyword.
 */
    Keyword(): string;

/**
 * Returns the keyword cards for the HexSpotweldAssem. Note that a carriage return is not added. See also HexSpotweldAssembly.Keyword()
 * @returns string containing the cards.
 */
    KeywordCards(): string;

/**
 * Returns the last DEFINE_HEX_SPOTWELD_ASSEMBLY in the model.
 * @param Model Model to get last DEFINE_HEX_SPOTWELD_ASSEMBLY in
 * @returns HexSpotweldAssembly object (or null if there are no DEFINE_HEX_SPOTWELD_ASSEMBLYs in the model).
 */
    static Last(Model: Model): HexSpotweldAssembly;

/**
 * Returns the last free DEFINE_HEX_SPOTWELD_ASSEMBLY label in the model. Also see HexSpotweldAssembly.FirstFreeLabel(), HexSpotweldAssembly.NextFreeLabel() and see Model.LastFreeItemLabel()
 * @param Model Model to get last free DEFINE_HEX_SPOTWELD_ASSEMBLY label in
 * @param layer Include file (0 for the main file) to search for labels in (Equivalent to Highest free in layer in editing panels). If omitted the whole model will be used.
 * @returns HexSpotweldAssembly label.
 */
    static LastFreeLabel(Model: Model, layer?: number): number;

/**
 * Returns the next DEFINE_HEX_SPOTWELD_ASSEMBLY in the model.
 * @returns HexSpotweldAssembly object (or null if there are no more DEFINE_HEX_SPOTWELD_ASSEMBLYs in the model).
 */
    Next(): HexSpotweldAssembly;

/**
 * Returns the next free (highest+1) DEFINE_HEX_SPOTWELD_ASSEMBLY label in the model. Also see HexSpotweldAssembly.FirstFreeLabel(), HexSpotweldAssembly.LastFreeLabel() and Model.NextFreeItemLabel()
 * @param Model Model to get next free DEFINE_HEX_SPOTWELD_ASSEMBLY label in
 * @param layer Include file (0 for the main file) to search for labels in (Equivalent to Highest+1 in layer in editing panels). If omitted the whole model will be used (Equivalent to Highest+1 in editing panels).
 * @returns HexSpotweldAssembly label.
 */
    static NextFreeLabel(Model: Model, layer?: number): number;

/**
 * Returns the previous DEFINE_HEX_SPOTWELD_ASSEMBLY in the model.
 * @returns HexSpotweldAssembly object (or null if there are no more DEFINE_HEX_SPOTWELD_ASSEMBLYs in the model).
 */
    Previous(): HexSpotweldAssembly;

/**
 * Renumbers all of the DEFINE_HEX_SPOTWELD_ASSEMBLYs in the model.
 * @param Model Model that all DEFINE_HEX_SPOTWELD_ASSEMBLYs will be renumbered in
 * @param start Start point for renumbering
 */
    static RenumberAll(Model: Model, start: number): void;

/**
 * Renumbers all of the flagged DEFINE_HEX_SPOTWELD_ASSEMBLYs in the model.
 * @param Model Model that all the flagged DEFINE_HEX_SPOTWELD_ASSEMBLYs will be renumbered in
 * @param flag Flag set on the DEFINE_HEX_SPOTWELD_ASSEMBLYs that you want to renumber
 * @param start Start point for renumbering
 */
    static RenumberFlagged(Model: Model, flag: number, start: number): void;

/**
 * Allows the user to select DEFINE_HEX_SPOTWELD_ASSEMBLYs using standard PRIMER object menus.
 * @param flag Flag to use when selecting DEFINE_HEX_SPOTWELD_ASSEMBLYs
 * @param prompt Text to display as a prompt to the user
 * @param limit If the argument is a Model then only DEFINE_HEX_SPOTWELD_ASSEMBLYs from that model can be selected. If the argument is a Flag then only DEFINE_HEX_SPOTWELD_ASSEMBLYs that are flagged with limit can be selected (limit should be different to flag). If omitted, or null, any DEFINE_HEX_SPOTWELD_ASSEMBLYs can be selected. from any model.
 * @param modal If selection is modal (blocks the user from doing anything else in PRIMER until this window is dismissed). If omitted the selection will be modal.
 * @returns Number of DEFINE_HEX_SPOTWELD_ASSEMBLYs selected or null if menu cancelled
 */
    static Select(flag: number, prompt: string, limit?: Model | number, modal?: boolean): number;

/**
 * Sets a flag on the DEFINE_HEX_SPOTWELD_ASSEMBLY.
 * @param flag Flag to set on the DEFINE_HEX_SPOTWELD_ASSEMBLY
 */
    SetFlag(flag: number): void;

/**
 * Returns the total number of DEFINE_HEX_SPOTWELD_ASSEMBLYs in the model.
 * @param Model Model to get total for
 * @param exists true if only existing DEFINE_HEX_SPOTWELD_ASSEMBLYs should be counted. If false or omitted referenced but undefined DEFINE_HEX_SPOTWELD_ASSEMBLYs will also be included in the total.
 * @returns number of DEFINE_HEX_SPOTWELD_ASSEMBLYs
 */
    static Total(Model: Model, exists?: boolean): number;

/**
 * Unsets a defined flag on all of the DEFINE_HEX_SPOTWELD_ASSEMBLYs in the model.
 * @param Model Model that the defined flag for all DEFINE_HEX_SPOTWELD_ASSEMBLYs will be unset in
 * @param flag Flag to unset on the DEFINE_HEX_SPOTWELD_ASSEMBLYs
 */
    static UnflagAll(Model: Model, flag: number): void;

/**
 * Object properties that are parameters are normally returned as the integer or float parameter values as that is virtually always what the user would want. This function temporarily changes the behaviour so that if a property is a parameter the parameter name is returned instead. This can be used with 'method chaining' (see the example below) to make sure a property argument is correct.
 * @returns HexSpotweldAssembly object.
 */
    ViewParameters(): HexSpotweldAssembly;

/**
 * Adds a warning for DEFINE_HEX_SPOTWELD_ASSEMBLY. For more details on checking see the Check class.
 * @param message The warning message to give
 * @param details An optional detailed warning message
 */
    Warning(message: string, details?: string): void;

/**
 * Returns the cross references for this DEFINE_HEX_SPOTWELD_ASSEMBLY.
 * @returns Xrefs object.
 */
    Xrefs(): Xrefs;

/**
 * Create a new *DEFINE_HEX_SPOTWELD_ASSEMBLY object.
 * @param Model Model that Hex Spotweld Assembly will be created in
 * @param options Options for creating the HexSpotweldAssembly
 * @returns HexSpotweldAssembly object
 */
    constructor(Model: Model, options: HexSpotweldAssembly_constructor_Argument_options);

/**
 * Creates a string containing the HexSpotweldAssem data in keyword format. Note that this contains the keyword header and the keyword cards. See also HexSpotweldAssembly.Keyword() and HexSpotweldAssembly.KeywordCards().
 * @returns string
 */
    toString(): string;

/** EID 1 */
    eid1: number;
/** EID 10 */
    eid10: number;
/** EID 11 */
    eid11: number;
/** EID 12 */
    eid12: number;
/** EID 13 */
    eid13: number;
/** EID 14 */
    eid14: number;
/** EID 15 */
    eid15: number;
/** EID 16 */
    eid16: number;
/** EID 2 */
    eid2: number;
/** EID 3 */
    eid3: number;
/** EID 4 */
    eid4: number;
/** EID 5 */
    eid5: number;
/** EID 6 */
    eid6: number;
/** EID 7 */
    eid7: number;
/** EID 8 */
    eid8: number;
/** EID 9 */
    eid9: number;
/** true if DEFINE_HEX_SPOTWELD_ASSEMBLY exists, false if referred to but not defined. */
    readonly exists: boolean;
/** *DEFINE_HEX_SPOTWELD_ASSEMBLY id */
    id: number;
/** The Include file number that the DEFINE_HEX_SPOTWELD_ASSEMBLY is in. */
    include: number;
/** The Model number that the DEFINE_HEX_SPOTWELD_ASSEMBLY is in. */
    readonly model: number;
/** *DEFINE_HEX_SPOTWELD_ASSEMBLY opt */
    opt: number;
/** Title (optional) */
    title: string;
}

declare class History {
/**
 * Blanks all of the database histories in the model.
 * @param Model Model that all database histories will be blanked in
 * @param type The database history type. Can be History.ACOUSTIC or History.BEAM or History.BEAM_SET or History.DISCRETE or History.DISCRETE_SET or History.NODE or History.NODE_SET or History.SEATBELT or History.SHELL or History.SHELL_SET or History.SOLID or History.SOLID_SET or History.SPH or History.SPH_SET or History.TSHELL or History.TSHELL_SET or History.ALL_TYPES. If omitted, applied to all database history types.
 * @param redraw If model should be redrawn or not. If omitted redraw is false. If you want to do several (un)blanks and only redraw after the last one then use false for all redraws apart from the last one. Alternatively you can redraw using View.Redraw().
 */
    static BlankAll(Model: Model, type?: number, redraw?: boolean): void;

/**
 * Blanks all of the flagged database histories in the model.
 * @param Model Model that all the flagged database histories will be blanked in
 * @param flag Flag set on the database histories that you want to blank
 * @param type The database history type. Can be History.ACOUSTIC or History.BEAM or History.BEAM_SET or History.DISCRETE or History.DISCRETE_SET or History.NODE or History.NODE_SET or History.SEATBELT or History.SHELL or History.SHELL_SET or History.SOLID or History.SOLID_SET or History.SPH or History.SPH_SET or History.TSHELL or History.TSHELL_SET or History.ALL_TYPES. If omitted, applied to all database history types.
 * @param redraw If model should be redrawn or not. If omitted redraw is false. If you want to do several (un)blanks and only redraw after the last one then use false for all redraws apart from the last one. Alternatively you can redraw using View.Redraw().
 */
    static BlankFlagged(Model: Model, flag: number, type?: number, redraw?: boolean): void;

/**
 * Checks if the database history is blanked or not.
 * @returns true if blanked, false if not.
 */
    Blanked(): boolean;

/**
 * Clears a flag on the database history.
 * @param flag Flag to clear on the database history
 */
    ClearFlag(flag: number): void;

/**
 * Starts an interactive editing panel to create a database history.
 * @param Model Model that the database history will be created in
 * @param type The database history type. Can be History.ACOUSTIC or History.BEAM or History.BEAM_SET or History.DISCRETE or History.DISCRETE_SET or History.NODE or History.NODE_SET or History.SEATBELT or History.SHELL or History.SHELL_SET or History.SOLID or History.SOLID_SET or History.SPH or History.SPH_SET or History.TSHELL or History.TSHELL_SET.
 * @param modal If this window is modal (blocks the user from doing anything else in PRIMER until this window is dismissed). If omitted the window will be modal.
 * @returns History object (or null if not made)
 */
    static Create(Model: Model, type: number, modal?: boolean): History;

/**
 * Starts an interactive editing panel to edit the database history.
 * @param modal If this window is modal (blocks the user from doing anything else in PRIMER until this window is dismissed). If omitted the window will be modal.
 */
    Edit(modal?: boolean): void;

/**
 * Returns the first database history in the model.
 * @param Model Model to get first database history in
 * @param type The database history type. Can be History.ACOUSTIC or History.BEAM or History.BEAM_SET or History.DISCRETE or History.DISCRETE_SET or History.NODE or History.NODE_SET or History.SEATBELT or History.SHELL or History.SHELL_SET or History.SOLID or History.SOLID_SET or History.SPH or History.SPH_SET or History.TSHELL or History.TSHELL_SET or History.ALL_TYPES. If omitted, applied to all database history types.
 * @returns History object (or null if there are no database histories in the model).
 */
    static First(Model: Model, type?: number): History;

/**
 * Flags all of the database histories in the model with a defined flag.
 * @param Model Model that all database histories will be flagged in
 * @param flag Flag to set on the database histories
 * @param type The database history type. Can be History.ACOUSTIC or History.BEAM or History.BEAM_SET or History.DISCRETE or History.DISCRETE_SET or History.NODE or History.NODE_SET or History.SEATBELT or History.SHELL or History.SHELL_SET or History.SOLID or History.SOLID_SET or History.SPH or History.SPH_SET or History.TSHELL or History.TSHELL_SET or History.ALL_TYPES. If omitted, applied to all database history types.
 */
    static FlagAll(Model: Model, flag: number, type?: number): void;

/**
 * Checks if the database history is flagged or not.
 * @param flag Flag to clear on the database history
 * @returns true if flagged, false if not.
 */
    Flagged(flag: number): boolean;

/**
 * Returns an array of History objects for all of the database histories in a models in PRIMER
 * @param Model Model to get database histories from
 * @param type The database history type. Can be History.ACOUSTIC or History.BEAM or History.BEAM_SET or History.DISCRETE or History.DISCRETE_SET or History.NODE or History.NODE_SET or History.SEATBELT or History.SHELL or History.SHELL_SET or History.SOLID or History.SOLID_SET or History.SPH or History.SPH_SET or History.TSHELL or History.TSHELL_SET or History.ALL_TYPES. If omitted, applied to all database history types.
 * @returns Array of History objects
 */
    static GetAll(Model: Model, type?: number): History[];

/**
 * Returns the History object for a database history ID.
 * @param Model Model to find the database history in
 * @param database_history_number number of the database history you want the History object for
 * @returns History object (or null if database history does not exist).
 */
    static GetFromID(Model: Model, database_history_number: number): History;

/**
 * Returns the keyword for this database history (*DATABASE_HISTORY). Note that a carriage return is not added. See also History.KeywordCards()
 * @returns string containing the keyword.
 */
    Keyword(): string;

/**
 * Returns the keyword cards for the database history. Note that a carriage return is not added. See also History.Keyword()
 * @returns string containing the cards.
 */
    KeywordCards(): string;

/**
 * Returns the last database history in the model.
 * @param Model Model to get last database history in
 * @param type The database history type. Can be History.ACOUSTIC or History.BEAM or History.BEAM_SET or History.DISCRETE or History.DISCRETE_SET or History.NODE or History.NODE_SET or History.SEATBELT or History.SHELL or History.SHELL_SET or History.SOLID or History.SOLID_SET or History.SPH or History.SPH_SET or History.TSHELL or History.TSHELL_SET or History.ALL_TYPES. If omitted, applied to all database history types.
 * @returns History object (or null if there are no database histories in the model).
 */
    static Last(Model: Model, type?: number): History;

/**
 * Returns the next database history in the model.
 * @returns History object (or null if there are no more database histories in the model).
 */
    Next(): History;

/**
 * Allows the user to pick a database history.
 * @param prompt Text to display as a prompt to the user
 * @param limit If the argument is a Model then only database histories from that model can be picked. If the argument is a Flag then only database histories that are flagged with limit can be selected. If omitted, or null, any database histories from any model can be selected. from any model.
 * @param modal If picking is modal (blocks the user from doing anything else in PRIMER until this window is dismissed). If omitted the pick will be modal.
 * @returns History object (or null if not picked)
 */
    static Pick(prompt: string, limit?: Model | number, modal?: boolean): History;

/**
 * Returns the previous database history in the model.
 * @returns History object (or null if there are no more database histories in the model).
 */
    Previous(): History;

/**
 * Allows the user to select database histories using standard PRIMER object menus.
 * @param flag Flag to use when selecting database histories
 * @param prompt Text to display as a prompt to the user
 * @param limit If the argument is a Model then only database histories from that model can be selected. If the argument is a Flag then only database histories that are flagged with limit can be selected (limit should be different to flag). If omitted, or null, any database histories from any model can be selected.
 * @param modal If selection is modal (blocks the user from doing anything else in PRIMER until this window is dismissed). If omitted the selection will be modal.
 * @returns Number of items selected or null if menu cancelled
 */
    static Select(flag: number, prompt: string, limit?: Model | number, modal?: boolean): number;

/**
 * Sets a flag on the database history.
 * @param flag Flag to set on the database history
 */
    SetFlag(flag: number): void;

/**
 * Sketches the database history. The database history will be sketched until you either call History.Unsketch(), History.UnsketchAll(), Model.UnsketchAll(), or delete the model
 * @param redraw If model should be redrawn or not after the database history is sketched. If omitted redraw is true. If you want to sketch several database histories and only redraw after the last one then use false for redraw and call View.Redraw().
 */
    Sketch(redraw?: boolean): void;

/**
 * Sketches all of the flagged database histories in the model.
 * @param Model Model that all the flagged database histories will be sketched in
 * @param flag Flag set on the database histories that you want to sketch
 * @param type The database history type. Can be History.ACOUSTIC or History.BEAM or History.BEAM_SET or History.DISCRETE or History.DISCRETE_SET or History.NODE or History.NODE_SET or History.SEATBELT or History.SHELL or History.SHELL_SET or History.SOLID or History.SOLID_SET or History.SPH or History.SPH_SET or History.TSHELL or History.TSHELL_SET or History.ALL_TYPES. If omitted, applied to all database history types.
 * @param redraw If model should be redrawn or not. If omitted redraw is true. If you want to do several (un)sketches and only redraw after the last one then use false for all redraws apart from the last one. Alternatively you can redraw using View.Redraw().
 */
    static SketchFlagged(Model: Model, flag: number, type?: number, redraw?: boolean): void;

/**
 * Unblanks all of the database histories in the model.
 * @param Model Model that all database histories will be unblanked in
 * @param redraw If model should be redrawn or not. If omitted redraw is false. If you want to do several (un)blanks and only redraw after the last one then use false for all redraws apart from the last one. Alternatively you can redraw using View.Redraw().
 * @param type The database history type. Can be History.ACOUSTIC or History.BEAM or History.BEAM_SET or History.DISCRETE or History.DISCRETE_SET or History.NODE or History.NODE_SET or History.SEATBELT or History.SHELL or History.SHELL_SET or History.SOLID or History.SOLID_SET or History.SPH or History.SPH_SET or History.TSHELL or History.TSHELL_SET or History.ALL_TYPES. If omitted, applied to all database history types.
 */
    static UnblankAll(Model: Model, redraw?: boolean, type?: number): void;

/**
 * Unblanks all of the flagged database histories in the model.
 * @param Model Model that the flagged database histories will be unblanked in
 * @param flag Flag set on the database histories that you want to unblank
 * @param type The database history type. Can be History.ACOUSTIC or History.BEAM or History.BEAM_SET or History.DISCRETE or History.DISCRETE_SET or History.NODE or History.NODE_SET or History.SEATBELT or History.SHELL or History.SHELL_SET or History.SOLID or History.SOLID_SET or History.SPH or History.SPH_SET or History.TSHELL or History.TSHELL_SET or History.ALL_TYPES. If omitted, applied to all database history types.
 * @param redraw If model should be redrawn or not. If omitted redraw is false. If you want to do several (un)blanks and only redraw after the last one then use false for all redraws apart from the last one. Alternatively you can redraw using View.Redraw().
 */
    static UnblankFlagged(Model: Model, flag: number, type?: number, redraw?: boolean): void;

/**
 * Unsets a defined flag on all of the database histories in the model.
 * @param Model Model that the defined flag for all database histories will be unset in
 * @param flag Flag to unset on the database histories
 * @param type The database history type. Can be History.ACOUSTIC or History.BEAM or History.BEAM_SET or History.DISCRETE or History.DISCRETE_SET or History.NODE or History.NODE_SET or History.SEATBELT or History.SHELL or History.SHELL_SET or History.SOLID or History.SOLID_SET or History.SPH or History.SPH_SET or History.TSHELL or History.TSHELL_SET or History.ALL_TYPES. If omitted, applied to all database history types.
 */
    static UnflagAll(Model: Model, flag: number, type?: number): void;

/**
 * Unsketches the database history.
 * @param redraw If model should be redrawn or not after the database history is unsketched. If omitted redraw is true. If you want to unsketch several database histories and only redraw after the last one then use false for redraw and call View.Redraw().
 */
    Unsketch(redraw?: boolean): void;

/**
 * Unsketches all database histories.
 * @param Model Model that all database histories will be unblanked in
 * @param redraw If model should be redrawn or not after the database histories are unsketched. If omitted redraw is true. If you want to unsketch several things and only redraw after the last one then use false for redraw and call View.Redraw().
 */
    static UnsketchAll(Model: Model, redraw?: boolean): void;

/**
 * Unsketches all flagged database histories.
 * @param Model Model that all database histories will be unblanked in
 * @param flag Flag set on the database histories that you want to sketch
 * @param type The database history type. Can be History.ACOUSTIC or History.BEAM or History.BEAM_SET or History.DISCRETE or History.DISCRETE_SET or History.NODE or History.NODE_SET or History.SEATBELT or History.SHELL or History.SHELL_SET or History.SOLID or History.SOLID_SET or History.SPH or History.SPH_SET or History.TSHELL or History.TSHELL_SET or History.ALL_TYPES. If omitted, applied to all database history types.
 * @param redraw If model should be redrawn or not after the database histories are unsketched. If omitted redraw is true. If you want to unsketch several things and only redraw after the last one then use false for redraw and call View.Redraw().
 */
    static UnsketchFlagged(Model: Model, flag: number, type?: number, redraw?: boolean): void;

/**
 * Returns the cross references for this database history.
 * @returns Xrefs object.
 */
    Xrefs(): Xrefs;

/**
 * Create a new History object.
 * @param Model Model that database history will be created in
 * @param type Entity type
 * @param id ID of the item
 * @param heading Optional heading
 * @returns History object
 */
    constructor(Model: Model, type: number, id: number, heading?: string);

/**
 * Creates a string containing the database history data in keyword format. Note that this contains the keyword header and the keyword cards. See also History.Keyword() and History.KeywordCards().
 * @returns string
 */
    toString(): string;

/** Coordinate system ID for _LOCAL */
    cid: number;
/** true if database history exists, false if referred to but not defined. */
    readonly exists: boolean;
/** Optional heading */
    heading: string;
/** High frequency flag for _LOCAL */
    hfo: number;
/** ID of the item */
    id: number;
/** The Include file number that the database history is in. */
    include: number;
/** Turns _LOCAL on or off */
    local: boolean;
/** The Model number that the database history is in. */
    model: number;
/** Output reference for _LOCAL */
    ref: number;
/** The database history type. Can be History.ACOUSTIC or History.BEAM or History.BEAM_SET or History.DISCRETE or History.DISCRETE_SET or History.NODE or History.NODE_SET or History.SEATBELT or History.SHELL or History.SHELL_SET or History.SOLID or History.SOLID_SET or History.SPH or History.SPH_SET or History.TSHELL or History.TSHELL_SET. */
    readonly type: number;
/** ACOUSTIC is *DATABASE_HISTORY_ACOUSTIC. */
    static ACOUSTIC: number;
/** All *DATABASE_HISTORY_ types. */
    static ALL_TYPES: number;
/** BEAM is *DATABASE_HISTORY_BEAM. */
    static BEAM: number;
/** BEAM_SET is *DATABASE_HISTORY_BEAM_SET. */
    static BEAM_SET: number;
/** DISCRETE is *DATABASE_HISTORY_DISCRETE. */
    static DISCRETE: number;
/** DISCRETE_SET is *DATABASE_HISTORY_DISCRETE_SET. */
    static DISCRETE_SET: number;
/** NODE is *DATABASE_HISTORY_NODE. */
    static NODE: number;
/** NODE_SET is *DATABASE_HISTORY_NODE_SET. */
    static NODE_SET: number;
/** SEATBELT is *DATABASE_HISTORY_SEATBELT. */
    static SEATBELT: number;
/** SHELL is *DATABASE_HISTORY_SHELL. */
    static SHELL: number;
/** SHELL_SET is *DATABASE_HISTORY_SHELL_SET. */
    static SHELL_SET: number;
/** SOLID is *DATABASE_HISTORY_SOLID. */
    static SOLID: number;
/** SOLID_SET is *DATABASE_HISTORY_SOLID_SET. */
    static SOLID_SET: number;
/** SPH is *DATABASE_HISTORY_SPH. */
    static SPH: number;
/** SPH_SET is *DATABASE_HISTORY_SPH_SET. */
    static SPH_SET: number;
/** TSHELL is *DATABASE_HISTORY_TSHELL. */
    static TSHELL: number;
/** TSHELL_SET is *DATABASE_HISTORY_TSHELL_SET. */
    static TSHELL_SET: number;
}

declare class Hourglass {
/**
 * Associates a comment with a hourglass.
 * @param Comment Comment that will be attached to the hourglass
 */
    AssociateComment(Comment: Comment): void;

/**
 * Starts an edit panel in Browse mode.
 * @param modal If this window is modal (blocks the user from doing anything else in PRIMER until this window is dismissed). If omitted the window will be modal.
 */
    Browse(modal?: boolean): void;

/**
 * Clears a flag on the hourglass.
 * @param flag Flag to clear on the hourglass
 */
    ClearFlag(flag: number): void;

/**
 * Copies the hourglass. The target include of the copied hourglass can be set using Options.copy_target_include.
 * @param range If you want to keep the copied item in the range specified for the current include. Default value is false. To set current include, use Include.MakeCurrentLayer().
 * @returns Hourglass object
 */
    Copy(range?: boolean): Hourglass;

/**
 * Starts an interactive editing panel to create a hourglass
 * @param Model Model that the hourglass will be created in.
 * @param modal If this window is modal (blocks the user from doing anything else in PRIMER until this window is dismissed). If omitted the window will be modal.
 * @returns Hourglass object (or null if not made).
 */
    static Create(Model: Model, modal?: boolean): Hourglass;

/**
 * Detaches a comment from a hourglass.
 * @param Comment Comment that will be detached from the hourglass
 */
    DetachComment(Comment: Comment): void;

/**
 * Starts an interactive editing panel.
 * @param modal If this window is modal (blocks the user from doing anything else in PRIMER until this window is dismissed). If omitted the window will be modal.
 */
    Edit(modal?: boolean): void;

/**
 * Adds an error for hourglass. For more details on checking see the Check class.
 * @param message The error message to give
 * @param details An optional detailed error message
 */
    Error(message: string, details?: string): void;

/**
 * Returns the first hourglass in the model.
 * @param Model Model to get first hourglass in
 * @returns Hourglass object (or null if there are no hourglasss in the model).
 */
    static First(Model: Model): Hourglass;

/**
 * Returns the first free hourglass label in the model. Also see Hourglass.LastFreeLabel(), Hourglass.NextFreeLabel() and Model.FirstFreeItemLabel().
 * @param Model Model to get first free hourglass label in
 * @param layer Include file (0 for the main file) to search for labels in (Equivalent to First free in layer in editing panels). If omitted the whole model will be used (Equivalent to First free in editing panels).
 * @returns Hourglass label.
 */
    static FirstFreeLabel(Model: Model, layer?: number): number;

/**
 * Flags all of the hourglasss in the model with a defined flag.
 * @param Model Model that all hourglasss will be flagged in
 * @param flag Flag to set on the hourglasss
 */
    static FlagAll(Model: Model, flag: number): void;

/**
 * Checks if the hourglass is flagged or not.
 * @param flag Flag to test on the hourglass
 * @returns true if flagged, false if not.
 */
    Flagged(flag: number): boolean;

/**
 * Calls a function for each hourglass in the model. Note that ForEach has been designed to make looping over hourglasss as fast as possible and so has some limitations. Firstly, a single temporary Hourglass object is created and on each function call it is updated with the current hourglass data. This means that you should not try to store the Hourglass object for later use (e.g. in an array) as it is temporary. Secondly, you cannot create new hourglasss inside a ForEach loop. 
 * @param Model Model that all hourglasss are in
 * @param func Function to call for each hourglass
 * @param extra An optional extra object/array/string etc that will appended to arguments when calling the function
 */
    static ForEach(Model: Model, func: () => void, extra?: any): void;

/**
 * Returns an array of Hourglass objects or properties for all of the hourglasss in a model in PRIMER. If the optional property argument is not given then an array of Hourglass objects is returned. If the property argument is given, that property value for each hourglass is returned in the array instead of a Hourglass object
 * @param Model Model to get hourglasss from
 * @param property Name for property to get for all hourglasss in the model
 * @returns Array of Hourglass objects or properties
 */
    static GetAll(Model: Model, property?: string): Hourglass[];

/**
 * Extracts the comments associated to a hourglass.
 * @returns Array of Comment objects (or null if there are no comments associated to the node).
 */
    GetComments(): Comment[];

/**
 * Returns an array of Hourglass objects for all of the flagged hourglasss in a model in PRIMER If the optional property argument is not given then an array of Hourglass objects is returned. If the property argument is given, then that property value for each hourglass is returned in the array instead of a Hourglass object
 * @param Model Model to get hourglasss from
 * @param flag Flag set on the hourglasss that you want to retrieve
 * @param property Name for property to get for all flagged hourglasss in the model
 * @returns Array of Hourglass objects or properties
 */
    static GetFlagged(Model: Model, flag: number, property?: string): Hourglass[];

/**
 * Returns the Hourglass object for a hourglass ID.
 * @param Model Model to find the hourglass in
 * @param number number of the hourglass you want the Hourglass object for
 * @returns Hourglass object (or null if hourglass does not exist).
 */
    static GetFromID(Model: Model, number: number): Hourglass;

/**
 * Checks if a Hourglass property is a parameter or not. Note that object properties that are parameters are normally returned as the integer or float parameter values as that is virtually always what the user would want. For this function to work the JavaScript interpreter must use the parameter name instead of the value. This can be done by setting the Options.property_parameter_names option to true before calling the function and then resetting it to false afterwards.. This behaviour can also temporarily be switched by using the Hourglass.ViewParameters() method and 'method chaining' (see the examples below).
 * @param prop hourglass property to get parameter for
 * @returns Parameter object if property is a parameter, null if not.
 */
    GetParameter(prop: string): Parameter;

/**
 * Returns the keyword for this hourglass (*HOURGLASS). Note that a carriage return is not added. See also Hourglass.KeywordCards()
 * @returns string containing the keyword.
 */
    Keyword(): string;

/**
 * Returns the keyword cards for the hourglass. Note that a carriage return is not added. See also Hourglass.Keyword()
 * @returns string containing the cards.
 */
    KeywordCards(): string;

/**
 * Returns the last hourglass in the model.
 * @param Model Model to get last hourglass in
 * @returns Hourglass object (or null if there are no hourglasss in the model).
 */
    static Last(Model: Model): Hourglass;

/**
 * Returns the last free hourglass label in the model. Also see Hourglass.FirstFreeLabel(), Hourglass.NextFreeLabel() and see Model.LastFreeItemLabel()
 * @param Model Model to get last free hourglass label in
 * @param layer Include file (0 for the main file) to search for labels in (Equivalent to Highest free in layer in editing panels). If omitted the whole model will be used.
 * @returns Hourglass label.
 */
    static LastFreeLabel(Model: Model, layer?: number): number;

/**
 * Returns the next hourglass in the model.
 * @returns Hourglass object (or null if there are no more hourglasss in the model).
 */
    Next(): Hourglass;

/**
 * Returns the next free (highest+1) hourglass label in the model. Also see Hourglass.FirstFreeLabel(), Hourglass.LastFreeLabel() and Model.NextFreeItemLabel()
 * @param Model Model to get next free hourglass label in
 * @param layer Include file (0 for the main file) to search for labels in (Equivalent to Highest+1 in layer in editing panels). If omitted the whole model will be used (Equivalent to Highest+1 in editing panels).
 * @returns Hourglass label.
 */
    static NextFreeLabel(Model: Model, layer?: number): number;

/**
 * Returns the previous hourglass in the model.
 * @returns Hourglass object (or null if there are no more hourglasss in the model).
 */
    Previous(): Hourglass;

/**
 * Renumbers all of the hourglasss in the model.
 * @param Model Model that all hourglasss will be renumbered in
 * @param start Start point for renumbering
 */
    static RenumberAll(Model: Model, start: number): void;

/**
 * Renumbers all of the flagged hourglasss in the model.
 * @param Model Model that all the flagged hourglasss will be renumbered in
 * @param flag Flag set on the hourglasss that you want to renumber
 * @param start Start point for renumbering
 */
    static RenumberFlagged(Model: Model, flag: number, start: number): void;

/**
 * Allows the user to select hourglasss using standard PRIMER object menus.
 * @param flag Flag to use when selecting hourglasss
 * @param prompt Text to display as a prompt to the user
 * @param limit If the argument is a Model then only hourglasss from that model can be selected. If the argument is a Flag then only hourglasss that are flagged with limit can be selected (limit should be different to flag). If omitted, or null, any hourglasss can be selected. from any model.
 * @param modal If selection is modal (blocks the user from doing anything else in PRIMER until this window is dismissed). If omitted the selection will be modal.
 * @returns Number of hourglasss selected or null if menu cancelled
 */
    static Select(flag: number, prompt: string, limit?: Model | number, modal?: boolean): number;

/**
 * Sets a flag on the hourglass.
 * @param flag Flag to set on the hourglass
 */
    SetFlag(flag: number): void;

/**
 * Returns the total number of hourglasss in the model.
 * @param Model Model to get total for
 * @param exists true if only existing hourglasss should be counted. If false or omitted referenced but undefined hourglasss will also be included in the total.
 * @returns number of hourglasss
 */
    static Total(Model: Model, exists?: boolean): number;

/**
 * Unsets a defined flag on all of the hourglasss in the model.
 * @param Model Model that the defined flag for all hourglasss will be unset in
 * @param flag Flag to unset on the hourglasss
 */
    static UnflagAll(Model: Model, flag: number): void;

/**
 * Object properties that are parameters are normally returned as the integer or float parameter values as that is virtually always what the user would want. This function temporarily changes the behaviour so that if a property is a parameter the parameter name is returned instead. This can be used with 'method chaining' (see the example below) to make sure a property argument is correct.
 * @returns Hourglass object.
 */
    ViewParameters(): Hourglass;

/**
 * Adds a warning for hourglass. For more details on checking see the Check class.
 * @param message The warning message to give
 * @param details An optional detailed warning message
 */
    Warning(message: string, details?: string): void;

/**
 * Returns the cross references for this hourglass.
 * @returns Xrefs object.
 */
    Xrefs(): Xrefs;

/**
 * Create a new Hourglass object.
 * @param Model Model that hourglass will be created in
 * @param hgid Hourglass number or character label
 * @param title Title for the hourglass
 * @returns Hourglass object
 */
    constructor(Model: Model, hgid: number | string, title?: string);

/**
 * Creates a string containing the hourglass data in keyword format. Note that this contains the keyword header and the keyword cards. See also Hourglass.Keyword() and Hourglass.KeywordCards().
 * @returns string
 */
    toString(): string;

/** true if hourglass exists, false if referred to but not defined. */
    readonly exists: boolean;
/** Hourglass number or character label. Also see the label property which is an alternative name for this. */
    hgid: number|string;
/** Bulk viscosity type */
    ibq: number;
/** Hourglass control type */
    ihq: number;
/** The Include file number that the hourglass is in. */
    include: number;
/** Hourglass number or character label. Also see the hgid property which is an alternative name for this. */
    label: number|string;
/** The Model number that the hourglass is in. */
    readonly model: number;
/** Quadratic bulk viscosity coefficient */
    q1: number;
/** Linear bulk viscosity coefficient */
    q2: number;
/** Coefficient for shell bending */
    qb: number;
/** Hourglass coefficient */
    qm: number;
/** Coefficient for shell warping */
    qw: number;
/** Title for hourglass */
    title: string;
/** Viscous damping coefficient */
    vdc: number;
}


/** Object function argument in IGA1DBrep constructor */
interface IGA1DBrep_constructor_Argument_details {
    /** ID of the IGA 1D Brep. */
    brid: number;
}

declare class IGA1DBrep {
/**
 * Associates a comment with a IGA 1D Brep.
 * @param Comment Comment that will be attached to the IGA 1D Brep
 */
    AssociateComment(Comment: Comment): void;

/**
 * Starts an edit panel in Browse mode.
 * @param modal If this window is modal (blocks the user from doing anything else in PRIMER until this window is dismissed). If omitted the window will be modal.
 */
    Browse(modal?: boolean): void;

/**
 * Clears a flag on the IGA 1D Brep.
 * @param flag Flag to clear on the IGA 1D Brep
 */
    ClearFlag(flag: number): void;

/**
 * Copies the IGA 1D Brep. The target include of the copied IGA 1D Brep can be set using Options.copy_target_include.
 * @param range If you want to keep the copied item in the range specified for the current include. Default value is false. To set current include, use Include.MakeCurrentLayer().
 * @returns IGA1DBrep object
 */
    Copy(range?: boolean): IGA1DBrep;

/**
 * Starts an interactive editing panel to create a IGA 1D Brep
 * @param Model Model that the IGA 1D Brep will be created in.
 * @param modal If this window is modal (blocks the user from doing anything else in PRIMER until this window is dismissed). If omitted the window will be modal.
 * @returns IGA1DBrep object (or null if not made).
 */
    static Create(Model: Model, modal?: boolean): IGA1DBrep;

/**
 * Detaches a comment from a IGA 1D Brep.
 * @param Comment Comment that will be detached from the IGA 1D Brep
 */
    DetachComment(Comment: Comment): void;

/**
 * Starts an interactive editing panel.
 * @param modal If this window is modal (blocks the user from doing anything else in PRIMER until this window is dismissed). If omitted the window will be modal.
 */
    Edit(modal?: boolean): void;

/**
 * Adds an error for IGA 1D Brep. For more details on checking see the Check class.
 * @param message The error message to give
 * @param details An optional detailed error message
 */
    Error(message: string, details?: string): void;

/**
 * Returns the first IGA 1D Brep in the model.
 * @param Model Model to get first IGA 1D Brep in
 * @returns IGA1DBrep object (or null if there are no IGA 1D Breps in the model).
 */
    static First(Model: Model): IGA1DBrep;

/**
 * Returns the first free IGA 1D Brep label in the model. Also see IGA1DBrep.LastFreeLabel(), IGA1DBrep.NextFreeLabel() and Model.FirstFreeItemLabel().
 * @param Model Model to get first free IGA 1D Brep label in
 * @param layer Include file (0 for the main file) to search for labels in (Equivalent to First free in layer in editing panels). If omitted the whole model will be used (Equivalent to First free in editing panels).
 * @returns IGA1DBrep label.
 */
    static FirstFreeLabel(Model: Model, layer?: number): number;

/**
 * Flags all of the IGA 1D Breps in the model with a defined flag.
 * @param Model Model that all IGA 1D Breps will be flagged in
 * @param flag Flag to set on the IGA 1D Breps
 */
    static FlagAll(Model: Model, flag: number): void;

/**
 * Checks if the IGA 1D Brep is flagged or not.
 * @param flag Flag to test on the IGA 1D Brep
 * @returns true if flagged, false if not.
 */
    Flagged(flag: number): boolean;

/**
 * Calls a function for each IGA 1D Brep in the model. Note that ForEach has been designed to make looping over IGA 1D Breps as fast as possible and so has some limitations. Firstly, a single temporary IGA1DBrep object is created and on each function call it is updated with the current IGA 1D Brep data. This means that you should not try to store the IGA1DBrep object for later use (e.g. in an array) as it is temporary. Secondly, you cannot create new IGA 1D Breps inside a ForEach loop. 
 * @param Model Model that all IGA 1D Breps are in
 * @param func Function to call for each IGA 1D Brep
 * @param extra An optional extra object/array/string etc that will appended to arguments when calling the function
 */
    static ForEach(Model: Model, func: () => void, extra?: any): void;

/**
 * Returns an array of IGA1DBrep objects or properties for all of the IGA 1D Breps in a model in PRIMER. If the optional property argument is not given then an array of IGA1DBrep objects is returned. If the property argument is given, that property value for each IGA 1D Brep is returned in the array instead of a IGA1DBrep object
 * @param Model Model to get IGA 1D Breps from
 * @param property Name for property to get for all IGA 1D Breps in the model
 * @returns Array of IGA1DBrep objects or properties
 */
    static GetAll(Model: Model, property?: string): IGA1DBrep[];

/**
 * Extracts the comments associated to a IGA 1D Brep.
 * @returns Array of Comment objects (or null if there are no comments associated to the node).
 */
    GetComments(): Comment[];

/**
 * Returns the data for eid in *IGA_1D_BREP.
 * @param index Index you want the data for. Note that indices start at 0.
 * @returns The ID of parametric edge, *IGA_EDGE_UVW.
 */
    GetData(index: number): number;

/**
 * Returns an array of IGA1DBrep objects for all of the flagged IGA 1D Breps in a model in PRIMER If the optional property argument is not given then an array of IGA1DBrep objects is returned. If the property argument is given, then that property value for each IGA 1D Brep is returned in the array instead of a IGA1DBrep object
 * @param Model Model to get IGA 1D Breps from
 * @param flag Flag set on the IGA 1D Breps that you want to retrieve
 * @param property Name for property to get for all flagged IGA 1D Breps in the model
 * @returns Array of IGA1DBrep objects or properties
 */
    static GetFlagged(Model: Model, flag: number, property?: string): IGA1DBrep[];

/**
 * Returns the IGA1DBrep object for a IGA 1D Brep ID.
 * @param Model Model to find the IGA 1D Brep in
 * @param number number of the IGA 1D Brep you want the IGA1DBrep object for
 * @returns IGA1DBrep object (or null if IGA 1D Brep does not exist).
 */
    static GetFromID(Model: Model, number: number): IGA1DBrep;

/**
 * Checks if a IGA1DBrep property is a parameter or not. Note that object properties that are parameters are normally returned as the integer or float parameter values as that is virtually always what the user would want. For this function to work the JavaScript interpreter must use the parameter name instead of the value. This can be done by setting the Options.property_parameter_names option to true before calling the function and then resetting it to false afterwards.. This behaviour can also temporarily be switched by using the IGA1DBrep.ViewParameters() method and 'method chaining' (see the examples below).
 * @param prop IGA 1D Brep property to get parameter for
 * @returns Parameter object if property is a parameter, null if not.
 */
    GetParameter(prop: string): Parameter;

/**
 * Returns the keyword for this IGA 1d brep (*IGA_1D_BREP). Note that a carriage return is not added. See also IGA1DBrep.KeywordCards()
 * @returns string containing the keyword.
 */
    Keyword(): string;

/**
 * Returns the keyword cards for the IGA 1d brep. Note that a carriage return is not added. See also IGA1DBrep.Keyword()
 * @returns string containing the cards.
 */
    KeywordCards(): string;

/**
 * Returns the last IGA 1D Brep in the model.
 * @param Model Model to get last IGA 1D Brep in
 * @returns IGA1DBrep object (or null if there are no IGA 1D Breps in the model).
 */
    static Last(Model: Model): IGA1DBrep;

/**
 * Returns the last free IGA 1D Brep label in the model. Also see IGA1DBrep.FirstFreeLabel(), IGA1DBrep.NextFreeLabel() and see Model.LastFreeItemLabel()
 * @param Model Model to get last free IGA 1D Brep label in
 * @param layer Include file (0 for the main file) to search for labels in (Equivalent to Highest free in layer in editing panels). If omitted the whole model will be used.
 * @returns IGA1DBrep label.
 */
    static LastFreeLabel(Model: Model, layer?: number): number;

/**
 * Returns the next IGA 1D Brep in the model.
 * @returns IGA1DBrep object (or null if there are no more IGA 1D Breps in the model).
 */
    Next(): IGA1DBrep;

/**
 * Returns the next free (highest+1) IGA 1D Brep label in the model. Also see IGA1DBrep.FirstFreeLabel(), IGA1DBrep.LastFreeLabel() and Model.NextFreeItemLabel()
 * @param Model Model to get next free IGA 1D Brep label in
 * @param layer Include file (0 for the main file) to search for labels in (Equivalent to Highest+1 in layer in editing panels). If omitted the whole model will be used (Equivalent to Highest+1 in editing panels).
 * @returns IGA1DBrep label.
 */
    static NextFreeLabel(Model: Model, layer?: number): number;

/**
 * Returns the previous IGA 1D Brep in the model.
 * @returns IGA1DBrep object (or null if there are no more IGA 1D Breps in the model).
 */
    Previous(): IGA1DBrep;

/**
 * Removes eid for an index in *IGA_1D_BREP.
 * @param index The index you want to delete eid for. Note that indices start at 0, not 1.
 */
    RemoveData(index: number): void;

/**
 * Renumbers all of the IGA 1D Breps in the model.
 * @param Model Model that all IGA 1D Breps will be renumbered in
 * @param start Start point for renumbering
 */
    static RenumberAll(Model: Model, start: number): void;

/**
 * Renumbers all of the flagged IGA 1D Breps in the model.
 * @param Model Model that all the flagged IGA 1D Breps will be renumbered in
 * @param flag Flag set on the IGA 1D Breps that you want to renumber
 * @param start Start point for renumbering
 */
    static RenumberFlagged(Model: Model, flag: number, start: number): void;

/**
 * Allows the user to select IGA 1D Breps using standard PRIMER object menus.
 * @param flag Flag to use when selecting IGA 1D Breps
 * @param prompt Text to display as a prompt to the user
 * @param limit If the argument is a Model then only IGA 1D Breps from that model can be selected. If the argument is a Flag then only IGA 1D Breps that are flagged with limit can be selected (limit should be different to flag). If omitted, or null, any IGA 1D Breps can be selected. from any model.
 * @param modal If selection is modal (blocks the user from doing anything else in PRIMER until this window is dismissed). If omitted the selection will be modal.
 * @returns Number of IGA 1D Breps selected or null if menu cancelled
 */
    static Select(flag: number, prompt: string, limit?: Model | number, modal?: boolean): number;

/**
 * Sets eid in *IGA_1D_BREP.
 * @param index Index you want to set the eid for. Note that indices start at 0.
 * @param eid The ID of parametric edge.
 */
    SetData(index: number, eid: number): void;

/**
 * Sets a flag on the IGA 1D Brep.
 * @param flag Flag to set on the IGA 1D Brep
 */
    SetFlag(flag: number): void;

/**
 * Sketches the IGA 1D Brep. The IGA 1D Brep will be sketched until you either call IGA1DBrep.Unsketch(), IGA1DBrep.UnsketchAll(), Model.UnsketchAll(), or delete the model
 * @param redraw If model should be redrawn or not after the IGA 1D Brep is sketched. If omitted redraw is true. If you want to sketch several IGA 1D Breps and only redraw after the last one then use false for redraw and call View.Redraw().
 */
    Sketch(redraw?: boolean): void;

/**
 * Sketches all of the flagged IGA 1D Breps in the model. The IGA 1D Breps will be sketched until you either call IGA1DBrep.Unsketch(), IGA1DBrep.UnsketchFlagged(), Model.UnsketchAll(), or delete the model
 * @param Model Model that all the flagged IGA 1D Breps will be sketched in
 * @param flag Flag set on the IGA 1D Breps that you want to sketch
 * @param redraw If model should be redrawn or not after the IGA 1D Breps are sketched. If omitted redraw is true. If you want to sketch flagged IGA 1D Breps several times and only redraw after the last one then use false for redraw and call View.Redraw().
 */
    static SketchFlagged(Model: Model, flag: number, redraw?: boolean): void;

/**
 * Returns the total number of IGA 1D Breps in the model.
 * @param Model Model to get total for
 * @param exists true if only existing IGA 1D Breps should be counted. If false or omitted referenced but undefined IGA 1D Breps will also be included in the total.
 * @returns number of IGA 1D Breps
 */
    static Total(Model: Model, exists?: boolean): number;

/**
 * Unsets a defined flag on all of the IGA 1D Breps in the model.
 * @param Model Model that the defined flag for all IGA 1D Breps will be unset in
 * @param flag Flag to unset on the IGA 1D Breps
 */
    static UnflagAll(Model: Model, flag: number): void;

/**
 * Unsketches the IGA 1D Brep.
 * @param redraw If model should be redrawn or not after the IGA 1D Brep is unsketched. If omitted redraw is true. If you want to unsketch several IGA 1D Breps and only redraw after the last one then use false for redraw and call View.Redraw().
 */
    Unsketch(redraw?: boolean): void;

/**
 * Unsketches all IGA 1D Breps.
 * @param Model Model that all IGA 1D Breps will be unblanked in
 * @param redraw If model should be redrawn or not after the IGA 1D Breps are unsketched. If omitted redraw is true. If you want to unsketch several things and only redraw after the last one then use false for redraw and call View.Redraw().
 */
    static UnsketchAll(Model: Model, redraw?: boolean): void;

/**
 * Unsketches all flagged IGA 1D Breps in the model.
 * @param Model Model that all IGA 1D Breps will be unsketched in
 * @param flag Flag set on the IGA 1D Breps that you want to unsketch
 * @param redraw If model should be redrawn or not after the IGA 1D Breps are unsketched. If omitted redraw is true. If you want to unsketch several things and only redraw after the last one then use false for redraw and call View.Redraw().
 */
    static UnsketchFlagged(Model: Model, flag: number, redraw?: boolean): void;

/**
 * Object properties that are parameters are normally returned as the integer or float parameter values as that is virtually always what the user would want. This function temporarily changes the behaviour so that if a property is a parameter the parameter name is returned instead. This can be used with 'method chaining' (see the example below) to make sure a property argument is correct.
 * @returns IGA1DBrep object.
 */
    ViewParameters(): IGA1DBrep;

/**
 * Adds a warning for IGA 1D Brep. For more details on checking see the Check class.
 * @param message The warning message to give
 * @param details An optional detailed warning message
 */
    Warning(message: string, details?: string): void;

/**
 * Returns the cross references for this IGA 1D Brep.
 * @returns Xrefs object.
 */
    Xrefs(): Xrefs;

/**
 * Create a new IGA1DBrep object.
 * @param Model Model that IGA 1d brep will be created in
 * @param details Details for creating the IGA1DBrep
 * @returns IGA1DBrep object
 */
    constructor(Model: Model, details: IGA1DBrep_constructor_Argument_details);

/**
 * Creates a string containing the IGA 1d brep data in keyword format. Note that this contains the keyword header and the keyword cards. See also IGA1DBrep.Keyword() and IGA1DBrep.KeywordCards().
 * @returns string
 */
    toString(): string;

/** IGA1DBrep number. Also see the label property which is an alternative name for this. */
    brid: number;
/** Number of Parametric Edge IDs (eid) added in IGA_1D_BREP. Can be used as index to add a new entry. */
    readonly entries: number;
/** true if IGA 1D Brep exists, false if referred to but not defined. */
    readonly exists: boolean;
/** The Include file number that the IGA 1D Brep is in. */
    include: number;
/** IGA1DBrep number. Also see the brid property which is an alternative name for this. */
    label: number;
/** The Model number that the IGA 1D Brep is in. */
    readonly model: number;
}


/** Object function argument in IGA1DNurbsUVW constructor */
interface IGA1DNurbsUVW_constructor_Argument_details {
    /** Number of control points in the local r-direction. */
    nr?: number;
    /** ID of the IGA 1D Nurbs UVW. */
    patchid: number;
    /** Polynomial degree of the basis in the local r-direction. */
    pr?: number;
    /** Knot vector type in the local r-direction. */
    unir?: number;
}

declare class IGA1DNurbsUVW {
/**
 * Associates a comment with a IGA 1D Nurbs UVW.
 * @param Comment Comment that will be attached to the IGA 1D Nurbs UVW
 */
    AssociateComment(Comment: Comment): void;

/**
 * Starts an edit panel in Browse mode.
 * @param modal If this window is modal (blocks the user from doing anything else in PRIMER until this window is dismissed). If omitted the window will be modal.
 */
    Browse(modal?: boolean): void;

/**
 * Clears a flag on the IGA 1D Nurbs UVW.
 * @param flag Flag to clear on the IGA 1D Nurbs UVW
 */
    ClearFlag(flag: number): void;

/**
 * Copies the IGA 1D Nurbs UVW. The target include of the copied IGA 1D Nurbs UVW can be set using Options.copy_target_include.
 * @param range If you want to keep the copied item in the range specified for the current include. Default value is false. To set current include, use Include.MakeCurrentLayer().
 * @returns IGA1DNurbsUVW object
 */
    Copy(range?: boolean): IGA1DNurbsUVW;

/**
 * Starts an interactive editing panel to create a IGA 1D Nurbs UVW
 * @param Model Model that the IGA 1D Nurbs UVW will be created in.
 * @param modal If this window is modal (blocks the user from doing anything else in PRIMER until this window is dismissed). If omitted the window will be modal.
 * @returns IGA1DNurbsUVW object (or null if not made).
 */
    static Create(Model: Model, modal?: boolean): IGA1DNurbsUVW;

/**
 * Detaches a comment from a IGA 1D Nurbs UVW.
 * @param Comment Comment that will be detached from the IGA 1D Nurbs UVW
 */
    DetachComment(Comment: Comment): void;

/**
 * Starts an interactive editing panel.
 * @param modal If this window is modal (blocks the user from doing anything else in PRIMER until this window is dismissed). If omitted the window will be modal.
 */
    Edit(modal?: boolean): void;

/**
 * Adds an error for IGA 1D Nurbs UVW. For more details on checking see the Check class.
 * @param message The error message to give
 * @param details An optional detailed error message
 */
    Error(message: string, details?: string): void;

/**
 * Returns the first IGA 1D Nurbs UVW in the model.
 * @param Model Model to get first IGA 1D Nurbs UVW in
 * @returns IGA1DNurbsUVW object (or null if there are no IGA 1D Nurbs UVWs in the model).
 */
    static First(Model: Model): IGA1DNurbsUVW;

/**
 * Returns the first free IGA 1D Nurbs UVW label in the model. Also see IGA1DNurbsUVW.LastFreeLabel(), IGA1DNurbsUVW.NextFreeLabel() and Model.FirstFreeItemLabel().
 * @param Model Model to get first free IGA 1D Nurbs UVW label in
 * @param layer Include file (0 for the main file) to search for labels in (Equivalent to First free in layer in editing panels). If omitted the whole model will be used (Equivalent to First free in editing panels).
 * @returns IGA1DNurbsUVW label.
 */
    static FirstFreeLabel(Model: Model, layer?: number): number;

/**
 * Flags all of the IGA 1D Nurbs UVWs in the model with a defined flag.
 * @param Model Model that all IGA 1D Nurbs UVWs will be flagged in
 * @param flag Flag to set on the IGA 1D Nurbs UVWs
 */
    static FlagAll(Model: Model, flag: number): void;

/**
 * Checks if the IGA 1D Nurbs UVW is flagged or not.
 * @param flag Flag to test on the IGA 1D Nurbs UVW
 * @returns true if flagged, false if not.
 */
    Flagged(flag: number): boolean;

/**
 * Calls a function for each IGA 1D Nurbs UVW in the model. Note that ForEach has been designed to make looping over IGA 1D Nurbs UVWs as fast as possible and so has some limitations. Firstly, a single temporary IGA1DNurbsUVW object is created and on each function call it is updated with the current IGA 1D Nurbs UVW data. This means that you should not try to store the IGA1DNurbsUVW object for later use (e.g. in an array) as it is temporary. Secondly, you cannot create new IGA 1D Nurbs UVWs inside a ForEach loop. 
 * @param Model Model that all IGA 1D Nurbs UVWs are in
 * @param func Function to call for each IGA 1D Nurbs UVW
 * @param extra An optional extra object/array/string etc that will appended to arguments when calling the function
 */
    static ForEach(Model: Model, func: () => void, extra?: any): void;

/**
 * Returns an array of IGA1DNurbsUVW objects or properties for all of the IGA 1D Nurbs UVWs in a model in PRIMER. If the optional property argument is not given then an array of IGA1DNurbsUVW objects is returned. If the property argument is given, that property value for each IGA 1D Nurbs UVW is returned in the array instead of a IGA1DNurbsUVW object
 * @param Model Model to get IGA 1D Nurbs UVWs from
 * @param property Name for property to get for all IGA 1D Nurbs UVWs in the model
 * @returns Array of IGA1DNurbsUVW objects or properties
 */
    static GetAll(Model: Model, property?: string): IGA1DNurbsUVW[];

/**
 * Extracts the comments associated to a IGA 1D Nurbs UVW.
 * @returns Array of Comment objects (or null if there are no comments associated to the node).
 */
    GetComments(): Comment[];

/**
 * Returns the data for a specific control point an an array. There are nr control points.
 * @param index Index you want the control data for. Note that indices start at 0.
 * @returns An array containing the control point data.
 */
    GetControlPoint(index: number): number[];

/**
 * Returns an array of IGA1DNurbsUVW objects for all of the flagged IGA 1D Nurbs UVWs in a model in PRIMER If the optional property argument is not given then an array of IGA1DNurbsUVW objects is returned. If the property argument is given, then that property value for each IGA 1D Nurbs UVW is returned in the array instead of a IGA1DNurbsUVW object
 * @param Model Model to get IGA 1D Nurbs UVWs from
 * @param flag Flag set on the IGA 1D Nurbs UVWs that you want to retrieve
 * @param property Name for property to get for all flagged IGA 1D Nurbs UVWs in the model
 * @returns Array of IGA1DNurbsUVW objects or properties
 */
    static GetFlagged(Model: Model, flag: number, property?: string): IGA1DNurbsUVW[];

/**
 * Returns the IGA1DNurbsUVW object for a IGA 1D Nurbs UVW ID.
 * @param Model Model to find the IGA 1D Nurbs UVW in
 * @param number number of the IGA 1D Nurbs UVW you want the IGA1DNurbsUVW object for
 * @returns IGA1DNurbsUVW object (or null if IGA 1D Nurbs UVW does not exist).
 */
    static GetFromID(Model: Model, number: number): IGA1DNurbsUVW;

/**
 * The data for Knot values in the local r-direction.
 * @param index Index you want the knot value for. Note that indices start at 0.
 * @returns Knot values in the local r-direction.
 */
    GetKnotVector(index: number): number;

/**
 * Checks if a IGA1DNurbsUVW property is a parameter or not. Note that object properties that are parameters are normally returned as the integer or float parameter values as that is virtually always what the user would want. For this function to work the JavaScript interpreter must use the parameter name instead of the value. This can be done by setting the Options.property_parameter_names option to true before calling the function and then resetting it to false afterwards.. This behaviour can also temporarily be switched by using the IGA1DNurbsUVW.ViewParameters() method and 'method chaining' (see the examples below).
 * @param prop IGA 1D Nurbs UVW property to get parameter for
 * @returns Parameter object if property is a parameter, null if not.
 */
    GetParameter(prop: string): Parameter;

/**
 * Returns the keyword for this IGA 1d nurbs uvw (*IGA_1D_NURBS_UVW). Note that a carriage return is not added. See also IGA1DNurbsUVW.KeywordCards()
 * @returns string containing the keyword.
 */
    Keyword(): string;

/**
 * Returns the keyword cards for the IGA 1d nurbs uvw. Note that a carriage return is not added. See also IGA1DNurbsUVW.Keyword()
 * @returns string containing the cards.
 */
    KeywordCards(): string;

/**
 * Returns the last IGA 1D Nurbs UVW in the model.
 * @param Model Model to get last IGA 1D Nurbs UVW in
 * @returns IGA1DNurbsUVW object (or null if there are no IGA 1D Nurbs UVWs in the model).
 */
    static Last(Model: Model): IGA1DNurbsUVW;

/**
 * Returns the last free IGA 1D Nurbs UVW label in the model. Also see IGA1DNurbsUVW.FirstFreeLabel(), IGA1DNurbsUVW.NextFreeLabel() and see Model.LastFreeItemLabel()
 * @param Model Model to get last free IGA 1D Nurbs UVW label in
 * @param layer Include file (0 for the main file) to search for labels in (Equivalent to Highest free in layer in editing panels). If omitted the whole model will be used.
 * @returns IGA1DNurbsUVW label.
 */
    static LastFreeLabel(Model: Model, layer?: number): number;

/**
 * Returns the next IGA 1D Nurbs UVW in the model.
 * @returns IGA1DNurbsUVW object (or null if there are no more IGA 1D Nurbs UVWs in the model).
 */
    Next(): IGA1DNurbsUVW;

/**
 * Returns the next free (highest+1) IGA 1D Nurbs UVW label in the model. Also see IGA1DNurbsUVW.FirstFreeLabel(), IGA1DNurbsUVW.LastFreeLabel() and Model.NextFreeItemLabel()
 * @param Model Model to get next free IGA 1D Nurbs UVW label in
 * @param layer Include file (0 for the main file) to search for labels in (Equivalent to Highest+1 in layer in editing panels). If omitted the whole model will be used (Equivalent to Highest+1 in editing panels).
 * @returns IGA1DNurbsUVW label.
 */
    static NextFreeLabel(Model: Model, layer?: number): number;

/**
 * Returns the previous IGA 1D Nurbs UVW in the model.
 * @returns IGA1DNurbsUVW object (or null if there are no more IGA 1D Nurbs UVWs in the model).
 */
    Previous(): IGA1DNurbsUVW;

/**
 * Renumbers all of the IGA 1D Nurbs UVWs in the model.
 * @param Model Model that all IGA 1D Nurbs UVWs will be renumbered in
 * @param start Start point for renumbering
 */
    static RenumberAll(Model: Model, start: number): void;

/**
 * Renumbers all of the flagged IGA 1D Nurbs UVWs in the model.
 * @param Model Model that all the flagged IGA 1D Nurbs UVWs will be renumbered in
 * @param flag Flag set on the IGA 1D Nurbs UVWs that you want to renumber
 * @param start Start point for renumbering
 */
    static RenumberFlagged(Model: Model, flag: number, start: number): void;

/**
 * Allows the user to select IGA 1D Nurbs UVWs using standard PRIMER object menus.
 * @param flag Flag to use when selecting IGA 1D Nurbs UVWs
 * @param prompt Text to display as a prompt to the user
 * @param limit If the argument is a Model then only IGA 1D Nurbs UVWs from that model can be selected. If the argument is a Flag then only IGA 1D Nurbs UVWs that are flagged with limit can be selected (limit should be different to flag). If omitted, or null, any IGA 1D Nurbs UVWs can be selected. from any model.
 * @param modal If selection is modal (blocks the user from doing anything else in PRIMER until this window is dismissed). If omitted the selection will be modal.
 * @returns Number of IGA 1D Nurbs UVWs selected or null if menu cancelled
 */
    static Select(flag: number, prompt: string, limit?: Model | number, modal?: boolean): number;

/**
 * Set the data for a specific control point. For each control point there will be control poin U, V, W and control weight WGT. There are nr control points.
 * @param index Index you want the control point data for. Note that indices start at 0.
 * @param data Array containing the control point data. The array length should be 4.
 */
    SetControlPoint(index: number, data: number[]): void;

/**
 * Sets a flag on the IGA 1D Nurbs UVW.
 * @param flag Flag to set on the IGA 1D Nurbs UVW
 */
    SetFlag(flag: number): void;

/**
 * Sets knot value, r in *IGA_1D_NURBS_UVW. There are [nr + pr + 1] knot values.
 * @param index Index you want to set the r for. Note that indices start at 0.
 * @param r Knot values in the local r-direction
 */
    SetKnotVector(index: number, r: number): void;

/**
 * Sketches the IGA 1D Nurbs UVW. The IGA 1D Nurbs UVW will be sketched until you either call IGA1DNurbsUVW.Unsketch(), IGA1DNurbsUVW.UnsketchAll(), Model.UnsketchAll(), or delete the model
 * @param redraw If model should be redrawn or not after the IGA 1D Nurbs UVW is sketched. If omitted redraw is true. If you want to sketch several IGA 1D Nurbs UVWs and only redraw after the last one then use false for redraw and call View.Redraw().
 */
    Sketch(redraw?: boolean): void;

/**
 * Sketches all of the flagged IGA 1D Nurbs UVWs in the model. The IGA 1D Nurbs UVWs will be sketched until you either call IGA1DNurbsUVW.Unsketch(), IGA1DNurbsUVW.UnsketchFlagged(), Model.UnsketchAll(), or delete the model
 * @param Model Model that all the flagged IGA 1D Nurbs UVWs will be sketched in
 * @param flag Flag set on the IGA 1D Nurbs UVWs that you want to sketch
 * @param redraw If model should be redrawn or not after the IGA 1D Nurbs UVWs are sketched. If omitted redraw is true. If you want to sketch flagged IGA 1D Nurbs UVWs several times and only redraw after the last one then use false for redraw and call View.Redraw().
 */
    static SketchFlagged(Model: Model, flag: number, redraw?: boolean): void;

/**
 * Returns the total number of IGA 1D Nurbs UVWs in the model.
 * @param Model Model to get total for
 * @param exists true if only existing IGA 1D Nurbs UVWs should be counted. If false or omitted referenced but undefined IGA 1D Nurbs UVWs will also be included in the total.
 * @returns number of IGA 1D Nurbs UVWs
 */
    static Total(Model: Model, exists?: boolean): number;

/**
 * Unsets a defined flag on all of the IGA 1D Nurbs UVWs in the model.
 * @param Model Model that the defined flag for all IGA 1D Nurbs UVWs will be unset in
 * @param flag Flag to unset on the IGA 1D Nurbs UVWs
 */
    static UnflagAll(Model: Model, flag: number): void;

/**
 * Unsketches the IGA 1D Nurbs UVW.
 * @param redraw If model should be redrawn or not after the IGA 1D Nurbs UVW is unsketched. If omitted redraw is true. If you want to unsketch several IGA 1D Nurbs UVWs and only redraw after the last one then use false for redraw and call View.Redraw().
 */
    Unsketch(redraw?: boolean): void;

/**
 * Unsketches all IGA 1D Nurbs UVWs.
 * @param Model Model that all IGA 1D Nurbs UVWs will be unblanked in
 * @param redraw If model should be redrawn or not after the IGA 1D Nurbs UVWs are unsketched. If omitted redraw is true. If you want to unsketch several things and only redraw after the last one then use false for redraw and call View.Redraw().
 */
    static UnsketchAll(Model: Model, redraw?: boolean): void;

/**
 * Unsketches all flagged IGA 1D Nurbs UVWs in the model.
 * @param Model Model that all IGA 1D Nurbs UVWs will be unsketched in
 * @param flag Flag set on the IGA 1D Nurbs UVWs that you want to unsketch
 * @param redraw If model should be redrawn or not after the IGA 1D Nurbs UVWs are unsketched. If omitted redraw is true. If you want to unsketch several things and only redraw after the last one then use false for redraw and call View.Redraw().
 */
    static UnsketchFlagged(Model: Model, flag: number, redraw?: boolean): void;

/**
 * Object properties that are parameters are normally returned as the integer or float parameter values as that is virtually always what the user would want. This function temporarily changes the behaviour so that if a property is a parameter the parameter name is returned instead. This can be used with 'method chaining' (see the example below) to make sure a property argument is correct.
 * @returns IGA1DNurbsUVW object.
 */
    ViewParameters(): IGA1DNurbsUVW;

/**
 * Adds a warning for IGA 1D Nurbs UVW. For more details on checking see the Check class.
 * @param message The warning message to give
 * @param details An optional detailed warning message
 */
    Warning(message: string, details?: string): void;

/**
 * Returns the cross references for this IGA 1D Nurbs UVW.
 * @returns Xrefs object.
 */
    Xrefs(): Xrefs;

/**
 * Create a new IGA1DNurbsUVW object.
 * @param Model Model that IGA 1d nurbs uvw will be created in
 * @param details Details for creating the IGA1DNurbsUVW
 * @returns IGA1DNurbsUVW object
 */
    constructor(Model: Model, details: IGA1DNurbsUVW_constructor_Argument_details);

/**
 * Creates a string containing the IGA 1d nurbs uvw data in keyword format. Note that this contains the keyword header and the keyword cards. See also IGA1DNurbsUVW.Keyword() and IGA1DNurbsUVW.KeywordCards().
 * @returns string
 */
    toString(): string;

/** true if IGA 1D Nurbs UVW exists, false if referred to but not defined. */
    readonly exists: boolean;
/** The Include file number that the IGA 1D Nurbs UVW is in. */
    include: number;
/** IGA1DNurbsUVW number. Also see the patchid property which is an alternative name for this. */
    label: number;
/** The Model number that the IGA 1D Nurbs UVW is in. */
    readonly model: number;
/** Number of control points in the local r-direction. */
    nr: number;
/** IGA1DNurbsUVW number. Also see the label property which is an alternative name for this. */
    patchid: number;
/** Polynomial degree of the basis in the local r-direction. */
    pr: number;
/** First knot value in the local r-direction. */
    rfirst: number;
/** Last knot value in the local r-direction. */
    rlast: number;
/** Knot vector type in the local r-direction. */
    unir: number;
}


/** Object function argument in IGA1DNurbsXYZ constructor */
interface IGA1DNurbsXYZ_constructor_Argument_details {
    /** Number of control points in the local r-direction. */
    nr?: number;
    /** ID of the IGA 1D Nurbs XYZ. */
    patchid: number;
    /** Polynomial degree of the basis in the local r-direction. */
    pr?: number;
    /** Knot vector type in the local r-direction. */
    unir?: number;
}

declare class IGA1DNurbsXYZ {
/**
 * Associates a comment with a IGA 1D Nurbs XYZ.
 * @param Comment Comment that will be attached to the IGA 1D Nurbs XYZ
 */
    AssociateComment(Comment: Comment): void;

/**
 * Starts an edit panel in Browse mode.
 * @param modal If this window is modal (blocks the user from doing anything else in PRIMER until this window is dismissed). If omitted the window will be modal.
 */
    Browse(modal?: boolean): void;

/**
 * Clears a flag on the IGA 1D Nurbs XYZ.
 * @param flag Flag to clear on the IGA 1D Nurbs XYZ
 */
    ClearFlag(flag: number): void;

/**
 * Copies the IGA 1D Nurbs XYZ. The target include of the copied IGA 1D Nurbs XYZ can be set using Options.copy_target_include.
 * @param range If you want to keep the copied item in the range specified for the current include. Default value is false. To set current include, use Include.MakeCurrentLayer().
 * @returns IGA1DNurbsXYZ object
 */
    Copy(range?: boolean): IGA1DNurbsXYZ;

/**
 * Starts an interactive editing panel to create a IGA 1D Nurbs XYZ
 * @param Model Model that the IGA 1D Nurbs XYZ will be created in.
 * @param modal If this window is modal (blocks the user from doing anything else in PRIMER until this window is dismissed). If omitted the window will be modal.
 * @returns IGA1DNurbsXYZ object (or null if not made).
 */
    static Create(Model: Model, modal?: boolean): IGA1DNurbsXYZ;

/**
 * Detaches a comment from a IGA 1D Nurbs XYZ.
 * @param Comment Comment that will be detached from the IGA 1D Nurbs XYZ
 */
    DetachComment(Comment: Comment): void;

/**
 * Starts an interactive editing panel.
 * @param modal If this window is modal (blocks the user from doing anything else in PRIMER until this window is dismissed). If omitted the window will be modal.
 */
    Edit(modal?: boolean): void;

/**
 * Adds an error for IGA 1D Nurbs XYZ. For more details on checking see the Check class.
 * @param message The error message to give
 * @param details An optional detailed error message
 */
    Error(message: string, details?: string): void;

/**
 * Returns the first IGA 1D Nurbs XYZ in the model.
 * @param Model Model to get first IGA 1D Nurbs XYZ in
 * @returns IGA1DNurbsXYZ object (or null if there are no IGA 1D Nurbs XYZs in the model).
 */
    static First(Model: Model): IGA1DNurbsXYZ;

/**
 * Returns the first free IGA 1D Nurbs XYZ label in the model. Also see IGA1DNurbsXYZ.LastFreeLabel(), IGA1DNurbsXYZ.NextFreeLabel() and Model.FirstFreeItemLabel().
 * @param Model Model to get first free IGA 1D Nurbs XYZ label in
 * @param layer Include file (0 for the main file) to search for labels in (Equivalent to First free in layer in editing panels). If omitted the whole model will be used (Equivalent to First free in editing panels).
 * @returns IGA1DNurbsXYZ label.
 */
    static FirstFreeLabel(Model: Model, layer?: number): number;

/**
 * Flags all of the IGA 1D Nurbs XYZs in the model with a defined flag.
 * @param Model Model that all IGA 1D Nurbs XYZs will be flagged in
 * @param flag Flag to set on the IGA 1D Nurbs XYZs
 */
    static FlagAll(Model: Model, flag: number): void;

/**
 * Checks if the IGA 1D Nurbs XYZ is flagged or not.
 * @param flag Flag to test on the IGA 1D Nurbs XYZ
 * @returns true if flagged, false if not.
 */
    Flagged(flag: number): boolean;

/**
 * Calls a function for each IGA 1D Nurbs XYZ in the model. Note that ForEach has been designed to make looping over IGA 1D Nurbs XYZs as fast as possible and so has some limitations. Firstly, a single temporary IGA1DNurbsXYZ object is created and on each function call it is updated with the current IGA 1D Nurbs XYZ data. This means that you should not try to store the IGA1DNurbsXYZ object for later use (e.g. in an array) as it is temporary. Secondly, you cannot create new IGA 1D Nurbs XYZs inside a ForEach loop. 
 * @param Model Model that all IGA 1D Nurbs XYZs are in
 * @param func Function to call for each IGA 1D Nurbs XYZ
 * @param extra An optional extra object/array/string etc that will appended to arguments when calling the function
 */
    static ForEach(Model: Model, func: () => void, extra?: any): void;

/**
 * Returns an array of IGA1DNurbsXYZ objects or properties for all of the IGA 1D Nurbs XYZs in a model in PRIMER. If the optional property argument is not given then an array of IGA1DNurbsXYZ objects is returned. If the property argument is given, that property value for each IGA 1D Nurbs XYZ is returned in the array instead of a IGA1DNurbsXYZ object
 * @param Model Model to get IGA 1D Nurbs XYZs from
 * @param property Name for property to get for all IGA 1D Nurbs XYZs in the model
 * @returns Array of IGA1DNurbsXYZ objects or properties
 */
    static GetAll(Model: Model, property?: string): IGA1DNurbsXYZ[];

/**
 * Extracts the comments associated to a IGA 1D Nurbs XYZ.
 * @returns Array of Comment objects (or null if there are no comments associated to the node).
 */
    GetComments(): Comment[];

/**
 * Returns the data for a specific control point an an array. There are nr control points.
 * @param index Index you want the control data for. Note that indices start at 0.
 * @returns An array containing the control point data.
 */
    GetControlPoint(index: number): number[];

/**
 * Returns an array of IGA1DNurbsXYZ objects for all of the flagged IGA 1D Nurbs XYZs in a model in PRIMER If the optional property argument is not given then an array of IGA1DNurbsXYZ objects is returned. If the property argument is given, then that property value for each IGA 1D Nurbs XYZ is returned in the array instead of a IGA1DNurbsXYZ object
 * @param Model Model to get IGA 1D Nurbs XYZs from
 * @param flag Flag set on the IGA 1D Nurbs XYZs that you want to retrieve
 * @param property Name for property to get for all flagged IGA 1D Nurbs XYZs in the model
 * @returns Array of IGA1DNurbsXYZ objects or properties
 */
    static GetFlagged(Model: Model, flag: number, property?: string): IGA1DNurbsXYZ[];

/**
 * Returns the IGA1DNurbsXYZ object for a IGA 1D Nurbs XYZ ID.
 * @param Model Model to find the IGA 1D Nurbs XYZ in
 * @param number number of the IGA 1D Nurbs XYZ you want the IGA1DNurbsXYZ object for
 * @returns IGA1DNurbsXYZ object (or null if IGA 1D Nurbs XYZ does not exist).
 */
    static GetFromID(Model: Model, number: number): IGA1DNurbsXYZ;

/**
 * The data for Knot values in the local r-direction.
 * @param index Index you want the knot value for. Note that indices start at 0.
 * @returns Knot values in the local r-direction.
 */
    GetKnotVector(index: number): number;

/**
 * Checks if a IGA1DNurbsXYZ property is a parameter or not. Note that object properties that are parameters are normally returned as the integer or float parameter values as that is virtually always what the user would want. For this function to work the JavaScript interpreter must use the parameter name instead of the value. This can be done by setting the Options.property_parameter_names option to true before calling the function and then resetting it to false afterwards.. This behaviour can also temporarily be switched by using the IGA1DNurbsXYZ.ViewParameters() method and 'method chaining' (see the examples below).
 * @param prop IGA 1D Nurbs XYZ property to get parameter for
 * @returns Parameter object if property is a parameter, null if not.
 */
    GetParameter(prop: string): Parameter;

/**
 * Returns the keyword for this IGA 1d nurbs xyz (*IGA_1D_NURBS_XYZ). Note that a carriage return is not added. See also IGA1DNurbsXYZ.KeywordCards()
 * @returns string containing the keyword.
 */
    Keyword(): string;

/**
 * Returns the keyword cards for the IGA 1d nurbs xyz. Note that a carriage return is not added. See also IGA1DNurbsXYZ.Keyword()
 * @returns string containing the cards.
 */
    KeywordCards(): string;

/**
 * Returns the last IGA 1D Nurbs XYZ in the model.
 * @param Model Model to get last IGA 1D Nurbs XYZ in
 * @returns IGA1DNurbsXYZ object (or null if there are no IGA 1D Nurbs XYZs in the model).
 */
    static Last(Model: Model): IGA1DNurbsXYZ;

/**
 * Returns the last free IGA 1D Nurbs XYZ label in the model. Also see IGA1DNurbsXYZ.FirstFreeLabel(), IGA1DNurbsXYZ.NextFreeLabel() and see Model.LastFreeItemLabel()
 * @param Model Model to get last free IGA 1D Nurbs XYZ label in
 * @param layer Include file (0 for the main file) to search for labels in (Equivalent to Highest free in layer in editing panels). If omitted the whole model will be used.
 * @returns IGA1DNurbsXYZ label.
 */
    static LastFreeLabel(Model: Model, layer?: number): number;

/**
 * Returns the next IGA 1D Nurbs XYZ in the model.
 * @returns IGA1DNurbsXYZ object (or null if there are no more IGA 1D Nurbs XYZs in the model).
 */
    Next(): IGA1DNurbsXYZ;

/**
 * Returns the next free (highest+1) IGA 1D Nurbs XYZ label in the model. Also see IGA1DNurbsXYZ.FirstFreeLabel(), IGA1DNurbsXYZ.LastFreeLabel() and Model.NextFreeItemLabel()
 * @param Model Model to get next free IGA 1D Nurbs XYZ label in
 * @param layer Include file (0 for the main file) to search for labels in (Equivalent to Highest+1 in layer in editing panels). If omitted the whole model will be used (Equivalent to Highest+1 in editing panels).
 * @returns IGA1DNurbsXYZ label.
 */
    static NextFreeLabel(Model: Model, layer?: number): number;

/**
 * Returns the previous IGA 1D Nurbs XYZ in the model.
 * @returns IGA1DNurbsXYZ object (or null if there are no more IGA 1D Nurbs XYZs in the model).
 */
    Previous(): IGA1DNurbsXYZ;

/**
 * Renumbers all of the IGA 1D Nurbs XYZs in the model.
 * @param Model Model that all IGA 1D Nurbs XYZs will be renumbered in
 * @param start Start point for renumbering
 */
    static RenumberAll(Model: Model, start: number): void;

/**
 * Renumbers all of the flagged IGA 1D Nurbs XYZs in the model.
 * @param Model Model that all the flagged IGA 1D Nurbs XYZs will be renumbered in
 * @param flag Flag set on the IGA 1D Nurbs XYZs that you want to renumber
 * @param start Start point for renumbering
 */
    static RenumberFlagged(Model: Model, flag: number, start: number): void;

/**
 * Allows the user to select IGA 1D Nurbs XYZs using standard PRIMER object menus.
 * @param flag Flag to use when selecting IGA 1D Nurbs XYZs
 * @param prompt Text to display as a prompt to the user
 * @param limit If the argument is a Model then only IGA 1D Nurbs XYZs from that model can be selected. If the argument is a Flag then only IGA 1D Nurbs XYZs that are flagged with limit can be selected (limit should be different to flag). If omitted, or null, any IGA 1D Nurbs XYZs can be selected. from any model.
 * @param modal If selection is modal (blocks the user from doing anything else in PRIMER until this window is dismissed). If omitted the selection will be modal.
 * @returns Number of IGA 1D Nurbs XYZs selected or null if menu cancelled
 */
    static Select(flag: number, prompt: string, limit?: Model | number, modal?: boolean): number;

/**
 * Set the data for a specific control point. For each control point there will be control point X, Y, Z and control weight WGT. There are nr control points.
 * @param index Index you want the control point data for. Note that indices start at 0.
 * @param data Array containing the control point data. The array length should be 4.
 */
    SetControlPoint(index: number, data: number[]): void;

/**
 * Sets a flag on the IGA 1D Nurbs XYZ.
 * @param flag Flag to set on the IGA 1D Nurbs XYZ
 */
    SetFlag(flag: number): void;

/**
 * Sets knot value, r in *IGA_1D_NURBS_XYZ. There are [nr + pr + 1] knot values.
 * @param index Index you want to set the r for. Note that indices start at 0.
 * @param r Knot values in the local r-direction
 */
    SetKnotVector(index: number, r: number): void;

/**
 * Sketches the IGA 1D Nurbs XYZ. The IGA 1D Nurbs XYZ will be sketched until you either call IGA1DNurbsXYZ.Unsketch(), IGA1DNurbsXYZ.UnsketchAll(), Model.UnsketchAll(), or delete the model
 * @param redraw If model should be redrawn or not after the IGA 1D Nurbs XYZ is sketched. If omitted redraw is true. If you want to sketch several IGA 1D Nurbs XYZs and only redraw after the last one then use false for redraw and call View.Redraw().
 */
    Sketch(redraw?: boolean): void;

/**
 * Sketches all of the flagged IGA 1D Nurbs XYZs in the model. The IGA 1D Nurbs XYZs will be sketched until you either call IGA1DNurbsXYZ.Unsketch(), IGA1DNurbsXYZ.UnsketchFlagged(), Model.UnsketchAll(), or delete the model
 * @param Model Model that all the flagged IGA 1D Nurbs XYZs will be sketched in
 * @param flag Flag set on the IGA 1D Nurbs XYZs that you want to sketch
 * @param redraw If model should be redrawn or not after the IGA 1D Nurbs XYZs are sketched. If omitted redraw is true. If you want to sketch flagged IGA 1D Nurbs XYZs several times and only redraw after the last one then use false for redraw and call View.Redraw().
 */
    static SketchFlagged(Model: Model, flag: number, redraw?: boolean): void;

/**
 * Returns the total number of IGA 1D Nurbs XYZs in the model.
 * @param Model Model to get total for
 * @param exists true if only existing IGA 1D Nurbs XYZs should be counted. If false or omitted referenced but undefined IGA 1D Nurbs XYZs will also be included in the total.
 * @returns number of IGA 1D Nurbs XYZs
 */
    static Total(Model: Model, exists?: boolean): number;

/**
 * Unsets a defined flag on all of the IGA 1D Nurbs XYZs in the model.
 * @param Model Model that the defined flag for all IGA 1D Nurbs XYZs will be unset in
 * @param flag Flag to unset on the IGA 1D Nurbs XYZs
 */
    static UnflagAll(Model: Model, flag: number): void;

/**
 * Unsketches the IGA 1D Nurbs XYZ.
 * @param redraw If model should be redrawn or not after the IGA 1D Nurbs XYZ is unsketched. If omitted redraw is true. If you want to unsketch several IGA 1D Nurbs XYZs and only redraw after the last one then use false for redraw and call View.Redraw().
 */
    Unsketch(redraw?: boolean): void;

/**
 * Unsketches all IGA 1D Nurbs XYZs.
 * @param Model Model that all IGA 1D Nurbs XYZs will be unblanked in
 * @param redraw If model should be redrawn or not after the IGA 1D Nurbs XYZs are unsketched. If omitted redraw is true. If you want to unsketch several things and only redraw after the last one then use false for redraw and call View.Redraw().
 */
    static UnsketchAll(Model: Model, redraw?: boolean): void;

/**
 * Unsketches all flagged IGA 1D Nurbs XYZs in the model.
 * @param Model Model that all IGA 1D Nurbs XYZs will be unsketched in
 * @param flag Flag set on the IGA 1D Nurbs XYZs that you want to unsketch
 * @param redraw If model should be redrawn or not after the IGA 1D Nurbs XYZs are unsketched. If omitted redraw is true. If you want to unsketch several things and only redraw after the last one then use false for redraw and call View.Redraw().
 */
    static UnsketchFlagged(Model: Model, flag: number, redraw?: boolean): void;

/**
 * Object properties that are parameters are normally returned as the integer or float parameter values as that is virtually always what the user would want. This function temporarily changes the behaviour so that if a property is a parameter the parameter name is returned instead. This can be used with 'method chaining' (see the example below) to make sure a property argument is correct.
 * @returns IGA1DNurbsXYZ object.
 */
    ViewParameters(): IGA1DNurbsXYZ;

/**
 * Adds a warning for IGA 1D Nurbs XYZ. For more details on checking see the Check class.
 * @param message The warning message to give
 * @param details An optional detailed warning message
 */
    Warning(message: string, details?: string): void;

/**
 * Returns the cross references for this IGA 1D Nurbs XYZ.
 * @returns Xrefs object.
 */
    Xrefs(): Xrefs;

/**
 * Create a new IGA1DNurbsXYZ object.
 * @param Model Model that IGA 1d nurbs xyz will be created in
 * @param details Details for creating the IGA1DNurbsXYZ
 * @returns IGA1DNurbsXYZ object
 */
    constructor(Model: Model, details: IGA1DNurbsXYZ_constructor_Argument_details);

/**
 * Creates a string containing the IGA 1d nurbs xyz data in keyword format. Note that this contains the keyword header and the keyword cards. See also IGA1DNurbsXYZ.Keyword() and IGA1DNurbsXYZ.KeywordCards().
 * @returns string
 */
    toString(): string;

/** true if IGA 1D Nurbs XYZ exists, false if referred to but not defined. */
    readonly exists: boolean;
/** The Include file number that the IGA 1D Nurbs XYZ is in. */
    include: number;
/** IGA1DNurbsXYZ number. Also see the patchid property which is an alternative name for this. */
    label: number;
/** The Model number that the IGA 1D Nurbs XYZ is in. */
    readonly model: number;
/** Number of control points in the local r-direction. */
    nr: number;
/** IGA1DNurbsXYZ number. Also see the label property which is an alternative name for this. */
    patchid: number;
/** Polynomial degree of the basis in the local r-direction. */
    pr: number;
/** First knot value in the local r-direction. */
    rfirst: number;
/** Last knot value in the local r-direction. */
    rlast: number;
/** Knot vector type in the local r-direction. */
    unir: number;
}


/** Object function argument in IGA2DBasisTransformXYZ constructor */
interface IGA2DBasisTransformXYZ_constructor_Argument_details {
    /** Name of file containing patch data. */
    filename?: string;
    /** ID of the IGA 2D Basis Transform XYZ. */
    patchid: number;
}

declare class IGA2DBasisTransformXYZ {
/**
 * Associates a comment with a IGA 2D Basis Transform XYZ.
 * @param Comment Comment that will be attached to the IGA 2D Basis Transform XYZ
 */
    AssociateComment(Comment: Comment): void;

/**
 * Starts an edit panel in Browse mode.
 * @param modal If this window is modal (blocks the user from doing anything else in PRIMER until this window is dismissed). If omitted the window will be modal.
 */
    Browse(modal?: boolean): void;

/**
 * Clears a flag on the IGA 2D Basis Transform XYZ.
 * @param flag Flag to clear on the IGA 2D Basis Transform XYZ
 */
    ClearFlag(flag: number): void;

/**
 * Copies the IGA 2D Basis Transform XYZ. The target include of the copied IGA 2D Basis Transform XYZ can be set using Options.copy_target_include.
 * @param range If you want to keep the copied item in the range specified for the current include. Default value is false. To set current include, use Include.MakeCurrentLayer().
 * @returns IGA2DBasisTransformXYZ object
 */
    Copy(range?: boolean): IGA2DBasisTransformXYZ;

/**
 * Starts an interactive editing panel to create a IGA 2D Basis Transform XYZ
 * @param Model Model that the IGA 2D Basis Transform XYZ will be created in.
 * @param modal If this window is modal (blocks the user from doing anything else in PRIMER until this window is dismissed). If omitted the window will be modal.
 * @returns IGA2DBasisTransformXYZ object (or null if not made).
 */
    static Create(Model: Model, modal?: boolean): IGA2DBasisTransformXYZ;

/**
 * Detaches a comment from a IGA 2D Basis Transform XYZ.
 * @param Comment Comment that will be detached from the IGA 2D Basis Transform XYZ
 */
    DetachComment(Comment: Comment): void;

/**
 * Starts an interactive editing panel.
 * @param modal If this window is modal (blocks the user from doing anything else in PRIMER until this window is dismissed). If omitted the window will be modal.
 */
    Edit(modal?: boolean): void;

/**
 * Adds an error for IGA 2D Basis Transform XYZ. For more details on checking see the Check class.
 * @param message The error message to give
 * @param details An optional detailed error message
 */
    Error(message: string, details?: string): void;

/**
 * Returns the first IGA 2D Basis Transform XYZ in the model.
 * @param Model Model to get first IGA 2D Basis Transform XYZ in
 * @returns IGA2DBasisTransformXYZ object (or null if there are no IGA 2D Basis Transform XYZs in the model).
 */
    static First(Model: Model): IGA2DBasisTransformXYZ;

/**
 * Returns the first free IGA 2D Basis Transform XYZ label in the model. Also see IGA2DBasisTransformXYZ.LastFreeLabel(), IGA2DBasisTransformXYZ.NextFreeLabel() and Model.FirstFreeItemLabel().
 * @param Model Model to get first free IGA 2D Basis Transform XYZ label in
 * @param layer Include file (0 for the main file) to search for labels in (Equivalent to First free in layer in editing panels). If omitted the whole model will be used (Equivalent to First free in editing panels).
 * @returns IGA2DBasisTransformXYZ label.
 */
    static FirstFreeLabel(Model: Model, layer?: number): number;

/**
 * Flags all of the IGA 2D Basis Transform XYZs in the model with a defined flag.
 * @param Model Model that all IGA 2D Basis Transform XYZs will be flagged in
 * @param flag Flag to set on the IGA 2D Basis Transform XYZs
 */
    static FlagAll(Model: Model, flag: number): void;

/**
 * Checks if the IGA 2D Basis Transform XYZ is flagged or not.
 * @param flag Flag to test on the IGA 2D Basis Transform XYZ
 * @returns true if flagged, false if not.
 */
    Flagged(flag: number): boolean;

/**
 * Calls a function for each IGA 2D Basis Transform XYZ in the model. Note that ForEach has been designed to make looping over IGA 2D Basis Transform XYZs as fast as possible and so has some limitations. Firstly, a single temporary IGA2DBasisTransformXYZ object is created and on each function call it is updated with the current IGA 2D Basis Transform XYZ data. This means that you should not try to store the IGA2DBasisTransformXYZ object for later use (e.g. in an array) as it is temporary. Secondly, you cannot create new IGA 2D Basis Transform XYZs inside a ForEach loop. 
 * @param Model Model that all IGA 2D Basis Transform XYZs are in
 * @param func Function to call for each IGA 2D Basis Transform XYZ
 * @param extra An optional extra object/array/string etc that will appended to arguments when calling the function
 */
    static ForEach(Model: Model, func: () => void, extra?: any): void;

/**
 * Returns an array of IGA2DBasisTransformXYZ objects or properties for all of the IGA 2D Basis Transform XYZs in a model in PRIMER. If the optional property argument is not given then an array of IGA2DBasisTransformXYZ objects is returned. If the property argument is given, that property value for each IGA 2D Basis Transform XYZ is returned in the array instead of a IGA2DBasisTransformXYZ object
 * @param Model Model to get IGA 2D Basis Transform XYZs from
 * @param property Name for property to get for all IGA 2D Basis Transform XYZs in the model
 * @returns Array of IGA2DBasisTransformXYZ objects or properties
 */
    static GetAll(Model: Model, property?: string): IGA2DBasisTransformXYZ[];

/**
 * Extracts the comments associated to a IGA 2D Basis Transform XYZ.
 * @returns Array of Comment objects (or null if there are no comments associated to the node).
 */
    GetComments(): Comment[];

/**
 * Returns an array of IGA2DBasisTransformXYZ objects for all of the flagged IGA 2D Basis Transform XYZs in a model in PRIMER If the optional property argument is not given then an array of IGA2DBasisTransformXYZ objects is returned. If the property argument is given, then that property value for each IGA 2D Basis Transform XYZ is returned in the array instead of a IGA2DBasisTransformXYZ object
 * @param Model Model to get IGA 2D Basis Transform XYZs from
 * @param flag Flag set on the IGA 2D Basis Transform XYZs that you want to retrieve
 * @param property Name for property to get for all flagged IGA 2D Basis Transform XYZs in the model
 * @returns Array of IGA2DBasisTransformXYZ objects or properties
 */
    static GetFlagged(Model: Model, flag: number, property?: string): IGA2DBasisTransformXYZ[];

/**
 * Returns the IGA2DBasisTransformXYZ object for a IGA 2D Basis Transform XYZ ID.
 * @param Model Model to find the IGA 2D Basis Transform XYZ in
 * @param number number of the IGA 2D Basis Transform XYZ you want the IGA2DBasisTransformXYZ object for
 * @returns IGA2DBasisTransformXYZ object (or null if IGA 2D Basis Transform XYZ does not exist).
 */
    static GetFromID(Model: Model, number: number): IGA2DBasisTransformXYZ;

/**
 * Checks if a IGA2DBasisTransformXYZ property is a parameter or not. Note that object properties that are parameters are normally returned as the integer or float parameter values as that is virtually always what the user would want. For this function to work the JavaScript interpreter must use the parameter name instead of the value. This can be done by setting the Options.property_parameter_names option to true before calling the function and then resetting it to false afterwards.. This behaviour can also temporarily be switched by using the IGA2DBasisTransformXYZ.ViewParameters() method and 'method chaining' (see the examples below).
 * @param prop IGA 2D Basis Transform XYZ property to get parameter for
 * @returns Parameter object if property is a parameter, null if not.
 */
    GetParameter(prop: string): Parameter;

/**
 * Returns the keyword for this IGA 2d basis transform xyz (*IGA_2D_BASIS_TRANSFORM_XYZ). Note that a carriage return is not added. See also IGA2DBasisTransformXYZ.KeywordCards()
 * @returns string containing the keyword.
 */
    Keyword(): string;

/**
 * Returns the keyword cards for the IGA 2d basis transform xyz. Note that a carriage return is not added. See also IGA2DBasisTransformXYZ.Keyword()
 * @returns string containing the cards.
 */
    KeywordCards(): string;

/**
 * Returns the last IGA 2D Basis Transform XYZ in the model.
 * @param Model Model to get last IGA 2D Basis Transform XYZ in
 * @returns IGA2DBasisTransformXYZ object (or null if there are no IGA 2D Basis Transform XYZs in the model).
 */
    static Last(Model: Model): IGA2DBasisTransformXYZ;

/**
 * Returns the last free IGA 2D Basis Transform XYZ label in the model. Also see IGA2DBasisTransformXYZ.FirstFreeLabel(), IGA2DBasisTransformXYZ.NextFreeLabel() and see Model.LastFreeItemLabel()
 * @param Model Model to get last free IGA 2D Basis Transform XYZ label in
 * @param layer Include file (0 for the main file) to search for labels in (Equivalent to Highest free in layer in editing panels). If omitted the whole model will be used.
 * @returns IGA2DBasisTransformXYZ label.
 */
    static LastFreeLabel(Model: Model, layer?: number): number;

/**
 * Returns the next IGA 2D Basis Transform XYZ in the model.
 * @returns IGA2DBasisTransformXYZ object (or null if there are no more IGA 2D Basis Transform XYZs in the model).
 */
    Next(): IGA2DBasisTransformXYZ;

/**
 * Returns the next free (highest+1) IGA 2D Basis Transform XYZ label in the model. Also see IGA2DBasisTransformXYZ.FirstFreeLabel(), IGA2DBasisTransformXYZ.LastFreeLabel() and Model.NextFreeItemLabel()
 * @param Model Model to get next free IGA 2D Basis Transform XYZ label in
 * @param layer Include file (0 for the main file) to search for labels in (Equivalent to Highest+1 in layer in editing panels). If omitted the whole model will be used (Equivalent to Highest+1 in editing panels).
 * @returns IGA2DBasisTransformXYZ label.
 */
    static NextFreeLabel(Model: Model, layer?: number): number;

/**
 * Returns the previous IGA 2D Basis Transform XYZ in the model.
 * @returns IGA2DBasisTransformXYZ object (or null if there are no more IGA 2D Basis Transform XYZs in the model).
 */
    Previous(): IGA2DBasisTransformXYZ;

/**
 * Renumbers all of the IGA 2D Basis Transform XYZs in the model.
 * @param Model Model that all IGA 2D Basis Transform XYZs will be renumbered in
 * @param start Start point for renumbering
 */
    static RenumberAll(Model: Model, start: number): void;

/**
 * Renumbers all of the flagged IGA 2D Basis Transform XYZs in the model.
 * @param Model Model that all the flagged IGA 2D Basis Transform XYZs will be renumbered in
 * @param flag Flag set on the IGA 2D Basis Transform XYZs that you want to renumber
 * @param start Start point for renumbering
 */
    static RenumberFlagged(Model: Model, flag: number, start: number): void;

/**
 * Allows the user to select IGA 2D Basis Transform XYZs using standard PRIMER object menus.
 * @param flag Flag to use when selecting IGA 2D Basis Transform XYZs
 * @param prompt Text to display as a prompt to the user
 * @param limit If the argument is a Model then only IGA 2D Basis Transform XYZs from that model can be selected. If the argument is a Flag then only IGA 2D Basis Transform XYZs that are flagged with limit can be selected (limit should be different to flag). If omitted, or null, any IGA 2D Basis Transform XYZs can be selected. from any model.
 * @param modal If selection is modal (blocks the user from doing anything else in PRIMER until this window is dismissed). If omitted the selection will be modal.
 * @returns Number of IGA 2D Basis Transform XYZs selected or null if menu cancelled
 */
    static Select(flag: number, prompt: string, limit?: Model | number, modal?: boolean): number;

/**
 * Sets a flag on the IGA 2D Basis Transform XYZ.
 * @param flag Flag to set on the IGA 2D Basis Transform XYZ
 */
    SetFlag(flag: number): void;

/**
 * Sketches the IGA 2D Basis Transform XYZ. The IGA 2D Basis Transform XYZ will be sketched until you either call IGA2DBasisTransformXYZ.Unsketch(), IGA2DBasisTransformXYZ.UnsketchAll(), Model.UnsketchAll(), or delete the model
 * @param redraw If model should be redrawn or not after the IGA 2D Basis Transform XYZ is sketched. If omitted redraw is true. If you want to sketch several IGA 2D Basis Transform XYZs and only redraw after the last one then use false for redraw and call View.Redraw().
 */
    Sketch(redraw?: boolean): void;

/**
 * Sketches all of the flagged IGA 2D Basis Transform XYZs in the model. The IGA 2D Basis Transform XYZs will be sketched until you either call IGA2DBasisTransformXYZ.Unsketch(), IGA2DBasisTransformXYZ.UnsketchFlagged(), Model.UnsketchAll(), or delete the model
 * @param Model Model that all the flagged IGA 2D Basis Transform XYZs will be sketched in
 * @param flag Flag set on the IGA 2D Basis Transform XYZs that you want to sketch
 * @param redraw If model should be redrawn or not after the IGA 2D Basis Transform XYZs are sketched. If omitted redraw is true. If you want to sketch flagged IGA 2D Basis Transform XYZs several times and only redraw after the last one then use false for redraw and call View.Redraw().
 */
    static SketchFlagged(Model: Model, flag: number, redraw?: boolean): void;

/**
 * Returns the total number of IGA 2D Basis Transform XYZs in the model.
 * @param Model Model to get total for
 * @param exists true if only existing IGA 2D Basis Transform XYZs should be counted. If false or omitted referenced but undefined IGA 2D Basis Transform XYZs will also be included in the total.
 * @returns number of IGA 2D Basis Transform XYZs
 */
    static Total(Model: Model, exists?: boolean): number;

/**
 * Unsets a defined flag on all of the IGA 2D Basis Transform XYZs in the model.
 * @param Model Model that the defined flag for all IGA 2D Basis Transform XYZs will be unset in
 * @param flag Flag to unset on the IGA 2D Basis Transform XYZs
 */
    static UnflagAll(Model: Model, flag: number): void;

/**
 * Unsketches the IGA 2D Basis Transform XYZ.
 * @param redraw If model should be redrawn or not after the IGA 2D Basis Transform XYZ is unsketched. If omitted redraw is true. If you want to unsketch several IGA 2D Basis Transform XYZs and only redraw after the last one then use false for redraw and call View.Redraw().
 */
    Unsketch(redraw?: boolean): void;

/**
 * Unsketches all IGA 2D Basis Transform XYZs.
 * @param Model Model that all IGA 2D Basis Transform XYZs will be unblanked in
 * @param redraw If model should be redrawn or not after the IGA 2D Basis Transform XYZs are unsketched. If omitted redraw is true. If you want to unsketch several things and only redraw after the last one then use false for redraw and call View.Redraw().
 */
    static UnsketchAll(Model: Model, redraw?: boolean): void;

/**
 * Unsketches all flagged IGA 2D Basis Transform XYZs in the model.
 * @param Model Model that all IGA 2D Basis Transform XYZs will be unsketched in
 * @param flag Flag set on the IGA 2D Basis Transform XYZs that you want to unsketch
 * @param redraw If model should be redrawn or not after the IGA 2D Basis Transform XYZs are unsketched. If omitted redraw is true. If you want to unsketch several things and only redraw after the last one then use false for redraw and call View.Redraw().
 */
    static UnsketchFlagged(Model: Model, flag: number, redraw?: boolean): void;

/**
 * Object properties that are parameters are normally returned as the integer or float parameter values as that is virtually always what the user would want. This function temporarily changes the behaviour so that if a property is a parameter the parameter name is returned instead. This can be used with 'method chaining' (see the example below) to make sure a property argument is correct.
 * @returns IGA2DBasisTransformXYZ object.
 */
    ViewParameters(): IGA2DBasisTransformXYZ;

/**
 * Adds a warning for IGA 2D Basis Transform XYZ. For more details on checking see the Check class.
 * @param message The warning message to give
 * @param details An optional detailed warning message
 */
    Warning(message: string, details?: string): void;

/**
 * Returns the cross references for this IGA 2D Basis Transform XYZ.
 * @returns Xrefs object.
 */
    Xrefs(): Xrefs;

/**
 * Create a new IGA2DBasisTransformXYZ object.
 * @param Model Model that IGA 2d basis transform xyz will be created in
 * @param details Details for creating the IGA2DBasisTransformXYZ
 * @returns IGA2DBasisTransformXYZ object
 */
    constructor(Model: Model, details: IGA2DBasisTransformXYZ_constructor_Argument_details);

/**
 * Creates a string containing the IGA 2d basis transform xyz data in keyword format. Note that this contains the keyword header and the keyword cards. See also IGA2DBasisTransformXYZ.Keyword() and IGA2DBasisTransformXYZ.KeywordCards().
 * @returns string
 */
    toString(): string;

/** true if IGA 2D Basis Transform XYZ exists, false if referred to but not defined. */
    readonly exists: boolean;
/** Number of control points in the local r-direction. */
    filename: string;
/** The Include file number that the IGA 2D Basis Transform XYZ is in. */
    include: number;
/** IGA2DBasisTransformXYZ number. Also see the patchid property which is an alternative name for this. */
    label: number;
/** The Model number that the IGA 2D Basis Transform XYZ is in. */
    readonly model: number;
/** IGA2DBasisTransformXYZ number. Also see the label property which is an alternative name for this. */
    patchid: number;
}


/** Object function argument in IGA2DBrep constructor */
interface IGA2DBrep_constructor_Argument_details {
    /** ID of the IGA 2D Brep. */
    brid: number;
}

declare class IGA2DBrep {
/**
 * Associates a comment with a IGA 2D Brep.
 * @param Comment Comment that will be attached to the IGA 2D Brep
 */
    AssociateComment(Comment: Comment): void;

/**
 * Starts an edit panel in Browse mode.
 * @param modal If this window is modal (blocks the user from doing anything else in PRIMER until this window is dismissed). If omitted the window will be modal.
 */
    Browse(modal?: boolean): void;

/**
 * Clears a flag on the IGA 2D Brep.
 * @param flag Flag to clear on the IGA 2D Brep
 */
    ClearFlag(flag: number): void;

/**
 * Copies the IGA 2D Brep. The target include of the copied IGA 2D Brep can be set using Options.copy_target_include.
 * @param range If you want to keep the copied item in the range specified for the current include. Default value is false. To set current include, use Include.MakeCurrentLayer().
 * @returns IGA2DBrep object
 */
    Copy(range?: boolean): IGA2DBrep;

/**
 * Starts an interactive editing panel to create a IGA 2D Brep
 * @param Model Model that the IGA 2D Brep will be created in.
 * @param modal If this window is modal (blocks the user from doing anything else in PRIMER until this window is dismissed). If omitted the window will be modal.
 * @returns IGA2DBrep object (or null if not made).
 */
    static Create(Model: Model, modal?: boolean): IGA2DBrep;

/**
 * Detaches a comment from a IGA 2D Brep.
 * @param Comment Comment that will be detached from the IGA 2D Brep
 */
    DetachComment(Comment: Comment): void;

/**
 * Starts an interactive editing panel.
 * @param modal If this window is modal (blocks the user from doing anything else in PRIMER until this window is dismissed). If omitted the window will be modal.
 */
    Edit(modal?: boolean): void;

/**
 * Adds an error for IGA 2D Brep. For more details on checking see the Check class.
 * @param message The error message to give
 * @param details An optional detailed error message
 */
    Error(message: string, details?: string): void;

/**
 * Returns the first IGA 2D Brep in the model.
 * @param Model Model to get first IGA 2D Brep in
 * @returns IGA2DBrep object (or null if there are no IGA 2D Breps in the model).
 */
    static First(Model: Model): IGA2DBrep;

/**
 * Returns the first free IGA 2D Brep label in the model. Also see IGA2DBrep.LastFreeLabel(), IGA2DBrep.NextFreeLabel() and Model.FirstFreeItemLabel().
 * @param Model Model to get first free IGA 2D Brep label in
 * @param layer Include file (0 for the main file) to search for labels in (Equivalent to First free in layer in editing panels). If omitted the whole model will be used (Equivalent to First free in editing panels).
 * @returns IGA2DBrep label.
 */
    static FirstFreeLabel(Model: Model, layer?: number): number;

/**
 * Flags all of the IGA 2D Breps in the model with a defined flag.
 * @param Model Model that all IGA 2D Breps will be flagged in
 * @param flag Flag to set on the IGA 2D Breps
 */
    static FlagAll(Model: Model, flag: number): void;

/**
 * Checks if the IGA 2D Brep is flagged or not.
 * @param flag Flag to test on the IGA 2D Brep
 * @returns true if flagged, false if not.
 */
    Flagged(flag: number): boolean;

/**
 * Calls a function for each IGA 2D Brep in the model. Note that ForEach has been designed to make looping over IGA 2D Breps as fast as possible and so has some limitations. Firstly, a single temporary IGA2DBrep object is created and on each function call it is updated with the current IGA 2D Brep data. This means that you should not try to store the IGA2DBrep object for later use (e.g. in an array) as it is temporary. Secondly, you cannot create new IGA 2D Breps inside a ForEach loop. 
 * @param Model Model that all IGA 2D Breps are in
 * @param func Function to call for each IGA 2D Brep
 * @param extra An optional extra object/array/string etc that will appended to arguments when calling the function
 */
    static ForEach(Model: Model, func: () => void, extra?: any): void;

/**
 * Returns an array of IGA2DBrep objects or properties for all of the IGA 2D Breps in a model in PRIMER. If the optional property argument is not given then an array of IGA2DBrep objects is returned. If the property argument is given, that property value for each IGA 2D Brep is returned in the array instead of a IGA2DBrep object
 * @param Model Model to get IGA 2D Breps from
 * @param property Name for property to get for all IGA 2D Breps in the model
 * @returns Array of IGA2DBrep objects or properties
 */
    static GetAll(Model: Model, property?: string): IGA2DBrep[];

/**
 * Extracts the comments associated to a IGA 2D Brep.
 * @returns Array of Comment objects (or null if there are no comments associated to the node).
 */
    GetComments(): Comment[];

/**
 * Returns the data for fid in *IGA_2D_BREP.
 * @param index Index you want the data for. Note that indices start at 0.
 * @returns The ID of parametric face, *IGA_FACE_UVW.
 */
    GetData(index: number): number;

/**
 * Returns an array of IGA2DBrep objects for all of the flagged IGA 2D Breps in a model in PRIMER If the optional property argument is not given then an array of IGA2DBrep objects is returned. If the property argument is given, then that property value for each IGA 2D Brep is returned in the array instead of a IGA2DBrep object
 * @param Model Model to get IGA 2D Breps from
 * @param flag Flag set on the IGA 2D Breps that you want to retrieve
 * @param property Name for property to get for all flagged IGA 2D Breps in the model
 * @returns Array of IGA2DBrep objects or properties
 */
    static GetFlagged(Model: Model, flag: number, property?: string): IGA2DBrep[];

/**
 * Returns the IGA2DBrep object for a IGA 2D Brep ID.
 * @param Model Model to find the IGA 2D Brep in
 * @param number number of the IGA 2D Brep you want the IGA2DBrep object for
 * @returns IGA2DBrep object (or null if IGA 2D Brep does not exist).
 */
    static GetFromID(Model: Model, number: number): IGA2DBrep;

/**
 * Checks if a IGA2DBrep property is a parameter or not. Note that object properties that are parameters are normally returned as the integer or float parameter values as that is virtually always what the user would want. For this function to work the JavaScript interpreter must use the parameter name instead of the value. This can be done by setting the Options.property_parameter_names option to true before calling the function and then resetting it to false afterwards.. This behaviour can also temporarily be switched by using the IGA2DBrep.ViewParameters() method and 'method chaining' (see the examples below).
 * @param prop IGA 2D Brep property to get parameter for
 * @returns Parameter object if property is a parameter, null if not.
 */
    GetParameter(prop: string): Parameter;

/**
 * Returns the keyword for this IGA 2d brep (*IGA_2D_BREP). Note that a carriage return is not added. See also IGA2DBrep.KeywordCards()
 * @returns string containing the keyword.
 */
    Keyword(): string;

/**
 * Returns the keyword cards for the IGA 2d brep. Note that a carriage return is not added. See also IGA2DBrep.Keyword()
 * @returns string containing the cards.
 */
    KeywordCards(): string;

/**
 * Returns the last IGA 2D Brep in the model.
 * @param Model Model to get last IGA 2D Brep in
 * @returns IGA2DBrep object (or null if there are no IGA 2D Breps in the model).
 */
    static Last(Model: Model): IGA2DBrep;

/**
 * Returns the last free IGA 2D Brep label in the model. Also see IGA2DBrep.FirstFreeLabel(), IGA2DBrep.NextFreeLabel() and see Model.LastFreeItemLabel()
 * @param Model Model to get last free IGA 2D Brep label in
 * @param layer Include file (0 for the main file) to search for labels in (Equivalent to Highest free in layer in editing panels). If omitted the whole model will be used.
 * @returns IGA2DBrep label.
 */
    static LastFreeLabel(Model: Model, layer?: number): number;

/**
 * Returns the next IGA 2D Brep in the model.
 * @returns IGA2DBrep object (or null if there are no more IGA 2D Breps in the model).
 */
    Next(): IGA2DBrep;

/**
 * Returns the next free (highest+1) IGA 2D Brep label in the model. Also see IGA2DBrep.FirstFreeLabel(), IGA2DBrep.LastFreeLabel() and Model.NextFreeItemLabel()
 * @param Model Model to get next free IGA 2D Brep label in
 * @param layer Include file (0 for the main file) to search for labels in (Equivalent to Highest+1 in layer in editing panels). If omitted the whole model will be used (Equivalent to Highest+1 in editing panels).
 * @returns IGA2DBrep label.
 */
    static NextFreeLabel(Model: Model, layer?: number): number;

/**
 * Returns the previous IGA 2D Brep in the model.
 * @returns IGA2DBrep object (or null if there are no more IGA 2D Breps in the model).
 */
    Previous(): IGA2DBrep;

/**
 * Removes fid for an index in *IGA_2D_BREP.
 * @param index The index you want to delete fid for. Note that indices start at 0, not 1.
 */
    RemoveData(index: number): void;

/**
 * Renumbers all of the IGA 2D Breps in the model.
 * @param Model Model that all IGA 2D Breps will be renumbered in
 * @param start Start point for renumbering
 */
    static RenumberAll(Model: Model, start: number): void;

/**
 * Renumbers all of the flagged IGA 2D Breps in the model.
 * @param Model Model that all the flagged IGA 2D Breps will be renumbered in
 * @param flag Flag set on the IGA 2D Breps that you want to renumber
 * @param start Start point for renumbering
 */
    static RenumberFlagged(Model: Model, flag: number, start: number): void;

/**
 * Allows the user to select IGA 2D Breps using standard PRIMER object menus.
 * @param flag Flag to use when selecting IGA 2D Breps
 * @param prompt Text to display as a prompt to the user
 * @param limit If the argument is a Model then only IGA 2D Breps from that model can be selected. If the argument is a Flag then only IGA 2D Breps that are flagged with limit can be selected (limit should be different to flag). If omitted, or null, any IGA 2D Breps can be selected. from any model.
 * @param modal If selection is modal (blocks the user from doing anything else in PRIMER until this window is dismissed). If omitted the selection will be modal.
 * @returns Number of IGA 2D Breps selected or null if menu cancelled
 */
    static Select(flag: number, prompt: string, limit?: Model | number, modal?: boolean): number;

/**
 * Sets fid in *IGA_2D_BREP.
 * @param index Index you want to set the fid for. Note that indices start at 0.
 * @param fid The ID of parametric face.
 */
    SetData(index: number, fid: number): void;

/**
 * Sets a flag on the IGA 2D Brep.
 * @param flag Flag to set on the IGA 2D Brep
 */
    SetFlag(flag: number): void;

/**
 * Sketches the IGA 2D Brep. The IGA 2D Brep will be sketched until you either call IGA2DBrep.Unsketch(), IGA2DBrep.UnsketchAll(), Model.UnsketchAll(), or delete the model
 * @param redraw If model should be redrawn or not after the IGA 2D Brep is sketched. If omitted redraw is true. If you want to sketch several IGA 2D Breps and only redraw after the last one then use false for redraw and call View.Redraw().
 */
    Sketch(redraw?: boolean): void;

/**
 * Sketches all of the flagged IGA 2D Breps in the model. The IGA 2D Breps will be sketched until you either call IGA2DBrep.Unsketch(), IGA2DBrep.UnsketchFlagged(), Model.UnsketchAll(), or delete the model
 * @param Model Model that all the flagged IGA 2D Breps will be sketched in
 * @param flag Flag set on the IGA 2D Breps that you want to sketch
 * @param redraw If model should be redrawn or not after the IGA 2D Breps are sketched. If omitted redraw is true. If you want to sketch flagged IGA 2D Breps several times and only redraw after the last one then use false for redraw and call View.Redraw().
 */
    static SketchFlagged(Model: Model, flag: number, redraw?: boolean): void;

/**
 * Returns the total number of IGA 2D Breps in the model.
 * @param Model Model to get total for
 * @param exists true if only existing IGA 2D Breps should be counted. If false or omitted referenced but undefined IGA 2D Breps will also be included in the total.
 * @returns number of IGA 2D Breps
 */
    static Total(Model: Model, exists?: boolean): number;

/**
 * Unsets a defined flag on all of the IGA 2D Breps in the model.
 * @param Model Model that the defined flag for all IGA 2D Breps will be unset in
 * @param flag Flag to unset on the IGA 2D Breps
 */
    static UnflagAll(Model: Model, flag: number): void;

/**
 * Unsketches the IGA 2D Brep.
 * @param redraw If model should be redrawn or not after the IGA 2D Brep is unsketched. If omitted redraw is true. If you want to unsketch several IGA 2D Breps and only redraw after the last one then use false for redraw and call View.Redraw().
 */
    Unsketch(redraw?: boolean): void;

/**
 * Unsketches all IGA 2D Breps.
 * @param Model Model that all IGA 2D Breps will be unblanked in
 * @param redraw If model should be redrawn or not after the IGA 2D Breps are unsketched. If omitted redraw is true. If you want to unsketch several things and only redraw after the last one then use false for redraw and call View.Redraw().
 */
    static UnsketchAll(Model: Model, redraw?: boolean): void;

/**
 * Unsketches all flagged IGA 2D Breps in the model.
 * @param Model Model that all IGA 2D Breps will be unsketched in
 * @param flag Flag set on the IGA 2D Breps that you want to unsketch
 * @param redraw If model should be redrawn or not after the IGA 2D Breps are unsketched. If omitted redraw is true. If you want to unsketch several things and only redraw after the last one then use false for redraw and call View.Redraw().
 */
    static UnsketchFlagged(Model: Model, flag: number, redraw?: boolean): void;

/**
 * Object properties that are parameters are normally returned as the integer or float parameter values as that is virtually always what the user would want. This function temporarily changes the behaviour so that if a property is a parameter the parameter name is returned instead. This can be used with 'method chaining' (see the example below) to make sure a property argument is correct.
 * @returns IGA2DBrep object.
 */
    ViewParameters(): IGA2DBrep;

/**
 * Adds a warning for IGA 2D Brep. For more details on checking see the Check class.
 * @param message The warning message to give
 * @param details An optional detailed warning message
 */
    Warning(message: string, details?: string): void;

/**
 * Returns the cross references for this IGA 2D Brep.
 * @returns Xrefs object.
 */
    Xrefs(): Xrefs;

/**
 * Create a new IGA2DBrep object.
 * @param Model Model that IGA 2d brep will be created in
 * @param details Details for creating the IGA2DBrep
 * @returns IGA2DBrep object
 */
    constructor(Model: Model, details: IGA2DBrep_constructor_Argument_details);

/**
 * Creates a string containing the IGA 2d brep data in keyword format. Note that this contains the keyword header and the keyword cards. See also IGA2DBrep.Keyword() and IGA2DBrep.KeywordCards().
 * @returns string
 */
    toString(): string;

/** IGA2DBrep number. Also see the label property which is an alternative name for this. */
    brid: number;
/** Number of Parametric Face IDs (fid) added in IGA_2D_BREP. Can be used as index to add a new entry. */
    readonly entries: number;
/** true if IGA 2D Brep exists, false if referred to but not defined. */
    readonly exists: boolean;
/** The Include file number that the IGA 2D Brep is in. */
    include: number;
/** IGA2DBrep number. Also see the brid property which is an alternative name for this. */
    label: number;
/** The Model number that the IGA 2D Brep is in. */
    readonly model: number;
}


/** Object function argument in IGA2DNurbsUVW constructor */
interface IGA2DNurbsUVW_constructor_Argument_details {
    /** Number of control points in the local r-direction. */
    nr?: number;
    /** Number of control points in the local s-direction. */
    ns?: number;
    /** ID of the IGA 2D Nurbs UVW. */
    patchid: number;
    /** Polynomial degree of the basis in the local r-direction. */
    pr?: number;
    /** Polynomial degree of the basis in the local s-direction. */
    ps?: number;
    /** Knot vector type in the local r-direction. */
    unir?: number;
    /** Knot vector type in the local s-direction. */
    unis?: number;
}

declare class IGA2DNurbsUVW {
/**
 * Associates a comment with a IGA 2D Nurbs UVW.
 * @param Comment Comment that will be attached to the IGA 2D Nurbs UVW
 */
    AssociateComment(Comment: Comment): void;

/**
 * Starts an edit panel in Browse mode.
 * @param modal If this window is modal (blocks the user from doing anything else in PRIMER until this window is dismissed). If omitted the window will be modal.
 */
    Browse(modal?: boolean): void;

/**
 * Clears a flag on the IGA 2D Nurbs UVW.
 * @param flag Flag to clear on the IGA 2D Nurbs UVW
 */
    ClearFlag(flag: number): void;

/**
 * Copies the IGA 2D Nurbs UVW. The target include of the copied IGA 2D Nurbs UVW can be set using Options.copy_target_include.
 * @param range If you want to keep the copied item in the range specified for the current include. Default value is false. To set current include, use Include.MakeCurrentLayer().
 * @returns IGA2DNurbsUVW object
 */
    Copy(range?: boolean): IGA2DNurbsUVW;

/**
 * Starts an interactive editing panel to create a IGA 2D Nurbs UVW
 * @param Model Model that the IGA 2D Nurbs UVW will be created in.
 * @param modal If this window is modal (blocks the user from doing anything else in PRIMER until this window is dismissed). If omitted the window will be modal.
 * @returns IGA2DNurbsUVW object (or null if not made).
 */
    static Create(Model: Model, modal?: boolean): IGA2DNurbsUVW;

/**
 * Detaches a comment from a IGA 2D Nurbs UVW.
 * @param Comment Comment that will be detached from the IGA 2D Nurbs UVW
 */
    DetachComment(Comment: Comment): void;

/**
 * Starts an interactive editing panel.
 * @param modal If this window is modal (blocks the user from doing anything else in PRIMER until this window is dismissed). If omitted the window will be modal.
 */
    Edit(modal?: boolean): void;

/**
 * Adds an error for IGA 2D Nurbs UVW. For more details on checking see the Check class.
 * @param message The error message to give
 * @param details An optional detailed error message
 */
    Error(message: string, details?: string): void;

/**
 * Returns the first IGA 2D Nurbs UVW in the model.
 * @param Model Model to get first IGA 2D Nurbs UVW in
 * @returns IGA2DNurbsUVW object (or null if there are no IGA 2D Nurbs UVWs in the model).
 */
    static First(Model: Model): IGA2DNurbsUVW;

/**
 * Returns the first free IGA 2D Nurbs UVW label in the model. Also see IGA2DNurbsUVW.LastFreeLabel(), IGA2DNurbsUVW.NextFreeLabel() and Model.FirstFreeItemLabel().
 * @param Model Model to get first free IGA 2D Nurbs UVW label in
 * @param layer Include file (0 for the main file) to search for labels in (Equivalent to First free in layer in editing panels). If omitted the whole model will be used (Equivalent to First free in editing panels).
 * @returns IGA2DNurbsUVW label.
 */
    static FirstFreeLabel(Model: Model, layer?: number): number;

/**
 * Flags all of the IGA 2D Nurbs UVWs in the model with a defined flag.
 * @param Model Model that all IGA 2D Nurbs UVWs will be flagged in
 * @param flag Flag to set on the IGA 2D Nurbs UVWs
 */
    static FlagAll(Model: Model, flag: number): void;

/**
 * Checks if the IGA 2D Nurbs UVW is flagged or not.
 * @param flag Flag to test on the IGA 2D Nurbs UVW
 * @returns true if flagged, false if not.
 */
    Flagged(flag: number): boolean;

/**
 * Calls a function for each IGA 2D Nurbs UVW in the model. Note that ForEach has been designed to make looping over IGA 2D Nurbs UVWs as fast as possible and so has some limitations. Firstly, a single temporary IGA2DNurbsUVW object is created and on each function call it is updated with the current IGA 2D Nurbs UVW data. This means that you should not try to store the IGA2DNurbsUVW object for later use (e.g. in an array) as it is temporary. Secondly, you cannot create new IGA 2D Nurbs UVWs inside a ForEach loop. 
 * @param Model Model that all IGA 2D Nurbs UVWs are in
 * @param func Function to call for each IGA 2D Nurbs UVW
 * @param extra An optional extra object/array/string etc that will appended to arguments when calling the function
 */
    static ForEach(Model: Model, func: () => void, extra?: any): void;

/**
 * Returns an array of IGA2DNurbsUVW objects or properties for all of the IGA 2D Nurbs UVWs in a model in PRIMER. If the optional property argument is not given then an array of IGA2DNurbsUVW objects is returned. If the property argument is given, that property value for each IGA 2D Nurbs UVW is returned in the array instead of a IGA2DNurbsUVW object
 * @param Model Model to get IGA 2D Nurbs UVWs from
 * @param property Name for property to get for all IGA 2D Nurbs UVWs in the model
 * @returns Array of IGA2DNurbsUVW objects or properties
 */
    static GetAll(Model: Model, property?: string): IGA2DNurbsUVW[];

/**
 * Extracts the comments associated to a IGA 2D Nurbs UVW.
 * @returns Array of Comment objects (or null if there are no comments associated to the node).
 */
    GetComments(): Comment[];

/**
 * Returns the data for a specific control point an an array. There are nr x ns control points.
 * @param index Index you want the control data for. Note that indices start at 0.
 * @returns An array containing the control point data.
 */
    GetControlPoint(index: number): number[];

/**
 * Returns an array of IGA2DNurbsUVW objects for all of the flagged IGA 2D Nurbs UVWs in a model in PRIMER If the optional property argument is not given then an array of IGA2DNurbsUVW objects is returned. If the property argument is given, then that property value for each IGA 2D Nurbs UVW is returned in the array instead of a IGA2DNurbsUVW object
 * @param Model Model to get IGA 2D Nurbs UVWs from
 * @param flag Flag set on the IGA 2D Nurbs UVWs that you want to retrieve
 * @param property Name for property to get for all flagged IGA 2D Nurbs UVWs in the model
 * @returns Array of IGA2DNurbsUVW objects or properties
 */
    static GetFlagged(Model: Model, flag: number, property?: string): IGA2DNurbsUVW[];

/**
 * Returns the IGA2DNurbsUVW object for a IGA 2D Nurbs UVW ID.
 * @param Model Model to find the IGA 2D Nurbs UVW in
 * @param number number of the IGA 2D Nurbs UVW you want the IGA2DNurbsUVW object for
 * @returns IGA2DNurbsUVW object (or null if IGA 2D Nurbs UVW does not exist).
 */
    static GetFromID(Model: Model, number: number): IGA2DNurbsUVW;

/**
 * Checks if a IGA2DNurbsUVW property is a parameter or not. Note that object properties that are parameters are normally returned as the integer or float parameter values as that is virtually always what the user would want. For this function to work the JavaScript interpreter must use the parameter name instead of the value. This can be done by setting the Options.property_parameter_names option to true before calling the function and then resetting it to false afterwards.. This behaviour can also temporarily be switched by using the IGA2DNurbsUVW.ViewParameters() method and 'meth