/* Utils.Version was added for REPORTER in Version 21. So based on if this function is present or not we can check which version we are running */
if (typeof Utils.Version != "function") {
    let message = `This workflow is only available for Oasys REPORTER version 21 and above.`;
    if (Batch()) {
        LogError(message);
    } else {
        Window.Message("Workflow not available", message);
    }
    Exit();
}

/**
 * Cross-platform function for the Window.GetFile method. Instead of taking a string, it takes
 * in a list of extensions and constructs the appropriate filter string based on the OS. It also takes
 * the parameters for save and initial.
 * @param {string[]|string} [extensions=[]] - List of file extensions (e.g., [".csv", ".txt"]) or a single extension string (e.g., ".csv")
 * @param {boolean} [save=false] - Whether the dialog is for saving a file
 * @param {string} [initial=""] - Initial file path or directory name. If a file path is provided, the parent directory is used. If a directory is provided, it's used as-is.
 * @returns {string|null} - Selected filename as a string, or null if the user cancels
 */
function GetFile(extensions = [], save = false, initial = "") {
    /* Check that extensions is a string or an array. */
    if (typeof extensions === "string") {
        /* If the user passes a string of delimited values (with ; commas or spaces to separate the extensions) instead of an array,
         * split the string into an array using the appropriate delimiter. The regex splits on commas, semicolons, or spaces. */
        extensions = extensions.split(/[,; ]+/);
    } else if (!Array.isArray(extensions)) {
        LogError("Function GetFile: extensions parameter must be a string or an array of strings.");
        return null;
    }

    // Replace all the *'s at the start of the string (this is generally before the first dot of the extension).
    // We do this because we add the * later on using the joiner variable.
    extensions = extensions.map((ext) => ext.replace(/^\**/, ""));

    const is_linux = Unix();

    /**
     * The filter adds a * before each extension (excluding the first) and joins them with either spaces (Linux) or semicolons (Windows).
     */
    let file, filter;
    const joiner = is_linux ? " *" : "; *";
    const file_separator = is_linux ? "/" : "\\";

    // Adjust initial path for Linux and Windows if needed. On Linux, convert backslashes to forward slashes.
    // On Windows, convert forward slashes to backslashes. This ensures the initial path is in the correct format for the OS.
    if (is_linux && initial) {
        initial = initial.replace(/\\/g, file_separator);
    }

    if (!is_linux && initial) {
        initial = initial.replace(/\//g, file_separator);
    }

    filter = extensions.map((ext) => `${ext}`).join(joiner);

    if (initial && File.Exists(initial)) {
        // For Window.GetFile, if a directory is used but is not properly specified with a / or \\ at the end, then the appropriate
        // delimiter (e.g. forward slash) is added to force it to use the correct directory. Otherwise the function may misinterpret it as a file.
        if (File.IsDirectory(initial) && !initial.endsWith(file_separator)) {
            initial = initial + file_separator;
        }
        file = Window.GetFile(filter, save, initial);
    } else {
        file = Window.GetFile(filter, save);
    }
    return file;
}

/* Contain the entire script within a function because REPORTER only has a single JavaScript realm
 * for the entire session. */
on_load_script();

/**
 * #AAW REPORTER on load script
 *
 * Runs as soon as the template is opened.
 *
 * In batch mode, will proceed to generate if:
 *
 * 1. Keyword file can be found from %KEYWORD_FILE% or via %DEFAULT_DIR%/%DEFAULT_JOB%
 * 2. Valid %OUTPUT_DIR% defined or valid default (subdirectory can be created)
 * 3. Valid %RESULTS_DIR% defined or valid default containing results files
 *
 * Otherwise will terminate generation with a log error.
 *
 * In interactive mode, the script asks the user two questions:
 *
 * 1. Select the keyword file of the job you wish to post-process
 * 2. Use default results and output directories or choose to specify them in PRIMER
 *
 * If proceeding to template generation, the script will deactivate itelf to avoid entering an
 * infinite loop, and it also controls generation and (de)activation of various other items.
 */
function on_load_script() {
    let templ = Template.GetCurrent();

    /* Delete AAW_job_control_variables.csv in case it has data from previous runs */
    let reporter_temp = get_expanded_variable_value(templ, `REPORTER_TEMP`);
    if (reporter_temp == null || (reporter_temp != null && !File.IsDirectory(reporter_temp))) {
        finish_script(false, `Could not find REPORTER_TEMP directory.`);
    }
    let f_csv_name = `${reporter_temp}/AAW_job_control_variables.csv`;
    if (File.Exists(f_csv_name)) {
        let success = File.Delete(f_csv_name);
        if (!success) {
            finish_script(false, `Unable to delete existing variables file: ${f_csv_name}`);
        }
    }
    /* Replace it with an empty file so that `#AAW REPORTER read variables from PRIMER job control`
     * item always has something to read. */
    let f_csv = new File(f_csv_name, File.WRITE);
    f_csv.Close();

    /* To begin with, activate all master #AAW page items and deactivate all normal page #AAW items */
    activate_master_page_aaw_items(true);
    activate_normal_page_aaw_items(false);

    /* Generate the subjective modifiers item (if present) to ensure that modifiers are initialised
     * and functions declared in case user aborts generation and then clicks `Set Modifiers`
     * button. */
    generate_item(`#VTC Recalculate Script`, false);

    /* Get keyword file and set %DEFAULT_DIR% and %DEFAULT_JOB% */
    if (!get_keyword_file()) {
        finish_script(false, `Keyword file not selected. Report will not be generated.`);
    } else {
        configure_results_and_outputs();
        finish_script(true);
    }
}

/**
 * Gets the keyword file and sets %KEYWORD_FILE%, %DEFAULT_DIR% and %DEFAULT_JOB%.
 * Returns whether successful (true/false).
 *
 * REPORTER variables for KEYWORD_FILE and DEFAULT_DIR/DEFAULT_JOB might already be defined if:
 * 1. They have been hardwired (possibly)
 * 2. They have been passed through as arguments in batch
 * 3. The template is being re-run.
 *
 * In batch mode, the template only needs one of KEYWORD_FILE or (DEFAULT_DIR and DEFAULT_JOB) to
 * proceed. If running from SHELL, it is convenient to use DEFAULT_DIR and DEFAULT_JOB, whereas it
 * may be more convenient to define KEYWORD_FILE if running from the command line.
 *
 * In interactive mode, the user is always prompted to specify the keyword file, regardless of
 * whether the variables are already defined. We could change the logic here if we wanted so that
 * the template runs automatically in interactive mode if all variables are defined, but we think
 * it is more helpful if the user is prompted, in case they want to change any inputs that have
 * been deliberately or accidentally saved previously.
 *
 * @returns {Boolean}
 * @example
 * let success = get_keyword_file();
 */
function get_keyword_file() {
    let templ = Template.GetCurrent();

    let keyword_file;
    if (Batch()) {
        /* First check to see if we can find a keyword file.
         * We check KEYWORD_FILE first, and then DEFAULT_DIR and DEFAULT_JOB.
         * We make sure they're updated to be consistent with each other.
         */
        keyword_file = get_expanded_variable_value(templ, `KEYWORD_FILE`);
        let found_key_file = false;
        let default_dir;
        let default_job;
        if (keyword_file != null && File.Exists(keyword_file)) {
            LogPrint(`Found keyword file: ${keyword_file}`);
            found_key_file = true;
            /* Got a file, so set %KEYWORD_FILE%, %DEFAULT_DIR% and %DEFAULT_JOB% */
            set_keyword_file_variables(keyword_file);
        } else {
            if (keyword_file == null) {
                LogPrint(`REPORTER Variable %KEYWORD_FILE% was not defined.`);
            } else if (!File.Exists(keyword_file)) {
                LogPrint(
                    `Keyword file specified by REPORTER Variable %KEYWORD_FILE% could not be found: ${keyword_file}`
                );
            }
            LogPrint(`Searching for keyword file via %DEFAULT_DIR% and %DEFAULT_JOB% instead...`);
            /* If KEYWORD_FILE doesn't exist, check DEFAULT_DIR and DEFAULT_JOB. */
            default_dir = get_expanded_variable_value(templ, `DEFAULT_DIR`);
            default_job = get_expanded_variable_value(templ, `DEFAULT_JOB`);

            if (default_dir != null && File.IsDirectory(default_dir) && default_job != null) {
                let possible_extensions = [
                    `k`,
                    `key`,
                    `kby`,
                    `k.gz`,
                    `key.gz`,
                    `kby.gz`,
                    `k.zip`,
                    `key.zip`,
                    `kby.zip`
                ];
                for (let ext of possible_extensions) {
                    keyword_file = `${default_dir}/${default_job}.${ext}`;
                    if (File.Exists(keyword_file)) {
                        LogPrint(`Found keyword file: ${keyword_file}`);
                        found_key_file = true;
                        /* Got a file, so set %KEYWORD_FILE%, %DEFAULT_DIR% and %DEFAULT_JOB% */
                        set_keyword_file_variables(keyword_file);
                        break;
                    }
                }
                if (!found_key_file) {
                    LogPrint(
                        `Could not find a keyword file via %DEFAULT_DIR% and %DEFAULT_JOB% using any of the possible extentions "${possible_extensions.join(
                            `", "`
                        )}".`
                    );
                }
            } else {
                if (default_dir == null) {
                    LogPrint(`REPORTER Variable %DEFAULT_DIR% was not defined.`);
                } else if (!File.IsDirectory(default_dir)) {
                    LogPrint(`Directory specified by REPORTER Variable %DEFAULT_DIR% could not be found.`);
                }
                if (default_job == null) {
                    LogPrint(`REPORTER Variable %DEFAULT_JOB% was not defined.`);
                }
            }
        }

        if (!found_key_file) {
            return false;
        }
    } else {
        let ans = Window.Message(
            `Select keyword file`,
            `Select the LS-DYNA keyword file of the job you wish to post-process.`,
            Window.OK | Window.CANCEL
        );
        if (ans == Window.CANCEL) {
            return false;
        }
        keyword_file = GetFile([".k", ".key", ".dyn"]);

        if (!keyword_file) {
            /* User clicked cancel */
            return false;
        }

        /* Got a file, so set %KEYWORD_FILE%, %DEFAULT_DIR% and %DEFAULT_JOB% */
        set_keyword_file_variables(keyword_file);
    }

    return true;
}

/**
 * Sets the REPORTER variables %KEYWORD_FILE%, %DEFAULT_DIR%, and %DEFAULT_JOB% to match the
 * specified keyword file.
 * @param {string} keyword_file Absolute path and filename of keyword file
 */
function set_keyword_file_variables(keyword_file) {
    let templ = Template.GetCurrent();

    /* If keyword_file is relative path, make it absolute path. We need to do this so that
     * any other variables that end up being derived from it (DEFAULT_DIR, RESULTS_DIR, OUTPUT_DIR,
     * etc.) are absolute paths too. When REPORTER generates an Oasys item, it sets -start_in to
     * the job file directory, so it gets confused if the relative paths we have defined are
     * relative to a different starting directory. Converting everything to absolute paths avoids
     * this problem. */
    keyword_file = convert_to_absolute_path(keyword_file, `%KEYWORD_FILE%`);

    let path_index = Math.max(keyword_file.lastIndexOf("/"), keyword_file.lastIndexOf("\\"));
    let default_dir = keyword_file.substring(0, path_index);
    let filename = keyword_file.substring(path_index + 1);
    let default_job = filename.substring(0, filename.lastIndexOf("."));

    /* Assign to REPORTER variables. Constructor will overwrite existing variable.
     * KEYWORD_FILE should be temporary; DEFAULT_DIR and DEFAULT_JOB are not temporary. */
    let keyword_file_var = new Variable(
        templ,
        `KEYWORD_FILE`,
        `Keyword file`,
        keyword_file,
        `File(absolute)`,
        false,
        true
    );
    let default_dir_var = new Variable(
        templ,
        `DEFAULT_DIR`,
        `Reporter default directory`,
        default_dir,
        `Directory`,
        false,
        false
    );
    let default_job_var = new Variable(
        templ,
        `DEFAULT_JOB`,
        `Reporter default jobname`,
        default_job,
        `File(basename)`,
        false,
        false
    );
}

/**
 * Configures and checks the results and output directories.
 *
 * By default, RESULTS_DIR and OUTPUT_DIR can be inferred from the keyword file location.
 * However, the user may wish to specify different locations for RESULTS_DIR and OUTPUT_DIR,
 * in which case these are configured by the subsequent `#AAW PRIMER job control GUI` item.
 * Depending on the user's preference, this script activates/deactivates that item.
 *
 * In interactive mode, if hte user chooses to use the defaults but there are issues, they are
 * prompted to use the PRIMER GUI to rectify.
 *
 * In batch mode, supplied values of RESULTS_DIR and OUTPUT_DIR will be used if provided, or it
 * will attempt to use the defaults. Generation will terminate with a log error if there are any
 * issues.
 */
function configure_results_and_outputs() {
    let templ = Template.GetCurrent();

    /* By default, show PRIMER job control GUI */
    let show_job_control_gui = true;
    if (Batch()) {
        /* If we are running in batch, always skip job control GUI. */
        show_job_control_gui = false;
    }

    /* Default OUTPUT_DIR is in subdirectory matching name of template. */
    let output_dir_name = templ.filename.substring(0, templ.filename.lastIndexOf(`.`));

    /* DEFAULT_DIR should already have been defined in get_keyword_file() */
    let default_dir = get_expanded_variable_value(templ, `DEFAULT_DIR`);
    if (default_dir == null) {
        finish_script(false, `Could not find REPORTER Variable %DEFAULT_DIR% in configure_results_and_outputs().`);
    }

    /* Set initial answer to NO so that in Batch mode, RESULTS_DIR and OUTPUT_DIR only reset if not
     * provided. */
    let ans = Window.NO;

    /* In interactive mode, give user option */
    if (!Batch()) {
        ans = Window.Question(
            `Results and output directories`,
            `By default, REPORTER will search for results in the same directory as the keyword file, ` +
                `and write images and other files to a subdirectory named "${output_dir_name}".\n\n` +
                `Proceed with these defaults (Yes)? Or configure directories in PRIMER (No).`
        );
    }

    /* In Batch mode, if %RESULTS_DIR% and %OUTPUT_DIR% have been provided, we will use them.
     * Otherwise, we create them with default values.
     *
     * In interactive mode, if the user said YES, then we always reset to default values. If they
     * said NO, use values if provided, otherwise use defaults - and show PRIMER GUI. */

    let results_dir = get_expanded_variable_value(templ, `RESULTS_DIR`);
    if (results_dir == null || ans == Window.YES) {
        results_dir = default_dir;
        if (Batch()) {
            LogPrint(`REPORTER variable %RESULTS_DIR% unspecified. Setting default %RESULTS_DIR% = ${results_dir}`);
        }
        let results_dir_var = new Variable(
            templ,
            `RESULTS_DIR`,
            `Directory containing LS-DYNA results`,
            `%DEFAULT_DIR%`,
            `Directory`,
            false,
            true
        );
    } else {
        /* If results_dir was provided, make sure it is an absolute path */
        results_dir = convert_to_absolute_path(results_dir, `%RESULTS_DIR%`);
        let results_dir_var = new Variable(
            templ,
            `RESULTS_DIR`,
            `Directory containing LS-DYNA results`,
            results_dir,
            `Directory`,
            false,
            true
        );
    }

    let output_dir = get_expanded_variable_value(templ, `OUTPUT_DIR`);
    if (output_dir == null || ans == Window.YES) {
        output_dir = `${default_dir}/${output_dir_name}`;
        if (Batch()) {
            LogPrint(`REPORTER variable %OUTPUT_DIR% unspecified. Setting default %OUTPUT_DIR% = ${output_dir}`);
        }
        let output_dir_var = new Variable(
            templ,
            `OUTPUT_DIR`,
            `Directory where images and other output files will be written`,
            `%DEFAULT_DIR%/${output_dir_name}`,
            `Directory`,
            false,
            true
        );
    } else {
        /* If output_dir was provided, make sure it is an absolute path */
        output_dir = convert_to_absolute_path(output_dir, `%OUTPUT_DIR%`);
        let output_dir_var = new Variable(
            templ,
            `OUTPUT_DIR`,
            `Directory where images and other output files will be written`,
            output_dir,
            `Directory`,
            false,
            true
        );
    }

    if (Batch() || ans == Window.YES) {
        /* If it doesn't already exist, check that we can create the output_dir */
        let output_dir_exists = true;
        if (!File.IsDirectory(output_dir)) {
            LogPrint(`Output directory doesn't yet exist. Creating...`);
            let success = File.Mkdir(output_dir);
            if (success) {
                LogPrint(`Created output directory: ${output_dir}`);
            } else {
                output_dir_exists = false;

                if (Batch()) {
                    finish_script(false, `Unable to create output directory: ${output_dir}`);
                } else {
                    Window.Warning(
                        `Output directory`,
                        `Unable to create output directory. Please configure directories in PRIMER.\n\n${output_dir}`
                    );
                }
            }
        }

        /* Find D3PLOT results files in RESULTS_DIR and set RESULTS_FILE_D3PLOT */
        let results_file_d3plot_exists = true;

        let default_job = get_expanded_variable_value(templ, `DEFAULT_JOB`);
        let results_file_d3plot = find_lsdyna_files(results_dir, default_job, `D3PLOT`);
        if (results_file_d3plot == null) {
            results_file_d3plot = ``;
            results_file_d3plot_exists = false;
            if (Batch()) {
                finish_script(false, `Could not find D3PLOT results files in results directory: ${results_dir}`);
            } else {
                Window.Warning(
                    `Results file`,
                    `Could not find D3PLOT results files in results directory. Please configure directories in PRIMER.\n\n${results_dir}`
                );
            }
        } else {
            LogPrint(`Found D3PLOT results file: ${results_file_d3plot}`);
        }
        let results_file_d3plot_var = new Variable(
            templ,
            `RESULTS_FILE_D3PLOT`,
            `Results file for D3PLOT`,
            results_file_d3plot,
            `File(absolute)`,
            false,
            true
        );

        if (output_dir_exists && results_file_d3plot_exists) {
            show_job_control_gui = false;
        }
    }

    /* If running in batch, we skip the check in `#AAW D3PLOT check and do assessment` and proceed
     * directly to doing the assessment with whatever user data we can find. If running
     * interactively, regardless of whether we are showing the PRIMER job control GUI, set
     * JOB_CONTROL to `Check` in preparation for `#AAW D3PLOT check and do assessment` item. This
     * will be overridden by the PRIMER job control GUI if it is shown and the user clicks `Cancel`
     * (which would cause an abort). */
    let job_control = `Check`;
    if (Batch()) job_control = `Run`;
    let job_control_var = new Variable(
        templ,
        `JOB_CONTROL`,
        `Controls "Check", "Run", or "Skip" for various items, or "Abort" template generation entirely.`,
        job_control,
        `String`,
        false,
        true
    );

    /* Activate or deactivate `#AAW PRIMER job control GUI` item depending on whether or not we
     * want to show it. Always deactivate `#AAW PRIMER user data GUI`, which will be optionally
     * generated by `#AAW REPORTER run PRIMER rerun D3PLOT` item. */
    activate_master_page_item(`#AAW PRIMER job control GUI`, show_job_control_gui);
    activate_master_page_item(`#AAW PRIMER user data GUI`, false);
}

/**
 * Common function for finishing the script.
 * Complete the generation of the template if we have all the required information.
 * @param {Boolean} can_generate Whether or not to generate the template
 * @param {string} [msg] Error message (provide when not generating)
 * @example
 * finish_script(Template.GetCurrent(), false, `Keyword file not provided`);
 */
function finish_script(can_generate, msg) {
    /* Complete the template generation if we have the required information,
     * otherwise end with message */
    if (can_generate) {
        /* Deactivate this script item to avoid entering an infinite loop */
        activate_master_page_item(`#AAW REPORTER on load script`, false);
        /* Also deactivate `#AAW REPORTER subjective modifiers` because we have already generated
         * it. */
        activate_master_page_item(`#AAW REPORTER subjective modifiers`, false, false);

        Template.GetCurrent().Generate();
    } else {
        if (Batch()) LogError(msg);
        else {
            Window.Message(`Report automation`, msg);
        }
    }

    /* Whether or not we generated, deactivate all the script items except this script item. */
    activate_all_aaw_items(false);
    activate_master_page_item(`#AAW REPORTER on load script`, true);

    // TODO
    // Reactivate buttons on modifiers page

    //reactivate_modifier_buttons();

    Exit();
}

/**
 * Activates or deactivates #AAW (Automotive Assessment Workflow) items on all pages (master page
 * and normal pages) so they are only generated when required.
 * @param {boolean} active Whether to activate or deactivate the items
 * @example
 * activate_all_aaw_items(false);
 */
function activate_all_aaw_items(active) {
    activate_master_page_aaw_items(active);
    activate_normal_page_aaw_items(active);
}

/**
 * Activates or deactivates #AAW (Automotive Assessment Workflow) items on normal pages so that
 * they are only generated when required.
 * @param {boolean} active Whether to activate or deactivate the items
 * @example
 * activate_normal_page_aaw_items(false);
 */
function activate_normal_page_aaw_items(active) {
    let templ = Template.GetCurrent();

    /* We will search for items with names beginning with an identifiable string */
    let aaw_item_str = `#AAW`;

    /* Search for relevant items on other pages */
    let pages = templ.GetAllPages();
    for (let page of pages) {
        let items = page.GetAllItems();
        for (let item of items) {
            if (item.name.substring(0, aaw_item_str.length) == aaw_item_str) {
                /* (De)activate all #AAW items */
                item.active = active;
            }
        }
    }
}

/**
 * Activates or deactivates #AAW (Automotive Assessment Workflow) master page items so that they
 * are only generated once (they sit on the Master page so would be generated for every page otherwise).
 * @param {boolean} active Whether to activate or deactivate the items
 * @example
 * activate_master_page_aaw_items(false);
 */
function activate_master_page_aaw_items(active) {
    let templ = Template.GetCurrent();

    /* We will search for items with names beginning with an identifiable string */
    let aaw_item_str = `#AAW`;

    /* Search for relevant items on the Master page */
    let master = templ.GetMaster();

    let items = master.GetAllItems();

    for (let item of items) {
        if (item.name.substring(0, aaw_item_str.length) == aaw_item_str) {
            /* (De)activate all #AAW items */
            item.active = active;
        }
    }
}

/**
 * Activates or deactivates a specific item on the Master page so that it is only generated when
 * required.
 * @param {string} item_name The item name (the first item matching this name will be actioned)
 * @param {boolean} active Whether to activate or deactivate the item
 * @param {boolean} [expected = true] If true (default), will exit with an error if the requested item cannot be found
 * @example
 * activate_master_page_item(`#AAW REPORTER on load script`, false);
 */
function activate_master_page_item(item_name, active, expected = true) {
    let templ = Template.GetCurrent();

    /* Syntax for log messages */
    let de = `de`;
    if (active) de = ``;

    /* Search for relevant items on the Master page */
    let master = templ.GetMaster();

    let match = false;
    let items = master.GetAllItems();
    for (let item of items) {
        /* Deactivate this script item */
        if (item.name.substring(0, item_name.length) == item_name) {
            item.active = active;
            match = true;
            LogPrint(`Successfully ${de}activated master page item "${item_name}".`);
            break;
        }
    }

    if (!match && expected) {
        LogError(`Could not find master page item "${item_name}" to ${de}activate it.`);
        Exit();
    }
}

/**
 * Finds the first item in the template with the specified <item_name> and generates it.
 * Looks at the master page first, then all the normal pages.
 * @param {string} item_name The name of the item to be generated
 * @param {boolean} [expected = true] If true (default), will exit with an error if the requested item cannot be found
 * @example
 * generate_item(`#AAW REPORTER read variables from PRIMER job control`);
 */
function generate_item(item_name, expected = true) {
    let template = Template.GetCurrent();
    /* Loop through all pages in the template, searching for relevant items */
    let pages = template.GetAllPages();
    /* Add Master page to front so we consider it first */
    pages.unshift(template.GetMaster());
    let match = false;
    for (let p = 0; p < pages.length; p++) {
        let items = pages[p].GetAllItems();
        for (let item of items) {
            if (item.name.substring(0, item_name.length) == item_name) {
                LogPrint(`Found item "${item_name}" on page ${p}. Generating...`);
                match = true;
                /* Activate the item before generating it, then return it to its original status */
                let active = item.active;
                item.active = true;
                item.Generate();
                item.active = active;
                break;
            }
        }
        if (match) break;
    }
    if (!match && expected) {
        LogError(`Could not find item "${item_name}" in order to generate it.`);
        Exit();
    }
}

/**
 * If relative, converts a path to an absolute path based on the current working directory.
 * @param {string} path A path to a directory or filename.
 * @param {string} label The name of the path (e.g. "KEYWORD_FILE") used for print statements.
 */
function convert_to_absolute_path(path, label) {
    if (!File.Exists(path)) {
        LogError(`In function convert_to_absolute_path: specified path for ${label} does not exist: ${path}`);
        return path;
    }
    if (File.IsAbsolute(path)) {
        /* If path is already absolute, just return it. */
        return path;
    } else {
        LogPrint(`Converting ${label} to absolute path...`);
        LogPrint(`Relative path: ${path}`);
        let current_dir = GetCurrentDirectory();
        let abs_path = `${current_dir}/${path}`;
        LogPrint(`Absolute path: ${abs_path}`);
        if (!File.Exists(abs_path)) {
            /* Trap the unexpected case where the conversion hasn't worked. */
            LogError(
                `In function convert_to_absolute_path: converted absolute path does not exist. Reverting to relative path.`
            );
            return path;
        }
        return abs_path;
    }
}

/**
 * Searches a directory for filenames matching either Arup or LSTC filename conventions.
 * Searches for files in directory <dir> of type <file_type>, possibly containing
   <job_name>, and returns the first match in the priority lists below.
   @param {string} dir Directory to search
   @param {string} job_name Root filename to search for
   @param {string} file_type File type to search for (can be "PRIMER", "D3PLOT", "T/HIS", "OTF")
   @returns {?string}
   @example
   let absolute_filename = find_lsdyna_files("C:/my/results/directory", "job_001", "D3PLOT");
 */
function find_lsdyna_files(dir, job_name, file_type) {
    let filters = [];
    let filename;

    switch (file_type) {
        case "PRIMER":
            filters = [
                new RegExp("^" + job_name + ".key$"),
                new RegExp("^" + job_name + ".k$"),
                new RegExp("^" + job_name + ".*.key$"),
                new RegExp("^" + job_name + ".*.k$"),
                new RegExp(".*.key$"),
                new RegExp(".*.k$"),
                new RegExp(".*.dyn$")
            ];

            break;

        case "D3PLOT":
            filters = [
                new RegExp("^" + job_name + ".ptf$"),
                new RegExp("^" + job_name + ".d3plot$"),
                new RegExp("^d3plot$"),
                new RegExp("^" + job_name + ".*.ptf$"),
                new RegExp("^" + job_name + ".*.d3plot$"),
                new RegExp(".*.ptf$"),
                new RegExp(".*.d3plot$"),
                new RegExp(".*.f(em)?z(ip)?$")
            ];
            break;

        case "T/HIS":
            filters = [
                new RegExp("^" + job_name + ".thf$"),
                new RegExp("^" + job_name + ".d3thdt$"),
                new RegExp("^d3thdt$"),
                new RegExp("^" + job_name + ".*.thf$"),
                new RegExp("^" + job_name + ".*.d3thdt$"),
                new RegExp(".*.thf$"),
                new RegExp(".*.d3thdt$"),
                new RegExp("^" + job_name + ".binout.*$"),
                new RegExp("^binout.*$"),
                new RegExp("^" + job_name + ".*.binout.*$"),
                new RegExp(".*.binout.*$")
            ];
            break;

        case "OTF":
            filters = [
                new RegExp("^" + job_name + ".otf$"),
                new RegExp("^" + job_name + ".d3hsp$"),
                new RegExp("^d3hsp$"),
                new RegExp("^" + job_name + ".*.otf$"),
                new RegExp("^" + job_name + ".*.d3hsp$"),
                new RegExp(".*.otf$"),
                new RegExp(".*.d3hsp$")
            ];
            break;

        default:
            LogError(`Unexpected <file_type = "${file_type}" in function find_lsdyna_files.`);
            Exit();
            break;
    }

    let filestore = [];
    for (let filter of filters) {
        let filelist = File.FindFiles(dir, `*`, false);
        for (let file of filelist) {
            if (filter.test(file) == true) {
                filestore.push(file);
            }
        }
        if (filestore.length > 0) {
            filestore.sort();
            /* Pick first matching file and strip off rest of path */
            filename = filestore[0].substring(
                Math.max(filestore[0].lastIndexOf("/"), filestore[0].lastIndexOf("\\")) + 1
            );
            break;
        }
    }

    if (filestore.length == 0) return null;

    let absolute_filename = dir + "/" + filename;

    return absolute_filename;
}

/**
 * Wrapper function to return the expanded variable value, handling the case where it doesn't exist
 * @param {Template} templ The REPORTER template object
 * @param {string} name Variable name you want to get value for
 * @returns {?string}
 */
function get_expanded_variable_value(templ, name) {
    let value = templ.GetVariableValue(name);
    if (value == null) {
        return null;
    } else {
        let expanded_value = templ.ExpandVariablesInString(value);

        /* recursively expand variables until no more variables are found */
        while (expanded_value != value) {
            LogPrint(`Expanded variable ${name} from "${value}" -> "${expanded_value}"`);
            value = expanded_value;
            expanded_value = templ.ExpandVariablesInString(value);
        }

        /** there is a bug where the templ.ExpandVariablesInString() does not handle slashes correctly
         * and accidentally puts double slashes in the path.
         * This is not generally a problem except when working with UNC file paths
         * The workaround is to check if it starts with 4 slashes then remove 2 of them
         */
        if (/^[\/]{4}/.test(expanded_value)) {
            LogPrint(`Removing 2 slashes from the expanded variable value: ${expanded_value}`);
            expanded_value = expanded_value.substring(2); // Remove first two slashes
        }

        return expanded_value;
    }
}

/**
 * Wrapper function for Template.GetVariableValue that also:
 * 1. Checks that "int" and "float" variables are valid
 * 2. Updates a status object with information about any missing or invalid variables
 * 3. Creates a variable for any missing variables with the value "Missing"
 * @param {object} status Status object, used to track any missing or invalid variables
 * @param {string} name Variable name
 * @param {string} [type = "string"] Variable type. Must be "string", "int", or "float".
 * @param {boolean} [required = true] Whether or not the varialbe is required (true) or optional (false)
 * @returns {*} Variable value
 * @example
 * let head_hic_score = get_variable_value(status, `${m}_${occ}_HEAD_HIC_SCORE`, "float");
 */
function get_variable_value(status, name, type = "string", required = true) {
    let templ = Template.GetCurrent();
    let value = templ.GetVariableValue(name);
    if (value == null) {
        if (required) {
            status.success = false;
            status.missing.push(name);
        }
        /* Create variable with value "Missing" so that REPORTER doesn't complain about missing
         * variables. */
        let missing_var = new Variable(templ, name, "Requested variable is missing", `Missing`, "String", false, true);

        return value;
    }
    if (type == "string") {
        /* For string variables, just return value without any further checks */
        return value;
    } else if (type == "int" || type == "float") {
        /* For numeric values, check they are valid integers or floats */
        let parsed;
        if (type == "int") {
            parsed = Number.parseInt(value);
        }
        if (type == "float") {
            parsed = Number.parseFloat(value);
        }
        if (Number.isNaN(parsed)) {
            status.success = false;
            status.invalid.push(name);
        }
        return parsed;
    } else {
        /* We shouldn't get here so print error and exit */
        LogError(`Unexpected argument <type = ${type}> in function get_variable_value.`);
        Exit();
    }
}

/**
 * Prints warnings about missing and/or invalid REPORTER variables that have been requested during
 * score calculations.
 * @param {object} status Contains information about missing/invalid variables, expected in the form { success: false, missing: [...], invalid: [...] }
 * @param {string} descriptor String describing what is affected
 * @example
 * let body_region_label = `knee, femur and pelvis`;
 * let status = { success: true, missing: [], invalid: [] };
 * status.success = false;
 * status.missing.push("M1_DRIVER_FEMUR_COMPRESSION_EXCEEDENCE_SCORE");
 * status.invalid.push("M1_DRIVER_KNEE_COMPRESSION_SCORE");
 * if (!status.success) {
 *     warn_about_missing_or_invalid_variables(status, `M1 DRIVER ${body_region_label} score calculation`);
 * }
 */
function warn_about_missing_or_invalid_variables(status, descriptor) {
    if (status.missing && status.missing.length > 0) {
        LogWarning(`The following variables required for the ${descriptor} are missing:`);
        for (let name of status.missing) {
            LogPrint(name);
        }
    }
    if (status.invalid && status.invalid.length > 0) {
        LogWarning(`The following variables used in the ${descriptor} have invalid values:`);
        for (let name of status.invalid) {
            LogPrint(name);
        }
    }
}

/**
 * Returns a list of models based on the value of the %MODEL_LIST% variable.
 * Expect the value of %MODEL_LIST% to be in the form:
 *
 *     "M1, M2, M3"
 *
 *  Return null if missing
 *  @returns {?string[]}
 */
function get_model_list() {
    let templ = Template.GetCurrent();
    let models_list_str = templ.GetVariableValue("MODEL_LIST");
    if (models_list_str == null) {
        return null;
    } else {
        return models_list_str.split(/,\s*/);
    }
}
