/**
 * Cross-platform function for the Window.GetFile method. Instead of taking a string, it takes
 * in a list of extensions and constructs the appropriate filter string based on the OS. It also takes
 * the parameters for save and initial.
 * @param {string[]|string} [extensions=[]] - List of file extensions (e.g., [".csv", ".txt"]) or a single extension string (e.g., ".csv")
 * @param {boolean} [save=false] - Whether the dialog is for saving a file
 * @param {string} [initial=""] - Initial file path or directory name. If a file path is provided, the parent directory is used. If a directory is provided, it's used as-is.
 * @returns {string|null} - Selected filename as a string, or null if the user cancels
 */
function GetFile(extensions = [], save = false, initial = "") {
    /* Check that extensions is a string or an array. */
    if (typeof extensions === "string") {
        /* If the user passes a string of delimited values (with ; commas or spaces to separate the extensions) instead of an array,
         * split the string into an array using the appropriate delimiter. The regex splits on commas, semicolons, or spaces. */
        extensions = extensions.split(/[,; ]+/);
    } else if (!Array.isArray(extensions)) {
        LogError("Function GetFile: extensions parameter must be a string or an array of strings.");
        return null;
    }

    // Replace all the *'s at the start of the string (this is generally before the first dot of the extension).
    // We do this because we add the * later on using the joiner variable.
    extensions = extensions.map((ext) => ext.replace(/^\**/, ""));

    const is_linux = Unix();

    /**
     * The filter adds a * before each extension (excluding the first) and joins them with either spaces (Linux) or semicolons (Windows).
     */
    let file, filter;
    const joiner = is_linux ? " *" : "; *";
    const file_separator = is_linux ? "/" : "\\";

    // Adjust initial path for Linux and Windows if needed. On Linux, convert backslashes to forward slashes.
    // On Windows, convert forward slashes to backslashes. This ensures the initial path is in the correct format for the OS.
    if (is_linux && initial) {
        initial = initial.replace(/\\/g, file_separator);
    }

    if (!is_linux && initial) {
        initial = initial.replace(/\//g, file_separator);
    }

    filter = extensions.map((ext) => `${ext}`).join(joiner);

    if (initial && File.Exists(initial)) {
        // For Window.GetFile, if a directory is used but is not properly specified with a / or \\ at the end, then the appropriate
        // delimiter (e.g. forward slash) is added to force it to use the correct directory. Otherwise the function may misinterpret it as a file.
        if (File.IsDirectory(initial) && !initial.endsWith(file_separator)) {
            initial = initial + file_separator;
        }
        file = Window.GetFile(filter, save, initial);
    } else {
        file = Window.GetFile(filter, save);
    }
    return file;
}

function clean_up_table() {
    let summary_table_item = get_named_item_on_named_page("MAIN_SUMMARY_TABLE", "Summary");

    for (let row = 2; row < summary_table_item.rows; row++) {
        for (let col = 2; col < summary_table_item.columns - 1; col++) {
            set_cell_text(summary_table_item, row, col, "");
        }
    }
}

/**
 * There is a bug with REPORTER 21.1 where some unicode characters are not displayed correctly in the text box.
 * This function adds padding to the end of the text to ensure that the unicode characters are displayed correctly
 * @param {String} [text="😀"] - The text to be displayed in the text box
 * @param {Boolean} [debug=false] - If true, the function will print the text with padding and the variable value
 * @returns {String} text_with_padding
 */
function get_unicode_text_with_padding(text = "😀", debug = false) {
    let templ = Template.GetCurrent();
    /** create a temporary variable to set the value of */
    let rep_var = new Variable(templ, "UNICODE_TEXT_WITH_PADDING");
    /** set the value to the string plus padding of $ characters for 4 times the length */
    rep_var.value = text + "$".repeat(text.length * 4);
    /** get the value after setting and subtract the original length to see how many
     *  $ characters are "swallowed up" in the unicode conversion */
    let num_trailing_stars = rep_var.value.length - text.length;
    let text_with_padding = text + "$".repeat(text.length * 4 - num_trailing_stars);
    /** if debugging set the REPORTER variable value to confirm that it worked
     * otherwise remove the variable*/
    if (debug) {
        rep_var.value = text_with_padding;
        /** log print the string with padding and the variable value */
        LogPrint(text_with_padding);
        /** note that the variable value will not have the padding so it will need to have the padding
         * added again using get_unicode_text_with_padding() */
        LogPrint(rep_var.value);
        LogPrint(get_unicode_text_with_padding(rep_var.value, false));
    } else {
        rep_var.Remove();
    }
    return text_with_padding;
}

function run_on_load() {
    let templ = Template.GetCurrent();
    let master = templ.GetMaster();
    let on_load_item = Item.GetFromName(master, "#SIMVT REPORTER on load script");
    on_load_item.Generate();
}

/**
 * @param {String} extension e.g. ".simvt"
 * @returns {?String} file path or null
 */
function load_file_item(extension) {
    let file = GetFile(extension);

    if (!file) return null;
    if (!File.Exists(file)) {
        Window.Information(`Missing file`, `"${file}" does not exist`);
        return null;
    }

    LogPrint(file);

    return file;
}

/**
 * open a window to request a directory
 * @returns {?String} directory path or null
 */
function load_directory_item() {
    let directory = Window.GetDirectory();

    if (!directory) return null;
    if (!File.IsDirectory(directory)) {
        Window.Information(`Missing file`, `"${directory}" does not exist`);
        return null;
    }
    LogPrint(directory);

    return directory;
}

/**
 * Get the first item on the page where the name matches contains the given name (note that exact match is not required as when we duplicate a page the
 * items get a suffix added (e.g. "_1") and we still want to get the item
 * @param {String} name
 * @param {Page} page
 * @returns {Item}
 */
function get_item_by_name(name, page) {
    if (name == null || name == "") {
        throw Error(`get_item_by_name() called with empty name`);
    }
    for (let item of page.GetAllItems()) {
        if (RegExp(`^${name}`, "i").test(item.name)) return item;
    }
    throw Error(`No items contain the name "${name}" on page ${page.name}`);
}
/**
 * Get a page from the template which matches the name
 * @param {String} name
 * @param {Boolean} [throw_error = true]
 * @param {Template} templ
 * @returns {?Page}
 */
function get_page_by_name(name, throw_error = true, templ = Template.GetCurrent()) {
    let pages = templ.pages;
    for (let i = 0; i < pages; i++) {
        let page = templ.GetPage(i);
        if (page.name == name) return page;
    }
    if (throw_error) throw Error(`No page exists with a name that contains "${name}"`);

    return null;
}

/**
 * Get the first item on the page where the name matches contains the given name (note that exact match is not required as when we duplicate a page the
 * items get a suffix added (e.g. "_1") and we still want to get the item
 * @param {String} item_name
 * @param {String} page_name
 * @returns {Item}
 */
function get_named_item_on_named_page(item_name, page_name) {
    let page = get_page_by_name(page_name);
    return get_item_by_name(item_name, page);
}

/**
 * Wrapper function to return the expanded variable value, handling the case where it doesn't exist
 * @param {Template} templ The REPORTER template object
 * @param {string} name Variable name you want to get value for
 * @returns {?string}
 */
function get_expanded_variable_value(templ, name) {
    let value = templ.GetVariableValue(name);
    if (value == null) {
        return null;
    } else {
        let expanded_value = templ.ExpandVariablesInString(get_unicode_text_with_padding(value));

        /* recursively expand variables until no more variables are found */
        while (expanded_value != value) {
            LogPrint(`Expanded variable ${name} from "${value}" -> "${expanded_value}"`);
            value = expanded_value;
            expanded_value = templ.ExpandVariablesInString(get_unicode_text_with_padding(value));
        }

        /** there is a bug where the templ.ExpandVariablesInString() does not handle slashes correctly
         * and accidentally puts double slashes in the path.
         * This is not generally a problem except when working with UNC file paths
         * The workaround is to check if it starts with 4 slashes then remove 2 of them
         */
        if (/^[\/]{4}/.test(expanded_value)) {
            LogPrint(`Removing 2 slashes from the expanded variable value: ${expanded_value}`);
            expanded_value = expanded_value.substring(2); // Remove first two slashes
        }

        return expanded_value;
    }
}

/**
 * Delete the first item on the page where the name matches contains the given name (note that exact match is not required as when we duplicate a page the
 * items get a suffix added (e.g. "_1") and we still want to delete the item
 * @param {String} name
 * @param {Page} page
 * @returns {Boolean} true if deleted
 */
function delete_item_by_name(name, page) {
    let index = 0;
    for (let item of page.GetAllItems()) {
        if (RegExp(name, "i").test(item.name)) {
            page.DeleteItem(index);
            return true;
        }
        index++;
    }
    return false;
}

/**
 * Delete pages from the template which matches the name. Do this in reverse order or we get a crash
 * @param {String} name
 * @param {Template} templ
 */
function delete_pages_by_name(name = "DELETEME", templ = Template.GetCurrent()) {
    let delete_count = 0;
    let pages = templ.pages;
    /** this must be in reverse order */
    let indices_of_pages_to_delete = [];
    for (let i = pages - 1; i >= 0; i--) {
        if (templ.GetPage(i).name == name) {
            indices_of_pages_to_delete.push(i);
        }
    }

    for (let index of indices_of_pages_to_delete) {
        if (index >= templ.pages) break; // give up trying to delete pages
        LogWarning(`Delete page ${index}`);
        templ.DeletePage(index);
    }
    templ.Update();
    if (indices_of_pages_to_delete.length)
        LogPrint(`Deleted ${indices_of_pages_to_delete.length} pages with the name "${name}"`);
    else LogPrint(`No pages found to delete with the name "${name}"`);
}

/** functions for Item.TABLE */

/**
 * Get the cell text from the table at row and col
 * @param {Item} table
 * @param {Number} row zero indexed row number
 * @param {Number} col zero indexed column number
 * @returns {String} text
 */
function get_cell_text(table, row, col) {
    if (!(table instanceof Item)) {
        throw Error(`table arg is not an Item in set_cell_text()`);
    }
    if (table.type != Item.TABLE) {
        throw Error(`table arg is not an Item.TABLE in set_cell_text()`);
    }

    if (row >= table.rows) {
        throw Error(`table "${table.name}" does not have row index ${row} in set_cell_text()`);
    }

    if (col >= table.columns) {
        throw Error(`table "${table.name}" does not have column index ${row} in set_cell_text()`);
    }

    return table.GetCellProperties(row, col).text;
}

/**
 * Set the cell text
 * @param {Item} table
 * @param {Number} row zero indexed row number
 * @param {Number} col zero indexed column number
 * @param {?String} text
 * @param {?String} [undefined_var_text = null] value to set text to if the variable is undefined or an empty string
 *
 */
function set_cell_text(table, row, col, text, undefined_var_text = null) {
    if (!(table instanceof Item)) {
        throw Error(`table arg is not an Item in set_cell_text()`);
    }
    if (table.type != Item.TABLE) {
        throw Error(`table arg is not an Item.TABLE in set_cell_text()`);
    }

    if (row >= table.rows) {
        throw Error(`table "${table.name}" does not have row index ${row} in set_cell_text()`);
    }

    if (col >= table.columns) {
        throw Error(`table "${table.name}" does not have column index ${col} in set_cell_text()`);
    }

    if (text == null) {
        /** if the text field is null it essentially means skip setting the text (i.e. leave the default) */
        return;
    }

    let match;
    if ((match = text.match(/%(.*)%/))) {
        let var_text = get_expanded_variable_value(Template.GetCurrent(), match[1]);
        if (var_text == null) {
            LogWarning(
                `variable ${text} does not exist so cell (${row},${col}) in "${table.name}" will be set to "${undefined_var_text}" in set_cell_text()`
            );
            if (typeof undefined_var_text == "string") text = undefined_var_text;
            else {
                text = `MISSING:${match[1]}`;
            }
        } else {
            //the variable is defined, but we overwrite the value if undefined_var_text is a string
            if (typeof undefined_var_text == "string") text = undefined_var_text;
        }
    }

    let props = { text: get_unicode_text_with_padding(text) };
    table.SetCellProperties(props, row, col);
}

/**
 * Set the cell conditions from a JSON object defined in the reporter JSON file
 * @param {Item} table
 * @param {Number} row
 * @param {Number} col
 * @param {Object} conditions_json if expected values are numbers (e.g. 0.5) or strings (e.g. "Pass")
 */
function set_cell_conditions(table, row, col, conditions_json) {
    if (!(conditions_json instanceof Array)) {
        LogPrint(`Conditions ${conditions_json} is not an array so will be skipped`);
        return;
    }

    let default_props = table.GetCellProperties(row, col);

    let conditions = [];

    for (let condition of conditions_json) {
        if (condition.hasOwnProperty("name") && condition.hasOwnProperty("type") && condition.hasOwnProperty("value")) {
            let condition_props = {
                name: get_unicode_text_with_padding(condition.name),
                type: eval(`Reporter.${condition.type}`), //convert string to Reporter.CONDITION_### constant
                value: get_unicode_text_with_padding(condition.value),
                fillColour: Colour.RGB(condition.fillColour.red, condition.fillColour.green, condition.fillColour.blue),
                fontName: condition.fontName == undefined ? default_props.fontName : condition.fontName,
                fontSize: condition.fontSize == undefined ? default_props.fontSize : condition.fontSize,
                fontStyle: condition.fontStyle == undefined ? default_props.fontStyle : condition.fontStyle,
                justify: condition.justify == undefined ? default_props.justify : condition.justify,
                textColour: condition.textColour == undefined ? default_props.textColour : condition.textColour
            };
            conditions.push(condition_props);
        } else {
            LogWarning(
                `Condition ${condition} is missing a required properties (name, type and value) so will be skipped`
            );
        }

        let num_conditions = Math.max(default_props.conditions, conditions.length);
        for (let i = 0; i < num_conditions; i++) {
            if (i < conditions.length) table.SetCondition(i, row, col, conditions[i]);
            else {
                /** clear any surplus conditions that were already set that we no longer need */
                try {
                    table.RemoveCondition(conditions.length, row, col);
                } catch (error) {
                    LogWarning(
                        `Could not remove condition for cell (${row},${col}) in "${table.name}"\n${error.message}`
                    );
                }
            }
        }
    }
}

/** functions for duplication */

function copy_table(table_to_copy, page) {
    let copied_table = new Item(page, Item.TABLE, table_to_copy.name);

    copied_table.DeleteColumn(1);
    copied_table.DeleteRow(1);

    for (let row = 0; row < table_to_copy.rows - 1; row++) copied_table.InsertRow(0);
    for (let col = 0; col < table_to_copy.columns - 1; col++) copied_table.InsertColumn(0);

    let properties_to_copy = [
        "x",
        "y",
        "bottomMargin",
        "leftMargin",
        "lineColour",
        "lineWidth",
        "rightMargin",
        // "saveCSV",
        // "saveCSVFilename",
        // "saveXlsx",
        // "saveXlsxFilename",
        "topMargin",
        "width",
        "height"
    ];

    for (let property of properties_to_copy) {
        copied_table[property] = table_to_copy[property];
    }

    for (let row = 0; row < copied_table.rows; row++) {
        for (let col = 0; col < copied_table.columns; col++) {
            let props = table_to_copy.GetCellProperties(row, col);
            copied_table.SetCellProperties(props, row, col);
            copied_table.MergeCells(
                props.rowMergeOrigin,
                props.columnMergeOrigin,
                row - props.rowMergeOrigin + 1,
                col - props.columnMergeOrigin + 1
            );
        }
    }

    return copied_table;
}

function copy_item(item_to_copy, page) {
    if (item_to_copy.type == Item.TABLE) return copy_table(item_to_copy, page);
    let copied_item = new Item(page, item_to_copy.type, item_to_copy.name);

    let read_only_props = ["columns", "conditions", "filetype", "rows", "type"];
    for (let prop in item_to_copy) {
        if (read_only_props.includes(prop)) continue;
        try {
            copied_item[prop] = item_to_copy[prop];
        } catch (error) {
            LogPrint(error.toString());
        }
    }

    return copied_item;
}
