/** set up global template variable for the current template */
var templ = Template.GetCurrent();
var colour_light_blue_grey = Colour.RGB(243, 247, 251); // #F3F7FB
var colour_dark_blue_grey = Colour.RGB(100, 115, 130); //#647382

var normal_props = {
    fontSize: 8,
    fillColour: colour_light_blue_grey,
    textColour: Colour.Black(),
    fontStyle: Reporter.TEXT_NORMAL,
    fontName: "Microsoft YaHei UI",
    justify: Reporter.JUSTIFY_CENTRE | Reporter.JUSTIFY_MIDDLE
};

var bold_props = {
    fontSize: 12,
    fillColour: colour_dark_blue_grey,
    textColour: Colour.White(),
    fontStyle: Reporter.TEXT_BOLD,
    fontName: "Microsoft YaHei UI",
    justify: Reporter.JUSTIFY_CENTRE | Reporter.JUSTIFY_MIDDLE
};

var header_props = {
    fontSize: 8,
    fillColour: colour_dark_blue_grey,
    textColour: Colour.White(),
    fontStyle: Reporter.TEXT_BOLD,
    fontName: "Microsoft YaHei UI",
    justify: Reporter.JUSTIFY_CENTRE | Reporter.JUSTIFY_MIDDLE
};

/** FUNCTIONS */

/**
 * Read a JSON file and return the parsed object
 * @param {String} filepath
 * @returns {Object} json
 */
function ReadJSON(filepath) {
    function unicodeReplacer(key, value) {
        if (typeof value === "string") {
            return value.replace(/[\u007F-\uFFFF]/g, function (c) {
                return "\\u" + ("0000" + c.charCodeAt(0).toString(16)).slice(-4);
            });
        }
        return value;
    }

    function unicodeReviver(key, value) {
        if (typeof value === "string") {
            return value.replace(/\\u[\dA-Fa-f]{4}/g, function (match) {
                return String.fromCharCode(parseInt(match.replace("\\u", ""), 16));
            });
        }
        return value;
    }

    try {
        if (File.Exists(filepath)) {
            let json_file = new File(filepath, File.READ);

            let line;
            let lines = [];

            //@ts-ignore
            while ((line = json_file.ReadLongLine()) != File.EOF) {
                lines.push(line);
            }

            json_file.Close();

            let json = JSON.parse(lines.join(`\n`), unicodeReviver);

            return json;
        }

        throw Error(`${filepath} does not exist`);
    } catch (error) {
        // Handle error
        throw Error(`Failed to read JSON from ${filepath}: ${error.message}`);
    }
}

/**
 * Attempt to parse the reporter json. Throw an Error if we encounter an issue (e.g. the reporter_json_file_path does not exist)
 */
function parse_reporter_json(reporter_json_file_path = null) {
    /** if the file path is not passed in then we use the reporter_data.json file in the temporary folder
     * this is the normal case. We onlt pass in the path when testing */

    if (!reporter_json_file_path) {
        let reporter_temp = Variable.GetFromName(Template.GetCurrent(), "REPORTER_TEMP").value;
        reporter_json_file_path = `${reporter_temp}/reporter_data.json`;
    }

    /** @type {REPORTER_JSON_DATA} */
    let reporter_json = ReadJSON(reporter_json_file_path);

    /** set the output directory based on the reporter json property*/
    set_output_file_variable(reporter_json.output_directory);

    /** create the variables for populating the results tables*/
    create_reporter_variables(reporter_json);

    /** store the number of original pages before creating any new pages */
    let num_pages = templ.pages;

    /** create the pages if they are missing */
    create_and_update_pages(reporter_json);

    /** delete the original pages */
    for (let index = 0; index < num_pages; index++) templ.DeletePage(0);
}

/**
 * create reporter variables from the variables array in reporter_data.json
 * @param {REPORTER_SIMVT_JSON_DATA} reporter_json
 */
function create_reporter_variables(reporter_json) {
    let temp_var;

    if (!reporter_json.hasOwnProperty("variables")) {
        throw Error(`REPORTER data is missing the variables Object`);
    }

    /** loop through every variable and convert it */
    for (let var_name in reporter_json.variables) {
        let var_value = get_unicode_text_with_padding(reporter_json.variables[var_name]);
        let type = Number.isNaN(var_value) ? "String" : "Number";

        temp_var = new Variable(
            Template.GetCurrent(),
            var_name.toUpperCase(),
            "",
            var_value,
            Number.isNaN(var_value) ? "String" : "Number",
            false, // READ_ONLY should be false, otherwise the variable will not be written when saving a REPORT
            true // TEMPORARY
        );

        // set the format and precision for the variable if it is a number
        if (type === "Number") {
            temp_var.precision = 3;
            temp_var.format = Variable.FORMAT_GENERAL;
        }
    }
}

/** this function creates the REPORTER pages from the pages array in the reporter_data.json file
 * @param {Object} reporter_json
 */
function create_and_update_pages(reporter_json) {
    let i = 0;
    for (let page_json of reporter_json.pages) {
        i++;
        /** the order of pages is the order they are created */

        if (!page_json.placeholder_page_name || !page_json.title) {
            LogWarning(
                `page placeholder_page_name and title properties are both required, but not defined so page ${i} will be skipped `
            );
            continue;
        }

        try {
            switch (page_json.placeholder_page_name) {
                case "DETAILED_RESULTS":
                    create_detailed_results_pages(page_json);
                    break;
                case "Summary": // the first page in a report that summarises the results
                    create_summary_results_pages(page_json);
                    break;
                case "PLACEHOLDER_1_CHANNEL": // the placeholder for sensors with only 1 channel
                case "PLACEHOLDER_2_CHANNEL": // the placeholder for sensors with 2 channels
                case "PLACEHOLDER_3_CHANNEL": // the placeholder for sensors with 3 channels
                case "Head Excursion": // the head excursion page for Euro NCAP
                    create_generic_page_from_placeholder(page_json);
                    break;
                default:
                    LogWarning(
                        `"${
                            page_json.placeholder_page_name
                        }" page type is not yet supported so ${get_unicode_text_with_padding(
                            page_json.title
                        )} page will be skipped `
                    );
                    break;
            }
        } catch (error) {
            LogError(`Error creating page ${get_unicode_text_with_padding(page_json.title)} : ${error.message}`);
        }
    }
}

function create_generic_page_from_placeholder(page_json) {
    /** get_page_by_name will throw an error if the page does not exist */
    let placeholder_page = get_page_by_name(page_json.placeholder_page_name);
    /** place the duplicate page at the end - we will delete all the original pages later */
    let page = placeholder_page.Duplicate(templ.pages);
    /** set the name (a.k.a. title) of the new page */
    page.name = get_unicode_text_with_padding(page_json.title);

    /** edit each item on the page - first check the item exists. If it does not then skip with a warning */
    for (let item_json of page_json.items) {
        let item;
        try {
            item = get_placholder_item(item_json, page);
        } catch (error) {
            // we get here if we cannot find the placeholder item or if the item type does not match the expected type
            LogWarning(`${error.message} - we will attempt to create it instead`);
            item = create_item_from_json(item_json, page);
            if (!(item instanceof Item)) {
                LogWarning(`Item "${item_json.name}" was not created so it will be skipped`);
            }
            // we have either created the item or failed to. In both cases we will skip the code below which only
            // applies to updating existing placeholder items
            continue;
        }

        LogPrint(`Creating ${item_json.name} (${item_json.type}) item`);
        switch (item.type) {
            case Item.TABLE:
                update_table_from_json(item, item_json);
                break;
            case Item.TEXT:
                update_text_from_json(item, item_json);
                break;
            case Item.TEXTBOX:
                if (item_json.hasOwnProperty("placeholder_name")) {
                    if (/image.*/i.test(item_json.placeholder_name)) {
                        update_placeholder_image_from_json(item, item_json, page);
                        break;
                    }
                }
                update_text_from_json(item, item_json);
                break;
            case Item.IMAGE_FILE:
                item.file = item_json.file;
                break;
            default:
                LogWarning(`Item "${item_json.name}" is not supported so it will be skipped`);
                break;
        }
    }
}

/**
 * If there are a lot of sensors then the summary page may overflow and we need to create
 * additional pages to hold the detailed results.
 * @param {Object} page_json
 */
function create_summary_results_pages(page_json, index = 0) {
    let main_summary_table_json = page_json.items.find((item) => item.placeholder_name == "MAIN_TABLE");

    if (!main_summary_table_json) {
        LogError(
            `No TABLE item with placeholder_name "MAIN_TABLE" found so the summary results page(s) will be skipped`
        );
        return;
    }

    let max_rows = 23; // maximum number of rows in the summary table (excluding the 2 header rows) before we
    /** create a new page if the main summary table has more rows than the maximum then we need to create additional pages */
    if (main_summary_table_json.rows.length > max_rows) {
        let original_rows = main_summary_table_json.rows;
        let first_page_rows = original_rows.slice(0, max_rows);
        let remaining_rows = original_rows.slice(max_rows);

        main_summary_table_json.rows = first_page_rows;
        /** create the first page from the placeholder */
        create_generic_page_from_placeholder(page_json);

        /** loop over the remaining rows and create a new page for each set of max_rows */
        main_summary_table_json.rows = remaining_rows;
        create_summary_results_pages(page_json, index + 1);

        /** restore the original rows so that the next page can be created correctly */
        main_summary_table_json.rows = original_rows;
    } else {
        create_generic_page_from_placeholder(page_json);
    }

    let PLACEHOLDER_DETAILED_RESULTS_TABLE = "DETAILED_RESULTS_TABLE";
    // row, col, cell_text

    let page_detailed_results = get_page_by_name("DETAILED_RESULTS");
    let summary_table_item = get_item_by_name("DETAILED_RESULTS_TABLE", page_detailed_results);

    let table_json;
    for (let item of page_json.items) {
        if (item.placeholder_name == PLACEHOLDER_DETAILED_RESULTS_TABLE && item.type == "TABLE") {
            table_json = item;
            break;
        }
    }

    if (!table_json) {
        throw Error(
            `No TABLE item with placeholder_name "${PLACEHOLDER_DETAILED_RESULTS_TABLE}" found so the detailed results page(s) will be skipped`
        );
    }

    /** loop over all the rows in the table and create a new page if the row number exceeds the number of rows in the summary table */
    for (let row = 0; row < table_json.rows.length; row++) {
        let num_header_rows = 2;
        let remainder = row % (summary_table_item.rows - num_header_rows);
        let table_index = (row - remainder) / (summary_table_item.rows - num_header_rows) + 1;

        // no suffix if table_index == 0 otherwise it is added with underscore e.g. DETAILED_RESULTS_1;
        let page_suffix = table_index ? `_${table_index}` : "";

        let new_page_detailed_results = get_page_by_name(`DETAILED_RESULTS${page_suffix}`, false);
        if (!new_page_detailed_results) {
            new_page_detailed_results = page_detailed_results.Duplicate(templ.pages);
            /**set the new page name so that it can be found the next time */
            new_page_detailed_results.name = `DETAILED_RESULTS${page_suffix}`;
            let header = get_item_by_name("Detailed Results", new_page_detailed_results);
            header.text = table_index > 1 ? "%DETAILED_RESULTS% %CONTINUED%" : "%DETAILED_RESULTS%";
            summary_table_item = get_item_by_name("DETAILED_RESULTS_TABLE", new_page_detailed_results);
            /** clear the table since we are starting a new page and the page we duplicated will have a full table*/
            for (let r = num_header_rows; r < summary_table_item.rows; r++) {
                for (let c = 0; c < summary_table_item.columns; c++) {
                    set_cell_text(summary_table_item, r, c, "");
                }
            }
        }
        summary_table_item = get_item_by_name("DETAILED_RESULTS_TABLE", new_page_detailed_results);

        /** set the detailed summary page table text*/
        for (let col = 0; col < table_json.rows[row].length; col++) {
            let cell_json = table_json.rows[row][col];
            set_cell_text(summary_table_item, remainder + num_header_rows, col, cell_json.text);
            if (cell_json.conditions)
                set_cell_conditions(summary_table_item, remainder + num_header_rows, col, cell_json.conditions);
        }
    }
}

function create_detailed_results_pages(page_json) {
    let PLACEHOLDER_DETAILED_RESULTS_TABLE = "DETAILED_RESULTS_TABLE";
    // row, col, cell_text

    let page_detailed_results = get_page_by_name("DETAILED_RESULTS");
    let summary_table_item = get_item_by_name("DETAILED_RESULTS_TABLE", page_detailed_results);

    let table_json;
    for (let item of page_json.items) {
        if (item.placeholder_name == PLACEHOLDER_DETAILED_RESULTS_TABLE && item.type == "TABLE") {
            table_json = item;
            break;
        }
    }

    if (!table_json) {
        throw Error(
            `No TABLE item with placeholder_name "${PLACEHOLDER_DETAILED_RESULTS_TABLE}" found so the detailed results page(s) will be skipped`
        );
    }

    /** loop over all the rows in the table and create a new page if the row number exceeds the number of rows in the summary table */
    for (let row = 0; row < table_json.rows.length; row++) {
        let num_header_rows = 2;
        let remainder = row % (summary_table_item.rows - num_header_rows);
        let table_index = (row - remainder) / (summary_table_item.rows - num_header_rows) + 1;

        // no suffix if table_index == 0 otherwise it is added with underscore e.g. DETAILED_RESULTS_1;
        let page_suffix = table_index ? `_${table_index}` : "";

        let new_page_detailed_results = get_page_by_name(`DETAILED_RESULTS${page_suffix}`, false);
        if (!new_page_detailed_results) {
            new_page_detailed_results = page_detailed_results.Duplicate(templ.pages);
            /**set the new page name so that it can be found the next time */
            new_page_detailed_results.name = `DETAILED_RESULTS${page_suffix}`;
            let header = get_item_by_name("Detailed Results", new_page_detailed_results);
            header.text = table_index > 1 ? "%DETAILED_RESULTS% %CONTINUED%" : "%DETAILED_RESULTS%";
            summary_table_item = get_item_by_name("DETAILED_RESULTS_TABLE", new_page_detailed_results);
            /** clear the table since we are starting a new page and the page we duplicated will have a full table*/
            for (let r = num_header_rows; r < summary_table_item.rows; r++) {
                for (let c = 0; c < summary_table_item.columns; c++) {
                    set_cell_text(summary_table_item, r, c, "");
                }
            }
        }
        summary_table_item = get_item_by_name("DETAILED_RESULTS_TABLE", new_page_detailed_results);

        /** set the detailed summary page table text*/
        for (let col = 0; col < table_json.rows[row].length; col++) {
            let cell_json = table_json.rows[row][col];
            set_cell_text(summary_table_item, remainder + num_header_rows, col, cell_json.text);
            if (cell_json.conditions)
                set_cell_conditions(summary_table_item, remainder + num_header_rows, col, cell_json.conditions);
        }
    }
}

/**
 * Get the placeholder item from the page or throw an error if we cannot find it
 * @param {Object} item_json
 * @param {Page} page
 * @returns {Item}
 */
function get_placholder_item(item_json, page) {
    /** check if the placeholder_name propert exists */
    if (!item_json.hasOwnProperty("placeholder_name")) {
        throw Error(`Item "${item_json.name}" is missing the placeholder_name property`);
    }
    /** get_item_by_name will throw an error if the item does not exist
     * note that the name does not need to match exactly as when we duplicate a page the items get a suffix added (e.g. "_1")
     * so we still want to get the item */
    let item = get_item_by_name(item_json.placeholder_name, page);

    /** check the item type matches the expected type*/
    let expected_type = Item[item_json.type.toUpperCase()];
    if (item.type != expected_type) {
        function convert_item_type_number_to_string(item_type) {
            switch (item_type) {
                case Item.ARROW:
                    return "ARROW";
                case Item.AUTO_TABLE:
                    return "AUTO_TABLE";
                case Item.D3PLOT:
                    return "D3PLOT";
                case Item.ELLIPSE:
                    return "ELLIPSE";
                case Item.IMAGE:
                    return "IMAGE";
                case Item.IMAGE_FILE:
                    return "IMAGE_FILE";
                case Item.LIBRARY_IMAGE:
                    return "LIBRARY_IMAGE";
                case Item.LIBRARY_PROGRAM:
                    return "LIBRARY_PROGRAM";
                case Item.LINE:
                    return "LINE";
                case Item.NOTE:
                    return "NOTE";
                case Item.PLACEHOLDER:
                    return "PLACEHOLDER";
                case Item.PRIMER:
                    return "PRIMER";
                case Item.PROGRAM:
                    return "PROGRAM";
                case Item.RECTANGLE:
                    return "RECTANGLE";
                case Item.SCRIPT:
                    return "SCRIPT";
                case Item.SCRIPT_FILE:
                    return "SCRIPT_FILE";
                case Item.TABLE:
                    return "TABLE";
                case Item.TEXT:
                    return "TEXT";
                case Item.TEXTBOX:
                    return "TEXTBOX";
                case Item.TEXT_FILE:
                    return "TEXT_FILE";
                default:
                    return "UNKNOWN";
            }
        }

        /** image items may use a textbox as a placeholder = we knoe this if the placeholder name starts with "IMAGE".
         * similarly if the image path does not exist then it will be replaced with a texbox and descriptive message
         *  to indicate that the image is missing. Because of this we need to allow the types to be different
         */
        if (
            item.type == Item.TEXTBOX &&
            expected_type == Item.IMAGE_FILE &&
            item_json.placeholder_name.toUpperCase().startsWith("IMAGE")
        ) {
            return item;
        }

        throw Error(
            `Item "${item.name}" is type "${convert_item_type_number_to_string(
                item.type
            )}" but expected type is "${convert_item_type_number_to_string(expected_type)}".`
        );
    }

    return item;
}

/**
 * Get the placeholder item from the page or throw an error if we cannot find it
 * @param {Object} item_json
 * @param {Page} page
 * @returns {?Item}
 */
function create_item_from_json(item_json, page) {
    /** default item colours */

    let item = null;
    try {
        let type_const = Item[item_json.type.toUpperCase()];
        let x1 = item_json.x ? item_json.x : 0;
        let y1 = item_json.y ? item_json.y : 0;
        let x2 = item_json.x2 ? item_json.x2 : x1 + (item_json.width ? item_json.width : 0);
        let y2 = item_json.y2 ? item_json.y2 : y1 + (item_json.height ? item_json.height : 0);
        item = new Item(page, type_const, item_json.name, x1, x2, y1, y2);

        LogPrint(`Created item "${item_json.name}": x:${x1}, y:${y1}, x2:${x2}, height:${y2}`);

        switch (item.type) {
            case Item.TABLE:
                set_default_style_properties(item);
                item.lineColour = Colour.White();
                break;
            case Item.TEXTBOX:
                for (let prop in bold_props) {
                    item[prop] = bold_props[prop];
                }
                break;
        }

        // try and set other properties if they exist in the item_json

        let prop = "";
        let props_to_skip = ["x", "y", "x2", "y2", "width", "height", "name", "type", "placeholder_name"];
        for (prop in item_json) {
            try {
                if (item_json.hasOwnProperty(prop)) {
                    // skip properties that are not relevant to the item type
                    if (props_to_skip.includes(prop)) continue;

                    if (item_json[prop] === null || item_json[prop] === undefined) {
                        LogWarning(`Skipping setting ${item_json.name} "${prop}" property as it is null or undefined`);
                        continue;
                    }

                    // set the property on the item
                    LogPrint(`Setting "${prop}" property for item ${item_json.name}`);

                    if (/colour/i.test(prop)) {
                        let colour_json = item_json[prop];
                        if (/none/i.test(colour_json)) {
                            item[prop] = Colour.None();
                            continue;
                        }
                        LogPrint(
                            `Setting "${prop}" property for item ${item_json.name} to Colour.RGB(${colour_json.red}, ${colour_json.green}, ${colour_json.blue})`
                        );
                        item[prop] = Colour.RGB(colour_json.red, colour_json.green, colour_json.blue);
                    } else if (/^Reporter\.[A-Z]/.test(item_json[prop])) {
                        //extract all the constants after Reporter.
                        //e.g. "Reporter.JUSTIFY_LEFT | Reporter.JUSTIFY_MIDDLE" -> ["JUSTIFY_LEFT", "JUSTIFY_MIDDLE"]
                        item[prop] = eval(item_json[prop]);
                        // let constants = item_json[prop]
                        //     .replace(/^Reporter\./, "")
                        //     .split("|")
                        //     .map((constant) => constant.trim());
                        // // convert the constants to their corresponding values
                        // let constant_values = constants.map((constant) => {
                        //     // check if the constant exists in the Reporter object
                        //     if (Reporter.hasOwnProperty(constant)) {
                        //         return Reporter[constant];
                        //     } else {
                        //         throw Error(`Constant "${constant}" does not exist in Reporter object`);
                        //     }
                        // });
                        // // set the property on the item
                        // LogPrint(
                        //     `Setting "${prop}" property for item ${item_json.name} to ${constant_values.join(" | ")}`
                        // );
                        // item[prop] = constant_values.reduce((acc, val) => acc | val, 0); // combine the constants using bitwise OR
                    } else {
                        LogPrint(`Setting "${prop}" property for item ${item_json.name} to ${item_json[prop]}`);
                        item[prop] = item_json[prop];
                    }
                }
            } catch (error) {
                LogWarning(`Error setting "${prop}" property for item ${item_json.name}:\n${error.message}`);
            }
        }
    } catch (error) {
        LogError(`Error creating item ${item_json.name} : ${error.message}`);
    }

    return item;
}

/**
 *
 * @param {Item} item
 */
function set_default_style_properties(item) {
    if (!(item instanceof Item)) {
        throw Error(`cell_or_item is not an instance of Item`);
    }

    if (item.type !== Item.TABLE) {
        for (let row = 0; row < item.rows; row++) {
            for (let col = 0; col < item.columns; col++) {
                item.SetCellProperties(bold_props, row, col);
            }
        }
    }
}

/**
 * This function updates the cells in a REPORTER table from a JSON object
 * @param {Item} table
 * @param {Object} table_json
 */
function update_table_from_json(table, table_json) {
    /**
     * Add a column and set the cell properties to match the neighbour cell properties from the column to the left
     * @param {Item} table
     */
    function AddColumn(table) {
        // add a column ensuring that the maximum width is not exceeded
        let original_width = table.width;
        table.width = 0.5 * original_width;
        LogPrint(`Width: ${table.width}, original width: ${original_width}`);
        LogPrint(`Adding column to ${table.name}`);
        //@ts-ignore
        table.InsertColumn();
        table.width = original_width;

        try {
            for (let row = 0; row < table.rows; row++) {
                let neighbour_cell_props = table.GetCellProperties(row, table.columns - 2);
                table.SetCellProperties(neighbour_cell_props, row, table.columns - 1);
            }
        } catch (error) {
            LogWarning(error.message);
        }
    }

    /**
     * Add a row and set the cell properties to match the neighbour cell properties from the row above
     * @param {Item} table
     */
    function AddRow(table) {
        // add a row ensuring that the maximum height is not exceeded
        let original_height = table.height;
        table.height = 0.5 * original_height;
        LogPrint(`Height: ${table.height}, original height: ${original_height}`);
        LogPrint(`Adding row to ${table.name}`);
        //@ts-ignore
        table.InsertRow();
        table.height = original_height;

        try {
            for (let col = 0; col < table.columns; col++) {
                let neighbour_cell_props = table.GetCellProperties(table.rows - 2, col);
                table.SetCellProperties(neighbour_cell_props, table.rows - 1, col);
            }
        } catch (error) {
            LogWarning(error.message);
        }
    }

    // if (table_json.hasOwnProperty("header")) {
    //     let header_row = 0;

    //     /** set the table values */
    //     for (let cell_json of table_json.header) {
    //         /** default start column is 0, but this can be overriden by table_json.start_col */
    //         let col = isNaN(table_json.start_col) ? 0 : table_json.start_col;
    //         /** add a new row to the bottom of the table if we need more */
    //         if (table.rows <= header_row) AddRow(table);
    //         /** add a new column to the right of the table if we need more */
    //         if (table.columns <= col) AddColumn(table);
    //         set_cell_text(table, header_row, col, cell_json.text);
    //         table.SetCellProperties(header_props, header_row, col);
    //         if (cell_json.conditions) set_cell_conditions(table, header_row, col, cell_json.conditions);
    //         col++;
    //     }
    //     header_row++;
    // }

    /** default start row is 2 as the first 2 rows are usually headers, but this can be overriden by table_json.start_row */
    let row = isNaN(table_json.start_row) ? 2 : table_json.start_row;

    /** set the table values */
    for (let row_json of table_json.rows) {
        /** default start column is 0, but this can be overriden by table_json.start_col */
        let col = isNaN(table_json.start_col) ? 0 : table_json.start_col;
        /** add a new row to the bottom of the table if we need more */
        let new_cell = false;
        if (table.rows <= row) {
            AddRow(table);
            new_cell = true;
        }
        for (let cell_json of row_json) {
            /** add a new column to the right of the table if we need more */
            if (table.columns <= col) {
                AddColumn(table);
                new_cell = true;
            }
            set_cell_text(table, row, col, cell_json.text);
            /** only change the cell properties if it is a new cell */
            // if (new_cell) table.SetCellProperties(normal_props, row, col);
            if (cell_json.conditions) set_cell_conditions(table, row, col, cell_json.conditions);
            col++;
        }
        row++;
    }
}

/**
 * This function updates the text in a REPORTER text item (e.g. page header text) from a JSON object
 * @param {Item} text_item
 * @param {Object} text_item_json
 */
function update_text_from_json(text_item, text_item_json) {
    text_item.text = get_unicode_text_with_padding(text_item_json.text);
    if (text_item_json.hasOwnProperty("x")) text_item.x = text_item_json.x;
    if (text_item_json.hasOwnProperty("y")) text_item.y = text_item_json.y;
    if (text_item_json.hasOwnProperty("width")) text_item.width = text_item_json.width;
    if (text_item_json.hasOwnProperty("height")) text_item.height = text_item_json.height;
}

/**
 * This function updates the image in a REPORTER image item from a JSON object
 * @param {Item} placeholder_item
 * @param {Object} image_item_json
 */
function update_placeholder_image_from_json(placeholder_item, image_item_json, page) {
    // image_item.file = image_item_json.image;
    // image_item.x = image_item_json.x;
    // image_item.y = image_item_json.y;
    // image_item.width = image_item_json.width;
    // image_item.height = image_item_json.height;

    /** convert the description to the image name */
    let filepath = image_item_json.file;
    let name = image_item_json.name;
    let axis = image_item_json.axis;

    let img_file_txt = filepath;
    let img_file_path = img_file_txt;

    /** expand the variables in the image file path if it contains "%" characters */
    try {
        if (img_file_path && img_file_path.includes("%")) img_file_path = templ.ExpandVariablesInString(img_file_txt);
    } catch (error) {
        LogWarning(`Error expanding variables in image file path: ${error.message}`);
    }

    if (image_item_json.missing_text) {
        LogPrint(`Image file ${img_file_path} does not exist so it will not be generated`);
        /** if the file does not exist image_item_json.missing_text will be defined and will say
         * "Missing <channel_code> Corridor Plot data" or the Chinese equivalent if Language.language is set to Chinese */
        placeholder_item.text = get_unicode_text_with_padding(image_item_json.missing_text);
        return;
    }

    let image_item = new Item(
        page,
        Item.IMAGE_FILE,
        name,
        placeholder_item.x,
        placeholder_item.x2,
        placeholder_item.y,
        placeholder_item.y2
    );

    /** ensure that the border line colour is off for the images */
    image_item.lineColour = Colour.None();

    /** generate the image */
    image_item.file = img_file_txt;
    image_item.Generate();

    /** delete the placeholder item */
    let placeholder_name = placeholder_item.name;
    delete_item_by_name(placeholder_name, page);
}

/** create a REPORTER PAGE CLASS */

// class ReporterPage {
//     /** create an array to store all current pages in the report */
//     static pages = [];

//     constructor(name) {
//         this.template = Template.GetCurrent();
//         for (let page of ReporterPage.pages) {
//             if (page.name === name) {
//                 throw Error(`Page with name ${name} already exists`);
//             }
//         }
//     }

//     static MovePage() {
//         templ.Duplicate();
//     }

//     /**
//      * Get a page from the template which matches the name
//      * @param {String} name the exact name of the page
//      * @param {Boolean} [throw_error = true]
//      * @param {Template} templ
//      * @returns {?Page}
//      */
//     static get_page_by_name(name, throw_error = true, templ = Template.GetCurrent()) {
//         let pages = templ.pages;
//         for (let i = 0; i < pages; i++) {
//             let page = templ.GetPage(i);
//             if (page.name == name) return page;
//         }
//         if (throw_error) throw Error(`No page exists with a name that contains "${name}"`);

//         return null;
//     }

//     FromPageJSON;
// }

/** typedefs */

/**
 * @typedef {Object} REPORTER_SIMVT_JSON_DATA
 * @property {String} output_directory
 * @property {Object} variables
 * @property {String} ref_model_tag
 * @property {String} sim_model_tag
 * @property {String} ref_file_path
 * @property {String} sim_file_path
 * @property {Object} protocol_channel_tags keys are channels value is true/false (present/missing)
 * @property {Object} he_data head excursion data
 * @property {String} head_excursion_image
 * @property {String} settings_file
 * @property {Object[]} correlations
 * @property {String} results_csv
 *
 */

/** extra function dependencies */

/**
 * If relative, converts a path to an absolute path based on the current working directory.
 * @param {string} path A path to a directory or filename.
 * @param {string} label The name of the path (e.g. "KEYWORD_FILE") used for print statements.
 */
function convert_to_absolute_path(path, label) {
    if (!File.Exists(path)) {
        LogError(`In function convert_to_absolute_path: specified path for ${label} does not exist: ${path}`);
        return path;
    }
    if (File.IsAbsolute(path)) {
        /* If path is already absolute, just return it. */
        return path;
    } else {
        LogPrint(`Converting ${label} to absolute path...`);
        LogPrint(`Relative path: ${path}`);
        let current_dir = GetCurrentDirectory();
        let abs_path = `${current_dir}/${path}`;
        LogPrint(`Absolute path: ${abs_path}`);
        if (!File.Exists(abs_path)) {
            /* Trap the unexpected case where the conversion hasn't worked. */
            LogError(
                `In function convert_to_absolute_path: converted absolute path does not exist. Reverting to relative path.`
            );
            return path;
        }
        return abs_path;
    }
}

/**
 * Sets the REPORTER variables %OUTPUT_DIR%
 * The latter 2 are extracted from the settings file
 * specified keyword file.
 * @param {string} output_dir Absolute path and filename of settings file
 */
function set_output_file_variable(output_dir) {
    let templ = Template.GetCurrent();

    /* If output_dir is relative path, make it absolute path. We need to do this so that
     * any other variables that end up being derived from it (DEFAULT_DIR, RESULTS_DIR, OUTPUT_DIR,
     * etc.) are absolute paths too. When REPORTER generates an Oasys item, it sets -start_in to
     * the job file directory, so it gets confused if the relative paths we have defined are
     * relative to a different starting directory. Converting everything to absolute paths avoids
     * this problem. */
    output_dir = convert_to_absolute_path(output_dir, `%OUTPUT_DIR%`);

    let path_index = Math.max(output_dir.lastIndexOf("/"), output_dir.lastIndexOf("\\"));
    let filename = output_dir.substring(path_index + 1);
    let default_job = filename; //the name of the REPORT will be the same as the output folder by default

    /* Assign to REPORTER variables. Constructor will overwrite existing variable.
     * OUTPUT_DIR should be temporary; DEFAULT_DIR and DEFAULT_JOB are not temporary. */
    let output_dir_var = new Variable(templ, `OUTPUT_DIR`, `Output Directory`, output_dir, `Directory`, false, true);
    let default_dir_var = new Variable(
        templ,
        `DEFAULT_DIR`,
        `Reporter default directory`,
        output_dir,
        `Directory`,
        false,
        false
    );
    let default_job_var = new Variable(
        templ,
        `DEFAULT_JOB`,
        `Reporter default jobname`,
        default_job,
        `File(basename)`,
        false,
        false
    );
}

/**
 * @typedef {Object} REPORTER_JSON_DATA
 * @property {String} output_directory
 * @property {Object} variables
 * @property {String} ref_model_tag
 * @property {String} sim_model_tag
 * @property {String} ref_file_path
 * @property {String} sim_file_path
 * @property {Object} protocol_channel_tags keys are channels value is true/false (present/missing)
 * @property {Object} he_data head excursion data
 * @property {String} head_excursion_image
 * @property {String} settings_file
 * @property {Object[]} correlations
 * @property {String} results_csv
 *
 */
