import {
    USER_CRV_SETTINGS,
    USER_CRV_SETTINGS_1MODEL,
    USER_STOREY_DATUM_SETTINGS
} from "../../classes/plot_settings.js";
import Plot_Storey_Graph, {
    Import_Workflow_JSON_Data,
    Generate_Drift_Curves,
    Generate_Aggregate_Curves,
    Rearrange_Storey_Curves
} from "../../this/storey_plots/storey_curves_plotter.js";
import utils from "../../common/utils.js";
import { STOREY_DRIFT_VARS } from "../../classes/workflow_variables.js";
import Drift from "../../classes/drift.js";
import { Graph_Curve } from "../../classes/graph_setup.js";

/**
 * REPORTER T/HIS JavaScript item that generates storey drift plots and exports them to image
 * files.
 * @param {boolean} REPORTER Whether running from REPORTER
 * @param {string} default_dir REPORTER default directory (absolute path)
 * @param {string} output_dir Name of output directory subfolder
 * @param {string} images_dir Name of images directory subfolder
 * @param {string} workflow_dir Location of seismic scripts in workflows definition folder
 * @param {number} num_models Number of models to process
 */
export function Reporter_Generate_Storey_Drifts(
    REPORTER,
    default_dir,
    output_dir,
    images_dir,
    workflow_dir,
    num_models
) {
    // Read reporter variables file
    let fname_reporter_variables = `${default_dir}/${output_dir}/reporter_variables`;
    let variables = utils.read_reporter_variables_file(fname_reporter_variables);
    if (!variables) {
        // @ts-ignore
        ErrorMessage(`Could not find reporter_variables file: ${fname_reporter_variables}`);
        Exit();
    }

    // Get paths of all models loaded in Reporter apart from the first one
    // The first model should already be loaded in T/HIS (i.e. model_dir/default_job.thf)
    /** @type {String[]} */
    let added_model_paths = [];
    if (num_models > 1) {
        for (let i = 1; i < num_models; i++) {
            let mdl_var = variables.find((x) => x.name == `MODEL_PATH_${i + 1}`);
            added_model_paths.push(mdl_var.value);
        }
    }

    // Choose WORKFLOW, along with it's specific inputs...
    let workflow_vars = STOREY_DRIFT_VARS;

    // Import json data
    let data = Import_Workflow_JSON_Data(workflow_vars, added_model_paths, workflow_dir);

    // Generate storey drift curves, per model
    /** @type {Graph_Curve[]} */
    let graph_curves = [];
    // let drifts = data.base_entities.map(ent => { if (ent instanceof Drift) { return ent } }); // Array.map does not work on Reporter T/HIS Script Item. Use alternate script below.
    let drifts = Drift.map_entities_to_drifts(data.base_entities);
    for (let i = 0; i < data.t_his_models.length; i++) {
        const m = data.t_his_models[i];
        let graph_crvs = Generate_Drift_Curves(m, drifts, REPORTER);
        graph_curves.push(...graph_crvs);
    }

    // Write all T/HIS curve data back to JSON file
    Workflow.Refresh();
    /** @type {any} */
    let workflow_data = Workflow.ModelUserDataFromIndex(0, STOREY_DRIFT_VARS.workflow_name);
    workflow_data.curves = graph_curves;
    let extra = { model_unit_system: Workflow.ModelUnitSystemFromIndex(0, STOREY_DRIFT_VARS.workflow_name) };
    let workflow_definition_filename = Workflow.WorkflowDefinitionFilename(STOREY_DRIFT_VARS.workflow_name);
    Workflow.WriteToFile(workflow_data, data.json_file, workflow_definition_filename, extra);

    // Generate aggregate curves (e.g. Avg, Enve)
    Generate_Aggregate_Curves(workflow_vars, data.base_entities);

    // Rearrange curves (arrangement of curves affects order of labels in T/HIS graph)
    Rearrange_Storey_Curves(workflow_vars, data.base_entities);

    // Plot all graphs on JSON file, and export images
    let graph_sel = data.graphs[0];
    let curve_settings = data.t_his_models.length > 1 ? USER_CRV_SETTINGS : USER_CRV_SETTINGS_1MODEL;
    let storey_datum_settings = USER_STOREY_DATUM_SETTINGS;
    Plot_Storey_Graph(
        workflow_vars,
        data.graphs,
        data.base_entities,
        data.storeys,
        data.storey_datums,
        graph_sel,
        0,
        curve_settings,
        storey_datum_settings,
        data.limit_datums,
        data.limit_curves,
        REPORTER,
        default_dir,
        output_dir,
        images_dir
    );

    // Write data back to Reporter Variables file
    // Update NUM_GRAPHS variable
    Message(`Writing 'NUM_GRAPHS' Reporter variable to external file.`);
    for (let i = 0; i < variables.length; i++) {
        const v = variables[i];
        if (v.name == "NUM_GRAPHS") {
            v.value = data.graphs.length.toString();
            break;
        }
    }
    // Define new variables to hold title (entity) and subtitle (component) of each graph
    Message(`Writing graph titles/subtitles Reporter variables to external file.`);
    data.graphs.forEach((g) => {
        let title = {
            name: `G${String(g.id).padStart(3, "0")}_TITLE`,
            desc: "",
            value: g.entity_tag
        };
        let subtitle = {
            name: `G${String(g.id).padStart(3, "0")}_SUBTITLE`,
            desc: "",
            value: g.component_tag
        };
        variables.push(title);
        variables.push(subtitle);
    });
    utils.write_reporter_variables(variables, fname_reporter_variables);
}
