// DESCRIPTION: This REPORTER Javascript sets the Reporter template variables from the user-selected model.
//              Afterwards, it will initiate the template generation, thru calling the next script item,
//              'AutoTemplateRun'

// Main code is enclosed in IIFE so that variables declared on each script item will not be
// exposed to other script items in the Reporter template.
/* Utils.Version was added for REPORTER in Version 21. So based on if this function is present or not we can check which version we are running */
if (typeof Utils.Version != "function") {
    let message = `This workflow is only available for Oasys REPORTER version 21 and above.`;
    if (Batch()) {
        LogError(message);
    } else {
        Window.Message("Workflow not available", message);
    }
    Exit();
}
(() => {
    let templ = Template.GetCurrent();
    let page = templ.GetPage(0);

    // In case template was aborted when opened for the first time, reactivate
    //  the necessary script items to ensure it runs when template is generated manually by user
    // On successful run, 'ScriptOnLoad' will be reactivated by 'ReactivateScriptItems*
    activate_script_items(page, true, [
        "WriteVariablesFile",
        "GenerateStoreyPlots",
        "HighlightInModel",
        "ReadVariablesFile",
        "GenerateReport",
        "ISO_VIEW",
        "FRONT_VIEW",
        "RIGHT_SIDE_VIEW"
    ]);

    let default_dir;
    let default_job;
    let model_dir;
    let num_selected_models;

    /* If running in batch, expect user to provide DEFAULT_DIR and at least one model path
     * MODEL_PATH_1, with any number of MODEL_PATH_2, MODEL_PATH_3, etc.
     * If running interactively, prompt user for folder/file locations. */
    if (Batch()) {
        // Check required variables, keeping track of missing and invalid variables
        let missing_vars = [];
        let invalid_vars = [];

        // DEFAULT_DIR is used as parent directory
        default_dir = templ.GetVariableValue(`DEFAULT_DIR`);
        if (default_dir == null) {
            missing_vars.push(`DEFAULT_DIR`);
        } else if (!File.IsDirectory(default_dir)) {
            invalid_vars.push(`DEFAULT_DIR`);
        }
        /* If default_dir is relative path, make it absolute path. We need to do this so that any
         * other variables that end up being derived from it are absolute paths too. When REPORTER
         * generates an Oasys item, it sets -start_in to the job file directory, so it gets
         * confused if the relative paths we have defined are relative to a different starting
         * directory. Converting everything to absolute paths avoids this problem. */
        default_dir = convert_to_absolute_path(default_dir, `DEFAULT_DIR`);

        /* Search for all the MODEL_PATH_1, MODEL_PATH_2, etc. variables.
         * In batch mode, at least one should be defined. */
        num_selected_models = 0;
        let all_vars = Variable.GetAll(templ);
        for (let v of all_vars) {
            if (v.name.startsWith("MODEL_PATH_")) {
                let model_path = v.value;
                if (!File.Exists(model_path) || !File.IsFile(model_path)) {
                    invalid_vars.push(v.name);
                } else {
                    /* As with default_dir, we make sure model paths are absolute */
                    model_path = convert_to_absolute_path(model_path, v.name);
                    v.value = model_path; // Update the variable value
                    if (v.name == "MODEL_PATH_1") {
                        model_dir = model_path.match(/(.*)[\/\\]/)[1];
                        default_job = model_path.match(/([^(\/\\)]*$)/)[0].match(/(.*)[.]/)[1]; //model filename without extension
                    }
                }
                num_selected_models++;
            }
        }
        if (num_selected_models == 0) {
            missing_vars.push(`MODEL_PATH_1`);
        }

        if (missing_vars.length > 0 || invalid_vars.length > 0) {
            let msg = "";
            if (missing_vars.length > 0) {
                msg += `The following required variables are missing:\n\n${missing_vars.join(", ")}\n\n`;
            }
            if (invalid_vars.length > 0) {
                msg += `The following required variables are invalid:\n\n${invalid_vars.join(", ")}\n\n`;
            }
            msg += `Make sure that you supply valid variables during REPORTER job submission.`;
            LogError(msg);
            Exit();
        }
    } else {
        // Prompt user to select a parent directory, containing .key files
        let ans = Window.Message(
            "Select parent directory",
            "Please select the parent directory that holds all the desired LS-DYNA keyword files (which may be in subdirectories).",
            Window.OK | Window.CANCEL
        );

        // VALIDATING USER INPUTS...
        if (ans == Window.CANCEL) {
            template_aborted(templ, page);
            Window.Message("Select parent directory", "Operation cancelled. Report will not be generated.");
            Exit();
        }
        default_dir = Window.GetDirectory();
        if (default_dir == null) {
            template_aborted(templ, page);
            Window.Message("Select parent directory", "Parent directory not selected. Report will not be generated.");
            Exit();
        }

        // Get models from root directory and let users choose models to process
        let selected_models = user_select_file(default_dir, "*.key", "Model", true);
        if (selected_models == null) {
            template_aborted(templ, page);
            Exit();
        }

        // Create temporary variables for model full paths (if more than one exist)
        templ.DeleteTemporaryVariables(); // Remove existing
        if (selected_models.length > 1) {
            for (let i = 0; i < selected_models.length; i++) {
                const m = selected_models[i];
                LogPrint(`Creating MODEL_PATH_${i + 1} with value of: ${m.file}`);
                new Variable(templ, `MODEL_PATH_${i + 1}`, "", m.file, "String", false, true);
            }
        }
        num_selected_models = selected_models.length;

        // Set variables (set model_dir and default_job to the first model in sel_models)
        model_dir = selected_models[0].file.match(/(.*)[\/\\]/)[1];
        default_job = selected_models[0].file.match(/([^(\/\\)]*$)/)[0].match(/(.*)[.]/)[1]; //model filename without extension
    }

    // Update template variables
    let def_dir_var = Variable.GetFromName(templ, "DEFAULT_DIR");
    let mdl_dir_var = Variable.GetFromName(templ, "MODEL_DIR");
    let job_var = Variable.GetFromName(templ, "DEFAULT_JOB");
    let num_mdl_var = Variable.GetFromName(templ, "NUM_MODELS");

    LogPrint("Setting DEFAULT_DIR to: " + default_dir);
    LogPrint("Setting MODEL_DIR to: " + model_dir);
    LogPrint("Setting DEFAULT_JOB to: " + default_job);
    LogPrint("Setting NUM_MODELS to: " + num_selected_models);

    def_dir_var.value = default_dir;
    mdl_dir_var.value = model_dir;
    job_var.value = default_job;
    num_mdl_var.value = num_selected_models.toString();

    // Create output directory if it does not already exist
    let output_dir = templ.GetVariableValue(`OUTPUT_DIR`);
    let output_dir_path = `${default_dir}/${output_dir}`;
    if (!File.Exists(output_dir_path)) {
        LogPrint(`Creating output directory: ${output_dir_path}`);
        let success = File.Mkdir(output_dir_path);
        if (!success) {
            LogError(`Unable to create output directory: ${output_dir_path}`);
            Exit();
        }
    }

    // CONTINUE TEMPLATE GENERATION...
    //  Set this script inactive so when we generate the whole template, we don't generate this script again
    let this_script_item = Item.GetFromName(page, "ScriptOnLoad");
    this_script_item.active = false;

    // Generate the rest of the template
    let run_template = Item.GetFromName(page, "AutoRunTemplate");
    run_template.Generate();
})();

/**
 * Termination function when user aborts the template generation or invalid inputs have been provided.
 * This function ensures that all items in the template reverts back to their original state,
 * so the template can be safely run again.
 *
 * 'AutoRunTemplate' script item however, is disabled, as the template will be generated
 * manually at this point.
 * @param {Template} templ - Current template
 * @param {Page} page - Page where script_items are located.
 */
function template_aborted(templ, page) {
    activate_page_items(page, false); // deactivate all items in the FIRST page
    activate_script_items(page, true, ["ScriptOnLoad", "ReactivateScriptItems"]);

    // Deactivate all items on other pages
    for (let i = 1; i < templ.pages; i++) {
        activate_page_items(templ.GetPage(i), false);
    }
}

/**
 * Searches files from root and lets users choose which files to use via checkboxes
 * @param {String} root - Root directory to search files from
 * @param {String} search_pattern - File search pattern (e.g. *.key, *.json)
 * @param {String} search_entity - Entity being searched. This will be displayed in Message boxes.
 * It is recommended to input a string in 'ProperCase'.
 * @param {Boolean=} show_full_path - True if full file path is desired to be shown in UI. False if only filename is to be shown.
 * @param {Boolean=} single_select - Enforce only one being selected among the options
 */
function user_select_file(root, search_pattern, search_entity, show_full_path = false, single_select = false) {
    /** @type {{text:string,type:string,value:string,selected:boolean,file:string}[]} */
    let selected_files = [];
    let initial = search_entity[0].toUpperCase();

    let searched_files = File.FindFiles(root, search_pattern, true);

    // Remove files detected from shortcut links.
    root = root.replace(/\\/g, "/");
    searched_files = searched_files.map((f) => f.replace(/\\/g, "/")).filter((f) => f.startsWith(root));

    if (searched_files.length == 0) {
        Window.Warning(
            `Select ${search_entity.toLowerCase()}s`,
            `No '${search_pattern}' files found within the parent directory.`
        );
        selected_files = null;
    }
    // Automatically select the only file searched in root
    else if (searched_files.length == 1) {
        const file = searched_files[0];
        let filename = file.replace(/^.*[\\\/]/, "");
        selected_files.push({
            text: `${initial}1`,
            type: "checkbox",
            value: show_full_path ? file : filename,
            selected: true,
            file: file
        });
    }
    // Invoke UI to let user choose the file
    else {
        selected_files = show_selection_ui(searched_files, search_entity, show_full_path, single_select);
    }
    return selected_files;

    /**
     * Show UI to select options, and returns an array of selected items or null if none selected.
     * @param {String[]} files - Files to show in selection window.
     * @param {String} search_entity - Entity being searched. This will be displayed in Message boxes
     * @param {Boolean=} show_file - True if full file path is desired to be shown in UI. False if only filename is to be shown.
     * @param {Boolean=} single_select - Enforce only one being selected among the options
     */
    function show_selection_ui(files, search_entity, show_file = false, single_select = false) {
        /** @type {{text:string,type:string,value:string,selected:boolean,file:string}[]} */
        let selected_files = [];
        let entity_PCASE = `${search_entity}${single_select ? "" : "s"}`;
        let entity_LCASE = `${search_entity.toLowerCase()}${single_select ? "" : "s"}`;

        let options = [];
        for (let i = 0; i < files.length; i++) {
            const file = files[i];
            let filename = file.replace(/^.*[\\\/]/, "");
            options.push({
                text: `${initial}${i + 1}`,
                type: "checkbox",
                value: show_file ? file : filename,
                selected: single_select ? (i == 0 ? true : false) : true,
                file: file
            });
        }
        let proceed = Window.GetOptions(
            `Select ${entity_PCASE}`,
            `Select ${entity_LCASE} that will be used to generate report...`,
            options
        );
        if (!proceed) {
            Window.Message(`Select ${entity_PCASE}`, "Operation cancelled. Report will not be generated.");
            return null;
        }

        selected_files = options.filter((x) => x.selected);

        if (selected_files.length == 0) {
            Window.Message(`Select ${entity_PCASE}`, `No ${entity_LCASE} selected. Report will not be generated.`);
            return null;
        } else if (single_select && selected_files.length > 1) {
            Window.Message(`Select ${entity_PCASE}`, `Only one ${entity_LCASE} should be selected.`);
            return show_selection_ui(files, search_entity, show_file, single_select); // Show UI again
        }

        return selected_files;
    }
}

/**
 * Activate (or deactivate) all items in the current page of the given template.
 * @param {Page} page - Page to find script items
 * @param {Boolean=} activate - True to activate, false to deactivate
 * @param {String[]=} item_names - List of item names in the page to activate/deactivate.
 * If undefined, this function will take all the items in the page instead.
 */
function activate_page_items(page, activate = true, item_names = []) {
    let page_items =
        item_names.length == 0
            ? Item.GetAll(page)
            : item_names.map((x) => Item.GetFromName(page, x)).filter((y) => y !== null);

    // Reactivate script items
    page_items.forEach((item) => (item.active = activate));
}

/**
 * Activate (or deactivate) all script items in the current page of the given template.
 *
 * This only supports items of type: {@link Item.SCRIPT}, {@link Item.PRIMER}, {@link Item.D3PLOT},
 * {@link Item.THIS}, {@link Item.PROGRAM}, {@link Item.LIBRARY_PROGRAM}
 * @param {Page} page - Page to find script items
 * @param {Boolean=} activate - True to activate, false to deactivate
 * @param {String[]=} item_names - List of item names in the page to activate/deactivate.
 * If undefined, this function will take all the items in the page instead.
 */
function activate_script_items(page, activate = true, item_names = []) {
    // Get only script items
    let script_items = (
        item_names.length == 0
            ? Item.GetAll(page)
            : item_names.map((x) => Item.GetFromName(page, x)).filter((y) => y !== null)
    ).filter(
        (x) =>
            x.type == Item.SCRIPT ||
            x.type == Item.PRIMER ||
            x.type == Item.THIS ||
            x.type == Item.D3PLOT ||
            x.type == Item.PROGRAM ||
            x.type == Item.LIBRARY_PROGRAM
    );

    // Reactivate script items
    script_items.forEach((item) => (item.active = activate));
}

/**
 * Read a json file from Reporter
 * @param {string} filename - full filename
 * @returns
 */
function read_json(filename) {
    if (File.Exists(filename)) {
        let file = new File(filename, File.READ);
        let json = "";
        let line;
        // @ts-ignore
        while ((line = file.ReadLine()) != File.EOF) {
            json += line;
        }
        file.Close();

        let data = JSON.parse(json);
        return data;
    } else {
        LogError("File does not exist!");
        return null;
    }
}

/**
 * If relative, converts a path to an absolute path based on the current working directory.
 * @param {string} path A path to a directory or filename.
 * @param {string} label The name of the path (e.g. "KEYWORD_FILE") used for print statements.
 */
function convert_to_absolute_path(path, label) {
    if (!File.Exists(path)) {
        LogError(`In function convert_to_absolute_path: specified path for ${label} does not exist: ${path}`);
        return path;
    }
    if (File.IsAbsolute(path)) {
        /* If path is already absolute, just return it. */
        return path;
    } else {
        LogPrint(`Converting ${label} to absolute path...`);
        LogPrint(`Relative path: ${path}`);
        let current_dir = GetCurrentDirectory();
        let abs_path = `${current_dir}/${path}`;
        LogPrint(`Absolute path: ${abs_path}`);
        if (!File.Exists(abs_path)) {
            /* Trap the unexpected case where the conversion hasn't worked. */
            LogError(
                `In function convert_to_absolute_path: converted absolute path does not exist. Reverting to relative path.`
            );
            return path;
        }
        return abs_path;
    }
}
