// module: TRUE

// JavaScript: reporter_utils.js
// Name: Utility functions
// Author: Ian-Keith Relucio
// Date: 21 September 2021
// Job:
//
// DESCRIPTION: Reporter Generic helper functions

export { Utilities as default };

const Utilities = (() => {
    return {
        /**
         * Read a json file from Reporter
         * @param {string} filename - full filename
         * @returns
         */
        read_json: (filename) => {
            if (File.Exists(filename)) {
                let file = new File(filename, File.READ);
                let json = "";
                let line;
                // @ts-ignore
                while ((line = file.ReadLine()) != File.EOF) {
                    json += line;
                }
                file.Close();

                let data = JSON.parse(json);
                return data;
            } else {
                LogError("File does not exist!");
                return null;
            }
        },

        /**
         * Activate (or deactivate) all items in the current page of the given template.
         * @param {Page} page - Page to find script items
         * @param {Boolean=} activate - True to activate, false to deactivate
         * @param {String[]=} item_names - List of item names in the page to activate/deactivate.
         * If undefined, this function will take all the items in the page instead.
         */
        activate_page_items: (page, activate = true, item_names = []) => {
            let page_items =
                item_names.length == 0
                    ? Item.GetAll(page)
                    : item_names.map((x) => Item.GetFromName(page, x)).filter((y) => y !== null);

            // Reactivate script items
            page_items.forEach((item) => (item.active = activate));
        },

        /**
         * Activate (or deactivate) all script items in the current page of the given template.
         *
         * This only supports items of type: {@link Item.SCRIPT}, {@link Item.PRIMER}, {@link Item.D3PLOT},
         * {@link Item.THIS}, {@link Item.PROGRAM}, {@link Item.LIBRARY_PROGRAM}
         * @param {Page} page - Page to find script items
         * @param {Boolean=} activate - True to activate, false to deactivate
         * @param {String[]=} item_names - List of item names in the page to activate/deactivate.
         * If undefined, this function will take all the items in the page instead.
         */
        activate_script_items: (page, activate = true, item_names = []) => {
            // Get only script items
            let script_items = (
                item_names.length == 0
                    ? Item.GetAll(page)
                    : item_names.map((x) => Item.GetFromName(page, x)).filter((y) => y !== null)
            ).filter(
                (x) =>
                    x.type == Item.SCRIPT ||
                    x.type == Item.PRIMER ||
                    x.type == Item.THIS ||
                    x.type == Item.D3PLOT ||
                    x.type == Item.PROGRAM ||
                    x.type == Item.LIBRARY_PROGRAM
            );

            // Reactivate script items
            script_items.forEach((item) => (item.active = activate));
        }
    };
})();
