/* Contain the entire script within a function because REPORTER only has a single JavaScript realm
 * for the entire session. */
head_score();

/**
 * Performs the head score calculation for the specified models
 */
function head_score() {
    let templ = Template.GetCurrent();
    let models = get_model_list();
    let body_region_label = `head`;

    let occupants = ["DRIVER", "REAR_PASSENGER_SIDE"];

    /* Weighting factors */
    let driver_head_wt_factor = 0.923;
    let passenger_head_wt_factor = 0.8;

    new Variable(
        templ,
        "DRIVER_HEAD_WEIGHTING_FACTOR",
        "Driver head weighting factor",
        driver_head_wt_factor.toString(),
        "String",
        false,
        true
    );

    new Variable(
        templ,
        "REAR_PASSENGER_SIDE_HEAD_WEIGHTING_FACTOR",
        "Rear passenger side head weighting factor",
        passenger_head_wt_factor.toString(),
        "String",
        false,
        true
    );

    for (let m of models) {
        for (let occ of occupants) {
            LogPrint(`Calculating ${m} ${occ} ${body_region_label} score...`);

            /* Create a status object to track whether REPORTER Variables are all present and valid.
             * <success> is initially true but will be set to false if anything missing or invalid. */
            let status = { success: true, missing: [], invalid: [] };

            /* Head score and modifiers */

            /* Shared */
            let head_hic_score = 0;

            /*Driver */
            let steering_column_upper_displacement_score;

            /*Passenger*/
            let secondary_collision_modifier;

            if (occ == "DRIVER") {
                head_hic_score = get_variable_value(status, `${m}_${occ}_HEAD_HIC36_SCORE`, "float");

                steering_column_upper_displacement_score = get_variable_value(
                    status,
                    `${m}_STRUCTURE_STEERING_COLUMN_VERTICAL_INTRUSION_SCORE`,
                    "float",
                    true
                );
            } else if (occ == "REAR_PASSENGER_SIDE") {
                head_hic_score = get_variable_value(status, `${m}_${occ}_HEAD_HIC15_SCORE`, "float");
                secondary_collision_modifier = get_variable_value(
                    status,
                    `${m}_${occ}_SECONDARY_COLLISION_MODIFIER`,
                    "string",
                    true
                );
            } else {
                LogError(`Unexpected occupant type "${occ}" in ${body_region_label} calculation.`);
                Exit();
            }

            /* Final scores all start at zero and will remain so if any variables were missing or invalid */
            let head_hic_score_weighted = 0;
            let steering_column_upper_displacement_score_weighted = 0;

            let head_score = 0;

            /* If we have all the required variables, calculate the final scores */
            if (status.success) {
                /* "Head score" is :
                 * For the driver, ( head_hic_score - steering_column_upper_displacement_score ) * weighting factor
                 * For the rear passenger
                 *   if secondary collision = "No", then score is set to 4 * weighting factor
                 *   if secondary collision = "Yes", then score is head_hic_score * weighting factor
                 */

                if (occ == "DRIVER") {
                    head_hic_score_weighted = head_hic_score * driver_head_wt_factor;
                    steering_column_upper_displacement_score_weighted =
                        steering_column_upper_displacement_score * driver_head_wt_factor;

                    head_score = head_hic_score_weighted + steering_column_upper_displacement_score_weighted;
                } else {
                    head_hic_score_weighted = head_hic_score * passenger_head_wt_factor;

                    if (secondary_collision_modifier == "Yes") {
                        head_score = head_hic_score_weighted;
                    } else {
                        head_score = 4.0 * passenger_head_wt_factor;
                    }
                }

                /* Bound the score between upper and lower limits */
                head_score = Math.max(Math.min(head_score, 4.0), 0.0);
                LogPrint(`${m} ${occ} head score = ${head_score}`);
            } else {
                warn_about_missing_or_invalid_variables(status, `${m} ${occ} ${body_region_label} score calculation`);
            }

            /* Weighted scores */
            new Variable(
                templ,
                `${m}_${occ}_HEAD_HIC_SCORE_WEIGHTED`,
                "Head HIC score weighted",
                head_hic_score_weighted.toString(),
                "String",
                false,
                true
            );

            if (occ == "DRIVER") {
                new Variable(
                    templ,
                    `${m}_${occ}_STEERING_COLUMN_UPPER_DISPLACEMENT_SCORE_WEIGHTED`,
                    "Steering column upper displacement score weighted",
                    steering_column_upper_displacement_score_weighted.toString(),
                    "String",
                    false,
                    true
                );
            }

            /* Overall scores */
            new Variable(
                templ,
                `${m}_${occ}_HEAD_FINAL_SCORE`,
                "Final head score",
                head_score.toString(),
                "String",
                false,
                true
            );
        }
    }
}
