/* Contain the entire script within a function because REPORTER only has a single JavaScript realm
 * for the entire session. */
chest_score();

/**
 * Performs the chest score calculation for the specified models
 */
function chest_score() {
    let templ = Template.GetCurrent();
    let models = get_model_list();
    let body_region_label = `chest`;

    /* NOTE:
     * The variables, scoring, and weighting factors are
     * same for driver and front passenger; for JNCAP-MDB Protocol.
     */

    /* Weighting factors */
    let chest_weighting_factor = 1.0;

    new Variable(
        templ,
        "CHEST_WEIGHTING_FACTOR",
        "Chest weighting factor",
        chest_weighting_factor.toString(),
        "String",
        false,
        true
    );

    let occupants = ["DRIVER", "FRONT_PASSENGER"];

    for (let m of models) {
        for (let occ of occupants) {
            LogPrint(`Calculating ${m} ${occ} ${body_region_label} score...`);

            /* Create a status object to track whether REPORTER Variables are all present and valid.
             * <success> is initially true but will be set to false if anything missing or invalid. */
            let status = { success: true, missing: [], invalid: [] };

            /** Chest score and modifiers **/

            let comp_scores = get_multiple_variable_values(
                status,
                [
                    `${m}_${occ}_CHEST_COMPRESSION_MAX_01_SCORE`,
                    `${m}_${occ}_CHEST_COMPRESSION_MAX_02_SCORE`,
                    `${m}_${occ}_CHEST_COMPRESSION_MAX_03_SCORE`
                ],
                "float",
                true
            );

            let shoulder_lat_force_scores = get_multiple_variable_values(
                status,
                [
                    `${m}_${occ}_SHOULDER_LATERAL_FORCES_LEFT_MAX_SCORE`,
                    `${m}_${occ}_SHOULDER_LATERAL_FORCES_RIGHT_MAX_SCORE`
                ],
                "float",
                true
            );

            /* Final scores all start at zero and will remain so if any variables were missing or invalid */
            let chest_comp_01_score_weighted = 0;
            let chest_comp_02_score_weighted = 0;
            let chest_comp_03_score_weighted = 0;
            let chest_score = 0;
            let chest_score_bfr_modifier = 0;
            let shoulder_total_lat_force_score = 0;

            /* If we have all the required variables, calculate the final scores */
            if (status.success) {
                /* Weighted scores */
                chest_comp_01_score_weighted =
                    comp_scores[`${m}_${occ}_CHEST_COMPRESSION_MAX_01_SCORE`] * chest_weighting_factor;
                chest_comp_02_score_weighted =
                    comp_scores[`${m}_${occ}_CHEST_COMPRESSION_MAX_02_SCORE`] * chest_weighting_factor;
                chest_comp_03_score_weighted =
                    comp_scores[`${m}_${occ}_CHEST_COMPRESSION_MAX_03_SCORE`] * chest_weighting_factor;

                /* Chest Score calculation */
                chest_score_bfr_modifier = Math.min(
                    chest_comp_01_score_weighted,
                    chest_comp_02_score_weighted,
                    chest_comp_03_score_weighted
                );

                /* Shoulder modifier */
                shoulder_total_lat_force_score = Math.min(...Object.values(shoulder_lat_force_scores));

                /* Applying modifier score */
                chest_score = Math.max(0, chest_score_bfr_modifier + shoulder_total_lat_force_score);
            } else {
                warn_about_missing_or_invalid_variables(status, `${m} ${occ} ${body_region_label} score calculation`);
            }

            /*Sending values to reporter*/

            new Variable(
                templ,
                `${m}_${occ}_CHEST_COMPRESSION_MAX_01_SCORE_WEIGHTED`,
                "Chest score 01 weighted",
                chest_comp_01_score_weighted.toString(),
                "String",
                false,
                true
            );

            new Variable(
                templ,
                `${m}_${occ}_CHEST_COMPRESSION_MAX_02_SCORE_WEIGHTED`,
                "Chest score 02 weighted",
                chest_comp_02_score_weighted.toString(),
                "String",
                false,
                true
            );

            new Variable(
                templ,
                `${m}_${occ}_CHEST_COMPRESSION_MAX_03_SCORE_WEIGHTED`,
                "Chest score 03 weighted",
                chest_comp_03_score_weighted.toString(),
                "String",
                false,
                true
            );

            new Variable(
                templ,
                `${m}_${occ}_CHEST_TOTAL_COMPRESSION_SCORE`,
                "Chest score before modifier",
                chest_score_bfr_modifier.toString(),
                "String",
                false,
                true
            );
            new Variable(
                templ,
                `${m}_${occ}_SHOULDER_MODIFIER`,
                "Shoulder modifier",
                shoulder_total_lat_force_score.toString(),
                "String",
                false,
                true
            );
            new Variable(
                templ,
                `${m}_${occ}_CHEST_FINAL_SCORE`,
                "Final chest score",
                chest_score.toString(),
                "String",
                false,
                true
            );
        }
    }
}
