/* Contain the entire script within a function because REPORTER only has a single JavaScript realm
 * for the entire session. */
lower_leg_score();

/**
 * Performs the lower leg score calculation for the specified models
 */
function lower_leg_score() {
    let templ = Template.GetCurrent();
    let models = get_model_list();
    let body_region_label = `lower_leg`;

    let occupants = ["DRIVER", "FRONT_PASSENGER"];
    let sides = ["LEFT", "RIGHT"];

    /* Weighting factors */
    let driver_lower_leg_weighting_factor = 0.923;
    let passenger_lower_leg_weighting_factor = 0.4;

    new Variable(
        templ,
        "DRIVER_LOWER_LEG_WEIGHTING_FACTOR",
        "Driver lower leg weighting factor",
        driver_lower_leg_weighting_factor.toString(),
        "String",
        false,
        true
    );

    new Variable(
        templ,
        "FRONT_PASSENGER_LOWER_LEG_WEIGHTING_FACTOR",
        "Front passenger lower leg weighting factor",
        passenger_lower_leg_weighting_factor.toString(),
        "String",
        false,
        true
    );

    let driver_lower_leg_max_limit = 3.692;
    let passenger_lower_leg_max_limit = 1.6;

    new Variable(
        templ,
        "DRIVER_LOWER_LEG_MAX_POINTS",
        "Driver lower leg max points",
        driver_lower_leg_max_limit.toString(),
        "String",
        false,
        true
    );

    new Variable(
        templ,
        "FRONT_PASSENGER_LOWER_LEG_MAX_POINTS",
        "Front passenger lower leg max points",
        passenger_lower_leg_max_limit.toString(),
        "String",
        false,
        true
    );

    /* Shared */
    let femur_score = {};

    /*Driver*/
    let tibia_compression_score = {};
    let tibia_index_score = {};
    let brake_pedal_fore_aft_score, brake_pedal_vertical_score;
    let femur_score_wt;

    for (let m of models) {
        for (let occ of occupants) {
            /* Create a status object to track whether REPORTER Variables are all present and valid.
             * <success> is initially true but will be set to false if anything missing or invalid. */
            let status = { success: true, missing: [], invalid: [] };

            switch (occ) {
                case "DRIVER":
                    brake_pedal_fore_aft_score = get_variable_value(
                        status,
                        `${m}_STRUCTURE_BRAKE_PEDAL_FORE_AFT_INTRUSION_SCORE`,
                        "float"
                    );

                    brake_pedal_vertical_score = get_variable_value(
                        status,
                        `${m}_STRUCTURE_BRAKE_PEDAL_VERTICAL_INTRUSION_SCORE`,
                        "float"
                    );

                    let brake_pedal_fore_aft_wt = brake_pedal_fore_aft_score * driver_lower_leg_weighting_factor;
                    let brake_pedal_vertical_wt = brake_pedal_vertical_score * driver_lower_leg_weighting_factor;

                    new Variable(
                        templ,
                        `${m}_STRUCTURE_BRAKE_PEDAL_FORE_AFT_INTRUSION_WEIGHTED_SCORE`,
                        "Brake pedal fore aft weighted score",
                        brake_pedal_fore_aft_wt.toString(),
                        "String",
                        false,
                        true
                    );

                    new Variable(
                        templ,
                        `${m}_STRUCTURE_BRAKE_PEDAL_VERTICAL_INTRUSION_WEIGHTED_SCORE`,
                        "Brake pedal vertical weighted score",
                        brake_pedal_vertical_wt.toString(),
                        "String",
                        false,
                        true
                    );

                    break;

                case "FRONT_PASSENGER":
                    break;

                default:
                    LogError(`Unexpected occupant type "${occ}" in ${body_region_label} calculation.`);
                    Exit();
            }
            for (let side of sides) {
                LogPrint(`Calculating ${m} ${occ} ${side} ${body_region_label} score...`);

                /* Final scores all start at zero and will remain so if any variables were missing or invalid */

                femur_score[side] = 0;
                tibia_index_score[side] = 0;
                tibia_compression_score[side] = 0;

                switch (occ) {
                    /* Score calculation is different for different occupants */
                    case "DRIVER":
                        femur_score[side] = get_variable_value(
                            status,
                            `${m}_${occ}_${side}_FEMUR_COMPRESSION_SCORE`,
                            "float"
                        );

                        femur_score_wt = femur_score[side] * driver_lower_leg_weighting_factor;

                        new Variable(
                            templ,
                            `${m}_${occ}_${side}_FEMUR_COMPRESSION_WEIGHTED_SCORE`,
                            "Driver femur compression weighted score",
                            femur_score_wt.toString(),
                            "String",
                            false,
                            true
                        );

                        let tibia_index_upper_max_score = get_variable_value(
                            status,
                            `${m}_${occ}_${side}_TIBIA_INDEX_UPPER_MAX_SCORE`,
                            "float"
                        );
                        let tibia_index_lower_max_score = get_variable_value(
                            status,
                            `${m}_${occ}_${side}_TIBIA_INDEX_LOWER_MAX_SCORE`,
                            "float"
                        );

                        let tibia_index_upper_wt = tibia_index_upper_max_score * driver_lower_leg_weighting_factor;
                        let tibia_index_lower_wt = tibia_index_lower_max_score * driver_lower_leg_weighting_factor;

                        new Variable(
                            templ,
                            `${m}_${occ}_${side}_TIBIA_INDEX_UPPER_MAX_WEIGHTED_SCORE`,
                            "Driver tibia index weighted score",
                            tibia_index_upper_wt.toString(),
                            "String",
                            false,
                            true
                        );

                        new Variable(
                            templ,
                            `${m}_${occ}_${side}_TIBIA_INDEX_LOWER_MAX_WEIGHTED_SCORE`,
                            "Driver tibia index weighted score",
                            tibia_index_lower_wt.toString(),
                            "String",
                            false,
                            true
                        );

                        tibia_index_score[side] = Math.min(tibia_index_upper_max_score, tibia_index_lower_max_score);

                        let tibia_compression_lower_max_score = get_variable_value(
                            status,
                            `${m}_${occ}_${side}_TIBIA_COMPRESSION_LOWER_MAX_SCORE`,
                            "float"
                        );

                        let tibia_compression_upper_max_score = get_variable_value(
                            status,
                            `${m}_${occ}_${side}_TIBIA_COMPRESSION_UPPER_MAX_SCORE`,
                            "float"
                        );

                        tibia_compression_score[side] = Math.min(
                            tibia_compression_upper_max_score,
                            tibia_compression_lower_max_score
                        );

                        break;
                    case "FRONT_PASSENGER":
                        femur_score[side] = get_variable_value(
                            status,
                            `${m}_${occ}_${side}_FEMUR_COMPRESSION_PASSENGER_SCORE`,
                            "float"
                        );

                        femur_score_wt = femur_score[side] * passenger_lower_leg_weighting_factor;

                        new Variable(
                            templ,
                            `${m}_${occ}_${side}_FEMUR_COMPRESSION_PASSENGER_WEIGHTED_SCORE`,
                            "Front passenger femur compression weighted score",
                            femur_score_wt.toString(),
                            "String",
                            false,
                            true
                        );

                        break;

                    default:
                        LogError(`Unexpected occupant type "${occ}" in ${body_region_label} calculation.`);
                        Exit();
                }
            }

            let lower_leg_score = 0;

            /* If we have all the required variables, calculate the final scores */
            if (status.success) {
                /* Score calculation is different for different occupants */
                switch (occ) {
                    case "DRIVER":
                        let tibia_final_score = Math.min(tibia_index_score.LEFT, tibia_index_score.RIGHT);
                        let femur_final_score = Math.min(femur_score.LEFT, femur_score.RIGHT);
                        lower_leg_score =
                            (tibia_final_score +
                                femur_final_score +
                                (brake_pedal_fore_aft_score + brake_pedal_vertical_score)) *
                                driver_lower_leg_weighting_factor +
                            Math.min(tibia_compression_score.LEFT, tibia_compression_score.RIGHT);

                        /* Bound the score between upper and lower limits */
                        lower_leg_score = Math.max(Math.min(lower_leg_score, driver_lower_leg_max_limit), 0.0);

                        break;
                    case "FRONT_PASSENGER":
                        lower_leg_score =
                            Math.min(femur_score.LEFT, femur_score.RIGHT) * passenger_lower_leg_weighting_factor;

                        /* Bound the score between upper and lower limits */
                        lower_leg_score = Math.max(Math.min(lower_leg_score, passenger_lower_leg_max_limit), 0.0);

                        break;
                    default:
                        LogError(`Unexpected occupant type "${occ}" in ${body_region_label} calculation.`);
                        Exit();
                }
                LogPrint(`${m} ${occ} lower leg score = ${lower_leg_score}`);
            } else {
                warn_about_missing_or_invalid_variables(status, `${m} ${occ} ${body_region_label} score calculation`);
            }

            new Variable(
                templ,
                `${m}_${occ}_LOWER_LEG_FINAL_SCORE`,
                `Final ${body_region_label} score`,
                lower_leg_score.toString(),
                "String",
                false,
                true
            );
        }
    }
}
