/* Contain the entire script within a function because REPORTER only has a single JavaScript realm
 * for the entire session. */
draw_structural_graphs();

/**
 * Draw structural graphs for the specified occupants
 */
function draw_structural_graphs() {
    let templ = Template.GetCurrent();
    let models = get_model_list();
    let output_dir = get_expanded_variable_value(templ, `OUTPUT_DIR`);
    let occupants;
    /* Create a status object to track whether REPORTER Variables are all present and valid.
     * <success> is initially true but will be set to false if anything missing or invalid. */
    let status = { success: true, missing: [], invalid: [] };

    is_valid_output_dir(output_dir);

    let crash_test_type = get_variable_value(status, `TEMPLATE_CRASH_TEST`, "string");

    if (!status.success) {
        LogWarning(`Variable "TEMPLATE_CRASH_TEST" is missing for draw image `);
        return;
    }

    if (crash_test_type == "ODB" || crash_test_type == "ODB_STRUCTURE") {
        occupants = ["DRIVER"];
    } else if (crash_test_type == "SOB" || crash_test_type == "SOB_STRUCTURE") {
        occupants = ["DRIVER", "PASSENGER"];
    } else {
        LogWarning(`Variable "TEMPLATE_CRASH_TEST" value "${crash_test_type}" is invalid for draw image`);
        return;
    }

    for (let m of models) {
        for (let occ of occupants) {
            switch (crash_test_type) {
                case "ODB_STRUCTURE":
                case "ODB":
                    var brake_resultant = get_variable_value(
                        status,
                        `${m}_BRAKE_PEDAL_BRAKE_PEDAL_ALL_INTRUSION_RESULTANT_FINAL_VALUE`,
                        "float"
                    );
                    var clutch_resultant = get_variable_value(
                        status,
                        `${m}_CLUTCH_PEDAL_CLUTCH_PEDAL_ALL_INTRUSION_RESULTANT_FINAL_VALUE`,
                        "float",
                        false
                    );

                    // If the Brake resultant is Greater than Clutch for Manual transmission the text in the image will Brake Pedal
                    var pedal_text, pedal_value;
                    if (brake_resultant >= clutch_resultant || isNaN(clutch_resultant)) {
                        pedal_value = brake_resultant;
                        pedal_text = "Brake pedal";
                    } else {
                        pedal_value = clutch_resultant;
                        pedal_text = "Clutch pedal";
                    }

                    var x_axis_labels = [
                        "Footrest",
                        "Left Toepan",
                        "Centre toepan",
                        "Right toepan",
                        pedal_text,
                        "Left Inst. panel",
                        "Right Inst. panel",
                        "Door"
                    ];

                    // Values

                    var values = new Array(8);
                    values[0] = get_variable_value(
                        status,
                        `${m}_${occ}_FOOTREST_DRIVER_FOOTREST_ALL_INTRUSION_RESULTANT_FINAL_VALUE`,
                        "float"
                    );
                    values[1] = get_variable_value(
                        status,
                        `${m}_${occ}_LEFT_TOEPAN_DRIVER_LEFT_TOEPAN_ALL_INTRUSION_RESULTANT_FINAL_VALUE`,
                        "float"
                    );
                    values[2] = get_variable_value(
                        status,
                        `${m}_${occ}_CENTRE_TOEPAN_DRIVER_CENTRE_TOEPAN_ALL_INTRUSION_RESULTANT_FINAL_VALUE`,
                        "float"
                    );
                    values[3] = get_variable_value(
                        status,
                        `${m}_${occ}_RIGHT_TOEPAN_DRIVER_RIGHT_TOEPAN_ALL_INTRUSION_RESULTANT_FINAL_VALUE`,
                        "float"
                    );
                    values[4] = pedal_value;

                    values[5] = get_variable_value(
                        status,
                        `${m}_STRUCTURE_LEFT_INSTRUMENT_PANEL_FORE_AFT_INTRUSION_X_FINAL_VALUE`,
                        "float"
                    );
                    values[6] = get_variable_value(
                        status,
                        `${m}_STRUCTURE_RIGHT_INSTRUMENT_PANEL_FORE_AFT_INTRUSION_X_FINAL_VALUE`,
                        "float"
                    );
                    values[7] = get_variable_value(
                        status,
                        `${m}_STRUCTURE_DOOR_FORE_AFT_INTRUSION_X_FINAL_VALUE`,
                        "float"
                    );

                    // If negative, just set to zero

                    if (values[5] < 0.0) values[5] = 0.0;
                    if (values[6] < 0.0) values[6] = 0.0;
                    if (values[7] < 0.0) values[7] = 0.0;
                    var filename = "driver_structural_rating";

                    var pointsList = [
                        [0, 0, 7, 0, 7, 40, 0, 40], //good
                        [0, 15, 4, 15, 5, 5, 7, 5, 7, 40, 0, 40], //acceptable
                        [0, 22.5, 4, 22.5, 5, 10, 7, 10, 7, 40, 0, 40], //marginal
                        [0, 30, 4, 30, 5, 15, 7, 15, 7, 40, 0, 40] //poor
                    ];

                    draw_structural_graph(filename, pointsList, x_axis_labels, values);
                    break;

                case "SOB_STRUCTURE":
                case "SOB":
                    switch (occ) {
                        case "DRIVER":
                            var brake_resultant = get_variable_value(
                                status,
                                `${m}_BRAKE_PEDAL_BRAKE_PEDAL_ALL_INTRUSION_RESULTANT_FINAL_VALUE`,
                                "float"
                            );
                            var clutch_resultant = get_variable_value(
                                status,
                                `${m}_CLUTCH_PEDAL_CLUTCH_PEDAL_ALL_INTRUSION_RESULTANT_FINAL_VALUE`,
                                "float",
                                false
                            );

                            // If the Brake resultant is Greater than Clutch for Manual transmission the text in the image will Brake Pedal
                            var pedal_text, pedal_value;
                            if (brake_resultant >= clutch_resultant || isNaN(clutch_resultant)) {
                                pedal_value = brake_resultant;
                                pedal_text = "Brake pedal";
                            } else {
                                pedal_value = clutch_resultant;
                                pedal_text = "Clutch pedal";
                            }

                            var x_axis_labels = [
                                "Lower hinge pillar",
                                "Footrest",
                                "Left Toepan",
                                pedal_text,
                                "Parking brake pedal",
                                "Rocker panel (lat)",
                                "Steering column (long)",
                                "Upper hinge pillar",
                                "Upper dash",
                                "Instrument panel"
                            ];

                            var values = new Array(10);
                            values[0] = get_variable_value(
                                status,
                                `${m}_${occ}_LOWER_HINGE_MAX_RESULTANT_FINAL_VALUE`,
                                "float"
                            );
                            values[1] = get_variable_value(
                                status,
                                `${m}_${occ}_FOOTREST_DRIVER_FOOTREST_ALL_INTRUSION_RESULTANT_FINAL_VALUE`,
                                "float"
                            );
                            values[2] = get_variable_value(
                                status,
                                `${m}_${occ}_LEFT_TOEPAN_DRIVER_LEFT_TOEPAN_ALL_INTRUSION_RESULTANT_FINAL_VALUE`,
                                "float"
                            );
                            values[3] = pedal_value;
                            values[4] = get_variable_value(
                                status,
                                `${m}_PARKING_BRAKE_PARKING_BRAKE_ALL_INTRUSION_RESULTANT_FINAL_VALUE`,
                                "float"
                            );
                            values[5] = get_variable_value(
                                status,
                                `${m}_STRUCTURE_DRIVER_ROCKER_PANEL_AVERAGE_VALUE`,
                                "float"
                            );
                            values[6] = get_variable_value(
                                status,
                                `${m}_STRUCTURE_STEERING_COLUMN_FORE_AFT_INTRUSION_X_FINAL_VALUE`,
                                "float"
                            );
                            values[7] = get_variable_value(
                                status,
                                `${m}_${occ}_UPPER_HINGE_MAX_RESULTANT_FINAL_VALUE`,
                                "float"
                            );
                            values[8] = get_variable_value(
                                status,
                                `${m}_${occ}_UPPER_DASH_DRIVER_UPPER_DASH_ALL_INTRUSION_RESULTANT_FINAL_VALUE`,
                                "float"
                            );
                            values[9] = get_variable_value(
                                status,
                                `${m}_LEFT_INSTRUMENT_PANEL_LEFT_INSTRUMENT_PANEL_ALL_INTRUSION_RESULTANT_FINAL_VALUE`,
                                "float"
                            );

                            // If steering wheel is negative, just set to zero
                            if (values[6] < 0.0) values[6] = 0.0;
                            var filename = "driver_structural_rating";

                            var pointsList = [
                                [0, 0, 9, 0, 9, 40, 0, 40], //good
                                [0, 15, 4, 15, 5, 5, 6, 5, 7, 7.5, 9, 7.5, 9, 40, 0, 40], //acceptable
                                [0, 22.5, 4, 22.5, 5, 10, 6, 10, 7, 12.5, 9, 12.5, 9, 40, 0, 40], //marginal
                                [0, 30, 4, 30, 5, 15, 6, 15, 7, 17.5, 9, 17.5, 9, 40, 0, 40] //poor
                            ];

                            draw_structural_graph(filename, pointsList, x_axis_labels, values);
                            break;
                        case "PASSENGER":
                            var x_axis_labels = [
                                "Lower hinge pillar",
                                "Footrest",
                                "Right Toepan",
                                "Center Toepan",
                                "Rocker Panel (lat) ",
                                "Center Dash",
                                "Upper Hinge Pillar",
                                "Upper Dash",
                                "Right Lower Dash"
                            ];

                            var values = new Array(9);
                            values[0] = get_variable_value(
                                status,
                                `${m}_${occ}_LOWER_HINGE_MAX_RESULTANT_FINAL_VALUE`,
                                "float"
                            );
                            values[1] = get_variable_value(
                                status,
                                `${m}_${occ}_FOOTREST_PASSENGER_FOOTREST_ALL_INTRUSION_RESULTANT_FINAL_VALUE`,
                                "float"
                            );
                            values[2] = get_variable_value(
                                status,
                                `${m}_${occ}_RIGHT_TOEPAN_PASSENGER_RIGHT_TOEPAN_ALL_INTRUSION_RESULTANT_FINAL_VALUE`,
                                "float"
                            );
                            values[3] = get_variable_value(
                                status,
                                `${m}_${occ}_CENTRE_TOEPAN_PASSENGER_CENTRE_TOEPAN_ALL_INTRUSION_RESULTANT_FINAL_VALUE`,
                                "float"
                            );
                            values[4] = get_variable_value(
                                status,
                                `${m}_STRUCTURE_PASSENGER_ROCKER_PANEL_AVERAGE_VALUE`,
                                "float"
                            );
                            values[5] = get_variable_value(
                                status,
                                `${m}_${occ}_CENTRE_DASH_PASSENGER_CENTRE_DASH_ALL_INTRUSION_RESULTANT_FINAL_VALUE`,
                                "float"
                            );
                            values[6] = get_variable_value(
                                status,
                                `${m}_${occ}_UPPER_HINGE_MAX_RESULTANT_FINAL_VALUE`,
                                "float"
                            );
                            values[7] = get_variable_value(
                                status,
                                `${m}_${occ}_UPPER_DASH_PASSENGER_UPPER_DASH_ALL_INTRUSION_RESULTANT_FINAL_VALUE`,
                                "float"
                            );
                            values[8] = get_variable_value(
                                status,
                                `${m}_${occ}_RIGHT_LOWER_DASH_PASSENGER_RIGHT_LOWER_DASH_ALL_INTRUSION_RESULTANT_FINAL_VALUE`,
                                "float"
                            );

                            var filename = "passenger_structural_rating";

                            var pointsList = [
                                [0, 0, 8, 0, 8, 40, 0, 40], //good
                                [0, 15, 3, 15, 4, 5, 5, 7.5, 8, 7.5, 8, 40, 0, 40], //acceptable
                                [0, 22.5, 3, 22.5, 4, 10, 5, 12.5, 8, 12.5, 8, 40, 0, 40], //marginal
                                [0, 30, 3, 30, 4, 15, 5, 17.5, 8, 17.5, 8, 40, 0, 40] //poor
                            ];

                            draw_structural_graph(filename, pointsList, x_axis_labels, values);
                            break;
                    }
            }
        }
    }
}
/**
 * Draw the structural graphs and plots the relavnt points for the specfied occupant
 * @param {string} filename The name of the structural graphs' output file
 * @param {number[][]} pointsList An array is used define polygons that represent the different rating
 * @param {string[]} x_axis_labels An array of the x-axis labels for the various structural components
 * @param {Array} values An array of the strtructal values to be plotted on the graphs
 */
function draw_structural_graph(filename, pointsList, x_axis_labels, values) {
    let status = { success: true, missing: [], invalid: [] };
    let templ = Template.GetCurrent();
    let m = get_model_list();
    let output_dir = get_expanded_variable_value(templ, `OUTPUT_DIR`);
    let crash_test_type = get_variable_value(status, `TEMPLATE_CRASH_TEST`, "string");

    //IIHS colours
    var GREEN = "#52B442"; // Green
    var YELLOW = "#FCE702"; // Yellow
    var ORANGE = "#F79800"; // Orange
    var RED = "#F40000"; // Red

    // Get the default directory where we will write the images
    var dir = get_variable_value(status, "IMAGES_DIR");
    var scaling = 2; //use this to increase image resolution

    var width = 600 * scaling;
    var height = 450 * scaling;
    var padding = 5 * scaling;
    var x_border = 20 * scaling;
    var y_border = 30 * scaling;
    var tick_length = 3 * scaling;

    var y_axis_range = 400;
    var y_axis_tick_spacing = 50;

    var img = new Image(width, height);

    img.lineWidth = 1.5 * scaling;
    var font_height_in_pixels = 7 * scaling;
    img.fontSize = pixels_to_font_size(font_height_in_pixels);
    img.font = "Segoe UI";

    var x_start = padding + x_border;
    var x_end = width - x_start;
    var y_start = padding + y_border;
    var y_end = height - y_start; //331;

    // Y axis marks and values
    img.fontJustify = Reporter.JUSTIFY_LEFT;
    var x_mark = x_start - tick_length;
    var x_text = padding; //Y axis text is 1 unit away from tick end
    var y = y_start;
    var y_text_offset = Math.round(font_height_in_pixels / 2); //offset down half height of text so centred on tick

    var number_of_y_axis_gaps = y_axis_range / y_axis_tick_spacing;
    for (var i = 0; i <= number_of_y_axis_gaps; i++) {
        var text = y_axis_range - i * y_axis_tick_spacing;

        img.Text(x_text, y + y_text_offset, text.toString());
        img.Line(x_mark, y, x_start, y);

        y += (y_end - y_start) / number_of_y_axis_gaps;
    }

    // X axis marks and values

    var x_mark = x_start;
    var x_text = padding; //x_start; x_start is same as Y axis x_start for first label
    var y_mark = y_end;
    var y_text = y_mark + tick_length + Math.round(1.5 * font_height_in_pixels);

    var number_of_x_axis_gaps = x_axis_labels.length - 1;

    for (var i = 0; i <= number_of_x_axis_gaps; i++) {
        if (i == 0) img.fontJustify = Reporter.JUSTIFY_LEFT;
        else if (i == number_of_x_axis_gaps) img.fontJustify = Reporter.JUSTIFY_RIGHT;
        else img.fontJustify = Reporter.JUSTIFY_CENTRE;

        if (i != 0) x_text = x_mark;
        if (i == number_of_x_axis_gaps) x_text = x_end + x_border;

        img.Text(x_text, Math.round(y_text + 1.5 * font_height_in_pixels * ((i + 1) % 2)), x_axis_labels[i]); //test is alternatively offset by 10 units for every odd label
        img.Line(
            x_mark,
            y_mark,
            x_mark,
            Math.round(y_mark + tick_length + 0.6 * font_height_in_pixels * ((i + 1) % 2))
        ); //ticks are either 2 (even) or 7 (odd) units long

        x_mark += (x_end - x_start) / number_of_x_axis_gaps;
    }

    // GOOD Zone
    var coords_good = graphPointsToImageCoords(
        pointsList[0],
        x_start,
        x_end,
        number_of_x_axis_gaps,
        y_start,
        y_end,
        y_axis_range
    );
    img.fillColour = GREEN;
    img.Polygon(coords_good);

    // ACCEPTABLE Zone
    var coords_acceptable = graphPointsToImageCoords(
        pointsList[1],
        x_start,
        x_end,
        number_of_x_axis_gaps,
        y_start,
        y_end,
        y_axis_range
    );
    img.fillColour = YELLOW;
    img.Polygon(coords_acceptable);

    // MARGINAL Zone

    var coords_marginal = graphPointsToImageCoords(
        pointsList[2],
        x_start,
        x_end,
        number_of_x_axis_gaps,
        y_start,
        y_end,
        y_axis_range
    );
    img.fillColour = ORANGE;
    img.Polygon(coords_marginal);

    // POOR Zone
    var coords_poor = graphPointsToImageCoords(
        pointsList[3],
        x_start,
        x_end,
        number_of_x_axis_gaps,
        y_start,
        y_end,
        y_axis_range
    );
    img.fillColour = RED;
    img.Polygon(coords_poor);

    // Grid lines - don't draw over top of zone boundaries

    var x = x_start;

    img.lineColour = "#647382"; //'white';// 'black'//'f3f7fb';
    //img.lineWidth = 1.5;//2*scaling;
    img.lineStyle = Reporter.LINE_DASH;

    //draw vertical grid lines
    for (var i = 1; i < number_of_x_axis_gaps; i++) {
        x += (x_end - x_start) / number_of_x_axis_gaps;
        img.Line(x, y_start, x, y_end);
    }

    var y = y_start;

    //draw horizontal grid lines
    //
    for (var i = 1; i < number_of_y_axis_gaps; i++) {
        y += (y_end - y_start) / number_of_y_axis_gaps;

        img.Line(x_start, y, x_end, y);
    }

    // Outline
    //the lines have already been drawn in the polygons but we redraw with no fill (i.e. Polyline)
    //so that we cover gridlines where they overlap
    img.lineColour = "black";
    img.lineStyle = Reporter.LINE_SOLID;
    img.Polyline(coords_good);
    img.Polyline(coords_acceptable);
    img.Polyline(coords_marginal);
    img.Polyline(coords_poor);

    // Values

    img.lineColour = "blue";
    img.fillColour = "blue";
    img.lineWidth = 3 * scaling;
    img.lineStyle = Reporter.LINE_SOLID;

    var square_width = 4 * scaling;
    var shw = square_width / 2; //square half width
    var x = x_start;

    for (var i = 0; i < values.length; i++) {
        if (isNaN(values[i])) values[i] = 0.0;

        if (values[i] > y_axis_range) {
            // Off the top of the graph - draw an arrow
            y = y_start;
            img.Line(x, y, x, y + 15 * scaling);
            img.Line(x, y, x - 3 * scaling, y + 5 * scaling);
            img.Line(x, y, x + 3 * scaling, y + 5 * scaling);
        } // On the graph - draw a square
        else {
            y = y_end - (values[i] * (y_end - y_start)) / y_axis_range;
            //img.Polygon(x-shw,y-shw, x-shw,y+shw, x+shw,y+shw, x+shw,y-shw);
            img.Rectangle(x - shw, y - shw, x + shw, y + shw);
        }

        x += (x_end - x_start) / number_of_x_axis_gaps;
        LogPrint(" x = " + x + " i " + i);
    }

    // Write out image

    img.Save(`${output_dir}/${m}_${crash_test_type}_${filename}.png`, Image.PNG);
}

/**
 * Transforms graph points into image coordinates
 * @param {number[]} points An array containing x and y coordinates in graph coordinates. The array structure is x, y, x1, y1, x2, y2
 * @param {number} x_start The starting x-coordinate in the graph
 * @param {number} x_end The ending x-coordinate in the graph
 * @param {number} number_of_x_axis_gaps The number of intervals (gaps) on the x-axis
 * @param {number} y_start The starting y-coordinate in the graph
 * @param {number} y_end The ending y-coordinate in the graph
 * @param {number} y_axis_range The total range of y-values
 * @returns {number[]} An array of the converted image coordinates
 */
function graphPointsToImageCoords(points, x_start, x_end, number_of_x_axis_gaps, y_start, y_end, y_axis_range) {
    ///points is an array x, y, x1, y1, x2, y2 ... in graph coords (labels on x axis are at x = 0, 1, 2, 3 etc... )
    var coords = [];

    for (var i = 0; i < points.length; i++) {
        if (i % 2 == 0) {
            //then x coord
            //start from left side i.e. x_start
            var x_width = x_end - x_start; //x_end is > x_start as x = 0 at left of image
            var x_current = points[i];

            var x = x_start + (x_width * x_current) / number_of_x_axis_gaps;
            coords.push(x);
        } else {
            //y coord
            //start from the bottom i.e. y_end
            var y_height = y_end - y_start; //y_end is > y_start as y = 0 at top of image
            var y_current = points[i];

            var y = y_end - (y_height * y_current * 10) / y_axis_range;
            coords.push(y);
        }
    }

    return coords;
}
