/* Contain the entire script within a function because REPORTER only has a single JavaScript realm
 * for the entire session. */
head_and_neck_score();

/**
 * Performs the head and neck score calculation for the specified models
 */
function head_and_neck_score() {
    let templ = Template.GetCurrent();
    let models = get_model_list();
    let body_region_label = `head and neck`;
    let occupants;

    /* Create a status object to track whether REPORTER Variables are all present and valid.
     * <success> is initially true but will be set to false if anything missing or invalid. */
    let status = { success: true, missing: [], invalid: [] };

    let crash_test_type = get_variable_value(status, `TEMPLATE_CRASH_TEST`, "string");

    if (!status.success) {
        LogWarning(`Variable "TEMPLATE_CRASH_TEST" is missing for ${body_region_label} score calculation`);
        return;
    }

    if (crash_test_type == "ODB") {
        occupants = ["DRIVER"];
    } else if (crash_test_type == "SOB") {
        occupants = ["DRIVER", "FRONT_PASSENGER"];
    } else {
        LogWarning(
            `Variable "TEMPLATE_CRASH_TEST" value "${crash_test_type}" is invalid for ${body_region_label} score calculation`
        );
        return;
    }

    for (let m of models) {
        for (let occ of occupants) {
            LogPrint(`Calculating ${m} ${occ} ${body_region_label} score...`);
            /* Head score and modifiers */

            /* Shared */
            let head_hic_rating;
            let head_peak_accn_value;

            /* Neck score */
            let neck_axial_rating;
            let nij_rating;

            let ten_exc_rating;
            let com_exc_rating;
            let shr_exc_rating;

            // Calulating nij rating
            let Nij_value;
            let N_TE = get_variable_value(status, `${m}_${occ}_NECK_NIJ_TENSION-EXTENSION_VALUE`, "float");
            let N_TF = get_variable_value(status, `${m}_${occ}_NECK_NIJ_TENSION-FLEXION_VALUE`, "float");
            let N_CE = get_variable_value(status, `${m}_${occ}_NECK_NIJ_COMPRESSION-EXTENSION_VALUE`, "float");
            let N_CF = get_variable_value(status, `${m}_${occ}_NECK_NIJ_COMPRESSION-FLEXION_VALUE`, "float");

            Nij_value = Math.max(N_TE, N_TF, N_CE, N_CF);
            if (Nij_value <= 0.8) {
                nij_rating = "Good";
            } else if (Nij_value > 0.8 && Nij_value <= 1) {
                nij_rating = "Acceptable";
            } else if (Nij_value > 1 && Nij_value <= 1.2) {
                nij_rating = "Marginal";
            } else if (Nij_value > 1.2) {
                nij_rating = "Poor";
            }

            //
            head_hic_rating = get_variable_value(status, `${m}_${occ}_HEAD_HIC_RATING`, "string");
            head_peak_accn_value = get_variable_value(status, `${m}_${occ}_HEAD_PEAK_ACCELERATION_PEAK_VALUE`, "float");
            neck_axial_rating = get_variable_value(status, `${m}_${occ}_NECK_AXIAL_RATING`, "string");
            ten_exc_rating = get_variable_value(status, `${m}_${occ}_NECK_TENSION_EXCEEDENCE_RATING`, "string");
            com_exc_rating = get_variable_value(status, `${m}_${occ}_NECK_COMPRESSION_EXCEEDENCE_RATING`, "string");
            shr_exc_rating = get_variable_value(status, `${m}_${occ}_NECK_SHEAR_EXCEEDENCE_RATING`, "string");

            /* Final scores all start at zero and will remain so if any variables were missing or invalid */
            let head_rating = "Not Computed";
            let neck_rating = "Not Computed";
            let head_and_neck_rating = "Not Computed";
            let head_peak_rating = "";

            let head_and_neck_text = "The overall rating for the head and neck is the lower of the two ratings.";

            /* If we have all the required variables, calculate the final scores */
            if (status.success) {
                head_rating = head_hic_rating;

                /* If the peak acceleration is > 70g then the head rating is downgraded */

                if (head_peak_accn_value > 70.0) {
                    // Only applies if > 70g. If acceleration is above peak value, the hic rating gets downgraded
                    if (head_hic_rating == "Good") {
                        head_rating = "Acceptable";
                    } else if (head_hic_rating == "Acceptable") {
                        head_rating = "Marginal";
                    } else if (head_hic_rating == "Marginal") {
                        head_rating = "Poor";
                    }

                    head_peak_rating = "Exceeded Limit";

                    head_and_neck_text +=
                        "The head rating was downgraded one level because of a peak acceleration greater than 70g.";
                }

                let variable = new Variable(
                    templ,
                    `${m}_${occ}_HEAD_PEAK_ACCN_RATING`,
                    `Head peak acc Rating`,
                    head_peak_rating.toString(),
                    "string",
                    false,
                    true
                );

                new Variable(
                    templ,
                    `${m}_${occ}_HEAD_NECK_TEXT`,
                    `Head and Neck text`,
                    head_and_neck_text,
                    "string",
                    false,
                    true
                );

                if ((crash_test_type = "SOB")) {
                    neck_rating = return_worst_IIHS_rating([neck_axial_rating, nij_rating]);
                } else if ((crash_test_type = "ODB")) {
                    neck_axial_rating = neck_rating;
                }

                // not sure if the below is part of the 2021 regulation
                if (neck_rating == "Good") {
                    if (
                        ten_exc_rating == "Acceptable" ||
                        com_exc_rating == "Acceptable" ||
                        shr_exc_rating == "Acceptable"
                    ) {
                        neck_rating = "Acceptable";
                    }
                }
                LogPrint(`${m} ${occ} head score = ${head_rating}`);
                LogPrint(`${m} ${occ} neck score = ${neck_rating}`);

                /* Overall score is minimum of head and neck scores */

                // HEAD and NECK combined rating -> lowest of two
                head_and_neck_rating = return_worst_IIHS_rating([head_rating, neck_rating]);

                /* Subjective downgrade */

                if (crash_test_type == "ODB") {
                    let subjective_downgrade_from = head_and_neck_rating;

                    let subjective_downgrade_level = get_variable_value(status, `${m}_HEAD_AND_NECK_DOWNGRADE_DEMERIT`);
                    head_and_neck_rating = downgrade_IIHS_rating(head_and_neck_rating, subjective_downgrade_level);

                    let subjective_downgrade_to = head_and_neck_rating;

                    let text = "";

                    if (subjective_downgrade_level > 0) {
                        text += `The head and neck rating was downgraded because of qualitative observations from ${subjective_downgrade_from} to ${subjective_downgrade_to}.`;
                    }

                    new Variable(
                        templ,
                        "HEAD_AND_NECK_TEXT",
                        "Text to explain how head and neck rating was calculated",
                        text,
                        "General"
                    );
                }
            } else {
                warn_about_missing_or_invalid_variables(status, `${m} ${occ} ${body_region_label} score calculation`);
            }

            /* Overall scores */
            let head_score_var = new Variable(
                templ,
                `${m}_${occ}_HEAD_FINAL_RATING`,
                "Final head rating",
                head_rating.toString(),
                "String",
                false,
                true
            );
            let neck_score_var = new Variable(
                templ,
                `${m}_${occ}_NECK_FINAL_RATING`,
                "Final neck rating",
                neck_rating.toString(),
                "String",
                false,
                true
            );
            let head_neck_score_var = new Variable(
                templ,
                `${m}_${occ}_HEAD_NECK_FINAL_RATING`,
                `Final ${body_region_label} rating`,
                head_and_neck_rating.toString(),
                "String",
                false,
                true
            );

            Nij_value = Math.max(N_TE, N_TF, N_CE, N_CF);
            let Nij_value_var = new Variable(
                templ,
                `${m}_${occ}_NECK_NIJ_MAX_VALUE`,
                `Maximum of all the N_ij values`,
                Nij_value.toString(),
                "String",
                false,
                true
            );
            let Nij_rating_var = new Variable(
                templ,
                `${m}_${occ}_NECK_NIJ_RATING`,
                `Maximum of all the N_ij values`,
                nij_rating.toString(),
                "String",
                false,
                true
            );
        }
    }
}
