/* Contain the entire script within a function because REPORTER only has a single JavaScript realm
 * for the entire session. */
structural_rating();

/**
 * Performs the structural rating calculation for the specified models
 */
function structural_rating() {
    let templ = Template.GetCurrent();
    let models = get_model_list();

    let occupants = ["DRIVER"];
    for (let m of models) {
        for (let occ of occupants) {
            LogPrint(`Calculating ${m} ${occ} structural rating...`);
            /* Create a status object to track whether REPORTER Variables are all present and valid.
             * <success> is initially true but will be set to false if anything missing or invalid. */
            let status = { success: true, missing: [], invalid: [] };

            let brake_resultant = get_variable_value(
                status,
                `${m}_BRAKE_PEDAL_BRAKE_PEDAL_ALL_INTRUSION_RESULTANT_FINAL_VALUE`,
                "float"
            );

            /* The clutch is optional, so use a dummy <status> object so that if it is missing it doesn't
             * trip over the status.success check further down. */

            let clutch_resultant = get_variable_value(
                { success: true, missing: [], invalid: [] },
                `${m}_CLUTCH_PEDAL_CLUTCH_PEDAL_ALL_INTRUSION_RESULTANT_FINAL_VALUE`,
                "float",
                false
            );
            let structural_rating;

            var ratings = new Array(8);

            ratings[0] = get_variable_value(
                status,
                `${m}_${occ}_FOOTREST_${occ}_FOOTREST_ALL_INTRUSION_RESULTANT_FINAL_RATING`,
                "string"
            );
            ratings[1] = get_variable_value(
                status,
                `${m}_${occ}_LEFT_TOEPAN_${occ}_LEFT_TOEPAN_ALL_INTRUSION_RESULTANT_FINAL_RATING`,
                "string"
            );
            ratings[2] = get_variable_value(
                status,
                `${m}_${occ}_CENTRE_TOEPAN_${occ}_CENTRE_TOEPAN_ALL_INTRUSION_RESULTANT_FINAL_RATING`,
                "string"
            );
            ratings[3] = get_variable_value(
                status,
                `${m}_${occ}_RIGHT_TOEPAN_${occ}_RIGHT_TOEPAN_ALL_INTRUSION_RESULTANT_FINAL_RATING`,
                "string"
            );
            if (brake_resultant >= clutch_resultant || isNaN(clutch_resultant)) {
                ratings[4] = get_variable_value(
                    status,
                    `${m}_BRAKE_PEDAL_BRAKE_PEDAL_ALL_INTRUSION_RESULTANT_FINAL_RATING`,
                    "string"
                );

                /* The clutch is optional (i.e. for an automatic car) and if the clutch structure isn't specified
                 * the clutch resultant will be null. In addition the X, Y, Z and RESULTANT displacement variables
                 * and the rating variable that would normally come from T/HIS won't exist, so create them and set
                 * their values to a blank string so we don't get any error messages about missing variables when
                 * the table with the values is generated. */
                if (!clutch_resultant) {
                    for (let dir of ["X", "Y", "Z", "RESULTANT"]) {
                        new Variable(
                            templ,
                            `${m}_CLUTCH_PEDAL_CLUTCH_PEDAL_ALL_INTRUSION_${dir}_FINAL_VALUE`,
                            "Blank string for cars with no clutch",
                            "",
                            "String",
                            false,
                            true
                        );
                    }

                    new Variable(
                        templ,
                        `${m}_CLUTCH_PEDAL_CLUTCH_PEDAL_ALL_INTRUSION_RESULTANT_FINAL_RATING`,
                        "Blank string for cars with no clutch",
                        "",
                        "String",
                        false,
                        true
                    );
                }
            } else if (clutch_resultant >= brake_resultant) {
                ratings[4] = get_variable_value(
                    status,
                    `${m}_CLUTCH_PEDAL_CLUTCH_PEDAL_ALL_INTRUSION_RESULTANT_FINAL_RATING`,
                    "string",
                    false
                );
            }

            ratings[5] = get_variable_value(
                status,
                `${m}_STRUCTURE_LEFT_INSTRUMENT_PANEL_FORE_AFT_INTRUSION_RATING`,
                "string"
            );
            ratings[6] = get_variable_value(
                status,
                `${m}_STRUCTURE_RIGHT_INSTRUMENT_PANEL_FORE_AFT_INTRUSION_RATING`,
                "string"
            );
            ratings[7] = get_variable_value(status, `${m}_STRUCTURE_DOOR_FORE_AFT_INTRUSION_RATING`, "string");

            let ratings_count = { poor: 0, marginal: 0, acceptable: 0, good: 0 };

            if (status.success) {
                /* Count number of Poor, Marginal, Acceptable, Good and get overall rating */
                ratings_count = count_IIHS_ratings(ratings);

                structural_rating = get_overall_IIHS_rating_from_count(ratings_count);
            } else {
                warn_about_missing_or_invalid_variables(status, `${m} ${occ} calculation`);
            }

            // Rating cannot be more than one rating level better than the worst rating.

            var downgrade_from = structural_rating;

            let clamp = clamp_IIHS_strucural_rating(structural_rating, ratings_count);

            var downgraded = clamp.downgraded;
            var worst_rating = clamp.worst_rating;
            var downgrade_to = clamp.downgrade_to;

            structural_rating = downgrade_to;

            // Also check for subjective downgrade

            var subjective_downgrade = false;

            var subjective_downgrade_level = get_variable_value(status, `${m}_STRUCTURAL_DOWNGRADE_DEMERIT`);

            var subjective_downgrade_from = structural_rating;

            if (subjective_downgrade_level > 0) {
                structural_rating = downgrade_IIHS_rating(structural_rating, subjective_downgrade_level);

                subjective_downgrade = true;
            }

            var subjective_downgrade_to = structural_rating;

            // Pass rating back to Reporter

            let overall_structural_rating_var = new Variable(
                templ,
                `${m}_OVERALL_STRUCTURAL_RATING`,
                "The overall structural rating based on the Driver and Passenger",
                structural_rating,
                "String",
                false,
                true
            );

            // Set text to explain how the rating was calculated - explain about any downgrades

            var text = "";

            if (!downgraded) text += "";
            if (!subjective_downgrade) text += "";

            text +=
                "The overall rating for the Structure is taken as the rating with the most measures. Where there is a tie, the lower rating is used.";

            if (downgraded)
                text +=
                    "The rating cannot be more than one rating level better than the worst measurement (" +
                    worst_rating +
                    "), so the overall rating was downgraded from " +
                    downgrade_from +
                    " to " +
                    downgrade_to +
                    ".";
            if (subjective_downgrade)
                text +=
                    "The rating was downgraded because of qualitiative observations from " +
                    subjective_downgrade_from +
                    " to " +
                    subjective_downgrade_to +
                    ".";
            var variable = new Variable(
                templ,
                "STRUCTURAL_TEXT",
                "Text to explain how structural rating was calculated",
                text,
                "General"
            );
        }
    }
}
