/* Contain the entire script within a function because REPORTER only has a single JavaScript realm
 * for the entire session. */
head_protection_rating();

/**
 * Performs the head protection rating calculation for the specified models
 */
function head_protection_rating() {
    let templ = Template.GetCurrent();
    let models = get_model_list();

    let occupants = ["DRIVER", "REAR_DRIVER_SIDE"];
    for (let m of models) {
        for (let occ of occupants) {
            LogPrint(`Calculating ${m} ${occ} head protection rating...`);
            /* Create a status object to track whether REPORTER Variables are all present and valid.
             * <success> is initially true but will be set to false if anything missing or invalid. */
            let status = { success: true, missing: [], invalid: [] };

            let head_peak_acceleration;
            let head_protection_rating;

            head_peak_acceleration = get_variable_value(
                status,
                `${m}_${occ}_HEAD_PEAK_ACCELERATION_PEAK_VALUE`,
                "float"
            );

            let contact_is_hard = false; //contact with any non FMVSS 201-compliant surface or structure?
            let head_leaves_vehicle_or_contacts_barrier = false; //this is immediate "Poor"

            var head_protection_system_is_adequate = WindowAnswerToBool(
                "Head Protection system",
                "Is there an adequate head protection system for the " +
                    `${occ}` +
                    " head?\n(e.g. air bag that deploys properly)"
            );
            var contact_between_head_and_vehicle = WindowAnswerToBool(
                "Contact",
                "Is there contact between the " + `${occ}` + " head and vehicle?"
            );

            if (contact_between_head_and_vehicle) {
                head_leaves_vehicle_or_contacts_barrier = WindowAnswerToBool(
                    "Head leave vehicle",
                    "Does the " + `${occ}` + " head leave the vehicle or contact the MDB in any way?"
                );
                if (!head_leaves_vehicle_or_contacts_barrier) {
                    contact_is_hard = WindowAnswerToBool(
                        "Hard Contact",
                        "Does the " + `${occ}` + " head contact with any non FMVSS 201-compliant surface or structure?"
                    );
                }
            }

            if (head_leaves_vehicle_or_contacts_barrier) {
                head_protection_rating = "Poor";
            } else if ((head_peak_acceleration > 70 && contact_between_head_and_vehicle) || contact_is_hard) {
                head_protection_rating = "Marginal";
            } else if (head_protection_system_is_adequate) {
                head_protection_rating = "Good";
            } else {
                /*i.e. acceleration is <= 70G, but head protection (air bags) do not provide not
             adequate and there are no hard contacts (with any non FMVSS 201-compliant surface or structure)*/
                head_protection_rating = "Acceptable";
            }

            if (occ == "DRIVER") {
                let head_protection_rating_final = new Variable(
                    templ,
                    `${m}_${occ}_HEAD_PROTECTION_RATING`,
                    `${occ} head protection rating`,
                    head_protection_rating,
                    "String",
                    false,
                    true
                );
            } else if (occ == "REAR_DRIVER_SIDE") {
                let head_protection_rating_final = new Variable(
                    templ,
                    `${m}_${occ}_HEAD_PROTECTION_RATING`,
                    `${occ} head protection rating`,
                    head_protection_rating,
                    "String",
                    false,
                    true
                );
            }

            recalculate_results_IIHS();
        }
    }
}

/**
 * This function is used to create small windows with questions in it. There are two buttons YES or NO. The user can click on either button based on the question.
 * @param {string} heading Heading or title for the window box
 * @param {string} question A question to the user
 * @returns {boolean} returns true or false based on the users input
 */
function WindowAnswerToBool(heading, question) {
    //window returns Window.YES|Window.No but it is more convenient to have true|false
    var answer = Window.Message(heading, question, Window.YES | Window.NO);
    if (answer == Window.YES) return true;
    return false;
}
