/* Contain the entire script within a function because REPORTER only has a single JavaScript realm
 * for the entire session. */
lower_leg_foot_and_ankle_score();

/**
 * Performs the lower leg, foot and ankle score calculation for the specified models
 */
function lower_leg_foot_and_ankle_score() {
    let templ = Template.GetCurrent();
    let models = get_model_list();
    let body_region_label = `lower leg, foot and ankle`;

    let occupants = ["DRIVER", "FRONT_PASSENGER"];
    let sides = ["LEFT", "RIGHT"];
    for (let m of models) {
        let lower_leg = {}; // To be populated with score for each side
        let tibia_compression_max_val = {}; // To be populated with max value for each side
        let tibia_compression_score = {}; // To be populated with score for each side

        for (let occ of occupants) {
            /* First, extract the pedal scores. Afterwards, we will extract the tibia scores for
             * both left and right sides. */

            /* Create a status object to track whether REPORTER Variables are all present and valid.
             * <success> is initially true but will be set to false if anything missing or invalid. */
            let pedal_status = { success: true, missing: [], invalid: [] };

            /* Driver */
            let pedal_blocking_mod;
            let footwell_rupture_mod;
            let accel_pedal_vertical_score;
            let accel_pedal_vertical_val;
            let accel_pedal_fore_aft_score;
            let accel_pedal_fore_aft_val;
            let brake_pedal_vertical_score;
            let brake_pedal_vertical_val;
            let brake_pedal_fore_aft_score;
            let brake_pedal_fore_aft_val;
            /* Clutch pedal is optional so initialise with maximum modifier of 0.0 */
            let clutch_pedal_vertical_score = 0.0;
            let clutch_pedal_vertical_val = 0.0;
            let clutch_pedal_fore_aft_score = 0.0;
            let clutch_pedal_fore_aft_val = 0.0;

            /* Foot and ankle scores all start at zero and will remain so if any variables were missing or invalid */
            let max_pedal_vertical_intrusion = "[Not computed]";
            let max_pedal_fore_aft_intrusion = "[Not computed]";
            let overall_pedal_vertical_score = 0.0;
            let overall_pedal_fore_aft_score = 0.0;
            let overall_pedal_blocking_penalty = 0.0;
            let foot_and_ankle_score = 0.0;

            switch (occ) {
                case "DRIVER":
                    LogPrint(`Processing pedal intrusions for ${body_region_label} score...`);

                    pedal_blocking_mod = get_variable_value(pedal_status, `${m}_${occ}_PEDAL_BLOCKING_MODIFIER`);
                    footwell_rupture_mod = get_variable_value(
                        pedal_status,
                        `${m}_${occ}_FOOTWELL_RUPTURE_MODIFIER`,
                        "float"
                    );
                    accel_pedal_vertical_score = get_variable_value(
                        pedal_status,
                        `${m}_STRUCTURE_ACCELERATOR_PEDAL_VERTICAL_INTRUSION_SCORE`,
                        "float"
                    );
                    accel_pedal_vertical_val = get_variable_value(
                        pedal_status,
                        `${m}_STRUCTURE_ACCELERATOR_PEDAL_VERTICAL_INTRUSION_Z_FINAL_VALUE`,
                        "float"
                    );
                    accel_pedal_fore_aft_score = get_variable_value(
                        pedal_status,
                        `${m}_STRUCTURE_ACCELERATOR_PEDAL_FORE_AFT_INTRUSION_SCORE`,
                        "float"
                    );
                    accel_pedal_fore_aft_val = get_variable_value(
                        pedal_status,
                        `${m}_STRUCTURE_ACCELERATOR_PEDAL_FORE_AFT_INTRUSION_X_FINAL_VALUE`,
                        "float"
                    );
                    brake_pedal_vertical_score = get_variable_value(
                        pedal_status,
                        `${m}_STRUCTURE_BRAKE_PEDAL_VERTICAL_INTRUSION_SCORE`,
                        "float"
                    );
                    brake_pedal_vertical_val = get_variable_value(
                        pedal_status,
                        `${m}_STRUCTURE_BRAKE_PEDAL_VERTICAL_INTRUSION_Z_FINAL_VALUE`,
                        "float"
                    );
                    brake_pedal_fore_aft_score = get_variable_value(
                        pedal_status,
                        `${m}_STRUCTURE_BRAKE_PEDAL_FORE_AFT_INTRUSION_SCORE`,
                        "float"
                    );
                    brake_pedal_fore_aft_val = get_variable_value(
                        pedal_status,
                        `${m}_STRUCTURE_BRAKE_PEDAL_FORE_AFT_INTRUSION_X_FINAL_VALUE`,
                        "float"
                    );
                    /* Clutch pedal is optional */
                    clutch_pedal_vertical_score = get_variable_value(
                        pedal_status,
                        `${m}_STRUCTURE_CLUTCH_PEDAL_VERTICAL_INTRUSION_SCORE`,
                        "float",
                        false
                    );
                    clutch_pedal_vertical_val = get_variable_value(
                        pedal_status,
                        `${m}_STRUCTURE_CLUTCH_PEDAL_VERTICAL_INTRUSION_Z_FINAL_VALUE`,
                        "float",
                        false
                    );
                    clutch_pedal_fore_aft_score = get_variable_value(
                        pedal_status,
                        `${m}_STRUCTURE_CLUTCH_PEDAL_FORE_AFT_INTRUSION_SCORE`,
                        "float",
                        false
                    );
                    clutch_pedal_fore_aft_val = get_variable_value(
                        pedal_status,
                        `${m}_STRUCTURE_CLUTCH_PEDAL_FORE_AFT_INTRUSION_X_FINAL_VALUE`,
                        "float",
                        false
                    );
                    if (pedal_status.success) {
                        /* We report the maximum vertical intrusion */
                        let max_pedal_vertical_intrusion_val = Math.max(
                            accel_pedal_vertical_val,
                            brake_pedal_vertical_val
                        );
                        /* if the clutch is present, then include it in the pedal intrusion calculation */
                        if (clutch_pedal_vertical_val != null) {
                            max_pedal_vertical_intrusion_val = Math.max(
                                max_pedal_vertical_intrusion_val,
                                clutch_pedal_vertical_val
                            );
                        }
                        max_pedal_vertical_intrusion = max_pedal_vertical_intrusion_val.toString();

                        /* Vertical pedal score is worst of individual pedals */
                        overall_pedal_vertical_score = Math.min(
                            accel_pedal_vertical_score,
                            brake_pedal_vertical_score,
                            clutch_pedal_vertical_score
                        );
                        /* Fore-aft pedal score is worst of individual pedals */
                        overall_pedal_fore_aft_score = Math.min(
                            accel_pedal_fore_aft_score,
                            brake_pedal_fore_aft_score,
                            clutch_pedal_fore_aft_score
                        );
                        /* Consider pedal blocking modifier */
                        let max_pedal_fore_aft_intrusion_val = Math.max(
                            accel_pedal_fore_aft_val,
                            brake_pedal_fore_aft_val,
                            clutch_pedal_fore_aft_val
                        );
                        /* Pedal blocking penalty according to GNCAP AOP v7.1.2 */
                        max_pedal_fore_aft_intrusion = max_pedal_fore_aft_intrusion_val.toString();
                        if (pedal_blocking_mod == "Yes") {
                            overall_pedal_blocking_penalty = pedal_blocking_sliding_scale(
                                max_pedal_fore_aft_intrusion_val,
                                50,
                                175,
                                0,
                                -1
                            );
                        }
                        /* Foot and ankle score is sum of fore-aft pedal score with footwell
                         * rupture and pedal blocking modifiers */
                        foot_and_ankle_score =
                            overall_pedal_fore_aft_score + footwell_rupture_mod + overall_pedal_blocking_penalty;
                    } else {
                        warn_about_missing_or_invalid_variables(
                            pedal_status,
                            `${m} ${occ} ${body_region_label} score calculation`
                        );
                    }

                    new Variable(
                        templ,
                        `${m}_${occ}_MAX_PEDAL_VERTICAL_INTRUSION`,
                        `Maximum pedal vertical intrusion`,
                        max_pedal_vertical_intrusion,
                        "String",
                        false,
                        true
                    );
                    new Variable(
                        templ,
                        `${m}_${occ}_OVERALL_PEDAL_VERTICAL_INTRUSION_SCORE`,
                        `Overall pedal vertical intrusion score`,
                        overall_pedal_vertical_score.toString(),
                        "String",
                        false,
                        true
                    );
                    new Variable(
                        templ,
                        `${m}_${occ}_MAX_PEDAL_FORE_AFT_INTRUSION`,
                        `Maximum pedal fore-aft intrusion`,
                        max_pedal_fore_aft_intrusion,
                        "String",
                        false,
                        true
                    );
                    new Variable(
                        templ,
                        `${m}_${occ}_OVERALL_PEDAL_FORE_AFT_INTRUSION_SCORE`,
                        `Overall pedal fore-aft intrusion score`,
                        overall_pedal_fore_aft_score.toString(),
                        "String",
                        false,
                        true
                    );
                    new Variable(
                        templ,
                        `${m}_${occ}_OVERALL_PEDAL_BLOCKING_PENALTY`,
                        `Overall pedal blocking penalty`,
                        overall_pedal_blocking_penalty.toString(),
                        "String",
                        false,
                        true
                    );
                    new Variable(
                        templ,
                        `${m}_${occ}_FOOT_AND_ANKLE_SCORE`,
                        `Foot and ankle score`,
                        foot_and_ankle_score.toString(),
                        "String",
                        false,
                        true
                    );

                    break;
                case "FRONT_PASSENGER":
                    /* No variables specific to front passenger */
                    break;
                default:
                    LogError(`Unexpected occupant type "${occ}" in ${body_region_label} calculation.`);
                    Exit();
            }

            for (let side of sides) {
                LogPrint(`Calculating ${m} ${occ} ${side} ${body_region_label} score...`);

                /* Create a status object to track whether REPORTER Variables are all present and valid.
                 * <success> is initially true but will be set to false if anything missing or invalid. */
                let status = { success: true, missing: [], invalid: [] };

                /* Lower leg, foot and ankle score and modifiers */

                /* Variables extracted for all occupants */

                let tibia_comp_values = get_multiple_variable_values(
                    status,
                    [
                        `${m}_${occ}_${side}_TIBIA_COMPRESSION_UPPER_MAX_VALUE`,
                        `${m}_${occ}_${side}_TIBIA_COMPRESSION_LOWER_MAX_VALUE`
                    ],
                    "float",
                    true
                );

                let tibia_comp_scores = get_multiple_variable_values(
                    status,
                    [
                        `${m}_${occ}_${side}_TIBIA_COMPRESSION_UPPER_MAX_SCORE`,
                        `${m}_${occ}_${side}_TIBIA_COMPRESSION_LOWER_MAX_SCORE`
                    ],
                    "float",
                    true
                );

                let tibia_index_scores = get_multiple_variable_values(
                    status,
                    [
                        `${m}_${occ}_${side}_TIBIA_INDEX_UPPER_MAX_SCORE`,
                        `${m}_${occ}_${side}_TIBIA_INDEX_LOWER_MAX_SCORE`
                    ],
                    "float",
                    true
                );

                /* Final scores all start at zero and will remain so if any variables were missing or invalid */
                tibia_compression_max_val[side] = "[Not computed]";
                tibia_compression_score[side] = "[Not computed]";
                lower_leg[side] = 0;

                /* If we have all the required variables, calculate the final scores */
                if (status.success && pedal_status.success) {
                    /* We report a single tibia compression max value and score */
                    tibia_compression_max_val[side] = Math.max(...Object.values(tibia_comp_values));
                    tibia_compression_score[side] = Math.min(...Object.values(tibia_comp_scores));

                    /* Score calculation is different for different occupants */
                    switch (occ) {
                        case "DRIVER":
                            /* Left/right score is minimum of tibia compression and tibia index scores, summed with modifier */
                            lower_leg[side] = Math.max(
                                0,
                                Math.min(...Object.values(tibia_comp_scores), ...Object.values(tibia_index_scores)) +
                                    overall_pedal_vertical_score
                            );
                            break;
                        case "FRONT_PASSENGER":
                            /* Left/right score is minimum of tibia compression and tibia index scores, summed with modifier*/
                            let passenger_footwell_rupture = get_variable_value(
                                status,
                                `${m}_${occ}_FOOTWELL_RUPTURE_MODIFIER`,
                                "int"
                            );

                            lower_leg[side] = Math.max(
                                0,
                                Math.min(...Object.values(tibia_comp_scores), ...Object.values(tibia_index_scores)) +
                                    passenger_footwell_rupture
                            );
                            break;
                        default:
                            LogError(`Unexpected occupant type "${occ}" in ${body_region_label} calculation.`);
                            Exit();
                    }

                    LogPrint(`${m} ${occ} ${side} lower leg score = ${lower_leg[side]}`);
                } else {
                    warn_about_missing_or_invalid_variables(
                        status,
                        `${m} ${occ} ${body_region_label} score calculation`
                    );
                }

                new Variable(
                    templ,
                    `${m}_${occ}_${side}_TIBIA_COMPRESSION_MAX_VALUE`,
                    `${side} tibia compression max value`,
                    tibia_compression_max_val[side],
                    "String",
                    false,
                    true
                );
                new Variable(
                    templ,
                    `${m}_${occ}_${side}_TIBIA_COMPRESSION_SCORE`,
                    `${side} tibia compression score`,
                    tibia_compression_score[side],
                    "String",
                    false,
                    true
                );
                new Variable(
                    templ,
                    `${m}_${occ}_LOWER_LEG_${side}_FINAL_SCORE`,
                    `Final ${side} lower leg score`,
                    lower_leg[side].toString(),
                    "String",
                    false,
                    true
                );
            }

            /* Overall score for each occupant is minimum of left and right scores */
            let lower_leg_foot_and_ankle_final_score = Math.min(lower_leg.LEFT, lower_leg.RIGHT);
            /* For driver, we also consider the foot and ankle score */
            if (occ == "DRIVER") {
                lower_leg_foot_and_ankle_final_score = Math.min(
                    lower_leg_foot_and_ankle_final_score,
                    foot_and_ankle_score
                );
            }
            new Variable(
                templ,
                `${m}_${occ}_LOWER_LEG_FOOT_ANKLE_FINAL_SCORE`,
                `Final ${body_region_label} score`,
                lower_leg_foot_and_ankle_final_score.toString(),
                "String",
                false,
                true
            );
        }
    }
}

/**
 * From the GNCAP Assessment Protocol (v7.1.2):
 *
 *     Where the rearward displacement of a ‘blocked’ pedal exceeds 175mm relative to the pre-test
 *     measurement, a one point penalty is applied to the driver’s foot and ankle assessment. A pedal is
 *     blocked when the forward movement of the intruded pedal under a load of 200N is <25mm.
 *     Between 50mm and 175mm of rearward displacement the penalty is calculated using a sliding scale
 *     between 0 to 1 points.
 *
 * @param {number} val Rearward displacement of blocked pedal
 * @param {number} hi_perf Displacement corresponding to high performance (50 mm)
 * @param {number} lo_perf Displacement corresponding to low performance (175 mm)
 * @param {number} hi_score Penalty corresponding to high performance (0)
 * @param {number} lo_score Penalty corresponding to low performance (-1)
 * @returns {number}
 */
function pedal_blocking_sliding_scale(val, hi_perf, lo_perf, hi_score, lo_score) {
    var retval = 0.0;

    if (val < hi_perf) retval = hi_score;
    else if (val > lo_perf) retval = lo_score;
    else retval = hi_score + ((val - hi_perf) * (lo_score - hi_score)) / (lo_perf - hi_perf);

    return retval;
}
