/* Contain the entire script within a function because REPORTER only has a single JavaScript realm
 * for the entire session. */
chest_score();

/**
 * Performs the chest and abdomen score calculation for the specified models
 */
function chest_score() {
    let templ = Template.GetCurrent();
    let models = get_model_list();
    let body_region_label = `chest`;

    let occupants = ["DRIVER", "FRONT_PASSENGER"];
    for (let m of models) {
        /* Later we create a REPORTER variable to display an explanatory message if any capping
         * limit exceeded */
        let capping_limit_explanation = "";
        for (let occ of occupants) {
            LogPrint(`Calculating ${m} ${occ} ${body_region_label} score...`);

            /* Create a status object to track whether REPORTER Variables are all present and valid.
             * <success> is initially true but will be set to false if anything missing or invalid. */
            let status = { success: true, missing: [], invalid: [] };

            /* Chest score and modifiers */

            /* Driver */
            let steering_wheel_contact_mod;

            /* Shared */

            let scores = get_multiple_variable_values(
                status,
                [`${m}_${occ}_CHEST_COMPRESSION_SCORE`, `${m}_${occ}_CHEST_VISCOUS_CRITERION_SCORE`],
                "float",
                true
            );

            let capping_limit_values = get_multiple_variable_values(
                status,
                [`${m}_${occ}_CHEST_COMPRESSION_CAPPING_LIMIT`, `${m}_${occ}_CHEST_VISCOUS_CRITERION_CAPPING_LIMIT`],
                "string",
                true
            );

            let incorrect_airbag_mod = get_variable_value(
                status,
                `${m}_${occ}_CHEST_INCORRECT_AIRBAG_DEPLOYMENT_MODIFIER`,
                "int"
            );

            let unstable_compartment_mod = get_variable_value(
                status,
                `${m}_${occ}_CHEST_UNSTABLE_PASSENGER_COMPARTMENT_MODIFIER`,
                "int"
            );

            let belt_load_mod = get_variable_value(
                status,
                `${m}_${occ}_SHOULDER_BELT_B3_SHOULDER_BELT_FORCE_SCORE`,
                "int"
            );

            let a_pillar_score;

            switch (occ) {
                case "DRIVER":
                    steering_wheel_contact_mod = get_variable_value(
                        status,
                        `${m}_${occ}_CHEST_STEERING_WHEEL_CONTACT_MODIFIER`,
                        "int"
                    );
                    a_pillar_score = get_variable_value(
                        status,
                        `${m}_STRUCTURE_A_PILLAR_FORE_AFT_INTRUSION_SCORE`,
                        "float"
                    );
                    break;
                case "FRONT_PASSENGER":
                    a_pillar_score = get_variable_value(
                        status,
                        `${m}_STRUCTURE_PASSENGER_SIDE_A_PILLAR_FORE_AFT_INTRUSION_SCORE`,
                        "float"
                    );
                    break;
                default:
                    LogError(`Unexpected occupant type "${occ}" in ${body_region_label} calculation.`);
                    Exit();
            }

            /* Final scores all start at zero and will remain so if any variables were missing or invalid */
            let chest_score_before_mods = 0;
            let chest_score = 0;
            /* Capping limits start as exceeded and will remain so if any variables were missing or invalid */
            let chest_capping_limit;

            /* If we have all the required variables, calculate the final scores */
            if (status.success) {
                /* Score calculation is different for different occupants */
                switch (occ) {
                    case "DRIVER":
                        /* Chest score is minimum of compression and viscous criterion scores, summed with modifiers */
                        chest_score_before_mods = Math.min(...Object.values(scores));
                        chest_score = Math.max(
                            0,
                            chest_score_before_mods +
                                a_pillar_score +
                                unstable_compartment_mod +
                                steering_wheel_contact_mod +
                                incorrect_airbag_mod +
                                belt_load_mod
                        );

                        /* Only report abdomen score for driver */
                        LogPrint(`${m} ${occ} chest score = ${chest_score}`);

                        /* Check capping limits. If any individual limits are exceeded, an asterisk is
                         * displayed for all parent body regions. */

                        chest_capping_limit = return_capping_limit(Object.values(capping_limit_values));
                        if (chest_capping_limit == "*") {
                            capping_limit_explanation = `*Capping limit exceeded`;
                        }

                        break;
                    case "FRONT_PASSENGER":
                        /* Chest score is minimum of compression and viscous criterion scores, summed with modifiers */
                        chest_score_before_mods = Math.min(...Object.values(scores));
                        /* only modifiers relevant to the front passenger are applied, i.e. not steering wheel, a-pillar or unstable compartment modifier*/
                        chest_score = Math.max(
                            0,
                            chest_score_before_mods +
                                incorrect_airbag_mod +
                                belt_load_mod +
                                a_pillar_score +
                                unstable_compartment_mod
                        );

                        /* Check capping limits. If any individual limits are exceeded, an asterisk is
                         * displayed for all parent body regions. */
                        chest_capping_limit = return_capping_limit(Object.values(capping_limit_values));
                        if (chest_capping_limit == "*") {
                            capping_limit_explanation = `*Capping limit exceeded`;
                        }

                        break;
                    default:
                        LogError(`Unexpected occupant type "${occ}" in ${body_region_label} calculation.`);
                        Exit();
                }

                chest_score = Math.max(Math.min(chest_score, 4.0), 0.0);

                LogPrint(`${m} ${occ} ${body_region_label} score = ${chest_score}`);
            } else {
                warn_about_missing_or_invalid_variables(status, `${m} ${occ} ${body_region_label} score calculation`);
            }

            /* Only write some variables for driver and passenger */
            new Variable(
                templ,
                `${m}_${occ}_CHEST_FINAL_SCORE`,
                "Final chest score",
                chest_score.toString(),
                "String",
                false,
                true
            );
            new Variable(
                templ,
                `${m}_${occ}_CHEST_CAPPING_LIMIT`,
                `Chest capping limit`,
                chest_capping_limit,
                "String",
                false,
                true
            );
        }

        new Variable(
            templ,
            `${m}_CHEST_CAPPING_LIMIT_EXPLANATION`,
            `Capping limit explanation`,
            capping_limit_explanation,
            "String",
            false,
            true
        );
    }
}
