/* Contain the entire script within a function because REPORTER only has a single JavaScript realm
 * for the entire session. */
overall_score();

/**
 * Performs the overall score calculation for the specified models
 */
function overall_score() {
    let templ = Template.GetCurrent();
    let models = get_model_list();

    let occupants = ["DRIVER", "FRONT_PASSENGER"];

    /* The following objects will be populated with values for each occupant */
    let head_neck_score = {};
    let chest_score = {};
    let knee_femur_pelvis_score = {};
    let lower_leg_foot_ankle_score = {};
    let occ_score = {};
    let occ_capping_limit = {};

    for (let m of models) {
        for (let occ of occupants) {
            LogPrint(`Calculating ${m} ${occ} overall score...`);

            /* Create a status object to track whether REPORTER Variables are all present and valid.
             * <success> is initially true but will be set to false if anything missing or invalid. */
            let status = { success: true, missing: [], invalid: [] };

            /* Body region final scores */
            head_neck_score[occ] = get_variable_value(status, `${m}_${occ}_HEAD_NECK_FINAL_SCORE`, "float");
            chest_score[occ] = get_variable_value(status, `${m}_${occ}_CHEST_FINAL_SCORE`, "float");
            knee_femur_pelvis_score[occ] = get_variable_value(
                status,
                `${m}_${occ}_KNEE_FEMUR_PELVIS_FINAL_SCORE`,
                "float"
            );
            lower_leg_foot_ankle_score[occ] = get_variable_value(
                status,
                `${m}_${occ}_LOWER_LEG_FOOT_ANKLE_FINAL_SCORE`,
                "float"
            );

            /* Body region capping limits */
            let head_neck_capping_limit = get_variable_value(status, `${m}_${occ}_HEAD_NECK_CAPPING_LIMIT`, "string");
            let chest_capping_limit = get_variable_value(status, `${m}_${occ}_CHEST_CAPPING_LIMIT`, "string");

            /* Final scores all start at zero and will remain so if any variables were missing or invalid */
            occ_score[occ] = 0;
            occ_capping_limit[occ] = "*";

            /* If we have all the required variables, calculate the final scores */
            if (status.success) {
                /* Overall occupant score is sum of body region scores. */
                occ_score[occ] =
                    head_neck_score[occ] +
                    chest_score[occ] +
                    knee_femur_pelvis_score[occ] +
                    lower_leg_foot_ankle_score[occ];
                LogPrint(`${m} ${occ} overall score = ${occ_score[occ]}`);

                /* Only remove the overall occupant capping limit if none of the body region
                 * capping limits were exceeded. */
                if (head_neck_capping_limit == "" && chest_capping_limit == "") {
                    occ_capping_limit[occ] = "";
                    LogPrint(`Capping limit not applied to ${m} ${occ}.`);
                } else {
                    LogPrint(`Capping limit applied to ${m} ${occ}.`);
                }
            } else {
                warn_about_missing_or_invalid_variables(status, `${m} ${occ} overall score calculation`);
            }

            let overall_occ_score_var = new Variable(
                templ,
                `${m}_${occ}_OVERALL_SCORE`,
                `Overall ${occ} score`,
                occ_score[occ].toString(),
                "String",
                false,
                true
            );
        }

        LogPrint(`Retrieving door opening modifier...`);
        /* Create a status object to track whether REPORTER Variables are all present and valid.
         * <success> is initially true but will be set to false if anything missing or invalid. */
        let status = { success: true, missing: [], invalid: [] };
        let door_opening_mod = get_variable_value(status, `${m}_DOOR_OPENING_MODIFIER`, "int");
        if (!status.success) {
            warn_about_missing_or_invalid_variables(status, `${m} door opening modifier`);
            door_opening_mod = 0;
        }

        let overall_score = 0.0;
        let overall_capping_limit = "";

        /* Overall capping limit applied if either occupant's capping limit exceeded */
        if (occ_capping_limit.DRIVER == "*" || occ_capping_limit.FRONT_PASSENGER == "*") {
            overall_capping_limit = "*";
            LogPrint(`Overall capping limit applied. ${m} overall score = ${overall_score}`);
        } else {
            /* Else (if capping limit not applied):
             * Overall score = 0.5 * (Sum of lowest score in each body region) + door opening modifier
             * With a minimum of zero (door opening modifier can't take score below zero).
             */
            overall_score = Math.max(
                0.0,
                0.5 *
                    (Math.min(head_neck_score.DRIVER, head_neck_score.FRONT_PASSENGER) +
                        Math.min(chest_score.DRIVER, chest_score.FRONT_PASSENGER) +
                        Math.min(knee_femur_pelvis_score.DRIVER, knee_femur_pelvis_score.FRONT_PASSENGER) +
                        Math.min(lower_leg_foot_ankle_score.DRIVER, lower_leg_foot_ankle_score.FRONT_PASSENGER) +
                        door_opening_mod
                    )
            );
            LogPrint(`${m} overall score = ${overall_score}`);
        }

        let overall_score_var = new Variable(
            templ,
            `${m}_OVERALL_SCORE`,
            `Overall score`,
            overall_score.toString(),
            "String",
            false,
            true
        );

        let overall_capping_limit_var = new Variable(
            templ,
            `${m}_OVERALL_CAPPING_LIMIT`,
            `Overall capping limit`,
            overall_capping_limit,
            "String",
            false,
            true
        );
        let capping_limit_explanation = `*Capping limit exceeded`;
        if (overall_capping_limit == "") {
            capping_limit_explanation = "";
        }
        let capping_limit_explanation_var = new Variable(
            templ,
            `${m}_OVERALL_CAPPING_LIMIT_EXPLANATION`,
            `Capping limit explanation`,
            capping_limit_explanation,
            "String",
            false,
            true
        );
    }
}
