/* Contain the entire script within a function because REPORTER only has a single JavaScript realm
 * for the entire session. */
overall_score();

/**
 * Performs the overall score calculation for the specified models
 */
function overall_score() {
    let templ = Template.GetCurrent();
    let models = get_model_list();

    let occupants = ["DRIVER", "FRONT_PASSENGER", "REAR_DRIVER_SIDE"];

    /* The following objects will be populated with values for each occupant */
    let head_score = {};
    let neck_score = {};
    let chest_abdomen_score = {};
    let knee_femur_pelvis_score = {};
    let occ_score = {};
    let occ_capping_limit = {};
    let front_passenger_percentage = 0;
    let final_driver_score = 0;
    let final_front_passenger_score = 0;
    let final_Rear_driver_side_score = 0;

    for (let m of models) {
        for (let occ of occupants) {
            LogPrint(`Calculating ${m} ${occ} overall score...`);

            /* Create a status object to track whether REPORTER Variables are all present and valid.
             * <success> is initially true but will be set to false if anything missing or invalid. */
            let status = { success: true, missing: [], invalid: [] };

            /* Body region final scores */
            head_score[occ] = get_variable_value(status, `${m}_${occ}_HEAD_FINAL_SCORE`, "float");
            neck_score[occ] = get_variable_value(status, `${m}_${occ}_NECK_FINAL_SCORE`, "float");
            chest_abdomen_score[occ] = get_variable_value(status, `${m}_${occ}_CHEST_FINAL_SCORE`, "float");
            knee_femur_pelvis_score[occ] = get_variable_value(
                status,
                `${m}_${occ}_KNEE_FEMUR_PELVIS_FINAL_SCORE`,
                "float"
            );

            /* Body region capping limits */
            let head_capping_limit = get_variable_value(status, `${m}_${occ}_HEAD_CAPPING_LIMIT`, "string");
            let neck_capping_limit = get_variable_value(status, `${m}_${occ}_NECK_CAPPING_LIMIT`, "string");
            let chest_capping_limit = get_variable_value(status, `${m}_${occ}_CHEST_CAPPING_LIMIT`, "string");

            /* Final scores all start at zero and will remain so if any variables were missing or invalid */
            occ_score[occ] = 0;
            occ_capping_limit[occ] = "*";

            /* If we have all the required variables, calculate the final scores */
            if (status.success) {
                /* Overall occupant score is sum of body region scores. */
                occ_score[occ] =
                    head_score[occ] + neck_score[occ] + chest_abdomen_score[occ] + knee_femur_pelvis_score[occ];
                LogPrint(`${m} ${occ} overall score = ${occ_score[occ]}`);

                if (occupants.includes("DRIVER")) {
                    final_driver_score = occ_score[occ];
                }
                LogPrint("the final driver score" + final_driver_score);

                if (occupants.includes("FRONT_PASSENGER")) {
                    final_front_passenger_score = occ_score[occ];
                }
                LogPrint("the final front passenger score" + final_front_passenger_score);

                if (occupants.includes("REAR_DRIVER_SIDE")) {
                    final_Rear_driver_side_score = occ_score[occ];
                }
                LogPrint("the final rear driver side score" + final_Rear_driver_side_score);

                /* Only remove the overall occupant capping limit if none of the body region
                 * capping limits were exceeded. */
                if (head_capping_limit == "" && chest_capping_limit == "") {
                    occ_capping_limit[occ] = "";
                    LogPrint(`Capping limit not applied to ${m} ${occ}.`);
                } else {
                    LogPrint(`Capping limit applied to ${m} ${occ}.`);
                }
            } else {
                warn_about_missing_or_invalid_variables(status, `${m} ${occ} overall score calculation`);
            }

            let overall_occ_score_var = new Variable(
                templ,
                `${m}_${occ}_OVERALL_SCORE`,
                `Overall ${occ} score`,
                occ_score[occ].toString(),
                "String",
                false,
                true
            );
        }

        LogPrint(`Retrieving door opening modifier...`);
        /* Create a status object to track whether REPORTER Variables are all present and valid.
         * <success> is initially true but will be set to false if anything missing or invalid. */
        let status = { success: true, missing: [], invalid: [] };
        let door_opening_mod = get_variable_value(status, `${m}_DOOR_OPENING_MODIFIER`, "int");
        if (!status.success) {
            warn_about_missing_or_invalid_variables(status, `${m} door opening modifier`);
            door_opening_mod = 0;
        }

        let overall_score = 0.0;
        let overall_capping_limit = "";

        /* Overall capping limit applied if either occupant's capping limit exceeded */
        if (
            occ_capping_limit.DRIVER == "*" ||
            occ_capping_limit.FRONT_PASSENGER == "*" ||
            occ_capping_limit.REAR_DRIVER_SIDE == "*"
        ) {
            overall_capping_limit = "*";
            LogPrint(`Overall capping limit applied. ${m} overall score = ${overall_score}`);
        } else {
            /* Else (if capping limit not applied):
             * Overall score = 0.5 * (Sum of lowest score in each body region) + door opening modifier
             * With a minimum of zero (door opening modifier can't take score below zero).
             */

            let front_passenger_percentage = (final_front_passenger_score / final_driver_score) * 100;

            LogPrint("the front passenger percentage" + front_passenger_percentage + "% ");

            if (front_passenger_percentage < 0.9 * final_driver_score) {
                overall_score =
                    (0.5 *
                        (Math.min(head_score.DRIVER, head_score.FRONT_PASSENGER) +
                            Math.min(neck_score.DRIVER, neck_score.FRONT_PASSENGER) +
                            Math.min(chest_abdomen_score.DRIVER, chest_abdomen_score.FRONT_PASSENGER) +
                            Math.min(knee_femur_pelvis_score.DRIVER, knee_femur_pelvis_score.FRONT_PASSENGER) +
                            final_Rear_driver_side_score) +
                        door_opening_mod) /
                    2;
            } else {
                overall_score = (0.5 * (final_driver_score + final_Rear_driver_side_score) + door_opening_mod) / 2;
            }

            LogPrint(`${m} overall score = ${overall_score}`);
            let front_passenger_percentage_var = new Variable(
                templ,
                `${m}_FRONT_PASSENGER_PERCENTAGE`,
                `front passenger percentage`,
                front_passenger_percentage.toString(),
                "String",
                false,
                true
            );
        }

        let overall_score_var = new Variable(
            templ,
            `${m}_OVERALL_SCORE`,
            `Overall score`,
            overall_score.toString(),
            "String",
            false,
            true
        );

        let overall_capping_limit_var = new Variable(
            templ,
            `${m}_OVERALL_CAPPING_LIMIT`,
            `Overall capping limit`,
            overall_capping_limit,
            "String",
            false,
            true
        );
        let capping_limit_explanation = `*Capping limit exceeded`;
        if (overall_capping_limit == "") {
            capping_limit_explanation = "";
        }
        let capping_limit_explanation_var = new Variable(
            templ,
            `${m}_OVERALL_CAPPING_LIMIT_EXPLANATION`,
            `Capping limit explanation`,
            capping_limit_explanation,
            "String",
            false,
            true
        );
    }
}
