/* Contain the entire script within a function because REPORTER only has a single JavaScript realm
 * for the entire session. */
overall_score();

/**
 * Performs the overall score calculation for the specified models
 */
function overall_score() {
    let templ = Template.GetCurrent();
    let models = get_model_list();

    let occupants = ["DRIVER", "FRONT_PASSENGER", "REAR_DRIVER_SIDE", "REAR_PASSENGER_SIDE"];

    /* The following objects will be populated with values for each occupant */
    let head_neck_score = {};
    let chest_abdomen_score = {};
    let knee_femur_pelvis_score = {};
    let lower_leg_foot_ankle_score = {};
    let restraint_system_modifier = {};
    let occ_score = {};
    let occ_capping_limit = {};

    for (let m of models) {
        for (let occ of occupants) {
            LogPrint(`Calculating ${m} ${occ} overall score...`);

            /* Create a status object to track whether REPORTER Variables are all present and valid.
             * <success> is initially true but will be set to false if anything missing or invalid. */
            let status = { success: true, missing: [], invalid: [] };

            /* Body region final scores */
            head_neck_score[occ] = get_variable_value(status, `${m}_${occ}_HEAD_NECK_FINAL_SCORE`, "float");
            chest_abdomen_score[occ] = get_variable_value(status, `${m}_${occ}_CHEST_ABDOMEN_FINAL_SCORE`, "float");
            if (occ == "DRIVER" || occ == "FRONT_PASSENGER") {
                knee_femur_pelvis_score[occ] = get_variable_value(
                    status,
                    `${m}_${occ}_KNEE_FEMUR_PELVIS_FINAL_SCORE`,
                    "float"
                );
            }
            if (occ == "DRIVER") {
                lower_leg_foot_ankle_score[occ] = get_variable_value(
                    status,
                    `${m}_${occ}_LOWER_LEG_FOOT_ANKLE_FINAL_SCORE`,
                    "float"
                );
            }

            /* Body region capping limits */
            let head_neck_capping_limit;
            let chest_capping_limit;
            if (occ == "DRIVER" || "REAR_Driver_SIDE" || "FRONT_PASSENGER") {
                head_neck_capping_limit = get_variable_value(
                    status,
                    `${m}_${occ}_HEAD_NECK_CAPPING_LIMIT`,
                    "string"
                );
                chest_capping_limit = get_variable_value(status, `${m}_${occ}_CHEST_CAPPING_LIMIT`, "string");
            }

            /* Restraint system modifier */
            let seat_system_mod;
            let seat_belt_mod;
            let Restraint_mod;
            if (occ == "REAR_PASSENGER_SIDE"){
                Restraint_mod = get_variable_value(
                    status,
                    `${m}_${occ}_RESTRAINT_SYSTEM_FAILS_MODIFIER`,
                    "float"
                );
                seat_belt_mod = get_variable_value(status, `${m}_${occ}_SEAT_BELT_FAILS_MODIFIER`, "float");
            } else{
                seat_system_mod = get_variable_value(status, `${m}_${occ}_SEAT_SYSTEM_FAILS_MODIFIER`, "float");
                seat_belt_mod = get_variable_value(status, `${m}_${occ}_SEAT_BELT_SYSTEM_FAILS_MODIFIER`, "float");
            }
            

            /* Final scores all start at zero and will remain so if any variables were missing or invalid */
            occ_score[occ] = 0;
            occ_capping_limit[occ] = "*";
            restraint_system_modifier[occ]= 0;

            /* If we have all the required variables, calculate the final scores */
            if (status.success) {
                /* Overall occupant score is sum of body region scores. */
                switch (occ) {
                    case "DRIVER":
                        restraint_system_modifier[occ] = seat_system_mod + seat_belt_mod;
                        occ_score[occ] = Math.max(
                            0,
                            head_neck_score[occ] +
                                chest_abdomen_score[occ] +
                                knee_femur_pelvis_score[occ] +
                                lower_leg_foot_ankle_score[occ] +
                                restraint_system_modifier[occ]
                        );
                        LogPrint(`${m} ${occ} overall score = ${occ_score[occ]}`);
                        break;
                    case "FRONT_PASSENGER":
                        restraint_system_modifier[occ] = seat_system_mod + seat_belt_mod;
                        occ_score[occ] = Math.max(
                            0,
                            head_neck_score[occ] +
                                chest_abdomen_score[occ] +
                                knee_femur_pelvis_score[occ] +
                                restraint_system_modifier[occ]
                        );
                        LogPrint(`${m} ${occ} overall score = ${occ_score[occ]}`);
                        break;
                    case "REAR_DRIVER_SIDE":
                        restraint_system_modifier[occ] = seat_system_mod + seat_belt_mod;
                        occ_score[occ] = Math.max(
                            0,
                            head_neck_score[occ] +
                                chest_abdomen_score[occ] +
                                restraint_system_modifier[occ]
                        );
                        LogPrint(`${m} ${occ} overall score = ${occ_score[occ]}`);
                        break;
                    case "REAR_PASSENGER_SIDE":
                        restraint_system_modifier[occ] = Restraint_mod + seat_belt_mod;
                        occ_score[occ] = Math.max(
                            0,
                            head_neck_score[occ] +
                                chest_abdomen_score[occ] +
                                restraint_system_modifier[occ]
                        );
                        break;
                }

                /* Only remove the overall occupant capping limit if none of the body region
                 * capping limits were exceeded. */
                if (head_neck_capping_limit == "" && chest_capping_limit == "") {
                    occ_capping_limit[occ] = "";
                    LogPrint(`Capping limit not applied to ${m} ${occ}.`);
                } else {
                    LogPrint(`Capping limit applied to ${m} ${occ}.`);
                }
            } else {
                warn_about_missing_or_invalid_variables(status, `${m} ${occ} overall score calculation`);
            }

            let restraint_system_modifier_var = new Variable(
                templ,
                `${m}_${occ}_RESTRAINT_SYSTEM_MODIFIER`,
                `Restraint system modifier`,
                restraint_system_modifier[occ].toString(),
                "String",
                false,
                true
            );

            let overall_occ_score_var = new Variable(
                templ,
                `${m}_${occ}_OVERALL_SCORE`,
                `Overall ${occ} score`,
                occ_score[occ].toString(),
                "String",
                false,
                true
            );
        }

        LogPrint(`Retrieving general modifiers...`);
        /* Create a status object to track whether REPORTER Variables are all present and valid.
         * <success> is initially true but will be set to false if anything missing or invalid. */
        let status = { success: true, missing: [], invalid: [] };
        let door_opening_during_impact_mod = get_variable_value(
            status,
            `${m}_DOOR_OPENING_DURING_IMPACT_MODIFIER`,
            "float"
        );
        let door_opening_after_impact_mod = get_variable_value(
            status,
            `${m}_DOOR_OPENING_AFTER_IMPACT_MODIFIER`,
            "float"
        );
        let seat_belt_buckle_force_mod = get_variable_value(status, `${m}_SEAT_BELT_BUCKLE_FORCE_MODIFIER`, "float");
        let locking_feature_fails_mod = get_variable_value(status, `${m}_LOCKING_FEATURE_FAILS_MODIFIER`, "float");
        let general_modifiers;
        if (!status.success) {
            warn_about_missing_or_invalid_variables(status, `${m} general modifiers`);
            general_modifiers = 0;
        } else {
            general_modifiers = Math.max(
                -4,
                door_opening_during_impact_mod +
                    door_opening_after_impact_mod +
                    seat_belt_buckle_force_mod +
                    locking_feature_fails_mod
            );
        }

        let general_modifier_var = new Variable(
            templ,
            `${m}_GENERAL_MODIFIER`,
            `General modifier`,
            general_modifiers.toString(),
            "String",
            false,
            true
        );

        let front_row_score ;
        let overall_capping_limit = "";

        /* Front row capping limit applied if either front row occupant's capping limit exceeded */
        if (occ_capping_limit.DRIVER == "*" || occ_capping_limit.FRONT_PASSENGER == "*") {
            overall_capping_limit = "*";
            front_row_score = 0;
            LogPrint(`Front row occupants capping limit applied.`);
        } else {
            /* Else (if capping limit not applied for front row occupants):
             * Overall score = Sum of lowest score in each body region + door opening modifier
             * With a minimum of zero (door opening modifier can't take score below zero).
             */
            front_row_score = Math.max(
                0.0,
                Math.min(head_neck_score.DRIVER, head_neck_score.FRONT_PASSENGER) +
                    Math.min(chest_abdomen_score.DRIVER, chest_abdomen_score.FRONT_PASSENGER) +
                    Math.min(knee_femur_pelvis_score.DRIVER, knee_femur_pelvis_score.FRONT_PASSENGER) +
                    lower_leg_foot_ankle_score.DRIVER +
                    Math.min(restraint_system_modifier.DRIVER, restraint_system_modifier.FRONT_PASSENGER)
            );
        }

        let rear_row_score = 0.0;
        /* Rear row capping limit applied if rear row occupant's capping limit exceeded */
        if (occ_capping_limit.REAR_DRIVER_SIDE == "*") {
            rear_row_score = Math.max(0.0, occ_score.REAR_PASSENGER_SIDE);
        } else {
            rear_row_score = Math.max(0.0, occ_score.REAR_DRIVER_SIDE + occ_score.REAR_PASSENGER_SIDE);
        }

        let overall_score = Math.max(0.0, front_row_score + rear_row_score + general_modifiers);
        LogPrint(`${m} overall score = ${overall_score}`);

        let front_row_score_var = new Variable(
            templ,
            `${m}_FRONT_ROW_SCORE`,
            `Front row score`,
            front_row_score.toString(),
            "String",
            false,
            true
        );

        let overall_score_var = new Variable(
            templ,
            `${m}_OVERALL_SCORE`,
            `Overall score`,
            overall_score.toString(),
            "String",
            false,
            true
        );

        let overall_capping_limit_var = new Variable(
            templ,
            `${m}_OVERALL_CAPPING_LIMIT`,
            `Overall capping limit`,
            overall_capping_limit,
            "String",
            false,
            true
        );
        let capping_limit_explanation = `*Capping limit exceeded`;
        if (overall_capping_limit == "") {
            capping_limit_explanation = "";
        }
        let capping_limit_explanation_var = new Variable(
            templ,
            `${m}_OVERALL_CAPPING_LIMIT_EXPLANATION`,
            `Capping limit explanation`,
            capping_limit_explanation,
            "String",
            false,
            true
        );
    }
}
