/* Contain the entire script within a function because REPORTER only has a single JavaScript realm
 * for the entire session. */
overall_score();

/**
 * Performs the overall score calculation for the specified models
 */
function overall_score() {
    let templ = Template.GetCurrent();
    let models = get_model_list();
    let occupants = ["DRIVER", "FRONT_PASSENGER", "REAR_DRIVER_SIDE"];

    /* The following objects will be populated with values for each occupant */
    let head_score = {};
    let neck_score = {};
    let chest_score = {};
    let femur_and_knee_score = {};
    let tibia_score = {};
    let restraint_system_modifier = {};
    let occ_score = {};
    let occ_capping_limit = {};

    for (let m of models) {
        for (let occ of occupants) {
            LogPrint(`Calculating ${m} ${occ} overall score...`);

            /* Create a status object to track whether REPORTER Variables are all present and valid.
             * <success> is initially true but will be set to false if anything missing or invalid. */
            let status = { success: true, missing: [], invalid: [] };

            /* Body region final scores - common for all 3 occupants */
            head_score[occ] = get_variable_value(status, `${m}_${occ}_HEAD_FINAL_SCORE`, "float");
            neck_score[occ] = get_variable_value(status, `${m}_${occ}_NECK_FINAL_SCORE`, "float");
            chest_score[occ] = get_variable_value(status, `${m}_${occ}_CHEST_FINAL_SCORE`, "float");

            /* Body region final scores - common for front row occupants */
            if (occ == "DRIVER" || occ == "FRONT_PASSENGER") {
                femur_and_knee_score[occ] = get_variable_value(
                    status,
                    `${m}_${occ}_FEMUR_AND_KNEE_FINAL_SCORE`,
                    "float"
                );
                tibia_score[occ] = get_variable_value(status, `${m}_${occ}_TIBIA_FINAL_SCORE`, "float");
            }

            /* Body region capping limits - common for all 3 occupants */
            let head_capping_limit = get_variable_value(status, `${m}_${occ}_HEAD_CAPPING_LIMIT`, "string");
            let neck_capping_limit = get_variable_value(status, `${m}_${occ}_NECK_CAPPING_LIMIT`, "string");
            let chest_capping_limit = get_variable_value(status, `${m}_${occ}_CHEST_CAPPING_LIMIT`, "string");

            /* Restraint system modifier - common for all 3 occupants */
            let seat_system_mod = get_variable_value(status, `${m}_${occ}_SEAT_SYSTEM_FAILURE_MODIFIER`, "int");
            let seat_belt_mod = get_variable_value(status, `${m}_${occ}_SEATBELT_FAILURE_MODIFIER`, "int");

            /* Pelvic submarining mod only for 2nd row female */
            let submarine_mod = 0.0;
            if (occ == "REAR_DRIVER_SIDE") {
                let submarine_mod = get_variable_value(status, `${m}_${occ}_PELVIC_SUBMARINING_MODIFIER`, "int");
            }
            /* Final scores all start at zero and will remain so if any variables were missing or invalid */
            occ_score[occ] = 0;
            occ_capping_limit[occ] = "*";

            /* If we have all the required variables, calculate the final scores */
            if (status.success) {
                /* Overall occupant score is sum of body region scores. */
                switch (occ) {
                    case "DRIVER":
                        restraint_system_modifier[occ] = seat_system_mod + seat_belt_mod;
                        occ_score[occ] = Math.max(
                            0,
                            head_score[occ] +
                                neck_score[occ] +
                                chest_score[occ] +
                                femur_and_knee_score[occ] +
                                tibia_score[occ] +
                                restraint_system_modifier[occ]
                        );
                        LogPrint(`${m} ${occ} overall score = ${occ_score[occ]}`);
                        break;
                    case "FRONT_PASSENGER":
                        restraint_system_modifier[occ] = seat_system_mod + seat_belt_mod;
                        occ_score[occ] = Math.max(
                            0,
                            head_score[occ] +
                                neck_score[occ] +
                                chest_score[occ] +
                                femur_and_knee_score[occ] +
                                tibia_score[occ] +
                                restraint_system_modifier[occ]
                        );
                        LogPrint(`${m} ${occ} overall score = ${occ_score[occ]}`);
                        break;
                    case "REAR_DRIVER_SIDE":
                        restraint_system_modifier[occ] = seat_system_mod + seat_belt_mod;
                        occ_score[occ] = Math.max(
                            0,
                            head_score[occ] +
                                neck_score[occ] +
                                chest_score[occ] +
                                submarine_mod +
                                restraint_system_modifier[occ]
                        );
                        LogPrint(`${m} ${occ} overall score = ${occ_score[occ]}`);
                        break;
                }

                /* Only remove the overall occupant capping limit if none of the body region
                 * capping limits were exceeded. */
                if (head_capping_limit == "" && neck_capping_limit == "" && chest_capping_limit == "") {
                    occ_capping_limit[occ] = "";
                    LogPrint(`Capping limit not applied to ${m} ${occ}.`);
                } else {
                    LogPrint(`Capping limit applied to ${m} ${occ}.`);
                }
            } else {
                warn_about_missing_or_invalid_variables(status, `${m} ${occ} overall score calculation`);
            }

            let restraint_system_modifier_var = new Variable(
                templ,
                `${m}_${occ}_RESTRAINT_SYSTEM_MODIFIER`,
                `Restraint system modifier`,
                restraint_system_modifier[occ].toString(),
                "String",
                false,
                true
            );

            let overall_occ_score_var = new Variable(
                templ,
                `${m}_${occ}_OVERALL_SCORE`,
                `Overall ${occ} score`,
                occ_score[occ].toString(),
                "String",
                false,
                true
            );
        }

        let frt_row_res_mod = Math.min(restraint_system_modifier.DRIVER, restraint_system_modifier.FRONT_PASSENGER);
        let frt_row_res_mod_var = new Variable(
            templ,
            `${m}_FRONT_ROW_RESTRAINT_SYSTEM_MODIFIER`,
            `Front row restraint system modifier taking the lower between driver and front passenger`,
            frt_row_res_mod.toString(),
            "String",
            false,
            true
        );

        LogPrint(`Retrieving general modifiers...`);
        /* Create a status object to track whether REPORTER Variables are all present and valid.
         * <success> is initially true but will be set to false if anything missing or invalid. */
        let status = { success: true, missing: [], invalid: [] };
        let door_open_mod = get_variable_value(status, `${m}_DOOR_OPENING_MODIFIER`, "int");
        let inc_res_rel_mod = get_variable_value(status, `${m}_INCORRECT_RESTRAINT_RELEASE_MODIFIER`, "int");
        let row_door_jam_mod = get_variable_value(status, `${m}_ROW_OF_DOORS_JAMMED_MODIFIER`, "int");
        let loc_feat_fail_mod = get_variable_value(status, `${m}_DOOR_LOCKING_FEATURE_FAILURE_MODIFIER`, "int");
        let general_modifiers;
        if (!status.success) {
            warn_about_missing_or_invalid_variables(status, `${m} general modifiers`);
            general_modifiers = 0;
        } else {
            general_modifiers = Math.max(-4, door_open_mod + inc_res_rel_mod + row_door_jam_mod + loc_feat_fail_mod);
        }

        let general_modifier_var = new Variable(
            templ,
            `${m}_GENERAL_MODIFIER`,
            `General modifier`,
            general_modifiers.toString(),
            "String",
            false,
            true
        );

        let front_row_score = 0.0;

        let overall_capping_limit = "";
        let frt_row_capping_limit = "";
        let sec_row_female_cap_lim = "";

        /* Front row capping limit applied if either front row occupant's capping limit exceeded */
        if (occ_capping_limit.DRIVER == "*" || occ_capping_limit.FRONT_PASSENGER == "*") {
            frt_row_capping_limit = "*";
            overall_capping_limit = "*";
            LogPrint(`Front row occupants capping limit applied.`);
        } else {
            /* Else (if capping limit not applied for front row occupants):
             * Overall score = Sum of lowest score in each body region + restraint system modifiers
             * With a minimum of zero (door opening modifier can't take score below zero).
             */
            front_row_score = Math.max(
                0.0,
                Math.min(head_score.DRIVER, head_score.FRONT_PASSENGER) +
                    Math.min(neck_score.DRIVER, neck_score.FRONT_PASSENGER) +
                    Math.min(chest_score.DRIVER, chest_score.FRONT_PASSENGER) +
                    Math.min(femur_and_knee_score.DRIVER, femur_and_knee_score.FRONT_PASSENGER) +
                    Math.min(tibia_score.DRIVER, tibia_score.FRONT_PASSENGER) +
                    Math.min(restraint_system_modifier.DRIVER, restraint_system_modifier.FRONT_PASSENGER)
            );
        }

        let rear_row_score = 0.0;
        /* Rear row capping limit applied if rear row occupant's capping limit exceeded */
        if (occ_capping_limit.REAR_DRIVER_SIDE == "*") {
            sec_row_female_cap_lim = "*";
            LogPrint(`Rear row occupant capping limit applied.`);
        } else {
            rear_row_score = Math.max(0.0, occ_score.REAR_DRIVER_SIDE);
        }

        let overall_score = Math.max(0.0, front_row_score + rear_row_score + general_modifiers);
        LogPrint(`${m} overall score = ${overall_score}`);

        let front_row_score_var = new Variable(
            templ,
            `${m}_FRONT_ROW_SCORE`,
            `Overall score for front row dummies`,
            front_row_score.toString(),
            "String",
            false,
            true
        );

        let rear_row_score_var = new Variable(
            templ,
            `${m}_REAR_ROW_SCORE`,
            `Overall score for 2nd row female`,
            rear_row_score.toString(),
            "String",
            false,
            true
        );

        let overall_score_var = new Variable(
            templ,
            `${m}_OVERALL_SCORE`,
            `Overall score`,
            overall_score.toString(),
            "String",
            false,
            true
        );

        let overall_capping_limit_var = new Variable(
            templ,
            `${m}_OVERALL_CAPPING_LIMIT`,
            `Overall capping limit`,
            overall_capping_limit,
            "String",
            false,
            true
        );

        let frt_row_capping_limit_var = new Variable(
            templ,
            `${m}_FRONT_ROW_CAPPING_LIMIT`,
            `Overall capping limit`,
            frt_row_capping_limit,
            "String",
            false,
            true
        );

        let sec_row_female_cap_lim_var = new Variable(
            templ,
            `${m}_SECOND_ROW_FEMALE_CAPPING_LIMIT`,
            `Overall capping limit`,
            sec_row_female_cap_lim,
            "String",
            false,
            true
        );

        let capping_limit_explanation = `*Capping limit exceeded`;
        if (overall_capping_limit == "") {
            capping_limit_explanation = "";
        }
        let capping_limit_explanation_var = new Variable(
            templ,
            `${m}_OVERALL_CAPPING_LIMIT_EXPLANATION`,
            `Capping limit explanation`,
            capping_limit_explanation,
            "String",
            false,
            true
        );
    }
}
