/* Contain the entire script within a function because REPORTER only has a single JavaScript realm
 * for the entire session. */
neck_score();

/**
 * Performs the neck score calculation for the specified models
 * Note that we assume 2nd row female sits on Rear Driver Side. In future need to allow sitting on either side
 */
function neck_score() {
    let templ = Template.GetCurrent();
    let models = get_model_list();
    let body_region_label = `neck`;

    let occupants = ["DRIVER", "FRONT_PASSENGER", "REAR_DRIVER_SIDE"];

    for (let m of models) {
        /* Later we create a REPORTER variable to display an explanatory message if any capping
         * limit exceeded. Please note that for C-NCAP 2024 Front cases (FRB, MPDB),
         * the total score consists of: (1) front row score for each body region,
         * (2) 2nd row female score for each body region, (3) 2nd row child
         * To support both front row occupants + 2nd row female,
         * two separate capping limit messages are created as follows: */
        let capping_limit_explanation_frt_row = "";
        let capping_limit_explanation_sec_row_female = "";

        let front_row_score_neck = 0.0;
        let frt_row_cap_neck = "";
        let neck_score_driver = 0.0;
        let neck_score_front_passenger = 0.0;

        for (let occ of occupants) {
            LogPrint(`Calculating ${m} ${occ} ${body_region_label} score...`);

            /* Create a status object to track whether REPORTER Variables are all present and valid.
             * <success> is initially true but will be set to false if anything missing or invalid. */
            let status = { success: true, missing: [], invalid: [] };

            /* Neck scores */
            let neck_axial_tension_score;
            let neck_shear_score;
            let neck_extension_score;
            let neck_axial_tension_capping_limit;
            let neck_shear_capping_limit;
            let neck_extension_capping_limit;

            let head_2nd_impact;

            /* Front row variables */
            if (occ == "DRIVER" || occ == "FRONT_PASSENGER") {
                /* Shared */
                neck_axial_tension_score = get_variable_value(
                    status,
                    `${m}_${occ}_NECK_TENSION_EXCEEDENCE_SCORE`,
                    "float"
                );
                neck_axial_tension_capping_limit = get_variable_value(
                    status,
                    `${m}_${occ}_NECK_TENSION_EXCEEDENCE_CAPPING_LIMIT`
                );
                neck_shear_score = get_variable_value(status, `${m}_${occ}_NECK_SHEAR_EXCEEDENCE_SCORE`, "float");
                neck_shear_capping_limit = get_variable_value(
                    status,
                    `${m}_${occ}_NECK_SHEAR_EXCEEDENCE_CAPPING_LIMIT`
                );
                neck_extension_score = get_variable_value(status, `${m}_${occ}_NECK_EXTENSION_SCORE`, "float");
                neck_extension_capping_limit = get_variable_value(status, `${m}_${occ}_NECK_EXTENSION_CAPPING_LIMIT`);
            }
            /* 2nd row female variables */
            if (occ == "REAR_DRIVER_SIDE") {
                /* 2nd row female */
                neck_axial_tension_score = get_variable_value(
                    status,
                    `${m}_${occ}_NECK_AXIAL_REAR_PASSENGER_SCORE`,
                    "float"
                );
                neck_axial_tension_capping_limit = get_variable_value(
                    status,
                    `${m}_${occ}_NECK_AXIAL_REAR_PASSENGER_CAPPING_LIMIT`
                );
                neck_shear_score = get_variable_value(status, `${m}_${occ}_NECK_SHEAR_REAR_PASSENGER_SCORE`, "float");
                neck_shear_capping_limit = get_variable_value(
                    status,
                    `${m}_${occ}_NECK_SHEAR_REAR_PASSENGER_CAPPING_LIMIT`
                );
                neck_extension_score = get_variable_value(
                    status,
                    `${m}_${occ}_NECK_EXTENSION_REAR_PASSENGER_SCORE`,
                    "float"
                );
                neck_extension_capping_limit = get_variable_value(
                    status,
                    `${m}_${occ}_NECK_EXTENSION_REAR_PASSENGER_CAPPING_LIMIT`
                );
                head_2nd_impact = get_variable_value(status, `${m}_${occ}_HEAD_SECONDARY_IMPACT_MODIFIER`);
            }

            /* Final scores all start at zero and will remain so if any variables were missing or invalid */
            let neck_score = 0.0;

            /* Previously, capping limits start as exceeded and will remain so if any variables were missing or invalid
             * However, now we want them to start from blank, so if any dummies missing it will not show "*" */
            let neck_capping_limit = "";

            /* If we have all the required variables, calculate the final scores */
            /* Ref. Table 3-12 on Chapter.III of C-NCAP 2024 Management Regulation */
            if (status.success) {
                /* Neck front row */
                if (occ == "DRIVER" || occ == "FRONT_PASSENGER") {
                    /* Neck score is sum of scores, no modifiers for this body region. */
                    neck_score = Math.min(neck_axial_tension_score, neck_shear_score, neck_extension_score);
                    /* Bound the score between upper and lower limits */
                    neck_score = Math.max(neck_score, 0.0);
                    LogPrint(`${m} ${occ} neck score = ${neck_score}`);
                    if (occ == "DRIVER") {
                        neck_score_driver = neck_score;
                    }
                    if (occ == "FRONT_PASSENGER") {
                        neck_score_front_passenger = neck_score;
                    }
                    /* Check capping limits. If any individual limits are exceeded, an asterisk is
                     * displayed for all parent body regions. */
                    if (
                        neck_axial_tension_capping_limit == "" &&
                        neck_shear_capping_limit == "" &&
                        neck_extension_capping_limit == ""
                    ) {
                        neck_capping_limit = "";
                    } else {
                        neck_capping_limit = "*";
                        capping_limit_explanation_frt_row = `*Capping limit exceeded`;
                        frt_row_cap_neck = "*";
                    }
                }
                /* 2nd row female neck score
                 * Note that we assume 2nd row female sits on Rear Driver Side
                 * In future need to allow sitting on either side */
                if (occ == "REAR_DRIVER_SIDE") {
                    /* According to 1.2.1.1.2.2 in Chapter III of C-NCAP 2024,
                     * if no head secondary impact, only neck tension taken in scoring */
                    if (head_2nd_impact == "Yes") {
                        neck_score = Math.min(neck_axial_tension_score, neck_shear_score, neck_extension_score);
                    }
                    if (head_2nd_impact == "No") {
                        neck_score = neck_axial_tension_score;
                    }
                    /* Bound the score between upper and lower limits */
                    neck_score = Math.max(neck_score, 0.0);
                    LogPrint(`${m} ${occ} neck score = ${neck_score}`);

                    /* Check capping limits. If any individual limits are exceeded, an asterisk is
                     * displayed for all parent body regions. */
                    if (
                        neck_axial_tension_capping_limit == "" &&
                        neck_shear_capping_limit == "" &&
                        neck_extension_capping_limit == ""
                    ) {
                        neck_capping_limit = "";
                    } else {
                        neck_capping_limit = "*";
                        capping_limit_explanation_sec_row_female = `*Capping limit exceeded`;
                        frt_row_cap_neck = "*";
                    }
                }
            } else {
                warn_about_missing_or_invalid_variables(status, `${m} ${occ} ${body_region_label} score calculation`);
            }

            /* Overall scores */
            let neck_score_var = new Variable(
                templ,
                `${m}_${occ}_NECK_FINAL_SCORE`,
                "Final neck score",
                neck_score.toString(),
                "String",
                false,
                true
            );

            /* Capping limits */
            let neck_capping_limit_var = new Variable(
                templ,
                `${m}_${occ}_NECK_CAPPING_LIMIT`,
                "Neck capping limit (*asterisk if limit exceeded; empty string otherwise)",
                neck_capping_limit,
                "String",
                false,
                true
            );
        }

        front_row_score_neck = Math.min(neck_score_driver, neck_score_front_passenger);
        let ront_row_score_neck_var = new Variable(
            templ,
            `${m}_FRONT_ROW_NECK_SCORE`,
            "Final femur and knee score",
            front_row_score_neck.toString(),
            "String",
            false,
            true
        );
        let frt_row_cap_head_var = new Variable(
            templ,
            `${m}_FRONT_ROW_NECK_CAPPING_LIMIT`,
            `Front row neck capping limit asterisk`,
            frt_row_cap_neck,
            "String",
            false,
            true
        );
        let capping_limit_frt_row_explanation_var = new Variable(
            templ,
            `${m}_FRONT_ROW_NECK_CAPPING_LIMIT_EXPLANATION`,
            `Capping limit explanation - Neck - Front row`,
            capping_limit_explanation_frt_row,
            "String",
            false,
            true
        );
        let capping_limit_sec_row_female_explanation_var = new Variable(
            templ,
            `${m}_SECOND_ROW_FEMALE_NECK_CAPPING_LIMIT_EXPLANATION`,
            `Capping limit explanation - Neck - 2nd row female`,
            capping_limit_explanation_sec_row_female,
            "String",
            false,
            true
        );
    }
}
