/* Contain the entire script within a function because REPORTER only has a single JavaScript realm
 * for the entire session. */
chest_score();

/**
 * Performs the chest score calculation for the specified models
 * Note that we assume 2nd row female sits on Rear Driver Side. In future need to allow sitting on either side
 */
function chest_score() {
    let templ = Template.GetCurrent();
    let models = get_model_list();
    let body_region_label = `chest`;

    let occupants = ["DRIVER", "FRONT_PASSENGER", "REAR_DRIVER_SIDE"];

    for (let m of models) {
        /* Later we create a REPORTER variable to display an explanatory message if any capping
         * limit exceeded. Please note that for C-NCAP 2024 Front cases (FRB, MPDB),
         * the total score consists of: (1) front row score for each body region,
         * (2) 2nd row female score for each body region, (3) 2nd row child
         * To support both front row occupants + 2nd row female,
         * two separate capping limit messages are created as follows: */
        let capping_limit_explanation_frt_row = "";
        let capping_limit_explanation_sec_row_female = "";

        /* Use this var to compare and store the front row score and capping notice */
        let frt_row_score_chest = 0.0;
        let frt_row_cap_chest = "";
        let chest_score_driver = 0.0;
        let chest_score_front_passenger = 0.0;

        for (let occ of occupants) {
            LogPrint(`Calculating ${m} ${occ} ${body_region_label} score...`);

            /* Create a status object to track whether REPORTER Variables are all present and valid.
             * <success> is initially true but will be set to false if anything missing or invalid. */
            let status = { success: true, missing: [], invalid: [] };

            /* Chest scores */
            let chest_comp_score;
            let chest_comp_capping_limit;
            let chest_vc_score;
            let chest_vc_capping_limit;
            /* Chest modifiers, 2nd row female seatbelt wearing position modifier (-0.5) instead of (-1) */
            let steering_column_status = { success: true, missing: [], invalid: [] };
            let shoulder_belt_status = { success: true, missing: [], invalid: [] };
            let steering_column_intrusion_x_score;
            let shoulder_belt_load_mod;
            let steering_wheel_contact_mod;
            let seatbelt_wear_pos_frt_row_mod; // -1.0 int
            let seatbelt_wear_pos_sec_row_female_mod; // -0.5 float
            let incorrect_airbag_mod;

            /* Front row variables */
            if (occ == "DRIVER" || occ == "FRONT_PASSENGER") {
                /* Front row score variables */
                chest_comp_score = get_variable_value(status, `${m}_${occ}_CHEST_COMPRESSION_SCORE`, "float");
                chest_comp_capping_limit = get_variable_value(status, `${m}_${occ}_CHEST_COMPRESSION_CAPPING_LIMIT`);
                chest_vc_score = get_variable_value(status, `${m}_${occ}_CHEST_VISCOUS_CRITERION_SCORE`, "float");
                chest_vc_capping_limit = get_variable_value(
                    status,
                    `${m}_${occ}_CHEST_VISCOUS_CRITERION_CAPPING_LIMIT`
                );
                /* Front row modifier variables */
                shoulder_belt_load_mod = get_variable_value(
                    shoulder_belt_status,
                    `${m}_${occ}_SHOULDER_BELT_B3_SHOULDER_BELT_FORCE_SCORE`,
                    "float"
                );
                if (!shoulder_belt_status.success) {
                    //Window.Message("Message", occ + " shoulder belt load missing, modifier set to zero.");
                    shoulder_belt_load_mod = 0.0;
                }
                seatbelt_wear_pos_frt_row_mod = get_variable_value(
                    status,
                    `${m}_${occ}_SEATBELT_WEARING_POSITION_MODIFIER`,
                    "int"
                );
                incorrect_airbag_mod = get_variable_value(
                    status,
                    `${m}_${occ}_CHEST_INCORRECT_AIRBAG_DEPLOYMENT_MODIFIER`,
                    "int"
                );
                /* Driver specific modifiers */
                if (occ == "DRIVER") {
                    steering_column_intrusion_x_score = get_variable_value(
                        steering_column_status,
                        `${m}_STRUCTURE_STEERING_COLUMN_FORE_AFT_INTRUSION_SCORE`,
                        "float"
                    );
                    if (!steering_column_status.success) {
                        //Window.Message("Message", occ + " steering column intrusion missing, modifier set to zero.");
                        steering_column_intrusion_x_score = 0.0;
                    }
                    steering_wheel_contact_mod = get_variable_value(
                        status,
                        `${m}_${occ}_CHEST_STEERING_WHEEL_CONTACT_MODIFIER`,
                        "int"
                    );
                }
            }
            /* 2nd row female variables */
            if (occ == "REAR_DRIVER_SIDE") {
                /* 2nd row female score variables */
                chest_comp_score = get_variable_value(
                    status,
                    `${m}_${occ}_CHEST_COMPRESSION_REAR_PASSENGER_SCORE`,
                    "float"
                );
                chest_comp_capping_limit = get_variable_value(
                    status,
                    `${m}_${occ}_CHEST_COMPRESSION_REAR_PASSENGER_CAPPING_LIMIT`
                );
                chest_vc_score = get_variable_value(
                    status,
                    `${m}_${occ}_CHEST_VISCOUS_CRITERION_REAR_PASSENGER_SCORE`,
                    "float"
                );
                chest_vc_capping_limit = get_variable_value(
                    status,
                    `${m}_${occ}_CHEST_VISCOUS_CRITERION_REAR_PASSENGER_CAPPING_LIMIT`
                );
                /* 2nd row female modifier variables */
                shoulder_belt_load_mod = get_variable_value(
                    shoulder_belt_status,
                    `${m}_${occ}_SHOULDER_BELT_B3_SHOULDER_BELT_REAR_PASSENGER_FORCE_SCORE`,
                    "float"
                );
                if (!shoulder_belt_status.success) {
                    //Window.Message("Message", occ + " shoulder belt load missing, modifier set to zero.");
                    shoulder_belt_load_mod = 0.0;
                }
                seatbelt_wear_pos_sec_row_female_mod = get_variable_value(
                    status,
                    `${m}_${occ}_SEATBELT_WEARING_POSITION_MODIFIER`,
                    "float" /* (-0.5) */
                );
            }

            /* Final scores all start at zero and will remain so if any variables were missing or invalid */
            let chest_score = 0.0;

            /* Record the modifiers */
            let chest_mod = 0.0;
            let chest_score_no_mod = 0.0;

            /* Previously, capping limits start as exceeded and will remain so if any variables were missing or invalid
             * However, now we want them to start from blank, so if any dummies missing it will not show "*" */
            let chest_capping_limit = "";

            /* If we have all the required variables, calculate the final scores */
            /* Ref. Table 3-12 on Chapter.III of C-NCAP 2024 Management Regulation */
            if (status.success) {
                /* Chest front row */
                /* Head score is sum of score with modifiers.
                 * For the driver, there is an extra steering column vertical displacement modifer 0~-1 points.
                 * Start with front row occupants*/
                if (occ == "DRIVER" || occ == "FRONT_PASSENGER") {
                    if (occ == "DRIVER") {
                        /* For the driver, also add the steering column intrusion and steering wheel contact modifier */
                        chest_mod =
                            steering_column_intrusion_x_score +
                            shoulder_belt_load_mod +
                            steering_wheel_contact_mod +
                            seatbelt_wear_pos_frt_row_mod +
                            incorrect_airbag_mod;
                        chest_score_no_mod = Math.min(chest_comp_score, chest_vc_score);

                        chest_score_driver = Math.max(0, chest_score_no_mod + chest_mod);
                        chest_score = chest_score_driver;
                    }

                    if (occ == "FRONT_PASSENGER") {
                        chest_mod = shoulder_belt_load_mod + seatbelt_wear_pos_frt_row_mod + incorrect_airbag_mod;
                        chest_score_no_mod = Math.min(chest_comp_score, chest_vc_score);

                        chest_score_front_passenger = Math.max(0, chest_score_no_mod + chest_mod);
                        chest_score = chest_score_front_passenger;
                    }

                    LogPrint(`${m} ${occ} chest score = ${chest_score}`);
                    /* Check capping limits. If any individual limits are exceeded, an asterisk is
                     * displayed for all parent body regions. */
                    if (chest_comp_capping_limit == "" && chest_vc_capping_limit == "") {
                        chest_capping_limit = "";
                    } else {
                        chest_capping_limit = "*";
                        capping_limit_explanation_frt_row = `*Capping limit exceeded`;
                        /* If any of front row dummies capped, the front row cap is on with asterisk */
                        frt_row_cap_chest = "*";
                    }
                }
                /* 2nd row female chest score
                 * Note that we assume 2nd row female sits on Rear Driver Side
                 * In future need to allow sitting on either side */
                if (occ == "REAR_DRIVER_SIDE") {
                    chest_mod = shoulder_belt_load_mod + seatbelt_wear_pos_sec_row_female_mod;
                    chest_score = Math.min(chest_comp_score, chest_vc_score) + chest_mod;
                    chest_score_no_mod = Math.min(chest_comp_score, chest_vc_score);
                    /* Bound the score between upper and lower limits */
                    chest_score = Math.max(chest_score, 0.0);
                    LogPrint(`${m} ${occ} chest score = ${chest_score}`);
                    /* Check capping limits. If any individual limits are exceeded, an asterisk is
                     * displayed for all parent body regions. */
                    if (chest_comp_capping_limit == "" && chest_vc_capping_limit == "") {
                        chest_capping_limit = "";
                    } else {
                        chest_capping_limit = "*";
                        capping_limit_explanation_sec_row_female = `*Capping limit exceeded`;
                    }
                }
            } else {
                warn_about_missing_or_invalid_variables(status, `${m} ${occ} ${body_region_label} score calculation`);
            }

            /* Overall scores */
            let chest_score_var = new Variable(
                templ,
                `${m}_${occ}_CHEST_FINAL_SCORE`,
                "Final chest score",
                chest_score.toString(),
                "String",
                false,
                true
            );
            let chest_mod_var = new Variable(
                templ,
                `${m}_${occ}_CHEST_MODIFIERS`,
                "Final chest modifiers",
                chest_mod.toString(),
                "String",
                false,
                true
            );
            let chest_score_no_mod_var = new Variable(
                templ,
                `${m}_${occ}_CHEST_SCORE_NO_MOD`,
                "Chest score before applying modifiers",
                chest_score_no_mod.toString(),
                "String",
                false,
                true
            );
            /* Capping limits */
            let chest_capping_limit_var = new Variable(
                templ,
                `${m}_${occ}_CHEST_CAPPING_LIMIT`,
                "Chest capping limit (*asterisk if limit exceeded; empty string otherwise)",
                chest_capping_limit,
                "String",
                false,
                true
            );
        }

        frt_row_score_chest = Math.min(chest_score_driver, chest_score_front_passenger);

        let frt_row_score_chest_var = new Variable(
            templ,
            `${m}_FRONT_ROW_CHEST_SCORE`,
            `Front row chest final score`,
            frt_row_score_chest.toString(),
            "String",
            false,
            true
        );
        let frt_row_cap_chest_var = new Variable(
            templ,
            `${m}_FRONT_ROW_CHEST_CAPPING_LIMIT`,
            `Front row chest capping limit asterisk`,
            frt_row_cap_chest,
            "String",
            false,
            true
        );

        let capping_limit_frt_row_explanation_var = new Variable(
            templ,
            `${m}_FRONT_ROW_CHEST_CAPPING_LIMIT_EXPLANATION`,
            `Capping limit explanation - Chest - Front row`,
            capping_limit_explanation_frt_row,
            "String",
            false,
            true
        );
        let capping_limit_sec_row_female_explanation_var = new Variable(
            templ,
            `${m}_SECOND_ROW_FEMALE_CHEST_CAPPING_LIMIT_EXPLANATION`,
            `Capping limit explanation - Chest - 2nd row female`,
            capping_limit_explanation_sec_row_female,
            "String",
            false,
            true
        );
    }
}
