/* Utils.Version was added for REPORTER in Version 21. So based on if this function is present or not we can check which version we are running */
if (typeof Utils.Version != "function") {
    let message = `This workflow is only available for Oasys REPORTER version 21 and above.`;
    if (Batch()) {
        LogError(message);
    } else {
        Window.Message("Workflow not available", message);
    }
    Exit();
}

/**
 * Cross-platform function for the Window.GetFile method. Instead of taking a string, it takes
 * in a list of extensions and constructs the appropriate filter string based on the OS. It also takes
 * the parameters for save and initial.
 * @param {string[]|string} [extensions=[]] - List of file extensions (e.g., [".csv", ".txt"]) or a single extension string (e.g., ".csv")
 * @param {boolean} [save=false] - Whether the dialog is for saving a file
 * @param {string} [initial=""] - Initial file path or directory name. If a file path is provided, the parent directory is used. If a directory is provided, it's used as-is.
 * @returns {string|null} - Selected filename as a string, or null if the user cancels
 */
function GetFile(extensions = [], save = false, initial = "") {
    /* Check that extensions is a string or an array. */
    if (typeof extensions === "string") {
        /* If the user passes a string of delimited values (with ; commas or spaces to separate the extensions) instead of an array,
         * split the string into an array using the appropriate delimiter. The regex splits on commas, semicolons, or spaces. */
        extensions = extensions.split(/[,; ]+/);
    } else if (!Array.isArray(extensions)) {
        LogError("Function GetFile: extensions parameter must be a string or an array of strings.");
        return null;
    }

    // Replace all the *'s at the start of the string (this is generally before the first dot of the extension).
    // We do this because we add the * later on using the joiner variable.
    extensions = extensions.map((ext) => ext.replace(/^\**/, ""));

    const is_linux = Unix();

    /**
     * The filter adds a * before each extension (excluding the first) and joins them with either spaces (Linux) or semicolons (Windows).
     */
    let file, filter;
    const joiner = is_linux ? " *" : "; *";
    const file_separator = is_linux ? "/" : "\\";

    // Adjust initial path for Linux and Windows if needed. On Linux, convert backslashes to forward slashes.
    // On Windows, convert forward slashes to backslashes. This ensures the initial path is in the correct format for the OS.
    if (is_linux && initial) {
        initial = initial.replace(/\\/g, file_separator);
    }

    if (!is_linux && initial) {
        initial = initial.replace(/\//g, file_separator);
    }

    filter = extensions.map((ext) => `${ext}`).join(joiner);

    if (initial && File.Exists(initial)) {
        // For Window.GetFile, if a directory is used but is not properly specified with a / or \\ at the end, then the appropriate
        // delimiter (e.g. forward slash) is added to force it to use the correct directory. Otherwise the function may misinterpret it as a file.
        if (File.IsDirectory(initial) && !initial.endsWith(file_separator)) {
            initial = initial + file_separator;
        }
        file = Window.GetFile(filter, save, initial);
    } else {
        file = Window.GetFile(filter, save);
    }
    return file;
}

/* Contain the entire script within a function because REPORTER only has a single JavaScript realm
 * for the entire session. */
on_load_script();

/**
 * #AAW REPORTER on load script
 *
 * Runs as soon as the template is opened.
 *
 * In batch mode, will proceed to generate if:
 *
 * 1. Keyword file can be found from %KEYWORD_FILE% or via %DEFAULT_DIR%/%DEFAULT_JOB%
 * 2. Valid %OUTPUT_DIR% defined or valid default (subdirectory can be created)
 * 3. Valid %RESULTS_DIR% defined or valid default containing results files
 *
 * Otherwise will terminate generation with a log error.
 *
 * In interactive mode, the script asks the user two questions:
 *
 * 1. Select the keyword file of the job you wish to post-process
 * 2. Use default results and output directories or choose to specify them in PRIMER
 *
 * If proceeding to template generation, the script will deactivate itelf to avoid entering an
 * infinite loop, and it also controls generation and (de)activation of various other items.
 */
function on_load_script() {
    let templ = Template.GetCurrent();

    // try {
    //     // .generating property added in v22
    //     if (templ.generating) {
    //         LogPrint(`Template Generating on load script again so exiting`);
    //         Exit();
    //     }
    // } catch (error) {}

    /** ensure that variables which may be used in the template are created if they don't exist */
    initialise_language_constant_reporter_variables();

    /* Delete AAW_job_control_variables.csv in case it has data from previous runs */
    let reporter_temp = get_expanded_variable_value(templ, `REPORTER_TEMP`);
    if (reporter_temp == null || (reporter_temp != null && !File.IsDirectory(reporter_temp))) {
        finish_script(false, `Could not find REPORTER_TEMP directory.`);
    }
    let f_csv_name = `${reporter_temp}/AAW_job_control_variables.csv`;
    if (File.Exists(f_csv_name)) {
        let success = File.Delete(f_csv_name);
        if (!success) {
            finish_script(false, `Unable to delete existing variables file: ${f_csv_name}`);
        }
    }
    /* Replace it with an empty file so that `#AAW REPORTER read variables from PRIMER job control`
     * item always has something to read. */
    let f_csv = new File(f_csv_name, File.WRITE);
    f_csv.Close();

    /* To begin with, activate all master #AAW page items and deactivate all normal page #AAW items */
    activate_master_page_aaw_items(true);
    activate_normal_page_aaw_items(false);

    /* Generate the subjective modifiers item (if present) to ensure that modifiers are initialised
     * and functions declared in case user aborts generation and then clicks `Set Modifiers`
     * button. */
    generate_item(`#AAW REPORTER subjective modifiers`, false);

    if (!Batch()) {
        /* Generate the CNCAP working condition item (if present) so that the working condition number can be selected for VTC purposes */
        generate_item(`#AAW REPORTER working condition`, false);
        // If not in batch mode, make a variable to identify this
        new Variable(templ, "BATCH", "Batch true/false", Batch() ? "TRUE" : "FALSE", "General", false, true);
    } else {
        // If in batch mode, make a variable to identify this, and pass to multiple model selector to omit T/HIS GUI.
        new Variable(templ, "BATCH", "Batch true/false", Batch() ? "TRUE" : "FALSE", "General", false, true);
        /** in batch mode either SUMMARY_NAME or WORKING_CONDITION is given */
        let working_condition = get_expanded_variable_value(templ, `WORKING_CONDITION`);
        /** if working condition is not defined then it is not relevant to the template as it is defined by default for the relevant templates" */
        if (working_condition != null) {
            let summary_name = get_expanded_variable_value(templ, `WORKING_CONDITION`);
            /** if the working condition contains a dash (i.e. "1 - 6" or "7 - 8") then we assume it has not been set via the command line
             *  so we want to update it based on the the first group of numbers in the value of %SUMMARY_NAME%
             *  e.g. %SUMMARY_NAME% ="Working Condition 1" would set %WORKING_CONDITION% = "1"*/
            if (working_condition.includes("-")) {
                /** working condition is not defined so we can try and extract it from the SUMMARY_NAME */
                if (typeof summary_name == "string") {
                    let working_condition_num_match = summary_name.match(/\d+/);
                    if (working_condition_num_match != null) {
                        working_condition = working_condition_num_match[0];
                        new Variable(
                            templ,
                            "WORKING_CONDITION",
                            "Working condition number used in CNCAP protocol",
                            working_condition,
                            "Number",
                            false,
                            true
                        );
                    }
                }
            } else {
                /** if %WORKING_CONDITION% is defined then we don't require the SUMMARY_NAME argument and we construct it with the
                 * working condition language constant and the working condition number. If %SUMMARY_NAME% is also defined then
                 * print a warning message to let the user know that %SUMMARY_NAME% will be ignored */
                new Variable(
                    templ,
                    "SUMMARY_NAME",
                    "Working condition name for summary template",
                    "%[WORKING_CONDITION]% %WORKING_CONDITION%",
                    "General",
                    false,
                    true
                );
                let new_summary_name = get_expanded_variable_value(templ, `SUMMARY_NAME`);
                if (summary_name != null && summary_name != new_summary_name) {
                    LogWarning(
                        `WORKING_CONDITION command argument overrides SUMMARY_NAME which has been set to "${new_summary_name}"` +
                            ` instead of using the value from command line argument ("${summary_name}") which will be ignored.`
                    );
                }
            }
        }
    }
    if (get_expanded_variable_value(templ, `SUMMARY_NAME`) == "CANCEL") {
        finish_script(false, `Working condition not selected. Report will not be generated.`);
    } else if (get_expanded_variable_value(templ, `SUMMARY_NAME`) == null) {
        /* Create an empty working condition variable so we don't get an error in templates where the working condition is not used */
        new Variable(templ, "SUMMARY_NAME", "Not used", "", "String", false, true);
    }

    if (!Batch()) {
        /* Generate the test file item (if present) so the test file can be selected for VTC purposes */
        generate_item(`#AAW REPORTER test file`, false);
    } else {
        // To support both names in Batch: in AA the 'TEST_FILE' is same as SimVT 'TEST_DATA_PATH' 3-Sep-2025
        // Take care of all scenarios where user may pass any one of the 2 variables in batch
        // This will happen in templates where both AA and SimVT are run (combined templates)
        // e.g. automotive_assessments\CNCAP_2024_Far_Side_VTC_O2O_LHS_POLE_ES2RE_Passenger_CN.ort
        let test_file = get_expanded_variable_value(templ, `TEST_FILE`);
        let test_data_file = get_expanded_variable_value(templ, `TEST_DATA_PATH`);
        if (test_file && test_data_file && test_file != test_data_file) {
            LogWarning(
                `Both %TEST_FILE% and %TEST_DATA_PATH% defined. Overriding value of %TEST_DATA_PATH% with ${test_file}`
            );
            new Variable(templ, "TEST_DATA_PATH", "Test file", test_file, "File(absolute)", false, true);
        } else if (test_file && !test_data_file) {
            new Variable(templ, "TEST_DATA_PATH", "Test file", test_file, "File(absolute)", false, true);
        } else if (!test_file && test_data_file) {
            new Variable(templ, "TEST_FILE", "Test file", test_data_file, "File(absolute)", false, true);
        } else if (test_file && test_data_file && test_file == test_data_file) {
            LogPrint(`Both %TEST_FILE% and %TEST_DATA_PATH% are the same path`);
        } else if (!test_file && !test_data_file) {
            LogWarning(`Both %TEST_FILE% and %TEST_DATA_PATH% are missing`);
            new Variable(templ, "TEST_FILE", "Not used", "", "String", false, true);
            new Variable(templ, "TEST_DATA_PATH", "Not used", "", "String", false, true);
        }
    }
    if (get_expanded_variable_value(templ, `TEST_FILE`) == "CANCEL") {
        finish_script(false, `Test file not selected. Report will not be generated.`);
    } else if (get_expanded_variable_value(templ, `TEST_FILE`) == null) {
        /* Create an empty test_file variable so we don't get an error in templates where the test file is not used */
        new Variable(templ, "TEST_FILE", "Not used", "", "String", false, true);
    }

    if (get_expanded_variable_value(templ, `RED_ZONE`) == null) {
        /* Create an empty red_zone variable so we don't get an error in templates where the red zone is not used */
        new Variable(templ, "RED_ZONE", "Not used", "", "String", false, true);
    }

    // In multiple models templates, we want to pass a models list file in Batch
    let lst = Variable.GetFromName(templ, `MODELS_LST`);
    // If the variable of models.lst file path has not been passed in, set as null
    if (!lst) {
        new Variable(templ, "MODELS_LST", "Multiple models list", null, "General", false, true);
    }
    /* Generate the multiple models item (if present) to ensure that the user can select multiple models if required */
    let output_dir = get_expanded_variable_value(templ, `OUTPUT_DIR`);
    if (!output_dir) {
        new Variable(templ, "OUTPUT_DIR", "Results output directory", null, "General", false, true);
    }
    get_multiple_file_inputs(reporter_temp, output_dir);

    /* Get keyword file and set %DEFAULT_DIR% and %DEFAULT_JOB% */
    if (!get_keyword_file()) {
        finish_script(false, `Keyword file not selected. Report will not be generated.`);
    } else {
        configure_results_and_outputs();
        finish_script(true);
    }
}

/**
 * some templates may require multiple models/files to be loaded
 * this will be the case if "#AAWD T/HIS Multiple Model Selector" T/HIS Item is present
 * If this is the case then we do not ask the normal questions but go straight to the GUI
 * in T/HIS and we create the REPORTER variables for from the GUI inputs
 * @param {String} reporter_temp REPORTER_TEMP directory path
 * @param {String} output_dir OUTPUT_DIR directory path
 */
function get_multiple_file_inputs(reporter_temp, output_dir) {
    let templ = Template.GetCurrent();

    if (get_item_by_name(`#AAWD T/HIS Multiple Model Selector`) != null) {
        if (Batch()) {
            create_output_dir_if_missing(output_dir);
            let models_lst_file = get_expanded_variable_value(templ, `MODELS_LST`);
            if (!models_lst_file) {
                finish_script(false, `-varMODELS_LST command line argument must be provided in batch mode`);
            } else if (!File.Exists(models_lst_file)) {
                finish_script(
                    false,
                    `-varMODELS_LST file provided in command line argument does not exist: ${models_lst_file}`
                );
            }
        }
        /** ensure that the assesment item exists first as we will need it to update the .job property*/
        let this_assessment_item = get_item_by_name(`#AAW T/HIS check and do assessment`);
        let d3plot_assessment_item = get_item_by_name(`#AAW D3PLOT do assessment`);

        generate_item(`#AAWD T/HIS Multiple Model Selector`, true);
        /** check if the JSON exists in the REPORTER temp directory */
        let filename = `${reporter_temp}/multiple_models_selector.json`;
        if (File.Exists(filename)) {
            /** read the JSON file and set the variables in the template */

            /**
             * Read JSON file and return the parsed object
             * (suitable for REPORTER scripts which does not support <File>.ReadAll())
             * @param {String} filename
             * @returns
             */
            function REPORTER_ReadJSON(filename) {
                /** read the JSON file and set the variables in the template */
                let json = new File(filename, File.READ);
                let line;
                let json_string_array = [];

                //@ts-ignore
                while ((line = json.ReadLine()) != File.EOF) {
                    json_string_array.push(line);
                }

                json.Close();
                let json_string = json_string_array.join("\n");
                return JSON.parse(json_string);
            }

            // commented out as importing not used yet /** @type {MultipleModelSelector} */
            let json = REPORTER_ReadJSON(filename);
            /** set the output directory */
            let output_dir_var = new Variable(
                templ,
                `OUTPUT_DIR`,
                `Directory where images and other output files will be written`,
                json.output_dir,
                `Directory`,
                false,
                true
            );

            /** generate the job file path by joining all the file paths */
            let model_paths = [];
            let primer_model_paths = [];
            let this_model_paths = [];
            let d3plot_model_paths = [];
            let model_obj;
            let output_dir = get_expanded_variable_value(templ, `OUTPUT_DIR`);
            let success = false; // If no valid file path in any of 8 models

            // case_num will be "CASE_1", "CASE_2", etc.
            for (let case_obj of json.cases) {
                let case_num = case_obj.case_num;
                let file_path = case_obj.path && case_obj.valid ? case_obj.path : null;

                let keyword_file_path_var = new Variable(
                    templ,
                    `${case_num}_MODEL_PATH`, // was M${i}_MODEL_PATH
                    `${case_num} *KEYWORD file path - prefix:"${case_obj.prefix}"`,
                    file_path || "N/A",
                    `File(absolute)`,
                    false,
                    true
                );

                // skip if the file path is null (after creating the variable)
                if (!file_path) continue;

                let file_path_is_keyword = /.*\.k(ey)?$/i.test(file_path);
                if (file_path_is_keyword) primer_model_paths.push(file_path);

                success = true;
                model_paths.push(file_path);

                // Create THIS path variable for each CASE_#
                if (case_obj.this_file) this_model_paths.push(case_obj.this_file);

                // if the filepath is not a keyword or this_assessment_item does not exists then THIS is irrelevant so use "N/A"
                let this_fallback_value = file_path_is_keyword && this_assessment_item ? "Missing" : "N/A";

                let this_model_paths_var = new Variable(
                    templ,
                    `${case_num}_THIS_PATH`, // was `THIS_WC${i}_PATH`,
                    `${case_num} THIS file path: prefix:"${case_obj.prefix}"`,
                    case_obj.this_file || this_fallback_value,
                    `File(absolute)`,
                    false,
                    true
                );

                // if the filepath is not a keyword or d3plot_assessment_item does not exists then D3PLOT is irrelevant so use "N/A"
                let d3plot_fallback_value = file_path_is_keyword && d3plot_assessment_item ? "Missing" : "N/A";

                if (case_obj.d3plot_file) d3plot_model_paths.push(case_obj.d3plot_file);
                let d3plot_model_paths_var = new Variable(
                    templ,
                    `${case_num}_D3PLOT_PATH`, // was `D3PLOT_M${i}_MODEL_PATH`
                    `${case_num} D3PLOT file path: prefix:"${case_obj.prefix}"`,
                    case_obj.d3plot_file || d3plot_fallback_value,
                    `File(absolute)`,
                    false,
                    true
                );
            }

            /** update the job file path in the assessment item so that multiple models are loaded*/
            if (this_assessment_item == null && d3plot_assessment_item == null) {
                finish_script(false, `No T/HIS or D3PLOT assessment items found to update job file path.`);
            }

            if (this_assessment_item != null) {
                /** if the assessment item does not exist then we cannot proceed with the template generation */
                this_assessment_item.job = `"${this_model_paths.join('" "')}"`;
            }

            if (d3plot_assessment_item != null) {
                /** update the job file path in the D3PLOT assessment item so that multiple models are loaded*/
                // 30-Jul-2025: Because d3plot will process multiple model in sequence, we start with M1
                //              Later in finish_script function, we loop the D3PLOT do assessment item to go through M2, M3, ...
                if (Variable.GetFromName(templ, `D3PLOT_M1_MODEL_PATH`) != null) {
                    d3plot_assessment_item.job = `%D3PLOT_M1_MODEL_PATH%`;
                } else {
                    finish_script(false, `No D3PLOT assessment items found to update job file path.`);
                }
            }
            if (success == true) {
                let job_control = `Run`;
                LogPrint(
                    `${model_paths.length} models have been loaded successfully.\nThe job control status is set to "${job_control}".`
                );

                let model_paths_var = new Variable(
                    templ,
                    `PRIMER_MODEL_PATHS`,
                    `JSON array of model paths for the PRIMER workflow.`,
                    JSON.stringify(model_paths),
                    `String`,
                    false,
                    true
                );

                let job_control_var = new Variable(
                    templ,
                    `JOB_CONTROL`,
                    `Controls "Check", "Run", or "Skip" for various items, or "Abort" template generation entirely.`,
                    job_control,
                    `String`,
                    false,
                    true
                );
                /* Activate or deactivate `#AAW PRIMER job control GUI` item depending on whether or not we
                 * want to show it. Always deactivate `#AAW PRIMER user data GUI`, which will be optionally
                 * generated by `#AAW REPORTER run PRIMER rerun T/HIS` item. */
                activate_master_page_item(`#AAW PRIMER job control GUI`, false);
                activate_master_page_item(`#AAW PRIMER user data GUI`, false);
                /** may need to change the JOB control status... */
                finish_script(true);
            } else {
                // If no valid file path in any of 8 models
                finish_script(false, `No valid file path in any of 8 models.`);
            }
        } else {
            /** if in batch mode then we are expecting a lst file to be provided
             *  if the %MODELS_LST% is
             */
            if (Batch()) {
                let models_lst_file = get_expanded_variable_value(templ, `MODELS_LST`);
                finish_script(
                    false,
                    `-varMODELS_LST file provided in command line argument was not valid: ${models_lst_file}`
                );
            }
            /** If the JSON file does not exist (i.e. the user closed the GUI or T/HIS prematurely), then we cannot proceed with the template generation*/
            finish_script(false, `No model/data source files provided.`);
        }
    }
}

/**
 * Gets the keyword file and sets %KEYWORD_FILE%, %DEFAULT_DIR% and %DEFAULT_JOB%.
 * Returns whether successful (true/false).
 *
 * REPORTER variables for KEYWORD_FILE and DEFAULT_DIR/DEFAULT_JOB might already be defined if:
 * 1. They have been hardwired (possibly)
 * 2. They have been passed through as arguments in batch
 * 3. The template is being re-run.
 *
 * In batch mode, the template only needs one of KEYWORD_FILE or (DEFAULT_DIR and DEFAULT_JOB) to
 * proceed. If running from SHELL, it is convenient to use DEFAULT_DIR and DEFAULT_JOB, whereas it
 * may be more convenient to define KEYWORD_FILE if running from the command line.
 *
 * In interactive mode, the user is always prompted to specify the keyword file, regardless of
 * whether the variables are already defined. We could change the logic here if we wanted so that
 * the template runs automatically in interactive mode if all variables are defined, but we think
 * it is more helpful if the user is prompted, in case they want to change any inputs that have
 * been deliberately or accidentally saved previously.
 *
 * @returns {Boolean}
 * @example
 * let success = get_keyword_file();
 */
function get_keyword_file() {
    let templ = Template.GetCurrent();

    let keyword_file;
    if (Batch()) {
        /* First check to see if we can find a keyword file.
         * We check KEYWORD_FILE first, and then DEFAULT_DIR and DEFAULT_JOB.
         * We make sure they're updated to be consistent with each other.
         */
        keyword_file = get_expanded_variable_value(templ, `KEYWORD_FILE`);
        let found_key_file = false;
        let default_dir;
        let default_job;
        if (keyword_file != null && File.Exists(keyword_file)) {
            LogPrint(`Found keyword file: ${keyword_file}`);
            found_key_file = true;
            /* Got a file, so set %KEYWORD_FILE%, %DEFAULT_DIR% and %DEFAULT_JOB% */
            set_keyword_file_variables(keyword_file);
        } else {
            if (keyword_file == null) {
                LogPrint(`REPORTER Variable %KEYWORD_FILE% was not defined.`);
            } else if (!File.Exists(keyword_file)) {
                LogPrint(
                    `Keyword file specified by REPORTER Variable %KEYWORD_FILE% could not be found: ${keyword_file}`
                );
            }
            LogPrint(`Searching for keyword file via %DEFAULT_DIR% and %DEFAULT_JOB% instead...`);
            /* If KEYWORD_FILE doesn't exist, check DEFAULT_DIR and DEFAULT_JOB. */
            default_dir = get_expanded_variable_value(templ, `DEFAULT_DIR`);
            default_job = get_expanded_variable_value(templ, `DEFAULT_JOB`);

            if (default_dir != null && File.IsDirectory(default_dir) && default_job != null) {
                let possible_extensions = [
                    `k`,
                    `key`,
                    `kby`,
                    `k.gz`,
                    `key.gz`,
                    `kby.gz`,
                    `k.zip`,
                    `key.zip`,
                    `kby.zip`
                ];
                for (let ext of possible_extensions) {
                    keyword_file = `${default_dir}/${default_job}.${ext}`;
                    if (File.Exists(keyword_file)) {
                        LogPrint(`Found keyword file: ${keyword_file}`);
                        found_key_file = true;
                        /* Got a file, so set %KEYWORD_FILE%, %DEFAULT_DIR% and %DEFAULT_JOB% */
                        set_keyword_file_variables(keyword_file);
                        break;
                    }
                }
                if (!found_key_file) {
                    LogPrint(
                        `Could not find a keyword file via %DEFAULT_DIR% and %DEFAULT_JOB% using any of the possible extentions "${possible_extensions.join(
                            `", "`
                        )}".`
                    );
                }
            } else {
                if (default_dir == null) {
                    LogPrint(`REPORTER Variable %DEFAULT_DIR% was not defined.`);
                } else if (!File.IsDirectory(default_dir)) {
                    LogPrint(`Directory specified by REPORTER Variable %DEFAULT_DIR% could not be found.`);
                }
                if (default_job == null) {
                    LogPrint(`REPORTER Variable %DEFAULT_JOB% was not defined.`);
                }
            }
        }

        if (!found_key_file) {
            return false;
        }
    } else {
        let ans = Window.Message(
            `Select keyword file`,
            `Select the LS-DYNA keyword file of the job you wish to post-process.`,
            Window.OK | Window.CANCEL
        );
        if (ans == Window.CANCEL) {
            return false;
        }
        keyword_file = GetFile([".k", ".key", ".dyn"]);

        if (!keyword_file) {
            /* User clicked cancel */
            return false;
        }

        /* Got a file, so set %KEYWORD_FILE%, %DEFAULT_DIR% and %DEFAULT_JOB% */
        set_keyword_file_variables(keyword_file);
    }

    return true;
}

/**
 * Sets the REPORTER variables %KEYWORD_FILE%, %DEFAULT_DIR%, and %DEFAULT_JOB% to match the
 * specified keyword file.
 * @param {string} keyword_file Absolute path and filename of keyword file
 */
function set_keyword_file_variables(keyword_file) {
    let templ = Template.GetCurrent();

    /* If keyword_file is relative path, make it absolute path. We need to do this so that
     * any other variables that end up being derived from it (DEFAULT_DIR, RESULTS_DIR, OUTPUT_DIR,
     * etc.) are absolute paths too. When REPORTER generates an Oasys item, it sets -start_in to
     * the job file directory, so it gets confused if the relative paths we have defined are
     * relative to a different starting directory. Converting everything to absolute paths avoids
     * this problem. */
    keyword_file = convert_to_absolute_path(keyword_file, `%KEYWORD_FILE%`);

    let path_index = Math.max(keyword_file.lastIndexOf("/"), keyword_file.lastIndexOf("\\"));
    let default_dir = keyword_file.substring(0, path_index);
    let filename = keyword_file.substring(path_index + 1);
    let default_job = filename.substring(0, filename.lastIndexOf("."));

    /* Assign to REPORTER variables. Constructor will overwrite existing variable.
     * KEYWORD_FILE should be temporary; DEFAULT_DIR and DEFAULT_JOB are not temporary. */
    let keyword_file_var = new Variable(
        templ,
        `KEYWORD_FILE`,
        `Keyword file`,
        keyword_file,
        `File(absolute)`,
        false,
        true
    );
    let default_dir_var = new Variable(
        templ,
        `DEFAULT_DIR`,
        `Reporter default directory`,
        default_dir,
        `Directory`,
        false,
        false
    );
    let default_job_var = new Variable(
        templ,
        `DEFAULT_JOB`,
        `Reporter default jobname`,
        default_job,
        `File(basename)`,
        false,
        false
    );
}

/**
 * Configures and checks the results and output directories.
 *
 * By default, RESULTS_DIR and OUTPUT_DIR can be inferred from the keyword file location.
 * However, the user may wish to specify different locations for RESULTS_DIR and OUTPUT_DIR,
 * in which case these are configured by the subsequent `#AAW PRIMER job control GUI` item.
 * Depending on the user's preference, this script activates/deactivates that item.
 *
 * In interactive mode, if hte user chooses to use the defaults but there are issues, they are
 * prompted to use the PRIMER GUI to rectify.
 *
 * In batch mode, supplied values of RESULTS_DIR and OUTPUT_DIR will be used if provided, or it
 * will attempt to use the defaults. Generation will terminate with a log error if there are any
 * issues.
 */
function configure_results_and_outputs() {
    let templ = Template.GetCurrent();

    /* By default, show PRIMER job control GUI */
    let show_job_control_gui = true;
    if (Batch()) {
        /* If we are running in batch, always skip job control GUI. */
        show_job_control_gui = false;
    }

    /* Default OUTPUT_DIR is in subdirectory matching name of template. */
    let output_dir_name = templ.filename.substring(0, templ.filename.lastIndexOf(`.`));

    /* DEFAULT_DIR should already have been defined in get_keyword_file() */
    let default_dir = get_expanded_variable_value(templ, `DEFAULT_DIR`);
    if (default_dir == null) {
        finish_script(false, `Could not find REPORTER Variable %DEFAULT_DIR% in configure_results_and_outputs().`);
    }

    /* Set initial answer to NO so that in Batch mode, RESULTS_DIR and OUTPUT_DIR only reset if not
     * provided. */
    let use_default_directories = Window.NO;

    /* In interactive mode, give user option */
    if (!Batch()) {
        use_default_directories = Window.Question(
            `Results and output directories`,
            `By default, REPORTER will search for results in the same directory as the keyword file, ` +
                `and write images and other files to a subdirectory named "${output_dir_name}".\n\n` +
                `Proceed with these defaults (Yes)? Or configure directories in PRIMER (No).`
        );
    }

    /* In Batch mode, if %RESULTS_DIR% and %OUTPUT_DIR% have been provided, we will use them.
     * Otherwise, we create them with default values.
     *
     * In interactive mode, if the user said YES, then we always reset to default values. If they
     * said NO, use values if provided, otherwise use defaults - and show PRIMER GUI. */

    let results_dir = get_expanded_variable_value(templ, `RESULTS_DIR`);
    if (results_dir == null || results_dir == "" || use_default_directories == Window.YES) {
        results_dir = default_dir;
        if (Batch()) {
            LogPrint(`REPORTER variable %RESULTS_DIR% unspecified. Setting default %RESULTS_DIR% = ${results_dir}`);
        }
        let results_dir_var = new Variable(
            templ,
            `RESULTS_DIR`,
            `Directory containing LS-DYNA results`,
            `%DEFAULT_DIR%`,
            `Directory`,
            false,
            true
        );
    } else {
        /* If results_dir was provided, make sure it is an absolute path */
        results_dir = convert_to_absolute_path(results_dir, `%RESULTS_DIR%`);
        let results_dir_var = new Variable(
            templ,
            `RESULTS_DIR`,
            `Directory containing LS-DYNA results`,
            results_dir,
            `Directory`,
            false,
            true
        );
    }

    let output_dir = get_expanded_variable_value(templ, `OUTPUT_DIR`);
    if (output_dir == null || output_dir == "" || use_default_directories == Window.YES) {
        output_dir = `${default_dir}/${output_dir_name}`;
        if (Batch()) {
            LogPrint(`REPORTER variable %OUTPUT_DIR% unspecified. Setting default %OUTPUT_DIR% = ${output_dir}`);
        }
        let output_dir_var = new Variable(
            templ,
            `OUTPUT_DIR`,
            `Directory where images and other output files will be written`,
            `%DEFAULT_DIR%/${output_dir_name}`,
            `Directory`,
            false,
            true
        );
    } else {
        /* If output_dir was provided, make sure it is an absolute path */
        output_dir = convert_to_absolute_path(output_dir, `%OUTPUT_DIR%`);
        let output_dir_var = new Variable(
            templ,
            `OUTPUT_DIR`,
            `Directory where images and other output files will be written`,
            output_dir,
            `Directory`,
            false,
            true
        );
    }

    if (Batch() || use_default_directories == Window.YES) {
        let output_dir_exists = create_output_dir_if_missing(output_dir);

        /* Find T/HIS results files in RESULTS_DIR and set RESULTS_FILE_THIS */
        let results_file_this_exists = true;

        let default_job = get_expanded_variable_value(templ, `DEFAULT_JOB`);
        let results_file_this = find_lsdyna_files(results_dir, default_job, `T/HIS`);
        if (results_file_this == null) {
            results_file_this = ``;
            results_file_this_exists = false;
            if (Batch()) {
                finish_script(false, `Could not find T/HIS results files in results directory: ${results_dir}`);
            } else {
                Window.Warning(
                    `Results file`,
                    `Could not find T/HIS results files in results directory. Please configure directories in PRIMER.\n\n${results_dir}`
                );
            }
        } else {
            LogPrint(`Found T/HIS results file: ${results_file_this}`);
        }
        let results_file_this_var = new Variable(
            templ,
            `RESULTS_FILE_THIS`,
            `Results file for T/HIS`,
            results_file_this,
            `File(absolute)`,
            false,
            true
        );

        /* Find D3PLOT results files in RESULTS_DIR and set RESULTS_FILE_D3PLOT */
        let results_file_d3plot_exists = true;

        let results_file_d3plot = find_lsdyna_files(results_dir, default_job, `D3PLOT`);
        if (results_file_d3plot == null) {
            results_file_d3plot = ``;
            results_file_d3plot_exists = false;
            if (Batch()) {
                //finish_script(false, `Could not find D3PLOT results files in results directory: ${results_dir}`);
            } else {
                /* Commenting out as most templates don't need D3PLOT results. Need to think of a way for the template to check if they're needed or not. */
                //Window.Warning(
                //    `Results file`,
                //    `Could not find D3PLOT results files in results directory. Please configure directories in PRIMER.\n\n${results_dir}`
                //);
            }
        } else {
            LogPrint(`Found D3PLOT results file: ${results_file_d3plot}`);
        }
        let results_file_d3plot_var = new Variable(
            templ,
            `RESULTS_FILE_D3PLOT`,
            `Results file for D3PLOT`,
            results_file_d3plot,
            `File(absolute)`,
            false,
            true
        );

        /* No need to show PRIMER job control GUI if we have all the information we need */
        if (output_dir_exists && results_file_this_exists) {
            show_job_control_gui = false;
        }
    }

    /* If running in batch, we skip the check in `#AAW T/HIS check and do assessment` and proceed
     * directly to doing the assessment with whatever user data we can find. If running
     * interactively, regardless of whether we are showing the PRIMER job control GUI, set
     * JOB_CONTROL to `Check` in preparation for `#AAW T/HIS check and do assessment` item. This
     * will be overridden by the PRIMER job control GUI if it is shown and the user clicks `Cancel`
     * (which would cause an abort). */
    let job_control = `Check`;
    if (Batch()) job_control = `Run`;
    let job_control_var = new Variable(
        templ,
        `JOB_CONTROL`,
        `Controls "Check", "Run", or "Skip" for various items, or "Abort" template generation entirely.`,
        job_control,
        `String`,
        false,
        true
    );

    /* Activate or deactivate `#AAW PRIMER job control GUI` item depending on whether or not we
     * want to show it. Always deactivate `#AAW PRIMER user data GUI`, which will be optionally
     * generated by `#AAW REPORTER run PRIMER rerun T/HIS` item. */
    activate_master_page_item(`#AAW PRIMER job control GUI`, show_job_control_gui);
    activate_master_page_item(`#AAW PRIMER user data GUI`, false);
    activate_master_page_item(`#AAW PRIMER cor user data GUI`, false, false); // required for JSS Chapter 4 combined template
    activate_master_page_item(`#AAW PRIMER user data GUI_1`, false, false); // required for CNCAP 2024 Far Side VTC O2O RHS POLE template
    activate_master_page_item(`#AAW PRIMER user data GUI_2`, false, false); // required for CNCAP O2O Picture template
}

/**
 *
 * @returns attempts to the
 */
function create_output_dir_if_missing(output_dir) {
    /* If it doesn't already exist, check that we can create the output_dir */
    let output_dir_exists = true;
    if (!File.IsDirectory(output_dir)) {
        LogPrint(`Output directory doesn't yet exist. Creating...`);
        let success = File.Mkdir(output_dir);
        if (success) {
            LogPrint(`Created output directory: ${output_dir}`);
        } else {
            output_dir_exists = false;

            if (Batch()) {
                finish_script(false, `Unable to create output directory: ${output_dir}`);
            } else {
                Window.Warning(
                    `Output directory`,
                    `Unable to create output directory. Please configure directories in PRIMER.\n\n${output_dir}`
                );
            }
        }
    }
    return output_dir_exists;
}

/**
 * Finds all the variables defined in the template and sets them to empty string if they are missing
 * to prevent error from being thrown when generating the template.
 * Looks at the master page first, then all the normal pages.
 */
function catch_all_missing_variables() {
    let template = Template.GetCurrent();
    /* Loop through all pages in the template, searching for relevant items */
    let pages = template.GetAllPages();
    /* Add Master page to front so we consider it first */
    let variables_found = {};
    pages.unshift(template.GetMaster());

    function extract_variables_from_text(text) {
        /** text could be %VAR_1%   %VAR_2%%VAR_3(2F)%%A(0G)% */
        let regex = /%([^%\n]+?)(?:\(..+\))?%/g;
        let matches;
        while ((matches = regex.exec(text)) !== null) {
            let var_name = matches[1]; // Use capture group directly
            /** set value to false initially as we have not checked if the variable exists */
            variables_found[var_name] = false;
        }
    }

    for (let p = 0; p < pages.length; p++) {
        let items = pages[p].GetAllItems();
        for (let item of items) {
            /** extract any reporter variables from the item - only do this for tables, textboxes and text*/
            switch (item.type) {
                case Item.TABLE:
                    /** loop over all cells and rows in the table */

                    for (let row = 0; row < item.rows; row++) {
                        for (let col = 0; col < item.columns; col++) {
                            let cell_text = item.GetCellProperties(row, col).text;
                            extract_variables_from_text(cell_text);
                        }
                    }

                    break;
                case Item.TEXTBOX:
                case Item.TEXT:
                    /** extract variables from the text in the textbox */
                    extract_variables_from_text(item.text);
                    break;
                default:
                    break;
            }
        }
    }

    /** now we have an object with all the variables it found we can check which ones do not exist */
    let count = 0;
    for (let var_name in variables_found) {
        if (template.GetVariableValue(var_name) == null) {
            /** create the missing variable */
            new Variable(template, var_name, "Missing variable");
            count++;
        } else {
            variables_found[var_name] = true;
        }
    }

    if (count) LogWarning(`Missing ${count} REPORTER variables - they have been created with empty values.`);
}

/**
 * Common function for finishing the script.
 * Complete the generation of the template if we have all the required information.
 * @param {Boolean} can_generate Whether or not to generate the template
 * @param {string} [msg] Error message (provide when not generating)
 * @example
 * finish_script(Template.GetCurrent(), false, `Keyword file not provided`);
 */
function finish_script(can_generate, msg) {
    /* Complete the template generation if we have the required information,
     * otherwise end with message */
    if (can_generate) {
        /* Deactivate this script item to avoid entering an infinite loop */
        activate_master_page_item(`#AAW REPORTER on load script`, false);
        /* Also deactivate a few items we have already generated */
        activate_master_page_item(`#AAW REPORTER subjective modifiers`, false, false);
        activate_master_page_item(`#AAW REPORTER working condition`, false, false);
        activate_master_page_item(`#AAW REPORTER test file`, false, false);
        activate_master_page_item(`#AAWD T/HIS Multiple Model Selector`, false, false);

        activate_master_page_item(`#AAW SIMVT REPORTER on load script`, true, false);

        let templ = Template.GetCurrent();
        /** before generating, loop over all REPORTER variables and set them to empty string if they are missing to prevent
         * error from being thrown when generating the template
         */

        /** loop over all items */
        catch_all_missing_variables();
        templ.Generate();

        /** replace any hard coded language constants in the template with the corresponding REPORTER variable */
        replace_language_constants();

        /** there is an issue importing the language translation variables which have this format %[....]%
         * when using the auto-table item in the summary template. To get around this issue for now we expand all
         * the variable value strings so that language translation variables are inserted literally rather than
         * by reference to another REPORTER variable
         */

        let template_has_summary_variables = false;

        for (let rep_var of Variable.GetAll(templ)) {
            if (rep_var.name.startsWith("SUMMARY_")) {
                template_has_summary_variables = true;
                rep_var.value = get_unicode_text_with_padding(get_expanded_variable_value(templ, rep_var.name));
            }
        }

        /** If the template has summary variables then we assume that a reporter_variables file is writted
         * IMPORTANT reporter_variables file is only written when the template is generated
         * but we need the variables to be updated/created before writing reporter_variables file
         * so the workaround it to call templ.Generate() once to update/created the variables
         * and then (if required) call templ.Generate() again after replace_language_constants() and replacing summary variables.
         * To prevent being stuck in a loop or regenerating any script items unnecessarily we deactivating all the script items first
         */
        if (template_has_summary_variables) {
            let items_to_reactivate = deactivate_all_items_and_return_arry_with_items_that_are_currently_active();
            templ.Generate();
            for (let item of items_to_reactivate) {
                item.active = true; // Reactivate all items that were active before
            }
        }
    } else {
        if (Batch()) {
            LogError(msg);
        } else {
            Window.Message(`Report automation`, msg);
        }
    }

    /* Whether or not we generated, deactivate all the script items except this script item. */
    activate_all_aaw_items(false);
    activate_master_page_item(`#AAW REPORTER on load script`, true);

    // TODO
    // Reactivate buttons on modifiers page

    //reactivate_modifier_buttons();

    Exit();
}

/**
 * Activates or deactivates #AAW (Automotive Assessment Workflow) items on all pages (master page
 * and normal pages) so they are only generated when required.
 * @param {boolean} active Whether to activate or deactivate the items
 * @example
 * activate_all_aaw_items(false);
 */
function activate_all_aaw_items(active) {
    activate_master_page_aaw_items(active);
    activate_normal_page_aaw_items(active);
}

/**
 * Activates or deactivates #AAW (Automotive Assessment Workflow) items on normal pages so that
 * they are only generated when required.
 * @param {boolean} active Whether to activate or deactivate the items
 * @example
 * activate_normal_page_aaw_items(false);
 */
function activate_normal_page_aaw_items(active) {
    let templ = Template.GetCurrent();

    /* We will search for items with names beginning with an identifiable string */
    let aaw_item_str = `#AAW`;

    /* Search for relevant items on other pages */
    let pages = templ.GetAllPages();
    for (let page of pages) {
        let items = page.GetAllItems();
        for (let item of items) {
            if (item.name.substring(0, aaw_item_str.length) == aaw_item_str) {
                /* (De)activate all #AAW items */
                item.active = active;
            }
        }
    }
}

function deactivate_all_items_and_return_arry_with_items_that_are_currently_active() {
    let templ = Template.GetCurrent();
    let items = templ.GetMaster().GetAllItems();
    let pages = templ.GetAllPages();
    for (let page of pages) {
        items = items.concat(page.GetAllItems());
    }
    let active_items = [];
    for (let item of items) {
        if (item.active) active_items.push(item);
        item.active = false; // Deactivate all items
    }
    return active_items;
}

/**
 * Activates or deactivates #AAW (Automotive Assessment Workflow) master page items so that they
 * are only generated once (they sit on the Master page so would be generated for every page otherwise).
 * @param {boolean} active Whether to activate or deactivate the items
 * @example
 * activate_master_page_aaw_items(false);
 */
function activate_master_page_aaw_items(active) {
    let templ = Template.GetCurrent();

    /* We will search for items with names beginning with an identifiable string */
    let aaw_item_str = `#AAW`;

    /* Search for relevant items on the Master page */
    let master = templ.GetMaster();

    let items = master.GetAllItems();

    for (let item of items) {
        if (item.name.substring(0, aaw_item_str.length) == aaw_item_str) {
            /* (De)activate all #AAW items */
            item.active = active;
        }
    }
}

/**
 * Activates or deactivates a specific item on the Master page so that it is only generated when
 * required.
 * @param {string} item_name The item name (the first item matching this name will be actioned)
 * @param {boolean} active Whether to activate or deactivate the item
 * @param {boolean} [expected = true] If true (default), will exit with an error if the requested item cannot be found
 * @example
 * activate_master_page_item(`#AAW REPORTER on load script`, false);
 */
function activate_master_page_item(item_name, active, expected = true) {
    let templ = Template.GetCurrent();

    /* Syntax for log messages */
    let de = `de`;
    if (active) de = ``;

    /* Search for relevant items on the Master page */
    let master = templ.GetMaster();

    let match = false;
    let items = master.GetAllItems();
    for (let item of items) {
        /* Deactivate this script item */
        if (item.name.substring(0, item_name.length) == item_name) {
            item.active = active;
            match = true;
            LogPrint(`Successfully ${de}activated master page item "${item_name}".`);
            break;
        }
    }

    if (!match && expected) {
        LogError(`Could not find master page item "${item_name}" to ${de}activate it.`);
        Exit();
    }
}

/**
 * Finds the first item in the template with the specified <item_name> and generates it.
 * Looks at the master page first, then all the normal pages.
 * @param {string} item_name The name of the item to be generated
 * @param {boolean} [expected = true] If true (default), will exit with an error if the requested item cannot be found
 * @example
 * generate_item(`#AAW REPORTER read variables from PRIMER job control`);
 */
function generate_item(item_name, expected = true) {
    let template = Template.GetCurrent();
    /* Loop through all pages in the template, searching for relevant items */
    let pages = template.GetAllPages();
    /* Add Master page to front so we consider it first */
    pages.unshift(template.GetMaster());
    let match = false;
    for (let p = 0; p < pages.length; p++) {
        let items = pages[p].GetAllItems();
        for (let item of items) {
            if (item.name == item_name) {
                LogPrint(`Found item "${item_name}" on page ${p + 1}. Generating...`);
                match = true;
                /* Activate the item before generating it, then return it to its original status */
                let active = item.active;
                item.active = true;
                item.Generate();
                item.active = active;
                break;
            }
        }
        if (match) break;
    }
    if (!match && expected) {
        LogError(`Could not find item "${item_name}" in order to generate it.`);
        Exit();
    }
}

/**
 * If relative, converts a path to an absolute path based on the current working directory.
 * @param {string} path A path to a directory or filename.
 * @param {string} label The name of the path (e.g. "KEYWORD_FILE") used for print statements.
 */
function convert_to_absolute_path(path, label) {
    if (!File.Exists(path)) {
        LogError(`In function convert_to_absolute_path: specified path for ${label} does not exist: ${path}`);
        return path;
    }
    if (File.IsAbsolute(path)) {
        /* If path is already absolute, just return it. */
        return path;
    } else {
        LogPrint(`Converting ${label} to absolute path...`);
        LogPrint(`Relative path: ${path}`);
        let current_dir = GetCurrentDirectory();
        let abs_path = `${current_dir}/${path}`;
        LogPrint(`Absolute path: ${abs_path}`);
        if (!File.Exists(abs_path)) {
            /* Trap the unexpected case where the conversion hasn't worked. */
            LogError(
                `In function convert_to_absolute_path: converted absolute path does not exist. Reverting to relative path.`
            );
            return path;
        }
        return abs_path;
    }
}

/**
 * Searches a directory for filenames matching either Arup or LSTC filename conventions.
 * Searches for files in directory <dir> of type <file_type>, possibly containing
   <job_name>, and returns the first match in the priority lists below.
   @param {string} dir Directory to search
   @param {string} job_name Root filename to search for
   @param {string} file_type File type to search for (can be "PRIMER", "D3PLOT", "T/HIS", "OTF")
   @returns {?string}
   @example
   let absolute_filename = find_lsdyna_files("C:/my/results/directory", "job_001", "D3PLOT");
 */
function find_lsdyna_files(dir, job_name, file_type) {
    let filters = [];
    let filename;

    switch (file_type) {
        case "PRIMER":
            filters = [
                new RegExp("^" + job_name + ".key$"),
                new RegExp("^" + job_name + ".k$"),
                new RegExp("^" + job_name + ".*.key$"),
                new RegExp("^" + job_name + ".*.k$"),
                new RegExp(".*.key$"),
                new RegExp(".*.k$"),
                new RegExp(".*.dyn$")
            ];

            break;

        case "D3PLOT":
            filters = [
                new RegExp("^" + job_name + ".ptf$"),
                new RegExp("^" + job_name + ".d3plot$"),
                new RegExp("^d3plot$"),
                new RegExp("^" + job_name + ".*.ptf$"),
                new RegExp("^" + job_name + ".*.d3plot$"),
                new RegExp(".*.ptf$"),
                new RegExp(".*.d3plot$"),
                new RegExp(".*.f(em)?z(ip)?$")
            ];
            break;

        case "T/HIS":
            filters = [
                new RegExp("^" + job_name + ".thf$"),
                new RegExp("^" + job_name + ".d3thdt$"),
                new RegExp("^d3thdt$"),
                new RegExp("^" + job_name + ".*.thf$"),
                new RegExp("^" + job_name + ".*.d3thdt$"),
                new RegExp(".*.thf$"),
                new RegExp(".*.d3thdt$"),
                new RegExp("^" + job_name + ".binout.*$"),
                new RegExp("^binout.*$"),
                new RegExp("^" + job_name + ".*.binout.*$"),
                new RegExp(".*.binout.*$")
            ];
            break;

        case "OTF":
            filters = [
                new RegExp("^" + job_name + ".otf$"),
                new RegExp("^" + job_name + ".d3hsp$"),
                new RegExp("^d3hsp$"),
                new RegExp("^" + job_name + ".*.otf$"),
                new RegExp("^" + job_name + ".*.d3hsp$"),
                new RegExp(".*.otf$"),
                new RegExp(".*.d3hsp$")
            ];
            break;

        default:
            LogError(`Unexpected <file_type = "${file_type}" in function find_lsdyna_files.`);
            Exit();
            break;
    }

    let filestore = [];
    for (let filter of filters) {
        let filelist = File.FindFiles(dir, `*`, false);
        for (let file of filelist) {
            if (filter.test(file) == true) {
                filestore.push(file);
            }
        }
        if (filestore.length > 0) {
            filestore.sort();
            /* Pick first matching file and strip off rest of path */
            filename = filestore[0].substring(
                Math.max(filestore[0].lastIndexOf("/"), filestore[0].lastIndexOf("\\")) + 1
            );
            break;
        }
    }

    if (filestore.length == 0) return null;

    let absolute_filename = dir + "/" + filename;

    return absolute_filename;
}

/**
 * Wrapper function to return the expanded variable value, handling the case where it doesn't exist
 * @param {Template} templ The REPORTER template object
 * @param {string} name Variable name you want to get value for
 * @returns {?string}
 */
function get_expanded_variable_value(templ, name) {
    let value = templ.GetVariableValue(name);
    if (value == null) {
        return null;
    } else {
        let expanded_value = templ.ExpandVariablesInString(get_unicode_text_with_padding(value));

        /* recursively expand variables until no more variables are found */
        while (expanded_value != value) {
            LogPrint(`Expanded variable ${name} from "${value}" -> "${expanded_value}"`);
            value = expanded_value;
            expanded_value = templ.ExpandVariablesInString(get_unicode_text_with_padding(value));
        }

        /** there is a bug where the templ.ExpandVariablesInString() does not handle slashes correctly
         * and accidentally puts double slashes in the path.
         * This is not generally a problem except when working with UNC file paths
         * The workaround is to check if it starts with 4 slashes then remove 2 of them
         */
        if (/^[\\/]{4}/.test(expanded_value)) {
            LogPrint(`Removing 2 slashes from the expanded variable value: ${expanded_value}`);
            expanded_value = expanded_value.substring(2); // Remove first two slashes
        }

        return expanded_value;
    }
}

/**
 * Wrapper function for Template.GetVariableValue that also:
 * 1. Checks that "int" and "float" variables are valid
 * 2. Updates a status object with information about any missing or invalid variables
 * 3. Creates a variable for any missing variables with the value "%MISSING%"
 * @param {object} status Status object, used to track any missing or invalid variables
 * @param {string} name Variable name
 * @param {string} [type = "string"] Variable type. Must be "string", "int", or "float".
 * @param {boolean} [required = true] Whether or not the variable is required (true) or optional (false)
 * @returns {*} Variable value
 * @example
 * let head_hic_score = get_variable_value(status, `${m}_${occ}_HEAD_HIC_SCORE`, "float");
 */
function get_variable_value(status, name, type = "string", required = true) {
    let templ = Template.GetCurrent();
    let value = templ.GetVariableValue(name);
    /** if the description contains "Missing variable" then we set value to null
     * so it does not break code which assumes missing variables return null
     * note that description will be null if the variable does not exist hence the check for value != null */
    if (value != null && templ.GetVariableDescription(name).includes("Missing variable")) value = null;

    value = replaceValueWithEnglishLanguageValue(value);
    /** We have N/A here, because when the scoring happens first time and it's null and not required, it gets set to N/A.
     * Next time it's ran, this means it's not null, if it was supposed to be a number and it's been set to N/A from this before,
     * then we get to Number isNaN and status success is set to false which we don't want. As it is not required, N/A is ok.
     */

    if (value == null || value == "N/A") {
        if (required) {
            status.success = false;
            status.missing.push(name);

            /* Create variable with value "%MISSING%" so that REPORTER doesn't complain about missing
             * variables. */
            let missing_var = new Variable(
                templ,
                name,
                "Requested variable is missing",
                "%[MISSING]%",
                "String",
                false,
                true
            );
        } else {
            /* If the variable was not required, create variable with value "N/A" so that REPORTER
             * doesn't complain about missing variables, and so that user can see it is not
             * required. */
            let optional_var = new Variable(
                templ,
                name,
                "Requested variable is missing but optional so mark as N/A",
                `%[NA]%`,
                "String",
                false,
                true
            );
        }

        return value;
    }
    if (type == "string") {
        /* For string variables, just return value without any further checks */
        return value;
    } else if (type == "int" || type == "float") {
        /* For numeric values, check they are valid integers or floats */
        let parsed;
        if (type == "int") {
            parsed = Number.parseInt(value);
        }
        if (type == "float") {
            parsed = Number.parseFloat(value);
        }
        if (Number.isNaN(parsed)) {
            status.success = false;
            status.invalid.push(name);
        }
        return parsed;
    } else {
        /* We shouldn't get here so print error and exit */
        LogError(`Unexpected argument <type = ${type}> in function get_variable_value.`);
        Exit();
    }
}

/**
 * Replaces a given value with its corresponding English language value from a set of language constants.
 * The function assumes the input value contains a constant name wrapped in `%[]%` and looks up the English translation
 * from a pre-defined language constants object. If the constant is not found or value is same as constantName, the function returns the original value.
 * @param {String} value
 * @returns {String}
 */
function replaceValueWithEnglishLanguageValue(value) {
    /** if value is falsey (e.g. null) or not a string just return the value without performing any replacements*/
    if (!value || typeof value != "string") return value;
    /** Get the language constants */
    let constants = get_language_constants();

    /** Extract the constant name from the value, e.g., "PASS_LOWER" */
    let constantName = value.replace(/%\[|\]%/g, "");

    /** If the constaName is same as value we didn't replace the %[]%, so we return the value as it is as it is not a language constant. */
    if (value == constantName) {
        return value;
    }

    /** Look up the constant and return the English value */
    if (constants[constantName]) {
        return constants[constantName].English;
    }

    /** If the constant is not found, return the original value */
    return value;
}

/**
 * This function retrieves the values of multiple variables and returns them in an
 * object with properties for each variable name and its corresponding value.
 *
 * @param {object} status Status object, used to track any missing or invalid variables
 * @param {string[]} names An array of variable names.
 * @param {string} [type = "string"] Variable type. Must be "string", "int", or "float".
 * @param {boolean} [required = true] Whether or not the varialbe is required (true) or optional (false)
 *
 * @returns {object} An object with properties for each variable name and its corresponding value.
 *
 * @example
 * let names = ["MY_VARIABLE_1", "MY_VARIABLE_2", "MY_VARIABLE_3"];
 * let variable_values = get_multiple_variable_values(status, names, "string", true);
 *
 * let my_variable_1_value = variable_values["MY_VARIABLE_1"];
 */
function get_multiple_variable_values(status, names, type = "string", required = true) {
    let values = {};

    for (let name of names) {
        let value = get_variable_value(status, name, type, required);
        values[name] = value;
    }

    return values;
}

/**
 * Prints warnings about missing and/or invalid REPORTER variables that have been requested during
 * score calculations.
 * @param {object} status Contains information about missing/invalid variables, expected in the form { success: false, missing: [...], invalid: [...] }
 * @param {string} descriptor String describing what is affected
 * @example
 * let body_region_label = `knee, femur and pelvis`;
 * let status = { success: true, missing: [], invalid: [] };
 * status.success = false;
 * status.missing.push("M1_DRIVER_FEMUR_COMPRESSION_EXCEEDENCE_SCORE");
 * status.invalid.push("M1_DRIVER_KNEE_COMPRESSION_SCORE");
 * if (!status.success) {
 *     warn_about_missing_or_invalid_variables(status, `M1 DRIVER ${body_region_label} score calculation`);
 * }
 */
function warn_about_missing_or_invalid_variables(status, descriptor) {
    if (status.missing && status.missing.length > 0) {
        LogWarning(`The following variables required for the ${descriptor} are missing:`);
        for (let name of status.missing) {
            LogPrint(name);
        }
    }
    if (status.invalid && status.invalid.length > 0) {
        LogWarning(`The following variables used in the ${descriptor} have invalid values:`);
        for (let name of status.invalid) {
            LogPrint(name);
        }
    }
}

/**
 * Returns a list of models based on the value of the %MODEL_LIST% variable.
 * Expect the value of %MODEL_LIST% to be in the form:
 *
 *     "M1, M2, M3"
 *
 *  Return null if missing
 *  @returns {?string[]}
 */
function get_model_list() {
    let templ = Template.GetCurrent();
    let models_list_str = templ.GetVariableValue("MODEL_LIST");
    if (models_list_str == null) {
        return null;
    } else {
        return models_list_str.split(/,\s*/);
    }
}

/**
 * This function initialises the REPORTER variables that are used in the script.
 * @param {Boolean} [override = false] - If true, the function will overwrite existing variables
 * otherwise it will only create them if they don't exist
 */
function initialise_language_constant_reporter_variables(override = false) {
    let templ = Template.GetCurrent();

    let language_constants = get_language_constants();

    let language = "English";

    /** if the template filename contains "_CN.or" then set the language to Chinese */
    if (/_CN\.or/i.test(templ.filename)) {
        language = "Chinese";
    }

    for (let const_name in language_constants) {
        /** note that language constants are wrapped in [] so that they are all grouped together in the variable list
         * and to reduce the risk of unintentionally overwriting a variable with the same name
         */
        let var_name = `[${const_name}]`;
        /** only create the language constant variables if they don't exist yet or if override is true */
        if (get_expanded_variable_value(templ, var_name) == null || override) {
            let temp_var = new Variable(
                templ,
                var_name,
                `${language} Language constant. The value may be changed to support a different language`,
                get_unicode_text_with_padding(language_constants[const_name][language]),
                "String",
                false,
                false // make language constant variables permanent
            );
        }
    }
}

/**
 * Returns an object containing the language constants used in the template
 * Add support for new languages and constants here
 * @returns {Object} -
 */
function get_language_constants() {
    /** note that the case matters */
    return {
        PASS: { English: "Pass", Chinese: "通过" },
        FAIL: { English: "Fail", Chinese: "未通过" },
        PASS_ALL_CAPS: { English: "PASS", Chinese: "通过" },
        FAIL_ALL_CAPS: { English: "FAIL", Chinese: "未通过" },
        MISSING: { English: "Missing", Chinese: "缺失" },
        MISSING_UPPER: { English: "MISSING", Chinese: "缺失" },
        YES: { English: "Yes", Chinese: "是" },
        NO: { English: "No", Chinese: "否" },
        NO_UPPER: { English: "NO", Chinese: "否" },
        YES_UPPER: { English: "YES", Chinese: "是" },
        NA: { English: "N/A", Chinese: "无" },
        CAPPING: { English: "CAPPING", Chinese: "上限区" },
        CAPPING_LIMIT_EXCEEDED: { English: "Capping limit exceeded", Chinese: "超出极限值" },
        STAR_CAPPING_LIMIT_EXCEEDED: { English: "*Capping limit exceeded", Chinese: "*超出极限值" },
        RED: { English: "RED", Chinese: "红色区" },
        BROWN: { English: "BROWN", Chinese: "棕色区" },
        CNCAP_RED: { English: "RED**", Chinese: "红色区**" },
        ORANGE: { English: "ORANGE", Chinese: "橙色区" },
        YELLOW: { English: "YELLOW", Chinese: "黄色区" },
        GREEN: { English: "GREEN", Chinese: "绿色区" },
        //Note that the following language constants are used in the summary template so you may
        //need to update the places they are used in cncap_summary.js if you make any changes below
        WORKING_CONDITION: { English: "Working Condition", Chinese: "滑台工况" },
        DRIVER_AIRBAG_PENALTY: { English: "Driver (Airbag) Penalty", Chinese: "主驾（气囊）罚分" },
        PASSENGER_PENALTY: { English: "Passenger Penalty", Chinese: "乘员罚分" },
        DESIGN_LOCATION: { English: "Design Location", Chinese: "设计位置" },
        HIGHEST_LEVEL: { English: "Highest Level", Chinese: "调至最高" },
        POLE_IMPACT: { English: "pole impact", Chinese: "柱碰" }
    };
}

/**
 * loop through all the template variables and replace all the strings which match the language constants with the corresponding
 * reporter variable language constant
 */
function replace_language_constants() {
    let templ = Template.GetCurrent();
    let language_constants = get_language_constants();
    let variables = Variable.GetAll(templ);
    for (let variable of variables) {
        /** skip the REPORTER variables which are language constants themselves
         * note that the language constants are wrapped in square brackets so we can easily identify them e.g. %[variable.name]%
         * but to check that they are valid language constants and not some other variable that happens to be wrapped in square brackets
         * we need to also check if the variable name minus square brackets can be found in the language_constants object
         */
        let var_name_minus_sqr_brackets = variable.name.replace(/^\[(.*)\]$/, "$1");
        if (var_name_minus_sqr_brackets != variable.name && language_constants[var_name_minus_sqr_brackets]) continue;
        /** only replace temporary variables */
        if (!variable.temporary) continue;
        /** get the variable value for replacing the language constants */
        let value = variable.value;
        /** skip the variable if it contains a REPORTER variable */
        if (/%\S+%/.test(value)) continue;
        for (let language_constant in language_constants) {
            /** if the variable value exactly matches the English|Chinese language constant then replace it with the REPORTER variable */
            if (
                value == language_constants[language_constant]["English"] ||
                value == language_constants[language_constant]["Chinese"]
            ) {
                variable.value = `%[${language_constant}]%`;
                LogPrint(`Replaced language constants in variable ${variable.name} ${value} -> ${variable.value}`);
                break;
            }
        }
    }
}

/**
 * loop through all the template variables and replace all the strings which are language constant variables (e.g. %[VAR_NAME]%) with the corresponding
 * English language constant string
 */
function replace_language_constant_variable_strings_with_english() {
    let templ = Template.GetCurrent();
    let language_constants = get_language_constants();
    let variables = Variable.GetAll(templ);
    for (let variable of variables) {
        /** skip the REPORTER variables which are language constants themselves */
        if (language_constants[variable.name]) continue;
        /** only replace temporary variables */
        if (!variable.temporary) continue;
        /** get the variable value for replacing the language constants */
        let value = variable.value;
        // /** skip the variable if it contains a REPORTER variable */
        // if (/%\S+%/.test(value)) continue;
        for (let language_constant in language_constants) {
            /** if the variable value exactly matches the English language constant then replace it with the REPORTER variable */
            if (value == `%[${language_constant}]%`) {
                variable.value = language_constants[language_constant]["English"];
                LogPrint(`Replaced language constants in variable ${variable.name} ${value} -> ${variable.value}`);

                break;
            }
        }
    }
}

/**
 * loop through all given array of template variable names and replace all the strings which are language constant variables (e.g. %[VAR_NAME]%) with the corresponding
 * language constant string value
 * @param {String[]} variable_names
 */
function replace_language_constant_variable_strings_with_value(variable_names) {
    let templ = Template.GetCurrent();

    for (let variable_name of variable_names) {
        let variable = Variable.GetFromName(templ, variable_name);
        if (!variable) {
            LogError(`Variable ${variable_name} not found in the template`);
            continue;
        }
        let value = get_expanded_variable_value(templ, variable_name);
        if (!value) {
            LogError(`Variable ${variable_name} has no value`);
            continue;
        }
        variable.value = value;
    }
}

/**
 * There is a bug with REPORTER 21.1 where some unicode characters are not displayed correctly in the text box.
 * This function adds padding to the end of the text to ensure that the unicode characters are displayed correctly
 * @param {String} [text="😀"] - The text to be displayed in the text box
 * @param {Boolean} [debug=false] - If true, the function will print the text with padding and the variable value
 * @returns {String} text_with_padding
 */
function get_unicode_text_with_padding(text = "😀", debug = false) {
    let templ = Template.GetCurrent();
    /** create a temporary variable to set the value of */
    let rep_var = new Variable(templ, "UNICODE_TEXT_WITH_PADDING");
    /** set the value to the string plus padding of $ characters for 4 times the length */
    rep_var.value = text + "$".repeat(text.length * 4);
    /** get the value after setting and subtract the original length to see how many
     *  $ characters are "swallowed up" in the unicode conversion */
    let num_trailing_stars = rep_var.value.length - text.length;
    let text_with_padding = text + "$".repeat(text.length * 4 - num_trailing_stars);
    /** if debugging set the REPORTER variable value to confirm that it worked
     * otherwise remove the variable*/
    if (debug) {
        rep_var.value = text_with_padding;
        /** log print the string with padding and the variable value */
        LogPrint(text_with_padding);
        /** note that the variable value will not have the padding so it will need to have the padding
         * added again using get_unicode_text_with_padding() */
        LogPrint(rep_var.value);
        LogPrint(get_unicode_text_with_padding(rep_var.value, false));
    } else {
        rep_var.Remove();
    }
    return text_with_padding;
}

/**
 * Get the first item on the page where the name matches contains the given name (note that exact match is not required as when we duplicate a page the
 * items get a suffix added (e.g. "_1") and we still want to get the item
 * @param {String} name
 * @param {Page} [page = Template.GetCurrent().GetMaster()] - The page to search for the item. If not provided, the master page is used.
 * @returns {?Item}
 */
function get_item_by_name(name, page = Template.GetCurrent().GetMaster()) {
    for (let item of page.GetAllItems()) {
        if (RegExp(name, "i").test(item.name)) return item;
    }
    return null;
}

/**
 * Replaces NaN with "Missing" for either an object or a scalar.
 * If passed an object, replaces any NaN values on its keys.
 * If passed a scalar, returns "Missing" if NaN.
 */
function replaceNaNWithMissing(value) {
    // const isActuallyNaN = (v) => typeof v === "number" && isNaN(v);
    const isActuallyNaN = (v) => isNaN(v);

    if (typeof value === "object" && value !== null) {
        for (const key in value) {
            if (isActuallyNaN(value[key])) {
                value[key] = "Missing";
            }
        }
        return value;
    } else {
        return isActuallyNaN(value) ? "Missing" : value;
    }
}

function get_EuroNCAP_band_score(s) {
    if (s <= 0) return "0";
    if (s <= 33.3) return "20";
    if (s <= 66.67) return "40";
    if (s < 100) return "80";
    if (s >= 100) return "100";
    if (s == "") return "Missing";

    return "Missing";
}

/**
 *
 * @param {object} status
 * @param {string} name
 * @param {string} type
 * @param {object} templ
 * @returns varValue, value of variable that was requested
 * This function is a wrapper for get_variable_value() so that when a request is made
 * to get a SCORE variable, the score will be processed and binned into the EuroNCAP
 * scoring groups, and create a new variable with "_ENCAP_BANDED" added on to the variable name.
 */
function get_variable_value_and_ENCAP_band_score(status, name, type, templ) {
    let euroNCAPBandedScore;
    let varValue = get_variable_value(status, name, type);
    if (varValue === undefined || varValue === null || varValue === "") {
        euroNCAPBandedScore = "Missing";
        varValue = "Missing";
    }
    if (type == "string") {
        if (!varValue.includes("Missing") && !varValue.includes("NaN")) {
            euroNCAPBandedScore = get_EuroNCAP_band_score(parseFloat(varValue));
        } else {
            euroNCAPBandedScore = "Missing";
            varValue = "Missing";
        }
    } else if (varValue == 0 || Boolean(varValue)) {
        euroNCAPBandedScore = get_EuroNCAP_band_score(parseFloat(varValue));
    } else {
        euroNCAPBandedScore = "Missing";
    }

    let ENCAP_BAND_VAR = new Variable(
        templ,
        name + "_ENCAP_BANDED",
        `Score converted to Euro NCAP Bands`,
        euroNCAPBandedScore.toString(),
        "String",
        false,
        true
    );

    return varValue;
}

/**
 * Function is a wrapper for creating a new variable. It creates the original variable but then it takes the
 * value and bins it according to the EuroNCAP bins to create an additional variable with the appended string
 * "_ENCAP_BANDED"
 * @param {*} template
 * @param {*} name
 * @param {*} description
 * @param {*} value
 * @param {*} type
 * @param {*} readonly
 * @param {*} temporary
 * @returns
 */

function new_variable_and_new_euroNCAP_banded_score(template, name, description, value, type, readonly, temporary) {
    let overall_occ_score_pass_var = new Variable(template, name, description, value, type, readonly, temporary);

    let euroNCAPBandedScore = "Missing";
    if (type == "string") {
        if (!value.includes("Missing") && !value.includes("NaN")) {
            euroNCAPBandedScore = get_EuroNCAP_band_score(parseFloat(value));
        } else {
            euroNCAPBandedScore = "Missing";
        }
    } else if (Boolean(value)) {
        euroNCAPBandedScore = get_EuroNCAP_band_score(parseFloat(value));
    }

    let ENCAP_BAND_VAR = new Variable(
        template,
        name + "_ENCAP_BANDED",
        `Score converted to Euro NCAP Bands`,
        euroNCAPBandedScore.toString(),
        "String",
        false,
        true
    );
    return value;
}

/**
 * Returns the path to parent directory or the file or directory
 * @param {string} path
 * @returns {string}
 */
function GetParent(path) {
    let parent_path;
    if (path.endsWith("/..") || path.endsWith("\\..")) {
        parent_path = path + "/..";
    } else {
        parent_path = path.substring(0, Math.max(path.lastIndexOf("/"), path.lastIndexOf("\\")));
        if (parent_path === "" || !File.IsDirectory(parent_path)) {
            parent_path = path + "/..";
        }
    }

    try {
        if (File.IsDirectory(parent_path)) {
            return parent_path;
        } else {
            throw new Error(`Parent path is not a directory: ${parent_path}`);
        }
    } catch (error) {
        LogError(`${error}\nreturning original path: ${path}`);
        return path;
    }
}

/**
 * Read a JSON file and return the parsed object
 * (located in workflow_definitions\scripts\automotive_assessments\post\reporter\reporter_on_load.js)
 * @param {String} filepath
 * @returns {Object} json
 */
function ReadJSON(filepath) {
    function unicodeReplacer(key, value) {
        if (typeof value === "string") {
            return value.replace(/[\u007F-\uFFFF]/g, function (c) {
                return "\\u" + ("0000" + c.charCodeAt(0).toString(16)).slice(-4);
            });
        }
        return value;
    }

    function unicodeReviver(key, value) {
        if (typeof value === "string") {
            return value.replace(/\\u[\dA-Fa-f]{4}/g, function (match) {
                return String.fromCharCode(parseInt(match.replace("\\u", ""), 16));
            });
        }
        return value;
    }

    try {
        if (File.Exists(filepath)) {
            let json_file = new File(filepath, File.READ);

            let line;
            let lines = [];

            //@ts-ignore
            while ((line = json_file.ReadLongLine()) != File.EOF) {
                lines.push(line);
            }

            json_file.Close();

            let json = JSON.parse(lines.join(`\n`), unicodeReviver);

            return json;
        }

        throw Error(`${filepath} does not exist`);
    } catch (error) {
        // Handle error
        throw Error(`Failed to read JSON from ${filepath}: ${error.message}`);
    }
}
