// module: TRUE
// memory: 1000
// ^^^^^^^^^^^^ important - as importing scripts as modules

// T/HIS is able to import modules from paths that are relative to the
// T/HIS scripts library so the cora.mjs module can be imported as shown below.
// Alternatively, if you have changed you script library using oa_prefs you can
// use an absolute path and import from:
// "path/to/oasys/installation/this_library/scripts/CORA/modules/cora.mjs"
import { correlation_analysis, read_json } from "CORA/modules/cora.mjs";

// salient file paths as global variables
let script_dir = parent_directory(arguments[0]);
let model_file_path = script_dir + "/analysis_model/bouncer.thf";
let test_file_path = script_dir + "/test_data/bouncer.cur";
let cora_config_file =
    script_dir + "/cora_and_iso18571_workflow_test_input.json";

// read in the test curves - 3 tests - force and disp i.e. 6 curves total
let options = new Object(); // options for curve file read
let cur_id = Curve.FirstFreeID();
options.outputOpt = cur_id; // read in starting at first free id
Read.Cur(test_file_path, options);
// assign the curves to objects so can be manipulated
let cur_d_test = []; // array of test curves - displacement
let cur_f_test = []; // array of test curves - force
for (let i = 0; i < 3; i++) cur_d_test[i] = Curve.GetFromID(i + cur_id);
for (let i = 0; i < 3; i++) cur_f_test[i] = Curve.GetFromID(i + cur_id + 3);

// read in the simulation model
let m = Model.Read(model_file_path);

// first, extract displacement-time curve
let f_d = AllocateFlag();
m.SetFlag(f_d, Entity.NODE, -875);
let cur_d = m.GetDataFlagged(f_d, Component.DZ)[0];

// then, extract force-time curve
let f_f = AllocateFlag();
m.SetFlag(f_f, Entity.CONTACT, 1);
let cur_f = m.GetDataFlagged(f_f, Component.CFZ)[0];

// flip displacement time
let cur_d_flipped = Operate.Mul(cur_d, -1);

// filter force-time
let cur_f_filtered = Operate.C180(cur_f);

// C180 filter the test force-time curves
let cur_f_test_filtered = [];
for (let i = 0; i < 3; i++)
    cur_f_test_filtered[i] = Operate.C180(cur_f_test[i]);

// remove all of these curves from the graph
let f = AllocateFlag();
Curve.FlagAll(f);
Curve.RemoveFlaggedFromGraph(f);

// run the correlation analysis. Note: cora function is defined below
cora();

// finished so say bye
Message("bye");
Exit();

function cora() {
    let cora_input = read_json(cora_config_file);

    if (cora_input == false) {
        ErrorMessage(`Error reading "${cora_config_file}" config file`);
        Exit();
    }

    /** overwrite file_output and b_delta_end property values
     * (other values can be overwritten in the same way)  */
    cora_input.file_output = script_dir + "/workflow_test.csv";
    cora_input.b_delta_end = 0.0;

    // turn on curve_output but only plot corridor (not cross-correlation) graphs
    cora_input.curve_output = true;
    cora_input.corridor_graph = true;
    cora_input.correlation_graph = false;

    // perform the pdb cora correlation analysis
    correlation_analysis(cora_input);

    // now update the properties to perform the ISO/TS 18571:2014 correlation analysis

    // first overwrite the method field
    cora_input.method = "ISO/TS 18571:2014";

    // use the cora_input vairables
    cora_input.use_iso18571_vars = true;

    // overwrite the case titles by appending the method to the existing case titles
    cora_input.cases = cora_input.cases.map((c) => {
        c.title += " (ISO/TS 18571:2014)";
        return c;
    });

    // ensure append is true so that the output is appended to the existing file: cora_input.file_output
    cora_input.append = true;

    // perform the ISO/TS 18571:2014 correlation analysis
    correlation_analysis(cora_input);
}

/**
 * Return the parent directory of the given path.
 * @param {String} path
 * @returns {String}
 */
function parent_directory(path) {
    return path.substring(
        0,
        Math.max(path.lastIndexOf("/"), path.lastIndexOf("\\"))
    );
}
