/* Minesweeper example
 *
 * Miles Thornton
 * 10/1/2008
 */
var mines = 10;
var rows  = 9;
var cols  = 9;


var row, col, time, cleared, marked, exploded;

function changed()
{
    Message("Text has changed to " + this.text);
}


var w = new Window("Minesweeper", 0.85, 1.0, 0.8, 1.0);

// Exit button
var exit_w = new Widget(w, Widget.BUTTON,    1, 30, 1, 7, "Exit");
exit_w.onClick  = confirm_exit;

// Help button
var help_w = new Widget(w, Widget.BUTTON,   31, 60, 1, 7, "Help");
help_w.onClick  = show_help;

// Count widget to show how many mines there are
var count_w = new Widget(w, Widget.LABEL,    1, 20, 10, 20, mines.toString());
count_w.background = Widget.BLACK;
count_w.foreground = Widget.RED;

// Smile widget
var smile_w = new Widget(w, Widget.BUTTON,  25, 35, 10, 20, "");
smile_w.onClick  = start_game;
smile_w.timerDelay = 1000;
smile_w.timerRepeat= true;
draw_smile();

// Time widget
var time_w  = new Widget(w, Widget.LABEL,   41, 60, 10, 20, "000");
time_w.background = Widget.BLACK;
time_w.foreground = Widget.RED;

// Make cells for mine grid
var cells = new Array();
var labs  = new Array();
for (row=0; row<rows; row++)
{
    for (col=0; col<cols; col++)
    {
        cells[row*cols+col] = new Widget(w, Widget.BUTTON, 4+(col*6), 10+(col*6), 25+(row*6), 31+(row*6), "")
        cells[row*cols+col].onClick = cell_clicked;
        cells[row*cols+col].row     = row;
        cells[row*cols+col].col     = col;
        cells[row*cols+col].active  = false;
        labs[row*cols+col] = new Widget(w, Widget.LABEL, 4+(col*6), 10+(col*6), 25+(row*6), 31+(row*6), "")
        labs[row*cols+col].Hide();
        labs[row*cols+col].background = Widget.LIGHTGREY;
        labs[row*cols+col].Rectangle(Widget.DARKGREY, false, 0, 0, 100, 100);
    }
}

w.Show();

////////////////////////////////////////////////////////////////////////////////

function confirm_exit()
{
    var ret = Window.Question("Confirm exit", "Are you sure you want to quit?");
    if (ret == Window.YES)
    {
        smile_w.onTimer = null;
        Exit();
    }
}

function show_help()
{
    var ret = Window.Message("Help",
                             "Locate all of the mines!\n" +
                             "To mark a mine hold down the\n" +
                             "Ctrl or Shift key when pressing\n" +
                             "the left mouse button.\n" +
                             "\n"
                            );
}

function cell_clicked()
{
    var mark = false;

    if (Widget.CtrlPressed() || Widget.ShiftPressed())
    {
        if (this.mark)
        {
            this.mark = false;
            this.Clear();
            mines++;
            marked--;
        }
        else
        {
            this.mark = true;
            draw_mark(this)
            mines--;
            marked++;
        }
        this.Show();
        count_w.text = mines.toString();
    }
    else
    {
        if (this.mine)
        {
            reveal_mines(this);
        }
        else
        {
            check_cell(this);
        }
    }

// Check for end of game
    if (marked+cleared == rows*cols) end_game();
    
}

function allocate_mines()
{
    var n = mines;

    while (n)
    {
        var pos = Math.round(Math.floor(rows*cols*Math.random()));
        if (pos >= 0 && pos < rows*cols)
        {
            if (cells[pos].mine == false)
            {
                cells[pos].mine = true;
//              draw_mine(cells[pos]);
                n--;
            }
        }
    }
}

function draw_smile()
{
    smile_w.Clear();
    smile_w.line_width = 1;
    smile_w.Circle(Widget.YELLOW, true, 50, 50, 39);

// Eyes
    smile_w.Circle(Widget.BLACK,  true,  35, 35,  3);
    smile_w.Circle(Widget.BLACK,  true,  65, 35,  3);

// Border
    smile_w.Circle(Widget.BLACK,  false, 50, 50, 40);

// Smile
    smile_w.Line(Widget.BLACK, 30, 60, 43, 70);
    smile_w.Line(Widget.BLACK, 70, 60, 55, 70);
    smile_w.Line(Widget.BLACK, 43, 70, 55, 70);

    smile_w.Show();
}

function draw_shades()
{
    smile_w.Clear();
    smile_w.line_width = 1;
    smile_w.Circle(Widget.YELLOW, true, 50, 50, 39);

// Shades
    smile_w.Circle(Widget.BLACK,  true,  35, 40, 10);
    smile_w.Circle(Widget.BLACK,  true,  65, 40, 10);
    smile_w.Line(Widget.BLACK, 90, 50, 65, 30);
    smile_w.Line(Widget.BLACK, 10, 50, 35, 30);
    smile_w.Line(Widget.BLACK, 35, 30, 65, 30);

// Border
    smile_w.Circle(Widget.BLACK,  false, 50, 50, 40);

// Smile
    smile_w.Line(Widget.BLACK, 30, 60, 43, 70);
    smile_w.Line(Widget.BLACK, 70, 60, 55, 70);
    smile_w.Line(Widget.BLACK, 43, 70, 55, 70);

    smile_w.Show();
}

function draw_frown()
{
    smile_w.Clear();
    smile_w.line_width = 1;
    smile_w.Circle(Widget.YELLOW, true, 50, 50, 39);

// Eyes
    smile_w.Line(Widget.BLACK,  30, 30, 40, 40);
    smile_w.Line(Widget.BLACK,  30, 40, 40, 30);

    smile_w.Line(Widget.BLACK,  60, 30, 70, 40);
    smile_w.Line(Widget.BLACK,  60, 40, 70, 30);

// Border
    smile_w.Circle(Widget.BLACK,  false, 50, 50, 40);

// Frown
    smile_w.Line(Widget.BLACK, 30, 70, 43, 60);
    smile_w.Line(Widget.BLACK, 70, 70, 55, 60);
    smile_w.Line(Widget.BLACK, 43, 60, 55, 60);

    smile_w.Show();
}



function draw_mine(cell)
{
    cell.Clear();
    cell.line_width = 1;
    cell.Circle(Widget.BLACK, true, 50, 50, 25);
    cell.Line(Widget.BLACK, 50, 10, 50, 90);
    cell.Line(Widget.BLACK, 10, 50, 90, 50);
    cell.Line(Widget.BLACK, 20, 20, 85, 85);
    cell.Line(Widget.BLACK, 20, 80, 85, 20);

    cell.Circle(Widget.WHITE, true, 45, 45, 3);

    cell.Show();
}

function draw_mark(cell)
{
    cell.Clear();
    cell.line_width = 1;
    cell.Line(Widget.BLACK, 50, 50, 50, 90);
    cell.Polygon(Widget.BLACK, true, 10, 90, 50, 70, 90, 90);
    cell.Polygon(Widget.RED,   true, 10, 30, 50, 10, 50, 50);
}

function start_game()
{
    for (row=0; row<rows; row++)
    {
        for (col=0; col<cols; col++)
        {
            labs[row*cols+col].Hide();

            cells[row*cols+col].active     = true;
            cells[row*cols+col].mine       = false;
            cells[row*cols+col].cleared    = false;
            cells[row*cols+col].mark       = false;
            cells[row*cols+col].pushed     = false;
            cells[row*cols+col].background = Widget.DEFAULT;
            cells[row*cols+col].Clear();
            cells[row*cols+col].Show();
        }
    }

    mines    = 10;
    cleared  = 0;
    marked   = 0;
    exploded = false;

    count_w.text = mines.toString();

// Allocate mines
    allocate_mines();

    time = 0;

    draw_smile();
    smile_w.onTimer = update_time;
}

function end_game()
{
    draw_shades();
    smile_w.onTimer = null;
    Window.Message("Congratulations!",
                   "You cleared all of the mines\n" +
                   "in a time of " + time + " seconds\n",
                   Window.OK)
}

function update_time()
{
    time++;
    if (time < 10)
        time_w.text = "00" + time;
    else if (time < 100)
        time_w.text = "0" + time;
    else
        time_w.text = time;
}

function reveal_mines(cell)
{
    exploded = true;

    smile_w.onTimer = null;
    cell.background = Widget.RED;

// Show where all of the mines are
    for (row=0; row<rows; row++)
    {
        for (col=0; col<cols; col++)
        {
            if (cells[row*cols+col].mine) draw_mine(cells[row*cols+col]);
        }
    }

    draw_frown();
}

function check_cell(cell)
{
    if (cell.cleared || cell.mark || exploded) return;

    if (cell.mine)
    {
        reveal_mines(cell);
        return;
    }

    var r=cell.row;	// Make easier to read
    var c=cell.col;	// Make easier to read
    cell.cleared = true;
    cleared++;

    var nmines = 0;

// Find how many mines are around this cell
    if ( (r-1)>=0   && (c-1)>=0   && cells[(r-1)*cols+(c-1)].mine) nmines++;
    if ( (r-0)>=0   && (c-1)>=0   && cells[(r-0)*cols+(c-1)].mine) nmines++;
    if ( (r+1)<rows && (c-1)>=0   && cells[(r+1)*cols+(c-1)].mine) nmines++;

    if ( (r-1)>=0   && (c-0)>=0   && cells[(r-1)*cols+(c-0)].mine) nmines++;

    if ( (r+1)<rows && (c-0)>=0   && cells[(r+1)*cols+(c-0)].mine) nmines++;

    if ( (r-1)>=0   && (c+1)<cols && cells[(r-1)*cols+(c+1)].mine) nmines++;
    if ( (r-0)>=0   && (c+1)<cols && cells[(r-0)*cols+(c+1)].mine) nmines++;
    if ( (r+1)<rows && (c+1)<cols && cells[(r+1)*cols+(c+1)].mine) nmines++;

// Hide the button, show the label and set colour
    cell.Hide();
    if (nmines)
    {
        if      (nmines == 1) labs[r*cols+c].foreground=Widget.BLUE;
        else if (nmines == 2) labs[r*cols+c].foreground=Widget.DARKGREEN;
        else if (nmines == 3) labs[r*cols+c].foreground=Widget.RED;
        else if (nmines == 4) labs[r*cols+c].foreground=Widget.DARKBLUE;
        else if (nmines >= 5) labs[r*cols+c].foreground=Widget.MAGENTA;

        labs[r*cols+c].text=nmines.toString();
    }
    else
        labs[r*cols+c].text="";
    labs[r*cols+c].Show();

    if (nmines) return;

// If there are no mines around this cell then check the cells around it.
    if ( (r-1)>=0   && (c-1)>=0  ) check_cell(cells[(r-1)*cols+(c-1)]);
    if ( (r-0)>=0   && (c-1)>=0  ) check_cell(cells[(r-0)*cols+(c-1)]);
    if ( (r+1)<rows && (c-1)>=0  ) check_cell(cells[(r+1)*cols+(c-1)]);

    if ( (r-1)>=0   && (c-0)>=0  ) check_cell(cells[(r-1)*cols+(c-0)]);

    if ( (r+1)<rows && (c-0)>=0  ) check_cell(cells[(r+1)*cols+(c-0)]);

    if ( (r-1)>=0   && (c+1)<cols) check_cell(cells[(r-1)*cols+(c+1)]);
    if ( (r-0)>=0   && (c+1)<cols) check_cell(cells[(r-0)*cols+(c+1)]);
    if ( (r+1)<rows && (c+1)<cols) check_cell(cells[(r+1)*cols+(c+1)]);
}
