// These functions are used by the standard REPORTER IIHS templates
// in OA_INSTALL/reporter_dir/library/templates


function IIHS_get_constant_limit_results(c, good, accp, marg, symmetric)
{
	// Calculates results for criteria that has constant limits for the IIHS protocol
	// Returns the rating and value. 
	//
	// <c> is the curve
	// <good>, <accp> and <marg> are the constant limits

	var ret = new Array(2);

    // If symmetric variable has a value then the curve should be checked against positive and negative limits. 
    // This is done by taking the absolute maximum value of the curve and check that against the positive limits

	var val;
	var rating;

    if (symmetric != undefined && symmetric == true) { val = Math.max(c.ymax,Math.abs(c.ymin)); rating = IIHS_get_rating(val, good, accp, marg, symmetric); }
    else if(good > 0) { val =  c.ymax; rating = IIHS_get_rating(val,  good,  accp,  marg); }
	else         { val = -c.ymin; rating = IIHS_get_rating(val, -good, -accp, -marg); }

	ret[0] = rating;
	ret[1] = val;

	return ret;  
}



function IIHS_get_variable_limit_results(c, time, good, accp, marg)
{
	// Calculates results for criteria that have variable limits
	// Returns the rating, value and time of value in an array. 
	//
	// <c> is the curve.
	// <time>, <good>, <accp> and <marg> are arrays defining the limits

	var ret = new Array(3);

    var rating, level, duratn;
    
	rating = "GOOD";
	level  = 0.0;
	duratn = 0.0;

	var n = c.npoints;

	for(var i=1; i<=n; i++)
	{
		var p = c.GetPoint(i);

		var t = p[0];
		var f = p[1];

		var limits = IIHS_get_variable_limits(t, time, good, accp, marg);

		var temp = IIHS_get_rating(f, limits[0], limits[1], limits[2]);

		// Store point if the rating is lower than previous

		var isLower = false;

		if(rating == "GOOD"       && (temp == "ACCEPTABLE" || temp == "MARGINAL" || temp == "POOR")) isLower = true;
		else if(rating == "ACCEPTABLE" && (                        temp == "MARGINAL" || temp == "POOR")) isLower = true;
		else if(rating == "MARGINAL"   && (                                              temp == "POOR")) isLower = true;

		if(i==1 || isLower)
		{
			rating = temp;
			level  = f;
			duratn = t;
		}
	}

	ret[0] = rating;
	ret[1] = level;
	ret[2] = duratn;

	return ret;
}



function IIHS_get_rating(val, good_val, acceptable_val, marginal_val)
{
	var retval = "GOOD";

	if(val <= good_val)       retval = "GOOD";
	else if(val <= acceptable_val) retval = "ACCEPTABLE";
	else if(val <= marginal_val)   retval = "MARGINAL";
	else                           retval = "POOR";

	return retval;
}



function IIHS_get_variable_limits(t, time, good, accp, marg)
{
	// Returns the limits at time <t> for variable limits

	var ret = new Array(3);

	// Initialise values, as arguments can be null

	if(good[0] > 0) { ret[0] =  1.0e20; ret[1] =  1.0e20;  ret[2] =  1.0e20; }
	else            { ret[0] = -1.0e20; ret[1] = -1.0e20;  ret[2] = -1.0e20; }

	for(var i=1; i<time.length; i++)
	{
		if(t < time[i])
		{
			if(good) ret[0] = good[i] + ((good[i-1] - good[i]) * (t - time[i])) / (time[i-1] - time[i]);
			if(accp) ret[1] = accp[i] + ((accp[i-1] - accp[i]) * (t - time[i])) / (time[i-1] - time[i]);
			if(marg) ret[2] = marg[i] + ((marg[i-1] - marg[i]) * (t - time[i])) / (time[i-1] - time[i]);

			return ret;
		}
	}

	// <t> is bigger than the last time defined for the limits, so just return the limits at the last time

	if(good) ret[0] = good[good.length - 1];
	if(accp) ret[1] = accp[accp.length - 1];
	if(marg) ret[2] = marg[marg.length - 1];

	return ret;
}

