/// <reference path="C:/SOURCE14/this_dir/this_js_api/this.intellisense.js" />
/// <reference path="C:/SOURCE14/reporter_dir/library/templates/scripts/this_common1.js" />

// These functions are used by the standard REPORTER EuroNCAP templates
// in OA_INSTALL/reporter_dir/library/templates


function get_variable_limit_results(c, time, good, weak, lo_score, hi_score)
{
	// Calculates results for criteria that have variable limits
	// Returns the score, value and time of value in an array. 
	//
	// <c> is the curve.
	// <time>, <good>, <adqt> and <marg> are arrays defining the limits

	var ret = new Array(3);

	var score  = 0.0;
	var level  = 0.0;
	var duratn = 0.0;

	var n = c.npoints;

	for(var i=1; i<=n; i++)
	{
		var p = c.GetPoint(i);

		var t = p[0];
		var f = p[1];

		var limits = get_variable_limits(t, time, good, weak);

		var temp = sliding_scale(f, limits[0], limits[1], hi_score, lo_score);

		// Store point if the score is lower than previous

		if(i==1 || temp < score)
		{
			score = temp;
			level  = f;
			duratn = t;
		}
	}

	ret[0] = score;
	ret[1] = level;
	ret[2] = duratn;

	return ret;
}



function get_variable_limits(t, time, good, weak)
{
	// Returns the limits at time <t> for variable limits

	var ret = new Array(2);

	// Initialise values, as arguments can be null

	if(good[0] > 0) { ret[0] =  1.0e20; ret[1] =  1.0e20; }
	else            { ret[0] = -1.0e20; ret[1] = -1.0e20; }

	for(var i=1; i<time.length; i++)
	{
		if(t < time[i])
		{
			if(good) ret[0] = good[i] + ((good[i-1] - good[i]) * (t - time[i])) / (time[i-1] - time[i]);
			if(weak) ret[1] = weak[i] + ((weak[i-1] - weak[i]) * (t - time[i])) / (time[i-1] - time[i]);

			return ret;
		}
	}

	// <t> is bigger than the last time defined for the limits, so just return the limits at the last time

	if(good) ret[0] = good[good.length - 1];
	if(weak) ret[1] = weak[weak.length - 1];

	return ret;
}



function draw_head_limits(t, out_data, airbag)
{
	/// <signature>
	/// <summary>Creates curve(s) from time 0 to t at certain limit(s).
	/// <para>If only the &lt;t&gt; argument is used then one curve is drawn at PEAK_HEAD_ACCN_HI_LIMIT.</para>
	/// <para>If both arguments are used and airbag is false then one curve is drawn at PEAK_HEAD_ACCN_SOFT.</para>
	/// <para>If both arguments are used and airbag is true then two curvs are drawn at PEAK_HEAD_ACCN_LO_LIMIT and PEAK_HEAD_ACCN_HI_LIMIT.</para>
	/// </summary>
	/// <param name="t" type="Number" optional="false">End time</param>
	/// <param name="airbag" type="Boolean" optional="true">Steering wheel airbag present</param>
	/// <param name="out_data" type="Object" optional="false">OutputData object to store curve list.</param>
	/// </signature>

	var curve_id = Curve.FirstFreeID();

	if(arguments.length == 2)
	{
		var c_upper_limit = new Curve(curve_id++);

		c_upper_limit.AddPoint(0.0, PEAK_HEAD_ACCN_HI_LIMIT);
		c_upper_limit.AddPoint(t, PEAK_HEAD_ACCN_HI_LIMIT);

		set_line_style(c_upper_limit, Colour.RED);
		set_labels(c_upper_limit, "Head Acceleration Upper Limit",  "Time (" + oGlblData.unit_time + ")", "Acceleration (g)");
		out_data.curveList.push(c_upper_limit.id);
	}
	else
	{
		if(airbag)
		{
			// Steering wheel airbag present (or passenger) so just draw soft contact limit
			
			var c_soft_limit = new Curve(curve_id++);

			c_soft_limit.AddPoint(0.0, PEAK_HEAD_ACCN_SOFT);
			c_soft_limit.AddPoint(t,   PEAK_HEAD_ACCN_SOFT);

			c_soft_limit.colour = Colour.RED;
			c_soft_limit.style = LineStyle.SOLID;
			c_soft_limit.width = LineWidth.BOLD;
			c_soft_limit.symbol = Symbol.NONE;
			c_soft_limit.label = "Soft contact Limit";
			c_soft_limit.x_axis_label = "Time (" + oGlblData.unit_time + ")";
			c_soft_limit.y_axis_label = "Acceleration (g)";
			out_data.curveList.push(c_soft_limit.id);
		}
		else
		{
			// Steering wheel airbag NOT present so draw lower and upper limits

			var c_lower_limit = new Curve(curve_id++);
			var c_upper_limit = new Curve(curve_id++);

			c_lower_limit.AddPoint(0.0, PEAK_HEAD_ACCN_LO_LIMIT);
			c_lower_limit.AddPoint(t,   PEAK_HEAD_ACCN_LO_LIMIT);

			c_upper_limit.AddPoint(0.0, PEAK_HEAD_ACCN_HI_LIMIT);
			c_upper_limit.AddPoint(t,   PEAK_HEAD_ACCN_HI_LIMIT);

			set_line_style(c_lower_limit, Colour.GREEN);
			set_line_style(c_upper_limit, Colour.RED);

			set_labels(c_lower_limit, "Head Acceleration Lower Limit", "Time (" + oGlblData.unit_time + ")", "Acceleration (g)");
			set_labels(c_upper_limit, "Head Acceleration Upper Limit", "Time (" + oGlblData.unit_time + ")", "Acceleration (g)");

			out_data.curveList.push(c_lower_limit.id, c_upper_limit.id);
		}
	}
}
