/// <reference path="C:/SOURCE14/this_dir/this_js_api/this.intellisense.js" />

// These functions are used by the standard REPORTER templates
// in OA_INSTALL/reporter_dir/library/templates

// Define global constants for Euro NCAP colours (to match colours on euroncap.com)
var EuroNCAPColour = {};
EuroNCAPColour.GREEN  = Colour.RGB(38,155,41);
EuroNCAPColour.YELLOW = Colour.RGB(255,204,0);
EuroNCAPColour.ORANGE = Colour.RGB(255,151,0);
EuroNCAPColour.BROWN  = Colour.RGB(117,63,42);
EuroNCAPColour.RED    = Colour.RGB(224,7,0);



var OutputData = function(title, fname)
{
    /// <signature>
    /// <summary>Create a new OutputData object.</summary>
    /// <param name="title" type="String" optional="false">Title to put on graph</param>
    /// <param name="fname" type="String" optional="false">Filename for writing image</param>
    /// <returns type="OutputData"/>
    /// </signature>

    this.title = title;
    this.fname = fname;
    this.curveList = [];
}



var DummyData = function(label, entity_id_1, entity_type_1, entity_id_2, entity_type_2, entity_id_3, entity_type_3, entity_id_4, entity_type_4)
{
    /// <signature>
    /// <summary>Create a new DummyData object.</summary>
    /// <param name="label" type="String" optional="false">Dummy prefix string for chart titles (any spaces are replaced with underscores before use in variable names)</param>
    /// <param name="entity_id" type="Integer" optional="false">Entity ID (e.g. node ID) for rating calc</param>
    /// <param name="entity_type" type="String" optional="false">Type of entity. Can be "node", "beam", or "spring".</param>
    /// <returns type="DummyData"/>
    /// </signature>

    var variable_label = label.replace(/\s/g, "_");
    this.chart_label = label;
    this.variable_label = variable_label;
    // Loops through pairs of entity IDs and types, so that you can assign any/all of node_id, beam_id and spring_id
    for(var i=1; i<arguments.length; i+=2)
    {
        switch(arguments[i+1])
        {
            case "node":
                this.node_id = arguments[i];
                break;
            case "node_y":
                this.node_y_id = arguments[i];
                break;
            case "node_z":
                this.node_z_id = arguments[i];
                break;
            case "beam":
                this.beam_id = arguments[i];
                break;
            case "beam_2": // Added a second beam ID for THOR dummy knee data
                this.beam_2_id = arguments[i];
                break;
            case "spring":
                this.spring_id = arguments[i];
                break;
            case "section":
                this.section_id = arguments[i];
                break;
            // case "section_2": // Added a second cross section ID for THOR dummy pelvis data
            //     this.section_2_id = arguments[i];
            //     break;
            case "version":
                this.version = arguments[i];
                break;
            default:
                ErrorMessage("Unexpected entity_type in DummyData.");
                Exit();
                break;
        }
    }
}



function do_common_single_analysis_setup(obj)
{
    // Does the setup required in all the single analysis templates
    //
    // Returns the csv file that T/HIS will write variables to for Reporter to pick up.
    // It is up to the calling function to close the file.

    // First load the model
    var res_file = results_dir + "/" + default_job;

    DialogueInput("/MODEL READ", res_file);

    if(!Model.Exists(1))
    {
        ErrorMessage("Cannot find model results: " + res_file);
        Exit();
    }

    // Read the csv file containing entity IDS, etc and assign values to the <obj> object

    parse_csv_file(fname_csv, obj);

    // Open a csv file in the %IMAGES_DIR% to write variables to, for Reporter to pick up

    var f_vars = new File(images_dir + "/this_vars.csv", File.WRITE);

    // Check we have unit valid unit data and write a variable with the units used.
    // If the units are not defined/valid then we cannot go any further.  

    if(!check_unit_values())
    {
        write_unit_vars(f_vars, false);
        ErrorMessage("Cannot find the model unit values. They could be missing from the input csv file or post-*END data." );
        Exit();
    }

    // Write the unit data to the variables file
    write_missing_result_variables(f_vars, false);

    write_unit_vars(f_vars, true);

    // Calculate constants now we have unit data and store on <oGlblData> object

    calculate_unit_constants();

    // Now we've read the data
    //
    // 1. Setup the graph

    setup_this_graph();

    // Make sure any units set in T/HIS (e.g. via preferences) are unset

    unset_this_units(1);

    // Return the variables file that T/HIS will write to

    return f_vars;
}



function parse_csv_file(fname, obj)
{
    // Reads the CSV file <fname> storing the data on the <obj> object
    //
    // Lines beginning with '$' are comments otherwise they should be of the form:
    //
    // variable_name,variable_value
    //
    // Most variables are just entity IDs, so the default is to call parse_id().
    // Other variables that need to treated as strings are dealt with on a case by case basis.

    Message("Parsing input file: "+fname);
    var f_csv = new File(fname, File.READ);
    var line;

    while ( (line = f_csv.ReadLine()) != undefined)
    {

        if(line[0] != '$')
        {
            var list = line.split(",");
            var name = list[0].toLowerCase();

            if(list[1] != undefined)
            {
                switch(name)
                {
                    case "unit_length":             { obj.unit_length           = list[1];               break; }
                    case "unit_mass":               { obj.unit_mass             = list[1];               break; }
                    case "unit_time":               { obj.unit_time             = list[1];               break; }

                    case "driver_dummy":            { obj.driver_dummy          = list[1];               break; }
                    case "passenger_dummy":         { obj.passenger_dummy       = list[1];               break; }
                    case "front_passenger_dummy":   { obj.front_passenger_dummy = list[1];               break; }	
                    case "rear_passenger_dummy":    { obj.rear_passenger_dummy  = list[1];               break; }
                    case "steering_wheel_airbag":   { obj.steering_wheel_airbag = list[1].toUpperCase(); break; }

                    default:                        { obj[name]                 = parse_id(list[1]);     break; }
                }
            }
        }
    }

    // Close the file

    f_csv.Close();
    Message("Finished parsing file.");
}



function parse_id(id)
{
    // Parse an id string.  Could be a number, name or blank

    var ret_id;

    // Is it a number

    if(!(isNaN(id)))
    {

        ret_id = parseInt(id);

        if(ret_id != undefined && !(isNaN(ret_id))) return ret_id;
    }

    // Not a number so return the string if it's not blank

    if(id.length >  0)
    {
        for(var i=0; i<id.length; i++)
        {
            if(id[i] != ' ') return id;
        }
    }

    // Get here and it's not a number or non-blank string so return undefined.

    return undefined;
}



function write_blank_images(output_data, title_prefix)
{
    /// <signature>
    /// <summary>Writes images of blank graphs.</summary>
    /// <param name="output_data" type="Object" optional="false">Object containing properties which are set to OutputData objects.</param>
    /// <param name="title_prefix" type="String" optional="false">Prefix to add to title.</param>
    /// </signature>

    for(var prop in output_data)
    {
        create_blank_image(title_prefix + " " + output_data[prop].title, output_data[prop].fname);
    }
}



function create_blank_image(title, fname)
{
    // Writes a blank image to file <fname> with graph title <title>

    remove_all_curves_and_datums();
    DialogueInput("/DEFAULT TI", title);
    DialogueInput("/AU");
    DialogueInput("/IMAGE PNG24", fname + ".png", "GRAPH", "1");

}



function write_image(title, fname, c, max, min)
{
    /// <signature>
    /// <summary>Writes out an image of the graph.</summary>
    /// <param name="title" type="String" optional="false">Title to add to graph.</param>
    /// <param name="fname" type="String" optional="false">Filename to write image.</param>
    /// </signature>
    /// <signature>
    /// <summary>Writes out an image of the graph.</summary>
    /// <param name="title" type="String" optional="false">Title to add to graph.</param>
    /// <param name="fname" type="String" optional="false">Filename to write image.</param>
    /// <param name="c" type="Curve" optional="false">Curve to scale image to.</param>
    /// <param name="max" type="Number" optional="false">The maximum Y value to set the graph to as a minimum.</param>
    /// <param name="min" type="Number" optional="true">The minimum Y value to set the graph to as a minimum.</param>
    /// </signature>
    /// <signature>
    /// <summary>Writes out an image of the graph.</summary>
    /// <param name="title" type="String" optional="false">Title to add to graph.</param>
    /// <param name="fname" type="String" optional="false">Filename to write image.</param>
    /// <param name="c" type="Array" elementType="Curve" optional="false">Array of curves to scale image to.</param>
    /// <param name="max" type="Number" optional="false">The maximum Y value to set the graph to as a minimum.</param>
    /// <param name="min" type="Number" optional="true">The minimum Y value to set the graph to as a minimum.</param>
    /// </signature>

    if(File.Exists(fname + ".png")) File.Delete(fname + ".png");

    DialogueInput("/DEFAULT TI", title);

    if(arguments.length == 2) DialogueInput("/AU");
    else if(arguments.length == 4) scale_graph(c, max);
    else if(arguments.length == 5) scale_graph_2(c, max, min);

    DialogueInput("/IMAGE PNG24", fname + ".png", "GRAPH", "1");
}



function write_curve(format, curveList, fname)
{
    /// <signature>
    /// <summary>Writes out a curve file.</summary>
    /// <param name="format" type="String" optional="false">Format for output - cur or csv.</param>
    /// <param name="curveList" type="Array" optional="false">List of curves to write.</param>
    /// <param name="fname" type="String" optional="false">Filename to write curve.</param>
    /// </signature>

    if(curveList) {
        if(format == "cur" || format == "wr" || format == "WR")
        {
            if(File.Exists(fname + ".cur")) File.Delete(fname + ".cur");
            DialogueInput("WR WR", "#" + curveList, "", fname + ".cur");
        }
        else 
        {  
            if(File.Exists(fname + ".csv")) File.Delete(fname + ".csv");
            DialogueInput("WR", format, "#" + curveList, "", fname + ".csv");
        }
    }
}



function remove_all_curves_and_datums()
{
    // Removes all curves and datums from graphs

    remove_all_curves();
    remove_all_datums();
}



function remove_all_curves()
{
    // Removes all curves from graphs

    var id = Curve.HighestID();

    if(id > 0)
    {
        var c = Curve.First();

        while(c)
        {
            c.RemoveFromGraph();
            c = c.Next();
        }
    }
}



function remove_all_datums()
{
    // Removes all datums from graphs

    var d = Datum.First();

    while(d)
    {
        d.RemoveFromGraph();
        d = d.Next();
    }
}



function write_unit_vars(f, valid)
{
    // Writes variables for units to the variables file <f>

    var name  = "MODEL_UNITS"
    var desc  = "Model units";
    var type  = "String";
    var value;

    if(valid) value = oGlblData.unit_length + "  " + oGlblData.unit_mass + "  " + oGlblData.unit_time;
    else      value = "NOT DEFINED / INVALID. UNABLE TO POST-PROCESS RESULTS.";

    f.Writeln(name + "," + value + "," + desc + "," + type);
}



function write_variable(f, name, value, desc, type)
{
    // Writes variable to file <f>

    f.Writeln(name + "," + value + "," + desc + "," + type);
}



function scale_graph(curves, max_limit)
{

    // Get limits of curves

    var max_x = -1e20;
    var min_x =  1e20;
    var max_y = -1e20;
    var min_y =  1e20;

    var p;

    if(Array.isArray(curves))
    {
        // Array of curves

        for(var i=0; i<curves.length; i++)
        {
            if(curves[i])
            {
                var n = curves[i].npoints;

                for(var j=1; j<=n; j++)
                {
                    p = curves[i].GetPoint(j);

                    max_x = Math.max(max_x, p[0]);
                    min_x = Math.min(min_x, p[0]);

                    max_y = Math.max(max_y, p[1]);
                    min_y = Math.min(min_y, p[1]);
                }
            }
        }
    }
    else
    {
        // Single curve

        if(curves) 
        {
            var n = curves.npoints;

            for(var j=1; j<=n; j++)
            {
                p = curves.GetPoint(j);

                max_x = Math.max(max_x, p[0]);
                min_x = Math.min(min_x, p[0]);

                max_y = Math.max(max_y, p[1]);
                min_y = Math.min(min_y, p[1]);
            }
        }
    }

    max_y *= 1.1;
    min_y *= 1.1;

    // Make sure we'll see the datum lines as well

    max_limit *= 1.1;

    if(max_limit > 0 &&
        max_limit > max_y)
    {
        max_y = max_limit;
    }
    else if(max_limit < 0 &&
        max_limit < min_y)
    {
        min_y = max_limit;
    }

    // Now scale graph

    if(max_x == -1e20 || min_x == 1e20 ||
        max_y == -1e20 || min_y == 1e20)
    {
        DialogueInput("/AU");
    }
    else
    {
        max_x = max_x.toFixed(2);
        min_x = min_x.toFixed(2);
        max_y = max_y.toFixed(2);
        min_y = min_y.toFixed(2);

        DialogueInput("/DE AU XMX", max_x);
        DialogueInput("/DE AU XMN", min_x);
        DialogueInput("/DE AU YMX", max_y);
        DialogueInput("/DE AU YMN", min_y);

        DialogueInput("/PLOT");
    }
}



function set_labels(c, label, xlabel, ylabel)
{
    // Sets label of curve <c> and x and y-axis labels

    c.label        = label;
    c.x_axis_label = xlabel;
    c.y_axis_label = ylabel;
}



function set_line_style(c, col)
{
    // Sets line style of curve <c>

    c.colour = col;
    c.style  = LineStyle.SOLID;
    c.width  = LineWidth.BOLD;
    c.symbol = Symbol.NONE;
}



function get_row_and_column_from_fname(fname, oData)
{
    // Get row and column numbers from filename: <zone>_<row>_<col>.key
    // returning them on the <oData> object as properties <.row> and <.col>
    //
    // Returns true on success, false otherwise

    var rc = fname.split("_");

    if(rc.length < 3)      return false;
    if(rc[1] == undefined) return false;
    if(rc[2] == undefined) return false;

    if (isNaN(parseInt(rc[1]))) return false;
    oData.row = rc[1];

    var end = rc[2].split(".");  // Get <col> from <col>.key or <col>.k
    if (isNaN(parseInt(end[0]))) return false;
    oData.col = end[0];

    return true;
}



function is_lst_case_option(fname)
{
    // Function to check if the LST file <fname> is in the format used for *CASE model build.
    // Returns <true> if it is <false> otherwise

    var retval = false;

    var f_lst = new File(fname, File.READ);

    var line;

    // Look for a line beginning with "$case

    while ( (line = f_lst.FindLineStarting("$case") ) != File.EOF)
    {
        retval = true;
    }

    f_lst.Close();

    return retval;
}



function get_case_res_dir(fname)
{
    // Gets the results directory for *CASE models from the LST file <fname>

    var dir = "";

    var f_lst = new File(fname, File.READ);

    var line;

    // Look for a line not beginning with '$'

    while ( (line = f_lst.ReadLine()) != File.EOF)
    {
        if(line[0] != '$')
        {
            // Split the line

            var list = line.split(",");

            // Get the directory

            if(list.length >= 1)
            {
                // Get directory of results

                var dirs;

                dirs = list[0].split("/");                         // Unix
                if(dirs.length == 1) dirs = list[0].split("\\");   // Windows

                // Assemble path

                dir = dirs[0] + "/";
                for(var j=1; j<(dirs.length-1); j++) dir += dirs[j] + "/";
            }
        }
    }

    f_lst.Close();

    return dir;
}



function get_case_master_file(fname)
{
    // Gets the master file for *CASE models from the LST file <fname>

    var master_file = "";

    var f_lst = new File(fname, File.READ);

    var line;

    // Look for a line not beginning with '$'

    while ( (line = f_lst.ReadLine()) != File.EOF)
    {
        if(line[0] != '$')
        {
            // Split the line

            var list = line.split(",");

            // Get the master filename

            if(list.length >= 1)
            {
                // Split into directories

                var dirs;

                dirs = list[0].split("/");                         // Unix
                if(dirs.length == 1) dirs = list[0].split("\\");   // Windows

                // Get filename

                master_file = dirs[dirs.length-1];

                // Remove anything after '.'

                var root = master_file.split(".");

                master_file = root[0];
            }
        }
    }

    f_lst.Close();

    return master_file;
}



function get_case_key_file(dir, master_file, case_id)
{
    // Tries to create a keyword filename for the current case <case_id>
    // which will pick up results in directory <dir>. 
    //
    // Looks for LSTC or ARUP type results to construct a valid keyword filename.


    var retval = "";

    // Look for LSTC file: case<case_id>.d3thdt

    var fname = dir + "/case" + case_id + ".d3thdt";

    if(File.Exists(fname)) { retval = dir + "/case" + case_id + ".key";  return retval; }

    // Look for ARUP file: case<case_id>.<master_file>.thf

    var fname = dir + "/case" + case_id + "." + master_file + ".thf";

    if(File.Exists(fname)) { retval = dir + "/case" + case_id + "." + master_file + ".key";  return retval; }

    // Look for binout: case<case_id>.binout  (same for LSTC or ARUP)

    var fname = dir + "/case" + case_id + ".binout";

    if(File.Exists(fname)) { retval = dir + "/case" + case_id + ".key";  return retval; }

    // Look for binoutxxxx: case<case_id>.binoutxxxx (same for LSTC or ARUP)  

    for(var i=0; i<10000; i++)
    {
        var ext;

        if(i < 10)    ext = "000" + i;
        else if(i < 100)   ext = "00"  + i;
        else if(i < 1000)  ext = "0"   + i;
        else if(i < 10000) ext = ""    + i;

        var fname = dir + "/case" + case_id + ".binout" + ext;

        if(File.Exists(fname)) { retval = dir + "/case" + case_id + ".key";  return retval; }
    }

    return retval;
}



function unset_this_units(model_id)
{
    // The templates assume units are not set in T/HIS, so make sure
    // they are undefined as they could have been set by preferences

    DialogueInput("/UNITS MO " + model_id, "UN");
    DialogueInput("/UNITS DI UN");
}


function set_x_axis_precision()
{
    //this sets number of decimal places on x axis to 3 if time in seconds and 0 otherwise
    
    if(oGlblData.unit_time == "s")
    {
        DialogueInput("/LAYOUT GRAPH 1 X-AXIS PRECISION 3");
    }
    else
    {
        DialogueInput("/LAYOUT GRAPH 1 X-AXIS PRECISION 0");
    }
}


function calculate_unit_constants()
{
    // Calculate constants based on the model units

    var len  = oGlblData.unit_length;
    var mass = oGlblData.unit_mass;
    var time = oGlblData.unit_time;

    oGlblData.len_factor  = 1.0;
    oGlblData.mass_factor = 1.0;
    oGlblData.time_factor = 1.0;

    // Factors to convert from metres, kgs to model units
    var len_factor;

    if(len == "m")    len_factor = 1.0;
    else if(len == "cm")   len_factor = 100.0;
    else if(len == "mm")   len_factor = 1000.0;
    else if(len == "inch") len_factor = 39.37008;
    else if(len == "ft")   len_factor = 3.28084;

    var mass_factor;
    if(mass == "tonne") mass_factor = 0.001;
    else if(mass == "kg")    mass_factor = 1.0;
    else if(mass == "lb")    mass_factor = 2.204586;
    else if(mass == "slug")  mass_factor = 0.06852178;
    else if(mass == "gram")  mass_factor = 1000.0;

    var time_factor;
    if(time == "s")   time_factor = 1.0;
    else if(time == "ms")  time_factor = 1000.0;
    else if(time == "us")  time_factor = 1000000.0;

    oGlblData.len_factor  = len_factor;
    oGlblData.mass_factor = mass_factor;
    oGlblData.time_factor = time_factor;

    // Gravity constant in model units (9.81 m/s^2)

    oGlblData.g_constant = 9.81 * len_factor / (time_factor*time_factor);

    // Factor to divide force by to convert to kN

    oGlblData.kn_factor = 1000 * (mass_factor * len_factor / (time_factor * time_factor));

    // Factor to divide moment by to convert to Nm

    oGlblData.nm_factor = mass_factor * len_factor * len_factor / (time_factor * time_factor);

    // Factor to divide length by to convert to mm

    oGlblData.mm_factor = len_factor / 1000;

    // Factor to divide length by to convert to cm

    oGlblData.cm_factor = len_factor / 100;

}



function check_unit_values(model_id)
{
    // Checks that the model units specified by the user are valid.
    // Returns false if not.

    // Valid lengths: m, cm, mm, inch, ft

    switch(oGlblData.unit_length)
    {
        case "m":
        case "cm":
        case "mm":
        case "inch":
        case "ft":    break;

        default:      return false;
    }

    // Valid mass: tonne, kg, lb, slug, gm

    switch(oGlblData.unit_mass)
    {
        case "tonne":
        case "kg":
        case "lb":
        case "slug":
        case "gm":    break;

        default:      return false;
    }

    // Valid time: s, ms, us

    switch(oGlblData.unit_time)
    {
        case "s":
        case "ms":
        case "us":    

            var curve_id = Curve.FirstFreeID();

            if (Model.Total()==1)
            {
                DialogueInput("/MODEL DATA GLOBAL DT ", "#" + curve_id);
            }
            else if (Model.Total()>1)
            {
                ErrorMessage("Warning: Time unit check can only handle one model at a time. Report generation would be terminated." );
                if (batch_mode=="false")
                    Window.Error("Error", "Time unit check can only handle one model at a time. Report generation would be terminated.");
                return false;
            }
            else
            {
                ErrorMessage("Warning: No model loaded to check the model and user input time units. Report generation would be terminated." );
                if (batch_mode=="false")
                    Window.Error("Error", "No model loaded to check the model and user input time units. Report generation would be terminated.");
                return false;
            }	

            if (Curve.Exists(curve_id))
                var c_time = Curve.GetFromID(curve_id);
            else
                var c_time = undefined;

            if (c_time)
            {
// Specified time is seconds but time duration looks more like milliseconds
                if ( c_time.xmax > 15 && c_time.xmax <= 15000 && oGlblData.unit_time == "s" )
                {
                    if (batch_mode=="false")
                    {
                        var ans = Window.Warning("Time units", "Based on the analysis duration of "+c_time.xmax.toPrecision(4)+", " +
                            "the time units appear to be milliseconds rather than the seconds you specified. " +
                            "Do you want to convert to milliseconds instead?", Window.YES | Window.NO);

                        if (ans == Window.YES)
                            oGlblData.unit_time = "ms";
                    }
                    else
                        WarningMessage("Time units appear to be milliseconds rather than seconds. This could affect the results." );

                }
// Specified time is seconds or milliseconds but time duration looks more like microseconds
                else if ( c_time.xmax > 15000 && ( oGlblData.unit_time == "ms" || oGlblData.unit_time == "s" ) ) 
                {
                    if (batch_mode == "false")
                    {
                        var ans = Window.Warning("Time units", "Based on the analysis duration of "+c_time.xmax.toPrecision(4)+", " +
                            "the time units appear to be microseconds rather than the \""+ oGlblData.unit_time +"\" you specified. " +
                            "Do you want to convert to microseconds instead?", Window.YES | Window.NO);

                        if (ans == Window.YES)
                            oGlblData.unit_time = "us";
                    }
                    else
                        WarningMessage("Time units appear to be microseconds rather than \"" + oGlblData.unit_time 
                            + "\". This could affect the results." );
                }
// Specified time is milliseconds or microseconds but time duration looks more like seconds
                else if ( c_time.xmax < 15 && ( oGlblData.unit_time == "ms" || oGlblData.unit_time == "us" ) ) 
                {
                    if (batch_mode == "false")
                    {
                        var ans = Window.Warning("Time units", "Based on the analysis duration of "+c_time.xmax.toPrecision(4)+", " +
                            "the time units appear to be seconds rather than the \"" + oGlblData.unit_time +"\" you specified. " +
                            "Do you want to convert to seconds instead?", 
                            Window.YES|Window.NO);

                        if (ans == Window.YES)
                            oGlblData.unit_time = "s";
            }
                    else
                        WarningMessage("Time units appear to be seconds rather than \"" + oGlblData.unit_time 
                            + "\". This could affect the results." );
                }
            }
            break;

        default:      return false;
    }

    return true;
}



function setup_this_graph()
{
    // Sets up the graph in T/His
    //
    //  - background colour to white and foreground to black
    //  - turn off model prefix for curve labels
    //  - font size = 14, 18 and 24 for title
    //  - show HIC and 3ms values
    //  - don't convert from ms->s when filtering

    DialogueInput("/DEFAULT BA WHITE");
    DialogueInput("/DEFAULT FO BLACK");

    DialogueInput("/DEFAULT MP OFF");

    setFontSizeBasedOnPPU();

    DialogueInput("/PREFERENCES SHOW HIC YES");
    DialogueInput("/PREFERENCES SHOW 3MS YES");
    DialogueInput("/PREFERENCES CONV NO");

    DialogueInput("/LAYOUT GRAPH 1 LEGEND FORMAT AUTO");
    DialogueInput("/LAYOUT GRAPH 1 LEGEND COLUMNS 1");
    DialogueInput("/LAYOUT GRAPH 1 Y-AXIS PRECISION 1");
    
    set_x_axis_precision();  //this sets number of decimal places on x axis to 3 if time in seconds and 0 otherwise
}



// This function returns this data curve. 
// 
function read_data(type, model_id, id, comp)
{
    /// <signature>
    /// <summary>Reads in data from a model.</summary>
    /// <param name="type" type="String" optional="false">The component type, e.g. "NODE", "BEAM BASIC", "SPRING"</param>
    /// <param name="model_id" type="Number" optional="false">Model number</param>
    /// <param name="id" type="Number" optional="false">Entity ID to read</param>
    /// <param name="comp" type="String" optional="false">Component name, e.g. "AX"</param>
    /// <returns type="Curve"/>
    /// </signature>

    // Reads in data for entity <type> <id>, component <comp>
    //
    // Assumes a model has been read in.
    //
    // Returns the curve object the data was read into.

    // Get model. 

    if (Model.Total() == 1)
        var current_model = Model.GetFromID(Model.HighestID());
    else
        var current_model = Model.GetFromID(model_id);

    var entity_type = new Array();


    if (type.indexOf(" ")>=0)
        entity_type = type.split(" ");
    else
        entity_type[0] = type;

    // check if the entity id/label entered by the user exists
    // and contain the desired component
    if ( entity_type[0].toUpperCase() == "SPRING" && entity_type[1].toUpperCase() == "ROT")
        var exists = check_entity_if_exists(entity_type[0].toUpperCase(), Entity.SPRING, id, current_model, comp);
    else if ( entity_type[0].toUpperCase() == "SPRING" && ( entity_type[1].toUpperCase() == "TR" || entity_type[1].toUpperCase() == "TRANS" ))
        var exists = check_entity_if_exists(entity_type[0].toUpperCase(), Entity.SPRING, id, current_model, comp);
    else if ( entity_type[0].toUpperCase() == "BEAM" && entity_type[1].toUpperCase() == "BASIC" )
        var exists = check_entity_if_exists(entity_type[0].toUpperCase(), Entity.BEAM, id, current_model, comp);
    else if ( entity_type[0].toUpperCase() == "NODE" )
        var exists = check_entity_if_exists(entity_type[0].toUpperCase(), Entity.NODE, id, current_model, comp);
    else if ( entity_type[0].toUpperCase() == "SECTION" || entity_type[0].toUpperCase() == "SEC" )
        var exists = check_entity_if_exists(entity_type[0].toUpperCase(), Entity.X_SECTION, id, current_model, comp);
    else if ( entity_type[0].toUpperCase() == "JOINT" && entity_type[1].toUpperCase() == "TR" )
        var exists = check_entity_if_exists(entity_type[0].toUpperCase(), Entity.JOINT, id, current_model, comp);
    else if ( entity_type[0].toUpperCase() == "GLOBAL" )
        var exists = true;

    if (exists)
    {
        var curve_id = Curve.FirstFreeID();

        if (Model.Total() == 1)   // for single models you don't need to specify the model id
        {
            if(!isNaN(id)) DialogueInput("/MODEL DATA " + type + " "  + id,       comp, "#" + curve_id);  // ID as number
            else           DialogueInput("/MODEL DATA " + type + ' "' + id + '"', comp, "#" + curve_id);  // ID as string (name)
        }
        else 
        {
            if(!isNaN(id)) DialogueInput("/MODEL DATA " + type + " m" + model_id + "/" +       id,       comp, "#" + curve_id);  // ID as number
            else           DialogueInput("/MODEL DATA " + type + " m" + model_id + "/" + '"' + id + '"', comp, "#" + curve_id);  // ID as string (name)
        }

        if (Curve.Exists(curve_id))
            return Curve.GetFromID(curve_id);
        else
            return undefined;
    }
    else
        return undefined;
}



function scale_graph_2(curves, max_limit, min_limit)
{
    // As above, but scaling for graphs with +ve and -ve datums

    // Get limits of curves

    var max_x = -1e20;
    var min_x =  1e20;
    var max_y = -1e20;
    var min_y =  1e20;

    var p;

    if(Array.isArray(curves))
    {
        // Array of curves

        for(var i=0; i<curves.length; i++)
        {
            if(curves[i])
            {
                var n = curves[i].npoints;

                for(var j=1; j<=n; j++)
                {
                    p = curves[i].GetPoint(j);

                    max_x = Math.max(max_x, p[0]);
                    min_x = Math.min(min_x, p[0]);

                    max_y = Math.max(max_y, p[1]);
                    min_y = Math.min(min_y, p[1]);
                }
            }
        }
    }
    else
    {
        // Single curve

        if(curves)
        {
            var n = curves.npoints;

            for(var j=1; j<=n; j++)
            {
                p = curves.GetPoint(j);

                max_x = Math.max(max_x, p[0]);
                min_x = Math.min(min_x, p[0]);

                max_y = Math.max(max_y, p[1]);
                min_y = Math.min(min_y, p[1]);
            }
        }
    }

    max_y *= 1.1;
    min_y *= 1.1;

    // Make sure we'll see the datum lines as well

    max_limit *= 1.1;
    min_limit *= 1.1;

    if(max_limit > 0 &&
        max_limit > max_y)
    {
        max_y = max_limit;
    }
    else if(max_limit < 0 &&
        max_limit < min_y)
    {
        min_y = max_limit;
    }

    if(min_limit > 0 &&
        min_limit > max_y)
    {
        max_y = min_limit;
    }
    else if(min_limit < 0 &&
        min_limit < min_y)
    {
        min_y = min_limit;
    }

    // Now scale graph

    if(max_x == -1e20 || min_x == 1e20 ||
        max_y == -1e20 || min_y == 1e20)
    {
        DialogueInput("/AU");
    }
    else
    {
        max_x = max_x.toFixed(2);
        min_x = min_x.toFixed(2);
        max_y = max_y.toFixed(2);
        min_y = min_y.toFixed(2);

        DialogueInput("/DE AU XMX", max_x);
        DialogueInput("/DE AU XMN", min_x);
        DialogueInput("/DE AU YMX", max_y);
        DialogueInput("/DE AU YMN", min_y);

        DialogueInput("/PLOT");
    }
}



function draw_constant_datums(good, adqt, marg, weak, symmetric)
{
    // Creates datum lines to show ratings
    // Draw both positive and negative datums if symmetric variable has a value


    // Delete any datums that might already exist 

    if(Datum.Exists("GOOD"))       Datum.Delete("GOOD"); 
    if(Datum.Exists("ADEQUATE"))   Datum.Delete("ADEQUATE"); 
    if(Datum.Exists("ACCEPTABLE")) Datum.Delete("ACCEPTABLE");
    if(Datum.Exists("MARGINAL"))   Datum.Delete("MARGINAL"); 
    if(Datum.Exists("WEAK"))       Datum.Delete("WEAK"); 
    if(Datum.Exists("GOOD2"))       Datum.Delete("GOOD2"); 
    if(Datum.Exists("ADEQUATE2"))   Datum.Delete("ADEQUATE2"); 
    if(Datum.Exists("ACCEPTABLE2")) Datum.Delete("ACCEPTABLE2");
    if(Datum.Exists("MARGINAL2"))   Datum.Delete("MARGINAL2"); 
    if(Datum.Exists("WEAK2"))       Datum.Delete("WEAK2"); 

    // Make limits positive if plotting both positive and negative datums
    if(symmetric != undefined && symmetric == true)
    {
        if(good!=null) good = Math.abs(good);
        if(adqt!=null) adqt = Math.abs(adqt);
        if(marg!=null) marg = Math.abs(marg);
        if(weak!=null) weak = Math.abs(weak);
    }
    
    // Create positive datums if limits are positive
    
    if(good > 0)
    {
        // Create new datums and colour in. The order which these are defined in matters

        var d_good;
        var d_adqt;
        var d_marg;
        var d_weak;

        if(weak != null) d_weak = new Datum("WEAK",     Datum.CONSTANT_Y, weak);
        if(marg != null) d_marg = new Datum("MARGINAL", Datum.CONSTANT_Y, marg); 
        if(adqt != null) d_adqt = new Datum("ADEQUATE", Datum.CONSTANT_Y, adqt);
        if(good != null) d_good = new Datum("GOOD",     Datum.CONSTANT_Y, good);

        if(weak != null) d_weak.line_colour = EuroNCAPColour.BROWN;
        if(marg != null) d_marg.line_colour = EuroNCAPColour.ORANGE;
        if(adqt != null) d_adqt.line_colour = EuroNCAPColour.YELLOW;
        if(good != null) d_good.line_colour = EuroNCAPColour.GREEN;

        if(weak != null) d_weak.fill_colour_above = EuroNCAPColour.RED;

        if(marg != null && weak != null)
            d_weak.fill_colour_below = EuroNCAPColour.BROWN;
        else if(marg != null)    
            d_marg.fill_colour_above = EuroNCAPColour.RED;

        if(marg != null) d_marg.fill_colour_below = EuroNCAPColour.ORANGE;
        if(adqt != null) d_adqt.fill_colour_below = EuroNCAPColour.YELLOW;
        if(good != null) d_good.fill_colour_below = EuroNCAPColour.GREEN;
    }

    // Make limits negative to plot negative datums as well if plotting both positive and negative datums
    if(symmetric != undefined && symmetric == true)
    {
        if(good!=null) good = -good;
        if(adqt!=null) adqt = -adqt;
        if(marg!=null) marg = -marg;
        if(weak!=null) weak = -weak;
    }

    // Create negative datums if limits are negative

    if(good < 0)
    {    

        // Create new datums and colour in. 
        // Using differnt variables so that positive and negative datums can be plotted on the same plot.

        var d_good2;
        var d_adqt2;
        var d_marg2;
        var d_weak2;
        
        if(good != null) d_good2 = new Datum("GOOD2",     Datum.CONSTANT_Y, good);
        if(adqt != null) d_adqt2 = new Datum("ADEQUATE2", Datum.CONSTANT_Y, adqt);
        if(marg != null) d_marg2 = new Datum("MARGINAL2", Datum.CONSTANT_Y, marg);
        if(weak != null) d_weak2 = new Datum("WEAK2",     Datum.CONSTANT_Y, weak);
        
        if(good != null) d_good2.line_colour = EuroNCAPColour.GREEN;
        if(adqt != null) d_adqt2.line_colour = EuroNCAPColour.YELLOW;
        if(marg != null) d_marg2.line_colour = EuroNCAPColour.ORANGE;
        if(weak != null) d_weak2.line_colour = EuroNCAPColour.BROWN; 
        
        // if only plotting negative datums, the area above the good datum needs to be filled green
        if(symmetric == undefined || (symmetric != undefined && symmetric == false))
        {
           if(good != null) d_good2.fill_colour_above = EuroNCAPColour.GREEN;
        }
        if(good != null) d_good2.fill_colour_below = EuroNCAPColour.YELLOW;
        if(adqt != null) d_adqt2.fill_colour_below = EuroNCAPColour.ORANGE;

        if(marg != null && weak != null)
            d_marg2.fill_colour_below = EuroNCAPColour.BROWN;
        else if(marg != null)
            d_marg2.fill_colour_below = EuroNCAPColour.RED;

        if(weak != null) d_weak2.fill_colour_below = EuroNCAPColour.RED;   
    }      
}



function draw_variable_datums(t, time, good, adqt, marg, weak)
{
    // Creates datums for variable limits up to time <t>;


    // Delete any that might already exist

    if(Datum.Exists("GOOD"))       Datum.Delete("GOOD");
    if(Datum.Exists("ADEQUATE"))   Datum.Delete("ADEQUATE");
    if(Datum.Exists("ACCEPTABLE")) Datum.Delete("ACCEPTABLE");
    if(Datum.Exists("MARGINAL"))   Datum.Delete("MARGINAL");
    if(Datum.Exists("WEAK"))       Datum.Delete("WEAK");

    // Create new ones and colour in

    var good_limit = new Array();
    var adqt_limit = new Array();
    var marg_limit = new Array();
    var weak_limit = new Array();


    for(var i=0; i<time.length; i++)
    {
        if(good)
        {
            good_limit[2*i + 0] = time[i];
            good_limit[2*i + 1] = good[i];
        }

        if(adqt)
        {
            adqt_limit[2*i + 0] = time[i];
            adqt_limit[2*i + 1] = adqt[i];
        }

        if(marg)
        {
            marg_limit[2*i + 0] = time[i];
            marg_limit[2*i + 1] = marg[i];
        }

        if(weak)
        {
            weak_limit[2*i + 0] = time[i];
            weak_limit[2*i + 1] = weak[i];
        }
    }

    // Points at time <t>

    if(good) { good_limit[2*good.length] = t;  good_limit[2*good.length + 1] = good[good.length-1]; }
    if(adqt) { adqt_limit[2*adqt.length] = t;  adqt_limit[2*adqt.length + 1] = adqt[adqt.length-1]; }
    if(marg) { marg_limit[2*marg.length] = t;  marg_limit[2*marg.length + 1] = marg[marg.length-1]; }
    if(weak) { weak_limit[2*weak.length] = t;  weak_limit[2*weak.length + 1] = weak[weak.length-1]; }

    // Create datums

    var d_good, d_adqt, d_marg, d_weak;

    if(good)         { d_good = new Datum("GOOD",     Datum.POINTS, good_limit); d_good.line_colour = EuroNCAPColour.GREEN;  d_good.fill_colour_below = EuroNCAPColour.GREEN;  d_good.fill_colour_above = EuroNCAPColour.YELLOW; }
    if(adqt)         { d_adqt = new Datum("ADEQUATE", Datum.POINTS, adqt_limit); d_adqt.line_colour = EuroNCAPColour.YELLOW; d_adqt.fill_colour_above = EuroNCAPColour.ORANGE;                                                   }
    if(marg && weak) { d_marg = new Datum("MARGINAL", Datum.POINTS, marg_limit); d_marg.line_colour = EuroNCAPColour.ORANGE; d_marg.fill_colour_above = EuroNCAPColour.BROWN;                                                    }
    else if(marg)    { d_marg = new Datum("MARGINAL", Datum.POINTS, marg_limit); d_marg.line_colour = EuroNCAPColour.ORANGE; d_marg.fill_colour_above = EuroNCAPColour.RED;                                                      }
    if(weak)         { d_weak = new Datum("WEAK",     Datum.POINTS, weak_limit); d_weak.line_colour = EuroNCAPColour.BROWN;  d_weak.fill_colour_above = EuroNCAPColour.RED;                                                      }
}



function write_missing_result_variables(f, missing)
{
    if(missing)
    {
        var name  = "MISSING_RESULTS_MESSAGE";
        var value = "Unable to find results for RESULTS_DIR/DEFAULT_JOB";
        var desc  = "Could not find any results";
        var type  = "String";

        f.Writeln(name + "," + value + "," + desc + "," + type);
    }
    else
    {
        var name  = "MISSING_RESULTS_MESSAGE";
        var value = "";
        var desc  = "Blank as no results were missing";
        var type  = "String";

        f.Writeln(name + "," + value + "," + desc + "," + type);
    }
}



function sliding_scale(val, hi_perf, lo_perf, hi_score, lo_score)
{
    var retval = 0.0;

    if(val < hi_perf) retval = hi_score;
    else if(val > lo_perf) retval = lo_score;
    else                   retval = hi_score + ( (val-hi_perf) * (lo_score-hi_score) / (lo_perf-hi_perf) );

    return retval;
}



function get_constant_limit_results(c, lo_limit, hi_limit, lo_score, hi_score, symmetric)
{
    // Calculates results for criteria that /as constant limits
    // Returns the rating and value. 
    //
    // <c> is the curve
    // <good>, <adqt>, <marg> and <weak> are the constant limits

    var ret = new Array(2);

    var val    = 0.0;
    var rating = 0.0; 
    
    // If symmetric variable has a value then the curve should be checked against positive and negative limits. 
    // This is done by taking the absolute maximum value of the curve and check that against the positive limits
    
    if (symmetric != undefined && symmetric == true) { val = Math.max(c.ymax,Math.abs(c.ymin)); rating = sliding_scale(val, lo_limit, hi_limit,  hi_score, lo_score); }
    else if (lo_limit > 0) { val =  c.ymax; rating = sliding_scale(val,  lo_limit,  hi_limit,  hi_score, lo_score); }
    else             { val = -c.ymin; rating = sliding_scale(val, -lo_limit, -hi_limit,  hi_score, lo_score); }    

    ret[0] = rating;
    ret[1] = val;

    return ret;  
}



function read_xyz_accelerations(model_id, node_x, node_y, node_z)
{
    /// <signature>
    /// <summary>Reads in X, Y, Z accelerations from a model.</summary>
    /// <param name="model_id" type="Number" optional="false">Model number</param>
    /// <param name="node" type="Number" optional="false">Node id to read X, Y and Z accelerations</param>
    /// <returns type="Object"/>
    /// </signature>

    /// <signature>
    /// <summary>Reads in X, Y, Z accelerations from a model.</summary>
    /// <param name="model_id" type="Number" optional="false">Model number</param>
    /// <param name="node_x" type="Number" optional="false">Node id to read X acceleration</param>
    /// <param name="node_y" type="Number" optional="false">Node id to read Y acceleration</param>
    /// <param name="node_z" type="Number" optional="false">Node id to read Z acceleration</param>
    /// <returns type="Object"/>
    /// </signature>

    var obj = {};

    if(arguments.length == 2)
    {
        node_x = node_x;
        node_y = node_x;
        node_z = node_x;
    }

    var current_model = Model.GetFromID(model_id) ;

    obj.ax = read_data("NODE", model_id, node_x, "AX");
    obj.ay = read_data("NODE", model_id, node_y, "AY");
    obj.az = read_data("NODE", model_id, node_z, "AZ");

    return obj;
}



function convert_xyz_acc_to_g(curves)
{
    /// <signature>
    /// <summary>Convert X, Y, Z accelerations from model units to g.</summary>
    /// <param name="curves" type="Object" optional="false">Object with acceleration Curves .ax, .ay, .az properties</param>
    /// <returns type="Object"/>
    /// </signature>

    var obj = {};

    obj.gx = Operate.Div(curves.ax, oGlblData.g_constant);
    obj.gy = Operate.Div(curves.ay, oGlblData.g_constant);
    obj.gz = Operate.Div(curves.az, oGlblData.g_constant);

    return obj;
}



function convert_to_s_and_filter(c, filter_func, reg_dt)
{
    /// <signature>
    /// <summary>Converts a curve from model units to seconds then filters it.</summary>
    /// <param name="c" type="Curve" optional="false">Curve to convert and filter.</param>
    /// <param name="filter_func" type="Function" optional="false">Function used to filter curve. Expects two arguments, the curve and the regularisation timestep.</param>
    /// <param name="reg_dt" type="Number" optional="false">Regularisation timestep.</param>
    /// <returns type="Curve"/>
    /// </signature>

    var c_gx_s = Operate.Dix(c, oGlblData.time_factor);

    return filter_func(c_gx_s, reg_dt);
}



function check_entity_if_exists(entity, entity_type, id, model, comp)
{
    var num_of_entities = model.GetNumberOf(entity_type);

    // check if the desired component exists
    if (entity == "SPRING" )
        var component_const = get_Spring_Component(comp);
    else if (entity == "BEAM")
        var component_const = get_Beam_Component(comp);
    else if (entity == "NODE")
        var component_const = get_Node_Component(comp);
    else if (entity == "SECTION")
        var component_const = get_Sec_Component(comp);
    else if (entity == "JOINT")
        var component_const = get_Joint_Component(comp);		
    if (component_const == undefined )
        return false;


    var component_exists = model.QueryDataPresent(component_const, entity_type)

    if (!component_exists)
        return false;
    
    for (var i = 1; i <= num_of_entities; i++)
    {
        var ext_label = model.GetLabel(entity_type, i);

        if (isNaN(id))
        {
            if (id == model.GetName(entity_type, i))
                return true;
        }

        if (id == ext_label)
            return true;

    }

    return false;	
}



function get_Spring_Component (comp)
{
    if (comp == "ET" )
        return Component.SP_E;
    else if (comp == "RT" )
        return Component.SP_R;
    else
        return undefined;
}



function get_Beam_Component (comp)
{
    if (comp == "NX")
        return Component.BFX;
    else if (comp == "NY")
        return Component.BFY;
    else if (comp == "NZ")
        return Component.BFZ;
    else if (comp == "MX")
        return Component.BMXX;
    else if (comp == "MY")
        return Component.BMYY;
    else if (comp == "MZ")
        return Component.BMZZ;
    else 
        return undefined;
}



function get_Node_Component (comp)
{
    if (comp == "DX")
        return Component.DX;
    else if (comp == "DY")
        return Component.DY;
    else if (comp == "DZ")
        return Component.DZ;
    else if (comp == "DM")
        return Component.DM;
    else if (comp == "VX")
        return Component.VX;
    else if (comp == "VY")
        return Component.VY;
    else if (comp == "VZ")
        return Component.VZ;
    else if (comp == "VM")
        return Component.VM;
    else if (comp == "AX")
        return Component.AX;
    else if (comp == "AY")
        return Component.AY;
    else if (comp == "AZ")
        return Component.AZ;
    else if (comp == "AM")
        return Component.AM;
    else 
        return undefined;
}



function get_Sec_Component (comp)
{ 
    if (comp == "FX")
        return Component.XSEC_FX;
    else if (comp == "FY")
        return Component.XSEC_FY;
    else if (comp == "FZ")
        return Component.XSEC_FZ;
    else if (comp == "FM")
        return Component.XSEC_FM;
    else if (comp == "MX")
        return Component.XSEC_MX;
    else if (comp == "MY")
        return Component.XSEC_MY;
    else if (comp == "MZ")
        return Component.XSEC_MZ;
    else if (comp == "MM")
        return Component.XSEC_MM;
    else
        return undefined;

}



function get_Joint_Component (comp)
{ 
    if (comp == "XD")
        return Component.DX;
    else if (comp == "YD")
        return Component.DY;
    else
        return undefined;
}


//this screen resolution fixer for T/HIS image font size

function setFontSizeBasedOnPPU(){
 
    var ppu = Widget.PixelsPerUnit();

    //var supportedFontSizes = [8, 10, 12, 14, 18, 24];

    //1920x1080p
    //ppu = 3.15
    var big = 24;
    var med = 18;

    if (ppu > 6)
    { //4k screen
    //ppu = 6.3
        big = 12;
        med = 8;
    }
    else if (ppu > 5)
    { //betweeon 4k and 1440p
        big = 14;
        med = 10;
    }
    else if (ppu > 4)
    { //1440p
        big = 18;
        med = 12;
    }

    //set fonts
    DialogueInput("/FONT TIT DE "+big+" FOREGROUND");
    DialogueInput("/FONT XLA DE "+big+" FOREGROUND");
    DialogueInput("/FONT YLA DE "+big+" FOREGROUND");
    DialogueInput("/FONT LEG DE "+med+" FOREGROUND");
    DialogueInput("/FONT XUN DE "+med+" FOREGROUND");
    DialogueInput("/FONT YUN DE "+med+" FOREGROUND");

}
