/// <reference path="C:/SOURCE14/this_dir/this_js_api/this.intellisense.js" />
/// <reference path="C:/SOURCE14/reporter_dir/library/templates/scripts/this_common1.js" />

// Common functions for EuroNCAP Legform templates
var zone;

// Now try to open LST files 

if( (File.Exists(fname_lower) && File.IsFile(fname_lower)) ||
	(File.Exists(fname_upper) && File.IsFile(fname_upper)) )
{
	// Read the LST files and store the data

	var oGlblData = new Object();
	var data = new Array();

	for(var i=0; i<2; i++)
	{
		var fname;

		if(i==0) fname = fname_lower;
		if(i==1) fname = fname_upper;

		if( !File.Exists(fname) || !File.IsFile(fname) ) continue;


		// ------------------------------------------------------------------------------------------------
		// If models were built with the *CASE option then the LST file should contain:
		//
		//    MASTER_KEY_FILE,,,,                                       { this is the master keyword file that gets run }
		//    $case1:KEY_FILE,TEMPLATE,ZONE,X-COORD,Y-COORD             {   Case 1 keyword file                         }
		//    $case2:KEY_FILE,TEMPLATE,ZONE,X-COORD,Y-COORD             {   Case 2 keyword file                         }
		//    $case3:KEY_FILE,TEMPLATE,ZONE,X-COORD,Y-COORD             {   Case 3 keyword file                         }
		//    $case<n>:KEY_FILE,TEMPLATE,ZONE,X-COORD,Y-COORD     {   Case n keyword file                         }
		//
		// Lines starting with '$case<n>:' will have a fake KEY_FILE, but it will be in the form
		// <directory>/A_2_0.key, so we can use them to get the row and columns for each case.
		//
		// The results for each case will be in the same directory as the MASTER_KEY_FILE.
		//
		// The actual filenames of the results files for each case will depend on the format that they
		// were written out in.  We can try both and use the one that exists on the filesystem.
		//
		//   LSTC (default)        ARUP
		//   -------------         -----
		//   case1.d3thdt          case1.<master_key_file>.thf
		//   case2.d3thdt          case2.<master_key_file>.thf
		//   case3.d3thdt          case3.<master_key_file>.thf
		//
		// So we can construct a keyword file for each case which we store on the oData object.
		// T/His will then find the correct results files:
		//
		//   case1.key     or     case1.<master_key_file>.key
		//   case2.key            case2.<master_key_file>.key
		//   case3.key            case3.<master_key_file>.key
		// ------------------------------------------------------------------------------------------------

		// We need to detect if this is a *CASE LST file, so we know what to do when we read the file
		// If it is then get the directory where the results will be

		var case_option = is_lst_case_option(fname);

		var case_res_dir     = "";
		var case_master_file = "";

		if(case_option)
		{
			case_res_dir     = get_case_res_dir(fname);
			case_master_file = get_case_master_file(fname);
		}

		// Now read the LST file and store the data

		var f_lst = new File(fname, File.READ);

		var line;

		while ( (line = f_lst.ReadLine()) != undefined)
		{
			// Lines beginning with '$' are comments or 
			// beams or xsections to read data for in T/His
			//
			// or if they begin with '$case<id>:' they are lines for *CASE models

			if(line[0] == '$')
			{
				var list = line.split(",");
				var name = list[0].toLowerCase();

				if(list[1] != undefined)
				{
					if(case_option && 
							line[1] == "c" && line[2] == "a" && line[3] == "s" && line[4] == "e")
					{
						// *CASE model data
						// $case<id>:KEY_FILE,TEMPLATE,ZONE,X-COORD,Y-COORD

						// Get case id

						var case_id = "";
						var j = 5;
						while(line[j] != ":")
						{
							case_id += line[j];
							j++;
						}

						// Split the line

						var list = line.split(",");

						// Store an object with the data into the <data> array. (Must have at least three columns of data)

						if(list.length >= 3)
						{
							if ( check_zone_name(list[2],i) )
							{
								var oData = new Object();

								oData.res_file  = get_case_key_file(case_res_dir, case_master_file, case_id);
								oData.zone      = zone;    // 3rd column

								data.push(oData);
							}
						}
					}
					else
					{
						switch(name)
						{
							case "$unit_length":                 { oGlblData.unit_length                  = list[1];            break; }        
							case "$unit_mass":                   { oGlblData.unit_mass                    = list[1];            break; }
							case "$unit_time":                   { oGlblData.unit_time                    = list[1];            break; }

							case "$beam_femur1_id":              { oGlblData.beam_femur1_id               = parse_id(list[1]);  break; }
							case "$beam_femur2_id":              { oGlblData.beam_femur2_id               = parse_id(list[1]);  break; }

							case "$xsection_femur_upper_id":     { oGlblData.xsection_femur_upper_id      = parse_id(list[1]);  break; }
							case "$xsection_femur_centre_id":    { oGlblData.xsection_femur_centre_id     = parse_id(list[1]);  break; }
							case "$xsection_femur_lower_id":     { oGlblData.xsection_femur_lower_id      = parse_id(list[1]);  break; }

							case "$xsection_tibia_upper_id":     { oGlblData.xsection_tibia_upper_id      = parse_id(list[1]);  break; }
							case "$xsection_tibia_mid_upper_id": { oGlblData.xsection_tibia_mid_upper_id  = parse_id(list[1]);  break; }
							case "$xsection_tibia_mid_lower_id": { oGlblData.xsection_tibia_mid_lower_id  = parse_id(list[1]);  break; }
							case "$xsection_tibia_lower_id":     { oGlblData.xsection_tibia_lower_id      = parse_id(list[1]);  break; }

							case "$spring_pcl_id":               { oGlblData.spring_pcl_id                = parse_id(list[1]);  break; }
							case "$spring_acl_id":               { oGlblData.spring_acl_id                = parse_id(list[1]);  break; }
							case "$spring_mcl_id":               { oGlblData.spring_mcl_id                = parse_id(list[1]);  break; }
						}
					}
				}
			}
			else // Filenames (colum 1) and zones (column 3)
			{
				if(!case_option)  // Ignore if it's a lst file for *CASE models - this is the line for the master file
				{

					// Split the line

					var list = line.split(",");

					// Store an object with the data into the <data> array. (Must have at least three columns of data)

					if(list.length >= 3)
					{
						if ( check_zone_name(list[2],i) )
						{							
							var oData = new Object();

							oData.res_file  = list[0];
							oData.zone      = zone;    // 3rd column

							data.push(oData);
						}
					}
				}
			}
		}

		f_lst.Close();
    }
    
	// Open a csv file in the %IMAGES_DIR% to write variables to, for Reporter to pick up

	var f_vars = new File(images_dir + "/this_vars.csv", File.WRITE);

	// Setup graph

	setup_this_graph();

    // Switch graph borders off for this template (if we end up removing graph borders for all
    // library templates, we can move this to setup_this_graph() ).
    DialogueInput("/DEFAULT BD OFF");

	// Initialise variables to hold upper and lower scores

	oGlblData.upper_score = 0.0;
	oGlblData.femur_score = 0.0;
	oGlblData.knee_and_tibia_score = 0.0;
	oGlblData.score_sum_per_zone = 0.0;   //variable for CNCAP

	oGlblData.upper_num = 0;
	oGlblData.lower_num = 0;

	oGlblData.upper_min_col = 0;
	oGlblData.lower_min_col = 0;
	oGlblData.upper_max_col = 0;
	oGlblData.lower_max_col = 0;

	// Read the model in into T/HIS and do the calculations
    
    // Do different calculations depending on the regulatory body
    switch (template_reg)
    {
        case "KNCAP":
        case "EuroNCAP":
	        for(var i=0; i<data.length; i++)
	        {
	            if( (File.Exists(data[i].res_file) && File.IsFile(data[i].res_file)) ||
	                    case_option)
	            {
	                if(data[i].zone.substring(0,1).toUpperCase() == "U") do_upper_legform_calc(f_vars, data[i].zone);
	                else if (data[i].zone.substring(0,1).toUpperCase() == "L") do_lower_legform_calc(f_vars, data[i].zone);
	            }
	            else // Exit script and print an error message
	            {
	                ErrorMessage("Invalid keyword file in list file: " + data[i].res_file);
	                Exit();
	            }
	        }
    


			// Write out values for overall scores

	        if(oGlblData.upper_num > 0) oGlblData.upper_score = 4.5 * oGlblData.upper_score / oGlblData.upper_num;
			if(oGlblData.lower_num > 0) oGlblData.femur_score = 4.5 * oGlblData.femur_score / oGlblData.lower_num;
			if(oGlblData.lower_num > 0) oGlblData.knee_and_tibia_score = 9.0 * oGlblData.knee_and_tibia_score / oGlblData.lower_num;
    
			var total_legform = oGlblData.upper_score + oGlblData.femur_score + oGlblData.knee_and_tibia_score;
    
            //Variables specific to EuroNCAP/KNCAP
	        write_variable(f_vars, "UPPER_LEGFORM_TOTAL", oGlblData.upper_score.toFixed(3), "Upper Legform Score", "String");        
	        write_variable(f_vars, "UPPER_MIN_COL",       oGlblData.upper_min_col,          "Minimum column for upper leg", "String");
	        write_variable(f_vars, "UPPER_MAX_COL",       oGlblData.upper_max_col,          "Maximum column for upper leg", "String");
  

			//Variables common to both JNCAP and EuroNCAP/KNCAP
			write_variable(f_vars, "FEMUR_SCORE",         oGlblData.femur_score.toFixed(3),           "Total Femur Score", "String");
			write_variable(f_vars, "KNEE_AND_TIBIA_SCORE",oGlblData.knee_and_tibia_score.toFixed(3),  "Total Knee and Tibia Score", "String");
			write_variable(f_vars, "TOTAL_LEGFORM",       total_legform.toFixed(3),         "Total Legform Score", "String");

			write_variable(f_vars, "LOWER_MIN_COL",       oGlblData.lower_min_col,          "Minimum column for lower leg", "String");
			write_variable(f_vars, "LOWER_MAX_COL",       oGlblData.lower_max_col,          "Maximum column for lower leg", "String");
			
			write_variable(f_vars, "UPPER_MAX_POSS_SCORE",       legform_data.UPPER_MAX_POSS_SCORE,          "Total Legform Score", "String");          
			
			write_variable(f_vars, "FEMUR_MAX_POSS_SCORE",       legform_data.FEMUR_MAX_POSS_SCORE,          "Femur max possible Score", "String");
			write_variable(f_vars, "KNEE_AND_TIBIA_MAX_POSS_SCORE",legform_data.KNEE_AND_TIBIA_MAX_POSS_SCORE,"Knee and Tibia max possible Score", "String");          
			
			write_variable(f_vars, "UPPER_SINGLE_MAX_POSS_SCORE",legform_data.UPPER_SINGLE_MAX_POSS_SCORE,   "Total Legform Score", "String");               
			
			write_variable(f_vars, "APLI_MAX_POSS_SCORE",        legform_data.APLI_MAX_POSS_SCORE,           "Max APLI Score", "String");
			write_variable(f_vars, "MAX_POSS_SCORE",             legform_data.MAX_POSS_SCORE,                "Total Legform Score", "String");  

	        break; //End of EuroNCAP calcs

        case "JNCAP":
            var lower_assess_array_negative = [];
            var lower_assess_array_zero = [];
            var lower_assess_array_positive = [];

            // We loop through <data> twice.
            // The first time, we calculate the score for each impact point and determine the maximmum zone value.
            // The second time, we assign the (assumed to be Euro NCAP) impact points to the most appropriate
            // JNCAP zones (three zones L1, L2, L3 and the six segments L1A, L1B, L2A, L2B, L3A, L3B).
            // The user might not have provided a full sweep of leg impact points. The maximum zone value is 
            // assumed to be at the bumper reference line e.g. if the user provides Euro NCAP points
            // L-7, L-5, L-1, L0, L1 and L3, then the vehicle width is assumed to be 2*700 = 1400 mm.
            // In the second loop through <data>, the Euro NCAP points are assigned to the JNCAP zones based on
            // the assumption that each Euro NCAP point is spaced 100 mm from the vehicle centre line at L0.

            // First loop through <data> finds the maximum zone value and calculate the lower impact score
            var max_zone_value = 0;
            for(var i=0; i<data.length; i++)
	        {
                if( (File.Exists(data[i].res_file) && File.IsFile(data[i].res_file)) ||
	                    case_option)
	            {
                    if(data[i].zone.substring(0,1).toUpperCase() == "L")
                    {
                        // Calculate the lower leg score for this impact point and push it to an array so we can do hte overall calculation later
                        data[i].lower_assess_score = do_lower_legform_calc(f_vars, data[i].zone);
                        zone_value = Math.abs(parseInt(data[i].zone.substr(2)));
                        max_zone_value = Math.max(max_zone_value, zone_value);
                    }
	            }
	            else // Exit script and print an error message
	            {
	                ErrorMessage("Invalid keyword file in list file: " + data[i].res_file);
	                Exit();
	            }
            }
    
            
            //set up bins which separate the boundaries between the 6 segments. Using greater than or equal to, start at the highest boundary (one down from the max value) and work down
            var JNCAP_bins = [max_zone_value*(2/3), max_zone_value*(1/3), 0, -max_zone_value*(1/3), -max_zone_value*(2/3), -max_zone_value];

            //set up an array to put the JNCAP L1a/b L2a/b L3a/b results into, [0] = L1a, [1] = L1b ... [5] = L3b.
            // Initialise the array with <empty_segment_score> values to default to grey in the draw image
            var empty_segment_score = -1;
            var JNCAP_assess_array = new Array(6);
            for (var i = 0; i < JNCAP_bins.length; i++) JNCAP_assess_array[i] = empty_segment_score;

            // Second loop through <data> to put EuroNCAP results into JNCAP segments
            // The score for each JNCAP segment is taken as the worst score from the impact points found in that segment
            for(var i=0; i<data.length; i++)
            {
                var zone_value = parseInt(data[i].zone.substr(2));

                // loop through each time reducing the boundary until the zone fits into a segment. Then break out of the for loop
                for(var j=0; j < JNCAP_bins.length;j++)
                {
                    if(zone_value >= JNCAP_bins[j])
                    {
                        if (JNCAP_assess_array[j] == empty_segment_score) //if the segment is empty, set it to the assessment score
                        {
                            JNCAP_assess_array[j] = data[i].lower_assess_score;
                        }
                        else //if the segment already has a value, take the lower of the 2
                        {
                            JNCAP_assess_array[j] = Math.min(JNCAP_assess_array[i],data[i].lower_assess_score)
                        }
                        
                        break;
                    }
                }

            }

            //calculating results for JNCAP zones L1, L2 and L3 which is the average of results in LXa and LXb
            
            var JNCAP_zones = new Array(3);
            
            // If values in both L1a and L1b, average them both. If only values in one, take that value
            for (var i = 0; i<JNCAP_zones.length; i++)
            {
                var j = i*2;
                if(JNCAP_assess_array[j] != empty_segment_score && JNCAP_assess_array[j+1] != empty_segment_score )
                {
                    JNCAP_zones[i] = (JNCAP_assess_array[j] + JNCAP_assess_array[j+1]) / 2;
                }
                else if (JNCAP_assess_array[j] != empty_segment_score)
                {
                    JNCAP_zones[i] = JNCAP_assess_array[j];
                }
                else if (JNCAP_assess_array[j+1] != empty_segment_score)
                {
                    JNCAP_zones[i] = JNCAP_assess_array[j+1];
                }
                else 
                {
                    JNCAP_zones[i] = 0;
                }
            }


            //calculating the overall score
            var JNCAP_score_temp = 0.0;
            for (var i=0; i<JNCAP_zones.length; i++)
            {
                JNCAP_score_temp += JNCAP_zones[i]; 
            }
            
            var JNCAP_score = JNCAP_score_temp/JNCAP_zones.length; //averaging over L1 L2 and L3

            write_variable(f_vars, "JNCAP_L1",       JNCAP_zones[0].toFixed(3),          "JNCAP score for Zone L1", "String");
            write_variable(f_vars, "JNCAP_L2",       JNCAP_zones[1].toFixed(3),          "JNCAP score for Zone L2", "String");
            write_variable(f_vars, "JNCAP_L3",       JNCAP_zones[2].toFixed(3),          "JNCAP score for Zone L3", "String");
            write_variable(f_vars, "JNCAP_L1a",       JNCAP_assess_array[0].toFixed(3),          "JNCAP score for Zone L1a", "String");
            write_variable(f_vars, "JNCAP_L1b",       JNCAP_assess_array[1].toFixed(3),          "JNCAP score for Zone L1b", "String");
            write_variable(f_vars, "JNCAP_L2a",       JNCAP_assess_array[2].toFixed(3),          "JNCAP score for Zone L2a", "String");
            write_variable(f_vars, "JNCAP_L2b",       JNCAP_assess_array[3].toFixed(3),          "JNCAP score for Zone L2b", "String");
            write_variable(f_vars, "JNCAP_L3a",       JNCAP_assess_array[4].toFixed(3),          "JNCAP score for Zone L3a", "String");
            write_variable(f_vars, "JNCAP_L3b",       JNCAP_assess_array[5].toFixed(3),          "JNCAP score for Zone L3b", "String");

 

			// Write out values for overall scores

	        oGlblData.upper_score = ""; //No upper leg test for JNCAP to give reporter a blank string for front page to look as we want 
	        if(oGlblData.lower_num > 0) oGlblData.lower_score = 6.25 * oGlblData.lower_score / oGlblData.lower_num;
    
	        var total_legform = oGlblData.lower_score;          
            
            //Variables specific to JNCAP
	        write_variable(f_vars, "UPPER_LEGFORM_TOTAL", oGlblData.upper_score, "Upper Legform Score", "String");

	        break;
			
		case "CNCAP":
	        for(var i=0; i<data.length; i++)
	        {
	            if( (File.Exists(data[i].res_file) && File.IsFile(data[i].res_file)) ||
	                    case_option)
	            {
	                if (data[i].zone.substring(0,1).toUpperCase() == "L") do_lower_legform_calc(f_vars, data[i].zone);
	            }
	            else // Exit script and print an error message
	            {
	                ErrorMessage("Invalid keyword file in list file: " + data[i].res_file);
	                Exit();
	            }
	        }
    


			// Write out values for overall scores


			if(oGlblData.lower_num > 0) oGlblData.score_sum_per_zone = 5.0 * oGlblData.score_sum_per_zone / oGlblData.lower_num;
    
			var total_legform = oGlblData.score_sum_per_zone;
  


			write_variable(f_vars, "FEMUR_SCORE",         oGlblData.femur_score.toFixed(3),           "Total Femur Score", "String");
			write_variable(f_vars, "KNEE_AND_TIBIA_SCORE",oGlblData.knee_and_tibia_score.toFixed(3),  "Total Knee and Tibia Score", "String");
			write_variable(f_vars, "TOTAL_LEGFORM",       total_legform.toFixed(3),         "Total Legform Score", "String");

			write_variable(f_vars, "LOWER_MIN_COL",       oGlblData.lower_min_col,          "Minimum column for lower leg", "String");
			write_variable(f_vars, "LOWER_MAX_COL",       oGlblData.lower_max_col,          "Maximum column for lower leg", "String");

			write_variable(f_vars, "FEMUR_MAX_POSS_SCORE",       legform_data.FEMUR_MAX_POSS_SCORE,    "Femur max possible Score", "String");
			write_variable(f_vars, "TIBIA_MAX_POSS_SCORE",       legform_data.TIBIA_MAX_POSS_SCORE,    "Tibia max possible Score", "String");          
			write_variable(f_vars, "SINGLE_MAX_POSS_SCORE",      legform_data.SINGLE_MAX_POSS_SCORE,   "Single max Score", "String");               
			write_variable(f_vars, "MCL_MAX_POSS_SCORE",         legform_data.MCL_MAX_POSS_SCORE,      "Max MCL Score", "String");
			write_variable(f_vars, "MAX_POSS_SCORE",             legform_data.MAX_POSS_SCORE,          "Max Legform Score", "String");  

	        break; //End of CNCAP calcs

		default:
			//***Error message here***
			break;
    }
        
    f_vars.Close();

}

else // Exit script and print an error message
{
	ErrorMessage("Neither a valid upper nor lower list file was found.\n" +
				 "Lower: " + fname_lower + "\n" +
				 "Upper: " + fname_upper);
	Exit();
}


function do_upper_legform_calc(f, zone)
{
	// Calculates the scores for upper leg <zone> and writes relevant values
	//into the Reporter variable file <f>.

	// Get the results file for this <zone>

	var res_file;

	for(var i=0; i<data.length; i++)
	{
		if(data[i].zone == zone) res_file = data[i].res_file;
	}

	// Read in the model

	DialogueInput("/MODEL READ", res_file);

	// Make sure any units set in T/HIS (e.g. via preferences) are unset

	unset_this_units(1);
	
	// check if the model time units and input time units are consistent
	// If the units are not defined/valid then we cannot go any further.  

	if(!check_unit_values())
	{
	write_unit_vars(f_vars, false);
    ErrorMessage("Cannot find the model unit values. They could be missing from the input csv file or post-*END data." );
	Exit();
	}

	// Write the unit data to the variables file

	write_unit_vars(f_vars, true);

	// Calculate constants now we have unit data and store on <oGlblData> object

	calculate_unit_constants();

	var u_assess_score = 999.0;

	// Zone name for reporter variables (replace '-' with 'MINUS_'

	var zone_for_var = zone.replace('-', 'MINUS_');

	// Read in force 1 and 2
	
	if(oGlblData.dummy == undefined)
		oGlblData.dummy = "<Unspecified>";

	for (var i=0; i<2; i++)
	{
		if (i==0) beam = oGlblData.beam_femur1_id;
		else      beam = oGlblData.beam_femur2_id;

		// Check data components are available for this beam, if not create a blank image

		if(beam != undefined)
		{
			// Read in beam forces

			switch(oGlblData.dummy)
			{
				case "<Unspecified>":
				case "Arup-Cellbond Pedestrian Upper Legform Model v2.0":
					var c_force = read_data("BEAM BASIC", 0, beam, "NX");
					break;
				case "LSTC Upper Leg impactor v2.3.3":
					var c_force = read_data("CONTACT FORCES", 0, beam, "FM");
					break;
				case "LSTC Upper Leg impactor v2.3.2":
					var c_force = read_data("CONTACT FORCES", 0, beam, "FM");
					break;
				default:
					ErrorMessage("Unsupported dummy "+oGlblData.dummy);
					break;
			}

			if(i == 0)
			{
				var c_f1 = c_force;

				if(c_f1)
				{
					var c_f1      = Operate.Dix(c_f1, oGlblData.time_factor); // Convert to seconds
					var c_f1      = Operate.Div(c_f1, oGlblData.kn_factor);   // Convert to kN
					var c_f1_c180 = Operate.C180(c_f1, REG_DT);               // C180 filter
				}
			}
			else
			{
				var c_f2 = c_force;

				if(c_f2)
				{
					var c_f2      = Operate.Dix(c_f2, oGlblData.time_factor);  // Convert to seconds
					var c_f2      = Operate.Div(c_f2, oGlblData.kn_factor);    // Convert to kN
					var c_f2_c180 = Operate.C180(c_f2, REG_DT);                // C180 filter
				}
			}
		}
	}

	var f_sum       = 0.0;
	var f_sum_score = 0.0;

	var output_data = {};

	output_data.sum = new OutputData(zone + " Sum of Forces", images_dir + "/" + zone + "_Sum_Force");

	if(c_f1_c180 && c_f2_c180)
	{
		// Combine both the force curves into a total force curve

		var all_curves = new Array();

		all_curves[0] = c_f1_c180;
		all_curves[1] = c_f2_c180;

		var c_sum_f = Operate.Sum(all_curves);

		// Get the sum of forces value and score

		DialogueInput("/AU");
		var results = get_constant_limit_results(c_sum_f, legform_data.FEMUR_FORCE_GOOD, legform_data.FEMUR_FORCE_WEAK, legform_data.FEMUR_LO_SCORE, legform_data.FEMUR_HI_SCORE);

		f_sum_score = results[0];
		f_sum       = results[1];

		// Remove all curves and datums

		remove_all_curves_and_datums();

		// Convert from seconds back to model time

		c_sum_f = Operate.Mux(c_sum_f, oGlblData.time_factor);

		// Set label and style

		set_labels(c_sum_f, "Femur Sum of Forces", "Time (" + oGlblData.unit_time + ")", "Force (kN)");
		set_line_style(c_sum_f, Colour.BLACK);
        
        legform_draw_constant_datums(legform_data.FEMUR_FORCE_GOOD,legform_data.FEMUR_FORCE_WEAK, "FEMUR_FORCE");

		// Create image and curve files
		write_image(output_data.sum.title, output_data.sum.fname, c_sum_f, legform_data.FEMUR_FORCE_WEAK);

		output_data.sum.curveList.push(c_sum_f.id);
		write_curve("cur", output_data.sum.curveList, output_data.sum.fname);
		write_curve("csv", output_data.sum.curveList, output_data.sum.fname);
	}
	else
	{
		// No BEAM defines-variables should be set to blank by the first script in this template.
		// Create a blank image to let the user know.

		write_blank_images(output_data, "INVALID/NO BEAM ID DEFINED FOR");
	}

	remove_all_curves_and_datums();
	
	// Femur U1 Sum of forces kN VALUES

	write_variable(f, "FEMUR_FORCE_" + zone_for_var,        f_sum.toFixed(3),        "Femur Sum of Forces Value " + zone,  "String");
	write_variable(f, "FEMUR_FORCE_SCORE_" + zone_for_var,  f_sum_score.toFixed(3),  "Femur Sum of Forces Score " + zone,  "String");



	// Delete model

	DialogueInput("/MODEL DELETE ALL");

	write_variable(f, "UPPER_ASSESS_SCORE_" + zone_for_var,  f_sum_score.toFixed(3),  "Femur Assessment Score " + zone,  "String");

	// Add to overall score stored on oGlblData and increment number of points so we can average

	oGlblData.upper_score += f_sum_score;

	oGlblData.upper_num++;

	// Get the column number for this point and store it if it's greater or less than the current max/min
	// <zone> = U_<num>.  So get the string from the second character on.

	var col = parseInt(zone.substr(2));

	if(!isNaN(col))
	{
		if(col < 0) oGlblData.upper_min_col = Math.min(oGlblData.upper_min_col, col);
		else        oGlblData.upper_max_col = Math.max(oGlblData.upper_max_col, col);
	}
}



function do_lower_legform_calc(f, zone)
{
	// Calculates the scores for the <zone> and writes relevant values
    // into the Reporter variable file <f>.
    
    // This function returns the assessment score for the lower legform to be used for JNCAP overall calculations
    // (the return value is ignored in Euro NCAP and KNCAP)


	var res_file;

	for(var i=0; i<data.length; i++)
	{
		if(data[i].zone == zone) res_file = data[i].res_file;
	}

	DialogueInput("/MODEL READ", res_file);

	// Make sure any units set in T/HIS (e.g. via preferences) are unset

	unset_this_units(1);
	
	// check if the model time units and input time units are consistent
	// If the units are not defined/valid then we cannot go any further.  

	if(!check_unit_values())
	{
	write_unit_vars(f_vars, false);
    ErrorMessage("Cannot find the model unit values. They could be missing from the input csv file or post-*END data." );
	Exit();
	}

	// Write the unit data to the variables file

	write_unit_vars(f_vars, true);

	// Calculate constants now we have unit data and store on <oGlblData> object

	calculate_unit_constants();

	var total_elongation_score = 999.0;
	var total_mom_score = 999.0;

	// Zone name for reporter variables (replace '-' with 'MINUS_'

	var zone_for_var = zone.replace('-', 'MINUS_');

	// Cruciate ligaments elongation - MCL, ACL and PCL

    var spring_MCL, acro_MCL, good_MCL, weak_MCL;

    spring_MCL = oGlblData.spring_mcl_id;
    acro_MCL = "MCL";
    good_MCL = legform_data.MCL_ELONGATION_GOOD;
    weak_MCL = legform_data.MCL_ELONGATION_WEAK;
 
    var disp_MCL       = 0.0;
    var disp_MCL_score = 0.0;

    var output_data_MCL = {};

    output_data_MCL.disp = new OutputData(zone + " " + acro_MCL + " Displacement", images_dir + "/" + zone + "_" + acro_MCL + "_Displacement");

    //Calcs for MCL and plotting MCL graph
    if (spring_MCL != undefined)
    {
        var c_disp_MCL = read_data("SPRING TRANS", 0, spring_MCL, "ET");
        // Check that the curve read in - i.e. the spring id and component are valid
        // Create a blank image to let the user know.

        if(c_disp_MCL == undefined)
        {
            write_blank_images(output_data, "NO DATA FOR SPRING " + spring_MCL);
        }
        else
        {
            var c_disp_MCL      = Operate.Dix(c_disp_MCL, oGlblData.time_factor);     // Convert to seconds
            var c_disp_MCL_c180 = Operate.C180(c_disp_MCL, REG_DT);                   // C180 filter
            var c_disp_MCL      = Operate.Div(c_disp_MCL_c180, oGlblData.mm_factor);  // Convert to mm

            remove_all_curves_and_datums();

            // Convert back from seconds to model time

            c_disp_MCL = Operate.Mux(c_disp_MCL, oGlblData.time_factor);

            // Set line style and label graph

            set_labels(c_disp_MCL, acro_MCL + " Displacement", "Time (" + oGlblData.unit_time + ")", "Displacement (mm)");
            set_line_style(c_disp_MCL, Colour.BLACK);
            
            // Draw acceleration limit datums

            legform_draw_constant_datums(good_MCL, weak_MCL, acro_MCL);

            // Create image and curve files       
            write_image(output_data_MCL.disp.title, output_data_MCL.disp.fname, c_disp_MCL, weak_MCL);

            output_data_MCL.disp.curveList.push(c_disp_MCL.id);
            write_curve("cur", output_data_MCL.disp.curveList, output_data_MCL.disp.fname);
            write_curve("csv", output_data_MCL.disp.curveList, output_data_MCL.disp.fname);

            // Get value and score
            var results_MCL = legform_get_constant_limit_results(c_disp_MCL, good_MCL, weak_MCL, legform_data.LOW_LO_SCORE, legform_data.LOW_HI_SCORE_KNEE, "KNEE");

            disp_MCL_score = results_MCL[0];
            disp_MCL       = results_MCL[1];
        }
    }
    else
    {
        // No Spring defined - variables should be set to blank by the first script in this template.
        // Create a blank image to let the user know

        write_blank_images(output_data_MCL, "NO SPRING ID DEFINED FOR");
    }

    // Values and scores

    write_variable(f, "LOWER_TIBIA_" + acro_MCL + "_ELONGATION_" + zone_for_var, disp_MCL.toFixed(3), "Tibia " + acro_MCL + " Elongation " + zone, "String");
    write_variable(f, "LOWER_TIBIA_" + acro_MCL + "_ELONGATION_SCORE_" + zone_for_var, disp_MCL_score.toFixed(3), "Tibia " + acro_MCL + " Elongation Score " + zone, "String");


    // Running total for overall score
    total_elongation_score = Math.min(total_elongation_score, disp_MCL_score);



	// Bending moments - upper, mid-upper, mid-lower and lower

	for(var pass = 0; pass < 4; pass++)
	{  
		var xsec, acro, good, adqt, mgnl, weak;

		if(pass == 0) { xsec = oGlblData.xsection_tibia_upper_id;      acro = "Upper";     }
		else if(pass == 1) { xsec = oGlblData.xsection_tibia_mid_upper_id;  acro = "Mid_Upper"; }
		else if(pass == 2) { xsec = oGlblData.xsection_tibia_mid_lower_id;  acro = "Mid_Lower"; }
		else if(pass == 3) { xsec = oGlblData.xsection_tibia_lower_id;      acro = "Lower";     }

		var mom       = 0.0;
		var mom_score = 0.0;

		var output_data = {};

		output_data.mom = new OutputData(zone + " " + acro + " Tibia Moment", images_dir + "/" + zone + "_" + acro + "_Moment");

		if (xsec != undefined)
		{
			var c_mom = read_data("SECTION", 0, xsec, "MX");

			// Check that the curve read in - i.e. the xsec id and component are valid
			// Create a blank image to let the user know.

			if(!c_mom)
			{
				write_blank_images(output_data, "NO DATA FOR XSEC " + xsec);
			}
			else
			{
				c_mom = Operate.Dix(c_mom, oGlblData.time_factor);  // Convert to seconds
				c_mom = Operate.Div(c_mom, oGlblData.nm_factor);    // Convert to Nm
				c_mom = Operate.C180(c_mom, REG_DT);                // C180 filter

				c_mom = Operate.Mul(c_mom, -1);                     // Multiply by -1

				remove_all_curves_and_datums();

				// Convert back from seconds to model time

				c_mom = Operate.Mux(c_mom, oGlblData.time_factor);

				// Set line style and label graph

				set_labels(c_mom, acro + " Tibia Moment", "Time (" + oGlblData.unit_time + ")", "Moment (Nm)");
				set_line_style(c_mom, Colour.BLACK);

				// Draw both positive and negative datums

				legform_draw_constant_datums(legform_data.TIBIA_BENDING_MOMENT_GOOD, legform_data.TIBIA_BENDING_MOMENT_WEAK, "TIBIA",true);

				// Create image and curve files       
				write_image(output_data.mom.title, output_data.mom.fname, c_mom, legform_data.TIBIA_BENDING_MOMENT_WEAK, -legform_data.TIBIA_BENDING_MOMENT_WEAK);

				output_data.mom.curveList.push(c_mom.id);
				write_curve("cur", output_data.mom.curveList, output_data.mom.fname);
				write_curve("csv", output_data.mom.curveList, output_data.mom.fname);

				// Get value and score. Check against both positive and negative limits.

				DialogueInput("/AU");
				var results = legform_get_constant_limit_results(c_mom, legform_data.TIBIA_BENDING_MOMENT_GOOD, legform_data.TIBIA_BENDING_MOMENT_WEAK, legform_data.LOW_LO_SCORE, legform_data.LOW_HI_SCORE_TIBIA,"TIBIA",true);

				mom_score = results[0];
				mom       = results[1];
			}
		}
		else
		{
			// No Xsec defined-variables should be set to blank by the first script in this template.
			// Create a blank image to let the user know

			write_blank_images(output_data, "NO XSEC ID DEFINED FOR");
		}

		// Values and scores

		write_variable(f, "LOWER_TIBIA_" + acro.toUpperCase() + "_MOMENT_" + zone_for_var,        mom.toFixed(3),       "Tibia " + acro + " Moment " + zone,       "String");
		write_variable(f, "LOWER_TIBIA_" + acro.toUpperCase() + "_MOMENT_SCORE_" + zone_for_var,  mom_score.toFixed(3), "Tibia " + acro + " Moment Score " + zone, "String");

		// Running total for overall score

		total_mom_score = Math.min(total_mom_score, mom_score);
	}
	
    write_variable(f, "LOWER_TIBIA_SCORE_" + zone_for_var,  total_mom_score.toFixed(3), "Lower Legform Tibia Score " + zone,  "String");
    write_variable(f, "LOWER_KNEE_SCORE_" + zone_for_var,  total_elongation_score.toFixed(3), "Lower Legform Knee Score " + zone,  "String");
	
	var knee_and_tibia_score = Math.min(total_mom_score, total_elongation_score);
	write_variable(f, "KNEE_AND_TIBIA_SCORE_" + zone_for_var,  knee_and_tibia_score.toFixed(3), "Lower Legform Knee Score " + zone,  "String");
	
	
// Bending moments

	var f_upperbend        = 0.0;
	var f_centrebend       = 0.0;
	var f_lowerbend        = 0.0;
	var f_upperbend_score  = 0.0;
	var f_centrebend_score = 0.0;
	var f_lowerbend_score  = 0.0;

	for (i=0; i<3; i++)
	{    
		if      (i == 0) { xsection = oGlblData.xsection_femur_upper_id;  }
		else if (i == 1) { xsection = oGlblData.xsection_femur_centre_id; }
		else             { xsection = oGlblData.xsection_femur_lower_id;  }

		if(xsection != undefined)
		{
			// The Humanetics aPLI impactor model has a local coordinate system for the database cross section.
            // MX is the relevant bending moment component.
            // The relevant component may differ for other aPLI FE models added in future.
			var c_my = read_data("SECTION", 0, xsection, "MX");

			if(c_my)
			{
				var c_my = Operate.Dix(c_my, oGlblData.time_factor);  // Convert to seconds
				var c_my = Operate.Div(c_my, oGlblData.nm_factor);    // Convert to Nm
				var c_my = Operate.C180(c_my, REG_DT);                // C180 filter
				var c_my = Operate.Abs(c_my);                         // Absolute value

				var results = get_constant_limit_results(c_my, legform_data.FEMUR_BENDING_MOMENT_GOOD, legform_data.FEMUR_BENDING_MOMENT_WEAK, legform_data.FEMUR_LO_SCORE, legform_data.FEMUR_HI_SCORE);

				if (i == 0)
				{
					//Get Peak value and score

					var c_xu = c_my;

					f_upperbend_score = results[0];
					f_upperbend       = results[1];
				}
				else if (i == 1)
				{
					//Get Peak value and score

					var c_xc = c_my;

					f_centrebend_score = results[0];
					f_centrebend       = results[1];
				}
				else 
				{
					//Get Peak value and score

					var c_xl = c_my;

					f_lowerbend_score = results[0];
					f_lowerbend       = results[1];
				}
			}
		}
	}

	var output_data = {};

	output_data.mom = new OutputData(zone + " Femur Bending Moments", images_dir + "/" + zone + "_Femur_Bending_Moment");

	if(c_xu && c_xc && c_xl)
	{

		// Convert lines back to original time

		c_xu_s = Operate.Mux(c_xu, oGlblData.time_factor);
		c_xc_s = Operate.Mux(c_xc, oGlblData.time_factor);
		c_xl_s = Operate.Mux(c_xl, oGlblData.time_factor);

		// Set line style

		set_labels(c_xu_s, "Femur Upper Bending Moment", "Time (" + oGlblData.unit_time + ")", "Bending Moment (Nm)");
		set_line_style(c_xu_s, Colour.BLUE);

		set_labels(c_xc_s, "Femur Centre Bending Moment", "Time (" + oGlblData.unit_time + ")", "Bending Moment (Nm)");
		set_line_style(c_xc_s, Colour.MAGENTA);

		set_labels(c_xl_s, "Femur Lower Bending Moment", "Time (" + oGlblData.unit_time + ")", "Bending Moment (Nm)");
		set_line_style(c_xl_s, Colour.BLACK);

		// Remove all curves and datums

		remove_all_curves_and_datums();

		// Draw moment limit datums

		legform_draw_constant_datums(legform_data.FEMUR_BENDING_MOMENT_GOOD, legform_data.FEMUR_BENDING_MOMENT_WEAK, "FEMUR_BENDING");

		// Create image and curve files of all three bending moments

		c_xu_s.AddToGraph();
		c_xc_s.AddToGraph();
		c_xl_s.AddToGraph();

		write_image(output_data.mom.title, output_data.mom.fname, [ c_xu_s, c_xc_s, c_xl_s ], legform_data.FEMUR_BENDING_MOMENT_WEAK)

			output_data.mom.curveList.push(c_xu_s.id, c_xc_s.id, c_xl_s.id);
		write_curve("cur", output_data.mom.curveList, output_data.mom.fname);
		write_curve("csv", output_data.mom.curveList, output_data.mom.fname);
	}
	else
	{
		// No Xsection defined-variables should be set to blank by the first script in this template.
		// Create a blank image to let the user know.

		write_blank_images(output_data, "MISSING/INVALID XSECTION ID DEFINED FOR");
	}


	//Write out values, calculating the assessemnt values and the colour to go with it

	//Femur Upper Bending moments VALUES and SCORES

	write_variable(f, "FEMUR_UPPER_BENDING_" + zone_for_var,        f_upperbend.toFixed(3),        "Femur Upper Bend Value " + zone,  "String");
	write_variable(f, "FEMUR_UPPER_BENDING_SCORE_" + zone_for_var,  f_upperbend_score.toFixed(3),  "Femur Upper Bend Score " + zone,  "String");

	write_variable(f, "FEMUR_CENTRE_BENDING_" + zone_for_var,       f_centrebend.toFixed(3),       "Femur Centre Bend Value " + zone, "String");
	write_variable(f, "FEMUR_CENTRE_BENDING_SCORE_" + zone_for_var, f_centrebend_score.toFixed(3), "Femur Centre Bend Score " + zone, "String");

	write_variable(f, "FEMUR_LOWER_BENDING_" + zone_for_var,        f_lowerbend.toFixed(3),        "Femur Lower Bend Value " + zone,  "String");
	write_variable(f, "FEMUR_LOWER_BENDING_SCORE_" + zone_for_var,  f_lowerbend_score.toFixed(3),  "Femur Lower Bend Score " + zone,  "String");

	var femur_score = Math.min(f_upperbend_score, f_centrebend_score, f_lowerbend_score);
	write_variable(f, "FEMUR_BENDING_SCORE_" + zone_for_var,  femur_score.toFixed(3),  "Femur Bend Score " + zone,  "String");


	// Delete model

	DialogueInput("/MODEL DELETE ALL");

	oGlblData.femur_score += femur_score
	oGlblData.knee_and_tibia_score += knee_and_tibia_score
	if(template_reg=='CNCAP')
	{
		var score_sum_per_zone = femur_score+total_mom_score+total_elongation_score
		write_variable(f, "SCORE_SUM_PER_ZONE_" + zone_for_var,  score_sum_per_zone.toFixed(3),  "Score Sum for zone " + zone,  "String");
		oGlblData.score_sum_per_zone += score_sum_per_zone
	}

	oGlblData.lower_num++;

	// Get the column number for this point and store it if it's greater or less than the current max/min
	// <zone> = L_<num>.  So get the string from the second character on.

	var col = parseInt(zone.substr(2));

	if(!isNaN(col))
	{
		if(col < 0) oGlblData.lower_min_col = Math.min(oGlblData.lower_min_col, col);
		else        oGlblData.lower_max_col = Math.max(oGlblData.lower_max_col, col);
    }
    
    //return l_assess_score; // Return score for JNCAP calculation
}

// zone_name should follow the format U/L_<integer>
// zone_index = 0 for lower and 1 for upper
function check_zone_name(zone_name, zone_index)
{
	var list = zone_name.split("_");

	if (zone_index==0 && zone_name[0].toUpperCase() != 'L')
	{
		if (batch_mode == "false")
		{
			var ans = Window.Warning("Warning", "Lower legform gridpoint should start with L instead " + zone_name 
			+ " starts with " + zone_name[0].toUpperCase() + ". Do you want to correct it else this grid point will be skipped.", 
			Window.YES|Window.NO);
			
			if (ans == Window.YES)
				zone_name='L_'+list[1];
			else
				return false;
		}
		else
		{
			WarningMessage("Warning: Lower legform gridpoint should start with L instead " + zone_name 
			+ " starts with " + zone_name[0].toUpperCase() + ". This grid point will be skipped.");
			return false;
		}
	}
	else if (zone_index==1 && zone_name[0].toUpperCase() != 'U')
	{
		if (batch_mode == "false")
		{
			var ans = Window.Warning("Warning", "Upper legform gridpoint should start with U instead " + zone_name 
			+ " starts with " + zone_name[0].toUpperCase() + ". Do you want to correct it else this grid point will be skipped.", 
			Window.YES|Window.NO);
			
			if (ans == Window.YES)
				zone_name='U_'+list[1];
			else
				return false;
		}
		else
		{
			WarningMessage("Warning: Upper legform gridpoint should start with U instead " + zone_name 
			+ " starts with " + zone_name[0].toUpperCase() + ". This grid point will be skipped.");
			return false;
		}
	}
			
	if (isNaN(parseInt(list[1])))
	{
		if (batch_mode == "false")
		{
			if (zone_index == 0)
				Window.Warning("Warning", zone_name + " does not follow the format L_<integer>. This grid point will be skipped");
			else if (zone_index == 1)
				Window.Warning("Warning", zone_name + " does not follow the format U_<integer>. This grid point will be skipped");
		}
		else 
		{
			if (zone_index == 0)
				WarningMessage("Warning: "+ zone_name + " does not follow the format L_<integer>. This grid point will be skipped");
			else if (zone_index == 1)
				WarningMessage("Warning: "+ zone_name + " does not follow the format U_<integer>. This grid point will be skipped");
		}
		return false;
	}
	
	zone = zone_name;
	return true;
}


function legform_draw_constant_datums(good, weak, part_being_tested, symmetric)
{
    // this function is instead of using draw_constant_datums directly, as legform testing for EuroNCAP needs red from 0.25 points and lower
    // whereas other tests are red for 0 points. This function takes the min and max limits, and then calls draw_constant_datums
    // with the correct values for legform. JNCAP however has arbitrary limits for the colour mapping. 

    // check that good is greater than weak, this should be the case for all legform calcs except those with a simple pass/fail.
    if (good < weak)
    {
        //find the difference between the limits
        var dif_min_max = weak - good;
            
        //calculate the intermediate boundaries. As red starts at 0.25 points, this is 3/4 of the difference
        switch (template_reg)
        {
            //EuroNCAP case
            case "KNCAP":
			case "CNCAP":
            case "EuroNCAP":
                var temp_adequate = good + dif_min_max/4;
                var temp_marginal = good + 2*dif_min_max/4;
                var temp_weak = good + 3*dif_min_max/4;
                break;

            //JNCAP case
            case "JNCAP":
                if (part_being_tested == "ACL" | part_being_tested == "PCL")
                {
                    var temp_adequate = good;
                    var temp_marginal = good;
                    var temp_weak = weak;
                }

                else if (part_being_tested == "MCL")
                {
                    var temp_adequate = legform_data.MCL_ELONGATION_ADEQUATE;
                    var temp_marginal = legform_data.MCL_ELONGATION_MARGINAL;
                    var temp_weak = weak;
                }

                else if (part_being_tested == "TIBIA")
                {
                    var temp_adequate = legform_data.TIBIA_BENDING_MOMENT_ADEQUATE;
                    var temp_marginal = legform_data.TIBIA_BENDING_MOMENT_MARGINAL;
                    var temp_weak = weak;
                }


                else
                {
                    ErrorMessage("JNCAP unexpected limits, potential limitation with the legform_draw_constant_datums script")
                }
                break;
        }
        
    	// Now run the draw_constant_datums with adjusted values
        if (symmetric!=undefined && symmetric==true) {draw_constant_datums(good, temp_adequate, temp_marginal, temp_weak, symmetric);}
        else                   {draw_constant_datums(good, temp_adequate, temp_marginal, temp_weak);}
    }
    
    // This occurs for some where it's just a pass/fail, and all values can just be passed through with no calculation required
    else if (weak == good)
    {
        draw_constant_datums(good, good, good, good);
    }
    else 
    {
        ErrorMessage("Function legform_draw_constant_datums does not accept good value ("+good+") greater than weak value ("+weak+").");
        Exit();
    }
}

//This function allows different maths to be done depending on EuroNCAP or JNCAP etc
function legform_get_constant_limit_results(c, lo_limit, hi_limit, lo_score, hi_score, body_part, symmetric)
{

    switch (template_reg)
    {
        // for these cases get_constant_limits_results can solve them normally with linear interpolation for the scores
        case "KNCAP":  
		case "CNCAP":
        case "EuroNCAP":
            //This function below does linear interpolation
			if(symmetric!=undefined && symmetric == true){var return_array = get_constant_limit_results(c, lo_limit, hi_limit, lo_score, hi_score, symmetric);}
            else                 {var return_array = get_constant_limit_results(c, lo_limit, hi_limit, lo_score, hi_score);}
            return return_array;
    

        //for JNCAP there are different equations for each, hence Knee and tibia done separately
        //JNCAP also assigned 4 points to each place instead of 1, to make the code easier, only 1 point is assigned here, and at a different step a factor
        // of 4 is introduced.
        case "JNCAP":
            //This function follows JNCAP score calculation rules which are not linear interpolation
            if(symmetric!=undefined && symmetric == true){var return_array = get_constant_limit_results_JNCAP(c, lo_limit, hi_limit, lo_score, hi_score, body_part, symmetric);}
            else                 {var return_array = get_constant_limit_results_JNCAP(c, lo_limit, hi_limit, lo_score, hi_score, body_part);}
            return return_array;    
        
    }
}

function get_constant_limit_results_JNCAP(c, lo_limit, hi_limit, lo_score, hi_score, body_part, symmetric)
// This is a variant of the common function <get_constant_limit_results> because for JNCAP,
// The interpolation is done differently (see maths below).
{
    var return_array = new Array(2);
    var val = 0.0;
    var rating = 0.0;
    if(symmetric!=undefined && symmetric == true) {val = Math.max(c.ymax,Math.abs(c.ymin));}
    else                {val = c.ymax;}
       
    if (val < lo_limit)
    {
        rating = hi_score*4;
    }
    else if (val > hi_limit)
    {
        rating = lo_score;
    }
    else
    {
        if (body_part == "KNEE")
        {
            rating = hi_score*(4 - (4/(hi_limit-lo_limit)) * (val-lo_limit)); //this is doing 4 - 4/5 (MCL stretch amount - 14.8)
        }
        else if(body_part == "TIBIA")
        {
            rating = hi_score *(4 - (val-lo_limit)/((hi_limit-lo_limit)/4)); //this is doing 4 - (Tibia Flexion moment - 202)/26

        }
        else
        {
            ErrorMessage("Body part invalid - needs to be Tibia or Knee, coding error not a user error");
            Exit();
        }
    }

    return_array[0] = rating;
    return_array[1] = val;

    return return_array;  
}
