// Common functions for US NCAP

function do_head_rating_calc(f, num_dummies)
{
    // Calculates the head values and writes them into the Reporter csv variables file <f>.
    
    // dummy no. 1 is driver
    // dummy no. 2 is passenger
        
    for (var pass=0; pass<num_dummies; pass++)
    {
        if (pass == 0)
        {
            var node_x = oGlblData.driver_head_node;
            var node_y = oGlblData.driver_head_node_y;
            var node_z = oGlblData.driver_head_node_z;

            dummy = "Driver ";
            dummy_ = "Driver_";
        }
        
        if (pass == 1) 
        {
            var node_x = oGlblData.passenger_head_node;
            var node_y = oGlblData.passenger_head_node_y;
            var node_z = oGlblData.passenger_head_node_z;
            
            dummy = "Passenger ";
            dummy_ = "Passenger_";
        }
        
        var output_data = {};        
        
        output_data.acc = new OutputData(dummy + "Head Acceleration", images_dir + "/" + dummy_ + "Head_Acceleration");

        if( node_x != undefined )
        { 
            // Read in acceleration X, Y and Z

            // If nodes for three accelerometer positions defined, use all three, otherwise, read all components from same node
            if(node_y != undefined && node_z != undefined && node_y != 0 && node_z != 0)
            {
                var c_acc = read_xyz_accelerations(1, node_x, node_y, node_z);
            }
            else
            {
                var c_acc = read_xyz_accelerations(1, node_x);
            }
            
            // Read in acceleration X, Y and Z
            // var one_node = true;
            
            // Check that the curves read in - i.e. the node id and component are valid
            // Create a blank image to let the user know.

            if(!c_acc.ax || !c_acc.ay || !c_acc.az)
            {

                var blank_image_msg;
                // If only one node supplied, just report that node ID
                // if(one_node) blank_image_msg = "NO ACCEL DATA (NODE " + node + ") ";
                // else report each of the node IDs that were invalid
                // else
                // {
                var acc_comp_lst = [];
                var acc_id_lst = [];
                
                if(!c_acc.ax)
                {
                    acc_comp_lst.push("X");
                    acc_id_lst.push(" "+node_x);
                }
                if(!c_acc.ay)
                {
                    acc_comp_lst.push("Y");
                    acc_id_lst.push(" "+node_y);
                }
                if(!c_acc.az)
                {
                    acc_comp_lst.push("Z");
                    acc_id_lst.push(" "+node_z);
                }
                blank_image_msg = "NO " + dummy.toUpperCase() + acc_comp_lst + " ACCEL DATA FOR NODE(s)" + acc_id_lst;
                // }
                create_blank_image(blank_image_msg, output_data.acc.fname);
            }
            else
            {
                // Convert to g from model units

                var c_g = convert_xyz_acc_to_g(c_acc);

                // C1000 filter (convert to seconds first)

                var c_gx_c1000 = convert_to_s_and_filter(c_g.gx, Operate.C1000, REG_DT);
                var c_gy_c1000 = convert_to_s_and_filter(c_g.gy, Operate.C1000, REG_DT);
                var c_gz_c1000 = convert_to_s_and_filter(c_g.gz, Operate.C1000, REG_DT);

                // Vector combine

                var c_vec_g = Operate.Vec(c_gx_c1000, c_gy_c1000, c_gz_c1000);

                // Remove all curves and datums

                remove_all_curves_and_datums();

                // Convert from seconds back to model time

                c_vec_g = Operate.Mux(c_vec_g, oGlblData.time_factor);

                // Get peak acceleration (in g) and time (in seconds)

                DialogueInput("/AU");
                var peak_accn      = c_vec_g.ymax;
                // var peak_accn_time = c_vec_g.x_at_ymax / oGlblData.time_factor;

                // For HIC36 value 36 ms set to HIC_WINDOW.
                // Similarly, for HIC15 value 15 ms set to HIC_WINDOW.

                var hic            = Operate.Hic(c_vec_g, HIC_WINDOW, 1.0);

                // Set label and style

                set_labels(c_vec_g, "Head Acceleration Magnitude", "Time (" + oGlblData.unit_time + ")", "Acceleration (g)");
                set_line_style(c_vec_g, Colour.BLACK);

                // Create image/curve

                write_image(output_data.acc.title, output_data.acc.fname);

                output_data.acc.curveList.push(c_vec_g.id);
                write_curve("cur", output_data.acc.curveList, output_data.acc.fname);
                write_curve("csv", output_data.acc.curveList, output_data.acc.fname);
                
                // Write variables to csv file for Reporter

                // Peak acceleration and time
                write_variable(f, dummy_.toUpperCase() + "HEAD_PEAK_ACCN",      peak_accn.toFixed(2),  dummy + "head peak acceleration value",      "General");
                if (HIC_WINDOW == 0.036)
                    write_variable(f, dummy_.toUpperCase() + "HEAD_HIC_36",            hic.toFixed(2),            dummy + "head HIC36 value",            "General");
                else if (HIC_WINDOW == 0.015)
                    write_variable(f, dummy_.toUpperCase() + "HEAD_HIC_15",            hic.toFixed(2),            dummy + "head HIC15 value",            "General");
            }

        }
        else
        {

            var blank_image_msg;

            var node_id_lst = [];

            blank_image_msg = "UNDEFINED " + dummy.toUpperCase() + "ACCELERATION NODE-X";
            create_blank_image(blank_image_msg , output_data.acc.fname);
        }
    }
}


function do_neck_rating_calc(f, num_dummies, nij_flag)
{
    // Calculates the neck values and writes them into the Reporter csv variables file <f>.
    
    // dummy no. 1 is driver
    // dummy no. 2 is passenger
    
    Message("Doing neck rating calculation...");
    
    var dummy = [];
    dummy.push(new DummyData("Driver", oGlblData.driver_neck_beam, "beam", oGlblData.driver_dummy, "version"));
    dummy.push(new DummyData("Passenger", oGlblData.passenger_neck_beam, "beam", oGlblData.passenger_dummy, "version"));

    for(var i = 0; i<dummy.length; i++)
    {
        var output_data = {};

        output_data.shr_exc = new OutputData(dummy[i].chart_label + "Neck Shear Exceedence",  images_dir + "/" + dummy[i].variable_label+ "_Neck_Shear_Exceedence");
        output_data.ten_exc = new OutputData(dummy[i].chart_label + "Neck Tension Force",     images_dir + "/" + dummy[i].variable_label + "_Neck_Tension_Force");
        output_data.com_exc = new OutputData(dummy[i].chart_label + "Neck Compression Force", images_dir + "/" + dummy[i].variable_label + "_Neck_Compression_Force");
        output_data.mom     = new OutputData(dummy[i].chart_label + "Neck Bending Moment",    images_dir + "/" + dummy[i].variable_label + "_Neck_Moment");
        output_data.axl     = new OutputData(dummy[i].chart_label + "Neck Axial Force",       images_dir + "/" + dummy[i].variable_label + "_Neck_Axial_Force");
        output_data.shr     = new OutputData(dummy[i].chart_label + "Neck Shear Force",       images_dir + "/" + dummy[i].variable_label + "_Neck_Shear_Force");
        
        if (nij_flag)
            output_data.nij = new OutputData(dummy[i].chart_label + "Neck Nij",               images_dir + "/" + dummy[i].variable_label + "_Neck_Nij");

// If dummy version was undefined, assume BEAM data
        if (dummy[i].version == undefined)
            dummy[i].version = "<Unspecified>";

        if( dummy[i].beam_id != undefined )
        {
            // Read in beam shear, tension and  MY
            // For Humanetics dummies this is measured from a beam;
            // for LSTC dummies this is measured from a cross section
            switch (dummy[i].version)
            {
                case "<Unspecified>":
                case "V7_HUMANETICS_HIII_50TH_MALE":
                case "V8_HUMANETICS_HIII_50TH_MALE":
                case "Humanetics HIII 50M v1.5":
                case "Humanetics HIII 50M v1.5.1":
                case "V6_HUMANETICS_HIII_5TH_FEMALE":
                case "V7_HUMANETICS_HIII_5TH_FEMALE":
                    var c_shr = read_data("BEAM BASIC", 0, dummy[i].beam_id, "NX");  // NX is shear
                    var c_axl = read_data("BEAM BASIC", 0, dummy[i].beam_id, "NZ");
                    var c_mom = read_data("BEAM BASIC", 0, dummy[i].beam_id, "MY");
                    var entity_str = "BEAM"; // Used for error message below
                    break;

                case "LSTC HIII 5F v2.0":
                case "LSTC HIII 5F Fast v2.0":
                case "LSTC HIII 50M Detailed v190217_beta":
                case "LSTC HIII 50M Fast V2.0":
                case "LSTC HIII 50M v130528_beta":                
                case "Humanetics HIII 5F v2.02":
                    var c_shr = read_data("SECTION", 0, dummy[i].beam_id, "FX");  // FX is shear
                    var c_axl = read_data("SECTION", 0, dummy[i].beam_id, "FZ");
                    var c_mom = read_data("SECTION", 0, dummy[i].beam_id, "MY");
                    var entity_str = "SECTION"; // Used for error message below
                    break;

                default:
                    ErrorMessage("Unsupported dummy \""+dummy[i].version+"\" in do_neck_rating_calc.");
                    write_blank_images(output_data, "UNSUPPORTED DUMMY \""+dummy[i].version+"\".");
                    return;
                    break;
            }

            // Check that the curves read in - i.e. the beam id and component are valid
            // Create blank images to let the user know.

            if( !c_shr || !c_axl || !c_mom )
            {
                write_blank_images(output_data, "NO DATA FOR " + entity_str + " " + dummy[i].beam_id);
            }
            else
            {
                // Convert forces to kN and
                //         moment to Nm from model units
                
                var c_shr_kn = Operate.Div(c_shr, oGlblData.kn_factor);                
                var c_axl_kn = Operate.Div(c_axl, oGlblData.kn_factor);
                var c_mom_nm = Operate.Div(c_mom, oGlblData.nm_factor);
                
                // C1000/C600 filter (convert to seconds first)                
                var c_shr_c1000 = convert_to_s_and_filter(c_shr_kn, Operate.C1000, REG_DT);                
                var c_axl_c1000 = convert_to_s_and_filter(c_axl_kn, Operate.C1000, REG_DT);
                var c_mom_c600  = convert_to_s_and_filter(c_mom_nm, Operate.C600,  REG_DT);

                // Exceedence curves for shear, tension and compression. For shear and compression take absolute values.
                var c_shr_abs = Operate.Abs(c_shr_c1000);
                var c_shr_exc = Operate.Exc(c_shr_abs, "positive");
                
                // tension, compression
                var c_ten_exc = Operate.Exc(c_axl_c1000, "positive");
                var c_com_exc = Operate.Exc(c_axl_c1000, "negative");                
                //shear
                var c_shr_exc_abs = Operate.Abs(c_shr_exc);
                // Compression
                var c_com_exc_abs = Operate.Abs(c_com_exc);

                // Transpose the bending moment to the occupital condoyle moment
                // Neck assessment constants

                var NECK_SHR_ADJ_FAC = 17.78;   // Factor to multiply shear by to adjust bending moment

                var c_shr_adj = Operate.Mul(c_shr_c1000, NECK_SHR_ADJ_FAC);
                var c_mom_adj = Operate.Sub(c_mom_c600, c_shr_adj);
                
                // Regularise curves first so they have the same number of points

                var max_points = Math.max(c_axl_c1000.npoints,
                        c_mom_adj.npoints);
                        
                max_points = Math.max(c_shr_c1000.npoints,
                    max_points);
                
                var p = c_shr_c1000.GetPoint(c_shr_c1000.npoints);       
                var t = p[0];

                var dt = t / max_points;

                c_shr_c1000 = Operate.Reg(c_shr_c1000, dt);

                var p = c_axl_c1000.GetPoint(c_axl_c1000.npoints);       
                var t = p[0];

                var dt = t / max_points;

                c_axl_c1000 = Operate.Reg(c_axl_c1000, dt);

                var p = c_mom_adj.GetPoint(c_mom_adj.npoints);       
                var t = p[0];

                var dt = t / max_points;

                c_mom_adj = Operate.Reg(c_mom_adj, dt);

                // Calculate Nij

                if (nij_flag)
                {
                switch (dummy[i].version)
                {
                    case "<Unspecified>":
                    case "V7_HUMANETICS_HIII_50TH_MALE":
                    case "V8_HUMANETICS_HIII_50TH_MALE":
                    case "Humanetics HIII 50M v1.5":
                    case "Humanetics HIII 50M v1.5.1":
                    case "LSTC HIII 50M Detailed v190217_beta":
                    case "LSTC HIII 50M Fast V2.0":
                    case "LSTC HIII 50M v130528_beta":

                        // neck injury critical values for HIII 50% dummy
                        var NIJ_TEN_FCRIT = 6.806;  // Critical neck tension     force  for Nij (kN)
                        var NIJ_COM_FCRIT = 6.160;  // Critical neck compression force  for Nij (kN)
                        var NIJ_FLX_MCRIT = 310;    // Critical neck flexion     moment for Nij (Nm)
                        var NIJ_EXT_MCRIT = 135;    // Critical neck extension   moment for Nij (Nm)
                        break;

                    case "LSTC HIII 5F v2.0":
                    case "LSTC HIII 5F Fast v2.0":
                    case "V6_HUMANETICS_HIII_5TH_FEMALE":
                    case "V7_HUMANETICS_HIII_5TH_FEMALE":
                    case "Humanetics HIII 5F v2.02":
                        
                        // neck injury critical values for HIII 5% dummy
                        var NIJ_TEN_FCRIT = 4.287;  // Critical neck tension     force  for Nij (kN)
                        var NIJ_COM_FCRIT = 3.880;  // Critical neck compression force  for Nij (kN)
                        var NIJ_FLX_MCRIT = 155;    // Critical neck flexion     moment for Nij (Nm)
                        var NIJ_EXT_MCRIT = 67;    // Critical neck extension   moment for Nij (Nm)
                        break;

                    default:
                        ErrorMessage("Unsupported dummy \""+dummy[i].version+"\" in do_neck_rating_calc.");
                        write_blank_images(output_data, "UNSUPPORTED DUMMY \""+dummy[i].version+"\".");
                        return;
                        break;
                }
                    
                    var nij_curves = Operate.Nij(c_shr_c1000,
                            c_axl_c1000,
                            c_mom_adj,
                            NIJ_TEN_FCRIT,
                            NIJ_COM_FCRIT,
                            NIJ_FLX_MCRIT,
                            NIJ_EXT_MCRIT,
                            0.0);

                    for(var k=0; k<4; k++)
                        nij_curves[k] = Operate.Mux(nij_curves[k], oGlblData.time_factor);
                
                }
                
                // Remove all curves and datums

                remove_all_curves_and_datums();

                // Convert from seconds back to model time
                c_shr_exc_abs = Operate.Mux(c_shr_exc_abs, oGlblData.time_factor);
                c_shr_c1000   = Operate.Mux(c_shr_c1000,   oGlblData.time_factor);
                c_ten_exc     = Operate.Mux(c_ten_exc,     oGlblData.time_factor);
                c_com_exc_abs = Operate.Mux(c_com_exc_abs, oGlblData.time_factor);
                c_mom_adj     = Operate.Mux(c_mom_adj,     oGlblData.time_factor);
                c_axl_c1000   = Operate.Mux(c_axl_c1000,   oGlblData.time_factor);

                // Set labels and style
                set_labels(c_shr_exc_abs, "Neck Shear Exceedence",       "Cumulative Time (" + oGlblData.unit_time + ")", "Force (kN)");
                set_labels(c_shr_c1000,   "neck Shear Force",            "Time (" + oGlblData.unit_time + ")",            "Force (kN)");
                set_labels(c_ten_exc,     "Neck Tension Force",     "Cumulative Time (" + oGlblData.unit_time + ")", "Force (kN)");
                set_labels(c_com_exc_abs, "Neck Compression Force", "Cumulative Time (" + oGlblData.unit_time + ")", "Force (kN)");
                set_labels(c_mom_adj,     "Neck Bending Moment",         "Time (" + oGlblData.unit_time + ")",            "Bending Moment (Nm)");
                set_labels(c_axl_c1000,   "Neck Axial Force",            "Time (" + oGlblData.unit_time + ")",            "Force (kN)");
                
                if (nij_flag)
                {
                    set_labels(nij_curves[0], "Nij - Tension-Extension",     "Time (" + oGlblData.unit_time + ")",            "Nij");
                    set_labels(nij_curves[1], "Nij - Tension-Flexion",       "Time (" + oGlblData.unit_time + ")",            "Nij");
                    set_labels(nij_curves[2], "Nij - Compression-Extension", "Time (" + oGlblData.unit_time + ")",            "Nij");
                    set_labels(nij_curves[3], "Nij - Compression-Flexion",   "Time (" + oGlblData.unit_time + ")",            "Nij");
                }
                
                set_line_style(c_shr_exc_abs, Colour.BLACK);
                set_line_style(c_shr_c1000,   Colour.BLACK);
                set_line_style(c_ten_exc,     Colour.BLACK);
                set_line_style(c_com_exc_abs, Colour.BLACK);
                set_line_style(c_mom_adj,     Colour.BLACK);
                set_line_style(c_axl_c1000,   Colour.BLACK);
                
                if (nij_flag)
                {
                    set_line_style(nij_curves[0], Colour.MAGENTA);
                    set_line_style(nij_curves[1], Colour.MAGENTA);
                    set_line_style(nij_curves[2], Colour.BLACK);
                    set_line_style(nij_curves[3], Colour.BLACK);
                    
                    nij_curves[1].style  = LineStyle.DASH;
                    nij_curves[3].style  = LineStyle.DASH;

                    // Remove all curves and datums and datums

                    remove_all_curves_and_datums();

                    // Create image/curve of neck Nij curves

                    for(var j=0; j<4; j++)
                    {
                        nij_curves[j].AddToGraph();
                        output_data.nij.curveList.push(nij_curves[j].id);
                    }

                    write_image(output_data.nij.title, output_data.nij.fname);

                    write_curve("cur", output_data.nij.curveList, output_data.nij.fname);
                    write_curve("csv", output_data.nij.curveList, output_data.nij.fname);            
                }

                // Remove all curves and datums

                remove_all_curves_and_datums();

                // Draw neck tension datums

                c_ten_exc.AddToGraph();
                write_image(output_data.ten_exc.title, output_data.ten_exc.fname);
                
                output_data.ten_exc.curveList.push(c_ten_exc.id);
                write_curve("cur", output_data.ten_exc.curveList, output_data.ten_exc.fname);
                write_curve("csv", output_data.ten_exc.curveList, output_data.ten_exc.fname);

                // Remove all curves and datums

                remove_all_curves_and_datums();

                // Draw neck compression datums

                var p = c_com_exc_abs.GetPoint(c_com_exc_abs.npoints);

                // Create image/curve of neck tension curves

                c_com_exc_abs.AddToGraph();
                write_image(output_data.com_exc.title, output_data.com_exc.fname);
                
                output_data.com_exc.curveList.push(c_com_exc_abs.id);
                write_curve("cur", output_data.com_exc.curveList, output_data.com_exc.fname);
                write_curve("csv", output_data.com_exc.curveList, output_data.com_exc.fname);

                // Remove all curves and datums

                remove_all_curves_and_datums();

                // Calc values

                var ten_results     = c_axl_c1000.ymax;
                var com_results     = -c_axl_c1000.ymin;
                
                if (nij_flag)
                    var nij_results = get_neck_nij_results(nij_curves);
                
                // Write variables

                // TENSION

                write_variable(f, dummy[i].variable_label.toUpperCase() + "_NECK_TENSION_VALUE",  ten_results.toFixed(3), dummy + "Neck tension value",  "General");

                // COMPRESSION

                write_variable(f, dummy[i].variable_label.toUpperCase() + "_NECK_COMPRESSION_VALUE",  com_results.toFixed(3), dummy + "Neck compression value",  "General");

                // NIJ
                if (nij_flag)
                {
                    write_variable(f, dummy[i].variable_label.toUpperCase()+"_NECK_INJURY_VALUE" ,  nij_results.toFixed(3), dummy + "Neck injury value",  "General");
                }
            }
        }
        else
        {
            // No beam defined - variables should be set to blank by first script in template.
            // Create blank images to let the user know.
            
            output_data.shr_exc.title = "";
            output_data.ten_exc.title = "";
            output_data.com_exc.title = "";
            output_data.mom.title     = "";
            output_data.axl.title     = "";
            output_data.shr.title     = "";
            if (nij_flag)
            output_data.nij.title     = "";

            write_blank_images(output_data, "UNDEFINED " + dummy[i].variable_label.toUpperCase() + "NECK LOADCELL");
        }
    }
}

function do_chest_3_sensors_rating_calc(f, num_dummies)
{
    for(var pass = 0; pass < num_dummies; pass++)
    {
        if (pass == 0)
        {
            spring1 = oGlblData.driver_chest_spring_upp;
            spring2 = oGlblData.driver_chest_spring_mid;
            spring3 = oGlblData.driver_chest_spring_low;
            
            dummy = "Driver ";
            dummy_ = "Driver_";
        }
        
        if (pass == 1) 
        {
            spring1 = oGlblData.passenger_chest_spring_upp;
            spring2 = oGlblData.passenger_chest_spring_mid;
            spring3 = oGlblData.passenger_chest_spring_low;
            
            dummy = "Passenger ";
            dummy_ = "Passenger_";
        }
        
        var output_data = {};
        
        output_data.rib = new OutputData(dummy + "Chest Rib Deflections",   images_dir + "/" + dummy_ + "Rib_Deflections");
        
        if( spring1 != undefined && spring2 != undefined && spring3 != undefined )
        {
            c_disp1 = read_data("SPRING TR", 0, spring1, "ET");
            c_disp2 = read_data("SPRING TR", 0, spring2, "ET");
            c_disp3 = read_data("SPRING TR", 0, spring3, "ET");
            
            if( !c_disp1 || !c_disp2 || !c_disp3 )
            {
                var blank_image_msg;

                var rib_disp_id_lst = [];
                
                if( spring1_exists == null )
                {
                    rib_disp_id_lst.push(" "+spring1);
                }
                if( spring2_exists == null )
                {
                    rib_disp_id_lst.push(" "+spring2);
                }
                if( spring3_exists == null )
                {
                    rib_disp_id_lst.push(" "+spring3);
                }
                blank_image_msg = "NO " + dummy.toUpperCase() + "CHEST RIB DISP DATA FOR SPRING(s)" + rib_disp_id_lst;
                create_blank_image(blank_image_msg, output_data.rib.fname);
            }
            else
            {
                var c_disp1 = Operate.Div(c_disp1, oGlblData.mm_factor);    // Convert to mm
                    c_disp1 = Operate.Mul(c_disp1, -1.0);                   // Compression +ve

                var c_disp2 = Operate.Div(c_disp2, oGlblData.mm_factor);    // Convert to mm
                    c_disp2 = Operate.Mul(c_disp2, -1.0);                   // Compression +ve    

                var c_disp3 = Operate.Div(c_disp3, oGlblData.mm_factor);    // Convert to mm
                    c_disp3 = Operate.Mul(c_disp3, -1.0);                   // Compression +ve            
                    
                // C180 filter (convert to second first)
                
                var c_disp1_mm_s    = Operate.Dix(c_disp1, oGlblData.time_factor);
                var c_disp1_c180_mm = Operate.C180(c_disp1_mm_s, REG_DT);   // in mm for compression calc
           
                
                var c_disp2_mm_s    = Operate.Dix(c_disp2, oGlblData.time_factor);
                var c_disp2_c180_mm = Operate.C180(c_disp2_mm_s, REG_DT);   // in mm for compression calc

                var c_disp3_mm_s    = Operate.Dix(c_disp3, oGlblData.time_factor);
                var c_disp3_c180_mm = Operate.C180(c_disp3_mm_s, REG_DT);   // in mm for compression calc
               
              
                // Remove all curves and datums
                
                remove_all_curves_and_datums();
                
                // Convert from seconds back to model time
                
                var c_pf1 = Operate.Mux(c_disp1_c180_mm, oGlblData.time_factor);
                var c_pf2 = Operate.Mux(c_disp2_c180_mm, oGlblData.time_factor);
                var c_pf3 = Operate.Mux(c_disp3_c180_mm, oGlblData.time_factor);           
              
                // Set labels and style

                set_labels(c_pf1, "Chest Upper Rib Deflection", "Time (" + oGlblData.unit_time + ")", "Compression (mm)");
                set_line_style(c_pf1, Colour.BLACK);
                set_labels(c_pf2, "Chest Middle Rib Deflection", "Time (" + oGlblData.unit_time + ")", "Compression (mm)");
                set_line_style(c_pf2, Colour.BLUE);
                set_labels(c_pf3, "Chest Lower Rib Deflection", "Time (" + oGlblData.unit_time + ")", "Compression (mm)");
                set_line_style(c_pf3, Colour.CYAN);
               
                write_image(output_data.rib.title, output_data.rib.fname);

                output_data.rib.curveList.push(c_pf1.id);
                output_data.rib.curveList.push(c_pf2.id);
                output_data.rib.curveList.push(c_pf3.id);
                write_curve("cur", output_data.rib.curveList, output_data.rib.fname);
                write_curve("csv", output_data.rib.curveList, output_data.rib.fname);
                
                // Calc values
                
                var chest_compression = Math.max(Math.max(c_pf1.ymax, c_pf2.ymax), c_pf3.ymax);
                
                write_variable(f, dummy_.toUpperCase() + "RIB_DEFLECTION", chest_compression.toFixed(3), dummy + "rib deflection value",  "String");
            
                // Remove all curves and datums

                remove_all_curves_and_datums();
                
            }
        }
        else
        {
            var blank_image_msg;

            blank_image_msg = "UNDEFINED " + dummy.toUpperCase() + "CHEST RIB DEFLECTION SPRING(s)";
            create_blank_image(blank_image_msg, output_data.rib.fname);    
        }
    }                
}

function do_chest_1_sensor_rating_calc(f, num_dummies)
{
    for(var pass = 0; pass < num_dummies; pass++)
    {
        if (pass == 0)
        {
            spring = oGlblData.driver_chest_spring;
            var version = oGlblData.driver_dummy;
            
            dummy = "Driver ";
            dummy_ = "Driver_";
        }
        
        if (pass == 1) 
        {
            spring = oGlblData.passenger_chest_spring;
            var version = oGlblData.passenger_dummy;
            
            dummy = "Passenger ";
            dummy_ = "Passenger_";
        }
        
        var output_data = {};
        
        output_data.rib = new OutputData(dummy + "Chest Rib Deflections",   images_dir + "/" + dummy_ + "Rib_Deflections");
        
        if( spring != undefined )
        {
            var c_rot = read_data("SPRING ROT", 0, spring, "RT");
            
            if( !c_rot )
            {
                var blank_image_msg;

                blank_image_msg = "NO " + dummy.toUpperCase() + "CHEST ROTATION DATA FOR TRANSDUCER " + spring;
                create_blank_image(blank_image_msg, output_data.rib.fname);
            }
            else
            {
                
                // Chest compression is calculated by converting the rotation in radians to a deflection using
                // a transfer function.
                //
                // For V7 dummies and earlier, the conversion is linear
                // For V8 dummies and later, the conversion uses a 3rd order polynomial
                //
                // The dummy manuals say that a negative rotation indicates compression, however testing has shown
                // that in some versions it is actually the other way around. If the maximum absolute value is +ve
                // then multiply the rotation by -1 so the conversion to compression works correctly.

                var rmax = c_rot.ymax;
                var rmin = c_rot.ymin;

                if(Math.abs(rmax) > Math.abs(rmin)) c_rot = Operate.Mul(c_rot, -1);

                var c_disp_mm;

                switch (version)
                {
                    case "LSTC HIII 50M Detailed v190217_beta":
                    case "LSTC HIII 50M Fast V2.0":
                    case "LSTC HIII 50M v130528_beta":
                        c_disp_mm = Operate.Mul(c_rot, CHEST_LSTC_HIII_50TH_MALE_DETAILED_190217_BETA_ROT_TO_COM_FACTOR);
                        break;
                    case "LSTC HIII 5F v2.0":
                        c_disp_mm = Operate.Mul(c_rot, CHEST_V2_LSTC_ROT_TO_COM_FACTOR);
                        break;
                    case "V7_HUMANETICS_HIII_50TH_MALE":
                    case "V6_HUMANETICS_HIII_5TH_FEMALE":
                        c_disp_mm = Operate.Mul(c_rot, CHEST_V7_ROT_TO_COM_FACTOR);
                        break;
                    case "V8_HUMANETICS_HIII_50TH_MALE":   
                    case "Humanetics HIII 50M v1.5":
                    case "Humanetics HIII 50M v1.5.1":
                        var curve_id = Curve.FirstFreeID();
                        c_disp_mm = new Curve(curve_id);
                        var n = c_rot.npoints;
                        for(var i=1; i<=n; i++)
                        {
                            var p_rot = c_rot.GetPoint(i);

                            var time = p_rot[0];
                            var com  = CHEST_V8_ROT_TO_COM_FACTOR_A * p_rot[1] * p_rot[1] * p_rot[1] +
                                CHEST_V8_ROT_TO_COM_FACTOR_B * p_rot[1] * p_rot[1] + 
                                CHEST_V8_ROT_TO_COM_FACTOR_C * p_rot[1];

                            c_disp_mm.AddPoint(time, com);
                        }
                        break;
                    case "V7_HUMANETICS_HIII_5TH_FEMALE":
                    case "Humanetics HIII 5F v2.02":
                        var curve_id = Curve.FirstFreeID();
                        c_disp_mm = new Curve(curve_id);
                        var n = c_rot.npoints;
                        for(var i=1; i<=n; i++)
                        {
                            var p_rot = c_rot.GetPoint(i);

                            var time = p_rot[0];
                            var com  = CHEST_V7_5TH_FEMALE_ROT_TO_COM_FACTOR_A * p_rot[1] * p_rot[1] * p_rot[1] +
                                CHEST_V7_5TH_FEMALE_ROT_TO_COM_FACTOR_B * p_rot[1] * p_rot[1] + 
                                CHEST_V7_5TH_FEMALE_ROT_TO_COM_FACTOR_C * p_rot[1];

                            c_disp_mm.AddPoint(time, com);
                        }
                        break;
                    default:
                        ErrorMessage("Unsupported dummy \""+version+"\" in do_chest_1_sensor_rating_calc.");
                        write_blank_images(output_data, "UNSUPPORTED DUMMY \""+version+"\".");
                        return;
                        break;
                }
                
                // C180 filter (convert to second first)
                
                var c_disp1_mm_s    = Operate.Dix(c_disp_mm, oGlblData.time_factor);
                var c_disp1_c180_mm = Operate.C180(c_disp1_mm_s, REG_DT);   // in mm for compression calc
              
                // Remove all curves and datums
                
                remove_all_curves_and_datums();
                
                // Convert from seconds back to model time
                
                var c_pf1 = Operate.Mux(c_disp1_c180_mm, oGlblData.time_factor);
                
                // Set labels and style

                set_labels(c_pf1, "Chest Deflection", "Time (" + oGlblData.unit_time + ")", "Compression (mm)");
                set_line_style(c_pf1, Colour.BLACK);
                
                write_image(output_data.rib.title, output_data.rib.fname);

                output_data.rib.curveList.push(c_pf1.id);
                write_curve("cur", output_data.rib.curveList, output_data.rib.fname);
                write_curve("csv", output_data.rib.curveList, output_data.rib.fname);
                
                // Calc values
                
                var chest_compression = c_pf1.ymax;
                
                write_variable(f, dummy_.toUpperCase() + "RIB_DEFLECTION", chest_compression.toFixed(3), dummy + "Chest Deflection value",  "String");
            
                // Remove all curves and datums

                remove_all_curves_and_datums();
                
            }
        }
        else
        {
            var blank_image_msg;

            blank_image_msg = "UNDEFINED " + dummy.toUpperCase() + "CHEST RIB TRANSDUCER";
            create_blank_image(blank_image_msg , output_data.rib.fname);    
        }
    }                
}

function do_abdomen_rating_calc(f, num_dummies)
{
//Calculates the abdomen scores for the driver dummy and writes relevant 
//values into the Reporter csv variable file <f>.
      
  for(var pass = 0; pass < num_dummies; pass++)
    {
        if (pass == 0)
        {
            var beam1 = oGlblData.driver_abdomen_beam_frnt;
            var beam2 = oGlblData.driver_abdomen_beam_mid;
            var beam3 = oGlblData.driver_abdomen_beam_back;
            
            dummy = "Driver ";
            dummy_ = "Driver_";
        }
        
        if (pass == 1) 
        {
            var beam1 = oGlblData.passenger_abdomen_beam_frnt;
            var beam2 = oGlblData.passenger_abdomen_beam_mid;
            var beam3 = oGlblData.passenger_abdomen_beam_back;
            
            dummy = "Passenger ";
            dummy_ = "Passenger_";
        }
        
        var output_data = {};
        
        output_data.abdomen = new OutputData(dummy + "Abdomen Force",   images_dir + "/" + dummy_ + "Abdomen_Force");
        
        if( beam1 != undefined && beam2 != undefined && beam3 != undefined )
        {
            var c_pf1 = read_data("BEAM BASIC", 0, beam1, "NY");
            var c_pf2 = read_data("BEAM BASIC", 0, beam2, "NY");
            var c_pf3 = read_data("BEAM BASIC", 0, beam3, "NY");
        
            if( !c_pf1 || !c_pf2 || !c_pf3 )
            {
                var blank_image_msg;

                var abd_force_id_lst = [];
                
                if( beam1_exists == null )
                {
                    abd_force_id_lst.push(" "+spring1);
                }
                if( beam2_exists == null )
                {
                    abd_force_id_lst.push(" "+spring2);
                }
                if( beam3_exists == null )
                {
                    abd_force_id_lst.push(" "+spring3);
                }
            
                blank_image_msg = "NO " + dummy.toUpperCase() + "ABDOMEN FORCE DATA FOR BEAM(s)" + abd_force_id_lst;
                create_blank_image(blank_image_msg, output_data.abdomen.fname);
            }
            else
            {   
            // C600 filter (convert to seconds) and convert force into kN

                var c_pf1 = Operate.Dix(c_pf1, oGlblData.time_factor);
                var c_pf1 = Operate.C600(c_pf1, REG_DT);
                var c_pf1 = Operate.Div(c_pf1, oGlblData.kn_factor);

                var c_pf2 = Operate.Dix(c_pf2, oGlblData.time_factor);
                var c_pf2 = Operate.C600(c_pf2, REG_DT);
                var c_pf2 = Operate.Div(c_pf2, oGlblData.kn_factor);

                var c_pf3 = Operate.Dix(c_pf3, oGlblData.time_factor);
                var c_pf3 = Operate.C600(c_pf3, REG_DT);
                var c_pf3 = Operate.Div(c_pf3, oGlblData.kn_factor);
      
                // Remove all curves and datums
        
                remove_all_curves_and_datums();
        
                // Convert from seconds back to model time
        
                c_pf1 = Operate.Mux(c_pf1, oGlblData.time_factor);
                c_pf2 = Operate.Mux(c_pf2, oGlblData.time_factor);
                c_pf3 = Operate.Mux(c_pf3, oGlblData.time_factor);
      
                //  Total Abdominal force
      
                c_pf = Operate.Add(c_pf1, c_pf2);
                c_pf.RemoveFromGraph();
                c_pf = Operate.Add(c_pf, c_pf3);
                           
                // Set labels and style

                set_labels(c_pf1, "Front Abdomen Force ", "Time (" + oGlblData.unit_time + ")", "Force (kN)");
                set_line_style(c_pf1, Colour.BLUE);
                set_labels(c_pf2, "Middle Abdomen Force ", "Time (" + oGlblData.unit_time + ")", "Force (kN)");
                set_line_style(c_pf2, Colour.CYAN);
                set_labels(c_pf3, "Back Abdomen Force ",        "Time (" + oGlblData.unit_time + ")", "Force (kN)");
                set_line_style(c_pf3, Colour.GREEN);
                set_labels(c_pf, "Resultant Force ", "Time (" + oGlblData.unit_time + ")", "Force (kN)");
                set_line_style(c_pf, Colour.BLACK);      
         

                write_image(output_data.abdomen.title, output_data.abdomen.fname);

                output_data.abdomen.curveList.push(c_pf1.id);
                output_data.abdomen.curveList.push(c_pf2.id);
                output_data.abdomen.curveList.push(c_pf3.id);
                output_data.abdomen.curveList.push(c_pf.id);
                write_curve("cur", output_data.abdomen.curveList, output_data.abdomen.fname);
                write_curve("csv", output_data.abdomen.curveList, output_data.abdomen.fname);
        
                // Calc values
        
                var abdomen_force = (c_pf.ymax)*1000;
        
                // Write out variables

                write_variable(f, dummy_ + "ABDOMEN_FORCE", abdomen_force.toFixed(3), dummy + "total abdomen force value",  "String");

                // Remove all curves and datums

                remove_all_curves_and_datums();
            }
        }
        else
        {
            var blank_image_msg;

            blank_image_msg = "UNDEFINED " + dummy.toUpperCase() + "ABDOMEN FORCE BEAM(s)";
            create_blank_image(blank_image_msg , output_data.abdomen.fname);
        }
    }
}

function do_pubic_symphysis_rating_calc ( f, num_dummies )
{
//Calculates the pubic symphysis scores for the driver dummy and writes relevant 
//values into the Reporter csv variable file <f>.

  for(var pass = 0; pass < num_dummies; pass++)
    {
        if (pass == 0)
        {
            var beam = oGlblData.driver_pubic_symphysis_beam;
            
            dummy = "Driver ";
            dummy_ = "Driver_";
        }
        
        if (pass == 1) 
        {
            var beam = oGlblData.passenger_pubic_symphysis_beam;
            
            dummy = "Passenger ";
            dummy_ = "Passenger_";
        }
        
        var output_data = {};
        
        output_data.pubic_symphysis = new OutputData(dummy + "Pubic Symphysis Force",   images_dir + "/" + dummy_ + "Pubic_Symphysis_Force");
        
        if (beam == undefined) 
        {
            blank_image_msg = "UNDEFINED " + dummy.toUpperCase() + "PUBIC SYMPHYSIS BEAM";
            create_blank_image(blank_image_msg , output_data.pubic_symphysis.fname);
        }
        else
        {
        // Should check data components are available for this beam

        // Read in forces
                
            var c_pf = read_data("BEAM BASIC", 0, beam, "NY");
                    
        // Check that the curves read in - i.e. the beam id and component are valid
        // Create a blank image to let the user know.

            if( !c_pf )
            {
                var title = "NO DATA FOR" + dummy.toUpperCase() + "PUBIC SYMPHYSIS FORCE BEAM " + beam;
                
                create_blank_image(title, output_data.pubic_symphysis.fname);
            }
            else
            {
        
            // C600 filter (convert to seconds) and convert force into kN

                var c_pf = Operate.Dix(c_pf, oGlblData.time_factor);
                var c_pf = Operate.C600(c_pf, REG_DT);
                var c_pf = Operate.Div(c_pf, oGlblData.kn_factor);
          
            // Multiply by -1.0 to make compressive force +ve

            // var c_pf1 = Operate.Mul(c_pf1, -1.0);
            // var c_pf2 = Operate.Mul(c_pf2, -1.0);       
          
            // Remove all curves and datums
            
                remove_all_curves_and_datums();
            
            // Convert from seconds back to model time
            
                c_pf = Operate.Mux(c_pf, oGlblData.time_factor);
          
          
            // Set labels and style

                set_labels(c_pf, "Pubic Symphysis Force", "Time (" + oGlblData.unit_time + ")", "Force (kN)");
                set_line_style(c_pf, Colour.BLUE);
      
                write_image(output_data.pubic_symphysis.title, output_data.pubic_symphysis.fname);

                output_data.pubic_symphysis.curveList.push(c_pf.id);
                write_curve("cur", output_data.pubic_symphysis.curveList, output_data.pubic_symphysis.fname);
                write_curve("csv", output_data.pubic_symphysis.curveList, output_data.pubic_symphysis.fname);
                
            
            // Calc values
            
                var pelvis_force = Math.max(c_pf.ymax, Math.abs(c_pf.ymin))*1000;
            

            // Write out variables


                write_variable(f, dummy_ + "PUBIC_FORCE",  pelvis_force.toFixed(3), dummy + "pubic symphysis Force value",  "String");

        
            // Remove all curves and datums

                remove_all_curves_and_datums();
            }
        }
    }
}

function do_pelvis_rating_calc (f, num_dummies, passenger_only)
{
//Calculates the pelvis scores for the dummy and writes relevant 
//values into the Reporter csv variable file <f>.

	Message("Doing pelvis rating calculation...");
    var dummy = [];
    dummy.push(new DummyData("Driver", oGlblData.driver_xsection_acetabular_id, "beam", oGlblData.driver_xsection_iliac_id, "beam", oGlblData.driver_dummy, "version"));
    dummy.push(new DummyData("Passenger", oGlblData.passenger_xsection_acetabular_id, "beam", oGlblData.passenger_xsection_iliac_id, "beam", oGlblData.passenger_dummy, "version"));
	for(var i = 0; i<dummy.length; i++)
    {	
        if(dummy[i].chart_label== "Driver")
		{
			var xsection1 = oGlblData.driver_xsection_iliac_id;
			var xsection2 = oGlblData.driver_xsection_acetabular_id;
        }
        else
        {
         var xsection1 = oGlblData.passenger_xsection_iliac_id;
		 var xsection2 = oGlblData.passenger_xsection_acetabular_id;  
        }   
      
        
        var output_data = {};
        
        output_data.pelvis = new OutputData(dummy[i].chart_label + " Pelvis Force",   images_dir + "/" + dummy[i].chart_label + "_Pelvis_Force");
 
        if (dummy[i].version == undefined)
            dummy[i].version = "<Unspecified>";
        
        if (xsection1 != undefined && xsection2 != undefined ) 
        {
        // Should check data components are available for this beam

        // Read in iliac and acetabular forces
        
            switch(dummy[i].version)
            {  
                case "<Unspecified>":
				case "Humanetics SID IIs SBLD v.4.3.1":
                case "Humanetics SID IIs SBLD v.4.3.2":
                case "Humanetics SID IIs SBLD v.4.3.5":
  			       var c_pf1 = read_data("SECTION", 0, xsection1, "FY");
			       var c_pf2 = read_data("SECTION", 0, xsection2, "FY");
                    break;

                case "LSTC SID IIS-D v0.150.beta":                 
  			       var c_pf1 = read_data("BEAM BASIC", 0, xsection1, "NY");
			       var c_pf2 = read_data("BEAM BASIC", 0, xsection2, "NY");
				   break;
                default:
                    ErrorMessage("Unsupported dummy \""+dummy[i].version+"\".");
                    write_blank_images(output_data, "UNSUPPORTED DUMMY \""+dummy[i].version+"\"");
                    break;
            }
    
        // Check that the curves read in - i.e. the beam id and component are valid
        // Create a blank image to let the user know.

            if( !c_pf1 || !c_pf2 )
            {
                var blank_image_msg;

                var pelvis_force_id_lst = [];
                
                if( xsec1_exists == null )
                {
                    pelvis_force_id_lst.push(" "+xsection1);
                }
                if( xsec2_exists == null )
                {
                    pelvis_force_id_lst.push(" "+xsection2);
                }
                blank_image_msg = "NO " + dummy[i].variable_label.toUpperCase() + "PELVIS FORCE DATA FOR CUT-SECTION(s)" + pelvis_force_id_lst;
                create_blank_image(blank_image_msg, output_data.pelvis.fname);
            }
            else
            {
    
            // C180 filter (convert to seconds) and convert force into kN

                var c_pf1 = Operate.Dix(c_pf1, oGlblData.time_factor);
                var c_pf1 = Operate.C180(c_pf1, REG_DT);
                var c_pf1 = Operate.Div(c_pf1, oGlblData.kn_factor);

                var c_pf2 = Operate.Dix(c_pf2, oGlblData.time_factor);
                var c_pf2 = Operate.C180(c_pf2, REG_DT);
                var c_pf2 = Operate.Div(c_pf2, oGlblData.kn_factor);    
      
            // Remove all curves and datums
        
                remove_all_curves_and_datums();
        
            // Convert from seconds back to model time
        
                c_pf1 = Operate.Mux(c_pf1, oGlblData.time_factor);
                c_pf2 = Operate.Mux(c_pf2, oGlblData.time_factor);
      
            //  Total Pelvic force
      
                var c_pf = Operate.Add(c_pf1, c_pf2);           
      
            // Set labels and style

                set_labels(c_pf1, "Iliac Wing Force Y", "Time (" + oGlblData.unit_time + ")", "Force (kN)");
                set_line_style(c_pf1, Colour.BLUE);
                set_labels(c_pf2, "Acetabulum Force Y", "Time (" + oGlblData.unit_time + ")", "Force (kN)");
                set_line_style(c_pf2, Colour.CYAN);
                set_labels(c_pf, "Total Pelvis Force", "Time (" + oGlblData.unit_time + ")", "Force (kN)");
                set_line_style(c_pf, Colour.BLACK);      
         
            // Save Image
                write_image(output_data.pelvis.title, output_data.pelvis.fname);

                output_data.pelvis.curveList.push(c_pf1.id);
                output_data.pelvis.curveList.push(c_pf2.id);
                output_data.pelvis.curveList.push(c_pf.id);
                write_curve("cur", output_data.pelvis.curveList, output_data.pelvis.fname);
                write_curve("csv", output_data.pelvis.curveList, output_data.pelvis.fname);
        
            // Calc values
        
                var pelvis_force_results = Math.max(c_pf.ymax,Math.abs(c_pf.ymin))*1000;
        

            // Write out variables

                write_variable(f, dummy[i].variable_label + "_PELVIS_FORCE",       pelvis_force_results.toFixed(3), dummy[i].variable_label + "pelvis force value",  "String");

            // Remove all curves and datums

                remove_all_curves_and_datums();
            }
        }
        else
        {
        // No beam defined - variables should be set to blank by first script in template.
        // Create a blank image to let the user know. 
            var blank_image_msg;

            blank_image_msg = "UNDEFINED " + dummy[i].variable_label.toUpperCase() + "PELVIS FORCE CUT-SECTION(s)";
            create_blank_image(blank_image_msg , output_data.pelvis.fname);
        }
    }
}

function do_femur_rating_calc(f, num_dummies)
{
    // Calculates the upper legs and knee values and writes them into the Reporter csv variables file <f>.

    var dummy = {};
    dummy.driver = {};
    dummy.driver.Left  = new DummyData("Driver", oGlblData.driver_left_femur_beam,  "beam", oGlblData.driver_dummy, "version");
    dummy.driver.Right = new DummyData("Driver", oGlblData.driver_right_femur_beam, "beam", oGlblData.driver_dummy, "version");
    dummy.passenger = {};
    dummy.passenger.Left  = new DummyData("Passenger", oGlblData.passenger_left_femur_beam,  "beam", oGlblData.passenger_dummy, "version");
    dummy.passenger.Right = new DummyData("Passenger", oGlblData.passenger_right_femur_beam, "beam", oGlblData.passenger_dummy, "version");
    
    for(var i in dummy) // loop through driver/passenger
    {
        for(var lr in dummy[i]) // loop through left/right side of dummy
        {            
            var output_data = {};

            output_data.com = new OutputData(dummy[i][lr].chart_label + " " + lr.toLowerCase() + " Femur Compression Force", images_dir + "/" + dummy[i][lr].variable_label + "_" + lr + "_Femur_Compression_Force");

// If dummy version was undefined, assume BEAM data
            if (dummy[i][lr].version == undefined)
                dummy[i][lr].version = "<Unspecified>";

            // Femur compression forces

            if (dummy[i][lr].beam_id != undefined)
            {
                // Read in beam longitudinal force
                // For Humanetics dummies this is measured from a beam;
                // for LSTC dummies this is measured from a cross section    
                switch (dummy[i][lr].version)
                {
                    case "<Unspecified>":
                    case "V7_HUMANETICS_HIII_50TH_MALE":
                    case "V8_HUMANETICS_HIII_50TH_MALE":
                    case "Humanetics HIII 50M v1.5":
                    case "Humanetics HIII 50M v1.5.1":
                    case "V6_HUMANETICS_HIII_5TH_FEMALE":
                    case "V7_HUMANETICS_HIII_5TH_FEMALE":    
                        var c_com_x = read_data("BEAM BASIC", 0, dummy[i][lr].beam_id, "NX");
                        var c_com_y = read_data("BEAM BASIC", 0, dummy[i][lr].beam_id, "NY");
                        var c_com_z = read_data("BEAM BASIC", 0, dummy[i][lr].beam_id, "NZ");
                        var entity_str = "BEAM"; // Used for error message below
                        break;

                    case "LSTC HIII 5F v2.0":
                    case "Humanetics HIII 5F v2.02":
                    case "LSTC HIII 50M Detailed v190217_beta":                        
                        var c_com_x = read_data("SECTION", 0, dummy[i][lr].beam_id, "FX");
                        var c_com_y = read_data("SECTION", 0, dummy[i][lr].beam_id, "FY");
                        var c_com_z = read_data("SECTION", 0, dummy[i][lr].beam_id, "FZ");
                        var entity_str = "SECTION"; // Used for error message below
                        break;

                    default:
                        ErrorMessage("Unsupported dummy \""+dummy[i][lr].version+"\" in do_femur_rating_calc.");
                        write_blank_images(output_data, "UNSUPPORTED DUMMY \""+dummy[i][lr].version+"\".");
                        return;
                        break;
                }
                    
                // Check that the curves read in - i.e. the beam id and component are valid
                // Create a blank image to let the user know.

                if( !c_com_x || !c_com_y || !c_com_z )
                {
                    write_blank_images(output_data, "NO DATA FOR " + entity_str + " " + dummy[i].beam_id);
                }
                else
                {
                    // Convert forces to kN from model units

                    var c_com_x_kn = Operate.Div(c_com_x, oGlblData.kn_factor);
                    var c_com_y_kn = Operate.Div(c_com_y, oGlblData.kn_factor);
                    var c_com_z_kn = Operate.Div(c_com_z, oGlblData.kn_factor);

                    // C600 filter (convert to seconds first)

                    var c_com_x_kn_s = Operate.Dix(c_com_x_kn, oGlblData.time_factor);
                    var c_com_y_kn_s = Operate.Dix(c_com_y_kn, oGlblData.time_factor);
                    var c_com_z_kn_s = Operate.Dix(c_com_z_kn, oGlblData.time_factor);

                    var c_com_x_c600 = Operate.C600(c_com_x_kn_s, REG_DT);
                    var c_com_y_c600 = Operate.C600(c_com_y_kn_s, REG_DT);
                    var c_com_z_c600 = Operate.C600(c_com_z_kn_s, REG_DT);

                    // Vector combine

                    var c_com_res = Operate.Vec(c_com_x_c600, c_com_y_c600, c_com_z_c600);    

                    // Remove all curves and datums

                    remove_all_curves_and_datums();

                    // Set labels and style

                    set_labels(c_com_res, "Femur " + lr +  " Force",    "Time (" + oGlblData.unit_time + ")",         "Force (kN)");
                    set_line_style(c_com_res, Colour.BLACK);

                    // Remove all curves and datums

                    remove_all_curves_and_datums();

                    // Create image/curve of femur compression exceedence curves

                    c_com_res.AddToGraph();

                    write_image(output_data.com.title, output_data.com.fname);
                    
                    output_data.com.curveList.push(c_com_res.id);                    
                    write_curve("cur", output_data.com.curveList, output_data.com.fname);
                    write_curve("csv", output_data.com.curveList, output_data.com.fname);
                    
                    // Remove all curves and datums
                    remove_all_curves_and_datums(); 

                    // Calc values
                    var com_results     = c_com_res.ymax;

                    // Write compression variables

                    write_variable(f, dummy[i][lr].variable_label.toUpperCase()+ "_" + lr.toUpperCase() + "_FEMUR_FORCE_VALUE",  com_results.toFixed(3), dummy + " " + lr + " Femur compression value",  "General");
                }
            }
            else
            {
                // No beam defined - variables should be set to blank by first script in template.
                // Create a blank image to let the user know.
                var blank_image_msg;
                
                blank_image_msg = "UNDEFINED " + dummy[i][lr].variable_label.toUpperCase() + " " + lr.toUpperCase() + " FEMUR LOADCELL";
                create_blank_image(blank_image_msg , output_data.com.fname);
            }
        }        

    }
}
 
function get_neck_nij_results(curves)
{
    // Calculates results for the neck Nij assessment returning
    // the rating and maximum Nij. 
    //
    // <curves> is an array containing the 4 Nij curves:
    //
    //    Tension-Extension
    //    Tension-Flexion
    //    Compression-Extension
    //    Compression-Flexion
    //

    var ret = new Array(2); 

    nij_max = 0.0;

    for(var j=0; j<curves.length; j++)
    {
        if(curves[j].ymax > nij_max)
        {
            nij_max = curves[j].ymax;
        }
    }

    return nij_max;
}
