/// <reference path="C:/SOURCE14/this_dir/this_js_api/this.intellisense.js" />
/// <reference path="C:/SOURCE14/reporter_dir/library/templates/scripts/this_common1.js" />
/// <reference path="C:/SOURCE14/reporter_dir/library/templates/scripts/this_common_EuroNCAP.js" />


// Common functions for EuroNCAP, C-NCAP templates
if(File.Exists(fname_csv) && File.IsFile(fname_csv))
{
	// Do the common setup of assigning values to the oGlblData object,
	// setting the T/HIS graph up and reading in the model.

	var oGlblData = new Object();

	var f_vars = do_common_single_analysis_setup(oGlblData);

    if (assessment == "Occupant")
    {
		// Convert variable datums to model time
		convert_variable_datum_times();

		// For each body region, do the calculation, writing results to the variables file.

		do_head_rating_calc(f_vars);

		do_neck_rating_calc(f_vars);

		do_chest_rating_calc(f_vars);

		do_knee_femur_pelvis_rating_calc(f_vars);

        do_lower_leg_foot_and_ankle_rating_calc(f_vars);
    }
    else if (assessment == "Compatibility")
    {
        var compatibility_output = compatibilityCalcs(f_vars)
        
        scoresAndPlots(f_vars, compatibility_output);
    }
    
	f_vars.Close();
}


function compatibilityCalcs(f_vars)
//=================================================
//  compatibilityCalcs:
//
//  Compute compatibility parameters: 
//      1) standard deviation: "standard_dev"
//      2) bottoming out: "bottomed_out"
//      3) OLC: "OccupantLoadCriterion"
//      4) Produce OLC time history graph
//=================================================
{
    var inputs = {};
	inputs.f_vars = f_vars;
	inputs.f_intrusions = images_dir + "/MPDB_barrier_def.csv";  // List of intrusion displacements read from MPDB_barrier_def.csv
	
    // Read the side of the impact (left or right), vehicle width, and in C-NCAP the longitudinal height from the input %CSV_MODEL_INFO_FILE%
	inputs.driver_side = oGlblData.left_or_right.toLowerCase();  // vehicle drive side
    if (inputs.driver_side == undefined)
    {
        ErrorMessage("Variable LEFT_OR_RIGHT for driver side undefined - expected LEFT or RIGHT.");
        Exit();
    }
	else if (inputs.driver_side != "left" && inputs.driver_side != "right")
    {
        ErrorMessage("Variable LEFT_OR_RIGHT for driver side \""+oGlblData.left_or_right+"\" - expected LEFT or RIGHT.");
        Exit();
    }
	
    inputs.imp_width = parseFloat(oGlblData.vehicle_width) * MPDB.EVAL_AREA_WIDTH_FRAC;
    if (isNaN(inputs.imp_width))
    {
        ErrorMessage("Failed to parse variable VEHICLE_WIDTH = \"" + oGlblData.vehicle_width + "\" as a number.");
        Exit();
    }
	
	if (template_reg === "CNCAP")
	{
		inputs.long_height_check = oGlblData.longitudinal_height_check;
		if (inputs.long_height_check == undefined)
		{
			WarningMessage("Variable LONGITUDINAL_HEIGHT_CHECK undefined - setting to TRUE.");
            inputs.long_height_check = "TRUE";
		}
		else if (inputs.long_height_check != "TRUE" && inputs.long_height_check != "FALSE")
		{
			WarningMessage("Variable LONGITUDINAL_HEIGHT_CHECK = \"" + inputs.long_height_check + "\" - expected TRUE or FALSE. Setting to TRUE.");
            inputs.long_height_check = "TRUE";
		}
	}
    
	// Perform several calculations based on intrusion depths
    var intrusion_calcs = intrusionCalcs(inputs);

    // Do OLC calculations
	var OLC = OLC_calcs(inputs);
    
    // Write variables for the template to be used within the Reporter script updating the intrusion table
    write_variable(inputs.f_vars, "LEFT_OR_RIGHT", oGlblData.left_or_right, "Left or right hand side driving", "String");
	write_variable(inputs.f_vars, "VEHICLE_WIDTH", oGlblData.vehicle_width, "Vehicle width", "String");
	if (template_reg === "CNCAP")
	{
		write_variable(inputs.f_vars, "LONGITUDINAL_HEIGHT_CHECK", inputs.long_height_check, "Longitudinal height exceeds 508 mm?", "String");
		write_variable(inputs.f_vars, "LONG_INTRUSION_MAX_POSS_SCORE", MPDB.NO_ATTENUATION_PENALTY, "Maximum score for the barrier height intrusion range", "String");
    }
    write_variable(inputs.f_vars, "RESULTS_SCHEME", oGlblData.results_scheme, "DEFAULT contour colours or euroncap.com SPREADSHEET colours with 633 mm cap", "String");
    
    // Create the object "results" containing the MPDB intrusion measures.
    // The "results" object is passed to the do_compatibility_score_and_plot function
    var results = {};
    results.standard_dev = intrusion_calcs.standard_dev;
	results.bottomed_out = intrusion_calcs.bottomed_out;
	if (template_reg === "CNCAP") results.not_attenuated = intrusion_calcs.not_attenuated;
    results.OLC = OLC;    
	
    return results;
	
}


function intrusionCalcs(inputs)
// Do calculations based on the calculated intrusion depths based on deformation depths stored in <inputs.f_intrusions>.
// - Standard deviation of intrusion depths within assessment area
// - Check for bottoming out
// - Attenuation at barrier height (C-NCAP ONLY)
// Write outputs to file with filename <inputs.f_vars>.
{
	var results = {};
	
	// Assign the appropriate variable values if the intrusion displacement list "MPDB_barrier_def.csv" is missing
    if (!File.Exists(inputs.f_intrusions))
    {
        ErrorMessage("Cannot find the intrusion displacements list: " + inputs.f_intrusions);
        
        write_variable(inputs.f_vars, "STANDARD_DEVIATION", "N/A", "Standard deviation of intrusion displacement", "String");
        write_variable(inputs.f_vars, "BOTTOMING_OUT", "N/A", "Barrier bottoming out boolean", "String");
		
		results.standard_dev = 0.0;
		results.bottomed_out = "N/A";
		
		if (template_reg === "CNCAP")
		{
			write_variable(inputs.f_vars, "LONGITUDINAL_INTRUSION_ATTENUATION", "N/A", "Barrier height intrusion boolean", "String");
			
			results.not_attenuated = "N/A";
		}
    }
    
    // If the MPDB_barrier_def.csv file is found then perform the standard deviation and bottoming out calculations.
	// Also calculate barrier intrusion height attenation (C-NCAP ONLY).
    else
    {		
		var num_rows 			= parseFloat(MPDB.NUM_ROWS);
		var num_columns 		= parseFloat(MPDB.NUM_COLUMNS);
		var bottom_out_limit 	= parseFloat(MPDB.BOTTOMING_OUT_LIMIT);
		var attenuation_limit 	= parseFloat(MPDB.ATTENUATION_LIMIT);
	
        var intrusions = readIntrusionCSV(inputs.f_intrusions, num_rows, num_columns);
        
        var bottomed_out = new Array(intrusions.length);
		var attenuation = new Array(intrusions.length);
		
        var intrusions_sum = 0;
        var assessment_area_points = 0;
        
        // Loop through the intrusion values to check for bottoming out / attenuation and to count the points   
        // within the assessment area
        for (var i = 0; i < intrusions.length; i++)
        {
            // Get the respective Z and Y coordinates in the barrier local reference system
            var y = getY(i);
			var z = getZ(i);
            var int_max = MPDB.SD_MAX_INTRUSION;
            bottomed_out[i] = false;
			attenuation[i] = false;
            
            // If the intrusion at position i is within the main assessment area
            if (checkAssessmentArea(y, z, inputs.driver_side, inputs.imp_width).assess_flag)
            {
                // Keep track of the number of points falling within the assessment area: assessment_area_points.
                // assessment_area_points is used to compute the intrusion standard deviation.
                assessment_area_points++;
                intrusions_sum += Math.min(intrusions[i], int_max);
              
                // If the bottoming out limit of the intrusion is exceeded  
                if (intrusions[i] > bottom_out_limit) bottomed_out[i] = true;
            }
			
			// If the intrusion at position i is within the region above the assessment area
            else if (checkAssessmentArea(y, z, inputs.driver_side, inputs.imp_width).intrusion_height_flag)
            {
				// If the attenuation limit of the intrusion is exceeded  
                if (intrusions[i] > attenuation_limit) attenuation[i] = true;
            }
        }
		
		// BOTTOMING OUT
		// Determine whether barrier has bottomed out
		// If a series of adjacent values in a rectangle are found to be bottomed_out[i] = true
		// then set the barrier to be bottomed out: barrier_bottomed_out = "Yes"
			
		// Set shape of area which defines 'bottoming out'
		// In C-NCAP, this area is 40mmX60mm, or a 2x3 rectangle at 20mm spacing 
		// In Euro NCAP, this area is 40mmX40mm, or a 2x2 square at 20mm spacing
		var shape = MPDB.BOTTOMING_OUT_SHAPE;
		var barrier_bottomed_out = checkArrayForRectangles(bottomed_out, shape, num_rows, num_columns);
        
        // Compute the standard deviation of the intrusion displacement over the assessment area
        var average = intrusions_sum / assessment_area_points;
        var standard_dev = StandardDeviation(intrusions, average, assessment_area_points, inputs.driver_side, inputs.imp_width);
        
        write_variable(inputs.f_vars, "STANDARD_DEVIATION", standard_dev.toFixed(1), "Standard deviation of intrusion displacement", "String");
        write_variable(inputs.f_vars, "BOTTOMING_OUT", barrier_bottomed_out, "Barrier bottoming out boolean", "String");
		
		results.standard_dev = standard_dev;
		results.bottomed_out = barrier_bottomed_out;
		
		// ATTENUATION (C-NCAP ONLY)
		if (template_reg === "CNCAP")
		{
			var not_attenuated;
			var score = 0.0;
			
			if (inputs.long_height_check === "FALSE")  // i.e. vehicle is not under 'curb mass status'
			{
				not_attenuated = "N/A";
			}
			
			else
			{
				// The penalty is applied if attenuation is NOT shown. As defined, attenuation occurs if:
				// - the intrusion depths in the area above the rectangle are below the limit; AND
				// - the intrusion depths on both sides of the rectangle are below the limit.
				// For a rectangle that is AxB, if it is not attenuated...
				// - in the vertical direction, then we should find a rectangle of dimensions (A+1)xB where
				//   all positions are above the limit.
				// - in the transverse direction, then we should find a rectangle of dimensions Ax(B+1) where
				//   all positions are above the limit.
				
				// Set shape of area which defines 'barrier intrusion at height'.
				// In C-NCAP, this area is defined as a 'continuous 6 elements' and is 
				// ASSUMED TO BE 40mmX60mm, or a 2x3 rectangle at 20mm spacing.
				// Therefore, the penalty should be applied if we find a rectangle that is 3x3, 
				// or one that is 2x4 (or 4x2).
				
				for (var i = 0; i < 2; i++)  // i.e. check each orientation (vertical, transverse)
				{
					var shape = Object.create(MPDB.ATTENUATION_SHAPE);  // Object.create so as to copy, not reference
					shape[i] += 1;  // i=0 checks vertical, i=1 checks transverse
					
					// Returns "Yes" if rectangle is found
					not_attenuated = checkArrayForRectangles(attenuation, shape, num_rows, num_columns);
					
					if (not_attenuated == "Yes") break; // i.e. if no attenuation --> no need to check transverse
				}
			}
			
			write_variable(inputs.f_vars, "LONGITUDINAL_INTRUSION_ATTENUATION", not_attenuated, "Barrier height intrusion boolean", "String");
			
			results.not_attenuated = not_attenuated;
			
		}	
    }
	
	return results;
	
}


function OLC_calcs(inputs)
// Calculate occupant load criterion and generate plots
{
	var output_data = {};
	var OccupantLoadCriterion;

    // Create the objects associates to the titles of the pictures created within the do_compatibility_parameters_calc() function
    output_data.vel =  new OutputData("Velocity and Acceleration", images_dir + "/" + "OLC_and_Virtual_Occupant_Velocity");
    output_data.disp = new OutputData("Displacement", images_dir + "/" + "OLC_and_Virtual_Occupant_Displacement");
	
	// Read the barrier CoG ID
    var node = "accelerometer: 1";

    // Perform OLC calculations only if the CoG node ID exists
    if (node != undefined)
	{
        var blank_image_msg;
        // read_xyz_accelerations(model_id, node_x, node_y, node_z)
        var cog_acc = read_xyz_accelerations(1, node);
        // read_data(type, model_id, id, comp)
        var cog_vx = read_data("NODE", 1, node, "VX"); // read_data("NODE", model_id, node_x, "AX");
        
        // If the X acceleration is undefined then create a blank image  
        if(!cog_acc.ax)
        {
            blank_image_msg = "NO X ACCELERATION DATA (NODE(s) " + node + ") ";
            create_blank_image(blank_image_msg + output_data.vel.title, output_data.vel.fname);
            create_blank_image(blank_image_msg + output_data.disp.title, output_data.disp.fname);
        }
		
        else
        {
            // Convert to g from model units
            var cog_g = convert_xyz_acc_to_g(cog_acc);
            
            // C180 filter (convert to seconds first)
            var c_gx_c180 = convert_to_s_and_filter(cog_g.gx, Operate.C180, MPDB.REG_DT);
            
            // read barrier initial velocity and convert in SI units
            var vel_init = cog_vx.YatX(0.0)*oGlblData.time_factor/oGlblData.len_factor;
            
            // Define barrier velocity by c180 filtered acceleration integration, 
            // and barrier initial velocity in X direction
			var OLC_g = MPDB.OLC_G;
            var vel_x = Operate.Int(c_gx_c180);
            var vel_x = Operate.Mul(vel_x, OLC_g);
            var vel_x = Operate.Add(vel_x, vel_init);
           
            // Define the MPDB displacement in [mm] by velocity integration 
            var disp_MPDB = Operate.Int(vel_x);
            var disp_MPDB_mm = Operate.Mul(disp_MPDB, 1000);
            
            // The free_flight_time curve is defined as the "transposed" time history of the 
            // relative displacement of the virtual occupant w.r.t. the MPDB displacement.
            // The free_flight_time is used to define the time at which the free flight phase ends:
            // transposition is required as only the *.YatX() method exists and not .XatY . 
            var free_flight_time = new Curve(Curve.FirstFreeID());
            
            // The curve free_flight_disp is defined by combining the relative displacement 
            // of the virtual occupant w.r.t. the MPDB displ. 
            // with the MPDB displacement time history at the initial velocity.
            // The free_flight_disp curve is used to define the virt. occupant total displacement 
            // when the relative displacement between the barrier and the occupant is 65mm (end of the free flight phase).
            var free_flight_disp = new Curve(Curve.FirstFreeID());
            
            // Create the free_flight_time and free_flight_disp curves by looping through
            // the MPDB displacement time history and the displ. time history of the virtual occupant
            // during the free flight phase: vel_init*1000*coord[0]
            for (var i=1; i<=disp_MPDB_mm.npoints; i++)
            {
                var coord = disp_MPDB_mm.GetPoint(i);
                var free_flight_disp_value = vel_init * 1000 * coord[0];
                var rel_disp_value = free_flight_disp_value-coord[1];
                
                free_flight_time.AddPoint(rel_disp_value,coord[0]);
                free_flight_disp.AddPoint(rel_disp_value,free_flight_disp_value);
            }
            
            // Get the maximum relative displacement between the displacement time history at the 
            // barrier initial velocity and the MPDB at the end of the simulation.
            // The max_relative_displ is used to check that the initial velocity and the analysis  
            // duration are sufficiently large to reach a relative disp. (in Euro NCAP: 65mm).
            var relative_displ_t_hist = free_flight_time.GetPoint(free_flight_time.npoints);
            var max_relative_displ = relative_displ_t_hist[0];
            var max_time = relative_displ_t_hist[1];
            
			// Collect variables for legibility
			var t1_disp = MPDB.OLC_FFT_T1_DISP * 1000;  // units: mm
			var t2_disp = MPDB.OLC_FFT_T2_DISP;  // units: m
			
            // If the barrier initial velocity and analysis termination time  
            // are sufficiently large to reach specified rel. displacement then:			
            if (max_relative_displ > t1_disp)
            {
                // Define t1 as the TIME when the rel. displacement 
                // between the virt. occ. and MPDB reaches specified displacement (mm):
                var t1 = free_flight_time.YatX(t1_disp);
             
                // Define s1 as the total VIRTUAL OCCUPANT TOTAL DISPLACEMENT when the rel. 
                // displacement between the barrier and MPDB reaches 65.0mm displacement
                var s1 = free_flight_disp.YatX(t1_disp);
             
                var v1 = vel_x.YatX(0.0);
             
                // The first guess of t2 assumes that the virtual occupant 
                // is moving at a speed equal to the initial barrier speed.
                // That is overestimating the OLC (~virt. occupant deceleration) as 
                // it is assuming zero stiffness of the ideal restraint system.
                var t2 = t2_disp/v1 + t1;
                
                // Check if the first guess of t2 (a lowerbound to the converged t2 value)
                // is smaller then the termination time
                if (t2 <= max_time)
                {
                    var v2 = vel_x.YatX(t2);

                    OccupantLoadCriterion = (v1 - v2)  / (t2 - t1) / OLC_g;
                    
                    // Compute the relative displacement between virtual occupant and MPDB for the guessed t2
                    var delta_L = (v1 + v2) * (t2 - t1) / 2 - (disp_MPDB_mm.YatX(t2) - disp_MPDB_mm.YatX(t1))/1000;
                    
                    // Assume how much of t2 is varied based on the displacement residual: t2_ratio.
                    // This value has been calibrated by trials and errors - it`s convergence properties 
                    // have no theoretical back up - although the set of problems involving the MPDB 
                    // crash seem to be a sufficiently well posed problem to guarantee convergence
                    // of any reasonable/conceivable scenario.
                    var t2_ratio = 0.1;
                    
                    // Update t2 based on the relative displacemnt delta_L.
                    // If delta_L is larger then the converged value 0.235m then 
                    // the ideal restrint phase should terminate earlier, conversely if 
                    // delta_L<0.235m the ideal restraint phase should last longer
                    t2 = t2 * ( 1 + t2_ratio * ( t2_disp / delta_L - 1 ) );
                    
                    // iterations are counted to escape the while loop if the converged solution is not reached
                    // after a limited number of t2 updates.
                    // The converged solution is generally reached in less then 50 iterations.
                    // A maximum number of 100 iterations is allowed.
                    var iter = 0;
                    
                    while ( Math.abs(t2_disp - delta_L)/ t2_disp > 0.0000001 && t2 <= max_time && iter < 100)
                    {
                        
                        v2 = vel_x.YatX(t2);

                        OccupantLoadCriterion = (v1 - v2)  / (t2 - t1) / OLC_g;
                        
                        delta_L = (v1 + v2) * (t2 - t1) / 2 - (disp_MPDB_mm.YatX(t2) - disp_MPDB_mm.YatX(t1))/1000;
                        
                        t2 = t2 * ( 1 + t2_ratio * ( t2_disp / delta_L - 1 ) );
                        
                        iter++;
                        
                    }
                    
                    WarningMessage("Total number of iterations " + iter);
                    
                    if (iter == 100)
                    {
                        ErrorMessage("Maximum number of iterations reached.");
                    }
                    // t2, v2 and the OLC are assigned arbitrarily if the relative displacement accumulated 
                    // during the ideal restraint phase cannot reach 235mm by the end of the simulation.
                    // That is checked: 
                    // 1) over the while loop 
                    if (t2 > max_time) 
                    {
                        var t2 = max_time;
                        var v2 = vel_x.YatX(max_time);
                        var delta_L = t2_disp; 
                        OccupantLoadCriterion = 0.0;
                    }
                }
                //     2) over the 1st guess of t2 with a zero stifness ideal restraint.
                else
                {
                    var t2 = max_time;
                    var v2 = vel_x.YatX(max_time);
                    var delta_L = t2_disp; 
                    OccupantLoadCriterion = 0.0;
                }
            }
            // t1, t2, v1, v2 and the OLC are assigned arbitrary values in case the maximum relative
            // displacement between the MPDB barrier and the virt. occupant during the free flight phase 
            // does not reach 65mm by the end of the simulation: 
            // i.e. low initial velocities or very short analysis time termination.
            else
            {
                var t1 = max_time;
                var t2 = max_time;
                var v1 = vel_x.YatX(0.0);
                var v2 = v1; 
                var delta_L = t2_disp; 
                OccupantLoadCriterion = 0.0;
            }

            // Write variables to this_vars.csv
            write_variable(inputs.f_vars, "OCCUPANT_LOAD_CRITERION", OccupantLoadCriterion.toFixed(1), "Occupant Load Criterion", "String");

            write_variable(inputs.f_vars, "OCCUPANT_LOAD_CRITERION_VEL1", v1.toFixed(1), "Occupant velocity during the free flight phase", "String");
            write_variable(inputs.f_vars, "OCCUPANT_LOAD_CRITERION_DIST1", (t1_disp).toFixed(1), "Time at the end of free flight phase", "String");
            write_variable(inputs.f_vars, "OCCUPANT_LOAD_CRITERION_TIME1", (t1*1000).toFixed(1), "Time at the end of free flight phase", "String");
            
            write_variable(inputs.f_vars, "OCCUPANT_LOAD_CRITERION_VEL2", v2.toFixed(1), "Occupant velocity at the end of the ideal restraint phase", "String");
            write_variable(inputs.f_vars, "OCCUPANT_LOAD_CRITERION_DIST2", (delta_L*1000+t1_disp).toFixed(1), "Time at the end of ideal restraint phase", "String");
            write_variable(inputs.f_vars, "OCCUPANT_LOAD_CRITERION_TIME2", (t2*1000).toFixed(1), "Time at the end of ideal restraint phase", "String");
            
			// Create virtual occupant velocity time history
			var prescribed_virt_occ_vel = new Curve(Curve.FirstFreeID());
			prescribed_virt_occ_vel.AddPoint(0,v1);
			prescribed_virt_occ_vel.AddPoint(t1,v1);
			prescribed_virt_occ_vel.AddPoint(t2,v2);
			prescribed_virt_occ_vel.AddPoint(vel_x.xmax,v2);
			
			// Create virtual occupant displacement time history by time integration of prescribed_virt_occ_vel 
			prescribed_virt_occ_vel = Operate.Reg(prescribed_virt_occ_vel, 0.1E-4);
			var Virt_Occ_Disp = Operate.Int(prescribed_virt_occ_vel);
			Virt_Occ_Disp = Operate.Mul(Virt_Occ_Disp, 1000);
			
			// create a relative displacement time history: "virtual occupant" - "MPDB displacemnt"
			rel_disp = Operate.Sub(Virt_Occ_Disp,disp_MPDB_mm);

			yaxis_min = c_gx_c180.ymin;
			yaxis_max = v1;
			DialogueInputNoEcho("/dou ","on"); 
			DialogueInputNoEcho("/y2 ",c_gx_c180.id.toString(),""); 
			
			// Set the primary and secondary axis limits
			var graph = Graph.GetFromID(1);
			graph.y2min = yaxis_min*1.1;
			graph.y2max = Math.max(yaxis_max*1.1,c_gx_c180.ymax);
			graph.ymin = yaxis_min;
			graph.ymax = yaxis_max;
			graph.y2_unit_size = Graph.FONT_SIZE_18;
			graph.y2label_size = Graph.FONT_SIZE_24;
			graph.x_unit_decimals  = 0;
			graph.y_unit_decimals  = 0;
			graph.y2_unit_decimals = 0;
			
			// Move gravity curve to the end of the curve lists for clarity of the legend
			var MPDB_CoG_gx = new Curve(Curve.HighestID()+1);
			Curve.Copy(c_gx_c180.id,MPDB_CoG_gx.id);
						
			// Remove all curves and datums
			remove_all_curves_and_datums();
			
			// Add free flight phase datum
			//================================================================
			if(Datum.Exists("free flight phase"))       Datum.Delete("free flight phase");

			var d_free_flight = new Datum("free flight phase", Datum.CONSTANT_X, t1*1000);
			
			d_free_flight.style = LineStyle.DASH2;
			d_free_flight.label = "free flight phase";
			d_free_flight.label_position = Datum.LABEL_BOTTOM_LEFT;
			d_free_flight.label_size = Datum.LABEL_18_POINT;
			//================================================================
			if(Datum.Exists("ideal restraint phase"))       Datum.Delete("ideal restraint phase");

			var d_ideal_restraint = new Datum("ideal restraint phase", Datum.CONSTANT_X, t2*1000);
			
			d_ideal_restraint.style = LineStyle.DASH2;
			d_ideal_restraint.label = "ideal restraint phase";
			d_ideal_restraint.label_position = Datum.LABEL_BOTTOM_LEFT;
			d_ideal_restraint.label_size = Datum.LABEL_18_POINT;
			//================================================================
			
			// Convert x axis to ms
			vel_x                   = Operate.Mux(vel_x, 1000.0, vel_x);
			prescribed_virt_occ_vel = Operate.Mux(prescribed_virt_occ_vel, 1000.0, prescribed_virt_occ_vel);
			MPDB_CoG_gx             = Operate.Mux(MPDB_CoG_gx, 1000.0, MPDB_CoG_gx);
			
			// Set labels and style of velocity [m/s] and acceleration [g]
			set_labels(vel_x,                   "X velocity of the barrier CoG",      "Time (ms)", "Velocity (m/s)");
			set_labels(prescribed_virt_occ_vel, "X velocity of the virtual occupant", "Time (ms)", "Velocity (m/s)");
			set_labels(MPDB_CoG_gx,             "X acceleration of the barrier CoG",  "Time (ms)", "Acceleration (g)");

			set_line_style(vel_x, Colour.BLACK);
			set_line_style(prescribed_virt_occ_vel, Colour.RED);
			set_line_style(MPDB_CoG_gx, Colour.BLUE);

			
			// Create image and curve files of velocities` time history plot
			vel_x.AddToGraph();
			prescribed_virt_occ_vel.AddToGraph();
			MPDB_CoG_gx.AddToGraph();

			// Set up different blank images error messages for the different scenarios:
            // The virtual occupant max relative displacement is less then 65mm:
            if (t1 == max_time)
            {
                output_data.vel.title = "End of free-flight phase not reached";
                output_data.disp.title = "End of free-flight phase not reached";
            }
            // The virtual occupant max relative displacement is less then 65 + 235mm:
            else if(t2 == max_time)
            {
                output_data.vel.title = "End of ideal restraint phase not reached";
                output_data.disp.title = "End of ideal restraint phase not reached";
            }
            // The maximum number of iterations is reached without achieving convergence:
            else if (iter == 100)
            {
                output_data.vel.title = "Iteration limit reached in OLC calculation – check your D3PLOT results";
				output_data.disp.title = "Iteration limit reached in OLC calculation – check your D3PLOT results";
            }

			write_image(output_data.vel.title, output_data.vel.fname, [ vel_x, MPDB_CoG_gx, prescribed_virt_occ_vel ],yaxis_max);

			write_curve("cur", [ vel_x.id, MPDB_CoG_gx.id, prescribed_virt_occ_vel.id ], output_data.vel.fname);
			write_curve("csv", [ vel_x.id, MPDB_CoG_gx.id, prescribed_virt_occ_vel.id ], output_data.vel.fname);
			
			// Turn off second y axis for displacements plot
			DialogueInputNoEcho("/dou ","off");
			
			// Convert x axis to ms
			Virt_Occ_Disp  = Operate.Mux(Virt_Occ_Disp,1000.0,Virt_Occ_Disp);
			disp_MPDB_mm_ms= Operate.Mux(disp_MPDB_mm,1000.0);
			rel_disp       = Operate.Mux(rel_disp,1000.0,rel_disp);           
							
			// Set labels and style of displacement time histories [mm]
			set_labels(Virt_Occ_Disp,   "Displacement of the virtual occupant", "Time (ms)", "Displacement (mm)");
			set_labels(disp_MPDB_mm_ms, "Displacement of the barrier CoG",      "Time (ms)", "Displacement (mm)");
			set_labels(rel_disp,        "Relative displacement",                "Time (ms)", "Displacement (mm)");

			set_line_style(Virt_Occ_Disp, Colour.RED);
			set_line_style(disp_MPDB_mm_ms, Colour.BLACK);
			set_line_style(rel_disp, Colour.BLUE);
			
			// Remove all curves and datums
			remove_all_curves_and_datums();
			
			yaxis_max = Virt_Occ_Disp.ymax;
			
			// Add back free flight and ideal restraint phases datum
			d_free_flight.AddToGraph();
			d_ideal_restraint.AddToGraph();
			
			// Create image and curve files of displacements` time history plot
			Virt_Occ_Disp.AddToGraph();
			disp_MPDB_mm_ms.AddToGraph(); 
			rel_disp.AddToGraph(); 

			DialogueInputNoEcho("AU");
                
			write_image(output_data.disp.title, output_data.disp.fname, [ Virt_Occ_Disp, disp_MPDB_mm_ms, rel_disp],yaxis_max);

			write_curve("cur", [ Virt_Occ_Disp.id, disp_MPDB_mm_ms.id, rel_disp.id ], output_data.disp.fname);
			write_curve("csv", [ Virt_Occ_Disp.id, disp_MPDB_mm_ms.id, rel_disp.id ], output_data.disp.fname);
		}
    }
	
    else
    {
        // No node defined - variables should be set to blank by first script in template.
        // Create a blank image to let the user know.
        create_blank_image("NO NODE ID DEFINED FOR " + output_data.disp.title, output_data.disp.fname);
        create_blank_image("NO NODE ID DEFINED FOR " + output_data.vel.title, output_data.vel.fname);
    }
	
	return OccupantLoadCriterion;
	
}


function checkArrayForRectangles(array, shape, num_rows, num_columns)
// Check array <array> for any rectangles of dimensions given by <shape> where all positions
// within the rectangle have value True.
// Used to identify bottoming out and attenuation at barrier height.
{	
	for (var or = 0; or < 2; or++)  // check different orientations of shape, i.e. switch rows/columns if rectangular
	{
		// Check original orientation, i.e. rectangle that is 2 rows by 3 columns
		var r = shape[0];
		var c = shape[1];
		var ar = r * c;  // area of rectangle
		
		for (var i = 0; i < array.length; i++)
		{
			if (array[i] == false) continue;
							
			// Position of intrusion
			var row = i % num_rows;
			var col = Math.floor(i / num_rows);
			
			// If too close to edge to construct rectangle, skip;
			if (row + r > num_rows || col + c > num_columns) continue;

			// Get position of other instrusions in area
			// one row below: 			i + 1
			// one column to the right: i + num_rows				
			
			var neighbours = 0;				
			
			var idx = i;
			for (var i_c = 0; i_c < c; i_c++)  // iterate across columns
			{
				for (var i_r = idx; i_r < idx + r; i_r++)  // iterate across rows
				{						
					// If other intrusion is bottomed out, keep searching, otherwise break
					// If all other intrusions are bottomed out, list of neighbours will have length == area of rectangle
					if (array[i_r] == true) neighbours++;  // collect indices of neighbour intrusion measurements							
				}
				
				idx += num_rows;  // index of next column;
			}
			
			if (neighbours == ar)  // i.e. cell in question and neighbours in rectangle are all bottomed out, so length of list == area
			{
				return "Yes";
			}
		}
		
		if (r == c ) break;  // i.e. shape is a square so orientations are identical
		else  // if shape is a rectangle, transpose, i.e. rectangle that is 3 rows by 2 columns
		{
			shape[0] = c;
			shape[1] = r;
		}
	}
	
	return "No";
	
}


function scoresAndPlots(f_vars, out)
//=================================================
//  scoresAndPlots:
//
//  Compute compatibility scores and plots: 
//      1) Compute standard deviation, bottoming out and OLC scores
//      2) Produce the compatibility score graph
//=================================================
{
	
//========================
//================ SCORING
//========================

//=================================================  	
    // Assign bottoming out score
	var bottomed_out = out.bottomed_out;
	var bottoming_score;
	
	var BO_pen = MPDB.BOTTOMING_OUT_PENALTY * scores_SF;  // maximum possible penalty (scaled)
	
	if (bottomed_out == "No") bottoming_score = 0;
    else bottoming_score = BO_pen;  // scaled
	
	write_variable(f_vars, "B_OUT_MAX_POSS_SCORE", BO_pen.toFixed(1), "Maximum score for bottoming out", "String");

//=================================================
	// Assign OLC score - linear interpolation between minimum and maximum
	var OccupantLoadCriterion = out.OLC;
	var OLC_score;
	
	var OLC_scale_min = MPDB.OLC_SCALE_MIN;
	var OLC_scale_max = MPDB.OLC_SCALE_MAX;
	var OLC_scale_diff = OLC_scale_max - OLC_scale_min;
	var OLC_pen = MPDB.OLC_PENALTY * scores_SF;  // maximum possible penalty (scaled)

    // Compute OLC score: compatibility modifier component related to Occupant Load Criterium (for 0% Intrusion Standard Deviation)
    var OLC_pct = (OccupantLoadCriterion - OLC_scale_min) / OLC_scale_diff;  // percentage over which OLC score is interpolated
	var OLC_util = OLC_pct * 100;  // as % out of 100 for plotting
    
	if (OLC_pct <= 0)  // i.e. less than minimum
	{
		OLC_score = 0;
		OLC_util = 0.0;  // bound the OLC_util between 0% and 100%
	}
    else if (OLC_pct > 0 && OLC_pct < 1) 
	{
		OLC_score = OLC_pct * OLC_pen;  // i.e. interpolated, scaled
	}
    else  // i.e. full penalty
	{
		OLC_score = OLC_pen;  // scaled
		OLC_util = 100.0;
	}

    write_variable(f_vars, "COMPATIBILITY_OCCUPANT_LOAD_CRITERION_SCORE", OLC_score.toFixed(3), "Compatibility Occupant Load Criterion score", "String");
	write_variable(f_vars, "OLC_MAX_POSS_SCORE", OLC_pen.toFixed(3), "Maximum score for the Occupant Load Criterion", "String");
	write_variable(f_vars, "COMPATIBILITY_OCCUPANT_LOAD_CRITERION_PCT", OLC_util.toFixed(0), "Compatibility Occupant Load Criterion percentage utilization", "String");

//=================================================  
    // Compute compatibility modifier component related to the value of Intrusion Standard Deviation
	// "Max modifier for this is 2 points. If SD is lower than 50mm, no modifier; if more than 150mm, 2 points modifier. 
	// For value between the limits, linear interpolation will be applied to calculate the results and it will be rounded to three decimal points."
    var SD = out.standard_dev;
	var SD_score;
	
	var SD_scale_min = MPDB.SD_SCALE_MIN;
	var SD_scale_max = MPDB.SD_SCALE_MAX;
	var SD_scale_diff = SD_scale_max - SD_scale_min;
	var SD_min_pen = MPDB.SD_PENALTY_MIN_OLC * scores_SF;  // scaled
	var SD_max_pen = MPDB.SD_PENALTY_MAX_OLC * scores_SF;  // scaled
	var SD_pen_diff = SD_max_pen - SD_min_pen;    // scaled
	
	var SD_pct = (SD - SD_scale_min) / SD_scale_diff; // percentage in range over which the SD score is interpolated
	var SD_util  = SD_pct * 100;  // as % out of 100 for plotting
	
	// Find max possible SD penalty depending on OLC % -- range_SD is already scaled
    if (OLC_pct <= 0) range_SD = SD_min_pen; 
    else if (OLC_pct > 0 && OLC_pct <= 1) range_SD = (SD_pen_diff * OLC_pct + SD_min_pen);  // i.e. interpolate between extremes when OLC=0% and OLC=100%
    else range_SD = SD_max_pen;
    
    write_variable(f_vars, "SD_MAX_POSS_SCORE", range_SD.toFixed(3), "Maximum score for the SD for the current OLC range", "String"); 			 
	
    if (SD_pct < 0)  // i.e. SD < SD_scale_min
	{
		SD_pct = 0.0;
		SD_score = 0;
		SD_util = 0.0;  // bound the SD_util between 0% and 100%
	}
    else if (SD_pct >= 0 && SD_pct <= 1) 
	{
		SD_score = SD_pct * range_SD;  // scaled
	}
    else  // i.e. SD > SD_scale_max; full penalty
	{
		SD_pct = 1.0;
		SD_score = range_SD;  // scaled
		SD_util = 100.0;
	}
	
//=================================================
	// Assign barrier intrusion height score - C-NCAP only
	if (template_reg === "CNCAP")
	{
		var not_attenuated = out.not_attenuated;		
		var not_attenuated_score;
	
		var NA_pen = MPDB.NO_ATTENUATION_PENALTY * scores_SF;  // maximum possible penalty (scaled)
		
		if (not_attenuated == "Yes") not_attenuated_score = NA_pen;  // scaled
		else not_attenuated_score = 0;
		
		write_variable(f_vars, "LONGITUDINAL_INTRUSION_SCORE", not_attenuated_score, "Barrier height intrusion score", "String");
	}

//=================================================

	// The total score is required for the score plot
	if (template_reg === "CNCAP") var addl_penalties = bottoming_score + not_attenuated_score;  // scaled
	else var addl_penalties = bottoming_score;  // scaled
	
    var tot_score = Math.min(MPDB.MAX_TOTAL_PENALTY, OLC_score + SD_score + addl_penalties);

//========================
//============== PLOTTING
//========================

    remove_all_curves_and_datums();
	
	// Create the single point curve associated to the total compatibility modifier score
    var mod = new Curve(Curve.FirstFreeID());
	mod.AddPoint(OccupantLoadCriterion, tot_score);
	set_line_style(mod, Colour.RED);
	mod.symbol = Symbol.CROSS;
	mod.label=("compatibility modifier = " + tot_score.toFixed(2))
	
    // Create the piecewise linear curve going through to the total compatibility
    // modifier dot ("mod" curve)
    var mod_domain = new Curve(Curve.FirstFreeID());
    
    min_OLC = Math.min(OccupantLoadCriterion, MPDB.OLC_SCALE_MIN - 2.0);
    max_OLC = Math.max(OccupantLoadCriterion, MPDB.OLC_SCALE_MAX + 3.0);    
    
    mod_domain.AddPoint(min_OLC-1, 0.0);
    mod_domain.AddPoint(max_OLC+1, 0.0);
    mod_domain.AddPoint(max_OLC+1, MPDB.MAX_TOTAL_PENALTY * scores_SF);
 
    mod_domain.style = LineStyle.NONE;
    mod_domain.symbol = Symbol.NONE;
    
    DialogueInputNoEcho("/de ","sy","on");
	DialogueInputNoEcho("/de ","gr","th", "1");		
    DialogueInputNoEcho("/AU");

    // Create a series of datum to define the domain of admissible
    // compatibility modifier scores. Do in three stages rather than all at once
    // so that the label can be placed visible position and not on the
	// right/left boundaries of the graph.
    //================================================================
	
	function DrawDatum(name, points, style, fill_colour, fill_dir, label, label_size, label_pos, line_colour, line_width)
	// Draw datum
	{
		if (Datum.Exists(name)) Datum.Delete(name);
		
		var d = new Datum(name, Datum.POINTS, points);
		if (style !== undefined)        d.style = style;
		if (fill_dir === "above" && fill_colour !== undefined) d.fill_colour_above = fill_colour;
		if (fill_dir === "below" && fill_colour !== undefined) d.fill_colour_below = fill_colour;
		if (label !== undefined)        d.label = label;
		if (label_size !== undefined)   d.label_size = label_size;
		if (label_pos !== undefined)    d.label_position = label_pos;
		if (line_colour !== undefined)  d.line_colour = line_colour;		
		if (line_width !== undefined)   d.line_width = line_width;

		return d;
		
	}

	//================================================================
	
	// Draw a piecewise line in three parts showing the upper bound at SD=100% with no bottoming out/barrier height penalty.
	// Drawn as piecewise because labels are always attached to the last point. Since some of the labels need to be placed
	// at the front of the line, i.e. positioned relative to the first section, the line needs to be divided into smaller sections.
	
	// 1. Left
	var points_sd_100_l = new Array(4);
	points_sd_100_l[0] = min_OLC-1;	
    points_sd_100_l[1] = MPDB.SD_PENALTY_MIN_OLC * scores_SF;
	points_sd_100_l[2] = OLC_scale_min;
    points_sd_100_l[3] = points_sd_100_l[1];
	
	d_sd_100_l = DrawDatum("SD_100_pct_left", 
							points_sd_100_l, 
							LineStyle.DASH2,
							undefined,
							undefined,
							"SD = 100%",
							Datum.LABEL_18_POINT);
							
	// 2. Centre
	var points_sd_100_c = new Array(4);
	points_sd_100_c[0] = points_sd_100_l[2];
    points_sd_100_c[1] = points_sd_100_l[3];
	points_sd_100_c[2] = OLC_scale_max;
	points_sd_100_c[3] = (MPDB.OLC_PENALTY + MPDB.SD_PENALTY_MAX_OLC) * scores_SF;
	
	d_sd_100_c = DrawDatum("SD_100_pct_centre", 
							points_sd_100_c, 
							LineStyle.DASH2);
							
	// 3. Right
	var points_sd_100_r = new Array(4);
	points_sd_100_r[0] = points_sd_100_c[2];
    points_sd_100_r[1] = points_sd_100_c[3];
	points_sd_100_r[2] = max_OLC+1;
	points_sd_100_r[3] = points_sd_100_r[1];
	
	d_sd_100_r = DrawDatum("SD_100_pct_right", 
							points_sd_100_r, 
							LineStyle.DASH2);
							
	//================================================================

	// Draw a piecewise line in three parts showing the lower bound at SD=0% with no bottoming out/barrier height penalty.
	// 1. Left
    points_sd_0_l = new Array(4);
    points_sd_0_l[0] = min_OLC-1;
    points_sd_0_l[1] = 0.0;
    points_sd_0_l[2] = OLC_scale_min;
    points_sd_0_l[3] = points_sd_0_l[1];
	
	d_sd_0_l = DrawDatum("SD_0_pct_l", 
						points_sd_0_l, 
						LineStyle.DASH2,
						Colour.LIGHT_GREY,
						"below");
						
	// 2. Centre
	points_sd_0_c = new Array(4);
    points_sd_0_c[0] = points_sd_0_l[2];
    points_sd_0_c[1] = points_sd_0_l[3];
    points_sd_0_c[2] = OLC_scale_max;
    points_sd_0_c[3] = MPDB.OLC_PENALTY * scores_SF;
	
	d_sd_0_c = DrawDatum("SD_0_pct_c", 
						points_sd_0_c, 
						LineStyle.DASH2,
						Colour.LIGHT_GREY,
						"below");
						
	// 3. Right
    points_sd_0_r = new Array(4);
    points_sd_0_r[0] = points_sd_0_c[2];
    points_sd_0_r[1] = points_sd_0_c[3];
    points_sd_0_r[2] = max_OLC+1;
    points_sd_0_r[3] = points_sd_0_r[1];
	
	d_sd_0_r = DrawDatum("SD_0_pct_r", 
						points_sd_0_r, 
						LineStyle.DASH2,
						Colour.LIGHT_GREY,
						"below");
						
	// 4. Dummy -- for purposes of plotting label
	points_sd_0_d = new Array(4);
    points_sd_0_d[0] = points_sd_0_r[2];
    points_sd_0_d[1] = points_sd_0_r[3];
	points_sd_0_d[2] = points_sd_0_d[0];
    points_sd_0_d[3] = points_sd_0_d[1];
	
	// Variable positioning to prevent labels from overlapping
	if (SD_util <= 50) var pos = Datum.LABEL_BELOW_LEFT;
	else var pos = Datum.LABEL_ABOVE_LEFT;
	
	d_sd_0_d = DrawDatum("SD_0_pct_d", 
						points_sd_0_d, 
						LineStyle.DASH2,
						Colour.LIGHT_GREY,
						"below",
						"SD = 0.0%",
						Datum.LABEL_18_POINT,
						pos);

	//================================================================
	
	// Draw a piecewise line in three parts showing the curve considering actual SD %, with no bottoming out/barrier height penalty.
	// 1. Left
	points_sd_l = new Array(4);
	points_sd_l[0] = min_OLC-1;
	points_sd_l[1] = SD_min_pen * SD_pct;
	points_sd_l[2] = OLC_scale_min;
	points_sd_l[3] = points_sd_l[1];
	
	d_sd_l = DrawDatum("SD_l", 
						points_sd_l, 
						LineStyle.DASH,
						undefined,
						undefined,
						undefined,
						undefined,
						undefined,
						Colour.BLUE,
						LineWidth.W3);						
	d_sd_l.line_style = LineStyle.DASH;  // for some reason, this has to be overwritten to be implemented successfully
	
	// 2. Centre
	points_sd_c = new Array(4);
	points_sd_c[0] = points_sd_l[2];
    points_sd_c[1] = points_sd_l[3];
	points_sd_c[2] = OLC_scale_max;
	points_sd_c[3] = (MPDB.OLC_PENALTY * scores_SF) + (SD_max_pen * SD_pct);
	
	d_sd_c = DrawDatum("SD_c", 
						points_sd_c, 
						LineStyle.DASH,
						undefined,
						undefined,
						undefined,
						undefined,
						undefined,
						Colour.BLUE,
						LineWidth.W3);
	d_sd_c.line_style = LineStyle.DASH;
						
	// 3. Right
	points_sd_r = new Array(4);
	points_sd_r[0] = points_sd_c[2];
    points_sd_r[1] = points_sd_c[3];
	points_sd_r[2] = max_OLC+1;
    points_sd_r[3] = points_sd_r[1];
	
	d_sd_r = DrawDatum("SD_r", 
						points_sd_r, 
						LineStyle.DASH,
						undefined,
						undefined,
						undefined,
						undefined,
						undefined,
						Colour.BLUE,
						LineWidth.W3);
	d_sd_r.line_style = LineStyle.DASH;
	
	// 4. Dummies -- for purposes of plotting label
	points_sd_d1 = new Array(4);
    points_sd_d1[0] = points_sd_r[2];
    points_sd_d1[1] = points_sd_r[3];
	points_sd_d1[2] = points_sd_d1[0];
	points_sd_d1[3] = points_sd_d1[1];
	
	// Variable positioning to prevent labels from overlapping
	if (SD_util <= 50) var pos = Datum.LABEL_ABOVE_LEFT;
	else var pos = Datum.LABEL_BELOW_LEFT;
	
	d_sd_d1 = DrawDatum("SD_d1", 
						points_sd_d1, 
						LineStyle.DASH,
						undefined,
						undefined,						
						"SD = "+ SD_util.toFixed(0)+"%",
						Datum.LABEL_18_POINT,
						pos);
						
	// d_sd_d2 = DrawDatum("SD_d2", 
						// points_sd_d2, 
						// LineStyle.DASH,
						// undefined,
						// undefined,
						// "SD = "+ SD_util.toFixed(0)+"%",
						// Datum.LABEL_18_POINT,
						// Datum.LABEL_BELOW_LEFT);
						
	//================================================================
						
	// Draw a piecewise line in three parts showing the transposition if additional penalties 
	// (bottoming out and/or barrier height penalty) occur.
	
	// Condition describing which additional penalties are applied:
	// 1. Bottomed out only
	// 2. Not attenuated only
	// 3. Bottomed out and not attenuated
	var cond = 0;
	var y_trans = 0;  // vertical transformation of curve
	var lbl_str;  // label string
	
	if (bottomed_out == "Yes")
	{
		cond += 1;
		y_trans += MPDB.BOTTOMING_OUT_PENALTY * scores_SF;
		lbl_str = "bottoming out"
	}
	
	if (template_reg === "CNCAP")
	{
		if (not_attenuated === "Yes") 
		{
			cond += 2;
			y_trans += MPDB.NO_ATTENUATION_PENALTY * scores_SF;
			if (cond === 2) lbl_str = "intrusion pen.";
			else lbl_str = lbl_str.concat("+ intrusion pen.");
		}
	}
	
	if (cond > 0)
	{	
		// 1. Left-hand side of line - d_sd_l transformed vertically
		var points_p_l = new Array(4);
		points_p_l[0] = min_OLC-1;  // min x
		points_p_l[1] = (SD_min_pen * SD_pct) + y_trans;  // penalty if OLC=0%, SD=100% + additional penalties
		points_p_l[2] = OLC_scale_min;
		points_p_l[3] = points_p_l[1];
		
		d_adl_l = DrawDatum("addl_pens_left", 
							points_p_l, 
							LineStyle.DASH2,
							undefined,
							undefined,
							undefined,
							undefined,
							undefined,
							Colour.BLUE,
							LineWidth.W5);
		
		// 2. Center of line:
		var points_p_c = new Array(4);
		points_p_c[0] = points_p_l[2];
		points_p_c[1] = points_p_l[3];
		
		// Try vertical transformation of d_sd_c -- if exceeds max, find point where intersects with max
		points_p_c[2] = OLC_scale_max;
		points_p_c[3] = (MPDB.OLC_PENALTY * scores_SF) + (SD_max_pen * SD_pct) + y_trans;	
		
		if (points_p_c[3] > MPDB.MAX_TOTAL_PENALTY)
		{
			// Identify x coordinate by drawing slope (given by SD curve) starting from previous point
			// continuing up to MAX_TOTAL_PENALTY.
			// m = delY / delX --> delX = delY / m --> xf = xi + delY / m
			var xi = points_p_c[0];
			var delY = MPDB.MAX_TOTAL_PENALTY - points_p_c[1];
			var m = ((MPDB.OLC_PENALTY * scores_SF) + (SD_max_pen * SD_pct) - (SD_min_pen * SD_pct)) / OLC_scale_diff;
			
			points_p_c[2] = xi + (delY / m);
			points_p_c[3] = MPDB.MAX_TOTAL_PENALTY;
		}		
		
		d_adl_c = DrawDatum("addl_pens_centre", 
							points_p_c, 
							LineStyle.DASH2,
							undefined,
							undefined,
							undefined,
							undefined,
							undefined,
							Colour.BLUE,
							LineWidth.W5);
		
		// 3. Right-hand side of line: flat at MAX_TOTAL_PENALTY until OLC_SCALE_MAX
		var points_p_r = new Array(4);
		points_p_r[0] = points_p_c[2];
		points_p_r[1] = points_p_c[3];
		points_p_r[2] = max_OLC+1;
		points_p_r[3] = points_p_r[1];
		
		d_adl_r = DrawDatum("addl_pens_right", 
							points_p_r, 
							LineStyle.DASH2,
							undefined,
							undefined,
							undefined,
							undefined,
							undefined,
							Colour.BLUE,
							LineWidth.W5);
	}
	
	// WHY DOESN'T THIS WORK?
	// else  // i.e. SD_(actual)% curve is the curve from which the total penalty value is chosen
	// {
		// d_sd_l.line_width = LineWidth.W5;
		// d_sd_c.line_width = LineWidth.W5;
		// d_sd_r.line_width = LineWidth.W5;
		
		// d_sd_l.style = LineStyle.DASH2;
		// d_sd_c.style = LineStyle.DASH2;
		// d_sd_r.style = LineStyle.DASH2;
	// }
	
	//================================================================
	
	// Draw grey section above uppermost line(s) and assign labels to correct positions
	
	// There is a chance that the the SD_100% curve and the SD_(actual)%+penalties curve cross each other	
	// Check whether SD_100% curve crosses SD_(actual%)+penalties curve	in sloped centre section
	var cond_int = (cond > 0 &&  // i.e. penalties are applied
					points_p_c[1] > points_sd_100_c[1] &&  // i.e. at left-hand side, SD_(actual)%+penalties curve is higher
					points_sd_100_c[3] > points_p_c[3]);   // i.e. at right-hand side, SD_100% curve is higher
					
	if (cond_int)    
	{
		// Find point of intersection
		var m_p   = (points_p_c[3] - points_p_c[1]) / (points_p_c[2] - points_p_c[0]);  // slope
		var m_100 = (points_sd_100_c[3] - points_sd_100_c[1]) / (points_sd_100_c[2] - points_sd_100_c[0]);
		
		var x_int = ((points_sd_100_c[1] - points_p_c[1]) / (m_p - m_100)) + points_p_c[0];
		var y_int = points_p_c[1] + m_p * (x_int - points_p_c[0]);
		
		// Draw piece-wise line in four parts across the top
		// 1. Left -- equivalent to left section of SD_(actual%)+penalties curve
		points_int_l = new Array(4);
		points_int_l[0] = points_p_l[0];
		points_int_l[1] = points_p_l[1];
		points_int_l[2] = points_p_l[2];
		points_int_l[3] = points_p_l[3];
		
		d_int_l = DrawDatum("SD_int_l", 
							points_int_l, 
							LineStyle.DASH2,
							Colour.LIGHT_GREY,
							"above");
							
		// 2. Left of intersection
		points_int_lx = new Array(4);
		points_int_lx[0] = points_int_l[2];
		points_int_lx[1] = points_int_l[3];
		points_int_lx[2] = x_int;
		points_int_lx[3] = y_int;
		
		d_int_lx = DrawDatum("SD_int_lx", 
							points_int_lx, 
							LineStyle.DASH2,
							Colour.LIGHT_GREY,
							"above");
							
		// 3. Right of intersection
		points_int_rx = new Array(4);
		points_int_rx[0] = points_int_lx[2];
		points_int_rx[1] = points_int_lx[3];
		points_int_rx[2] = points_sd_100_r[0];
		points_int_rx[3] = points_sd_100_r[1];
		
		d_int_rx = DrawDatum("SD_int_rx", 
							points_int_rx, 
							LineStyle.DASH2,
							Colour.LIGHT_GREY,
							"above");
							
		// 4. Right -- equivalent to left section of SD_100 curve
		points_int_r = new Array(4);
		points_int_r[0] = points_sd_100_r[0];
		points_int_r[1] = points_sd_100_r[1];
		points_int_r[2] = points_sd_100_r[2];
		points_int_r[3] = points_sd_100_r[3];
		
		d_int_r = DrawDatum("SD_int_r", 
							points_int_r, 
							LineStyle.DASH2,
							Colour.LIGHT_GREY,
							"above");

		// Draw SD_100 label
		d_sd_100_l.label_position = Datum.LABEL_BELOW_RIGHT;
		
		// Create dummy curve on right-hand side and draw SD_(actual%)+penalties label there
		var points_p_d = new Array(4);
		points_p_d[0] = points_p_r[2];
		points_p_d[1] = points_p_r[3];
		points_p_d[2] = points_p_d[0];
		points_p_d[3] = points_p_d[1];
		
		d_adl_d1 = DrawDatum("adl_d1", 
							points_p_d,
							undefined,
							undefined,
							undefined,
							"SD = "+ SD_util.toFixed(0)+"% + "+lbl_str,
							Datum.LABEL_18_POINT,				
							Datum.LABEL_BELOW_LEFT);
	}
	
	else  // i.e. the two curves do not cross, one is uppermost the entire time
	{
		// Identify which curve is uppermost -- the area above this curve should be filled grey
		if (tot_score > OLC_score + range_SD) uppermost = "penalties";
		else uppermost = "SD_100";

		// Shade area in grey and assign labels with variable positioning to prevent labels from clashing/overlapping.
		if (uppermost === "penalties")
		{
			d_adl_l.fill_colour_above = Colour.LIGHT_GREY;
			d_adl_c.fill_colour_above = Colour.LIGHT_GREY;
			d_adl_r.fill_colour_above = Colour.LIGHT_GREY;
			
			// Draw SD_100 label
			d_sd_100_l.label_position = Datum.LABEL_BELOW_RIGHT;
			
			// Create dummy curve on right-hand side and draw SD_(actual%)+penalties label there
			var points_p_d = new Array(4);
			points_p_d[0] = points_p_r[2];
			points_p_d[1] = points_p_r[3];
			points_p_d[2] = points_p_d[0];
			points_p_d[3] = points_p_d[1];
			
			d_adl_d1 = DrawDatum("adl_d1", 
								points_p_d,
								undefined,
								undefined,
								undefined,
								"SD = "+ SD_util.toFixed(0)+"% + "+lbl_str,
								Datum.LABEL_18_POINT,				
								Datum.LABEL_ABOVE_LEFT);
		}
		
		else
		{
			d_sd_100_l.fill_colour_above = Colour.LIGHT_GREY;
			d_sd_100_c.fill_colour_above = Colour.LIGHT_GREY;
			d_sd_100_r.fill_colour_above = Colour.LIGHT_GREY;
			
			// Draw SD_100 label
			d_sd_100_l.label_position = Datum.LABEL_ABOVE_RIGHT;
			
			if (cond > 0)  // i.e. penalties are applied
			{
				// Draw SD_(actual%)+penalties label from left-hand side
				d_adl_l.label = "SD = "+ SD_util.toFixed(0)+"% + "+lbl_str;
				d_adl_l.label_size = Datum.LABEL_18_POINT;
				d_adl_l.label_position = Datum.LABEL_BELOW_RIGHT;
			}
			
			// else - label has already been applied to d_sd_d1
		}
	}	
	
	//================================================================
    
    var graph = Graph.GetFromID(1);
    graph.y_unit_decimals = 1;
    graph.xlabel = "OLC (g)";
    graph.ylabel = "Penalty points";
    graph.title = "";   
   
    // Create en error message picture of the scoring and assign the appropriate variable values if the 
    // intrusion displacement list "MPDB_barrier_def.csv" is missing
	var output_data = new OutputData("", images_dir + "/" + "Compatibility_modifier_scoring");
	
    if (SD == 0.0 && bottomed_out == "N/A")
    {
        write_variable(f_vars, "COMPATIBILITY_TOTAL_SCORE",         "N/A", "Compatibility modifier total score",      "String");
        write_variable(f_vars, "COMPATIBILITY_BOTTOMING_OUT_SCORE", "N/A", "Bottoming out score",                     "String");
        write_variable(f_vars, "COMPATIBILITY_SD_SCORE",            "N/A", "Compatibility SD score",                  "String");
        write_variable(f_vars, "COMPATIBILITY_SD_PCT",              "N/A", "Compatibility SD percentage utilization", "String");

        blank_image_msg = "No intrusion displacement list defined on "+ images_dir + "/" + "/MPDB_barrier_def.csv";
        create_blank_image(blank_image_msg, output_data.fname);

        return;
    }
    else
    {
		if (template_reg === "CNCAP") var figs = 3;
		else var figs = 2;
        write_variable(f_vars, "COMPATIBILITY_TOTAL_SCORE",         tot_score.toFixed(figs),    "Compatibility modifier total score",      "String");
        write_variable(f_vars, "COMPATIBILITY_BOTTOMING_OUT_SCORE", bottoming_score.toFixed(1), "Bottoming out score",                     "String");
        write_variable(f_vars, "COMPATIBILITY_SD_SCORE",            SD_score.toFixed(3),        "Compatibility SD score",                  "String");
        write_variable(f_vars, "COMPATIBILITY_SD_PCT",              SD_util.toFixed(0),         "Compatibility SD percentage utilization", "String");
    
        write_image(output_data.title, output_data.fname, [mod, mod_domain], MPDB.MAX_TOTAL_PENALTY);
    }
}


function readIntrusionCSV(f, num_rows, num_columns)
// Reads CSV file <f> containing information about intrusions. Return array of size <num_rows> * <num_columns> with values.
{
	// Read MPDB_barrier_def.csv
	var f_csv = new File(f, File.READ);
	var line = f_csv.ReadLine();
	var line = f_csv.ReadLine();
	
	var intrusions = new Array(num_rows * num_columns);
	var i = 0;

	while ((line = f_csv.ReadLine())!== undefined && i < intrusions.length)  // read data from .csv into array
	{
		intrusions[i] = parseFloat(line);
		i++;
	}

	f_csv.Close()
	
	return intrusions;
}


function getZ(i)
// Get height to centre of square i from bottom of barrier (i=0 is at top of barrier)
// i.e. scan down rows
// NOTE: THIS IS NOT GENERALIZED FOR CASES WHERE THERE ARE OTHER THAN 1400 POINTS ASSESSED LIKE IN Euro NCAP AND C-NCAP
{
	nr = parseFloat(MPDB.NUM_ROWS)
    return (nr*20 - 10) - (Math.floor(i%nr*20));
}


function getY(i)
// Get distance to centre of center of square i from edge of barrier
// Counts up using RHS drive convention, i.e. for all values of i that return getY(i) < 200,
// checkAssessmentArea() will always find them outside assessment area.
// NOTE: THIS IS NOT GENERALIZED FOR CASES WHERE THERE ARE OTHER THAN 1400 POINTS ASSESSED LIKE IN Euro NCAP AND C-NCAP
{
	nr = parseFloat(MPDB.NUM_ROWS)
    return Math.floor(  (i/nr+1))*20 - 10;
}


function checkAssessmentArea(y, z, driver_side, imp_width)
// NOTE: THIS IS NOT GENERALIZED FOR CASES WHERE THERE ARE OTHER THAN 1400 POINTS ASSESSED LIKE IN Euro NCAP AND C-NCAP
{
	var flags = {};
    flags.assess_flag = false;  // checks whether is within assessment area
	flags.intrusion_height_flag = false;  // checks whether is above assessment area (USED FOR C-NCAP BARRIER HEIGHT INTRUSION CHECK ONLY)
    
	if (100 < z)
	{
		if (y < 200)  // i.e. near the edge of barrier
		{
			if (driver_side == "right")  // too close, on side of fixed edge of assessment area
			{
				flags.assess_flag           = false;
				flags.intrusion_height_flag = false;
			} 
			else 
			{
				if((imp_width + y - 1010) < 0)  // width of barrier is 50 cols*20mm = 1000mm
				{
					flags.assess_flag           = false;  // too far, on side of variable edge of assessment area
					flags.intrusion_height_flag = false;
				}
				else  // i.e. in appropriate lateral position
				{
					if (z < 500)  // i.e. height is within assessment area
					{
						flags.assess_flag           = true;
						flags.intrusion_height_flag = false;
					}
					else  // height puts it above assessment area, i.e. in 'intrustion barrier height' area
					{
						flags.assess_flag           = false;
						flags.intrusion_height_flag = true;
					}
				}
			}
		}

		else if (200 < y && y < 800)  // i.e. towards the centre of the barrier
		{
			if (driver_side == "right")  
			{
				if((imp_width - y - 10) < 0)  // too far, on side of variable edge of assessment area
				{
					flags.assess_flag           = false;
					flags.intrusion_height_flag = false; 
				}
				else  // i.e. appropriate lateral position
				{
					if (z < 500)  // i.e. height is within assessment area
						{
							flags.assess_flag           = true;
							flags.intrusion_height_flag = false;
						}
					else  // height puts it above assessment area, i.e. in 'intrustion barrier height' area
					{
						flags.assess_flag           = false;
						flags.intrusion_height_flag = true;
					}
				}
			}
			else 
			{
				if((imp_width + y - 1010) < 0)  // too far, on side of variable edge of assessment area
				{
					flags.assess_flag           = false;
					flags.intrusion_height_flag = false; 
				}
				else  // i.e. appropriate lateral position
				{
					if (z < 500)  // i.e. height is within assessment area
					{
						flags.assess_flag           = true;
						flags.intrusion_height_flag = false;
					}
					else  // height puts it above assessment area, i.e. in 'intrustion barrier height' area
					{
						flags.assess_flag           = false;
						flags.intrusion_height_flag = true;
					}
				} 
			} 
		}
		else // if y > 800
		{
			if (driver_side == "left")  // too close, on side of fixed edge of assessment area
			{
				flags.assess_flag           = false;
				flags.intrusion_height_flag = false;
			}
			else 
			{
				if((imp_width - y - 10) < 0)  // too far, on side of variable edge of assessment area
				{
					flags.assess_flag           = false;
					flags.intrusion_height_flag = false;
				}
				else  // i.e. appropriate lateral position
				{
					if (z < 500)  // i.e. height is within assessment area
					{
						flags.assess_flag           = true;
						flags.intrusion_height_flag = false;
					}
					else  // height puts it above assessment area, i.e. in 'intrustion barrier height' area
					{
						flags.assess_flag           = false;
						flags.intrusion_height_flag = true;
					}
				}
			} 
		}
	}
			
    return flags;
	
}


function StandardDeviation(intrusion, average, assessment_area_points, driver_side, imp_width)
{
    var sd_value = 0;
    var int_max = MPDB.SD_MAX_INTRUSION;
    
    for (var i = 0; i < intrusion.length; i++)
    {
		var y = getY(i);
        var z = getZ(i);        

        if ( checkAssessmentArea(y, z, driver_side, imp_width).assess_flag )
        {
            sd_value += Math.pow((Math.min(intrusion[i], int_max)) - average, 2);
            
        }
    }
    
    sd_value /= assessment_area_points;
    
    sd_value = Math.sqrt(sd_value);
    
    return sd_value
    
}

//============================
// IMPACT ASSESSMENT FUNCTIONS
//============================

function do_head_rating_calc(f)
{
    // Calculates the head scores for both driver and passenger and writes
    // relevant values into the Reporter csv variables file <f>.

    // Capping limit for all passengers
    var capping_limit_total = "FALSE";
    
    // Two passes, one for driver, one for passenger
    // should be four passengers in C-NCAP and EuroNCAP codes 2021; however, now only considers the front two passengers.
    for(var pass = 0; pass<2; pass++)
    {
        var node;
        var node_y;
        var node_z;
        var dummy;
        var capping_limit = "FALSE"; // for individual (Driver OR Passenger)

        if(pass == 0)
        {
            node = oGlblData.driver_head_node;
            node_y = oGlblData.driver_head_node_y;
            node_z = oGlblData.driver_head_node_z;
            dummy = "Driver";
        }

        if(pass == 1)
        {
            node = oGlblData.passenger_head_node;
            node_y = oGlblData.passenger_head_node_y;
            node_z = oGlblData.passenger_head_node_z;
            dummy = "Passenger";
        }

        var output_data = {};

        output_data.acc = new OutputData(dummy + " Head Acceleration",         images_dir + "/" + dummy + "_Head_Acceleration");
        output_data.int = new OutputData(dummy + " Steering Column Intrusion", images_dir + "/" + dummy + "_Steering_Column_Intrusion");

        if(node != undefined)
        { 
            // Read in acceleration X, Y and Z
            var one_node = true;

            // If nodes for three accelerometer positions defined, use all three, otherwise, read all components from same node
            if(node_y != undefined && node_z != undefined && node_y != 0 && node_z != 0)
            {
                var c_acc = read_xyz_accelerations(1, node, node_y, node_z);
                one_node = false;
            }
            else
            {
                var c_acc = read_xyz_accelerations(1, node);
            }

            // Check that the curves read in - i.e. the node id and component are valid
            // Create a blank image to let the user know.
            
            // Generate a relevant message for the blank image depending on which acceleration data are missing
            if(!c_acc.ax || !c_acc.ay || !c_acc.az)
            {
                var blank_image_msg;
                // If only one node supplied, just report that node ID
                if(one_node) blank_image_msg = "NO ACCEL DATA (NODE " + node + ") ";
                // else report each of the node IDs that were invalid
                else
                {
                    var acc_comp_lst = [];
                    var acc_id_lst = [];
                    if(!c_acc.ax)
                    {
                        acc_comp_lst.push(" X");
                        acc_id_lst.push(" "+node);
                    }
                    if(!c_acc.ay)
                    {
                        acc_comp_lst.push(" Y");
                        acc_id_lst.push(" "+node_y);
                    }
                    if(!c_acc.az)
                    {
                        acc_comp_lst.push(" Z");
                        acc_id_lst.push(" "+node_z);
                    }
                    blank_image_msg = "NO" + acc_comp_lst + " ACCEL DATA (NODE(s) " + acc_id_lst + ") ";
                }
                create_blank_image(blank_image_msg + output_data.acc.title, output_data.acc.fname);

                if(pass == 0)
                {
                    create_blank_image(blank_image_msg + output_data.int.title, output_data.int.fname);
                }
            }
            else
            {
                // Convert to g from model units
                var c_g = convert_xyz_acc_to_g(c_acc);

                // C1000 filter (convert to seconds first)
                var c_gx_c1000 = convert_to_s_and_filter(c_g.gx, Operate.C1000, MPDB.REG_DT);
                var c_gy_c1000 = convert_to_s_and_filter(c_g.gy, Operate.C1000, MPDB.REG_DT);
                var c_gz_c1000 = convert_to_s_and_filter(c_g.gz, Operate.C1000, MPDB.REG_DT);

                // Vector combine
                var c_vec_g = Operate.Vec(c_gx_c1000, c_gy_c1000, c_gz_c1000);

                // Remove all curves and datums
                remove_all_curves_and_datums();

                // Convert from seconds back to model time
                c_vec_g = Operate.Mux(c_vec_g, oGlblData.time_factor);

                // Get peak acceleration (in g)
                DialogueInput("/AU");
                var peak_accn = c_vec_g.ymax;

                // Get HIC value
                var hic = Operate.Hic(c_vec_g, MPDB.HIC_WINDOW, 1.0);

                // Get 3ms exceedance
                Operate.Tms(c_vec_g, MPDB.TMS_PERIOD*oGlblData.time_factor);
                var tms = c_vec_g.tms;

                // Set label and style
                set_labels(c_vec_g, dummy + " Head Acceleration Magnitude", "Time (" + oGlblData.unit_time + ")", "Acceleration (g)");
                set_line_style(c_vec_g, Colour.BLACK);

                // Draw limit curves => up to final time
                var p = c_vec_g.GetPoint(c_vec_g.npoints);
                switch (template_reg)
                {
                    case "EuroNCAP":
                        if((pass == 0 && oGlblData.steering_wheel_airbag == "YES") || pass == 1) draw_head_limits(p[0], output_data.acc, true);   // Normal limits
                        else                                                                     draw_head_limits(p[0], output_data.acc, false);  // Limits for driver with no airbag
                        break;

                    case "CNCAP":
                        draw_head_limits(p[0], output_data.acc, false);  // C-NCAP doesn't have the soft/hard contact distinction so draw limits as if no airbag
                        break;

                    default:
                        ErrorMessage("Unexpected template regulation in do_head_rating_calc.");
                        return;
                        break; 
                }

                // Create image and curve files
                write_image(output_data.acc.title, output_data.acc.fname);

                output_data.acc.curveList.push(c_vec_g.id);
                write_curve("cur", output_data.acc.curveList, output_data.acc.fname);
                write_curve("csv", output_data.acc.curveList, output_data.acc.fname);

                // Remove all curves and datums
                remove_all_curves_and_datums();

                // Steering wheel displacements - driver only
                if(pass == 0)
                {
                    var head_steering_wheel_fa_disp   = 0.0;
                    var head_steering_wheel_lat_disp  = 0.0;
                    var head_steering_wheel_vert_disp = 0.0;

                    var head_steering_wheel_fa_mod   = 0.0;
                    var head_steering_wheel_lat_mod  = 0.0;
                    var head_steering_wheel_vert_mod = 0.0;

                    var spring;

                    for(var sp=0; sp<3; sp++)
                    {
                        if(sp == 0) spring = oGlblData.steering_col_intrusion_spring_x; // Fore-aft spring
                        else if(sp == 1) spring = oGlblData.steering_col_intrusion_spring_y; // Lateral spring
                        else if(sp == 2) spring = oGlblData.steering_col_intrusion_spring_z; // Vertical spring

                        if(spring != undefined)
                        {
                            var c_el = read_data("SPRING TR", 0, spring, "ET");

                            if(c_el)
                            {
                                // Convert to mm, take final value and calculate modifier
                                var c_el_mm = Operate.Div(c_el, oGlblData.mm_factor);

                                var c_fa;
                                var c_lat;
                                var c_vert;

                                if(sp == 0)
                                {
                                    // Fore/aft intrusion = -ve elongation is rearward so multiply by -1.0 first
                                    c_fa = Operate.Mul(c_el_mm, -1);

                                    var p = c_fa.GetPoint(c_fa.npoints);
                                    head_steering_wheel_fa_disp = p[1];
                                    head_steering_wheel_fa_mod = sliding_scale(head_steering_wheel_fa_disp, 
                                            MPDB.HEAD_FA_DISP_LO_LIMIT, MPDB.HEAD_FA_DISP_HI_LIMIT,
                                            MPDB.STEERING_WHEEL_HI_SCORE, MPDB.STEERING_WHEEL_LO_SCORE);
                                }
                                else if(sp == 1)
                                {
                                    // Lateral intrusion - use absolute value (left or right)
                                    c_lat = c_el_mm;

                                    var p = c_lat.GetPoint(c_lat.npoints);
                                    head_steering_wheel_lat_disp = p[1];
                                    head_steering_wheel_lat_mod = sliding_scale(Math.abs(head_steering_wheel_lat_disp), 
                                            MPDB.HEAD_LAT_DISP_LO_LIMIT, MPDB.HEAD_LAT_DISP_HI_LIMIT,
                                            MPDB.STEERING_WHEEL_HI_SCORE, MPDB.STEERING_WHEEL_LO_SCORE);
                                }
                                else if(sp == 2)
                                {
                                    // Vertical intrusion - +ve elongation is upward
                                    c_vert = c_el_mm;

                                    var p = c_vert.GetPoint(c_vert.npoints);
                                    head_steering_wheel_vert_disp = p[1];
                                    head_steering_wheel_vert_mod = sliding_scale(head_steering_wheel_vert_disp, 
                                            MPDB.HEAD_VERT_DISP_LO_LIMIT, MPDB.HEAD_VERT_DISP_HI_LIMIT,
                                            MPDB.STEERING_WHEEL_HI_SCORE, MPDB.STEERING_WHEEL_LO_SCORE);
                                }
                            }
                        }
                    }

                    // Overall steering wheel modifier - minimum of three modifiers
                    var head_steering_wheel_mod = Math.min(head_steering_wheel_fa_mod, head_steering_wheel_lat_mod, head_steering_wheel_vert_mod);

                    // Draw intrusion curves

                    // Remove all curves and datums
                    remove_all_curves_and_datums();

                    // Set labels and styles
                    if(c_fa)   set_labels(c_fa,   dummy + " Fore/Aft Intrusion (+ve rearward)", "Time (" + oGlblData.unit_time + ")", "Intrusion (mm)");
                    if(c_lat)  set_labels(c_lat,  dummy + " Lateral Intrusion",                 "Time (" + oGlblData.unit_time + ")", "Intrusion (mm)");
                    if(c_vert) set_labels(c_vert, dummy + " Vertical Intrusion (+ve rearward)", "Time (" + oGlblData.unit_time + ")", "Intrusion (mm)");

                    if(c_fa)   set_line_style(c_fa,   Colour.RED);
                    if(c_lat)  set_line_style(c_lat,  Colour.BLUE);
                    if(c_vert) set_line_style(c_vert, Colour.GREEN);

                    // Create image and curve files
                    if(c_fa)   c_fa.AddToGraph();
                    if(c_lat)  c_lat.AddToGraph();
                    if(c_vert) c_vert.AddToGraph();

                    if(c_fa)   output_data.int.curveList.push(c_fa.id);
                    if(c_lat)  output_data.int.curveList.push(c_lat.id);
                    if(c_vert) output_data.int.curveList.push(c_vert.id);

                    var title;
                    if(c_fa && c_lat && c_vert) title = output_data.int.title;
                    else                        title = "SOME SPRING IDs MISSING/DO NOT EXIST FOR " + output_data.int.title;

                    write_image(title, output_data.int.fname);
                    write_curve("cur", output_data.int.curveList, output_data.int.fname);
                    write_curve("csv", output_data.int.curveList, output_data.int.fname);
                }

                // Calculate head points

                // Driver with steering wheel airbag or passenger
                var hic_score       = 0.0;
                var tms_score       = 0.0;
                var score           = 0.0;

                // EuroNCAP should consider the peak_accn and steering wheel airbag being existing or not; 
                // but CNCAP doesn't consider the two
                switch (template_reg)
                {
                    case "EuroNCAP":
                        var peak_accn_score = 0.0;

                        if(pass == 0)
                        {       
                            if(oGlblData.steering_wheel_airbag == "YES")
                            {
                                if(peak_accn < MPDB.PEAK_HEAD_ACCN_SOFT)
                                {
                                    // 'Soft contact' -> 4.0
                                    peak_accn_score = score = MPDB.HEAD_HI_SCORE;
                                    write_variable(f, dummy.toUpperCase() + "_PEAK_HEAD_ACCN_SCORE", peak_accn_score.toFixed(3), "Peak head acceleration score", "String");
                                }
                                else
                                {
                                    hic_score = sliding_scale(hic, MPDB.HIC_LO_LIMIT, MPDB.HIC_HI_LIMIT, MPDB.HEAD_HI_SCORE, MPDB.HEAD_LO_SCORE);
                                    tms_score = sliding_scale(tms, MPDB.TMS_LO_LIMIT, MPDB.TMS_HI_LIMIT, MPDB.HEAD_HI_SCORE, MPDB.HEAD_LO_SCORE);
                
                                    score = Math.min(hic_score, tms_score);
                                    write_variable(f, dummy.toUpperCase() + "_PEAK_HEAD_ACCN_SCORE", "", "Peak head acceleration score is not considered in hard contact scenario", "String");
                
                                    // Capping limit should be applied to overall score if these are zero
                                    if(hic_score == 0.0)
                                    {
                                        capping_limit = "TRUE";
                                        write_variable(f, dummy.toUpperCase() + "_HEAD_HIC15_CAPPING_LIMIT", "*", dummy + " HIC15 capping limit", "String");
                                    }
                                    if(tms_score == 0.0)
                                    {
                                        capping_limit = "TRUE";
                                        write_variable(f, dummy.toUpperCase() + "_HEAD_3MS_CAPPING_LIMIT", "*", dummy + " ACC 3ms exceedance capping limit", "String");
                                    }
                                    if(capping_limit == "TRUE")
                                    {
                                        write_variable(f, dummy.toUpperCase() + "_HEAD_CAPPING_LIMIT", "*", dummy + " head capping limit", "String");
                                        write_variable(f, dummy.toUpperCase() + "_HEAD_NECK_CAPPING_LIMIT", "*", dummy + " head and neck capping limit", "String");
                                    }
                                }
                            }
                            else
                            {
                                // If no steering wheel airbag is fitted, 
                                // the driver will be awarded 0 points for the head and neck.        
                                peak_accn_score = 0.0;
                                tms_score = 0.0;
                                score = 0.0;
    
                                write_variable(f, dummy.toUpperCase() + "_PEAK_HEAD_ACCN_SCORE", peak_accn_score.toFixed(3), "Peak head acceleration score", "String");                                
                                // Capping limit should be applied to overall score if these are zero        
                                capping_limit = "TRUE";
                                write_variable(f, dummy.toUpperCase() + "_PEAK_HEAD_ACCN_CAPPING_LIMIT", "*", dummy + " ACCN capping limit", "String");
                                write_variable(f, dummy.toUpperCase() + "_HEAD_CAPPING_LIMIT", "*", dummy + " head capping limit", "String");
                                write_variable(f, dummy.toUpperCase() + "_HEAD_NECK_CAPPING_LIMIT", "*", dummy + " head and neck capping limit", "String");
                                // in which case the HIC and 3ms limit should also be 0, show capping limit here
                                write_variable(f, dummy.toUpperCase() + "_HEAD_HIC15_CAPPING_LIMIT", "*", dummy + " HIC15 capping limit", "String");
                                write_variable(f, dummy.toUpperCase() + "_HEAD_3MS_CAPPING_LIMIT", "*", dummy + " ACC 3ms exceedance capping limit", "String");
                            }
                        }
                        else if (pass == 1)
                        {
                            hic_score = sliding_scale(hic, MPDB.HIC_LO_LIMIT, MPDB.HIC_HI_LIMIT, MPDB.HEAD_HI_SCORE, MPDB.HEAD_LO_SCORE);
                            tms_score = sliding_scale(tms, MPDB.TMS_LO_LIMIT, MPDB.TMS_HI_LIMIT, MPDB.HEAD_HI_SCORE, MPDB.HEAD_LO_SCORE);

                            score = Math.min(hic_score, tms_score);
        
                            // Capping limit should be applied to overall score if these are zero        
                            if(hic_score == 0.0)
                            {
                                capping_limit = "TRUE";
                                write_variable(f, dummy.toUpperCase() + "_HEAD_HIC15_CAPPING_LIMIT", "*", dummy + " HIC15 capping limit", "String");
                            }
                            if(tms_score == 0.0)
                            {
                                capping_limit = "TRUE";
                                write_variable(f, dummy.toUpperCase() + "_HEAD_3MS_CAPPING_LIMIT", "*", dummy + " ACC 3ms exceedance capping limit", "String");
                            }
                            if(capping_limit == "TRUE")
                            {
                                write_variable(f, dummy.toUpperCase() + "_HEAD_CAPPING_LIMIT", "*", dummy + " head capping limit", "String");
                                write_variable(f, dummy.toUpperCase() + "_HEAD_NECK_CAPPING_LIMIT", "*", dummy + " head and neck capping limit", "String");
                            }
                        }
                        break;

                    case "CNCAP":
                        // C-NCAP does not have the airbag, hard/soft contact considerations
                        // Head score is the minimum of the hic and tms scores 
                        hic_score = sliding_scale(hic, MPDB.HIC_LO_LIMIT, MPDB.HIC_HI_LIMIT, MPDB.HEAD_HI_SCORE, MPDB.HEAD_LO_SCORE);
                        tms_score = sliding_scale(tms, MPDB.TMS_LO_LIMIT, MPDB.TMS_HI_LIMIT, MPDB.HEAD_HI_SCORE, MPDB.HEAD_LO_SCORE);

                        score = Math.min(hic_score, tms_score);
                        write_variable(f, dummy.toUpperCase() + "_PEAK_HEAD_ACCN_SCORE", "", "Peak head acceleration score is not used in C-NCAP", "String");

                        // Capping limit should be applied to overall score if these are zero
                        if(hic_score == 0.0)
                        {
                            capping_limit = "TRUE";
                            write_variable(f, dummy.toUpperCase() + "_HEAD_HIC15_CAPPING_LIMIT", "*", dummy + " HIC15 capping limit", "String");
                        }
                        if(tms_score == 0.0)
                        {
                            capping_limit = "TRUE";
                            write_variable(f, dummy.toUpperCase() + "_HEAD_3MS_CAPPING_LIMIT", "*", dummy + " ACC 3ms exceedance capping limit", "String");
                        }
                        if(capping_limit == "TRUE")
                        {
                            write_variable(f, dummy.toUpperCase() + "_HEAD_CAPPING_LIMIT", "*", dummy + " head capping limit", "String");
                            write_variable(f, dummy.toUpperCase() + "_HEAD_NECK_CAPPING_LIMIT", "*", dummy + " head and neck capping limit", "String");
                        }
                        break;

                    default:
                        ErrorMessage("Calculating HIC and TMS score has errors");
                        return;
                        break; 
                } 
                
                // Add on steering wheel modifier to get final score (before applying subjective modifiers, this
                // is done in a script in the Reporter template).

                if(pass == 0) score += head_steering_wheel_mod;

                // Bound score between limits
                if((pass == 0 && oGlblData.steering_wheel_airbag == "YES") || pass == 1)
                {
                    if(score < MPDB.HEAD_LO_SCORE) score = MPDB.HEAD_LO_SCORE;
                    if(score > MPDB.HEAD_HI_SCORE) score = MPDB.HEAD_HI_SCORE;
                }

                // Write variables to csv file for Reporter
                if(pass == 0)
                {
                    // Driver steering wheel airbag
                    var value;
                    if(oGlblData.steering_wheel_airbag == "YES") value = "Yes";
                    else                                         value = "No";

                    write_variable(f, "DRIVER_STEERING_WHEEL_AIRBAG", value, "Driver steering wheel airbag present", "String");
                }

                // Peak head acceleration (score already written out above)
                write_variable(f, dummy.toUpperCase() + "_PEAK_HEAD_ACCN",       peak_accn.toFixed(2),       "Peak head acceleration value", "String");
                    
                // HIC value and score

                if(MPDB.HIC_WINDOW == 0.036)
                {
                    write_variable(f, dummy.toUpperCase() + "_HEAD_HIC36",       hic.toFixed(2),       "HIC36 value", "String");
                    write_variable(f, dummy.toUpperCase() + "_HEAD_HIC36_SCORE", hic_score.toFixed(2), "HIC36 score", "String");
                }
                else
                {
                    write_variable(f, dummy.toUpperCase() + "_HEAD_HIC15",       hic.toFixed(2),       "HIC15 value", "String");
                    write_variable(f, dummy.toUpperCase() + "_HEAD_HIC15_SCORE", hic_score.toFixed(2), "HIC15 score", "String");
                }

                // 3ms value and score
                write_variable(f, dummy.toUpperCase() + "_HEAD_3MS",       tms.toFixed(2),       "3ms value", "String");
                write_variable(f, dummy.toUpperCase() + "_HEAD_3MS_SCORE", tms_score.toFixed(3), "3ms score", "String");

                // Steering wheel displacement values and modifier value
                if(pass == 0)
                {
                    write_variable(f, "DRIVER_STEERING_WHEEL_FA_DISP",   head_steering_wheel_fa_disp.toFixed(1),   "Steering wheel fore/aft displacement", "String");
                    write_variable(f, "DRIVER_STEERING_WHEEL_VERT_DISP", head_steering_wheel_vert_disp.toFixed(1), "Steering wheel vertical displacement", "String");
                    write_variable(f, "DRIVER_STEERING_WHEEL_LAT_DISP",  head_steering_wheel_lat_disp.toFixed(1),  "Steering wheel lateral displacement",  "String");
                    write_variable(f, "DRIVER_STEERING_WHEEL_DISP_MOD",  head_steering_wheel_mod.toFixed(1),       "Steering wheel displacement modifier", "String");
                }

                // Final score value (before subjective modifiers) - write a variable and store on <oGlblData> object
                write_variable(f, dummy.toUpperCase() + "_HEAD_PRE_MODS_SCORE",   score.toFixed(3),   "Final score", "String");
            }
        }
        else
        {
            // No node defined - variables should be set to blank by first script in template.
            // Create a blank image to let the user know.
            create_blank_image("NO NODE ID DEFINED FOR " + output_data.acc.title, output_data.acc.fname);

            if(pass == 0)
            {
                create_blank_image("NO NODE ID DEFINED FOR " + output_data.int.title, output_data.int.fname);
            }
        }
        
        // Check Capping limit
        if(capping_limit == "TRUE") capping_limit_total = "TRUE";
    }
    // Should capping limit be applied
    write_variable(f, "HEAD_CAPPING_LIMIT", capping_limit_total, "Head capping limit", "String");   
}

function do_neck_rating_calc(f)
{
    // Calculates the head scores for both driver and passenger and writes
    // relevant values into the Reporter csv variables file <f>.
    
    // Capping limit for all passengers
    var capping_limit_total = "FALSE";
    
    // Two passes, one for driver, one for passenger
    Message("Doing neck rating calculation...");
    var dummy = [];
    dummy.push(new DummyData("Driver", oGlblData.driver_neck_loadcell, "beam", oGlblData.driver_dummy, "version"));
    dummy.push(new DummyData("Passenger", oGlblData.passenger_neck_loadcell, "beam", oGlblData.passenger_dummy, "version"));

    for(var i = 0; i<dummy.length; i++)
    {
        var capping_limit = "FALSE"; // for individual (Driver OR Passenger)
        
        // Data for images that we will create
        var output_data = {};

        // For Driver/Passenger of EuroNCAP and CNCAP, only Passenger of EuroNCAP uses exceedance for neck shear and tension
        if (dummy[i].chart_label == "Driver" || template_reg == "CNCAP") // Driver or CNCAP Not exceedance
        {
            output_data.shr = new OutputData(dummy[i].chart_label + " Neck Shear",   images_dir + "/" + dummy[i].variable_label + "_Neck_Shear");
            output_data.ten = new OutputData(dummy[i].chart_label + " Neck Tension", images_dir + "/" + dummy[i].variable_label + "_Neck_Tension");
        }
        else if (dummy[i].chart_label == "Passenger" && template_reg == "EuroNCAP") // Passenger of EuroNCAP with exceedance
        {
            output_data.shr_exc = new OutputData(dummy[i].chart_label + " Neck Shear exceedance",   images_dir + "/" + dummy[i].variable_label + "_Neck_Shear");
            output_data.ten_exc = new OutputData(dummy[i].chart_label + " Neck Tension exceedance", images_dir + "/" + dummy[i].variable_label + "_Neck_Tension");    
        }
        output_data.mom     = new OutputData(dummy[i].chart_label + " Neck Bending Moment",     images_dir + "/" + dummy[i].variable_label + "_Neck_Moment");

        if(dummy[i].beam_id != undefined)
        {
            // Read in beam shear, tension and  MY
            switch (dummy[i].version)
            {
                case "V6_HUMANETICS_HIII_50TH_MALE":
                case "V7_HUMANETICS_HIII_50TH_MALE":
                case "V8_HUMANETICS_HIII_50TH_MALE":
                case "Humanetics HIII 50M v1.5":
                case "Humanetics HIII 50M v1.5.1":
                    var c_shr = read_data("BEAM BASIC", 0, dummy[i].beam_id, "NX");  // NX is shear
                    var c_ten = read_data("BEAM BASIC", 0, dummy[i].beam_id, "NZ");
                    var c_mom = read_data("BEAM BASIC", 0, dummy[i].beam_id, "MY");
                    var entity_str = "BEAM"; // Used for error message below
                    break;

                case "LSTC HIII 50M Detailed v190217_beta":
                case "LSTC HIII 50M Fast V2.0":
                case "LSTC HIII 50M v130528_beta":
                case "LSTC HIII 5F v2.0":
                case "Humanetics HIII 5F v2.02":
                case "Humanetics THOR 50M v1.8":
                case "Humanetics THOR 50M v1.9":
                    var c_shr = read_data("SECTION", 0, dummy[i].beam_id, "FX");  // FX is shear
                    var c_ten = read_data("SECTION", 0, dummy[i].beam_id, "FZ");
                    var c_mom = read_data("SECTION", 0, dummy[i].beam_id, "MY");
                    var entity_str = "SECTION"; // Used for error message below
                    break;

                default:
                    ErrorMessage("Unsupported dummy \""+dummy[i].version+"\" in do_neck_rating_calc.");
                    write_blank_images(output_data, "UNSUPPORTED DUMMY \""+dummy[i].version+"\"");
                    return;
                    break;
            }

            // Check that the curves read in - i.e. the beam id and component are valid
            // Create blank images to let the user know.

            if(!c_shr || !c_ten || !c_mom)
            {
               write_blank_images(output_data, "NO DATA FOR " + entity_str + " " + dummy[i].beam_id);
            }
            
            else
            {
                // Convert forces to kN and
                // Convert moment to Nm from model units
                var c_shr_kn = Operate.Div(c_shr, oGlblData.kn_factor);
                var c_ten_kn = Operate.Div(c_ten, oGlblData.kn_factor);
                var c_mom_nm = Operate.Div(c_mom, oGlblData.nm_factor);

                // C1000/C600 filter (convert to seconds first)
                var c_shr_c1000 = convert_to_s_and_filter(c_shr_kn, Operate.C1000, MPDB.REG_DT);
                var c_ten_c1000 = convert_to_s_and_filter(c_ten_kn, Operate.C1000, MPDB.REG_DT);
                var c_mom_c600  = convert_to_s_and_filter(c_mom_nm, Operate.C600,  MPDB.REG_DT);

                // do exceedance for neck shear and tension in case Passenger of EuroNCAP
                if (dummy[i].chart_label == "Passenger" && template_reg == "EuroNCAP") // Passenger of EuroNCAP
                {
                    // exceedance curves for shear and tension. For shear take absolute values in case -ve Y values.
                    var c_shr_abs = Operate.Abs(c_shr_c1000);
                    var c_shr_exc = Operate.Exc(c_shr_abs, "positive");
                    var c_ten_exc = Operate.Exc(c_ten_c1000, "positive");

                    var c_shr_exc_abs = Operate.Abs(c_shr_exc);
                }
    
                if (dummy[i].chart_label == "Passenger") // Passenger uses HIII Dummy
                {
                    // Transpose the bending moment to the occupital condyle moment
                    var c_ten_adj = Operate.Mul(c_shr_c1000, MPDB.NECK_SHR_ADJ_FAC); 
                    var c_mom_adj = Operate.Sub(c_mom_c600, c_ten_adj);
                }
                else if (dummy[i].chart_label == "Driver") // Driver uses THOR Dummy, read the moment curve directly
                {
                    var c_mom_adj = c_mom_c600;
                }

                // Remove all curves and datums
                remove_all_curves_and_datums();

                // Convert from seconds back to model time
                if (dummy[i].chart_label == "Passenger" && template_reg == "EuroNCAP") // Passenger of EuroNCAP
                {
                    c_shr_exc_abs = Operate.Mux(c_shr_exc_abs, oGlblData.time_factor);
                    c_ten_exc     = Operate.Mux(c_ten_exc,     oGlblData.time_factor);
                    
                    // Set labels and style
                    set_labels(c_shr_exc_abs, dummy[i].chart_label + " Neck Shear exceedance",         "Cumulative Time (" + oGlblData.unit_time + ")", "Force (kN)");
                    set_labels(c_ten_exc,     dummy[i].chart_label + " Neck Tension exceedance",       "Cumulative Time (" + oGlblData.unit_time + ")", "Force (kN)");
                    set_line_style(c_shr_exc_abs, Colour.BLACK);
                    set_line_style(c_ten_exc,     Colour.BLACK);
                }
                else if(dummy[i].chart_label == "Driver" || template_reg == "CNCAP") // Driver or CNCAP
                {
                    c_shr = Operate.Mux(c_shr_c1000, oGlblData.time_factor);
                    // change values from negative to positive for shear curves
                    c_shr = Operate.Mul(c_shr_c1000, -1);
                    c_ten = Operate.Mux(c_ten_c1000, oGlblData.time_factor);
            
                    // Set labels and style
                    set_labels(c_shr, dummy[i].chart_label + " Neck Shear",         "Time (" + oGlblData.unit_time + ")", "Force (kN)");
                    set_labels(c_ten, dummy[i].chart_label + " Neck Tension",       "Time (" + oGlblData.unit_time + ")", "Force (kN)");
                    set_line_style(c_shr, Colour.BLACK);
                    set_line_style(c_ten, Colour.BLACK);
                }

                c_mom_adj     = Operate.Mux(c_mom_adj,     oGlblData.time_factor);
                set_labels(c_mom_adj, dummy[i].chart_label + " Neck Extension Bending Moment", "Time (" + oGlblData.unit_time + ")", "Bending Moment (Nm)");
                set_line_style(c_mom_adj, Colour.BLACK);

                // Remove all curves and datums
                remove_all_curves_and_datums();
                
                // Draw neck shear limit curves
                if(dummy[i].chart_label == "Driver")
                {
                    // Neck shear
                    // Draw neck shear limit curves
                    draw_constant_datums(MPDB.NECK_SHR_DR_GOOD, MPDB.NECK_SHR_DR_ADEQUATE, MPDB.NECK_SHR_DR_MARGINAL, MPDB.NECK_SHR_DR_WEAK);

                    // Create image/curve of neck shear curves
                    c_shr.AddToGraph();
                    write_image(output_data.shr.title, output_data.shr.fname, c_shr, MPDB.NECK_SHR_DR_WEAK);

                    output_data.shr.curveList.push(c_shr.id);
                    write_curve("cur", output_data.shr.curveList, output_data.shr.fname);
                    write_curve("csv", output_data.shr.curveList, output_data.shr.fname);

                    // Remove all curves and datums
                    remove_all_curves_and_datums();

                    // Neck tension
                    // Draw neck tension limit curves
                    draw_constant_datums(MPDB.NECK_TEN_DR_GOOD, MPDB.NECK_TEN_DR_ADEQUATE, MPDB.NECK_TEN_DR_MARGINAL, MPDB.NECK_TEN_DR_WEAK);

                    // Create image/curve of neck shear curves
                    c_ten.AddToGraph();
                    write_image(output_data.ten.title, output_data.ten.fname, c_ten, MPDB.NECK_TEN_DR_WEAK);

                    output_data.ten.curveList.push(c_ten.id);
                    write_curve("cur", output_data.ten.curveList, output_data.ten.fname);
                    write_curve("csv", output_data.ten.curveList, output_data.ten.fname);

                    // Remove all curves and datums
                    remove_all_curves_and_datums();

                    // Moment
                    // Draw moment limit datums
                    draw_constant_datums(MPDB.NECK_MOM_DR_GOOD, MPDB.NECK_MOM_DR_ADEQUATE, MPDB.NECK_MOM_DR_MARGINAL, MPDB.NECK_MOM_DR_WEAK);

                    // Create image/curve of moment curves
                    c_mom_adj.AddToGraph();
                    write_image(output_data.mom.title, output_data.mom.fname, c_mom_adj, MPDB.NECK_MOM_DR_WEAK);

                    output_data.mom.curveList.push(c_mom_adj.id);
                    write_curve("cur", output_data.mom.curveList, output_data.mom.fname);
                    write_curve("csv", output_data.mom.curveList, output_data.mom.fname);

                    // Remove all curves and datums
                    remove_all_curves_and_datums();

                    // Calc values
					//sliding_scale(val, hi_perf, lo_perf, hi_score, lo_score)
					var shr_results = get_constant_limit_results(c_shr, MPDB.NECK_SHR_DR_GOOD, MPDB.NECK_SHR_DR_WEAK, MPDB.NECK_LO_SCORE, MPDB.NECK_HI_SCORE);
					var ten_results = get_constant_limit_results(c_ten, MPDB.NECK_TEN_DR_GOOD, MPDB.NECK_TEN_DR_WEAK, MPDB.NECK_LO_SCORE, MPDB.NECK_HI_SCORE);
                    var mom_results = get_constant_limit_results(c_mom_adj, MPDB.NECK_MOM_DR_GOOD, MPDB.NECK_MOM_DR_WEAK, MPDB.NECK_LO_SCORE, MPDB.NECK_HI_SCORE);
                }          
                else if(dummy[i].chart_label == "Passenger")
                {    
                    // EuroNCAP and CNCAP have different assessment methods for neck shear/tension (with or without exceedance)
                    // EuroNCAP has the same value for lower limit and Capping limit; CNCAP has differnet values.
                    switch (template_reg)
                    {
                        case "EuroNCAP": // exceedance
                            // Neck shear
                            // Draw neck shear limit curves => up to final time
                            var p = c_shr_exc_abs.GetPoint(c_shr_exc_abs.npoints);       
                            draw_variable_datums(p[0], MPDB.NECK_SHR_PASS_EXC_TIME, MPDB.NECK_SHR_PASS_EXC_GOOD, MPDB.NECK_SHR_PASS_EXC_ADEQUATE, MPDB.NECK_SHR_PASS_EXC_MARGINAL, MPDB.NECK_SHR_PASS_EXC_WEAK);
                            
                            // Create image/curve of neck shear curves
                            c_shr_exc_abs.AddToGraph();
                            write_image(output_data.shr_exc.title, output_data.shr_exc.fname, c_shr_exc_abs, MPDB.NECK_SHR_PASS_EXC_WEAK[0]);

                            output_data.shr_exc.curveList.push(c_shr_exc_abs.id);
                            write_curve("cur", output_data.shr_exc.curveList, output_data.shr_exc.fname);
                            write_curve("csv", output_data.shr_exc.curveList, output_data.shr_exc.fname);

                            // Remove all curves and datums
                            remove_all_curves_and_datums();

                            // Neck tension
                            // Draw neck tension limit curves => up to final time
                            var p = c_ten_exc.GetPoint(c_ten_exc.npoints);       
                            draw_variable_datums(p[0], MPDB.NECK_TEN_PASS_EXC_TIME, MPDB.NECK_TEN_PASS_EXC_GOOD, MPDB.NECK_TEN_PASS_EXC_ADEQUATE, MPDB.NECK_TEN_PASS_EXC_MARGINAL, MPDB.NECK_PASS_TEN_EXC_WEAK);
        
                            // Create image/curve of neck shear curves
                            c_ten_exc.AddToGraph();
                            write_image(output_data.ten_exc.title, output_data.ten_exc.fname, c_ten_exc, MPDB.NECK_TEN_PASS_EXC_WEAK[0]);

                            output_data.ten_exc.curveList.push(c_ten_exc.id);
                            write_curve("cur", output_data.ten_exc.curveList, output_data.ten_exc.fname);
                            write_curve("csv", output_data.ten_exc.curveList, output_data.ten_exc.fname);
                            
                            // Remove all curves and datums
                            remove_all_curves_and_datums();

                            // Moment
                            // Draw moment limit datums
                            draw_constant_datums(MPDB.NECK_MOM_PASS_GOOD, MPDB.NECK_MOM_PASS_ADEQUATE, MPDB.NECK_MOM_PASS_MARGINAL, MPDB.NECK_MOM_PASS_WEAK);

                            // Create image/curve of moment curves
                            c_mom_adj.AddToGraph();
                            write_image(output_data.mom.title, output_data.mom.fname, c_mom_adj, MPDB.NECK_MOM_PASS_WEAK);

                            output_data.mom.curveList.push(c_mom_adj.id);
                            write_curve("cur", output_data.mom.curveList, output_data.mom.fname);
                            write_curve("csv", output_data.mom.curveList, output_data.mom.fname);
                            
                            // Remove all curves and datums
                            remove_all_curves_and_datums();                            
                            
                            // Calc values
                            var shr_results = get_variable_limit_results(c_shr_exc_abs, MPDB.NECK_SHR_PASS_EXC_TIME, MPDB.NECK_SHR_PASS_EXC_GOOD, MPDB.NECK_SHR_PASS_EXC_WEAK, MPDB.NECK_LO_SCORE, MPDB.NECK_HI_SCORE);
                            var ten_results = get_variable_limit_results(c_ten_exc,     MPDB.NECK_TEN_PASS_EXC_TIME, MPDB.NECK_TEN_PASS_EXC_GOOD, MPDB.NECK_TEN_PASS_EXC_WEAK, MPDB.NECK_LO_SCORE, MPDB.NECK_HI_SCORE);
                            var mom_results = get_constant_limit_results(c_mom_adj,     MPDB.NECK_MOM_PASS_GOOD, MPDB.NECK_MOM_PASS_WEAK, MPDB.NECK_LO_SCORE, MPDB.NECK_HI_SCORE);                       

                            break;

                        case "CNCAP": // No exceedance
                            // Neck shear
                            // Draw neck shear limit curves
                            draw_constant_datums(MPDB.NECK_SHR_PASS_GOOD, MPDB.NECK_SHR_PASS_ADEQUATE, MPDB.NECK_SHR_PASS_MARGINAL, MPDB.NECK_SHR_PASS_WEAK);

                            // Create image/curve of neck shear curves
                            c_shr.AddToGraph();
                            write_image(output_data.shr.title, output_data.shr.fname, c_shr, MPDB.NECK_SHR_PASS_WEAK);

                            output_data.shr.curveList.push(c_shr.id);
                            write_curve("cur", output_data.shr.curveList, output_data.shr.fname);
                            write_curve("csv", output_data.shr.curveList, output_data.shr.fname);

                            // Remove all curves and datums
                            remove_all_curves_and_datums();

                            // Neck tension
                            // Draw neck tension limit curves
                            draw_constant_datums(MPDB.NECK_TEN_PASS_GOOD, MPDB.NECK_TEN_PASS_ADEQUATE, MPDB.NECK_TEN_PASS_MARGINAL, MPDB.NECK_TEN_PASS_WEAK);

                            // Create image/curve of neck shear curves
                            c_ten.AddToGraph();
                            write_image(output_data.ten.title, output_data.ten.fname, c_ten, MPDB.NECK_TEN_PASS_WEAK);

                            output_data.ten.curveList.push(c_ten.id);
                            write_curve("cur", output_data.ten.curveList, output_data.ten.fname);
                            write_curve("csv", output_data.ten.curveList, output_data.ten.fname);

                            // Remove all curves and datums
                            remove_all_curves_and_datums();

                            // Moment
                            // Draw moment limit datums
                            draw_constant_datums(MPDB.NECK_MOM_PASS_GOOD, MPDB.NECK_MOM_PASS_ADEQUATE, MPDB.NECK_MOM_PASS_MARGINAL, MPDB.NECK_MOM_PASS_WEAK);

                            // Create image/curve of moment curves
                            c_mom_adj.AddToGraph();
                            write_image(output_data.mom.title, output_data.mom.fname, c_mom_adj, MPDB.NECK_MOM_PASS_WEAK);

                            output_data.mom.curveList.push(c_mom_adj.id);
                            write_curve("cur", output_data.mom.curveList, output_data.mom.fname);
                            write_curve("csv", output_data.mom.curveList, output_data.mom.fname);
                            
                            // Remove all curves and datums
                            remove_all_curves_and_datums();
        
                            // Calc values
                            // CNCAP has different lower limit and capping limit
                            // calculate the max value, and compare it with the capping limit
                            // Note here the cuver c_shr, c_ten, c_mom_adj, need consider the negative/positive values,
                            // which is the reason why there is a line to mul -1 for curv c_shr
                            var shr_results = get_constant_limit_results(c_shr, MPDB.NECK_SHR_PASS_GOOD, MPDB.NECK_SHR_PASS_WEAK, MPDB.NECK_LO_SCORE, MPDB.NECK_HI_SCORE);
                            if(c_shr.ymax > MPDB.NECK_SHR_PASS_WEAK_OVERALL)
                            {
                                capping_limit = "TRUE";
                                write_variable(f, dummy[i].chart_label.toUpperCase() + "_NECK_SHEAR_CAPPING_LIMIT", "*", dummy[i].chart_label + " Neck shear capping limit", "String");
                            }

                            var ten_results = get_constant_limit_results(c_ten, MPDB.NECK_TEN_PASS_GOOD, MPDB.NECK_TEN_PASS_WEAK, MPDB.NECK_LO_SCORE, MPDB.NECK_HI_SCORE);                                                   
                            if(c_ten.ymax > MPDB.NECK_TEN_PASS_WEAK_OVERALL)
                            {
                                capping_limit = "TRUE";
                                write_variable(f, dummy[i].chart_label.toUpperCase() + "_NECK_TENSION_CAPPING_LIMIT", "*", dummy[i].chart_label + " Neck tension capping limit", "String");
                            }

                            var mom_results = get_constant_limit_results(c_mom_adj, MPDB.NECK_MOM_PASS_GOOD, MPDB.NECK_MOM_PASS_WEAK, MPDB.NECK_LO_SCORE, MPDB.NECK_HI_SCORE);
                            if(c_mom_adj.ymax > MPDB.NECK_MOM_EXC_WEAK_OVERALL)
                            {
                                capping_limit = "TRUE";
                                write_variable(f, dummy[i].chart_label.toUpperCase() + "_NECK_EXTENSION_CAPPING_LIMIT", "*", dummy[i].chart_label + " Neck extension moment capping limit", "String");
                            }
                            if(capping_limit == "TRUE")
                            {
                                write_variable(f, dummy[i].chart_label.toUpperCase() + "_NECK_CAPPING_LIMIT", "*", dummy[i].chart_label + " Neck capping limit", "String");
                                write_variable(f, dummy[i].chart_label.toUpperCase() + "_HEAD_NECK_CAPPING_LIMIT", "*", dummy[i].chart_label + " head and neck capping limit", "String");
                            }

                            break;

                        default:
                            ErrorMessage("Code version is not assigned, should be either EuroNCAP or CNCAP");
                            return;
                            break;
                    }
                }
                
                // Write variables

                // SHEAR
                write_variable(f, dummy[i].variable_label.toUpperCase() + "_NECK_SHEAR_SCORE",  shr_results[0].toFixed(3), "Neck shear score",    "String");
                write_variable(f, dummy[i].variable_label.toUpperCase() + "_NECK_SHEAR",        shr_results[1].toFixed(3), "Neck shear",    "String");
                
                switch (template_reg)
                {
                    case "EuroNCAP":
                        if (dummy[i].chart_label == "Passenger") // for Passenger fo EuroNCAP, one more variable should be written out
                        {                
                            write_variable(f, dummy[i].variable_label.toUpperCase() + "_NECK_SHEAR_DURATION",   shr_results[2].toFixed(3), "Neck shear duration", "String");
                        }
                        break;
                    
                    case "CNCAP":
                        // No need to write SHEAR_DURATION out because of no exceedance
                        break;
                    
                    default:
                        ErrorMessage("NECK shear duration has errors");
                        return;
                        break;
                }

                // TENSION
                write_variable(f, dummy[i].variable_label.toUpperCase() + "_NECK_TENSION_SCORE",    ten_results[0].toFixed(3), "Neck tension score",    "String");
                write_variable(f, dummy[i].variable_label.toUpperCase() + "_NECK_TENSION",          ten_results[1].toFixed(3), "Neck tension",    "String");
                
                switch (template_reg)
                {
                    case "EuroNCAP":
                        if (dummy[i].chart_label == "Passenger") // for Passenger fo EuroNCAP, one more variable should be written out
                        {                
                            write_variable(f, dummy[i].variable_label.toUpperCase() + "_NECK_TENSION_DURATION",       ten_results[2].toFixed(3), "Neck tension duration", "String");
                        }
                        break;
                    
                    case "CNCAP":
                        // No need to write TENSION_DURATION out because of no exceedance
                        break;
                    
                    default:
                        ErrorMessage("NECK tension duration has errors");
                        return;
                        break;
                }

                // MOMENT
                write_variable(f, dummy[i].variable_label.toUpperCase() + "_NECK_EXTENSION_SCORE",  mom_results[0].toFixed(3), "Neck extension score", "String");
                write_variable(f, dummy[i].variable_label.toUpperCase() + "_NECK_BENDING_MOMENT",   mom_results[1].toFixed(3), "Neck bending moment",  "String");

                // Final score is minimum of all scores
                var value = Math.min(shr_results[0], ten_results[0], mom_results[0]).toFixed(3);
                write_variable(f, dummy[i].variable_label.toUpperCase() + "_NECK_FINAL_SCORE",   value, "Neck final score",  "String");

                switch (template_reg)
                {
                    case "EuroNCAP":
                        // Capping limit should be applied to overall score if these are zero because lower limit equals capping limit
                        if(shr_results[0] == 0.0)
                        {
                            capping_limit = "TRUE";
                            write_variable(f, dummy[i].chart_label.toUpperCase() + "_NECK_SHEAR_CAPPING_LIMIT", "*", dummy[i].chart_label + " Neck shear capping limit", "String");
                        }
                        if(ten_results[0] == 0.0)
                        {
                            capping_limit = "TRUE";
                            write_variable(f, dummy[i].chart_label.toUpperCase() + "_NECK_TENSION_CAPPING_LIMIT", "*", dummy[i].chart_label + " Neck tension capping limit", "String");
                        }
                        if(mom_results[0] == 0.0)
                        {
                            capping_limit = "TRUE";
                            write_variable(f, dummy[i].chart_label.toUpperCase() + "_NECK_EXTENSION_CAPPING_LIMIT", "*", dummy[i].chart_label + " Neck extension moment capping limit", "String");
                        }
                        if(capping_limit == "TRUE")
                        {
                            write_variable(f, dummy[i].chart_label.toUpperCase() + "_NECK_CAPPING_LIMIT", "*", dummy[i].chart_label + " Neck capping limit", "String");
                            write_variable(f, dummy[i].chart_label.toUpperCase() + "_HEAD_NECK_CAPPING_LIMIT", "*", dummy[i].chart_label + " head and neck capping limit", "String");
                        }
                        break;
                    
                    case "CNCAP":
                        if (dummy[i].chart_label == "Driver")
                        {
                            if(shr_results[0] == 0.0)
                            {
                                capping_limit = "TRUE";
                                write_variable(f, dummy[i].chart_label.toUpperCase() + "_NECK_SHEAR_CAPPING_LIMIT", "*", dummy[i].chart_label + " Neck shear capping limit", "String");
                            }
                            if(ten_results[0] == 0.0)
                            {
                                capping_limit = "TRUE";
                                write_variable(f, dummy[i].chart_label.toUpperCase() + "_NECK_TENSION_CAPPING_LIMIT", "*", dummy[i].chart_label + " Neck tension capping limit", "String");
                            }
                            if(mom_results[0] == 0.0)
                            {
                                capping_limit = "TRUE";
                                write_variable(f, dummy[i].chart_label.toUpperCase() + "_NECK_EXTENSION_CAPPING_LIMIT", "*", dummy[i].chart_label + " Neck extension moment capping limit", "String");
                            }
                            if(capping_limit == "TRUE")
                            {
                                write_variable(f, dummy[i].chart_label.toUpperCase() + "_NECK_CAPPING_LIMIT", "*", dummy[i].chart_label + " Neck capping limit", "String");
                                write_variable(f, dummy[i].chart_label.toUpperCase() + "_HEAD_NECK_CAPPING_LIMIT", "*", dummy[i].chart_label + " head and neck capping limit", "String");
                            }
                        }
                        // Capping limit of CNCAP Passenger has been done by previous codes
                        break;
                    
                    default:
                        ErrorMessage("NECK tension duration has errors");
                        return;
                        break;
                }
            }
        }
        else
        {
            // No node defined - variables should be set to blank by first script in template.
            // Create blank images to let the user know.
            write_blank_images(output_data, "NO ID DEFINED FOR NECK");
        }

        // Check Capping limit
        if(capping_limit == "TRUE") capping_limit_total = "TRUE";      
    }
    // Should capping limit be applied
    write_variable(f, "NECK_CAPPING_LIMIT", capping_limit_total, "Neck capping limit", "String");
}


function do_chest_rating_calc(f)
{
    // Calculates the chest scores for both driver and passenger and writes
    // relevant values into the Reporter csv variables file <f>.

    // Capping limit will considered for Driver's 4rib max compression, and Passenger's VC
    // Capping limit for all passengers
    var capping_limit_total = "FALSE";

    // Two passes, one for driver, one for passenger
    // For driver, make four passes, one for each rib compression   
    var dummy = {};
    dummy.driver = {};
    dummy.driver.Left = {}; 
    // Reading  node 1 to entity_type "node" and node 2 (N2) to entity_type "node_y" in the existing DummyData function in this_common1.js 
    dummy.driver.Left.Lower = new DummyData("Driver", oGlblData.driver_chest_compression_lower_left_node_1, "node",oGlblData.driver_chest_compression_lower_left_node_2, "node_y", oGlblData.driver_dummy, "version"); 
    dummy.driver.Left.Upper = new DummyData("Driver", oGlblData.driver_chest_compression_upper_left_node_1, "node",oGlblData.driver_chest_compression_upper_left_node_2, "node_y", oGlblData.driver_dummy, "version");
    dummy.driver.Right = {};
    dummy.driver.Right.Lower = new DummyData("Driver", oGlblData.driver_chest_compression_lower_right_node_1, "node",oGlblData.driver_chest_compression_lower_right_node_2, "node_y", oGlblData.driver_dummy, "version");
    dummy.driver.Right.Upper = new DummyData("Driver", oGlblData.driver_chest_compression_upper_right_node_1, "node",oGlblData.driver_chest_compression_upper_right_node_2, "node_y", oGlblData.driver_dummy, "version");
    
    dummy.driver.abdomen = {};
    dummy.driver.abdomen.Left = new DummyData("Driver", oGlblData.driver_abdomen_compression_left_node_1, "node",oGlblData.driver_abdomen_compression_left_node_2, "node_y", oGlblData.driver_dummy, "version");
    dummy.driver.abdomen.Right = new DummyData("Driver", oGlblData.driver_abdomen_compression_right_node_1, "node",oGlblData.driver_abdomen_compression_right_node_2, "node_y", oGlblData.driver_dummy, "version");
    
    dummy.passenger = new DummyData("Passenger", oGlblData.passenger_chest_transducer, "spring", oGlblData.passenger_dummy, "version");

    for(var i in dummy)
    {
        var capping_limit = "FALSE"; // for individual (Driver OR Passenger)

        var output_data = {};        
        var spring = {};        
        if (i == "driver")
        {
            // Calc Abdomen compression value is exceeding the lower limit or not; if so, use score = 0 for chest
            // No capping limit should be considered for abdomen however
            
            // with reference to the "HUMANETICS_THOR_50M_EuroNCAP_V1.8 " dummy nodes are used to track the IR-TRACC real-time length change or the ribs compression, (i. e) IRTRACC Mount Node (N1) and IRTRACC End Node (N2) .
            // The IR-TRACC real lengths for is done by following formula
            //      Length= sqrt( (n1x-n2x)^2 + (n1y-n2y)^2 + (n1z-n2z)^2)
            //   1. Left Upper Thorax IR-TRACC  
            //   2. Right Upper Thorax IR-TRACC 
            //   3. Left Lower Thorax IR-TRACC 
            //   4. Right Lower Thorax IR-TRACC 
            //   5. Left Abdomen IR-TRACC  
            //   6. Right Abdomen IR-TRACC 
            
            output_data.abdmcom = {};
            var driver_abdm_com = {};
            
            for (var j in dummy[i])
            {
                if(j == "abdomen")
                {
                    for (var lr in dummy[i][j])
                    {
                        output_data.abdmcom[lr] = new OutputData("Driver Abdomen Compression "+lr, images_dir + "/Driver_Abdomen_Compression_"+lr);

                        if(dummy[i][j][lr].node_id != undefined && dummy[i][j][lr].node_y_id != undefined)
                        {
                            // Read abdomen compression data IRTRACC Mount Node (N1)
                            var abdm_n1_x = read_data("NODE ", 0, dummy[i][j][lr].node_id, "DX");
                            var abdm_n1_y = read_data("NODE ", 0, dummy[i][j][lr].node_id, "DY");
                            var abdm_n1_z = read_data("NODE ", 0, dummy[i][j][lr].node_id, "DZ");
                           // Read abdomen compression data IRTRACC End Node (N2)
                            var abdm_n2_x = read_data("NODE ", 0, dummy[i][j][lr].node_y_id, "DX");
                            var abdm_n2_y = read_data("NODE ", 0, dummy[i][j][lr].node_y_id, "DY");
                            var abdm_n2_z = read_data("NODE ", 0, dummy[i][j][lr].node_y_id, "DZ");                                                                              
                                                          
                            // Check that the curve read in - i.e. the node id and component are valid
                            // Create blank images to let the user know.
                            if(!abdm_n1_x && !abdm_n2_x )
                            {
                                create_blank_image("NO DATA FOR nodes " + dummy[i][j][lr].node_id + " and " + dummy[i][j][lr].node_y_id + " for " + output_data.abdmcom[lr].title, output_data.abdmcom[lr].fname);
                            }
                            
                            else if ( !abdm_n2_x )
                            {
                                create_blank_image("NO DATA FOR node " + dummy[i][j][lr].node_y_id + " for " + output_data.abdmcom[lr].title, output_data.abdmcom[lr].fname);
                            } 
                            else if ( !abdm_n1_x )
                            {
                                create_blank_image("NO DATA FOR node " + dummy[i][j][lr].node_id + " for " + output_data.abdmcom[lr].title, output_data.abdmcom[lr].fname);
                            }
                            else
                            {   
                                var abdm_res = [];
                            
                                abdm_res[0] = Operate.Sub(abdm_n1_x, abdm_n2_x);  
                                abdm_res[1] = Operate.Sub(abdm_n1_y, abdm_n2_y);
                                abdm_res[2] = Operate.Sub(abdm_n1_z, abdm_n2_z);
                                
                                var abdm_dm = Operate.Res(abdm_res);
                                // Note the Calculation here is totally copied from Chest part. Not sure they are right or not.
                                var abdm_dm_c180_mm = convert_to_s_and_filter(abdm_dm, Operate.C180, MPDB.REG_DT);  // in mm (for compression calc)
    
                                // Remove all curves and datums    
                                remove_all_curves_and_datums();
    
                                // Convert from seconds back to model time    
                                abdm_dm_c180_mm = Operate.Mux(abdm_dm_c180_mm, oGlblData.time_factor);
                                
                                // Set labels and style                                
                                set_labels(abdm_dm_c180_mm, output_data.abdmcom[lr].title , "Time (" + oGlblData.unit_time + ")", "Compression (mm)");
                                set_line_style(abdm_dm_c180_mm, Colour.BLACK);
    
                                // Remove all curves and datums    
                                remove_all_curves_and_datums();
    
                                // Draw datums    
                                draw_constant_datums(MPDB.DR_ABDOMEN_COM_WEAK,MPDB.DR_ABDOMEN_COM_WEAK,MPDB.DR_ABDOMEN_COM_WEAK, MPDB.DR_ABDOMEN_COM_WEAK);
    
                                // Create image and curve files of chest compression curves    
                                abdm_dm_c180_mm.AddToGraph();
                                write_image(output_data.abdmcom[lr].title, output_data.abdmcom[lr].fname, abdm_dm_c180_mm, MPDB.DR_ABDOMEN_COM_WEAK);
    
                                output_data.abdmcom[lr].curveList.push(abdm_dm_c180_mm.id);
                                write_curve("cur", output_data.abdmcom[lr].curveList, output_data.abdmcom[lr].fname);
                                write_curve("csv", output_data.abdmcom[lr].curveList, output_data.abdmcom[lr].fname);
    
                                // Remove all curves and datums    
                                remove_all_curves_and_datums(); 
                                
                                // Calc values                               
                                driver_abdm_com[lr] = Math.abs(abdm_dm_c180_mm.ymax); // compression is negative
    
                                // Write variables                                
                                write_variable(f, "DRIVER_ABDOMEN_COMPRESSION_"+ lr.toUpperCase(), driver_abdm_com[lr].toFixed(3), output_data.abdmcom[lr].title, "String"); 
                            }
                        }
                    }
                    
                    var driver_abdo_com_list = [driver_abdm_com.Left, driver_abdm_com.Right];
                    var max_abdomen_com_results = Math.max.apply(null, driver_abdo_com_list);
                    write_variable(f, "DRIVER_ABDOMEN_MAX_COMPRESSION", max_abdomen_com_results.toFixed(3), "Abdomen max compression", "String"); 
                    
                    if (max_abdomen_com_results > MPDB.DR_ABDOMEN_COM_WEAK)
                    {
                       var abdm_com_results = MPDB.CHEST_LO_SCORE;
                       write_variable(f, "DRIVER_ABDOMEN_COMPRESSION_SCORE", abdm_com_results.toFixed(3), "Abdomen compression score", "String");
                    }
                    else
                    {
                        write_variable(f, "DRIVER_ABDOMEN_COMPRESSION_SCORE", "-", "Abdomen compression score", "String");
                    }
                }
            }

            // Calc chest compression for 4 ribs
            output_data.com = {};
            var driver_chest_com = {};
            var all_rib_data_present = true;

            for (var lr in dummy[i])
            {
                if (lr != "abdomen")
                {
                    output_data.com[lr] = {};
                    driver_chest_com[lr] = {};
                    for (var ul in dummy[i][lr])
                    {
                        output_data.com[lr][ul] = new OutputData("Driver Chest Compression "+ul+" "+lr, images_dir + "/Driver_Chest_Compression_"+ul+"_"+lr);

                        if ( dummy[i][lr][ul].node_id != undefined && dummy[i][lr][ul].node_y_id != undefined )
                        {
                            // // Read chest compression data IRTRACC Mount Node (N1)                            
                            var c_n1_x = read_data("NODE ", 0, dummy[i][lr][ul].node_id, "DX");
                            var c_n1_y = read_data("NODE ", 0, dummy[i][lr][ul].node_id, "DY");
                            var c_n1_z = read_data("NODE ", 0, dummy[i][lr][ul].node_id, "DZ");
                            // Read abdomen compression data IRTRACC End Node (N2)
                            var c_n2_x = read_data("NODE ", 0, dummy[i][lr][ul].node_y_id, "DX");
                            var c_n2_y = read_data("NODE ", 0, dummy[i][lr][ul].node_y_id, "DY");
                            var c_n2_z = read_data("NODE ", 0, dummy[i][lr][ul].node_y_id, "DZ");                           
                            
                            // Check that the curve read in - i.e. the node id and component are valid
                            // Create blank images to let the user know.

                            if(!c_n1_x && !c_n2_x)
                            {
                                create_blank_image("NO DATA FOR nodes " + dummy[i][lr][ul].node_id + " and " + dummy[i][lr][ul].node_y_id + " for " + output_data.com[lr][ul].title, output_data.com[lr][ul].fname);
                                all_rib_data_present = false;
                            }
                            else if( !c_n1_x )
                            {
                                create_blank_image("NO DATA FOR node " + dummy[i][lr][ul].node_id + " for "+ output_data.com[lr][ul].title, output_data.com[lr][ul].fname);
                                all_rib_data_present = false;
                            }
                            else if( !c_n2_x )
                            {
                                create_blank_image("NO DATA FOR node " + dummy[i][lr][ul].node_y_id + " for "+ output_data.com[lr][ul].title, output_data.com[lr][ul].fname);
                                all_rib_data_present = false;
                            }
                            else
                            {   
                                var c_res = [];                            
                                c_res[0] = Operate.Sub(c_n1_x, c_n2_x);  
                                c_res[1] = Operate.Sub(c_n1_y, c_n2_y);
                                c_res[2] = Operate.Sub(c_n1_z, c_n2_z);
                                
                                var c_dm = Operate.Res(c_res);
                                var c_dm_c180_mm = convert_to_s_and_filter(c_dm, Operate.C180, MPDB.REG_DT);  // in mm (for compression calc)

                                // Remove all curves and datums
                                remove_all_curves_and_datums();

                                // Convert from seconds back to model time
                                c_dm_c180_mm = Operate.Mux(c_dm_c180_mm, oGlblData.time_factor);
                                
                                // Set labels and style                                
                                set_labels(c_dm_c180_mm, output_data.com[lr][ul].title , "Time (" + oGlblData.unit_time + ")", "Compression (mm)");
                                set_line_style(c_dm_c180_mm, Colour.BLACK);

                                // Remove all curves and datums
                                remove_all_curves_and_datums();

                                // Draw chest compression datums
                                draw_constant_datums(MPDB.DR_CHEST_COM_GOOD, MPDB.DR_CHEST_COM_ADEQUATE, MPDB.DR_CHEST_COM_MARGINAL, MPDB.DR_CHEST_COM_WEAK);

                                // Create image and curve files of chest compression curves
                                c_dm_c180_mm.AddToGraph();
                                write_image(output_data.com[lr][ul].title, output_data.com[lr][ul].fname, c_dm_c180_mm, MPDB.DR_CHEST_COM_WEAK);

                                output_data.com[lr][ul].curveList.push(c_dm_c180_mm.id);
                                write_curve("cur", output_data.com[lr][ul].curveList, output_data.com[lr][ul].fname);
                                write_curve("csv", output_data.com[lr][ul].curveList, output_data.com[lr][ul].fname);

                                // Remove all curves and datums
                                remove_all_curves_and_datums(); 
                                
                                // Calc values                                
                                driver_chest_com[lr][ul] = c_dm_c180_mm.ymax;

                                // Write variables                                
                                write_variable(f, "DRIVER_CHEST_COMPRESSION_"+ ul.toUpperCase() +"_"+ lr.toUpperCase(), driver_chest_com[lr][ul].toFixed(3), output_data.com[lr][ul].title, "String"); 
                            }
                        }
                        else
                        {
                            // No node defined - variables should be set to blank by first script in template.
                            // Create blank images to let the user know.
                            create_blank_image("NO NODE ID DEFINED FOR " + output_data.com[lr][ul].title, output_data.com[lr][ul].fname);
                            all_rib_data_present = false;
                        }
                    }
                }
            }
                
            // Calc Chest values + Abdomen scores
            if (all_rib_data_present == true)
            {
                var driver_chest_com_list = [driver_chest_com.Left.Upper, driver_chest_com.Left.Lower, driver_chest_com.Right.Upper, driver_chest_com.Right.Lower];
                var max_chest_com_results = Math.max.apply(null, driver_chest_com_list);
                write_variable(f, "DRIVER_CHEST_MAX_COMPRESSION", max_chest_com_results.toFixed(3), "Chest max compression", "String"); 

                var chest_com_results = sliding_scale(max_chest_com_results, 
                                                      MPDB.DR_CHEST_COM_GOOD, MPDB.DR_CHEST_COM_WEAK, 
                                                      MPDB.CHEST_HI_SCORE, MPDB.CHEST_LO_SCORE);

                // Capping limit should be applied to overall score of compression of 4 ribs if either of these are zero
                // note here should be in front of abdomen assessment --- because assessment could also make score equal to 0 but no capping limit is applied on abdomen
                if(chest_com_results == 0.0 ) 
                {
                    capping_limit = "TRUE";
                    write_variable(f, "DRIVER_CHEST_CAPPING_LIMIT", "*", "Driver chest capping limit", "String"); 
                }

                // If abdomen is out-of-limit, then use the score of abdomen (equal 0) as the chest score
                if(abdm_com_results != undefined)
                {
                    chest_com_results = Math.min(chest_com_results, abdm_com_results);
                }
            
                write_variable(f, "DRIVER_CHEST_COMPRESSION_SCORE", chest_com_results.toFixed(3), "Chest compression score", "String");              
            }
            else
            {
                capping_limit = "TRUE";
                write_variable(f, "DRIVER_CHEST_CAPPING_LIMIT", "*", "Driver chest capping limit", "String"); 
            } 
                
        }
        else // Passenger
        {
            output_data.com = new OutputData(dummy[i].chart_label + " Chest compression",       images_dir + "/" + dummy[i].chart_label + "_Chest_Compression");
            output_data.vc  = new OutputData(dummy[i].chart_label + " Chest Viscous Criterion", images_dir + "/" + dummy[i].chart_label + "_Chest_Viscous_Criterion");

            if (dummy[i].spring_id != undefined)
            {
                // Read in spring rotation
                var c_rot = read_data("SPRING ROT", 0, dummy[i].spring_id, "RT");

                // Check that the curve read in - i.e. the spring id and component are valid
                // Create blank images to let the user know.

                if(!c_rot)
                {
                    write_blank_images(output_data, "NO DATA FOR SPRING " + spring);
                }
                else
                {
                    // Chest compression is calculated by converting the rotation in radians to a deflection using
                    // a transfer function.
                    //
                    // HIII 50 M (passenger dummy in Euro NCAP MPDB template)
                    // For Humanetics V7 dummies and earlier, the conversion is linear
                    // For LSTC dummies, a different linear conversion factor is used
                    // For Humanetics V8 dummies and later, the conversion uses a 3rd order polynomial
                    //
                    // HIII 5F (passenger dummy in C-NCAP MPDB template)
                    // For V2 LSTC and V6 Humanetics dummies and earlier, the conversion is linear
                    // For V7 Humanetics dummies and later, the conversion uses a 3rd order polynomial
                    //
                    // The dummy manuals say that a negative rotation indicates compression, however testing has shown
                    // that in some versions it is actually the other way around. If the maximum absolute value is +ve
                    // then multiply the rotation by -1 so the conversion to compression works correctly.

                    var rmax = c_rot.ymax;
                    var rmin = c_rot.ymin;

                    if(Math.abs(rmax) > Math.abs(rmin)) c_rot = Operate.Mul(c_rot, -1);

                    var c_disp_mm;

                    // Depending on the protocol (C-NCAP or Euro NCAP) we expect a different passenger dummy.
                    // The separate cases in this switch statement allow us to check that the user hasn't
                    // accidentally supplied a C-NCAP CSV file for a Euro NCAP template or vice versa.
                    switch(template_reg)
                    {
                        case "EuroNCAP":
                            switch (dummy[i].version) 
                            {
                                // HIII 50M (passenger dummy in Euro NCAP MPDB template)
                                case "V7_HUMANETICS_HIII_50TH_MALE": 
                                    c_disp_mm = Operate.Mul(c_rot, MPDB.CHEST_V7_ROT_TO_COM_FACTOR);
                                    break;
                                case "LSTC HIII 50M Detailed v190217_beta":                
                                case "LSTC HIII 50M Fast V2.0": 
                                case "LSTC HIII 50M v130528_beta":
                                    c_disp_mm = Operate.Mul(c_rot, MPDB.CHEST_LSTC_HIII_50TH_MALE_DETAILED_190217_BETA_ROT_TO_COM_FACTOR);
                                    break;
                                case "V8_HUMANETICS_HIII_50TH_MALE":
                                case "Humanetics HIII 50M v1.5":
                                case "Humanetics HIII 50M v1.5.1":            
                                    var curve_id = Curve.FirstFreeID();
                                    c_disp_mm = new Curve(curve_id);
                                    var n = c_rot.npoints;
                                    for(var i = 1; i <= n; i++)
                                    {
                                        var p_rot = c_rot.GetPoint(i);

                                        var time = p_rot[0];
                                        var com  = MPDB.CHEST_V8_ROT_TO_COM_FACTOR_A * p_rot[1] * p_rot[1] * p_rot[1] +
                                                   MPDB.CHEST_V8_ROT_TO_COM_FACTOR_B * p_rot[1] * p_rot[1] + 
                                                   MPDB.CHEST_V8_ROT_TO_COM_FACTOR_C * p_rot[1];

                                        c_disp_mm.AddPoint(time, com);
                                    }
                                    break;
                                // Catch user error where HIII 5F dummy given by mistake (passenger dummy in C-NCAP MPDB template)
                                case "LSTC HIII 5F v2.0":
                                case "Humanetics HIII 5F v2.02":
                                    ErrorMessage("Unsupported dummy \""+dummy[i].version+"\" in do_chest_rating_calc. Euro NCAP MPDB passenger should be HIII 50M, not HIII 5F.");
                                    write_blank_images(output_data, "UNSUPPORTED DUMMY \""+dummy[i].version+"\"");
                                    return;
                                    break;
                                default:
                                    ErrorMessage("Unsupported dummy \""+dummy[i].version+"\" in do_chest_rating_calc.");
                                    write_blank_images(output_data, "UNSUPPORTED DUMMY \""+dummy[i].version+"\"");
                                    return;
                                    break;
                            }
                            break;

                        case "CNCAP":
                            switch (dummy[i].version) 
                            {
                                // Catch user error where HIII 50M given by mistake (passenger dummy in Euro NCAP MPDB template)
                                case "V7_HUMANETICS_HIII_50TH_MALE": 
                                case "LSTC HIII 50M Detailed v190217_beta":                
                                case "LSTC HIII 50M Fast V2.0": 
                                case "LSTC HIII 50M v130528_beta":
                                case "V8_HUMANETICS_HIII_50TH_MALE":
                                case "Humanetics HIII 50M v1.5":
                                case "Humanetics HIII 50M v1.5.1":
                                    ErrorMessage("Unsupported dummy \""+dummy[i].version+"\" in do_chest_rating_calc. C-NCAP MPDB passenger should be HIII 5F, not HIII 50M.");
                                    write_blank_images(output_data, "UNSUPPORTED DUMMY \""+dummy[i].version+"\"");
                                    return;
                                    break;
                                // HIII 5F (passenger dummy in C-NCAP MPDB template)
                                case "LSTC HIII 5F v2.0":
                                    c_disp_mm = Operate.Mul(c_rot, MPDB.CHEST_V2_LSTC_ROT_TO_COM_FACTOR);
                                    break;
                                case "Humanetics HIII 5F v2.02":
                                    var curve_id = Curve.FirstFreeID();
                                    c_disp_mm = new Curve(curve_id);
                                    var n = c_rot.npoints;
                                    for(var i = 1; i <= n; i++)
                                    {
                                        var p_rot = c_rot.GetPoint(i);

                                        var time = p_rot[0];
                                        var com  = MPDB.CHEST_V7_ROT_TO_COM_FACTOR_A * p_rot[1] * p_rot[1] * p_rot[1] +
                                                   MPDB.CHEST_V7_ROT_TO_COM_FACTOR_B * p_rot[1] * p_rot[1] + 
                                                   MPDB.CHEST_V7_ROT_TO_COM_FACTOR_C * p_rot[1];

                                        c_disp_mm.AddPoint(time, com);
                                    }
                                    break;
                                default:
                                    ErrorMessage("Unsupported dummy \""+dummy[i].version+"\" in do_chest_rating_calc.");
                                    write_blank_images(output_data, "UNSUPPORTED DUMMY \""+dummy[i].version+"\"");
                                    return;
                                    break;
                            }
                            break;                            

                        default:
                            ErrorMessage("Unexpected regulation \""+ template_reg + "\" in do_chest_rating_calc calculation.");
                            return;
                            break;
                    }

                    // C180 filter (convert to seconds first)
                    var c_disp_c180_mm = convert_to_s_and_filter(c_disp_mm, Operate.C180, MPDB.REG_DT);  // in mm (for compression calc)
                    var c_disp_c180_m  = Operate.Div(c_disp_c180_mm, 1000);  // in m  (for viscous criterion calc)

                    // Calculate viscous criterion
                    var c_vc = Operate.Vc(c_disp_c180_m, MPDB.CHEST_VC_A, MPDB.CHEST_VC_B, MPDB.CHEST_VC_METHOD);

                    // Remove all curves and datums
                    remove_all_curves_and_datums();

                    // Convert from seconds back to model time
                    c_disp_c180_mm = Operate.Mux(c_disp_c180_mm, oGlblData.time_factor);
                    c_vc           = Operate.Mux(c_vc,           oGlblData.time_factor);

                    // Set labels and style
                    set_labels(c_disp_c180_mm, "Passenger Chest Compression", "Time (" + oGlblData.unit_time + ")", "Compression (mm)");
                    set_labels(c_vc,           "Passenger Chest Viscous Criterion", "Time (" + oGlblData.unit_time + ")", "Velocity (m/s)");

                    set_line_style(c_disp_c180_mm, Colour.BLACK);
                    set_line_style(c_vc,           Colour.BLACK);

                    // Remove all curves and datums
                    remove_all_curves_and_datums();

                    // Draw chest compression datums
                    draw_constant_datums(MPDB.PASS_CHEST_COM_GOOD, MPDB.PASS_CHEST_COM_ADEQUATE, MPDB.PASS_CHEST_COM_MARGINAL, MPDB.PASS_CHEST_COM_WEAK);

                    // Create image and curve files of chest compression curves
                    c_disp_c180_mm.AddToGraph();
                    write_image(output_data.com.title, output_data.com.fname, c_disp_c180_mm, MPDB.CHEST_COM_WEAK);

                    output_data.com.curveList.push(c_disp_c180_mm.id);
                    write_curve("cur", output_data.com.curveList, output_data.com.fname);
                    write_curve("csv", output_data.com.curveList, output_data.com.fname);

                    // Remove all curves and datums
                    remove_all_curves_and_datums(); 

                    // Draw viscous criterion datums
                    draw_constant_datums(MPDB.CHEST_VC_GOOD, MPDB.CHEST_VC_ADEQUATE, MPDB.CHEST_VC_MARGINAL, MPDB.CHEST_VC_WEAK);

                    // Create image and curve files of viscous criterion curves
                    c_vc.AddToGraph();
                    write_image(output_data.vc.title, output_data.vc.fname, c_vc, MPDB.CHEST_VC_WEAK);

                    output_data.vc.curveList.push(c_vc.id);
                    write_curve("cur", output_data.vc.curveList, output_data.vc.fname);
                    write_curve("csv", output_data.vc.curveList, output_data.vc.fname);

                    // Remove all curves and datums
                    remove_all_curves_and_datums(); 

                    // Calc values
                    var chest_com_results = get_constant_limit_results(c_disp_c180_mm, MPDB.PASS_CHEST_COM_GOOD, MPDB.PASS_CHEST_COM_WEAK, MPDB.CHEST_LO_SCORE, MPDB.CHEST_HI_SCORE);
                    var chest_vc_results  = get_constant_limit_results(c_vc,           MPDB.CHEST_VC_GOOD,  MPDB.CHEST_VC_WEAK,  MPDB.CHEST_LO_SCORE, MPDB.CHEST_HI_SCORE);

                    // Write variables
                    write_variable(f, "PASSENGER_CHEST_COMPRESSION_SCORE",       chest_com_results[0].toFixed(3), "Passenger chest compression score", "String");        
                    write_variable(f, "PASSENGER_CHEST_COMPRESSION",             chest_com_results[1].toFixed(3), "Passenger chest compression",       "String");   

                    write_variable(f, "PASSENGER_CHEST_VISCOUS_CRITERION_SCORE", chest_vc_results[0].toFixed(3),  "Passenger chest viscous criterion score", "String");        
                    write_variable(f, "PASSENGER_CHEST_VISCOUS_CRITERION",       chest_vc_results[1].toFixed(3),  "Passenger chest viscous criterion",       "String");           

                    // Capping limit should be applied to overall score if either of these are zero because lower limit equals capping limit
                    if(chest_com_results[0] == 0.0)
                    {
                        capping_limit = "TRUE";
                        write_variable(f, "PASSENGER_CHEST_COMPRESSION_CAPPING_LIMIT", "*", "Passenger chest compression capping limit", "String"); 
                    }
                    else if(chest_vc_results[0] == 0.0) 
                    {
                        capping_limit = "TRUE";
                        write_variable(f, "PASSENGER_CHEST_VC_CAPPING_LIMIT", "*", "Passenger chest VC capping limit", "String");
                    }
                    if(capping_limit == "TRUE")
                    {
                        write_variable(f, "PASSENGER_CHEST_CAPPING_LIMIT", "*", "Passenger chest capping limit", "String");  
                    }
                }
            }
            else
            {
                // No spring defined - variables should be set to blank by first script in template.
                // Create blank images to let the user know.
                write_blank_images(output_data, "NO SPRING ID DEFINED FOR");
            }
        }

        // Calculate A-Pillar intrusion modifier - driver only
        var driver_output_data = {};
        driver_output_data.ap  = new OutputData("Driver Fore/Aft A-Pillar Intrusion", images_dir + "/" + "Driver_Fore_Aft_A_Pillar_Intrusion");

        if (i == "driver")
        {
            var a_pillar_disp = 0.0;
            var spring = oGlblData.a_pillar_intrusion_spring;   // Acceleration pedal fore/aft spring

            var c_a_pillar_fa;

            if(spring != undefined)
            {
                var c_el = read_data("SPRING TR", 0, spring, "ET");

                if(c_el)
                {
                    // Convert to mm 
                    var c_el_mm = Operate.Div(c_el, oGlblData.mm_factor);

                    // Fore/aft intrusion = -ve elongation is rearward so multiply by -1.0 first    
                    c_a_pillar_fa = Operate.Mul(c_el_mm, -1.0);
                    var p = c_a_pillar_fa.GetPoint(c_a_pillar_fa.npoints);
                    a_pillar_disp = p[1];

                    set_labels(c_a_pillar_fa, "Driver A-Pillar Fore/Aft Intrusion (+ve rearward)", "Time (" + oGlblData.unit_time + ")", "Intrusion (mm)");
                    set_line_style(c_a_pillar_fa, Colour.BLACK);
                }
            }

            // Calculate modifier
            var a_pillar_mod = sliding_scale(a_pillar_disp, 
                                MPDB.CHEST_A_PILLAR_FA_DISP_GOOD, MPDB.CHEST_A_PILLAR_FA_DISP_WEAK,
                                MPDB.CHEST_A_PILLAR_HI_SCORE, MPDB.CHEST_A_PILLAR_LO_SCORE);

            // Remove all curves and datums
            remove_all_curves_and_datums();

            // Draw fore/aft displacment datums
            if(c_a_pillar_fa) draw_constant_datums(MPDB.CHEST_A_PILLAR_FA_DISP_GOOD, MPDB.CHEST_A_PILLAR_FA_DISP_ADEQUATE, MPDB.CHEST_A_PILLAR_FA_DISP_MARGINAL, MPDB.CHEST_A_PILLAR_FA_DISP_WEAK);

            // Create image and curve files of fore/aft intrusion curves
            if(c_a_pillar_fa) c_a_pillar_fa.AddToGraph();

            if(c_a_pillar_fa) write_image(driver_output_data.ap.title, driver_output_data.ap.fname, c_a_pillar_fa, MPDB.CHEST_A_PILLAR_FA_DISP_WEAK);
            else              write_blank_images(driver_output_data, "SPRING ID MISSING/DOES NOT EXIST FOR Driver Fore/Aft A-Pillar Intrusion");

            if(c_a_pillar_fa) driver_output_data.ap.curveList.push(c_a_pillar_fa.id);
            write_curve("cur", driver_output_data.ap.curveList, driver_output_data.ap.fname);
            write_curve("csv", driver_output_data.ap.curveList, driver_output_data.ap.fname);

            // Write variables
            write_variable(f, "DRIVER_CHEST_A_PILLAR_DISPLACEMENT", a_pillar_disp.toFixed(3), "A-Pillar fore/aft displacement",       "String");
            write_variable(f, "DRIVER_CHEST_A_PILLAR_MOD",          a_pillar_mod.toFixed(3),  "A-Pillar fore/aft displacement score", "String");
        }
        
        // Check Capping limit
        if(capping_limit == "TRUE") capping_limit_total = "TRUE";  
    }
    // Should capping limit be applied
    write_variable(f, "CHEST_CAPPING_LIMIT", capping_limit_total, "Chest capping limit", "String");
}


function do_knee_femur_pelvis_rating_calc(f)
{
    // Calculates the knee, femur and pelvis scores for both driver and passenger and writes
    // relevant values into the Reporter csv variables file <f>.

    // Two passes, one for driver, one for passenger.
    // And for each one do the left and right hand side
    var dummy = {};
    dummy.driver = {};
    dummy.driver.Left = new DummyData("Driver",
                                        oGlblData.driver_left_femur_loadcell, "beam",  // should be "section", but are modified by codes below
                                        oGlblData.driver_left_knee_transducer, "spring",
                                        oGlblData.driver_left_acetabulum_loadcell, "beam_2", // should be "section_2", but are modified by codes below
                                        oGlblData.driver_dummy, "version");
    dummy.driver.Right = new DummyData("Driver",
                                        oGlblData.driver_right_femur_loadcell, "beam",  // should be "section", but are modified by codes below
                                        oGlblData.driver_right_knee_transducer, "spring",
                                        oGlblData.driver_right_acetabulum_loadcell, "beam_2", // should be "section_2", but are modified by codes below
                                        oGlblData.driver_dummy, "version");
    dummy.passenger = {};
    dummy.passenger.Left = new DummyData("Passenger",
                                        oGlblData.passenger_left_femur_loadcell, "beam",
                                        oGlblData.passenger_left_knee_transducer, "spring",
                                        oGlblData.passenger_dummy, "version");
    dummy.passenger.Right = new DummyData("Passenger",
                                        oGlblData.passenger_right_femur_loadcell, "beam",
                                        oGlblData.passenger_right_knee_transducer, "spring",
                                        oGlblData.passenger_dummy, "version");

    var fpass_knee_femur_pelvis_exists = "TRUE";
    for(var i in dummy)
    {
        for(var lr in dummy[i])
        {
            // Femur compression forces
            var output_data = {};

            // place-holder for only CNCAP-Passenger that should NOT do exceedance
            if (dummy[i][lr].chart_label == "Passenger" && template_reg == "CNCAP")
            {
                output_data.com = new OutputData(dummy[i][lr].chart_label + " " + lr + " Femur Compression", images_dir + "/" + dummy[i][lr].variable_label + "_" + lr + "_Femur_Compression");
            }
            // for the others, should do exceedance
            else
            {
                output_data.com_exc = new OutputData(dummy[i][lr].chart_label + " " + lr + " Femur Compression exceedance", images_dir + "/" + dummy[i][lr].variable_label + "_" + lr + "_Femur_Compression");
            }

            if(dummy[i][lr].beam_id != undefined)
            {
                // Read in beam longitudinal force
                // For Humanetics dummies this is measured from a beam;
                // for LSTC dummies this is measured from a cross section

                switch (dummy[i][lr].version)
                {
                    case "V6_HUMANETICS_HIII_50TH_MALE":
                    case "V7_HUMANETICS_HIII_50TH_MALE":
                    case "V8_HUMANETICS_HIII_50TH_MALE":
                    case "Humanetics HIII 50M v1.5":
                    case "Humanetics HIII 50M v1.5.1":
                        var c_com = read_data("BEAM BASIC", 0, dummy[i][lr].beam_id, "NZ");  // NZ is longitudinal
                        var entity_str = "BEAM"; // Used for error message below
                        break;

                    case "LSTC HIII 5F v2.0":
                    case "LSTC HIII 50M Detailed v190217_beta":
                    case "LSTC HIII 50M Fast V2.0":
                    case "LSTC HIII 50M v130528_beta":
                    case "Humanetics HIII 5F v2.02":
                    case "Humanetics THOR 50M v1.8":
                    case "Humanetics THOR 50M v1.9":
                        var c_com = read_data("SECTION", 0, dummy[i][lr].beam_id, "FZ");  // FZ is longitudinal
                        var entity_str = "SECTION"; // Used for error message below
                        break;

                    default:
                        ErrorMessage("Unsupported dummy \""+dummy[i][lr].version+"\" in do_knee_femur_pelvis_rating_calc for longitudinal force.");
                        write_blank_images(output_data, "UNSUPPORTED DUMMY \""+dummy[i][lr].version+"\"");
                        return;
                        break;
                }

                // Check that the curve read in - i.e. the beam id and component are valid
                // Create blank image to let the user know.

                if(!c_com)
                {   
                        write_blank_images(output_data, "NO DATA FOR " + entity_str + " " + dummy[i][lr].beam_id);
                }
                else
                {
                    // Convert forces to kN from model units
                    var c_com_kn = Operate.Div(c_com, oGlblData.kn_factor);

                    // C600 filter (convert to seconds first)
                    var c_com_c600 = convert_to_s_and_filter(c_com_kn, Operate.C600, MPDB.REG_DT);
                    
                    // for CNCAP Passener, no need to do exceedance
                    if (dummy[i][lr].chart_label == "Passenger" && template_reg == "CNCAP")
                    { 
                        // Remove all curves and datums
                        remove_all_curves_and_datums();

                        // Convert from seconds back to model time
                        var c_com = Operate.Mux(c_com_c600, oGlblData.time_factor);
                        // Set labels and style
                        set_labels(c_com, dummy[i][lr].chart_label + " " + lr + " Femur Compression", "Time (" + oGlblData.unit_time + ")", "Compression (kN)");
                        set_line_style(c_com, Colour.BLACK);

                        // Remove all curves and datums
                        remove_all_curves_and_datums();
                    }
                    else if(dummy[i][lr].chart_label == "Driver" || template_reg == "EuroNCAP")
                    {
                        // exceedance curve for compression - negative values.
                        var c_com_exc = Operate.Exc(c_com_c600, "negative");
                        // Take absolute values
                        var c_com_exc_abs = Operate.Abs(c_com_exc);
                        // Remove all curves and datums
                        remove_all_curves_and_datums();

                        // Convert from seconds back to model time
                        c_com_exc_abs = Operate.Mux(c_com_exc_abs, oGlblData.time_factor);
                        // Set labels and style
                        set_labels(c_com_exc_abs, dummy[i][lr].chart_label + " " + lr + " Femur Compression exceedance", "Cumulative Time (" + oGlblData.unit_time + ")", "Compression (kN)");
                        set_line_style(c_com_exc_abs, Colour.BLACK);

                        // Remove all curves and datums
                        remove_all_curves_and_datums();
                    }

                    // Draw femur compression limit curves => up to final time
                    // Femur of EuroNCAP, Driver and Passenger are the same;
                    // Femur of CNCAP, Driver uses exceedance, and Passenger uses direct curve (no exceedance). The limits are also different.
                    switch(template_reg)
                    {
                        case "EuroNCAP":
                            var p = c_com_exc_abs.GetPoint(c_com_exc_abs.npoints);       
                            draw_variable_datums(p[0], MPDB.FEMUR_COM_EXC_TIME, MPDB.FEMUR_COM_EXC_GOOD, MPDB.FEMUR_COM_EXC_ADEQUATE, MPDB.FEMUR_COM_EXC_MARGINAL, MPDB.FEMUR_COM_EXC_WEAK);
                            
                            // Create image/curve of femur compression curves
                            c_com_exc_abs.AddToGraph();
                            write_image(output_data.com_exc.title, output_data.com_exc.fname, c_com_exc_abs, MPDB.FEMUR_COM_EXC_WEAK[0]);

                            output_data.com_exc.curveList.push(c_com_exc_abs.id);
                            write_curve("cur", output_data.com_exc.curveList, output_data.com_exc.fname);
                            write_curve("csv", output_data.com_exc.curveList, output_data.com_exc.fname);

                            // Remove all curves and datums
                            remove_all_curves_and_datums(); 

                            // Calc values
                            var com_results = get_variable_limit_results(c_com_exc_abs, MPDB.FEMUR_COM_EXC_TIME, MPDB.FEMUR_COM_EXC_GOOD, MPDB.FEMUR_COM_EXC_WEAK, MPDB.KNEE_FEMUR_PELVIS_LO_SCORE, MPDB.KNEE_FEMUR_PELVIS_HI_SCORE);

                            // Write compression variables
                            write_variable(f, dummy[i][lr].variable_label.toUpperCase() + "_" + lr.toUpperCase() + "_FEMUR_COMPRESSION_SCORE",          com_results[0].toFixed(3), "Femur compression score",    "String");
                            write_variable(f, dummy[i][lr].variable_label.toUpperCase() + "_" + lr.toUpperCase() + "_FEMUR_COMPRESSION_LEVEL_EXCEEDED", com_results[1].toFixed(3), "Femur compression level",    "String");
                            write_variable(f, dummy[i][lr].variable_label.toUpperCase() + "_" + lr.toUpperCase() + "_FEMUR_COMPRESSION_DURATION",       com_results[2].toFixed(3), "Femur compression duration", "String");
                            break;

                        case "CNCAP":
                            if (dummy[i][lr].chart_label == "Driver") // Driver, with exceedance
                            {
                                var p = c_com_exc_abs.GetPoint(c_com_exc_abs.npoints);       
                                draw_variable_datums(p[0], MPDB.FEMUR_COM_EXC_TIME, MPDB.FEMUR_COM_EXC_GOOD, MPDB.FEMUR_COM_EXC_ADEQUATE, MPDB.FEMUR_COM_EXC_MARGINAL, MPDB.FEMUR_COM_EXC_WEAK);
                                
                                // Create image/curve of femur compression curves
                                c_com_exc_abs.AddToGraph();                 
                        
                                write_image(output_data.com_exc.title, output_data.com_exc.fname, c_com_exc_abs, MPDB.FEMUR_COM_EXC_WEAK[0]);
                                
                                output_data.com_exc.curveList.push(c_com_exc_abs.id);
                                write_curve("cur", output_data.com_exc.curveList, output_data.com_exc.fname);
                                write_curve("csv", output_data.com_exc.curveList, output_data.com_exc.fname);
            
                                // Remove all curves and datums
                                remove_all_curves_and_datums(); 
            
                                // Calc values
                                var com_results = get_variable_limit_results(c_com_exc_abs, MPDB.FEMUR_COM_EXC_TIME, MPDB.FEMUR_COM_EXC_GOOD, MPDB.FEMUR_COM_EXC_WEAK, MPDB.KNEE_FEMUR_PELVIS_LO_SCORE, MPDB.KNEE_FEMUR_PELVIS_HI_SCORE);
        
                                // Write compression variables        
                                write_variable(f, dummy[i][lr].variable_label.toUpperCase() + "_" + lr.toUpperCase() + "_FEMUR_COMPRESSION_SCORE",          com_results[0].toFixed(3), "Femur compression score",    "String");
                                write_variable(f, dummy[i][lr].variable_label.toUpperCase() + "_" + lr.toUpperCase() + "_FEMUR_COMPRESSION_LEVEL_EXCEEDED", com_results[1].toFixed(3), "Femur compression level",    "String");
                                write_variable(f, dummy[i][lr].variable_label.toUpperCase() + "_" + lr.toUpperCase() + "_FEMUR_COMPRESSION_DURATION",       com_results[2].toFixed(3), "Femur compression duration", "String");
                            }
                            else if (dummy[i][lr].chart_label == "Passenger") // Passenger
                            {
                                // No exceedance, so constant datums, constant variables, and so on
                                draw_constant_datums(MPDB.FEMUR_COM_PASS_GOOD, MPDB.FEMUR_COM_PASS_ADEQUATE, MPDB.FEMUR_COM_PASS_MARGINAL, MPDB.FEMUR_COM_PASS_WEAK); 
                            
                                // Create image/curve of femur compression curves        
                                c_com.AddToGraph();
                                write_image(output_data.com.title, output_data.com.fname, c_com, MPDB.FEMUR_COM_PASS_WEAK);                      
                            
                                output_data.com.curveList.push(c_com.id);
                                write_curve("cur", output_data.com.curveList, output_data.com.fname);
                                write_curve("csv", output_data.com.curveList, output_data.com.fname);
        
                                // Remove all curves and datums        
                                remove_all_curves_and_datums(); 
        
                                // Calc values        
                                var com_results = get_constant_limit_results(c_com, MPDB.FEMUR_COM_PASS_GOOD, MPDB.FEMUR_COM_PASS_WEAK, MPDB.KNEE_FEMUR_PELVIS_LO_SCORE, MPDB.KNEE_FEMUR_PELVIS_HI_SCORE);
                            
                                // Write compression variables        
                                write_variable(f, dummy[i][lr].variable_label.toUpperCase() + "_" + lr.toUpperCase() + "_FEMUR_COMPRESSION_NON_EXC_SCORE",          com_results[0].toFixed(3), "Femur compression score",    "String");
                                // probably the below row is useless 
                                write_variable(f, dummy[i][lr].variable_label.toUpperCase() + "_" + lr.toUpperCase() + "_FEMUR_COMPRESSION_FORCE",          com_results[1].toFixed(3), "Femur compression force",    "String");
                            
                                // Variables below are only for avoid REPORTER log errors      
                                write_variable(f, dummy[i][lr].variable_label.toUpperCase() + "_" + lr.toUpperCase() + "_FEMUR_COMPRESSION_SCORE",          "-", "Femur compression score not used in C-NCAP",    "String");
                                write_variable(f, dummy[i][lr].variable_label.toUpperCase() + "_" + lr.toUpperCase() + "_FEMUR_COMPRESSION_LEVEL_EXCEEDED", "-", "Femur compression level not used in C-NCAP",    "String");
                                write_variable(f, dummy[i][lr].variable_label.toUpperCase() + "_" + lr.toUpperCase() + "_FEMUR_COMPRESSION_DURATION",       "-", "Femur compression duration not used in C-NCAP", "String");
                            }
                            break;

                        default:
                            ErrorMessage("Unexpected regulation \""+ template_reg + "\" in do_knee_femur_pelvis_rating_calc femur calculation.");
                            return;
                            break;
                    }   
                }
            }
            else
            {
                // No beam defined - variables should be set to blank by first script in template.
                // Create a blank image to let the user know.
                write_blank_images(output_data, "NO BEAM ID DEFINED FOR");
            }


            // Knee slide
            var output_data = {};

            output_data.disp = new OutputData(dummy[i][lr].chart_label + " " + lr + " Knee Displacement", images_dir + "/" + dummy[i][lr].variable_label + "_" + lr + "_Knee_Displacement");

            // If valid spring ID defined and if not C-NCAP Passenger (knee not assessed for C-NCAP Passenger)
            if(dummy[i][lr].spring_id != undefined && !(dummy[i][lr].chart_label == "Passenger" && template_reg == "CNCAP"))
            {
                // Read in spring elongation force             
                // For Humanetics dummies this is measured from a Spring;
                // for LSTC dummies this is measured from a Translational Joint
                switch (dummy[i][lr].version)
                {
                    case "V6_HUMANETICS_HIII_50TH_MALE":
                    case "V7_HUMANETICS_HIII_50TH_MALE":
                    case "V8_HUMANETICS_HIII_50TH_MALE":
                    case "Humanetics HIII 50M v1.5" :
                    case "Humanetics THOR 50M v1.8":
                    case "Humanetics THOR 50M v1.9":
                        var c_disp = read_data("SPRING TR", 0, dummy[i][lr].spring_id, "ET");  
                        break;

                    case "LSTC HIII 50M Detailed v190217_beta":
                    case "LSTC HIII 50M Fast V2.0":
                    case "LSTC HIII 50M v130528_beta":
                        var c_disp = read_data("JOINT TR", 0, dummy[i][lr].spring_id, "XD");  //
                        var entity_str = "SECTION"; // Used for error message below        
                        break;

                    default:
                        ErrorMessage("Unsupported dummy \""+dummy[i][lr].version+"\" in do_knee_femur_pelvis_rating_calc for elongation force.");
                        write_blank_images(output_data, "UNSUPPORTED DUMMY \""+dummy[i][lr].version+"\"");
                        return;                       
                        break;
                }
                // Check that the curve read in - i.e. the spring id and component are valid
                // Create blank image to let the user know.

                if(!c_disp)
                {
                    write_blank_images(output_data, "NO DATA FOR " + entity_str + " " + dummy[i][lr].beam_id); // better change to "dummy[i][lr].spring_id", but should be okay
                }
                else
                {   
                    // Knee of EuroNCAP, Driver and Passenger use the same limits
                    // Knee of CNCAP, only Driver needs to be assessed
                    switch(template_reg)
                    {
                        case "EuroNCAP":
                            // Convert to mm
                            var c_disp_mm = Operate.Div(c_disp, oGlblData.mm_factor);

                            // C180 filter (convert to seconds first)
                            var c_disp_c180 = convert_to_s_and_filter(c_disp_mm, Operate.C180, MPDB.REG_DT);

                            // Remove all curves and datums
                            remove_all_curves_and_datums();

                            // Convert from seconds back to model time
                            c_disp_c180 = Operate.Mux(c_disp_c180, oGlblData.time_factor);

                            // Set labels and style
                            set_labels(c_disp_c180, dummy[i][lr].chart_label + " " + lr + " Knee Displacement", "Time (" + oGlblData.unit_time + ")", "Displacement (mm)");
                            set_line_style(c_disp_c180, Colour.BLACK);

                            // Remove all curves and datums
                            remove_all_curves_and_datums();

                            // Draw knee displacement datums
                            draw_constant_datums(MPDB.KNEE_DISP_GOOD, MPDB.KNEE_DISP_ADEQUATE, MPDB.KNEE_DISP_MARGINAL, MPDB.KNEE_DISP_WEAK);

                            // Create image/curve of femur compression curves
                            c_disp_c180.AddToGraph();
                            write_image(output_data.disp.title, output_data.disp.fname, c_disp_c180, MPDB.KNEE_DISP_WEAK);

                            output_data.disp.curveList.push(c_disp_c180.id);
                            write_curve("cur", output_data.disp.curveList, output_data.disp.fname);
                            write_curve("csv", output_data.disp.curveList, output_data.disp.fname);

                            // Remove all curves and datums
                            remove_all_curves_and_datums(); 

                            // Calc values
                            var knee_disp_results = get_constant_limit_results(c_disp_c180, MPDB.KNEE_DISP_GOOD, MPDB.KNEE_DISP_WEAK, MPDB.KNEE_FEMUR_PELVIS_LO_SCORE, MPDB.KNEE_FEMUR_PELVIS_HI_SCORE);

                            // Write displacement variables
                            write_variable(f, dummy[i][lr].variable_label.toUpperCase() + "_" + lr.toUpperCase() + "_KNEE_DISPLACEMENT_SCORE", knee_disp_results[0].toFixed(3), "Knee displacement score", "String");
                            write_variable(f, dummy[i][lr].variable_label.toUpperCase() + "_" + lr.toUpperCase() + "_KNEE_DISPLACEMENT",       knee_disp_results[1].toFixed(3), "Knee displacement",       "String");
                            break;

                        case "CNCAP":
                            // Convert to mm
                            var c_disp_mm = Operate.Div(c_disp, oGlblData.mm_factor);

                            // C180 filter (convert to seconds first)
                            var c_disp_c180 = convert_to_s_and_filter(c_disp_mm, Operate.C180, MPDB.REG_DT);

                            // Remove all curves and datums
                            remove_all_curves_and_datums();

                            // Convert from seconds back to model time
                            c_disp_c180 = Operate.Mux(c_disp_c180, oGlblData.time_factor);

                            // Set labels and style
                            set_labels(c_disp_c180, dummy[i][lr].chart_label + " " + lr + " Knee Displacement", "Time (" + oGlblData.unit_time + ")", "Displacement (mm)");
                            set_line_style(c_disp_c180, Colour.BLACK);

                            // Remove all curves and datums
                            remove_all_curves_and_datums();

                            // Draw knee displacement datums
                            draw_constant_datums(MPDB.KNEE_DISP_GOOD, MPDB.KNEE_DISP_ADEQUATE, MPDB.KNEE_DISP_MARGINAL, MPDB.KNEE_DISP_WEAK);

                            // Create image/curve of femur compression curves
                            c_disp_c180.AddToGraph();
                            write_image(output_data.disp.title, output_data.disp.fname, c_disp_c180, MPDB.KNEE_DISP_WEAK);

                            output_data.disp.curveList.push(c_disp_c180.id);
                            write_curve("cur", output_data.disp.curveList, output_data.disp.fname);
                            write_curve("csv", output_data.disp.curveList, output_data.disp.fname);

                            // Remove all curves and datums
                            remove_all_curves_and_datums(); 

                            // Calc values
                            var knee_disp_results = get_constant_limit_results(c_disp_c180, MPDB.KNEE_DISP_GOOD, MPDB.KNEE_DISP_WEAK, MPDB.KNEE_FEMUR_PELVIS_LO_SCORE, MPDB.KNEE_FEMUR_PELVIS_HI_SCORE);

                            // Write displacement variables
                            write_variable(f, dummy[i][lr].variable_label.toUpperCase() + "_" + lr.toUpperCase() + "_KNEE_DISPLACEMENT_SCORE", knee_disp_results[0].toFixed(3), "Knee displacement score", "String");
                            write_variable(f, dummy[i][lr].variable_label.toUpperCase() + "_" + lr.toUpperCase() + "_KNEE_DISPLACEMENT",       knee_disp_results[1].toFixed(3), "Knee displacement",       "String");
                            break; 
                        
                        default:
                            ErrorMessage("Unexpected regulation \""+ template_reg + "\" in do_knee_femur_pelvis_rating_calc knee calculation.");
                            return;
                            break;  
                    }
                }
            }
            else
            {
                // CNCAP Knee spring IDs for passengers are not defined because it doesn't need be assessed
                // so blank figures are not needed here
                if(dummy[i][lr].chart_label == "Passenger" && template_reg == "CNCAP")
                {
                    // Variables are created to avoid REPORTER log errors
                    write_variable(f, dummy[i][lr].variable_label.toUpperCase() + "_" + lr.toUpperCase() + "_KNEE_DISPLACEMENT_SCORE", "-", "Knee displacement score not used in C-NCAP", "String");
                    write_variable(f, dummy[i][lr].variable_label.toUpperCase() + "_" + lr.toUpperCase() + "_KNEE_DISPLACEMENT",       "-", "Knee displacement not used in C-NCAP",       "String");
                }
                else if(!(dummy[i][lr].chart_label == "Passenger" && template_reg == "CNCAP"))
                {
                    write_blank_images(output_data, "NO SPRING ID DEFINED FOR");
                }
            }

            // Acetabulum compression
            
            var output_data = {};

			output_data.com = new OutputData("Driver " + lr + " Acetabulum compression", images_dir + "/" + "Driver_" + lr + "_Acetabulum_Compression");
            
            if (i == "driver") // Only Driver for both CNCAP and EuroNCAP needs to assess
            {
                if(dummy[i][lr].beam_2_id != undefined)
                {
                    // For THOR Dummy, Read in Section X, Y, Z forces,
                    var c_com_x = read_data("SECTION", 0, dummy[i][lr].beam_2_id, "FX");  
                    var c_com_y = read_data("SECTION", 0, dummy[i][lr].beam_2_id, "FY");  
                    var c_com_z = read_data("SECTION", 0, dummy[i][lr].beam_2_id, "FZ"); 
                    var entity_str = "SECTION"; // Used for error message below

                    // The resultant force is
                    var c_com_res = Operate.Vec(c_com_x, c_com_y, c_com_z);

                    // For THOR Dummy, further calculation is needed
                    // For left pelvis,     when Fx>0, c_com = 1 * c_com_res; when Fx <= 0, c_com = 0 * c_com_res = 0;
                    // For right pelvis,    when Fx<0, c_com = 1 * c_com_res; when Fx >= 0, c_com = 0 * c_com_res = 0;
                    var c_com = new Curve(Curve.FirstFreeID());
                    if(lr == "Left")
                    {
                        for (var k = 1; k <= c_com_x.npoints; k++)
                        {
                            var com_x_data = c_com_x.GetPoint(k);
                            var com_res_data = c_com_res.GetPoint(k);
                            if(com_x_data[1] > 0)
                            {
                                c_com.AddPoint(com_x_data[0], com_res_data[1]);
                            }
                            else
                            {
                                c_com.AddPoint(com_x_data[0], 0);
                            }

                        }
                    }
                    else if(lr == "Right")
                    {
                        for (var k = 1; k <= c_com_x.npoints; k++)
                        {
                            var com_x_data = c_com_x.GetPoint(k);
                            var com_res_data = c_com_res.GetPoint(k);
                            if(com_x_data[1] < 0)
                            {
                                c_com.AddPoint(com_x_data[0], com_res_data[1]);
                            }
                            else
                            {
                                c_com.AddPoint(com_x_data[0], 0);
                            }

                        }
                    }

                    // Check that the curve read in - i.e. the beam id and component are valid
                    // Create blank image to let the user know.

                    if(!c_com)
                    {
                        write_blank_images(output_data, "NO DATA FOR BEAM " + entity_str + " " + dummy[i][lr].beam_2_id);
                    }
                    else
                    {
                        // Convert forces to kN from model units
                        var c_com_kn = Operate.Div(c_com, oGlblData.kn_factor);

                        // C600 filter (convert to seconds first)
                        var c_com_c600 = convert_to_s_and_filter(c_com_kn, Operate.C600, MPDB.REG_DT);

                        // Remove all curves and datums
                        remove_all_curves_and_datums();

                        // Convert from seconds back to model time
                        c_com_adj = Operate.Mux(c_com_c600, oGlblData.time_factor);

                        // Set labels and style
                        set_labels(c_com_adj, dummy[i][lr].chart_label + " Acetabulum compression", "Time (" + oGlblData.unit_time + ")", "Compression (kN)");
                        set_line_style(c_com_adj, Colour.BLACK);

                        // Remove all curves and datums
                        remove_all_curves_and_datums();

                        // Draw aetabulum compression datum
                        draw_constant_datums(MPDB.ACETABULUM_COM_GOOD, MPDB.ACETABULUM_COM_ADEQUATE, MPDB.ACETABULUM_COM_MARGINAL, MPDB.ACETABULUM_COM_WEAK);

                        // Create image and curve files of aetabulum compression curves
                        c_com_adj.AddToGraph();
                        write_image(output_data.com.title, output_data.com.fname, c_com_adj, MPDB.ACETABULUM_COM_WEAK);

                        output_data.com.curveList.push(c_com_adj.id);
                        write_curve("cur", output_data.com.curveList, output_data.com.fname);
                        write_curve("csv", output_data.com.curveList, output_data.com.fname);

                        // Remove all curves and datums
                        remove_all_curves_and_datums(); 

                        // Calc values
                        var com_results = get_constant_limit_results(c_com_adj, MPDB.ACETABULUM_COM_GOOD, MPDB.ACETABULUM_COM_WEAK, MPDB.KNEE_FEMUR_PELVIS_LO_SCORE, MPDB.KNEE_FEMUR_PELVIS_HI_SCORE);

                        // Write compression variables
                        write_variable(f, "DRIVER_" + lr.toUpperCase() + "_ACETABULUM_COMPRESSION_SCORE", com_results[0].toFixed(3),  "Acetabulum compression score",    "String");
                        write_variable(f, "DRIVER_" + lr.toUpperCase() + "_ACETABULUM_COMPRESSION", com_results[1].toFixed(3),  "Acetabulum compression level",    "String");
                    }
                }
                else
                {
                    // No beam defined - variables should be set to blank by first script in template.
                    // Create a blank image to let the user know.
                    write_blank_images(output_data, "NO BEAM ID DEFINED FOR");
                }
            }
        }
    }
}



function do_lower_leg_foot_and_ankle_rating_calc(f)
{   // Calculates the lower_leg, foot and ankle scores for both driver and passenger and writes
    // relevant values into the Reporter csv variables file <f>.

    // Two passes, one for driver, one for passenger.
    // And for each one do the left and right hand side
    // And for each one do upper and lower tibia
    var dummy = {};
    dummy.driver = {};
    dummy.driver.Left = {};
    dummy.driver.Left.Lower = new DummyData("Driver", oGlblData.driver_left_lower_tibia_loadcell, "beam", oGlblData.driver_dummy, "version");
    dummy.driver.Left.Upper = new DummyData("Driver", oGlblData.driver_left_upper_tibia_loadcell, "beam", oGlblData.driver_dummy, "version");
    dummy.driver.Right = {};
    dummy.driver.Right.Lower = new DummyData("Driver", oGlblData.driver_right_lower_tibia_loadcell, "beam", oGlblData.driver_dummy, "version");
    dummy.driver.Right.Upper = new DummyData("Driver", oGlblData.driver_right_upper_tibia_loadcell, "beam", oGlblData.driver_dummy, "version");
    dummy.passenger = {};
    dummy.passenger.Left = {};
    dummy.passenger.Left.Lower = new DummyData("Passenger", oGlblData.passenger_left_lower_tibia_loadcell, "beam", oGlblData.passenger_dummy, "version");
    dummy.passenger.Left.Upper = new DummyData("Passenger", oGlblData.passenger_left_upper_tibia_loadcell, "beam", oGlblData.passenger_dummy, "version");
    dummy.passenger.Right = {};
    dummy.passenger.Right.Lower = new DummyData("Passenger", oGlblData.passenger_right_lower_tibia_loadcell, "beam", oGlblData.passenger_dummy, "version");
    dummy.passenger.Right.Upper = new DummyData("Passenger", oGlblData.passenger_right_upper_tibia_loadcell, "beam", oGlblData.passenger_dummy, "version");  


    for(var i in dummy) // loop through driver/passenger for EuroNCAP; only driver for CNCAP
    {
        if(!(template_reg == "CNCAP" && i == "passenger")) // CNCAP Passenger does NOT need assess tibia
        {      
            for(var lr in dummy[i]) // loop through left/right side of dummy
            {
                var com_upper_results;
                var com_lower_results;

                for(var ul in dummy[i][lr]) // loop through lower/upper tibia
                {
                    var output_data = {};
                    output_data.com = new OutputData(dummy[i][lr][ul].chart_label + " " + lr + " " + ul + " Tibia Compression", images_dir + "/" + dummy[i][lr][ul].variable_label + "_" + lr + "_" + ul + "_Tibia_Compression");
                    output_data.ti  = new OutputData(dummy[i][lr][ul].chart_label + " " + lr + " " + ul + " Tibia Index",       images_dir + "/" + dummy[i][lr][ul].variable_label + "_" + lr + "_" + ul + "_Tibia_Index");

                    if(dummy[i][lr][ul].beam_id != undefined)
                    {
                        // Read in beam longitudinal force, MX and MY
                        // For Humanetics dummies this is measured from a Beam;
                        // for LSTC dummies this is measured from a Section

                        switch (dummy[i][lr][ul].version)
                        {
                            case "V6_HUMANETICS_HIII_50TH_MALE":
                            case "V7_HUMANETICS_HIII_50TH_MALE":
                            case "V8_HUMANETICS_HIII_50TH_MALE":
                            case "Humanetics HIII 50M v1.5":
                            case "Humanetics HIII 50M v1.5.1":
                                var c_com   = read_data("BEAM BASIC", 0, dummy[i][lr][ul].beam_id, "NZ");  // NZ is longitudinal
                                var c_mom_x = read_data("BEAM BASIC", 0, dummy[i][lr][ul].beam_id, "MX");
                                var c_mom_y = read_data("BEAM BASIC", 0, dummy[i][lr][ul].beam_id, "MY");
                                var entity_str = "BEAM"; // Used for error message below
                                break;

                            case "LSTC HIII 50M Detailed v190217_beta":
                            case "LSTC HIII 50M Fast V2.0":
                            case "LSTC HIII 50M v130528_beta":
                            case "Humanetics THOR 50M v1.8":
                            case "Humanetics THOR 50M v1.9":
                                var c_com   = read_data("SECTION", 0, dummy[i][lr][ul].beam_id, "FZ");  // FZ is longitudinal
                                var c_mom_x = read_data("SECTION", 0, dummy[i][lr][ul].beam_id, "MX");
                                var c_mom_y = read_data("SECTION", 0, dummy[i][lr][ul].beam_id, "MY");
                                var entity_str = "SECTION"; // Used for error message below
                                break;

                            default:
                                ErrorMessage("Unsupported dummy \""+dummy[i][lr][ul].version+"\" in do_lower_leg_foot_and_ankle_rating_calc.");
                                write_blank_images(output_data, "UNSUPPORTED DUMMY \""+dummy[i][lr][ul].version+"\"");
                                return;
                                break;
                        }

                        // Check that the curves read in - i.e. the beam id and components are valid
                        // Create blank images to let the user know.

                        if(!c_com || !c_mom_x || !c_mom_y)
                        {   
                            write_blank_images(output_data, "NO DATA FOR " + entity_str + " " + dummy[i][lr][ul].beam_id);
                        }
                        else
                        {
                            // Convert forces to kN (dividing by -ve number so compression is +ve)
                            //         moments to Nm from model units
                            var c_com_kn   = Operate.Div(c_com, -oGlblData.kn_factor);
                            var c_mom_x_nm = Operate.Div(c_mom_x, oGlblData.nm_factor);
                            var c_mom_y_nm = Operate.Div(c_mom_y, oGlblData.nm_factor);         

                            // C600 filter (convert to seconds first)
                            var c_com_c600   = convert_to_s_and_filter(c_com_kn,   Operate.C600, MPDB.REG_DT);
                            var c_mom_x_c600 = convert_to_s_and_filter(c_mom_x_nm, Operate.C600, MPDB.REG_DT);
                            var c_mom_y_c600 = convert_to_s_and_filter(c_mom_y_nm, Operate.C600, MPDB.REG_DT);

                            // Calculate resultant moment
                            var c_mom_res = Operate.Vec2d(c_mom_x_c600, c_mom_y_c600);

                            // Remove all curves and datums
                            remove_all_curves_and_datums();

                            // Convert from seconds back to model time
                            c_com_c600 = Operate.Mux(c_com_c600, oGlblData.time_factor);
                            c_mom_res  = Operate.Mux(c_mom_res,  oGlblData.time_factor);

                            //
                            // Calculate Tibia index = |Mr/Mrc| + |Fz/Fzc|
                            //
                            //  Where Mr  = Resultant moment
                            //        Mrc = Critical resultant moment = 225Nm
                            //        Fz  = Compression force
                            //        Fzc = Critical compressive force = 35.9kN
                            //
                            var curve_id = Curve.FirstFreeID();

                            var c_ti = new Curve(curve_id);
                            var n = c_mom_res.npoints;

                            for(var j = 1; j <= n; j++)
                            {
                                var p_com = c_com_c600.GetPoint(j);
                                var p_mom = c_mom_res.GetPoint(j);

                                var time = p_mom[0];
                                var ti   = Math.abs(p_mom[1]/MPDB.TIBIA_MRC) + Math.abs(p_com[1]/MPDB.TIBIA_FZC);

                                c_ti.AddPoint(time, ti);
                            }

                            // Set labels and style
                            set_labels(c_com_c600, dummy[i][lr][ul].chart_label + " " + lr + " " + ul + " Tibia Compression", "Time (" + oGlblData.unit_time + ")", "Compression (kN)");
                            set_labels(c_ti,       dummy[i][lr][ul].chart_label + " " + lr + " " + ul + " Tibia Index",       "Time (" + oGlblData.unit_time + ")", "Tibia Index");

                            set_line_style(c_com_c600, Colour.BLACK);
                            set_line_style(c_ti, Colour.BLACK);

                            // Remove all curves and datums
                            remove_all_curves_and_datums();

                            // Draw Tibia compression datums
                            draw_constant_datums(MPDB.TIBIA_COM_GOOD, MPDB.TIBIA_COM_ADEQUATE, MPDB.TIBIA_COM_MARGINAL, MPDB.TIBIA_COM_WEAK);

                            // Create image/curve of tibia compresion curves
                            c_com_c600.AddToGraph();
                            write_image(output_data.com.title, output_data.com.fname, c_com_c600, MPDB.TIBIA_COM_WEAK);

                            output_data.com.curveList.push(c_com_c600.id);
                            write_curve("cur", output_data.com.curveList, output_data.com.fname);
                            write_curve("csv", output_data.com.curveList, output_data.com.fname);

                            // Remove all curves and datums
                            remove_all_curves_and_datums(); 

                            // Draw Tibia index datums
                            draw_constant_datums(MPDB.TIBIA_INDEX_GOOD, MPDB.TIBIA_INDEX_ADEQUATE, MPDB.TIBIA_INDEX_MARGINAL, MPDB.TIBIA_INDEX_WEAK);

                            // Create image/curve of tibia index curves
                            c_ti.AddToGraph();
                            write_image(output_data.ti.title, output_data.ti.fname, c_ti, MPDB.TIBIA_INDEX_WEAK);

                            output_data.ti.curveList.push(c_ti.id);
                            write_curve("cur", output_data.ti.curveList, output_data.ti.fname);
                            write_curve("csv", output_data.ti.curveList, output_data.ti.fname);

                            // Remove all curves and datums
                            remove_all_curves_and_datums(); 

                            // Calc values
                            if(ul == "Upper") com_upper_results = get_constant_limit_results(c_com_c600, MPDB.TIBIA_COM_GOOD, MPDB.TIBIA_COM_WEAK, MPDB.LOWER_LEG_FOOT_AND_ANKLE_LO_SCORE, MPDB.LOWER_LEG_FOOT_AND_ANKLE_HI_SCORE);
                            else              com_lower_results = get_constant_limit_results(c_com_c600, MPDB.TIBIA_COM_GOOD, MPDB.TIBIA_COM_WEAK, MPDB.LOWER_LEG_FOOT_AND_ANKLE_LO_SCORE, MPDB.LOWER_LEG_FOOT_AND_ANKLE_HI_SCORE);

                            var ti_results = get_constant_limit_results(c_ti, MPDB.TIBIA_INDEX_GOOD, MPDB.TIBIA_INDEX_WEAK, MPDB.LOWER_LEG_FOOT_AND_ANKLE_LO_SCORE, MPDB.LOWER_LEG_FOOT_AND_ANKLE_HI_SCORE);

                            // Write tibia index variables - and store on <oGlblData> object
                            write_variable(f, dummy[i][lr][ul].variable_label.toUpperCase() + "_" + lr.toUpperCase() + "_" + ul.toUpperCase() + "_TIBIA_INDEX_SCORE", ti_results[0].toFixed(3), "Tibia index score", "String");
                            write_variable(f, dummy[i][lr][ul].variable_label.toUpperCase() + "_" + lr.toUpperCase() + "_" + ul.toUpperCase() + "_TIBIA_INDEX",       ti_results[1].toFixed(3), "Tibia index",       "String");

                            // Write compression variables to file and store in <oGlblData> object - worst out of upper and lower
                            if(com_upper_results != undefined && com_lower_results != undefined)
                            {
                                var tibia_com_score = Math.min(com_upper_results[0], com_lower_results[0])
                                write_variable(f, dummy[i][lr][ul].variable_label.toUpperCase() + "_" + lr.toUpperCase() + "_TIBIA_COMPRESSION_SCORE", tibia_com_score.toFixed(3), "Tibia compression score", "String");
                                var value = Math.max(com_upper_results[1], com_lower_results[1]).toFixed(3);
                                write_variable(f, dummy[i][lr][ul].variable_label.toUpperCase() + "_" + lr.toUpperCase() + "_TIBIA_COMPRESSION", value, "Tibia compression value", "String");
                            }
                        }
                    }
                    else
                    {
                        // No beam defined - variables should be set to blank by first script in template.
                        // Create blank images to let the user know.
                        write_blank_images(output_data, "NO ID DEFINED FOR TIBIA");
                    }
                }
            }
        }
        else if(template_reg == "CNCAP" && i == "passenger") // CNCAP Passenger does NOT need assess tibia, but dummy variables are created to avoid REPORTER log errors
        {
            for(var lr in dummy[i]) // loop through left/right side of dummy
            {
                for(var ul in dummy[i][lr]) // loop through lower/upper tibia
                {
                    write_variable(f, dummy[i][lr][ul].variable_label.toUpperCase() + "_" + lr.toUpperCase() + "_" + ul.toUpperCase() + "_TIBIA_INDEX_SCORE", "-", "Tibia index score not used in C-NCAP", "String");
                    write_variable(f, dummy[i][lr][ul].variable_label.toUpperCase() + "_" + lr.toUpperCase() + "_" + ul.toUpperCase() + "_TIBIA_INDEX", "-", "Tibia index not used in C-NCAP",       "String");
                    write_variable(f, dummy[i][lr][ul].variable_label.toUpperCase() + "_" + lr.toUpperCase() + "_TIBIA_COMPRESSION_SCORE", "-", "Tibia compression score not used in C-NCAP", "String");
                    write_variable(f, dummy[i][lr][ul].variable_label.toUpperCase() + "_" + lr.toUpperCase() + "_TIBIA_COMPRESSION", "-", "Tibia compression value not used in C-NCAP", "String");
                }
            }
        }
    }

    // Calculate pedal vertical and fore/aft displacements to get modifier - for driver only
    // Do 6 passes - acceleration, brake and clutch pedal for vertical and fore/aft
    var accn_vert_disp   = 0.0;
    var brake_vert_disp  = 0.0;
    var clutch_vert_disp = 0.0;

    var accn_fa_disp     = 0.0;
    var brake_fa_disp    = 0.0;
    var clutch_fa_disp   = 0.0;

    for(var sp = 0; sp < 6; sp++)
    {
        if(sp == 0)      spring = oGlblData.accn_pedal_intrusion_spring_z;   // Acceleration pedal vertical spring
        else if(sp == 1) spring = oGlblData.brake_pedal_intrusion_spring_z;  // Brake pedal vertical spring
        else if(sp == 2) spring = oGlblData.clutch_pedal_intrusion_spring_z; // Clutch pedal vertical spring
        else if(sp == 3) spring = oGlblData.accn_pedal_intrusion_spring_x;   // Acceleration  pedal fore/aft spring
        else if(sp == 4) spring = oGlblData.brake_pedal_intrusion_spring_x;  // Brake pedal fore/aft spring
        else if(sp == 5) spring = oGlblData.clutch_pedal_intrusion_spring_x; // Clutch pedal fore/aft spring

        var c_accn_vert,   c_accn_fa;
        var c_brake_vert,  c_brake_fa;
        var c_clutch_vert, c_clutch_fa;

        if(spring != undefined)
        {
            var c_el = read_data("SPRING TR", 0, spring, "ET");

            if(c_el)
            {
                // Convert to mm and take final value
                var c_el_mm = Operate.Div(c_el, oGlblData.mm_factor);
                var p = c_el_mm.GetPoint(c_el_mm.npoints);

                if(sp == 0)
                {
                    c_accn_vert = c_el_mm;
                    accn_vert_disp = p[1]; // the final value would be the vertical intrusion

                    set_labels(c_accn_vert, "Driver Acceleration Pedal Vertical Intrusion (+ve upward)", "Time (" + oGlblData.unit_time + ")", "Intrusion (mm)");
                    set_line_style(c_accn_vert, Colour.CYAN);
                }
                else if(sp == 1)
                {
                    c_brake_vert = c_el_mm;
                    brake_vert_disp = p[1]; // the final value would be the vertical intrusion

                    set_labels(c_brake_vert, "Driver Brake Pedal Vertical Intrusion (+ve upward)", "Time (" + oGlblData.unit_time + ")", "Intrusion (mm)");
                    set_line_style(c_brake_vert, Colour.BLUE);
                }
                else if(sp == 2)
                {
                    c_clutch_vert = c_el_mm;
                    clutch_vert_disp = p[1]; // the final value would be the vertical intrusion

                    set_labels(c_clutch_vert, "Driver Clutch Pedal Vertical Intrusion (+ve upward)", "Time (" + oGlblData.unit_time + ")", "Intrusion (mm)");
                    set_line_style(c_clutch_vert, Colour.MAGENTA);
                }
                else if(sp == 3)
                {
                    c_accn_fa = Operate.Mul(c_el_mm, -1.0);
                    var p = c_accn_fa.GetPoint(c_accn_fa.npoints);
                    accn_fa_disp = p[1];

                    set_labels(c_accn_fa, "Driver Acceleration Pedal Fore/Aft Intrusion (+ve rearward)", "Time (" + oGlblData.unit_time + ")", "Intrusion (mm)");
                    set_line_style(c_accn_fa, Colour.CYAN);
                }
                else if(sp == 4)
                {
                    c_brake_fa = Operate.Mul(c_el_mm, -1.0);
                    var p = c_brake_fa.GetPoint(c_brake_fa.npoints);
                    brake_fa_disp  = p[1];

                    set_labels(c_brake_fa, "Driver Brake Pedal Fore/Aft Intrusion (+ve rearward)", "Time (" + oGlblData.unit_time + ")", "Intrusion (mm)");
                    set_line_style(c_brake_fa, Colour.BLUE);
                }
                else if(sp == 5)
                {
                    c_clutch_fa = Operate.Mul(c_el_mm, -1.0);
                    var p = c_clutch_fa.GetPoint(c_clutch_fa.npoints);
                    clutch_fa_disp = p[1];  

                    set_labels(c_clutch_fa, "Driver Clutch Pedal Fore/Aft Intrusion (+ve rearward)", "Time (" + oGlblData.unit_time + ")", "Intrusion (mm)");
                    set_line_style(c_clutch_fa, Colour.MAGENTA);
                }
            }
        }
    }

    // Get maximum vertical and fore/aft displacement and calculate modifier

    var max_pedal_vert_disp = Math.max(accn_vert_disp, brake_vert_disp, clutch_vert_disp);
    var max_pedal_fa_disp   = Math.max(accn_fa_disp,   brake_fa_disp,   clutch_fa_disp);

    var pedal_vert_disp_mod = sliding_scale(max_pedal_vert_disp, 
            MPDB.LOWER_LEG_PEDAL_VERT_DISP_GOOD, MPDB.LOWER_LEG_PEDAL_VERT_DISP_WEAK,
            MPDB.LOWER_LEG_PEDAL_VERT_DISP_HI_SCORE, MPDB.LOWER_LEG_PEDAL_VERT_DISP_LO_SCORE);

    var pedal_fa_disp_score   = sliding_scale(max_pedal_fa_disp, 
            MPDB.LOWER_LEG_PEDAL_FA_DISP_GOOD, MPDB.LOWER_LEG_PEDAL_FA_DISP_WEAK,
            MPDB.LOWER_LEG_PEDAL_FA_DISP_HI_SCORE, MPDB.LOWER_LEG_PEDAL_FA_DISP_LO_SCORE);

    //var pedal_disp_mod = Math.min(pedal_vert_disp_mod, pedal_fa_disp_mod);

    // Remove all curves and datums
    remove_all_curves_and_datums();

    // Draw datums
    draw_constant_datums(MPDB.LOWER_LEG_PEDAL_VERT_DISP_GOOD, MPDB.LOWER_LEG_PEDAL_VERT_DISP_ADEQUATE, MPDB.LOWER_LEG_PEDAL_VERT_DISP_MARGINAL, MPDB.LOWER_LEG_PEDAL_VERT_DISP_WEAK);

    // Create image/curve of vertical intrusion curves
    if(c_accn_vert)   c_accn_vert.AddToGraph();
    if(c_brake_vert)  c_brake_vert.AddToGraph();
    if(c_clutch_vert) c_clutch_vert.AddToGraph();

    var curve_list = [];    // There is no OutputData object for this result, so the curve list is defined locally here
    if(c_accn_vert)   curve_list.push(c_accn_vert.id);
    if(c_brake_vert)  curve_list.push(c_brake_vert.id);
    if(c_clutch_vert) curve_list.push(c_clutch_vert.id);

    var title;
    // Considering sometimes there is no clutch padel, so do changes as below
    // if(c_accn_vert && c_brake_vert && c_clutch_vert) title = "Driver Vertical Pedal Intrusion";
    if(c_accn_vert && c_brake_vert) title = "Driver Vertical Pedal Intrusion";
    else                            title = "SOME SPRING IDs MISSING/DO NOT EXIST FOR Driver Vertical Pedal Intrusion";

    var curves = new Array(3);
    curves[0] = c_accn_vert;
    curves[1] = c_brake_vert;
    curves[2] = c_clutch_vert;

    write_image(title, images_dir + "/Driver_Vertical_Pedal_Intrusion", curves, MPDB.LOWER_LEG_PEDAL_VERT_DISP_WEAK);

    if(curve_list) write_curve("cur", curve_list, images_dir + "/Driver_Vertical_Pedal_Intrusion");
    if(curve_list) write_curve("csv", curve_list, images_dir + "/Driver_Vertical_Pedal_Intrusion");

    remove_all_curves_and_datums();

    // Draw datums
    draw_constant_datums(MPDB.LOWER_LEG_PEDAL_FA_DISP_GOOD, MPDB.LOWER_LEG_PEDAL_FA_DISP_ADEQUATE, MPDB.LOWER_LEG_PEDAL_FA_DISP_MARGINAL, MPDB.LOWER_LEG_PEDAL_FA_DISP_WEAK);

    // Create image/curve of fore/aft intrusion curves
    if(c_accn_fa)   c_accn_fa.AddToGraph();
    if(c_brake_fa)  c_brake_fa.AddToGraph();
    if(c_clutch_fa) c_clutch_fa.AddToGraph();

    var curve_list = [];    // There is no OutputData object for this result, so the curve list is defined locally here

    if(c_accn_fa)   curve_list.push(c_accn_fa.id);
    if(c_brake_fa)  curve_list.push(c_brake_fa.id);
    if(c_clutch_fa) curve_list.push(c_clutch_fa.id);

    var title;
    // Considering sometimes there is no clutch padel, so do changes as below
    // if(c_accn_fa && c_brake_fa && c_clutch_fa) title = "Driver Fore/Aft Pedal Intrusion";
    if(c_accn_fa && c_brake_fa) title = "Driver Fore/Aft Pedal Intrusion";
    else                        title = "SOME SPRING IDs MISSING/DO NOT EXIST FOR Driver Fore/Aft Pedal Intrusion";

    var curves = new Array(3);
    curves[0] = c_accn_fa;
    curves[1] = c_brake_fa;
    curves[2] = c_clutch_fa;

    write_image(title, images_dir + "/Driver_Fore_Aft_Pedal_Intrusion", curves, MPDB.LOWER_LEG_PEDAL_FA_DISP_WEAK);

    if(curve_list) write_curve("cur", curve_list, images_dir + "/Driver_Fore_Aft_Pedal_Intrusion");
    if(curve_list) write_curve("csv", curve_list, images_dir + "/Driver_Fore_Aft_Pedal_Intrusion");

    // Write variables

    write_variable(f, "DRIVER_ACCELERATION_PEDAL_VERTICAL_DISPLACEMENT", accn_vert_disp.toFixed(3),   "Acceleration pedal vertical displacement", "String");
    write_variable(f, "DRIVER_BRAKE_PEDAL_VERTICAL_DISPLACEMENT",        brake_vert_disp.toFixed(3),  "Brake pedal vertical displacement",        "String");
    if(c_clutch_vert != undefined)
    {
        write_variable(f, "DRIVER_CLUTCH_PEDAL_VERTICAL_DISPLACEMENT",   clutch_vert_disp.toFixed(3), "Clutch pedal vertical displacement",       "String");
    }
    else
    {
        write_variable(f, "DRIVER_CLUTCH_PEDAL_VERTICAL_DISPLACEMENT",   "", "Clutch pedal vertical displacement",       "String");  
    }
    write_variable(f, "DRIVER_ACCELERATION_PEDAL_FA_DISPLACEMENT",       accn_fa_disp.toFixed(3),     "Acceleration pedal fore/aft displacement", "String");
    write_variable(f, "DRIVER_BRAKE_PEDAL_FA_DISPLACEMENT",              brake_fa_disp.toFixed(3),    "Brake pedal fore/aft displacement",        "String");
    if(c_clutch_fa != undefined)
    {
        write_variable(f, "DRIVER_CLUTCH_PEDAL_FA_DISPLACEMENT",         clutch_fa_disp.toFixed(3),   "Clutch pedal fore/aft displacement",       "String");
    }
    else
    {
        write_variable(f, "DRIVER_CLUTCH_PEDAL_FA_DISPLACEMENT",         "",   "Clutch pedal fore/aft displacement",       "String");   
    }
    //write_variable(f, "DRIVER_PEDAL_DISPLACEMENT_MOD",                   pedal_disp_mod.toFixed(3),   "Pedal displacement modifier",              "String");
    write_variable(f, "DRIVER_MAX_PEDAL_VERTICAL_DISPLACEMENT",          max_pedal_vert_disp.toFixed(3), "Maximum pedal vertical displacement",      "String");
    write_variable(f, "DRIVER_MAX_PEDAL_VERTICAL_DISPLACEMENT_MOD",      pedal_vert_disp_mod.toFixed(3), "Maximum pedal vertical modifier",          "String");
    write_variable(f, "DRIVER_MAX_PEDAL_FA_DISPLACEMENT",          max_pedal_fa_disp.toFixed(3),   "Maximum pedal fore/aft displacement",      "String");
    
    // For EuroNCAP, the front/rear displacement should be socre; but for CNCAP, should be "_Modifier", but here using the same variable name
    write_variable(f, "DRIVER_MAX_PEDAL_FA_DISPLACEMENT_SCORE",    pedal_fa_disp_score.toFixed(3), "Maximum pedal fore/aft score",             "String");
}



function convert_variable_datum_times()
{
    // Convert the variable datum times from seconds to the model units
    // Related to differences between EuroNCAP and CNCAP, especially CNCAP doesn't need consider exceedance for NECK shear and tension
    switch (template_reg)
    {
        case "EuroNCAP":
            for(var i=0; i<MPDB.NECK_SHR_PASS_EXC_TIME.length;  i++) MPDB.NECK_SHR_PASS_EXC_TIME[i]  *= oGlblData.time_factor;
            for(var i=0; i<MPDB.NECK_TEN_PASS_EXC_TIME.length;  i++) MPDB.NECK_TEN_PASS_EXC_TIME[i]  *= oGlblData.time_factor;
            for(var i=0; i<MPDB.FEMUR_COM_EXC_TIME.length; i++) MPDB.FEMUR_COM_EXC_TIME[i] *= oGlblData.time_factor;
            break;

        case "CNCAP":
            // Convert the variable datum times from seconds to the model units
            for(var i=0; i<MPDB.FEMUR_COM_EXC_TIME.length; i++) MPDB.FEMUR_COM_EXC_TIME[i] *= oGlblData.time_factor;
            break;

        default:
            ErrorMessage("function convert_variable_datum_times have errors");
            return;
            break;    
    }
}
