function GetBarrierPartLists()
// Sort barrier part names to identify IDs of backplate, honeycomb, and parallel plates
// Hardcoded with part names and numbers corresponding to the Arup Cellbond MPDB Shell Model:
// https://www.oasys-software.com/dyna/models/arup-cellbond-barrier-models/
// And part names for the LSTC MPDB Model (v3.0): https://lsdyna.ansys.com/lstc-barrier-models/
// To use a different barrier model, enter part names or numbers corresponding to current model in the region marked USER INPUT as an additional case for each switch statement
// Mark the appropriate option in lines 15-16 below.
// Select the barrier model to use by setting the barrierModel variable on line 17. 
{
    var backplate; var hc; var plates; var excluded; var seed;
    
    ////////////////////////
    // USER INPUT
    
    var option1 = true;   // user will define parts by name (requires .ztf file)
    var option2 = false;  // user will define parts by number
    var barrierModel = 'Arup-Cellbond';     // barrier model to use, can be 'Arup-Cellbond' or 'LSTC' or user defined
    
    // -------- OPTION 1: Define parts by name
    // NOTE: This option requires that a .ztf file is included in the folder alongside the solve results.
    
    if (option1)
    {
        // Name of part that defines rigid backplate
        // NOTE: Only one part will be accepted. If more than one part is named, the script will close with an error.
        switch (barrierModel) {
            case 'Arup-Cellbond':
                backplate = ["Backplate_grid"];
                break;
            case 'LSTC':
                backplate = ["Rear_Mounting_Wall"];
                break;
        }
        
        // Name(s) of parts that define honeycomb elements (i.e. shell normals are parallel to plane of backplate)
        switch (barrierModel) {
            case 'Arup-Cellbond':
                hc = ["Block A single thickness", "Block A double thickness", "Block B single thickness", "Block B double thickness",
                "Block C single thickness", "Block C double thickness", "Adhesive Block A single thickness", "Adhesive Block A double thickness",
                "Adhesive Block B single thickness", "Adhesive Block B double thickness", "Adhesive Block C double thickness", 
                "Adhesive Block C single thickness", "1.7mm adhesive  HIGH STRENGTH", "1.15mm adhesive",  
                "Adhesive Block C double thickness STRONGER", "Adhesive Block C single thickness STRONGER", "1.15mm adhesive  HIGH STRENGTH"];
                break;
            case 'LSTC':
                hc = ["1t-Front_Core_Horizontal", "2t-Front_Core_Vertical", "1t-Middle_Core_Down_Horizontal", "2t-Middle_Core_Down_Vertical",
                "2t-Rear_Middle_Core_Up_Vertical", "1t-Rear_Core_Horizontal",
                "Adhesive_FRONT_BLOCK_TO_CONTACT_PLATE", "Adhesive_FRONT_BLOCK_TO_FRONT_PLATE",
                "Adhesive_MIDDLE_BLOCK_TO_FRONT_PLATE", "Adhesive_MIDDLE_BLOCK_TO_MIDDLE_PLATE",
                "Adhesive_REAR_BLOCK_TO_MIDDLE_PLATE", "Adhesive_REAR_BLOCK_TO_REAR_PLATE"]; 
                break;
        }
        
        // Name(s) of parts that define intermediate plates parallel to backplate
        switch (barrierModel) {
            case 'Arup-Cellbond':
                plates = ["Intermediate plate front", "Intermediate plate rear", "Trolley-MountPlate-Rigid", "Backplate",
                "Contact plate OVERLAPPING LAYER1", "Cladding plate OVERLAPPING LAYER1: Front"];
                break;
            case 'LSTC':
                plates = ["Front_Plate", "Middle_Plate", "Outer_Cladding_Thick_Shell"];
                break;
        }
        
        // Name(s) of parts that are connected by tied contact to other parts named elsewhere and are therefore redundant
        switch (barrierModel) {
            case 'Arup-Cellbond':
                excluded = ["Contact plate OVERLAPPING LAYER2", "Contact plate OVERLAPPING LAYER3", 
                "Cladding plate OVERLAPPING LAYER3", "Cladding plate OVERLAPPING LAYER2"];
                break;
            case 'LSTC':
                excluded = ["Rear_Plate", "Outer_Cladding"];
                break;
        }
        
        // Name(s) of parts that will be used to identify a seed element.
        // The seed element is used to identify the elements still attached to the main body of the barrier 
        // and exclude elements which have detached during impact.
        // Pick one part with elements near to the rear of the barrier which are unlikely to be detached in the final state.
        // This part MUST also be named in one of the lists above.
        // It MUST be made up of shell elements.
        switch (barrierModel) {
            case 'Arup-Cellbond':
                seed = ["Cladding plate OVERLAPPING LAYER1: Front"];
                break;
            case 'LSTC':
                seed = ["1t-Rear_Core_Horizontal"];
                break;
        }
	}
    
    // -------- OPTION 2: Define parts by number
    
    // If there is no .ztf file, parts are not listed by name, only by number.
    // User can redefine using appropriate part numbers for different barrier model as needed.
    else if (option2)
    {
        switch (barrierModel) {
            case 'Arup-Cellbond':
                backplate = [136];
                hc        = [1, 2, 3, 4, 5, 6, 8, 10, 11, 13, 14, 15, 16, 24, 25, 26, 135];
                plates    = [7, 9, 12, 17, 18, 22];
                excluded  = [19, 20, 21, 23];
                
                seed      = [22];
                break;
            case 'LSTC':
                backplate = [16];
                hc        = [1, 2, 3, 4, 10, 11, 19, 20, 21, 22, 23, 24];
                plates    = [13, 14, 17];
                excluded  = [15, 18];

                seed      = [11];
                break;
        }
    }
    
    // END USER INPUT
    ////////////////////////
    
    if ((option1 && option2) || (!option1 && !option2))  // i.e. both are true, or both are false
    {
        throw "ERROR: please mark only one option as true - option1 (ID by part name) or option2 (ID by part number)."
    }
    
    // Create object defining all part lists
    var partLists = {};
    partLists.backplate = [];  // part that contains the rigid backplate -- internal IDs
    partLists.hc = [];  // all parts that describe the honeycomb -- internal IDs
    partLists.plates = [];  // all parts that describe plates parallel to backplate -- internal IDs
    partLists.excluded = [];  // all parts that are excluded from solve (e.g. solids, tied contacts, etc.) -- internal IDs
    partLists.seed = [];  // all parts that are used to identify a seed element
    
    var partsNo = GetNumberOf(PART);

    for (var pid = 1; pid <= partsNo; pid++)
    {
        // Parse for relevant info (part name or part external ID)
        if (option1)  // i.e. if the user has defined the barrier parts by name (Option 1)
        {
            var check_data = GetPartInfo(pid).title;  // i.e. check against part name
        }
        
        else  // i.e. if the user has defined the barrier parts by external ID (Option 2)
        {
            var check_data = GetLabel(PART, pid);  // i.e. check against external part ID
        }
        
        // Only consider shells and thick shells - if a part in one of the lists is not in fact shells or tshells, exclude it
        if ( (GetElemsInPart(pid).type !== SHELL && GetElemsInPart(pid).type !== TSHELL) &&
            (backplate.includes(check_data)     ||
             hc.includes(check_data)            ||
             plates.includes(check_data)        ) )
        {
            partLists.excluded.push(pid); // ignore solids and other
            continue;
        }
        
        // Add part to correct list
        if (backplate.includes(check_data))  // i.e. if part name in backplate list 
        {
            partLists.backplate.push(pid);
        }
        
        else if (hc.includes(check_data))
        {
            partLists.hc.push(pid);
        }
        
        else if (plates.includes(check_data))
        {
            partLists.plates.push(pid);
        }
            
        else if (excluded.includes(check_data)) 
        {
            partLists.excluded.push(pid)
        }
        
        // Check if specifies seed part
        if (seed.includes(check_data))
        {
            partLists.seed.push(pid);
            
            // Check that seed part is also included in one of the other part lists         
            if (! (partLists.backplate.includes(pid) || 
            partLists.hc.includes(pid) || 
            partLists.plates.includes(pid) || 
            partLists.excluded.includes(pid)))
            {
                throw "ERROR: seed part must also be listed in another part list."
            }
        }
    }
	
	if (partLists.backplate.length !== 1)
    {
        throw "ERROR: one backplate part must be specified. Check barrier type in d3plot_MPDB_barrier_parts_blanking.js."
    }
    
    if (partLists.seed.length !== 1)
    {
        throw "ERROR: one seed part must be specified. Check barrier type in d3plot_MPDB_barrier_parts_blanking.js."
    }

    // Add the barrier model as a property of partLists so that we can use it in d3plot_MPDB_deformation.js
    // If the barrier model is anything other than Arup-Cellbond or LSTC, we will treat it as though it is Arup-Cellbond
    switch (barrierModel) {
        case 'Arup-Cellbond':
            partLists.barrierModel = 'Arup-Cellbond';
            break;
        case 'LSTC':
            partLists.barrierModel = 'LSTC';
            break;
        default:    // if the barrier model is anything else
            partLists.barrierModel = 'Arup-Cellbond';
            break;
    }
    
    return partLists;
    
}


function UnblankBarrier(view)
{
    var partLists = GetBarrierPartLists();  // list of part IDs associated with the different parts of the barrier model
    
    var backplatePIDsList = partLists.backplate;
    var barrierPIDsList = partLists.backplate.concat(partLists.hc.concat(partLists.plates.concat(partLists.excluded)));  // relevant internal part IDs

    Message("MPDB deformation - "+view+" view");
    Message("Blanking entire model...");
    DialogueInputNoEcho("/Blank All");

    var num_p = GetNumberOf(PART);

    Message("Unblanking the rigid backplate...");
    for (var i = 0; i < backplatePIDsList.length; i++)
    {
        Unblank(PART, backplatePIDsList[i]);
    }

    Message("Orienting backplate...");
    DialogueInput("/state last");       // go to final plot state
    DialogueInput("/+yz");              // front elevation
    DialogueInput("/ac");               // auto-centre (these first commands get correct initial zoom)
    DialogueInput("/+iso");             // isometric view
    DialogueInput("/rs -30 0 0");       // tilt upwards for better view
    if (view == "left")     
    {       
        DialogueInput("/rm 0 0 90");    // for "left" view, rotate 90deg about Z-axis in model space
    }
    DialogueInput("/ac");               // auto-centre again
    DialogueInput("/mg 0.60");          // zoom out a bit to make sure everything fits in the frame when we unblank the other parts
    if (view == "left")
    {
		DialogueInput("/tr -0.4 0.2");  // translate into the upper-left corner for "left" view
    }
    else
    {
        DialogueInput("/tr 0.2 0.2");   // translate into the upper-right corner for "right" view
    }

    Message("Unblanking the other relevant barrier parts...");
    for (var i = 0; i < barrierPIDsList.length; i++)
    {
        Unblank(PART, barrierPIDsList[i]);
    }
    
    Message("Done.");
    Exit();

}
