// roundAccurately is a function defined to perform rounding reliably. Rounding is assumed for the IIHS values
const roundAccurately = (number, decimalPlaces) => Number(Math.round(Number(number + "e" + decimalPlaces)) + "e" + decimalPlaces * -1);

// Constants used throughout the script

var REG_DT     = 0.00001; // dt for regularising curves (in seconds)
var HIC_WINDOW = 0.015;   // HIC time window (in seconds)
var TMS_PERIOD = 0.003;   // 3ms period (in seconds)

// Structural intrusion ratings for B-pillar to seat centerline constants (cm) 

var B_PILLAR_DEFORMATION_GOOD       = 18.0; // >=18.0 cm
var B_PILLAR_DEFORMATION_ACCEPTABLE = 14.0; // >=14.0 cm
var B_PILLAR_DEFORMATION_MARGINAL   = 10.0; // >=10.0 cm

// Head assessment constants

var HIC_GOOD       = 623.0;
var HIC_ACCEPTABLE = 779.0;
var HIC_MARGINAL   = 935.0;

// Neck assessment constants

var NECK_TEN_GOOD       = 2.1;
var NECK_TEN_ACCEPTABLE = 2.5;
var NECK_TEN_MARGINAL   = 2.9;

var NECK_COM_GOOD       = 2.5;
var NECK_COM_ACCEPTABLE = 3.0;
var NECK_COM_MARGINAL   = 3.5;


// Torso assessment
// The following are updated to match SID-IIs dummy
var SHOULDER_VC_A = 1.0;         // Viscous Criterion A constant (for m/s)
var SHOULDER_VC_B = 0.138;       // Viscous Criterion B constant (for m/s)
var SHOULDER_VC_METHOD = "IIHS"; // Viscous Criterion calculation method

var TORSO_DEF_GOOD       = 28.0;  // 28mm
var TORSO_DEF_ACCEPTABLE = 38.0;  // 38mm
var TORSO_DEF_MARGINAL   = 48.0;  // 48mm

var TORSO_WORST_DEF_GOOD       = 28.0;  // 28mm
var TORSO_WORST_DEF_ACCEPTABLE = 50.0;  // 50mm
var TORSO_WORST_DEF_MARGINAL   = 55.0;  // 55mm

var TORSO_DEF_RATE_GOOD       = 8.2;   // 8.2  m/s
var TORSO_DEF_RATE_ACCEPTABLE = 9.8;   // 9.8  m/s
var TORSO_DEF_RATE_MARGINAL   = 11.5;  // 11.5 m/s

var TORSO_VC_GOOD       = 1;      // 1   m/s
var TORSO_VC_ACCEPTABLE = 1.2;    // 1.2 m/s
var TORSO_VC_MARGINAL   = 1.4;    // 1.4 m/s


// Pelvis assessments

var PELVIS_FORCE_GOOD         = 4; // 4 kN
var PELVIS_FORCE_ACCEPTABLE   = 5; // 5 kN
var PELVIS_FORCE_MARGINAL     = 6; // 6 kN
var PELVIS_LO_SCORE = 0.0;

// Reporter variables

if(File.Exists(fname_csv) && File.IsFile(fname_csv))
{
    var line;
    var oGlblData = new Object();

    var f_vars = do_common_single_analysis_setup(oGlblData);

    // For each body region, do the calculation, writing results to the variables file.
    // 2 here denotes two dummies: driver and passenger
    // Third argument false for nij flag
    do_head_rating_calc(f_vars, 2);

    do_neck_axial_force_rating_calc(f_vars, 2);

    do_torso_rating_calc(f_vars);

    do_pelvis_rating_calc(f_vars);

    add_b_pillar_variables(f_vars);

    f_vars.Close();
}
else
{
    ErrorMessage("Could not locate a valid input file: "+fname_csv);
    Exit();
}

function add_b_pillar_variables(f)
{
    // NOTE these used to be hard coded and are used in B-pillar image and calc of B-pillar rating. This funtion exports the values to REPORTER variables
    write_variable(f, "B_PILLAR_DEFORMATION_GOOD",       B_PILLAR_DEFORMATION_GOOD,       "B_pillar intrusion GOOD/ACCEPTABLE boundary (cm)",     "General");
    write_variable(f, "B_PILLAR_DEFORMATION_ACCEPTABLE", B_PILLAR_DEFORMATION_ACCEPTABLE, "B_pillar intrusion ACCEPTABLE/MARGINAL boundary (cm)", "General");
    write_variable(f, "B_PILLAR_DEFORMATION_MARGINAL",   B_PILLAR_DEFORMATION_MARGINAL,   "B_pillar intrusion MARGINAL/POOR boundary (cm)",       "General");
}

function do_torso_rating_calc(f)
{
    // Calculates the chest values and writes them into the Reporter csv variables file <f>.

    //var capping_limit = "FALSE";
    var c_disp1       = "FALSE";
    var c_disp2       = "FALSE";
    var c_disp3       = "FALSE";
    var c_disp4       = "FALSE";
    var c_disp5       = "FALSE";
    var c_disp6       = "FALSE";
    var c01, c02, c03;
    var c11, c12, c13;
    var c21, c22, c23;
    var c31, c32, c33;
    var c41, c42, c43;
    var c51, c52, c53;

    for(var pass = 0; pass<2; pass++)
    {
        var spring;
        var dummy;

        for (var fp = 0; fp<6; fp++)
        {
            if(pass == 0) 
            {
                if      (fp == 0) {var spring = oGlblData.driver_shoulder_transducer;}
                else if (fp == 1) {var spring = oGlblData.driver_top_thorax_transducer;}
                else if (fp == 2) {var spring = oGlblData.driver_middle_thorax_transducer;}
                else if (fp == 3) {var spring = oGlblData.driver_bottom_thorax_transducer;} 
                else if (fp == 4) {var spring = oGlblData.driver_top_abdomen_transducer;} 
                else if (fp == 5) {var spring = oGlblData.driver_bottom_abdomen_transducer;}             
                dummy = "Driver";    
            }
            if(pass == 1) 
            {
                if      (fp == 0) {var spring = oGlblData.passenger_shoulder_transducer;}
                else if (fp == 1) {var spring = oGlblData.passenger_top_thorax_transducer;}
                else if (fp == 2) {var spring = oGlblData.passenger_middle_thorax_transducer;}
                else if (fp == 3) {var spring = oGlblData.passenger_bottom_thorax_transducer;} 
                else if (fp == 4) {var spring = oGlblData.passenger_top_abdomen_transducer;} 
                else if (fp == 5) {var spring = oGlblData.passenger_bottom_abdomen_transducer;}             
                dummy = "Passenger"; 
            }

            var output_data = {};

            output_data.rib = new OutputData(dummy + " Torso Deflections",   images_dir + "/" + dummy + "_Torso_Deflections");
            output_data.rib_rat = new OutputData(dummy + " Torso Deflection Rates", images_dir + "/" + dummy + "_Torso_Deflection_Rates");
            output_data.rib_vc = new OutputData(dummy + " Torso Viscous Criteria", images_dir + "/" + dummy + "_Torso_Viscous_Criteria");
            
            //Check data components are available for this spring, if not create a blank image

            if(spring != undefined)
            {
                var c_disp;

                //Read in data
                
                if      (fp == 0)
                {
                    c_disp1 = read_data("SPRING TR", 0, spring, "ET");
                    c_disp = c_disp1; 
                }
                else if (fp == 1)
                {
                    c_disp2 = read_data("SPRING TR", 0, spring, "ET");
                    c_disp = c_disp2; 
                }
                else if (fp == 2)
                {
                    c_disp3 = read_data("SPRING TR", 0, spring, "ET");
                    c_disp = c_disp3; 
                }
                else if (fp == 3)
                {
                    c_disp4 = read_data("SPRING TR", 0, spring, "ET");
                    c_disp = c_disp4; 
                }
                else if (fp == 4)
                {
                    c_disp5 = read_data("SPRING TR", 0, spring, "ET");
                    c_disp = c_disp5; 
                }
                else if (fp == 5)
                {
                    c_disp6 = read_data("SPRING TR", 0, spring, "ET");
                    c_disp = c_disp6; 
                }    
                    
                if(!c_disp)
                {
                    switch(fp)
                    {    
                        case 0: var title = "NO DATA FOR SPRING " + spring + " Driver Shoulder Deflection";
                            var fname = images_dir + "/" + dummy + "_Shoulder_Deflection";
                            
                            c01 = undefined;
                            c02 = undefined;
                            c03 = undefined;
                            
                            break;
                        case 1: var title = "NO DATA FOR SPRING " + spring + " Driver Top Thorax Deflection";
                            var fname = images_dir + "/" + dummy + "_Top_Thorax_Deflection";
                            
                            c11 = undefined;
                            c12 = undefined;
                            c13 = undefined;
                            
                            break;
                        case 2: var title = "NO DATA FOR SPRING " + spring + " Driver Middle Thorax Deflection";
                            var fname = images_dir + "/" + dummy + "_Middle_Thorax_Deflection";
                            
                            c21 = undefined;
                            c22 = undefined;
                            c23 = undefined;
                            
                            break;
                        case 3: var title = "NO DATA FOR SPRING " + spring + " Driver Bottom_ Thorax Deflection";
                            var fname = images_dir + "/" + dummy + "_Bottom_Thorax_Deflection";
                            
                            c31 = undefined;
                            c32 = undefined;
                            c33 = undefined;
                            
                            break;
                        case 4: var title = "NO DATA FOR SPRING " + spring + " Driver Top Abdomen Deflection";
                            var fname = images_dir + "/" + dummy + "_Top_Abdomen_Deflection";
                            
                            c41 = undefined;
                            c42 = undefined;
                            c43 = undefined;
                            
                            break;
                        case 5: var title = "NO DATA FOR SPRING " + spring + " Driver Bottom Abdomen Deflection";
                            var fname = images_dir + "/" + dummy + "_Bottom_Abdomen_Deflection";
                            
                            c51 = undefined;
                            c52 = undefined;
                            c53 = undefined;
                            
                            break;    
                            
                    }
   
                    create_blank_image(title, fname);

                }
                else
                {
                    var c_disp_mm = Operate.Div(c_disp, oGlblData.mm_factor);       // Convert to mm
                    c_disp_mm     = Operate.Mul(c_disp_mm, -1.0);                   // Compression +ve

                    // C180 filter (convert to seconds first)

                    var c_disp_mm_s    = Operate.Dix(c_disp_mm, oGlblData.time_factor);
                    var c_disp_c180_mm = Operate.C180(c_disp_mm_s, REG_DT);  // in mm
                    var c_disp_c180_m  = Operate.Div(c_disp_c180_mm, 1000);  // in m  (for viscous criterion calc and deflection rate)


                    // Calculate viscous criterion

                    var c_vc = Operate.Vc(c_disp_c180_m, SHOULDER_VC_A, SHOULDER_VC_B, SHOULDER_VC_METHOD);


                    // Set labels and style

                    switch(fp)
                    {
                        case 0:c01 = Operate.Mux(c_disp_c180_mm, oGlblData.time_factor);
                               set_labels(c01, " Shoulder Deflection",      "Time (" + oGlblData.unit_time + ")",        "Deflection (mm)");
                               set_line_style(c01, Colour.BLACK);

                               c02 = Operate.Dif(c_disp_c180_m);
                               set_labels(c02, " Shoulder Deflection Rate", "Time (" + oGlblData.unit_time + ")",        "Deflection Rate (m/s)");
                               set_line_style(c02, Colour.BLACK);

                               c03 = Operate.Mux(c_vc, oGlblData.time_factor);
                               set_labels(c03, " Shoulder Viscous Criterion","Time (" + oGlblData.unit_time + ")",       "Velocity (m/s)");
                               set_line_style(c03, Colour.BLACK);                    
                               break;

                        case 1:c11 = Operate.Mux(c_disp_c180_mm, oGlblData.time_factor);
                               set_labels(c11, " Top Thorax Deflection",      "Time (" + oGlblData.unit_time + ")",        "Deflection (mm)");
                               set_line_style(c11, Colour.BLUE);

                               c12 = Operate.Dif(c_disp_c180_m);
                               set_labels(c12, " Top Thorax Deflection Rate", "Time (" + oGlblData.unit_time + ")",        "Deflection Rate (m/s)");
                               set_line_style(c12, Colour.BLUE);

                               c13 = Operate.Mux(c_vc, oGlblData.time_factor);
                               set_labels(c13, " Top Thorax Viscous Criterion",      "Time (" + oGlblData.unit_time + ")", "Velocity (m/s)");
                               set_line_style(c13, Colour.BLUE);                    
                               break;

                        case 2:c21 = Operate.Mux(c_disp_c180_mm, oGlblData.time_factor);
                               set_labels(c21, " Middle Thorax Deflection",      "Time (" + oGlblData.unit_time + ")",        "Deflection (mm)");
                               set_line_style(c21, Colour.CYAN);

                               c22 = Operate.Dif(c_disp_c180_m);
                               set_labels(c22, " Middle Thorax Deflection Rate", "Time (" + oGlblData.unit_time + ")",        "Deflection Rate (m/s)");
                               set_line_style(c22, Colour.CYAN);

                               c23 = Operate.Mux(c_vc, oGlblData.time_factor);
                               set_labels(c23, " Middle Thorax Viscous Criterion",      "Time (" + oGlblData.unit_time + ")", "Velocity (m/s)");
                               set_line_style(c23, Colour.CYAN);                    
                               break;

                        case 3:c31 = Operate.Mux(c_disp_c180_mm, oGlblData.time_factor);
                               set_labels(c31, " Bottom Thorax Deflection",      "Time (" + oGlblData.unit_time + ")",        "Deflection (mm)");
                               set_line_style(c31, Colour.MAGENTA);

                               c32 = Operate.Dif(c_disp_c180_m);
                               set_labels(c32, " Bottom Thorax Deflection Rate", "Time (" + oGlblData.unit_time + ")",        "Deflection Rate (m/s)");
                               set_line_style(c32, Colour.MAGENTA);

                               c33 = Operate.Mux(c_vc, oGlblData.time_factor);
                               set_labels(c33, " Bottom Thorax Viscous Criterion",      "Time (" + oGlblData.unit_time + ")", "Velocity (m/s)");
                               set_line_style(c33, Colour.MAGENTA);                    
                               break;

                        case 4:c41 = Operate.Mux(c_disp_c180_mm, oGlblData.time_factor);
                               set_labels(c41, " Top Abdomen Deflection",      "Time (" + oGlblData.unit_time + ")",        "Deflection (mm)");
                               set_line_style(c41, Colour.INDIGO);

                               c42 = Operate.Dif(c_disp_c180_m);
                               set_labels(c42, " Top Abdomen Deflection Rate", "Time (" + oGlblData.unit_time + ")",        "Deflection Rate (m/s)");
                               set_line_style(c42, Colour.INDIGO);

                               c43 = Operate.Mux(c_vc, oGlblData.time_factor);
                               set_labels(c43, " Top Abdomen Viscous Criterion",      "Time (" + oGlblData.unit_time + ")", "Velocity (m/s)");
                               set_line_style(c43, Colour.INDIGO);                    
                               break;

                        case 5:c51 = Operate.Mux(c_disp_c180_mm, oGlblData.time_factor);
                               set_labels(c51, " Bottom Abdomen Deflection",      "Time (" + oGlblData.unit_time + ")",        "Deflection (mm)");
                               set_line_style(c51, Colour.ORANGE);

                               c52 = Operate.Dif(c_disp_c180_m);
                               set_labels(c52, " Bottom Abdomen Deflection Rate", "Time (" + oGlblData.unit_time + ")",        "Deflection Rate (m/s)");
                               set_line_style(c52, Colour.ORANGE);

                               c53 = Operate.Mux(c_vc, oGlblData.time_factor);
                               set_labels(c53, " Bottom Abdomen Viscous Criterion",      "Time (" + oGlblData.unit_time + ")", "Velocity (m/s)");
                               set_line_style(c53, Colour.ORANGE);                    
                               break;                    
                    }            
                }
            }
            else
            {
                if     (fp == 0)
                {
                    var title = "INVALID/NO SPRING ID for " + dummy + " Shoulder Deflection";
                    var fname = images_dir+ "/" + dummy + "_Shoulder_Deflection";
                    
                    c01 = undefined;
                    c02 = undefined;
                    c03 = undefined;
                }
                else if(fp == 1)
                {
                    var title = "INVALID/NO SPRING ID for " + dummy + " Top Thorax Deflection";
                    var fname = images_dir+ "/" + dummy + "_Top_Thorax_Deflection";
                    
                    c11 = undefined;
                    c12 = undefined;
                    c13 = undefined;
                }
                else if(fp == 2)
                {
                    var title = "INVALID/NO SPRING ID for " + dummy + " Middle Thorax Deflection";
                    var fname = images_dir+ "/" + dummy + "_Middle_Thorax_Deflection";
                    
                    c21 = undefined;
                    c22 = undefined;
                    c23 = undefined;
                }
                else if(fp == 3)
                {
                    var title = "INVALID/NO SPRING ID for " + dummy + " Bottom_ Thorax Deflection";
                    var fname = images_dir+ "/" + dummy + "_Bottom_Thorax_Deflection";
                    
                    c31 = undefined;
                    c32 = undefined;
                    c33 = undefined;
                }
                else if(fp == 4)
                {
                    var title = "INVALID/NO SPRING ID for " + dummy + " Top Abdomen Deflection";
                    var fname = images_dir+ "/" + dummy + "_Top_Abdomen_Deflection";
                    
                    c41 = undefined;
                    c42 = undefined;
                    c43 = undefined;
                }                
                else
                {
                    var title = "INVALID/NO SPRING ID for " + dummy + " Bottom Abdomen Deflection";
                    var fname = images_dir+ "/" + dummy + "_Bottom_Abdomen_Deflection";
                    
                    c51 = undefined;
                    c52 = undefined;
                    c53 = undefined;
                }        
                create_blank_image(title, fname);

            }
        }

        // Remove all curves and datums
        remove_all_curves_and_datums();

        // Draw chest deflection datums
        draw_constant_datums(TORSO_DEF_GOOD, TORSO_DEF_ACCEPTABLE, TORSO_DEF_MARGINAL, null);

        var curves = new Array(7);
        curves[0] = c01;
        curves[1] = c11;
        curves[2] = c21;
        curves[3] = c31;
        curves[4] = c41;
        curves[5] = c51;
        
        var num_valid_curves = 0;

        for (var fp = 0; fp<6; fp++)
        {
            if (curves[fp] != undefined)
            {
                curves[fp].AddToGraph();
                output_data.rib.curveList.push(curves[fp].id);
                num_valid_curves ++;
            }
        }
        
        if (num_valid_curves > 0)
        {
            // Create image/curve files        
            write_image(output_data.rib.title, output_data.rib.fname, curves, TORSO_DEF_MARGINAL);
            
            write_curve("cur", output_data.rib.curveList, output_data.rib.fname);
            write_curve("csv", output_data.rib.curveList, output_data.rib.fname);
        } 
        else 
        {
            create_blank_image("NO SPRING ID DEFINED FOR " + output_data.rib.title, output_data.rib.fname );            
        }

        // Remove all curves and datums
        remove_all_curves_and_datums(); 

        // Draw chest deflection rate datums
        draw_constant_datums(TORSO_DEF_RATE_GOOD, TORSO_DEF_RATE_ACCEPTABLE, TORSO_DEF_RATE_MARGINAL, null);

        var curves1 = new Array(7);
        curves1[0] = c02;
        curves1[1] = c12;
        curves1[2] = c22;
        curves1[3] = c32;
        curves1[4] = c42;
        curves1[5] = c52;

        num_valid_curves = 0;
        for (var fp = 0; fp<6; fp++)
        {
            if (curves1[fp] != undefined)
            {
                curves1[fp].AddToGraph();
                output_data.rib_rat.curveList.push(curves1[fp].id);
                num_valid_curves ++;
            }
        }    
        
        if (num_valid_curves > 0)
        {
            // Create image/curve files        
            write_image(output_data.rib_rat.title, output_data.rib_rat.fname, curves1, TORSO_DEF_RATE_MARGINAL);
            
            write_curve("cur", output_data.rib_rat.curveList, output_data.rib_rat.fname);
            write_curve("csv", output_data.rib_rat.curveList, output_data.rib_rat.fname);
        } 
        else 
            create_blank_image("NO SPRING ID DEFINED FOR " + output_data.rib_rat.title, output_data.rib_rat.fname );            


        // Remove all curves and datums
        remove_all_curves_and_datums(); 

        // Draw chest deflection rate datums
        draw_constant_datums(TORSO_VC_GOOD, TORSO_VC_ACCEPTABLE, TORSO_VC_MARGINAL, null);

        var curves2 = new Array(7);
        curves2[0] = c03;
        curves2[1] = c13;
        curves2[2] = c23;
        curves2[3] = c33;
        curves2[4] = c43;
        curves2[5] = c53;    

        num_valid_curves = 0;
        for (var fp = 0; fp<6; fp++)
        {
            if (curves2[fp] != undefined)
            {
                curves2[fp].AddToGraph();
                output_data.rib_vc.curveList.push(curves2[fp].id);
                num_valid_curves ++;
            }
        }
        
        if (num_valid_curves > 0)
        {
            // Create image/curve files        
            write_image(output_data.rib_vc.title, output_data.rib_vc.fname, curves2, TORSO_VC_MARGINAL);
            
            write_curve("cur", output_data.rib_vc.curveList, output_data.rib_vc.fname);
            write_curve("csv", output_data.rib_vc.curveList, output_data.rib_vc.fname);
        } 
        else 
            create_blank_image("NO SPRING ID DEFINED FOR " + output_data.rib_vc.title, output_data.rib_vc.fname );            

        // Remove all curves and datums
        remove_all_curves_and_datums(); 
        
        // Shoulder
        var shoulder_deflection;
        
        // Top thorax rib
        var top_thorax_deflection;
        var top_thorax_deflection_rate;
        var top_thorax_vc;
        
        // Middle thorax rib
        var middle_thorax_deflection;
        var middle_thorax_deflection_rate;
        var middle_thorax_vc;
        
        // Bottom thorax rib
        var bottom_thorax_deflection;
        var bottom_thorax_deflection_rate;
        var bottom_thorax_vc;
        
        // Top abdomen rib
        var top_abd_deflection;
        var top_abd_deflection_rate;
        var top_abd_vc;
        
        // Bottom abdomen rib
        var bottom_abd_deflection;
        var bottom_abd_deflection_rate;
        var bottom_abd_vc;
        
        //Note on rounding: 
        //rounding to number of significant figures displayed in Table 1 of the Side Impact 2.0 ratings guidelines
        //is used before assesment of rating. Hence the use of the roundAccurately() method below to 0 and 1 dp.

        // Check if shoulder is defined in the input file
        if (c01 != undefined)
        {
            //NOTE: if shoulder deflection > 60mm or bottoms out the torso rating is decreased by one category
            shoulder_deflection              = roundAccurately(Math.abs(c01.ymax),0); 
        }
        
        // Checks if top thorax rib is defined in the input file
        if (c11 != undefined)
        {
            top_thorax_deflection            = roundAccurately(Math.abs(c11.ymax),0);
            top_thorax_deflection_rate       = roundAccurately(Math.abs(c12.ymax),1);
            top_thorax_vc                    = roundAccurately(Math.abs(c13.ymax),1);
        }
        
        // Checks if middle thorax rib is defined in the input file
        if (c21 != undefined)
        {
            middle_thorax_deflection         = roundAccurately(Math.abs(c21.ymax),0);
            middle_thorax_deflection_rate    = roundAccurately(Math.abs(c22.ymax),1); 
            middle_thorax_vc                 = roundAccurately(Math.abs(c23.ymax),1);
        }
        
        // Checks if bottom thorax rib is defined in the input file
        if (c31 != undefined)
        {
            bottom_thorax_deflection         = roundAccurately(Math.abs(c31.ymax),0);
            bottom_thorax_deflection_rate    = roundAccurately(Math.abs(c32.ymax),1);
            bottom_thorax_vc                 = roundAccurately(Math.abs(c33.ymax),1);
        }
        
        // Checks if top abdomen rib is defined in the input file
        if (c41 != undefined)
        {
            top_abd_deflection               = roundAccurately(Math.abs(c41.ymax),0);
            top_abd_deflection_rate          = roundAccurately(Math.abs(c42.ymax),1);
            top_abd_vc                       = roundAccurately(Math.abs(c43.ymax),1);
        }
        
        // Checks if bottom abdomen rib is defined in the input file
        if (c51 != undefined)
        {
            bottom_abd_deflection            = roundAccurately(Math.abs(c51.ymax),0);
            bottom_abd_deflection_rate       = roundAccurately(Math.abs(c52.ymax),1);
            bottom_abd_vc                    = roundAccurately(Math.abs(c53.ymax),1);
        }
        
        var def_val, worst_def;

        if(top_thorax_deflection > 50 || middle_thorax_deflection > 50 || bottom_thorax_deflection > 50 || top_abd_deflection > 50 || bottom_abd_deflection > 50)
        {
            def_val = Math.max(top_thorax_deflection, middle_thorax_deflection, bottom_thorax_deflection, top_abd_deflection, bottom_abd_deflection); 
            worst_def = 1;
        }
        else
        {
            def_val = (top_thorax_deflection + middle_thorax_deflection + bottom_thorax_deflection + top_abd_deflection + bottom_abd_deflection)/5;
            worst_def = 0;
        }

        var def_rating_val = (top_thorax_deflection_rate + middle_thorax_deflection_rate + bottom_thorax_deflection_rate + top_abd_deflection_rate + bottom_abd_deflection_rate)/5;

        var vc_val         = (top_thorax_vc + middle_thorax_vc + bottom_thorax_vc + top_abd_vc + bottom_abd_vc)/5;

        // Write variables
        
        var driver_torso_def_results;
        if(worst_def == 0)  driver_torso_def_results    = get_constant_val_results(def_val,        TORSO_DEF_GOOD,       TORSO_DEF_ACCEPTABLE,       TORSO_DEF_MARGINAL);
        else                driver_torso_def_results    = get_constant_val_results(def_val,        TORSO_WORST_DEF_GOOD, TORSO_WORST_DEF_ACCEPTABLE, TORSO_WORST_DEF_MARGINAL);
        
        var driver_torso_def_rate_results               = get_constant_val_results(def_rating_val, TORSO_DEF_RATE_GOOD, TORSO_DEF_RATE_ACCEPTABLE, TORSO_DEF_RATE_MARGINAL);
        var driver_torso_vc_results                     = get_constant_val_results(vc_val,         TORSO_VC_GOOD,       TORSO_VC_ACCEPTABLE,       TORSO_VC_MARGINAL);
        
        // Write shoulder deflection rating and value only if shoulder spring ID is defined in the defined in the input
        if ( (c01 != undefined) ) 
        {
            //shoulder deflection does not have a rating but a requirement to be below a threshold of 60mm and not bottom out
            write_variable(f, dummy.toUpperCase()+ "_SHOULDER_DEFLECTION_VALUE",    shoulder_deflection.toFixed(3),      dummy + " shoulder deflection value",       "General");      
        }
        
        // Write torso rating and value only if top thorax, middle thorax, bottom thorax, top and bottom abdomen are defined
        if ( (c11 != undefined) && (c21 != undefined) && (c31 != undefined) && (c41 != undefined) && (c51 != undefined) ) 
        {
            write_variable(f, dummy.toUpperCase()+ "_TORSO_DEFLECTION_RATING",      driver_torso_def_results[0],                    dummy + " torso deflection rating",      "General");
            write_variable(f, dummy.toUpperCase()+ "_TORSO_DEFLECTION_VALUE",       driver_torso_def_results[1].toFixed(3),         dummy + " torso deflection value",       "General");

            write_variable(f, dummy.toUpperCase()+ "_TORSO_DEFLECTION_RATE_RATING", driver_torso_def_rate_results[0],               dummy + " torso deflection rate rating", "General");
            write_variable(f, dummy.toUpperCase()+ "_TORSO_DEFLECTION_RATE_VALUE",  driver_torso_def_rate_results[1].toFixed(3),    dummy + " torso deflection rate value",  "General");

            write_variable(f, dummy.toUpperCase()+ "_TORSO_VISCOUS_RATING",         driver_torso_vc_results[0],                     dummy + " torso viscous rating",         "General");
            write_variable(f, dummy.toUpperCase()+ "_TORSO_VISCOUS_VALUE",          driver_torso_vc_results[1].toFixed(3),          dummy + " torso viscous value",          "General");              
        }
    }
}

function do_pelvis_rating_calc (f)
{
    //Calculates the pelvis scores for the dummy and writes relevant 
    //values into the Reporter csv variable file <f>.
    Message("Doing pelvis rating calculation...");
    var dummy = [];
    dummy.push(new DummyData("Driver", oGlblData.xsection_driver_acetabular_id, "beam", oGlblData.xsection_driver_iliac_id, "beam",oGlblData.driver_dummy, "version"));
    dummy.push(new DummyData("Passenger", oGlblData.xsection_passenger_acetabular_id, "beam", oGlblData.xsection_passenger_iliac_id, "beam", oGlblData.passenger_dummy, "version"));
    
    
    for(var i = 0; i<dummy.length; i++)
    {
        var xsec_iliac;
        var xsec_acetab;
        
        if( i == 0)
        { 
            xsec_iliac = oGlblData.xsection_driver_iliac_id;
            xsec_acetab = oGlblData.xsection_driver_acetabular_id;
            
        }
        if( i == 1)
        { 
            xsec_iliac = oGlblData.xsection_passenger_iliac_id;
            xsec_acetab = oGlblData.xsection_passenger_acetabular_id;  
            
        }
        
        var output_data = {};

        output_data.pel = new OutputData(dummy[i].chart_label + " Pelvic Force", images_dir + "/" + dummy[i].variable_label + "_Pelvic_Force");
        
        //var capping_limit = "FALSE";
    
        // Originally, it was not necessary to specify the dummy version for IIHS Side
        // MDB Impact (it always extracted SECTION data) so if version is undefined, mark
        // it as unspecified and treat as SECTION data

        if (dummy[i].version == undefined)
            dummy[i].version = "<Unspecified>";

        if( xsec_iliac != undefined && xsec_acetab != undefined)
        {
            // Should check data components are available for this beam
            // Read in iliac and acetabular Section forces for Humanitics SID dummy 
            // Read in iliac and acetabular beam forces for LSTC SID dummy
            switch(dummy[i].version)
            {
                case "<Unspecified>":
                case "Humanetics SID IIs SBLD v.4.3.1":
                case "Humanetics SID IIs SBLD v.4.3.2":
                case "Humanetics SID IIs SBLD v.4.3.5":
                    var c_iliac_f = read_data("SECTION", 0, xsec_iliac, "FY");
                    var c_acetab_f = read_data("SECTION", 0, xsec_acetab, "FY");
                    var entity_str = "SECTION"; // Used for error message below
                    break;

                case "LSTC SID IIS-D v0.150.beta": 
                    var c_iliac_f = read_data("BEAM BASIC", 0, xsec_iliac, "NY");
                    var c_acetab_f = read_data("BEAM BASIC", 0, xsec_acetab, "NY");
                    var entity_str = "BEAM"; // Used for error message below
                    break;
                default:
                    ErrorMessage("Unsupported dummy \""+dummy[i].version+"\" in do_pelvis_rating_calc.");
                    write_blank_images(output_data, "UNSUPPORTED DUMMY \""+dummy[i].version+"\".");
                    return;
                    break;
            }

            // Check that the curves read in - i.e. the beam id and component are valid
            // Create a blank image to let the user know.

            if(!c_iliac_f || !c_acetab_f)
            {
                create_blank_image("NO DATA FOR "+ entity_str + xsec_iliac +" "+ xsec_acetab+" " + output_data.pel.title, output_data.pel.fname);
            }
            else
            {
                // C600 filter (convert to seconds) and convert force into kN

                var c_iliac_f = Operate.Dix(c_iliac_f, oGlblData.time_factor);
                var c_iliac_f = Operate.C600(c_iliac_f, REG_DT);
                var c_iliac_f = Operate.Div(c_iliac_f, oGlblData.kn_factor);

                var c_acetab_f = Operate.Dix(c_acetab_f, oGlblData.time_factor);
                var c_acetab_f = Operate.C600(c_acetab_f, REG_DT);
                var c_acetab_f = Operate.Div(c_acetab_f, oGlblData.kn_factor);

                // Remove all curves and datums

                remove_all_curves_and_datums();

                // Convert from seconds back to model time

                c_iliac_f  = Operate.Mux(c_iliac_f,  oGlblData.time_factor);
                c_acetab_f = Operate.Mux(c_acetab_f, oGlblData.time_factor);

                // ignore negative values and set them to zero

                var this_max = 1e20; //this is maximum number in T/HIS
                c_iliac_f  = Operate.Clip(c_iliac_f, -this_max,this_max,0.0,this_max);
                c_acetab_f = Operate.Clip(c_acetab_f,-this_max,this_max,0.0,this_max);

                //  Total Pelvic force
                
                var c_pf = Operate.Add(c_iliac_f, c_acetab_f);           

                // Set labels and style

                set_labels(c_iliac_f, " Iliac Wing Force Y", "Time (" + oGlblData.unit_time + ")", "Force (kN)");
                set_line_style(c_iliac_f, Colour.BLACK);
                set_labels(c_acetab_f, " Acetabulum Force Y", "Time (" + oGlblData.unit_time + ")", "Force (kN)");
                set_line_style(c_acetab_f, Colour.BLACK);
                set_labels(c_pf,  " Pelvis Force",       "Time (" + oGlblData.unit_time + ")", "Force (kN)");
                set_line_style(c_pf, Colour.BLACK);    

                // Remove all curves and datums
                remove_all_curves_and_datums();


                // Draw datums

                draw_constant_datums(PELVIS_FORCE_GOOD, PELVIS_FORCE_ACCEPTABLE, PELVIS_FORCE_MARGINAL, null, true);
                c_pf.AddToGraph();

                // Create image/curve files        
                write_image(output_data.pel.title, output_data.pel.fname, c_pf, PELVIS_FORCE_MARGINAL, -PELVIS_FORCE_MARGINAL);
                
                output_data.pel.curveList.push(c_pf.id);
                write_curve("cur", output_data.pel.curveList, output_data.pel.fname);
                write_curve("csv", output_data.pel.curveList, output_data.pel.fname);


                // Write out variables  

                var driver_pelv_force           = IIHS_get_constant_limit_results(c_pf,  PELVIS_FORCE_GOOD, PELVIS_FORCE_ACCEPTABLE,  PELVIS_FORCE_MARGINAL, true);

                write_variable(f, dummy[i].variable_label.toUpperCase()+ "_PELVIS_FORCE_RATING",        driver_pelv_force[0],                   dummy[i].variable_label + " Pelvis force rating",        "General");
                write_variable(f, dummy[i].variable_label.toUpperCase()+ "_PELVIS_FORCE_VALUE",         driver_pelv_force[1].toFixed(3),        dummy[i].variable_label + " Pelvis force value",         "General");

                // Remove all curves and datums

                remove_all_curves_and_datums();
            }
        }
        else
        {
            // No beam defined - variables should be set to blank by first script in template.
            // Create a blank image to let the user know.   

            create_blank_image("NO SECTION ID DEFINED FOR Pelvis Forces", output_data.pel.fname);
        }
    }
}

function get_constant_val_results(c, good, accp, marg)
{
    // Calculates results for criteria that has constant limits
    // Returns the rating and value. 
    //
    // <c> is the value
    // <good>, <accp> and <marg> are the constant limits

    var ret = new Array(2);

    var val;
    var rating;

    if(good > 0) { val =  c; rating = IIHS_get_rating(val,  good,  accp,  marg); }
    else         { val = -c;    rating = IIHS_get_rating(val, -good, -accp, -marg); }

    ret[0] = rating;
    ret[1] = val;

    return ret;  
}


function do_neck_axial_force_rating_calc(f, num_dummies)
{
    // Calculates the neck values and writes them into the Reporter csv variables file <f>.
    
    // dummy no. 1 is driver
    // dummy no. 2 is passenger
    var dummy = [];
    dummy.push(new DummyData("Driver", oGlblData.driver_neck_loadcell, "beam", oGlblData.driver_dummy, "version"));
    if (num_dummies == 2)
    {
        dummy.push(new DummyData("Passenger", oGlblData.passenger_neck_loadcell, "beam", oGlblData.passenger_dummy, "version"));
    }

        
    for(var i = 0; i<dummy.length; i++)
    {
        var output_data = {};

        output_data.axl      = new OutputData(dummy[i].chart_label + " Neck Axial Force",            images_dir + "/" + dummy[i].variable_label + "_Neck_Axial_Force");
        
// Originally, it was not necessary to specify the dummy version for IIHS Side
// MDB Impact (it always extracted BEAM data) so if version is undefined, mark
// it as unspecified and treat as BEAM data
        if (dummy[i].version == undefined)
            dummy[i].version = "<Unspecified>";

        if( dummy[i].beam_id != undefined )
        {
            // Read in beam shear, tension and  MY
            // For Humanetics dummies this is measured from a beam;
            // for LSTC dummies this is measured from a cross section
            switch (dummy[i].version)
            {
                case "<Unspecified>":
                case "LSTC SID IIS-D v0.150.beta":                
                    //var c_shr = read_data("BEAM BASIC", 0, dummy[i].beam_id, "NX");  // NX is shear
                    var c_axl = read_data("BEAM BASIC", 0, dummy[i].beam_id, "NZ");
                    //var c_mom = read_data("BEAM BASIC", 0, dummy[i].beam_id, "MY");
                    var entity_str = "BEAM"; // Used for error message below
                    break;

                case "Humanetics SID IIs SBLD v.4.3.1":
                case "Humanetics SID IIs SBLD v.4.3.2":
                case "Humanetics SID IIs SBLD v.4.3.5":
                    //Shifted Humanetics-SID-Dummy to X-sec because the neck load cells are x-sections
                    //var c_shr = read_data("SECTION", 0, dummy[i].beam_id, "FX");  // FX is shear
                    var c_axl = read_data("SECTION", 0, dummy[i].beam_id, "FZ");
                    //var c_mom = read_data("SECTION", 0, dummy[i].beam_id, "MY");
                    var entity_str = "SECTION"; // Used for error message below
                    break;

                default:
                    ErrorMessage("Unsupported dummy \""+dummy[i].version+"\" in do_neck_axial_force_rating_calc.");
                    write_blank_images(output_data, "UNSUPPORTED DUMMY \""+dummy[i].version+"\".");
                    return;
                    break;
            }

            // Check that the curve reads in - i.e. the beam id and component are valid
            // Create blank images to let the user know.

            if(!c_axl)
            {
                write_blank_images(output_data, "NO DATA FOR " + entity_str + " " + dummy[i].beam_id);
            }
            else
            {
                // Convert forces to kN from model units
                               
                var c_axl_kn = Operate.Div(c_axl, oGlblData.kn_factor);
                
                // C1000/C600 filter (convert to seconds first)                
                var c_axl_c1000 = convert_to_s_and_filter(c_axl_kn, Operate.C1000, REG_DT);
                
                // Remove all curves and datums

                remove_all_curves_and_datums();

                // Convert from seconds back to model time
                c_axl_c1000 = Operate.Mux(c_axl_c1000,   oGlblData.time_factor);

                // Set labels and style
                set_labels(c_axl_c1000, " Neck Axial Force", "Time (" + oGlblData.unit_time + ")","Force (kN)");
                set_line_style(c_axl_c1000,   Colour.BLACK);                   

                // Remove all curves and datums

                remove_all_curves_and_datums();

                // Draw neck axial force datums

                var p = c_axl_c1000.GetPoint(c_axl_c1000.npoints);       
                draw_neck_axl_datums(p[0]);

                // Create image/curve of axial forces curves

                c_axl_c1000.AddToGraph();
                write_image(output_data.axl.title, output_data.axl.fname, c_axl_c1000, NECK_TEN_MARGINAL, -NECK_COM_MARGINAL);
                
                output_data.axl.curveList.push(c_axl_c1000.id);
                write_curve("cur", output_data.axl.curveList, output_data.axl.fname);
                write_curve("csv", output_data.axl.curveList, output_data.axl.fname);

                // Remove all curves and datums

                remove_all_curves_and_datums();

                // Calc values

                var ten_results     = IIHS_get_constant_limit_results(c_axl_c1000,  NECK_TEN_GOOD,  NECK_TEN_ACCEPTABLE,  NECK_TEN_MARGINAL);
                var com_results     = IIHS_get_constant_limit_results(c_axl_c1000, -NECK_COM_GOOD, -NECK_COM_ACCEPTABLE, -NECK_COM_MARGINAL);
                
                // Write variables

                // TENSION

                write_variable(f, dummy[i].variable_label.toUpperCase() + "_NECK_TENSION_RATING", ten_results[0],            dummy[i].variable_label + " Neck tension rating", "String");
                write_variable(f, dummy[i].variable_label.toUpperCase() + "_NECK_TENSION_VALUE",  ten_results[1].toFixed(3), dummy[i].variable_label + " Neck tension value",  "String");

                // COMPRESSION

                write_variable(f, dummy[i].variable_label.toUpperCase() + "_NECK_COMPRESSION_RATING", com_results[0],            dummy[i].variable_label + " Neck compression rating", "General");
                write_variable(f, dummy[i].variable_label.toUpperCase() + "_NECK_COMPRESSION_VALUE",  com_results[1].toFixed(3), dummy[i].variable_label + " Neck compression value",  "General");

            }
        }
        else
        {

            // No beam defined - variables should be set to blank by first script in template.
            // Create blank images to let the user know.

            write_blank_images(output_data, "NO BEAM ID DEFINED FOR");
        }
    }
}
