// Constants used throughout the script

var REG_DT     = 0.00001; // dt for regularising curves (in seconds)
var HIC_WINDOW = 0.015;   // HIC time window (in seconds)
var TMS_PERIOD = 0.003;   // 3ms period (in seconds)

// Head assessment constants

var HIC_GOOD       = 623.0;
var HIC_ACCEPTABLE = 779.0;
var HIC_MARGINAL   = 935.0;

// Neck assessment constants

var NECK_SHR_ADJ_FAC = 0.01778;   // Factor to multiply shear by to adjust bending moment

// Neck shear exceedence limit curves

var NECK_SHR_EXC_TIME = new Array(4);
var NECK_SHR_EXC_GOOD = new Array(4);

NECK_SHR_EXC_TIME[0] = 0.0;    NECK_SHR_EXC_GOOD[0] = 3.1;
NECK_SHR_EXC_TIME[1] = 25.0;   NECK_SHR_EXC_GOOD[1] = 1.5;
NECK_SHR_EXC_TIME[2] = 35.0;   NECK_SHR_EXC_GOOD[2] = 1.5;
NECK_SHR_EXC_TIME[3] = 45.0;   NECK_SHR_EXC_GOOD[3] = 1.1;


// Neck tension exceedence limit curves

var NECK_TEN_EXC_TIME = new Array(3);
var NECK_TEN_EXC_GOOD = new Array(3);

NECK_TEN_EXC_TIME[0] = 0.0;    NECK_TEN_EXC_GOOD[0] = 3.3;
NECK_TEN_EXC_TIME[1] = 35.0;   NECK_TEN_EXC_GOOD[1] = 2.9;
NECK_TEN_EXC_TIME[2] = 45.0;   NECK_TEN_EXC_GOOD[2] = 1.1;


// Neck compression exceedence limit curves

var NECK_COM_EXC_TIME = new Array(2);
var NECK_COM_EXC_GOOD = new Array(2);

NECK_COM_EXC_TIME[0] = 0.0;    NECK_COM_EXC_GOOD[0] = 4.0;
NECK_COM_EXC_TIME[1] = 30.0;   NECK_COM_EXC_GOOD[1] = 1.1;


var NECK_TEN_GOOD       = 2.1;
var NECK_TEN_ACCEPTABLE = 2.5;
var NECK_TEN_MARGINAL   = 2.9;

var NECK_COM_GOOD       = 2.5;
var NECK_COM_ACCEPTABLE = 3.0;
var NECK_COM_MARGINAL   = 3.5;


// Torso assessment
// The following are updated to match SID-IIs dummy
var SHOULDER_VC_A = 1.0;           // Viscous Criterion A constant (for m/s)
var SHOULDER_VC_B = 0.138;         // Viscous Criterion B constant (for m/s)
var SHOULDER_VC_METHOD = "IIHS"; // Viscous Criterion calculation method

var TORSO_DEF_GOOD       = 34.0;  // 34mm
var TORSO_DEF_ACCEPTABLE = 42.0;  // 42mm
var TORSO_DEF_MARGINAL   = 50.0;  // 50mm

var TORSO_WORST_DEF_GOOD       = 34.0;  // 34mm
var TORSO_WORST_DEF_ACCEPTABLE = 50.0;  // 50mm
var TORSO_WORST_DEF_MARGINAL   = 55.0;  // 55mm

var TORSO_DEF_RATE_GOOD       = 8.2;    // 6.6   m/s
var TORSO_DEF_RATE_ACCEPTABLE = 9.84;   // 9.84  m/s
var TORSO_DEF_RATE_MARGINAL   = 11.48;  // 11.48 m/s

var TORSO_VC_GOOD       = 1;      // 1m/s
var TORSO_VC_ACCEPTABLE = 1.2;    // 1.2m/s
var TORSO_VC_MARGINAL   = 1.4;    // 1.4m/s


// Pelvis and left femur assessments

var ACETAB_FORCE_GOOD         = 4.0;
var ACETAB_FORCE_ACCEPTABLE   = 4.8;
var ACETAB_FORCE_MARGINAL     = 5.6;

var ILIAC_FORCE_GOOD         = 4.0;
var ILIAC_FORCE_ACCEPTABLE   = 4.8;
var ILIAC_FORCE_MARGINAL     = 5.6;

var PELVIS_FORCE_GOOD         = 5.1;
var PELVIS_FORCE_ACCEPTABLE   = 6.1;
var PELVIS_FORCE_MARGINAL     = 7.1;
var PELVIS_LO_SCORE = 0.0;

// Upper Leg and knee assessments

var FEMUR_COM_GOOD       = 2.8;
var FEMUR_COM_ACCEPTABLE = 3.4;
var FEMUR_COM_MARGINAL   = 3.9;

var FEMUR_MOM_GOOD       = 254;
var FEMUR_MOM_ACCEPTABLE = 305;
var FEMUR_MOM_MARGINAL   = 356;


// Reporter variables

if(File.Exists(fname_csv) && File.IsFile(fname_csv))
{
	var line;
	var oGlblData = new Object();

	var f_vars = do_common_single_analysis_setup(oGlblData);

	// For each body region, do the calculation, writing results to the variables file.
	// 2 here denotes two dummies: driver and passenger
	// Third argument false for nij flag
	do_head_rating_calc(f_vars, 2);

	do_neck_rating_calc(f_vars, 2, false);

	do_torso_rating_calc(f_vars);

	do_pelvis_rating_calc(f_vars);

	do_upper_leg_and_knee_rating_calc(f_vars);

	f_vars.Close();
}
else
{
    ErrorMessage("Could not locate a valid input file: "+fname_csv);
    Exit();
}

function do_torso_rating_calc(f)
{
	// Calculates the chest values and writes them into the Reporter csv variables file <f>.

	var capping_limit = "FALSE";
	var c_disp1       = "FALSE";
	var c_disp2       = "FALSE";
	var c_disp3       = "FALSE";
	var c_disp4       = "FALSE";
	var c_disp5       = "FALSE";
	var c_disp6       = "FALSE";
	var c01, c02, c03;
	var c11, c12, c13;
	var c21, c22, c23;
	var c31, c32, c33;
	var c41, c42, c43;
	var c51, c52, c53;

	for(var pass = 0; pass<2; pass++)
	{
		var spring;
		var dummy;

		for (var fp = 0; fp<6; fp++)
		{
			if(pass == 0) 
			{
				if      (fp == 0) {var spring = oGlblData.driver_shoulder_transducer;}
				else if (fp == 1) {var spring = oGlblData.driver_top_thorax_transducer;}
				else if (fp == 2) {var spring = oGlblData.driver_middle_thorax_transducer;}
				else if (fp == 3) {var spring = oGlblData.driver_bottom_thorax_transducer;} 
				else if (fp == 4) {var spring = oGlblData.driver_top_abdomen_transducer;} 
				else if (fp == 5) {var spring = oGlblData.driver_bottom_abdomen_transducer;} 			
				dummy = "Driver";    
			}
			if(pass == 1) 
			{
				if      (fp == 0) {var spring = oGlblData.passenger_shoulder_transducer;}
				else if (fp == 1) {var spring = oGlblData.passenger_top_thorax_transducer;}
				else if (fp == 2) {var spring = oGlblData.passenger_middle_thorax_transducer;}
				else if (fp == 3) {var spring = oGlblData.passenger_bottom_thorax_transducer;} 
				else if (fp == 4) {var spring = oGlblData.passenger_top_abdomen_transducer;} 
				else if (fp == 5) {var spring = oGlblData.passenger_bottom_abdomen_transducer;} 			
				dummy = "Passenger"; 
			}

			var output_data = {};

			output_data.rib = new OutputData(dummy + " Torso Deflections",   images_dir + "/" + dummy + "_Torso_Deflections");
			output_data.rib_rat = new OutputData(dummy + " Torso Deflection Rates", images_dir + "/" + dummy + "_Torso_Deflection_Rates");
			output_data.rib_vc = new OutputData(dummy + " Torso Viscous Criteria", images_dir + "/" + dummy + "_Torso_Viscous_Criteria");
			
			//Check data components are available for this spring, if not create a blank image

			if(spring != undefined)
			{
				var c_disp;

				//Read in data
				
				if      (fp == 0)
				{
					c_disp1 = read_data("SPRING TR", 0, spring, "ET");
					c_disp = c_disp1; 
				}
				else if (fp == 1)
				{
					c_disp2 = read_data("SPRING TR", 0, spring, "ET");
					c_disp = c_disp2; 
				}
				else if (fp == 2)
				{
					c_disp3 = read_data("SPRING TR", 0, spring, "ET");
					c_disp = c_disp3; 
				}
				else if (fp == 3)
				{
					c_disp4 = read_data("SPRING TR", 0, spring, "ET");
					c_disp = c_disp4; 
				}
				else if (fp == 4)
				{
					c_disp5 = read_data("SPRING TR", 0, spring, "ET");
					c_disp = c_disp5; 
				}
				else if (fp == 5)
				{
					c_disp5 = read_data("SPRING TR", 0, spring, "ET");
					c_disp = c_disp5; 
				}	
					
				if(!c_disp)
				{
					switch(fp)
					{    
						case 0: var title = "NO DATA FOR SPRING " + spring + " Driver Shoulder Deflection";
							var fname = images_dir + "/" + dummy + "_Shoulder_Deflection";
							
							c01 = undefined;
							c02 = undefined;
							c03 = undefined;
							
							break;
						case 1: var title = "NO DATA FOR SPRING " + spring + " Driver Top Thorax Deflection";
							var fname = images_dir + "/" + dummy + "_Top_Thorax_Deflection";
							
							c11 = undefined;
							c12 = undefined;
							c13 = undefined;
							
							break;
						case 2: var title = "NO DATA FOR SPRING " + spring + " Driver Middle Thorax Deflection";
							var fname = images_dir + "/" + dummy + "_Middle_Thorax_Deflection";
							
							c21 = undefined;
							c22 = undefined;
							c23 = undefined;
							
							break;
						case 3: var title = "NO DATA FOR SPRING " + spring + " Driver Bottom_ Thorax Deflection";
							var fname = images_dir + "/" + dummy + "_Bottom_Thorax_Deflection";
							
							c31 = undefined;
							c32 = undefined;
							c33 = undefined;
							
							break;
						case 4: var title = "NO DATA FOR SPRING " + spring + " Driver Top Abdomen Deflection";
							var fname = images_dir + "/" + dummy + "_Top_Abdomen_Deflection";
							
							c41 = undefined;
							c42 = undefined;
							c43 = undefined;
							
							break;
						case 5: var title = "NO DATA FOR SPRING " + spring + " Driver Bottom Abdomen Deflection";
							var fname = images_dir + "/" + dummy + "_Bottom_Abdomen_Deflection";
							
							c51 = undefined;
							c52 = undefined;
							c53 = undefined;
							
							break;	
							
					}
   
					create_blank_image(title, fname);

				}
				else
				{
					var c_disp_mm = Operate.Div(c_disp, oGlblData.mm_factor);       // Convert to mm
					c_disp_mm     = Operate.Mul(c_disp_mm, -1.0);                   // Compression +ve

					// C180 filter (convert to seconds first)

					var c_disp_mm_s    = Operate.Dix(c_disp_mm, oGlblData.time_factor);
					var c_disp_c180_mm = Operate.C180(c_disp_mm_s, REG_DT);  // in mm
					var c_disp_c180_m  = Operate.Div(c_disp_c180_mm, 1000);  // in m  (for viscous criterion calc and deflection rate)


					// Calculate viscous criterion

					var c_vc = Operate.Vc(c_disp_c180_m, SHOULDER_VC_A, SHOULDER_VC_B, SHOULDER_VC_METHOD);


					// Set labels and style

					switch(fp)
					{
						case 0:c01 = Operate.Mux(c_disp_c180_mm, oGlblData.time_factor);
						       set_labels(c01, " Shoulder Deflection",      "Time (" + oGlblData.unit_time + ")",        "Deflection (mm)");
						       set_line_style(c01, Colour.BLACK);

						       c02 = Operate.Dif(c_disp_c180_m);
						       set_labels(c02, " Shoulder Deflection Rate", "Time (" + oGlblData.unit_time + ")",        "Deflection Rate (m/s)");
						       set_line_style(c02, Colour.BLACK);

						       c03 = Operate.Mux(c_vc, oGlblData.time_factor);
						       set_labels(c03, " Shoulder Viscous Criterion","Time (" + oGlblData.unit_time + ")",       "Velocity (m/s)");
						       set_line_style(c03, Colour.BLACK);					
						       break;

						case 1:c11 = Operate.Mux(c_disp_c180_mm, oGlblData.time_factor);
						       set_labels(c11, " Top Thorax Deflection",      "Time (" + oGlblData.unit_time + ")",        "Deflection (mm)");
						       set_line_style(c11, Colour.BLUE);

						       c12 = Operate.Dif(c_disp_c180_m);
						       set_labels(c12, " Top Thorax Deflection Rate", "Time (" + oGlblData.unit_time + ")",        "Deflection Rate (m/s)");
						       set_line_style(c12, Colour.BLUE);

						       c13 = Operate.Mux(c_vc, oGlblData.time_factor);
						       set_labels(c13, " Top Thorax Viscous Criterion",      "Time (" + oGlblData.unit_time + ")", "Velocity (m/s)");
						       set_line_style(c13, Colour.BLUE);					
						       break;

						case 2:c21 = Operate.Mux(c_disp_c180_mm, oGlblData.time_factor);
						       set_labels(c21, " Middle Thorax Deflection",      "Time (" + oGlblData.unit_time + ")",        "Deflection (mm)");
						       set_line_style(c21, Colour.CYAN);

						       c22 = Operate.Dif(c_disp_c180_m);
						       set_labels(c22, " Middle Thorax Deflection Rate", "Time (" + oGlblData.unit_time + ")",        "Deflection Rate (m/s)");
						       set_line_style(c22, Colour.CYAN);

						       c23 = Operate.Mux(c_vc, oGlblData.time_factor);
						       set_labels(c23, " Middle Thorax Viscous Criterion",      "Time (" + oGlblData.unit_time + ")", "Velocity (m/s)");
						       set_line_style(c23, Colour.CYAN);					
						       break;

						case 3:c31 = Operate.Mux(c_disp_c180_mm, oGlblData.time_factor);
						       set_labels(c31, " Bottom Thorax Deflection",      "Time (" + oGlblData.unit_time + ")",        "Deflection (mm)");
						       set_line_style(c31, Colour.MAGENTA);

						       c32 = Operate.Dif(c_disp_c180_m);
						       set_labels(c32, " Bottom Thorax Deflection Rate", "Time (" + oGlblData.unit_time + ")",        "Deflection Rate (m/s)");
						       set_line_style(c32, Colour.MAGENTA);

						       c33 = Operate.Mux(c_vc, oGlblData.time_factor);
						       set_labels(c33, " Bottom Thorax Viscous Criterion",      "Time (" + oGlblData.unit_time + ")", "Velocity (m/s)");
						       set_line_style(c33, Colour.MAGENTA);					
						       break;

						case 4:c41 = Operate.Mux(c_disp_c180_mm, oGlblData.time_factor);
						       set_labels(c41, " Top Abdomen Deflection",      "Time (" + oGlblData.unit_time + ")",        "Deflection (mm)");
						       set_line_style(c41, Colour.INDIGO);

						       c42 = Operate.Dif(c_disp_c180_m);
						       set_labels(c42, " Top Abdomen Deflection Rate", "Time (" + oGlblData.unit_time + ")",        "Deflection Rate (m/s)");
						       set_line_style(c42, Colour.INDIGO);

						       c43 = Operate.Mux(c_vc, oGlblData.time_factor);
						       set_labels(c43, " Top Abdomen Viscous Criterion",      "Time (" + oGlblData.unit_time + ")", "Velocity (m/s)");
						       set_line_style(c43, Colour.INDIGO);					
						       break;

						case 5:c51 = Operate.Mux(c_disp_c180_mm, oGlblData.time_factor);
						       set_labels(c51, " Bottom Abdomen Deflection",      "Time (" + oGlblData.unit_time + ")",        "Deflection (mm)");
						       set_line_style(c51, Colour.ORANGE);

						       c52 = Operate.Dif(c_disp_c180_m);
						       set_labels(c52, " Bottom Abdomen Deflection Rate", "Time (" + oGlblData.unit_time + ")",        "Deflection Rate (m/s)");
						       set_line_style(c52, Colour.ORANGE);

						       c53 = Operate.Mux(c_vc, oGlblData.time_factor);
						       set_labels(c53, " Bottom Abdomen Viscous Criterion",      "Time (" + oGlblData.unit_time + ")", "Velocity (m/s)");
						       set_line_style(c53, Colour.ORANGE);					
						       break;					
					}			
				}
			}
			else
			{
				if     (fp == 0)
				{
					var title = "INVALID/NO SPRING ID for " + dummy + " Shoulder Deflection";
					var fname = images_dir+ "/" + dummy + "_Shoulder_Deflection";
					
					c01 = undefined;
					c02 = undefined;
					c03 = undefined;
				}
				else if(fp == 1)
				{
					var title = "INVALID/NO SPRING ID for " + dummy + " Top Thorax Deflection";
					var fname = images_dir+ "/" + dummy + "_Top_Thorax_Deflection";
					
					c11 = undefined;
					c12 = undefined;
					c13 = undefined;
				}
				else if(fp == 2)
				{
					var title = "INVALID/NO SPRING ID for " + dummy + " Middle Thorax Deflection";
					var fname = images_dir+ "/" + dummy + "_Middle_Thorax_Deflection";
					
					c21 = undefined;
					c22 = undefined;
					c23 = undefined;
				}
				else if(fp == 3)
				{
					var title = "INVALID/NO SPRING ID for " + dummy + " Bottom_ Thorax Deflection";
					var fname = images_dir+ "/" + dummy + "_Bottom_Thorax_Deflection";
					
					c31 = undefined;
					c32 = undefined;
					c33 = undefined;
				}
				else if(fp == 4)
				{
					var title = "INVALID/NO SPRING ID for " + dummy + " Top Abdomen Deflection";
					var fname = images_dir+ "/" + dummy + "_Top_Abdomen_Deflection";
					
					c41 = undefined;
					c42 = undefined;
					c43 = undefined;
				}				
				else
				{
					var title = "INVALID/NO SPRING ID for " + dummy + " Bottom Abdomen Deflection";
					var fname = images_dir+ "/" + dummy + "_Bottom_Abdomen_Deflection";
					
					c51 = undefined;
					c52 = undefined;
					c53 = undefined;
				}		
				create_blank_image(title, fname);

			}
		}

		// Remove all curves and datums
		remove_all_curves_and_datums();

		// Draw chest deflection datums
		draw_constant_datums(TORSO_DEF_GOOD, TORSO_DEF_ACCEPTABLE, TORSO_DEF_MARGINAL, null);

		var curves = new Array(7);
		curves[0] = c01;
		curves[1] = c11;
		curves[2] = c21;
		curves[3] = c31;
		curves[4] = c41;
		curves[5] = c51;
		
		var num_valid_curves = 0;

		for (var fp = 0; fp<6; fp++)
		{
			if (curves[fp] != undefined)
			{
				curves[fp].AddToGraph();
				output_data.rib.curveList.push(curves[fp].id);
				num_valid_curves ++;
			}
		}
		
		if (num_valid_curves > 0)
		{
			// Create image/curve files		
			write_image(output_data.rib.title, output_data.rib.fname, curves, TORSO_DEF_MARGINAL);
			
			write_curve("cur", output_data.rib.curveList, output_data.rib.fname);
			write_curve("csv", output_data.rib.curveList, output_data.rib.fname);
		} 
		else 
		{
			create_blank_image("NO SPRING ID DEFINED FOR " + output_data.rib.title, output_data.rib.fname );			
		}

		// Remove all curves and datums
		remove_all_curves_and_datums(); 

		// Draw chest deflection rate datums
		draw_constant_datums(TORSO_DEF_RATE_GOOD, TORSO_DEF_RATE_ACCEPTABLE, TORSO_DEF_RATE_MARGINAL, null);

		var curves1 = new Array(7);
		curves1[0] = c02;
		curves1[1] = c12;
		curves1[2] = c22;
		curves1[3] = c32;
		curves1[4] = c42;
		curves1[5] = c52;

		num_valid_curves = 0;
		for (var fp = 0; fp<6; fp++)
		{
			if (curves1[fp] != undefined)
			{
				curves1[fp].AddToGraph();
                output_data.rib_rat.curveList.push(curves1[fp].id);
				num_valid_curves ++;
			}
		}	
		
		if (num_valid_curves > 0)
		{
			// Create image/curve files		
			write_image(output_data.rib_rat.title, output_data.rib_rat.fname, curves1, TORSO_DEF_RATE_MARGINAL);
			
			write_curve("cur", output_data.rib_rat.curveList, output_data.rib_rat.fname);
			write_curve("csv", output_data.rib_rat.curveList, output_data.rib_rat.fname);
		} 
		else 
			create_blank_image("NO SPRING ID DEFINED FOR " + output_data.rib_rat.title, output_data.rib_rat.fname );			


		// Remove all curves and datums
		remove_all_curves_and_datums(); 

		// Draw chest deflection rate datums
		draw_constant_datums(TORSO_VC_GOOD, TORSO_VC_ACCEPTABLE, TORSO_VC_MARGINAL, null);

		var curves2 = new Array(7);
		curves2[0] = c03;
		curves2[1] = c13;
		curves2[2] = c23;
		curves2[3] = c33;
		curves2[4] = c43;
		curves2[5] = c53;	

		num_valid_curves = 0;
		for (var fp = 0; fp<6; fp++)
		{
			if (curves2[fp] != undefined)
			{
				curves2[fp].AddToGraph();
                output_data.rib_vc.curveList.push(curves2[fp].id);
				num_valid_curves ++;
			}
		}
		
		if (num_valid_curves > 0)
		{
			// Create image/curve files		
			write_image(output_data.rib_vc.title, output_data.rib_vc.fname, curves2, TORSO_VC_MARGINAL);
			
			write_curve("cur", output_data.rib_vc.curveList, output_data.rib_vc.fname);
			write_curve("csv", output_data.rib_vc.curveList, output_data.rib_vc.fname);
		} 
		else 
			create_blank_image("NO SPRING ID DEFINED FOR " + output_data.rib_vc.title, output_data.rib_vc.fname );			

		// Remove all curves and datums
		remove_all_curves_and_datums(); 
		
		// Shoulder
		var shoulder_deflection;
		var shoulder_deflection_rate;
		var shoulder_vc;
		
		// Top thorax rib
		var top_thorax_deflection;
		var top_thorax_deflection_rate;
		var top_thorax_vc;
		
		// Middle thorax rib
		var middle_thorax_deflection;
		var middle_thorax_deflection_rate;
		var middle_thorax_vc;
		
		// Bottom thorax rib
		var bottom_thorax_deflection;
		var bottom_thorax_deflection_rate;
		var bottom_thorax_vc;
		
		// Top abdomen rib
		var top_abd_deflection;
		var top_abd_deflection_rate;
		var top_abd_vc;
		
		// Bottom abdomen rib
		var bottom_abd_deflection;
		var bottom_abd_deflection_rate;
		var bottom_abd_vc;
		
		// Check if shoulder is defined in the input file
		if (c01 != undefined)
		{
			shoulder_deflection 		= Math.abs(c01.ymax);
			shoulder_deflection_rate 	= Math.abs(c02.ymax);
			shoulder_vc 				= Math.abs(c03.ymax);
		}
		
		// Checks if top thorax rib is defined in the input file
		if (c11 != undefined)
		{
			top_thorax_deflection 		= Math.abs(c11.ymax);
			top_thorax_deflection_rate 	= Math.abs(c12.ymax);
			top_thorax_vc 				= Math.abs(c13.ymax);
		}
		
		// Checks if middle thorax rib is defined in the input file
		if (c21 != undefined)
		{
			middle_thorax_deflection 		= Math.abs(c21.ymax);
			middle_thorax_deflection_rate 	= Math.abs(c22.ymax); 
			middle_thorax_vc 				= Math.abs(c23.ymax);
		}
		
		// Checks if bottom thorax rib is defined in the input file
		if (c31 != undefined)
		{
			bottom_thorax_deflection 		= Math.abs(c31.ymax);
			bottom_thorax_deflection_rate 	= Math.abs(c32.ymax);
			bottom_thorax_vc 				= Math.abs(c33.ymax);
		}
		
		// Checks if top abdomen rib is defined in the input file
		if (c41 != undefined)
		{
			top_abd_deflection 				= Math.abs(c41.ymax);
			top_abd_deflection_rate 		= Math.abs(c42.ymax);
			top_abd_vc 						= Math.abs(c43.ymax);
		}
		
		// Checks if bottom abdomen rib is defined in the input file
		if (c51 != undefined)
		{
		bottom_abd_deflection =          Math.abs(c51.ymax);
		bottom_abd_deflection_rate =     Math.abs(c52.ymax);
		bottom_abd_vc =                  Math.abs(c53.ymax);
		}
		
		var def_val, worst_def;

		if(top_thorax_deflection > 50 || middle_thorax_deflection > 50 || bottom_thorax_deflection > 50 || top_abd_deflection > 50 || bottom_abd_deflection > 50)
		{
			def_val = Math.max(top_thorax_deflection, middle_thorax_deflection, bottom_thorax_deflection, top_abd_deflection, bottom_abd_deflection); 
			worst_def = 1;
		}
		else
		{
			def_val = (top_thorax_deflection + middle_thorax_deflection + bottom_thorax_deflection + top_abd_deflection + bottom_abd_deflection)/5;
			worst_def = 0;
		}

		var def_rating_val = (top_thorax_deflection_rate + middle_thorax_deflection_rate + bottom_thorax_deflection_rate + top_abd_deflection_rate + bottom_abd_deflection_rate)/5;

		var vc_val         = (top_thorax_vc + middle_thorax_vc + bottom_thorax_vc + top_abd_vc + bottom_abd_vc)/5;

		// Write variables
		var driver_shoulder_def_results            = get_constant_val_results(shoulder_deflection,        TORSO_DEF_GOOD,       TORSO_DEF_ACCEPTABLE,       TORSO_DEF_MARGINAL);
		var driver_torso_def_results;
		if(worst_def == 0)  driver_torso_def_results            = get_constant_val_results(def_val,        TORSO_DEF_GOOD,       TORSO_DEF_ACCEPTABLE,       TORSO_DEF_MARGINAL);
		else                driver_torso_def_results            = get_constant_val_results(def_val,        TORSO_WORST_DEF_GOOD, TORSO_WORST_DEF_ACCEPTABLE, TORSO_WORST_DEF_MARGINAL);
		var driver_torso_def_rate_results       = get_constant_val_results(def_rating_val, TORSO_DEF_RATE_GOOD, TORSO_DEF_RATE_ACCEPTABLE, TORSO_DEF_RATE_MARGINAL);
		var driver_torso_vc_results             = get_constant_val_results(vc_val,         TORSO_VC_GOOD,       TORSO_VC_ACCEPTABLE,       TORSO_VC_MARGINAL);
		
		// Write shoulder deflection rating and value only if shoulder spring ID is defined in the defined in the input
		if ( (c01 != undefined) ) 
		{
			write_variable(f, dummy.toUpperCase()+ "_SHOULDER_DEFLECTION_RATING",   driver_shoulder_def_results[0],                 dummy + " torso deflection rating",      "General");
			write_variable(f, dummy.toUpperCase()+ "_SHOULDER_DEFLECTION_VALUE",    driver_shoulder_def_results[1].toFixed(3),      dummy + " torso deflection value",       "General");	  
		}
		
		// Write torso rating and value only if top thorax, middle thorax, bottom thorax, top and bottom abdomen are defined
		if ( (c11 != undefined) && (c21 != undefined) && (c31 != undefined) && (c41 != undefined) && (c51 != undefined) ) 
		{
			write_variable(f, dummy.toUpperCase()+ "_TORSO_DEFLECTION_RATING",      driver_torso_def_results[0],                    dummy + " torso deflection rating",      "General");
			write_variable(f, dummy.toUpperCase()+ "_TORSO_DEFLECTION_VALUE",       driver_torso_def_results[1].toFixed(3),         dummy + " torso deflection value",       "General");

			write_variable(f, dummy.toUpperCase()+ "_TORSO_DEFLECTION_RATE_RATING", driver_torso_def_rate_results[0],               dummy + " torso deflection rate rating", "General");
			write_variable(f, dummy.toUpperCase()+ "_TORSO_DEFLECTION_RATE_VALUE",  driver_torso_def_rate_results[1].toFixed(3),    dummy + " torso deflection rate value",  "General");

			write_variable(f, dummy.toUpperCase()+ "_TORSO_VISCOUS_RATING",         driver_torso_vc_results[0],                     dummy + " torso viscous rating",         "General");
			write_variable(f, dummy.toUpperCase()+ "_TORSO_VISCOUS_VALUE",          driver_torso_vc_results[1].toFixed(3),          dummy + " torso viscous value",          "General");	  		
		}
	}
}


function do_pelvis_rating_calc (f)
{
	//Calculates the pelvis scores for the dummy and writes relevant 
	//values into the Reporter csv variable file <f>.
	Message("Doing pelvis rating calculation...");
    var dummy = [];
    dummy.push(new DummyData("Driver", oGlblData.xsection_driver_acetabular_id, "beam", oGlblData.xsection_driver_iliac_id, "beam",oGlblData.driver_dummy, "version"));
    dummy.push(new DummyData("Passenger", oGlblData.xsection_passenger_acetabular_id, "beam", oGlblData.xsection_passenger_iliac_id, "beam", oGlblData.passenger_dummy, "version"));
    
    
	for(var i = 0; i<dummy.length; i++)
    {
		var xsec_iliac;
		var xsec_acetab;
		
		if( i == 0)
		{ 
			xsec_iliac = oGlblData.xsection_driver_iliac_id;
			xsec_acetab = oGlblData.xsection_driver_acetabular_id;
			
		}
		if( i == 1)
		{ 
			xsec_iliac = oGlblData.xsection_passenger_iliac_id;
			xsec_acetab = oGlblData.xsection_passenger_acetabular_id;  
			
		}
        
		var output_data = {};

		output_data.ace = new OutputData(dummy[i].chart_label + " Acetabulum Force",   images_dir + "/" + dummy[i].variable_label + "_Acetabulum_Force");
		output_data.ili = new OutputData(dummy[i].chart_label+ " Ilium Force", images_dir + "/" + dummy[i].variable_label + "_Ilium_Force");
		output_data.pel = new OutputData(dummy[i].chart_label + " Pelvic Force", images_dir + "/" + dummy[i].variable_label + "_Pelvic_Force");
		
		var capping_limit = "FALSE";
    
// Originally, it was not necessary to specify the dummy version for IIHS Side
// MDB Impact (it always extracted SECTION data) so if version is undefined, mark
// it as unspecified and treat as SECTION data
        if (dummy[i].version == undefined)
            dummy[i].version = "<Unspecified>";

		if( xsec_iliac != undefined && xsec_acetab != undefined)
		{
			// Should check data components are available for this beam
			// Read in iliac and acetabular Section forces for Humanitics SID dummy 
		   // Read in iliac and acetabular beam forces for LSTC SID dummy
            switch(dummy[i].version)
            {
				case "<Unspecified>":
				case "Humanetics SID IIs SBLD v.4.3.1":
				case "Humanetics SID IIs SBLD v.4.3.2":
				case "Humanetics SID IIs SBLD v.4.3.5":
  			       var c_iliac_f = read_data("SECTION", 0, xsec_iliac, "FY");
			       var c_acetab_f = read_data("SECTION", 0, xsec_acetab, "FY");
                   var entity_str = "SECTION"; // Used for error message below
                    break;

                case "LSTC SID IIS-D v0.150.beta": 
  			       var c_iliac_f = read_data("BEAM BASIC", 0, xsec_iliac, "NY");
			       var c_acetab_f = read_data("BEAM BASIC", 0, xsec_acetab, "NY");
                   var entity_str = "BEAM"; // Used for error message below
				   break;
                default:
                    ErrorMessage("Unsupported dummy \""+dummy[i].version+"\" in do_pelvis_rating_calc.");
                    write_blank_images(output_data, "UNSUPPORTED DUMMY \""+dummy[i].version+"\".");
                    return;
                    break;
            }

			// Check that the curves read in - i.e. the beam id and component are valid
			// Create a blank image to let the user know.

			if(!c_iliac_f || !c_acetab_f)
			{
				create_blank_image("NO DATA FOR "+ entity_str + xsec_iliac +" "+ xsec_acetab+" " + output_data.pel.title, output_data.pel.fname);

			}
			else
			{

				// C600 filter (convert to seconds) and convert force into kN

				var c_iliac_f = Operate.Dix(c_iliac_f, oGlblData.time_factor);
				var c_iliac_f = Operate.C600(c_iliac_f, REG_DT);
				var c_iliac_f = Operate.Div(c_iliac_f, oGlblData.kn_factor);

				var c_acetab_f = Operate.Dix(c_acetab_f, oGlblData.time_factor);
				var c_acetab_f = Operate.C600(c_acetab_f, REG_DT);
				var c_acetab_f = Operate.Div(c_acetab_f, oGlblData.kn_factor);

				// Remove all curves and datums

				remove_all_curves_and_datums();

				// Convert from seconds back to model time

				c_iliac_f = Operate.Mux(c_iliac_f, oGlblData.time_factor);
				c_acetab_f = Operate.Mux(c_acetab_f, oGlblData.time_factor);

				//  Total Pelvic force

				var c_pf = Operate.Add(c_iliac_f, c_acetab_f);	  	 

				// Set labels and style

				set_labels(c_iliac_f, " Iliac Wing Force Y", "Time (" + oGlblData.unit_time + ")", "Force (kN)");
				set_line_style(c_iliac_f, Colour.BLACK);
				set_labels(c_acetab_f, " Acetabulum Force Y", "Time (" + oGlblData.unit_time + ")", "Force (kN)");
				set_line_style(c_acetab_f, Colour.BLACK);
				set_labels(c_pf,  " Pelvis Force",       "Time (" + oGlblData.unit_time + ")", "Force (kN)");
				set_line_style(c_pf, Colour.BLACK);	

				// Remove all curves and datums
				remove_all_curves_and_datums();


				// Draw datums

				draw_constant_datums(ILIAC_FORCE_GOOD, ILIAC_FORCE_ACCEPTABLE, ILIAC_FORCE_MARGINAL, null,true);
				c_iliac_f.AddToGraph();
				
				// Create image/curve files		
				write_image(output_data.ili.title, output_data.ili.fname, c_iliac_f, ILIAC_FORCE_MARGINAL, -ILIAC_FORCE_MARGINAL);
				
				output_data.ili.curveList.push(c_iliac_f.id);
				write_curve("cur", output_data.ili.curveList, output_data.ili.fname);
				write_curve("csv", output_data.ili.curveList, output_data.ili.fname);

				// Remove all curves and datums
				remove_all_curves_and_datums();

				// Draw datums

				draw_constant_datums(ACETAB_FORCE_GOOD, ACETAB_FORCE_ACCEPTABLE, ACETAB_FORCE_MARGINAL, null, true);
				c_acetab_f.AddToGraph()
				
				// Create image/curve files		
				write_image(output_data.ace.title, output_data.ace.fname, c_acetab_f, ACETAB_FORCE_MARGINAL, -ACETAB_FORCE_MARGINAL);
				
				output_data.ace.curveList.push(c_acetab_f.id);
				write_curve("cur", output_data.ace.curveList, output_data.ace.fname);
				write_curve("csv", output_data.ace.curveList, output_data.ace.fname);

				// Remove all curves and datums
				remove_all_curves_and_datums();

				// Draw datums

				draw_constant_datums(PELVIS_FORCE_GOOD, PELVIS_FORCE_ACCEPTABLE, PELVIS_FORCE_MARGINAL, null, true);
				c_pf.AddToGraph();

				// Create image/curve files		
				write_image(output_data.pel.title, output_data.pel.fname, c_pf, PELVIS_FORCE_MARGINAL, -PELVIS_FORCE_MARGINAL);
				
				output_data.pel.curveList.push(c_pf.id);
				write_curve("cur", output_data.pel.curveList, output_data.pel.fname);
				write_curve("csv", output_data.pel.curveList, output_data.pel.fname);


				// Write out variables  

				var driver_acetab_force         = IIHS_get_constant_limit_results(c_acetab_f, ACETAB_FORCE_GOOD, ACETAB_FORCE_ACCEPTABLE, ACETAB_FORCE_MARGINAL, true);
				var driver_ilium_force          = IIHS_get_constant_limit_results(c_iliac_f, ILIAC_FORCE_GOOD, ILIAC_FORCE_ACCEPTABLE, ILIAC_FORCE_MARGINAL, true);
				var driver_pelv_force           = IIHS_get_constant_limit_results(c_pf,  PELVIS_FORCE_GOOD, PELVIS_FORCE_ACCEPTABLE,  PELVIS_FORCE_MARGINAL, true);



				write_variable(f, dummy[i].variable_label.toUpperCase()+ "_ACETABULUM_FORCE_RATING",    driver_acetab_force[0],                 dummy[i].variable_label + " Acetabulum force rating",    "General");
				write_variable(f, dummy[i].variable_label.toUpperCase()+ "_ACETABULUM_FORCE_VALUE",     driver_acetab_force[1].toFixed(3),      dummy[i].variable_label + " Acetabulum force value",     "General");

				write_variable(f, dummy[i].variable_label.toUpperCase()+ "_ILIUM_FORCE_RATING",         driver_ilium_force[0],                  dummy[i].variable_label + " Ilium force rating",         "General");
				write_variable(f, dummy[i].variable_label.toUpperCase()+ "_ILIUM_FORCE_VALUE",          driver_ilium_force[1].toFixed(3),       dummy[i].variable_label + " Ilium force value",          "General");

				write_variable(f, dummy[i].variable_label.toUpperCase()+ "_PELVIS_FORCE_RATING",        driver_pelv_force[0],                   dummy[i].variable_label + " Pelvis force rating",        "General");
				write_variable(f, dummy[i].variable_label.toUpperCase()+ "_PELVIS_FORCE_VALUE",         driver_pelv_force[1].toFixed(3),        dummy[i].variable_label + " Pelvis force value",         "General");

				// Remove all curves and datums

				remove_all_curves_and_datums();
			}
		}
		else
		{
			// No beam defined - variables should be set to blank by first script in template.
			// Create a blank image to let the user know.   
			
			create_blank_image("NO SECTION ID DEFINED FOR Acetabulum Force", output_data.ace.fname);
			create_blank_image("NO SECTION ID DEFINED FOR Iliac Force", output_data.ili.fname);
			create_blank_image("NO SECTION ID DEFINED FOR Pelvis Forces", output_data.pel.fname);
		}
	}
}

function do_upper_leg_and_knee_rating_calc(f)
{
	// Calculates the upper legs and knee values and writes them into the Reporter csv variables file <f>.

	// Two passes for the left and right hand side
	for(var pass = 0; pass<2; pass++)
	{
		var beam;
		var dummy;
		var side;


		for(var lr=0; lr<2; lr++)
		{          		
			if(pass == 0)
			{
				if(lr == 0) side = "A_P";
				else        side = "L_M";

				if(lr == 0) beam   = oGlblData.driver_upper_femur_beam;
				else        beam   = oGlblData.driver_bottom_femur_beam;			
				dummy = "Driver";
				var version = oGlblData.driver_dummy;
			}
			else
			{
				if(lr == 0) side = "A_P";
				else        side = "L_M";

				if(lr == 0) beam   = oGlblData.passenger_upper_femur_beam;
				else        beam   = oGlblData.passenger_bottom_femur_beam;			
				dummy = "Passenger";
				var version = oGlblData.passenger_dummy;
			}
			
			var output_data = {};

			output_data.com_exc = new OutputData(dummy + " Femur " + side + " Compression Exceedence", images_dir + "/" + dummy + "_" + side + "_Femur_Compression_Exceedence");
			output_data.mom = new OutputData(dummy + " Femur " + side + " Bending Moment", images_dir + "/" + dummy + "_" + side + "_Femur_Bending_Moment");

			var curve_id = Curve.FirstFreeID();

			// Femur compression forces
            //Over here dummy version is checked because Humanetics SID dummy 
            //uses femur load cells as X-Sections instead of Beam 
            if (version == undefined)
            version = "<Unspecified>";
        
			if(beam != undefined)
			{
				switch(version)
                {
                    case "<Unspecified>":
                    case "LSTC SID IIS-D v0.150.beta":  
                        var c_com_x = read_data("BEAM BASIC", 0, beam, "NX");
                        var c_com_y = read_data("BEAM BASIC", 0, beam, "NY");
                        var c_com_z = read_data("BEAM BASIC", 0, beam, "NZ");
                        
                        var c_mom_x = read_data("BEAM BASIC", 0, beam, "MX");
                        var c_mom_y = read_data("BEAM BASIC", 0, beam, "MY");
                        var c_mom_z = read_data("BEAM BASIC", 0, beam, "MZ");
                        var entity_str = "BEAM"; // Used for error message below
                        break;
                        
                    case "Humanetics SID IIs SBLD v.4.3.1":
                    case "Humanetics SID IIs SBLD v.4.3.2": 
                    case "Humanetics SID IIs SBLD v.4.3.5": 
                    
                        var c_com_x = read_data("SECTION", 0, beam, "FX");
                        var c_com_y = read_data("SECTION", 0, beam, "FY");
                        var c_com_z = read_data("SECTION", 0, beam, "FZ");
                        
                        var c_mom_x = read_data("SECTION", 0, beam, "MX");
                        var c_mom_y = read_data("SECTION", 0, beam, "MY");
                        var c_mom_z = read_data("SECTION", 0, beam, "MZ");
                        var entity_str = "SECTION"; // Used for error message below
                        break;
                        
                    default:
                        ErrorMessage("Unsupported dummy \""+ version +"\" in do_upper_leg_and_knee_rating_calc.");
                        write_blank_images(output_data, "UNSUPPORTED DUMMY \""+ version +"\".");
                        return;
                        break;
                }
                // Check that the curve read in - i.e. the beam id and component are valid
				// Create blank image to let the user know.

				if(!c_com_x || !c_com_y || !c_com_z || !c_mom_x || !c_mom_y || !c_mom_z)
				{
					// No beam defined - variables should be set to blank by first script in template.
					// Create a blank image to let the user know.

					write_blank_images(output_data, "NO DATA FOR BEAM")
				}
				else
				{
					// Convert forces to kN from model units

					var c_com_x_kn = Operate.Div(c_com_x, oGlblData.kn_factor);
					var c_com_y_kn = Operate.Div(c_com_y, oGlblData.kn_factor);
					var c_com_z_kn = Operate.Div(c_com_z, oGlblData.kn_factor);

					// Convert forces to kN and
					//       moment to Nm from model units

					var c_mom_x_nm = Operate.Div(c_mom_x, oGlblData.nm_factor);
					var c_mom_y_nm = Operate.Div(c_mom_y, oGlblData.nm_factor);
					var c_mom_z_nm = Operate.Div(c_mom_z, oGlblData.nm_factor);

					// C600 filter (convert to seconds first)

					var c_com_x_kn_s = Operate.Dix(c_com_x_kn, oGlblData.time_factor);
					var c_com_y_kn_s = Operate.Dix(c_com_y_kn, oGlblData.time_factor);
					var c_com_z_kn_s = Operate.Dix(c_com_z_kn, oGlblData.time_factor);

					var c_com_x_c600 = Operate.C600(c_com_x_kn_s, REG_DT);
					var c_com_y_c600 = Operate.C600(c_com_y_kn_s, REG_DT);
					var c_com_z_c600 = Operate.C600(c_com_z_kn_s, REG_DT);

					var c_mom_x_nm_s = Operate.Dix(c_mom_x_nm, oGlblData.time_factor);
					var c_mom_y_nm_s = Operate.Dix(c_mom_y_nm, oGlblData.time_factor);
					var c_mom_z_nm_s = Operate.Dix(c_mom_y_nm, oGlblData.time_factor);

					var c_mom_x_c600 = Operate.C600(c_mom_x_nm_s, REG_DT);
					var c_mom_y_c600 = Operate.C600(c_mom_y_nm_s, REG_DT);
					var c_mom_z_c600 = Operate.C600(c_mom_z_nm_s, REG_DT);

					// Vector combine

					var c_com_res = Operate.Vec(c_com_x_c600, c_com_y_c600, c_com_z_c600);	
					var c_mom_res = Operate.Vec(c_mom_x_c600, c_mom_y_c600, c_mom_z_c600);		

					// Remove all curves and datums

					remove_all_curves_and_datums();

					// Convert from seconds back to model time

					c_com_res     = Operate.Mux(c_com_res,     oGlblData.time_factor);
					c_com_res     = Operate.Tms(c_com_res,0.003);
					var c_com_3ms_res = c_com_res.tms;
					c_mom_res     = Operate.Mux(c_mom_res,     oGlblData.time_factor);
					c_mom_res     = Operate.Tms(c_mom_res,0.003);	
					var c_mom_3ms_res = c_mom_res.tms;					

					// Set labels and style

					set_labels(c_com_res, "Femur " + side +  " Force",    "Time (" + oGlblData.unit_time + ")",         "Force (kN)");
					set_line_style(c_com_res, Colour.BLACK);

					set_labels(c_mom_res, "Femur " + side +  " Bending Moment", "Time (" + oGlblData.unit_time + ")",  "Bending Moment (Nm)");		
					set_line_style(c_mom_res, Colour.BLACK);

					// Remove all curves and datums

					remove_all_curves_and_datums();

					// Draw femur compression datums

					draw_constant_datums(FEMUR_COM_GOOD, FEMUR_COM_ACCEPTABLE, FEMUR_COM_MARGINAL, null);

					// Create image/curve of femur compression exceedence curves

					c_com_res.AddToGraph();

					write_image(output_data.com_exc.title, output_data.com_exc.fname, c_com_res, FEMUR_COM_MARGINAL);
					
					output_data.com_exc.curveList.push(c_com_res.id);					
					write_curve("cur", output_data.com_exc.curveList, output_data.com_exc.fname);
					write_curve("csv", output_data.com_exc.curveList, output_data.com_exc.fname);
					
					// Remove all curves and datums
					remove_all_curves_and_datums(); 

					// Draw femur compression datums

					draw_constant_datums(FEMUR_MOM_GOOD, FEMUR_MOM_ACCEPTABLE, FEMUR_MOM_MARGINAL, null);		
					
					
					// Create image/curve of moment curves

					c_mom_res.AddToGraph();
					write_image(output_data.mom.title, output_data.mom.fname, c_mom_res, FEMUR_MOM_MARGINAL);
					
					output_data.mom.curveList.push(c_mom_res.id);					
					write_curve("cur", output_data.mom.curveList, output_data.mom.fname);
					write_curve("csv", output_data.mom.curveList, output_data.mom.fname);   

					
					// Calc values
					var driver_com_results     = get_constant_val_results(c_com_3ms_res,  FEMUR_COM_GOOD,     FEMUR_COM_ACCEPTABLE,    FEMUR_COM_MARGINAL);
					var driver_mom_results     = get_constant_val_results(c_mom_3ms_res,  FEMUR_MOM_GOOD,     FEMUR_MOM_ACCEPTABLE,    FEMUR_MOM_MARGINAL);

					// Write compression variables

					write_variable(f, dummy.toUpperCase()+ "_" + side.toUpperCase() + "_FEMUR_FORCE_RATING", driver_com_results[0],            "Femur compression rating", "General");
					write_variable(f, dummy.toUpperCase()+ "_" + side.toUpperCase() + "_FEMUR_FORCE_VALUE",  driver_com_results[1].toFixed(3), "Femur compression value",  "General");

					write_variable(f, dummy.toUpperCase()+ "_" + side.toUpperCase() + "_FEMUR_MOMENT_RATING",      driver_mom_results[0],            "Femur moment rating",      "General");
					write_variable(f, dummy.toUpperCase()+ "_" + side.toUpperCase() + "_FEMUR_MOMENT_VALUE",       driver_mom_results[1].toFixed(3), "Femur moment value",       "General");		

				}
			}
			else
			{

				// No beam defined - variables should be set to blank by first script in template.
				// Create a blank image to let the user know.

				write_blank_images(output_data, "UNDEFINED BEAM ID")
			}
		}		

	}
}


function get_constant_val_results(c, good, accp, marg)
{
	// Calculates results for criteria that has constant limits
	// Returns the rating and value. 
	//
	// <c> is the value
	// <good>, <accp> and <marg> are the constant limits

	var ret = new Array(2);

	var val;
	var rating;

	if(good > 0) { val =  c; rating = IIHS_get_rating(val,  good,  accp,  marg); }
	else         { val = -c;    rating = IIHS_get_rating(val, -good, -accp, -marg); }

	ret[0] = rating;
	ret[1] = val;

	return ret;  
}

function draw_neck_axl_datums(t)
{
	// Creates datums for the neck axial force limits up to time <t>

	// Delete any that might already exist

	if(Datum.Exists("TEN GOOD"))       Datum.Delete("TEN GOOD");
	if(Datum.Exists("TEN ACCEPTABLE")) Datum.Delete("TEN ACCEPTABLE");
	if(Datum.Exists("TEN MARGINAL"))   Datum.Delete("TEN MARGINAL");
	if(Datum.Exists("COM GOOD"))       Datum.Delete("COM GOOD");
	if(Datum.Exists("COM ACCEPTABLE")) Datum.Delete("COM ACCEPTABLE");
	if(Datum.Exists("COM MARGINAL"))   Datum.Delete("COM MARGINAL");

	// Note that the order these are defined is important
	var d_com_marg = new Datum("COM MARGINAL",   Datum.CONSTANT_Y, -NECK_COM_MARGINAL);
	var d_com_accp = new Datum("COM ACCEPTABLE", Datum.CONSTANT_Y, -NECK_COM_ACCEPTABLE);
	var d_com_good = new Datum("COM GOOD",       Datum.CONSTANT_Y, -NECK_COM_GOOD);
	var d_ten_good = new Datum("TEN GOOD",       Datum.CONSTANT_Y,  NECK_TEN_GOOD);
	var d_ten_accp = new Datum("TEN ACCEPTABLE", Datum.CONSTANT_Y,  NECK_TEN_ACCEPTABLE);
	var d_ten_marg = new Datum("TEN MARGINAL",   Datum.CONSTANT_Y,  NECK_TEN_MARGINAL);

	d_ten_good.line_colour = EuroNCAPColour.GREEN;
	d_ten_accp.line_colour = EuroNCAPColour.YELLOW;
	d_ten_marg.line_colour = EuroNCAPColour.ORANGE;
	d_com_good.line_colour = EuroNCAPColour.GREEN;
	d_com_accp.line_colour = EuroNCAPColour.YELLOW;
	d_com_marg.line_colour = EuroNCAPColour.ORANGE;

	d_com_marg.fill_colour_below = EuroNCAPColour.RED;
	d_com_marg.fill_colour_above = EuroNCAPColour.ORANGE;
	d_com_accp.fill_colour_above = EuroNCAPColour.YELLOW;
	d_com_good.fill_colour_above = EuroNCAPColour.GREEN;
	d_ten_good.fill_colour_above = EuroNCAPColour.YELLOW;
	d_ten_accp.fill_colour_above = EuroNCAPColour.ORANGE;
	d_ten_marg.fill_colour_above = EuroNCAPColour.RED;
}
