/// <reference path="C:/SOURCE14/this_dir/this_js_api/this.intellisense.js" />
/// <reference path="C:/SOURCE14/reporter_dir/library/templates/scripts/this_common1.js" />
/// <reference path="C:/SOURCE14/reporter_dir/library/templates/scripts/this_common_IIHS.js" />


// Constants used throughout the script

var REG_DT     = 0.00001; // dt for regularising curves (in seconds)
var HIC_WINDOW = 0.015;   // HIC time window (in seconds)
var TMS_PERIOD = 0.003;   // 3ms period (in seconds)

// Head assessment constants

var HIC_GOOD       = 560.0;
var HIC_ACCEPTABLE = 700.0;
var HIC_MARGINAL   = 840.0;

// Neck assessment constants

var NECK_SHR_ADJ_FAC = 17.78;   // Factor to multiply shear by to adjust bending moment

var NIJ_TEN_FCRIT = 6.806;  // Critical neck tension     force  for Nij (kN)
var NIJ_COM_FCRIT = 6.160;  // Critical neck compression force  for Nij (kN)
var NIJ_FLX_MCRIT = 310;    // Critical neck flexion     moment for Nij (Nm)
var NIJ_EXT_MCRIT = 135;    // Critical neck extension   moment for Nij (Nm)

var NECK_NIJ_GOOD       = 0.80;
var NECK_NIJ_ACCEPTABLE = 1.00;
var NECK_NIJ_MARGINAL   = 1.20;

// Neck shear exceedence limit curves

var NECK_SHR_EXC_TIME = new Array(4);
var NECK_SHR_EXC_GOOD = new Array(4);

NECK_SHR_EXC_TIME[0] = 0.0;     NECK_SHR_EXC_GOOD[0] = 3.1;
NECK_SHR_EXC_TIME[1] = 0.025;   NECK_SHR_EXC_GOOD[1] = 1.5;
NECK_SHR_EXC_TIME[2] = 0.035;   NECK_SHR_EXC_GOOD[2] = 1.5;
NECK_SHR_EXC_TIME[3] = 0.045;   NECK_SHR_EXC_GOOD[3] = 1.1;

// Neck tension exceedence limit curves  

var NECK_TEN_EXC_TIME = new Array(3);
var NECK_TEN_EXC_GOOD = new Array(3);

NECK_TEN_EXC_TIME[0] = 0.0;     NECK_TEN_EXC_GOOD[0] = 3.3;
NECK_TEN_EXC_TIME[1] = 0.035;   NECK_TEN_EXC_GOOD[1] = 2.9;
NECK_TEN_EXC_TIME[2] = 0.045;   NECK_TEN_EXC_GOOD[2] = 1.1;

// Neck compression exceedence limit curves

var NECK_COM_EXC_TIME = new Array(2);
var NECK_COM_EXC_GOOD = new Array(2);

NECK_COM_EXC_TIME[0] = 0.0;     NECK_COM_EXC_GOOD[0] = 4.0;
NECK_COM_EXC_TIME[1] = 0.030;   NECK_COM_EXC_GOOD[1] = 1.1;


var NECK_TEN_GOOD       = 2.6;
var NECK_TEN_ACCEPTABLE = 3.3;
var NECK_TEN_MARGINAL   = 4.0;

var NECK_COM_GOOD       = 3.2;
var NECK_COM_ACCEPTABLE = 4.0;
var NECK_COM_MARGINAL   = 4.8;


// Chest assessment

// V7 and earlier dummies use a linear conversion: D = A*theta

var CHEST_V7_ROT_TO_COM_FACTOR = -139.0;   // Factor to convert chest transducer rotation to compression (mm)

//LSTC Dummy

var CHEST_LSTC_HIII_50TH_MALE_DETAILED_190217_BETA_ROT_TO_COM_FACTOR = -146.0; // Factor to convert chest transducer rotation to compression (mm)


// V8 dummies use a 3rd order polynomial conversion D = A*theta^3 - B*theta^2 - C*theta

var CHEST_V8_ROT_TO_COM_FACTOR_A =   25.13;   // First polynomial factor to convert chest transducer rotation to compression (mm)
var CHEST_V8_ROT_TO_COM_FACTOR_B =  -35.77;   // Second polynomial factor to convert chest transducer rotation to compression (mm)
var CHEST_V8_ROT_TO_COM_FACTOR_C = -136.26;   // Third polynomial factor to convert chest transducer rotation to compression (mm)

var CHEST_VC_A = 1.3;           // Viscous Criterion A constant (for m/s)
var CHEST_VC_B = 0.229;         // Viscous Criterion B constant (for m/s)
var CHEST_VC_METHOD = "ECER95"; // Viscous Criterion calculation method

var CHEST_COM_GOOD       = 50.0;  // 50mm
var CHEST_COM_ACCEPTABLE = 60.0;  // 60mm
var CHEST_COM_MARGINAL   = 75.0;  // 75mm

var CHEST_COM_RATE_GOOD       = 6.6;  // 6.6m/s
var CHEST_COM_RATE_ACCEPTABLE = 8.2;  // 8.2m/s
var CHEST_COM_RATE_MARGINAL   = 9.8;  // 9.8m/s

var CHEST_VC_GOOD       = 0.8;    // 0.8m/s
var CHEST_VC_ACCEPTABLE = 1.0;    // 1.0m/s
var CHEST_VC_MARGINAL   = 1.2;    // 1.2m/s

var CHEST_TMS_GOOD       = 60.0;  // 60g
var CHEST_TMS_ACCEPTABLE = 75.0;  // 75g
var CHEST_TMS_MARGINAL   = 90.0;  // 90g


// Thigh and Hip assessments

var THIGH_AND_HIP_GOOD_FORCE_1 = 5.22;
var THIGH_AND_HIP_GOOD_FORCE_2 = 5.69;

var THIGH_AND_HIP_ACCEPTABLE_FORCE_1 = 5.92;
var THIGH_AND_HIP_ACCEPTABLE_FORCE_2 = 7.69;

var THIGH_AND_HIP_MARGINAL_FORCE_1 = 6.38;
var THIGH_AND_HIP_MARGINAL_FORCE_2 = 8.92;

var THIGH_AND_HIP_GOOD_IMPULSE       = 113.5;
var THIGH_AND_HIP_ACCEPTABLE_IMPULSE = 127.7;
var THIGH_AND_HIP_MARGINAL_IMPULSE   = 137.1;


// Lower leg, knee and foot assessments

var KNEE_DISP_GOOD       = 12;
var KNEE_DISP_ACCEPTABLE = 15;
var KNEE_DISP_MARGINAL   = 18;

var TIBIA_COM_GOOD       = 4.0;
var TIBIA_COM_ACCEPTABLE = 6.0;
var TIBIA_COM_MARGINAL   = 8.0;

var TIBIA_INDEX_GOOD       = 0.8;
var TIBIA_INDEX_ACCEPTABLE = 1.0;
var TIBIA_INDEX_MARGINAL   = 1.2;

var TIBIA_MRC = 225;    // Tibia index critical moment = 225Nm
var TIBIA_FZC = 35.9;   // Tibia index critical force  = 35.9kN

var FOOT_ACCN_GOOD       = 150;
var FOOT_ACCN_ACCEPTABLE = 200;
var FOOT_ACCN_MARGINAL   = 260;

// Structural assessment

var LOWER_HINGE_PILLAR_INTRUSION_GOOD       = 15;
var LOWER_HINGE_PILLAR_INTRUSION_ACCEPTABLE = 22.5;
var LOWER_HINGE_PILLAR_INTRUSION_MARGINAL   = 30;

var FOOTREST_INTRUSION_GOOD       = 15;
var FOOTREST_INTRUSION_ACCEPTABLE = 22.5;
var FOOTREST_INTRUSION_MARGINAL   = 30;

var TOEPAN_INTRUSION_GOOD       = 15;
var TOEPAN_INTRUSION_ACCEPTABLE = 22.5;
var TOEPAN_INTRUSION_MARGINAL   = 30;

var BRAKE_PEDAL_INTRUSION_GOOD       = 15;
var BRAKE_PEDAL_INTRUSION_ACCEPTABLE = 22.5;
var BRAKE_PEDAL_INTRUSION_MARGINAL   = 30;

var PARKING_BRAKE_PEDAL_INTRUSION_GOOD       = 15;
var PARKING_BRAKE_PEDAL_INTRUSION_ACCEPTABLE = 22.5;
var PARKING_BRAKE_PEDAL_INTRUSION_MARGINAL   = 30;

var ROCKER_PANEL_INTRUSION_GOOD       = 5;
var ROCKER_PANEL_INTRUSION_ACCEPTABLE = 10;
var ROCKER_PANEL_INTRUSION_MARGINAL   = 15;

var STEERING_COLUMN_INTRUSION_GOOD       = 5;
var STEERING_COLUMN_INTRUSION_ACCEPTABLE = 10;
var STEERING_COLUMN_INTRUSION_MARGINAL   = 15;

var UPPER_HINGE_PILLAR_INTRUSION_GOOD       = 7.5;
var UPPER_HINGE_PILLAR_INTRUSION_ACCEPTABLE = 12.5;
var UPPER_HINGE_PILLAR_INTRUSION_MARGINAL   = 17.5;

var UPPER_DASH_INTRUSION_GOOD       = 7.5;
var UPPER_DASH_INTRUSION_ACCEPTABLE = 12.5;
var UPPER_DASH_INTRUSION_MARGINAL   = 17.5;

var INSTRUMENT_PANEL_INTRUSION_GOOD       = 7.5;
var INSTRUMENT_PANEL_INTRUSION_ACCEPTABLE = 12.5;
var INSTRUMENT_PANEL_INTRUSION_MARGINAL   = 17.5;


if(File.Exists(fname_csv) && File.IsFile(fname_csv))
{
    // Do the common setup of assigning values to the oGlblData object,
    // setting the T/HIS graph up and reading in the model.

    var oGlblData = new Object();

    var f_vars = do_common_single_analysis_setup(oGlblData);

    // Convert variable datums to model time

    convert_variable_datum_times();

    // For each body region, do the calculation, writing results to the variables file.
    // 1 here denotes the number of dummies
    do_head_rating_calc(f_vars, 1);

    do_neck_rating_calc(f_vars, 1, true);

    do_chest_rating_calc(f_vars);

    do_thigh_and_hip_calc(f_vars);

    do_lower_leg_and_foot_rating_calc(f_vars, true);

    do_structural_rating_calc(f_vars);

    f_vars.Close();
}



function do_thigh_and_hip_calc(f)
{
    // Calculates the thigh and hip values and writes them into the Reporter csv variables file <f>.

    // Two passes for the left and right hand side
    
    Message("Doing thigh and hip rating calculation...");
    
    var dummy = {};
    dummy.driver = {};
    dummy.driver.Left  = new DummyData("Driver", oGlblData.driver_left_femur_loadcell, "beam", oGlblData.driver_left_knee_transducer, "spring", oGlblData.driver_dummy, "version");
    dummy.driver.Right = new DummyData("Driver", oGlblData.driver_right_femur_loadcell, "beam", oGlblData.driver_right_knee_transducer, "spring", oGlblData.driver_dummy, "version");
 
    for(var i in dummy)
    {
        for(var lr in dummy[i])
        {
            // Femur compression forces

            var output_data = {};

            output_data.com = new OutputData(dummy[i][lr].chart_label + " " + lr + " Femur Axial Force",                   images_dir + "/" + dummy[i][lr].variable_label + "_" + lr + "_Femur_Compression");
            output_data.kth = new OutputData(dummy[i][lr].chart_label + " " + lr + " Femur Compressive Force vs. Impulse", images_dir + "/" + dummy[i][lr].variable_label + "_" + lr + "_KTH_Rating.png");

            if(dummy[i][lr].beam_id != undefined)
            {
                // Read in beam longitudinal force

                // For Humanetics dummies this is measured from a beam;
                // for LSTC dummies this is measured from a cross section
                switch (dummy[i][lr].version)
                {
                    case "V6_HUMANETICS_HIII_50TH_MALE":
                    case "V7_HUMANETICS_HIII_50TH_MALE":
                    case "V8_HUMANETICS_HIII_50TH_MALE":
                    case "Humanetics HIII 50M v1.5":
                    case "Humanetics HIII 50M v1.5.1":
                        var c_com = read_data("BEAM BASIC", 0, dummy[i][lr].beam_id, "NZ");  // NZ is longitudinal
                        var entity_str = "BEAM"; // Used for error message below
                        break;

                    case "LSTC HIII 50M Detailed v190217_beta":
                    case "LSTC HIII 50M Fast V2.0":
                    case "LSTC HIII 50M v130528_beta":                    
                        var c_com = read_data("SECTION", 0, dummy[i][lr].beam_id, "FZ");  // FZ is longitudinal
                        var entity_str = "SECTION"; // Used for error message below
                        break;

                    default:
                        ErrorMessage("Unsupported dummy \""+dummy[i][lr].version+"\".");
                        write_blank_images(output_data, "UNSUPPORTED DUMMY \""+dummy[i][lr].version+"\"");
                        break;
                }

                // Check that the curve read in - i.e. the beam id and component are valid
                // Create blank image to let the user know.

                if(!c_com)
                {        
                    write_blank_images(output_data, "NO DATA FOR " + entity_str + " " + dummy[i][lr].beam_id);
                }
                else
                {
                    // Convert forces to kN from model units

                    var c_com_kn = Operate.Div(c_com, oGlblData.kn_factor);

                    // C600 filter (convert to seconds first)

                    var c_com_c600 = convert_to_s_and_filter(c_com_kn, Operate.C600, REG_DT);

                    // Remove all curves and datums

                    remove_all_curves_and_datums();

                    // Convert from seconds back to model time

                    c_com_c600 = Operate.Mux(c_com_c600, oGlblData.time_factor);

                    // Set labels and style

                    set_labels(c_com_c600, lr + " Femur Axial Force", "Time (" + oGlblData.unit_time + ")", "Force (kN)");
                    set_line_style(c_com_c600, Colour.BLACK);

                    // Remove all curves and datums

                    remove_all_curves_and_datums();

                    // Create image/curve

                    c_com_c600.AddToGraph();
                    write_image(output_data.com.title, output_data.com.fname);
                    
                    output_data.com.curveList.push(c_com_c600.id);
                    write_curve("cur", output_data.com.curveList, output_data.com.fname);
                    write_curve("csv", output_data.com.curveList, output_data.com.fname);

                    // Calculate values and rating

                    var kth_results = get_kth_results(c_com_c600);

                    // Create image/curve of peak compressive force v impulse, with corridors

                    remove_all_curves_and_datums();

                    draw_kth_datums(kth_results[2]);

                    // Add in curve to show force v impulse point

                    var curve_id = Curve.FirstFreeID();
                    var c_force_v_impulse = new Curve(curve_id);

                    c_force_v_impulse.AddPoint(kth_results[1], kth_results[2]);

                    set_labels(c_force_v_impulse, lr + " Femur Compressive Force vs. Impulse", "Force (kN)", "Impulse (Ns)");
                    set_line_style(c_force_v_impulse, Colour.BLACK);

                    c_force_v_impulse.symbol = Symbol.CROSS;


                    // Autoscale so that x goes from 0 to max(force, marginal limit * 1.2)
                    //                   y goes from 0 to max(impulse, marginal limit *1.5)

                    var xmin = 0.0;
                    var xmax = Math.max(kth_results[1], 1.2 * THIGH_AND_HIP_MARGINAL_FORCE_2);

                    var ymin = 0.0;
                    var ymax = Math.max(kth_results[2], 1.5 * THIGH_AND_HIP_MARGINAL_IMPULSE);

                    DialogueInput("DE AU XMN " + xmin);
                    DialogueInput("DE AU XMX " + xmax);
                    DialogueInput("DE AU YMN " + ymin);
                    DialogueInput("DE AU YMX " + ymax);
                    DialogueInput("/PLOT");

                    DialogueInput("/DEFAULT TI",  output_data.kth.title);
                    DialogueInput("/IMAGE PNG24", output_data.kth.fname, "GRAPH", "1");
                    
                    output_data.kth.curveList.push(c_force_v_impulse.id);
                    write_curve("cur", output_data.kth.curveList, output_data.kth.fname);
                    write_curve("csv", output_data.kth.curveList, output_data.kth.fname);

                    // Write variables

                    write_variable(f, dummy[i][lr].variable_label.toUpperCase() + "_" + lr.toUpperCase() + "_KTH_RATING",               kth_results[0],            lr + " Knee Thigh Hip rating",   "General");
                    write_variable(f, dummy[i][lr].variable_label.toUpperCase() + "_" + lr.toUpperCase() + "_FEMUR_COMPRESSION_VALUE",  kth_results[1].toFixed(2), lr + " Femur Compression value", "General");
                    write_variable(f, dummy[i][lr].variable_label.toUpperCase() + "_" + lr.toUpperCase() + "_FEMUR_IMPULSE_VALUE",      kth_results[2].toFixed(2), lr + " Femur Impulse value",     "General");
                }
            }
            else
            {
                // No beam defined - variables should be set to blank by first script in template.
                // Create a blank image to let the user know.

                write_blank_images(output_data, "NO BEAM ID DEFINED FOR");
            }
        }
    }
}



function do_structural_rating_calc(f)
{
    // Calculates the structural values and writes them into the Reporter csv variables file <f>.


    // 6 Upper Compartment structural measurements

    var max_hinge_resultant = -1e20;

    for(var pass=0; pass<6; pass++)
    {
        var springs   = new Array(3);
        var structure = "";

        if(pass == 0)
        {
            springs[0] = oGlblData.steering_column_intrusion_spring_x;

            structure = "Steering_Column";
        }
        else if(pass == 1)
        {
            springs[0] = oGlblData.instrument_panel_intrusion_spring_x;
            springs[1] = oGlblData.instrument_panel_intrusion_spring_y;
            springs[2] = oGlblData.instrument_panel_intrusion_spring_z;

            structure = "Instrument_Panel";
        }
        else if(pass == 2)
        {
            springs[0] = oGlblData.upper_dash_intrusion_spring_x;
            springs[1] = oGlblData.upper_dash_intrusion_spring_y;
            springs[2] = oGlblData.upper_dash_intrusion_spring_z;

            structure = "Upper_Dash";
        }
        else if(pass == 3)
        {
            springs[0] = oGlblData.upper_hinge_1_intrusion_spring_x;
            springs[1] = oGlblData.upper_hinge_1_intrusion_spring_y;
            springs[2] = oGlblData.upper_hinge_1_intrusion_spring_z;

            structure = "Upper_Hinge_Pillar_1";
        }
        else if(pass == 4)
        {
            springs[0] = oGlblData.upper_hinge_2_intrusion_spring_x;
            springs[1] = oGlblData.upper_hinge_2_intrusion_spring_y;
            springs[2] = oGlblData.upper_hinge_2_intrusion_spring_z;

            structure = "Upper_Hinge_Pillar_2";
        }
        else if(pass == 5)
        {
            springs[0] = oGlblData.upper_hinge_3_intrusion_spring_x;
            springs[1] = oGlblData.upper_hinge_3_intrusion_spring_y;
            springs[2] = oGlblData.upper_hinge_3_intrusion_spring_z;

            structure = "Upper_Hinge_Pillar_3";
        }

        var output_data = {};

        output_data.int = new OutputData(structure + " Intrusion", images_dir + "/" + structure + "_Intrusion");

        if( (pass > 0 && ((springs[0] != undefined)   ||
                        (springs[1] != undefined)   ||
                        (springs[2] != undefined))) ||
                (pass == 0 && (springs[0] != undefined)) )
        {

            var c_fa = read_data("SPRING TR", 0, springs[0], "ET");

            if(pass > 0)  // Get Y and Z displacements
            {
                var c_lat  = read_data("SPRING TR", 0, springs[1], "ET");
                var c_vert = read_data("SPRING TR", 0, springs[2], "ET");
            }

            if(!c_fa || (pass > 0 && (!c_lat || !c_vert)))
            {
                var spring_id = [];
                if(!c_fa){spring_id.push(springs[0]);} //Distinguishing the spring axis having no data
                if(pass>0 && !c_lat){spring_id.push(springs[1]);}
                if(pass>0 && !c_vert){spring_id.push(springs[2]);}
                write_blank_images(output_data, "NO DATA FOR SPRING: " + spring_id.join(", "));
            }
            else
            {
                // Remove all curves and datums

                remove_all_curves_and_datums();

                // Convert to mm

                var c_fa_cm = Operate.Div(c_fa, oGlblData.cm_factor);

                if(pass > 0)
                {
                    var c_lat_cm  = Operate.Div(c_lat,  oGlblData.cm_factor);
                    var c_vert_cm = Operate.Div(c_vert, oGlblData.cm_factor);
                }

                // Get final displacements

                var p = c_fa_cm.GetPoint(c_fa_cm.npoints);
                var fa_disp = p[1];

                if(pass > 0)
                {
                    var p = c_lat_cm.GetPoint(c_lat_cm.npoints);
                    var lat_disp = p[1];

                    var p = c_vert_cm.GetPoint(c_vert_cm.npoints);
                    var vert_disp = p[1];

                    // Resultant displacement 
                    //
                    // For upper hinge pillar, upper dash and instrument panel, if X is forward (+ve)
                    // don't use X.
                    //
                    // For the upper hinge pillar, if Y is outboard (-ve), don't use Y.
                    //
                    var res_disp;

                    if(pass == 1 || pass == 2)  // Instrument panel and upper dash
                    {
                        if(fa_disp < 0.0)  // Use X, Y and Z
                        {
                            res_disp = Math.sqrt(fa_disp   * fa_disp  + 
                                    lat_disp  * lat_disp +
                                    vert_disp * vert_disp);
                        }
                        else  // Only use Y and Z
                        {
                            res_disp = Math.sqrt(lat_disp  * lat_disp +
                                    vert_disp * vert_disp);
                        }
                    }
                    else // Upper hinge pillar
                    {
                        if(fa_disp < 0.0 && lat_disp > 0.0)  // Use X, Y and Z
                        {
                            res_disp = Math.sqrt(fa_disp   * fa_disp  + 
                                    lat_disp  * lat_disp +
                                    vert_disp * vert_disp);
                        }
                        else
                        {
                            if(fa_disp < 0.0)  // Use X and Z
                            {
                                res_disp = Math.sqrt(fa_disp   * fa_disp  + 
                                        vert_disp * vert_disp);
                            }
                            else if(lat_disp > 0.0) // Use Y and Z
                            {
                                res_disp = Math.sqrt(lat_disp  * lat_disp +
                                        vert_disp * vert_disp);
                            }
                            else // Just use Z
                            {
                                res_disp = Math.sqrt(vert_disp * vert_disp);
                            }
                        }
                    }
                }


                // Get the maximum hinge resultant

                if(pass > 2) max_hinge_resultant = Math.max(max_hinge_resultant, res_disp);

                // Set label and style

                set_labels(c_fa_cm, "Fore/Aft Intrusion (-ve rearward)", "Time (" + oGlblData.unit_time + ")", "Intrusion (cm)");
                set_line_style(c_fa_cm, Colour.RED);
                output_data.int.curveList.push(c_fa_cm.id);

                if(pass > 0)
                {
                    set_labels(c_lat_cm, "Lateral Intrusion", "Time (" + oGlblData.unit_time + ")", "Intrusion (cm)");
                    set_line_style(c_lat_cm, Colour.BLUE);
                    output_data.int.curveList.push(c_lat_cm.id);

                    set_labels(c_vert_cm, "Vertical Intrusion (+ve upward)", "Time (" + oGlblData.unit_time + ")", "Intrusion (cm)");
                    set_line_style(c_vert_cm, Colour.GREEN);
                    output_data.int.curveList.push(c_vert_cm.id);
                }

                // Create image/curve

                write_image(output_data.int.title, output_data.int.fname);
                
                write_curve("cur", output_data.int.curveList, output_data.int.fname);
                write_curve("csv", output_data.int.curveList, output_data.int.fname);

                // Calculate ratings

                var rating = "GOOD";

                if(pass == 0) rating = IIHS_get_rating(-fa_disp, STEERING_COLUMN_INTRUSION_GOOD,    STEERING_COLUMN_INTRUSION_ACCEPTABLE,    STEERING_COLUMN_INTRUSION_MARGINAL);
                if(pass == 1) rating = IIHS_get_rating(res_disp, INSTRUMENT_PANEL_INTRUSION_GOOD,   INSTRUMENT_PANEL_INTRUSION_ACCEPTABLE,   INSTRUMENT_PANEL_INTRUSION_MARGINAL);
                if(pass == 2) rating = IIHS_get_rating(res_disp, UPPER_DASH_INTRUSION_GOOD,         UPPER_DASH_INTRUSION_ACCEPTABLE,         UPPER_DASH_INTRUSION_MARGINAL);
                if(pass == 3) rating = IIHS_get_rating(res_disp, UPPER_HINGE_PILLAR_INTRUSION_GOOD, UPPER_HINGE_PILLAR_INTRUSION_ACCEPTABLE, UPPER_HINGE_PILLAR_INTRUSION_MARGINAL);
                if(pass == 4) rating = IIHS_get_rating(res_disp, UPPER_HINGE_PILLAR_INTRUSION_GOOD, UPPER_HINGE_PILLAR_INTRUSION_ACCEPTABLE, UPPER_HINGE_PILLAR_INTRUSION_MARGINAL);
                if(pass == 5) rating = IIHS_get_rating(res_disp, UPPER_HINGE_PILLAR_INTRUSION_GOOD, UPPER_HINGE_PILLAR_INTRUSION_ACCEPTABLE, UPPER_HINGE_PILLAR_INTRUSION_MARGINAL);

                // Write variables

                write_variable(f, "STRUCTURAL_" + structure.toUpperCase() + "_X", fa_disp.toFixed(2), structure + " X Intrusion", "General");

                if(pass > 0)
                {
                    write_variable(f, "STRUCTURAL_" + structure.toUpperCase() + "_Y",         lat_disp.toFixed(2),  structure + " Y Intrusion",         "General");
                    write_variable(f, "STRUCTURAL_" + structure.toUpperCase() + "_Z",         vert_disp.toFixed(2), structure + " Z Intrusion",         "General");
                    write_variable(f, "STRUCTURAL_" + structure.toUpperCase() + "_RESULTANT", res_disp.toFixed(2),  structure + " resultant Intrusion", "General");
                }

                write_variable(f, "STRUCTURAL_" + structure.toUpperCase() + "_RATING", rating, structure + " Intrusion Rating", "General");
            }
        }
        else
        {
            write_blank_images(output_data, "NO SPRING ID DEFINED FOR");
        }
    }

    // For the upper hinge, we want the worst rating

    rating = IIHS_get_rating(max_hinge_resultant, UPPER_HINGE_PILLAR_INTRUSION_GOOD, UPPER_HINGE_PILLAR_INTRUSION_ACCEPTABLE, UPPER_HINGE_PILLAR_INTRUSION_MARGINAL);

    write_variable(f, "STRUCTURAL_UPPER_HINGE_PILLAR_RESULTANT", max_hinge_resultant.toFixed(2), "Upper Hinge Pillar resultant Intrusion",        "General");
    write_variable(f, "STRUCTURAL_UPPER_HINGE_PILLAR_RATING",    rating,                         "Upper Hinge Pillar resultant Intrusion Rating", "General");  




    // Now do lower compartment - 10 Lower Compartment structural measurements

    var max_hinge_resultant = -1e20;
    var tot_rocker_panel    =  0.0;

    for(var pass=0; pass<10; pass++)
    {
        var springs   = new Array(3);
        var structure = "";

        if(pass == 0)
        {
            springs[0] = oGlblData.brake_pedal_intrusion_spring_x;
            springs[1] = oGlblData.brake_pedal_intrusion_spring_y;
            springs[2] = oGlblData.brake_pedal_intrusion_spring_z;

            structure = "Brake_Pedal";
        }
        else if(pass == 1)
        {
            springs[0] = oGlblData.parking_brake_pedal_intrusion_spring_x;
            springs[1] = oGlblData.parking_brake_pedal_intrusion_spring_y;
            springs[2] = oGlblData.parking_brake_pedal_intrusion_spring_z;

            structure = "Parking_Brake_Pedal";
        }
        else if(pass == 2)
        {
            springs[0] = oGlblData.footrest_intrusion_spring_x;
            springs[1] = oGlblData.footrest_intrusion_spring_y;
            springs[2] = oGlblData.footrest_intrusion_spring_z;

            structure = "Footrest";
        }
        else if(pass == 3)
        {
            springs[0] = oGlblData.toepan_intrusion_spring_x;
            springs[1] = oGlblData.toepan_intrusion_spring_y;
            springs[2] = oGlblData.toepan_intrusion_spring_z;

            structure = "Toepan";
        }
        else if(pass == 4)
        {
            springs[0] = oGlblData.lower_hinge_1_intrusion_spring_x;
            springs[1] = oGlblData.lower_hinge_1_intrusion_spring_y;
            springs[2] = oGlblData.lower_hinge_1_intrusion_spring_z;

            structure = "Lower_Hinge_Pillar_1";
        }
        else if(pass == 5)
        {
            springs[0] = oGlblData.lower_hinge_2_intrusion_spring_x;
            springs[1] = oGlblData.lower_hinge_2_intrusion_spring_y;
            springs[2] = oGlblData.lower_hinge_2_intrusion_spring_z;

            structure = "Lower_Hinge_Pillar_2";
        }
        else if(pass == 6)
        {
            springs[0] = oGlblData.lower_hinge_3_intrusion_spring_x;
            springs[1] = oGlblData.lower_hinge_3_intrusion_spring_y;
            springs[2] = oGlblData.lower_hinge_3_intrusion_spring_z;

            structure = "Lower_Hinge_Pillar_3";
        }
        else if(pass == 7)
        {
            springs[1] = oGlblData.rocker_panel_1_intrusion_spring_y;

            structure = "Rocker_Panel_1";
        }
        else if(pass == 8)
        {
            springs[1] = oGlblData.rocker_panel_2_intrusion_spring_y;

            structure = "Rocker_Panel_2";
        }
        else if(pass == 9)
        {
            springs[1] = oGlblData.rocker_panel_3_intrusion_spring_y;

            structure = "Rocker_Panel_3";
        }


        var output_data = {};

        output_data.int = new OutputData(structure + " Intrusion", images_dir + "/" + structure + "_Intrusion");

        if( (pass < 7 && ((springs[0] != undefined)   ||
                        (springs[1] != undefined)   ||
                        (springs[2] != undefined))) ||
                (pass >= 7 && (springs[1] != undefined)) )
        {

            var c_lat = read_data("SPRING TR", 0, springs[1], "ET");

            if(pass < 7)  // Get X and Z displacements
            {
                var c_fa   = read_data("SPRING TR", 0, springs[0], "ET");
                var c_vert = read_data("SPRING TR", 0, springs[2], "ET");
            }

            if(!c_lat || (pass < 7 && (!c_fa || !c_vert)))
            {
                var spring_id = [];
                if(!c_lat){spring_id.push(springs[1]);} //Distinguishing the spring axis having no data
                if(pass<7 && !c_fa){spring_id.push(springs[0]);}
                if(pass<7 && !c_vert){spring_id.push(springs[2]);}
                write_blank_images(output_data, "NO DATA FOR SPRING: " + spring_id.join(", "));
            }
            else
            {
                // Remove all curves and datums

                remove_all_curves_and_datums();

                // Convert to mm

                var c_lat_cm  = Operate.Div(c_lat, oGlblData.cm_factor);

                if(pass < 7)
                {
                    var c_fa_cm   = Operate.Div(c_fa,   oGlblData.cm_factor);
                    var c_vert_cm = Operate.Div(c_vert, oGlblData.cm_factor);
                }

                // Get final displacements

                var p = c_lat_cm.GetPoint(c_lat_cm.npoints);
                var lat_disp = p[1];

                if(pass < 7)
                {
                    var p = c_fa_cm.GetPoint(c_fa_cm.npoints);
                    var fa_disp = p[1];

                    var p = c_vert_cm.GetPoint(c_vert_cm.npoints);
                    var vert_disp = p[1];

                    // Resultant displacement 
                    //
                    // For lower hinge pillar, footrest, toepan, brake pedal and parking brake,
                    // if X is forward (+ve) don't use X.
                    //
                    // For the lower hinge pillar and parking brake, 
                    // if Y is outboard (-ve), don't use Y.
                    //
                    var res_disp;

                    if(pass == 0 || pass == 2 || pass == 3)  // Brake pedal, Footrest or Toepan
                    {
                        if(fa_disp < 0.0)  // Use X, Y and Z
                        {
                            res_disp = Math.sqrt(fa_disp   * fa_disp  + 
                                    lat_disp  * lat_disp +
                                    vert_disp * vert_disp);
                        }
                        else  // Only use Y and Z
                        {
                            res_disp = Math.sqrt(lat_disp  * lat_disp +
                                    vert_disp * vert_disp);
                        }
                    }
                    else if((pass >= 4 && pass <= 6) || pass == 1)  // Lower Hinge Pillar or Parking Brake
                    {
                        if(fa_disp < 0.0 && lat_disp > 0.0)  // Use X, Y and Z
                        {
                            res_disp = Math.sqrt(fa_disp   * fa_disp  + 
                                    lat_disp  * lat_disp +
                                    vert_disp * vert_disp);
                        }
                        else
                        {
                            if(fa_disp < 0.0)  // Use X and Z
                            {
                                res_disp = Math.sqrt(fa_disp   * fa_disp  + 
                                        vert_disp * vert_disp);
                            }
                            else if(lat_disp > 0.0) // Use Y and Z
                            {
                                res_disp = Math.sqrt(lat_disp  * lat_disp +
                                        vert_disp * vert_disp);
                            }
                            else // Just use Z
                            {
                                res_disp = Math.sqrt(vert_disp * vert_disp);
                            }
                        }
                    }
                }
                else // Rocker panel total
                {
                    tot_rocker_panel += lat_disp;
                }

                // Get the maximum hinge resultant

                if(pass >= 4 && pass <= 6) max_hinge_resultant = Math.max(max_hinge_resultant, res_disp);

                // Set label and style

                set_labels(c_lat_cm, "Lateral Intrusion", "Time (" + oGlblData.unit_time + ")", "Intrusion (cm)");
                set_line_style(c_lat_cm, Colour.BLUE);
                output_data.int.curveList.push(c_lat_cm.id);

                if(pass < 7)
                {
                    set_labels(c_fa_cm, "Fore/Aft Intrusion (-ve rearward)", "Time (" + oGlblData.unit_time + ")", "Intrusion (cm)");
                    set_line_style(c_fa_cm, Colour.RED);
                    output_data.int.curveList.push(c_fa_cm.id);

                    set_labels(c_vert_cm, "Vertical Intrusion (-ve rearward)", "Time (" + oGlblData.unit_time + ")", "Intrusion (cm)");
                    set_line_style(c_vert_cm, Colour.GREEN);
                    output_data.int.curveList.push(c_vert_cm.id);
                }

                // Create image/curve

                write_image(output_data.int.title, output_data.int.fname);
                
                write_curve("cur", output_data.int.curveList, output_data.int.fname);
                write_curve("csv", output_data.int.curveList, output_data.int.fname);

                // Calculate ratings

                var rating = "GOOD";

                if(pass == 0) rating = IIHS_get_rating(res_disp, BRAKE_PEDAL_INTRUSION_GOOD,         BRAKE_PEDAL_INTRUSION_ACCEPTABLE,         BRAKE_PEDAL_INTRUSION_MARGINAL);
                if(pass == 1) rating = IIHS_get_rating(res_disp, PARKING_BRAKE_PEDAL_INTRUSION_GOOD, PARKING_BRAKE_PEDAL_INTRUSION_ACCEPTABLE, PARKING_BRAKE_PEDAL_INTRUSION_MARGINAL);
                if(pass == 2) rating = IIHS_get_rating(res_disp, FOOTREST_INTRUSION_GOOD,            FOOTREST_INTRUSION_ACCEPTABLE,            FOOTREST_INTRUSION_MARGINAL);
                if(pass == 3) rating = IIHS_get_rating(res_disp, TOEPAN_INTRUSION_GOOD,              TOEPAN_INTRUSION_ACCEPTABLE,              TOEPAN_INTRUSION_MARGINAL);
                if(pass == 4) rating = IIHS_get_rating(res_disp, LOWER_HINGE_PILLAR_INTRUSION_GOOD,  LOWER_HINGE_PILLAR_INTRUSION_ACCEPTABLE,  LOWER_HINGE_PILLAR_INTRUSION_MARGINAL);
                if(pass == 5) rating = IIHS_get_rating(res_disp, LOWER_HINGE_PILLAR_INTRUSION_GOOD,  LOWER_HINGE_PILLAR_INTRUSION_ACCEPTABLE,  LOWER_HINGE_PILLAR_INTRUSION_MARGINAL);
                if(pass == 6) rating = IIHS_get_rating(res_disp, LOWER_HINGE_PILLAR_INTRUSION_GOOD,  LOWER_HINGE_PILLAR_INTRUSION_ACCEPTABLE,  LOWER_HINGE_PILLAR_INTRUSION_MARGINAL);
                if(pass == 7) rating = IIHS_get_rating(lat_disp, ROCKER_PANEL_INTRUSION_GOOD,        ROCKER_PANEL_INTRUSION_ACCEPTABLE,        ROCKER_PANEL_INTRUSION_MARGINAL);
                if(pass == 8) rating = IIHS_get_rating(lat_disp, ROCKER_PANEL_INTRUSION_GOOD,        ROCKER_PANEL_INTRUSION_ACCEPTABLE,        ROCKER_PANEL_INTRUSION_MARGINAL);
                if(pass == 9) rating = IIHS_get_rating(lat_disp, ROCKER_PANEL_INTRUSION_GOOD,        ROCKER_PANEL_INTRUSION_ACCEPTABLE,        ROCKER_PANEL_INTRUSION_MARGINAL);

                // Write variables

                write_variable(f, "STRUCTURAL_" + structure.toUpperCase() + "_Y", lat_disp.toFixed(2), structure + " Y Intrusion", "General");

                if(pass < 7)
                {
                    write_variable(f, "STRUCTURAL_" + structure.toUpperCase() + "_X",         fa_disp.toFixed(2),   structure + " X Intrusion",         "General");
                    write_variable(f, "STRUCTURAL_" + structure.toUpperCase() + "_Z",         vert_disp.toFixed(2), structure + " Z Intrusion",         "General");
                    write_variable(f, "STRUCTURAL_" + structure.toUpperCase() + "_RESULTANT", res_disp.toFixed(2),  structure + " resultant Intrusion", "General");
                }

                write_variable(f, "STRUCTURAL_" + structure.toUpperCase() + "_RATING", rating, structure + " Intrusion Rating", "General");
            }
        }
        else
        {
            write_blank_images(output_data, "NO SPRING ID DEFINED FOR");
        }
    }

    // For the lower hinge, we want the worst rating

    rating = IIHS_get_rating(max_hinge_resultant, LOWER_HINGE_PILLAR_INTRUSION_GOOD, LOWER_HINGE_PILLAR_INTRUSION_ACCEPTABLE, LOWER_HINGE_PILLAR_INTRUSION_MARGINAL);

    write_variable(f, "STRUCTURAL_LOWER_HINGE_PILLAR_RESULTANT", max_hinge_resultant.toFixed(2), "Lower Hinge Pillar resultant Intrusion",        "General");
    write_variable(f, "STRUCTURAL_LOWER_HINGE_PILLAR_RATING",    rating,                         "Lower Hinge Pillar resultant Intrusion Rating", "General");  


    // For the rocker panel, we want the average value of the three measurements

    var avg_rocker_panel = tot_rocker_panel / 3;

    rating = IIHS_get_rating(avg_rocker_panel, ROCKER_PANEL_INTRUSION_GOOD, ROCKER_PANEL_INTRUSION_ACCEPTABLE, ROCKER_PANEL_INTRUSION_MARGINAL);

    write_variable(f, "STRUCTURAL_ROCKER_PANEL_Y_AVERAGE", avg_rocker_panel.toFixed(2), "Rocker Panel average Y Intrusion",        "General");
    write_variable(f, "STRUCTURAL_ROCKER_PANEL_RATING",    rating,                      "Rocker Panel average Y Intrusion Rating", "General"); 

}



function get_kth_results(c)
{
    // Calculates results for the thigh and hip assessment returning
    // the rating, peak compressive force and impulse in an array. 
    //
    // <c> is the femur axial force curve.
    //
    // To calculate the Knee-Thigh-Hip (KTH) rating we need the peak compressive force
    // and the time when the force last equals zero prior to the peak compressive force.
    //
    // The curve is then integrated from this time up to the time after the peak force when
    // the compressive force equals 4.050kN. This is the force impulse.
    //
    // The peak and the impulse are then compared to corridor limits to get a rating.
    //

    var ret = new Array(3); 

    var rating = "GOOD";

    var peak_com         = -c.ymin;
    var time_of_peak_com =  c.x_at_ymin;

    // Find the last time when the force equals zero prior to the peak compressive force
    // and the first time when the compressive force equals 4050N after the peak force

    var time_pre_peak_com  = 0.0;
    var time_post_peak_com = 0.0;
    var stop = false;

    for(var i=1; i<=c.npoints; i++)
    {
        var p = c.GetPoint(i);

        if(p[0] < time_of_peak_com)
        {
            if(p[1] >= 0.0) time_pre_peak_com = p[0];
        }

        if(p[0] > time_of_peak_com)
        {
            if(p[1] <= -4.050) time_post_peak_com = p[0];
            else               stop = true;
        }

        if(stop) break;
    }

    // If the peak compressive force is the very last point, then <time_post_peak_com>
    // will still be zero, so set it to <time_of_peak_com>

    if(time_post_peak_com == 0.0) time_post_peak_com = time_of_peak_com;

    Message("CLIP between: " + time_pre_peak_com + ", " + time_post_peak_com);

    // Clip the curve between these times and then integrate

    var c_com_clipped = Operate.Clip(c, time_pre_peak_com,   // Minimum X
            time_post_peak_com,  // Maximum X
            -1e20,               // Minimum Y
            0.0);               // Maximum Y

    var c_com_int = Operate.Int(c_com_clipped);

    c_com_int = Operate.Mul(c_com_int, -oGlblData.kn_factor); // Convert from -ve kNs to +ve Ns

    // Get last value to get impulse

    var p = c_com_int.GetPoint(c_com_int.npoints);

    var impulse = p[1];

    // Calculate rating based on defined corridors

    // Split the corridors up into rectangles and see which one the values lie within.
    //
    //  (1) and (2)      are GOOD
    //  (3), (4) and (5) are ACCEPTABLE
    //  (6), (7) and (8) are MARGINAL
    //  (9) and (10)     are POOR
    //
    //
    //                 ^
    //                 |        |      |   |          *
    //                 |        |      |(6)|   (9)    *
    //                 |        |      |   |          *
    //                 |   (1)  |  (3) |***+----------+
    //                 |        |      |     (7)      |
    //                 |        |******+---+**********|
    //   Impulse(Ns)   |        |    (4)   |          |    (10)
    //                 |        +----+*****|          |
    //                 |        *    |     |          |
    //                 |        *    |     |          |
    //                 |        *    |     |          |
    //                 |        *(2) | (5) |   (8)    |
    //                 |        *    |     |          |
    //                 |        *    |     |          |
    //                 +------------------------------------------>
    //                                   Force(kN)
    //

    if(peak_com <= THIGH_AND_HIP_GOOD_FORCE_1)              // Rectangle (1)
    {
        rating = "GOOD";
    }
    else if(peak_com <= THIGH_AND_HIP_GOOD_FORCE_2 && 
            impulse  <= THIGH_AND_HIP_GOOD_IMPULSE)         // Rectangle (2)
            {
                rating = "GOOD";
            }
    else if(peak_com <= THIGH_AND_HIP_ACCEPTABLE_FORCE_1 && 
            peak_com >= THIGH_AND_HIP_GOOD_FORCE_1       &&
            impulse  >= THIGH_AND_HIP_ACCEPTABLE_IMPULSE)   // Rectangle (3)
            {
                rating = "ACCEPTABLE";
            }
    else if(peak_com <= THIGH_AND_HIP_ACCEPTABLE_FORCE_2 && 
            peak_com >= THIGH_AND_HIP_GOOD_FORCE_1       &&
            impulse  <= THIGH_AND_HIP_ACCEPTABLE_IMPULSE &&
            impulse  >= THIGH_AND_HIP_GOOD_IMPULSE)         // Rectangle (4)
            {
                rating = "ACCEPTABLE";
            }
    else if(peak_com <= THIGH_AND_HIP_ACCEPTABLE_FORCE_2 && 
            peak_com >= THIGH_AND_HIP_GOOD_FORCE_2       &&
            impulse  <= THIGH_AND_HIP_GOOD_IMPULSE)         // Rectangle (5)
            {
                rating = "ACCEPTABLE";
            }
    else if(peak_com <= THIGH_AND_HIP_MARGINAL_FORCE_1   && 
            peak_com >= THIGH_AND_HIP_ACCEPTABLE_FORCE_1 &&
            impulse  >= THIGH_AND_HIP_MARGINAL_IMPULSE)     // Rectangle (6)
            {
                rating = "MARGINAL";
            }
    else if(peak_com <= THIGH_AND_HIP_MARGINAL_FORCE_2   && 
            peak_com >= THIGH_AND_HIP_ACCEPTABLE_FORCE_1 &&
            impulse  <= THIGH_AND_HIP_MARGINAL_IMPULSE   &&
            impulse  >= THIGH_AND_HIP_ACCEPTABLE_IMPULSE)   // Rectangle (7)
            {
                rating = "MARGINAL";
            }
    else if(peak_com <= THIGH_AND_HIP_MARGINAL_FORCE_2   && 
            peak_com >= THIGH_AND_HIP_ACCEPTABLE_FORCE_2 &&
            impulse  <= THIGH_AND_HIP_ACCEPTABLE_IMPULSE)   // Rectangle (8)
            {
                rating = "MARGINAL";
            }
    else
    {
        rating = "POOR";
    }

    ret[0] = rating;
    ret[1] = peak_com;
    ret[2] = impulse;

    return ret;
}



function draw_kth_datums(max_impulse)
{
    // Creates curves for the KTH injury criteria limits up to impulse <max_impulse>

    // Make sure we show all limits

    if(max_impulse < 1.5 * THIGH_AND_HIP_MARGINAL_IMPULSE) 
        max_impulse = 1.5 * THIGH_AND_HIP_MARGINAL_IMPULSE;

    // Delete any that might already exist

    if(Datum.Exists("GOOD"))       Datum.Delete("GOOD");
    if(Datum.Exists("ACCEPTABLE")) Datum.Delete("ACCEPTABLE");
    if(Datum.Exists("MARGINAL"))   Datum.Delete("MARGINAL");

    // Create new ones and colour in

    var good_limit = new Array();
    var accp_limit = new Array();
    var marg_limit = new Array();

    good_limit[0] = THIGH_AND_HIP_GOOD_FORCE_1;  good_limit[1] = max_impulse;
    good_limit[2] = THIGH_AND_HIP_GOOD_FORCE_1;  good_limit[3] = THIGH_AND_HIP_GOOD_IMPULSE;
    good_limit[4] = THIGH_AND_HIP_GOOD_FORCE_2;  good_limit[5] = THIGH_AND_HIP_GOOD_IMPULSE;
    good_limit[6] = THIGH_AND_HIP_GOOD_FORCE_2;  good_limit[7] = 0.0;

    accp_limit[0] = THIGH_AND_HIP_ACCEPTABLE_FORCE_1;  accp_limit[1] = max_impulse;
    accp_limit[2] = THIGH_AND_HIP_ACCEPTABLE_FORCE_1;  accp_limit[3] = THIGH_AND_HIP_ACCEPTABLE_IMPULSE;
    accp_limit[4] = THIGH_AND_HIP_ACCEPTABLE_FORCE_2;  accp_limit[5] = THIGH_AND_HIP_ACCEPTABLE_IMPULSE;
    accp_limit[6] = THIGH_AND_HIP_ACCEPTABLE_FORCE_2;  accp_limit[7] = 0.0;

    marg_limit[0] = THIGH_AND_HIP_MARGINAL_FORCE_1;  marg_limit[1] = max_impulse;
    marg_limit[2] = THIGH_AND_HIP_MARGINAL_FORCE_1;  marg_limit[3] = THIGH_AND_HIP_MARGINAL_IMPULSE;
    marg_limit[4] = THIGH_AND_HIP_MARGINAL_FORCE_2;  marg_limit[5] = THIGH_AND_HIP_MARGINAL_IMPULSE;
    marg_limit[6] = THIGH_AND_HIP_MARGINAL_FORCE_2;  marg_limit[7] = 0.0;


    var d_good = new Datum("GOOD",       Datum.POINTS, good_limit);
    var d_accp = new Datum("ACCEPTABLE", Datum.POINTS, accp_limit);
    var d_marg = new Datum("MARGINAL",   Datum.POINTS, marg_limit);

    d_good.line_colour = EuroNCAPColour.GREEN;
    d_accp.line_colour = EuroNCAPColour.YELLOW;
    d_marg.line_colour = EuroNCAPColour.ORANGE;

    d_good.fill_type = Datum.FILL_RIGHT_LEFT;
    d_accp.fill_type = Datum.FILL_RIGHT_LEFT;
    d_marg.fill_type = Datum.FILL_RIGHT_LEFT;

    d_good.fill_colour_left  = EuroNCAPColour.GREEN;
    d_good.fill_colour_right = EuroNCAPColour.YELLOW;
    d_accp.fill_colour_right = EuroNCAPColour.ORANGE;
    d_marg.fill_colour_right = EuroNCAPColour.RED;
}



function convert_variable_datum_times()
{
    // Convert the variable datum times from seconds to the model units

    for(var i=0; i<NECK_SHR_EXC_TIME.length; i++) NECK_SHR_EXC_TIME[i] *= oGlblData.time_factor;
    for(var i=0; i<NECK_TEN_EXC_TIME.length; i++) NECK_TEN_EXC_TIME[i] *= oGlblData.time_factor;
    for(var i=0; i<NECK_COM_EXC_TIME.length; i++) NECK_COM_EXC_TIME[i] *= oGlblData.time_factor;
}

