/// <reference path="C:/SOURCE14/this_dir/this_js_api/this.intellisense.js" />
/// <reference path="C:/SOURCE14/reporter_dir/library/templates/scripts/this_common1.js" />
/// <reference path="C:/SOURCE14/reporter_dir/library/templates/scripts/this_common_IIHS.js" />


// Constants used throughout the script

var REG_DT     = 0.00001; // dt for regularising curves (in seconds)
var HIC_WINDOW = 0.015;   // HIC time window (in seconds)
var TMS_PERIOD = 0.003;   // 3ms period (in seconds)

// Head assessment constants

var HIC_GOOD       = 560.0;
var HIC_ACCEPTABLE = 700.0;
var HIC_MARGINAL   = 840.0;

// Neck assessment constants

var NECK_SHR_ADJ_FAC = 17.78;   // Factor to multiply shear by to adjust bending moment

var NIJ_TEN_FCRIT = 6.806;  // Critical neck tension     force  for Nij (kN)
var NIJ_COM_FCRIT = 6.160;  // Critical neck compression force  for Nij (kN)
var NIJ_FLX_MCRIT = 310;    // Critical neck flexion     moment for Nij (Nm)
var NIJ_EXT_MCRIT = 135;    // Critical neck extension   moment for Nij (Nm)

var NECK_NIJ_GOOD       = 0.80;
var NECK_NIJ_ACCEPTABLE = 1.00;
var NECK_NIJ_MARGINAL   = 1.20;

// Neck shear exceedence limit curves

var NECK_SHR_EXC_TIME = new Array(4);
var NECK_SHR_EXC_GOOD = new Array(4);

NECK_SHR_EXC_TIME[0] = 0.0;     NECK_SHR_EXC_GOOD[0] = 3.1;
NECK_SHR_EXC_TIME[1] = 0.025;   NECK_SHR_EXC_GOOD[1] = 1.5;
NECK_SHR_EXC_TIME[2] = 0.035;   NECK_SHR_EXC_GOOD[2] = 1.5;
NECK_SHR_EXC_TIME[3] = 0.045;   NECK_SHR_EXC_GOOD[3] = 1.1;

// Neck tension exceedence limit curves

var NECK_TEN_EXC_TIME = new Array(3);
var NECK_TEN_EXC_GOOD = new Array(3);

NECK_TEN_EXC_TIME[0] = 0.0;     NECK_TEN_EXC_GOOD[0] = 3.3;
NECK_TEN_EXC_TIME[1] = 0.035;   NECK_TEN_EXC_GOOD[1] = 2.9;
NECK_TEN_EXC_TIME[2] = 0.045;   NECK_TEN_EXC_GOOD[2] = 1.1;

// Neck compression exceedence limit curves

var NECK_COM_EXC_TIME = new Array(2);
var NECK_COM_EXC_GOOD = new Array(2);

NECK_COM_EXC_TIME[0] = 0.0;     NECK_COM_EXC_GOOD[0] = 4.0;
NECK_COM_EXC_TIME[1] = 0.030;   NECK_COM_EXC_GOOD[1] = 1.1;


var NECK_TEN_GOOD       = 2.6;
var NECK_TEN_ACCEPTABLE = 3.3;
var NECK_TEN_MARGINAL   = 4.0;

var NECK_COM_GOOD       = 3.2;
var NECK_COM_ACCEPTABLE = 4.0;
var NECK_COM_MARGINAL   = 4.8;


// Chest assessment

// V7 and earlier dummies use a linear conversion: D = A*theta

var CHEST_V7_ROT_TO_COM_FACTOR = -139.0;   // Factor to convert chest transducer rotation to compression (mm)

// V8 dummies use a 3rd order polynomial conversion D = A*theta^3 - B*theta^2 - C*theta

var CHEST_V8_ROT_TO_COM_FACTOR_A =   25.13;   // First polynomial factor to convert chest transducer rotation to compression (mm)
var CHEST_V8_ROT_TO_COM_FACTOR_B =  -35.77;   // Second polynomial factor to convert chest transducer rotation to compression (mm)
var CHEST_V8_ROT_TO_COM_FACTOR_C = -136.26;   // Third polynomial factor to convert chest transducer rotation to compression (mm)

var CHEST_VC_A = 1.3;           // Viscous Criterion A constant (for m/s)
var CHEST_VC_B = 0.229;         // Viscous Criterion B constant (for m/s)
var CHEST_VC_METHOD = "ECER95"; // Viscous Criterion calculation method

var CHEST_COM_GOOD       = 50.0;  // 50mm
var CHEST_COM_ACCEPTABLE = 60.0;  // 60mm
var CHEST_COM_MARGINAL   = 75.0;  // 75mm

var CHEST_COM_RATE_GOOD       = 6.6;  // 6.6m/s
var CHEST_COM_RATE_ACCEPTABLE = 8.2;  // 8.2m/s
var CHEST_COM_RATE_MARGINAL   = 9.8;  // 9.8m/s

var CHEST_VC_GOOD       = 0.8;    // 0.8m/s
var CHEST_VC_ACCEPTABLE = 1.0;    // 1.0m/s
var CHEST_VC_MARGINAL   = 1.2;    // 1.2m/s

var CHEST_TMS_GOOD       = 60.0;  // 60g
var CHEST_TMS_ACCEPTABLE = 75.0;  // 75g
var CHEST_TMS_MARGINAL   = 90.0;  // 90g



// Upper Leg and knee assessments

var FEMUR_COM_GOOD       = 7.3;
var FEMUR_COM_ACCEPTABLE = 9.1;
var FEMUR_COM_MARGINAL   = 10.9;

var FEMUR_COM_EXC_TIME       = new Array(2);
var FEMUR_COM_EXC_GOOD       = new Array(2);
var FEMUR_COM_EXC_ACCEPTABLE = new Array(2);
var FEMUR_COM_EXC_MARGINAL   = new Array(2);

FEMUR_COM_EXC_TIME[0] = 0.0;     FEMUR_COM_EXC_GOOD[0] = 7.3;    FEMUR_COM_EXC_ACCEPTABLE[0] = 9.1;    FEMUR_COM_EXC_MARGINAL[0] = 10.9;
FEMUR_COM_EXC_TIME[1] = 0.010;   FEMUR_COM_EXC_GOOD[1] = 6.1;    FEMUR_COM_EXC_ACCEPTABLE[1] = 7.6;    FEMUR_COM_EXC_MARGINAL[1] = 9.1;

var KNEE_DISP_GOOD       = 12;
var KNEE_DISP_ACCEPTABLE = 15;
var KNEE_DISP_MARGINAL   = 18;

// Lower leg and foot assessments

var TIBIA_COM_GOOD       = 4.0;
var TIBIA_COM_ACCEPTABLE = 6.0;
var TIBIA_COM_MARGINAL   = 8.0;

var TIBIA_INDEX_GOOD       = 0.8;
var TIBIA_INDEX_ACCEPTABLE = 1.0;
var TIBIA_INDEX_MARGINAL   = 1.2;

var TIBIA_MRC = 225;    // Tibia index critical moment = 225Nm
var TIBIA_FZC = 35.9;   // Tibia index critical force  = 35.9kN

var FOOT_ACCN_GOOD       = 150;
var FOOT_ACCN_ACCEPTABLE = 200;
var FOOT_ACCN_MARGINAL   = 260;

// Structural assessment

var FOOTREST_INTRUSION_GOOD       = 15;
var FOOTREST_INTRUSION_ACCEPTABLE = 22.5;
var FOOTREST_INTRUSION_MARGINAL   = 30;

var LEFT_TOEPAN_INTRUSION_GOOD       = 15;
var LEFT_TOEPAN_INTRUSION_ACCEPTABLE = 22.5;
var LEFT_TOEPAN_INTRUSION_MARGINAL   = 30;

var CENTRE_TOEPAN_INTRUSION_GOOD       = 15;
var CENTRE_TOEPAN_INTRUSION_ACCEPTABLE = 22.5;
var CENTRE_TOEPAN_INTRUSION_MARGINAL   = 30;

var RIGHT_TOEPAN_INTRUSION_GOOD       = 15;
var RIGHT_TOEPAN_INTRUSION_ACCEPTABLE = 22.5;
var RIGHT_TOEPAN_INTRUSION_MARGINAL   = 30;

var BRAKE_PEDAL_INTRUSION_GOOD       = 15;
var BRAKE_PEDAL_INTRUSION_ACCEPTABLE = 22.5;
var BRAKE_PEDAL_INTRUSION_MARGINAL   = 30;

var LEFT_INST_PANEL_INTRUSION_GOOD       = 5;
var LEFT_INST_PANEL_INTRUSION_ACCEPTABLE = 10;
var LEFT_INST_PANEL_INTRUSION_MARGINAL   = 15;

var RIGHT_INST_PANEL_INTRUSION_GOOD       = 5;
var RIGHT_INST_PANEL_INTRUSION_ACCEPTABLE = 10;
var RIGHT_INST_PANEL_INTRUSION_MARGINAL   = 15;

var DOOR_INTRUSION_GOOD       = 5;
var DOOR_INTRUSION_ACCEPTABLE = 10;
var DOOR_INTRUSION_MARGINAL   = 15;



if(File.Exists(fname_csv) && File.IsFile(fname_csv))
{
	// Do the common setup of assigning values to the oGlblData object,
	// setting the T/HIS graph up and reading in the model.

	var oGlblData = new Object();

	var f_vars = do_common_single_analysis_setup(oGlblData);

	// Convert variable datums to model time

	convert_variable_datum_times();

	// For each body region, do the calculation, writing results to the variables file.
	// 1 here denotes the number of dummies


	do_structural_rating_calc(f_vars);

	f_vars.Close();
}




function do_upper_leg_and_knee_rating_calc(f)
{
	// Calculates the upper legs and knee values and writes them into the Reporter csv variables file <f>.

	// Two passes for the left and right hand side

	for(var lr=0; lr<2; lr++)
	{
		var beam;
		var spring;
		var side;

		if(lr == 0) side = "Left";
		else        side = "Right";

		if(lr == 0)
		{
			beam   = oGlblData.driver_left_femur_loadcell;
			spring = oGlblData.driver_left_knee_transducer;
		}
		else
		{
			beam   = oGlblData.driver_right_femur_loadcell;
			spring = oGlblData.driver_right_knee_transducer;
		}

		var output_data = {};

		output_data.com     = new OutputData(side + " Femur Axial Force",            images_dir + "/" + side + "_Femur_Compression");
		output_data.com_exc = new OutputData(side + " Femur Compression Exceedence", images_dir + "/" + side + "_Femur_Compression_Exceedence");

		// Femur compression forces

		if(beam != undefined)
		{
			// Read in beam longitudinal force

			var c_com = read_data("BEAM BASIC", 0, beam, "NZ");  // NZ is longitudinal

			// Check that the curve read in - i.e. the beam id and component are valid
			// Create blank image to let the user know.

			if(!c_com)
			{
				write_blank_images(output_data, "NO DATA FOR BEAM " + beam);
			}
			else
			{
				// Convert forces to kN from model units

				var c_com_kn = Operate.Div(c_com, oGlblData.kn_factor);

				// C600 filter (convert to seconds first)

				var c_com_c600 = convert_to_s_and_filter(c_com_kn, Operate.C600, REG_DT);

				// Exceedence curve for compression - negative values.

				var c_com_exc = Operate.Exc(c_com_c600, "negative");

				// Take absolute values

				var c_com_exc_abs = Operate.Abs(c_com_exc);

				// Remove all curves and datums

				remove_all_curves_and_datums();

				// Convert from seconds back to model time

				c_com_c600    = Operate.Mux(c_com_c600,    oGlblData.time_factor);
				c_com_exc_abs = Operate.Mux(c_com_exc_abs, oGlblData.time_factor);

				// Set labels and style

				set_labels(c_com_c600, side + " Femur Axial Force", "Time (" + oGlblData.unit_time + ")", "Force (kN)");
				set_line_style(c_com_c600, Colour.BLACK);

				// Remove all curves and datums

				remove_all_curves_and_datums();

				// Draw femur compression datums

				draw_constant_datums(-FEMUR_COM_GOOD, -FEMUR_COM_ACCEPTABLE, -FEMUR_COM_MARGINAL, null);

				// Create image/curve of femur compression exceedence curves

				c_com_c600.AddToGraph();
				write_image(output_data.com.title, output_data.com.fname, c_com_c600, -FEMUR_COM_MARGINAL);
				
				output_data.com.curveList.push(c_com_c600.id);
				write_curve("cur", output_data.com.curveList, output_data.com.fname);
				write_curve("csv", output_data.com.curveList, output_data.com.fname);

				// Remove all curves and datums

				remove_all_curves_and_datums();

				// Set labels and style

				set_labels(c_com_exc_abs, side + " Femur Compression Exceedence", "Cumulative Time (" + oGlblData.unit_time + ")", "Compression (kN)");
				set_line_style(c_com_exc_abs, Colour.BLACK);

				// Remove all curves and datums

				remove_all_curves_and_datums();

				// Draw femur compression exceedence datums

				var p = c_com_exc_abs.GetPoint(c_com_exc_abs.npoints);       
				draw_variable_datums(p[0], FEMUR_COM_EXC_TIME, FEMUR_COM_EXC_GOOD, FEMUR_COM_EXC_ACCEPTABLE, FEMUR_COM_EXC_MARGINAL, null);

				// Create image/curve of femur compression exceedence curves

				c_com_exc_abs.AddToGraph();
				write_image(output_data.com_exc.title, output_data.com_exc.fname, c_com_exc_abs, FEMUR_COM_EXC_MARGINAL[0]);
				
				output_data.com_exc.curveList.push(c_com_exc_abs.id);
				write_curve("cur", output_data.com_exc.curveList, output_data.com_exc.fname);
				write_curve("csv", output_data.com_exc.curveList, output_data.com_exc.fname);

				// Remove all curves and datums

				remove_all_curves_and_datums(); 

				// Calc values

				var com_results     = IIHS_get_constant_limit_results(c_com_c600,                       -FEMUR_COM_GOOD,     -FEMUR_COM_ACCEPTABLE,    -FEMUR_COM_MARGINAL);
				var com_exc_results = IIHS_get_variable_limit_results(c_com_exc_abs, FEMUR_COM_EXC_TIME, FEMUR_COM_EXC_GOOD,  FEMUR_COM_EXC_ACCEPTABLE, FEMUR_COM_EXC_MARGINAL);

				// Write compression variables

				write_variable(f, side.toUpperCase() + "_FEMUR_COMPRESSION_RATING", com_results[0],            "Femur compression rating", "General");
				write_variable(f, side.toUpperCase() + "_FEMUR_COMPRESSION_VALUE",  com_results[1].toFixed(3), "Femur compression value",  "General");

				write_variable(f, side.toUpperCase() + "_FEMUR_COMPRESSION_EXCEEDED_RATING", com_exc_results[0],            "Femur compression exceeded rating",         "General");
				write_variable(f, side.toUpperCase() + "_FEMUR_COMPRESSION_EXCEEDED_VALUE",  com_exc_results[1].toFixed(3), "Femur compression exceeded value",          "General");
				write_variable(f, side.toUpperCase() + "_FEMUR_COMPRESSION_DURATION_VALUE",  com_exc_results[2].toFixed(3), "Femur compression exceeded duration value", "General");
			}
		}
		else
		{
			write_blank_images(output_data, "NO BEAM ID DEFINED FOR");
		}

		// Knee slide

		var output_data = {};

		output_data.disp = new OutputData(side + " Knee Displacement", images_dir + "/" + side + "_Knee_Displacement");

		if(spring != undefined)
		{
			// Read in spring elongation force

			var c_disp = read_data("SPRING TR", 0, spring, "ET");

			// Check that the curve read in - i.e. the spring id and component are valid
			// Create blank image to let the user know.

			if(!c_disp)
			{
				write_blank_images(output_data, "NO DATA FOR SPRING " + spring);
			}
			else
			{
				// Convert to mm

				var c_disp_mm = Operate.Div(c_disp, oGlblData.mm_factor);

				// C180 filter (convert to seconds first)

				var c_disp_c180 = convert_to_s_and_filter(c_disp_mm, Operate.C180, REG_DT);

				// Remove all curves and datums

				remove_all_curves_and_datums();

				// Convert from seconds back to model time

				c_disp_c180 = Operate.Mux(c_disp_c180, oGlblData.time_factor);

				// Set labels and style

				set_labels(c_disp_c180, side + " Knee Displacement", "Time (" + oGlblData.unit_time + ")", "Displacement (mm)");
				set_line_style(c_disp_c180, Colour.BLACK);

				// Remove all curves and datums

				remove_all_curves_and_datums();

				// Draw knee displacement datums

				draw_constant_datums(KNEE_DISP_GOOD, KNEE_DISP_ACCEPTABLE, KNEE_DISP_MARGINAL, null);

				// Create image/curve of femur compression curves

				c_disp_c180.AddToGraph();
				write_image(output_data.disp.title, output_data.disp.fname, c_disp_c180, KNEE_DISP_MARGINAL);
				
				output_data.disp.curveList.push(c_disp_c180.id);
				write_curve("cur", output_data.disp.curveList, output_data.disp.fname);
				write_curve("csv", output_data.disp.curveList, output_data.disp.fname);

				// Remove all curves and datums

				remove_all_curves_and_datums(); 

				// Calc values

				var knee_disp_results = IIHS_get_constant_limit_results(c_disp_c180, KNEE_DISP_GOOD, KNEE_DISP_ACCEPTABLE, KNEE_DISP_MARGINAL);

				// Write displacement variables

				write_variable(f, side.toUpperCase() + "_KNEE_DISPLACEMENT_RATING", knee_disp_results[0],            "Knee displacement rating", "General");
				write_variable(f, side.toUpperCase() + "_KNEE_DISPLACEMENT_VALUE",  knee_disp_results[1].toFixed(3), "Knee displacement value",  "General");
			}
		}
		else
		{
			write_blank_images(output_data, "NO SPRING ID DEFINED FOR");
		}
	}
}



function do_structural_rating_calc(f)
{
	// Calculates the structural values and writes them into the Reporter csv variables file <f>.


	// 8 Structural measurements

	for(var pass=0; pass<8; pass++)
	{
		var springs   = new Array(3);
		var structure = "";

		if(pass == 0)
		{
			springs[0] = oGlblData.footrest_intrusion_spring_x;
			springs[1] = oGlblData.footrest_intrusion_spring_y;
			springs[2] = oGlblData.footrest_intrusion_spring_z;

			structure = "Footrest";
		}
		else if(pass == 1)
		{
			springs[0] = oGlblData.left_toepan_intrusion_spring_x;
			springs[1] = oGlblData.left_toepan_intrusion_spring_y;
			springs[2] = oGlblData.left_toepan_intrusion_spring_z;

			structure = "Left_Toepan";
		}
		else if(pass == 2)
		{
			springs[0] = oGlblData.centre_toepan_intrusion_spring_x;
			springs[1] = oGlblData.centre_toepan_intrusion_spring_y;
			springs[2] = oGlblData.centre_toepan_intrusion_spring_z;

			structure = "Centre_Toepan";
		}
		else if(pass == 3)
		{
			springs[0] = oGlblData.right_toepan_intrusion_spring_x;
			springs[1] = oGlblData.right_toepan_intrusion_spring_y;
			springs[2] = oGlblData.right_toepan_intrusion_spring_z;

			structure = "Right_Toepan";
		}
		else if(pass == 4)
		{
			springs[0] = oGlblData.brake_pedal_intrusion_spring_x;
			springs[1] = oGlblData.brake_pedal_intrusion_spring_y;
			springs[2] = oGlblData.brake_pedal_intrusion_spring_z;

			structure = "Brake_Pedal";
		}
		else if(pass == 5)
		{
			springs[0] = oGlblData.left_inst_panel_intrusion_spring_x;

			structure = "Left_Inst_Panel";
		}
		else if(pass == 6)
		{
			springs[0] = oGlblData.right_inst_panel_intrusion_spring_x;

			structure = "Right_Inst_Panel";
		}
		else if(pass == 7)
		{
			springs[0] = oGlblData.door_intrusion_spring_x;

			structure = "Door";
		}


		var output_data = {};

		output_data.int = new OutputData(structure + " Intrusion", images_dir + "/" + structure + "_Intrusion");

		if( (pass < 5 && ((springs[0] != undefined)   ||
						(springs[1] != undefined)   ||
						(springs[2] != undefined))) ||
				(pass >= 5 && (springs[0] != undefined)) )
		{

			var c_fa = read_data("SPRING TR", 0, springs[0], "ET");

			if(pass < 5)  // Get Y and Z displacements
			{
				var c_lat  = read_data("SPRING TR", 0, springs[1], "ET");
				var c_vert = read_data("SPRING TR", 0, springs[2], "ET");
			}

			if(!c_fa || (pass < 5 && (!c_lat || !c_vert)))
			{
                var spring_id = [];
                if(!c_fa){spring_id.push(springs[0]);} //Distinguishing the spring axis having no data
                if(pass<5 && !c_lat){spring_id.push(springs[1]);}
                if(pass<5 && !c_vert){spring_id.push(springs[2]);}
                write_blank_images(output_data, "NO DATA FOR SPRING: " + spring_id.join(", "));
			}
			else
			{
				// Remove all curves and datums

				remove_all_curves_and_datums();

				// Convert to mm

				var c_fa_cm = Operate.Div(c_fa, oGlblData.cm_factor);

				if(pass < 5)
				{
					var c_lat_cm  = Operate.Div(c_lat,  oGlblData.cm_factor);
					var c_vert_cm = Operate.Div(c_vert, oGlblData.cm_factor);
				}

				// Get final displacements

				var p = c_fa_cm.GetPoint(c_fa_cm.npoints);
				var fa_disp = p[1];

				if(pass < 5)
				{
					var p = c_lat_cm.GetPoint(c_lat_cm.npoints);
					var lat_disp = p[1];

					var p = c_vert_cm.GetPoint(c_vert_cm.npoints);
					var vert_disp = p[1];

					// Resultant displacement - use all three if X is rearward (-ve)
					//                          otherwise only use Y and Z.
					var res_disp;

					if(fa_disp < 0.0)
					{
						res_disp = Math.sqrt(fa_disp   * fa_disp  + 
								lat_disp  * lat_disp +
								vert_disp * vert_disp);
					}
					else
					{
						res_disp = Math.sqrt(lat_disp  * lat_disp +
								vert_disp * vert_disp);
					}
				}

				// Set label and style

				set_labels(c_fa_cm, "Fore/Aft Intrusion (-ve rearward)", "Time (" + oGlblData.unit_time + ")", "Intrusion (cm)");
				set_line_style(c_fa_cm, Colour.RED);
				output_data.int.curveList.push(c_fa_cm.id);

				if(pass < 5)
				{
					set_labels(c_lat_cm, "Lateral Intrusion", "Time (" + oGlblData.unit_time + ")", "Intrusion (cm)");
					set_line_style(c_lat_cm, Colour.BLUE);
					output_data.int.curveList.push(c_lat_cm.id);

					set_labels(c_vert_cm, "Vertical Intrusion (+ve upward)", "Time (" + oGlblData.unit_time + ")", "Intrusion (cm)");
					set_line_style(c_vert_cm, Colour.GREEN);
					output_data.int.curveList.push(c_vert_cm.id);
				}

				// Create image/curve

				write_image(output_data.int.title, output_data.int.fname); 

				write_curve("cur", output_data.int.curveList, output_data.int.fname);
                                
				write_curve("csv", output_data.int.curveList, output_data.int.fname);

				// Calculate ratings


				var rating = "GOOD";

				if(pass == 0) rating = IIHS_get_rating(res_disp, FOOTREST_INTRUSION_GOOD,      FOOTREST_INTRUSION_ACCEPTABLE,      FOOTREST_INTRUSION_MARGINAL);
				if(pass == 1) rating = IIHS_get_rating(res_disp, LEFT_TOEPAN_INTRUSION_GOOD,   LEFT_TOEPAN_INTRUSION_ACCEPTABLE,   LEFT_TOEPAN_INTRUSION_MARGINAL);
				if(pass == 2) rating = IIHS_get_rating(res_disp, CENTRE_TOEPAN_INTRUSION_GOOD, CENTRE_TOEPAN_INTRUSION_ACCEPTABLE, CENTRE_TOEPAN_INTRUSION_MARGINAL);
				if(pass == 3) rating = IIHS_get_rating(res_disp, RIGHT_TOEPAN_INTRUSION_GOOD,  RIGHT_TOEPAN_INTRUSION_ACCEPTABLE,  RIGHT_TOEPAN_INTRUSION_MARGINAL);
				if(pass == 4) rating = IIHS_get_rating(res_disp, BRAKE_PEDAL_INTRUSION_GOOD,   BRAKE_PEDAL_INTRUSION_ACCEPTABLE,   BRAKE_PEDAL_INTRUSION_MARGINAL);

				if(pass == 5) rating = IIHS_get_rating(-fa_disp, LEFT_INST_PANEL_INTRUSION_GOOD,  LEFT_INST_PANEL_INTRUSION_ACCEPTABLE,  LEFT_INST_PANEL_INTRUSION_MARGINAL);
				if(pass == 6) rating = IIHS_get_rating(-fa_disp, RIGHT_INST_PANEL_INTRUSION_GOOD, RIGHT_INST_PANEL_INTRUSION_ACCEPTABLE, RIGHT_INST_PANEL_INTRUSION_MARGINAL);
				if(pass == 7) rating = IIHS_get_rating(-fa_disp, DOOR_INTRUSION_GOOD,             DOOR_INTRUSION_ACCEPTABLE,             DOOR_INTRUSION_MARGINAL);


				// Write variables

				write_variable(f, "STRUCTURAL_" + structure.toUpperCase() + "_X", fa_disp.toFixed(2), structure + " X Intrusion", "General");

				if(pass < 5)
				{
					write_variable(f, "STRUCTURAL_" + structure.toUpperCase() + "_Y",         lat_disp.toFixed(2),  structure + " Y Intrusion",         "General");
					write_variable(f, "STRUCTURAL_" + structure.toUpperCase() + "_Z",         vert_disp.toFixed(2), structure + " Z Intrusion",         "General");
					write_variable(f, "STRUCTURAL_" + structure.toUpperCase() + "_RESULTANT", res_disp.toFixed(2),  structure + " resultant Intrusion", "General");
				}

				write_variable(f, "STRUCTURAL_" + structure.toUpperCase() + "_RATING", rating, structure + " Intrusion Rating", "General");
			}
		}
		else
		{
			write_blank_images(output_data, "NO SPRING ID DEFINED FOR");
		}

	}
}



function convert_variable_datum_times()
{
	// Convert the variable datum times from seconds to the model units

	for(var i=0; i<NECK_SHR_EXC_TIME.length;  i++) NECK_SHR_EXC_TIME[i]  *= oGlblData.time_factor;
	for(var i=0; i<NECK_TEN_EXC_TIME.length;  i++) NECK_TEN_EXC_TIME[i]  *= oGlblData.time_factor;
	for(var i=0; i<NECK_COM_EXC_TIME.length;  i++) NECK_COM_EXC_TIME[i]  *= oGlblData.time_factor;
	for(var i=0; i<FEMUR_COM_EXC_TIME.length; i++) FEMUR_COM_EXC_TIME[i] *= oGlblData.time_factor;
}

