/// <reference path="C:/SOURCE14/this_dir/this_js_api/this.intellisense.js" />
/// <reference path="C:/SOURCE14/reporter_dir/library/templates/scripts/this_common1.js" />
/// <reference path="C:/SOURCE14/reporter_dir/library/templates/scripts/this_common_IIHS.js" />


// Common functions for IIHS front impact templates



function do_head_rating_calc(f, num_dummies)
{
    // Calculates the head values and writes them into the Reporter csv variables file <f>.
    
    // dummy no. 1 is driver
    // dummy no. 2 is passenger
    
    var dummy, dummy_;
    
    for (var pass=0; pass<num_dummies; pass++)
    {
        if (pass == 0)
        {
            var node = oGlblData.driver_head_node;
            var node_y = oGlblData.driver_head_node_y;
            var node_z = oGlblData.driver_head_node_z;
            if (num_dummies>1)
            {
                dummy = "Driver ";
                dummy_ = "Driver_";
            }
            else
            {
                dummy = "";
                dummy_ = "";
            }
        }
        
        if (pass == 1) 
        {
            var node = oGlblData.passenger_head_node;
            var node_y = oGlblData.passenger_head_node_y;
            var node_z = oGlblData.passenger_head_node_z;
            dummy = "Passenger ";
            dummy_ = "Passenger_";
        }

        var output_data = {};
        
        
        output_data.acc = new OutputData(dummy + "Head Acceleration", images_dir + "/" + dummy_ + "Head_Acceleration");

        if(node != undefined)
        { 
            // Read in acceleration X, Y and Z
            var one_node = true;

            // If nodes for three accelerometer positions defined, use all three, otherwise, read all components from same node
            if(node_y != undefined && node_z != undefined && node_y != 0 && node_z != 0)
            {
                var c_acc = read_xyz_accelerations(1, node, node_y, node_z);
                one_node = false;
            }
            else
            {
                var c_acc = read_xyz_accelerations(1, node);
            }

            // Check that the curves read in - i.e. the node id and component are valid
            // Create a blank image to let the user know.
            
            // Generate a relevant message for the blank image depending on which acceleration data are missing
            if(!c_acc.ax || !c_acc.ay || !c_acc.az)
            {
                var blank_image_msg;
                // If only one node supplied, just report that node ID
                if(one_node) blank_image_msg = "NO ACCEL DATA (NODE " + node + ") ";
                // else report each of the node IDs that were invalid
                else
                {
                    var acc_comp_lst = [];
                    var acc_id_lst = [];
                    if(!c_acc.ax)
                    {
                        acc_comp_lst.push(" X");
                        acc_id_lst.push(" "+node);
                    }
                    if(!c_acc.ay)
                    {
                        acc_comp_lst.push(" Y");
                        acc_id_lst.push(" "+node_y);
                    }
                    if(!c_acc.az)
                    {
                        acc_comp_lst.push(" Z");
                        acc_id_lst.push(" "+node_z);
                    }
                    blank_image_msg = "NO" + acc_comp_lst + " ACCEL DATA (NODE(s) " + acc_id_lst + ") ";
                }
                create_blank_image(blank_image_msg + output_data.acc.title, output_data.acc.fname);
            }
            else
            {
                // Convert to g from model units

                var c_g = convert_xyz_acc_to_g(c_acc);

                // C1000 filter (convert to seconds first)

                var c_gx_c1000 = convert_to_s_and_filter(c_g.gx, Operate.C1000, REG_DT);
                var c_gy_c1000 = convert_to_s_and_filter(c_g.gy, Operate.C1000, REG_DT);
                var c_gz_c1000 = convert_to_s_and_filter(c_g.gz, Operate.C1000, REG_DT);

                // Vector combine

                var c_vec_g = Operate.Vec(c_gx_c1000, c_gy_c1000, c_gz_c1000);

                // Remove all curves and datums

                remove_all_curves_and_datums();

                // Convert from seconds back to model time

                c_vec_g = Operate.Mux(c_vec_g, oGlblData.time_factor);

                // Get peak acceleration (in g) and time (in seconds)

                DialogueInput("/AU");
                var peak_accn      = c_vec_g.ymax;
                var peak_accn_time = c_vec_g.x_at_ymax / oGlblData.time_factor;

                // Get HIC15 value and start and end time of window (in seconds)

                var hic            = Operate.Hic(c_vec_g, HIC_WINDOW, 1.0);
                var hic_start_time = c_vec_g.hic_tmin / oGlblData.time_factor;
                var hic_end_time   = c_vec_g.hic_tmax / oGlblData.time_factor;

                // Set label and style

                set_labels(c_vec_g, "Head Acceleration Magnitude", "Time (" + oGlblData.unit_time + ")", "Acceleration (g)");
                set_line_style(c_vec_g, Colour.BLACK);

                // Create image/curve

                write_image(output_data.acc.title, output_data.acc.fname);
                
                output_data.acc.curveList.push(c_vec_g.id);
                write_curve("cur", output_data.acc.curveList, output_data.acc.fname);
                write_curve("csv", output_data.acc.curveList, output_data.acc.fname);

                // Calculate HIC rating

                var hic_rating = IIHS_get_rating(hic, HIC_GOOD, HIC_ACCEPTABLE, HIC_MARGINAL);

                // Write variables to csv file for Reporter

                // Peak acceleration and time
                write_variable(f, dummy_.toUpperCase() + "HEAD_PEAK_ACCN_VALUE",      peak_accn.toFixed(2),  dummy + "Head peak acceleration value",      "General");
                write_variable(f, dummy_.toUpperCase() + "HEAD_PEAK_ACCN_TIME_VALUE", peak_accn_time.toFixed(3), dummy + "Head peak acceleration time value", "General");

                // HIC value, rating, start time and end time
                write_variable(f, dummy_.toUpperCase() + "HEAD_HIC15_VALUE",            hic.toFixed(2),            dummy + "Head HIC15 value",            "General");
                write_variable(f, dummy_.toUpperCase() + "HEAD_HIC15_RATING",           hic_rating,                dummy + "Head HIC15 rating",           "General");
                write_variable(f, dummy_.toUpperCase() + "HEAD_HIC15_START_TIME_VALUE", hic_start_time.toFixed(3), dummy + "Head HIC15 start time value", "General");
                write_variable(f, dummy_.toUpperCase() + "HEAD_HIC15_END_TIME_VALUE",   hic_end_time.toFixed(3),   dummy + "Head HIC15 end time value",   "General");
            }
        }
        else
        {

            // No node defined - variables should be set to blank by first script in template.
            // Create a blank image to let the user know.

            write_blank_images(output_data, "NO NODE ID DEFINED FOR");
        }
    }
}




function do_neck_rating_calc(f, num_dummies, nij_flag)
{
    // Calculates the neck values and writes them into the Reporter csv variables file <f>.
    
    // dummy no. 1 is driver
    // dummy no. 2 is passenger
    var dummy = [];
    dummy.push(new DummyData("Driver", oGlblData.driver_neck_loadcell, "beam", oGlblData.driver_dummy, "version"));
    if (num_dummies == 2)
    {
        dummy.push(new DummyData("Passenger", oGlblData.passenger_neck_loadcell, "beam", oGlblData.passenger_dummy, "version"));
    }

        
    for(var i = 0; i<dummy.length; i++)
    {
        var output_data = {};

        output_data.shr_exc  = new OutputData(dummy[i].chart_label + " Neck Shear Exceedence",       images_dir + "/" + dummy[i].variable_label + "_Neck_Shear_Exceedence");
        output_data.ten_exc  = new OutputData(dummy[i].chart_label + " Neck Tension Exceedence",     images_dir + "/" + dummy[i].variable_label + "_Neck_Tension_Exceedence");
        output_data.com_exc  = new OutputData(dummy[i].chart_label + " Neck Compression Exceedence", images_dir + "/" + dummy[i].variable_label + "_Neck_Compression_Exceedence");
        output_data.mom      = new OutputData(dummy[i].chart_label + " Neck Bending Moment",         images_dir + "/" + dummy[i].variable_label + "_Neck_Moment");
        output_data.axl      = new OutputData(dummy[i].chart_label + " Neck Axial Force",            images_dir + "/" + dummy[i].variable_label + "_Neck_Axial_Force");
        output_data.shr      = new OutputData(dummy[i].chart_label + " Neck Shear Force",            images_dir + "/" + dummy[i].variable_label + "_Neck_Shear_Force");
        
        if (nij_flag)
            output_data.nij  = new OutputData(dummy[i].chart_label + " Neck Nij",                    images_dir + "/" + dummy[i].variable_label + "_Neck_NIJ");

// Originally, it was not necessary to specify the dummy version for IIHS Side
// MDB Impact (it always extracted BEAM data) so if version is undefined, mark
// it as unspecified and treat as BEAM data
        if (dummy[i].version == undefined)
            dummy[i].version = "<Unspecified>";

        if( dummy[i].beam_id != undefined )
        {
            // Read in beam shear, tension and  MY
            // For Humanetics dummies this is measured from a beam;
            // for LSTC dummies this is measured from a cross section
            switch (dummy[i].version)
            {
                case "<Unspecified>":
                case "V6_HUMANETICS_HIII_50TH_MALE":
                case "V7_HUMANETICS_HIII_50TH_MALE":
                case "V8_HUMANETICS_HIII_50TH_MALE":
                case "Humanetics HIII 50M v1.5":
                case "Humanetics HIII 50M v1.5.1":
                case "LSTC SID IIS-D v0.150.beta":                
                    var c_shr = read_data("BEAM BASIC", 0, dummy[i].beam_id, "NX");  // NX is shear
                    var c_axl = read_data("BEAM BASIC", 0, dummy[i].beam_id, "NZ");
                    var c_mom = read_data("BEAM BASIC", 0, dummy[i].beam_id, "MY");
                    var entity_str = "BEAM"; // Used for error message below
                    break;

                case "LSTC HIII 50M Detailed v190217_beta":
                case "LSTC HIII 50M v130528_beta":
                case "LSTC HIII 50M Fast V2.0":
                case "Humanetics SID IIs SBLD v.4.3.1":
                case "Humanetics SID IIs SBLD v.4.3.2":
                case "Humanetics SID IIs SBLD v.4.3.5":
//Shifted Humanetics-SID-Dummy to X-sec because the neck load cells are x-sections
                    var c_shr = read_data("SECTION", 0, dummy[i].beam_id, "FX");  // FX is shear
                    var c_axl = read_data("SECTION", 0, dummy[i].beam_id, "FZ");
                    var c_mom = read_data("SECTION", 0, dummy[i].beam_id, "MY");
                    var entity_str = "SECTION"; // Used for error message below
                    break;

                default:
                    ErrorMessage("Unsupported dummy \""+dummy[i].version+"\" in do_neck_do_neck_rating_calc.");
                    write_blank_images(output_data, "UNSUPPORTED DUMMY \""+dummy[i].version+"\".");
                    return;
                    break;
            }

            // Check that the curves read in - i.e. the beam id and component are valid
            // Create blank images to let the user know.

            if(!c_shr || !c_axl || !c_mom)
            {
                write_blank_images(output_data, "NO DATA FOR " + entity_str + " " + dummy[i].beam_id);
            }
            else
            {
                // Convert forces to kN and
                //         moment to Nm from model units
                
                var c_shr_kn = Operate.Div(c_shr, oGlblData.kn_factor);                
                var c_axl_kn = Operate.Div(c_axl, oGlblData.kn_factor);
                var c_mom_nm = Operate.Div(c_mom, oGlblData.nm_factor);
                
                // C1000/C600 filter (convert to seconds first)                
                var c_shr_c1000 = convert_to_s_and_filter(c_shr_kn, Operate.C1000, REG_DT);                
                var c_axl_c1000 = convert_to_s_and_filter(c_axl_kn, Operate.C1000, REG_DT);
                var c_mom_c600  = convert_to_s_and_filter(c_mom_nm, Operate.C600,  REG_DT);

                // Exceedence curves for shear, tension and compression. For shear and compression take absolute values.
                var c_shr_abs = Operate.Abs(c_shr_c1000);
                var c_shr_exc = Operate.Exc(c_shr_abs, "positive");
                
                // tension, compression
                var c_ten_exc = Operate.Exc(c_axl_c1000, "positive");
                var c_com_exc = Operate.Exc(c_axl_c1000, "negative");                
                //shear
                var c_shr_exc_abs = Operate.Abs(c_shr_exc);
                // Compression
                var c_com_exc_abs = Operate.Abs(c_com_exc);

                // Transpose the bending moment to the occupital condoyle moment
                var c_shr_adj = Operate.Mul(c_shr_c1000, NECK_SHR_ADJ_FAC);
                var c_mom_adj = Operate.Sub(c_mom_c600, c_shr_adj);
                
                // Regularise curves first so they have the same number of points

                var max_points = Math.max(c_axl_c1000.npoints,
                        c_mom_adj.npoints);
                        
                max_points = Math.max(c_shr_c1000.npoints,
                    max_points);
                
                var p = c_shr_c1000.GetPoint(c_shr_c1000.npoints);       
                var t = p[0];

                var dt = t / max_points;

                c_shr_c1000 = Operate.Reg(c_shr_c1000, dt);

                var p = c_axl_c1000.GetPoint(c_axl_c1000.npoints);       
                var t = p[0];

                var dt = t / max_points;

                c_axl_c1000 = Operate.Reg(c_axl_c1000, dt);

                var p = c_mom_adj.GetPoint(c_mom_adj.npoints);       
                var t = p[0];

                var dt = t / max_points;

                c_mom_adj = Operate.Reg(c_mom_adj, dt);

                // Calculate Nij

                if (nij_flag)
                {
                    var nij_curves = Operate.Nij(c_shr_c1000,
                            c_axl_c1000,
                            c_mom_adj,
                            NIJ_TEN_FCRIT,
                            NIJ_COM_FCRIT,
                            NIJ_FLX_MCRIT,
                            NIJ_EXT_MCRIT,
                            0.0);

                    for(var k=0; k<4; k++)
                        nij_curves[k] = Operate.Mux(nij_curves[k], oGlblData.time_factor);
                
                }
                
                // Remove all curves and datums

                remove_all_curves_and_datums();

                // Convert from seconds back to model time
                c_shr_exc_abs = Operate.Mux(c_shr_exc_abs, oGlblData.time_factor);
                c_shr_c1000   = Operate.Mux(c_shr_c1000,   oGlblData.time_factor);
                c_ten_exc     = Operate.Mux(c_ten_exc,     oGlblData.time_factor);
                c_com_exc_abs = Operate.Mux(c_com_exc_abs, oGlblData.time_factor);
                c_mom_adj     = Operate.Mux(c_mom_adj,     oGlblData.time_factor);
                c_axl_c1000   = Operate.Mux(c_axl_c1000,   oGlblData.time_factor);

                // Set labels and style
                set_labels(c_shr_exc_abs, " Neck Shear Exceedence",       "Cumulative Time (" + oGlblData.unit_time + ")", "Force (kN)");
                set_labels(c_shr_c1000,   " Neck Shear Force",            "Time (" + oGlblData.unit_time + ")",            "Force (kN)");
                set_labels(c_ten_exc,     " Neck Tension Exceedence",     "Cumulative Time (" + oGlblData.unit_time + ")", "Force (kN)");
                set_labels(c_com_exc_abs, " Neck Compression Exceedence", "Cumulative Time (" + oGlblData.unit_time + ")", "Force (kN)");
                set_labels(c_mom_adj,     " Neck Bending Moment",         "Time (" + oGlblData.unit_time + ")",            "Bending Moment (Nm)");
                set_labels(c_axl_c1000,   " Neck Axial Force",            "Time (" + oGlblData.unit_time + ")",            "Force (kN)");
                
                if (nij_flag)
                {
                    set_labels(nij_curves[0], "Nij - Tension-Extension",     "Time (" + oGlblData.unit_time + ")",            "Nij");
                    set_labels(nij_curves[1], "Nij - Tension-Flexion",       "Time (" + oGlblData.unit_time + ")",            "Nij");
                    set_labels(nij_curves[2], "Nij - Compression-Extension", "Time (" + oGlblData.unit_time + ")",            "Nij");
                    set_labels(nij_curves[3], "Nij - Compression-Flexion",   "Time (" + oGlblData.unit_time + ")",            "Nij");
                }
                
                set_line_style(c_shr_exc_abs, Colour.BLACK);
                set_line_style(c_shr_c1000,   Colour.BLACK);
                set_line_style(c_ten_exc,     Colour.BLACK);
                set_line_style(c_com_exc_abs, Colour.BLACK);
                set_line_style(c_mom_adj,     Colour.BLACK);
                set_line_style(c_axl_c1000,   Colour.BLACK);
                
                if (nij_flag)
                {
                    set_line_style(nij_curves[0], Colour.MAGENTA);
                    set_line_style(nij_curves[1], Colour.MAGENTA);
                    set_line_style(nij_curves[2], Colour.BLACK);
                    set_line_style(nij_curves[3], Colour.BLACK);
                    
                    nij_curves[1].style  = LineStyle.DASH;
                    nij_curves[3].style  = LineStyle.DASH;

                    // Remove all curves and datums and datums

                    remove_all_curves_and_datums();

                    // Draw Nij datums

                    draw_constant_datums(NECK_NIJ_GOOD, NECK_NIJ_ACCEPTABLE, NECK_NIJ_MARGINAL, null);

                    // Create image/curve of neck Nij curves

                    for(var j=0; j<4; j++)
                    {
                        nij_curves[j].AddToGraph();
                        output_data.nij.curveList.push(nij_curves[j].id);
                    }

                    write_image(output_data.nij.title, output_data.nij.fname, nij_curves, NECK_NIJ_MARGINAL);

                    write_curve("cur", output_data.nij.curveList, output_data.nij.fname);
                    write_curve("csv", output_data.nij.curveList, output_data.nij.fname);
                }                    

                // Remove all curves and datums
                remove_all_curves_and_datums();
               
                // Draw neck shear datums
                var p = c_shr_exc_abs.GetPoint(c_shr_exc_abs.npoints);       
                draw_variable_datums(p[0], NECK_SHR_EXC_TIME, NECK_SHR_EXC_GOOD, null, null, null);

                // Create image/curve of neck shear curves

                c_shr_exc_abs.AddToGraph();
                write_image(output_data.shr_exc.title, output_data.shr_exc.fname, c_shr_exc_abs, NECK_SHR_EXC_GOOD[0]);
            
                output_data.shr_exc.curveList.push(c_shr_exc_abs.id);
                write_curve("cur", output_data.shr_exc.curveList, output_data.shr_exc.fname);
                write_curve("csv", output_data.shr_exc.curveList, output_data.shr_exc.fname);    

                // Remove all curves and datums

                remove_all_curves_and_datums();

                // Draw neck tension datums

                var p = c_ten_exc.GetPoint(c_ten_exc.npoints);    
                draw_variable_datums(p[0], NECK_TEN_EXC_TIME, NECK_TEN_EXC_GOOD, null, null, null);

                // Create image/curve of neck tension curves

                c_ten_exc.AddToGraph();
                write_image(output_data.ten_exc.title, output_data.ten_exc.fname, c_ten_exc, NECK_TEN_EXC_GOOD[0]);
                
                output_data.ten_exc.curveList.push(c_ten_exc.id);
                write_curve("cur", output_data.ten_exc.curveList, output_data.ten_exc.fname);
                write_curve("csv", output_data.ten_exc.curveList, output_data.ten_exc.fname);

                // Remove all curves and datums

                remove_all_curves_and_datums();

                // Draw neck compression datums

                var p = c_com_exc_abs.GetPoint(c_com_exc_abs.npoints);
                draw_variable_datums(p[0], NECK_COM_EXC_TIME, NECK_COM_EXC_GOOD, null, null, null);

                // Create image/curve of neck tension curves

                c_com_exc_abs.AddToGraph();
                write_image(output_data.com_exc.title, output_data.com_exc.fname, c_com_exc_abs, NECK_COM_EXC_GOOD[0]);
                
                output_data.com_exc.curveList.push(c_com_exc_abs.id);
                write_curve("cur", output_data.com_exc.curveList, output_data.com_exc.fname);
                write_curve("csv", output_data.com_exc.curveList, output_data.com_exc.fname);

                // Remove all curves and datums

                remove_all_curves_and_datums();
                
                 // Draw moment limit datums

                //draw_constant_datums(NECK_MOM_GOOD, NECK_MOM_ADEQUATE, NECK_MOM_MARGINAL, null);

                // Create image/curve of moment curves

                c_mom_adj.AddToGraph();
                write_image(output_data.mom.title, output_data.mom.fname);
                
                output_data.mom.curveList.push(c_mom_adj.id);
                write_curve("cur", output_data.mom.curveList, output_data.mom.fname);
                write_curve("csv", output_data.mom.curveList, output_data.mom.fname);

                // Remove all curves and datums

                remove_all_curves_and_datums();

                // Draw neck axial force datums

                var p = c_axl_c1000.GetPoint(c_axl_c1000.npoints);       
                draw_neck_axl_datums(p[0]);

                // Create image/curve of axial forces curves

                c_axl_c1000.AddToGraph();
                write_image(output_data.axl.title, output_data.axl.fname, c_axl_c1000, NECK_TEN_MARGINAL, -NECK_COM_MARGINAL);
                
                output_data.axl.curveList.push(c_axl_c1000.id);
                write_curve("cur", output_data.axl.curveList, output_data.axl.fname);
                write_curve("csv", output_data.axl.curveList, output_data.axl.fname);

                // Remove all curves and datums

                remove_all_curves_and_datums();

                // Create image/curve of shear force curve

                c_shr_c1000.AddToGraph();
                write_image(output_data.shr.title, output_data.shr.fname);
                
                output_data.shr.curveList.push(c_shr_c1000.id);
                write_curve("cur", output_data.shr.curveList, output_data.shr.fname);
                write_curve("csv", output_data.shr.curveList, output_data.shr.fname);

                // Remove all curves and datums

                remove_all_curves_and_datums();

                // Calc values

                var ten_results     = IIHS_get_constant_limit_results(c_axl_c1000,  NECK_TEN_GOOD,  NECK_TEN_ACCEPTABLE,  NECK_TEN_MARGINAL);
                var com_results     = IIHS_get_constant_limit_results(c_axl_c1000, -NECK_COM_GOOD, -NECK_COM_ACCEPTABLE, -NECK_COM_MARGINAL);
                
                if (nij_flag)
                    var nij_results     = get_neck_nij_results(nij_curves);
                
                var shr_exc_results = IIHS_get_variable_limit_results(c_shr_exc_abs, NECK_SHR_EXC_TIME, NECK_SHR_EXC_GOOD, null, null);
                
                var ten_exc_results = IIHS_get_variable_limit_results(c_ten_exc    , NECK_TEN_EXC_TIME, NECK_TEN_EXC_GOOD, null, null);
                var com_exc_results = IIHS_get_variable_limit_results(c_com_exc_abs, NECK_COM_EXC_TIME, NECK_COM_EXC_GOOD, null, null);

                // Write variables

                // TENSION

                write_variable(f, dummy[i].variable_label.toUpperCase() + "_NECK_TENSION_RATING", ten_results[0],            dummy[i].variable_label+ "Neck tension rating", "String");
                write_variable(f, dummy[i].variable_label.toUpperCase() + "_NECK_TENSION_VALUE",  ten_results[1].toFixed(3), dummy[i].variable_label + "Neck tension value",  "String");

                // COMPRESSION

                write_variable(f, dummy[i].variable_label.toUpperCase() + "_NECK_COMPRESSION_RATING", com_results[0],            dummy[i].variable_label + "Neck compression rating", "General");
                write_variable(f, dummy[i].variable_label.toUpperCase() + "_NECK_COMPRESSION_VALUE",  com_results[1].toFixed(3), dummy[i].variable_label + "Neck compression value",  "General");

                // NIJ
                if (nij_flag)
                {
                    write_variable(f, dummy[i].variable_label.toUpperCase() + "_NECK_INJURY_RATING", nij_results[0],            dummy[i].variable_label + "Neck injury rating", "General");
                    write_variable(f, dummy[i].variable_label.toUpperCase() + "_NECK_INJURY_VALUE",  nij_results[1].toFixed(3), dummy[i].variable_label + "Neck injury value",  "General");
                }
                
                // SHEAR EXCCEDENCE
                write_variable(f, dummy[i].variable_label.toUpperCase() + "_NECK_SHEAR_EXCEEDED_RATING", shr_exc_results[0],            dummy[i].variable_label + "Neck shear exceeded rating", "General");
                write_variable(f, dummy[i].variable_label.toUpperCase() + "_NECK_SHEAR_EXCEEDED_VALUE",  shr_exc_results[1].toFixed(3), dummy[i].variable_label + "Neck shear exceeded value",  "General");
                write_variable(f, dummy[i].variable_label.toUpperCase() + "_NECK_SHEAR_DURATION_VALUE",  shr_exc_results[2].toFixed(2), dummy[i].variable_label + "Neck shear duration value",  "General");

                // TENSION EXCEEDENCE
                write_variable(f, dummy[i].variable_label.toUpperCase() + "_NECK_TENSION_EXCEEDED_RATING", ten_exc_results[0],            dummy[i].variable_label + "Neck tension exceeded rating", "General");
                write_variable(f, dummy[i].variable_label.toUpperCase() + "_NECK_TENSION_EXCEEDED_VALUE",  ten_exc_results[1].toFixed(3), dummy[i].variable_label + "Neck tension exceeded value",  "General");
                write_variable(f, dummy[i].variable_label.toUpperCase() + "_NECK_TENSION_DURATION_VALUE",  ten_exc_results[2].toFixed(2), dummy[i].variable_label + "Neck tension duration value",  "General");

                // COMPRESSION EXCEEDENCE
                write_variable(f, dummy[i].variable_label.toUpperCase() + "_NECK_COMPRESSION_EXCEEDED_RATING", com_exc_results[0],            dummy[i].variable_label + "Neck compression exceeded rating", "General");
                write_variable(f, dummy[i].variable_label.toUpperCase() + "_NECK_COMPRESSION_EXCEEDED_VALUE",  com_exc_results[1].toFixed(3), dummy[i].variable_label + "Neck compression exceeded value",  "General");
                write_variable(f, dummy[i].variable_label.toUpperCase() + "_NECK_COMPRESSION_DURATION_VALUE",  com_exc_results[2].toFixed(2), dummy[i].variable_label + "Neck compression duration value",  "General");
            }
        }
        else
        {

            // No beam defined - variables should be set to blank by first script in template.
            // Create blank images to let the user know.

            write_blank_images(output_data, "NO BEAM ID DEFINED FOR");
        }
    }
}


function do_chest_rating_calc(f)
{
    // Calculates the chest values and writes them into the Reporter csv variables file <f>.

    var spring;

    // Compression

    spring = oGlblData.driver_chest_transducer;

    var output_data = {};

    output_data.com      = new OutputData("Chest compression",       images_dir + "/Chest_Compression");
    output_data.com_rate = new OutputData("Chest compression rate",  images_dir + "/Chest_Compression_Rate");
    output_data.vc       = new OutputData("Chest Viscous Criterion", images_dir + "/Chest_Viscous_Criterion");


    if( spring != undefined )
    {
        // Read in spring rotation

        var c_rot = read_data("SPRING ROT", 0, spring, "RT");
        
        if(!c_rot)
        {
            var blank_image_msg;

            blank_image_msg = "NO " + dummy.toUpperCase() + "CHEST ROTATION DATA FOR TRANSDUCER " + spring;
            create_blank_image(blank_image_msg, output_data.rib.fname);
        }
        else
        {
            // Chest compression is calculated by converting the rotation in radians to a deflection using
            // a transfer function.
            //
            // For V7 dummies and earlier, the conversion is linear
            // For V8 dummies and later, the conversion uses a 3rd order polynomial
            //
            // The dummy manuals say that a negative rotation indicates compression, however testing has shown
            // that in some versions it is actually the other way around. If the maximum absolute value is +ve
            // then multiply the rotation by -1 so the conversion to compression works correctly.

            var rmax = c_rot.ymax;
            var rmin = c_rot.ymin;

            if(Math.abs(rmax) > Math.abs(rmin)) c_rot = Operate.Mul(c_rot, -1);

            var version = oGlblData.driver_dummy;

            var c_disp_mm;
            switch (version)
            {
                case "V7_HUMANETICS_HIII_50TH_MALE": 
                  c_disp_mm = Operate.Mul(c_rot, CHEST_V7_ROT_TO_COM_FACTOR);
                break;
            
                case "LSTC HIII 50M Detailed v190217_beta": 
                case "LSTC HIII 50M Fast V2.0": 
                case "LSTC HIII 50M v130528_beta":
                c_disp_mm = Operate.Mul(c_rot,CHEST_LSTC_HIII_50TH_MALE_DETAILED_190217_BETA_ROT_TO_COM_FACTOR);
                break;
            
                case "V8_HUMANETICS_HIII_50TH_MALE":
                case "Humanetics HIII 50M v1.5":
                case "Humanetics HIII 50M v1.5.1":
                    curve_id = Curve.FirstFreeID();

                    c_disp_mm = new Curve(curve_id);
                    var n = c_rot.npoints;

                    for(var i=1; i<=n; i++)
                    {
                        var p_rot = c_rot.GetPoint(i);

                        var time = p_rot[0];
                        var com  = CHEST_V8_ROT_TO_COM_FACTOR_A * p_rot[1] * p_rot[1] * p_rot[1] +
                        CHEST_V8_ROT_TO_COM_FACTOR_B * p_rot[1] * p_rot[1] + 
                        CHEST_V8_ROT_TO_COM_FACTOR_C * p_rot[1];

                        c_disp_mm.AddPoint(time, com);
                    }
                 break;
                 
                 default:
                     ErrorMessage("Unsupported dummy \""+version+"\" in do_chest_rating_calc.");
                     write_blank_images(output_data, "UNSUPPORTED DUMMY \""+version+"\"");
                     return;
                     break;
            }

            // C180 filter (convert to seconds first)

            var c_disp_c180_mm = convert_to_s_and_filter(c_disp_mm, Operate.C180, REG_DT);  // in mm

            var c_disp_c180_m  = Operate.Div(c_disp_c180_mm, 1000);  // in m  (for viscous criterion calc and deflection rate)

            // Differentiate deflection to get rate

            var c_def_rate = Operate.Dif(c_disp_c180_m);

            // Calculate viscous criterion

            var c_vc = Operate.Vc(c_disp_c180_m, CHEST_VC_A, CHEST_VC_B, CHEST_VC_METHOD);

            // Remove all curves and datums

            remove_all_curves_and_datums();

            // Convert from seconds back to model time

            c_disp_c180_mm = Operate.Mux(c_disp_c180_mm, oGlblData.time_factor);
            c_vc           = Operate.Mux(c_vc,           oGlblData.time_factor);

            // Set labels and style

            set_labels(c_disp_c180_mm, "Chest Compression",      "Time (" + oGlblData.unit_time + ")", "Compression (mm)");
            set_labels(c_def_rate,     "Chest Compression Rate", "Time (" + oGlblData.unit_time + ")", "Compression Rate (m/s)");
            set_labels(c_vc,           "Viscous Criterion",      "Time (" + oGlblData.unit_time + ")", "Velocity (m/s)");

            set_line_style(c_disp_c180_mm, Colour.BLACK);
            set_line_style(c_def_rate,     Colour.BLACK);
            set_line_style(c_vc,           Colour.BLACK);

            // Remove all curves and datums

            remove_all_curves_and_datums();

            // Draw chest compression datums

            draw_constant_datums(CHEST_COM_GOOD, CHEST_COM_ACCEPTABLE, CHEST_COM_MARGINAL, null);

            // Create image/curve of chest compression curves

            c_disp_c180_mm.AddToGraph();
            write_image(output_data.com.title, output_data.com.fname, c_disp_c180_mm, CHEST_COM_MARGINAL);
            
            output_data.com.curveList.push(c_disp_c180_mm.id);
            write_curve("cur", output_data.com.curveList, output_data.com.fname);
            write_curve("csv", output_data.com.curveList, output_data.com.fname);

            // Remove all curves and datums

            remove_all_curves_and_datums(); 

            // Draw chest compression rate datums

            draw_constant_datums(CHEST_COM_RATE_GOOD, CHEST_COM_RATE_ACCEPTABLE, CHEST_COM_RATE_MARGINAL, null);

            // Create image/curve of chest compression curves

            c_def_rate.AddToGraph();
            write_image(output_data.com_rate.title, output_data.com_rate.fname, c_def_rate, CHEST_COM_RATE_MARGINAL);
            
            output_data.com_rate.curveList.push(c_def_rate.id);
            write_curve("cur", output_data.com_rate.curveList, output_data.com_rate.fname);
            write_curve("csv", output_data.com_rate.curveList, output_data.com_rate.fname);

            // Remove all curves and datums

            remove_all_curves_and_datums(); 

            // Draw viscous criterion datums

            draw_constant_datums(CHEST_VC_GOOD, CHEST_VC_ACCEPTABLE, CHEST_VC_MARGINAL, null);

            // Create image/curve of viscous criterion curves

            c_vc.AddToGraph();
            write_image(output_data.vc.title, output_data.vc.fname, c_vc, CHEST_VC_MARGINAL);
            
            output_data.vc.curveList.push(c_vc.id);
            write_curve("cur", output_data.vc.curveList, output_data.vc.fname);
            write_curve("csv", output_data.vc.curveList, output_data.vc.fname);

            // Remove all curves and datums

            remove_all_curves_and_datums(); 

            // Calc values

            var chest_com_results      = IIHS_get_constant_limit_results(c_disp_c180_mm, CHEST_COM_GOOD,      CHEST_COM_ACCEPTABLE,      CHEST_COM_MARGINAL);
            var chest_com_rate_results = IIHS_get_constant_limit_results(c_def_rate,     CHEST_COM_RATE_GOOD, CHEST_COM_RATE_ACCEPTABLE, CHEST_COM_RATE_MARGINAL);
            var chest_vc_results       = IIHS_get_constant_limit_results(c_vc,           CHEST_VC_GOOD,       CHEST_VC_ACCEPTABLE,       CHEST_VC_MARGINAL);

            // Write variables

            write_variable(f, "CHEST_COMPRESSION_RATING", chest_com_results[0],            "Chest compression rating", "General");
            write_variable(f, "CHEST_COMPRESSION_VALUE",  chest_com_results[1].toFixed(3), "Chest compression value",  "General");

            write_variable(f, "CHEST_DEFLECTION_RATE_RATING", chest_com_rate_results[0],            "Chest deflection rate rating", "General");
            write_variable(f, "CHEST_DEFLECTION_RATE_VALUE",  chest_com_rate_results[1].toFixed(3), "Chest deflection rate value",  "General");

            write_variable(f, "CHEST_VISCOUS_CRITERION_RATING", chest_vc_results[0],            "Chest viscous criterion rating", "General");
            write_variable(f, "CHEST_VISCOUS_CRITERION_VALUE",  chest_vc_results[1].toFixed(3), "Chest viscous criterion value",  "General");
        }
    }
    else
    {
        // No spring defined - variables should be set to blank by first script in template.
        // Create blank images to let the user know.

        write_blank_images(output_data, "NO SPRING ID DEFINED FOR");
    }



    // Acceleration 3ms exceedence

    var node_x, node_y, node_z;

    node_x = oGlblData.driver_chest_node_x;
    node_y = oGlblData.driver_chest_node_y;
    node_z = oGlblData.driver_chest_node_z;

    var output_data = {};

    output_data.acc = new OutputData("Chest Acceleration", images_dir + "/Chest_Acceleration");

    if( node_x != undefined &&
            node_y != undefined &&
            node_z != undefined )
    {
        // Read in acceleration X, Y and Z

        var c_acc = read_xyz_accelerations(1, node_x, node_y, node_z);

        // Check that the curves read in - i.e. the node id and component are valid
        // Create a blank image to let the user know.

        if(!c_acc.ax || !c_acc.ay || !c_acc.az)
        {
            write_blank_images(output_data, "NO ACCELERATION DATA FOR");
        }
        else
        {
            // Convert to g from model units

            var c_g = convert_xyz_acc_to_g(c_acc);

            // C1000 filter (convert to seconds first)

            var c_gx_c1000 = convert_to_s_and_filter(c_g.gx, Operate.C1000, REG_DT);
            var c_gy_c1000 = convert_to_s_and_filter(c_g.gy, Operate.C1000, REG_DT);
            var c_gz_c1000 = convert_to_s_and_filter(c_g.gz, Operate.C1000, REG_DT);

            // Vector combine

            var c_vec_g = Operate.Vec(c_gx_c1000, c_gy_c1000, c_gz_c1000);

            // Remove all curves and datums

            remove_all_curves_and_datums();

            // Convert from seconds back to model time

            c_vec_g = Operate.Mux(c_vec_g, oGlblData.time_factor);

            // Get 3ms exceedence

            DialogueInput("/AU");
            Operate.Tms(c_vec_g, TMS_PERIOD*oGlblData.time_factor);
            var tms = c_vec_g.tms;

            // Calculate rating

            var tms_rating = IIHS_get_rating(tms, CHEST_TMS_GOOD, CHEST_TMS_ACCEPTABLE, CHEST_TMS_MARGINAL);

            // Set label and style

            set_labels(c_vec_g, "Chest Acceleration Magnitude", "Time (" + oGlblData.unit_time + ")", "Acceleration (g)");
            set_line_style(c_vec_g, Colour.BLACK);

            // Create image/curve

            write_image(output_data.acc.title, output_data.acc.fname);
            
            output_data.acc.curveList.push(c_vec_g.id);
            write_curve("cur", output_data.acc.curveList, output_data.acc.fname);
            write_curve("csv", output_data.acc.curveList, output_data.acc.fname);

            // Write variables

            write_variable(f, "CHEST_ACCN_3MS_RATING", tms_rating,     "Chest Acceleration 3ms rating", "General");
            write_variable(f, "CHEST_ACCN_3MS_VALUE",  tms.toFixed(3), "Chest Acceleration 3ms value",  "General");
        }
    }
    else
    {
        // No nodes defined - variables should be set to blank by first script in template.
        // Create blank image to let the user know.

        write_blank_images(output_data, "NO NODE IDs DEFINED FOR");
    }
}



function do_lower_leg_and_foot_rating_calc(f, do_knee_slide_calc)
{
    // Calculates the lower leg and foot values and writes them into the Reporter csv variables file <f>.
    // If <do_knee_slide_calc> is true it does the Knee slide assessment

    // Do the left and right hand side
    // and the upper and lower tibia
    var dummy = {};
    dummy.driver = {};
    dummy.driver.Left = {};
    dummy.driver.Left.Lower = new DummyData("Driver", oGlblData.driver_left_lower_tibia_loadcell, "beam", oGlblData.driver_dummy, "version", oGlblData.driver_left_knee_transducer, "spring", oGlblData.driver_left_foot_node, "node");
    dummy.driver.Left.Upper = new DummyData("Driver", oGlblData.driver_left_upper_tibia_loadcell, "beam", oGlblData.driver_dummy, "version");
    dummy.driver.Right = {};
    dummy.driver.Right.Lower = new DummyData("Driver", oGlblData.driver_right_lower_tibia_loadcell, "beam", oGlblData.driver_dummy, "version", oGlblData.driver_right_knee_transducer, "spring", oGlblData.driver_right_foot_node, "node");
    dummy.driver.Right.Upper = new DummyData("Driver", oGlblData.driver_right_upper_tibia_loadcell, "beam", oGlblData.driver_dummy, "version");
    dummy.passenger = {};
    dummy.passenger.Left = {};
    dummy.passenger.Left.Lower = new DummyData("Passenger", oGlblData.passenger_left_lower_tibia_loadcell, "beam", oGlblData.passenger_dummy, "version");
    dummy.passenger.Left.Upper = new DummyData("Passenger", oGlblData.passenger_left_upper_tibia_loadcell, "beam", oGlblData.passenger_dummy, "version");
    dummy.passenger.Right = {};
    dummy.passenger.Right.Lower = new DummyData("Passenger", oGlblData.passenger_right_lower_tibia_loadcell, "beam", oGlblData.passenger_dummy, "version");
    dummy.passenger.Right.Upper = new DummyData("Passenger", oGlblData.passenger_right_upper_tibia_loadcell, "beam", oGlblData.passenger_dummy, "version");  

    for(var i in dummy) // loop through driver/passenger
    {
        for(var lr in dummy[i]) // loop through left/right side of dummy
        {
            var com_upper_results;
            var com_lower_results;

            for(var ul in dummy[i][lr]) // loop through lower/upper tibia
            {
                    
                var output_data = {};
                output_data.com = new OutputData(dummy[i][lr][ul].chart_label + " " + lr + " " + ul + " Tibia Compression", images_dir + "/" + dummy[i][lr][ul].variable_label + "_" + lr + "_" + ul + "_Tibia_Compression");
                output_data.ti  = new OutputData(dummy[i][lr][ul].chart_label + " " + lr + " " + ul + " Tibia Index",       images_dir + "/" + dummy[i][lr][ul].variable_label + "_" + lr + "_" + ul + "_Tibia_Index");

// Originally, it was not necessary to specify the dummy version for IIHS Side
// MDB Impact (it always extracted BEAM data) so if version is undefined, mark
// it as unspecified and treat as BEAM data
                if (dummy[i].version == undefined)
                    dummy[i].version = "<Unspecified>";

                if(dummy[i][lr][ul].beam_id != undefined)
                {
                    // Read in beam longitudinal force, MX and MY
                    
                       // for LSTC dummies this is measured from a Section

                    switch (dummy[i][lr][ul].version)
                    {
				        case "<Unspecified>":
                        case "V6_HUMANETICS_HIII_50TH_MALE":
                        case "V7_HUMANETICS_HIII_50TH_MALE":
                        case "V8_HUMANETICS_HIII_50TH_MALE":
                        case "Humanetics HIII 50M v1.5":
                        case "Humanetics HIII 50M v1.5.1":
                            var c_com   = read_data("BEAM BASIC", 0, dummy[i][lr][ul].beam_id, "NZ");  // NZ is longitudinal
                            var c_mom_x = read_data("BEAM BASIC", 0, dummy[i][lr][ul].beam_id, "MX");
                            var c_mom_y = read_data("BEAM BASIC", 0, dummy[i][lr][ul].beam_id, "MY");
                            var entity_str = "BEAM"; // Used for error message below
                            break;

                        case "LSTC HIII 50M Detailed v190217_beta":
                        case "LSTC HIII 50M v130528_beta":
                        case "LSTC HIII 50M Fast V2.0":
                            var c_com   = read_data("SECTION", 0, dummy[i][lr][ul].beam_id, "FZ");  // FZ is longitudinal
                            var c_mom_x = read_data("SECTION", 0, dummy[i][lr][ul].beam_id, "MX");
                            var c_mom_y = read_data("SECTION", 0, dummy[i][lr][ul].beam_id, "MY");
                            var entity_str = "SECTION"; // Used for error message below

                            // Mutiply compression data by -1 to so compression is -ve
                            c_com = Operate.Mul(c_com, -1)

                            break;

                        default:
                            ErrorMessage("Unsupported dummy \""+dummy[i][lr][ul].version+"\" in do_lower_leg_and_foot_rating_calc.");
                            write_blank_images(output_data, "UNSUPPORTED DUMMY \""+dummy[i][lr][ul].version+"\".");
                            break;
                    }
                    
                    // Check that the curves read in - i.e. the beam id and components are valid
                    // Create blank images to let the user know.

                    if(!c_com || !c_mom_x || !c_mom_y)
                    {
                        write_blank_images(output_data, "NO DATA FOR " + entity_str + " " + dummy[i][lr][ul].beam_id);
                    }
                    else
                    {
                        // Convert forces to kN
                        //         moments to Nm from model units

                        var c_com_kn   = Operate.Div(c_com,   oGlblData.kn_factor);
                        var c_mom_x_nm = Operate.Div(c_mom_x, oGlblData.nm_factor);
                        var c_mom_y_nm = Operate.Div(c_mom_y, oGlblData.nm_factor);         

                        // C600 filter (convert to seconds first)

                        var c_com_c600   = convert_to_s_and_filter(c_com_kn,   Operate.C600, REG_DT);
                        var c_mom_x_c600 = convert_to_s_and_filter(c_mom_x_nm, Operate.C600, REG_DT);
                        var c_mom_y_c600 = convert_to_s_and_filter(c_mom_y_nm, Operate.C600, REG_DT);

                        // Calculate resultant moment

                        var c_mom_res = Operate.Vec2d(c_mom_x_c600, c_mom_y_c600);

                        // Remove all curves and datums

                        remove_all_curves_and_datums();

                        // Convert from seconds back to model time

                        c_com_c600 = Operate.Mux(c_com_c600, oGlblData.time_factor);
                        c_mom_res  = Operate.Mux(c_mom_res,  oGlblData.time_factor);

                        //
                        // Calculate Tibia index = |Mr/Mrc| + |Fz/Fzc|
                        //
                        //  Where Mr  = Resultant moment
                        //        Mrc = Critical resultant moment = 225Nm
                        //        Fz  = Compression force
                        //        Fzc = Critical compressive force = 35.9kN
                        //
                        curve_id = Curve.FirstFreeID();

                        var c_ti = new Curve(curve_id);
                        var n = c_mom_res.npoints;

                        for(var j=1; j<=n; j++)
                        {
                            var p_com = c_com_c600.GetPoint(j);
                            var p_mom = c_mom_res.GetPoint(j);

                            var time = p_mom[0];
                            var ti   = Math.abs(p_mom[1]/TIBIA_MRC) + Math.abs(p_com[1]/-TIBIA_FZC);

                            c_ti.AddPoint(time, ti);
                        }

                        // Set labels and style

                        set_labels(c_com_c600, dummy[i][lr][ul].chart_label + " " + lr + " " + ul + " Tibia Axial Force", "Time (" + oGlblData.unit_time + ")", "Force (kN)");
                        set_labels(c_ti,       dummy[i][lr][ul].chart_label + " " + lr + " " + ul + " Tibia Index",       "Time (" + oGlblData.unit_time + ")", "Tibia Index");

                        set_line_style(c_com_c600, Colour.BLACK);
                        set_line_style(c_ti, Colour.BLACK);

                        // Remove all curves and datums

                        remove_all_curves_and_datums();

                        // Draw Tibia compression datums

                        draw_constant_datums(-TIBIA_COM_GOOD, -TIBIA_COM_ACCEPTABLE, -TIBIA_COM_MARGINAL, null);

                        // Create image/curve

                        c_com_c600.AddToGraph();
                        write_image(output_data.com.title, output_data.com.fname, c_com_c600, -TIBIA_COM_MARGINAL);
                        
                        output_data.com.curveList.push(c_com_c600.id);
                        write_curve("cur", output_data.com.curveList, output_data.com.fname);
                        write_curve("csv", output_data.com.curveList, output_data.com.fname);

                        // Remove all curves and datums

                        remove_all_curves_and_datums(); 

                        // Draw Tibia index datums

                        draw_constant_datums(TIBIA_INDEX_GOOD, TIBIA_INDEX_ACCEPTABLE, TIBIA_INDEX_MARGINAL, null);

                        // Create image/curve of tibia index curves

                        c_ti.AddToGraph();
                        write_image(output_data.ti.title, output_data.ti.fname, c_ti, TIBIA_INDEX_MARGINAL);
                        
                        output_data.ti.curveList.push(c_ti.id);
                        write_curve("cur", output_data.ti.curveList, output_data.ti.fname);
                        write_curve("csv", output_data.ti.curveList, output_data.ti.fname);

                        // Remove all curves and datums

                        remove_all_curves_and_datums(); 

                        // Calc values

                        if(ul == "Upper") com_upper_results = IIHS_get_constant_limit_results(c_com_c600, -TIBIA_COM_GOOD, -TIBIA_COM_ACCEPTABLE, -TIBIA_COM_MARGINAL);
                        else              com_lower_results = IIHS_get_constant_limit_results(c_com_c600, -TIBIA_COM_GOOD, -TIBIA_COM_ACCEPTABLE, -TIBIA_COM_MARGINAL);

                        var ti_results = IIHS_get_constant_limit_results(c_ti, TIBIA_INDEX_GOOD, TIBIA_INDEX_ACCEPTABLE, TIBIA_INDEX_MARGINAL);

                        // Write tibia index variables

                        write_variable(f, dummy[i][lr][ul].variable_label.toUpperCase() + "_" +lr.toUpperCase() + "_" + ul.toUpperCase() + "_TIBIA_INDEX_RATING", ti_results[0],            "Tibia index rating", "General");
                        write_variable(f, dummy[i][lr][ul].variable_label.toUpperCase() + "_" +lr.toUpperCase() + "_" + ul.toUpperCase() + "_TIBIA_INDEX_VALUE",  ti_results[1].toFixed(3), "Tibia index value",  "General");
                    }
                }
                else
                {
                    // No beam defined - variables should be set to blank by first script in template.
                    // Create blank images to let the user know.

                    write_blank_images(output_data, "NO BEAM ID DEFINED FOR"+ dummy[i][lr][ul].chart_label.toUpperCase());
                }
            }

            // Write compression variables to file - worst out of upper and lower

            if(com_upper_results != undefined && com_lower_results != undefined)
            {
                var tibia_com_rating = "GOOD";
                var tibia_com_value = Math.max(com_upper_results[1], com_lower_results[1]);

                if     (com_upper_results[0] == "POOR"       || com_lower_results[0] == "POOR")       tibia_com_rating = "POOR";
                else if(com_upper_results[0] == "MARGINAL"   || com_lower_results[0] == "MARGINAL")   tibia_com_rating = "MARGINAL";
                else if(com_upper_results[0] == "ACCEPTABLE" || com_lower_results[0] == "ACCEPTABLE") tibia_com_rating = "ACCEPTABLE";

                write_variable(f, dummy[i][lr][ul].variable_label.toUpperCase() + "_" + lr.toUpperCase() + "_TIBIA_COMPRESSION_RATING", tibia_com_rating,           "Tibia compression rating", "General");
                write_variable(f, dummy[i][lr][ul].variable_label.toUpperCase() + "_" + lr.toUpperCase() + "_TIBIA_COMPRESSION_VALUE",  tibia_com_value.toFixed(3), "Tibia compression value",  "General");
            }

            // Knee slide
            if (i == "driver") // Only do knee slide for driver, Lower
            {
                var output_data = {};
                output_data.disp = new OutputData(dummy[i][lr].Lower.chart_label + " " + lr + " Knee Displacement", images_dir + "/" + dummy[i][lr].Lower.variable_label + "_" + lr + "_Knee_Displacement");
                
// Originally, it was not necessary to specify the dummy version for IIHS Side
// MDB Impact (it always extracted SPRING data) so if version is undefined, mark
// it as unspecified and treat as SPRING data
                if (dummy[i].version == undefined)
                    dummy[i].version = "<Unspecified>";

                if(dummy[i][lr].Lower.spring_id != undefined)
                {
                    // Read in spring elongation force

                    // For Humanetics dummies this is measured from a Spring;
                    // for LSTC dummies this is measured from a Translational Joint
                    switch (dummy[i][lr].Lower.version)
                    {
				        case "<Unspecified>":
                        case "V6_HUMANETICS_HIII_50TH_MALE":
                        case "V7_HUMANETICS_HIII_50TH_MALE":
                        case "V8_HUMANETICS_HIII_50TH_MALE":
                        case "Humanetics HIII 50M v1.5":
                        case "Humanetics HIII 50M v1.5.1":
                            var c_disp = read_data("SPRING TR", 0, dummy[i][lr].Lower.spring_id, "ET"); 
                            var entity_str = "SPRING"; // Used for error message below
                            break;

                        case "LSTC HIII 50M Detailed v190217_beta":
                        case "LSTC HIII 50M v130528_beta":
                        case "LSTC HIII 50M Fast V2.0":
                            var c_disp = read_data("JOINT TR", 0, dummy[i][lr].Lower.spring_id, "XD");  //                             
                            var entity_str = "JOINT"; // Used for error message below
                            break;

                        default:
                            ErrorMessage("Unsupported dummy \""+dummy[i][lr][ul].version+"\" in do_lower_leg_and_foot_rating_calc.");
                            write_blank_images(output_data, "UNSUPPORTED DUMMY \""+dummy[i].version+"\".");
                            return;
                            break;
                    }
                    
                    if( !c_disp )
                    {
                        var blank_image_msg;

                        blank_image_msg = "NO " + dummy[i][lr].Lower.variable_label.toUpperCase() + "KNEE DISP DATA FOR " + entity_str + " " + dummy[i][lr][ul].spring_id;
                        create_blank_image(blank_image_msg, output_data.disp.fname);
                    }
                    else
                    {
                        // Convert to mm

                        var c_disp_mm = Operate.Div(c_disp, oGlblData.mm_factor);

                        // C180 filter (convert to seconds first)

                        var c_disp_c180 = convert_to_s_and_filter(c_disp_mm, Operate.C180, REG_DT);

                        // Remove all curves and datums

                        remove_all_curves_and_datums();

                        // Convert from seconds back to model time

                        c_disp_c180 = Operate.Mux(c_disp_c180, oGlblData.time_factor);

                        // Set labels and style

                        set_labels(c_disp_c180, dummy[i][lr].Lower.chart_label + " " + lr + " Knee Displacement", "Time (" + oGlblData.unit_time + ")", "Displacement (mm)");
                        set_line_style(c_disp_c180, Colour.BLACK);

                        // Remove all curves and datums

                        remove_all_curves_and_datums();

                        // Draw knee displacement datums

                        draw_constant_datums(KNEE_DISP_GOOD, KNEE_DISP_ACCEPTABLE, KNEE_DISP_MARGINAL, null);

                        // Create image/curve of femur compression curves

                        c_disp_c180.AddToGraph();
                        write_image(output_data.disp.title, output_data.disp.fname, c_disp_c180, KNEE_DISP_MARGINAL);
                        
                        output_data.disp.curveList.push(c_disp_c180.id);
                        write_curve("cur", output_data.disp.curveList, output_data.disp.fname);
                        write_curve("csv", output_data.disp.curveList, output_data.disp.fname);

                        // Remove all curves and datums

                        remove_all_curves_and_datums(); 

                        // Calc values

                        var knee_disp_results = IIHS_get_constant_limit_results(c_disp_c180, KNEE_DISP_GOOD, KNEE_DISP_ACCEPTABLE, KNEE_DISP_MARGINAL);

                        // Write displacement variables

                        write_variable(f, dummy[i][lr].Lower.variable_label.toUpperCase() + "_" + lr.toUpperCase() + "_KNEE_DISPLACEMENT_RATING", knee_disp_results[0],            "Knee displacement rating", "General");
                        write_variable(f, dummy[i][lr].Lower.variable_label.toUpperCase() + "_" + lr.toUpperCase() + "_KNEE_DISPLACEMENT_VALUE",  knee_disp_results[1].toFixed(3), "Knee displacement value",  "General");
                    }
                }
                else
                {
                    // No spring defined - variables should be set to blank by first script in template.
                    // Create a blank image to let the user know.

                    write_blank_images(output_data, "NO SPRING ID DEFINED FOR");
                }


                // Foot accelerations
                var output_data = {};
                output_data.acc = new OutputData(lr + " Foot Acceleration", images_dir + "/" + lr + "_Foot_Acceleration");

                if(dummy[i][lr].Lower.node_id != undefined)
                { 
                    // Read in acceleration X, Y and Z

                    var c_acc = read_xyz_accelerations(1, dummy[i][lr].Lower.node_id);

                    // Check that the curves read in - i.e. the node id and component are valid
                    // Create a blank image to let the user know.

                    if(!c_acc.ax || !c_acc.ay || !c_acc.az)
                    {
                        write_blank_images(output_data, "NO DATA FOR NODE " + dummy[i][lr].Lower.node_id);
                    }
                    else
                    {
                        // Convert to g from model units

                        var c_g = convert_xyz_acc_to_g(c_acc);

                        // C1000 filter (convert to seconds first)

                        var c_gx_c1000 = convert_to_s_and_filter(c_g.gx, Operate.C1000, REG_DT);
                        var c_gy_c1000 = convert_to_s_and_filter(c_g.gy, Operate.C1000, REG_DT);
                        var c_gz_c1000 = convert_to_s_and_filter(c_g.gz, Operate.C1000, REG_DT);

                        // Vector combine

                        var c_vec_g = Operate.Vec(c_gx_c1000, c_gy_c1000, c_gz_c1000);

                        // Convert from seconds back to model time

                        c_vec_g = Operate.Mux(c_vec_g, oGlblData.time_factor);

                        // Get peak acceleration (in g) and time (in seconds)

                        DialogueInput("/AU");
                        var peak_accn = c_vec_g.ymax;

                        // Set label and style

                        set_labels(c_vec_g, lr + " Foot Acceleration Magnitude", "Time (" + oGlblData.unit_time + ")", "Acceleration (g)");
                        set_line_style(c_vec_g, Colour.BLACK);

                        // Remove all curves and datums

                        remove_all_curves_and_datums();

                        // Draw foot acceleration datums

                        draw_constant_datums(FOOT_ACCN_GOOD, FOOT_ACCN_ACCEPTABLE, FOOT_ACCN_MARGINAL, null);      

                        // Create image/curve

                        c_vec_g.AddToGraph();
                        write_image(output_data.acc.title, output_data.acc.fname, c_vec_g, FOOT_ACCN_MARGINAL);
                        
                        output_data.acc.curveList.push(c_vec_g.id);
                        write_curve("cur", output_data.acc.curveList, output_data.acc.fname);
                        write_curve("csv", output_data.acc.curveList, output_data.acc.fname);

                        // Get rating

                        var foot_rating = IIHS_get_rating(peak_accn, FOOT_ACCN_GOOD, FOOT_ACCN_ACCEPTABLE, FOOT_ACCN_MARGINAL);

                        // Write variables to csv file for Reporter

                        write_variable(f, dummy[i][lr].Lower.variable_label.toUpperCase() + "_" + lr.toUpperCase() + "_FOOT_ACCELERATION_RATING", foot_rating,          "Foot acceleration rating", "General");
                        write_variable(f, dummy[i][lr].Lower.variable_label.toUpperCase() + "_" + lr.toUpperCase() + "_FOOT_ACCELERATION_VALUE",  peak_accn.toFixed(2), "Foot acceleration value",  "General");
                    }
                }
                else
                {

                    // No node defined - variables should be set to blank by first script in template.
                    // Create a blank image to let the user know.

                    write_blank_images(output_data, "NO NODE ID DEFINED FOR");
                }
            }
        }
    }
}




function get_neck_nij_results(curves)
{
    // Calculates results for the neck Nij assessment returning
    // the rating and maximum Nij. 
    //
    // <curves> is an array containing the 4 Nij curves:
    //
    //    Tension-Extension
    //    Tension-Flexion
    //    Compression-Extension
    //    Compression-Flexion
    //

    var ret = new Array(2); 

    rating  = "GOOD";
    nij_max = 0.0;

    var res = new Array(2);

    for(var j=0; j<curves.length; j++)
    {
        res = IIHS_get_constant_limit_results(curves[j], NECK_NIJ_GOOD, NECK_NIJ_ACCEPTABLE, NECK_NIJ_MARGINAL);

        if(res[1] > nij_max)
        {
            rating  = res[0];
            nij_max = res[1];
        }
    }

    ret[0] = rating;
    ret[1] = nij_max;

    return ret;
}



function draw_neck_axl_datums(t)
{
    // Creates datums for the neck axial force limits up to time <t>

    // Delete any that might already exist

    if(Datum.Exists("TEN GOOD"))       Datum.Delete("TEN GOOD");
    if(Datum.Exists("TEN ACCEPTABLE")) Datum.Delete("TEN ACCEPTABLE");
    if(Datum.Exists("TEN MARGINAL"))   Datum.Delete("TEN MARGINAL");
    if(Datum.Exists("COM GOOD"))       Datum.Delete("COM GOOD");
    if(Datum.Exists("COM ACCEPTABLE")) Datum.Delete("COM ACCEPTABLE");
    if(Datum.Exists("COM MARGINAL"))   Datum.Delete("COM MARGINAL");

    // Note that the order these are defined is important
    var d_com_marg = new Datum("COM MARGINAL",   Datum.CONSTANT_Y, -NECK_COM_MARGINAL);
    var d_com_accp = new Datum("COM ACCEPTABLE", Datum.CONSTANT_Y, -NECK_COM_ACCEPTABLE);
    var d_com_good = new Datum("COM GOOD",       Datum.CONSTANT_Y, -NECK_COM_GOOD);
    var d_ten_good = new Datum("TEN GOOD",       Datum.CONSTANT_Y,  NECK_TEN_GOOD);
    var d_ten_accp = new Datum("TEN ACCEPTABLE", Datum.CONSTANT_Y,  NECK_TEN_ACCEPTABLE);
    var d_ten_marg = new Datum("TEN MARGINAL",   Datum.CONSTANT_Y,  NECK_TEN_MARGINAL);

    d_ten_good.line_colour = EuroNCAPColour.GREEN;
    d_ten_accp.line_colour = EuroNCAPColour.YELLOW;
    d_ten_marg.line_colour = EuroNCAPColour.ORANGE;
    d_com_good.line_colour = EuroNCAPColour.GREEN;
    d_com_accp.line_colour = EuroNCAPColour.YELLOW;
    d_com_marg.line_colour = EuroNCAPColour.ORANGE;

    d_com_marg.fill_colour_below = EuroNCAPColour.RED;
    d_com_marg.fill_colour_above = EuroNCAPColour.ORANGE;
    d_com_accp.fill_colour_above = EuroNCAPColour.YELLOW;
    d_com_good.fill_colour_above = EuroNCAPColour.GREEN;
    d_ten_good.fill_colour_above = EuroNCAPColour.YELLOW;
    d_ten_accp.fill_colour_above = EuroNCAPColour.ORANGE;
    d_ten_marg.fill_colour_above = EuroNCAPColour.RED;
}


