/// <reference path="C:/SOURCE14/this_dir/this_js_api/this.intellisense.js" />
/// <reference path="C:/SOURCE14/reporter_dir/library/templates/scripts/this_common1.js" />
/// <reference path="C:/SOURCE14/reporter_dir/library/templates/scripts/this_common_EuroNCAP.js" />

/*NOTE: the calculation of the IR-TRACC length is not required as the length is fixed at 118.52235 mm [WSID v4.0] (take 118.5 for simplicity of hand-calc checking (~0.1% error))
        the original calculataion assumed the length was the same as the spring length but this is not the case (it is actually a bit shorter)
        from DYNAMORE on 15 July 2022:

        "The springs initial length is not the 2D IR_TRACC initial length(Reference length at t0):
            The 2D IR_TRACC inital length(Reference length at t0) is measured from Ball Joint IRTRACC to Pot axis in y-direchtion.
            These springs in dummy model are just used to measure the 2D IR_TRACC deflection (compression or extension).
            Both nodes of the spring element are fixed on the 2D IR_TRACC, the initial length will not effect the deflection output."
*/

//this value should be in mm but model may not be so x by factor to get to model scale (it is then added to the curve of change in length and divided by oGlblData.mm_factor to convert to mm)
var irtracc_length_abs_length_mm = 118.5; //HARD-CODED value valid for WSID v4.0

// Common functions for EuroNCAP 2015 side impact templates (MDB and Side Pole)
// This function considers World SID dummy for the calculation of Visocus Coefficient (VC)
function do_abdomen_rating_calc(f) {
    //Calculate the abdomen scores for the dummy and writes the relevant values
    //into the Reporter csv variables file <f>.

    var capping_limit = "FALSE";

    //note IRTRACC_length is same for all ribs
    var IRTRACC_length = irtracc_length_abs_length_mm * oGlblData.mm_factor;

    // Read in  forces front, mid and rear

    var spring;

    for (var fp = 0; fp < 2; fp++) {
        // upper and lower abdomen rotation and deformations from 2D IR-TRACC sensor are
        // used for the calculation of lateral deformation
        if (fp == 0) {
            var spring_rot = oGlblData.abdomen_upper_spring_rot;
            var spring_trans = oGlblData.abdomen_upper_spring_trans;
        } else if (fp == 1) {
            var spring_rot = oGlblData.abdomen_lower_spring_rot;
            var spring_trans = oGlblData.abdomen_lower_spring_trans;
        }

        //Check data components are available for this spring, if not create a blank image

        if (spring_rot != undefined || spring_trans != undefined) {
            //Read in SPRING rotation and compression

            var c_rot = read_data("SPRING ROT", 0, spring_rot, "RT");
            var c_disp = read_data("SPRING TR", 0, spring_trans, "ET");

            // Add the initial IRTRACT_length to get the actual length at a time (t)
            var abs_length = Operate.Add(c_disp, IRTRACC_length);

            // Converting the units of y-axis and x-axis into mm and sec respectively
            var abs_length_mm = Operate.Div(abs_length, oGlblData.mm_factor);
            var abs_length_mm_s = Operate.Dix(
                abs_length_mm,
                oGlblData.time_factor
            );
            var c_rot_s = Operate.Dix(c_rot, oGlblData.time_factor);

            // C180 filter and regularize the data using 0.000001 sec
            var abs_length_c180_mm = Operate.C180(abs_length_mm_s, REG_DT);
            var c_rot_c180 = Operate.C180(c_rot_s, REG_DT);

            // Calculating the Lateral displacement
            var c_cos_theta_c180 = Operate.Cos(c_rot_c180);
            var lat_length_mm = Operate.Mul(
                abs_length_c180_mm,
                c_cos_theta_c180
            );

            var lat_disp_elong_mm = Operate.Sub(lat_length_mm, IRTRACC_length);
            var lat_disp_mm = Operate.Mul(lat_disp_elong_mm, -1);

            if (fp == 0) {
                c_upp_c180_mm = lat_disp_mm;
                if (lat_disp_mm) {
                    var c_upp_c180_m = Operate.Div(lat_disp_mm, 1000); // in m (for viscous criterion calc)

                    // Calculate viscous Criterion for the upper

                    var c_upp_vc_s = Operate.Vc(
                        c_upp_c180_m,
                        CHEST_VC_A,
                        CHEST_VC_B,
                        CHEST_VC_METHOD
                    );
                }
            } else {
                c_bot_c180_mm = lat_disp_mm;
                if (lat_disp_mm) {
                    var c_bot_c180_m = Operate.Div(lat_disp_mm, 1000); // in m (for viscous criterion calc)

                    // Calculate viscous Criterion for the upper

                    var c_bot_vc_s = Operate.Vc(
                        c_bot_c180_m,
                        CHEST_VC_A,
                        CHEST_VC_B,
                        CHEST_VC_METHOD
                    );
                }
            }
        }
    }

    var output_data = {};

    output_data.com = new OutputData(
        "Abdomen Compression",
        images_dir + "/" + "Abdomen_Compression"
    );

    if (c_upp_c180_mm && c_bot_c180_mm) {
        // Remove all curves and datums

        remove_all_curves_and_datums();

        // Convert from seconds back to model time

        var c_upp_com = Operate.Mux(c_upp_c180_mm, oGlblData.time_factor);
        var c_bot_com = Operate.Mux(c_bot_c180_mm, oGlblData.time_factor);

        // Set label and style

        set_labels(
            c_upp_com,
            "Upper Abdomen Compression",
            "Time (" + oGlblData.unit_time + ")",
            "Compression (mm)"
        );
        set_labels(
            c_bot_com,
            "Bottom Abdomen Compression",
            "Time (" + oGlblData.unit_time + ")",
            "Compression (mm)"
        );

        set_line_style(c_upp_com, Colour.CYAN);
        set_line_style(c_bot_com, Colour.BLUE);

        // Draw datums

        draw_constant_datums(
            ABDOMEN_COMPRESSION_GOOD,
            ABDOMEN_COMPRESSION_ADEQUATE,
            ABDOMEN_COMPRESSION_MARGINAL,
            ABDOMEN_COMPRESSION_WEAK
        );

        // Create image and curve files of abdomen compresison curves

        write_image(
            output_data.com.title,
            output_data.com.fname,
            [c_upp_com, c_bot_com],
            ABDOMEN_COMPRESSION_WEAK
        );

        output_data.com.curveList.push(c_upp_com.id, c_bot_com.id);
        write_curve("cur", output_data.com.curveList, output_data.com.fname);
        write_curve("csv", output_data.com.curveList, output_data.com.fname);

        // Remove all curves and datums

        remove_all_curves_and_datums();

        // ABDOMEN COMPRESSION VARIABLES

        // Find the max compression of each rib

        var max_upp_rib_com = c_upp_com.ymax;
        var max_bottom_rib_com = c_bot_com.ymax;

        // Write variables for Reporter

        var topc = sliding_scale(
            max_upp_rib_com,
            ABDOMEN_COMPRESSION_GOOD,
            ABDOMEN_COMPRESSION_WEAK,
            ABDOMEN_HI_SCORE,
            ABDOMEN_LO_SCORE
        );

        write_variable(
            f,
            "ABDOMEN_TOP_COMPRESSION",
            max_upp_rib_com.toFixed(3),
            "Abdomen top compression value in mm",
            "String"
        );
        write_variable(
            f,
            "ABDOMEN_TOP_COMPRESSION_SCORE",
            topc.toFixed(3),
            "Abdomen top compression score",
            "String"
        );

        var botc = sliding_scale(
            max_bottom_rib_com,
            ABDOMEN_COMPRESSION_GOOD,
            ABDOMEN_COMPRESSION_WEAK,
            ABDOMEN_HI_SCORE,
            ABDOMEN_LO_SCORE
        );

        write_variable(
            f,
            "ABDOMEN_BOTTOM_COMPRESSION",
            max_bottom_rib_com.toFixed(3),
            "Abdomen bottom compression value in mm",
            "String"
        );
        write_variable(
            f,
            "ABDOMEN_BOTTOM_COMPRESSION_SCORE",
            botc.toFixed(3),
            "Abdomen bottom compression score",
            "String"
        );

        // Capping limit should be applied to overall score if these are zero

        if (topc == 0.0 || botc == 0.0) capping_limit = "TRUE";

        write_variable(
            f,
            "ABDOMEN_CAPPING_LIMIT",
            capping_limit,
            "Abdomen capping limit",
            "String"
        );
    } else {
        // No SPRING defined - variables should be set to blank by the first script in this template.
        // Create a blank image to let the user know.

        write_blank_images(output_data, "INVALID/NO SPRING ID DEFINED FOR");
    }

    // Do viscous criterion calculation

    var topvc = 0.0;
    var botvc = 0.0;

    var output_data = {};

    output_data.vc = new OutputData(
        "Abdomen Viscous Criterion",
        images_dir + "/" + "Abdomen_Viscous_Criterion"
    );

    if (c_upp_vc_s && c_bot_vc_s) {
        // Remove all curves and datums

        remove_all_curves_and_datums();

        // DRAWING THE VISCOUS CRITERION CURVES

        // Convert from seconds back to model time

        var c_upp_vc = Operate.Mux(c_upp_vc_s, oGlblData.time_factor);
        var c_bot_vc = Operate.Mux(c_bot_vc_s, oGlblData.time_factor);

        // Set label and style

        set_labels(
            c_upp_vc,
            "Upper Abdomen Viscous Criterion",
            "Time (" + oGlblData.unit_time + ")",
            "Abdomen Viscous Criterion"
        );
        set_labels(
            c_bot_vc,
            "Bottom Abdomen Viscous Criterion",
            "Time (" + oGlblData.unit_time + ")",
            "Abdomen Viscous Criterion"
        );

        set_line_style(c_upp_vc, Colour.CYAN);
        set_line_style(c_bot_vc, Colour.BLACK);

        // Draw datums

        draw_constant_datums(
            CHEST_VC_LIMIT,
            CHEST_VC_LIMIT,
            CHEST_VC_LIMIT,
            CHEST_VC_LIMIT
        );

        // Create image and curve files of viscous criterion curves

        write_image(
            output_data.vc.title,
            output_data.vc.fname,
            [c_upp_vc, c_bot_vc],
            CHEST_VC_LIMIT
        );

        output_data.vc.curveList.push(c_upp_vc.id, c_bot_vc.id);
        write_curve("cur", output_data.vc.curveList, output_data.vc.fname);
        write_curve("csv", output_data.vc.curveList, output_data.vc.fname);

        // Remove all curves and datums

        remove_all_curves_and_datums();

        // VISCOUS CRITERION VARIABLES

        var upp_vc = c_upp_vc.ymax;
        var bot_vc = c_bot_vc.ymax;

        write_variable(
            f,
            "ABDOMEN_TOP_VISCOUS_CRITERION",
            upp_vc.toFixed(3),
            "Upper abdomen viscous criterion value",
            "String"
        );
        write_variable(
            f,
            "ABDOMEN_BOTTOM_VISCOUS_CRITERION",
            bot_vc.toFixed(3),
            "Bottom abdomen viscous criterion value",
            "String"
        );

        if (upp_vc > CHEST_VC_LIMIT || bot_vc > CHEST_VC_LIMIT) {
            write_variable(
                f,
                "ABDOMEN_VISCOUS_CRITERION_MOD",
                -4.0,
                "Abdomen viscous criterion modifier",
                "String"
            );
        }
    } else {
        var title = "INVALID/NO SPRING ID DEFINED FOR ABDOMEN";
        var fname = images_dir + "/Abdomen_Viscous_Criterion";

        create_blank_image(title, fname);
    }
}

// This function considers World SID dummy for the calculation of Visocus Coefficient (VC)
function do_chest_rating_calc(f) {
    // Calclates the chest scores and writes
    // relevant values into the Reporter csv variable file <f>.

    var capping_limit = "FALSE";

    //note IRTRACC_length is same for all ribs
    var IRTRACC_length = irtracc_length_abs_length_mm * oGlblData.mm_factor;

    for (var rib = 0; rib < 3; rib++) {
        var spring;
        // chest rib rotation and deformations from 2D IR-TRACC sensor are
        // used for the calculation of lateral deformation of the chest ribs
        if (rib == 0) {
            var spring_rot = oGlblData.chest_upper_rib_spring_rot;
            var spring_trans = oGlblData.chest_upper_rib_spring_trans;
        } else if (rib == 1) {
            spring_rot = oGlblData.chest_middle_rib_spring_rot;
            spring_trans = oGlblData.chest_middle_rib_spring_trans;
        } else {
            spring_rot = oGlblData.chest_bottom_rib_spring_rot;
            spring_trans = oGlblData.chest_bottom_rib_spring_trans;
        }

        // Check data components are available for this SPRING, if not create a blank image

        if (spring_rot != undefined || spring_trans != undefined) {
            //Read in SPRING rotation and compression

            var c_rot = read_data("SPRING ROT", 0, spring_rot, "RT");
            var c_disp = read_data("SPRING TR", 0, spring_trans, "ET");

            // Add the initial IRTRACT_length to get the actual length at a time (t)
            var abs_length = Operate.Add(c_disp, IRTRACC_length);

            // Converting the units of y-axis and x-axis into mm and sec respectively
            var abs_length_mm = Operate.Div(abs_length, oGlblData.mm_factor);
            var abs_length_mm_s = Operate.Dix(
                abs_length_mm,
                oGlblData.time_factor
            );
            var c_rot_s = Operate.Dix(c_rot, oGlblData.time_factor);

            // C180 filter and regularize the data using 0.000001 sec
            var abs_length_c180_mm = Operate.C180(abs_length_mm_s, REG_DT);
            var c_rot_c180 = Operate.C180(c_rot_s, REG_DT);

            // Calculating the Lateral displacement
            var c_cos_theta_c180 = Operate.Cos(c_rot_c180);
            var lat_length_mm = Operate.Mul(
                abs_length_c180_mm,
                c_cos_theta_c180
            );

            var lat_disp_elong_mm = Operate.Sub(lat_length_mm, IRTRACC_length);
            var lat_disp_mm = Operate.Mul(lat_disp_elong_mm, -1);

            if (rib == 0) {
                c_upp_rib_c180_mm = lat_disp_mm;
                if (lat_disp_mm) {
                    var c_upp_rib_c180_m = Operate.Div(lat_disp_mm, 1000); // in m (for viscous criterion calc)

                    // Calculate viscous Criterion for the upper rib

                    var c_ur_vc_s = Operate.Vc(
                        c_upp_rib_c180_m,
                        CHEST_VC_A,
                        CHEST_VC_B,
                        CHEST_VC_METHOD
                    );
                }
            } else if (rib == 1) {
                c_mid_rib_c180_mm = lat_disp_mm;
                if (lat_disp_mm) {
                    var c_mid_rib_c180_m = Operate.Div(lat_disp_mm, 1000); //in m (for viscous criterion calc)

                    // Calculate viscous criterion for the middle rib

                    var c_mr_vc_s = Operate.Vc(
                        c_mid_rib_c180_m,
                        CHEST_VC_A,
                        CHEST_VC_B,
                        CHEST_VC_METHOD
                    );
                }
            } else {
                c_bot_rib_c180_mm = lat_disp_mm;
                if (lat_disp_mm) {
                    var c_bot_rib_c180_m = Operate.Div(lat_disp_mm, 1000); //in m (for viscous criterion calc)

                    // Calculate viscous criterion for the lower rib

                    var c_br_vc_s = Operate.Vc(
                        c_bot_rib_c180_m,
                        CHEST_VC_A,
                        CHEST_VC_B,
                        CHEST_VC_METHOD
                    );
                }
            }
        }
    }

    // Do compression calculation

    var topc = 0.0;
    var midc = 0.0;
    var botc = 0.0;

    var output_data = {};

    output_data.com = new OutputData(
        "Chest Compression",
        images_dir + "/Chest_Compression"
    );

    if (c_upp_rib_c180_mm && c_mid_rib_c180_mm && c_bot_rib_c180_mm) {
        // Remove all curves and datums

        remove_all_curves_and_datums();

        // Convert from seconds back to model time

        var c_upp_com = Operate.Mux(c_upp_rib_c180_mm, oGlblData.time_factor);
        var c_mid_com = Operate.Mux(c_mid_rib_c180_mm, oGlblData.time_factor);
        var c_bot_com = Operate.Mux(c_bot_rib_c180_mm, oGlblData.time_factor);

        // Set label and style

        set_labels(
            c_upp_com,
            "Upper Rib Compression",
            "Time (" + oGlblData.unit_time + ")",
            "Compression (mm)"
        );
        set_labels(
            c_mid_com,
            "Middle Rib Compression",
            "Time (" + oGlblData.unit_time + ")",
            "Compression (mm)"
        );
        set_labels(
            c_bot_com,
            "Bottom Rib Compression",
            "Time (" + oGlblData.unit_time + ")",
            "Compression (mm)"
        );

        set_line_style(c_upp_com, Colour.CYAN);
        set_line_style(c_mid_com, Colour.BLUE);
        set_line_style(c_bot_com, Colour.BLACK);

        // Draw datums

        draw_constant_datums(
            CHEST_COMPRESSION_GOOD,
            CHEST_COMPRESSION_ADEQUATE,
            CHEST_COMPRESSION_MARGINAL,
            CHEST_COMPRESSION_WEAK
        );

        // Create image and curve files of chest compresison curves

        write_image(
            output_data.com.title,
            output_data.com.fname,
            [c_upp_com, c_mid_com, c_bot_com],
            CHEST_COMPRESSION_WEAK
        );

        output_data.com.curveList.push(
            c_upp_com.id,
            c_mid_com.id,
            c_bot_com.id
        );
        write_curve("cur", output_data.com.curveList, output_data.com.fname);
        write_curve("csv", output_data.com.curveList, output_data.com.fname);

        // Remove all curves and datums

        remove_all_curves_and_datums();

        // CHEST COMPRESSION VARIABLES

        // Find the max compression of each rib

        var max_upp_rib_com = c_upp_com.ymax;
        var max_mid_rib_com = c_mid_com.ymax;
        var max_bottom_rib_com = c_bot_com.ymax;

        // Write variables to csv file for Reporter

        topc = sliding_scale(
            max_upp_rib_com,
            CHEST_COMPRESSION_GOOD,
            CHEST_COMPRESSION_WEAK,
            CHEST_TOP_HI_SCORE,
            CHEST_TOP_LO_SCORE
        );

        write_variable(
            f,
            "CHEST_TOP_COMPRESSION",
            max_upp_rib_com.toFixed(3),
            "Chest top rib compression value in mm",
            "String"
        );
        write_variable(
            f,
            "CHEST_TOP_COMPRESSION_SCORE",
            topc.toFixed(3),
            "Chest top rib compression score",
            "String"
        );

        midc = sliding_scale(
            max_mid_rib_com,
            CHEST_COMPRESSION_GOOD,
            CHEST_COMPRESSION_WEAK,
            CHEST_MID_HI_SCORE,
            CHEST_MID_LO_SCORE
        );

        write_variable(
            f,
            "CHEST_MIDDLE_COMPRESSION",
            max_mid_rib_com.toFixed(3),
            "Chest middle rib compression value in mm",
            "String"
        );
        write_variable(
            f,
            "CHEST_MIDDLE_COMPRESSION_SCORE",
            midc.toFixed(3),
            "Chest middle rib compression score",
            "String"
        );

        botc = sliding_scale(
            max_bottom_rib_com,
            CHEST_COMPRESSION_GOOD,
            CHEST_COMPRESSION_WEAK,
            CHEST_BOTTOM_HI_SCORE,
            CHEST_BOTTOM_LO_SCORE
        );

        write_variable(
            f,
            "CHEST_BOTTOM_COMPRESSION",
            max_bottom_rib_com.toFixed(3),
            "Chest bottom rib compression in mm",
            "String"
        );
        write_variable(
            f,
            "CHEST_BOTTOM_COMPRESSION_SCORE",
            botc.toFixed(3),
            "Chest bottom rib compression score",
            "String"
        );

        // Capping limit should be applied to overall score if compression is greater than capping limit value
        // (Note that this is limit may be different to the lower performance limit)

        if (
            max_upp_rib_com > CHEST_COMPRESSION_CAPPING_LIMIT ||
            max_mid_rib_com > CHEST_COMPRESSION_CAPPING_LIMIT ||
            max_bottom_rib_com > CHEST_COMPRESSION_CAPPING_LIMIT
        ) {
            capping_limit = "TRUE";
        }

        write_variable(
            f,
            "CHEST_CAPPING_LIMIT",
            capping_limit,
            "Chest capping limit",
            "String"
        );
    } else {
        write_blank_images(output_data, "INVALID/NO SPRING ID DEFINED FOR");
    }

    // Do viscous criterion calculation

    var topvc = 0.0;
    var midvc = 0.0;
    var botvc = 0.0;

    var output_data = {};

    output_data.vc = new OutputData(
        "Chest Viscous Criterion",
        images_dir + "/Chest_Viscous_Criterion"
    );

    if (c_ur_vc_s && c_mr_vc_s && c_br_vc_s) {
        // Remove all curves and datums

        remove_all_curves_and_datums();

        // DRAWING THE VISCOUS CRITERION CURVES

        // Convert from seconds back to model time

        var c_ur_vc = Operate.Mux(c_ur_vc_s, oGlblData.time_factor);
        var c_mr_vc = Operate.Mux(c_mr_vc_s, oGlblData.time_factor);
        var c_br_vc = Operate.Mux(c_br_vc_s, oGlblData.time_factor);

        // Set label and style

        set_labels(
            c_ur_vc,
            "Upper Rib Viscous Criterion",
            "Time (" + oGlblData.unit_time + ")",
            "Chest Viscous Criterion"
        );
        set_labels(
            c_mr_vc,
            "Middle Rib Viscous Criterion",
            "Time (" + oGlblData.unit_time + ")",
            "Chest Viscous Criterion"
        );
        set_labels(
            c_br_vc,
            "Bottom Rib Viscous Criterion",
            "Time (" + oGlblData.unit_time + ")",
            "Chest Viscous Criterion"
        );

        set_line_style(c_ur_vc, Colour.CYAN);
        set_line_style(c_mr_vc, Colour.BLUE);
        set_line_style(c_br_vc, Colour.BLACK);

        // Draw datums

        draw_constant_datums(
            CHEST_VC_LIMIT,
            CHEST_VC_LIMIT,
            CHEST_VC_LIMIT,
            CHEST_VC_LIMIT
        );

        // Create image and curve files of viscous criterion curves

        write_image(
            output_data.vc.title,
            output_data.vc.fname,
            [c_ur_vc, c_mr_vc, c_br_vc],
            CHEST_VC_LIMIT
        );

        output_data.vc.curveList.push(c_ur_vc.id, c_mr_vc.id, c_br_vc.id);
        write_curve("cur", output_data.vc.curveList, output_data.vc.fname);
        write_curve("csv", output_data.vc.curveList, output_data.vc.fname);

        // Remove all curves and datums

        remove_all_curves_and_datums();

        // VISCOUS CRITERION VARIABLES

        var ur_vc = c_ur_vc.ymax;
        var mr_vc = c_mr_vc.ymax;
        var br_vc = c_br_vc.ymax;

        write_variable(
            f,
            "CHEST_TOP_VISCOUS_CRITERION",
            ur_vc.toFixed(3),
            "Top rib viscous criterion value",
            "String"
        );
        write_variable(
            f,
            "CHEST_MIDDLE_VISCOUS_CRITERION",
            mr_vc.toFixed(3),
            "Middle rib viscous criterion value",
            "String"
        );
        write_variable(
            f,
            "CHEST_BOTTOM_VISCOUS_CRITERION",
            br_vc.toFixed(3),
            "Bottom rib viscous criterion value",
            "String"
        );

        if (
            ur_vc > CHEST_VC_LIMIT ||
            mr_vc > CHEST_VC_LIMIT ||
            br_vc > CHEST_VC_LIMIT
        ) {
            write_variable(
                f,
                "CHEST_VISCOUS_CRITERION_MOD",
                -4.0,
                "Chest viscous criterion modifier",
                "String"
            );
        }
    } else {
        write_blank_images(output_data, "INVALID/NO SPRING ID DEFINED FOR");
    }

    remove_all_curves_and_datums();
}

function do_shoulder_calc(f) {
    // Measures the lateral forces in the Shoulder
    // NOTE that LH beam and RH beam both have a local coordinate system with local Y pointing in the same direction as the
    // the a vector from the left shoulder to the right shoulder. This means that compression will be positive for the left
    // shoulder but it will be negative for the right shoulder. Consequently we need to flip the sign of the right shoulder
    // force to ensure consistency (i.e. compression is positive for both sides). This is done by dividing by - (minus) oGlblData.kn_factor
    // e.g. -1000 for a model where raw forces are in N

    // Creates a graph and writes the variables

    for (var sh = 0; sh < 2; sh++) {
        var beam;

        if (sh == 0) beam = oGlblData.shoulder_left_beam;
        else beam = oGlblData.shoulder_right_beam;

        if (beam != undefined) {
            //Read in the Shoulder shear force in Y

            var c_sh_fy = read_data("BEAM BASIC", 0, beam, "NY");

            if (sh == 0) {
                var c_sh_left_fy = c_sh_fy;

                // Check that the curves read in - i.e. the beam id and component are valid

                if (c_sh_left_fy) {
                    // Convert to kN

                    var c_sh_fy_kn = Operate.Div(
                        c_sh_left_fy,
                        oGlblData.kn_factor
                    );

                    //C600 filter (convert to seconds first)

                    var c_sh_fy_c600 = convert_to_s_and_filter(
                        c_sh_fy_kn,
                        Operate.C600,
                        REG_DT
                    );

                    // Convert from seconds back to model time

                    c_sh_left_fy = Operate.Mux(
                        c_sh_fy_c600,
                        oGlblData.time_factor
                    );
                }
            } else {
                var c_sh_right_fy = c_sh_fy;

                // Check that the curves read in - i.e. the beam id and component are valid

                if (c_sh_right_fy) {
                    // Convert to kN and compression +ve (right side compression is -ve by default so add a minus sign)

                    var c_sh_fy_kn = Operate.Div(
                        c_sh_right_fy,
                        -oGlblData.kn_factor
                    );

                    //C600 filter (convert to seconds first)

                    var c_sh_fy_c600 = convert_to_s_and_filter(
                        c_sh_fy_kn,
                        Operate.C600,
                        REG_DT
                    );

                    // Convert from seconds back to model time

                    c_sh_right_fy = Operate.Mux(
                        c_sh_fy_c600,
                        oGlblData.time_factor
                    );
                }
            }
        }
    }

    // Do compression calculation

    var left_sh = 0.0;
    var right_sh = 0.0;

    var output_data = {};

    output_data.force = new OutputData(
        "Shoulder Lateral Force",
        images_dir + "/" + "Shoulder_Force"
    );

    if (c_sh_left_fy && c_sh_right_fy) {
        // Remove all curves and datums

        remove_all_curves_and_datums();

        c_sh_left_fy.AddToGraph();
        c_sh_right_fy.AddToGraph();

        // Draw datums

        draw_constant_datums(
            SHOULDER_FORCE_LIMIT,
            SHOULDER_FORCE_LIMIT,
            SHOULDER_FORCE_LIMIT,
            SHOULDER_FORCE_LIMIT
        );

        // Set labels and Style

        set_labels(
            c_sh_left_fy,
            "Left Shoulder Lateral Force",
            "Time (" + oGlblData.unit_time + ")",
            "Compressive Force (kN)"
        );
        set_line_style(c_sh_left_fy, Colour.CYAN);

        set_labels(
            c_sh_right_fy,
            "Right Shoulder Lateral Force",
            "Time (" + oGlblData.unit_time + ")",
            "Compressive Force (kN)"
        );
        set_line_style(c_sh_right_fy, Colour.BLUE);

        // Create image and curve files of shoulder force curves

        write_image(
            output_data.force.title,
            output_data.force.fname,
            [c_sh_left_fy, c_sh_right_fy],
            SHOULDER_FORCE_LIMIT
        );

        output_data.force.curveList.push(c_sh_left_fy.id, c_sh_right_fy.id);
        write_curve(
            "cur",
            output_data.force.curveList,
            output_data.force.fname
        );
        write_curve(
            "csv",
            output_data.force.curveList,
            output_data.force.fname
        );

        // SHOULDER VARIABLES

        var left_fy = c_sh_left_fy.ymax;
        var right_fy = c_sh_right_fy.ymax;

        write_variable(
            f,
            "SHOULDER_LEFT_FORCE",
            left_fy.toFixed(3),
            "Left Shoulder lateral force in kN",
            "String"
        );
        write_variable(
            f,
            "SHOULDER_RIGHT_FORCE",
            right_fy.toFixed(3),
            "Right Shoulder lateral force in kN",
            "String"
        );

        if (left_fy > SHOULDER_FORCE_LIMIT || right_fy > SHOULDER_FORCE_LIMIT) {
            write_variable(
                f,
                "SHOULDER_MOD",
                -4.0,
                "Shoulder Force modifier",
                "String"
            );
        }
    } else {
        write_blank_images(output_data, "INVALID/NO BEAM ID DEFINED FOR");
    }
}

function do_pelvis_rating_calc(f) {
    //Calculates the pelvis scores for the dummy and writes relevant
    //values into the Reporter csv variable file <f>.

    var beam = oGlblData.pelvis_beam;

    var capping_limit = "FALSE";

    var output_data = {};

    output_data.force = new OutputData(
        "Pubic Symphysis Force",
        images_dir + "/Pelvis_Force"
    );

    if (beam != undefined) {
        // Should check data components are available for this beam

        // Read in Pubic Symphysis force

        var c_pf = read_data("BEAM BASIC", 0, beam, "NY");

        // Check that the curves read in - i.e. the beam id and component are valid
        // Create a blank image to let the user know.

        if (!c_pf) {
            write_blank_images(output_data, "NO DATA FOR BEAM " + beam);
        } else {
            // C600 filter (convert to seconds) and convert force into kN

            var c_pf = Operate.Dix(c_pf, oGlblData.time_factor);
            var c_pf = Operate.C600(c_pf, REG_DT);
            var c_pf = Operate.Div(c_pf, oGlblData.kn_factor);

            // Multiply by -1.0 to make compressive force +ve if required as
            // different versions can output the force in opposite directions.
            // Test for this by seeing if the absolute maximum value is -ve

            var pf_max = c_pf.ymax;
            var pf_min = c_pf.ymin;

            if (Math.abs(pf_min) > Math.abs(pf_max))
                c_pf = Operate.Mul(c_pf, -1.0);

            // Remove all curves and datums

            remove_all_curves_and_datums();

            // Convert from seconds back to model time

            c_pf = Operate.Mux(c_pf, oGlblData.time_factor);

            // Set labels and style

            set_labels(
                c_pf,
                "Pubic Symphysis Force",
                "Time (" + oGlblData.unit_time + ")",
                "Compressive Force (kN)"
            );
            set_line_style(c_pf, Colour.BLACK);

            // Draw datums

            draw_constant_datums(
                PELVIS_GOOD,
                PELVIS_ADEQUATE,
                PELVIS_MARGINAL,
                PELVIS_WEAK
            );

            write_image(
                output_data.force.title,
                output_data.force.fname,
                c_pf,
                PELVIS_WEAK
            );

            output_data.force.curveList.push(c_pf.id);
            write_curve(
                "cur",
                output_data.force.curveList,
                output_data.force.fname
            );
            write_curve(
                "csv",
                output_data.force.curveList,
                output_data.force.fname
            );

            // Calc values

            var pelvis_force_results = c_pf.ymax;

            // Calc pelvis force score

            // Write out variables

            var pelvis_score = sliding_scale(
                pelvis_force_results,
                PELVIS_GOOD,
                PELVIS_WEAK,
                PELVIS_HI_SCORE,
                PELVIS_LO_SCORE
            );

            write_variable(
                f,
                "PELVIS_FORCE",
                pelvis_force_results.toFixed(3),
                "Pelvis force value",
                "String"
            );
            write_variable(
                f,
                "PELVIS_FORCE_SCORE",
                pelvis_score.toFixed(3),
                "Pelvis Force score",
                "String"
            );

            // Capping limit should be applied to overall score if these are zero

            if (pelvis_score == 0.0) capping_limit = "TRUE";

            write_variable(
                f,
                "PELVIS_CAPPING_LIMIT",
                capping_limit,
                "Pelvis capping limit",
                "String"
            );

            // Remove all curves and datums

            remove_all_curves_and_datums();
        }
    } else {
        // No beam defined - variables should be set to blank by first script in template.
        // Create a blank image to let the user know.

        write_blank_images(output_data, "NO BEAM ID DEFINED FOR");
    }
}
