/// <reference path="C:/SOURCE14/this_dir/this_js_api/this.intellisense.js" />
/// <reference path="C:/SOURCE14/reporter_dir/library/templates/scripts/this_common1.js" />
/// <reference path="C:/SOURCE14/reporter_dir/library/templates/scripts/this_common_EuroNCAP.js" />


// Common functions for EuroNCAP ODB templates



if(File.Exists(fname_csv) && File.IsFile(fname_csv))
{
    // Do the common setup of assigning values to the oGlblData object,
    // setting the T/HIS graph up and reading in the model.

    var oGlblData = new Object();

    var f_vars = do_common_single_analysis_setup(oGlblData);

    // Convert variable datums to model time

    convert_variable_datum_times();

    // For each body region, do the calculation, writing results to the variables file.

    do_head_rating_calc(f_vars);

    do_neck_rating_calc(f_vars);

    do_chest_rating_calc(f_vars);

    do_knee_femur_pelvis_rating_calc(f_vars);

    do_lower_leg_foot_and_ankle_rating_calc(f_vars);

    f_vars.Close();
}


function do_head_rating_calc(f)
{
    // Calculates the head scores for both driver and passenger and writes
    // relevant values into the Reporter csv variables file <f>.

    // Capping limit

    var capping_limit = "FALSE";

    // Two passes, one for driver, one for passenger

    for(var pass = 0; pass<2; pass++)
    {
        var node;
        var node_y;
        var node_z;
        var dummy;

        if(pass == 0)
        {
            node = oGlblData.driver_head_node;
            node_y = oGlblData.driver_head_node_y;
            node_z = oGlblData.driver_head_node_z;
            dummy = "Driver";
        }

        if(pass == 1)
        {
            node = oGlblData.passenger_head_node;
            node_y = oGlblData.passenger_head_node_y;
            node_z = oGlblData.passenger_head_node_z;
            dummy = "Passenger";
        }

        var output_data = {};

        output_data.acc = new OutputData(dummy + " Head Acceleration",         images_dir + "/" + dummy + "_Head_Acceleration");
        output_data.int = new OutputData(dummy + " Steering Column Intrusion", images_dir + "/" + dummy + "_Steering_Column_Intrusion");

        if(node != undefined)
        { 
            // Read in acceleration X, Y and Z
            var one_node = true;

            // If nodes for three accelerometer positions defined, use all three, otherwise, read all components from same node
            if(node_y != undefined && node_z != undefined && node_y != 0 && node_z != 0)
            {
                var c_acc = read_xyz_accelerations(1, node, node_y, node_z);
                one_node = false;
            }
            else
            {
                var c_acc = read_xyz_accelerations(1, node);
            }

            // Check that the curves read in - i.e. the node id and component are valid
            // Create a blank image to let the user know.
            
            // Generate a relevant message for the blank image depending on which acceleration data are missing
            if(!c_acc.ax || !c_acc.ay || !c_acc.az)
            {
                var blank_image_msg;
                // If only one node supplied, just report that node ID
                if(one_node) blank_image_msg = "NO ACCEL DATA (NODE " + node + ") ";
                // else report each of the node IDs that were invalid
                else
                {
                    var acc_comp_lst = [];
                    var acc_id_lst = [];
                    if(!c_acc.ax)
                    {
                        acc_comp_lst.push(" X");
                        acc_id_lst.push(" "+node);
                    }
                    if(!c_acc.ay)
                    {
                        acc_comp_lst.push(" Y");
                        acc_id_lst.push(" "+node_y);
                    }
                    if(!c_acc.az)
                    {
                        acc_comp_lst.push(" Z");
                        acc_id_lst.push(" "+node_z);
                    }
                    blank_image_msg = "NO" + acc_comp_lst + " ACCEL DATA (NODE(s) " + acc_id_lst + ") ";
                }
                create_blank_image(blank_image_msg + output_data.acc.title, output_data.acc.fname);

                if(pass == 0)
                {
                    create_blank_image(blank_image_msg + output_data.int.title, output_data.int.fname);
                }
            }
            else
            {
                // Convert to g from model units

                var c_g = convert_xyz_acc_to_g(c_acc);

                // C1000 filter (convert to seconds first)

                var c_gx_c1000 = convert_to_s_and_filter(c_g.gx, Operate.C1000, REG_DT);
                var c_gy_c1000 = convert_to_s_and_filter(c_g.gy, Operate.C1000, REG_DT);
                var c_gz_c1000 = convert_to_s_and_filter(c_g.gz, Operate.C1000, REG_DT);

                // Vector combine

                var c_vec_g = Operate.Vec(c_gx_c1000, c_gy_c1000, c_gz_c1000);

                // Remove all curves and datums

                remove_all_curves_and_datums();

                // Convert from seconds back to model time

                c_vec_g = Operate.Mux(c_vec_g, oGlblData.time_factor);

                // Get peak acceleration (in g)

                DialogueInput("/AU");
                var peak_accn = c_vec_g.ymax;

                // Get HIC value

                var hic = Operate.Hic(c_vec_g, HIC_WINDOW, 1.0);

                // Get 3ms exceedence

                Operate.Tms(c_vec_g, TMS_PERIOD*oGlblData.time_factor);
                var tms = c_vec_g.tms;

                // Set label and style

                set_labels(c_vec_g, dummy + " Head Acceleration Magnitude", "Time (" + oGlblData.unit_time + ")", "Acceleration (g)");
                set_line_style(c_vec_g, Colour.BLACK);

                // Draw limit curves => up to final time

                var p = c_vec_g.GetPoint(c_vec_g.npoints); 
                if((pass == 0 && oGlblData.steering_wheel_airbag == "YES") || pass == 1) draw_head_limits(p[0], output_data.acc, true);   // Normal limits
                else                                                                     draw_head_limits(p[0], output_data.acc, false);  // Limits for driver with no airbag

                // Create image and curve files

                write_image(output_data.acc.title, output_data.acc.fname);

                output_data.acc.curveList.push(c_vec_g.id);
                write_curve("cur", output_data.acc.curveList, output_data.acc.fname);
                write_curve("csv", output_data.acc.curveList, output_data.acc.fname);

                // Remove all curves and datums

                remove_all_curves_and_datums();

                // Steering wheel displacements - driver only

                if(pass == 0)
                {
                    var head_steering_wheel_fa_disp   = 0.0;
                    var head_steering_wheel_lat_disp  = 0.0;
                    var head_steering_wheel_vert_disp = 0.0;

                    var head_steering_wheel_fa_mod   = 0.0;
                    var head_steering_wheel_lat_mod  = 0.0;
                    var head_steering_wheel_vert_mod = 0.0;

                    var spring;

                    for(var sp=0; sp<3; sp++)
                    {
                        if(sp == 0) spring = oGlblData.steering_col_intrusion_spring_x; // Fore-aft spring
                        else if(sp == 1) spring = oGlblData.steering_col_intrusion_spring_y; // Lateral spring
                        else if(sp == 2) spring = oGlblData.steering_col_intrusion_spring_z; // Vertical spring

                        if(spring != undefined)
                        {
                            var c_el = read_data("SPRING TR", 0, spring, "ET");

                            if(c_el)
                            {

                                // Convert to mm, take final value and calculate modifier

                                var c_el_mm = Operate.Div(c_el, oGlblData.mm_factor);

                                var c_fa;
                                var c_lat;
                                var c_vert;

                                if(sp == 0)
                                {
                                    // Fore/aft intrusion = -ve elongation is rearward so multiply by -1.0 first

                                    c_fa = Operate.Mul(c_el_mm, -1);

                                    var p = c_fa.GetPoint(c_fa.npoints);
                                    head_steering_wheel_fa_disp = p[1];
                                    head_steering_wheel_fa_mod = sliding_scale(head_steering_wheel_fa_disp, 
                                            HEAD_FA_DISP_LO_LIMIT, HEAD_FA_DISP_HI_LIMIT,
                                            STEERING_WHEEL_HI_SCORE, STEERING_WHEEL_LO_SCORE);
                                }
                                else if(sp == 1)
                                {
                                    // Lateral intrusion - use absolute value (left or right)

                                    c_lat = c_el_mm;

                                    var p = c_lat.GetPoint(c_lat.npoints);
                                    head_steering_wheel_lat_disp = p[1];
                                    head_steering_wheel_lat_mod = sliding_scale(Math.abs(head_steering_wheel_lat_disp), 
                                            HEAD_LAT_DISP_LO_LIMIT, HEAD_LAT_DISP_HI_LIMIT,
                                            STEERING_WHEEL_HI_SCORE, STEERING_WHEEL_LO_SCORE);
                                }
                                else if(sp == 2)
                                {
                                    // Vertical intrusion - +ve elongation is upward

                                    c_vert = c_el_mm;

                                    var p = c_vert.GetPoint(c_vert.npoints);
                                    head_steering_wheel_vert_disp = p[1];
                                    head_steering_wheel_vert_mod = sliding_scale(head_steering_wheel_vert_disp, 
                                            HEAD_VERT_DISP_LO_LIMIT, HEAD_VERT_DISP_HI_LIMIT,
                                            STEERING_WHEEL_HI_SCORE, STEERING_WHEEL_LO_SCORE);
                                }
                            }
                        }
                    }

                    // Overall steering wheel modifier - minimum of three modifiers

                    var head_steering_wheel_mod = Math.min(head_steering_wheel_fa_mod, head_steering_wheel_lat_mod, head_steering_wheel_vert_mod);

                    // Draw intrusion curves

                    // Remove all curves and datums

                    remove_all_curves_and_datums();

                    // Set labels and styles

                    if(c_fa)   set_labels(c_fa,   dummy + " Fore/Aft Intrusion (+ve rearward)", "Time (" + oGlblData.unit_time + ")", "Intrusion (mm)");
                    if(c_lat)  set_labels(c_lat,  dummy + " Lateral Intrusion",                 "Time (" + oGlblData.unit_time + ")", "Intrusion (mm)");
                    if(c_vert) set_labels(c_vert, dummy + " Vertical Intrusion (+ve rearward)", "Time (" + oGlblData.unit_time + ")", "Intrusion (mm)");

                    if(c_fa)   set_line_style(c_fa,   Colour.RED);
                    if(c_lat)  set_line_style(c_lat,  Colour.BLUE);
                    if(c_vert) set_line_style(c_vert, Colour.GREEN);

                    // Create image and curve files

                    if(c_fa)   c_fa.AddToGraph();
                    if(c_lat)  c_lat.AddToGraph();
                    if(c_vert) c_vert.AddToGraph();

                    if(c_fa)   output_data.int.curveList.push(c_fa.id);
                    if(c_lat)  output_data.int.curveList.push(c_lat.id);
                    if(c_vert) output_data.int.curveList.push(c_vert.id);

                    var title;
                    if(c_fa && c_lat && c_vert) title = output_data.int.title;
                    else                        title = "SOME SPRING IDs MISSING/DO NOT EXIST FOR " + output_data.int.title;

                    write_image(title, output_data.int.fname);
                    write_curve("cur", output_data.int.curveList, output_data.int.fname);
                    write_curve("csv", output_data.int.curveList, output_data.int.fname);
                }

                // Calculate head points

                // Driver with steering wheel airbag or passenger

                var peak_accn_score = 0.0;
                var hic_score       = 0.0;
                var tms_score       = 0.0;
                var score           = 0.0;

                if((pass == 0 && oGlblData.steering_wheel_airbag == "YES") || pass == 1)
                {       
                    if(peak_accn < PEAK_HEAD_ACCN_SOFT)
                    {
                        // 'Soft contact' -> 4.0
                        peak_accn_score = score = HEAD_HI_SCORE;
                    }
                    else
                    {
                        // 'Hard contact' -> minimum of hic and tms score

                        hic_score = sliding_scale(hic, HIC_LO_LIMIT, HIC_HI_LIMIT, HEAD_HI_SCORE, HEAD_LO_SCORE);
                        tms_score = sliding_scale(tms, TMS_LO_LIMIT, TMS_HI_LIMIT, HEAD_HI_SCORE, HEAD_LO_SCORE);

                        score = Math.min(hic_score, tms_score);

                        // Capping limit should be applied to overall score if these are zero

                        if(hic_score == 0.0 || tms_score == 0.0) capping_limit = "TRUE";
                    }
                }
                else
                {
                    // Driver with no steering wheel airbag - score is out of 2.0
                    // and is the minimum of the peak accn, hic and tms scores 

                    peak_accn_score = sliding_scale(peak_accn, PEAK_HEAD_ACCN_LO_LIMIT, PEAK_HEAD_ACCN_HI_LIMIT, HEAD_HI_SCORE_2, HEAD_LO_SCORE);
                    if(hic < HIC_HI_LIMIT) hic_score = HEAD_HI_SCORE_2;
                    tms_score = sliding_scale(tms, TMS_LO_LIMIT_2, TMS_HI_LIMIT_2, HEAD_HI_SCORE_2, HEAD_LO_SCORE);

                    score = Math.min(peak_accn_score, hic_score, tms_score);

                    // Capping limit should be applied to overall score if these are zero

                    if(peak_accn_score == 0.0 || hic_score == 0.0 || tms_score == 0.0) capping_limit = "TRUE";
                }

                // Add on steering wheel modifier to get final score (before applying subjective modifiers, this
                // is done in a script in the Reporter template).

                if(pass == 0) score += head_steering_wheel_mod;

                // Bound score between limits

                if((pass == 0 && oGlblData.steering_wheel_airbag == "YES") || pass == 1)
                {
                    if(score < HEAD_LO_SCORE) score = HEAD_LO_SCORE;
                    if(score > HEAD_HI_SCORE) score = HEAD_HI_SCORE;
                }
                else
                {
                    if(score < HEAD_LO_SCORE)   score = HEAD_LO_SCORE;
                    if(score > HEAD_HI_SCORE_2) score = HEAD_HI_SCORE_2;
                }

                // Write variables to csv file for Reporter

                if(pass == 0)
                {
                    // Driver steering wheel airbag

                    var value;
                    if(oGlblData.steering_wheel_airbag == "YES") value = "Yes";
                    else                                         value = "No";

                    write_variable(f, "DRIVER_STEERING_WHEEL_AIRBAG", value, "Driver steering wheel airbag present", "String");
                }

                // Peak head acceleration and score

                write_variable(f, dummy.toUpperCase() + "_PEAK_HEAD_ACCN",       peak_accn.toFixed(2),       "Peak head acceleration value", "String");
                write_variable(f, dummy.toUpperCase() + "_PEAK_HEAD_ACCN_SCORE", peak_accn_score.toFixed(3), "Peak head acceleration score", "String");

                // HIC value and score

                if(HIC_WINDOW == 0.036)
                {
                    write_variable(f, dummy.toUpperCase() + "_HEAD_HIC36",       hic.toFixed(2),       "HIC36 value", "String");
                    write_variable(f, dummy.toUpperCase() + "_HEAD_HIC36_SCORE", hic_score.toFixed(2), "HIC36 score", "String");
                }
                else
                {
                    write_variable(f, dummy.toUpperCase() + "_HEAD_HIC15",       hic.toFixed(2),       "HIC15 value", "String");
                    write_variable(f, dummy.toUpperCase() + "_HEAD_HIC15_SCORE", hic_score.toFixed(2), "HIC15 score", "String");
                }

                // 3ms value and score

                write_variable(f, dummy.toUpperCase() + "_HEAD_3MS",       tms.toFixed(2),       "3ms value", "String");
                write_variable(f, dummy.toUpperCase() + "_HEAD_3MS_SCORE", tms_score.toFixed(3), "3ms score", "String");

                // Steering wheel displacement values and modifier value

                if(pass == 0)
                {
                    write_variable(f, "DRIVER_STEERING_WHEEL_FA_DISP",   head_steering_wheel_fa_disp.toFixed(1),   "Steering wheel fore/aft displacement", "String");
                    write_variable(f, "DRIVER_STEERING_WHEEL_VERT_DISP", head_steering_wheel_vert_disp.toFixed(1), "Steering wheel vertical displacement", "String");
                    write_variable(f, "DRIVER_STEERING_WHEEL_LAT_DISP",  head_steering_wheel_lat_disp.toFixed(1),  "Steering wheel lateral displacement",  "String");
                    write_variable(f, "DRIVER_STEERING_WHEEL_DISP_MOD",  head_steering_wheel_mod.toFixed(1),       "Steering wheel displacement modifier", "String");
                }

                // Final score value (before subjective modifiers) - write a variable and store on <oGlblData> object

                write_variable(f, dummy.toUpperCase() + "_HEAD_PRE_MODS_SCORE",   score.toFixed(3),   "Final score", "String");
            }
        }
        else
        {

            // No node defined - variables should be set to blank by first script in template.
            // Create a blank image to let the user know.

            create_blank_image("NO NODE ID DEFINED FOR " + output_data.acc.title, output_data.acc.fname);

            if(pass == 0)
            {
                create_blank_image("NO NODE ID DEFINED FOR " + output_data.int.title, output_data.int.fname);
            }
        }

        // Should capping limit be applied

        write_variable(f, "HEAD_CAPPING_LIMIT", capping_limit, "Head capping limit", "String");
    }  
}

function do_neck_rating_calc(f)
{
    // Calculates the head scores for both driver and passenger and writes
    // relevant values into the Reporter csv variables file <f>.

    // Two passes, one for driver, one for passenger
    Message("Doing neck rating calculation...");
    var dummy = [];
    dummy.push(new DummyData("Driver", oGlblData.driver_neck_loadcell, "beam", oGlblData.driver_dummy, "version"));
    dummy.push(new DummyData("Passenger", oGlblData.passenger_neck_loadcell, "beam", oGlblData.passenger_dummy, "version"));

    for(var i = 0; i<dummy.length; i++)
    {

        // Data for images that we will create

        var output_data = {};

        output_data.shr_exc = new OutputData(dummy[i].chart_label + " Neck Shear Exceedence",   images_dir + "/" + dummy[i].variable_label + "_Neck_Shear");
        output_data.ten_exc = new OutputData(dummy[i].chart_label + " Neck Tension Exceedence", images_dir + "/" + dummy[i].variable_label + "_Neck_Tension");
        output_data.mom     = new OutputData(dummy[i].chart_label + " Neck Bending Moment",     images_dir + "/" + dummy[i].variable_label + "_Neck_Moment");

        if( dummy[i].beam_id != undefined )
        {
                        // Read in beam shear, tension and  MY
            // For Humanetics dummies this is measured from a beam;
            // for LSTC dummies this is measured from a cross section
            switch (dummy[i].version)
            {
                case "V6_HUMANETICS_HIII_50TH_MALE":
                case "V7_HUMANETICS_HIII_50TH_MALE":
                case "V8_HUMANETICS_HIII_50TH_MALE":
                case "Humanetics HIII 50M v1.5":
                case "Humanetics HIII 50M v1.5.1":
                    var c_shr = read_data("BEAM BASIC", 0, dummy[i].beam_id, "NX");  // NX is shear
                    var c_ten = read_data("BEAM BASIC", 0, dummy[i].beam_id, "NZ");
                    var c_mom = read_data("BEAM BASIC", 0, dummy[i].beam_id, "MY");
                    var entity_str = "BEAM"; // Used for error message below
                    break;

                case "LSTC HIII 50M Detailed v190217_beta":
                case "LSTC HIII 50M Fast V2.0":
                case "LSTC HIII 50M v130528_beta":
                    var c_shr = read_data("SECTION", 0, dummy[i].beam_id, "FX");  // FX is shear
                    var c_ten = read_data("SECTION", 0, dummy[i].beam_id, "FZ");
                    var c_mom = read_data("SECTION", 0, dummy[i].beam_id, "MY");
                    var entity_str = "SECTION"; // Used for error message below
                    break;

                default:
                    ErrorMessage("Unsupported dummy \""+dummy[i].version+"\".");
                    write_blank_images(output_data, "UNSUPPORTED DUMMY \""+dummy[i].version+"\"");
                    return;
                    break;
            }

            // Check that the curves read in - i.e. the beam id and component are valid
            // Create blank images to let the user know.

            if(!c_shr || !c_ten || !c_mom)
            {

               write_blank_images(output_data, "NO DATA FOR " + entity_str + " " + dummy[i].beam_id);

            }
            
            else
            {
                // Convert forces to kN and
                //         moment to Nm from model units

                var c_shr_kn = Operate.Div(c_shr, oGlblData.kn_factor);
                var c_ten_kn = Operate.Div(c_ten, oGlblData.kn_factor);
                var c_mom_nm = Operate.Div(c_mom, oGlblData.nm_factor);

                // C1000/C600 filter (convert to seconds first)

                var c_shr_c1000 = convert_to_s_and_filter(c_shr_kn, Operate.C1000, REG_DT);
                var c_ten_c1000 = convert_to_s_and_filter(c_ten_kn, Operate.C1000, REG_DT);
                var c_mom_c600  = convert_to_s_and_filter(c_mom_nm, Operate.C600,  REG_DT);

                // Exceedence curves for shear and tension. For shear take absolute values in case -ve Y values.

                var c_shr_abs = Operate.Abs(c_shr_c1000);
                var c_shr_exc = Operate.Exc(c_shr_abs, "positive");
                var c_ten_exc = Operate.Exc(c_ten_c1000, "positive");

                var c_shr_exc_abs = Operate.Abs(c_shr_exc);

                // Transpose the bending moment to the occupital condoyle moment

                var c_shr_adj = Operate.Mul(c_shr_c1000, NECK_SHR_ADJ_FAC);
                var c_mom_adj = Operate.Sub(c_mom_c600, c_shr_adj);

                // Remove all curves and datums

                remove_all_curves_and_datums();

                // Convert from seconds back to model time

                c_shr_exc_abs = Operate.Mux(c_shr_exc_abs, oGlblData.time_factor);
                c_ten_exc     = Operate.Mux(c_ten_exc,     oGlblData.time_factor);
                c_mom_adj     = Operate.Mux(c_mom_adj,     oGlblData.time_factor);

                // Set labels and style

                set_labels(c_shr_exc_abs, dummy[i].chart_label + " Neck Shear Exceedence",         "Cumulative Time (" + oGlblData.unit_time + ")", "Force (kN)");
                set_labels(c_ten_exc,     dummy[i].chart_label + " Neck Tension Exceedence",       "Cumulative Time (" + oGlblData.unit_time + ")", "Force (kN)");
                set_labels(c_mom_adj,     dummy[i].chart_label + " Neck Extension Bending Moment", "Time (" + oGlblData.unit_time + ")", "Bending Moment (Nm)");

                set_line_style(c_shr_exc_abs, Colour.BLACK);
                set_line_style(c_ten_exc,     Colour.BLACK);
                set_line_style(c_mom_adj,     Colour.BLACK);

                // Remove all curves and datums

                remove_all_curves_and_datums();

                // Draw neck shear limit curves => up to final time

                var p = c_shr_exc_abs.GetPoint(c_shr_exc_abs.npoints);       
                draw_variable_datums(p[0], NECK_SHR_EXC_TIME, NECK_SHR_EXC_GOOD, NECK_SHR_EXC_ADEQUATE, NECK_SHR_EXC_MARGINAL, NECK_SHR_EXC_WEAK);

                // Create image/curve of neck shear curves

                c_shr_exc_abs.AddToGraph();
                write_image(output_data.shr_exc.title, output_data.shr_exc.fname, c_shr_exc_abs, NECK_SHR_EXC_WEAK[0]);

                output_data.shr_exc.curveList.push(c_shr_exc_abs.id);
                write_curve("cur", output_data.shr_exc.curveList, output_data.shr_exc.fname);
                write_curve("csv", output_data.shr_exc.curveList, output_data.shr_exc.fname);

                // Remove all curves and datums

                remove_all_curves_and_datums();

                // Draw neck tension datums

                var p = c_ten_exc.GetPoint(c_ten_exc.npoints);       
                draw_variable_datums(p[0], NECK_TEN_EXC_TIME, NECK_TEN_EXC_GOOD, NECK_TEN_EXC_ADEQUATE, NECK_TEN_EXC_MARGINAL, NECK_TEN_EXC_WEAK);

                // Create image/curve of neck tension curves

                c_ten_exc.AddToGraph();
                write_image(output_data.ten_exc.title, output_data.ten_exc.fname, c_ten_exc, NECK_TEN_EXC_WEAK[0]);

                output_data.ten_exc.curveList.push(c_ten_exc.id);
                write_curve("cur", output_data.ten_exc.curveList, output_data.ten_exc.fname);
                write_curve("csv", output_data.ten_exc.curveList, output_data.ten_exc.fname);

                // Remove all curves and datums

                remove_all_curves_and_datums();

                // Draw moment limit datums

                draw_constant_datums(NECK_MOM_GOOD, NECK_MOM_ADEQUATE, NECK_MOM_MARGINAL, NECK_MOM_WEAK);

                // Create image/curve of moment curves

                c_mom_adj.AddToGraph();
                write_image(output_data.mom.title, output_data.mom.fname, c_mom_adj, NECK_MOM_WEAK);

                output_data.mom.curveList.push(c_mom_adj.id);
                write_curve("cur", output_data.mom.curveList, output_data.mom.fname);
                write_curve("csv", output_data.mom.curveList, output_data.mom.fname);

                // Remove all curves and datums

                remove_all_curves_and_datums();


                    // Calc values

                    var shr_results = get_variable_limit_results(c_shr_exc_abs, NECK_SHR_EXC_TIME, NECK_SHR_EXC_GOOD, NECK_SHR_EXC_MARGINAL, NECK_LO_SCORE, NECK_HI_SCORE);
                    var ten_results = get_variable_limit_results(c_ten_exc,     NECK_TEN_EXC_TIME, NECK_TEN_EXC_GOOD, NECK_TEN_EXC_MARGINAL, NECK_LO_SCORE, NECK_HI_SCORE);

                    var mom_results = get_constant_limit_results(c_mom_adj,     NECK_MOM_GOOD, NECK_MOM_MARGINAL, NECK_LO_SCORE, NECK_HI_SCORE);

                    // Write variables

                    // SHEAR

                    write_variable(f, dummy[i].variable_label.toUpperCase() + "_NECK_SHEAR_SCORE",          shr_results[0].toFixed(3), "Neck shear score",    "String");
                    write_variable(f, dummy[i].variable_label.toUpperCase() + "_NECK_SHEAR_LEVEL_EXCEEDED", shr_results[1].toFixed(3), "Neck shear level",    "String");
                    write_variable(f, dummy[i].variable_label.toUpperCase() + "_NECK_SHEAR_DURATION",       shr_results[2].toFixed(3), "Neck shear duration", "String");

                    // TENSION

                    write_variable(f, dummy[i].variable_label.toUpperCase() + "_NECK_TENSION_SCORE",          ten_results[0].toFixed(3), "Neck tension score",    "String");
                    write_variable(f, dummy[i].variable_label.toUpperCase() + "_NECK_TENSION_LEVEL_EXCEEDED", ten_results[1].toFixed(3), "Neck tension level",    "String");
                    write_variable(f, dummy[i].variable_label.toUpperCase() + "_NECK_TENSION_DURATION",       ten_results[2].toFixed(3), "Neck tension duration", "String");

                    // MOMENT

                    write_variable(f, dummy[i].variable_label.toUpperCase() + "_NECK_EXTENSION_SCORE",  mom_results[0].toFixed(3), "Neck extension score", "String");
                    write_variable(f, dummy[i].variable_label.toUpperCase() + "_NECK_BENDING_MOMENT",   mom_results[1].toFixed(3), "Neck bending moment",  "String");

                    // Final score is minimum of all scores

                    var value = Math.min(shr_results[0], ten_results[0], mom_results[0]).toFixed(3);
                    write_variable(f, dummy[i].variable_label.toUpperCase() + "_NECK_FINAL_SCORE",   value, "Neck final score",  "String");
                


            }
        }

        else
        {

            // No node defined - variables should be set to blank by first script in template.
            // Create blank images to let the user know.

            write_blank_images(output_data, "NO BEAM ID DEFINED FOR");
        }
    }
}


function do_chest_rating_calc(f)
{
    // Calculates the chest scores for both driver and passenger and writes
    // relevant values into the Reporter csv variables file <f>.

    // Capping limit

    var capping_limit = "FALSE";

    // Two passes, one for driver, one for passenger

    for(var pass = 0; pass<2; pass++)
    {
        var spring;
        var dummy;

        if(pass == 0) { spring = oGlblData.driver_chest_transducer;    dummy = "Driver";    }
        if(pass == 1) { spring = oGlblData.passenger_chest_transducer; dummy = "Passenger"; }

        var output_data = {};

        output_data.com = new OutputData(dummy + " Chest compression",       images_dir + "/" + dummy + "_Chest_Compression");
        output_data.vc  = new OutputData(dummy + " Chest Viscous Criterion", images_dir + "/" + dummy + "_Chest_Viscous_Criterion");

        if( spring != undefined )
        {
            // Read in spring rotation

            var c_rot = read_data("SPRING ROT", 0, spring, "RT");

            // Check that the curve read in - i.e. the spring id and component are valid
            // Create blank images to let the user know.

            if(!c_rot)
            {
                write_blank_images(output_data, "NO DATA FOR SPRING " + spring);
            }
            else
            {
                // Chest compression is calculated by converting the rotation in radians to a deflection using
                // a transfer function.
                //
                // For V7 dummies and earlier, the conversion is linear
                // For V8 dummies and later, the conversion uses a 3rd order polynomial
                //
                // The dummy manuals say that a negative rotation indicates compression, however testing has shown
                // that in some versions it is actually the other way around. If the maximum absolute value is +ve
                // then multiply the rotation by -1 so the conversion to compression works correctly.

                var rmax = c_rot.ymax;
                var rmin = c_rot.ymin;

                if(Math.abs(rmax) > Math.abs(rmin)) c_rot = Operate.Mul(c_rot, -1);

                var version;

                if(pass == 0) version = oGlblData.driver_dummy;
                if(pass == 1) version = oGlblData.passenger_dummy;

                var c_disp_mm;
                
                switch (version) 
                {
                    case "V7_HUMANETICS_HIII_50TH_MALE": 
                      c_disp_mm = Operate.Mul(c_rot, CHEST_V7_ROT_TO_COM_FACTOR);
                      break;
                    case "LSTC HIII 50M Detailed v190217_beta":                
                    case "LSTC HIII 50M Fast V2.0": 
                    case "LSTC HIII 50M v130528_beta":
                      c_disp_mm = Operate.Mul(c_rot, CHEST_LSTC_HIII_50TH_MALE_DETAILED_190217_BETA_ROT_TO_COM_FACTOR);
                      break;
                    case "V8_HUMANETICS_HIII_50TH_MALE":
                    case "Humanetics HIII 50M v1.5":
                    case "Humanetics HIII 50M v1.5.1":             
                      curve_id = Curve.FirstFreeID();
                      c_disp_mm = new Curve(curve_id);
                      var n = c_rot.npoints;

                      for(var i=1; i<=n; i++)
                      {
                        var p_rot = c_rot.GetPoint(i);

                        var time = p_rot[0];
                        var com  = CHEST_V8_ROT_TO_COM_FACTOR_A * p_rot[1] * p_rot[1] * p_rot[1] +
                            CHEST_V8_ROT_TO_COM_FACTOR_B * p_rot[1] * p_rot[1] + 
                            CHEST_V8_ROT_TO_COM_FACTOR_C * p_rot[1];

                        c_disp_mm.AddPoint(time, com);
                      }
                      break;
                    default:
                     ErrorMessage("Unsupported dummy \""+version+"\" in do_chest_rating_calc.");
                     write_blank_images(output_data, "UNSUPPORTED DUMMY \""+version+"\"");
                     return;
                     break;
                }

            
                         // C180 filter (convert to seconds first)

                var c_disp_c180_mm = convert_to_s_and_filter(c_disp_mm, Operate.C180, REG_DT);  // in mm (for compression calc)
                var c_disp_c180_m  = Operate.Div(c_disp_c180_mm, 1000);  // in m  (for viscous criterion calc)

                // Calculate viscous criterion

                c_vc = Operate.Vc(c_disp_c180_m, CHEST_VC_A, CHEST_VC_B, CHEST_VC_METHOD);

                // Remove all curves and datums

                remove_all_curves_and_datums();

                // Convert from seconds back to model time

                c_disp_c180_mm = Operate.Mux(c_disp_c180_mm, oGlblData.time_factor);
                c_vc           = Operate.Mux(c_vc,           oGlblData.time_factor);

                // Set labels and style

                set_labels(c_disp_c180_mm, dummy + " Chest Compression", "Time (" + oGlblData.unit_time + ")", "Compresion (mm)");
                set_labels(c_vc,           dummy + " Viscous Criterion", "Time (" + oGlblData.unit_time + ")", "Velocity (m/s)");

                set_line_style(c_disp_c180_mm, Colour.BLACK);
                set_line_style(c_vc,           Colour.BLACK);

                // Remove all curves and datums

                remove_all_curves_and_datums();

                // Draw chest compression datums

                draw_constant_datums(CHEST_COM_GOOD, CHEST_COM_ADEQUATE, CHEST_COM_MARGINAL, CHEST_COM_WEAK);

                // Create image and curve files of chest compression curves

                c_disp_c180_mm.AddToGraph();
                write_image(output_data.com.title, output_data.com.fname, c_disp_c180_mm, CHEST_COM_WEAK);

                output_data.com.curveList.push(c_disp_c180_mm.id);
                write_curve("cur", output_data.com.curveList, output_data.com.fname);
                write_curve("csv", output_data.com.curveList, output_data.com.fname);

                // Remove all curves and datums

                remove_all_curves_and_datums(); 

                // Draw viscous criterion datums

                draw_constant_datums(CHEST_VC_GOOD, CHEST_VC_ADEQUATE, CHEST_VC_MARGINAL, CHEST_VC_WEAK);

                // Create image and curve files of viscous criterion curves

                c_vc.AddToGraph();
                write_image(output_data.vc.title, output_data.vc.fname, c_vc, CHEST_VC_WEAK);

                output_data.vc.curveList.push(c_vc.id);
                write_curve("cur", output_data.vc.curveList, output_data.vc.fname);
                write_curve("csv", output_data.vc.curveList, output_data.vc.fname);

                // Remove all curves and datums

                remove_all_curves_and_datums(); 

                // Calc values

                var chest_com_results = get_constant_limit_results(c_disp_c180_mm, CHEST_COM_GOOD, CHEST_COM_WEAK, CHEST_LO_SCORE, CHEST_HI_SCORE);
                var chest_vc_results  = get_constant_limit_results(c_vc,           CHEST_VC_GOOD,  CHEST_VC_WEAK,  CHEST_LO_SCORE, CHEST_HI_SCORE);

                // Write variables

                write_variable(f, dummy.toUpperCase() + "_CHEST_COMPRESSION_SCORE",       chest_com_results[0].toFixed(3), "Chest compression score",        "String");        
                write_variable(f, dummy.toUpperCase() + "_CHEST_COMPRESSION",             chest_com_results[1].toFixed(3), "Chest compression",              "String");   

                write_variable(f, dummy.toUpperCase() + "_CHEST_VISCOUS_CRITERION_SCORE", chest_vc_results[0].toFixed(3),   "Chest viscous criterion score", "String");        
                write_variable(f, dummy.toUpperCase() + "_CHEST_VISCOUS_CRITERION",       chest_vc_results[1].toFixed(3),   "Chest viscous criterion",       "String");           

                // Capping limit should be applied to overall score if either of these are zero

                if(chest_com_results[0] == 0.0 || chest_vc_results[0] == 0.0) capping_limit = "TRUE";
            }
        }
        else
        {
            // No spring defined - variables should be set to blank by first script in template.
            // Create blank images to let the user know.

            write_blank_images(output_data, "NO SPRING ID DEFINED FOR");
        }
    

        // Calculate A-Pillar intrusion modifier - driver only
        var driver_output_data = {};
        driver_output_data.ap  = new OutputData(dummy + " Fore/Aft A-Pillar Intrusion", images_dir + "/" + dummy + "_Fore_Aft_A_Pillar_Intrusion");

        if(pass == 0)
        {
            var a_pillar_disp = 0.0;
            var spring = oGlblData.a_pillar_intrusion_spring;   // Acceleration pedal fore/aft spring

            var c_a_pillar_fa;

            if(spring != undefined)
            {
                var c_el = read_data("SPRING TR", 0, spring, "ET");

                if(c_el)
                {
                    // Convert to mm 

                    var c_el_mm = Operate.Div(c_el, oGlblData.mm_factor);

                    // Fore/aft intrusion = -ve elongation is rearward so multiply by -1.0 first       

                    c_a_pillar_fa = Operate.Mul(c_el_mm, -1.0);
                    var p = c_a_pillar_fa.GetPoint(c_a_pillar_fa.npoints);
                    a_pillar_disp = p[1];

                    set_labels(c_a_pillar_fa, dummy + " A-Pillar Fore/Aft Intrusion (+ve rearward)", "Time (" + oGlblData.unit_time + ")", "Intrusion (mm)");
                    set_line_style(c_a_pillar_fa, Colour.BLACK);
                }
            }


            // Calculate modifier

            var a_pillar_mod = sliding_scale(a_pillar_disp, 
                    CHEST_A_PILLAR_FA_DISP_GOOD, CHEST_A_PILLAR_FA_DISP_WEAK,
                    CHEST_A_PILLAR_HI_SCORE, CHEST_A_PILLAR_LO_SCORE);

            // Remove all curves and datums

            remove_all_curves_and_datums();

            // Draw fore/aft displacment datums

            if(c_a_pillar_fa) draw_constant_datums(CHEST_A_PILLAR_FA_DISP_GOOD, CHEST_A_PILLAR_FA_DISP_ADEQUATE, CHEST_A_PILLAR_FA_DISP_MARGINAL, CHEST_A_PILLAR_FA_DISP_WEAK);

            // Create image and curve files of fore/aft intrusion curves

            if(c_a_pillar_fa) c_a_pillar_fa.AddToGraph();

            if(c_a_pillar_fa) write_image(driver_output_data.ap.title, driver_output_data.ap.fname, c_a_pillar_fa, CHEST_A_PILLAR_FA_DISP_WEAK);
            else              write_blank_images(driver_output_data, "SPRING ID MISSING/DOES NOT EXIST FOR " + dummy + " Fore/Aft A-Pillar Intrusion");

            if(c_a_pillar_fa) driver_output_data.ap.curveList.push(c_a_pillar_fa.id);
            write_curve("cur", driver_output_data.ap.curveList, driver_output_data.ap.fname);
            write_curve("csv", driver_output_data.ap.curveList, driver_output_data.ap.fname);

            // Write variables

            write_variable(f, "DRIVER_CHEST_A_PILLAR_DISPLACEMENT", a_pillar_disp.toFixed(3), "A-Pillar fore/aft displacement",       "String");
            write_variable(f, "DRIVER_CHEST_A_PILLAR_MOD",          a_pillar_mod.toFixed(3),  "A-Pillar fore/aft displacement score", "String");
        }
      
    }
    // Should capping limit be applied

    write_variable(f, "CHEST_CAPPING_LIMIT", capping_limit, "Chest capping limit", "String");
}


function do_knee_femur_pelvis_rating_calc(f)
{
    // Calculates the knee, femur and pelvis scores for both driver and passenger and writes
    // relevant values into the Reporter csv variables file <f>.

    // Two passes, one for driver, one for passenger.
    // And for each one do the left and right hand side
    Message("Doing knee/femur/pelvis rating calculation...");
    var dummy = [];
    dummy[0] = [];
    dummy[0].push(new DummyData("Driver", oGlblData.driver_left_femur_loadcell, "beam", oGlblData.driver_left_knee_transducer, "spring", oGlblData.driver_dummy, "version"));
    dummy[0].push(new DummyData("Passenger", oGlblData.passenger_left_femur_loadcell, "beam", oGlblData.passenger_left_knee_transducer, "spring", oGlblData.passenger_dummy, "version"));

    dummy[1] = [];
    dummy[1].push(new DummyData("Driver",oGlblData.driver_right_femur_loadcell, "beam", oGlblData.driver_right_knee_transducer, "spring", oGlblData.driver_dummy, "version"));
    dummy[1].push(new DummyData("Passenger", oGlblData.passenger_right_femur_loadcell, "beam", oGlblData.passenger_right_knee_transducer, "spring", oGlblData.passenger_dummy, "version"));


    var fpass_knee_femur_pelvis_exists = "TRUE";
    for(var i = 0; i<dummy[0].length; i++)
    {
        for(var lr=0; lr<2; lr++)
        {
            var side;

            if(lr == 0) side = "Left";
            else        side = "Right";

            // Femur compression forces

            var output_data = {};

            output_data.com_exc = new OutputData(dummy[lr][i].chart_label + " " + side + " Femur Compression Exceedence", images_dir + "/" + dummy[lr][i].variable_label + "_" + side + "_Femur_Compression");

            if(dummy[lr][i].beam_id != undefined)
            {
                // Read in beam longitudinal force
                // For Humanetics dummies this is measured from a beam;
                // for LSTC dummies this is measured from a cross section

                switch (dummy[lr][i].version)
                {
                    case "V6_HUMANETICS_HIII_50TH_MALE":
                    case "V7_HUMANETICS_HIII_50TH_MALE":
                    case "V8_HUMANETICS_HIII_50TH_MALE":
                    case "Humanetics HIII 50M v1.5":
                    case "Humanetics HIII 50M v1.5.1":
                        var c_com = read_data("BEAM BASIC", 0, dummy[lr][i].beam_id, "NZ");  // NZ is longitudinal
                        var entity_str = "BEAM"; // Used for error message below
                        break;

                    case "LSTC HIII 50M Detailed v190217_beta":
                    case "LSTC HIII 50M Fast V2.0":
                    case "LSTC HIII 50M v130528_beta":
                        var c_com = read_data("SECTION", 0, dummy[lr][i].beam_id, "FZ");  // FZ is longitudinal
                        var entity_str = "SECTION"; // Used for error message below
                        break;

                    default:
                        ErrorMessage("Unsupported dummy \""+dummy[lr][i].version+"\".");
                        write_blank_images(output_data, "UNSUPPORTED DUMMY \""+dummy[lr][i].version+"\"");
                        return;
                        break;
                }

                // Check that the curve read in - i.e. the beam id and component are valid
                // Create blank image to let the user know.

                if(!c_com)
                {   
                        write_blank_images(output_data, "NO DATA FOR " + entity_str + " " + dummy[lr][i].beam_id);
                }
                else
                {
                    // Convert forces to kN from model units

                    var c_com_kn = Operate.Div(c_com, oGlblData.kn_factor);

                    // C600 filter (convert to seconds first)

                    var c_com_c600 = convert_to_s_and_filter(c_com_kn, Operate.C600, REG_DT);

                    // Exceedence curve for compression - negative values.

                    var c_com_exc = Operate.Exc(c_com_c600, "negative");

                    // Take absolute values

                    var c_com_exc_abs = Operate.Abs(c_com_exc);

                    // Remove all curves and datums

                    remove_all_curves_and_datums();

                    // Convert from seconds back to model time

                    c_com_exc_abs = Operate.Mux(c_com_exc_abs, oGlblData.time_factor);

                    // Set labels and style

                    set_labels(c_com_exc_abs, dummy[lr][i].chart_label + " " + side + " Femur Compression Exceedence", "Cumulative Time (" + oGlblData.unit_time + ")", "Compression (kN)");
                    set_line_style(c_com_exc_abs, Colour.BLACK);

                    // Remove all curves and datums

                    remove_all_curves_and_datums();

                    // Draw femur compression limit curves => up to final time

                    var p = c_com_exc_abs.GetPoint(c_com_exc_abs.npoints);       
                    draw_variable_datums(p[0], FEMUR_COM_EXC_TIME, FEMUR_COM_EXC_GOOD, FEMUR_COM_EXC_ADEQUATE, FEMUR_COM_EXC_MARGINAL, FEMUR_COM_EXC_WEAK);

                    // Create image/curve of femur compression curves

                    c_com_exc_abs.AddToGraph();
                    write_image(output_data.com_exc.title, output_data.com_exc.fname, c_com_exc_abs, FEMUR_COM_EXC_WEAK[0]);

                    output_data.com_exc.curveList.push(c_com_exc_abs.id);
                    write_curve("cur", output_data.com_exc.curveList, output_data.com_exc.fname);
                    write_curve("csv", output_data.com_exc.curveList, output_data.com_exc.fname);

                    // Remove all curves and datums

                    remove_all_curves_and_datums(); 

                    // Calc values

                    var com_results = get_variable_limit_results(c_com_exc_abs, FEMUR_COM_EXC_TIME, FEMUR_COM_EXC_GOOD, FEMUR_COM_EXC_MARGINAL, KNEE_FEMUR_PELVIS_LO_SCORE, KNEE_FEMUR_PELVIS_HI_SCORE);

                    // Write compression variables

                    write_variable(f, dummy[lr][i].variable_label.toUpperCase() + "_" + side.toUpperCase() + "_FEMUR_COMPRESSION_SCORE",          com_results[0].toFixed(3), "Femur compression score",    "String");
                    write_variable(f, dummy[lr][i].variable_label.toUpperCase() + "_" + side.toUpperCase() + "_FEMUR_COMPRESSION_LEVEL_EXCEEDED", com_results[1].toFixed(3), "Femur compression level",    "String");
                    write_variable(f, dummy[lr][i].variable_label.toUpperCase() + "_" + side.toUpperCase() + "_FEMUR_COMPRESSION_DURATION",       com_results[2].toFixed(3), "Femur compression duration", "String");
                }
            }
            else
            {

                // No beam defined - variables should be set to blank by first script in template.
                // Create a blank image to let the user know.

                write_blank_images(output_data, "NO BEAM ID DEFINED FOR");
            }


            // Knee slide

            var output_data = {};

            output_data.disp = new OutputData(dummy[lr][i].chart_label + " " + side + " Knee Displacement", images_dir + "/" + dummy[lr][i].variable_label + "_" + side + "_Knee_Displacement");

            if(dummy[lr][i].spring_id != undefined)
            {
                // Read in spring elongation force             
                // For Humanetics dummies this is measured from a Spring;
                // for LSTC dummies this is measured from a Translational Joint
                switch (dummy[lr][i].version)
                {
                    case "V6_HUMANETICS_HIII_50TH_MALE":
                    case "V7_HUMANETICS_HIII_50TH_MALE":
                    case "V8_HUMANETICS_HIII_50TH_MALE":
                    case "Humanetics HIII 50M v1.5" :
                        var c_disp = read_data("SPRING TR", 0, dummy[lr][i].spring_id, "ET");  
                        break;

                    case "LSTC HIII 50M Detailed v190217_beta":
                    case "LSTC HIII 50M Fast V2.0":
                        case "LSTC HIII 50M v130528_beta":
                        var c_disp = read_data("JOINT TR", 0, dummy[lr][i].spring_id, "XD");  //
                        var entity_str = "SECTION"; // Used for error message below

			// Multiply displacement by -1 so that compressive displacement is +ve
			c_disp = Operate.Mul(c_disp,-1)
                        break;

                    default:
                        ErrorMessage("Unsupported dummy \""+dummy[lr][i].version+"\".");
                        write_blank_images(output_data, "UNSUPPORTED DUMMY \""+dummy[lr][i].version+"\"");
                        return;                       
                        break;
                }
                // Check that the curve read in - i.e. the spring id and component are valid
                // Create blank image to let the user know.

                if(!c_disp)
                {

                    write_blank_images(output_data, "NO DATA FOR " + entity_str + " " + dummy[lr][i].beam_id);
                }
                else
                {
                    // Convert to mm

                    var c_disp_mm = Operate.Div(c_disp, oGlblData.mm_factor);

                    // C180 filter (convert to seconds first)

                    var c_disp_c180 = convert_to_s_and_filter(c_disp_mm, Operate.C180, REG_DT);

                    // Remove all curves and datums

                    remove_all_curves_and_datums();

                    // Convert from seconds back to model time

                    c_disp_c180 = Operate.Mux(c_disp_c180, oGlblData.time_factor);

                    // Set labels and style

                    set_labels(c_disp_c180, dummy[lr][i].chart_label + " " + side + " Knee Displacement", "Time (" + oGlblData.unit_time + ")", "Displacement (mm)");
                    set_line_style(c_disp_c180, Colour.BLACK);

                    // Remove all curves and datums

                    remove_all_curves_and_datums();

                    // Draw knee displacement datums

                    draw_constant_datums(KNEE_DISP_GOOD, KNEE_DISP_ADEQUATE, KNEE_DISP_MARGINAL, KNEE_DISP_WEAK);

                    // Create image/curve of knee displacement curves

                    c_disp_c180.AddToGraph();
                    write_image(output_data.disp.title, output_data.disp.fname, c_disp_c180, KNEE_DISP_WEAK);

                    output_data.disp.curveList.push(c_disp_c180.id);
                    write_curve("cur", output_data.disp.curveList, output_data.disp.fname);
                    write_curve("csv", output_data.disp.curveList, output_data.disp.fname);

                    // Remove all curves and datums

                    remove_all_curves_and_datums(); 

                    // Calc values

                    var knee_disp_results = get_constant_limit_results(c_disp_c180, KNEE_DISP_GOOD, KNEE_DISP_MARGINAL, KNEE_FEMUR_PELVIS_LO_SCORE, KNEE_FEMUR_PELVIS_HI_SCORE);

                    // Write displacement variables

                    write_variable(f, dummy[lr][i].variable_label.toUpperCase() + "_" + side.toUpperCase() + "_KNEE_DISPLACEMENT_SCORE", knee_disp_results[0].toFixed(3), "Knee displacement score", "String");
                    write_variable(f, dummy[lr][i].variable_label.toUpperCase() + "_" + side.toUpperCase() + "_KNEE_DISPLACEMENT",       knee_disp_results[1].toFixed(3), "Knee displacement",       "String");
                }
            }
            else
            {
                write_blank_images(output_data, "NO SPRING ID DEFINED FOR");
            }
        }
    }
}



function do_lower_leg_foot_and_ankle_rating_calc(f)

{    // Calculates the lower_leg, foot and ankle scores for both driver and passenger and writes
    // relevant values into the Reporter csv variables file <f>.

    // Two passes, one for driver, one for passenger.
    // And for each one do the left and right hand side
    // And for each one do upper and lower tibia
    var dummy = {};
    dummy.driver = {};
    dummy.driver.Left = {};
    dummy.driver.Left.Lower = new DummyData("Driver", oGlblData.driver_left_lower_tibia_loadcell, "beam", oGlblData.driver_dummy, "version");
    dummy.driver.Left.Upper = new DummyData("Driver", oGlblData.driver_left_upper_tibia_loadcell, "beam", oGlblData.driver_dummy, "version");
    dummy.driver.Right = {};
    dummy.driver.Right.Lower = new DummyData("Driver", oGlblData.driver_right_lower_tibia_loadcell, "beam", oGlblData.driver_dummy, "version");
    dummy.driver.Right.Upper = new DummyData("Driver", oGlblData.driver_right_upper_tibia_loadcell, "beam", oGlblData.driver_dummy, "version");
    dummy.passenger = {};
    dummy.passenger.Left = {};
    dummy.passenger.Left.Lower = new DummyData("Passenger", oGlblData.passenger_left_lower_tibia_loadcell, "beam", oGlblData.passenger_dummy, "version");
    dummy.passenger.Left.Upper = new DummyData("Passenger", oGlblData.passenger_left_upper_tibia_loadcell, "beam", oGlblData.passenger_dummy, "version");
    dummy.passenger.Right = {};
    dummy.passenger.Right.Lower = new DummyData("Passenger", oGlblData.passenger_right_lower_tibia_loadcell, "beam", oGlblData.passenger_dummy, "version");
    dummy.passenger.Right.Upper = new DummyData("Passenger", oGlblData.passenger_right_upper_tibia_loadcell, "beam", oGlblData.passenger_dummy, "version");  

    for(var i in dummy) // loop through driver/passenger
    {
        for(var lr in dummy[i]) // loop through left/right side of dummy
        {
            var com_upper_results;
            var com_lower_results;

            for(var ul in dummy[i][lr]) // loop through lower/upper tibia
            {
                var output_data = {};
                output_data.com = new OutputData(dummy[i][lr][ul].chart_label + " " + lr + " " + ul + " Tibia Compression", images_dir + "/" + dummy[i][lr][ul].variable_label + "_" + lr + "_" + ul + "_Tibia_Compression");
                output_data.ti  = new OutputData(dummy[i][lr][ul].chart_label + " " + lr + " " + ul + " Tibia Index",       images_dir + "/" + dummy[i][lr][ul].variable_label + "_" + lr + "_" + ul + "_Tibia_Index");

                if(dummy[i][lr][ul].beam_id != undefined)
                {
                    // Read in beam longitudinal force, MX and MY
                    // For Humanetics dummies this is measured from a Beam;
                    // for LSTC dummies this is measured from a Section

                    switch (dummy[i][lr][ul].version)
                    {
                        case "V6_HUMANETICS_HIII_50TH_MALE":
                        case "V7_HUMANETICS_HIII_50TH_MALE":
                        case "V8_HUMANETICS_HIII_50TH_MALE":
                        case "Humanetics HIII 50M v1.5":
                        case "Humanetics HIII 50M v1.5.1":
                            var c_com   = read_data("BEAM BASIC", 0, dummy[i][lr][ul].beam_id, "NZ");  // NZ is longitudinal
                            var c_mom_x = read_data("BEAM BASIC", 0, dummy[i][lr][ul].beam_id, "MX");
                            var c_mom_y = read_data("BEAM BASIC", 0, dummy[i][lr][ul].beam_id, "MY");
                            var entity_str = "BEAM"; // Used for error message below

                            // Mutiply compression data by -1 to so compression is +ve
                            c_com = Operate.Mul(c_com, -1)

                            break;

                        case "LSTC HIII 50M Detailed v190217_beta":
                        case "LSTC HIII 50M Fast V2.0":
                        case "LSTC HIII 50M v130528_beta":
                            var c_com   = read_data("SECTION", 0, dummy[i][lr][ul].beam_id, "FZ");  // FZ is longitudinal
                            var c_mom_x = read_data("SECTION", 0, dummy[i][lr][ul].beam_id, "MX");
                            var c_mom_y = read_data("SECTION", 0, dummy[i][lr][ul].beam_id, "MY");
                            var entity_str = "SECTION"; // Used for error message below	    
                            break;

                        default:
                            ErrorMessage("Unsupported dummy \""+dummy[i][lr][ul].version+"\".");
                            write_blank_images(output_data, "UNSUPPORTED DUMMY \""+dummy[i][lr][ul].version+"\"");
                            return;
                            break;
                    }

                    // Check that the curves read in - i.e. the beam id and components are valid
                    // Create blank images to let the user know.

                    if(!c_com || !c_mom_x || !c_mom_y)
                    {   
                      write_blank_images(output_data, "NO DATA FOR " + entity_str + " " + dummy[i][lr][ul].beam_id);
                    }
                    else
                    {
                        // Convert forces to kN
                        //         moments to Nm from model units

                        var c_com_kn   = Operate.Div(c_com, oGlblData.kn_factor);
                        var c_mom_x_nm = Operate.Div(c_mom_x, oGlblData.nm_factor);
                        var c_mom_y_nm = Operate.Div(c_mom_y, oGlblData.nm_factor);         

                        // C600 filter (convert to seconds first)

                        var c_com_c600   = convert_to_s_and_filter(c_com_kn,   Operate.C600, REG_DT);
                        var c_mom_x_c600 = convert_to_s_and_filter(c_mom_x_nm, Operate.C600, REG_DT);
                        var c_mom_y_c600 = convert_to_s_and_filter(c_mom_y_nm, Operate.C600, REG_DT);

                        // Calculate resultant moment

                        var c_mom_res = Operate.Vec2d(c_mom_x_c600, c_mom_y_c600);

                        // Remove all curves and datums

                        remove_all_curves_and_datums();

                        // Convert from seconds back to model time

                        c_com_c600 = Operate.Mux(c_com_c600, oGlblData.time_factor);
                        c_mom_res  = Operate.Mux(c_mom_res,  oGlblData.time_factor);

                        //
                        // Calculate Tibia index = |Mr/Mrc| + |Fz/Fzc|
                        //
                        //  Where Mr  = Resultant moment
                        //        Mrc = Critical resultant moment = 225Nm
                        //        Fz  = Compression force
                        //        Fzc = Critical compressive force = 35.9kN
                        //
                        curve_id = Curve.FirstFreeID();

                        var c_ti = new Curve(curve_id);
                        var n = c_mom_res.npoints;

                        for(var j = 1; j <= n; j++)
                        {
                            var p_com = c_com_c600.GetPoint(j);
                            var p_mom = c_mom_res.GetPoint(j);

                            var time = p_mom[0];
                            var ti   = Math.abs(p_mom[1]/TIBIA_MRC) + Math.abs(p_com[1]/TIBIA_FZC);

                            c_ti.AddPoint(time, ti);
                        }

                        // Set labels and style

                        set_labels(c_com_c600, dummy[i][lr][ul].chart_label + " " + lr + " " + ul + " Tibia Compression", "Time (" + oGlblData.unit_time + ")", "Compression (kN)");
                        set_labels(c_ti,       dummy[i][lr][ul].chart_label + " " + lr + " " + ul + " Tibia Index",       "Time (" + oGlblData.unit_time + ")", "Tibia Index");

                        set_line_style(c_com_c600, Colour.BLACK);
                        set_line_style(c_ti, Colour.BLACK);

                        // Remove all curves and datums

                        remove_all_curves_and_datums();

                        // Draw Tibia compression datums

                        draw_constant_datums(TIBIA_COM_GOOD, TIBIA_COM_ADEQUATE, TIBIA_COM_MARGINAL, TIBIA_COM_WEAK);

                        // Create image/curve of tibia compresion curves

                        c_com_c600.AddToGraph();
                        write_image(output_data.com.title, output_data.com.fname, c_com_c600, TIBIA_COM_WEAK);

                        output_data.com.curveList.push(c_com_c600.id);
                        write_curve("cur", output_data.com.curveList, output_data.com.fname);
                        write_curve("csv", output_data.com.curveList, output_data.com.fname);

                        // Remove all curves and datums

                        remove_all_curves_and_datums(); 

                        // Draw Tibia index datums

                        draw_constant_datums(TIBIA_INDEX_GOOD, TIBIA_INDEX_ADEQUATE, TIBIA_INDEX_MARGINAL, TIBIA_INDEX_WEAK);

                        // Create image/curve of tibia index curves

                        c_ti.AddToGraph();
                        write_image(output_data.ti.title, output_data.ti.fname, c_ti, TIBIA_INDEX_WEAK);

                        output_data.ti.curveList.push(c_ti.id);
                        write_curve("cur", output_data.ti.curveList, output_data.ti.fname);
                        write_curve("csv", output_data.ti.curveList, output_data.ti.fname);

                        // Remove all curves and datums

                        remove_all_curves_and_datums(); 

                        // Calc values

                        if(ul == "Upper") com_upper_results = get_constant_limit_results(c_com_c600, TIBIA_COM_GOOD, TIBIA_COM_WEAK, LOWER_LEG_FOOT_AND_ANKLE_LO_SCORE, LOWER_LEG_FOOT_AND_ANKLE_HI_SCORE);
                        else              com_lower_results = get_constant_limit_results(c_com_c600, TIBIA_COM_GOOD, TIBIA_COM_WEAK, LOWER_LEG_FOOT_AND_ANKLE_LO_SCORE, LOWER_LEG_FOOT_AND_ANKLE_HI_SCORE);

                        var ti_results = get_constant_limit_results(c_ti, TIBIA_INDEX_GOOD, TIBIA_INDEX_WEAK, LOWER_LEG_FOOT_AND_ANKLE_LO_SCORE, LOWER_LEG_FOOT_AND_ANKLE_HI_SCORE);

                        // Write tibia index variables - and store on <oGlblData> object

                        write_variable(f, dummy[i][lr][ul].variable_label.toUpperCase() + "_" + lr.toUpperCase() + "_" + ul.toUpperCase() + "_TIBIA_INDEX_SCORE", ti_results[0].toFixed(3), "Tibia index score", "String");
                        write_variable(f, dummy[i][lr][ul].variable_label.toUpperCase() + "_" + lr.toUpperCase() + "_" + ul.toUpperCase() + "_TIBIA_INDEX",       ti_results[1].toFixed(3), "Tibia index",       "String");

                        // Write compression variables to file and store in <oGlblData> object - worst out of upper and lower

                        if(com_upper_results != undefined && com_lower_results != undefined)
                        {
                            var tibia_com_score = Math.min(com_upper_results[0], com_lower_results[0])
                            write_variable(f, dummy[i][lr][ul].variable_label.toUpperCase() + "_" + lr.toUpperCase() + "_TIBIA_COMPRESSION_SCORE", tibia_com_score.toFixed(3), "Tibia compression score", "String");
                            var value = Math.max(com_upper_results[1], com_lower_results[1]).toFixed(3);
                            write_variable(f, dummy[i][lr][ul].variable_label.toUpperCase() + "_" + lr.toUpperCase() + "_TIBIA_COMPRESSION", value, "Tibia compression value", "String");
                        }
                    }
                }
                else
                {
                    // No beam defined - variables should be set to blank by first script in template.
                    // Create blank images to let the user know.

                    write_blank_images(output_data, "NO BEAM ID DEFINED FOR");
                }
            }
        }
    }

    // Calculate pedal vertical and fore/aft displacements to get modifier - for driver only
    // Do 6 passes - acceleration, brake and clutch pedal for vertical and fore/aft

    var accn_vert_disp   = 0.0;
    var brake_vert_disp  = 0.0;
    var clutch_vert_disp = 0.0;

    var accn_fa_disp     = 0.0;
    var brake_fa_disp    = 0.0;
    var clutch_fa_disp   = 0.0;

    for(var sp = 0; sp < 6; sp++)
    {
        if(sp == 0)      spring = oGlblData.accn_pedal_intrusion_spring_z;   // Acceleration pedal vertical spring
        else if(sp == 1) spring = oGlblData.brake_pedal_intrusion_spring_z;  // Brake pedal vertical spring
        else if(sp == 2) spring = oGlblData.clutch_pedal_intrusion_spring_z; // Clutch pedal vertical spring
        else if(sp == 3) spring = oGlblData.accn_pedal_intrusion_spring_x;   // Acceleration  pedal fore/aft spring
        else if(sp == 4) spring = oGlblData.brake_pedal_intrusion_spring_x;  // Brake pedal fore/aft spring
        else if(sp == 5) spring = oGlblData.clutch_pedal_intrusion_spring_x; // Clutch pedal fore/aft spring

        var c_accn_vert,   c_accn_fa;
        var c_brake_vert,  c_brake_fa;
        var c_clutch_vert, c_clutch_fa;

        if(spring != undefined)
        {
            var c_el = read_data("SPRING TR", 0, spring, "ET");

            if(c_el)
            {
                // Convert to mm and take final value

                var c_el_mm = Operate.Div(c_el, oGlblData.mm_factor);
                var p = c_el_mm.GetPoint(c_el_mm.npoints);

                if(sp == 0)
                {
                    c_accn_vert = c_el_mm;
                    accn_vert_disp = p[1];

                    set_labels(c_accn_vert, "Driver Acceleration Pedal Vertical Intrusion (+ve upward)", "Time (" + oGlblData.unit_time + ")", "Intrusion (mm)");
                    set_line_style(c_accn_vert, Colour.CYAN);
                }
                else if(sp == 1)
                {
                    c_brake_vert = c_el_mm;
                    brake_vert_disp = p[1];

                    set_labels(c_brake_vert, "Driver Brake Pedal Vertical Intrusion (+ve upward)", "Time (" + oGlblData.unit_time + ")", "Intrusion (mm)");
                    set_line_style(c_brake_vert, Colour.BLUE);
                }
                else if(sp == 2)
                {
                    c_clutch_vert = c_el_mm;
                    clutch_vert_disp = p[1];

                    set_labels(c_clutch_vert, "Driver Clutch Pedal Vertical Intrusion (+ve upward)", "Time (" + oGlblData.unit_time + ")", "Intrusion (mm)");
                    set_line_style(c_clutch_vert, Colour.MAGENTA);
                }
                else if(sp == 3)
                {
                    c_accn_fa = Operate.Mul(c_el_mm, -1.0);
                    var p = c_accn_fa.GetPoint(c_accn_fa.npoints);
                    accn_fa_disp = p[1];

                    set_labels(c_accn_fa, "Driver Acceleration Pedal Fore/Aft Intrusion (+ve rearward)", "Time (" + oGlblData.unit_time + ")", "Intrusion (mm)");
                    set_line_style(c_accn_fa, Colour.CYAN);
                }
                else if(sp == 4)
                {
                    c_brake_fa = Operate.Mul(c_el_mm, -1.0);
                    var p = c_brake_fa.GetPoint(c_brake_fa.npoints);
                    brake_fa_disp  = p[1];

                    set_labels(c_brake_fa, "Driver Brake Pedal Fore/Aft Intrusion (+ve rearward)", "Time (" + oGlblData.unit_time + ")", "Intrusion (mm)");
                    set_line_style(c_brake_fa, Colour.BLUE);
                }
                else if(sp == 5)
                {
                    c_clutch_fa = Operate.Mul(c_el_mm, -1.0);
                    var p = c_clutch_fa.GetPoint(c_clutch_fa.npoints);
                    clutch_fa_disp = p[1];  

                    set_labels(c_clutch_fa, "Driver Clutch Pedal Fore/Aft Intrusion (+ve rearward)", "Time (" + oGlblData.unit_time + ")", "Intrusion (mm)");
                    set_line_style(c_clutch_fa, Colour.MAGENTA);
                }
            }
        }
    }

    // Get maximum vertical and fore/aft displacement and calculate modifier

    var max_pedal_vert_disp = Math.max(accn_vert_disp, brake_vert_disp, clutch_vert_disp);
    var max_pedal_fa_disp   = Math.max(accn_fa_disp,   brake_fa_disp,   clutch_fa_disp);

    var pedal_vert_disp_mod = sliding_scale(max_pedal_vert_disp, 
            LOWER_LEG_PEDAL_VERT_DISP_GOOD, LOWER_LEG_PEDAL_VERT_DISP_WEAK,
            LOWER_LEG_PEDAL_VERT_DISP_HI_SCORE, LOWER_LEG_PEDAL_VERT_DISP_LO_SCORE);

    var pedal_fa_disp_score   = sliding_scale(max_pedal_fa_disp, 
            LOWER_LEG_PEDAL_FA_DISP_GOOD, LOWER_LEG_PEDAL_FA_DISP_WEAK,
            LOWER_LEG_PEDAL_FA_DISP_HI_SCORE, LOWER_LEG_PEDAL_FA_DISP_LO_SCORE);

    //var pedal_disp_mod = Math.min(pedal_vert_disp_mod, pedal_fa_disp_mod);

    // Remove all curves and datums

    remove_all_curves_and_datums();

    // Draw datums

    draw_constant_datums(LOWER_LEG_PEDAL_VERT_DISP_GOOD, LOWER_LEG_PEDAL_VERT_DISP_ADEQUATE, LOWER_LEG_PEDAL_VERT_DISP_MARGINAL, LOWER_LEG_PEDAL_VERT_DISP_WEAK);

    // Create image/curve of vertical intrusion curves

    if(c_accn_vert)   c_accn_vert.AddToGraph();
    if(c_brake_vert)  c_brake_vert.AddToGraph();
    if(c_clutch_vert) c_clutch_vert.AddToGraph();

    var curve_list = [];    // There is no OutputData object for this result, so the curve list is defined locally here

    if(c_accn_vert)   curve_list.push(c_accn_vert.id);
    if(c_brake_vert)  curve_list.push(c_brake_vert.id);
    if(c_clutch_vert) curve_list.push(c_clutch_vert.id);

    var title;
    if(c_accn_vert && c_brake_vert && c_clutch_vert) title = "Driver Vertical Pedal Intrusion";
    else                                             title = "SOME SPRING IDs MISSING/DO NOT EXIST FOR Driver Vertical Pedal Intrusion";

    var curves = new Array(3);
    curves[0] = c_accn_vert;
    curves[1] = c_brake_vert;
    curves[2] = c_clutch_vert;

    write_image(title, images_dir + "/Driver_Vertical_Pedal_Intrusion", curves, LOWER_LEG_PEDAL_VERT_DISP_WEAK);

    if(curve_list) write_curve("cur", curve_list, images_dir + "/Driver_Vertical_Pedal_Intrusion");
    if(curve_list) write_curve("csv", curve_list, images_dir + "/Driver_Vertical_Pedal_Intrusion");

    remove_all_curves_and_datums();

    // Draw datums

    draw_constant_datums(LOWER_LEG_PEDAL_FA_DISP_GOOD, LOWER_LEG_PEDAL_FA_DISP_ADEQUATE, LOWER_LEG_PEDAL_FA_DISP_MARGINAL, LOWER_LEG_PEDAL_FA_DISP_WEAK);

    // Create image/curve of fore/aft intrusion curves

    if(c_accn_fa)   c_accn_fa.AddToGraph();
    if(c_brake_fa)  c_brake_fa.AddToGraph();
    if(c_clutch_fa) c_clutch_fa.AddToGraph();

    var curve_list = [];    // There is no OutputData object for this result, so the curve list is defined locally here

    if(c_accn_fa)   curve_list.push(c_accn_fa.id);
    if(c_brake_fa)  curve_list.push(c_brake_fa.id);
    if(c_clutch_fa) curve_list.push(c_clutch_fa.id);

    var title;
    if(c_accn_fa && c_brake_fa && c_clutch_fa) title = "Driver Fore/Aft Pedal Intrusion";
    else                                       title = "SOME SPRING IDs MISSING/DO NOT EXIST FOR Driver Fore/Aft Pedal Intrusion";

    var curves = new Array(3);
    curves[0] = c_accn_fa;
    curves[1] = c_brake_fa;
    curves[2] = c_clutch_fa;

    write_image(title, images_dir + "/Driver_Fore_Aft_Pedal_Intrusion", curves, LOWER_LEG_PEDAL_FA_DISP_WEAK);

    if(curve_list) write_curve("cur", curve_list, images_dir + "/Driver_Fore_Aft_Pedal_Intrusion");
    if(curve_list) write_curve("csv", curve_list, images_dir + "/Driver_Fore_Aft_Pedal_Intrusion");

    // Write variables

    write_variable(f, "DRIVER_ACCELERATION_PEDAL_VERTICAL_DISPLACEMENT", accn_vert_disp.toFixed(3),   "Acceleration pedal vertical displacement", "String");
    write_variable(f, "DRIVER_BRAKE_PEDAL_VERTICAL_DISPLACEMENT",        brake_vert_disp.toFixed(3),  "Brake pedal vertical displacement",        "String");
    write_variable(f, "DRIVER_CLUTCH_PEDAL_VERTICAL_DISPLACEMENT",       clutch_vert_disp.toFixed(3), "Clutch pedal vertical displacement",       "String");
    write_variable(f, "DRIVER_ACCELERATION_PEDAL_FA_DISPLACEMENT",       accn_fa_disp.toFixed(3),     "Acceleration pedal fore/aft displacement", "String");
    write_variable(f, "DRIVER_BRAKE_PEDAL_FA_DISPLACEMENT",              brake_fa_disp.toFixed(3),    "Brake pedal fore/aft displacement",        "String");
    write_variable(f, "DRIVER_CLUTCH_PEDAL_FA_DISPLACEMENT",             clutch_fa_disp.toFixed(3),   "Clutch pedal fore/aft displacement",       "String");
    //write_variable(f, "DRIVER_PEDAL_DISPLACEMENT_MOD",                   pedal_disp_mod.toFixed(3),   "Pedal displacement modifier",              "String");
    write_variable(f, "DRIVER_MAX_PEDAL_VERTICAL_DISPLACEMENT",          max_pedal_vert_disp.toFixed(3), "Maximum pedal vertical displacement",      "String");
    write_variable(f, "DRIVER_MAX_PEDAL_VERTICAL_DISPLACEMENT_MOD",      pedal_vert_disp_mod.toFixed(3), "Maximum pedal vertical modifier",          "String");
    write_variable(f, "DRIVER_MAX_PEDAL_FA_DISPLACEMENT",          max_pedal_fa_disp.toFixed(3),   "Maximum pedal fore/aft displacement",      "String");
    write_variable(f, "DRIVER_MAX_PEDAL_FA_DISPLACEMENT_SCORE",    pedal_fa_disp_score.toFixed(3), "Maximum pedal fore/aft score",             "String");
}



function convert_variable_datum_times()
{
    // Convert the variable datum times from seconds to the model units

    for(var i=0; i<NECK_SHR_EXC_TIME.length;  i++) NECK_SHR_EXC_TIME[i]  *= oGlblData.time_factor;
    for(var i=0; i<NECK_TEN_EXC_TIME.length;  i++) NECK_TEN_EXC_TIME[i]  *= oGlblData.time_factor;
    for(var i=0; i<FEMUR_COM_EXC_TIME.length; i++) FEMUR_COM_EXC_TIME[i] *= oGlblData.time_factor;
}
