/// <reference path="C:/SOURCE14/this_dir/this_js_api/this.intellisense.js" />
/// <reference path="C:/SOURCE14/reporter_dir/library/templates/scripts/this_common1.js" />


// Common functions for EuroNCAP Head Impact templates

// Optionally, we can switch on a requirement for each head impact run to have
// a *.otf/d3hsp file present with normal termination, otherwise that run will
// be skipped. We switch this off by default here (for pre-2020 templates).
// The variable can be set to <true> in the master script in the REPORTER template.
if (typeof require_normal_termination == "undefined")
{
    var require_normal_termination = false;
}
// Set reporter_mode to EuroNCAP by default to support pre-2020 templates
if (typeof reporter_mode == "undefined")
{
    var reporter_mode = "EuroNCAP";
}
// Older templates (2014, 2015, 2017) don't require z-coord data in .lst files
if (typeof require_z_coord == 'undefined')
{
    var require_z_coord = true;
}

// Protocols prior to 2023 did not have a cyclist zone, so the fname_cyclist
// will be undefined, so just set it to a blank name

if (typeof fname_cyclist == 'undefined')
{
    var fname_cyclist = "";
}

// Now try to open LST files

if( (File.Exists(fname_adult) && File.IsFile(fname_adult)) ||
    (File.Exists(fname_child) && File.IsFile(fname_child)) ||
    (File.Exists(fname_cyclist) && File.IsFile(fname_cyclist)))
{
    // Read the LST files and store the data

    var oGlblData = new Object();
    var data = new Array();

    oGlblData.adult_head_node_id = -1;
    oGlblData.child_head_node_id = -1;
    oGlblData.cyclist_head_node_id = -1;
    oGlblData.max_row = -999;
    oGlblData.max_col = -999;
    oGlblData.min_col = 999;
    oGlblData.rear_reference_row = -1;

    var default_green = new Array();

    for(var i=0; i<3; i++)
    {
        var fname;

        if(i==0) fname = fname_adult;
        if(i==1) fname = fname_child;
        if(i==2) fname = fname_cyclist;

        if( !File.Exists(fname) || !File.IsFile(fname) ) continue;

        // ------------------------------------------------------------------------------------------------
        // If models were built with the *CASE option then the LST file should contain:
        //
        //    MASTER_KEY_FILE,,,,                               { this is the master keyword file that gets run }
        //    $case1:KEY_FILE,TEMPLATE,ZONE,X-COORD,Y-COORD     {   Case 1 keyword file                         }
        //    $case2:KEY_FILE,TEMPLATE,ZONE,X-COORD,Y-COORD     {   Case 2 keyword file                         }
        //    $case3:KEY_FILE,TEMPLATE,ZONE,X-COORD,Y-COORD     {   Case 3 keyword file                         }                               }
        //    $case<n>:KEY_FILE,TEMPLATE,ZONE,X-COORD,Y-COORD   {   Case n keyword file                         }
        //
        // Lines starting with '$case<n>:' will have a fake KEY_FILE, but it will be in the form
        // <directory>/A_2_0.key, so we can use them to get the row and columns for each case.
        //
        // The results for each case will be in the same directory as the MASTER_KEY_FILE.
        //
        // The actual filenames of the results files for each case will depend on the format that they
        // were written out in.  We can try both and use the one that exists on the filesystem.
        //
        //   LSTC (default)        ARUP
        //   -------------         -----
        //   case1.d3thdt          case1.<master_key_file>.thf
        //   case2.d3thdt          case2.<master_key_file>.thf
        //   case3.d3thdt          case3.<master_key_file>.thf
        //
        // So we can construct a keyword file for each case which we store on the oData object.
        // T/His will then find the correct results files:
        //
        //   case1.key     or     case1.<master_key_file>.key
        //   case2.key            case2.<master_key_file>.key
        //   case3.key            case3.<master_key_file>.key
        // ------------------------------------------------------------------------------------------------

        // We need to detect if this is a *CASE LST file, so we know what to do when we read the file
        // If it is then get the directory where the results will be

        var case_option = is_lst_case_option(fname);

        var case_res_dir     = "";
        var case_master_file = "";

        if(case_option)
        {
            case_res_dir     = get_case_res_dir(fname);
            case_master_file = get_case_master_file(fname);
        }

        // Now read the LST file and store the data

        var f_lst = new File(fname, File.READ);

        var line;

        if(i==0) oGlblData.type = "ADULT";
        if(i==1) oGlblData.type = "CHILD";
        if(i==2) oGlblData.type = "CYCLIST";

        while ( (line = f_lst.ReadLine()) != File.EOF)
        {    
            // Lines beginning with '$' are comments or 
            // $ADULT_HEAD_NODE_ID,xxx specifies the node ID for adult head
            // $CHILD_HEAD_NODE_ID,xxx specifies the node ID for child head
            // $ADULT_HEAD_NODE_ID,xxx specifies the node ID for cyclist head also
            // $UNIT_LENGTH, $UNIT_MASS and $UNIT_TIME
            // $REAR_REFERENCE_ROW
            //
            // or if they begin with '$case<id>:' they are lines for *CASE models
            // 
            // or if they begin with '$DG:' then they should be defaulted to green

            if(line[0] == '$')
            {
                var list = line.split(",");

                var name = list[0].substring(1,list[0].length);

                if(list[1] != undefined)
                {
                    if(case_option && 
                        line[1] == "c" && line[2] == "a" && line[3] == "s" && line[4] == "e")
                    {
                        // *CASE model data
                        // $case<id>:KEY_FILE,TEMPLATE,ZONE,X-COORD,Y-COORD

                        var oData = new Object();

                        // Get case id

                        var case_id = "";
                        var j = 5;
                        while(line[j] != ":")
                        {
                            case_id += line[j];
                            j++;
                        }

                        // Get dummy key file for this case

                        oData.key_file = get_case_key_file(case_res_dir, case_master_file, case_id);
                        oData.case_option = true;

                        // Results directory

                        oData.res_dir = case_res_dir;

                        // Get row and column numbers from filename: <zone>_<row>_<col>.key

                        var dirs;

                        dirs = list[0].split("/");                         // Unix
                        if(dirs.length == 1) dirs = list[0].split("\\");   // Windows

                        if(get_row_and_column_from_fname(dirs[dirs.length-1], oData))
                        {
                            oGlblData.max_row = Math.max(oGlblData.max_row, oData.row);
                            oGlblData.max_col = Math.max(oGlblData.max_col, oData.col);
                            oGlblData.min_col = Math.min(oGlblData.min_col, oData.col);

                            // HIC value to be calculated by T/His

                            oData.hic = 0.0;  

                            // Type

                            if(i==0) oData.type = "ADULT";
                            if(i==1) oData.type = "CHILD";
                            if(i==2) oData.type = "CYCLIST";
                            

                            // Store object in data array

                            data.push(oData);
                        }
                        // When file name fails to provide the necessary information about the row and column
                        // we then check the folder name to obtains the row and column details
                        else if(get_row_and_column_from_fname(dirs[dirs.length-2], oData))
                        {
                            oGlblData.max_row = Math.max(oGlblData.max_row, oData.row);
                            oGlblData.max_col = Math.max(oGlblData.max_col, oData.col);
                            oGlblData.min_col = Math.min(oGlblData.min_col, oData.col);

                            // HIC value to be calculated by T/His

                            oData.hic = 0.0;  

                            // Type

                            if(i==0) oData.type = "ADULT";
                            if(i==1) oData.type = "CHILD";
                            if(i==2) oData.type = "CYCLIST";

                            // Store object in data array

                            data.push(oData);
                        }
                        // When both file name and folder name does not satisfy the desired naming format
                        // a warning is thrown
                        else
                        {
                            WarningMessage("Warning: Neither the file name \"" , dirs[dirs.length-1] 
                                + "\" nor the folder name \"" + dirs[dirs.length-2] 
                                +"\" satisfy the format <zone>_<row>_<col>. "
                                +"This grid point will be skipped from the report." );

                            if( batch_mode=="false" )
                            {
                                var msg = "Neither the file name \"" + dirs[dirs.length-1] 
                                    + "\" nor the folder name \"" + dirs[dirs.length-2] 
                                    +"\" satisfy the format <zone>_<row>_<col>. "
                                    +"This grid point will be skipped from the report.";
                                var ans = Window.Warning("Warning", msg, Window.OK);
                            }

                        }
                    }
                    else if(line[1] == "D" && line[2] == "G" && line[3] == ":")
                    {
                        // Default green

                        var obj = new Object();

                        // Get row and column numbers from filename: <zone>_<row>_<col>.key

                        var dirs;

                        dirs = list[0].split("/");                         // Unix
                        if(dirs.length == 1) dirs = list[0].split("\\");   // Windows

                        if(get_row_and_column_from_fname(dirs[dirs.length-1], obj))
                        {
                            oGlblData.max_row = Math.max(oGlblData.max_row, obj.row);
                            oGlblData.max_col = Math.max(oGlblData.max_col, obj.col);
                            oGlblData.min_col = Math.min(oGlblData.min_col, obj.col);

                            // Store object

                            default_green.push(obj);
                        }
                        // When file name fails to provide the necessary information about the row and column
                        // we then check the folder name to obtains the row and column details
                        else if (get_row_and_column_from_fname(dirs[dirs.length-2], obj))
                        {
                            oGlblData.max_row = Math.max(oGlblData.max_row, obj.row);
                            oGlblData.max_col = Math.max(oGlblData.max_col, obj.col);
                            oGlblData.min_col = Math.min(oGlblData.min_col, obj.col);

                            // Store object

                            default_green.push(obj);
                        }
                        // When both file name and folder name does not satisfy the desired naming format
                        // a warning is thrown
                        else
                        {
                            WarningMessage("Warning: Neither the file name \"" + dirs[dirs.length-1] 
                                + "\" nor the folder name \"" + dirs[dirs.length-2] 
                                +"\" satisfy the format <zone>_<row>_<col>. "
                                +"This grid point will be skipped from the report." );

                            if( batch_mode=="false" )
                            {
                                var msg = "Neither the file name \"" + dirs[dirs.length-1] 
                                    + "\" nor the folder name \"" + dirs[dirs.length-2] 
                                    +"\" satisfy the format <zone>_<row>_<col>. "
                                    +"This grid point will be skipped from the report.";
                                var ans = Window.Warning("Warning", msg, Window.OK);
                            }

                        }
                    }
                    else
                    {
                        // Extra data
                        if(i == 0 && name.toLowerCase() == "adult_head_node_id") oGlblData.adult_head_node_id = parse_id(list[1]);
                        else if(name.toLowerCase() == "child_head_node_id") oGlblData.child_head_node_id = parse_id(list[1]);
                        //cyclist lst filse also use the Adult headform so we collect adult head node ids for cyclist too
                        else if(i == 2 && name.toLowerCase() == "adult_head_node_id") oGlblData.cyclist_head_node_id = parse_id(list[1]);
                        else if(name.toLowerCase() == "unit_length")        oGlblData.unit_length        = list[1];
                        else if(name.toLowerCase() == "unit_mass")          oGlblData.unit_mass          = list[1];
                        else if(name.toLowerCase() == "unit_time")          oGlblData.unit_time          = list[1];
                        else if(name.toLowerCase() == "rear_reference_row") oGlblData.rear_reference_row = list[1];
                    }
                }
            }

            // If it's not a comment line then it should be a comma separated list of:
            //
            //   KEY_FILE, TEMPLATE (Ignore), ZONE (Ignore), X-COORD, Y-COORD, Angle (Ignore), Z-COORD
            //
            // in that order.
            //
            // The filename should have been set by the pedestrian markup script and will be in the form
            // <directory>/A_2_0.key, where the first letter will be either 'A', 'B' or 'C' for Adult, Bicycle or Child,
            // then the row number and then the column.
            //
            // Ignore if it's a lst file for *CASE models - this is the line for the master file
            //
            else
            {
                if(!case_option)
                {
                    // Split the line

                    var list = line.split(",");

                    // Store an object with the dat  a into the <data> array. (Must have at least three columns of data)
           

                    if(list.length >= 1)
                    {
                        var oData = new Object();

                        oData.key_file = list[0];
                        oData.case_option = false;


                        // Get directory of results

                        var dirs;

                        dirs = oData.key_file.split("/");                         // Unix
                        if(dirs.length == 1) dirs = oData.key_file.split("\\");   // Windows

                        // Assemble path

                        var dir = dirs[0] + "/";
                        for(var j=1; j<(dirs.length-1); j++) dir += dirs[j] + "/";

                        oData.res_dir = dir;

                        if (File.IsDirectory(oData.res_dir) == false)
                        {
                            continue;
                        }


                        // Get row and column numbers from filename: <zone>_<row>_<col>.key

                        if(get_row_and_column_from_fname(dirs[dirs.length-1], oData))
                        {
                            oGlblData.max_row = Math.max(oGlblData.max_row, oData.row);
                            oGlblData.max_col = Math.max(oGlblData.max_col, oData.col);
                            oGlblData.min_col = Math.min(oGlblData.min_col, oData.col);

                            // HIC value to be calculated by T/His

                            oData.hic = 0.0;  

                            // Type

                            if(i==0) oData.type = "ADULT";
                            if(i==1) oData.type = "CHILD";
                            if(i==2) oData.type = "CYCLIST";

                            // Store object in data array

                            //data.push(oData);
                        }

                        // When file name fails to provide the necessary information about the row and column
                        // we then check the folder name to obtains the row and column details
                        else if (get_row_and_column_from_fname(dirs[dirs.length-2], oData))
                        {
                            oGlblData.max_row = Math.max(oGlblData.max_row, oData.row);
                            oGlblData.max_col = Math.max(oGlblData.max_col, oData.col);
                            oGlblData.min_col = Math.min(oGlblData.min_col, oData.col);

                            // HIC value to be calculated by T/His

                            oData.hic = 0.0;  

                            // Type

                            if(i==0) oData.type = "ADULT";
                            if(i==1) oData.type = "CHILD";
                            if(i==2) oData.type = "CYCLIST";

                            // Store object in data array

                            //data.push(oData);
                        }
                        // When both file name and folder name does not satisfy the desired naming format
                        // a warning is thrown
                        else
                        {
                            WarningMessage("Warning: Neither the file name \"" + dirs[dirs.length-1] 
                                + "\" nor the folder name \"" + dirs[dirs.length-2] 
                                +"\" satisfy the format <zone>_<row>_<col>. "
                                +"This grid point will be skipped from the report." );

                            if( batch_mode=="false" )
                            {
                                var msg = "Neither the file name \"" + dirs[dirs.length-1] 
                                    + "\" nor the folder name \"" + dirs[dirs.length-2] 
                                    +"\" satisfy the format <zone>_<row>_<col>. "
                                    +"This grid point will be skipped from the report.";
                                var ans = Window.Warning("Warning", msg, Window.OK);
                            }

                        }
                    }

                    // Get x coordinate from lst file            
                    if (list.length >= 4)
                    {
                        oData.x = parseFloat(list[3]);
                        if (isNaN(oData.x)) ErrorMessage("Invalid x-coordinate "+line[3]);
                    }
                    else ErrorMessage("List file missing x-coordinate data in column 4: "+fname);

                    // Get y coordinate from lst file 
                    if (list.length >= 5)
                    {
                        oData.y = parseFloat(list[4]);
                        if (isNaN(oData.y)) ErrorMessage("Invalid y-coordinate "+line[4]);
                    }
                    else ErrorMessage("List file missing y-coordinate data in column 5: "+fname);

// From 2020 template, we require z-coord for blob plots
                    if (require_z_coord == true)
                    {
                        // Get z coordinate from lst file 
                        if (list.length >= 7)
                        {
                            oData.z = parseFloat(list[6]);
                            if (isNaN(oData.z)) ErrorMessage("Invalid z-coordinate "+line[6]);
                        }
                        else ErrorMessage("List file missing z-coordinate data in column 7: "+fname);
// Abort script if we couldn't get X, Y and Z data
                        if (list.length < 7) Exit();
                    }

// Abort script if we couldn't get X and Y data
                    if (list.length < 5) Exit();

                    if (list.length >= 1)
                    {
                        data.push(oData);
                    }
                }
            }

        }

        f_lst.Close();
    }

    // Now we've read the data, for each line in the CSV file:
    //
    // 1. Read the model in <RESULTS_FILE> into T/His
    // 2. Plot the acceleration for <ADULT_HEAD_NODE_ID> or <CHILD_HEAD_NODE_ID>
    // 3. Filter the curve (C1000)
    // 4. Extract the HIC value (15ms) and store on data object
    //
    
    // make blob file name from the image_dir path 
    var path = images_dir;

    var path_split = path.split("/");                         // Unix
    if(dirs.length == 1) path.split = list[0].split("\\");   // Windows
    var blob_name = path_split[path_split.length - 1];

    // Open a csv file in the %IMAGES_DIR% to write variables to, for Reporter to pick up

    f_vars_csv = new File(images_dir + "/this_vars.csv", File.WRITE);
    f_vars_blob = new File(images_dir + "/" + blob_name + ".blob", File.WRITE);
    f_vars_blob_adult = new File(images_dir + "/" + blob_name + "_Adult.blob", File.WRITE);
    f_vars_blob_child = new File(images_dir + "/" + blob_name + "_Child.blob", File.WRITE);
    f_vars_blob_cyclist = new File(images_dir + "/" + blob_name + "_Cyclist.blob", File.WRITE);

    var model_id = 1;

    // Setup graph

    setup_this_graph();

    // Get key file root name and path ( key file path eg "C:/data/analysis/run.key")

    var invalid_points = []; 
    var k=0;
    for(var i=0; i<data.length; i++)
    {
        //find keyword files
        var root_name = data[i].key_file.substring(Math.max(data[i].key_file.lastIndexOf("/"), data[i].key_file.lastIndexOf("\\"))+1, data[i].key_file.lastIndexOf(".")); // eg "run"
        var termination_msg = "Normal";
        var OTF_absolute_filename = find_lsdyna_files(data[i].res_dir, root_name, "OTF");
        var point = "HIC_" + data[i].row + "_" + data[i].col;
        
        if (!data[i].case_option && find_lsdyna_files(data[i].res_dir, root_name, "KEY") == null)
        {
            WarningMessage(data[i].key_file + " could not be found.");
            invalid_points.push(point);
            continue;
        }
        var res_file = find_lsdyna_files(data[i].res_dir, root_name, "THF");
        if (res_file == null)
        {
            res_file = find_lsdyna_files(data[i].res_dir, root_name, "BINOUT");
            if (res_file == null)
            {
                WarningMessage("Results files (*.thf, *.d3thdt, d3thdt, *.binout* or binout*) could not be found in: " + data[i].res_dir);
                invalid_points.push(point);
                continue;
            }
        }
        if (require_normal_termination == true)
        {
            if (OTF_absolute_filename == null)
            {
                WarningMessage("Output file (*.otf, *.d3hsp or d3hsp) could not be found in: " + data[i].res_dir);
                invalid_points.push(point);
                continue;
            }
            if (TerminationStatus(OTF_absolute_filename) != termination_msg)
            {
                WarningMessage("otf / d3hsp file is incomplete, possible error termination in: " + data[i].res_dir);
                invalid_points.push(point);
                continue;
            }
        }

        // Create an acceleration magnitude curve
        // The previous code didnt give out variables for the child and cyclist head node ids
        // because the value of k was changing after the first loop so added a condition to check
        // any changes in data.type. 
        if(i != 0 )
        {
            if(data[i-1].type != data[i].type)
                {
                    k = i;
                }
        }
        if (i == 0)
        {
            f_vars_csv.Writeln("FIRST_LST_FILENAME, " + data[i].key_file);
        }
        var node_id;

        if(k == i && data[i].type == "ADULT") 
        {
            node_id = oGlblData.adult_head_node_id;
            f_vars_csv.Writeln("ADULT_HEAD_NODE_ID, " + node_id); 
        }

        if(k == i && data[i].type == "CHILD") 
        {
            node_id = oGlblData.child_head_node_id;
            f_vars_csv.Writeln("CHILD_HEAD_NODE_ID, " + node_id);
        }
        
        if(k == i && data[i].type == "CYCLIST") 
        {
            node_id = oGlblData.cyclist_head_node_id;
            f_vars_csv.Writeln("CYCLIST_HEAD_NODE_ID, " + node_id);
        }
        DialogueInput("/MODEL READ", res_file);

        // Make sure any units set in T/HIS (e.g. via preferences) are unset

        unset_this_units(model_id);

        // check if the model time units and input time units are consistent
        // If the units are not defined/valid then we cannot go any further.  

        if(!check_unit_values(model_id))
        {
            write_unit_vars(f_vars_csv, false);
            Message("Units check unsuccessful.");
            Exit();
        }

        // Write the unit data to the variables file
        if (k == 1)
        {
        write_unit_vars(f_vars_csv, true);
        }
        // Calculate constants now we have unit data and store on <oGlblData> object

        calculate_unit_constants();

        // Read in X, Y and Z acceleration.  Filter with C1000 and then vector combine.

        var c;

        var c_acc = read_xyz_accelerations(model_id, node_id);

        if(c_acc.ax && c_acc.ay && c_acc.az)
        {
            // Convert to g

            var c_g = convert_xyz_acc_to_g(c_acc);

            // C1000 filter (convert to seconds first)

            var c_gx_c1000 = convert_to_s_and_filter(c_g.gx, Operate.C1000, REG_DT);
            var c_gy_c1000 = convert_to_s_and_filter(c_g.gy, Operate.C1000, REG_DT);
            var c_gz_c1000 = convert_to_s_and_filter(c_g.gz, Operate.C1000, REG_DT);

            // Vector combine

            remove_all_curves();

            c = Operate.Vec(c_gx_c1000, c_gy_c1000, c_gz_c1000);
        }


        if(!c)
        {
            // No curve for some reason, so set HIC to -1 to indicate we could not calculate it

            data[i].hic = -1;
        }
        else
        {      
            // Convert from seconds back to model time

            var c_temp = Operate.Mux(c, oGlblData.time_factor);

            c.RemoveFromGraph(1);

            c = c_temp;

            // HIC15 calculation

            data[i].hic = Operate.Hic(c, HIC_WINDOW * oGlblData.time_factor, 1.0);

            // Change name of curve and update curve style colouring it based on the HIC value 

            var curve_label = "Grid Point " + data[i].row + "," + data[i].col;
            set_labels(c, curve_label, "Time (" + oGlblData.unit_time + ")", "Acceleration (g)");
            DialogueInput("/CUR LA #" + c.id + " " + curve_label); 

            var colour;

            if(data[i].hic < GREEN_LIMIT)  colour = Colour.GREEN;
            else if(data[i].hic < YELLOW_LIMIT) colour = Colour.YELLOW;
            else if(data[i].hic < ORANGE_LIMIT) colour = Colour.ORANGE;
            else if(data[i].hic < BROWN_LIMIT)  colour = Colour.OLIVE;
            else                                colour = Colour.RED;

            c.width  = LineWidth.BOLD;
            c.style  = LineStyle.SOLID;
            c.colour = colour;

            // Create image and curve files

            // No title needed as title already on reporter page
            write_image("", images_dir + "/HIC_curve_" + data[i].row + "_" + data[i].col)
            var type = data[i].type;
            var letter;
            if (type == "ADULT")
            {
                letter = "A_";
            }
            else if (type == "CHILD")
            {
                letter = "C_";
            }
            // The Letter for Cyclist in EuroNCAP 2023 Regulations is B
            else if (reporter_mode == "EuroNCAP" && type == "CYCLIST")
            {
                letter = "B_";
            
            }
            // The Letter for Cyclist in CNCAP 2021 Regulations is A          
            else if (reporter_mode == "C-NCAP" && type == "CYCLIST")
            {
                letter = "A_";
            }  
            write_curve("cur", c.id, images_dir + "/HIC_curve_" + letter + data[i].row + "_" + data[i].col)
            write_curve("csv", c.id, images_dir + "/HIC_curve_" + letter + data[i].row + "_" + data[i].col)

            // Blank the filtered curve, ready for the next results

            c.RemoveFromGraph(1);     
        }
        var current_model = Model.GetFromID(Model.HighestID());
        current_model.Delete();
        model_id++;
    }
    // We need to add a comment on top of .blob file which locates the .cur file from the keyword file
    // This would be used in primer HIC Area Tool Calculator later on
    //something like $cur_file ..\ENCAP_Images\HIC_$$NAME$.cur
     f_vars_blob.Writeln("$cur_file," + images_dir + "/" + "HIC_curve_$$NAME$.cur")
     f_vars_blob.Writeln("$abs_cur_file")
    
    // Now we've calculated the HIC values we need to write them to the csv and blob file for Reporter to pick up

    var missing_data = false;

    for(var i=0; i<data.length; i++)
    {
        // Write a HIC_<row>_<col> variable to store the HIC value
        var invalid = false;
        var name  = "HIC_" + data[i].row + "_" + data[i].col; 
        for (var j = 0; j < invalid_points.length; j++)
        {
            if (name == invalid_points[j])
            {
                invalid = true;
            }
            
        }
        if (invalid == true)
        {
            continue;
        }

        var value = data[i].hic;
        var desc  = "T/His calculated HIC value for row " + data[i].row + " column " + data[i].col;
        var type  = "Number";

        f_vars_csv.Writeln(name + "," + value + "," + desc + "," + type);

        // Write the .blob file "data"_<X>_<Y>_<Z>_HIC

        var coords = {x:0, y:0, z:0};
        for (var c in coords)
        {
            if (data[i][c] != undefined) coords[c] = data[i][c];
        }
        var HIC_value = data[i].hic; 
        var grid_point = data[i].row + "_" + data[i].col;
        var type = data[i].type;
        if (type == "ADULT")
        {
            letter = "A_";
        }
        else if (type == "CHILD")
        {
            letter = "C_";
        }
		// The Letter for Cyclist in EuroNCAP 2023 Regulations is B
        else if (reporter_mode == "EuroNCAP" && type == "CYCLIST")
        {
            letter = "B_";

        }       
        // The Letter for Cyclist in CNCAP 2021 Regulations is A       
        else if (reporter_mode == "C-NCAP" && type == "CYCLIST")
        {
            letter = "A_";
        }  

        // Write all points to the main blob file
        // and the Adult, Child or Cyclist points to the specific file for that type
        f_vars_blob.Writeln("data " + coords.x.toFixed(3) + " " + coords.y.toFixed(3) + " " + coords.z.toFixed(3) + " " + HIC_value.toFixed(3) + " " + letter + grid_point + " " + data[i].key_file);
        
        let f_vars_blob_ped_type = "";

        if (type == "ADULT") 
           f_vars_blob_ped_type = f_vars_blob_adult; 
        else if (type == "CHILD")
            f_vars_blob_ped_type = f_vars_blob_child;
        else if (type == "CYCLIST")
            f_vars_blob_ped_type = f_vars_blob_cyclist;

        if (f_vars_blob_ped_type != "")
        {
            f_vars_blob_ped_type.Writeln("data " + coords.x.toFixed(3) + " " + coords.y.toFixed(3) + " " + coords.z.toFixed(3) + " " + HIC_value.toFixed(3) + " " + letter + grid_point + " " + data[i].key_file);
        }

        // Increment the correct counter, depending on the HIC value

        if(value == -1)          { missing_data = true;     data[i].prediction = "GREY";   } // Grey (-1 indicates we could not find results)
        else if(value < GREEN_LIMIT)  { oGlblData.num_green++;   data[i].prediction = "GREEN";  } // Green
        else if(value < YELLOW_LIMIT) { oGlblData.num_yellow++;  data[i].prediction = "YELLOW"; } // Yellow
        else if(value < ORANGE_LIMIT) { oGlblData.num_orange++;  data[i].prediction = "ORANGE"; } // Orange
        else if(value < BROWN_LIMIT)  { oGlblData.num_brown++;   data[i].prediction = "BROWN";  } // Brown
        else                          { oGlblData.num_red++;     data[i].prediction = "RED";    } // Red

        // Write a HIC_<row>_<col>_file variable to store the image file name

        var name  = "HIC_" + data[i].row + "_" + data[i].col + "_file";
        var value = images_dir + "/HIC_curve_" + data[i].row + "_" + data[i].col + "";
        var desc  = "Image file name row " + data[i].row + " column " + data[i].col;
        var type  = "String";

        f_vars_csv.Writeln(letter + name + "," + value + "," + desc + "," + type);
    }

    // Write some HIC_<row>_<col>_DEFAULT variables, setting their values to a blank string. 
    // These can be changed by the user on the second page using the button to set them to 'RED' or 'GREEN'
    // We'll loop over the maximum number of rows and columns
    //
    // If we read any default values from the LST file, then set these here.

    var num_rows = oGlblData.max_row + 1;
    var num_max_cols = oGlblData.max_col;
    var num_min_cols = oGlblData.min_col;

    for(var r=0; r<num_rows; r++)
    {
        // Zero column
        var name  = "HIC_" + r + "_0_DEFAULT";
        var value = "";
        var desc  = "Default colour for row " + r + " column 0 [RED or GREEN]";
        var type  = "String";

        for(var i=0; i<default_green.length; i++)
        {
            if(default_green[i].row == r &&
                default_green[i].col == 0) value = "GREEN";
        }

        f_vars_csv.Writeln(name + "," + value + "," + desc + "," + type);

        // -ve columns
        for(var c=-1; c>num_min_cols-1; c--)
        {
            var name  = "HIC_" + r + "_" + c + "_DEFAULT";
            var value = "";
            var desc  = "Default colour for row " + r + " column " + c + " [RED or GREEN]";
            var type  = "String";

            for(var i=0; i<default_green.length; i++)
            {
                if(default_green[i].row == r &&
                    default_green[i].col == c) value = "GREEN";
            }

            f_vars_csv.Writeln(name + "," + value + "," + desc + "," + type);
        }

        // +ve columns
        for(var c=1; c<num_max_cols+1; c++)
        {
            var name  = "HIC_" + r + "_" + c + "_DEFAULT";
            var value = "";
            var desc  = "Default colour for row " + r + " column " + c + " [RED or GREEN]";
            var type  = "String";

            for(var i=0; i<default_green.length; i++)
            {
                if(default_green[i].row == r &&
                    default_green[i].col == c) value = "GREEN";
            }

            f_vars_csv.Writeln(name + "," + value + "," + desc + "," + type);
        }
    }


    // write some variables to indicate whether we have missing data or not.

    write_missing_hic_result_variables(f_vars_csv, missing_data);


    // Global data - calculate values for use in Reporter

    // Calculate final score

    // Write calculated values to variables file so Reporter can access them

    f_vars_csv.Writeln("MAX_ROW" + "," + oGlblData.max_row + ",Maximum rows,Number");
    f_vars_csv.Writeln("MAX_COL" + "," + oGlblData.max_col + ",Maximum cols,Number");
    f_vars_csv.Writeln("MIN_COL" + "," + oGlblData.min_col + ",Minimum cols,Number");
    f_vars_csv.Writeln("BLOB_FILE" + "," + blob_name);
    f_vars_csv.Writeln("BLOB_FILE_CHILD" + "," + blob_name + "_Child");
    f_vars_csv.Writeln("BLOB_FILE_CYCLIST" + "," + blob_name + "_Cyclist");

    if(oGlblData.rear_reference_row == -1) oGlblData.rear_reference_row = oGlblData.max_row;

    f_vars_csv.Writeln("REAR_REFERENCE_LINE_ROW" + "," + oGlblData.rear_reference_row + ",Rear reference line row [Set between 0 and " + oGlblData.max_row + "],Number");

    f_vars_csv.Close();


}
else Message("Reached the end of the script.");


function write_missing_hic_result_variables(f, missing)
{
    if(missing)
    {
        var name  = "MISSING_RESULTS";
        var value = 1;
        var desc  = "Could not find some results";
        var type  = "Number";

        f.Writeln(name + "," + value + "," + desc + "," + type);

        var name  = "MISSING_RESULTS_MESSAGE";
        var value = "Results were missing for some grid points. These have been coloured grey.";
        var desc  = "Could not find some results";
        var type  = "String";

        f.Writeln(name + "," + value + "," + desc + "," + type);
    }
    else
    {
        var name  = "MISSING_RESULTS_MESSAGE";
        var value = "";
        var desc  = "Blank as no results were missing";
        var type  = "String";

        f.Writeln(name + "," + value + "," + desc + "," + type);
    }
}

