/// <reference path="C:/SOURCE14/this_dir/this_js_api/this.intellisense.js" />
/// <reference path="C:/SOURCE14/reporter_dir/library/templates/scripts/this_common1.js" />
/// <reference path="C:/SOURCE14/reporter_dir/library/templates/scripts/this_common_EuroNCAP.js" />

/*
REQUIRED INPUTS
head_pid head_excursion -> head_exceedance_zone
red_zone <- output from AE-MDB OR Side Pole respectively


*/

/*NOTE: the calculation of the IR-TRACC length is not required as the length is fixed at 118.52235 mm [WSID v4.0] (take 118.5 for simplicity of hand-calc checking (~0.1% error))
        the original calculataion assumed the length was the same as the spring length but this is not the case (it is actually a bit shorter)
        from DYNAMORE on 15 July 2022:

        "The springs initial length is not the 2D IR_TRACC initial length(Reference length at t0):
            The 2D IR_TRACC inital length(Reference length at t0) is measured from Ball Joint IRTRACC to Pot axis in y-direchtion.
            These springs in dummy model are just used to measure the 2D IR_TRACC deflection (compression or extension).
            Both nodes of the spring element are fixed on the 2D IR_TRACC, the initial length will not effect the deflection output."

ADDITIONAL NODE - the rib compression calculations use the following equation:
            c_disp: rib displacement in m
            c_rot: rib rotation in radians
            
            abs_length_mm = irtracc_length_abs_length_mm + c_disp * oGlblData.mm_factor (to convert rib displacement to rib length in mm)
            
            filter:
            abs_length_mm_180 = C180 Filter of abs_length_mm
            c_rot_180         = C180 Filter of c_rot
            
            lat_length_mm = Cos(c_rot_c180) * abs_length_mm_180
        
            //convert back to displacment (subtract initial rib length)
            lat_disp_elong_mm = lat_length_mm - irtracc_length_abs_length_mm

            //multiply by -1  to make compression (negative displacement) positive
            lat_disp_mm = lat_disp_elong_mm * -1
            ^FINAL DISP^

            This only applies to WSID simulation models which do not have a +/- PI/2 initial offset in c_rot angle for right/left ribs
            The TB 021 Calculation uses (Sin(theta)*rib_length +/- rib_length but this only applies for hardware dummies with the PI/2 offsets for left and right ribs)
            */

//this value should be in mm but model may not be so x by factor to get to model scale (it is then added to the curve of change in length and divided by oGlblData.mm_factor to convert to mm)
var irtracc_length_abs_length_mm = 118.5; //HARD-CODED value valid for WSID v4.0

// Constants used through out the script

var REG_DT = 0.00001; // dt for regularising curves (in seconds)
var HIC_WINDOW = 0.015; // HIC time Window (in seconds)
var TMS_PERIOD = 0.003; // 3ms period (in seconds)

// Head Excursion table

//head score

//WITH COUNTERMEASURE (CM)
var HEAD_EXCURSION = {};

HEAD_EXCURSION.COUNTERMEASURE = {};

HEAD_EXCURSION.COUNTERMEASURE.HEAD_SCORE_ZONES = {
    CAPPING: 0.0,
    RED: 0.0, //RED ZONE LESS THAN (OR EQUAL TO) 125mm
    BROWN: 2.0, //RED ZONE GREATER THAN 125mm
    ORANGE: 3.0,
    YELLOW: 4.0,
    GREEN: 4.0,
};

HEAD_EXCURSION.COUNTERMEASURE.NECK_SCORE_ZONES = {
    CAPPING: 0.0,
    RED: 4.0, //RED ZONE LESS THAN (OR EQUAL TO) 125mm
    BROWN: 4.0, //RED ZONE GREATER THAN 125mm
    ORANGE: 3.0,
    YELLOW: 4.0,
    GREEN: 4.0,
};

HEAD_EXCURSION.COUNTERMEASURE.CHEST_SCORE_ZONES = {
    CAPPING: 0.0,
    RED: 0.0, //RED ZONE LESS THAN (OR EQUAL TO) 125mm
    BROWN: 0.0, //RED ZONE GREATER THAN 125mm
    ORANGE: 3.0,
    YELLOW: 4.0,
    GREEN: 4.0,
};

//WITHOUT COUNTERMEASURE (NOCM)

HEAD_EXCURSION.NO_COUNTERMEASURE = {};

HEAD_EXCURSION.NO_COUNTERMEASURE.HEAD_SCORE_ZONES = {
    CAPPING: 0.0,
    RED: 0.0,
    ORANGE: 1.0,
    YELLOW: 2.0,
    GREEN: 4.0,
};

HEAD_EXCURSION.NO_COUNTERMEASURE.NECK_SCORE_ZONES = {
    CAPPING: 0.0,
    RED: 1.0,
    ORANGE: 1.0,
    YELLOW: 2.0,
    GREEN: 4.0,
};

HEAD_EXCURSION.NO_COUNTERMEASURE.CHEST_SCORE_ZONES = {
    CAPPING: 0.0,
    RED: 0.0,
    ORANGE: 1.0,
    YELLOW: 2.0,
    GREEN: 4.0,
};

// Head assessment constants

var HIC_HI_LIMIT = 500.0;
var HIC_LO_LIMIT = 700.0;

var TMS_HI_LIMIT = 80.0; // 3 milisecond acceleration
var TMS_LO_LIMIT = 72.0; // 3 milisecond acceleration

// Upper Neck

var NECK_UPPER_TENSION_HI_LIMIT = 3.74; //kN
var NECK_UPPER_TENSION_LO_LIMIT = 3.74; //kN

var NECK_UPPER_LATERAL_FLEXION_HI_LIMIT = 162; //Nm
var NECK_UPPER_LATERAL_FLEXION_LO_LIMIT = 248; //Nm

var NECK_UPPER_NEG_EXTENSION_HI_LIMIT = 50; //Nm
var NECK_UPPER_NEG_EXTENSION_LO_LIMIT = 50; //Nm

// Lower Neck

var NECK_LOWER_TENSION_HI_LIMIT = 3.74; //kN
var NECK_LOWER_TENSION_LO_LIMIT = 3.74; //kN

var NECK_LOWER_LATERAL_FLEXION_HI_LIMIT = 162; //Nm
var NECK_LOWER_LATERAL_FLEXION_LO_LIMIT = 248; //Nm

var NECK_LOWER_NEG_EXTENSION_HI_LIMIT = 100; //Nm Note: only monitored in 2020-2022 (not used for scoring)
var NECK_LOWER_NEG_EXTENSION_LO_LIMIT = 100; //Nm Note: only monitored in 2020-2022 (not used for scoring)

//Chest assessment constants

var CHEST_COMPRESSION_HI_LIMIT = 28; // The compression limit for the chest in mm
var CHEST_COMPRESSION_LO_LIMIT = 50; // The compression limit for the chest in mm

var CHEST_COMPRESSION_CAPPING_LIMIT = 50; // Capping limit is 50mm (same as lower limit)

// Abdomen assessment constants

var ABDOMEN_COMPRESSION_HI_LIMIT = 47; // The abdominal compression limit in mm
var ABDOMEN_COMPRESSION_LO_LIMIT = 65; // The abdominal compression limit in mm
var ABDOMEN_COMPRESSION_CAPPING_LIMIT = 65; // Capping limit is 65mm (same as lower limit)

//Pelvis assessment constants

//performance limits
var PELVIS_PUBIC_SYMPHYSIS = 2.8; //kN
var PELVIS_LUMBAR_FY = 2; //kN
var PELVIS_LUMBAR_FZ = 3.5; //kN
var PELVIS_LUMBAR_MX = 120; //Nm

var PELVIS_EXCEEDANCE_MODIFIER = -4.0;

if (File.Exists(fname_csv) && File.IsFile(fname_csv)) {
    // Do the common setup of assigning values to the oGlblData object,
    // setting the T/HIS graph up and reading in the model.

    var oGlblData = new Object();

    var f_vars = do_common_single_analysis_setup(oGlblData);

    // For each body region, do the calculation, writing results to the variables file.

    if (countermeasure) {
        var countermeasure_string = "with counter measure.";
        var HEAD_HI_SCORE =
            HEAD_EXCURSION.COUNTERMEASURE.HEAD_SCORE_ZONES[
                head_exceedance_zone
            ];
        var NECK_HI_SCORE =
            HEAD_EXCURSION.COUNTERMEASURE.NECK_SCORE_ZONES[
                head_exceedance_zone
            ];
        var CHEST_HI_SCORE =
            HEAD_EXCURSION.COUNTERMEASURE.CHEST_SCORE_ZONES[
                head_exceedance_zone
            ];
    } else {
        var countermeasure_string = "with no counter measure.";
        var HEAD_HI_SCORE =
            HEAD_EXCURSION.NO_COUNTERMEASURE.HEAD_SCORE_ZONES[
                head_exceedance_zone
            ];
        var NECK_HI_SCORE =
            HEAD_EXCURSION.NO_COUNTERMEASURE.NECK_SCORE_ZONES[
                head_exceedance_zone
            ];
        var CHEST_HI_SCORE =
            HEAD_EXCURSION.NO_COUNTERMEASURE.CHEST_SCORE_ZONES[
                head_exceedance_zone
            ];
    }

    var HEAD_LO_SCORE = 0;
    var NECK_LO_SCORE = 0;
    var CHEST_LO_SCORE = 0;

    //write out HI_SCORE for the exceedance zone
    write_variable(
        f_vars,
        "HEAD_HI_SCORE",
        HEAD_HI_SCORE,
        "Max HEAD SCORE for head excursion in " +
            head_exceedance_zone +
            " zone " +
            countermeasure_string,
        "String"
    );
    write_variable(
        f_vars,
        "NECK_HI_SCORE",
        NECK_HI_SCORE,
        "Max NECK SCORE for head excursion in " +
            head_exceedance_zone +
            " zone " +
            countermeasure_string,
        "String"
    );
    write_variable(
        f_vars,
        "CHEST_HI_SCORE",
        CHEST_HI_SCORE,
        "Max CHEST SCORE for head excursion in " +
            head_exceedance_zone +
            " zone " +
            countermeasure_string,
        "String"
    );

    do_head_rating_calc(f_vars, HEAD_HI_SCORE, HEAD_LO_SCORE);

    do_upper_neck_rating_calc(f_vars, NECK_HI_SCORE, NECK_LO_SCORE);
    do_lower_neck_rating_calc(f_vars, NECK_HI_SCORE, NECK_LO_SCORE);

    do_chest_rating_calc(f_vars, CHEST_HI_SCORE, CHEST_LO_SCORE);

    do_abdomen_rating_calc(f_vars, CHEST_HI_SCORE, CHEST_LO_SCORE);

    do_pelvis_rating_calc(f_vars);

    f_vars.Close();
}

function get_head_exceedance_zone() {
    //this function extracts max displacement of the head relative to the threshold
    //colour bands in the passenger zone as outlined in EuroNCAP
}

function do_upper_neck_rating_calc(f, NECK_HI_SCORE, NECK_LO_SCORE) {
    /*output
    upper neck tension Fz
    upper neck flexion MxOC
    Upper neck extension negative MyOC

    existing neck rating calc extracts Fz and My but not Mx for upper neck. For lower neck it does not obviously extract anything...
    
    calc for neck criteria defined in section 3.2 of document linked below
    https://cdn.euroncap.com/media/58245/tb-021-data-acquisition-and-injury-calculation-v31.pdf

    MxOC => neck lateral flexion (section 3.2.2)
    MyOC => neck extension (section 3.2.1)

    MxOC = Mx + Fy*d
    MyOC = My - Fx*d

    */

    //HARD-CODED
    var d = 0.0195; // (WorldSID)
    //upper neck forces

    var beam = oGlblData.neck_upper_beam;

    var output_data = {};

    /*
    The lumbar load cell is a six-channel load cell. This means that the forces and the
    moments in each direction of axis can be evaluated in the hardware.

    see https://www.dummymodels.com/models/worldsid/documents/manuals/wsid50_pdb_v7-1_manual_v0-0.pdf pg.17
    */
    output_data.fz = new OutputData(
        "Upper Neck Tension",
        images_dir + "/upper_neck_tension"
    );
    output_data.mx = new OutputData(
        "Upper Neck Lateral Flexion",
        images_dir + "/upper_neck_lateral_flexion"
    );
    output_data.my = new OutputData(
        "Upper Neck Negative Extension",
        images_dir + "/upper_neck_extension"
    );

    if (beam != undefined) {
        // Should check data components are available for this beam

        // Read in Neck data force

        var un_fx = read_data("BEAM BASIC", 0, beam, "NX"); // Neck tension
        var un_fy = read_data("BEAM BASIC", 0, beam, "NY"); // Neck tension
        var un_fz = read_data("BEAM BASIC", 0, beam, "NZ"); // Neck tension
        var un_mx = read_data("BEAM BASIC", 0, beam, "MX"); // Neck Lateral Flexion
        var un_my = read_data("BEAM BASIC", 0, beam, "MY"); // Neck Negative Extension

        // Check that the curves read in - i.e. the beam id and component are valid
        // Create a blank image to let the user know.

        if (!un_fx || !un_fy || !un_fz || !un_mx || !un_my) {
            write_blank_images(output_data, "NO DATA FOR BEAM " + beam);
        } else {
            // C600 filter (convert to seconds) and convert force into kN
            // and torsion in to Nm

            //kN
            un_fx = FilterC600_kN(un_fx);
            un_fy = FilterC600_kN(un_fy);
            un_fz = FilterC600_kN(un_fz);
            //Nm
            un_mx = FilterC600_Nm(un_mx);
            un_my = FilterC600_Nm(un_my);

            // Remove all curves and datums

            remove_all_curves_and_datums();

            // Upper Neck Tension
            // Convert from seconds back to model time
            var un_fz = Operate.Mux(un_fz, oGlblData.time_factor);
            // Set label and style
            set_labels(
                un_fz,
                "Upper Neck Tension",
                "Time (" + oGlblData.unit_time + ")",
                "Tension (kN)"
            );
            set_line_style(un_fz, Colour.BLACK);

            // Draw datums
            // traffic light bands, green is full marks amber is sliding scale and red is 0
            draw_constant_datums(
                NECK_UPPER_TENSION_HI_LIMIT,
                NECK_UPPER_TENSION_LO_LIMIT,
                null,
                NECK_UPPER_TENSION_LO_LIMIT
            );
            write_image(
                output_data.fz.title,
                output_data.fz.fname,
                un_fz,
                1.1 * NECK_UPPER_TENSION_LO_LIMIT
            ); //set maximum y-axis value to at least 1.1 times the lower performance limit

            // output_data.fz.curveList.push(un_fz.id);
            // write_curve("cur", output_data.fz.curveList, output_data.fz.fname);
            // write_curve("csv", output_data.fz.curveList, output_data.fz.fname);

            //Upper Neck Lateral Flexion
            // Convert from seconds back to model time
            var un_fy = Operate.Mux(un_fy, oGlblData.time_factor);
            var un_mx = Operate.Mux(un_mx, oGlblData.time_factor);

            //MxOC = Mx + Fy*d
            var un_fy_d = Operate.Mul(un_fy, d * 1000); //note d in 'm' and un_fy in 'kN' so multiply by 1000 to get Nm for add operation on next line
            remove_all_curves_and_datums();
            var un_mx = Operate.Add(un_mx, un_fy_d);

            // Set label and style
            set_labels(
                un_mx,
                "Upper Neck Lateral Flexion",
                "Time (" + oGlblData.unit_time + ")",
                "Flexion (Nm)"
            );
            set_line_style(un_mx, Colour.BLACK);

            // Draw datums
            // traffic light bands, green is full marks amber is sliding scale and red is 0
            draw_constant_datums(
                NECK_UPPER_LATERAL_FLEXION_HI_LIMIT,
                NECK_UPPER_LATERAL_FLEXION_LO_LIMIT,
                null,
                NECK_UPPER_LATERAL_FLEXION_LO_LIMIT,
                true
            );
            write_image(
                output_data.mx.title,
                output_data.mx.fname,
                un_mx,
                1.1 * NECK_UPPER_LATERAL_FLEXION_LO_LIMIT,
                -1.1 * NECK_UPPER_LATERAL_FLEXION_LO_LIMIT
            ); //set maximum and minimum y-axis values to at least 1.1 times the lower performance limit

            // output_data.mx.curveList.push(un_mx.id);
            // write_curve("cur", output_data.mx.curveList, output_data.mx.fname);
            // write_curve("csv", output_data.mx.curveList, output_data.mx.fname);

            //Upper Neck Negative Extension
            // Convert from seconds back to model time
            var un_fx = Operate.Mux(un_fx, oGlblData.time_factor);
            var un_my = Operate.Mux(un_my, oGlblData.time_factor);

            //MyOC = My - Fx*d
            var un_fx_d = Operate.Mul(un_fx, -d * 1000); //note d in 'm' and un_fx in 'kN' so multiply by 1000 to get Nm for add operation on next line
            remove_all_curves_and_datums();
            var un_my = Operate.Add(un_my, un_fx_d); //node multiplied by -d*1000 in line above so addition is actually subtracting Fx*d

            // Set label and style
            set_labels(
                un_my,
                "Upper Neck Negative Extension",
                "Time (" + oGlblData.unit_time + ")",
                "Extension (Nm)"
            ); //TODO is Extension correct?
            set_line_style(un_my, Colour.BLACK);

            // Draw datums
            // traffic light bands, green is full marks amber is sliding scale and red is 0
            draw_constant_datums(
                NECK_UPPER_NEG_EXTENSION_HI_LIMIT,
                NECK_UPPER_NEG_EXTENSION_LO_LIMIT,
                null,
                NECK_UPPER_NEG_EXTENSION_LO_LIMIT,
                true
            ); //make symmetric as not obvious if +ve or -ve governs
            write_image(
                output_data.my.title,
                output_data.my.fname,
                un_my,
                1.1 * NECK_UPPER_NEG_EXTENSION_LO_LIMIT,
                -1.1 * NECK_UPPER_NEG_EXTENSION_LO_LIMIT
            ); //set maximum and minimum y-axis values to at least 1.1 times the lower performance limit

            // output_data.my.curveList.push(un_my.id);
            // write_curve("cur", output_data.my, output_data.my.fname);
            // write_curve("csv", output_data.my, output_data.my.fname);

            remove_all_curves_and_datums();

            // UPPER NECK OUTPUT VARIABLES

            // Find the max value of each curve

            var max_tension = un_fz.ymax;
            var max_flexion = Math.max(un_mx.ymax, -un_mx.ymin); //take the absolute peak value as neck could flex either way depending on which side the driver is on (LHD/RHD)
            var max_extension = Math.abs(un_my.ymin); //take the absolute peak negative value (this corresponds to moment of head rotating forward in the upper neck about OC)

            // Write variables to csv file for Reporter

            var tension_score = sliding_scale(
                max_tension,
                NECK_UPPER_TENSION_HI_LIMIT,
                NECK_UPPER_TENSION_LO_LIMIT,
                NECK_HI_SCORE,
                NECK_LO_SCORE
            );

            write_variable(
                f,
                "NECK_UPPER_TENSION",
                max_tension.toFixed(3),
                "Neck upper tension force in kN",
                "String"
            );
            write_variable(
                f,
                "NECK_UPPER_TENSION_SCORE",
                tension_score.toFixed(3),
                "Neck upper tension score",
                "String"
            );

            var flexion_score = sliding_scale(
                max_flexion,
                NECK_UPPER_LATERAL_FLEXION_HI_LIMIT,
                NECK_UPPER_LATERAL_FLEXION_LO_LIMIT,
                NECK_HI_SCORE,
                NECK_LO_SCORE
            );

            write_variable(
                f,
                "NECK_UPPER_FLEXION",
                max_flexion.toFixed(3),
                "Neck upper flexion moment in Nm",
                "String"
            );
            write_variable(
                f,
                "NECK_UPPER_FLEXION_SCORE",
                flexion_score.toFixed(3),
                "Neck upper flexion score",
                "String"
            );

            var extension_score = sliding_scale(
                max_extension,
                NECK_UPPER_NEG_EXTENSION_HI_LIMIT,
                NECK_UPPER_NEG_EXTENSION_LO_LIMIT,
                NECK_HI_SCORE,
                NECK_LO_SCORE
            );

            write_variable(
                f,
                "NECK_UPPER_EXTENSION",
                max_extension.toFixed(3),
                "Neck upper extension moment in Nm",
                "String"
            );
            write_variable(
                f,
                "NECK_UPPER_EXTENSION_SCORE",
                extension_score.toFixed(3),
                "Neck upper extension score",
                "String"
            );

            //upper neck worst score
            var un_score = Math.min(
                tension_score,
                flexion_score,
                extension_score
            );

            write_variable(
                f,
                "NECK_UPPER_SCORE",
                un_score.toFixed(3),
                "Neck upper worst score",
                "String"
            );
        }
    } else {
        // No beam defined - variables should be set to blank by first script in template.
        // Create a blank image to let the user know.

        write_blank_images(output_data, "NO BEAM ID DEFINED FOR");
    }
    remove_all_curves_and_datums();
}

function do_lower_neck_rating_calc(f, NECK_HI_SCORE, NECK_LO_SCORE) {
    /*output
    lower neck tension Fz
    lower neck lateral flexion Mx (base of neck)
    *lower neck extension neg. My (base of neck)    *monitoring for 2020 only
    */

    //HARD-CODED
    var Dz = 0.0145; //m (WorldSID)
    //lower neck forces

    var beam = oGlblData.neck_lower_beam;

    var output_data = {};

    /*
    The lumbar load cell is a six-channel load cell. This means that the forces and the
    moments in each direction of axis can be evaluated in the hardware.

    see https://www.dummymodels.com/models/worldsid/documents/manuals/wsid50_pdb_v7-1_manual_v0-0.pdf pg.17
    */
    output_data.fz = new OutputData(
        "Lower Neck Tension",
        images_dir + "/lower_neck_tension"
    );
    output_data.mx = new OutputData(
        "Lower Neck Lateral Flexion",
        images_dir + "/lower_neck_lateral_flexion"
    );
    output_data.my = new OutputData(
        "Lower Neck Negative Extension",
        images_dir + "/lower_neck_extension"
    );

    if (beam != undefined) {
        // Should check data components are available for this beam

        // Read in Neck data force

        var ln_fx = read_data("BEAM BASIC", 0, beam, "NX"); // Neck tension
        var ln_fy = read_data("BEAM BASIC", 0, beam, "NY"); // Neck tension
        var ln_fz = read_data("BEAM BASIC", 0, beam, "NZ"); // Neck tension
        var ln_mx = read_data("BEAM BASIC", 0, beam, "MX"); // Neck Lateral Flexion
        var ln_my = read_data("BEAM BASIC", 0, beam, "MY"); // Neck Negative Extension

        // Check that the curves read in - i.e. the beam id and component are valid
        // Create a blank image to let the user know.

        if (!ln_fx || !ln_fy || !ln_fz || !ln_mx || !ln_my) {
            write_blank_images(output_data, "NO DATA FOR BEAM " + beam);
        } else {
            // C600 filter (convert to seconds) and convert force into kN
            // and torsion in to Nm

            //kN
            ln_fx = FilterC600_kN(ln_fx);
            ln_fy = FilterC600_kN(ln_fy);
            ln_fz = FilterC600_kN(ln_fz);
            //Nm
            ln_mx = FilterC600_Nm(ln_mx);
            ln_my = FilterC600_Nm(ln_my);

            // Remove all curves and datums

            remove_all_curves_and_datums();

            // Lower Neck Tension
            // Convert from seconds back to model time
            var ln_fz = Operate.Mux(ln_fz, oGlblData.time_factor);
            // Set label and style
            set_labels(
                ln_fz,
                "Lower Neck Tension",
                "Time (" + oGlblData.unit_time + ")",
                "Tension (kN)"
            );
            set_line_style(ln_fz, Colour.BLACK);

            // Draw datums
            // traffic light bands, green is full marks amber is sliding scale and red is 0
            draw_constant_datums(
                NECK_LOWER_TENSION_HI_LIMIT,
                NECK_LOWER_TENSION_LO_LIMIT,
                null,
                NECK_LOWER_TENSION_LO_LIMIT
            );
            write_image(
                output_data.fz.title,
                output_data.fz.fname,
                ln_fz,
                1.1 * NECK_LOWER_TENSION_LO_LIMIT
            ); //set maximum y-axis value to at least 1.1 times the lower performance limit

            // output_data.fz.curveList.push(ln_fz.id);
            // write_curve("cur", output_data.fz.curveList, output_data.fz.fname);
            // write_curve("csv", output_data.fz.curveList, output_data.fz.fname);

            //remove_all_curves_and_datums();

            //Lower Neck Lateral Flexion
            // Convert from seconds back to model time
            var ln_mx = Operate.Mux(ln_mx, oGlblData.time_factor);
            var ln_fy = Operate.Mux(ln_fy, oGlblData.time_factor);
            var ln_fy_Dz = Operate.Mul(ln_fy, -Dz * 1000); //convert kN to N by *1000

            remove_all_curves_and_datums();
            //Mx(base of neck) = |min(Mx - Fy * Dz)|
            var ln_mx = Operate.Add(ln_mx, ln_fy_Dz);

            // Set label and style
            set_labels(
                ln_mx,
                "Lower Neck Lateral Flexion",
                "Time (" + oGlblData.unit_time + ")",
                "Flexion (Nm)"
            );
            set_line_style(ln_mx, Colour.BLACK);

            // Draw datums
            // traffic light bands, green is full marks amber is sliding scale and red is 0
            draw_constant_datums(
                NECK_LOWER_LATERAL_FLEXION_HI_LIMIT,
                NECK_LOWER_LATERAL_FLEXION_LO_LIMIT,
                null,
                NECK_LOWER_LATERAL_FLEXION_LO_LIMIT,
                true
            );
            write_image(
                output_data.mx.title,
                output_data.mx.fname,
                ln_mx,
                1.1 * NECK_LOWER_LATERAL_FLEXION_LO_LIMIT,
                -1.1 * NECK_LOWER_LATERAL_FLEXION_LO_LIMIT
            ); //set maximum (and min) y-axis value to at least 1.1 times the lower performance limit

            // output_data.mx.curveList.push(ln_mx.id);
            // write_curve("cur", output_data.mx.curveList, output_data.mx.fname);
            // write_curve("csv", output_data.mx.curveList, output_data.mx.fname);

            //remove_all_curves_and_datums();

            //Lower Neck Negative Extension
            // Convert from seconds back to model time
            var ln_my = Operate.Mux(ln_my, oGlblData.time_factor);
            var ln_fx = Operate.Mux(ln_fx, oGlblData.time_factor);
            var ln_fx_Dz = Operate.Mul(ln_fx, Dz * 1000); //convert kN to N by *1000

            remove_all_curves_and_datums();
            //My(base of neck) = |min(My + Fx * Dz)
            var ln_my = Operate.Add(ln_my, ln_fx_Dz);

            // Set label and style
            set_labels(
                ln_my,
                "Lower Neck Negative Extension",
                "Time (" + oGlblData.unit_time + ")",
                "Extension (Nm)"
            ); //TODO is Extension correct?
            set_line_style(ln_my, Colour.BLACK);

            // Draw datums
            // traffic light bands, green is full marks amber is sliding scale and red is 0
            draw_constant_datums(
                NECK_LOWER_NEG_EXTENSION_HI_LIMIT,
                NECK_LOWER_NEG_EXTENSION_LO_LIMIT,
                null,
                NECK_LOWER_NEG_EXTENSION_LO_LIMIT,
                true
            );
            write_image(
                output_data.my.title,
                output_data.my.fname,
                ln_my,
                1.1 * NECK_LOWER_NEG_EXTENSION_LO_LIMIT,
                -1.1 * NECK_LOWER_NEG_EXTENSION_LO_LIMIT
            ); //set maximum (and min) y-axis value to at least 1.1 times the lower performance limit

            // output_data.my.curveList.push(ln_my.id);
            // write_curve("cur", output_data.my, output_data.my.fname);
            // write_curve("csv", output_data.my, output_data.my.fname);

            remove_all_curves_and_datums();

            // LOWER NECK OUTPUT VARIABLES

            // Find the max value of each curve

            var max_tension = ln_fz.ymax;
            var max_flexion = Math.max(ln_mx.ymax, -ln_mx.ymin); //take the absolute peak value as neck could flex either way depending on which side the driver is on (LHD/RHD)
            var max_extension = Math.abs(ln_my.ymin); //take the absolute peak negative value (this corresponds to moment of head rotating forward about base of neck)

            // Write variables to csv file for Reporter

            var tension_score = sliding_scale(
                max_tension,
                NECK_LOWER_TENSION_HI_LIMIT,
                NECK_LOWER_TENSION_LO_LIMIT,
                NECK_HI_SCORE,
                NECK_LO_SCORE
            );

            write_variable(
                f,
                "NECK_LOWER_TENSION",
                max_tension.toFixed(3),
                "Neck lower tension force in kN",
                "String"
            );
            write_variable(
                f,
                "NECK_LOWER_TENSION_SCORE",
                tension_score.toFixed(3),
                "Neck lower tension score",
                "String"
            );

            var flexion_score = sliding_scale(
                max_flexion,
                NECK_LOWER_LATERAL_FLEXION_HI_LIMIT,
                NECK_LOWER_LATERAL_FLEXION_LO_LIMIT,
                NECK_HI_SCORE,
                NECK_LO_SCORE
            );

            write_variable(
                f,
                "NECK_LOWER_FLEXION",
                max_flexion.toFixed(3),
                "Neck lower flexion moment in Nm",
                "String"
            );
            write_variable(
                f,
                "NECK_LOWER_FLEXION_SCORE",
                flexion_score.toFixed(3),
                "Neck lower flexion score",
                "String"
            );

            var extension_score = sliding_scale(
                max_extension,
                NECK_LOWER_NEG_EXTENSION_HI_LIMIT,
                NECK_LOWER_NEG_EXTENSION_LO_LIMIT,
                NECK_HI_SCORE,
                NECK_LO_SCORE
            );

            write_variable(
                f,
                "NECK_LOWER_EXTENSION",
                max_extension.toFixed(3),
                "Neck lower extension moment in Nm",
                "String"
            );
            write_variable(
                f,
                "NECK_LOWER_EXTENSION_SCORE",
                extension_score.toFixed(3),
                "Neck lower extension score",
                "String"
            );

            //lower neck worst score
            var ln_score = Math.min(tension_score, flexion_score); // extension_score is not included as it is nonly monitored in 2020-2022

            write_variable(
                f,
                "NECK_LOWER_SCORE",
                ln_score.toFixed(3),
                "Neck lower worst score",
                "String"
            );
        }
    } else {
        // No beam defined - variables should be set to blank by first script in template.
        // Create a blank image to let the user know.

        write_blank_images(output_data, "NO BEAM ID DEFINED FOR");
    }
    remove_all_curves_and_datums();
}

function do_head_rating_calc(f, HEAD_HI_SCORE, HEAD_LO_SCORE) {
    //Calculates the head score and writes relevant values
    //into the Reporter csv variables file <f>.

    /*FAR SIDE HEAD CRITERIA
    HIC_15 with direct contact
    Resultant 3ms acceleration
    */

    var model_id = 1;

    var node = oGlblData.head_node;
    var node_y = oGlblData.head_node_y;
    var node_z = oGlblData.head_node_z;

    var capping_limit = "FALSE";

    var output_data = {};

    output_data.acc = new OutputData(
        "Head Acceleration",
        images_dir + "/Head_Acceleration"
    );

    if (node != undefined) {
        // Read in acceleration X, Y and Z
        var one_node = true;

        // If nodes for three accelerometer positions defined, use all three, otherwise, read all components from same node
        if (
            node_y != undefined &&
            node_z != undefined &&
            node_y != 0 &&
            node_z != 0
        ) {
            var c_acc = read_xyz_accelerations(model_id, node, node_y, node_z);
            one_node = false;
        } else {
            var c_acc = read_xyz_accelerations(model_id, node);
        }

        // Check that the curves read in - i.e. the node id and component are valid
        // Create a blank image to let the user know.

        // Generate a relevant message for the blank image depending on which acceleration data are missing
        if (!c_acc.ax || !c_acc.ay || !c_acc.az) {
            var blank_image_msg;
            // If only one node supplied, just report that node ID
            if (one_node)
                blank_image_msg = "NO ACCEL DATA (NODE " + node + ") ";
            // else report each of the node IDs that were invalid
            else {
                var acc_comp_lst = [];
                var acc_id_lst = [];
                if (!c_acc.ax) {
                    acc_comp_lst.push(" X");
                    acc_id_lst.push(" " + node);
                }
                if (!c_acc.ay) {
                    acc_comp_lst.push(" Y");
                    acc_id_lst.push(" " + node_y);
                }
                if (!c_acc.az) {
                    acc_comp_lst.push(" Z");
                    acc_id_lst.push(" " + node_z);
                }
                blank_image_msg =
                    "NO" +
                    acc_comp_lst +
                    " ACCEL DATA (NODE(s) " +
                    acc_id_lst +
                    ") ";
            }
            create_blank_image(
                blank_image_msg + output_data.acc.title,
                output_data.acc.fname
            );
        } else {
            // Convert to g from model units

            var c_g = convert_xyz_acc_to_g(c_acc);

            // C1000 filter (convert to seconds first)

            var c_gx_c1000 = convert_to_s_and_filter(
                c_g.gx,
                Operate.C1000,
                REG_DT
            );
            var c_gy_c1000 = convert_to_s_and_filter(
                c_g.gy,
                Operate.C1000,
                REG_DT
            );
            var c_gz_c1000 = convert_to_s_and_filter(
                c_g.gz,
                Operate.C1000,
                REG_DT
            );

            // Vector combine

            var c_vec_g = Operate.Vec(c_gx_c1000, c_gy_c1000, c_gz_c1000);

            // Remove all curves and datums

            remove_all_curves_and_datums();

            // Convert from seconds back to model time

            c_vec_g = Operate.Mux(c_vec_g, oGlblData.time_factor);

            // Get peak aceleration (in g)

            DialogueInput("/AU");
            var peak_accn = c_vec_g.ymax;

            // Get HIC15 value

            var hic = Operate.Hic(c_vec_g, HIC_WINDOW, 1.0);

            // Get 3ms exceedence

            Operate.Tms(c_vec_g, TMS_PERIOD * oGlblData.time_factor);
            var tms = c_vec_g.tms;

            // Set label and style

            set_labels(
                c_vec_g,
                "Head Acceleration Magnitude",
                "Time (" + oGlblData.unit_time + ")",
                "Acceleration (g)"
            );
            set_line_style(c_vec_g, Colour.BLACK);

            // Create image and curve files

            write_image(output_data.acc.title, output_data.acc.fname);

            output_data.acc.curveList.push(c_vec_g.id);
            write_curve(
                "cur",
                output_data.acc.curveList,
                output_data.acc.fname
            );
            write_curve(
                "csv",
                output_data.acc.curveList,
                output_data.acc.fname
            );

            // Remove all curves and datums

            remove_all_curves_and_datums();

            // Calculate Head points

            var hic_score = 0.0;
            var tms_score = 0.0;
            var score = 0.0;

            Message(
                "HIC_HI_LIMIT = " +
                    HIC_HI_LIMIT +
                    "; HIC_LO_LIMIT = " +
                    HIC_LO_LIMIT +
                    "; HEAD_HI_SCORE = " +
                    HEAD_HI_SCORE,
                "; HEAD_LO_SCORE = " + HEAD_LO_SCORE
            );
            hic_score = sliding_scale(
                hic,
                HIC_HI_LIMIT,
                HIC_LO_LIMIT,
                HEAD_HI_SCORE,
                HEAD_LO_SCORE
            );
            tms_score = sliding_scale(
                tms,
                TMS_HI_LIMIT,
                TMS_LO_LIMIT,
                HEAD_HI_SCORE,
                HEAD_LO_SCORE
            );

            score = Math.min(hic_score, tms_score);

            //Bound score between limits

            //HIC value and score

            write_variable(
                f,
                "HEAD_HIC15",
                hic.toFixed(3),
                "HIC15 value",
                "String"
            );
            write_variable(
                f,
                "HEAD_HIC15_SCORE",
                hic_score.toFixed(3),
                "HIC15 score",
                "String"
            );

            // 3ms value and score

            write_variable(
                f,
                "HEAD_3MS",
                tms.toFixed(3),
                "3ms value",
                "String"
            );
            write_variable(
                f,
                "HEAD_3MS_SCORE",
                tms_score.toFixed(3),
                "3ms score",
                "String"
            );

            // write overall head score

            write_variable(
                f,
                "HEAD_FINAL_SCORE",
                score.toFixed(3),
                "Final head score",
                "String"
            );

            // Capping limit should be applied to overall score if these are zero

            if (score == 0.0) capping_limit = "TRUE";

            write_variable(
                f,
                "HEAD_CAPPING_LIMIT",
                capping_limit,
                "Head capping limit",
                "String"
            );
        }
    } else {
        // No NODE defined-variables should be set to blank by the first script in this template.
        // Create a blank image to let the user know.

        write_blank_images(output_data, "NO NODE ID DEFINED FOR");
    }
}

// Common functions for EuroNCAP 2015 side impact templates (MDB and Side Pole) modified for FAR SIDE
// This function considers World SID dummy
function do_abdomen_rating_calc(f, ABDOMEN_HI_SCORE, ABDOMEN_LO_SCORE) {
    //Calculate the abdomen scores for the dummy and writes the relevant values
    //into the Reporter csv variables file <f>.

    /*FAR SIDE CHEST AND ABDOMEN CRITERIA
Chest lateral compression
Abdomen lateral compression
*/

    var capping_limit = "FALSE";

    //set perfomance limits
    var ABDOMEN_COMPRESSION_GOOD = ABDOMEN_COMPRESSION_HI_LIMIT; //higher performance limit
    var ABDOMEN_COMPRESSION_WEAK = ABDOMEN_COMPRESSION_LO_LIMIT; //lower performance limit

    //note IRTRACC_length is same for all ribs
    var IRTRACC_length = irtracc_length_abs_length_mm * oGlblData.mm_factor;

    // Read in forces front, mid and rear

    for (var fp = 0; fp < 2; fp++) {
        // upper and lower abdomen rotation and deformations from 2D IR-TRACC sensor are
        // used for the calculation of lateral deformation
        if (fp == 0) {
            //Upper
            var spring_rot = oGlblData.abdomen_upper_spring_rot;
            var spring_trans = oGlblData.abdomen_upper_spring_trans;
        } else if (fp == 1) {
            //Lower
            var spring_rot = oGlblData.abdomen_lower_spring_rot;
            var spring_trans = oGlblData.abdomen_lower_spring_trans;
        }

        //Check data components are available for this spring, if not create a blank image

        if (spring_rot != undefined || spring_trans != undefined) {
            //Read in SPRING rotation and compression

            var c_rot = read_data("SPRING ROT", 0, spring_rot, "RT");
            var c_disp = read_data("SPRING TR", 0, spring_trans, "ET");

            // Add the initial IRTRACT_length to get the actual length at a time (t)
            var abs_length = Operate.Add(c_disp, IRTRACC_length);

            // Converting the units of y-axis and x-axis into mm and sec respectively
            var abs_length_mm = Operate.Div(abs_length, oGlblData.mm_factor);
            var abs_length_mm_s = Operate.Dix(
                abs_length_mm,
                oGlblData.time_factor
            );
            var c_rot_s = Operate.Dix(c_rot, oGlblData.time_factor);

            // C180 filter and regularize the data using 0.000001 sec
            var abs_length_c180_mm = Operate.C180(abs_length_mm_s, REG_DT);
            var c_rot_c180 = Operate.C180(c_rot_s, REG_DT);

            // Calculating the Lateral displacement
            var c_cos_theta_c180 = Operate.Cos(c_rot_c180);
            var lat_length_mm = Operate.Mul(
                abs_length_c180_mm,
                c_cos_theta_c180
            );

            var lat_disp_elong_mm = Operate.Sub(lat_length_mm, IRTRACC_length);
            var lat_disp_mm = Operate.Mul(lat_disp_elong_mm, -1);

            if (fp == 0) {
                var c_upp_c180_mm = lat_disp_mm;
            } else {
                var c_bot_c180_mm = lat_disp_mm;
            }
        }
    }

    var output_data = {};

    output_data.com = new OutputData(
        "Abdomen Compression",
        images_dir + "/" + "Abdomen_Compression"
    );

    if (c_upp_c180_mm && c_bot_c180_mm) {
        // Remove all curves and datums

        remove_all_curves_and_datums();

        // Convert from seconds back to model time

        var c_upp_com = Operate.Mux(c_upp_c180_mm, oGlblData.time_factor);
        var c_bot_com = Operate.Mux(c_bot_c180_mm, oGlblData.time_factor);

        // Set label and style

        set_labels(
            c_upp_com,
            "Upper Abdomen Compression",
            "Time (" + oGlblData.unit_time + ")",
            "Compression (mm)"
        );
        set_labels(
            c_bot_com,
            "Bottom Abdomen Compression",
            "Time (" + oGlblData.unit_time + ")",
            "Compression (mm)"
        );

        set_line_style(c_upp_com, Colour.CYAN);
        set_line_style(c_bot_com, Colour.BLUE);

        // Draw datums

        // traffic light bands, green is full marks amber is sliding scale and red is 0
        draw_constant_datums(
            ABDOMEN_COMPRESSION_GOOD,
            ABDOMEN_COMPRESSION_WEAK,
            null,
            ABDOMEN_COMPRESSION_WEAK
        );

        // Create image and curve files of abdomen compresison curves

        write_image(
            output_data.com.title,
            output_data.com.fname,
            [c_upp_com, c_bot_com],
            ABDOMEN_COMPRESSION_WEAK
        );

        output_data.com.curveList.push(c_upp_com.id, c_bot_com.id);
        write_curve("cur", output_data.com.curveList, output_data.com.fname);
        write_curve("csv", output_data.com.curveList, output_data.com.fname);

        // Remove all curves and datums

        remove_all_curves_and_datums();

        // ABDOMEN COMPRESSION VARIABLES

        // Find the max compression of each rib

        var max_upp_rib_com = c_upp_com.ymax;
        var max_bottom_rib_com = c_bot_com.ymax;

        // Write variables for Reporter

        var topc = sliding_scale(
            max_upp_rib_com,
            ABDOMEN_COMPRESSION_GOOD,
            ABDOMEN_COMPRESSION_WEAK,
            ABDOMEN_HI_SCORE,
            ABDOMEN_LO_SCORE
        );

        write_variable(
            f,
            "ABDOMEN_TOP_COMPRESSION",
            max_upp_rib_com.toFixed(3),
            "Abdomen top compression value in mm",
            "String"
        );
        write_variable(
            f,
            "ABDOMEN_TOP_COMPRESSION_SCORE",
            topc.toFixed(3),
            "Abdomen top compression score",
            "String"
        );

        var botc = sliding_scale(
            max_bottom_rib_com,
            ABDOMEN_COMPRESSION_GOOD,
            ABDOMEN_COMPRESSION_WEAK,
            ABDOMEN_HI_SCORE,
            ABDOMEN_LO_SCORE
        );

        write_variable(
            f,
            "ABDOMEN_BOTTOM_COMPRESSION",
            max_bottom_rib_com.toFixed(3),
            "Abdomen bottom compression value in mm",
            "String"
        );
        write_variable(
            f,
            "ABDOMEN_BOTTOM_COMPRESSION_SCORE",
            topc.toFixed(3),
            "Abdomen bottom compression score",
            "String"
        );

        //Max compression ans score
        var maxc = Math.max(max_upp_rib_com, max_bottom_rib_com);
        write_variable(
            f,
            "ABDOMEN_MAX_COMPRESSION",
            maxc.toFixed(3),
            "Abdomen max of top and bottom compression value in mm",
            "String"
        );
        write_variable(
            f,
            "ABDOMEN_MAX_COMPRESSION_SCORE",
            Math.min(topc, botc).toFixed(3),
            "Abdomen lowest compression score",
            "String"
        );

        // Capping limit should be applied to overall score if compression is greater than capping limit value
        // (Note that this is limit may be different to the lower performance limit)

        if (maxc > ABDOMEN_COMPRESSION_CAPPING_LIMIT) {
            capping_limit = "TRUE";
        }

        write_variable(
            f,
            "ABDOMEN_CAPPING_LIMIT",
            capping_limit,
            "Abdomen capping limit",
            "String"
        );
    } else {
        // No SPRING defined - variables should be set to blank by the first script in this template.
        // Create a blank image to let the user know.

        write_blank_images(output_data, "INVALID/NO SPRING ID DEFINED FOR");
    }

    remove_all_curves_and_datums();
}

// This function considers World SID dummy for the calculation of Visocus Coefficient (VC)
function do_chest_rating_calc(f, CHEST_HI_SCORE, CHEST_LO_SCORE) {
    // Calclates the chest scores and writes
    // relevant values into the Reporter csv variable file <f>.

    var capping_limit = "FALSE";

    //set perfomance limits
    var CHEST_COMPRESSION_GOOD = CHEST_COMPRESSION_HI_LIMIT; //higher performance limit
    var CHEST_COMPRESSION_WEAK = CHEST_COMPRESSION_LO_LIMIT; //lower performance limit

    var CHEST_TOP_HI_SCORE = CHEST_HI_SCORE;
    var CHEST_TOP_LO_SCORE = CHEST_LO_SCORE;

    var CHEST_MID_HI_SCORE = CHEST_HI_SCORE;
    var CHEST_MID_LO_SCORE = CHEST_LO_SCORE;

    var CHEST_BOTTOM_HI_SCORE = CHEST_HI_SCORE;
    var CHEST_BOTTOM_LO_SCORE = CHEST_LO_SCORE;

    //note IRTRACC_length is same for all ribs
    var IRTRACC_length = irtracc_length_abs_length_mm * oGlblData.mm_factor;

    for (var rib = 0; rib < 3; rib++) {
        // chest rib rotation and deformations from 2D IR-TRACC sensor are
        // used for the calculation of lateral deformation of the chest ribs
        if (rib == 0) {
            var spring_rot = oGlblData.chest_upper_rib_spring_rot;
            var spring_trans = oGlblData.chest_upper_rib_spring_trans;
        } else if (rib == 1) {
            spring_rot = oGlblData.chest_middle_rib_spring_rot;
            spring_trans = oGlblData.chest_middle_rib_spring_trans;
        } else {
            spring_rot = oGlblData.chest_bottom_rib_spring_rot;
            spring_trans = oGlblData.chest_bottom_rib_spring_trans;
        }

        // Check data components are available for this SPRING, if not create a blank image

        if (spring_rot != undefined || spring_trans != undefined) {
            //Read in SPRING rotation and compression

            var c_rot = read_data("SPRING ROT", 0, spring_rot, "RT");
            var c_disp = read_data("SPRING TR", 0, spring_trans, "ET");

            // Add the initial IRTRACT_length to get the actual length at a time (t)
            var abs_length = Operate.Add(c_disp, IRTRACC_length);

            // Converting the units of y-axis and x-axis into mm and sec respectively
            var abs_length_mm = Operate.Div(abs_length, oGlblData.mm_factor);
            var abs_length_mm_s = Operate.Dix(
                abs_length_mm,
                oGlblData.time_factor
            );
            var c_rot_s = Operate.Dix(c_rot, oGlblData.time_factor);

            // C180 filter and regularize the data using 0.000001 sec
            var abs_length_c180_mm = Operate.C180(abs_length_mm_s, REG_DT);
            var c_rot_c180 = Operate.C180(c_rot_s, REG_DT);

            // Calculating the Lateral displacement
            var c_cos_theta_c180 = Operate.Cos(c_rot_c180);
            var lat_length_mm = Operate.Mul(
                abs_length_c180_mm,
                c_cos_theta_c180
            );

            var lat_disp_elong_mm = Operate.Sub(lat_length_mm, IRTRACC_length);
            var lat_disp_mm = Operate.Mul(lat_disp_elong_mm, -1);

            if (rib == 0) {
                var c_upp_rib_c180_mm = lat_disp_mm;
            } else if (rib == 1) {
                var c_mid_rib_c180_mm = lat_disp_mm;
            } else {
                var c_bot_rib_c180_mm = lat_disp_mm;
            }
        }
    }

    // Do compression calculation

    var topc = 0.0;
    var midc = 0.0;
    var botc = 0.0;

    var output_data = {};

    output_data.com = new OutputData(
        "Chest Compression",
        images_dir + "/Chest_Compression"
    );

    if (c_upp_rib_c180_mm && c_mid_rib_c180_mm && c_bot_rib_c180_mm) {
        // Remove all curves and datums

        remove_all_curves_and_datums();

        // Convert from seconds back to model time

        var c_upp_com = Operate.Mux(c_upp_rib_c180_mm, oGlblData.time_factor);
        var c_mid_com = Operate.Mux(c_mid_rib_c180_mm, oGlblData.time_factor);
        var c_bot_com = Operate.Mux(c_bot_rib_c180_mm, oGlblData.time_factor);

        // Set label and style

        set_labels(
            c_upp_com,
            "Upper Rib Compression",
            "Time (" + oGlblData.unit_time + ")",
            "Compression (mm)"
        );
        set_labels(
            c_mid_com,
            "Middle Rib Compression",
            "Time (" + oGlblData.unit_time + ")",
            "Compression (mm)"
        );
        set_labels(
            c_bot_com,
            "Bottom Rib Compression",
            "Time (" + oGlblData.unit_time + ")",
            "Compression (mm)"
        );

        set_line_style(c_upp_com, Colour.CYAN);
        set_line_style(c_mid_com, Colour.BLUE);
        set_line_style(c_bot_com, Colour.BLACK);

        // Draw datums
        // traffic light bands, green is full marks amber is sliding scale and red is 0
        draw_constant_datums(
            CHEST_COMPRESSION_GOOD,
            CHEST_COMPRESSION_WEAK,
            null,
            CHEST_COMPRESSION_WEAK
        );

        // Create image and curve files of chest compresison curves

        write_image(
            output_data.com.title,
            output_data.com.fname,
            [c_upp_com, c_mid_com, c_bot_com],
            CHEST_COMPRESSION_WEAK
        );

        output_data.com.curveList.push(
            c_upp_com.id,
            c_mid_com.id,
            c_bot_com.id
        );
        write_curve("cur", output_data.com.curveList, output_data.com.fname);
        write_curve("csv", output_data.com.curveList, output_data.com.fname);

        // Remove all curves and datums

        remove_all_curves_and_datums();

        // CHEST COMPRESSION VARIABLES

        // Find the max compression of each rib

        var max_upp_rib_com = c_upp_com.ymax;
        var max_mid_rib_com = c_mid_com.ymax;
        var max_bottom_rib_com = c_bot_com.ymax;

        // Write variables to csv file for Reporter

        topc = sliding_scale(
            max_upp_rib_com,
            CHEST_COMPRESSION_GOOD,
            CHEST_COMPRESSION_WEAK,
            CHEST_TOP_HI_SCORE,
            CHEST_TOP_LO_SCORE
        );

        write_variable(
            f,
            "CHEST_TOP_COMPRESSION",
            max_upp_rib_com.toFixed(3),
            "Chest top rib compression value in mm",
            "String"
        );
        write_variable(
            f,
            "CHEST_TOP_COMPRESSION_SCORE",
            topc.toFixed(3),
            "Chest top rib compression score",
            "String"
        );

        midc = sliding_scale(
            max_mid_rib_com,
            CHEST_COMPRESSION_GOOD,
            CHEST_COMPRESSION_WEAK,
            CHEST_MID_HI_SCORE,
            CHEST_MID_LO_SCORE
        );

        write_variable(
            f,
            "CHEST_MIDDLE_COMPRESSION",
            max_mid_rib_com.toFixed(3),
            "Chest middle rib compression value in mm",
            "String"
        );
        write_variable(
            f,
            "CHEST_MIDDLE_COMPRESSION_SCORE",
            midc.toFixed(3),
            "Chest middle rib compression score",
            "String"
        );

        botc = sliding_scale(
            max_bottom_rib_com,
            CHEST_COMPRESSION_GOOD,
            CHEST_COMPRESSION_WEAK,
            CHEST_BOTTOM_HI_SCORE,
            CHEST_BOTTOM_LO_SCORE
        );

        write_variable(
            f,
            "CHEST_BOTTOM_COMPRESSION",
            max_bottom_rib_com.toFixed(3),
            "Chest bottom rib compression in mm",
            "String"
        );
        write_variable(
            f,
            "CHEST_BOTTOM_COMPRESSION_SCORE",
            botc.toFixed(3),
            "Chest bottom rib compression score",
            "String"
        );

        var maxc = Math.max(
            max_upp_rib_com,
            max_mid_rib_com,
            max_bottom_rib_com
        );
        //max compression and associated (worst) score
        write_variable(
            f,
            "CHEST_MAX_COMPRESSION",
            maxc.toFixed(3),
            "Chest max rib compression in mm",
            "String"
        );
        write_variable(
            f,
            "CHEST_MAX_COMPRESSION_SCORE",
            Math.min(topc, midc, botc).toFixed(3),
            "Chest worst rib compression score",
            "String"
        );

        // Capping limit should be applied to overall score if compression is greater than capping limit value
        // (Note that this is limit may be different to the lower performance limit)

        if (maxc > CHEST_COMPRESSION_CAPPING_LIMIT) {
            capping_limit = "TRUE";
        }

        write_variable(
            f,
            "CHEST_CAPPING_LIMIT",
            capping_limit,
            "Chest capping limit",
            "String"
        );
    } else {
        write_blank_images(output_data, "INVALID/NO SPRING ID DEFINED FOR");
    }

    remove_all_curves_and_datums();
}

function do_pelvis_rating_calc(f) {
    //Calculates the pelvis scores for the dummy and writes relevant
    //values into the Reporter csv variable file <f>.

    var beam = oGlblData.pelvis_beam;

    var output_data = {};

    output_data.force = new OutputData(
        "Pubic Symphysis Force",
        images_dir + "/Pelvis_Force"
    );

    if (beam != undefined) {
        // Should check data components are available for this beam

        // Read in Pubic Symphysis force

        /*
        The pubic load cell is a one-channel load cell. This means that only the force in
        the local y-direction can be evaluated in the hardware.
        
        see https://www.dummymodels.com/models/worldsid/documents/manuals/wsid50_pdb_v7-1_manual_v0-0.pdf pg.20
        */
        var c_pf = read_data("BEAM BASIC", 0, beam, "NY");

        // Check that the curves read in - i.e. the beam id and component are valid
        // Create a blank image to let the user know.

        if (!c_pf) {
            write_blank_images(output_data, "NO DATA FOR BEAM " + beam);
        } else {
            // C600 filter (convert to seconds) and convert force into kN
            // AND
            // Multiply by -1.0 to make compressive force +ve if required as
            // different versions can output the force in opposite directions.
            // Test for this by seeing if the absolute maximum value is -ve
            c_pf = MakeCompressionPositive(FilterC600_kN(c_pf));

            // Remove all curves and datums

            remove_all_curves_and_datums();

            // Convert from seconds back to model time

            c_pf = Operate.Mux(c_pf, oGlblData.time_factor);

            // Pelvis force
            var pelvis_result = CreateLumbarCurve(
                output_data.force,
                c_pf,
                "Pubic Symphysis Force",
                "Compressive Force (kN)",
                PELVIS_PUBIC_SYMPHYSIS
            );
            // Write out variables
            write_variable(
                f,
                "PELVIS_FORCE",
                pelvis_result.value,
                "Pelvis force value",
                "String"
            );
            write_variable(
                f,
                "PELVIS_FORCE_SCORE",
                pelvis_result.score,
                "Pelvis Force score",
                "String"
            );
        }
    } else {
        // No beam defined - variables should be set to blank by first script in template.
        // Create a blank image to let the user know.

        write_blank_images(output_data, "NO BEAM ID DEFINED FOR");
    }

    //Lumbar forces

    var beam = oGlblData.lumbar_beam;

    var output_data = {};

    /*
    The lumbar load cell is a six-channel load cell. This means that the forces and the
    moments in each direction of axis can be evaluated in the hardware.

    see https://www.dummymodels.com/models/worldsid/documents/manuals/wsid50_pdb_v7-1_manual_v0-0.pdf pg.19
    */
    output_data.force_y = new OutputData(
        "Lumbar Beam Y Force",
        images_dir + "/Lumbar_Y_Force"
    );
    output_data.force_z = new OutputData(
        "Lumbar Beam Z Force",
        images_dir + "/Lumbar_Z_Force"
    );
    output_data.torsion_x = new OutputData(
        "Lumbar Beam X Torsion",
        images_dir + "/Lumbar_X_Torsion"
    );

    if (beam != undefined) {
        // Should check data components are available for this beam

        // Read in Pubic Symphysis force

        var lumbar_Fy = read_data("BEAM BASIC", 0, beam, "NY"); //y shear force (side-to-side)
        var lumbar_Fz = read_data("BEAM BASIC", 0, beam, "NZ"); //z axial force (vertical)
        var lumbar_Mx = read_data("BEAM BASIC", 0, beam, "MX"); //x torsion

        // Check that the curves read in - i.e. the beam id and component are valid
        // Create a blank image to let the user know.

        if (!lumbar_Fy || !lumbar_Fz || !lumbar_Mx) {
            write_blank_images(output_data, "NO DATA FOR BEAM " + beam);
        } else {
            // C600 filter (convert to seconds) and convert force into kN
            // and torsion in to Nm
            // AND
            // Multiply by -1.0 to make compressive force +ve if required as
            // different versions can output the force in opposite directions.
            // Test for this by seeing if the absolute maximum value is -ve

            lumbar_Fy = MakeCompressionPositive(FilterC600_kN(lumbar_Fy)); //kN
            lumbar_Fz = MakeCompressionPositive(FilterC600_kN(lumbar_Fz));
            //Nm
            lumbar_Mx = MakeCompressionPositive(FilterC600_Nm(lumbar_Mx));

            // Remove all curves and datums

            remove_all_curves_and_datums();

            // Convert from seconds back to model time

            // Y force (sideways motion of spine)
            lumbar_Fy = Operate.Mux(lumbar_Fy, oGlblData.time_factor);
            var fy_result = CreateLumbarCurve(
                output_data.force_y,
                lumbar_Fy,
                "Lumbar Y",
                "Force (kN)",
                PELVIS_LUMBAR_FY
            );
            // Write out variables
            write_variable(
                f,
                "LUMBAR_Y_FORCE",
                fy_result.value,
                "Lumbar Y Force value",
                "String"
            );
            write_variable(
                f,
                "LUMBAR_Y_FORCE_SCORE",
                fy_result.score,
                "Lumbar Y Force score",
                "String"
            );

            // Z force (vertical/axial movement of spine)
            lumbar_Fz = Operate.Mux(lumbar_Fz, oGlblData.time_factor);
            var fz_result = CreateLumbarCurve(
                output_data.force_z,
                lumbar_Fz,
                "Lumbar Z",
                "Force (kN)",
                PELVIS_LUMBAR_FZ
            );
            // Write out variables
            write_variable(
                f,
                "LUMBAR_Z_FORCE",
                fz_result.value,
                "Lumbar Z Force value",
                "String"
            );
            write_variable(
                f,
                "LUMBAR_Z_FORCE_SCORE",
                fz_result.score,
                "Lumbar Z Force score",
                "String"
            );

            // X Torsion (roataion from side-to-side of spine)
            lumbar_Mx = Operate.Mux(lumbar_Mx, oGlblData.time_factor);
            var mx_result = CreateLumbarCurve(
                output_data.torsion_x,
                lumbar_Mx,
                "Lumbar X Torsion",
                "Torsion (Nm)",
                PELVIS_LUMBAR_MX
            );
            // Write out variables
            write_variable(
                f,
                "LUMBAR_X_TORSION",
                mx_result.value,
                "Lumbar X Torsion value",
                "String"
            );
            write_variable(
                f,
                "LUMBAR_X_TORSION_SCORE",
                mx_result.score,
                "Lumbar X Torsion score",
                "String"
            );
        }
    } else {
        // No beam defined - variables should be set to blank by first script in template.
        // Create a blank image to let the user know.

        write_blank_images(output_data, "NO BEAM ID DEFINED FOR");
    }

    if (!pelvis_result || !fy_result || !fz_result || !mx_result) {
        ErrorMessage(
            "Not all results for pelvis and lumbar could be extracted.\npelvis_result  = " +
                pelvis_result +
                "\nfz_result  = " +
                fz_result +
                "\nmx_result = " +
                mx_result
        );
    } else {
        var pelvis_and_lumbar_modifier = Math.min(
            pelvis_result.score,
            fy_result.score,
            fz_result.score,
            mx_result.score
        );
        write_variable(
            f,
            "PELVIS_AND_LUMBAR_MODIFIER",
            pelvis_and_lumbar_modifier,
            "pelvis and lumbar modifier (0 or -4 points)",
            "String"
        );
    }
}

function FilterC600_kN(curve) {
    // C600 filter (convert to seconds) and convert force into kN
    curve = Operate.Dix(curve, oGlblData.time_factor);
    curve = Operate.C600(curve, REG_DT);
    curve = Operate.Div(curve, oGlblData.kn_factor);

    return curve;
}

function FilterC600_Nm(curve) {
    // C600 filter (convert to seconds) and convert force into kN
    curve = Operate.Dix(curve, oGlblData.time_factor);
    curve = Operate.C600(curve, REG_DT);
    curve = Operate.Div(curve, oGlblData.nm_factor);

    return curve;
}

function MakeCompressionPositive(curve) {
    // Multiply by -1.0 to make compressive force +ve if required as
    // different versions can output the force in opposite directions.
    // Test for this by seeing if the absolute maximum value is -ve

    var max = curve.ymax;
    var min = curve.ymin;

    if (Math.abs(min) > Math.abs(max)) return Operate.Mul(curve, -1.0);

    return curve;
}

function CreateLumbarCurve(
    output_object,
    curve,
    curve_label,
    y_axis_label,
    PERFORMANCE_LIMIT
) {
    // Y force
    // Set labels and style

    set_labels(
        curve,
        curve_label,
        "Time (" + oGlblData.unit_time + ")",
        y_axis_label
    );
    set_line_style(curve, Colour.BLACK);

    // Draw datums

    //green/red boundary
    draw_constant_datums(
        PERFORMANCE_LIMIT,
        null,
        null,
        PERFORMANCE_LIMIT,
        true
    );

    write_image(
        output_object.title,
        output_object.fname,
        curve,
        PERFORMANCE_LIMIT,
        -PERFORMANCE_LIMIT
    );

    output_object.curveList.push(curve.id);
    write_curve("cur", output_object.curveList, output_object.fname);
    write_curve("csv", output_object.curveList, output_object.fname);

    // Calc values

    var result = Math.max(curve.ymax, -curve.ymin); //take the absolute peak value

    var score = 0;

    if (result > PERFORMANCE_LIMIT) score = PELVIS_EXCEEDANCE_MODIFIER;

    // Remove all curves and datums

    remove_all_curves_and_datums();

    return { value: result.toFixed(3), score: score.toFixed(3) };
}
