/// <reference path="C:/SOURCE14/this_dir/this_js_api/this.intellisense.js" />
/// <reference path="C:/SOURCE14/reporter_dir/library/templates/scripts/this_common1.js" />
/// <reference path="C:/SOURCE14/reporter_dir/library/templates/scripts/this_common_EuroNCAP.js" />


// Constants used throughout the script

var REG_DT     = 0.00001; // dt for regularising curves (in seconds)
var HIC_WINDOW = 0.015;   // HIC time window (in seconds)
var TMS_PERIOD = 0.003;   // 3ms period (in seconds)

// Head assessment constants

var PEAK_HEAD_ACCN_SOFT = 80.0;       // 80g - soft contact

var PEAK_HEAD_ACCN_LO_LIMIT = 80.0;   // 80g  } limits if no steering wheel airbag present
var PEAK_HEAD_ACCN_HI_LIMIT = 120.0;  // 120g }

var HIC_LO_LIMIT = 500.0;
var HIC_HI_LIMIT = 700.0;

var TMS_LO_LIMIT   = 72.0;
var TMS_HI_LIMIT   = 80.0;
var TMS_LO_LIMIT_2 = 65.0;  // } Different 3ms limits for driver 
var TMS_HI_LIMIT_2 = 80.0;  // } if no steering wheel airbag present

var HEAD_FA_DISP_LO_LIMIT   = 90;   // 90mm  rearward displacement
var HEAD_FA_DISP_HI_LIMIT   = 110;  // 110mm rearward displacement
var HEAD_LAT_DISP_LO_LIMIT  = 90;   // 90mm  lateral displacement
var HEAD_LAT_DISP_HI_LIMIT  = 110;  // 110mm lateral displacement
var HEAD_VERT_DISP_LO_LIMIT = 72;   // 72mm  upward displacement
var HEAD_VERT_DISP_HI_LIMIT = 88;   // 88mm  upward displacement

var STEERING_WHEEL_LO_SCORE = -1.0;
var STEERING_WHEEL_HI_SCORE = 0.0;

var HEAD_LO_SCORE   = 0.0;
var HEAD_HI_SCORE   = 4.0;
var HEAD_HI_SCORE_2 = 2.0;  // High score limited to 2.0 for driver if no steering wheel airbag


// Neck assessment constants

var NECK_SHR_ADJ_FAC = 17.78;   // Factor to multiply shear by to adjust bending moment

// Neck shear limit curves

var NECK_SHR_GOOD     = 1.20;
var NECK_SHR_ADEQUATE = 1.45;
var NECK_SHR_MARGINAL = 1.70;
var NECK_SHR_WEAK     = 1.95;

// Neck tension limit curves

var NECK_TEN_GOOD     = 1.70;
var NECK_TEN_ADEQUATE = 2.01;
var NECK_TEN_MARGINAL = 2.31;
var NECK_TEN_WEAK     = 2.62;

var NECK_MOM_GOOD     = -36.0;
var NECK_MOM_ADEQUATE = -40.33;
var NECK_MOM_MARGINAL = -44.67;
var NECK_MOM_WEAK     = -49.0;

var NECK_SHR_CAPPING_LIMIT = 2.7;
var NECK_TEN_CAPPING_LIMIT = 2.9;
var NECK_MOM_CAPPING_LIMIT = -57.0;

var NECK_LO_SCORE   = 0.0;
var NECK_HI_SCORE   = 4.0;

var NECK_SHEAR_REAR_PASSENGER_HI_SCORE     = 1.0;
var NECK_TENSION_REAR_PASSENGER_HI_SCORE   = 1.0;
var NECK_EXTENSION_REAR_PASSENGER_HI_SCORE = 2.0;

var NECK_SHEAR_FRONT_PASSENGER_HI_SCORE     = 1.0;
var NECK_TENSION_FRONT_PASSENGER_HI_SCORE   = 1.0;
var NECK_EXTENSION_FRONT_PASSENGER_HI_SCORE = 2.0;


// Chest assessment

// V2_LSTC_HIII_5TH_FEMALE and V6_HUMANETICS_HIII_5TH_FEMALE use a linear conversion: D = A*theta

var CHEST_V2_LSTC_ROT_TO_COM_FACTOR       = -104.0;   // Factor to convert chest transducer rotation to compression (mm)
var CHEST_V6_HUMANETICS_ROT_TO_COM_FACTOR = -87.37;   // Factor to convert chest transducer rotation to compression (mm)

// V7_HUMANETICS_HIII_5TH_FEMALE dummies use a 3rd order polynomial conversion D = A*theta^3 - B*theta^2 - C*theta

var CHEST_V7_ROT_TO_COM_FACTOR_A = -15.61;   // First polynomial factor to convert chest transducer rotation to compression (mm)
var CHEST_V7_ROT_TO_COM_FACTOR_B =  33.84;   // Second polynomial factor to convert chest transducer rotation to compression (mm)
var CHEST_V7_ROT_TO_COM_FACTOR_C =  81.53;   // Third polynomial factor to convert chest transducer rotation to compression (mm)


var CHEST_VC_A = 1.3;           // Viscous Criterion A constant (for m/s)
var CHEST_VC_B = 0.229;         // Viscous Criterion B constant (for m/s)
var CHEST_VC_METHOD = "ECER95"; // Viscous Criterion calculation method

var CHEST_COM_GOOD     = 18.00;  // mm
var CHEST_COM_ADEQUATE = 26.00;  // mm
var CHEST_COM_MARGINAL = 34.00;  // mm
var CHEST_COM_WEAK     = 42.00;  // mm

var CHEST_VC_GOOD     = 0.5;   // 0.50 m/s
var CHEST_VC_ADEQUATE = 0.67;  // 0.67 m/s
var CHEST_VC_MARGINAL = 0.83;  // 0.83 m/s
var CHEST_VC_WEAK     = 1.0;   // 1.00 m/s

var CHEST_LO_SCORE = 0.0;
var CHEST_HI_SCORE = 4.0;


// Knee, Femur and Pelvis assessments

// Femur limit curves

var FEMUR_COM_GOOD     = 2.6;
var FEMUR_COM_ADEQUATE = 3.8;
var FEMUR_COM_MARGINAL = 5.0;
var FEMUR_COM_WEAK     = 6.2;

var KNEE_FEMUR_PELVIS_LO_SCORE = 0.0;
var KNEE_FEMUR_PELVIS_HI_SCORE = 4.0;



if(File.Exists(fname_csv) && File.IsFile(fname_csv))
{
    // Do the common setup of assigning values to the oGlblData object,
    // setting the T/HIS graph up and reading in the model.

    var oGlblData = new Object();

    var f_vars = do_common_single_analysis_setup(oGlblData);

    // For each body region, do the calculation, writing results to the variables file.

    var fpass_head_exists = do_head_rating_calc(f_vars);

    var fpass_neck_exists = do_neck_rating_calc(f_vars);

    var fpass_chest_exists = do_chest_rating_calc(f_vars);

    var fpass_knee_femur_pelvis_exists = do_knee_femur_pelvis_rating_calc(f_vars);

    if(fpass_head_exists == "FALSE" && fpass_neck_exists == "FALSE" && fpass_chest_exists == "FALSE" && fpass_knee_femur_pelvis_exists == "FALSE")
    {
        fpass_exists = "FALSE";
    }

    else
    {
        fpass_exists = "TRUE";
    }

    write_variable(f_vars, "FRONT_PASSENGER_EXISTS", fpass_exists, "Front Passenger Exists", "String");
    
    f_vars.Close();
}



function do_head_rating_calc(f)
{
    // Calculates the head scores for both driver and passenger and writes
    // relevant values into the Reporter csv variables file <f>.
    Message("Doing head rating calculation...");

    var dummy = [];
    dummy.push(new DummyData("Driver", oGlblData.driver_head_node, "node", oGlblData.driver_head_node_y, "node_y", oGlblData.driver_head_node_z, "node_z"));
    dummy.push(new DummyData("Front Passenger", oGlblData.front_passenger_head_node, "node", oGlblData.front_passenger_head_node_y, "node_y", oGlblData.front_passenger_head_node_z, "node_z"));
    dummy.push(new DummyData("Rear Passenger", oGlblData.rear_passenger_head_node, "node", oGlblData.rear_passenger_head_node_y, "node_y", oGlblData.rear_passenger_head_node_z, "node_z"));    

    // Two passes, one for driver, one for passenger

    var fpass_head_exists = "TRUE";

    for(var i=0; i<dummy.length; i++)
    {
        // Capping limit

        var capping_limit = "FALSE";
        var output_data = {};

        output_data.acc = new OutputData(dummy[i].chart_label + " Head Acceleration",         images_dir + "/" + dummy[i].variable_label + "_Head_Acceleration");
        output_data.int = new OutputData(dummy[i].chart_label + " Steering Column Intrusion", images_dir + "/" + dummy[i].variable_label + "_Steering_Column_Intrusion");

        if(dummy[i].node_id != undefined)
        { 

            // Read in acceleration X, Y and Z
            var one_node = true;

            // If nodes for three accelerometer positions defined, use all three, otherwise, read all components from same node
            if(dummy[i].node_y_id != undefined && dummy[i].node_z_id != undefined && dummy[i].node_y_id != 0 && dummy[i].node_z_id != 0)
            {
                var c_acc = read_xyz_accelerations(1, dummy[i].node_id, dummy[i].node_y_id, dummy[i].node_z_id);
                one_node = false;
            }
            else
            {
                var c_acc = read_xyz_accelerations(1, dummy[i].node_id);
            }

            // Check that the curves read in - i.e. the node id and component are valid
            // Create a blank image to let the user know.

            // Generate a relevant message for the blank image depending on which acceleration data are missing
            if(!c_acc.ax || !c_acc.ay || !c_acc.az)
            {
                var blank_image_msg;
                // If only one node supplied, just report that node ID
                if(one_node == true)
                {
                    blank_image_msg = "NO ACCEL DATA (NODE " + dummy[i].node_id + ") ";
                }

                // else report each of the node IDs that were invalid
                else
                {
                    var acc_comp_lst = [];
                    var acc_id_lst = [];
                    if(!c_acc.ax)
                    {
                        acc_comp_lst.push(" X");
                        acc_id_lst.push(" "+ dummy[i].node_id);
                    }
                    if(!c_acc.ay)
                    {
                        acc_comp_lst.push(" Y");
                        acc_id_lst.push(" "+ dummy[i].node_y_id);
                    }
                    if(!c_acc.az)
                    {
                        acc_comp_lst.push(" Z");
                        acc_id_lst.push(" "+ dummy[i].node_z_id);
                    }
                    blank_image_msg = "NO" + acc_comp_lst + " ACCEL DATA (NODE(s) " + acc_id_lst + ") ";
                }
                create_blank_image(blank_image_msg + output_data.acc.title, output_data.acc.fname);

                if(dummy[i].chart_label == "Driver")
                {
                    create_blank_image(blank_image_msg + output_data.int.title, output_data.int.fname);
                }
            }

            else
            {
                // Convert to g from model units

                var c_g = convert_xyz_acc_to_g(c_acc);

                // C1000 filter (convert to seconds first)

                var c_gx_c1000 = convert_to_s_and_filter(c_g.gx, Operate.C1000, REG_DT);
                var c_gy_c1000 = convert_to_s_and_filter(c_g.gy, Operate.C1000, REG_DT);
                var c_gz_c1000 = convert_to_s_and_filter(c_g.gz, Operate.C1000, REG_DT);

                // Vector combine

                var c_vec_g = Operate.Vec(c_gx_c1000, c_gy_c1000, c_gz_c1000);

                // Remove all curves and datums

                remove_all_curves_and_datums();

                // Convert from seconds back to model time

                c_vec_g = Operate.Mux(c_vec_g, oGlblData.time_factor);

                // Get peak acceleration (in g)

                DialogueInput("/AU");
                var peak_accn = c_vec_g.ymax;

                // Get HIC15 value

                var hic = Operate.Hic(c_vec_g, HIC_WINDOW, 1.0);

                // Get 3ms exceedence

                Operate.Tms(c_vec_g, TMS_PERIOD*oGlblData.time_factor);
                var tms = c_vec_g.tms;

                // Set label and style

                set_labels(c_vec_g, dummy[i].chart_label + " Head Acceleration Magnitude", "Time (" + oGlblData.unit_time + ")", "Acceleration (g)");
                set_line_style(c_vec_g, Colour.BLACK);

                // Draw limit curves => up to final time

                var p = c_vec_g.GetPoint(c_vec_g.npoints);
                if((dummy[i].chart_label == "Driver" && oGlblData.steering_wheel_airbag == "YES") || dummy[i].chart_label == "Rear Passenger") 
                {
                    draw_head_limits(p[0], output_data.acc, true);   // Normal limits
                }
                
                else 
                {
                    draw_head_limits(p[0], output_data.acc, false);  // Limits for driver with no airbag
                }

                // Create image and curve files

                write_image(output_data.acc.title, output_data.acc.fname);

                output_data.acc.curveList.push(c_vec_g.id);
                write_curve("cur", output_data.acc.curveList, output_data.acc.fname);
                write_curve("csv", output_data.acc.curveList, output_data.acc.fname);

                // Remove all curves and datums

                remove_all_curves_and_datums();

                // Steering wheel displacements - driver only

                if(dummy[i].chart_label == "Driver")
                {
                    var head_steering_wheel_fa_disp   = 0.0;
                    var head_steering_wheel_lat_disp  = 0.0;
                    var head_steering_wheel_vert_disp = 0.0;

                    var head_steering_wheel_fa_mod   = 0.0;
                    var head_steering_wheel_lat_mod  = 0.0;
                    var head_steering_wheel_vert_mod = 0.0;

                    var spring;

                    for(var sp=0; sp<3; sp++)
                    {
                        if(sp == 0) spring = oGlblData.steering_col_intrusion_spring_x; // Fore-aft spring
                        else if(sp == 1) spring = oGlblData.steering_col_intrusion_spring_y; // Lateral spring
                        else if(sp == 2) spring = oGlblData.steering_col_intrusion_spring_z; // Vertical spring

                        if(spring != undefined)
                        {
                            var c_el = read_data("SPRING TR", 0, spring, "ET");

                            if(c_el)
                            {

                                // Convert to mm, take final value and calculate modifier

                                var c_el_mm = Operate.Div(c_el, oGlblData.mm_factor);

                                var c_fa;
                                var c_lat;
                                var c_vert;

                                if(sp == 0)
                                {
                                    // Fore/aft intrusion = -ve elongation is rearward so multiply by -1.0 first

                                    c_fa = Operate.Mul(c_el_mm, -1);

                                    var p = c_fa.GetPoint(c_fa.npoints);
                                    head_steering_wheel_fa_disp = p[1];
                                    head_steering_wheel_fa_mod = sliding_scale(head_steering_wheel_fa_disp, 
                                            HEAD_FA_DISP_LO_LIMIT, HEAD_FA_DISP_HI_LIMIT,
                                            STEERING_WHEEL_HI_SCORE, STEERING_WHEEL_LO_SCORE);
                                }
                                else if(sp == 1)
                                {
                                    // Lateral intrusion - use absolute value (left or right)

                                    c_lat = c_el_mm;

                                    var p = c_lat.GetPoint(c_lat.npoints);
                                    head_steering_wheel_lat_disp = p[1];
                                    head_steering_wheel_lat_mod = sliding_scale(Math.abs(head_steering_wheel_lat_disp), 
                                            HEAD_LAT_DISP_LO_LIMIT, HEAD_LAT_DISP_HI_LIMIT,
                                            STEERING_WHEEL_HI_SCORE, STEERING_WHEEL_LO_SCORE);
                                }
                                else if(sp == 2)
                                {
                                    // Vertical intrusion - +ve elongation is upward

                                    c_vert = c_el_mm;

                                    var p = c_vert.GetPoint(c_vert.npoints);
                                    head_steering_wheel_vert_disp = p[1];
                                    head_steering_wheel_vert_mod = sliding_scale(head_steering_wheel_vert_disp, 
                                            HEAD_VERT_DISP_LO_LIMIT, HEAD_VERT_DISP_HI_LIMIT,
                                            STEERING_WHEEL_HI_SCORE, STEERING_WHEEL_LO_SCORE);
                                }
                            }
                        }
                    }

                    // Overall steering wheel modifier - minimum of three modifiers

                    var head_steering_wheel_mod = Math.min(head_steering_wheel_fa_mod, head_steering_wheel_lat_mod, head_steering_wheel_vert_mod);

                    // Draw intrusion curves

                    // Remove all curves and datums

                    remove_all_curves_and_datums();

                    // Set labels and styles

                    if(c_fa)   set_labels(c_fa,   dummy[i].chart_label + " Fore/Aft Intrusion (+ve rearward)", "Time (" + oGlblData.unit_time + ")", "Intrusion (mm)");
                    if(c_lat)  set_labels(c_lat,  dummy[i].chart_label + " Lateral Intrusion",                 "Time (" + oGlblData.unit_time + ")", "Intrusion (mm)");
                    if(c_vert) set_labels(c_vert, dummy[i].chart_label + " Vertical Intrusion (+ve rearward)", "Time (" + oGlblData.unit_time + ")", "Intrusion (mm)");

                    if(c_fa)   set_line_style(c_fa,   Colour.RED);
                    if(c_lat)  set_line_style(c_lat,  Colour.BLUE);
                    if(c_vert) set_line_style(c_vert, Colour.GREEN);

                    // Create image and curve files

                    if(c_fa)   c_fa.AddToGraph();
                    if(c_lat)  c_lat.AddToGraph();
                    if(c_vert) c_vert.AddToGraph();

                    if(c_fa)   output_data.int.curveList.push(c_fa.id);
                    if(c_lat)  output_data.int.curveList.push(c_lat.id);
                    if(c_vert) output_data.int.curveList.push(c_vert.id);

                    var title;

                    if(c_fa && c_lat && c_vert) title = output_data.int.title;
                    else                        title = "SOME SPRING IDs MISSING/DO NOT EXIST FOR " + output_data.int.title;

                    write_image(title, output_data.int.fname);

                    write_curve("cur", output_data.int.curveList, output_data.int.fname);
                    write_curve("csv", output_data.int.curveList, output_data.int.fname);


                }

                // Calculate head points

                // Driver with steering wheel airbag or passenger

                var peak_accn_score = 0.0;
                var hic_score       = 0.0;
                var tms_score       = 0.0;
                var score           = 0.0;

                if((dummy[i].chart_label == "Driver" && oGlblData.steering_wheel_airbag == "YES") || dummy[i].chart_label == "Front Passenger")
                {       
                    if(peak_accn < PEAK_HEAD_ACCN_SOFT)
                    {
                        // 'Soft contact' -> 4.0
                        peak_accn_score = score = HEAD_HI_SCORE;
                    }
                    else
                    {
                        // 'Hard contact' -> minimum of hic and tms score

                        hic_score = sliding_scale(hic, HIC_LO_LIMIT, HIC_HI_LIMIT, HEAD_HI_SCORE, HEAD_LO_SCORE);
                        tms_score = sliding_scale(tms, TMS_LO_LIMIT, TMS_HI_LIMIT, HEAD_HI_SCORE, HEAD_LO_SCORE);

                        score = Math.min(hic_score, tms_score);

                        // Capping limit should be applied to overall score if these are zero

                        if(hic_score == 0.0 || tms_score == 0.0) capping_limit = "TRUE";
                    }
                }
                
                else if(dummy[i].chart_label == "Driver" && oGlblData.steering_wheel_airbag == "NO")
                {
                    // Driver with no steering wheel airbag - score is out of 2.0
                    // and is the minimum of the peak accn, hic and tms scores 

                    peak_accn_score = sliding_scale(peak_accn, PEAK_HEAD_ACCN_LO_LIMIT, PEAK_HEAD_ACCN_HI_LIMIT, HEAD_HI_SCORE_2, HEAD_LO_SCORE);
                    if(hic < HIC_HI_LIMIT) hic_score = HEAD_HI_SCORE_2;
                    tms_score = sliding_scale(tms, TMS_LO_LIMIT_2, TMS_HI_LIMIT_2, HEAD_HI_SCORE_2, HEAD_LO_SCORE);

                    score = Math.min(peak_accn_score, hic_score, tms_score);

                    // Capping limit should be applied to overall score if these are zero

                    if(peak_accn_score == 0.0 || hic_score == 0.0 || tms_score == 0.0) capping_limit = "TRUE";
                }

                // Rear passenger

                else if(dummy[i].chart_label == "Rear Passenger")
                {
                    if(peak_accn < PEAK_HEAD_ACCN_SOFT)
                    {
                        // 'Soft contact' only 3ms exceedence measured

                        score = tms_score = sliding_scale(tms, TMS_LO_LIMIT, TMS_HI_LIMIT, HEAD_HI_SCORE, HEAD_LO_SCORE);
                    }
                    else
                    {
                        // 'Hard contact' -> minimum of hic and tms score

                        hic_score = sliding_scale(hic, HIC_LO_LIMIT, HIC_HI_LIMIT, HEAD_HI_SCORE, HEAD_LO_SCORE);
                        tms_score = sliding_scale(tms, TMS_LO_LIMIT, TMS_HI_LIMIT, HEAD_HI_SCORE, HEAD_LO_SCORE);


                        score = Math.min(hic_score, tms_score);

                        // Capping limit should be applied to overall score if these are zero

                        if(hic_score == 0.0 || tms_score == 0.0) capping_limit = "TRUE";
                    }
                }
                else
                {
                    Window.Error("Error!", "Unexpected Dummy type in head rating calc.");
                }

                // Add on steering wheel modifier to get final score (before applying subjective modifiers, this
                // is done in a script in the Reporter template).

                if(dummy[i].chart_label == "Driver") score += head_steering_wheel_mod;

                // Bound score between limits

                if(((dummy[i].chart_label == "Driver" && oGlblData.steering_wheel_airbag == "YES") || dummy[i].chart_label == "Front Passenger") || dummy[i].chart_label == "Rear Passenger")
                {
                    if(score < HEAD_LO_SCORE) score = HEAD_LO_SCORE;
                    if(score > HEAD_HI_SCORE) score = HEAD_HI_SCORE;
                }
                else
                {
                    if(score < HEAD_LO_SCORE)   score = HEAD_LO_SCORE;
                    if(score > HEAD_HI_SCORE_2) score = HEAD_HI_SCORE_2;
                }

                // Write variables to csv file for Reporter

                if(dummy[i].chart_label == "Driver")
                {
                    // Driver steering wheel airbag

                    var value;
                    if(oGlblData.steering_wheel_airbag == "YES") value = "Yes";
                    else                                         value = "No";

                    write_variable(f, "DRIVER_STEERING_WHEEL_AIRBAG", value, "Driver steering wheel airbag present", "String");
                }

                // Peak head acceleration and score

                write_variable(f, dummy[i].variable_label.toUpperCase() + "_PEAK_HEAD_ACCN",       peak_accn.toFixed(2),       "Peak head acceleration value", "String");
                write_variable(f, dummy[i].variable_label.toUpperCase() + "_PEAK_HEAD_ACCN_SCORE", peak_accn_score.toFixed(3), "Peak head acceleration score", "String");

                // HIC value and score

                write_variable(f, dummy[i].variable_label.toUpperCase() + "_HEAD_HIC15",       hic.toFixed(2),       "HIC15 value", "String");
                write_variable(f, dummy[i].variable_label.toUpperCase() + "_HEAD_HIC15_SCORE", hic_score.toFixed(3), "HIC15 score", "String");

                // 3ms value and score

                write_variable(f, dummy[i].variable_label.toUpperCase() + "_HEAD_3MS",       tms.toFixed(2),       "3ms value", "String");
                write_variable(f, dummy[i].variable_label.toUpperCase() + "_HEAD_3MS_SCORE", tms_score.toFixed(3), "3ms score", "String");

                // Steering wheel displacement values and modifier value

                if(dummy[i].chart_label == "Driver")
                {
                    write_variable(f, "DRIVER_STEERING_WHEEL_FA_DISP",   head_steering_wheel_fa_disp.toFixed(1),   "Steering wheel fore/aft displacement", "String");
                    write_variable(f, "DRIVER_STEERING_WHEEL_VERT_DISP", head_steering_wheel_vert_disp.toFixed(1), "Steering wheel vertical displacement", "String");
                    write_variable(f, "DRIVER_STEERING_WHEEL_LAT_DISP",  head_steering_wheel_lat_disp.toFixed(1),  "Steering wheel lateral displacement",  "String");
                    write_variable(f, "DRIVER_STEERING_WHEEL_DISP_MOD",  head_steering_wheel_mod.toFixed(1),       "Steering wheel displacement modifier", "String");
                }

                // Final score value (before subjective modifiers) - write a variable and store on <oGlblData> object

                write_variable(f, dummy[i].variable_label.toUpperCase() + "_HEAD_PRE_MODS_SCORE",   score.toFixed(3),   "Final score", "String");

                // Should capping limit be applied

                write_variable(f, dummy[i].variable_label.toUpperCase() + "_HEAD_CAPPING_LIMIT", capping_limit, dummy[i].chart_label + " Head capping limit", "String");
            }
        }
        else
        {

            // No node defined - variables should be set to blank by first script in template.
            // Create a blank image to let the user know.
            if(dummy[i].chart_label == "Front Passenger") fpass_head_exists = "FALSE";

            create_blank_image("NO NODE ID DEFINED FOR " + output_data.acc.title, output_data.acc.fname);

            if(dummy[i].chart_label == "Driver")
            {
                create_blank_image("NO NODE ID DEFINED FOR " + output_data.int.title, output_data.int.fname);
            }
        }
    }
    return fpass_head_exists;  
}



function do_neck_rating_calc(f)
{
    // Calculates the head scores for both driver and passenger and writes
    // relevant values into the Reporter csv variables file <f>.

    // Two passes, one for driver, one for passenger
    Message("Doing neck rating calculation...");

    var dummy = [];
    dummy.push(new DummyData("Driver", oGlblData.driver_neck_loadcell, "beam", oGlblData.driver_dummy, "version"));
    dummy.push(new DummyData("Front Passenger", oGlblData.front_passenger_neck_loadcell, "beam", oGlblData.front_passenger_dummy, "version"));
    dummy.push(new DummyData("Rear Passenger", oGlblData.rear_passenger_neck_loadcell, "beam", oGlblData.rear_passenger_dummy, "version"));

    //var dummy = [];
    //dummy.push("Driver");
    //dummy.push("Rear Passenger");
    //dummy.push("Front Passenger");

    //var output_dummy = [];
    //output_dummy.push("Driver");
    //output_dummy.push("Rear_Passenger");
    //output_dummy.push("Front_Passenger");

    var fpass_neck_exists = 1;
    for(var i = 0; i<dummy.length; i++)
    {
        // Capping limit

        var capping_limit = "FALSE";

        //var beam;
        //var dummy;

        //if(dummy[i].chart_label == "Driver") { beam = oGlblData.driver_neck_loadcell;    dummy = "Driver";    }
        //if(dummy[i].chart_label == "Rear Passenger") { beam = oGlblData.passenger_neck_loadcell; dummy = "Passenger"; }

        // Data for images that we will create

        var output_data = {};

        output_data.shr = new OutputData(dummy[i].chart_label + " Neck Shear",          images_dir + "/" + dummy[i].variable_label + "_Neck_Shear");
        output_data.ten = new OutputData(dummy[i].chart_label + " Neck Tension",        images_dir + "/" + dummy[i].variable_label + "_Neck_Tension");
        output_data.mom = new OutputData(dummy[i].chart_label + " Neck Bending Moment", images_dir + "/" + dummy[i].variable_label + "_Neck_Moment");

        if( dummy[i].beam_id != undefined )
        {
            // Read in beam shear, tension and  MY
            // For Humanetics dummies this is measured from a beam;
            // for LSTC dummies this is measured from a cross section
            switch (dummy[i].version)
            {
                case "V6_HUMANETICS_HIII_5TH_FEMALE":
                case "V7_HUMANETICS_HIII_5TH_FEMALE":
                    var c_shr = read_data("BEAM BASIC", 0, dummy[i].beam_id, "NX");  // NX is shear
                    var c_ten = read_data("BEAM BASIC", 0, dummy[i].beam_id, "NZ");
                    var c_mom = read_data("BEAM BASIC", 0, dummy[i].beam_id, "MY");
                    var entity_str = "BEAM"; // Used for error message below
                    break;

                case "LSTC HIII 5F v2.0":
                case "LSTC HIII 5F Fast v2.0":
                case "Humanetics HIII 5F v2.02":
                    var c_shr = read_data("SECTION", 0, dummy[i].beam_id, "FX");  // FX is shear
                    var c_ten = read_data("SECTION", 0, dummy[i].beam_id, "FZ");
                    var c_mom = read_data("SECTION", 0, dummy[i].beam_id, "MY");
                    var entity_str = "SECTION"; // Used for error message below
                    break;

                default:
                    ErrorMessage("Unsupported dummy \""+dummy[i].version+"\"in do_neck_rating_calc.");
                    write_blank_images(output_data, "UNSUPPORTED DUMMY \""+dummy[i].version+"\"");
                    return;
                    break;
            }

            // Check that the curves read in - i.e. the beam id and component are valid
            // Create blank images to let the user know.

            if(!c_shr || !c_ten || !c_mom)
            {
                write_blank_images(output_data, "NO DATA FOR " + entity_str + " " + dummy[i].beam_id);
            }
            else
            {
                // Convert forces to kN and
                //         moment to Nm from model units

                var c_shr_kn = Operate.Div(c_shr, oGlblData.kn_factor);
                var c_ten_kn = Operate.Div(c_ten, oGlblData.kn_factor);
                var c_mom_nm = Operate.Div(c_mom, oGlblData.nm_factor);

                // C1000/C600 filter (convert to seconds first)

                var c_shr_c1000 = convert_to_s_and_filter(c_shr_kn, Operate.C1000, REG_DT);
                var c_ten_c1000 = convert_to_s_and_filter(c_ten_kn, Operate.C1000, REG_DT);
                var c_mom_c600  = convert_to_s_and_filter(c_mom_nm, Operate.C600 , REG_DT);

                // Transpose the bending moment to the occupital condoyle moment

                var c_shr_adj = Operate.Mul(c_shr_c1000, NECK_SHR_ADJ_FAC);
                var c_mom_adj = Operate.Sub(c_mom_c600, c_shr_adj);

                // Remove all curves and datums

                remove_all_curves_and_datums();

                // Convert from seconds back to model time

                c_shr_c1000 = Operate.Mux(c_shr_c1000, oGlblData.time_factor);
                c_ten_c1000 = Operate.Mux(c_ten_c1000, oGlblData.time_factor);
                c_mom_adj   = Operate.Mux(c_mom_adj,   oGlblData.time_factor);

                // Set labels and style

                set_labels(c_shr_c1000, dummy[i].chart_label + " Neck Shear",                    "Time (" + oGlblData.unit_time + ")", "Force (kN)");
                set_labels(c_ten_c1000, dummy[i].chart_label + " Neck Tension",                  "Time (" + oGlblData.unit_time + ")", "Force (kN)");
                set_labels(c_mom_adj,   dummy[i].chart_label + " Neck Extension Bending Moment", "Time (" + oGlblData.unit_time + ")", "Bending Moment (Nm)");

                set_line_style(c_shr_c1000, Colour.BLACK);
                set_line_style(c_ten_c1000, Colour.BLACK);
                set_line_style(c_mom_adj,   Colour.BLACK);

                // Remove all curves and datums

                remove_all_curves_and_datums();

                // Draw neck shear datums

                draw_neck_shear_datums();

                // Create image and curve files of neck shear curves

                c_shr_c1000.AddToGraph();
                write_image(output_data.shr.title, output_data.shr.fname, c_shr_c1000, NECK_SHR_MARGINAL, -NECK_SHR_MARGINAL);

                output_data.shr.curveList.push(c_shr_c1000.id);
                write_curve("cur", output_data.shr.curveList, output_data.shr.fname);
                write_curve("csv", output_data.shr.curveList, output_data.shr.fname);

                // Remove all curves and datums

                remove_all_curves_and_datums();

                // Draw neck tension datums

                draw_constant_datums(NECK_TEN_GOOD, NECK_TEN_ADEQUATE, NECK_TEN_MARGINAL, NECK_TEN_WEAK);

                // Create image and curve files of neck tension curves

                c_ten_c1000.AddToGraph();
                write_image(output_data.ten.title, output_data.ten.fname, c_ten_c1000, NECK_TEN_WEAK);

                output_data.ten.curveList.push(c_ten_c1000.id);
                write_curve("cur", output_data.ten.curveList, output_data.ten.fname);
                write_curve("csv", output_data.ten.curveList, output_data.ten.fname);

                // Remove all curves and datums

                remove_all_curves_and_datums();

                // Draw moment limit datums

                draw_constant_datums(NECK_MOM_GOOD, NECK_MOM_ADEQUATE, NECK_MOM_MARGINAL, NECK_MOM_WEAK);

                // Create image and curve files of moment curves

                c_mom_adj.AddToGraph();
                write_image(output_data.mom.title, output_data.mom.fname, c_mom_adj, NECK_MOM_WEAK);

                output_data.mom.curveList.push(c_mom_adj.id);
                write_curve("cur", output_data.mom.curveList, output_data.mom.fname);
                write_curve("csv", output_data.mom.curveList, output_data.mom.fname);

                // Remove all curves and datums

                remove_all_curves_and_datums();

                // Calc values

                var shr_pos_results;
                var shr_neg_results;
                var ten_results;
                var mom_results;

                if(dummy[i].chart_label == "Driver")
                {
                    shr_pos_results = get_constant_limit_results(c_shr_c1000,  NECK_SHR_GOOD,  NECK_SHR_WEAK, NECK_LO_SCORE, NECK_HI_SCORE);
                    shr_neg_results = get_constant_limit_results(c_shr_c1000, -NECK_SHR_GOOD, -NECK_SHR_WEAK, NECK_LO_SCORE, NECK_HI_SCORE);
                    ten_results     = get_constant_limit_results(c_ten_c1000,  NECK_TEN_GOOD,  NECK_TEN_WEAK, NECK_LO_SCORE, NECK_HI_SCORE);
                    mom_results     = get_constant_limit_results(c_mom_adj,    NECK_MOM_GOOD,  NECK_MOM_WEAK, NECK_LO_SCORE, NECK_HI_SCORE);
                }
                else if (dummy[i].chart_label == "Rear Passenger")
                {
                    shr_pos_results = get_constant_limit_results(c_shr_c1000,  NECK_SHR_GOOD,  NECK_SHR_WEAK, NECK_LO_SCORE, NECK_SHEAR_REAR_PASSENGER_HI_SCORE);
                    shr_neg_results = get_constant_limit_results(c_shr_c1000, -NECK_SHR_GOOD, -NECK_SHR_WEAK, NECK_LO_SCORE, NECK_SHEAR_REAR_PASSENGER_HI_SCORE);
                    ten_results     = get_constant_limit_results(c_ten_c1000,  NECK_TEN_GOOD,  NECK_TEN_WEAK, NECK_LO_SCORE, NECK_TENSION_REAR_PASSENGER_HI_SCORE);
                    mom_results     = get_constant_limit_results(c_mom_adj,    NECK_MOM_GOOD,  NECK_MOM_WEAK, NECK_LO_SCORE, NECK_EXTENSION_REAR_PASSENGER_HI_SCORE);
                }
                else if (dummy[i].chart_label == "Front Passenger")
                {
                    shr_pos_results = get_constant_limit_results(c_shr_c1000,  NECK_SHR_GOOD,  NECK_SHR_WEAK, NECK_LO_SCORE, NECK_HI_SCORE);
                    shr_neg_results = get_constant_limit_results(c_shr_c1000, -NECK_SHR_GOOD, -NECK_SHR_WEAK, NECK_LO_SCORE, NECK_HI_SCORE);
                    ten_results     = get_constant_limit_results(c_ten_c1000,  NECK_TEN_GOOD,  NECK_TEN_WEAK, NECK_LO_SCORE, NECK_HI_SCORE);
                    mom_results     = get_constant_limit_results(c_mom_adj,    NECK_MOM_GOOD,  NECK_MOM_WEAK, NECK_LO_SCORE, NECK_HI_SCORE);
                }

                // Write variables

                // SHEAR

                var shr_results;

                if(Math.abs(shr_pos_results[1]) > Math.abs(shr_neg_results[1]))
                {
                    shr_results = shr_pos_results;
                }
                else
                {
                    shr_results = shr_neg_results;
                    shr_results[1] *= -1.0;   // So the value is displayed as negative
                }

                write_variable(f, dummy[i].variable_label.toUpperCase() + "_NECK_SHEAR_SCORE", shr_results[0].toFixed(3), "Neck shear score",    "String");
                write_variable(f, dummy[i].variable_label.toUpperCase() + "_NECK_SHEAR",       shr_results[1].toFixed(3), "Neck shear level",    "String");

                // TENSION
                write_variable(f, dummy[i].variable_label.toUpperCase() + "_NECK_TENSION_SCORE", ten_results[0].toFixed(3), "Neck tension score",    "String");
                write_variable(f, dummy[i].variable_label.toUpperCase() + "_NECK_TENSION",       ten_results[1].toFixed(3), "Neck tension level",    "String");

                // MOMENT
                write_variable(f, dummy[i].variable_label.toUpperCase() + "_NECK_EXTENSION_SCORE", mom_results[0].toFixed(3), "Neck extension score", "String");
                write_variable(f, dummy[i].variable_label.toUpperCase() + "_NECK_BENDING_MOMENT",  mom_results[1].toFixed(3), "Neck bending moment",  "String");
                // Capping limit should be applied to overall score if any of these exceed the capping limit value
                // Note that this is different to the lower performance limit in this case.

                if(Math.abs(shr_results[1]) > NECK_SHR_CAPPING_LIMIT ||
                        ten_results[1]  > NECK_TEN_CAPPING_LIMIT ||
                        mom_results[1]  < NECK_MOM_CAPPING_LIMIT) capping_limit = "TRUE";

                // Should capping limit be applied

                write_variable(f, dummy[i].variable_label.toUpperCase() + "_NECK_CAPPING_LIMIT", capping_limit, dummy[i].chart_label + " Neck capping limit", "String");
            }
        }
        else
        {

            // No node defined - variables should be set to blank by first script in template.
            // Create blank images to let the user know.
            if(dummy[i].chart_label == "Front Passenger") fpass_neck_exists = "FALSE";
            write_blank_images(output_data, "NO BEAM ID DEFINED FOR");
        }
    }
    return fpass_neck_exists;  
}


function do_chest_rating_calc(f)
{
    // Calculates the chest scores for both driver and passenger and writes
    // relevant values into the Reporter csv variables file <f>.

    // Two passes, one for driver, one for passenger
    Message("Doing chest rating calculation...");

    var dummy = [];
    dummy.push(new DummyData("Driver", oGlblData.driver_chest_transducer, "spring", oGlblData.driver_dummy, "version"));
    dummy.push(new DummyData("Front Passenger", oGlblData.front_passenger_chest_transducer, "spring", oGlblData.front_passenger_dummy, "version"));
    dummy.push(new DummyData("Rear Passenger", oGlblData.rear_passenger_chest_transducer, "spring", oGlblData.rear_passenger_dummy, "version"));

    var fpass_chest_exists = "TRUE";

    for(var i = 0; i<dummy.length; i++)
    {
        // Capping limit

        var capping_limit = "FALSE";

        var output_data = {};

        output_data.com = new OutputData(dummy[i].chart_label + " Chest compression",       images_dir + "/" + dummy[i].variable_label + "_Chest_Compression");
        output_data.vc  = new OutputData(dummy[i].chart_label + " Chest Viscous Criterion", images_dir + "/" + dummy[i].variable_label + "_Chest_Viscous_Criterion");

        if( dummy[i].spring_id != undefined )
        {
            // Read in spring rotation

            var c_rot = read_data("SPRING ROT", 0, dummy[i].spring_id, "RT");

            // Check that the curve read in - i.e. the spring id and component are valid
            // Create blank images to let the user know.

            if(!c_rot)
            {
                write_blank_images(output_data, "NO DATA FOR SPRING " + dummy[i].spring_id);
            }
            else
            {
                // Chest compression is calculated by converting the rotation in radians to a deflection using
                // a transfer function.
                //
                // For V2 LSTC and V6 Humanetics dummies and earlier, the conversion is linear
                // For V7 Humanetics dummies and later, the conversion uses a 3rd order polynomial

                var c_disp_mm;

                switch (dummy[i].version)
                {
                    case "LSTC HIII 5F Fast v2.0": 
                    case "LSTC HIII 5F v2.0":
                        c_disp_mm = Operate.Mul(c_rot, CHEST_V2_LSTC_ROT_TO_COM_FACTOR);
                        break;
                    case "V6_HUMANETICS_HIII_5TH_FEMALE":
                        c_disp_mm = Operate.Mul(c_rot, CHEST_V6_HUMANETICS_ROT_TO_COM_FACTOR);
                        break;
                    case "Humanetics HIII 5F v2.02":                   
                    case "V7_HUMANETICS_HIII_5TH_FEMALE":
                        curve_id = Curve.FirstFreeID();

                        c_disp_mm = new Curve(curve_id);
                        var n = c_rot.npoints;

                        for(var j=1; j<=n; j++)
                        {
                            var p_rot = c_rot.GetPoint(j);

                            var time = p_rot[0];
                            var com  = CHEST_V7_ROT_TO_COM_FACTOR_A * p_rot[1] * p_rot[1] * p_rot[1] +
                                CHEST_V7_ROT_TO_COM_FACTOR_B * p_rot[1] * p_rot[1] + 
                                CHEST_V7_ROT_TO_COM_FACTOR_C * p_rot[1];

                            c_disp_mm.AddPoint(time, com);
                        }
                        break;
                    default:
                        ErrorMessage("Unsupported dummy \""+dummy[i].version+"\"in do_chest_rating_calc.");
                        write_blank_images(output_data, "UNSUPPORTED DUMMY \""+dummy[i].version+"\"");
                        return;
                        break;
                }

                // C180 filter (convert to seconds first)

                var c_disp_c180_mm = convert_to_s_and_filter(c_disp_mm, Operate.C180, REG_DT);  // in mm (for compression calc)
                var c_disp_c180_m  = Operate.Div(c_disp_c180_mm, 1000);  // in m  (for viscous criterion calc)

                // Calculate viscous criterion

                c_vc = Operate.Vc(c_disp_c180_m, CHEST_VC_A, CHEST_VC_B, CHEST_VC_METHOD);

                // Remove all curves and datums

                remove_all_curves_and_datums();

                // Convert from seconds back to model time

                c_disp_c180_mm = Operate.Mux(c_disp_c180_mm, oGlblData.time_factor);
                c_vc           = Operate.Mux(c_vc,           oGlblData.time_factor);

                // Set labels and style

                set_labels(c_disp_c180_mm, dummy[i].chart_label + " Chest Compression", "Time (" + oGlblData.unit_time + ")", "Compresion (mm)");
                set_labels(c_vc,           dummy[i].chart_label + " Viscous Criterion", "Time (" + oGlblData.unit_time + ")", "Velocity (m/s)");

                set_line_style(c_disp_c180_mm, Colour.BLACK);
                set_line_style(c_vc,           Colour.BLACK);

                // Remove all curves and datums

                remove_all_curves_and_datums();

                // Draw chest compression datums

                draw_constant_datums(CHEST_COM_GOOD, CHEST_COM_ADEQUATE, CHEST_COM_MARGINAL, CHEST_COM_WEAK);

                // Create image and curve files of chest compression curves

                c_disp_c180_mm.AddToGraph();
                write_image(output_data.com.title, output_data.com.fname, c_disp_c180_mm, CHEST_COM_WEAK);

                output_data.com.curveList.push(c_disp_c180_mm.id);
                write_curve("cur", output_data.com.curveList, output_data.com.fname);
                write_curve("csv", output_data.com.curveList, output_data.com.fname);

                // Remove all curves and datums

                remove_all_curves_and_datums(); 

                // Draw viscous criterion datums

                draw_constant_datums(CHEST_VC_GOOD, CHEST_VC_ADEQUATE, CHEST_VC_MARGINAL, CHEST_VC_WEAK);

                // Create image and curve files of viscous criterion curves

                c_vc.AddToGraph();
                write_image(output_data.vc.title, output_data.vc.fname, c_vc, CHEST_VC_WEAK);

                output_data.vc.curveList.push(c_vc.id);
                write_curve("cur", output_data.vc.curveList, output_data.vc.fname);
                write_curve("csv", output_data.vc.curveList, output_data.vc.fname);

                // Remove all curves and datums

                remove_all_curves_and_datums(); 

                // Calc values

                var chest_com_results = get_constant_limit_results(c_disp_c180_mm, CHEST_COM_GOOD, CHEST_COM_WEAK, CHEST_LO_SCORE, CHEST_HI_SCORE);
                var chest_vc_results  = get_constant_limit_results(c_vc,           CHEST_VC_GOOD,  CHEST_VC_WEAK,  CHEST_LO_SCORE, CHEST_HI_SCORE);

                // Write variables

                write_variable(f, dummy[i].variable_label.toUpperCase() + "_CHEST_COMPRESSION_SCORE",       chest_com_results[0].toFixed(3), "Chest compression score",        "String");        
                write_variable(f, dummy[i].variable_label.toUpperCase() + "_CHEST_COMPRESSION",             chest_com_results[1].toFixed(3), "Chest compression",              "String");   

                write_variable(f, dummy[i].variable_label.toUpperCase() + "_CHEST_VISCOUS_CRITERION_SCORE", chest_vc_results[0].toFixed(3),   "Chest viscous criterion score", "String");        
                write_variable(f, dummy[i].variable_label.toUpperCase() + "_CHEST_VISCOUS_CRITERION",       chest_vc_results[1].toFixed(3),   "Chest viscous criterion",       "String");           

                // Capping limit should be applied to overall score if either of these are zero

                if(chest_com_results[0] == 0.0 || chest_vc_results[0] == 0.0) capping_limit = "TRUE";

                // Should capping limit be applied

                write_variable(f, dummy[i].variable_label.toUpperCase() + "_CHEST_CAPPING_LIMIT", capping_limit, dummy[i].chart_label + " Chest capping limit", "String");
            }
        }
        else
        {
            // No spring defined - variables should be set to blank by first script in template.
            // Create blank images to let the user know.
            if(dummy[i].chart_label == "Front Passenger") fpass_chest_exists = "FALSE";
            write_blank_images(output_data, "NO SPRING ID DEFINED FOR");
        }
    }  
    return fpass_chest_exists;  
}



function do_knee_femur_pelvis_rating_calc(f)
{
    // Calculates the knee, femur and pelvis scores for both driver and passenger and writes
    // relevant values into the Reporter csv variables file <f>.

    // Two passes, one for driver, one for passenger.
    // And for each one do the left and right hand side
    Message("Doing knee/femur/pelvis rating calculation...");

    var dummy = [];
    dummy[0] = [];
    dummy[0].push(new DummyData("Driver", oGlblData.driver_left_femur_loadcell, "beam", oGlblData.driver_left_knee_transducer, "spring", oGlblData.driver_dummy, "version"));
    dummy[0].push(new DummyData("Front Passenger", oGlblData.front_passenger_left_femur_loadcell, "beam", oGlblData.front_passenger_left_knee_transducer, "spring", oGlblData.front_passenger_dummy, "version"));
    dummy[0].push(new DummyData("Rear Passenger", oGlblData.rear_passenger_left_femur_loadcell, "beam", oGlblData.rear_passenger_left_knee_transducer, "spring", oGlblData.rear_passenger_dummy, "version"));

    dummy[1] = [];
    dummy[1].push(new DummyData("Driver",oGlblData.driver_right_femur_loadcell, "beam", oGlblData.driver_right_knee_transducer, "spring", oGlblData.driver_dummy, "version"));
    dummy[1].push(new DummyData("Front Passenger", oGlblData.front_passenger_right_femur_loadcell, "beam", oGlblData.front_passenger_right_knee_transducer, "spring", oGlblData.front_passenger_dummy, "version"));
    dummy[1].push(new DummyData("Rear Passenger", oGlblData.rear_passenger_right_femur_loadcell, "beam", oGlblData.rear_passenger_right_knee_transducer, "spring", oGlblData.rear_passenger_dummy, "version"));

    var fpass_knee_femur_pelvis_exists = "TRUE";

    for(var i = 0; i<dummy[0].length; i++)
    {
        for(var lr=0; lr<2; lr++)
        {
            var side;

            if(lr == 0) side = "Left";
            else        side = "Right";

            var output_data = {};

            output_data.com_exc = new OutputData(dummy[lr][i].chart_label + " " + side + " Femur Compression", images_dir + "/" + dummy[lr][i].variable_label + "_" + side + "_Femur_Compression");

            // Femur compression forces

            if(dummy[lr][i].beam_id != undefined)
            {
                // Read in beam longitudinal force
                // For Humanetics dummies this is measured from a beam;
                // for LSTC dummies this is measured from a cross section
                switch (dummy[lr][i].version)
                {
                    case "V6_HUMANETICS_HIII_5TH_FEMALE":
                    case "V7_HUMANETICS_HIII_5TH_FEMALE":
                        var c_com = read_data("BEAM BASIC", 0, dummy[lr][i].beam_id, "NZ");  // NZ is longitudinal
                        var entity_str = "BEAM"; // Used for error message below
                        break;

                    case "LSTC HIII 5F v2.0":
                    case "LSTC HIII 5F Fast v2.0":
                    case "Humanetics HIII 5F v2.02":
                        var c_com = read_data("SECTION", 0, dummy[lr][i].beam_id, "FZ");  // FZ is longitudinal
                        var entity_str = "SECTION"; // Used for error message below

                        // Mutiply compression data by -1 to so compression is +ve
                        c_com = Operate.Mul(c_com, -1)				
                        
			break;

                    default:
                        ErrorMessage("Unsupported dummy \""+dummy[i].version+"\"in do_knee_femur_pelvis_rating_calc.");
                        write_blank_images(output_data, "UNSUPPORTED DUMMY \""+dummy[lr][i].version+"\"");                      
                        return;                        
                        break;
                }

                // Check that the curve read in - i.e. the beam id and component are valid
                // Create blank image to let the user know.

                if(!c_com)
                {
                 
                    write_blank_images(output_data, "NO DATA FOR " + entity_str + " " + dummy[lr][i].beam_id);
                }
                else
                {
                    // Convert forces to kN from model units

                    var c_com_kn = Operate.Div(c_com, oGlblData.kn_factor);

                    // C600 filter (convert to seconds first)

                    var c_com_c600 = convert_to_s_and_filter(c_com_kn, Operate.C600, REG_DT);

                    // Remove all curves and datums

                    remove_all_curves_and_datums();

                    // Convert from seconds back to model time

                    c_com_c600 = Operate.Mux(c_com_c600, oGlblData.time_factor);

                    // Set labels and style

                    set_labels(c_com_c600, dummy[lr][i].chart_label + " Femur Compression", "Time (" + oGlblData.unit_time + ")", "Compression (kN)");
                    set_line_style(c_com_c600, Colour.BLACK);

                    // Remove all curves and datums

                    remove_all_curves_and_datums();

                    // Draw femur compression datum

                    draw_constant_datums(FEMUR_COM_GOOD, FEMUR_COM_ADEQUATE, FEMUR_COM_MARGINAL, FEMUR_COM_WEAK);

                    // Create image and curve files of femur compression curves

                    c_com_c600.AddToGraph();
                    write_image(output_data.com_exc.title, output_data.com_exc.fname, c_com_c600, FEMUR_COM_WEAK);

                    output_data.com_exc.curveList.push(c_com_c600.id);
                    write_curve("cur", output_data.com_exc.curveList, output_data.com_exc.fname);
                    write_curve("csv", output_data.com_exc.curveList, output_data.com_exc.fname);

                    // Remove all curves and datums

                    remove_all_curves_and_datums(); 

                    // Calc values

                    var com_results = get_constant_limit_results(c_com_c600, FEMUR_COM_GOOD, FEMUR_COM_WEAK, KNEE_FEMUR_PELVIS_LO_SCORE, KNEE_FEMUR_PELVIS_HI_SCORE);

                    // Write compression variables

                    write_variable(f, dummy[lr][i].variable_label.toUpperCase() + "_" + side.toUpperCase() + "_FEMUR_COMPRESSION_SCORE", com_results[0].toFixed(3),  "Femur compression score",    "String");
                    write_variable(f, dummy[lr][i].variable_label.toUpperCase() + "_" + side.toUpperCase() + "_FEMUR_COMPRESSION",       com_results[1].toFixed(3),  "Femur compression level",    "String");
                }
            }
            else
            {

                // No beam defined - variables should be set to blank by first script in template.
                // Create a blank image to let the user know.
                if(dummy[lr][i].chart_label == "Front Passenger") fpass_knee_femur_pelvis_exists = "FALSE";
                write_blank_images(output_data, "NO BEAM ID DEFINED FOR");
            }
        }
    }
    return fpass_knee_femur_pelvis_exists;  
}



function draw_neck_shear_datums()
{
    // Delete any that might already exist

    if(Datum.Exists("POS GOOD"))       Datum.Delete("POS GOOD");
    if(Datum.Exists("POS ACCEPTABLE")) Datum.Delete("POS ACCEPTABLE");
    if(Datum.Exists("POS MARGINAL"))   Datum.Delete("POS MARGINAL");
    if(Datum.Exists("NEG GOOD"))       Datum.Delete("NEG GOOD");
    if(Datum.Exists("NEG ACCEPTABLE")) Datum.Delete("NEG ACCEPTABLE");
    if(Datum.Exists("NEG MARGINAL"))   Datum.Delete("NEG MARGINAL");

    // Note that the order these are defined is important
    var d_neg_marg = new Datum("NEG MARGINAL",   Datum.CONSTANT_Y, -NECK_SHR_MARGINAL);
    var d_neg_accp = new Datum("NEG ACCEPTABLE", Datum.CONSTANT_Y, -NECK_SHR_ADEQUATE);
    var d_neg_good = new Datum("NEG GOOD",       Datum.CONSTANT_Y, -NECK_SHR_GOOD);
    var d_pos_good = new Datum("POS GOOD",       Datum.CONSTANT_Y,  NECK_SHR_GOOD);
    var d_pos_accp = new Datum("POS ACCEPTABLE", Datum.CONSTANT_Y,  NECK_SHR_ADEQUATE);
    var d_pos_marg = new Datum("POS MARGINAL",   Datum.CONSTANT_Y,  NECK_SHR_MARGINAL);

    d_pos_good.line_colour = EuroNCAPColour.GREEN;
    d_pos_accp.line_colour = EuroNCAPColour.YELLOW;
    d_pos_marg.line_colour = EuroNCAPColour.ORANGE;
    d_neg_good.line_colour = EuroNCAPColour.GREEN;
    d_neg_accp.line_colour = EuroNCAPColour.YELLOW;
    d_neg_marg.line_colour = EuroNCAPColour.ORANGE;

    d_neg_marg.fill_colour_below = EuroNCAPColour.RED;
    d_neg_marg.fill_colour_above = EuroNCAPColour.ORANGE;
    d_neg_accp.fill_colour_above = EuroNCAPColour.YELLOW;
    d_neg_good.fill_colour_above = EuroNCAPColour.GREEN;
    d_pos_good.fill_colour_above = EuroNCAPColour.YELLOW;
    d_pos_accp.fill_colour_above = EuroNCAPColour.ORANGE;
    d_pos_marg.fill_colour_above = EuroNCAPColour.RED;
}

