//calculate the maximum inboard intrusion

/*The maximum inboard point is determined within the intrusion area, the method to find this
point is described as follows. There is no compulsory procedure how to measure the
maximum inboard point. It is acceptable to use 3D scan, 3D arm or a tape measure.
In most cases the armrest will be the most inboard part. Therefore, the measurement will
be taken from the most inboard surface of the armrest.
At waistline level, if this is the most inboard area, the inner door trim/cover is ignored. The
point is defined as the most inboard metal part of the door structure +50mm inboard (see
example below)*/


//parts to check (these are selected by user and should include parts in the Intrusion area)
//see Appendix I of https://cdn.euroncap.com/media/64153/euro-ncap-far-side-test-and-assessment-protocol-v21.pdf

//Line A - Vertical line at x-position 700mm forward of the R-point
//Line B - Horizontal line at z-position of R-point
//Line C - Vertical line at x-position at the back of headrest stems
//Line D - Door waist line

//var R_point_node_id; //this should be input by user
var Door_parts; //this is input by the user 
var door_waist_line_node_id;//this is used determine the height above which to ignore nodes for intrusion 
//we do not actually need to fully define Line D as intrusion is calculated by defelction of door/pillar parts 
// (ignoring disp above door_waist_line_node) so line D is implicit

var line_A_offset = 700;//mm
var headrest_stem_node_id;//Used to define line C 

var inboard_additional_offset = 50;//mm add 50mm to measurement from metal (note measurement ignores internal trim etc.)
var seat_y;//occupant seat center assumed to be same as h-point y

var csv_data = new Object();

parse_in_csv_data();

//inputs from csv/post *END or PRIMER input macro
//R_point_node_id = csv_data.r_point_node_id;
var H_point_node_id  = csv_data.h_point_node_id;

// var H_point_left_node_id  = csv_data.h_point_left_node_id;
// var H_point_right_node_id = csv_data.h_point_right_node_id;

headrest_stem_node_id = csv_data.headrest_stem_node_id ;
Door_parts = csv_data.door_parts;// [1000812, 1000820, 1000829, 1000293, 1000291];
door_waist_line_node_id = csv_data.door_waist_line_node_id;

//end of inputs
var H_point_coords = getBasicCoords(-H_point_node_id);
// var H_point_left_coords = getBasicCoords(-H_point_left_node_id);
// var H_point_right_coords = getBasicCoords(-H_point_right_node_id);
// var R_point = {'x':(H_point_left_coords.x+H_point_right_coords.x)/2,'y':(H_point_left_coords.y+H_point_right_coords.y)/2,'z':(H_point_left_coords.z+H_point_right_coords.z)/2};

var R_point = H_point_coords;
var headrest_stem_node = getBasicCoords(-headrest_stem_node_id);
var door_waist_line_node = getBasicCoords(-door_waist_line_node_id);

seat_y = R_point.y; //seat_y is assumed to be at the y position of the R-point which assumes that the car is symmetric and the occupant is positioned on the center of the seat 
                    //and that the barrier strikes occupant side (note for Far side we care about the seat on the other side (assumed to be at -seat_y)) 

var line_A = R_point.x - line_A_offset;
var line_B = R_point.z;
var line_C = headrest_stem_node.x;
var line_D = door_waist_line_node.z;

//needed as a fixed frame
var n1 = csv_data.shift_deform_node_1;
var n2 = csv_data.shift_deform_node_2;
var n3 = csv_data.shift_deform_node_3;


//If n1, n2 or n3 are not defined then skip this step
if ( (n1!=undefined) && (n2!=undefined) && (n3!=undefined) )
    DialogueInput("/DEFORM SHIFT_DEFORMED DEFINE", n1, n2, n3);

var nstates = GetNumberOf(STATE);
var nnodes = GetNumberOf(NODE);

//make a node array with the same number of elements as num_nodes + 1 to act as a flag where:
//nodes[i] == true for i'th internal node if it is in a door part AND in the defined 'box' else nodes[i] == null

var nodes = new Array(nnodes+1);

SetCurrentState(1);

//get vector from shift deformed node 1 and node 2
var n1_coords = GetData(BV, NODE, -n1);
var v_n1n2 = getNormalisedVectorFromCoords(n1_coords, GetData(BV, NODE, -n2));

//rotation matrix initially aligned with +ve y direction
var rotation_matrix = getRotationMatrix(v_n1n2,[0,1,0]);

//get_local_datum() uses global n1, n2, rotation_matrix and is state dependent so run it after changing to final state
SetCurrentState(nstates);
//DialogueInput("STATE "+ nstates);

//initialise values
var min_v1_end_y = Number.MAX_VALUE;
var max_intrusion_node;

var local_datum_at_end_state = get_local_datum();

for (var p = 0; p<Door_parts.length; p++)
{
    //for each part
    var external_part_id = Door_parts[p];

    //get the elements that make up the part
    var elements = GetElemsInPart(-external_part_id);
    for (var e = 0; e<elements.nn; e++)
    {
        //and for each of the elements in the part
        var element_internal_id = elements.list[e];
        //get all of their nodes
        var nodes = GetTopology(elements.type, element_internal_id);
        for (var n = 0; n<nodes.nn; n++)
        {
            //for each node
            var node_internal_id = nodes.top[n];
            //if not already checked
            if (nodes[node_internal_id] == undefined)
            //if (node_internal_id == 1126638)
            {
                //check if basic coordinate is in box
                var initial_z_node = GetData(BZ, NODE, node_internal_id);

                if (initial_z_node > line_D) continue;
                if (initial_z_node < line_B) continue;

                var initial_x_node = GetData(BX, NODE, node_internal_id);

                if (initial_x_node < line_A) continue;
                if (initial_x_node > line_C) continue;

                //if we get here then node is inside the box

                //so set the element value to true so that we know it is one to check 
                //and we don't need to check it's initial coordinates again 
                
               
                //current_node_excursion is local y component of vector from nodes n to n1
                var v1_end_y  = Math.abs(get_current_excursion(local_datum_at_end_state, node_internal_id));
        
                //we care about the maximum intrusion (i.e. the smallest distance of abs(v1_end_y) )
                //initial y coord (this doesn't work because vehicle is moving in y so need relative )
                //var n_y_coord = GetData(BY, NODE, node_internal_id);
                
                if (v1_end_y<min_v1_end_y){
                    min_v1_end_y=v1_end_y;
                    max_intrusion_node=node_internal_id;
                }

            }
        }
    }
}


// +ve means outboard of seat centerline and -ve means inboard (i.e. intrusion goes past c/l beyond orange zone in to yellow)

//assume n1_y is < max_intrusion_node_y
if (n1_coords[1] < GetData(BY, NODE, max_intrusion_node))
{
    //the intrusion is in -ve y direction (i.e. in to the right side (passenger side) of the vehicle if the vehicle is pointing in -x with driver on left )
    var max_intrusion_y = n1_coords[1]+min_v1_end_y - inboard_additional_offset; //subtract inboard_additional_offset to make max_intrusion_y 50mm more in -y direction
    var distance_from_seat_centerline = max_intrusion_y - seat_y;
}
else
{
    //the intrusion is in +ve y direction (i.e. in to the left side (driver side) of the vehicle if the vehicle is pointing in -x with driver on left )
    //choice of shift deform node on the passenger door (assumed to be unaffect under impact)
    //the intrusion is in +ve y direction (not negative)
    var max_intrusion_y = n1_coords[1]-min_v1_end_y + inboard_additional_offset; //add inboard_additional_offset to make max_intrusion_y 50mm more in +y direction
    var distance_from_seat_centerline = -(max_intrusion_y - seat_y); //times by -1 so that +ve distance means intrusion does not reach seat c/l and -ve meant it passes it
}

Message("distance_from_seat_centerline = " + distance_from_seat_centerline);
Message("max max_intrusion_node = " + GetLabel(NODE, max_intrusion_node));
Message("max_intrusion_y = " + max_intrusion_y);
Message("seat_y = " + seat_y);
Message("inboard_additional_offset = " + inboard_additional_offset);

//var parts_to_blank = Door_parts.join(" ");
//DialogueInput("BLANK", "PART", parts_to_blank);

//write out side intrusion file

var f_intrusion = new File(images_dir+"/EuroNCAP_"+analysis+"_intrusion.csv", File.WRITE);
f_intrusion.writelin
write_variable(f_intrusion, "MAX_INTRUSION_Y", max_intrusion_y, "the y coordinate of the maximum intrusion (including 50mm additonal offset) relative to initial car position","Number");
write_variable(f_intrusion, "SEAT_Y", seat_y, "the initial y coordinate of the seat centreline (nearest impact)","Number");
write_variable(f_intrusion, "INTRUSION_FROM_SEAT_CENTER_Y", distance_from_seat_centerline, "the y distance of the max intrusion (+50mm offset) from seat centreline (nearest impact). +ve is away from driver and -ve is towards driver.","Number");
write_variable(f_intrusion, "BARRIER", analysis, "Side impact barrier (Pole or MDB)","String");
f_intrusion.Close()

//****************FUNCTIONS******************************************//

function write_variable(f, name, value, desc, type)
{
// Writes variable to file <f>

  f.Writeln(name + "," + value + "," + desc + "," + type);
}


function getBasicCoords(nid)
{
    //nid -ve is external id
    var n1_coords = GetData(BV, NODE, nid);
    return {'x':n1_coords[0],'y':n1_coords[1],'z':n1_coords[2]};
}

function get_local_datum()
{
    // var n1; //on far side door near seat (i.e. side not impacted) 
    // var n2; //on far side door at another X coordinate and approximately same Y (i.e. side not impacted) 
    // var n3; //used for shift deformed

    //note n1 and n2 are external node ids so must be -ve
    var n1_coords = GetData(CV, NODE, -n1);
    var n2_coords = GetData(CV, NODE, -n2);

    var origin_local = n1_coords;
    
    //local coordinate
    var v_n1n2 = getNormalisedVectorFromCoords(n1_coords,n2_coords);

    //y_axis_local is used to take dot product with to find excursion distance (relative to origin_local)
    var y_axis_local = vectorXmatrix(v_n1n2,rotation_matrix);

    var local = {"origin":origin_local, "y_axis":y_axis_local};

    return local

}

function getNormalisedVectorFromCoords(n1_coords,n2_coords)
{
    var v_n1n2 = new Array(3);

    //calculate n1->n2 vector
    for (var i=0; i<3; i++)  v_n1n2[i] = n2_coords[i] - n1_coords[i];

    var mag = magnitude(v_n1n2);

    //normalise n1->n2 vector
    for (var i=0; i<3; i++) v_n1n2[i] /= mag;

    return v_n1n2;
}

function get_current_excursion(local, head_internal_node_id)
{
    var head_node_coords = GetData(CV, NODE, head_internal_node_id);

    //vector from current n1 to current head node
    var origin_vector = [
        head_node_coords[0]-local.origin[0],
        head_node_coords[1]-local.origin[1],
        head_node_coords[2]-local.origin[2],           
    ]

    var distance_of_head_from_origin_in_local_y = dot(local.y_axis,origin_vector);

    return distance_of_head_from_origin_in_local_y;                               
}

//JSON.stringify(local.origin)
//matrix operations:

/*
getRotationMatrix function is based on logic from the following stackexchange question:
https://math.stackexchange.com/questions/180418/calculate-rotation-matrix-to-align-vector-a-to-vector-b-in-3d/897677#897677

the rotation matrix is used to rotate the n1->n2 vector used to define the shift deform to the global y axis at the first state.
the same rotation matrix is then used to calculate the local y axis which is the assumed direction to measure excursion in.

e.g. if the car rotates on impact it will not be correct/meaningful to calculate the y displacement of the head nodes as the excursion will be in
the 'cross-car' direction which is what we call local y because the initial cross car direction is assumed to be the same as global y (i.e. the car is pointing in the x direction)

*/

function getRotationMatrix(a,b)
{
    // var a = [1, 1, 1];
    // var b = [Math.sqrt(0.5), Math.sqrt(0.5), 0];//[0,0,1];

    var v = cross(a, b);
    var s = magnitude(v);
    var c = dot(a, b)
    var vx = [[0, -v[2], v[1]], [v[2], 0, -v[0]], [-v[1], v[0], 0]];
    var I = [[1, 0, 0],[0, 1, 0],[0, 0, 1]];
    var mult = (1-c)/(s*s);
    var w = [[mult, 0, 0],[0, mult, 0],[0, 0, mult]];
    var r = matrix3x3Add([I,vx,multiplyMatrices(w,multiplyMatrices(vx,vx))])
    var rotated_vector = vectorXmatrix(a,r);
    Message("rotation_matrix  = " + JSON.stringify(r));
    Message("b_rotated  = " + JSON.stringify(rotated_vector));
    Message("b_original = " + JSON.stringify(b));
    return r;
}

function vectorXmatrix(v,m)
{
    if (m[0][0]==null){
        //if the matrix has null elements then original n1->n2 vector was parallel to global Y
        //so no rotation required and just return v
        Message('Warning: First element in matrix is null which will occur if the choice of n1->n2 is parallel to global y. ')
        Message('The normalised n1->n2 vector will therefore be used so we return it (v).')        
        return v;
    }
    var result = [0,0,0];
    for (var row =0; row<3; row++){
        for (var i = 0; i <3; i++){
            result[row]+= m[row][i]*v[i];
        }
    }

    return result;

}

function matrix3x3Add(matrices){
    var totalMatrix = Array(3);

    for (var matrix =0; matrix<matrices.length; matrix++)
    {
        for (var row =0; row<3; row++)
        {
            if (matrix==0) totalMatrix[row] = [0,0,0];
            for (var column =0; column<3; column++)
            {
                totalMatrix[row][column]+=matrices[matrix][row][column];
            }
        }

    }

    return totalMatrix;

}

function multiplyMatrices(m1, m2) {
var result = [];
for (var i = 0; i < m1.length; i++) {
    result[i] = [];
    for (var j = 0; j < m2[0].length; j++) {
        var sum = 0;
        for (var k = 0; k < m1[0].length; k++) {
            sum += m1[i][k] * m2[k][j];
        }
        result[i][j] = sum;
    }
}
return result;
}

function cross(A, B){
    var a1, a2, a3, b1, b2, b3;
    [a1, a2, a3] = A;
    [b1, b2, b3] = B;
    return [ a2 * b3 - a3 * b2, a3 * b1 - a1 * b3, a1 * b2 - a2 * b1 ];
} 

function dot(A, B){
    var a1, a2, a3, b1, b2, b3;
    [a1, a2, a3] = A;
    [b1, b2, b3] = B;
    return a1*b1 + a2*b2 + a3*b3;
} 

function magnitude(A){
    var mag = Math.sqrt(A[0]*A[0] + 
                        A[1]*A[1] + 
                        A[2]*A[2]);

        return mag;
}



function parse_in_csv_data()
{
    if(File.Exists(fname_csv) && File.IsFile(fname_csv))
    {

        var f_csv = new File(fname_csv, File.READ);
        var line;
        
        csv_data.h_point_node_id = undefined;

        // csv_data.h_point_left_node_id = undefined;
        // csv_data.h_point_right_node_id = undefined;

        csv_data.headrest_stem_node_id = undefined;
        csv_data.door_waist_line_node_id = undefined;
        
        //csv_data.seat_centre_y = undefined;
        
        csv_data.shift_deform_node_1 = undefined;
        csv_data.shift_deform_node_2 = undefined;
        csv_data.shift_deform_node_3 = undefined;
        
        csv_data.door_parts  = new Array();

    // Read the CSV file and store the data
    
        while ( (line = f_csv.ReadLine()) != undefined)
        {
    // Lines beginning with '$' are comments otherwise they should be of the form:
    //
    // variable_name,variable_value
    //

            if(line[0] != '$')
            {
                var list = line.split(",");
                var name = list[0].toUpperCase();
            
                if(list[1] != undefined)
                {
                    switch(name)
                    {
                        case "H_POINT_NODE":                   { csv_data.h_point_node_id          = parse_id(list[1]);  break; }

                        // case "H_POINT_LEFT_NODE":              { csv_data.h_point_left_node_id     = parse_id(list[1]);  break; }
                        // case "H_POINT_RIGHT_NODE":             { csv_data.h_point_right_node_id    = parse_id(list[1]);  break; }

                        case "HEADREST_NODE":                  { csv_data.headrest_stem_node_id    = parse_id(list[1]);  break; }                            
                        case "DOOR_WAIST_LINE_NODE":           { csv_data.door_waist_line_node_id  = parse_id(list[1]);  break; }

                        case "SHIFT_DEFORM_NODE_1":            { csv_data.shift_deform_node_1      = parse_id(list[1]);  break; }          
                        case "SHIFT_DEFORM_NODE_2":            { csv_data.shift_deform_node_2      = parse_id(list[1]);  break; }
                        case "SHIFT_DEFORM_NODE_3":            { csv_data.shift_deform_node_3      = parse_id(list[1]);  break; }

                        //case "SEAT_CENTRE_Y":                  { csv_data.seat_centre_y            = parse_id(list[1]);  break; }
            
                        case "DOOR_PARTS":                     { csv_data.door_parts.push  (parse_id(list[1]));  break; }               
                    }
                }
            }
        }
        f_csv.Close();
    }
}


 
function parse_id(id)
{
// Parse an id string.  Could be a number, name or blank

    var ret_id;

// Is it a number

    if(!(isNaN(id)))
    {

        ret_id = parseInt(id);
  
        if(ret_id != undefined && !(isNaN(ret_id))) return ret_id;
    }
  
// Not a number so return the string if it's not blank

    if(id.length >  0)
    {
        for(var i=0; i<id.length; i++)
        {
            if(id[i] != ' ') return id;
        }
    }

// Get here and it's not a number or non-blank string so return undefined.
  
    return undefined;
}
 
