/* Reporter JavaScript to obtain Ansys LS-DYNA version and revision from OTF file
   Miles Thornton 2/3/2005

PROGRAM::version.js
DESC::Ansys LS-DYNA Version and Revision
FOLDER::OTF or d3hsp file
RETURN::text
ARG::OTF file name::%DEFAULT_DIR%/d3hsp
END_INFO
*/

if (arguments.length == 0) LogError("No otf file argument.");

var otf_file = new File(
  File.DriveMapFilename(arguments[0], Include.NATIVE),
  File.READ
);
var line;
var mpp_regex = new RegExp("Distributed Memory Parallel");
var smp_regex = new RegExp("Shared Memory Parallel");
var search_regex = new RegExp("(Version|Revision)");

// Regular expression should match Version and Revision numbers from the old
// Ansys LS-DYNA system (SVN revision numbers, Version on same line as Date, Revision
// on same line as Time) and the new Ansys LS-DYNA system (Git revision hashes,
// Version and Revision on separate lines from Date and Time):
//
// Example of old syntax:
//
//     |  LS-DYNA, A Program for Nonlinear Dynamic       |
//     |  Analysis of Structures in Three Dimensions     |
//     |  Version : mpp d R9.1.0    Date: 01/19/2017     |
//     |  Revision: 113698          Time: 13:22:48       |
//
// Example of new syntax from R11.2 onwards:
//
//     |  LS-DYNA, A Program for Nonlinear Dynamic       |
//     |  Analysis of Structures in Three Dimensions     |
//     |  Date    : 02/05/2021    Time: 12:22:50         |
//     |  Version : mpp d R11.2.1                        |
//     |  Revision: R11.2-181-gcb80e4d4cd                |
//
var match_regex = new RegExp(
  "(Version|Revision)\\s*:\\s+(.+?)(\\s+\\w+:|\\s\\s)"
);

var result;
var version = "";
var revision = "";
var memtype = "";
var ostring;
var count = 0;

while ((line = otf_file.ReadLine()) != File.EOF) {
  count++;

  // First to a simple quick search to eliminate lines
  if (line.search(search_regex) != -1) {
    // Now do the slower match
    if ((result = line.match(match_regex)) != null) {
      if (result[1] == "Version") version = result[2];
      else if (result[1] == "Revision") revision = result[2];
    }
  } else if (line.search(mpp_regex) != -1) memtype = " MPP";
  else if (line.search(smp_regex) != -1) memtype = " SMP";

  // Stop scanning once Version and Revision are found
  if (version != "" && revision != "") break;
  // We only scan the first 100 lines
  if (count == 100) break;
}

otf_file.Close();

// If version string starts with 'smp' or 'mpp' don't bother appending SMP/MPP...
if (version.match(/^(smp|mpp)/)) memtype = "";

if (version && revision) ostring = version + " (" + revision + ")" + memtype;
else ostring = version + memtype;

ostring.replace(/\s+$/, "");
output(ostring);
