/* Reporter script to create a us-ncap star graph
   Miles Thornton   21/11/2005

PROGRAM::us-ncap.js
DESC::Create a US-NCAP graph
FOLDER::NCAP
RETURN::none
ARG::HIC value::%HIC%
ARG::Chest acceleration (3ms clip)::%CH3MS%
ARG::Image file to write::%DEFAULT_DIR%/us-ncap.png
END_INFO
*/

if (arguments.length != 3)
    throw Error("Wrong number of arguments to us-ncap.js. Expected 3 but got " +
                arguments.length +
                "\n");

var hicMin  =    0;
var hicMax  = 1400;
var hicTick =  200;
var accMin  =    0;
var accMax  =   80;
var accTick =   10;

var imgX = 800;
var imgY = 600;

var pt = 14;
var tic = 5;

var drawFill = true;

var colours = new Array("red", "orange", "yellow", "green");
var colourgr = "grey";
var colourfg = "black";
var colourbg = "white";

var img = new Image(imgX, imgY);

var xMin = 60;
var xMax = imgX - 50;
var yMin = 50;
var yMax = imgY - 50;

// Set background
img.fillColour = colourbg;
img.lineColour = "none";
img.lineStyle  = Reporter.LineNone;
img.lineWidth = 0;
img.Rectangle(0,0, 800, 600);

img.fillColour = colourgr;
img.Rectangle(xMin,yMin, xMax, yMax);

img.font = "Helvetica";
img.fontSize = pt;

// Draw the fills/boundaries
var p   = new Array(0.45, 0.35, 0.2, 0.1);
var pts = new Array();

for (var i=0; i<p.length; i++)
{
    var c, hic, H;

    hic = 0;
    pts.length = 0;

    if (drawFill == true) pts.push(0,0);

    while (hic<1400)
    {
        H = (4.9 + 200/hic) - 0.00351*hic;
        var frac = 1/(1 + Math.exp(H));

        c = (5.55 - Math.log((1 - frac)/(p[i] - frac) - 1) ) / 0.0693;

        if (isNaN(c) || c < 0)
        {
            pts.push(hic, 0);
            break;
        }

        pts.push(hic, c);

        if      (c > 30) hic += 10;
        else if (c > 20) hic +=  5;
        else if (c > 10) hic +=  1;
        else             hic +=  0.1;
    }

    for (var j=0; j<pts.length; j+=2)
    {
        pts[j]   = xMin + (pts[j]-hicMin)*(xMax-xMin)/(hicMax-hicMin);
        pts[j+1] = yMax - (pts[j+1]-accMin)*(yMax-yMin)/(accMax-accMin);
    }

    img.fillColour = colours[i];
    img.lineColour = colours[i];
    img.lineStyle  = Reporter.LineSolid;

    if (drawFill == true)
    {
        img.lineWidth = 0;
        img.Polygon(pts);
    }
    else
    {
        img.lineWidth = 3;
        img.Polyline(pts);
    }
}

img.lineColour = colourfg;
img.fontColour = colourfg;
img.fontJustify = Reporter.JustifyCentre;

// X axis
for (i=hicMin; i<=hicMax; i+=hicTick)
{
    var x = xMin + (i-hicMin)*(xMax-xMin)/(hicMax-hicMin);
    img.lineStyle  = Reporter.LineSolid;
    img.lineWidth = 2;
    img.Line(x, yMax, x, yMax+tic);
    img.Line(x, yMin, x, yMin-tic);
    img.lineStyle  = Reporter.LineDash;
    img.lineWidth = 1;
    img.Line(x, yMin, x, yMax);
    img.Text(x, yMax+5+pt, i.toString());
}
img.Text((xMin+xMax)/2, yMax+2*tic+2*pt, "HIC (36ms)");

img.fontJustify = Reporter.JustifyRight;

// Y axis
for (i=accMin; i<=accMax; i+=accTick)
{
    var y = yMax - (i-accMin)*(yMax-yMin)/(accMax-accMin);
    img.lineStyle  = Reporter.LineSolid;
    img.lineWidth = 2;
    img.Line(xMax, y, xMax+tic, y);
    img.Line(xMin, y, xMin-tic, y);
    img.lineStyle  = Reporter.LineDash;
    img.lineWidth = 1;
    img.Line(xMin, y, xMax, y);
    img.Text(xMin-tic-1, y+pt/2, i.toString());
}

img.fontJustify = Reporter.JustifyCentre;
img.fontAngle = -90;
img.Text(xMin-35,(yMin+yMax)/2, "Chest Acceleration (g)");

// Graph border
img.lineColour = "black";
img.lineStyle  = Reporter.LineSolid;
img.lineWidth = 3;
img.Line(xMin, yMin, xMax, yMin);
img.Line(xMin, yMax, xMax, yMax);
img.Line(xMin, yMin, xMin, yMax);
img.Line(xMax, yMin, xMax, yMax);

// Title
img.fontAngle = 0;
img.Text((xMin+xMax)/2, yMin-3*tic, "US NCAP Star Rating");

// Star ratings
img.fontJustify = Reporter.JustifyLeft;
var y = yMax - (73-accMin)*(yMax-yMin)/(accMax-accMin);
img.Text(xMin+5, y, "2*");
var y = yMax - (63-accMin)*(yMax-yMin)/(accMax-accMin);
img.Text(xMin+5, y, "3*");
var y = yMax - (53-accMin)*(yMax-yMin)/(accMax-accMin);
img.Text(xMin+5, y, "4*");
var y = yMax - (43-accMin)*(yMax-yMin)/(accMax-accMin);
img.Text(xMin+5, y, "5*");

// Draw point
var hic = arguments[0];
if (hic == null) hic = 1400;
var acc = arguments[1];
if (acc == null) acc = 80;
var x = xMin + (hic-hicMin)*(xMax-xMin)/(hicMax-hicMin);
var y = yMax - (acc-accMin)*(yMax-yMin)/(accMax-accMin);

img.lineStyle  = Reporter.LineSolid;
img.lineWidth = 1;
img.lineColour = "black";
img.fillColour = "black";
img.Ellipse(x-10,y-10,x+10,y+10);

img.Save(arguments[2], Image.PNG);
