/* REPORTER JavaScript to extract modal frequencies from eigout file
PROGRAM::read_from_eigout.js
DESC::Extract eigenvalues (modal frequencies) from eigout file
FOLDER::eigout file
RETURN::text
ARG::eigout file::%DEFAULT_DIR%/eigout
END_INFO
*/

/* Check the input arguments */
if (arguments.length == 0)
{
    LogError(`No eigout file argument.`);
    Exit();
}
var f_eig_name = arguments[0];
if (!File.Exists(f_eig_name))
{
    LogError(`The specified eigout file cannot be found: ${f_eig_name}`);
    Exit();
}

/* Define the column headings and character widths as they appear in the eigout file */
var columns = [];
columns.push({attr:"MODE", width:12});
columns.push({attr:"EIGENVALUE", width:15});
columns.push({attr:"RADIANS", width:15});
columns.push({attr:"CYCLES", width:15});
columns.push({attr:"PERIOD", width:15});

LogPrint(`Extracting data from eigout file: ${f_eig_name}`);
var f_eig = new File(f_eig_name, File.READ);
var line;
var line_count = 0;
var mode = 0;
var read_data = false;
while ((line = f_eig.ReadLine()) != File.EOF)
{
    line_count++;
    if (line.match(" MODAL PARTICIPATION FACTORS"))
    {
        /* Stop processing lines once we get beyond the initial frequency table */
        break;
    }
    else if (line === "")
    {
        /* Skip empty lines */
        continue;
    }
    else if (read_data == true)
    {
        LogPrint(line);
        var char_count = 0;
        for (var c = 0; c < columns.length; c++)
        {
            var field = line.substr(char_count, columns[c].width);
            char_count += columns[c].width;
            if (isNaN(parseFloat(field)))
            {
                LogError(`Failed to parse ${columns[c].attr} value "${field}" in line ${line_count} of eigout file as a number.`);
                break;
            }
            else if (columns[c].attr == "MODE")
            {
                mode = parseInt(field);
                if (isNaN(mode))
                {
                    LogError(`Failed to parse ${columns[c].attr} value "${field}" in line ${line_count} of eigout file as an integer.`);
                    break;
                }
                var v = new Variable(Template.GetCurrent(), "EIGOUT_MODE_"+mode, "Output from eigout file", mode, "Number", false, true);
            }
            else if (columns[c].attr == "EIGENVALUE")
            {
                var v = new Variable(Template.GetCurrent(), "EIGOUT_MODE_"+mode+"_"+columns[c].attr, "Output from eigout file", parseFloat(field).toFixed(1), "Number", false, true);
            }
            else if (columns[c].attr == "RADIANS")
            {
                var v = new Variable(Template.GetCurrent(), "EIGOUT_MODE_"+mode+"_"+columns[c].attr, "Output from eigout file", parseFloat(field).toFixed(2), "Number", false, true);
            }
            else  if (columns[c].attr == "CYCLES")
            {
                var v = new Variable(Template.GetCurrent(), "EIGOUT_MODE_"+mode+"_"+columns[c].attr, "Output from eigout file", parseFloat(field).toFixed(2), "Number", false, true);
            }
            else if (columns[c].attr == "PERIOD")
            {
                var v = new Variable(Template.GetCurrent(), "EIGOUT_MODE_"+mode+"_"+columns[c].attr, "Output from eigout file", parseFloat(field).toFixed(4), "Number", false, true);
            }
        }
    }
    else if (line.match("        MODE    EIGENVALUE       RADIANS        CYCLES        PERIOD"))
    {
        /* Start processing lines after we reach the table header line */
        read_data = true;
        LogPrint("Found start of frequency table:");
        LogPrint(line);
    }
}

f_eig.Close();
LogPrint("Finished reading eigout file.");
Exit();
