// module: TRUE
/* Reporter script to create a zone area contour image of pedestrain results

PROGRAM::pedestrian_zone_area_from_variables.js
DESC::Create a contour image of HIC for pedestrian HIC results in reporter variable files
FOLDER::Pedestrian
RETURN::none
ARG::Directory/filename::%DEFAULT_DIR%
ARG::Variable for point X coord::%XCOORD%
ARG::Variable for point Y coord::%YCOORD%
ARG::Variable for point HIC::%HIC%
ARG::Perimeter CSV filename (X,Y)::%DEFAULT_DIR%/child_head_boundary.csv
ARG::Lower HIC limit for zone::1000
ARG::Upper HIC limit for zone::1700
ARG::Area target percentage::66.6
ARG::Zone name::CHILD
EXPAND_ARGS::false
DEPRECATED::20
END_INFO
*/

import {ArrayMax, ArrayMin, arrayPosition, pointInPolygon, triangleInterpolate, 
        squared, areaTriangle, smallestY, isLeftTurn, isInCircumcircle, increasingangle,
        increasingnumber, arrayFirstPosition, sumOfArray, perimCutsX, perimCutsY, 
        perimMultipleIntersections, perimIntersection, distanceToPerim, howManyIntersections,
        centroidify, interpolationPoint, triangleWeb, perimAreas} from "./pedestrian_zone_area_common_funcs.mjs";

if (arguments.length != 9)
    throw Error("Wrong number of arguments to pedestrian_zone_area_from_variables.js. Expected 9 but got " +
                arguments.length +
                "\n");

var filename = Template.GetCurrent().ExpandVariablesInString(arguments[0]);
filename = File.DriveMapFilename(filename, Include.NATIVE);

// Look for the variables file or directory
if (File.Exists(filename) == false)
    throw Error("Input file " + arguments[0] + " does not exist in pedestrian_zone_area_from_variables.js\n");

var inputFiles = new Array();
var fileIn;
var line;
var regex  = new RegExp("VAR\\s+([^!#]*)[!#]?\\s+DESC.*VALUE='(.*)'");
var vregex = new RegExp("%(\\w+)%");

// Check that x,y,z and value variables are sensible
if ( (result = arguments[1].match(vregex) ) == null)
    throw Error("Bad X variable '" + arguments[1] + "'");

if ( (result = arguments[2].match(vregex) ) == null)
    throw Error("Bad Y variable '" + arguments[2] + "'");

if ( (result = arguments[3].match(vregex) ) == null)
    throw Error("Bad HIC variable '" + arguments[3] + "'");

// Input is a directory so find any reporter_variables files recursively
// from this directory
if (File.IsDirectory(filename))
{
    inputFiles = File.FindFiles(filename, "reporter_variables", true);
}
// Input is a file so get the list of reporter_variables files from the file
else
{
    fileIn = new File(filename, File.READ);
    line;

    while ( (line = fileIn.ReadLine() ) != File.EOF)
    {
        inputFiles.push(line);
    }

    fileIn.Close();
}

var TXTlines2 = new Array();

// Loop over all of the reporter variables.generated files
for (var i=0; i<inputFiles.length; i++)
{
    fileIn = new File(inputFiles[i], File.READ);

    var varData = new Object;

    while ( (line = fileIn.ReadLine() ) != File.EOF)
    {
        if ( (result = line.match(regex) ) != null)
        { 
            varData[result[1]] = result[2];
        }
    }

    fileIn.Close();

// Write this data point to the TXTlines array
    write_data_point(varData, arguments, inputFiles[i], TXTlines2);
}


var perimInputFile = Template.GetCurrent().ExpandVariablesInString(arguments[4]);     //expected x,y, \n x,y
perimInputFile = File.DriveMapFilename(perimInputFile, Include.NATIVE);
var LOWER_LIMIT = arguments[5];
var UPPER_LIMIT = arguments[6];
var AREA_TARGET = arguments[7];
var ZONE_NAME = arguments[8];



mainScript:
{
// ========================== //
// == USER DEFINED OPTIONS == //
// ========================== //
        // some options as to the appearance of the contour image that user can choose

// BOOLEANS
var drawPerimeter = 1;
var colourInContours = 1;
var drawContourLines_Major = 1;        // you can specify "major" contour values later. 
var ThicknessContourLines_Major = 3;
var drawContourLines_Minor = 1;
var ThicknessContourLines_Minor = 1;
var drawGrid = 0;       // draw orig data points.
var contourBarWidth = 100;
var contourBarHeight = 600;
var outputPercentageAreas = 1;
var outputAbsoluteAreas = 1;

// NON_BOOLEANS:

var bordersize = 50;    // border round edge of picture.
var allContourValues = [0, 100,200,300,400,500,600,700,800,900,1000,1100,1200,1300,1400,1500,1600,1700,1800,1900,2000];
var majorContourValues = [LOWER_LIMIT,UPPER_LIMIT,2000];   //NB: if any of these "major" HIC values are not defined in "allContourLines" they will be ignored
                                        //NB2: order of either of these arrays is insignificant. 
var imageWidth = 1000;  // rough guide, actually goes to nearest manage-able width.
var colours = new Array(
"#0000ff",
"#0040ff",      
"#006aff",      // these defaults are based on D3Plot colours, but interpolated to get a few more.
"#008fff",
"#00cfff",
"#00ffff",
"#00ffd5",
"#00ffaa",
"#00ff85",
"#00ff00",
"#83ff00",
"#aaff00",
"#d5ff00",
"#ffff00",
"#ffdf00",
"#ffbf00",
"#ff7f00",
"#ff0000",
"#ff0063",
"#ff0094",
"#ff00c9");


// SOME DEFAULTS - AS PER INDIVIDUAL REQUESTS.

EUregulationKV=1        //areas expressed as HIC <1000, 1000 < HIC < 2000, 

// ======================== //
// == END OF USER INPUTS == //
// ======================== //



// =========================================================================================================================================== //
// ==| The following aredebugging options for troubleshooting. Some are now redundant, some must stay checked. Change at your own risk. |== //
// =========================================================================================================================================== //

var drawHull = 0;       //draws the convex hull of original data points
var drawTriangulation = 0;  //draws the delanunay triangulation of said points
var drawRectangles = 0;   //draw gridrectangles (i think maybe now redundant)
var hideOuter = 1;      //cover up the bit outside perim
var writeHICS = 0;      //write orig HICs on image
var writeHICS_all = 0;  //write ALL HICs (inc interpolated) on image
var drawGrid_all = 0;   //draw all grid POINTS

// Create the zone name variable
var pvar = new Variable(Template.GetCurrent(), "ZONE_NAME", "Zone name", ZONE_NAME);
var pvar = new Variable(Template.GetCurrent(), "LOWER_LIMIT", "Lower limit", LOWER_LIMIT);
var pvar = new Variable(Template.GetCurrent(), "UPPER_LIMIT", "Upper limit", UPPER_LIMIT);
var pvar = new Variable(Template.GetCurrent(), "AREA_TARGET", "Percetage area target for lower limit", AREA_TARGET);


//=================================================================================================================================================================================================

if (colourInContours == 1) {            //check we have enough colours..
    if (colours.length < allContourValues.length) {     
        LogError ("\nOnly ",colours.length," colours have been defined for ",allContourValues.length," contour levels\n");
    }           //log error if not
}
bordersize = Math.round(bordersize);
//make sure contour vals are in ascending order
allContourValues.sort(increasingnumber);  
    
//===================================//    
//==  get perimeter data from file ==//
//===================================//    

LogPrint("\nExtracting perimeter data...\n");

if (File.Exists(perimInputFile)) { 
    var f = new File(perimInputFile, File.READ);  //read in file
}       //if file exists, use it, else log an error.
else LogError ("Perimeter input file not found")

var TXTlines = new Array();     //array of lines of txt file
var TXTline;    //individual line as we read it
var a = 0;
var PerimCoordinates = new Array();

while ( (TXTline = f.ReadLine() ) != File.EOF)  {  //go through each line of file
    if((TXTline.length > 0)) {                          //if it contains anything
        TXTlines[a] = TXTline;                            //stick in an array of lines
        a++;
    }
}

for ( i=0; i < TXTlines.length; i++ ) {
    if (TXTlines[i].indexOf(",") != -1) {         // if it contains commas, must be csv,
        TXTlines[i] = TXTlines[i].split(",");      // so split with commas
    }
    else {                                     // otherwise must be white-space delimited,
        TXTlines[i] = TXTlines[i].split(/\s+/);   // so split accordingly
    }
    PerimCoordinates.push(TXTlines[i][0]);      //push each coordinate onto main array - will be x,y x,y etc
    PerimCoordinates.push(TXTlines[i][1]);
}


for ( i=0; i < PerimCoordinates.length; i++ ) {
    PerimCoordinates[i] = parseFloat(PerimCoordinates[i]);        //make sure they're numbers, not strings
}

//== draw perimeter ==//

var b = 0;
var XPerimCoords = new Array();
var YPerimCoords = new Array();

for ( i=0; i < PerimCoordinates.length; (i = i+2) ) {
    YPerimCoords[b] = PerimCoordinates[i];                //extract x and y coords separately, to 
    XPerimCoords[b] = PerimCoordinates[i+1];              // convert to Reporter-style axes  |-->x 
    b++;                                                  //                                 V y  
}

var minX = ArrayMin(XPerimCoords);
var maxX = ArrayMax(XPerimCoords);
var minY = ArrayMin(YPerimCoords);
var maxY = ArrayMax(YPerimCoords);
var scaleFactor = imageWidth/(maxX-minX);    //we shall scale up/down coords so image has width of imageWidth.
if (scaleFactor > 1) {
    scaleFactor = Math.round(scaleFactor);      //make sure an integer
}
else {
    var inverse = 1/scaleFactor;
    inverse = squared(Math.round(Math.sqrt(inverse)));  //make sure is a power of 1/2
    scaleFactor = 1/inverse;
}

for ( i=0; i < PerimCoordinates.length; i++ ) {
    PerimCoordinates[i] = PerimCoordinates[i]*scaleFactor;        //scale up this array too
}


for (i=0; i < XPerimCoords.length; i++) {
    XPerimCoords[i] = ((XPerimCoords[i] - minX + bordersize)*scaleFactor);   // re-zero, so no negatives - and then add (x,y) to get border
    YPerimCoords[i] = ((YPerimCoords[i] - minY + bordersize)*scaleFactor);   // //and scale up/down
}

PerimCoordinates = new Array();

for ( i=0; i < XPerimCoords.length; i++ ) {
    PerimCoordinates.push(XPerimCoords[i]);     //recreate this array with scaled & translated coords.
    PerimCoordinates.push(YPerimCoords[i]);             //need it in this form for polygon function (drawing perim)
}

var maxX = ArrayMax(XPerimCoords);   //recalculating maximums
var maxY = ArrayMax(YPerimCoords);

var tolerance = Math.sqrt(((maxX*maxY)/(XPerimCoords.length))/1000); //tolerance of 0.1% of average gridsize
var determinant_tolerance = Math.sqrt(((maxX*maxY)/(XPerimCoords.length))/1000000); //need a small tolerance on the determinant

var img = new Image(Math.round(maxX+bordersize),Math.round(maxY+bordersize));    //image sized according to perimeter dimensions, with border.
img.lineColour = "blue";

var a = 0;


//=============================//    
//==  get HIC data from file ==//
//=============================//    

LogPrint("Processing HIC data...\n");

var TXTline2;
var a = 0;
var XcoordOrig = new Array();
var Xcoord = new Array();
var YcoordOrig = new Array();
var Ycoord = new Array();
var HIC = new Array();


for ( i=0; i < TXTlines2.length; i++ ) {
    if (TXTlines2[i].indexOf(",") != -1) {         //if it contains commas, must be csv, so split with commas
        TXTlines2[i] = TXTlines2[i].split(",");
    }
    else {                                     //otherwise must be white-space delimited, so split accordingly
        TXTlines2[i] = TXTlines2[i].split(/\s+/);
    }
    XcoordOrig.push(TXTlines2[i][0]);    //separate into x, y and HIC - separate array for each 
    YcoordOrig.push(TXTlines2[i][1]);
    HIC.push(TXTlines2[i][2]);

}



for ( i=0; i < XcoordOrig.length; i++ ) {
    Xcoord[i] = (parseFloat(YcoordOrig[i]) - minX + bordersize)*scaleFactor;         //make sure they're numbers, not strings, re-zero, translate
    Ycoord[i] = (parseFloat(XcoordOrig[i]) - minY + bordersize)*scaleFactor;         // and convert to Reporter-style axes  |-->x      
    HIC[i] = parseFloat(HIC[i]);                                       //                                     V y  
    if (writeHICS == 1)  {
    }
}

//make a copy of this array of orig points to draw on over top of contours later.

var gridToDrawX = Xcoord.slice(0);
var gridToDrawY = Ycoord.slice(0);

img.lineWidth = 1;
img.fontSize = 10;
img.lineStyle = Reporter.LineSolid;

var XcoordIncPerim = Xcoord.slice(0);
var YcoordIncPerim = Ycoord.slice(0);

for (i = 0; i < XPerimCoords.length; i++)  {
    XcoordIncPerim.push(XPerimCoords[i]);
    YcoordIncPerim.push(YPerimCoords[i]);     
    HIC.push("null");                        
}


/////    ====================================================================    //
///        THIS SECTION SHOULD BE REDUNDANT NOW 
//make sure grid goes thru all perim coords, narrowing down options of how perim cuts thru rectangle.
for (var i = 0; i < Xcoord.length; i++)  {                      //for each point
    var IntersectionCoords1 = perimCutsX(Xcoord[i],XPerimCoords, YPerimCoords);  //find where a vertical line through it intersects perimeter
    if (IntersectionCoords1) {                  //if any intersections were found (may not be as may cut thru at exact point)
        for (var j = 0; j < IntersectionCoords1.length; j++) {  //then add these coords to array for later
            XcoordIncPerim.push(IntersectionCoords1[j]["x"]);
            YcoordIncPerim.push(IntersectionCoords1[j]["y"]);
        }
        
    }
    var IntersectionCoords2 = perimCutsY(Ycoord[i],XPerimCoords, YPerimCoords);  //ditto a horizontal line
    if (IntersectionCoords2) {
        for (var j = 0; j < IntersectionCoords2.length; j++) {  //ditto ditto 
            XcoordIncPerim.push(IntersectionCoords2[j]["x"]);
            YcoordIncPerim.push(IntersectionCoords2[j]["y"]);
        }
    }
}

/////    ====================================================================    //
 
var gridpointsX = new Array();
var gridpointsY = new Array();

//creating arrays of all x and y values present.
for ( i = 0; i < XcoordIncPerim.length; i++ )  {
    var Xpresent = 0;
    var Ypresent = 0;

    for ( j = 0; j < gridpointsX.length; j++ )  {
        if ( (gridpointsX[j] >= (XcoordIncPerim[i] - tolerance) ) && (gridpointsX[j] <= (XcoordIncPerim[i]) + tolerance) ) {             //is the current X value already stored in gridpointsX??
            Xpresent = 1;
        }
    }

    if (Xpresent != 1) {
        gridpointsX.push(XcoordIncPerim[i]);    //if its not already there, put it in gridpointsX
    }

    for ( j = 0; j < gridpointsY.length; j++ )  {
        if ((gridpointsY[j] >= (YcoordIncPerim[i] - tolerance) ) && (gridpointsY[j] <= (YcoordIncPerim[i]) + tolerance)) {              //ditto Y value
            Ypresent = 1;
        }
    }

    if (Ypresent != 1) {
        gridpointsY.push(YcoordIncPerim[i]);    //ditto Y
    }
}


gridpointsX.push(maxX,bordersize);   // adding to gridpoints the max/min x and y to define rectangle of grid that covers the perimeter.
gridpointsY.push(maxY,bordersize); 

// Sort into ascending order and remove duplicates

gridpointsX.sort(function(a,b) {return a-b});
gridpointsY.sort(function(a,b) {return a-b});

for(var i=0; i<gridpointsX.length-1; i++)
{
  if( (gridpointsX[i] >= (gridpointsX[i+1] - tolerance)) && 
      (gridpointsX[i] <= (gridpointsX[i+1] + tolerance)) )
  {
    gridpointsX.splice(i,1);
    i--;
  }
}

for(var i=0; i<gridpointsY.length-1; i++)
{
  if( (gridpointsY[i] >= (gridpointsY[i+1] - tolerance)) && 
      (gridpointsY[i] <= (gridpointsY[i+1] + tolerance)) )
  {
    gridpointsY.splice(i,1);
    i--;
  }
}


// Now store grid points in some more arrays and the HIC value (null if not yet calculated)

var XcoordNEW = new Array();
var YcoordNEW = new Array();
var HIC_NEW = new Array();

for ( i = 0; i < gridpointsX.length; i++ )  {
    for ( j = 0; j < gridpointsY.length; j++ )  {
        var COORDpresent = 0;
        var myHIC = "null";
        for ( k = 0; k < Xcoord.length; k++ )  {
            if (( (gridpointsX[i] >= (Xcoord[k] - tolerance) ) && (gridpointsX[i] <= (Xcoord[k]) + tolerance) ) && ( (gridpointsY[j] >= (Ycoord[k] - tolerance) ) && (gridpointsY[j] <= (Ycoord[k]) + tolerance) )) {  //does this coordinate already exist?
                COORDpresent = 1;
                myHIC = HIC[k]; 
            }
        }
        XcoordNEW.push(gridpointsX[i]);    //creating all the combinations of x, y coordinates to make grid.
        YcoordNEW.push(gridpointsY[j]); 
        HIC_NEW.push(myHIC);               //transferring HIC values if they already exist
    }
}
        

//defining different arrays of coordinates to keep track of.

var coordinatesORIG = new Array();      //combine x and y vals into 1 array, to keep together
for (var i = 0; i < Xcoord.length; i++)  {
    coordinatesORIG[i] = { x:Xcoord[i], y:Ycoord[i], HIC:HIC[i] } ;     //this is definitive list of all original coordinates from data.
}

// Combine x and y vals into 1 array, to keep together
var coordinatesGRID = new Array();
for (var i = 0; i < XcoordNEW.length; i++)  {
    coordinatesGRID[i] = { x:XcoordNEW[i], y:YcoordNEW[i], HIC:HIC_NEW[i] } ;     //this is definitive list of all coordinates of grid.
}



    //===============================\\
    //===| calculate convex hull |===\\  
    //===============================\\
// Takes the original points with HIC data, calculates convex hull and Delaunay triangulation to use for interpolation.
//       References Xcoord, Ycoord.

LogPrint("Triangulating points for interpolation...\n");

///=== GRAHAM SCAN ALGORITHM ===\\\

coordinatesHULL = coordinatesORIG.slice(0);        //make a copy of array

var coordinatesINNER = new Array();     //anything taken from hull array will go into this one

//    //==| find beginning point |==\\

coordinatesHULL.sort(smallestY);  //sort by smallest Y (and smallest x in case of a tie)

var point0_X = coordinatesHULL[0]["x"];  // identify starting point
var point0_Y = coordinatesHULL[0]["y"];

//    //==| order rest of coordinates by angle they and point0 make with X-axis |==\\

coordinatesHULL.sort(increasingangle);  //sort by increasing angle (see function "increasingangle")

//    //==| add coordinates one by one, in order, checking whether they represent left turn or right turn in path |==\\
                                //note "left turn" and "right turn" refer to standard axes, reporter is actually back-to-front
var i = 2;
while (i < coordinatesHULL.length) {
    if ((isLeftTurn(coordinatesHULL[i-2]["x"],coordinatesHULL[i-2]["y"],coordinatesHULL[i-1]["x"],coordinatesHULL[i-1]["y"],coordinatesHULL[i]["x"],coordinatesHULL[i]["y"])) == "true") {
        i++;                                  // do the last 3 points make a left turn?
    }           //if yes, carry on..
    else {                                    // else the previous point (middle of our three) must be wrong
        coordinatesINNER.push(coordinatesHULL[i-1]); //add to inner array
        coordinatesHULL.splice(i-1,1);        // remove from hull array
        if (i >0) {
            i = i-1;                              //lower i accordingly
        }
    }
}
// checking last points are valid
var a = 0;
while (a != 1)   {
    
    if ((isLeftTurn(coordinatesHULL[coordinatesHULL.length-2]["x"],coordinatesHULL[coordinatesHULL.length-2]["y"],coordinatesHULL[coordinatesHULL.length-1]["x"],coordinatesHULL[coordinatesHULL.length-1]["y"],coordinatesHULL[0]["x"],coordinatesHULL[0]["y"])) == "false") {
                                    //right turn?
        coordinatesHULL.splice(coordinatesHULL.length - 1,1);      //if so, remove last point from array
    }
    else {a = 1;}       //until it works/is left turn
}

 

// //recreate array in form (x,y, x,y, x,y) for Polygon function.
var HULLxy = new Array();
for (var i=0; i < coordinatesHULL.length; i++) {      
    HULLxy.push(coordinatesHULL[i]["x"],coordinatesHULL[i]["y"]); // (may draw later over the top of everything else)
}

img.fillColour = "none";
img.lineColour = "blue";
    //================================\\
    //===| Delaunay triangulation |===\\  
    //================================\\
// my algorithm in a nutshell:
// we start with 3 arrays - one contains ALL the coords in our list, and never changes.
//                          one contains the hull coordinates, and one the inner coords (not part of hull)
//      we go round all the hull coords, and see if we can cut off any corners. To test whether a triangle is valid, we have 
//      to check that NONE of the coords in our set are inside the tri's circumcircle (ON circumcircle is okay, just means an arbitrary choice 
//      between 2 equally valid solutions). If a triangle is formed, we store the triangle coords, and redefine the hull 
//      Next we go round all the inner coords, for each one find the closest hull coord, and try the 2 possible triangles with this coord
//      and a consecutive hull coord.
//      we loop round these two algorithms until we find  all the triangles. If anything goes wrong and we are unsuccessful in both algorithms
//      we escape the loop with a warning.

var coordinates = new Array();      //combine x and y vals into 1 array, to keep together
for (var i = 0; i < Xcoord.length; i++)  {
    coordinates[i] = { x:Xcoord[i], y:Ycoord[i] } ;     //this is definitive list of all coordinates
}

var coordinatesOUTER = new Array();     // Any coords that are not part of hull or inside hull (ie removed from consideration).       
                                        // Used for errorchecking.

if( coordinatesHULL.length + coordinatesINNER.length + coordinatesOUTER.length  != coordinates.length ) {
    LogWarning ("\n!!!!!!!!!!!!!!!!!!!!!!!!!!!flagA\nARRAYS GONE WRONG\nhull:",coordinatesHULL.length," inner:",coordinatesINNER.length," outer:",coordinatesOUTER.length," total:",coordinates.length,"!!!!!!!!!!!!!!!!!!!!!!!!!!!\n");
}

var triangles = new Array();    //will contain coords of all 3 points for each element
var NumberOfTriangles = (2*Xcoord.length) - 2 - coordinatesHULL.length;         // for set of points with n points
                                                                                // and convex hull with k points
var a = 0;    
// no. of triangles = 2n - 2 - k (see http://groups.csail.mit.edu/graphics/classes/6.838/F01/lectures/Delaunay/Delaunay2D.ppt?bcsi_scan_103D199360250874=0&bcsi_scan_filename=Delaunay2D.ppt)

while (triangles.length < NumberOfTriangles)  { //keep iterating until we find all of 'em.
    var b = 0;   //to keep track of whether iterations are creating triangles each time
    
 
    for (var i = 0; i < coordinatesHULL.length; i++) {

// == // go once round hull coordinates, seeing if any corner can be cut off.

//      define triangle to test
        loop:
        if( coordinatesHULL.length > 2) { //only try if you are guaranteed enough points for a triangle!
            if (i == 0)  {
                var pointA = coordinatesHULL[coordinatesHULL.length-2];            // ensuring we loop round beginning-to-end
                var pointB = coordinatesHULL[coordinatesHULL.length-1];            // (i - 2 would not be defined if i < 2 )
                var pointC = coordinatesHULL[0];
            }
            else if (i == 1)  {
                var pointA = coordinatesHULL[coordinatesHULL.length-1];             //ditto
                var pointB = coordinatesHULL[0];
                var pointC = coordinatesHULL[1];
            }
            else {
                var pointA = coordinatesHULL[i-2];
                var pointB = coordinatesHULL[i-1];
                var pointC = coordinatesHULL[i];
            }
            
            var inCircum = 0;
            
        // check for points in circumcircle
            for (var j = 0; (j < coordinates.length && inCircum == 0); j++)  {
                var inCircum = 0;
                    if (isInCircumcircle(pointA["x"],pointA["y"],pointB["x"],pointB["y"],pointC["x"],pointC["y"],coordinates[j]["x"],coordinates[j]["y"]) == "true")  {
                        inCircum = 1;                          //if any of the points are within circumcircle of A,B,C, cannot be correct so move on (not necessary to check all once we've found one point in circumcircle)
                }
            }
                
//            LogPrint (isInCircumcircle(pointA["x"],pointA["y"],pointB["x"],pointB["y"],pointC["x"],pointC["y"],coordinates[j]["x"],coordinates[j]["y"]) == "true");

        //make triangles if appropriate   
            if(inCircum == 0)  {    //ie nothing was found in last loop
                            triangles[a] = { Ax:pointA["x"],Ay:pointA["y"],  HIC_A:pointA["HIC"],  Bx:pointB["x"],By:pointB["y"],  HIC_B:pointB["HIC"], Cx:pointC["x"],Cy:pointC["y"],  HIC_C:pointC["HIC"] };
//                        LogPrint ("tri");
                a++;
                b++;
                if (drawTriangulation == 1)  {
                    img.Polygon ( pointA["x"],pointA["y"],     pointB["x"],pointB["y"],   pointC["x"],pointC["y"] );
                }
                if (i != 0) {
                    coordinatesOUTER.push(coordinatesHULL[i-1]);  //banish external point to outer
                    coordinatesHULL.splice(i-1,1);    //remove from hull
                }
                else {              //if i = 0, value [i-1] undefined, but we know it is last point in array, so use pop() to remove it.
                    coordinatesOUTER.push( coordinatesHULL.pop() );  // banish external point to outer
                }
                if (i > 0)  {
                    i = i -1; //adjust i accordingly
                }
            }
         
            else break loop; //if we found a point in circumcircle, escape from loop
            
        }
    }


// == // go once round inner coordinates, seeing if any can be joined to the edge.
    if( coordinatesINNER.length > 0 && coordinatesHULL.length > 3) { // duh.

        for (var i = 0; i < coordinatesINNER.length; i++) {         // to avoid stretching across hull and splitting hull into two,
//     LogPrint (i);                                            // we will find closest hull coord, then take the closest of the two next to it.
                                                                // (rather than just taking the closest 2 coords)
                                                                // update: why not try the closest and EACH of the neighbouring points. No harm in that,
       //     find the closest point                            // may pick up more triangles that way.
            var distance = new Array();                         
            for (j = 0; j < coordinatesHULL.length; j++) {          // calculate distance from current point    
                distance[j] = Math.sqrt(squared( coordinatesHULL[j]["x"] - coordinatesINNER[i]["x"] ) + squared( coordinatesHULL[j]["y"] - coordinatesINNER[i]["y"] )) ;
            }
                   
            distance2 = distance.slice(0);        // make a copy of array
            distance2.sort(increasingnumber);     // sort copied array, leaving original intact for reference
            
            var smallest = arrayFirstPosition(distance,distance2.shift())    // which j had smallest length?

            for (var k = 1; k <= 2; k++)  {
                if( coordinatesINNER.length > 0 && coordinatesHULL.length > 1) { // duh.
                
                    if (k == 1 )  {        //    first time we try taking first neighbour (previous coord in array) 
           
                        if ( smallest == 0)  {        //special case, loop round
                            var secondSmallest = coordinatesHULL.length - 1;
                            var lowest = secondSmallest;
                            var highest = smallest; 
                        }
                        else {                
                            var secondSmallest = smallest - 1;
                            var lowest = secondSmallest;
                            var highest = smallest;          //to identify what order they appear in hull (to know where to insert inner coord in array etc)
                        }
                    }
                    else if (k == 2 )  {            //    next we try with second neighbour (following coord in array)
                        if (smallest == coordinatesHULL.length - 1) { //special case, loop around
                            var secondSmallest = 0;
                            var lowest = smallest;
                            var highest = secondSmallest;     //to identify what order they appear in hull (to know where to insert inner coord in array etc) 
                        }
                        else {
                            var secondSmallest = smallest + 1;
                            var lowest = smallest;
                            var highest = secondSmallest;     //to identify what order they appear in hull (to know where to insert inner coord in array etc) 
                        }
                            
                    }
                
               //     define triangle to test (depending on how we defined those variables)
                    var pointA = coordinatesINNER[i];
                    var pointB = coordinatesHULL[lowest];       // to ensure triangle labelled A,B,C anticlockwise
                    var pointC = coordinatesHULL[highest];    
    
                    
                //    check for points in circumcircle
                    var inCircum = 0;
    
                    for (var j = 0; (j < coordinates.length && inCircum == 0); j++) {
                            var inCircum = 0;
                            if (isInCircumcircle(pointA["x"],pointA["y"],pointB["x"],pointB["y"],pointC["x"],pointC["y"],coordinates[j]["x"],coordinates[j]["y"]) == "true") {
                                inCircum = 1;               //if any of the points are within circumcircle of A,B,C, cannot be correct so move on (not necessary to check all once we've found one point in circumcircle)
                        }
                    }
    
                //make triangles if appropriate   
                    if (inCircum == 0) {
                            triangles[a] = { Ax:pointA["x"],Ay:pointA["y"],  HIC_A:pointA["HIC"],  Bx:pointB["x"],By:pointB["y"],  HIC_B:pointB["HIC"], Cx:pointC["x"],Cy:pointC["y"],  HIC_C:pointC["HIC"] };
                        a++;
                        b++;
                        if (drawTriangulation == 1)  {
                            img.Polygon ( pointA["x"],pointA["y"],     pointB["x"],pointB["y"],   pointC["x"],pointC["y"] );
                        }
                        coordinatesHULL.splice(highest,0,coordinatesINNER[i]);    // add point to hull in right position
                        coordinatesINNER.splice(i,1);    // remove from inner coords
                        if (i > 0 )  {
                            i = i -1; //adjust accordingly
                        }
    
                    }
                }
            }           
        } 
    } 

    if (b == 0)  break //if no method has yielded any triangles at all during this iteration
}
   
   //check we've found all our tris.
if (triangles.length < NumberOfTriangles) {
    LogWarning ("\n",NumberOfTriangles - triangles.length," triangles were not found\n(",triangles.length," triangles found so far)\n");
break mainScript;   
}

else if (triangles.length > NumberOfTriangles) {
   LogWarning ("\nWe somehow have ",triangles.length - NumberOfTriangles," too many triangles - possible duplicates??\n");
}

else if (triangles.length == NumberOfTriangles) {
    LogPrint ("All ",triangles.length," Delaunay triangles found\n");
}

if( coordinatesHULL.length + coordinatesINNER.length + coordinatesOUTER.length  != coordinates.length ) {
                    LogWarning ("\n!!!!!!!!!!!!!!!!!!!!!!!!!!!flagD\nARRAYS GONE WRONG\nhull:",coordinatesHULL.length," inner:",coordinatesINNER.length," outer:",coordinatesOUTER.length," total:",coordinates.length,"!!!!!!!!!!!!!!!!!!!!!!!!!!!\n");
}


//======================================//
//==| interpolate HICs at gridpoints |==//
//======================================//

LogPrint ("Interpolating...\n");

for ( i = 0; i < coordinatesGRID.length; i++ )  {   // for each point
    if (coordinatesGRID[i]["HIC"] =~ /null/i) { // if HIC is not defined
        loop3:
        for (j = 0; j < triangles.length; j++)  {   //go through triangles
           triangleXcoords = new Array (parseFloat(triangles[j]["Ax"]),parseFloat(triangles[j]["Bx"]),parseFloat(triangles[j]["Cx"])); //define coords as array for
           triangleYcoords = new Array (parseFloat(triangles[j]["Ay"]),parseFloat(triangles[j]["By"]),parseFloat(triangles[j]["Cy"])); //pointInPolygon function
           
           if ( pointInPolygon(coordinatesGRID[i]["x"],coordinatesGRID[i]["y"],triangleXcoords,triangleYcoords ) == "true" )  {
                coordinatesGRID[i]["HIC"] = triangleInterpolate(coordinatesGRID[i]["x"],coordinatesGRID[i]["y"],triangles[j]["Ax"],triangles[j]["Ay"], triangles[j]["HIC_A"],triangles[j]["Bx"],triangles[j]["By"], triangles[j]["HIC_B"],triangles[j]["Cx"],triangles[j]["Cy"], triangles[j]["HIC_C"]);
                HIC_NEW[i] = triangleInterpolate(coordinatesGRID[i]["x"],coordinatesGRID[i]["y"],triangles[j]["Ax"],triangles[j]["Ay"], triangles[j]["HIC_A"],triangles[j]["Bx"],triangles[j]["By"], triangles[j]["HIC_B"],triangles[j]["Cx"],triangles[j]["Cy"], triangles[j]["HIC_C"]);
                break loop3;  //if this point is inside current tri; interpolate using this tri's HICs and break out of loop (move onto next gridpoint) 
           }
           if (j == triangles.length-1) {       //if we've gone through them all with no  luck
                coordinatesGRID[i]["HIC"] = "not possible to interpolate";
                HIC_NEW[i] = "not possible to interpolate";
           }
        }
    }
}    


//=========================================//
//==| extrapolate HICs for any unknowns |==//
//=========================================//


var to_extrapolate = 0;
for ( i = 0; i < coordinatesGRID.length; i++ )  {   // for each point
    if (coordinatesGRID[i]["HIC"] == "not possible to interpolate") { // if HIC is still not defined
        to_extrapolate++;    //count how many we need to extrapolate for
    }
}



if (to_extrapolate > 0) {
    LogPrint ("Extrapolating...\n");

// Arrays are sorted in ascending X then Y - this will make the extrapolation calculations quicker.
// We need the number of rows and columns in the grid..

    var nrows = gridpointsY.length;
    var ncols = gridpointsX.length;
}

var extrapolated = 0;

loop6:
while ( extrapolated < to_extrapolate)
{

    var c = 0;
    
// Arrays to store HIC values and the index in the coordinatesGRID[] array.
// We don't store the values straight away in coordinatesGRID[], we want to grow the points
// out like an advancing wave front so don't want calculated values to have an influence on
// points we're still calculating.

    var HIC_vals = new Array();
    var indices  = new Array();
    
    for ( i = 0; i < coordinatesGRID.length; i++ )  // for each point
    {
// if HIC is still not defined
        if (coordinatesGRID[i]["HIC"] == "not possible to interpolate" || isNaN(coordinatesGRID[i]["HIC"]) )
        { 
            var nearby_toTheLeft = new Array();
            var nearby_toTheRight = new Array();
            var nearby_upAbove = new Array();
            var nearby_downBelow = new Array();
    
            var pointC = coordinatesGRID[i];
            
// Try to get two values from points adjacent to current point so we can extrapolate.
// Because the coordinatesGRID[] array is sorted in ascending X then Y we can easily get
// the two closest points.
            
            var row = i % nrows;
            var col = (i-row) / nrows;
            
            if(col >= 2)  // nearby_toTheLeft
            {
              if(typeof(coordinatesGRID[i-2*nrows]["HIC"]) == "number" &&
                 typeof(coordinatesGRID[i-nrows]["HIC"])   == "number")
              {
                nearby_toTheLeft[0] = coordinatesGRID[i-2*nrows];
                nearby_toTheLeft[1] = coordinatesGRID[i-nrows];
              }
            }
            
            if(col < ncols-2)  // nearby_toTheRight
            {
              if(typeof(coordinatesGRID[i+nrows]["HIC"])   == "number" &&
                 typeof(coordinatesGRID[i+2*nrows]["HIC"]) == "number")
              {
                nearby_toTheRight[0] = coordinatesGRID[i+nrows];
                nearby_toTheRight[1] = coordinatesGRID[i+2*nrows];
              }
            }
            
            if(row >= 2)  // nearby_upAbove
            {
              if(typeof(coordinatesGRID[i-2]["HIC"]) == "number" &&
                 typeof(coordinatesGRID[i-1]["HIC"]) == "number")
              {
                nearby_upAbove[0] = coordinatesGRID[i-2];
                nearby_upAbove[1] = coordinatesGRID[i-1];
              }
            }
            
            if(row < nrows-2)  // nearby_downBelow
            {
              if(typeof(coordinatesGRID[i+1]["HIC"])   == "number" &&
                 typeof(coordinatesGRID[i+2]["HIC"]) == "number")
              {
                nearby_downBelow[0] = coordinatesGRID[i+1];
                nearby_downBelow[1] = coordinatesGRID[i+2];
              }
            }

// Use all possible directions to extrapolate and then average...
// Average is weighted based on proximity of points extrapolating from...
            
            if(nearby_toTheLeft.length == 2 || nearby_toTheRight.length == 2 ||
               nearby_upAbove.length   == 2 || nearby_downBelow.length  == 2)
            {
              var vals     = new Array();
              var distance = new Array();
              var avg;  // Average distance of the two points to the extrapolated point
              var fraction;
              
              if(nearby_toTheLeft.length == 2)
              {
                fraction = (pointC["x"] - nearby_toTheLeft[0]["x"])/(nearby_toTheLeft[1]["x"] - nearby_toTheLeft[0]["x"]);
                vals.push(nearby_toTheLeft[0]["HIC"] + (fraction * (nearby_toTheLeft[1]["HIC"] - nearby_toTheLeft[0]["HIC"])));
                
                avg = 0.5 * (nearby_toTheLeft[0]["x"] + nearby_toTheLeft[1]["x"]);
                distance.push(Math.abs(avg - pointC["x"]));
              }
              
              if(nearby_toTheRight.length == 2)
              {
                fraction = (pointC["x"] - nearby_toTheRight[0]["x"])/(nearby_toTheRight[1]["x"] - nearby_toTheRight[0]["x"]);
                vals.push(nearby_toTheRight[0]["HIC"] + (fraction * (nearby_toTheRight[1]["HIC"] - nearby_toTheRight[0]["HIC"])));
                
                avg = 0.5 * (nearby_toTheRight[0]["x"] + nearby_toTheRight[1]["x"]);
                distance.push(Math.abs(avg - pointC["x"]));
              }
              
              if(nearby_upAbove.length == 2)
              {
                fraction = (pointC["y"] - nearby_upAbove[0]["y"])/(nearby_upAbove[1]["y"] - nearby_upAbove[0]["y"]);
                vals.push(nearby_upAbove[0]["HIC"] + (fraction * (nearby_upAbove[1]["HIC"] - nearby_upAbove[0]["HIC"])));
                
                avg = 0.5 * (nearby_upAbove[0]["y"] + nearby_upAbove[1]["y"]);
                distance.push(Math.abs(avg - pointC["y"]));
              }
              
              if(nearby_downBelow.length == 2)
              {
                fraction = (pointC["y"] - nearby_downBelow[0]["y"])/(nearby_downBelow[1]["y"] - nearby_downBelow[0]["y"]);
                vals.push(nearby_downBelow[0]["HIC"] + (fraction * (nearby_downBelow[1]["HIC"] - nearby_downBelow[0]["HIC"])));
                
                avg = 0.5 * (nearby_downBelow[0]["y"] + nearby_downBelow[1]["y"]);
                distance.push(Math.abs(avg - pointC["y"]));
              }
              
// Calculate weighting factors based on distance

              var total_dist = 0.0;
              
              for(var j=0; j<distance.length; j++) total_dist += distance[j];

              var weight = new Array();
              
              for(var j=0; j<distance.length; j++) weight.push(distance[j]/total_dist);
              
// Now sum weighted values
              
              var extrapolation = 0.0;
              
              for(var j=0; j<vals.length; j++) extrapolation += vals[j] * weight[j];
              
// Store values
              if (isFinite(extrapolation) == true)
              {             
                  HIC_vals.push(extrapolation);
                  indices.push(i);

                  extrapolated++;
                  c++;
              }
            }

        }
    }
      
// if this iteration hasn't got anywhere, we're never gonna.

    if (c == 0) break loop6;
    
// Now add values to arrays

    for(var i=0; i<HIC_vals.length; i++)
    {
      coordinatesGRID[indices[i]]["HIC"] = HIC_vals[i];
      HIC_NEW[indices[i]]                = HIC_vals[i];
    }
}

// Clamp HIC values to 0.01
for(var i=0; i<HIC_NEW.length; i++)
{
  if(HIC_NEW[i] <= 0.0) HIC_NEW[i] = 0.01;
  if(coordinatesGRID[i]["HIC"] <= 0.0) coordinatesGRID[i]["HIC"] = 0.01;
}

//======================//    
//==| define squares |==//
//======================//   
        // Joining up regular grid of coords into rectangles
        // Based on coords stored in XcoordNEW, YcoordNEW.
LogPrint ("Drawing contours & calculating areas...\n");


var pointAx = "";        //   quad defined by points      A       B       
var pointAy = "";        //                                 
var pointBx = "";        //                               C       D                              
var pointBy = "";        //    where: A is reference node;                                                       
var pointCx = "";        //           B has same y and smallest available greater x;
var pointCy = "";        //           C vice versa;       
var pointDx = "";        //           D either along from C or down from B
var pointDy = "";        // 
var AB_intersect = undefined;     //where perim cuts quad.
var BD_intersect = undefined;           //need to undefine them so if not defined  
var DC_intersect = undefined;                   //later we can tell.
var CA_intersect = undefined;
var areas = new Array();    
var vals = allContourValues; 


var areaTOTAL = new Array();

for (var valNo = 0; valNo < vals.length ; valNo++)  {
    val = vals[valNo];
    areas[val] = new Array();           //declaring each array.
}

// Bucket search
var buckets = new Array();
var minX = ArrayMin(XcoordNEW);
var maxX = ArrayMax(XcoordNEW);
var minY = ArrayMin(YcoordNEW);
var maxY = ArrayMax(YcoordNEW);

// use a tolerance to ensure to get stuff just outside the bucket as well
var tol = tolerance*50.0;

// 10x10 grid
for(var i=0; i<10; i++)
{
    for(var j=0; j<10; j++)
    {
        var bucket = new Object();

        bucket.minx = minX + i*((maxX-minX)/10.0);
        bucket.maxx = minX + (i+1)*((maxX-minX)/10.0);

        bucket.miny = minY + j*((maxY-minY)/10.0);
        bucket.maxy = minY + (j+1)*((maxY-minY)/10.0);

        bucket.points = new Array();

        for(var k=0; k < XcoordNEW.length; k++ )
        {
            var x = XcoordNEW[k];
            var y = YcoordNEW[k];

            if(x <= bucket.maxx+tol && x >= bucket.minx-tol && y <= bucket.maxy+tol && y >= bucket.miny-tol)
            {
                bucket.points.push(k);
            }
        }

        buckets.push(bucket);
    }
}

loop4:          // start at each coord, one by one, and define its associated quad.
for (var i=0; i < XcoordNEW.length; i++ ) {
    var pointAx = "";        //   quad defined by points      A       B       
    var pointAy = "";        //                                 
    var pointBx = "";        //                               C       D                              
    var pointBy = "";        //    where: A is reference node;                                                       
    var pointCx = "";        //           B has same y and smallest available greater x;
    var pointCy = "";        //           C vice versa;       
    var pointDx = "";        //           D either along from C or down from B
    var pointDy = "";        // 
    var AB_intersect = undefined;     //where perim cuts quad.
    var BD_intersect = undefined;           //need to undefine them so if not defined  
    var DC_intersect = undefined;                   //later we can tell.
    var CA_intersect = undefined;
    var AE_intersect = undefined;
    var BE_intersect = undefined;
    var CE_intersect = undefined;
    var DE_intersect = undefined;

    var area = 0;       // to reset each time
    if (HIC_NEW[i] == "not possible to interpolate") { continue loop4; } //if no HIC defined at A, no point continuing 

//  //== define point A ==//
    pointAx = XcoordNEW[i];             // reference node, just take coords from arrays
    pointAy= YcoordNEW[i];
    
//  //== define point B ==//
    var IndexBcoord = new Array();
    var PositionB = new Array();
    var ValueB = new Array();
    var minB = "";

//  //== define point C ==//
    var IndexCcoord = new Array();
    var PositionC = new Array();
    var ValueC = new Array();
    var minC = "";
        
// find appropriate buckets and just loop through those points
    for(var b = 0; b<buckets.length; b++)
    {
        var bucket = buckets[b];

        if(XcoordNEW[i] <= bucket.maxx && XcoordNEW[i] >= bucket.minx && YcoordNEW[i] <= bucket.maxy && YcoordNEW[i] >= bucket.miny)
        {
            var points = bucket.points;

            for(var p=0; p<bucket.points.length; p++)
            {
                j = bucket.points[p];

                if ((( YcoordNEW[j] >= YcoordNEW[i] - tolerance )&&( YcoordNEW[j] <= YcoordNEW[i] + tolerance ))&&( XcoordNEW[j] > XcoordNEW[i] ))  {
                    PositionB.push(j);       // returns an array of the positions of coordinates with same Y value as current point; & greater X.
                    ValueB.push(XcoordNEW[j]);   // keeps track of the actual x vals, with same positions.
                }
            }
        }

        if(XcoordNEW[i] <= bucket.maxx && XcoordNEW[i] >= bucket.minx && YcoordNEW[i] <= bucket.maxy && YcoordNEW[i] >= bucket.miny)
        {
            var points = bucket.points;

            for(var p=0; p<bucket.points.length; p++)
            {
                j = bucket.points[p];

                if (( XcoordNEW[j] <= XcoordNEW[i] + tolerance )&&( XcoordNEW[j] >= XcoordNEW[i] - tolerance )&&( YcoordNEW[j] > YcoordNEW[i] ))  {
                    PositionC.push(j);       // returns an array of the positions of coordinates with same X value as current & greater X.
                    ValueC.push(YcoordNEW[j]);   // keeps track of the actual x vals, with same positions.
                }
            }
        }
    }

    if (typeof(ValueB[0]) != "undefined") {     // if there are no arguments to feed to ArrayMin, it'll come back with - MAX_VALUE - so only perform if we have arguments
        var minB = ArrayMin(ValueB);            //return the smallest value that fulfils criteria
    }
    var PositionOfMin = arrayPosition(ValueB,minB);         // the position of the minimum within ValueB
    var forB = PositionB[PositionOfMin];                    // the position of this value within XcoordNEWs and YcoordNEWs, given by PositionB
        
        
    pointBx = XcoordNEW[(forB)];              // define point B 
    pointBy= YcoordNEW[(forB)];

    if (HIC_NEW[forB] == "not possible to interpolate" || typeof(pointBx) == "undefined" ) { continue loop4; } //if no HIC defined at B, or B not defined, no point continuing 

    if (typeof(ValueC[0]) != "undefined") {
        var minC = ArrayMin(ValueC);
    }
    var PositionOfMin = arrayPosition(ValueC,minC);         //the position of the minimum within orderC
    var forC = PositionC[PositionOfMin];                    //the position of this Xvalue within XcoordNEWs, given by PositionC
        
    pointCx = XcoordNEW[(forC)];              // define point C 
    pointCy= YcoordNEW[(forC)];

    if (HIC_NEW[forC] == "not possible to interpolate" || typeof(pointCx) == "undefined" ) { continue loop4; } //if no HIC defined at C, or C not defined, no point continuing 

//  //== define point D ==//
    // - - along from C
    var IndexD1coord = new Array();
    var PositionD1 = new Array();
    var OrderD1 = new Array();
    var minD1 = "";
        
// find appropriate buckets and just loop through those points
    for(var b = 0; b<buckets.length; b++)
    {
        var bucket = buckets[b];

        if(XcoordNEW[i] <= bucket.maxx && XcoordNEW[i] >= bucket.minx && YcoordNEW[i] <= bucket.maxy && YcoordNEW[i] >= bucket.miny)
        {
            var points = bucket.points;

            for(var p=0; p<bucket.points.length; p++)
            {
                j = bucket.points[p];

                if (( YcoordNEW[j] >= YcoordNEW[forC] - tolerance)&&( YcoordNEW[j] <= YcoordNEW[forC] + tolerance)&&( XcoordNEW[j] > XcoordNEW[forC] ))  {   // based on point C (D is to C what B is to A)
                    PositionD1.push(j);       // returns an array of the positions of coordinates with same Y value as current & greater X.
                    OrderD1.push(XcoordNEW[j]);   // keeps track of the actual x vals, with same positions.
                }
            }
        }
    }
    if (typeof(OrderD1[0]) != "undefined") {
        var minD1 = ArrayMin(OrderD1);
    }
    var PositionOfMin1 = arrayPosition(OrderD1,minD1);         //the position of the minimum within orderD
    var forD = PositionD1[PositionOfMin1];                    //the position of this Xvalue within XcoordNEWs, given by PositionD
        
        
    pointDx = XcoordNEW[forD];              // define point D 
    pointDy= YcoordNEW[forD];
    
    if (HIC_NEW[forD] == "not possible to interpolate" || typeof(pointDx) == "undefined" ) { continue loop4; } //if no HIC defined at D, or D not defined, no point continuing 
        

    var tolerance = (Math.abs((pointDx-pointCx)+(pointBy-pointDy))/2)/1000; //tolerance of 0.1% of gridsize
    

    if (pointInPolygon(pointAx,pointAy,XPerimCoords,YPerimCoords) == "false" &&  pointInPolygon(pointBx,pointBy,XPerimCoords,YPerimCoords) == "false" &&  pointInPolygon(pointCx,pointCy,XPerimCoords,YPerimCoords) == "false" &&  pointInPolygon(pointDx,pointDy,XPerimCoords,YPerimCoords) == "false" ) {
        continue loop4;         //if no corner is within perimeter, no point continuing
    }
    
    img.fillColour = "none";
    img.lineColour = "Red";
    
    if ((typeof(pointBx) != "undefined")&&(typeof(pointCx) != "undefined")&&(typeof(pointDx) != "undefined"))  {       //are there 4 points available? (Ax and Ay will always be defined)
        if (drawRectangles == 1 ) {
            img.Polygon(pointAx,pointAy,pointBx,pointBy,pointDx,pointDy,pointCx,pointCy);
        }
    }
    else {
        continue loop4;
    }


// =================================================================================================================
//  start to work a few things out ready for area calcs & contour plotting.
// =================================================================================================================

// ==  declare/define variables for area calcs
        var forA = i;       //simply for ease of naming
        var HIC_A = HIC_NEW[forA];      //get HICs
        var HIC_B = HIC_NEW[forB];
        var HIC_C = HIC_NEW[forC];              
        var HIC_D = HIC_NEW[forD];
    
        var A_in_perim = 0;        //declare variables, reset to 0 each time
        var B_in_perim = 0;
        var C_in_perim = 0;     //we are going to calculate all of these OUTside HIC loop
        var D_in_perim = 0;      
        var E_in_perim = 0;
    
        var A_on_PerimFlag = 0;
        var B_on_PerimFlag = 0;
        var C_on_PerimFlag = 0;
        var D_on_PerimFlag = 0;
        var E_on_PerimFlag = 0;
        
        pointEx = pointAx + ((pointBx - pointAx)/2);        //point E is midpoint of square
        pointEy = pointCy + ((pointAy - pointCy)/2); 

// =================================================================================================================
//  find whether corners ON perim and/or IN perim.   AND look at point E in middle of quad.
// =================================================================================================================
        
        if (pointInPolygon(pointAx,pointAy,XPerimCoords,YPerimCoords) == "true") {
            A_in_perim = 1;                      //is each point in the perimeter or not?
        }
        if (pointInPolygon(pointBx,pointBy,XPerimCoords,YPerimCoords) == "true") {
            B_in_perim = 1;                           //just calculate once, then just reference these.
        }
        if (pointInPolygon(pointCx,pointCy,XPerimCoords,YPerimCoords) == "true") {
            C_in_perim = 1;
        }
        if (pointInPolygon(pointDx,pointDy,XPerimCoords,YPerimCoords) == "true") {
            D_in_perim = 1;
        }
        if (pointInPolygon(pointEx,pointEy,XPerimCoords,YPerimCoords) == "true") {
            E_in_perim = 1;
        }

        // define if all are in perim, for easy reference
        var allCornersInPerim = 0;
        if ( A_in_perim == 1 && B_in_perim == 1 && C_in_perim == 1 && D_in_perim == 1 ) {
            allCornersInPerim = 1;
        }

        // find out if any corners are ON perim.   
        var nearperim = distanceToPerim(pointAx,pointAy,XPerimCoords,YPerimCoords);
        if (nearperim != -1  && nearperim < tolerance)  {       //if perimeter v close to corner of square (cannot say whether its ON or not, as floating points.) 
            A_on_PerimFlag = 1;
        }                 
        
        var nearperim = distanceToPerim(pointBx,pointBy,XPerimCoords,YPerimCoords);
        if (nearperim != -1  && nearperim < tolerance)  {
            B_on_PerimFlag = 1;
        }                
        
        var nearperim = distanceToPerim(pointCx,pointCy,XPerimCoords,YPerimCoords);
        if (nearperim != -1  && nearperim < tolerance)  {
            C_on_PerimFlag = 1;
        }               
        
        var nearperim = distanceToPerim(pointDx,pointDy,XPerimCoords,YPerimCoords);
        if (nearperim != -1  && nearperim < tolerance)  {
            D_on_PerimFlag = 1;
        }              
        var nearperim = distanceToPerim(pointEx,pointEy,XPerimCoords,YPerimCoords);
        if (nearperim != -1  && nearperim < tolerance)  {
            E_on_PerimFlag = 1;
        }               

 
// =======================================================================================================================
//  find perim intersection points. We only allow for a maximum of 1 non-corner intersection point on any side - but could output how many failed rectangles if we like for validity quote.
// =======================================================================================================================
                                    //may come back undefined if perim doesnt cross
        AB_intersect = perimIntersection(pointAx,pointAy,pointBx,pointBy,XPerimCoords,YPerimCoords);  
        var NoOfIntersections = howManyIntersections(pointAx,pointAy,pointBx,pointBy,XPerimCoords,YPerimCoords);
        
        if (NoOfIntersections == 0) { AB_intersect = undefined; }
        else if (NoOfIntersections == 1) {
            if (typeof(AB_intersect) != "undefined") {
                if (( (AB_intersect["x"] < pointAx+tolerance)  &&  (AB_intersect["x"] > pointAx-tolerance) &&  (AB_intersect["y"] < pointAy+tolerance)  &&  (AB_intersect["y"] > pointAy-tolerance) ) ||   ( (AB_intersect["x"] < pointBx+tolerance)  &&  (AB_intersect["x"] > pointBx-tolerance) &&  (AB_intersect["y"] < pointBy+tolerance)  &&  (AB_intersect["y"] > pointBy-tolerance) )    ) {
                        //if single intersection is defined and is equal to either point A or B, undefine it. //so we dont have flag at side and corner unnecessarily.
                    AB_intersect = undefined;  
                }
            }
        }
        else {
            AB_intersect_MULTIPLE= perimMultipleIntersections(pointAx,pointAy,pointBx,pointBy,XPerimCoords,YPerimCoords);
            for ( var j = 0; j <AB_intersect_MULTIPLE.length; j++ )  {
                if (typeof(AB_intersect_MULTIPLE[j]) != "undefined") {
                    if (( (AB_intersect_MULTIPLE[j]["x"] < pointAx+tolerance)  &&  (AB_intersect_MULTIPLE[j]["x"] > pointAx-tolerance) &&  (AB_intersect_MULTIPLE[j]["y"] < pointAy+tolerance)  &&  (AB_intersect_MULTIPLE[j]["y"] > pointAy-tolerance) ) ||   ( (AB_intersect_MULTIPLE[j]["x"] < pointBx+tolerance)  &&  (AB_intersect_MULTIPLE[j]["x"] > pointBx-tolerance) &&  (AB_intersect_MULTIPLE[j]["y"] < pointBy+tolerance)  &&  (AB_intersect_MULTIPLE[j]["y"] > pointBy-tolerance) )    ) {
                            //if single intersection is defined and is equal to either point A or B, undefine it. //so we dont have flag at side and corner unnecessarily.
                        AB_intersect_MULTIPLE.splice(j,1);       //remove from array
                        j = j-1;        //adjust accordingly
                    }
                } 
            }
            if (AB_intersect_MULTIPLE.length == 0)  {        //if none stood the test
                AB_intersect = undefined;  
            }
            else if (AB_intersect_MULTIPLE.length = 1)  { 
                AB_intersect = AB_intersect_MULTIPLE[0];
            }
            else {
                var warning = "failed";
            }
        }

            
        BD_intersect = perimIntersection(pointBx,pointBy,pointDx,pointDy,XPerimCoords,YPerimCoords);
        var NoOfIntersections = howManyIntersections(pointBx,pointBy,pointDx,pointDy,XPerimCoords,YPerimCoords);

        if (NoOfIntersections == 0) { BD_intersect = undefined; }
        else if (NoOfIntersections == 1) {
            if (typeof(BD_intersect) != "undefined")  {
                if (    ( (BD_intersect["x"] < pointAx+tolerance)  &&  (BD_intersect["x"] > pointAx-tolerance) &&  (BD_intersect["y"] < pointAy+tolerance)  &&  (BD_intersect["y"] > pointAy-tolerance) ) ||   ( (BD_intersect["x"] < pointBx+tolerance)  &&  (BD_intersect["x"] > pointBx-tolerance) &&  (BD_intersect["y"] < pointBy+tolerance)  &&  (BD_intersect["y"] > pointBy-tolerance) )  ) { 
                           //if intersection is defined and is equal to either point A or B, undefine it.       
                    BD_intersect = undefined;      
                }
            }
        }
        else {
            BD_intersect_MULTIPLE= perimMultipleIntersections(pointBx,pointBy,pointDx,pointDy,XPerimCoords,YPerimCoords);
            for ( var j = 0; j <BD_intersect_MULTIPLE.length; j++ )  {
               if (typeof(BD_intersect_MULTIPLE[j]) != "undefined")  {
                    if (    ( (BD_intersect_MULTIPLE[j]["x"] < pointBx+tolerance)  &&  (BD_intersect_MULTIPLE[j]["x"] > pointBx-tolerance) &&  (BD_intersect_MULTIPLE[j]["y"] < pointBy+tolerance)  &&  (BD_intersect_MULTIPLE[j]["y"] > pointBy-tolerance) ) ||   ( (BD_intersect_MULTIPLE[j]["x"] < pointDx+tolerance)  &&  (BD_intersect_MULTIPLE[j]["x"] > pointDx-tolerance) &&  (BD_intersect_MULTIPLE[j]["y"] < pointDy+tolerance)  &&  (BD_intersect_MULTIPLE[j]["y"] > pointDy-tolerance) )  ) { 
                            //if single intersection is defined and is equal to either point B or D, undefine it. //so we dont have flag at side and corner unnecessarily.
                        BD_intersect_MULTIPLE.splice(j,1);       //remove from array
                        j = j-1;        //adjust accordingly
                    }
                }
            }
            if (BD_intersect_MULTIPLE.length == 0)  {        //if none stood the test
                BD_intersect = undefined;  
            }
            else if (BD_intersect_MULTIPLE.length = 1)  { 
                BD_intersect = BD_intersect_MULTIPLE[0];
            }
            else {
                var warning = "failed";
            }
        }

        DC_intersect = perimIntersection(pointDx,pointDy,pointCx,pointCy,XPerimCoords,YPerimCoords);
        var NoOfIntersections = howManyIntersections(pointCx,pointCy,pointDx,pointDy,XPerimCoords,YPerimCoords);

        if (NoOfIntersections == 0) { DC_intersect = undefined; }
        else if (NoOfIntersections == 1) {
            if (typeof(DC_intersect) != "undefined")   { 
                if (   ( (DC_intersect["x"] < pointDx+tolerance)  &&  (DC_intersect["x"] > pointDx-tolerance) &&  (DC_intersect["y"] < pointDy+tolerance)  &&  (DC_intersect["y"] > pointDy-tolerance) ) || ( (DC_intersect["x"] < pointCx+tolerance)  &&  (DC_intersect["x"] > pointCx-tolerance) &&  (DC_intersect["y"] < pointCy+tolerance)  &&  (DC_intersect["y"] > pointCy-tolerance) )    ) {
                        //if intersection is defined and is equal to either point A or B, undefine it.       
                    DC_intersect = undefined;      
                }
            }
        }
        else  {
            DC_intersect_MULTIPLE= perimMultipleIntersections(pointDx,pointDy,pointCx,pointCy,XPerimCoords,YPerimCoords);
            for ( var j = 0; j <DC_intersect_MULTIPLE.length; j++ )  {
               if (typeof(DC_intersect_MULTIPLE[j]) != "undefined")  {
                    if (    ( (DC_intersect_MULTIPLE[j]["x"] < pointDx+tolerance)  &&  (DC_intersect_MULTIPLE[j]["x"] > pointDx-tolerance) &&  (DC_intersect_MULTIPLE[j]["y"] < pointDy+tolerance)  &&  (DC_intersect_MULTIPLE[j]["y"] > pointDy-tolerance) ) ||   ( (DC_intersect_MULTIPLE[j]["x"] < pointCx+tolerance)  &&  (DC_intersect_MULTIPLE[j]["x"] > pointCx-tolerance) &&  (DC_intersect_MULTIPLE[j]["y"] < pointCy+tolerance)  &&  (DC_intersect_MULTIPLE[j]["y"] > pointCy-tolerance) )  ) { 
                            //if single intersection is defined and is equal to either point D or C, undefine it. //so we dont have flag at side and corner unnecessarily.
                        DC_intersect_MULTIPLE.splice(j,1);       //remove from array
                        j = j-1;        //adjust accordingly
                    }
                }
            }
            if (DC_intersect_MULTIPLE.length == 0)  {        //if none stood the test
                DC_intersect = undefined;  
            }
            else if (DC_intersect_MULTIPLE.length = 1)  { 
                DC_intersect = DC_intersect_MULTIPLE[0];
            }
            else {
                var warning = "failed";
            }
        }
        CA_intersect = perimIntersection(pointCx,pointCy,pointAx,pointAy,XPerimCoords,YPerimCoords);
        var NoOfIntersections = howManyIntersections(pointCx,pointCy,pointAx,pointAy,XPerimCoords,YPerimCoords);
        if ( NoOfIntersections == 0 ) { CA_intersect = undefined; 
        }
        else if (NoOfIntersections == 1) {
            if (typeof(CA_intersect) != "undefined")  { 
                if (   ( (CA_intersect["x"] < pointCx+tolerance)  &&  (CA_intersect["x"] > pointCx-tolerance) &&  (CA_intersect["y"] < pointCy+tolerance)  &&  (CA_intersect["y"] > pointCy-tolerance) )  ||         ( (CA_intersect["x"] < pointAx+tolerance)  &&  (CA_intersect["x"] > pointAx-tolerance) &&  (CA_intersect["y"] < pointAy+tolerance)  &&  (CA_intersect["y"] > pointAy-tolerance) )    ) {
                        //if intersection is defined and is equal to either point A or B, undefine it.       
                    CA_intersect = undefined;                                       //   |==> we want this to be flagged as a corner only 
                }
            }
        }
        else {
            CA_intersect_MULTIPLE= perimMultipleIntersections(pointCx,pointCy,pointAx,pointAy,XPerimCoords,YPerimCoords);
            for (l = 0; l < CA_intersect_MULTIPLE.length; l++)  {
            }
            for ( var j = 0; j < CA_intersect_MULTIPLE.length; j++ )  {
                if (typeof(CA_intersect_MULTIPLE[j]) != "undefined")  { 
                    if (   ( (CA_intersect_MULTIPLE[j]["x"] < pointCx+tolerance)  &&  (CA_intersect_MULTIPLE[j]["x"] > pointCx-tolerance) &&  (CA_intersect_MULTIPLE[j]["y"] < pointCy+tolerance)  &&  (CA_intersect_MULTIPLE[j]["y"] > pointCy-tolerance) )  ||         ( (CA_intersect_MULTIPLE[j]["x"] < pointAx+tolerance)  &&  (CA_intersect_MULTIPLE[j]["x"] > pointAx-tolerance) &&  (CA_intersect_MULTIPLE[j]["y"] < pointAy+tolerance)  &&  (CA_intersect_MULTIPLE[j]["y"] > pointAy-tolerance) )    ) {
                            //if intersection is defined and is equal to either point A or B, undefine it.       
                        CA_intersect_MULTIPLE.splice(j,1);                        //   |==> we want this to be flagged as a corner only 
                        j = j-1;
                    }
                }
            }
            if (CA_intersect_MULTIPLE.length == 0)  {        //if none stood the test
                CA_intersect = undefined;  
            }
            else if (CA_intersect_MULTIPLE.length = 1)  { 
                CA_intersect = CA_intersect_MULTIPLE[0];
            }
            else {
                var warning = "failed";
            }
        }

//***// AE INTERSECT //***//
        AE_intersect = perimIntersection(pointAx,pointAy,pointEx,pointEy,XPerimCoords,YPerimCoords);
        var NoOfIntersections = howManyIntersections(pointAx,pointAy,pointEx,pointEy,XPerimCoords,YPerimCoords);
        if ( NoOfIntersections == 0 ) { AE_intersect = undefined; 
        }
        else if (NoOfIntersections == 1) {
            if (typeof(AE_intersect) != "undefined")  { 
                if (   ( (AE_intersect["x"] < pointAx+tolerance)  &&  (AE_intersect["x"] > pointAx-tolerance) &&  (AE_intersect["y"] < pointAy+tolerance)  &&  (AE_intersect["y"] > pointAy-tolerance) )  ||         ( (AE_intersect["x"] < pointEx+tolerance)  &&  (AE_intersect["x"] > pointEx-tolerance) &&  (AE_intersect["y"] < pointEy+tolerance)  &&  (AE_intersect["y"] > pointEy-tolerance) )    ) {
                        //if intersection is defined and is equal to either point E or B, undefine it.       
                    AE_intersect = undefined;                                       //   |==> we want this to be flagged as a corner only 
                }
            }
        }
        else {
            AE_intersect_MULTIPLE= perimMultipleIntersections(pointAx,pointAy,pointEx,pointEy,XPerimCoords,YPerimCoords);
            for (l = 0; l < AE_intersect_MULTIPLE.length; l++)  {
            }
            for ( var j = 0; j < AE_intersect_MULTIPLE.length; j++ )  {
                if (typeof(AE_intersect_MULTIPLE[j]) != "undefined")  { 
                    if (   ( (AE_intersect_MULTIPLE[j]["x"] < pointAx+tolerance)  &&  (AE_intersect_MULTIPLE[j]["x"] > pointAx-tolerance) &&  (AE_intersect_MULTIPLE[j]["y"] < pointAy+tolerance)  &&  (AE_intersect_MULTIPLE[j]["y"] > pointAy-tolerance) )  ||         ( (AE_intersect_MULTIPLE[j]["x"] < pointEx+tolerance)  &&  (AE_intersect_MULTIPLE[j]["x"] > pointEx-tolerance) &&  (AE_intersect_MULTIPLE[j]["y"] < pointEy+tolerance)  &&  (AE_intersect_MULTIPLE[j]["y"] > pointEy-tolerance) )    ) {
                            //if intersection is defined and is equal to either point E or B, undefine it.       
                        AE_intersect_MULTIPLE.splice(j,1);                        //   |==> we want this to be flagged as a corner only 
                        j = j-1;
                    }
                }
            }
            if (AE_intersect_MULTIPLE.length == 0)  {        //if none stood the test
                AE_intersect = undefined;  
            }
            else if (AE_intersect_MULTIPLE.length = 1)  { 
                AE_intersect = AE_intersect_MULTIPLE[0];
            }
            else {
                var warning = "failed";
            }
        }
//***// BE INTERSECT //***//
        BE_intersect = perimIntersection(pointBx,pointBy,pointEx,pointEy,XPerimCoords,YPerimCoords);
        var NoOfIntersections = howManyIntersections(pointBx,pointBy,pointEx,pointEy,XPerimCoords,YPerimCoords);
        if ( NoOfIntersections == 0 ) { BE_intersect = undefined; 
        }
        else if (NoOfIntersections == 1) {
            if (typeof(BE_intersect) != "undefined")  { 
                if (   ( (BE_intersect["x"] < pointBx+tolerance)  &&  (BE_intersect["x"] > pointBx-tolerance) &&  (BE_intersect["y"] < pointBy+tolerance)  &&  (BE_intersect["y"] > pointBy-tolerance) )  ||         ( (BE_intersect["x"] < pointEx+tolerance)  &&  (BE_intersect["x"] > pointEx-tolerance) &&  (BE_intersect["y"] < pointEy+tolerance)  &&  (BE_intersect["y"] > pointEy-tolerance) )    ) {
                        //if intersection is defined and is equal to either point E or B, undefine it.       
                    BE_intersect = undefined;                                       //   |==> we want this to be flagged as a corner only 
                }
            }
        }
        else {
            BE_intersect_MULTIPLE= perimMultipleIntersections(pointBx,pointBy,pointEx,pointEy,XPerimCoords,YPerimCoords);
            for (l = 0; l < BE_intersect_MULTIPLE.length; l++)  {
            }
            for ( var j = 0; j < BE_intersect_MULTIPLE.length; j++ )  {
                if (typeof(BE_intersect_MULTIPLE[j]) != "undefined")  { 
                    if (   ( (BE_intersect_MULTIPLE[j]["x"] < pointBx+tolerance)  &&  (BE_intersect_MULTIPLE[j]["x"] > pointBx-tolerance) &&  (BE_intersect_MULTIPLE[j]["y"] < pointBy+tolerance)  &&  (BE_intersect_MULTIPLE[j]["y"] > pointBy-tolerance) )  ||         ( (BE_intersect_MULTIPLE[j]["x"] < pointEx+tolerance)  &&  (BE_intersect_MULTIPLE[j]["x"] > pointEx-tolerance) &&  (BE_intersect_MULTIPLE[j]["y"] < pointEy+tolerance)  &&  (BE_intersect_MULTIPLE[j]["y"] > pointEy-tolerance) )    ) {
                            //if intersection is defined and is equal to either point E or B, undefine it.       
                        BE_intersect_MULTIPLE.splice(j,1);                        //   |==> we want this to be flagged as a corner only 
                        j = j-1;
                    }
                }
            }
            if (BE_intersect_MULTIPLE.length == 0)  {        //if none stood the test
                BE_intersect = undefined;  
            }
            else if (BE_intersect_MULTIPLE.length = 1)  { 
                BE_intersect = BE_intersect_MULTIPLE[0];
            }
            else {
                var warning = "failed";
            }
        }
//***// CE INTERSECT //***//
        CE_intersect = perimIntersection(pointCx,pointCy,pointEx,pointEy,XPerimCoords,YPerimCoords);
        var NoOfIntersections = howManyIntersections(pointCx,pointCy,pointEx,pointEy,XPerimCoords,YPerimCoords);
        if ( NoOfIntersections == 0 ) { CE_intersect = undefined; 
        }
        else if (NoOfIntersections == 1) {
            if (typeof(CE_intersect) != "undefined")  { 
                if (   ( (CE_intersect["x"] < pointCx+tolerance)  &&  (CE_intersect["x"] > pointCx-tolerance) &&  (CE_intersect["y"] < pointCy+tolerance)  &&  (CE_intersect["y"] > pointCy-tolerance) )  ||         ( (CE_intersect["x"] < pointEx+tolerance)  &&  (CE_intersect["x"] > pointEx-tolerance) &&  (CE_intersect["y"] < pointEy+tolerance)  &&  (CE_intersect["y"] > pointEy-tolerance) )    ) {
                        //if intersection is defined and is equal to either point E or C, undefine it.       
                    CE_intersect = undefined;                                       //   |==> we want this to be flagged as a corner only 
                }
            }
        }
        else {
            CE_intersect_MULTIPLE= perimMultipleIntersections(pointCx,pointCy,pointEx,pointEy,XPerimCoords,YPerimCoords);
            for (l = 0; l < CE_intersect_MULTIPLE.length; l++)  {
            }
            for ( var j = 0; j < CE_intersect_MULTIPLE.length; j++ )  {
                if (typeof(CE_intersect_MULTIPLE[j]) != "undefined")  { 
                    if (   ( (CE_intersect_MULTIPLE[j]["x"] < pointCx+tolerance)  &&  (CE_intersect_MULTIPLE[j]["x"] > pointCx-tolerance) &&  (CE_intersect_MULTIPLE[j]["y"] < pointCy+tolerance)  &&  (CE_intersect_MULTIPLE[j]["y"] > pointCy-tolerance) )  ||         ( (CE_intersect_MULTIPLE[j]["x"] < pointEx+tolerance)  &&  (CE_intersect_MULTIPLE[j]["x"] > pointEx-tolerance) &&  (CE_intersect_MULTIPLE[j]["y"] < pointEy+tolerance)  &&  (CE_intersect_MULTIPLE[j]["y"] > pointEy-tolerance) )    ) {
                            //if intersection is defined and is equal to either point E or C, undefine it.       
                        CE_intersect_MULTIPLE.splice(j,1);                        //   |==> we want this to be flagged as a corner only 
                        j = j-1;
                    }
                }
            }
            if (CE_intersect_MULTIPLE.length == 0)  {        //if none stood the test
                CE_intersect = undefined;  
            }
            else if (CE_intersect_MULTIPLE.length = 1)  { 
                CE_intersect = CE_intersect_MULTIPLE[0];
            }
            else {
                var warning = "failed";
            }
        }
        if (CE_intersect != undefined) {
        }
//***// DE INTERSECT //***//
        DE_intersect = perimIntersection(pointDx,pointDy,pointEx,pointEy,XPerimCoords,YPerimCoords);
        var NoOfIntersections = howManyIntersections(pointDx,pointDy,pointEx,pointEy,XPerimCoords,YPerimCoords);
        if ( NoOfIntersections == 0 ) { DE_intersect = undefined; 
        }
        else if (NoOfIntersections == 1) {
            if (typeof(DE_intersect) != "undefined")  { 
                if (   ( (DE_intersect["x"] < pointDx+tolerance)  &&  (DE_intersect["x"] > pointDx-tolerance) &&  (DE_intersect["y"] < pointDy+tolerance)  &&  (DE_intersect["y"] > pointDy-tolerance) )  ||         ( (DE_intersect["x"] < pointEx+tolerance)  &&  (DE_intersect["x"] > pointEx-tolerance) &&  (DE_intersect["y"] < pointEy+tolerance)  &&  (DE_intersect["y"] > pointEy-tolerance) )    ) {
                        //if intersection is defined and is equal to either point E or D, undefine it.       
                    DE_intersect = undefined;                                       //   |==> we want this to be flagged as a corner only 
                }
            }
        }
        else {
            DE_intersect_MULTIPLE= perimMultipleIntersections(pointDx,pointDy,pointEx,pointEy,XPerimCoords,YPerimCoords);
            for (l = 0; l < DE_intersect_MULTIPLE.length; l++)  {
            }
            for ( var j = 0; j < DE_intersect_MULTIPLE.length; j++ )  {
                if (typeof(DE_intersect_MULTIPLE[j]) != "undefined")  { 
                    if (   ( (DE_intersect_MULTIPLE[j]["x"] < pointDx+tolerance)  &&  (DE_intersect_MULTIPLE[j]["x"] > pointDx-tolerance) &&  (DE_intersect_MULTIPLE[j]["y"] < pointDy+tolerance)  &&  (DE_intersect_MULTIPLE[j]["y"] > pointDy-tolerance) )  ||         ( (DE_intersect_MULTIPLE[j]["x"] < pointEx+tolerance)  &&  (DE_intersect_MULTIPLE[j]["x"] > pointEx-tolerance) &&  (DE_intersect_MULTIPLE[j]["y"] < pointEy+tolerance)  &&  (DE_intersect_MULTIPLE[j]["y"] > pointEy-tolerance) )    ) {
                            //if intersection is defined and is equal to either point E or D, undefine it.       
                        DE_intersect_MULTIPLE.splice(j,1);                        //   |==> we want this to be flagged as a corner only 
                        j = j-1;
                    }
                }
            }
            if (DE_intersect_MULTIPLE.length == 0)  {        //if none stood the test
                DE_intersect = undefined;  
            }
            else if (DE_intersect_MULTIPLE.length = 1)  { 
                DE_intersect = DE_intersect_MULTIPLE[0];
            }
            else {
                var warning = "failed";
            }
        }
        
        var AB_flagPerim = 0;
        var BD_flagPerim = 0;
        var DC_flagPerim = 0;
        var CA_flagPerim = 0;
        var AE_flagPerim = 0;
        var BE_flagPerim = 0;
        var CE_flagPerim = 0;
        var DE_flagPerim = 0;
         
        if (AB_intersect != undefined) { AB_flagPerim = 1; }     //flag 'em up if they still exist okay.
        if (BD_intersect != undefined) { BD_flagPerim = 1; }
        if (DC_intersect != undefined) { DC_flagPerim = 1; }
        if (CA_intersect != undefined) { CA_flagPerim = 1; }
        if (AE_intersect != undefined) { AE_flagPerim = 1; }
        if (BE_intersect != undefined) { BE_flagPerim = 1; }
        if (CE_intersect != undefined) { CE_flagPerim = 1; }
        if (DE_intersect != undefined) { DE_flagPerim = 1; }

        var _1_on_PerimFlag = 0;         //we need these flags to later identify which bit of tri is inside perim.
        var _12_flagPerim = 0;   
        var _2_on_PerimFlag = 0;                 //we declare them globally here, but still reset them to zero 
        var _2E_flagPerim = 0;                //inside the functions.
        var _1E_flagPerim = 0; 

        var _12_intersect;
        var _1E_intersect;
        var _2E_intersect;

        var areaInPerim;

      //  ==  CALCULATE AREA OF QUAD IN PERIM  ==  //
        var areaInPerimVersion = new Array(); 
        var _1_on_PerimFlagVersion = new Array();        //we will store values for these for each 
        var _12_flagPerimVersion = new Array();       //of the four tris we define.   
        var _2_on_PerimFlagVersion = new Array();        //We then shan't have to recalculate them each time - these
        var _2E_flagPerimVersion = new Array();       //are the variables that are independent of HIC.
        var _1E_flagPerimVersion = new Array(); 
        var _12_intersectVersion = new Array(); 
        var _1E_intersectVersion = new Array(); 
        var _2E_intersectVersion = new Array(); 

        perimAreas();       //see function.
        
    //==| draw contours at regular HIC values |==//
    
    var contours = 0;
    loopContouring:

    for (var valNo = 0; valNo < vals.length ; valNo++)  {
        val = vals[valNo];
        img.fillColour = colours[valNo]; 
        img.lineColour = colours[valNo];
        lineJoinStyle = Reporter.JoinRound;

        drawContourLines = 0;   //for this particular HIC value, redefined each time.
        var majorHICvalue = 0;
        var contourThickness = 0;
        for ( var majorVal = 0; majorVal < majorContourValues.length; majorVal++ ) {
            if ((val == majorContourValues[majorVal])) {
                if (drawContourLines_Major == 1)  {
                    var majorHICvalue = 1; //we flag it up if this iteration represents a "major" HIC val
                }
            }
        }
        if (majorHICvalue == 0) {   //if just a regular HIC val
            if (drawContourLines_Minor == 1) {  //if flagged to draw minor lines,       
                drawContourLines = 1;           //we make a note of this
                contourThickness = ThicknessContourLines_Minor;     //and set thickness accordingly
            }
        }
        else {  //if a "major" HIC val
            if (drawContourLines_Major == 1) {
                drawContourLines = 1; 
                contourThickness = ThicknessContourLines_Major;     //and set thickness accordingly
            }
            else if (drawContourLines_Minor == 1) {
                drawContourLines = 1; 
                contourThickness = ThicknessContourLines_Minor;     //and set thickness accordingly
            }
        }
        
    
        img.lineStyle = Reporter.LineSolid;
        
        var AB_interpolationX = undefined;      
        var AB_interpolationY = undefined;      // Easier to test this way.
        var BD_interpolationX = undefined;
        var BD_interpolationY = undefined;
        var DC_interpolationX = undefined;
        var DC_interpolationY = undefined;      // if time, i should change all of these to use the interpolationPoint() function and 
        var CA_interpolationX = undefined;      // be coordinate arrays like diagonal interpolation points are.
        var CA_interpolationY = undefined;
        var AB_flag = 0;        //reset to zero each time round.
        var BD_flag = 0;
        var DC_flag = 0;
        var CA_flag = 0;

        
        //== verify on which sides the contour line cuts         //if on a corner, we flag NEITHER line...
        if( (typeof(HIC_A) != "undefined") && (typeof(HIC_B) != "undefined") && (((HIC_A > val)&&(HIC_B < val)) || (( HIC_A < val)&&(HIC_B > val))) || (HIC_A == val) ) {  
            AB_flag = 1;               // if the contour cuts through AB or exactly on A
            var fractionAB = ((Math.abs(HIC_A - val)) / (Math.abs(HIC_B - HIC_A)));     
            var AB_interpolationX = (XcoordNEW[forA] + (fractionAB * (XcoordNEW[forB] - XcoordNEW[forA])));  //finds point where HIC = val
            var AB_interpolationY = YcoordNEW[forA];   //or YcoordNEW[forB], they're gonna be the same.
        }
        
        if( (typeof(HIC_B) != "undefined") && (typeof(HIC_D) != "undefined") &&  (((HIC_B > val)&&(HIC_D < val)) || (( HIC_B < val)&&(HIC_D > val))) || (HIC_B == val) ) {   
            BD_flag = 1;              // if the contour cuts through BC or exactly on B
            var fractionBD = ((Math.abs(HIC_B - val)) / (Math.abs(HIC_D - HIC_B))); 
            var BD_interpolationX = XcoordNEW[forB];
            var BD_interpolationY = (YcoordNEW[forB] + (fractionBD * (YcoordNEW[forD] - YcoordNEW[forB])));
   
        }
    
        if( (typeof(HIC_D) != "undefined") && (typeof(HIC_C) != "undefined") && (((HIC_D > val)&&(HIC_C < val)) || (( HIC_D < val)&&(HIC_C > val))) || (HIC_D == val)) {
            DC_flag = 1;                // if the contour cuts through DC or exactly on D
            var fractionDC = ((Math.abs(HIC_D - val)) / (Math.abs(HIC_C - HIC_D))); 
            var DC_interpolationX = (XcoordNEW[forD] + (fractionDC * (XcoordNEW[forC] - XcoordNEW[forD])));
            var DC_interpolationY = YcoordNEW[forD];
        }
        
        if(  (typeof(HIC_C) != "undefined") && (typeof(HIC_A) != "undefined") && (((HIC_C > val)&&(HIC_A < val)) || (( HIC_C < val)&&(HIC_A > val))) || (HIC_C == val)) {
            CA_flag = 1;                // if the contour cuts through CA or exactly on C
            var fractionCA = ((Math.abs(HIC_C - val)) / (Math.abs(HIC_A - HIC_C))); 
            var CA_interpolationX = XcoordNEW[forC];
            var CA_interpolationY = (YcoordNEW[forC] + (fractionCA * (YcoordNEW[forA] - YcoordNEW[forC])));
        }
    
        var areaRECTANGLE = Math.abs(XcoordNEW[forB]-XcoordNEW[forA]) * Math.abs(YcoordNEW[forB]-YcoordNEW[forD]);      //whole area, to use later
        
        var flagContourScenario = undefined;
        var totalAreaInPerim = areaInPerimVersion["A"] + areaInPerimVersion["B"] + areaInPerimVersion["C"] + areaInPerimVersion["D"]
        
        var HIC_E = ( HIC_A + HIC_B + HIC_C + HIC_D )/4 ;     //average HICs for centre (point E has already been defiend way back
//=======| IF NO CONTOUR AT ALL |=========//
        if ((AB_flag == 0)&&(BD_flag == 0)&&(DC_flag == 0)&&(CA_flag == 0))  {  
            if (HIC_E < val) {  //either all is < val   
                areas[val].push(totalAreaInPerim);   //we already have calculated the area of rectangle inside perim.
            }
            else {      //else all is >val, so we colour it in
                img.lineColour = colours[valNo];
                img.lineWidth = 1; 
                if (colourInContours == 1) {
                    img.Polygon(pointAx,pointAy,pointBx,pointBy,pointDx,pointDy,pointCx,pointCy); //and colour it in 
                }
            }
        }
        else {

//=======| ELSE... |==========//
    //use saddle contouring stuff for EVERY square. Much easier this way...    
  
            var HIC_E = ( HIC_A + HIC_B + HIC_C + HIC_D )/4 ;     //average HICs for centre (point E has already been defiend way back

            var AE_flag = 0;    //declare more variables and rest at zero on each iteration. 
            var EA_flag = 0; 
            var BE_flag = 0;    
            var EB_flag = 0;    
            var DE_flag = 0;   
            var ED_flag = 0;   
            var CE_flag = 0;  
            var EC_flag = 0;  

            AE_interpolation = undefined;      
            EA_interpolation = undefined;      
            BE_interpolation = undefined;
            EB_interpolation = undefined;
            CE_interpolation = undefined;
            EC_interpolation = undefined;
            DE_interpolation = undefined;
            ED_interpolation = undefined;
        //subdivide quad into 4 tris fanning from centre - verify where contour line cuts each diagonal line
                //we already have defined AB,BD,DC etc, just need to sort inner lines
                        // We must define for example AE AND EA, so we can cope with when corner of tri has exact HIC
            if ( (((HIC_A > val)&&(HIC_E < val)) || (( HIC_A < val)&&(HIC_E > val)) || HIC_A == val ) ) {         // if the contour cuts through AE 
                AE_flag = 1;
                var AE_interpolation = interpolationPoint(pointAx,pointAy,HIC_A,pointEx,pointEy,HIC_E,val); 
            }
            if ( (((HIC_A > val)&&(HIC_E < val)) || (( HIC_A < val)&&(HIC_E > val)) || HIC_E == val) ) {         // if the contour cuts through AE 
                EA_flag = 1;
                var EA_interpolation = interpolationPoint(pointAx,pointAy,HIC_A,pointEx,pointEy,HIC_E,val); 
            }
            if ( (((HIC_B > val)&&(HIC_E < val)) || (( HIC_B < val)&&(HIC_E > val)) || HIC_B == val) ) {         // if the contour cuts through BE 
                BE_flag = 1;
                var BE_interpolation = interpolationPoint(pointBx,pointBy,HIC_B,pointEx,pointEy,HIC_E,val); 
            }
            if ( (((HIC_B > val)&&(HIC_E < val)) || (( HIC_B < val)&&(HIC_E > val)) || HIC_E == val) ) {         // if the contour cuts through BE 
                EB_flag = 1;
                var EB_interpolation = interpolationPoint(pointBx,pointBy,HIC_B,pointEx,pointEy,HIC_E,val); 
            }
            if ( (((HIC_C > val)&&(HIC_E < val)) || (( HIC_C < val)&&(HIC_E > val)) || HIC_C == val)  ) {          // if the contour cuts through CE  
                CE_flag = 1;
                var CE_interpolation = interpolationPoint(pointCx,pointCy,HIC_C,pointEx,pointEy,HIC_E,val); 
            }
            if ( (((HIC_C > val)&&(HIC_E < val)) || (( HIC_C < val)&&(HIC_E > val)) || HIC_E == val)  ) {          // if the contour cuts through CE  
                EC_flag = 1;
                var EC_interpolation = interpolationPoint(pointCx,pointCy,HIC_C,pointEx,pointEy,HIC_E,val); 
            }
            if ( (((HIC_D > val)&&(HIC_E < val)) || (( HIC_D < val)&&(HIC_E > val)) || HIC_D == val) ) {         // if the contour cuts through DE  
                DE_flag = 1;
                var DE_interpolation = interpolationPoint(pointDx,pointDy,HIC_D,pointEx,pointEy,HIC_E,val); 
            }
            if ( (((HIC_D > val)&&(HIC_E < val)) || (( HIC_D < val)&&(HIC_E > val)) || HIC_E == val) ) {         // if the contour cuts through DE  
                ED_flag = 1;
                var ED_interpolation = interpolationPoint(pointDx,pointDy,HIC_D,pointEx,pointEy,HIC_E,val); 
            }

           //triangle ABE:
           if (EA_flag == 1 && AB_flag == 1) {  //find out where contour line cuts
                if (HIC_B < val)  {
                    img.lineColour = colours[valNo]; 
                    img.lineWidth = 1; 
                    if (colourInContours == 1) {
                        img.Polygon(pointAx,pointAy,AB_interpolationX,AB_interpolationY,EA_interpolation["x"],EA_interpolation["y"]);//draw the triangle
                    }
                    flagSaddleScenario = "AB_AEtri";
                    var areaContourInTri = ( areaTriangle(AB_interpolationX,AB_interpolationY,EA_interpolation["x"],EA_interpolation["y"],pointEx,pointEy) +
                                            areaTriangle(pointBx,pointBy,AB_interpolationX,AB_interpolationY,pointEx,pointEy) );
                                //calculating area of quad by dividing into two tris.
               }
               else {
                    img.lineColour = colours[valNo]; 
                    img.lineWidth = 1; 
                    if (colourInContours == 1) {
                        img.Polygon(pointBx,pointBy,AB_interpolationX,AB_interpolationY,EA_interpolation["x"],EA_interpolation["y"],pointEx,pointEy);
                    }
                   //draw the quadrilateral
                    flagSaddleScenario = "AB_AEquad";
                   var areaContourInTri = areaTriangle(pointAx,pointAy,AB_interpolationX,AB_interpolationY,EA_interpolation["x"],EA_interpolation["y"]);
               }
               if (drawContourLines == 1)  {
                   img.lineColour = "black";
                   img.lineWidth = contourThickness; 
                   img.Line(AB_interpolationX,AB_interpolationY,EA_interpolation["x"],EA_interpolation["y"]);
               }
               var pointI = { x:AB_interpolationX, y:AB_interpolationY };        //define I and J for area calcs (see triangleWeb function)
               var pointJ = { x:EA_interpolation["x"], y:EA_interpolation["y"] };
           }
           else if (BE_flag == 1 && AB_flag == 1) {
               if (HIC_E < val)  {
                    img.lineColour = colours[valNo]; 
                    img.lineWidth = 1; 
                    if (colourInContours == 1) {
                        img.Polygon(pointBx,pointBy,AB_interpolationX,AB_interpolationY,BE_interpolation["x"],BE_interpolation["y"]);//draw the triangle
                    }
                    flagSaddleScenario = "AB_BEtri";
                    var areaContourInTri = ( areaTriangle(AB_interpolationX,AB_interpolationY,BE_interpolation["x"],BE_interpolation["y"],pointEx,pointEy) +
                                            areaTriangle(pointAx,pointAy,AB_interpolationX,AB_interpolationY,pointEx,pointEy) );
               }
               else {
                    img.lineColour = colours[valNo]; 
                    img.lineWidth = 1; 
                    if (colourInContours == 1) {
                        img.Polygon(pointAx,pointAy,AB_interpolationX,AB_interpolationY,BE_interpolation["x"],BE_interpolation["y"],pointEx,pointEy);
                    }
                   //draw the quadrilateral
                   flagSaddleScenario = "AB_BEquad";
                   var areaContourInTri = areaTriangle(pointBx,pointBy,AB_interpolationX,AB_interpolationY,BE_interpolation["x"],BE_interpolation["y"]);
               }
               if (drawContourLines == 1)  {
                    img.lineColour = "black";
                    img.lineWidth = contourThickness; 
                    img.Line(AB_interpolationX,AB_interpolationY,BE_interpolation["x"],BE_interpolation["y"]);
               }
               var pointI = { x:AB_interpolationX, y:AB_interpolationY };        //define I and J for area calcs (see triangleWeb function)
               var pointJ = { x:BE_interpolation["x"], y:BE_interpolation["y"] };
           }
           else if (BE_flag == 1 && EA_flag == 1) {
               if (HIC_A < val)  {
                   img.lineColour = colours[valNo]; 
                   img.lineWidth = 1; 
                    if (colourInContours == 1) {
                        img.Polygon(pointEx,pointEy,BE_interpolation["x"],BE_interpolation["y"],EA_interpolation["x"],EA_interpolation["y"]);//draw the triangle
                    }
                   flagSaddleScenario = "AE_BEtri";
                   var areaContourInTri = ( areaTriangle(BE_interpolation["x"],BE_interpolation["y"],EA_interpolation["x"],EA_interpolation["y"],pointAx,pointAy) +
                                            areaTriangle(pointAx,pointAy,BE_interpolation["x"],BE_interpolation["y"],pointBx,pointBy) );
               }
               else {
                    img.lineColour = colours[valNo]; 
                    img.lineWidth = 1; 
                    if (colourInContours == 1) {
                        img.Polygon(pointAx,pointAy,EA_interpolation["x"],EA_interpolation["y"],BE_interpolation["x"],BE_interpolation["y"],pointBx,pointBy);
                    }
                   //draw the quadrilateral
                   flagSaddleScenario = "AB_BEquad";
                   var areaContourInTri = areaTriangle(pointEx,pointEy,EA_interpolation["x"],EA_interpolation["y"],BE_interpolation["x"],BE_interpolation["y"]);
               }
               if (drawContourLines == 1)  {
                    img.lineColour = "black";
                    img.lineWidth = contourThickness; 
                    img.Line(EA_interpolation["x"],EA_interpolation["y"],BE_interpolation["x"],BE_interpolation["y"]);
               }
               var pointI = { x:EA_interpolation["x"], y:EA_interpolation["y"] };        //define I and J for area calcs (see triangleWeb function)
               var pointJ = { x:BE_interpolation["x"], y:BE_interpolation["y"] };
           }
           else {       //if none of the above were true, either all  of tri is < val or all is > val (but still not necessarily in perim!)
               var myCentroid = undefined; 
               var myCentroid = centroidify(pointAx,pointAy,pointBx,pointBy,pointEx,pointEy);
               var HIC_centroid = triangleInterpolate(myCentroid["x"],myCentroid["y"],pointAx,pointAy,HIC_A,pointBx,pointBy,HIC_B,pointEx,pointEy,HIC_E);
               if (HIC_centroid <= val)  {
                    flagSaddleScenario = "all_countableHIC";
                    var areaContourInTri = areaTriangle(pointAx,pointAy,pointBx,pointBy,pointEx,pointEy);
                    pointI = { x:pointAx, y:pointAy };  //define I and J arbitrarily; pick a line that doesnt cut tri at all
                    pointJ = { x:pointEx, y:pointEy };
               }
               else {
                   //colour all in..
                    img.lineColour = colours[valNo]; 
                    img.lineWidth = 1; 
                    if (colourInContours == 1) {
                        img.Polygon(pointAx,pointAy,pointBx,pointBy,pointEx,pointEy);
                    }
                    flagSaddleScenario = "all_non-countableHIC";
                    var areaContourInTri = 0;
                    pointI = { x:pointAx, y:pointAy };  //define I and J arbitrarily; pick a line that doesnt cut tri at all
                    pointJ = { x:pointEx, y:pointEy };
               }
           }
           triangleWeb(pointAx,pointAy,pointBx,pointBy,pointEx,pointEy,"A");

    
           //triangle BDE:
           if (EB_flag == 1 && BD_flag == 1) {
               if (HIC_D < val)  {
                    img.lineColour = colours[valNo]; 
                    img.lineWidth = 1; 
                    if (colourInContours == 1) {
                        img.Polygon(pointBx,pointBy,BD_interpolationX,BD_interpolationY,EB_interpolation["x"],EB_interpolation["y"]);//draw the triangle
                    }
                   flagSaddleScenario = "BD_BEtri";
                   var areaContourInTri = ( areaTriangle(BD_interpolationX,BD_interpolationY,EB_interpolation["x"],EB_interpolation["y"],pointEx,pointEy) +
                                            areaTriangle(pointDx,pointDy,BD_interpolationX,BD_interpolationY,pointEx,pointEy) );
               }
               else {
                    img.lineColour = colours[valNo]; 
                    img.lineWidth = 1; 
                    if (colourInContours == 1) {
                        img.Polygon(pointDx,pointDy,BD_interpolationX,BD_interpolationY,EB_interpolation["x"],EB_interpolation["y"],pointEx,pointEy);
                    }
                   //draw the quadrilateral
                   flagSaddleScenario = "BD_BEquad";
                   var areaContourInTri = areaTriangle(pointBx,pointBy,BD_interpolationX,BD_interpolationY,EB_interpolation["x"],EB_interpolation["y"]);
               }
               if (drawContourLines == 1)  {
                    img.lineColour = "black";
                    img.lineWidth = contourThickness; 
                    img.Line(BD_interpolationX,BD_interpolationY,EB_interpolation["x"],EB_interpolation["y"]);
               }
               var pointI = { x:BD_interpolationX, y:BD_interpolationY };        //define I and J for area calcs (see triangleWeb function)
               var pointJ = { x:EB_interpolation["x"], y:EB_interpolation["y"] };
           }        
           else if (DE_flag == 1 && BD_flag == 1) {
               if (HIC_E < val)  {
                    img.lineColour = colours[valNo]; 
                    img.lineWidth = 1; 
                    if (colourInContours == 1) {
                        img.Polygon(pointDx,pointDy,BD_interpolationX,BD_interpolationY,DE_interpolation["x"],DE_interpolation["y"]);//draw the triangle
                    }
                   flagSaddleScenario = "BD_DEtri";
                   var areaContourInTri = ( areaTriangle(BD_interpolationX,BD_interpolationY,DE_interpolation["x"],DE_interpolation["y"],pointEx,pointEy) +
                                            areaTriangle(pointBx,pointBy,BD_interpolationX,BD_interpolationY,pointEx,pointEy) );
               }
               else {
                    img.lineColour = colours[valNo]; 
                    img.lineWidth = 1; 
                    if (colourInContours == 1) {
                        img.Polygon(pointBx,pointBy,BD_interpolationX,BD_interpolationY,DE_interpolation["x"],DE_interpolation["y"],pointEx,pointEy);
                    }
                   flagSaddleScenario = "BD_DEquad";
                   var areaContourInTri = areaTriangle(pointDx,pointDy,BD_interpolationX,BD_interpolationY,DE_interpolation["x"],DE_interpolation["y"]);
               }
               if (drawContourLines == 1)  {
                    img.lineColour = "black";
                    img.lineWidth = contourThickness; 
                    img.Line(BD_interpolationX,BD_interpolationY,DE_interpolation["x"],DE_interpolation["y"]);
               }
               var pointI = { x:BD_interpolationX, y:BD_interpolationY };        //define I and J for area calcs (see triangleWeb function)
               var pointJ = { x:DE_interpolation["x"], y:DE_interpolation["y"] };
           }
           else if (DE_flag == 1 && EB_flag == 1) {
               if (HIC_B < val)  {
                    img.lineColour = colours[valNo]; 
                    img.lineWidth = 1; 
                    if (colourInContours == 1) {
                        img.Polygon(pointEx,pointEy,DE_interpolation["x"],DE_interpolation["y"],EB_interpolation["x"],EB_interpolation["y"]);//draw the triangle
                    }
                   flagSaddleScenario = "BE_DEtri";
                   var areaContourInTri = ( areaTriangle(DE_interpolation["x"],DE_interpolation["y"],EB_interpolation["x"],EB_interpolation["y"],pointDx,pointDy) +
                                            areaTriangle(pointDx,pointDy,EB_interpolation["x"],EB_interpolation["y"],pointBx,pointBy) );
               }
               else {
                    img.lineColour = colours[valNo]; 
                    img.lineWidth = 1; 
                    if (colourInContours == 1) {
                        img.Polygon(pointBx,pointBy,EB_interpolation["x"],EB_interpolation["y"],DE_interpolation["x"],DE_interpolation["y"],pointDx,pointDy);
                    }
                   //draw the quadrilateral
                   flagSaddleScenario = "BE_DEquad";
                   var areaContourInTri = areaTriangle(pointEx,pointEy,EB_interpolation["x"],EB_interpolation["y"],DE_interpolation["x"],DE_interpolation["y"]);
               }
               if (drawContourLines == 1)  {
                    img.lineColour = "black";
                    img.lineWidth = contourThickness; 
                    img.Line(EB_interpolation["x"],EB_interpolation["y"],DE_interpolation["x"],DE_interpolation["y"]);
               }
               var pointI = { x:EB_interpolation["x"], y:EB_interpolation["y"] };        //define I and J for area calcs (see triangleWeb function)
               var pointJ = { x:DE_interpolation["x"], y:DE_interpolation["y"] };
           }
           else {       //if none of the above were true, either all  of tri is < val or all is > val (but still not necessarily in perim!)
               var myCentroid = undefined; 
               var myCentroid = centroidify(pointDx,pointDy,pointBx,pointBy,pointEx,pointEy);
               var HIC_centroid = triangleInterpolate(myCentroid["x"],myCentroid["y"],pointDx,pointDy,HIC_D,pointBx,pointBy,HIC_B,pointEx,pointEy,HIC_E);
               if (HIC_centroid < val)  {
                    flagSaddleScenario = "all_countableHIC";
                    var areaContourInTri = areaTriangle(pointDx,pointDy,pointBx,pointBy,pointEx,pointEy);
                    pointI = { x:pointBx, y:pointBy };  //define I and J arbitrarily; pick a line that doesnt cut tri at all
                    pointJ = { x:pointEx, y:pointEy };
               }
               else {
                   //colour all in..
                    img.lineColour = colours[valNo]; 
                    img.lineWidth = 1; 
                    if (colourInContours == 1) {
                        img.Polygon(pointBx,pointBy,pointDx,pointDy,pointEx,pointEy);
                    }
                    
                    flagSaddleScenario = "all_non-countableHIC";
                    var areaContourInTri = 0;
                    pointI = { x:pointBx, y:pointBy };  //define I and J arbitrarily; pick a line that doesnt cut tri at all
                    pointJ = { x:pointEx, y:pointEy };
               }
           }
           triangleWeb(pointBx,pointBy,pointDx,pointDy,pointEx,pointEy,"B");

    
           //triangle DCE:
           if (ED_flag == 1 && DC_flag == 1) {
               if (HIC_C < val)  {
                    img.lineColour = colours[valNo]; 
                    img.lineWidth = 1; 
                    if (colourInContours == 1) {
                        img.Polygon(pointDx,pointDy,DC_interpolationX,DC_interpolationY,ED_interpolation["x"],ED_interpolation["y"]);//draw the triangle
                    }
                   flagSaddleScenario = "DC_DEtri";
                   var areaContourInTri = ( areaTriangle(DC_interpolationX,DC_interpolationY,ED_interpolation["x"],ED_interpolation["y"],pointEx,pointEy) +
                                            areaTriangle(pointCx,pointCy,DC_interpolationX,DC_interpolationY,pointEx,pointEy) );
               }
               else {
                    img.lineColour = colours[valNo]; 
                    img.lineWidth = 1; 
                    if (colourInContours == 1) {
                        img.Polygon(pointCx,pointCy,DC_interpolationX,DC_interpolationY,ED_interpolation["x"],ED_interpolation["y"],pointEx,pointEy);
                    }
                   flagSaddleScenario = "DC_DEquad";
                   var areaContourInTri = areaTriangle(pointDx,pointDy,DC_interpolationX,DC_interpolationY,ED_interpolation["x"],ED_interpolation["y"]);
               }
               if (drawContourLines == 1)  {
                    img.lineColour = "black";
                    img.lineWidth = contourThickness; 
                    img.Line(DC_interpolationX,DC_interpolationY,ED_interpolation["x"],ED_interpolation["y"]);
               }
                var pointI = { x:DC_interpolationX, y:DC_interpolationY };        //define I and J for area calcs (see triangleWeb function)
                var pointJ = { x:ED_interpolation["x"], y:ED_interpolation["y"] };
           }
           else if (CE_flag == 1 && DC_flag == 1) {
               if (HIC_E < val)  {
                    img.lineColour = colours[valNo]; 
                    img.lineWidth = 1; 
                    if (colourInContours == 1) {
                        img.Polygon(pointCx,pointCy,DC_interpolationX,DC_interpolationY,CE_interpolation["x"],CE_interpolation["y"]);//draw the triangle
                    }
                   flagSaddleScenario = "DC_CEtri";
                   var areaContourInTri = ( areaTriangle(DC_interpolationX,DC_interpolationY,CE_interpolation["x"],CE_interpolation["y"],pointEx,pointEy) +
                                            areaTriangle(pointDx,pointDy,DC_interpolationX,DC_interpolationY,pointEx,pointEy) );
               }
               else {
                    img.lineColour = colours[valNo]; 
                    img.lineWidth = 1; 
                    if (colourInContours == 1) {
                        img.Polygon(pointDx,pointDy,DC_interpolationX,DC_interpolationY,CE_interpolation["x"],CE_interpolation["y"],pointEx,pointEy);
                    }
                   flagSaddleScenario = "DC_CEquad";
                   var areaContourInTri = areaTriangle(pointCx,pointCy,DC_interpolationX,DC_interpolationY,CE_interpolation["x"],CE_interpolation["y"]);
               }
               if (drawContourLines == 1)  {
                    img.lineColour = "black";
                    img.lineWidth = contourThickness; 
                    img.Line(DC_interpolationX,DC_interpolationY,CE_interpolation["x"],CE_interpolation["y"]);
               }
               var pointI = { x:DC_interpolationX, y:DC_interpolationY };        //define I and J for area calcs (see triangleWeb function)
               var pointJ = { x:CE_interpolation["x"], y:CE_interpolation["y"] };
           }
           else if (CE_flag == 1 && ED_flag == 1) {
               if (HIC_D < val)  {
                    img.lineColour = colours[valNo]; 
                    img.lineWidth = 1; 
                    if (colourInContours == 1) {
                        img.Polygon(pointEx,pointEy,CE_interpolation["x"],CE_interpolation["y"],ED_interpolation["x"],ED_interpolation["y"]);//draw the triangle
                    }
                   flagSaddleScenario = "DE_CEtri";
                   var areaContourInTri = ( areaTriangle(CE_interpolation["x"],CE_interpolation["y"],ED_interpolation["x"],ED_interpolation["y"],pointDx,pointDy) +
                                            areaTriangle(pointDx,pointDy,ED_interpolation["x"],ED_interpolation["y"],pointCx,pointCy) );
               }
               else {
                    img.lineColour = colours[valNo]; 
                    img.lineWidth = 1; 
                    if (colourInContours == 1) {
                        img.Polygon(pointDx,pointDy,ED_interpolation["x"],ED_interpolation["y"],CE_interpolation["x"],CE_interpolation["y"],pointCx,pointCy);
                    }
                   //draw the quadrilateral
                   flagSaddleScenario = "DE_CEquad";
                   var areaContourInTri = areaTriangle(pointEx,pointEy,ED_interpolation["x"],ED_interpolation["y"],CE_interpolation["x"],CE_interpolation["y"]);
               }
               if (drawContourLines == 1)  {
                    img.lineColour = "black";
                    img.lineWidth = contourThickness; 
                    img.Line(ED_interpolation["x"],ED_interpolation["y"],CE_interpolation["x"],CE_interpolation["y"]);
               }
               var pointI = { x:ED_interpolation["x"], y:ED_interpolation["y"] };        //define I and J for area calcs (see triangleWeb function)
               var pointJ = { x:CE_interpolation["x"], y:CE_interpolation["y"] };
           }
           else {       //if none of the above were true, either all  of tri is < val or all is > val (but still not necessarily in perim!)
               var myCentroid = undefined; 
               var myCentroid = centroidify(pointCx,pointCy,pointDx,pointDy,pointEx,pointEy);
               var HIC_centroid = triangleInterpolate(myCentroid["x"],myCentroid["y"],pointCx,pointCy,HIC_C,pointDx,pointDy,HIC_D,pointEx,pointEy,HIC_E);
               if (HIC_centroid < val)  {
                    flagSaddleScenario = "all_countableHIC";
                    var areaContourInTri = areaTriangle(pointCx,pointCy,pointDx,pointDy,pointEx,pointEy);
                    pointI = { x:pointCx, y:pointCy };  //define I and J arbitrarily; pick a line that doesnt cut tri at all
                    pointJ = { x:pointEx, y:pointEy };
               }
               else {
                   //colour all in..
                    img.lineColour = colours[valNo]; 
                    img.lineWidth = 1; 
                    if (colourInContours == 1) {
                        img.Polygon(pointDx,pointDy,pointCx,pointCy,pointEx,pointEy);
                    }
                    
                    flagSaddleScenario = "all_non-countableHIC";
                    var areaContourInTri = 0;
                    pointI = { x:pointCx, y:pointCy };  //define I and J arbitrarily; pick a line that doesnt cut tri at all
                    pointJ = { x:pointEx, y:pointEy };
               }
           }
           triangleWeb(pointDx,pointDy,pointCx,pointCy,pointEx,pointEy,"D");
           //triangle CAE:

           if (EC_flag == 1 && CA_flag == 1) {
               if (HIC_A < val)  {
                    img.lineColour = colours[valNo]; 
                    img.lineWidth = 1; 
                    if (colourInContours == 1) {
                        img.Polygon(pointCx,pointCy,CA_interpolationX,CA_interpolationY,EC_interpolation["x"],EC_interpolation["y"]);//draw the triangle
                    } 
                   flagSaddleScenario = "CA_CEtri";
                   var areaContourInTri = ( areaTriangle(CA_interpolationX,CA_interpolationY,EC_interpolation["x"],EC_interpolation["y"],pointEx,pointEy) +
                                            areaTriangle(pointAx,pointAy,CA_interpolationX,CA_interpolationY,pointEx,pointEy) );
               }
               else {
                    img.lineColour = colours[valNo]; 
                    img.lineWidth = 1; 
                    if (colourInContours == 1) {
                        img.Polygon(pointAx,pointAy,CA_interpolationX,CA_interpolationY,EC_interpolation["x"],EC_interpolation["y"],pointEx,pointEy);
                    }
                   flagSaddleScenario = "CA_CEquad";
                   var areaContourInTri = areaTriangle(pointCx,pointCy,CA_interpolationX,CA_interpolationY,EC_interpolation["x"],EC_interpolation["y"]);
               }
               if (drawContourLines == 1)  {
                    img.lineColour = "black";
                    img.lineWidth = contourThickness; 
                    img.Line(CA_interpolationX,CA_interpolationY,EC_interpolation["x"],EC_interpolation["y"]);
               }
               var pointI = { x:CA_interpolationX, y:CA_interpolationY };        //define I and J for area calcs (see triangleWeb function)
               var pointJ = { x:EC_interpolation["x"], y:EC_interpolation["y"] };
           }
           else if (AE_flag == 1 && CA_flag == 1) {
               if (HIC_E < val)  {
                    img.lineColour = colours[valNo]; 
                    img.lineWidth = 1; 
                    if (colourInContours == 1) {
                        img.Polygon(pointAx,pointAy,CA_interpolationX,CA_interpolationY,AE_interpolation["x"],AE_interpolation["y"]);//draw the triangle
                    }
                   flagSaddleScenario = "CA_AEtri";
                   var areaContourInTri = ( areaTriangle(CA_interpolationX,CA_interpolationY,AE_interpolation["x"],AE_interpolation["y"],pointEx,pointEy) +
                                            areaTriangle(pointCx,pointCy,CA_interpolationX,CA_interpolationY,pointEx,pointEy) );
               }
               else {
                    img.lineColour = colours[valNo]; 
                    img.lineWidth = 1; 
                    if (colourInContours == 1) {
                        img.Polygon(pointCx,pointCy,CA_interpolationX,CA_interpolationY,AE_interpolation["x"],AE_interpolation["y"],pointEx,pointEy);
                    }
                   flagSaddleScenario = "CA_AEquad";
                   var areaContourInTri = areaTriangle(pointAx,pointAy,CA_interpolationX,CA_interpolationY,AE_interpolation["x"],AE_interpolation["y"]);
               }
               if (drawContourLines == 1)  {
                    img.lineColour = "black";
                    img.lineWidth = contourThickness; 
                    img.Line(CA_interpolationX,CA_interpolationY,AE_interpolation["x"],AE_interpolation["y"]);
               }
               var pointI = { x:CA_interpolationX, y:CA_interpolationY };        //define I and J for area calcs (see triangleWeb function)
               var pointJ = { x:AE_interpolation["x"], y:AE_interpolation["y"] };
           }
           else if (AE_flag == 1 && EC_flag == 1) {
               if (HIC_C < val)  {
                   img.lineColour = colours[valNo]; 
                   img.lineWidth = 1; 
                    if (colourInContours == 1) {
                        img.Polygon(pointEx,pointEy,AE_interpolation["x"],AE_interpolation["y"],EC_interpolation["x"],EC_interpolation["y"]);//draw the triangle
                    }
                   flagSaddleScenario = "CE_AEtri";
                   var areaContourInTri = ( areaTriangle(AE_interpolation["x"],AE_interpolation["y"],EC_interpolation["x"],EC_interpolation["y"],pointAx,pointAy) +
                                            areaTriangle(pointCx,pointCy,EC_interpolation["x"],EC_interpolation["y"],pointAx,pointAy) );
               }
               else {
                    img.lineColour = colours[valNo]; 
                    img.lineWidth = 1; 
                    if (colourInContours == 1) {
                        img.Polygon(pointAx,pointAy,AE_interpolation["x"],AE_interpolation["y"],EC_interpolation["x"],EC_interpolation["y"],pointCx,pointCy);
                    }
                   //draw the quadrilateral
                   flagSaddleScenario = "CE_AEquad";
                   var areaContourInTri = areaTriangle(pointEx,pointEy,EC_interpolation["x"],EC_interpolation["y"],AE_interpolation["x"],AE_interpolation["y"]);
               }
               if (drawContourLines == 1)  {
                    img.lineColour = "black";
                    img.lineWidth = contourThickness; 
                    img.Line(EC_interpolation["x"],EC_interpolation["y"],AE_interpolation["x"],AE_interpolation["y"]);
               }
               var pointI = { x:EC_interpolation["x"], y:EC_interpolation["y"] };        //define I and J for area calcs (see triangleWeb function)
               var pointJ = { x:AE_interpolation["x"], y:AE_interpolation["y"] };
           }
           else {       //if none of the above were true, either all  of tri is < val or all is > val (but still not necessarily in perim!)
               var myCentroid = undefined; 
               var myCentroid = centroidify(pointAx,pointAy,pointCx,pointCy,pointEx,pointEy);
               var HIC_centroid = triangleInterpolate(myCentroid["x"],myCentroid["y"],pointAx,pointAy,HIC_A,pointCx,pointCy,HIC_C,pointEx,pointEy,HIC_E);
               if (HIC_centroid < val)  {
                    flagSaddleScenario = "all_countableHIC";
                    var areaContourInTri = areaTriangle(pointAx,pointAy,pointCx,pointCy,pointEx,pointEy);
                    pointI = { x:pointDx, y:pointDy };  //define I and J arbitrarily; pick a line that doesnt cut tri at all
                    pointJ = { x:pointEx, y:pointEy };
               }
               else {
                   //colour all in..
                    img.lineColour = colours[valNo]; 
                    img.lineWidth = 1; 
                    if (colourInContours == 1) {
                        img.Polygon(pointCx,pointCy,pointAx,pointAy,pointEx,pointEy);
                    }
                    
                    flagSaddleScenario = "all_non-countableHIC";
                    var areaContourInTri = 0;
                    pointI = { x:pointDx, y:pointDy };  //define I and J arbitrarily; pick a line that doesnt cut tri at all
                    pointJ = { x:pointEx, y:pointEy };
               }
           }
           triangleWeb(pointCx,pointCy,pointAx,pointAy,pointEx,pointEy,"C");
        }
    }
}


var lower_limit_percent = 0.0;
var upper_limit_percent = 0.0;
 
for (var valNo = 0; valNo < vals.length ; valNo++)  {
    val = vals[valNo];
    var percentage = (sumOfArray(areas[val])/sumOfArray(areaTOTAL)*100)
    var absoluteArea = sumOfArray(areas[val])/squared(scaleFactor);
    var pvar = new Variable(Template.GetCurrent(), ZONE_NAME + "_ABSOLUTE_AREA_" + val, "Absolute area below HIC = " + val, absoluteArea);

    var absoluteArea = absoluteArea.toPrecision(4); //4 significant figures.
    if (outputPercentageAreas == 1 && outputAbsoluteAreas==1) {
        LogPrint ("Percentage below HIC = ",val,":\t",Math.round(percentage*10)/10,"%\t(absolute area = ",absoluteArea,")" );    // print to log
    }
    else {
        if (outputPercentageAreas == 1) {
            LogPrint ("Percentage below HIC = ",val,":\t",Math.round(percentage*10)/10,"%" );    // print to log
        }
        else 
            if (outputAbsoluteAreas == 1) {
                LogPrint ("Absolute area below HIC = ",val,":  ",absoluteArea);    // print to log
        }
    }

    var pvar = new Variable(Template.GetCurrent(), ZONE_NAME + "_PERCENT_BELOW_" + val, "Percent below HIC = " + val, Math.round(percentage*10)/10);

    if(val == LOWER_LIMIT) lower_limit_percent = percentage;
    if(val == UPPER_LIMIT) upper_limit_percent = percentage;
            
//    f.Write ("Percentage below HIC = ",val,":  ",Math.round(percentage*10)/10,"%");    // and to .txt
}
var myTotalArea=Math.round(sumOfArray(areaTOTAL))/squared(scaleFactor);
var total = sumOfArray(areaTOTAL)/squared(scaleFactor);
var pvar = new Variable(Template.GetCurrent(), ZONE_NAME + "_TOTAL_AREA", "Total area of zone", total);

var pvar = new Variable(Template.GetCurrent(), ZONE_NAME + "_PERCENTAGE_AREA_BELOW_LOWER_LIMIT", "Percentage of total area below lower limit specified", lower_limit_percent);
var pvar = new Variable(Template.GetCurrent(), ZONE_NAME + "_PERCENTAGE_AREA_ABOVE_UPPER_LIMIT", "Percentage of total area above upper limit specified", 100.0 - upper_limit_percent);

LogPrint("Total area = ",myTotalArea.toPrecision(4),"\n");
LogPrint("\n* All areas given to 4 significant figures; percentages to 1 d.p. *\n");
//f.Close();







//==  cover up anything outside perimeter ==//
if (hideOuter == 1) {

    // find a point on perim that can join up to origin without crossing over any other perim lines.
    myPerimCoords = PerimCoordinates.slice(0);  //use this to construct a polygon that equals the rectangle of image minus the inside of perimeter
    a=0;
    for (i=0; i < myPerimCoords.length; i=i) {
        perimXCoordsMinusOne = XPerimCoords.slice(0);
        perimXCoordsMinusOne.splice(a);
        perimYCoordsMinusOne = YPerimCoords.slice(0);
        perimYCoordsMinusOne.splice(a);
        pointIntersect=perimIntersection(0,0,PerimCoordinates["x"],PerimCoordinates["y"],perimXCoordsMinusOne,perimYCoordsMinusOne);
        if ( (pointIntersect==undefined) ) {
            break;
        }
        else {      //rearrange array of coords so correct point will be left at front.
            myPerimCoords.push(myPerimCoords.shift());     // y val
        myPerimCoords.push(myPerimCoords.shift());     // and x
            a++;    //so we know where we are in reference to original coords array
        }
    }
    
    myPerimCoords.push(0,0);
    myPerimCoords.unshift(0,0,0,(maxY+bordersize),(maxX+bordersize),(maxY+bordersize),(maxX+bordersize),0,0,0);
    
    img.lineStyle = Reporter.LineSolid;
    img.fillColour = "white";
    img.lineColour = "white";
    img.lineWidth = 1;
    img.Polygon(myPerimCoords);
}


img.fillColour = "none";
img.lineColour = "black";
img.lineStyle = Reporter.LineDashDot;
if (drawPerimeter == 1)  {
    img.Polygon(PerimCoordinates);       //draw perimeter
}

//draw grid etc over top of contour plot
if (drawGrid == 1)  {
    for (i = 0; i < gridToDrawX.length; i++)  {
        img.Polyline( gridToDrawX[i]+5, gridToDrawY[i]+5, gridToDrawX[i]-5,gridToDrawY[i]-5,gridToDrawX[i],gridToDrawY[i],gridToDrawX[i]+5,gridToDrawY[i]-5,gridToDrawX[i]-5,gridToDrawY[i]+5);
    }
}

for ( i=0; i < coordinatesGRID.length; i++ ) {
    if (writeHICS_all == 1)  {
        img.Text( coordinatesGRID[i]["x"],coordinatesGRID[i]["y"]," "+(10*Math.round(coordinatesGRID[i]["HIC"]/10)));
    }
    if (drawGrid_all == 1)  {
        img.Polyline( coordinatesGRID[i]["x"]+3, coordinatesGRID[i]["y"]+3, coordinatesGRID[i]["x"]-3,coordinatesGRID[i]["y"]-3,coordinatesGRID[i]["x"],coordinatesGRID[i]["y"],coordinatesGRID[i]["x"]+3,coordinatesGRID[i]["y"]-3,coordinatesGRID[i]["x"]-3,coordinatesGRID[i]["y"]+3);
    }
}

if (writeHICS == 1)  {
    for ( i=0; i < Xcoord.length; i++ ) {
        img.Text( Xcoord[i],Ycoord[i]," "+HIC[i]);
    }
}

//draw hull if required.
if (drawHull == 1) {
    img.fillColour = "none";
    img.lineColour = "red";
    img.lineStyle = Reporter.LineDash;
    img.Polygon(HULLxy);       //join the dots
}


//==  save the image ==//

var DEFAULT_DIR = Template.GetCurrent().GetVariableValue("DEFAULT_DIR");

img.Save(DEFAULT_DIR + "/" + ZONE_NAME + "_plot.png", Image.PNG);

}

//============================//
//==| DRAW THE CONTOUR BAR |==//
//============================//

//set width n height

//create image, make labels
var img = new Image(contourBarWidth+20,contourBarHeight+20);    //image sized according to perimeter dimensions, with border.
img.lineColour = "black";
img.lineWidth = 1;
img.fontSize = 13;

//specify values & associated colours
var vals = allContourValues;
//colours already defined too.

//draw,colour in & label
for (var i = 0; i < allContourValues.length; i++)  {
    var val = vals[i];
    img.fillColour = colours[i];
    img.lineColour = colours[i];
    img.lineWidth = 1;
    img.Polygon(0,10,  contourBarWidth/2,10,  contourBarWidth/2,contourBarHeight-(i*contourBarHeight/allContourValues.length)+10,  0,contourBarHeight-(i*contourBarHeight/allContourValues.length)+10);
    img.Text (contourBarWidth/2+10,contourBarHeight-((i-1)*contourBarHeight/allContourValues.length)-15,vals[i]+"")//label contour.
    drawContourLines = 0;   //for this particular HIC value, redefined each time.
    var majorHICvalue = 0;
    var contourThickness = 0;
    for ( var majorVal = 0; majorVal < majorContourValues.length; majorVal++ ) {
        if ((val == majorContourValues[majorVal])) {
            if (drawContourLines_Major == 1)  {
                var majorHICvalue = 1; //we flag it up if this iteration represents a "major" HIC val
            }
        }
    }
    if (majorHICvalue == 0) {   //if just a regular HIC val
        if (drawContourLines_Minor == 1) {  //if flagged to draw minor lines,       
            drawContourLines = 1;           //we make a note of this
            contourThickness = ThicknessContourLines_Minor;     //and set thickness accordingly
        }
    }
    else {  //if a "major" HIC val
        if (drawContourLines_Major == 1) {
            drawContourLines = 1; 
            contourThickness = ThicknessContourLines_Major;     //and set thickness accordingly
        }
        else if (drawContourLines_Minor == 1) {
            drawContourLines = 1; 
            contourThickness = ThicknessContourLines_Minor;     //and set thickness accordingly
        }
    }
    if (drawContourLines == 1) {
        img.lineColour = "black";
        img.lineWidth = contourThickness;
        img.Line(contourBarWidth/2,contourBarHeight-(i*contourBarHeight/allContourValues.length)+10,  0,contourBarHeight-(i*contourBarHeight/allContourValues.length)+10);
    }
}
img.Text (contourBarWidth/2+10,contourBarHeight-((allContourValues.length-1)*contourBarHeight/allContourValues.length)-15,">"+vals[vals.length-1])        //label last contour



//==  save the image ==//
var DEFAULT_DIR = Template.GetCurrent().GetVariableValue("DEFAULT_DIR");

img.Save(DEFAULT_DIR + "/" + ZONE_NAME + "_contour_plot.png", Image.PNG);






//___________________________________//
//>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>//
//============ FUNCTIONS ============//
//<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<//

function write_data_point(d, args, file, hic_array)
{
// Write a data point to hic_array. If any of the data is missing it is not written

    var x;
    var y;
    var hic;

    if ( (result = args[1].match(vregex) ) != null)
    {
        var str = result[1].toUpperCase();
        x = get_variable_value(str, d);
        if (x == undefined)
        {
            LogWarning("Missing X variable " + result[1] + " in file " + file);
            return;
        }
    }


    if ( (result = args[2].match(vregex) ) != null)
    {
        var str = result[1].toUpperCase();
        y = get_variable_value(str, d);
        if (y == undefined)
        {
            LogWarning("Missing Y variable " + result[1] + " in file " + file);
            return;
        }
    }

    if ( (result = args[3].match(vregex) ) != null)
    {
        var str = result[1].toUpperCase();
        hic = get_variable_value(str, d);
        if (hic == undefined)
        {
            LogWarning("Missing HIC variable " + result[1] + " in file " + file);
            return;
        }
    }

    var str = x + "," + y + "," + hic;

    hic_array.push(str);
}


//////////////////////////////////////////////////////////////////////

function get_variable_value(v, d)
{
// First look in the data we read from the reporter_variables file
    if (d[v] != undefined) return d[v];

// Not there so look for the variable value in the current template
    var tv = Template.GetCurrent().GetVariableValue(v);
    if (tv != null) return tv;

// Not there either so return undefined
    return undefined;
}
