/* Reporter script to create a data file for D3Plot from a variables file
   Miles Thornton   21/10/2005

PROGRAM::data_file_from_variables.js
DESC::Create a D3Plot data file from generated variables files
FOLDER::D3Plot data file
RETURN::text
ARG::Directory/filename::%DEFAULT_DIR%
ARG::Data file to create::%DEFAULT_DIR%/d3plot_data.file
ARG::Variable for data point X coord::%X%
ARG::Variable for data point Y coord::%Y%
ARG::Variable for data point Z coord::%Z%
ARG::Variable for data point value::%VALUE%
ARG::Link for reporter files (false = no link)::false
ARG::Link for html files (false = no link)::false
ARG::Link for pdf files (false = no link)::false
ARG::Additional data file (false = no file)::false
ARG::Data point name::External data
EXPAND_ARGS::false
END_INFO
*/

if (arguments.length < 10)
    throw Error("Wrong number of arguments to data_file_from_variables.js. Expected at least 10 but got " +
                arguments.length +
                "\n");

var filename = Template.GetCurrent().ExpandVariablesInString(arguments[0]);
filename = File.DriveMapFilename(filename, Include.NATIVE);

// Look for the variables file or directory
if (File.Exists(filename) == false)
    throw Error("Input file " + arguments[0] + " does not exist in data_file_from_variables.js\n");

var inputFiles = new Array();
var fileIn;
var fileOut;
var line;
var regex  = new RegExp("VAR\\s+([^!#]*)[!#]?\\s+DESC.*VALUE='(.*)'");
var vregex = new RegExp("%(\\w+)%");

// Check that x,y,z and value variables are sensible
if ( (result = arguments[2].match(vregex) ) == null)
    throw Error("Bad X variable '" + arguments[2] + "'");

if ( (result = arguments[3].match(vregex) ) == null)
    throw Error("Bad Y variable '" + arguments[3] + "'");

if ( (result = arguments[4].match(vregex) ) == null)
    throw Error("Bad Z variable '" + arguments[4] + "'");

if ( (result = arguments[5].match(vregex) ) == null)
    throw Error("Bad VALUE variable '" + arguments[5] + "'");


// Input is a directory so find any reporter_variables files recursively
// from this directory
if (File.IsDirectory(filename))
{
    inputFiles = File.FindFiles(filename, "reporter_variables", true);
}
// Input is a file so get the list of reporter_variables files from the file
else
{
    fileIn = new File(filename, File.READ);
    line;

    while ( (line = fileIn.ReadLine() ) != File.EOF)
    {
        inputFiles.push(line);
    }

    fileIn.Close();
}

// dataname is new optional argument for version 12
var dataname = "External data";
if (arguments.length > 10) dataname = arguments[10];

// Open output file
var filename = Template.GetCurrent().ExpandVariablesInString(arguments[1]);
filename = File.DriveMapFilename(filename, Include.NATIVE);

fileOut = new File(filename, File.WRITE);
fileOut.Write("name " + dataname +"\n");
fileOut.Write("$ \n");
fileOut.Write("$ Point data\n");

// Loop over all of the reporter variables.generated files

for (var i=0; i<inputFiles.length; i++)
{
    fileIn = new File(inputFiles[i], File.READ);

    var varData = new Object;

    while ( (line = fileIn.ReadLine() ) != File.EOF)
    {
        if ( (result = line.match(regex) ) != null)
            varData[result[1]] = result[2];
    }

    fileIn.Close();

// Write this data point to the file
    write_data_point(varData, arguments, inputFiles[i]);
}

// Add any additional data

var filename = Template.GetCurrent().ExpandVariablesInString(arguments[9]);
filename = File.DriveMapFilename(filename, Include.NATIVE);
if (arguments[9] != false && File.Exists(filename))
{
    fileIn = new File(filename, File.READ);

    while ( (line = fileIn.ReadLine() ) != File.EOF)
    {
        fileOut.Write(line + "\n");
    }

    fileIn.Close();
}

// Close the output file
fileOut.Close();

//////////////////////////////////////////////////////////////////////

function write_data_point(d, args, file)
{
// Write a data point. If any of the data is missing it is not written

    var x;
    var y;
    var z;
    var v;

    if ( (result = args[2].match(vregex) ) != null)
    {
        var str = result[1].toUpperCase();
        x = get_variable_value(str, d);
        if (x == undefined)
        {
            LogWarning("Missing X variable " + result[1] + " in file " + file);
            return;
        }
    }


    if ( (result = args[3].match(vregex) ) != null)
    {
        var str = result[1].toUpperCase();
        y = get_variable_value(str, d);
        if (y == undefined)
        {
            LogWarning("Missing Y variable " + result[1] + " in file " + file);
            return;
        }
    }

    if ( (result = args[4].match(vregex) ) != null)
    {
        var str = result[1].toUpperCase();
        z = get_variable_value(str, d);
        if (z == undefined)
        {
            LogWarning("Missing Z variable " + result[1] + " in file " + file);
            return;
        }
    }

    if ( (result = args[5].match(vregex) ) != null)
    {
        var str = result[1].toUpperCase();
        v = get_variable_value(str, d);
        if (v == undefined)
        {
            LogWarning("Missing value variable " + result[1] + " in file " + file);
            return;
        }
    }

    fileOut.Write("data " + x + " " + y + " " + z  + " " + v + "\n");

// Reporter hyperlink
    if (args[6] != "false")
    {
        v = expand_variables(args[6], d);
        fileOut.Write("link_opr " + v + "\n");
    }

// HTML hyperlink
    if (args[7] != "false")
    {
        v = expand_variables(args[7], d);
        fileOut.Write("link_html " + v + "\n");
    }

// pdf hyperlink
    if (args[8] != "false")
    {
        v = expand_variables(args[8], d);
        fileOut.Write("link_pdf " + v + "\n");
    }

    fileOut.Write("$\n");
}

//////////////////////////////////////////////////////////////////////

function get_variable_value(v, d)
{
// First look in the data we read from the reporter_variables file
    if (d[v] != undefined) return d[v];

// Not there so look for the variable value in the current template
    var tv = Template.GetCurrent().GetVariableValue(v);
    if (tv != null) return tv;

// Not there either so return undefined
    return undefined;
}

//////////////////////////////////////////////////////////////////////

function expand_variables(s, d)
{
// Expands any variables in the string

    var e = s;

    while ( (result = e.match(vregex) ) != null)
    {
         var search = new RegExp(result[0]);
         var replace = get_variable_value(result[1], d);
         if (replace == undefined) replace = "";

         e = e.replace(search, replace);
    }

    return e;
}
