/* Reporter script to obtain percentage final added mass from OTF file
   Dave Burton 13/4/2005

PROGRAM::check.js
DESC::Check on the quality of the run
FOLDER::OTF or d3hsp file
RETURN::text
ARG::OTF file name::%DEFAULT_DIR%/d3hsp
ARG::normal termination (false = no check)::true
ARG::% added initial mass limit (false = no check)::2
ARG::% added final mass limit (false = no check)::2
ARG::% total energy fluctuation limit (false = no check)::15
ARG::% hourglass energy limit (false = no check)::10
END_INFO
*/

if (arguments.length == 0) throw Error("no otf file argument\n");

// Look for any otf files from restarts e.g. otf01, otf02 etc
var otf_filename = File.DriveMapFilename(arguments[0], Include.NATIVE);
for (var i = 1; i < 99; i++) {
  var nstring = i.toString();
  if (nstring.length == 1) nstring = nstring.replace(/^/, "0");
  var test_filename = otf_filename + nstring;
  if (File.Exists(test_filename)) otf_filename = test_filename;
  else break;
}

var otf_file = new File(otf_filename, File.READ);
var line;
var mass_match_regex = new RegExp(
  "percentage increase\\s+=\\s+([\\w\\.\\-\\+]+)"
);
var term_match_regex = new RegExp("\\s+(.*)\\s+t e r m i n a t i o n");
var result;

var finl_added = 0;
var init_added = 0;
var init_added_limit = 0;
var finl_added_limit = 0;

var quality_ok = new Array();

// Set the initial values of the quality array
for (i = 0; i < 5; i++) {
  arguments[i + 1] = arguments[i + 1].toLowerCase();

  if (arguments[i + 1] == "false") {
    quality_ok[i] = 1;
  } else {
    quality_ok[i] = 0;
  }
}

// Figure out what is wanted to identify "quality" of run
if (arguments[2] != "false") {
  init_added_limit = parseFloat(arguments[2]);
}

if (arguments[3] != "false") {
  finl_added_limit = parseFloat(arguments[3]);
}

// Search through the otf file looking for the quality readings
while (
  (line = otf_file.FindLineContaining(
    "percentage increase",
    "t e r m i n a t i o n"
  )) != File.EOF
) {
  if ((result = line.match(mass_match_regex)) != null) {
    if (init_added_limit && !init_added) {
      init_added = parseFloat(result[1]);
    } else if (finl_added_limit) {
      finl_added = parseFloat(result[1]);
    }
  } else if ((result = line.match(term_match_regex)) != null) {
    var message = result[1];
    message = message.replace(/\s+/g, "");
  }
}
otf_file.Close();

// Calculate the quality of the run
if (arguments[1] != "false") {
  if (message == "Normal") {
    quality_ok[0] = 1;
    debug("Normal termination");
  } else {
    debug("NOT OK: Error termination");
  }
}

if (init_added_limit) {
  if (init_added < init_added_limit) {
    quality_ok[1] = 1;
    debug(
      "initial added mass: " +
        init_added +
        "% is less than " +
        init_added_limit +
        "%"
    );
  } else {
    debug(
      "NOT OK: initial added mass of " +
        init_added +
        "% is GREATER than " +
        init_added_limit +
        "%"
    );
  }
}

if (finl_added_limit) {
  if (finl_added < finl_added_limit) {
    quality_ok[2] = 1;
    debug(
      "final added mass: " +
        finl_added +
        "% is less than " +
        finl_added_limit +
        "%"
    );
  } else {
    debug(
      "NOT OK: final added mass of " +
        finl_added +
        "% is GREATER than " +
        finl_added_limit +
        "%"
    );
  }
}

// Now open glstat file to find out additional quality data
otf_filename = File.DriveMapFilename(otf_filename, Include.NATIVE);
otf_filename = File.SimplifyName(otf_filename);

var root_directory = File.Directory(otf_filename);

var glstat_filename = root_directory + "/glstat";
var glstat_file = new File(glstat_filename, File.READ);
var hg_match_regex = new RegExp("hourglass energy\\s+\\.+\\s+(.+)");
var te_match_regex = new RegExp("total energy\\.+\\s+(.+)");

var energy_init = 0;
var energy_fluc = 0;
var hourg_fluc = 0;
var energy_fluc_limit = 0;
var hourg_fluc_limit = 0;

if (arguments[4] != "false") {
  energy_fluc_limit = parseFloat(arguments[4]);
}

if (arguments[5] != "false") {
  hourg_fluc_limit = parseFloat(arguments[5]);
}

// Search through the glstat file looking for the quality readings
while (
  (line = glstat_file.FindLineContaining("hourglass energy", "total energy")) !=
  File.EOF
) {
  if ((result = line.match(te_match_regex)) != null) {
    if (!energy_init) {
      energy_init = parseFloat(result[1]);
    } else {
      energy_fluc = parseFloat(result[1]);
    }
  }
  if ((result = line.match(hg_match_regex)) != null) {
    hourg_fluc = parseFloat(result[1]);
  }
}
glstat_file.Close();

// If we have asked for hourglass checks, and there is no hourglass
// then give an error detail
if (hourg_fluc_limit && !hourg_fluc) {
  debug("NOT OK: HOURGLASS CHECK REQUESTED, BUT NO HOURGLASS IN GLSTAT FILE");
  quality_ok[4] = 0;
} else if (hourg_fluc_limit) {
  hourg_fluc = Math.abs(100 - (hourg_fluc * 100) / energy_init);

  if (hourg_fluc < hourg_fluc_limit) {
    quality_ok[4] = 1;
    debug(
      "hourglass " + hourg_fluc + "% is less than " + hourg_fluc_limit + "%"
    );
  } else {
    debug(
      "NOT OK: hourglass " +
        hourg_fluc +
        "% is GREATER than " +
        hourg_fluc_limit +
        "%"
    );
  }
}

// Total energy percentage check
if (energy_fluc_limit) {
  energy_fluc = Math.abs(100 - (energy_fluc * 100) / energy_init);

  if (energy_fluc < energy_fluc_limit) {
    quality_ok[3] = 1;
    debug(
      "total energy " +
        energy_fluc +
        "% is less than " +
        energy_fluc_limit +
        "%"
    );
  } else {
    debug(
      "NOT OK: total energy " +
        energy_fluc +
        "% is GREATER than " +
        energy_fluc_limit +
        "%"
    );
  }
}

// Final sum of checks
var total_quality = 0;

for (i = 0; i < 5; i++) {
  total_quality += quality_ok[i];
}

if (total_quality != 5) output("Not OK");
if (total_quality == 5) output("OK");
