// Script to read material properties from Primer Part Table and
// calculate yield fraction and failure strain fraction
//
// name:Yield
// description:Calculate yield fraction
//   
//
//New UI:
Window.Theme(Window.THEME_CURRENT);

var version = Utils.Version();
var install_dir = GetInstallDirectory();	

// Utils.Version returns a version in the form of XX.Y or XX
// The executable has the path in the for of XX_Y so we replace the . with a _ if exists
// It is assumed that d3plot and primer executables are in the same install directory
if (Unix())
{
    var primer = install_dir + "/" + "primer" + version.toString().replaceAll(".", "_") +"_64.exe";
}
else
{
    // Windows installations use x64
    var primer = install_dir + "\\" + "primer" + version.toString().replaceAll(".", "_") + "_x64.exe";
}

// Get Install Directory & Version and make it equals to var primer

if (!File.Exists(primer))
{
    primer = Window.GetFilename("Cannot find Primer", "Please give location of Primer executable", "exe");
    if (primer == null) Exit();
}

//  Find the keyword file
var filename = Window.GetFilename("Select file", "Keyword file?");
if (filename == null) Exit();
var direc = DirFromFilename(filename);

//  Assemble name of part table csv file, delete it if already exists
var pfile = direc + "temp.csv";
File.Delete(pfile);

//  Write command file
var cfile = direc + "temp.tcf";
var f = new File(cfile, File.WRITE);
f.Writeln("/READ DK " + '"' + filename + '"' + " 1");
f.Writeln("/PART_INFO NO " + '"' + pfile + '"' + " 1");
f.Close();

//  Call Primer and apply command file
if (Windows())
{
    var cmd = '""' + primer + '"' + ' -cf="' + cfile + '" -d=batch -exit"';    
}
else
{
    var cmd = "\"" + primer + "\"" + " -cf=\"" + cfile + "\" -d=batch -exit";
}
System(cmd);

// Check that the part_info file was created
if(!File.Exists(pfile))
{
  Warning("The part info csv file did not get written for some reason, exiting script");
  Exit();
}

//  Open the part_info file
var f2 = new File(pfile, File.READ);

var words = new Array();
var line = f2.ReadLongLine();

//  Split the header line into fields
words = line.split(",");

//   Find which columns contain yield stress and failure strain

var iyld = 0;   // column index for yield stress
var ifls = 0;   // column index for failure strain

for (var i=0; i<words.length; i++)
{
   if( words[i].match("Yield") )       iyld = i;
   if( words[i].match("Fail strain") ) ifls = i;
}

//  Check we have found the columns for yield and failure strain
if (iyld==0 || ifls==0) 
{
   Window.Error("Error message","Cannot find columns for yield & failure strain", Window.OK);
   Exit();
}

//  Arrays for storing material data (order of parts as in csv file)

var part_ID = new Array();
var yld_arr = new Array();
var failstr = new Array;

//   Keep reading lines from the file until there are no more
//   For each line, echo it to the dialog box

var icount = 0;

while ( (line = f2.ReadLongLine()) != undefined )
{
   words = line.split(",");
   if (words.length > 16)
   {
       icount++;
       part_ID[icount] = parseInt(words[0]);   // Read Part ID from first column
       
       yld_arr[icount] = 0.0;                    // Read yield, check it is a number 
       if (words[iyld] != "<undefined>") yld_arr[icount] = parseFloat(words[iyld]);
       if (isNaN(yld_arr[icount])) yld_arr[icount] = 0.0;
       
       failstr[icount] = 0.0;                  // Read fail strain, check it is a number
       if (words[ifls] != "<undefined>") failstr[icount] = parseFloat(words[ifls]);
       if (isNaN(failstr[icount])) failstr[icount] = 0.0;
       
    }
}

//  Close the file
f2.Close();

SetCurrentModel(1);                // Assume we are working with Model 1
var nshell = GetNumberOf(SHELL);   // Find number of shells in model
var nsolid = GetNumberOf(SOLID);   // Find number of solids in model
var npart  = GetNumberOf(PART);    // Find number of parts in model
var nstate = GetNumberOf(STATE);   // Get number of time-states in results file

//  Get the part IDs from D3PLOT; match to those from the csv file
var order = new Array();
for (i=1; i<=npart; i++)
{
   var pid = GetLabel(PART, i);
   order[i] = -1;
   for (var j=0; j<=icount; j++) if (part_ID[j]==pid) order[i] = j;
}

var yield_cum = new Array();        // Initialize variable for max yld ratio shells
for (i=0; i<nshell; i++) yield_cum[i] = 0.0;
var yield_cus = new Array();        // Initialize variable for max yld ratio solids
for (i=0; i<nsolid; i++) yield_cus[i] = 0.0;

//  Create UBIN components for failure indices
var icompy = CreateUbinComponent("Yield ratio", U_SOSH, U_SCALAR, REPLACE);
var icompc = CreateUbinComponent("Yield cumulative", U_SOSH, U_SCALAR, REPLACE);
var icompf = CreateUbinComponent("Fail strain ratio", U_SOSH, U_SCALAR, REPLACE);

//  Loop through states
for (var istate=1; istate<=nstate; istate++) 
{
   SetCurrentState(istate);
   Message("...processing state " + istate);

//  Loop through shells
   for (j=1; j<=nshell; j++)
   {
      var ipid = GetPid(SHELL, j);  // find part, linking to one from the table
      var jpid = order[ipid];
      if (jpid > 0)
      {

//  Find yield and failure strain from table
         var yld = yld_arr[jpid];
         var epf = failstr[jpid];
         
         for (var isurf=1; isurf<4; isurf++)   // Check top, middle and bottom surfaces
         {
            var svm = GetData(SVON, SHELL, j, isurf);  // Get Von Mises stress
            var epl = GetData(EPL,  SHELL, j, isurf);  // Get plastic strain
	    
            var test = 0.0;                            // calculate yield ratio
	    if (yld > 0.0) test = svm/yld; 
            yield_cum[j] =Math.max (yield_cum[j], test);

            PutUbinData(icompy, SHELL, j, isurf, test);  // store yield ratio
            PutUbinData(icompc, SHELL, j, isurf, yield_cum[j]);  // store yield ratio
	    
            var test = 0.0;                            // calculate strain ratio
	    if (epf > 0.0) test = epl/epf; 

            PutUbinData(icompf, SHELL, j, isurf, test);  // store strain ratio
         }
      }
   }

//  Loop through solids
   for (j=1; j<=nsolid; j++)
   {
      ipid = GetPid(SOLID, j);  // find part, linking to one from the table
      jpid = order[ipid];
      if (jpid > 0)
      {

//  Find yield and failure strain from table
         yld = yld_arr[jpid];
         epf = failstr[jpid];
         
         var svm = GetData(SVON, SOLID, j, 0);            // Get Von Mises stress
         var epl = GetData(EPL,  SOLID, j, 0);            // Get plastic strain
  
         var test = 0.0;                                  // calculate yield ratio
         if (yld > 0.0) test = svm/yld; 
         yield_cus[j] =Math.max (yield_cus[j], test);

         PutUbinData(icompy, SOLID, j, 1, test);          // store yield ratio
         PutUbinData(icompc, SOLID, j, 1, yield_cus[j]); 
	    
         var test = 0.0;                                  // calculate strain ratio
         if (epf > 0.0) test = epl/epf; 

         PutUbinData(icompf, SOLID, j, 1, test);          // store strain ratio
      }
   }

//  Next state
}

Exit();

////////////////////////////////////////////////////////////////////////////////

function DirFromFilename(filename)
{
// Extract directory from keyword filename
// Remove anything after a '/' or '\'
    var dir = filename.replace(/[^\/\\]+$/, "");

// If no match then we have no directory so return blank string
    if (dir == filename) dir = "";

    return dir;
}


